/*
 * Copyright © 2021-present Arcade Data Ltd (info@arcadedata.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-FileCopyrightText: 2021-present Arcade Data Ltd (info@arcadedata.com)
 * SPDX-License-Identifier: Apache-2.0
 */
package com.arcadedb.query.sql.executor;

import com.arcadedb.database.Database;
import com.arcadedb.exception.CommandExecutionException;
import com.arcadedb.exception.TimeoutException;
import com.arcadedb.schema.DocumentType;

/**
 * <p> This step is used just as a gate check to verify that a bucket belongs to a class. </p> <p> It accepts two values: a target
 * bucket (name or OCluster) and a class. If the bucket belongs to the class, then the syncPool() returns an empty result set,
 * otherwise it throws an PCommandExecutionException </p>
 *
 * @author Luigi Dell'Aquila (luigi.dellaquila-(at)-gmail.com)
 */
public class CheckClusterTypeStep extends AbstractExecutionStep {
  final String bucketName;
  final String targetType;
  boolean found = false;

  public CheckClusterTypeStep(final String targetBucketName, final String typez, final CommandContext context) {
    super(context);
    this.bucketName = targetBucketName;
    this.targetType = typez;
  }

  @Override
  public ResultSet syncPull(final CommandContext context, final int nRecords) throws TimeoutException {
    pullPrevious(context, nRecords);
    final long begin = context.isProfiling() ? System.nanoTime() : 0;
    try {
      if (found)
        return new InternalResultSet();

      final Database db = context.getDatabase();

      final com.arcadedb.engine.Bucket bucketObj;
      if (bucketName != null)
        bucketObj = db.getSchema().getBucketByName(bucketName);
      else
        bucketObj = null;

      if (bucketObj == null)
        throw new CommandExecutionException("Bucket '" + bucketName + "' not found");

      final DocumentType typez = db.getSchema().getType(targetType);
      if (typez == null)
        throw new CommandExecutionException("Type '" + targetType + "' not found");

      for (final Integer bucketId : typez.getBucketIds(true)) {
        if (bucketId == bucketObj.getFileId()) {
          found = true;
          break;
        }
      }
      if (!found)
        throw new CommandExecutionException("Bucket " + bucketObj.getFileId() + " does not belong to the type " + targetType);

      return new InternalResultSet();
    } finally {
      if ( context.isProfiling() )
        cost += (System.nanoTime() - begin);
    }
  }

  @Override
  public String prettyPrint(final int depth, final int indent) {
    final String spaces = ExecutionStepInternal.getIndent(depth, indent);
    final StringBuilder result = new StringBuilder();
    result.append(spaces);
    result.append("+ CHECK TARGET BUCKET FOR USERTYPE");
    if ( context.isProfiling() )
      result.append(" (").append(getCostFormatted()).append(")");

    result.append("\n");
    result.append(spaces);
    result.append("  ").append(this.targetType);
    return result.toString();
  }
}
