#!/bin/bash
#
# Copyright 2024 The Chromium Authors, the AUR, and Alex313031
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

# Let the wrapped binary know that it has been run through the wrapper.
export CHROME_WRAPPER="`readlink -f "$0"`"

HERE="`dirname "$CHROME_WRAPPER"`"

# We include some xdg utilities next to the binary, and we want to prefer them
# over the system versions when we know the system versions are very old. We
# detect whether the system xdg utilities are sufficiently new to be likely to
# work for us by looking for xdg-settings. If we find it, we leave $PATH alone,
# so that the system xdg utilities (including any distro patches) will be used.
if ! command -v xdg-settings &> /dev/null; then
  # Old xdg utilities. Prepend $HERE to $PATH to use ours instead.
  export PATH="$HERE:$PATH"
else
  # Use system xdg utilities. But first create mimeapps.list if it doesn't
  # exist; some systems have bugs in xdg-mime that make it fail without it.
  xdg_app_dir="${XDG_DATA_HOME:-$HOME/.local/share/applications}"
  mkdir -p "$xdg_app_dir"
  [ -f "$xdg_app_dir/mimeapps.list" ] || touch "$xdg_app_dir/mimeapps.list"
fi

# Always use our versions of ffmpeg libs.
# This also makes RPMs find the compatibly-named library symlinks.
if [[ -n "$LD_LIBRARY_PATH" ]]; then
  LD_LIBRARY_PATH="$HERE/lib:$LD_LIBRARY_PATH"
else
  LD_LIBRARY_PATH="$HERE/lib"
fi
export LD_LIBRARY_PATH

# Clean up old crash reports (see https://bugs.debian.org/1015931)
# after 30 days.
find "$(pwd)/.config/thorium/Crash Reports/pending/" -mtime +30 \
  \( -name "*.meta" -o -name "*.dmp" \) -exec rm \{\} \;

# APPNAME for GTK.
APPNAME=thorium

# Set DESKTOP variable
# DESKTOP="thorium-portable"

# Set XDG Title variable
TITLE="Thorium Portable"

# Set the correct file name for the desktop file
export CHROME_DESKTOP="thorium-portable.desktop"

# Set CHROME_VERSION_EXTRA text, which is displayed in the About dialog on chrome://help
# DIST=`cat /etc/debian_version`
export CHROME_VERSION_EXTRA="stable, (Portable)"

# We don't want bug-buddy intercepting our crashes. http://crbug.com/24120
export GNOME_DISABLE_CRASH_DIALOG=SET_BY_THORIUM

# Set config home.
XDG_CONFIG_HOME=${XDG_CONFIG_HOME:-$(pwd)/.config}

usage () {
  echo "thorium-browser [-h|--help] [--temp-profile] [options] [URL]"
  echo
  echo "        -h, -help, or --help       This help screen"
  echo "        --temp-profile             Start with a new and temporary profile"
  echo "        --safe-mode                Disable all chrome://flags flags"
  echo
  echo " Other supported options are:"
  MANWIDTH=80 man thorium-browser | sed -e '1,/OPTIONS/d; /ENVIRONMENT/,$d'
  echo " See 'man thorium-browser' for more details"
}

want_temp_profile=0
is_safe_mode=0
while [ $# -gt 0 ]; do
  case "$1" in
    -h | --help | -help )
      usage
      exit 0 ;;
    --temp-profile )
      want_temp_profile=1
      shift ;;
    --safe-mode )
      is_safe_mode=1
      shift ;;
    -- ) # Stop option processing
      shift
      break ;;
    * ) # Else
      break ;;
  esac
done

# Allow users to override command-line options with a file.
if [[ -f $XDG_CONFIG_HOME/thorium-flags.conf ]]; then
   CHROME_USER_FLAGS="$(cat $XDG_CONFIG_HOME/thorium-flags.conf)"
fi

# Sanitize std{in,out,err} because they'll be shared with untrusted child
# processes (http://crbug.com/376567).
exec < /dev/null
exec > >(exec cat)
exec 2> >(exec cat >&2)

if [ $want_temp_profile -eq 1 ] ; then
  TEMP_PROFILE=`mktemp -d`
  echo "Using temporary profile: $TEMP_PROFILE"
  PROFILE="$TEMP_PROFILE"
  CACHE="$TEMP_PROFILE/cache"
  export PROFILE
  export CACHE
  CHROME_USER_FLAGS="$CHROME_USER_FLAGS"
else
  PROFILE="$HERE/.config/thorium"
  CACHE="$HERE/.config/cache"
  export PROFILE
  export CACHE
  CHROME_USER_FLAGS="$CHROME_USER_FLAGS"
fi

if [ $is_safe_mode -eq 1 ] ; then
  CHROME_USER_FLAGS="$CHROME_USER_FLAGS --no-experiments"
fi

# Launch Thorium
# Note: exec -a below is a bashism.
exec -a "$0" "$HERE/thorium" "--disable-machine-id" "--disable-encryption" "--user-data-dir=$PROFILE" "--disk-cache-dir=$CACHE" "$CHROME_USER_FLAGS" "$@"
