//
// Copyright (c) Microsoft. All rights reserved.
// See https://aka.ms/csspeech/license for the full license information.
//

#import "SPXFoundation.h"
#import "SPXDialogServiceConnectorActivityReceivedEventArgs.h"
#import "SPXAudioConfiguration.h"
#import "SPXAutoDetectSourceLanguageConfiguration.h"
#import "SPXKeywordRecognitionModel.h"
#import "SPXSourceLanguageConfiguration.h"
#import "SPXDialogServiceConfiguration.h"
#import "SPXDialogServiceConnector.h"
#import "SPXDialogServiceConnectorTurnStatusReceivedEventArgs.h"
#import "SPXSpeechRecognitionResult.h"
#import "SPXSpeechRecognitionEventArgs.h"
#import "SPXSessionEventArgs.h"

/**
 * An object that communicates with a speech-enabled dialog system using either the Bot Framework or Custom Commands.
 * This type receives speech-to-text results and also facilitates the asynchronous sending and receiving of non-speech
 * dialog activity data.
 *
 * See also: 
 * [Integrate with a client application using Speech SDK](/azure/cognitive-services/speech-service/how-to-custom-commands-setup-speech-sdk)
 * [Voice assistants frequently asked questions](/azure/cognitive-services/speech-service/faq-voice-assistants)
 * [What is Custom Commands?](/azure/cognitive-services/speech-service/custom-commands)
 */
SPX_EXPORT
@interface SPXDialogServiceConnector : SPXRecognizer

/**
 * Handler type provided to listenOnceAsync that receives speech-to-text results upon completion.
 */
typedef void (^SPXDialogServiceConnectorRecognitionResultHandler)(SPXSpeechRecognitionResult * _Nonnull);

/**
 * Handler type provided to sendActivityAsync that receives interaction identifiers, as strings, upon completion.
 */
typedef void (^SPXDialogServiceConnectorInteractionIdHandler)(NSString * _Nonnull);

/**
 * Handler type provided to async methods without async result data that is invoked when an asynchronous stop operation
 * completes.
 */
typedef void (^SPXDialogServiceConnectorAsyncCompletionHandler)(void);

/**
 * Handler type provided to receieve event information for intermediate and final speech-to-text results.
 */
typedef void (^SPXDialogServiceConnectorRecognitionEventHandler)(SPXDialogServiceConnector * _Nonnull, SPXSpeechRecognitionEventArgs * _Nonnull);

/**
 * Handler type provided to receive event information for cancellation events as raised when errors or end of input
 * occurs.
 */
typedef void (^SPXDialogServiceConnectorCanceledEventHandler)(SPXDialogServiceConnector * _Nonnull, SPXSpeechRecognitionCanceledEventArgs * _Nonnull);

/**
 * Handler type provided to receive turn status information from the connected dialog implementation.
 */
typedef void (^SPXDialogServiceConnectorTurnStatusReceivedEventHandler)(SPXDialogServiceConnector * _Nonnull, SPXDialogServiceConnectorTurnStatusReceivedEventArgs * _Nonnull);

/**
 * Handler type provided to receive activity events from connected dialog implementation.
 */
typedef void (^SPXDialogServiceConnectorActivityReceivedEventHandler)(SPXDialogServiceConnector * _Nonnull, SPXDialogServiceConnectorActivityReceivedEventArgs * _Nonnull);

/**
 * The authorization token used to communicate with the speech service.
 *
 * Note: The caller needs to ensure that the authorization token is valid. Before the authorization token expires,
 * the caller needs to refresh it by calling this setter with a new valid token.
 */
@property (nonatomic, copy, nullable)NSString *authorizationToken;

/**
 * A JSON template for activity payload data that will be provided to the speech service for the next conversation.
 * The service will attempt to merge this template into all activities sent to the dialog backend, whether originated
 * by the client with SendActivityAsync or generated by the service, as is the case with speech-to-text results. This
 * allows a client to ensure consistent metadata information is provided to the dialog implementation across
 * interactions.
 */
@property (nonatomic, copy, nullable)NSString *speechActivityTemplate;

/**
 * Initializes a new instance of dialog service connector that will use the default audio input device for any
 * speech-to-text interactions.
 *
 * @param dialogServiceConfiguration the configuration object that includes information about the selected dialog
 * system.
 * @param outError error information.
 * @return an instance of a dialog service connector.
 */
- (nullable instancetype)initWithDialogServiceConfiguration
             :(nonnull SPXDialogServiceConfiguration *)dialogServiceConfiguration
        error:(NSError * _Nullable * _Nullable)outError;

/**
 * Initializes a new instance of dialog service connector that will use the default audio input device for any
 * speech-to-text interactions.
 *
 * @param dialogServiceConfiguration the configuration object that includes information about the selected dialog
 * system.
 * @return an instance of a dialog service connector.
 */
- (nonnull instancetype)initWithDialogServiceConfiguration
        :(nonnull SPXDialogServiceConfiguration *)dialogServiceConfiguration
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Initializes a new instance of dialog service connector using the specified audio configuration.
 *
 * @param dialogServiceConfiguration the configuration object that includes information about the selected dialog
 * system.
 * @param audioConfiguration the configuration object for input audio as used by speech-to-text interactions.
 * @param outError error information.
 * @return an instance of a dialog service connector.
 */
- (nullable instancetype)initWithDialogServiceConfiguration
                          :(nonnull SPXDialogServiceConfiguration *)dialogServiceConfiguration
        audioConfiguration:(nonnull SPXAudioConfiguration *)audioConfiguration
                     error:(NSError * _Nullable * _Nullable)outError;

/**
 * Initializes a new instance of dialog service connector using the specified audio configuration.
 *
 * @param dialogServiceConfiguration the configuration object that includes information about the selected dialog
 * system.
 * @param audioConfiguration the configuration object for input audio as used by speech-to-text interactions.
 * @return an instance of a dialog service connector.
 */
- (nonnull instancetype)initWithDialogServiceConfiguration
                          :(nonnull SPXDialogServiceConfiguration *)dialogServiceConfiguration
        audioConfiguration:(nonnull SPXAudioConfiguration *)audioConfiguration
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Begins a speech-to-text interaction with this connector and blocks until a single speech-to-text final result is
 * received. The speech-to-text result received is also provided to the configured dialog implementation and that dialog
 * system may produce any number of activity payloads in response to the speech interaction. Speech interactions may be
 * correlated with activities via dialog-specific data in the activity payload.
 *
 * @param outError error information.
 * @return the speech-to-text result from the single-shot speech interaction.
 */
- (nullable SPXSpeechRecognitionResult *)listenOnce:(NSError * _Nullable * _Nullable)outError NS_RETURNS_RETAINED;

/**
 * Begins a speech-to-text interaction with this connector and blocks until a single speech-to-text final result is
 * received. The speech-to-text result received is also provided to the configured dialog implementation and that dialog
 * system may produce any number of activity payloads in response to the speech interaction. Speech interactions may be
 * correlated with activities via dialog-specific data in the activity payload.
 *
 * @return the speech-to-text result from the single-shot speech interaction.
 */
- (nonnull SPXSpeechRecognitionResult *)listenOnce
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.") NS_RETURNS_RETAINED;

/**
 * Asynchronously begins a speech-to-text interaction with this connector and immediately returns execution to the
 * calling thread. When received, speech-to-text results may be processed by the provided result handler or retrieved
 * via a subscription to the recognized event. The speech-to-text result produced by this operation is also provided to
 * the configured dialog implementation and that dialog system may produce any number of activity payloads in response
 * to the speech interaction. Speech interactions may be correlated with activities via dialog-specific data in the
 * activity payload.
 *
 * @param resultReceivedHandler the handler function called when a final speech-to-text result is received.
 * @param outError error information.
 * @return a value indicating whether the request to listen started successfully. If NO, additional detail may be
 * available in outError.
 */
- (BOOL)listenOnceAsync
             :(nonnull SPXDialogServiceConnectorRecognitionResultHandler)resultReceivedHandler
        error:(NSError * _Nullable * _Nullable)outError;

/**
 * Asynchronously begins a speech-to-text interaction with this connector and immediately returns execution to the
 * calling thread. When received, speech-to-text results may be processed by the provided result handler or retrieved
 * via a subscription to the recognized event. The speech-to-text result produced by this operation is also provided to
 * the configured dialog implementation and that dialog system may produce any number of activity payloads in response
 * to the speech interaction. Speech interactions may be correlated with activities via dialog-specific data in the
 * activity payload.
 *
 * @param resultReceivedHandler the handler function called when a final speech-to-text result is received.
 */
- (void)listenOnceAsync:(nonnull SPXDialogServiceConnectorRecognitionResultHandler)resultReceivedHandler
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Requests an immediate stop to any active listening operation. This may interrupt a speech-to-text interaction in
 * progress and any speech-to-text result received may represent an incomplete speech input.
 *
 * Synchronous methods should not be called when handling an event. Use stopListeningAsync if a stop is desired in
 * response to an event.
 *
 * @param outError error information.
 * @return a value indicating whether a request to stop was received successfully. If NO, additional information may be
 * available in outError.
 */
- (BOOL)stopListening:(NSError * _Nullable * _Nullable)outError;

/**
 * Requests an immediate stop to any active listening operation. This may interrupt a speech-to-text interaction in
 * progress and any speech-to-text result received may represent an incomplete speech input.
 *
 * Synchronous methods should not be called when handling an event. Use stopListeningAsync if a stop is desired in
 * response to an event.
 */
- (void)stopListening
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Requests an immediate stop to any active listening operation. This may interrupt a speech-to-text interaction in
 * progress and any speech-to-text result received may represent an incomplete speech input.
 *
 * @param completionHandler the handler function called when the stop request has completed.
 * @param outError error information.
 * @return a value indicating whether a request to stop was received successfully. If NO, additional information may be
 * available in outError.
 */
- (BOOL)stopListeningAsync
             :(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
        error:(NSError * _Nullable * _Nullable)outError;
        
/**
 * Requests an immediate stop to any active listening operation. This may interrupt a speech-to-text interaction in
 * progress and any speech-to-text result received may represent an incomplete speech input.
 *
 * @param completionHandler the handler function called when the stop request has completed.
 */
- (void)stopListeningAsync:(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Sends a data payload to dialog implementation that this DialogServiceConnector instance is connected to. This is
 * usually a JSON document with its schema determined by the dialog implementation and the contents of a sent activity
 * should be populated with knowledge about the format and content expectations of the dialog system.
 *
 * Sent activities are not associated with any other interaction and will generate their own standalone interaction
 * identifier when processed by the service. Correlation of conversations and other interactions should be
 * accomplished via the activity payload itself using the capabilities of the dialog implementation used.
 *
 * See also:
 * <a href="https://github.com/microsoft/botframework-sdk/blob/main/specs/botframework-activity/botframework-activity.md">Bot Framework -- Activity schema</a>
 * <a href="https://docs.microsoft.com/azure/cognitive-services/speech-service/quickstarts/voice-assistants">Voice assistant quickstart (with SendActivityAsync example)</a>
 *
 * @param activity the serialized payload of an activity to send.
 * @param outError error information.
 * @return an interaction identifier acquired when the activity is sent to the service. This may occur before the
 * activity is processed and evaluated by the dialog implementation and the receipt of an interaction identifier does
 * not indicate any success or failure in processing the activity. Information about success or failure may be obtained
 * via response activities with correlation data or with TurnStatusReceived events that correlate to this interaction
 * identifier.
 */
- (nullable NSString *)sendActivity
             :(nonnull NSString *)activity
        error:(NSError * _Nullable * _Nullable)outError;

/**
 * Sends a data payload to dialog implementation that this DialogServiceConnector instance is connected to. This is
 * usually a JSON document with its schema determined by the dialog implementation and the contents of a sent activity
 * should be populated with knowledge about the format and content expectations of the dialog system.
 *
 * Sent activities are not associated with any other interaction and will generate their own standalone interaction
 * identifier when processed by the service. Correlation of conversations and other interactions should be
 * accomplished via the activity payload itself using the capabilities of the dialog implementation used.
 *
 * See also:
 * <a href="https://github.com/microsoft/botframework-sdk/blob/main/specs/botframework-activity/botframework-activity.md">Bot Framework -- Activity schema</a>
 * <a href="https://docs.microsoft.com/azure/cognitive-services/speech-service/quickstarts/voice-assistants">Voice assistant quickstart (with SendActivityAsync example)</a>
 *
 * @param activity the serialized payload of an activity to send.
 * @return an interaction identifier acquired when the activity is sent to the service. This may occur before the
 * activity is processed and evaluated by the dialog implementation and the receipt of an interaction identifier does
 * not indicate any success or failure in processing the activity. Information about success or failure may be obtained
 * via response activities with correlation data or with TurnStatusReceived events that correlate to this interaction
 * identifier.
 */
- (nonnull NSString *)sendActivity:(nonnull NSString *)activity
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Sends a data payload to dialog implementation that this DialogServiceConnector instance is connected to. This is
 * usually a JSON document with its schema determined by the dialog implementation and the contents of a sent activity
 * should be populated with knowledge about the format and content expectations of the dialog system.
 *
 * Sent activities are not associated with any other interaction and will generate their own standalone interaction
 * identifier when processed by the service. Correlation of conversations and other interactions should be
 * accomplished via the activity payload itself using the capabilities of the dialog implementation used.
 *
 * @param activity the serialized payload of an activity to send.
 * @param interactionIdReceivedHandler a handler function called when the activity is sent to the service. This may occur
 * before the activity is processed and evaluated by the dialog implementation and the receipt of an interaction
 * identifier does not indicate any success or failure in processing the activity. Information about success or failure
 * may be obtained via response activities with correlation data or with TurnStatusReceived events that correlate to
 * this interaction identifier.
 * @param outError error information.
 * @return a value indicating whether the request to send an activity started successfully. If NO, additional
 * information may be available in outError.
 */
- (BOOL)sendActivityAsync
                                    :(NSString * _Nonnull)activity
        interactionIdReceivedHandler:(nonnull SPXDialogServiceConnectorInteractionIdHandler)interactionIdReceivedHandler
                               error:(NSError * _Nullable * _Nullable)outError;

/**
 * Sends a data payload to dialog implementation that this DialogServiceConnector instance is connected to. This is
 * usually a JSON document with its schema determined by the dialog implementation and the contents of a sent activity
 * should be populated with knowledge about the format and content expectations of the dialog system.
 *
 * Sent activities are not associated with any other interaction and will generate their own standalone interaction
 * identifier when processed by the service. Correlation of conversations and other interactions should be
 * accomplished via the activity payload itself using the capabilities of the dialog implementation used.
 *
 * See also:
 * <a href="https://github.com/microsoft/botframework-sdk/blob/main/specs/botframework-activity/botframework-activity.md">Bot Framework -- Activity schema</a>
 * <a href="https://docs.microsoft.com/azure/cognitive-services/speech-service/quickstarts/voice-assistants">Voice assistant quickstart (with SendActivityAsync example)</a>
 *
 * @param activity the serialized payload of an activity to send.
 * @param interactionIdReceivedHandler a handler function called when the activity is sent to the service. This may occur
 * before the activity is processed and evaluated by the dialog implementation and the receipt of an interaction
 * identifier does not indicate any success or failure in processing the activity. Information about success or failure
 * may be obtained via response activities with correlation data or with TurnStatusReceived events that correlate to
 * this interaction identifier.
 */
- (void)sendActivityAsync
                                    :(NSString * _Nonnull)activity
        interactionIdReceivedHandler:(nonnull SPXDialogServiceConnectorInteractionIdHandler)interactionIdReceivedHandler
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Begins a speech-to-text interaction with this connector using a keyword. This interaction will use the provided
 * keyword model to listen for a keyword indefinitely, during which audio is not sent to the speech service and all
 * processing is performed locally. When a keyword is recognized, the DialogServiceConnector will automatically connect
 * to the speech service and begin sending audio data from just before the keyword as if listenOnceAsync() were invoked.
 * When received, speech-to-text results may be processed by the provided result handler or retrieved via a subscription
 * to the recognized event. The speech-to-text result produced by this operation is also provided to the configured
 * dialog implementation and that dialog system may produce any number of activity payloads in response to the speech
 * interaction. Speech interactions may be correlated with activities via dialog-specific data in the activity payload.
 *
 * @param keywordModel the keyword recognition model.
 * @param outError error information.
 * @return a value indicating whether the requested keyword recognition successfully started. If NO, outError may
 * contain additional information.
 */
- (BOOL)startKeywordRecognition
             :(nonnull SPXKeywordRecognitionModel *)keywordModel
        error:(NSError * _Nullable * _Nullable)outError;

/**
 * Begins a speech-to-text interaction with this connector using a keyword. This interaction will use the provided
 * keyword model to listen for a keyword indefinitely, during which audio is not sent to the speech service and all
 * processing is performed locally. When a keyword is recognized, the DialogServiceConnector will automatically connect
 * to the speech service and begin sending audio data from just before the keyword as if listenOnceAsync() were invoked.
 * When received, speech-to-text results may be processed by the provided result handler or retrieved via a subscription
 * to the recognized event. The speech-to-text result produced by this operation is also provided to the configured
 * dialog implementation and that dialog system may produce any number of activity payloads in response to the speech
 * interaction. Speech interactions may be correlated with activities via dialog-specific data in the activity payload.
 *
 * @param keywordModel the keyword recognition model.
 */
- (void)startKeywordRecognition:(nonnull SPXKeywordRecognitionModel *)keywordModel
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Asynchronously begins a speech-to-text interaction with this connector and immediately returns execution to the
 * calling thread. This interaction will use the provided keyword model to listen for a keyword indefinitely, during
 * which audio is not sent to the speech service and all processing is performed locally. When a keyword is recognized,
 * the DialogServiceConnector will automatically connect to the speech service and begin sending audio data from just
 * before the keyword as if listenOnceAsync() were invoked. When received, speech-to-text results may be processed by
 * the provided result handler or retrieved via a subscription to the recognized event. The speech-to-text result
 * produced by this operation is also provided to the configured dialog implementation and that dialog system may
 * produce any number of activity payloads in response to the speech interaction. Speech interactions may be correlated
 * with activities via dialog-specific data in the activity payload.
 *
 * @param keywordModel the keyword recognition model.
 * @param completionHandler the handler function called when keyword recognition has started.
 * @param outError error information.
 * @return a value indicating whether the request to start keyword recognition was received successfully. If NO,
 * additional information may available in outError.
 */
- (BOOL)startKeywordRecognitionAsync
                         :(nonnull SPXKeywordRecognitionModel *)keywordModel
        completionHandler:(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
                    error:(NSError * _Nullable * _Nullable)outError;

/**
 * Asynchronously begins a speech-to-text interaction with this connector and immediately returns execution to the
 * calling thread. This interaction will use the provided keyword model to listen for a keyword indefinitely, during
 * which audio is not sent to the speech service and all processing is performed locally. When a keyword is recognized,
 * the DialogServiceConnector will automatically connect to the speech service and begin sending audio data from just
 * before the keyword as if listenOnceAsync() were invoked. When received, speech-to-text results may be processed by
 * the provided result handler or retrieved via a subscription to the recognized event. The speech-to-text result
 * produced by this operation is also provided to the configured dialog implementation and that dialog system may
 * produce any number of activity payloads in response to the speech interaction. Speech interactions may be correlated
 * with activities via dialog-specific data in the activity payload.
 *
 * @param keywordModel the keyword recognition model.
 * @param completionHandler the handler function called when keyword recognition has started.
 */
- (void)startKeywordRecognitionAsync
                         :(nonnull SPXKeywordRecognitionModel *)keywordModel
        completionHandler:(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Stops any active keyword recognition.
 *
 * @param outError error information.
 * @return a value indicating whether keyword recognition was stopped successfully. If NO, additional information may
 * be available in outError.
 */
- (BOOL)stopKeywordRecognition:(NSError * _Nullable * _Nullable)outError;

/**
 * Stops any active keyword recognition.
 */
- (void)stopKeywordRecognition
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Stops any active keyword recognition.
 *
 * @param completionHandler the handler function called when keyword recognition has stopped.
 * @param outError error information.
 * @return a value indicating whether the the request to stop was received successfully. If NO, additional error
 * information may be available in outError.
 */
- (BOOL)stopKeywordRecognitionAsync
             :(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
        error:(NSError * _Nullable * _Nullable)outError;

/**
 * Stops any active keyword recognition.
 *
 * @param completionHandler the handler function called when keyword recognition has stopped.
 */
- (void)stopKeywordRecognitionAsync:(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Establishes a connection with the service. Connection is automatically performed when needed, but this manual call
 * can be useful to make sure the connection is active before its first use to help reduce inital latency.
 *
 * Calling connect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling open(false). The Connection object also provides events to monitor
 * connections and disconnections.
 * 
 * @param outError error information.
 * @return a value indicating whether connection was successful. If NO, additional information may be available in
 * outError.
 */
- (BOOL)connect:(NSError * _Nullable * _Nullable)outError;

/**
 * Establishes a connection with the service. Connection is automatically performed when needed, but this manual call
 * can be useful to make sure the connection is active before its first use to help reduce inital latency.
 *
 * Calling connect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling open(false). The Connection object also provides events to monitor
 * connections and disconnections.
 */
- (void)connect
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Establishes a connection with the service. Connection is automatically performed when needed, but this manual call
 * can be useful to make sure the connection is active before its first use to help reduce inital latency.
 *
 * Calling connect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling open(false). The Connection object also provides events to monitor
 * connections and disconnections.
 *
 * @param completionHandler the handler function called when connection is complete.
 * @param outError error information.
 * @return a value indicating whether the request to connect started successfully. If NO, additional information may
 * be available in outError.
 */
- (BOOL)connectAsync
             :(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
        error:(NSError * _Nullable * _Nullable)outError;

/**
 * Establishes a connection with the service. Connection is automatically performed when needed, but this manual call
 * can be useful to make sure the connection is active before its first use to help reduce inital latency.
 *
 * Calling connect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling open(false). The Connection object also provides events to monitor
 * connections and disconnections.
 *
 * @param completionHandler the handler function called when connection is complete.
 */
- (void)connectAsync:(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Disconnects from the service. Subsequent calls that require a connection will still automatically reconnect after
 * manual disconnection.
 *
 * Calling disconnect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling close(). The Connection object also provides events to monitor
 * connections and disconnections.
 *
 * @param outError error information.
 * @return a value indicating whether disconnection was successful. If NO, additional information may be available in
 * outError.
 */
- (BOOL)disconnect:(NSError * _Nullable * _Nullable)outError;

/**
 * Disconnects from the service. Subsequent calls that require a connection will still automatically reconnect after
 * manual disconnection.
 *
 * Calling disconnect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling close(). The Connection object also provides events to monitor
 * connections and disconnections.
 */
- (void)disconnect
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Disconnects from the service. Subsequent calls that require a connection will still automatically reconnect after
 * manual disconnection.
 *
 * Calling disconnect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling close(). The Connection object also provides events to monitor
 * connections and disconnections.
 *
 * @param completionHandler the handler function called when disconnection is complete.
 * @param outError error information.
 * @return a value indicating whether the request to disconnect was received successfully. If NO, additional
 * information may be available in outError.
 */
- (BOOL)disconnectAsync
             :(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
        error:(NSError * _Nullable * _Nullable)outError;

/**
 * Disconnects from the service. Subsequent calls that require a connection will still automatically reconnect after
 * manual disconnection.
 *
 * Calling disconnect() on a DialogServiceConnector is functionally equivalent to initializing a Connection using
 * initFromDialogServiceConnector and then calling close(). The Connection object also provides events to monitor
 * connections and disconnections.
 *
 * @param completionHandler the handler function called when disconnection is complete.
 */
- (void)disconnectAsync:(nonnull SPXDialogServiceConnectorAsyncCompletionHandler)completionHandler
NS_SWIFT_UNAVAILABLE("Use the method with Swift-compatible error handling.");

/**
 * Registers a provided handler function to be called when a final speech-to-text-result is received by this
 * DialogServiceConnector instance.
 */
- (void)addRecognizedEventHandler:(nonnull SPXDialogServiceConnectorRecognitionEventHandler)eventHandler;

/**
 * Registers a provided handler function to be called when an intermediate speech-to-text-result is received by this
 * DialogServiceConnector instance. A single speech-to-text interaction may receive many intermediate 'recognizing'
 * results that correspond to a single, final 'recognized' result.
 */
- (void)addRecognizingEventHandler:(nonnull SPXDialogServiceConnectorRecognitionEventHandler)eventHandler;

/**
 * Registers a provided handler function to be called when an interaction is abnormally terminated. This may happen for
 * expected conditions, such as the end of provided file or audio stream input, or for unexpected error conditions.
 * More information about the reason for the canceled event may be retrieved from the event data.
 */
- (void)addCanceledEventHandler:(nonnull SPXDialogServiceConnectorCanceledEventHandler)eventHandler;

/**
 * Registers a provided handler function to be called when a turn status update is received from the dialog
 * implementation that this DialogServiceConnector is connected to.
 * A turn is a single execution session within the dialog implementation that may generate
 * any number of activities over its course. The information in this payload represents success or failure conditions
 * encountered by the dialog implementation over the course of this execution. This data facilitates the indication
 * of completion or error conditions within the dialog implementation even when no explicit activity data is produced
 * as part of a turn.
 */
- (void)addTurnStatusReceivedEventHandler
        :(nonnull SPXDialogServiceConnectorTurnStatusReceivedEventHandler)eventHandler;

/**
 * Registers a provided handler function to be called when an activity payload is received from the dialog
 * implementation that this DialogServiceConnector is connected to. Activities may be sent by a dialog implementation
 * at any time during a connection and there may be a many-to-one relationship between activities received and input,
 * such as speech utterances, sent by this object.
 */
- (void)addActivityReceivedEventHandler:(nonnull SPXDialogServiceConnectorActivityReceivedEventHandler)eventHandler;

@end
