# 前端最佳实践之可维护性

在早期网站中，JavaScript主要用于实现一些小型动效或表单验证。今天的Web应用则动辄成千上万行JavaScript代码，用于完成各种各样复杂的处理。这些变化要求开发者把可维护能力放到重要位置上。正如更传统意义上的软件工程师一样，JavaScript工程师受雇是要为公司创造价值的。现代前端工程师的使命，不仅仅是要保证产品如期上线，更重要的是要随着时间推移为公司不断积累知识资产。

编写可维护的代码十分重要，因为很多开发者都会花大量时间去维护别人写的代码。实际开发中，从第一行代码开始写起的情况是非常少见的。通常都是要在别人的代码之上来构建自己的工作。让自己的代码容易维护，可以保证其他开发者更好地完成自己的工作。

> **注意** 可维护代码的概念并不止适用于JavaScript，其中很多概念适用于任何编程语言。当然，有部分概念可能确实是特定于JavaScript的。

## 1. 什么是可维护的代码

可维护的代码有几个特点。通常，说代码可维护就意味着它具备如下特点。

- 容易理解。无需求助原始开发者，任何人一看代码就知道是干什么的，怎么实现的。
- 符合常识。代码中的一切都显得自然而然，无论操作有多么复杂。
- 容易适配。即使数据发生变化也不用完全重写。
- 容易扩展。代码架构经过认真设计，支持未来扩展核心功能。
- 容易调试。出问题时，代码可以给出明确的信息，通过它能直接定位问题。

能够写出可维护的JavaScript代码是一项重要的专业技能。这个技能是一个周末就拼凑一个网站的业余爱好者和对自己所做的一切都深思熟虑的专业开发者的重要区别。

## 2. 编码规范

编写可维护代码的第一步是认真考虑编码规范。编码规范在多数编程语言中都会涉及，简单上网一搜，就可以找到成千上万的相关文章。专业组织都会有为开发者建立的编码规范，旨在让人写出更容易维护的代码。优秀开源项目都有严格的编码规范，能够让社区的所有人容易理解代码的组织。

编码规范对JavaScript而言非常重要，因为这门语言实在太灵活了。与多数面向对象语言不同，JavaScript并不强迫开发者把任何东西都定义为对象。相反，JavaScript支持任何编程风格，包括传统的面向对象编程和声明式编程，以及函数式编程。简单看几个开源的JavaScript库，就会发现有很多方式可以创建对象、定义方法和管理环境。

接下来的几节会讨论制定编码规范的一些基础方面。这些主题都很重要，当然每个人的需求不同，实现方式也可以不同。

### 2.1 可读性

要想让代码容易维护，首先必须让人容易看懂。可读性必须考虑代码是一种文本文件。为此，代码缩进是保证可读性的重要基础。如果所有人都使用相同的缩进，整个项目的代码就会更容易看懂。缩进通常要使用空格数而不是Tab（制表符）来定义，因为后者在不同文本编辑器中的显示会有差异。一般来说，缩进都是以4个空格为单位，当然具体多少个可以自己定。

可读性的另一方面是代码注释。在多数编程语言中，广泛接受的做法是为每个方法都编写注释。由于JavaScript可以在代码中任何地方创建函数，所以这一点容易被忽视。正因为如此，可能给JavaScript中的每个函数都写注释才更重要。一般来说，以下这些地方都是应该写注释的。

- 函数和方法。每个函数和方法都应该有注释来描述其用途，以及完成任务所用的算法。同时，也写清使用这个函数或方法的前提（假设）、每个参数的含义，以及函数是否返回值（因为通过函数定义看不出来）。
- 大型代码块。多行代码但用于完成单一任务的，应该在前面给出注释，把要完成的任务写清楚。
- 复杂的算法。如果使用了不同寻常的手法解决了问题，要通过注释解释明白。这样不仅可以帮到别人，也可以让自己今后再看的时候更快想起来。
- 使用黑科技。由于浏览器之间的差异，JavaScript代码中通常都会包含一些黑科技。不要假设其他人一看就能明白某个黑科技是为了解决某个浏览器的什么问题。如果对某个浏览器不能使用正常方式达到目的，那要在注释里把黑科技的用途写出来。这样可以避免别人误以为黑科技没有用而把它“修复”掉，结果你已经修好的问题又会复现。

缩进和注释可以让代码更容易理解，将来也更容易维护。

### 2.2 变量和函数命名

变量和函数的适当命名对于可读性和可维护性也是至关重要的。由于很多JavaScript开发者都是“草莽”出身，所以很容易用foo、bar命名变量，用doSomething来命名函数。专业JavaScript开发者必须改掉这些积习，这样才能写出可维护的代码。以下是关于命名的通用规则。

- 变量名应该是名词，例如car或person。
- 函数名应该以动词开始，例如getName()。返回布尔值的函数通常以is开头，比如isEnabled()。
- 对变量和函数都使用符合逻辑的名字，不用担心长度。长名字的问题可以通过后处理和压缩解决。
- 变量、函数和方法应该以小写字母开头，使用驼峰大小写形式，如getName()和isPerson。类名应该首字母大写，比如Person、RequestFactory。常量值应该全部大写并以下划线相接，比如REQUEST_TIMEOUT。
- 名字要尽量用描述性和直观的词汇，但不要过于冗长。getName()一看就知道会返回名字，而PersonFactory一看就知道会产生某个Person对象或实例。

要完全避免没有用的变量名，比如不能表示所包含数据的类型。通过适当命名，代码读起来就会像故事，因此更容易理解。

### 2.3 变量类型透明化

因为JavaScript是松散类型的语言，所以很容易忘记变量包含的数据类型。适当命名可以在某种程度上解决这个问题，但还不够。有三种方式可以表明变量的数据类型。

第一种方式是通过初始化。定义变量时，应该立即将其初始化为一个将来要使用类型的值。例如，要保存布尔值的变量可以将其初始化为true或false，而要保存数值的变量可以将其初始化为一个数值。再看几个例子：

```js
// 通过初始化表明变量类型
let found = false; 	// Boolean
let count = -1; 	// number
let name = ""; 		// string
let person = null; 	// object
```

初始化为特定数据类型的值可以明确表示变量的类型。在ES6之前，初始化方式不适合函数声明中函数的参数。ES6之后，可以在函数声明中为参数指定默认值来表明参数类型。

第二种表示变量类型的方式是使用匈牙利表示法。匈牙利表示法指的是在变量名前面前缀一个或多个字符表示数据类型。这种表示法曾经在脚本语言中非常流行，很长时间以来也是JavaScript首选的格式。对于基本数据类型，用o表示对象（object）、s表示字符串（string），i表示整数（integer），f表示浮点数（float）、b表示布尔值（boolean）。下面看几个例子。

```js
// 使用匈牙利表示法标明数据类型
let bFound; 	// Boolean
let iCount; 	// integer
let sName; 		// string
let oPerson; 	// object
```

匈牙利表示法也可以应用给函数参数。匈牙利表示法的缺点是让代码可读性有所下降，不够直观，破坏了类似句子的自然阅读流畅性。为此，匈牙利表示法已经被很多开发者抛弃。

最后一种表明数据类型的方式是使用类型注释。类型注释放在变量名后面，初始化表达式的前面。基本思路是在变量旁边使用注释说明类型，比如：

```js
// 使用类型注释标明数据类型
let found /*:Boolean*/ = false;
let count /*:int*/ = 10;
let name /*:String*/ = "Nicholas";
let person /*:Object*/ = null;
```

类型注释在保持整体可读性的同时向代码中注入了类型信息。类型注释的缺点是不能再使用多行注释把大型代码块注释掉了。因为类型注释也是多行注释，因此会造成干扰，如下面的例子所示：

```js
// 这样多行注释不会生效
/*
let found /*:Boolean*/ = false;
let count /*:int*/ = 10;
let name /*:String*/ = "Nicholas";
let person /*:Object*/ = null;
*/
```

这里本来是想使用多行注释把所有变量声明都注释掉。但类型注释产生了干扰，因为第一个/\*（第2行）会与第一个\*/（第3行）匹配，结果会导致语法错误。如果想注释掉包含类型注释的代码，只能使用单行注释一行一行地注释掉每一行（有的编辑器可以自动完成）。

以上是三种标明变量数据类型的最常用方式。每种方式都有优点和缺点，可以根据自己的情况选用。关键要看哪一种最适合自己的项目，并保证一致性。

## 3. 松散耦合

只要应用的某个部分对另一个部分依赖得过于紧密，代码就会变成强耦合，因而难以维护。典型的问题是在一个对象中直接引用另一个对象。这样，修改其中一个，可能必须还得修改另一个。紧密耦合的软件难于维护，肯定需要频繁地重写。

考虑到相关的技术，Web应用在某些情况下可能变得过于强耦合。关键在于有这个意识，随时注意不要让代码产生强耦合。

### 3.1 解耦HTML/JavaScript

Web开发中最常见的耦合是HTML/JavaScript耦合。在网页中，HTML和JavaScript分别代表不同层面的解决方案。HTML是数据，JavaScript是行为。因为它们之间要交互操作，需要通过不同的方式将这两种技术联系起来。可惜的是，其中一些方式会导致HTML与JavaScript强耦合。

把JavaScript直接嵌入在HTML中，包括使用`<script>`元素包含嵌入代码或使用HTML属性添加事件处理程序，都会造成强耦合。比如下面的例子：

```html
<!-- 使用<script>造成HTML/JavaScript强耦合 -->
<script>
	document.write("Hello world!");
</script>

<!-- 使用事件处理程序属性造成HTML/JavaScript强耦合 -->
<input type="button" value="Click Me" onclick="doSomething()"/>
```

虽然技术上这样做没有问题，但实践中这样会导致HTML的数据与JavaScript的行为紧密耦合在一起。理想情况下，HTML和JavaScript应该完全分开，通过外部文件引入JavaScript，然后使用DOM添加行为。

HTML与JavaScript强耦合的情况下，每次分析JavaScript的报错都要首先确定错误来自HTML还是JavaScript。而且，这样也会引入代码可用性的新错误。在这个例子中，用户可能会在doSomething()函数可用之前点击按钮，从而导致JavaScript报错。由于每次修改按钮的行为都需要既改HTML又改JavaScript，而实际上只有后者才是有必要修改的。这样就会降低代码的可维护性。

在相反的情况下，HTML和JavaScript也会变得强耦合：把HTML包含在JavaScript中。这种情况通常发生在把一段HTML通过innerHTML插入到页面中，比如：

```js
// HTML紧耦合到了JavaScript
function insertMessage(msg) {
  let container = document.getElementById("container");
    container.innerHTML = `<div class="msg">
      <p> class="post">${msg}</p>
      <p><em>Latest message above.</em></p>
    </div>`;
}
```

一般来说，应该避免在JavaScript中创建大量HTML。同样，这主要是为了做到数据层和行为层各司其职，在出错时更容易定位问题所在。如果使用上面的代码示例，那么如果动态插入的HTML格式不对，就会造成页面布局出错。但在这种情况下定位错误就更困难了。因为这时候通常首先会去找页面中出错的HTML源代码，但又找不到，因为它是动态生成的。而且修改数据或页面，还需要修改JavaScript，这说明两层是紧密耦合的。

HTML渲染应该尽可能与JavaScript分开。在使用JavaScript插入数据时，应该尽可能不要插入标记。相应的标记可以包含并隐藏在页面中，在需要的时候JavaScript可以直接用它来显示，而不需要动态生成。另一个办法是通过Ajax请求获取要显示的HTML，这样也可以保证同一个渲染层（PHP、JSP、Ruby等）负责输出标记，而不是把标记嵌在JavaScript中。

解耦HTML和JavaScript可以节省排错时间，因为更容易定位错误来源。同样解耦也有助于保证可维护性，对行为的修改只涉及JavaScript，而对标记的修改则只涉及要渲染的文件。

### 3.2 解耦CSS/JavaScript

Web应用的另一层是CSS，主要负责页面的外观。JavaScript和CSS是紧密相关的，它们都是建构在HTML之上的，因此也经常一起使用。与HTML和JavaScript的情况类似，CSS也可能与JavaScript产生强耦合。最常见的例子就是使用JavaScript修改个别样式，比如：

```js
// CSS紧耦合到了JavaScript
element.style.color = "red";
element.style.backgroundColor = "blue";
```

因为CSS负责页面外观，任何样式的问题都应该通过CSS文件解决。可是，如果JavaScript直接修改个别样式（比如颜色），就会增加一个排错时要考虑甚至要修改的因素。结果是JavaScript某种程度上承担了页面显示的任务，与CSS搞成了紧密耦合。如果将来有一天要修改样式，那么CSS和JavaScript都需要修改。这对负责维护的开发者来说是一个恶梦。层与层的清晰解耦是必需的。

现代Web应用经常使用JavaScript改变样式，因此虽然不太可能完全解耦CSS和JavaScript，但可以让这种耦合变成更松散。这主要可以通过动态修改类名而不是样式来实现，比如：

```js
// CSS与JavaScript松散耦合
element.className = "edit";
```

通过修改元素的CSS类名，可以把大部分样式限制在CSS文件里。JavaScript只负责修改应用样式的类名，而不直接影响元素的样式。只要应用的类名没错，那么外观的问题就只跟CSS有关，而跟JavaScript无关。

同样，保证层与层之间的适当分离是至关重要的。显示出问题就只应该去CSS里解决，行为出问题就只应该找JavaScript的问题。这些层之间的松散耦合可以提升整个应用的可维护性。

### 3.3 解耦应用逻辑/事件处理程序

每个Web应用中都会有大量事件处理程序在监听各种事件。可是，其中很少有真正做到应用逻辑与事件处理程序分离的。来看下面的例子：

```js
function handleKeyPress(event) {
  if (event.keyCode == 13) {
    let target = event.target;
    let value = 5 * parseInt(target.value);
    if (value > 10) {
      document.getElementById("error-msg").style.display = "block";
    }
  }
}
```

这个事件处理程序除了处理事件，还包含了应用逻辑。这样做的问题是双重的。首先，除了事件没有办法触发应用逻辑，结果造成调试困难。如果没有发生预期的结果怎么办？是因为没有调用事件处理程序，还是因为应用逻辑有错误？其次，如果后续事件也会对应相同的应用逻辑，就会导致代码重复，否则就要把它提取到一个函数中。无论如何，都会导致原本不必要的多余工作。

更好的做法是将应用逻辑与事件处理程序分开，各自只负责处理各自的事情。事件处理程序应该专注于event对象的相关信息，然后把这些信息传给处理应用逻辑的某些方法。例如，前面的例子可以重写成这样：

```js
function validateValue(value) {
  value = 5 * parseInt(value);
  if (value > 10) {
    document.getElementById("error-msg").style.display = "block";
  }
}

function handleKeyPress(event) {
  if (event.keyCode == 13) {
    let target = event.target;
    validateValue(target.value);
  }
}
```

这样修改之后，应用逻辑跟事件处理程序就分开了。handleKeyPress()函数只负责检查用户是不是按下了回车键（event.keyCode等于13），如果是则取得事件目标，并把目标的值传给validateValue()函数，由该函数处理应用逻辑。注意，validateValue()函数中不包含任何依赖事件处理程序的代码。这个函数只负责接收一个值，然后可以对这个值做任何处理。

把应用逻辑从事件处理程序中分离出来有很多好处。首先，可以方便地修改触发某个流程的事件。如果原来是通过鼠标单击触发流程，而现在又想增加键盘操作来触发，那么修改起来也很简单。其次，可以在不用添加事件的情况下测试代码，这样创建单元测试甚至与应用自动化整合都会更简单。

以下是在解耦应用和业务逻辑时应该注意的几点。

- 不要把event对象传给其他方法，而是只传递event对象中必要的数据。
- 应用中每个可能的操作都应该无需事件处理程序就可以执行。
- 事件处理程序应该处理事件，而把后续处理交给应用逻辑。

做到上述几点能够给任何代码的可维护性带来巨大的提升，同时也能为将来的测试和开发提供很多可能性。

## 4. 编码惯例

编写可维护的JavaScript不仅仅涉及代码格式和规范，也涉及代码做什么。大企业开发Web应用通常需要很多人协同工作。这时候就需要保证每个人的浏览器环境都有恒定不变的规则。为此，开发者应该遵守某些编码惯例。

### 4.1 尊重对象所有权

JavaScript的动态天性意味着几乎可以在任何时候修改任何东西。过去有人说，JavaScript中没有什么是神圣不可侵犯的，因为不能把任何东西标记为最终结果或者恒定不变。但ECMAScript 5引入防篡改对象之后，情况不同了。当然，对象默认还是可以修改的。在其他语言中，在没有源代码的情况下对象和类都是不可修改的。JavaScript则允许在任何时候修改任何对象，因此就可能导致意外地覆盖默认行为。既然这门语言没有什么限制，那就需要开发者自己限制自己。

在企业开发中，可能最最重的编码惯例就是尊重对象所有权，这意味着不要修改不属于你的对象。简单地讲，如果你不负责创建和维护某个对象，包括它的构造函数或它的方法，就不应该对它进行任何修改。更具体一点说，就是：

- 不要给实例或原型添加属性
- 不要给实例或原型添加方法
- 不要重定义已有的方法

问题在于，开发者会假设浏览器环境以某种方式运行。修改了多个人使用的对象也就意味着会有错误发生。如果有人希望某个函数叫stopEvent()，用于取消某个事件的默认行为。然后，你把它给改了，除了取消事件的默认行为，又添加了其他事件处理程序。可想而知，问题肯定会接踵而至。别人还会认为这个函数只做最开始的那点事，由于对它后来添加的副作用并不知情，很可能会用错或者造成损失。

以上规则不仅适用于自定义类型和对象，同样适用于原生类型和对象，比如Object、String、document、window，等等。考虑到浏览器厂商也有可能会在不宣布的情况下以非预期方式修改这些对象，那么潜在的风险就更大了。

以前有一个流行的Prototype库就发生过类似事件。当时，这个库在document对象上实现了getElementsByClassName()方法，返回一个Array的实例，而这个实例上还增加了each()方法。jQuery的作者John Resig后来在自己的博客上分析了这个问题造成的影响。他在博客中（[https://johnresig.com/blog/getelementsbyclassname-pre-prototype-16/](https://johnresig.com/blog/getelementsbyclassname-pre-prototype-16/)）指出这个问题是由于浏览器也原生实现了相同的getElementsByClassName()方法造成的。但Prototype的同名方法返回的是Array而非NodeList，后者没有each()方法。使用这个库的开发者之前会写这样的代码：

```js
document.getElementsByClassName("selected").each(Element.hide);
```

尽管这样写在没有原生实现getElementsByClassName()方法的浏览器里没有问题，但在实现它的浏览器里就会出问题。因为两个同名方法返回的结果不一样。我们不能预见浏览器厂商将来会怎么修改原生对象，因此不管怎么修改它们都可能在将来某个时刻出现冲突时导致问题。

为此，最好的方法是永远不要修改不属于你的对象。只有你自己创建的才是你的对象，包括自定义类型和对象字面量。Array、document等这些都不是你的，因为在你的代码执行之前它们已经存在了。可以这样为对象添加新功能：

- 创建包含想要功能的新对象，通过它与别人的对象交互。
- 创建新自定义类型继承本来想要修改的类型，给自定义类型添加新功能。

很多JavaScript库目前都赞同这个开发理念，这样无论浏览器怎样改变都可以发展和适应。

### 4.2 不声明全局变量

与尊重对象所有权密切相关的是尽可能不声明全局变量和函数。同样，这也关系到创建一致和可维护的脚本运行环境。最多可以创建一个全局变量，作为其他对象和函数的命名空间。来看下面的例子：

```js
// 两个全局变量——不要！
var name = "Nicholas";
function sayName() {
  console.log(name);
}
```

以上代码声明了两个全局变量：name和sayName()。可以像下面这样把它们包含在一个对象中：

```js
// 一个全局变量——推荐
var MyApplication = {
  name: "Nicholas",
  sayName: function() {
    console.log(this.name);
  }
};
```

这个重写后的版本只声明了一个全局对象MyApplication。在这个对象内部，又包含name和sayName()。这样可以避免之前版本的几个问题。首先，变量name会覆盖window.name属性，而这可能会影响其他功能。其次，有助于分清功能都集中在哪里。调用MyApplication.sayName()从逻辑上就会暗示出现任何问题，都可以在MyApplication的代码中找原因。

这样一个全局对象可以扩展为命名空间的概念。命名空间涉及创建一个对象，然后通过这个对象来暴露能力。比如，Google Closure库就利用了这样的命名空间来组织其代码。下面是几个例子：

- goog.string：用于操作字符串的方法。
- goog.html.utils：与HTML相关的方法。
- goog.i18n：与国际化（i18n）相关的方法。

对象goog就相当于一个容器，其他对象都包含在这里面。只要使用对象以这种方式来组织功能，就可以称该对象为命名空间。整个Google Closure库都构建在这个概念之上，能够在同一个页面上与其他JavaScript库共存。

关于命名空间，最重要的是确定一个所有人都同意的全局对象名称。这个名称要足够独特，不可能与其他人的冲突。多数情况下，可以使用开发者所在的公司名，例如goog或Wrox。下面的例子演示了使用Wrox作为命名空间来组织功能：

```js
// 创建全局对象
var Wrox = {};

// 为本书（Professional JavaScript）创建命名空间
Wrox.ProJS = {};

// 添加本书用到的其他对象
Wrox.ProJS.EventUtil = { ... };
Wrox.ProJS.CookieUtil = { ... };
```

在这个例子中，Wrox是全局变量，然后在它的下面又创建了命名空间。如果本书所有代码都保存在Wrox.ProJS命名空间中，那么其他作者的代码就可以使用自己的对象来保存。只要每个人都遵循这个模式，就不必担心有人会覆盖这里的EventUtil或CookieUtil，因为即使重名它们也只会出现在不同的命名空间中。比如下面的例子：

```js
// 为另一本书（Professional Ajax）创建命名空间
Wrox.ProAjax = {};

// 添加其他对象
Wrox.ProAjax.EventUtil = { ... };
Wrox.ProAjax.CookieUtil = { ... };

// 可以照常使用ProJS下面的对象
Wrox.ProJS.EventUtil.addHandler( ... );
                                
// 以及ProAjax下面的对象
Wrox.ProAjax.EventUtil.addHandler( ... );
```

虽然命名空间需要多写一点代码，但从可维护性角度看，这个代价还是非常值得的。命名空间可以确保代码与页面上的其他代码互不干扰。

### 4.3 不要比较null

JavaScript不会自动做任何类型检查，因此就需要开发者担起这个责任。结果，很多JavaScript代码都不会做类型检查。最常见的类型检查是看一个值是不是null。然而，与null进行比较的代码太多了，其中很多都因为类型检查不够而频繁引发错误。比如下面的例子：

```js
function sortArray(values) {
  if (values != null) { // 不要这样比较！
    values.sort(comparator);
  }
}
```

这个函数的目的是使用给定的比较函数对数组进行排序。为保证函数正常执行，values参数必须是数组。但是，if语句在这里只简单地检查了这个值不是null。实际上，字符串、数值还有其他很多值都可以通过这里的检查，结果就会导致错误。

现实当中，单纯比较null通常是不够的。检查值的类型就要真的检查类型，而不是检查它不能是什么。例如，在前面的代码中，values参数应该是数组。为此，应该检查它到底是不是数组，而不是检查它不是null。可以像下面这样重写那个函数：

```js
function sortArray(values) {
  if (values instanceof Array) { // 推荐
    values.sort(comparator);
  }
}
```

这个函数的这个版本可以过滤所有无效的值，根本不需要使用null。

如果看到比较null的代码，可以使用下列某种技术替换它。

- 如果值应该是引用类型，使用instanceof操作符检查其构造函数。
- 如果值应该是原始类型，使用typeof检查其类型。
- 如果希望值是有特定方法名的对象，使用typeof操作符确保对象上存在给定名字的对象。

代码中比较null的地方越少，就越容易明确类型检查的目的，从而消除不必要的错误。

### 4.4 使用常量

依赖常量的目标是从应用逻辑中分离数据，以便修改数据时不会引发错误。显示在用户界面上的字符串就应该以这种方式提取出来，可以方便实现国际化。URL也应该这样提取出来，因为随着应用越来越复杂，URL也极有可能变化。基本上，像这种地方因为这种或那种原因将来需要修改时，可能就要找到某个函数，然后修改其中的代码。而每次像这样修改应用逻辑，都可能引入新错误。为此，可以把这些可能会修改的数据提取出来，放在单独定义的常量中，以实现数据与逻辑分离。

关键在于把数据从使用它们的逻辑中分离出来。可以使用以下标准检查哪些数据需要提取。

- 重复出现的值。任何使用超过一次的值都应该提取到常量中。这样可以消除一个值改了而另一个值没改造成的错误。这里也包括CSS的类名。
- 用户界面字符串。任何会显示给用户的字符串都应该提取出来，以方便实现国际化。
- URL：Web应用中资源的地址经常会发生变化，因此建议把所有URL集中放在一个地方管理。
- 任何可能变化的值。任何时候，只要在代码中使用字面值，就问问自己这个值将来有没有可能会变。如果答案是有可能，那么就应该把它提取到常量中。

使用常量是企业级JavaScript的重要技术，因为它可以让代码更容易维护，同时可以让代码免受数据变化的影响。

---

本文摘自即将出版的《JavaScript高级程序设计（第4版）》，经出版方授权转发。

新书出版信息可以查询：https://www.ituring.com.cn/book/2472。

![](https://p0.ssl.qhimg.com/t01230f53898a31ee45.jpg)









