# setTimeout小尴尬的启示

本文通过分析一个小案例帮大家从一个侧面理解JavaScript的异步执行机制，从而可以在实践中避免类似的尴尬。

### 小背景

我们都知道，`alert`这种内置弹框会阻塞后续代码执行：

![](https://p1.ssl.qhimg.com/t01d06ca49db1d48ace.png)

之所以如此，就是因为JavaScript代码在浏览器中是单线程执行的。换句话说，浏览器中只有一个主线程负责运行所有JavaScript代码（不考虑Web Worker）。

提到浏览器中的JavaScript，基本上只有三个来源：

- BOM API的代码，让我们可以操作并利用浏览器提供的能力
- DOM API的代码，让我们可以操作网页内容
- 我们自己写的ECMAScript代码

这没什么。我们也知道，`setTimeout`用于“定时”执行代码，比如这样可以定时在3秒钟之后执行一段代码（函数）：

```javascript
setTimeout(delayCode, 3000)
```

当然，我们也都知道，`setTimeout`的“定时”并不精确，它只能保证`delayCode`函数在3秒**以后**执行，至于在3秒以后**还要等多长时间**才能执行，就跟它没关系了。

那跟什么有关系？我们知道，任务队列/事件循环是JavaScript执行异步任务的机制。`setTimeout`作为BOM API，它只负责设定一个计时器，到点把要执行的函数添加到任务队列。这样它就完成任务了。而把函数添加到任务队列并不能保证立即执行。什么时候能执行取决于事件循环什么时候把这个函数调度到主线程。事件循环调度异步函数的前提是主线程空闲。如果主线程被阻塞了，即使把函数添加到事件对列，事件循环也不会立即调度它到主线程。这就是`setTimeout`不能精确定时执行某个函数的原因。

显然，如果你的代码中存在依赖`setTimeout`精确定时的逻辑，就有可能遭遇尴尬。为此我们自己写代码时，除非绝对有把握，一定尽量不要依赖`setTimout`的精确定时。可是，问题在于我们能保证自己写的代码不依赖它，却很难保证我们代码依赖的第三方代码不依赖它。

### 小案例

下面我们就来介绍一个遭遇这种尴尬的真实案例。这个案例涉及的功能很简单，就是jQuery的`$.ajax()`函数在加载数据失败时重发请求。由于其超时逻辑依赖`setTimeout`的精确定时，结果导致超时设置失效。

> 说明一下，这个案例是在调试代码时把请求超时时间设置得很小（比如15毫秒）造成的。正常情况下应该设置为一个合理的值（如5000毫秒）。

相关代码也很简单，主要涉及3个函数：

```javascript
function asyncRequest() {
  $.ajax({
    url: 'https://api.example.rs',
    timeout: 15
  }).then(success, fail)
}

function success(data) {
  // 正常处理数据
}

function fail(xhr, errtext, errthrown) {
  // 重发请求
  asyncRequest()
  // 弹框提示；阻塞主进程
  alert('请求超时')
}
// 首次调用
asyncRequest()
```

- `asyncRequest`：包含Ajax请求的函数，会在`fail`中再次调用
- `success`：Ajax请求成功的回调
- `fail`：Ajax请求失败的回调

正常逻辑是这样的：调用`asyncRequest`发送请求，成功则浏览器将`success`添加到任务队列，失败则浏览器将`fail`添加到任务队列。之后由事件循环将它们调度到主线程执行。`success`就是正常处理数据，而`fail`会先调用`asyncRequest`重发请求，再调用`alert`弹框提示。

测试环境下Ajax请求100毫秒左右可以返回。而**为了测试超时失败后的逻辑**，我们故意将超时时间设置为15毫秒，确保一定会超时。实际测试时，首次请求超时，走`fail`分支，重发请求、弹框，都没问题。但是，在鼠标点击关闭弹框后，却发现重发的请求正常返回了，并没有因超时被取消掉。反复测试都是如此。

这就尴尬了，到底为什么呢？研究发现，jQuery干掉超时请求的代码是这样的（<https://j11y.io/jquery/#v=git&fn=jQuery.ajax>）：

```javascript
// Timeout
if (s.async && s.timeout > 0) {
  timeoutTimer = window.setTimeout(function () {
      jqXHR.abort("timeout");
  },
  s.timeout);
}
```

也就是说，在我们设置了`timeout`选项的情况下，jQuery会通过`setTimeout`设置一个15毫秒后定时执行的函数，用来中断（`abort`）请求，我们称其为**中断函数**。

正常情况下，执行完上面的代码，浏览器会在15毫秒后把中断函数添加到任务队列上。此时如果主线程是空闲的，则事件循环会立即把这个函数调度到主线程去执行，请求被取消，浏览器把`fail`添加到任务队列，事件循环把它调度到主线程执行。这正是首次调用`asyncRequet`的情况。

第二次调用`asyncRequest`时有什么不同呢？不同之处在于这次调用完`asyncRequest`之后，还弹框阻塞了主线程。调用`asyncRequest`的结果跟之前一样，浏览器仍然会在15毫秒后把中断函数添加到任务队列。但是，这里要注意，由于此时主线程因弹框阻塞一直处于被占用状态，事件循环只能等待。直到我们手拿鼠标花一两秒时间把弹框关闭，主线程空闲出来，中断函数才会被调度到主线程上执行。而在此之前，Ajax请求早已成功返回，同时浏览器把`success`添加到任务队列。

理论上，Ajax请求返回后`jqXHR`（`XMLHttpRequest`）对象的状态不应再有任何改变（改变也没意义）。因此，中断函数的执行并不会改变“请求已经成功返回”这个事实。更为尴尬的是——中断函数执行后，紧接着，事件循环又把`success`函数调度到主线程。而`fail`函数根本就没有进入任务队列，更谈不上执行了。

### 小收获

通过上面的案例分析，我们看到本该“超时”失败的请求，因为中断函数被耽误在任务队列上迟迟得不到执行，最终反而成功返回了数据。当然，问题的根源在于`alert`弹框阻塞了主线程，以及JavaScript的异步机制（事件循环）。

至于jQuery依赖`setTimeout`取消超时请求的逻辑，只要不是遇到像本文案例这样长时间阻塞主进程的情况就不会有问题。在本案例中，如果不是为了测试而把超时时间设置得那么短，而是设置为比如5000毫秒，这个尴尬的局面也不会出现。假如实际的服务器响应时间真超过了5秒，只要我们在Ajax请求返回前关掉弹框，中断函数还是会先一步执行，从而取消未完成的请求。当然，实践中使用系统弹框阻塞主进程本来也不是推荐的做法。

不管怎么样，机缘巧合，我们还是借这个小尴尬（重温或者）深入理解了`setTimeout`乃至JavaScript（应该说浏览器提供的JavaScript运行时）的异步代码执行机制。那么在今后的编程实践中，我们就可以有意识地在逻辑中避免依赖`setTimeout`精确定时，因为它的定时真的不可靠啊！

----

**感谢[hax](https://github.com/hax)帮忙审校本文**
