# Generated by Django 2.2.18 on 2021-02-15 21:59

import django.contrib.postgres.fields
import django.contrib.postgres.fields.jsonb
import django.core.validators
from django.db import migrations, models
import django.db.models.deletion
from django.utils.text import slugify
import zentral.contrib.osquery.models


def inventory_platform_to_osquery_platform(inventory_platform):
    if inventory_platform == "LINUX":
        return "linux"
    elif inventory_platform == "MACOS":
        return "darwin"
    elif inventory_platform == "WINDOWS":
        return "windows"
    else:
        return None


def probe_platform_to_osquery_platform(probe_platform):
    if probe_platform in ("arch", "amazon", "centos", "debian",
                          "fedora", "funtoo", "gentoo", "manjaro", "oracle", "redhat",
                          "rhel", "scientific", "ubuntu"):
        return "linux"
    else:
        return probe_platform


def convert_osquery_probes(apps, schema_editor):
    try:
        from zentral.contrib.osquery.probes import (OsqueryProbe,
                                                    OsqueryComplianceProbe,
                                                    OsqueryFIMProbe)
    except ImportError:
        return
    ProbeSource = apps.get_model("probes", "ProbeSource")

    # delete file carve probes
    ProbeSource.objects.filter(model="OsqueryFileCarveProbe").delete()

    Query = apps.get_model("osquery", "Query")

    # distributed query probes → queries
    for ps in ProbeSource.objects.filter(model="OsqueryDistributedQueryProbe"):
        if ps.status == "ACTIVE":
            sql = ps.body.get("distributed_query")
            if sql:
                if not Query.objects.filter(sql=sql).exists():
                    Query.objects.create(name=ps.name,
                                         sql=sql,
                                         description=ps.description)
        ps.delete()

    Pack = apps.get_model("osquery", "Pack")
    PackQuery = apps.get_model("osquery", "PackQuery")
    FileCategory = apps.get_model("osquery", "FileCategory")
    Tag = apps.get_model("inventory", "Tag")

    # other osquery probes → packs
    # pack key (discovery, tags, platform)
    packs = {}
    for ps in ProbeSource.objects.filter(model__in=["OsqueryProbe",
                                                    "OsqueryComplianceProbe",
                                                    "OsqueryFIMProbe"]):
        # load probe manually because we have deregistrered them
        if ps.model == "OsqueryProbe":
            p_model = OsqueryProbe
        elif ps.model == "OsqueryComplianceProbe":
            p_model = OsqueryComplianceProbe
        elif ps.model == "OsqueryFIMProbe":
            p_model = OsqueryFIMProbe
        try:
            p = p_model(ps)
        except Exception:
            print(f"Could not load probe {ps.name}")
            continue

        # calculate pack key
        pack_platforms = set([])
        pack_tags = set([])
        for inventory_filter in p.inventory_filters:
            pack_platforms.update(
                op for op in (inventory_platform_to_osquery_platform(ip) for ip in inventory_filter.platforms) if op
            )
            if inventory_filter.tag_ids:
                pack_tags.update(Tag.objects.select_related("taxonomy", "meta_business_unit")
                                            .filter(pk__in=inventory_filter.tag_ids))
        pack_key = (p.pack_key, frozenset(pack_platforms), frozenset(pack_tags))

        # get or create pack
        pack = packs.get(pack_key)
        if pack is None:
            pack_num = len(packs) + 1
            _, platforms, tags = pack_key
            description = f"Migrated legacy osquery probes #{pack_num}"
            if tags:
                tags_for_display_list = []
                for t in tags:
                    if t.taxonomy:
                        tags_for_display_list.append(f"{t.taxonomy}: {t.name}")
                    elif t.meta_business_unit:
                        tags_for_display_list.append(f"{t.meta_business_unit}/{t.name}")
                    else:
                        tags_for_display_list.append(t.name)
                tags_for_display = ", ".join(tags_for_display_list)
                description = f"{description}\nFormer tags: {tags_for_display}"
            name = f"Migrated legacy probes #{pack_num}"
            pack = Pack.objects.create(
                name=name,
                slug=slugify(name),
                description=description,
                discovery_queries=p.pack_discovery_queries,
                platforms=list(platforms)
            )
            packs[pack_key] = pack

        # add FileCategory
        file_category = None
        p_file_paths = getattr(p, "file_paths", [])
        if p_file_paths:
            description = f"File category for migrated legacy Osquery FIM probe '{p.name}'"
            if p.description:
                description = f"{description}\n\n{p.description}"
            file_category = FileCategory.objects.create(
                name=p.name,
                slug=slugify(p.name),
                description=description,
                file_paths=[p_file_path.file_path for p_file_path in p_file_paths],
                access_monitoring=any(p_file_path.file_access for p_file_path in p_file_paths)
            )

        # add queries
        scheduled_queries = list(p.iter_scheduled_queries())
        scheduled_query_count = len(scheduled_queries)
        for query_idx, osquery_query in enumerate(scheduled_queries):
            if PackQuery.objects.filter(slug=osquery_query.name).exists():
                # should not happen
                continue
            # Query
            name = p.name
            if scheduled_query_count > 1:
                name = "{} {}/{}".format(name, query_idx + 1, scheduled_query_count)
            sql = osquery_query.query
            if file_category:
                # substitue the older category hash with the new file_category slug
                for file_path in p_file_paths:
                    sql = sql.replace(file_path.category, file_category.slug)
            query = Query.objects.create(
                name=name,
                sql=sql,
                description=osquery_query.description or "",
                value=osquery_query.value or ""
            )
            # PackQuery
            PackQuery.objects.create(
                pack=pack,
                query=query,
                slug=slugify(name),
                interval=osquery_query.interval,
                log_removed_actions=osquery_query.removed if osquery_query.removed is not None else True,
                snapshot_mode=osquery_query.snapshot if osquery_query.snapshot is not None else False,
                platforms=[probe_platform_to_osquery_platform(pp) for pp in (osquery_query.platform or [])],
                minimum_osquery_version=osquery_query.version or None,
                shard=osquery_query.shard or None,
            )

        # delete probe source
        ps.delete()


def migrate_legacy_configuration_attributes(apps, schema_editor):
    Configuration = apps.get_model("osquery", "Configuration")
    for configuration in Configuration.objects.all():
        configuration.inventory_interval = 1001
        configuration.inventory_apps = True
        configuration.options = {
            "config_refresh": configuration.config_refresh,
            "distributed_interval": configuration.distributed_interval,
            "disable_carver": configuration.disable_carver,
            "buffered_log_max": configuration.buffered_log_max
        }
        configuration.save()


class Migration(migrations.Migration):

    dependencies = [
        ('inventory', '0052_auto_20210122_1239'),
        ('osquery', '0006_auto_20190925_1934'),
    ]

    operations = [
        migrations.CreateModel(
            name='AutomaticTableConstruction',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('name', models.CharField(max_length=256, unique=True)),
                ('description', models.TextField(blank=True)),
                ('table_name', models.CharField(max_length=64, unique=True, validators=[django.core.validators.RegexValidator('[a-z_]+')])),
                ('query', models.TextField()),
                ('path', models.CharField(max_length=256)),
                ('columns', django.contrib.postgres.fields.ArrayField(base_field=models.CharField(max_length=64, validators=[django.core.validators.RegexValidator('[a-z_]')]), size=None)),
                ('platforms', django.contrib.postgres.fields.ArrayField(base_field=models.CharField(choices=[('darwin', 'darwin'), ('freebsd', 'freebsd'), ('linux', 'linux'), ('posix', 'posix'), ('windows', 'windows')], max_length=32), blank=True, default=list, help_text="Restrict this automatic table construction to some platforms, default is 'all' platforms", size=None)),
                ('created_at', models.DateTimeField(auto_now_add=True)),
                ('updated_at', models.DateTimeField(auto_now=True)),
            ],
        ),
        migrations.CreateModel(
            name='ConfigurationPack',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
            ],
        ),
        migrations.CreateModel(
            name='DistributedQuery',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('query_version', models.IntegerField(editable=False)),
                ('sql', models.TextField(editable=False)),
                ('valid_from', models.DateTimeField()),
                ('valid_until', models.DateTimeField(blank=True, null=True)),
                ('serial_numbers', django.contrib.postgres.fields.ArrayField(base_field=models.TextField(), blank=True, default=list, size=None)),
                ('shard', models.IntegerField(default=100, help_text='Restrict this query to a percentage (1-100) of target hosts', validators=[django.core.validators.MinValueValidator(1), django.core.validators.MaxValueValidator(100)])),
                ('created_at', models.DateTimeField(auto_now_add=True)),
                ('updated_at', models.DateTimeField(auto_now=True)),
            ],
        ),
        migrations.CreateModel(
            name='DistributedQueryMachine',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('serial_number', models.TextField(db_index=True)),
                ('status', models.IntegerField(null=True)),
                ('error_message', models.TextField(null=True)),
                ('created_at', models.DateTimeField(auto_now_add=True)),
                ('updated_at', models.DateTimeField(auto_now=True)),
                ('distributed_query', models.ForeignKey(on_delete=django.db.models.deletion.CASCADE, to='osquery.DistributedQuery')),
            ],
        ),
        migrations.CreateModel(
            name='DistributedQueryResult',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('serial_number', models.TextField()),
                ('row', django.contrib.postgres.fields.jsonb.JSONField()),
                ('distributed_query', models.ForeignKey(on_delete=django.db.models.deletion.CASCADE, to='osquery.DistributedQuery')),
            ],
        ),
        migrations.CreateModel(
            name='FileCarvingBlock',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('block_id', models.IntegerField()),
                ('file', models.FileField(upload_to=zentral.contrib.osquery.models.file_carving_block_path)),
                ('created_at', models.DateTimeField(auto_now_add=True)),
            ],
        ),
        migrations.CreateModel(
            name='FileCategory',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('name', models.CharField(max_length=256, unique=True)),
                ('slug', models.CharField(editable=False, max_length=256, unique=True)),
                ('description', models.TextField(blank=True)),
                ('file_paths', django.contrib.postgres.fields.ArrayField(base_field=models.CharField(max_length=256), blank=True, default=list, size=None)),
                ('exclude_paths', django.contrib.postgres.fields.ArrayField(base_field=models.CharField(max_length=256), blank=True, default=list, size=None)),
                ('file_paths_queries', django.contrib.postgres.fields.ArrayField(base_field=models.TextField(), blank=True, default=list, size=None)),
                ('access_monitoring', models.BooleanField(default=False)),
                ('created_at', models.DateTimeField(auto_now_add=True)),
                ('updated_at', models.DateTimeField(auto_now=True)),
            ],
        ),
        migrations.CreateModel(
            name='Pack',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('name', models.CharField(max_length=256, unique=True)),
                ('slug', models.CharField(editable=False, max_length=256, unique=True)),
                ('description', models.TextField(blank=True)),
                ('discovery_queries', django.contrib.postgres.fields.ArrayField(base_field=models.TextField(), blank=True, default=list, help_text='This pack will only execute if all discovery queries return results.', size=None)),
                ('platforms', django.contrib.postgres.fields.ArrayField(base_field=models.CharField(choices=[('darwin', 'darwin'), ('freebsd', 'freebsd'), ('linux', 'linux'), ('posix', 'posix'), ('windows', 'windows')], max_length=32), blank=True, default=list, help_text="Restrict this pack to some platforms, default is 'all' platforms", size=None)),
                ('minimum_osquery_version', models.CharField(blank=True, help_text='This pack will only execute on osquery versions greater than or equal-to this version string', max_length=14, null=True, validators=[django.core.validators.RegexValidator('[0-9]{1,4}\\.[0-9]{1,4}\\.[0-9]{1,4}')])),
                ('shard', models.IntegerField(blank=True, help_text='Restrict every pack queries to a percentage (1-100) of target hosts', null=True, validators=[django.core.validators.MinValueValidator(1), django.core.validators.MaxValueValidator(100)])),
                ('created_at', models.DateTimeField(auto_now_add=True)),
                ('updated_at', models.DateTimeField(auto_now=True)),
            ],
        ),
        migrations.CreateModel(
            name='PackQuery',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('slug', models.CharField(editable=False, max_length=256)),
                ('interval', models.IntegerField(help_text='interval in seconds to run the query (subject to splay/smoothing)', validators=[django.core.validators.MinValueValidator(10), django.core.validators.MaxValueValidator(86400)])),
                ('log_removed_actions', models.BooleanField(default=True, help_text="If 'removed' action should be logged")),
                ('snapshot_mode', models.BooleanField(default=False, help_text="Run this query in 'snapshot' mode")),
                ('platforms', django.contrib.postgres.fields.ArrayField(base_field=models.CharField(choices=[('darwin', 'darwin'), ('freebsd', 'freebsd'), ('linux', 'linux'), ('posix', 'posix'), ('windows', 'windows')], max_length=32), blank=True, default=list, help_text="restrict this query to some platforms, default is 'all' platforms", size=None)),
                ('minimum_osquery_version', models.CharField(blank=True, help_text='only run this query on osquery versions greater than or equal-to this version string', max_length=14, null=True, validators=[django.core.validators.RegexValidator('[0-9]{1,4}\\.[0-9]{1,4}\\.[0-9]{1,4}')])),
                ('shard', models.IntegerField(blank=True, help_text='restrict this query to a percentage (1-100) of target hosts', null=True, validators=[django.core.validators.MinValueValidator(1), django.core.validators.MaxValueValidator(100)])),
                ('can_be_denylisted', models.BooleanField(default=True, help_text='If this query can be denylisted when stopped for excessive resource consumption.')),
                ('created_at', models.DateTimeField(auto_now_add=True)),
                ('updated_at', models.DateTimeField(auto_now=True)),
                ('pack', models.ForeignKey(editable=False, on_delete=django.db.models.deletion.CASCADE, to='osquery.Pack')),
            ],
        ),
        migrations.CreateModel(
            name='FileCarvingSession',
            fields=[
                ('id', models.UUIDField(primary_key=True, serialize=False)),
                ('distributed_query',  models.ForeignKey(null=True, on_delete=django.db.models.deletion.CASCADE, to='osquery.DistributedQuery')),
                ('pack_query', models.ForeignKey(null=True, on_delete=django.db.models.deletion.CASCADE, to='osquery.PackQuery')),
                ('serial_number', models.TextField(db_index=True)),
                ('carve_guid', models.TextField()),
                ('carve_size', models.BigIntegerField()),
                ('block_size', models.IntegerField()),
                ('block_count', models.IntegerField()),
                ('archive', models.FileField(null=True, upload_to=zentral.contrib.osquery.models.file_carving_session_archive_path)),
                ('created_at', models.DateTimeField(auto_now_add=True)),
            ],
        ),
        migrations.CreateModel(
            name='Query',
            fields=[
                ('id', models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')),
                ('name', models.CharField(max_length=256, unique=True)),
                ('sql', models.TextField()),
                ('description', models.TextField(blank=True)),
                ('value', models.TextField(blank=True)),
                ('version', models.PositiveIntegerField(default=1, editable=False)),
                ('created_at', models.DateTimeField(auto_now_add=True)),
                ('updated_at', models.DateTimeField(auto_now=True)),
            ],
        ),
        migrations.RemoveField(
            model_name='carvesession',
            name='probe_source',
        ),
        migrations.RemoveField(
            model_name='distributedqueryprobemachine',
            name='probe_source',
        ),
        migrations.AddField(
            model_name='configuration',
            name='description',
            field=models.TextField(blank=True),
        ),
        migrations.AddField(
            model_name='configuration',
            name='inventory',
            field=models.BooleanField(default=True, help_text='Schedule regular inventory queries'),
        ),
        migrations.AddField(
            model_name='configuration',
            name='inventory_apps',
            field=models.BooleanField(default=False, help_text='Include macOS apps or linux packages in the inventory'),
        ),
        migrations.AddField(
            model_name='configuration',
            name='inventory_interval',
            field=models.IntegerField(default=86400, help_text='Inventory refresh interval in seconds', validators=[django.core.validators.MinValueValidator(300), django.core.validators.MaxValueValidator(172800)]),
        ),
        migrations.AddField(
            model_name='configuration',
            name='options',
            field=django.contrib.postgres.fields.jsonb.JSONField(blank=True, default=dict, help_text='Osquery options'),
        ),
        migrations.AddField(
            model_name='enrolledmachine',
            name='osquery_version',
            field=models.CharField(blank=True, max_length=14, null=True),
        ),
        migrations.AddField(
            model_name='enrolledmachine',
            name='updated_at',
            field=models.DateTimeField(auto_now=True),
        ),
        migrations.AlterUniqueTogether(
            name='enrolledmachine',
            unique_together={('enrollment', 'serial_number')},
        ),
        migrations.DeleteModel(
            name='CarveBlock',
        ),
        migrations.DeleteModel(
            name='CarveSession',
        ),
        migrations.DeleteModel(
            name='DistributedQueryProbeMachine',
        ),
        migrations.AddField(
            model_name='filecarvingblock',
            name='file_carving_session',
            field=models.ForeignKey(on_delete=django.db.models.deletion.CASCADE, to='osquery.FileCarvingSession'),
        ),
        migrations.AddField(
            model_name='distributedquery',
            name='query',
            field=models.ForeignKey(editable=False, null=True, on_delete=django.db.models.deletion.SET_NULL, to='osquery.Query'),
        ),
        migrations.AddField(
            model_name='distributedquery',
            name='tags',
            field=models.ManyToManyField(blank=True, to='inventory.Tag'),
        ),
        migrations.AddField(
            model_name='configurationpack',
            name='configuration',
            field=models.ForeignKey(editable=False, on_delete=django.db.models.deletion.CASCADE, to='osquery.Configuration'),
        ),
        migrations.AddField(
            model_name='configurationpack',
            name='pack',
            field=models.ForeignKey(on_delete=django.db.models.deletion.CASCADE, to='osquery.Pack'),
        ),
        migrations.AddField(
            model_name='configurationpack',
            name='tags',
            field=models.ManyToManyField(blank=True, to='inventory.Tag'),
        ),
        migrations.AddField(
            model_name='configuration',
            name='automatic_table_constructions',
            field=models.ManyToManyField(blank=True, to='osquery.AutomaticTableConstruction'),
        ),
        migrations.AddField(
            model_name='configuration',
            name='file_categories',
            field=models.ManyToManyField(blank=True, to='osquery.FileCategory'),
        ),
        migrations.AddField(
            model_name='packquery',
            name='query',
            field=models.OneToOneField(on_delete=django.db.models.deletion.PROTECT, to='osquery.Query')
        ),
        migrations.AlterUniqueTogether(
            name='packquery',
            unique_together={('pack', 'slug')},
        ),
        migrations.AlterUniqueTogether(
            name='filecarvingblock',
            unique_together={('file_carving_session', 'block_id')},
        ),
        migrations.AddIndex(
            model_name='distributedqueryresult',
            index=models.Index(fields=['distributed_query', 'serial_number'], name='osquery_dis_distrib_fac7c0_idx'),
        ),
        migrations.AlterUniqueTogether(
            name='distributedquerymachine',
            unique_together={('distributed_query', 'serial_number')},
        ),
        migrations.AlterUniqueTogether(
            name='configurationpack',
            unique_together={('configuration', 'pack')},
        ),
        migrations.RunPython(convert_osquery_probes),
        migrations.RunPython(migrate_legacy_configuration_attributes),
        migrations.RemoveField(
            model_name='configuration',
            name='buffered_log_max',
        ),
        migrations.RemoveField(
            model_name='configuration',
            name='config_refresh',
        ),
        migrations.RemoveField(
            model_name='configuration',
            name='disable_carver',
        ),
        migrations.RemoveField(
            model_name='configuration',
            name='distributed_interval',
        ),
    ]
