/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Unit tests for our TalkTable_GFF class.
 */

#include "gtest/gtest.h"

#include "src/common/util.h"
#include "src/common/encoding.h"
#include "src/common/memreadstream.h"

#include "src/aurora/types.h"
#include "src/aurora/talktable.h"
#include "src/aurora/talktable_gff.h"

// --- TLK V0.2 ---

static const byte kTLKV02[] = {
	0x47,0x46,0x46,0x20,0x56,0x34,0x2E,0x30,0x50,0x43,0x20,0x20,0x54,0x4C,0x4B,0x20,
	0x56,0x30,0x2E,0x32,0x02,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x54,0x4C,0x4B,0x20,
	0x01,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x53,0x54,0x52,0x4E,
	0x02,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x39,0x4A,0x00,0x00,
	0x01,0x00,0x00,0xC0,0x00,0x00,0x00,0x00,0x3A,0x4A,0x00,0x00,0x04,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x3B,0x4A,0x00,0x00,0x0E,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x50,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,0x30,0x00,0x00,0x00,
	0x06,0x00,0x00,0x00,0x46,0x00,0x6F,0x00,0x6F,0x00,0x62,0x00,0x61,0x00,0x72,0x00,
	0x06,0x00,0x00,0x00,0x42,0x00,0x61,0x00,0x72,0x00,0x66,0x00,0x6F,0x00,0x6F,0x00,
	0x00,0x00,0x00,0x00
};

GTEST_TEST(TalkTable_TLK02, hasEntry) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV02), Common::kEncodingUTF16LE);

	EXPECT_TRUE (tlk.hasEntry(0));
	EXPECT_TRUE (tlk.hasEntry(1));
	EXPECT_TRUE (tlk.hasEntry(2));
	EXPECT_FALSE(tlk.hasEntry(3));
	EXPECT_TRUE (tlk.hasEntry(4));

	EXPECT_FALSE(tlk.hasEntry(5));
}

GTEST_TEST(TalkTable_TLK02, getString) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV02), Common::kEncodingUTF16LE);

	EXPECT_STREQ(tlk.getString(0).c_str(), "Foobar");
	EXPECT_STREQ(tlk.getString(1).c_str(), "");
	EXPECT_STREQ(tlk.getString(2).c_str(), "");
	EXPECT_STREQ(tlk.getString(3).c_str(), "");
	EXPECT_STREQ(tlk.getString(4).c_str(), "Barfoo");

	EXPECT_STREQ(tlk.getString(5).c_str(), "");
}

GTEST_TEST(TalkTable_TLK02, getSoundResRef) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV02), Common::kEncodingUTF16LE);

	EXPECT_STREQ(tlk.getSoundResRef(0).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(1).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(2).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(3).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(4).c_str(), "");

	EXPECT_STREQ(tlk.getSoundResRef(5).c_str(), "");
}

GTEST_TEST(TalkTable_TLK02, getSoundID) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV02), Common::kEncodingUTF16LE);

	EXPECT_EQ(tlk.getSoundID(0), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(1), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(2), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(3), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(4), Aurora::kFieldIDInvalid);

	EXPECT_EQ(tlk.getSoundID(5), Aurora::kFieldIDInvalid);
}

GTEST_TEST(TalkTable_TLK02, fromGeneric) {
	Common::MemoryReadStream *stream = new Common::MemoryReadStream(kTLKV02);

	Aurora::TalkTable *tlk = Aurora::TalkTable::load(stream, Common::kEncodingUTF16LE);
	EXPECT_NE(dynamic_cast<Aurora::TalkTable_GFF *>(tlk), static_cast<Aurora::TalkTable_GFF *>(0));

	delete tlk;
}

// --- TLK V0.5 ---

static const byte kTLKV05[] = {
	0x47,0x46,0x46,0x20,0x56,0x34,0x2E,0x30,0x50,0x43,0x20,0x20,0x54,0x4C,0x4B,0x20,
	0x56,0x30,0x2E,0x35,0x02,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x54,0x4C,0x4B,0x20,
	0x03,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x53,0x54,0x52,0x4E,
	0x02,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x3E,0x4A,0x00,0x00,
	0x01,0x00,0x00,0xC0,0x00,0x00,0x00,0x00,0x3F,0x4A,0x00,0x00,0x05,0x00,0x00,0x80,
	0x04,0x00,0x00,0x00,0x40,0x4A,0x00,0x00,0x04,0x00,0x00,0x80,0x08,0x00,0x00,0x00,
	0x3C,0x4A,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3D,0x4A,0x00,0x00,
	0x04,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x38,0x00,0x00,0x00,
	0x74,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x28,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x28,0x00,0x00,0x00,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x0E,0x00,0x00,0x00,0xB9,0xFF,0xFF,0xFF,0x9D,0xFF,0xFF,0xFF,0xBD,0xFF,0xFF,0xFF,
	0x99,0xFF,0xFF,0xFF,0x9E,0xFF,0xFF,0xFF,0x8D,0xFF,0xFF,0xFF,0x90,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
	0x03,0x9B,0x7A,0xE9,0x83,0x02,0x00,0x00
};

GTEST_TEST(TalkTable_TLK05, hasEntry) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV05), Common::kEncodingUTF16LE);

	EXPECT_TRUE (tlk.hasEntry(0));
	EXPECT_TRUE (tlk.hasEntry(1));
	EXPECT_TRUE (tlk.hasEntry(2));
	EXPECT_FALSE(tlk.hasEntry(3));
	EXPECT_TRUE (tlk.hasEntry(4));

	EXPECT_FALSE(tlk.hasEntry(5));
}

GTEST_TEST(TalkTable_TLK05, getString) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV05), Common::kEncodingUTF16LE);

	EXPECT_STREQ(tlk.getString(0).c_str(), "Foobar");
	EXPECT_STREQ(tlk.getString(1).c_str(), "");
	EXPECT_STREQ(tlk.getString(2).c_str(), "");
	EXPECT_STREQ(tlk.getString(3).c_str(), "");
	EXPECT_STREQ(tlk.getString(4).c_str(), "Barfoo");

	EXPECT_STREQ(tlk.getString(5).c_str(), "");
}

GTEST_TEST(TalkTable_TLK05, getSoundResRef) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV05), Common::kEncodingUTF16LE);

	EXPECT_STREQ(tlk.getSoundResRef(0).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(1).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(2).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(3).c_str(), "");
	EXPECT_STREQ(tlk.getSoundResRef(4).c_str(), "");

	EXPECT_STREQ(tlk.getSoundResRef(5).c_str(), "");
}

GTEST_TEST(TalkTable_TLK05, getSoundID) {
	Aurora::TalkTable_GFF tlk(new Common::MemoryReadStream(kTLKV05), Common::kEncodingUTF16LE);

	EXPECT_EQ(tlk.getSoundID(0), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(1), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(2), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(3), Aurora::kFieldIDInvalid);
	EXPECT_EQ(tlk.getSoundID(4), Aurora::kFieldIDInvalid);

	EXPECT_EQ(tlk.getSoundID(5), Aurora::kFieldIDInvalid);
}

GTEST_TEST(TalkTable_TLK05, fromGeneric) {
	Common::MemoryReadStream *stream = new Common::MemoryReadStream(kTLKV05);

	Aurora::TalkTable *tlk = Aurora::TalkTable::load(stream, Common::kEncodingUTF16LE);
	EXPECT_NE(dynamic_cast<Aurora::TalkTable_GFF *>(tlk), static_cast<Aurora::TalkTable_GFF *>(0));

	delete tlk;
}
