/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Unit tests for our IFOFile class.
 */

#include "gtest/gtest.h"

#include "src/common/util.h"
#include "src/common/error.h"
#include "src/common/memreadstream.h"

#include "src/aurora/ifofile.h"
#include "src/aurora/language.h"

static const byte kIFOFile[] = {
	0x49,0x46,0x4F,0x20,0x56,0x33,0x2E,0x32,0x38,0x00,0x00,0x00,0x16,0x00,0x00,0x00,
	0x40,0x01,0x00,0x00,0x4E,0x00,0x00,0x00,0xE8,0x04,0x00,0x00,0x40,0x00,0x00,0x00,
	0xE8,0x08,0x00,0x00,0x0D,0x02,0x00,0x00,0xF5,0x0A,0x00,0x00,0xE4,0x00,0x00,0x00,
	0xD9,0x0B,0x00,0x00,0x7C,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,
	0x39,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x2E,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x06,0x00,0x00,0x00,0x2F,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x06,0x00,0x00,0x00,
	0x30,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x32,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x33,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x08,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x09,0x00,0x00,0x00,
	0x36,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x37,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x3A,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x3B,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3E,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x3F,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x40,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x42,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x43,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x46,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x47,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x0D,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
	0x02,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x03,0x00,0x00,0x00,
	0x02,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,
	0x05,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x06,0x00,0x00,0x00,
	0x3D,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x47,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,
	0x09,0x00,0x00,0x00,0x6A,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,
	0x71,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x00,0x00,0x28,0x42,
	0x08,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x66,0x66,0x28,0x42,0x08,0x00,0x00,0x00,
	0x0D,0x00,0x00,0x00,0xCD,0xCC,0x28,0x42,0x08,0x00,0x00,0x00,0x0E,0x00,0x00,0x00,
	0x00,0x00,0x50,0x42,0x08,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0x66,0x66,0x50,0x42,
	0x0F,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x11,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x12,0x00,0x00,0x00,
	0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x13,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x15,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x00,0x00,0x00,
	0x0D,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x17,0x00,0x00,0x00,0x26,0x09,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x19,0x00,0x00,0x00,0x77,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x1A,0x00,0x00,0x00,
	0x83,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x1B,0x00,0x00,0x00,0x8D,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x98,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x1D,0x00,0x00,0x00,0xA5,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x1E,0x00,0x00,0x00,
	0xB2,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x1F,0x00,0x00,0x00,0xBE,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0xCB,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x21,0x00,0x00,0x00,0xD8,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x22,0x00,0x00,0x00,
	0xE5,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x23,0x00,0x00,0x00,0xF0,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x24,0x00,0x00,0x00,0xFB,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x25,0x00,0x00,0x00,0x06,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x26,0x00,0x00,0x00,
	0x11,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x27,0x00,0x00,0x00,0x1E,0x01,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x28,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x29,0x00,0x00,0x00,0x35,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x2A,0x00,0x00,0x00,
	0x42,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,0x2B,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x0F,0x00,0x00,0x00,0x2C,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x2D,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x2E,0x00,0x00,0x00,
	0x48,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x2E,0x00,0x00,0x00,0x4E,0x01,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x2E,0x00,0x00,0x00,0x54,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x2F,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x30,0x00,0x00,0x00,
	0x5A,0x01,0x00,0x00,0x0A,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x62,0x01,0x00,0x00,
	0x0A,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x6A,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x31,0x00,0x00,0x00,0x2C,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x32,0x00,0x00,0x00,
	0x72,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x32,0x00,0x00,0x00,0x77,0x01,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x32,0x00,0x00,0x00,0x7C,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x33,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x34,0x00,0x00,0x00,
	0x81,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x88,0x01,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x8F,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x35,0x00,0x00,0x00,0x4C,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x36,0x00,0x00,0x00,
	0x96,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x36,0x00,0x00,0x00,0xA1,0x01,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x36,0x00,0x00,0x00,0xAC,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x37,0x00,0x00,0x00,0x5C,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x38,0x00,0x00,0x00,
	0xB7,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0xC1,0x01,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0xCB,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x39,0x00,0x00,0x00,0x6C,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x3A,0x00,0x00,0x00,
	0xD5,0x01,0x00,0x00,0x0B,0x00,0x00,0x00,0x3A,0x00,0x00,0x00,0xE1,0x01,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x3A,0x00,0x00,0x00,0xED,0x01,0x00,0x00,0x0D,0x00,0x00,0x00,
	0x3B,0x00,0x00,0x00,0xF9,0x01,0x00,0x00,0x05,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,
	0x17,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x3D,0x00,0x00,0x00,0x18,0x00,0x00,0x00,
	0x05,0x00,0x00,0x00,0x3E,0x00,0x00,0x00,0x19,0x00,0x00,0x00,0x05,0x00,0x00,0x00,
	0x3F,0x00,0x00,0x00,0x1A,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x49,0x44,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4D,0x69,0x6E,0x47,
	0x61,0x6D,0x65,0x56,0x65,0x72,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x43,0x72,0x65,0x61,
	0x74,0x6F,0x72,0x5F,0x49,0x44,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x56,0x65,0x72,0x73,
	0x69,0x6F,0x6E,0x00,0x00,0x00,0x00,0x00,0x45,0x78,0x70,0x61,0x6E,0x73,0x69,0x6F,
	0x6E,0x5F,0x50,0x61,0x63,0x6B,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4E,0x61,0x6D,0x65,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x54,0x61,0x67,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x44,0x65,0x73,0x63,
	0x72,0x69,0x70,0x74,0x69,0x6F,0x6E,0x00,0x4D,0x6F,0x64,0x5F,0x49,0x73,0x53,0x61,
	0x76,0x65,0x47,0x61,0x6D,0x65,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x43,0x75,0x73,0x74,
	0x6F,0x6D,0x54,0x6C,0x6B,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x45,0x6E,0x74,0x72,
	0x79,0x5F,0x41,0x72,0x65,0x61,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x45,0x6E,0x74,0x72,
	0x79,0x5F,0x58,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x45,0x6E,0x74,0x72,
	0x79,0x5F,0x59,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x45,0x6E,0x74,0x72,
	0x79,0x5F,0x5A,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x45,0x6E,0x74,0x72,
	0x79,0x5F,0x44,0x69,0x72,0x5F,0x58,0x00,0x4D,0x6F,0x64,0x5F,0x45,0x6E,0x74,0x72,
	0x79,0x5F,0x44,0x69,0x72,0x5F,0x59,0x00,0x4D,0x6F,0x64,0x5F,0x45,0x78,0x70,0x61,
	0x6E,0x5F,0x4C,0x69,0x73,0x74,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x44,0x61,0x77,0x6E,
	0x48,0x6F,0x75,0x72,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x44,0x75,0x73,0x6B,
	0x48,0x6F,0x75,0x72,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4D,0x69,0x6E,0x50,
	0x65,0x72,0x48,0x6F,0x75,0x72,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x53,0x74,0x61,0x72,
	0x74,0x4D,0x6F,0x6E,0x74,0x68,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x53,0x74,0x61,0x72,
	0x74,0x44,0x61,0x79,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x53,0x74,0x61,0x72,
	0x74,0x48,0x6F,0x75,0x72,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x53,0x74,0x61,0x72,
	0x74,0x59,0x65,0x61,0x72,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x58,0x50,0x53,0x63,
	0x61,0x6C,0x65,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x48,0x65,
	0x61,0x72,0x74,0x62,0x65,0x61,0x74,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x4D,0x6F,
	0x64,0x4C,0x6F,0x61,0x64,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x4D,0x6F,
	0x64,0x53,0x74,0x61,0x72,0x74,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x43,0x6C,
	0x69,0x65,0x6E,0x74,0x45,0x6E,0x74,0x72,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x43,0x6C,
	0x69,0x65,0x6E,0x74,0x4C,0x65,0x61,0x76,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x41,0x63,
	0x74,0x76,0x74,0x49,0x74,0x65,0x6D,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x41,0x63,
	0x71,0x75,0x69,0x72,0x49,0x74,0x65,0x6D,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x55,0x73,
	0x72,0x44,0x65,0x66,0x69,0x6E,0x65,0x64,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x55,0x6E,
	0x41,0x71,0x72,0x65,0x49,0x74,0x65,0x6D,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x50,0x6C,
	0x72,0x44,0x65,0x61,0x74,0x68,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x50,0x6C,
	0x72,0x44,0x79,0x69,0x6E,0x67,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x50,0x6C,
	0x72,0x45,0x71,0x49,0x74,0x6D,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x50,0x6C,
	0x72,0x4C,0x76,0x6C,0x55,0x70,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x53,0x70,
	0x61,0x77,0x6E,0x42,0x74,0x6E,0x44,0x6E,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x50,0x6C,
	0x72,0x52,0x65,0x73,0x74,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x50,0x6C,
	0x72,0x55,0x6E,0x45,0x71,0x49,0x74,0x6D,0x4D,0x6F,0x64,0x5F,0x4F,0x6E,0x43,0x75,
	0x74,0x73,0x6E,0x41,0x62,0x6F,0x72,0x74,0x4D,0x6F,0x64,0x5F,0x53,0x74,0x61,0x72,
	0x74,0x4D,0x6F,0x76,0x69,0x65,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x43,0x75,0x74,0x53,
	0x63,0x65,0x6E,0x65,0x4C,0x69,0x73,0x74,0x4D,0x6F,0x64,0x5F,0x47,0x56,0x61,0x72,
	0x5F,0x4C,0x69,0x73,0x74,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x41,0x72,0x65,0x61,
	0x5F,0x6C,0x69,0x73,0x74,0x00,0x00,0x00,0x41,0x72,0x65,0x61,0x5F,0x4E,0x61,0x6D,
	0x65,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x48,0x61,0x6B,0x4C,
	0x69,0x73,0x74,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x48,0x61,0x6B,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x43,0x61,0x63,0x68,
	0x65,0x4E,0x53,0x53,0x4C,0x69,0x73,0x74,0x52,0x65,0x73,0x52,0x65,0x66,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0x75,0x65,0x73,0x74,0x5F,0x6C,0x69,
	0x73,0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0x75,0x65,0x73,0x74,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0x75,0x65,0x73,0x74,0x42,0x61,0x73,
	0x65,0x5F,0x6C,0x69,0x73,0x74,0x00,0x00,0x51,0x75,0x65,0x73,0x74,0x42,0x61,0x73,
	0x65,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x53,0x74,0x6F,0x72,0x79,0x4E,0x50,0x43,
	0x5F,0x6C,0x69,0x73,0x74,0x00,0x00,0x00,0x53,0x74,0x6F,0x72,0x79,0x4E,0x50,0x43,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x6E,0x73,0x74,0x65,0x72,0x4E,
	0x50,0x43,0x5F,0x6C,0x69,0x73,0x74,0x00,0x4D,0x6F,0x6E,0x73,0x74,0x65,0x72,0x4E,
	0x50,0x43,0x00,0x00,0x00,0x00,0x00,0x00,0x43,0x61,0x6D,0x70,0x61,0x69,0x67,0x6E,
	0x5F,0x49,0x44,0x00,0x00,0x00,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4D,0x61,0x78,0x57,
	0x65,0x61,0x74,0x49,0x6E,0x74,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x4D,0x69,0x6E,0x57,
	0x65,0x61,0x74,0x49,0x6E,0x74,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x52,0x61,0x69,0x6E,
	0x43,0x68,0x61,0x6E,0x63,0x65,0x00,0x00,0x4D,0x6F,0x64,0x5F,0x53,0x6E,0x6F,0x77,
	0x43,0x68,0x61,0x6E,0x63,0x65,0x00,0x00,0x10,0x00,0x00,0x00,0x55,0x55,0x55,0x55,
	0x55,0x55,0x55,0x55,0x55,0x55,0x55,0x55,0x55,0x55,0x55,0x55,0x04,0x00,0x00,0x00,
	0x31,0x2E,0x36,0x30,0x1D,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x01,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0D,0x00,0x00,0x00,0x41,0x20,0x6D,0x6F,0x64,0x75,0x6C,0x65,
	0x20,0x6E,0x61,0x6D,0x65,0x06,0x00,0x00,0x00,0x4D,0x4F,0x44,0x55,0x4C,0x45,0x1F,
	0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0F,
	0x00,0x00,0x00,0x4D,0x6F,0x64,0x20,0x64,0x65,0x73,0x63,0x72,0x69,0x70,0x74,0x69,
	0x6F,0x6E,0x03,0x00,0x00,0x00,0x74,0x6C,0x6B,0x05,0x61,0x72,0x65,0x61,0x31,0x0B,
	0x6F,0x6E,0x68,0x65,0x61,0x72,0x74,0x62,0x65,0x61,0x74,0x09,0x6F,0x6E,0x6D,0x6F,
	0x64,0x6C,0x6F,0x61,0x64,0x0A,0x6F,0x6E,0x6D,0x6F,0x64,0x73,0x74,0x61,0x72,0x74,
	0x0C,0x6F,0x6E,0x63,0x6C,0x69,0x65,0x6E,0x74,0x65,0x6E,0x74,0x72,0x0C,0x6F,0x6E,
	0x63,0x6C,0x69,0x65,0x6E,0x74,0x6C,0x65,0x61,0x76,0x0B,0x6F,0x6E,0x61,0x63,0x74,
	0x76,0x74,0x69,0x74,0x65,0x6D,0x0C,0x6F,0x6E,0x61,0x63,0x71,0x75,0x69,0x72,0x69,
	0x74,0x65,0x6D,0x0C,0x6F,0x6E,0x75,0x73,0x72,0x64,0x65,0x66,0x69,0x6E,0x65,0x64,
	0x0C,0x6F,0x6E,0x75,0x6E,0x61,0x71,0x72,0x65,0x69,0x74,0x65,0x6D,0x0A,0x6F,0x6E,
	0x70,0x6C,0x72,0x64,0x65,0x61,0x74,0x68,0x0A,0x6F,0x6E,0x70,0x6C,0x72,0x64,0x79,
	0x69,0x6E,0x67,0x0A,0x6F,0x6E,0x70,0x6C,0x72,0x65,0x71,0x69,0x74,0x6D,0x0A,0x6F,
	0x6E,0x70,0x6C,0x72,0x6C,0x76,0x6C,0x75,0x70,0x0C,0x6F,0x6E,0x73,0x70,0x61,0x77,
	0x6E,0x62,0x74,0x6E,0x64,0x6E,0x09,0x6F,0x6E,0x70,0x6C,0x72,0x72,0x65,0x73,0x74,
	0x0C,0x6F,0x6E,0x70,0x6C,0x72,0x75,0x6E,0x65,0x71,0x69,0x74,0x6D,0x0C,0x6F,0x6E,
	0x63,0x75,0x74,0x73,0x6E,0x61,0x62,0x6F,0x72,0x74,0x05,0x6D,0x6F,0x76,0x69,0x65,
	0x05,0x61,0x72,0x65,0x61,0x31,0x05,0x61,0x72,0x65,0x61,0x32,0x05,0x61,0x72,0x65,
	0x61,0x33,0x04,0x00,0x00,0x00,0x68,0x61,0x6B,0x31,0x04,0x00,0x00,0x00,0x68,0x61,
	0x6B,0x32,0x04,0x00,0x00,0x00,0x68,0x61,0x6B,0x33,0x04,0x6E,0x73,0x73,0x31,0x04,
	0x6E,0x73,0x73,0x32,0x04,0x6E,0x73,0x73,0x33,0x06,0x71,0x75,0x65,0x73,0x74,0x31,
	0x06,0x71,0x75,0x65,0x73,0x74,0x32,0x06,0x71,0x75,0x65,0x73,0x74,0x33,0x0A,0x71,
	0x75,0x65,0x73,0x74,0x62,0x61,0x73,0x65,0x31,0x0A,0x71,0x75,0x65,0x73,0x74,0x62,
	0x61,0x73,0x65,0x32,0x0A,0x71,0x75,0x65,0x73,0x74,0x62,0x61,0x73,0x65,0x33,0x09,
	0x73,0x74,0x6F,0x72,0x79,0x6E,0x70,0x63,0x31,0x09,0x73,0x74,0x6F,0x72,0x79,0x6E,
	0x70,0x63,0x32,0x09,0x73,0x74,0x6F,0x72,0x79,0x6E,0x70,0x63,0x33,0x0B,0x6D,0x6F,
	0x6E,0x73,0x74,0x65,0x72,0x6E,0x70,0x63,0x31,0x0B,0x6D,0x6F,0x6E,0x73,0x74,0x65,
	0x72,0x6E,0x70,0x63,0x32,0x0B,0x6D,0x6F,0x6E,0x73,0x74,0x65,0x72,0x6E,0x70,0x63,
	0x33,0x10,0x00,0x00,0x00,0xAA,0xAA,0xAA,0xAA,0xAA,0xAA,0xAA,0xAA,0xAA,0xAA,0xAA,
	0xAA,0xAA,0xAA,0xAA,0xAA,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x02,0x00,0x00,
	0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x06,0x00,0x00,
	0x00,0x07,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x0A,0x00,0x00,
	0x00,0x0B,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x0D,0x00,0x00,0x00,0x0E,0x00,0x00,
	0x00,0x0F,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x11,0x00,0x00,0x00,0x12,0x00,0x00,
	0x00,0x13,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x15,0x00,0x00,0x00,0x16,0x00,0x00,
	0x00,0x17,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x19,0x00,0x00,0x00,0x1A,0x00,0x00,
	0x00,0x1B,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x1D,0x00,0x00,0x00,0x1E,0x00,0x00,
	0x00,0x1F,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x21,0x00,0x00,0x00,0x22,0x00,0x00,
	0x00,0x23,0x00,0x00,0x00,0x24,0x00,0x00,0x00,0x25,0x00,0x00,0x00,0x26,0x00,0x00,
	0x00,0x27,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x29,0x00,0x00,0x00,0x2A,0x00,0x00,
	0x00,0x2B,0x00,0x00,0x00,0x2C,0x00,0x00,0x00,0x2D,0x00,0x00,0x00,0x31,0x00,0x00,
	0x00,0x35,0x00,0x00,0x00,0x39,0x00,0x00,0x00,0x3D,0x00,0x00,0x00,0x41,0x00,0x00,
	0x00,0x45,0x00,0x00,0x00,0x49,0x00,0x00,0x00,0x4A,0x00,0x00,0x00,0x4B,0x00,0x00,
	0x00,0x4C,0x00,0x00,0x00,0x4D,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x02,0x00,0x00,
	0x00,0x03,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x05,0x00,0x00,
	0x00,0x06,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x08,0x00,0x00,
	0x00,0x09,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x0B,0x00,0x00,
	0x00,0x0C,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x0D,0x00,0x00,0x00,0x0E,0x00,0x00,
	0x00,0x0F,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x11,0x00,0x00,
	0x00,0x12,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x13,0x00,0x00,0x00,0x14,0x00,0x00,
	0x00,0x15,0x00,0x00,0x00
};

class IFOFile : public ::testing::Test {
protected:
	static void SetUpTestCase() {
		LangMan.clear();

		LangMan.addLanguage(Aurora::kLanguageEnglish, 0, Common::kEncodingUTF8);
	}

	static void TearDownTestCase() {
		Aurora::LanguageManager::destroy();
	}
};

GTEST_TEST_F(IFOFile, unload) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	ifo.unload();

	EXPECT_STREQ(ifo.getTag().c_str(), "");
}

GTEST_TEST_F(IFOFile, getGFF) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const Aurora::GFF3Struct *gff = ifo.getGFF();
	ASSERT_NE(gff, static_cast<const Aurora::GFF3Struct *>(0));
}

GTEST_TEST_F(IFOFile, getVersion) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getVersion(), 3);
}

GTEST_TEST_F(IFOFile, getCreatorID) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getCreatorID(), 2);
}

GTEST_TEST_F(IFOFile, isSave) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_FALSE(ifo.isSave());
}

GTEST_TEST_F(IFOFile, getTag) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_STREQ(ifo.getTag().c_str(), "MODULE");
}

GTEST_TEST_F(IFOFile, getName) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_STREQ(ifo.getName().getString().c_str(), "A module name");
}

GTEST_TEST_F(IFOFile, getDescription) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_STREQ(ifo.getDescription().getString().c_str(), "Mod description");
}

GTEST_TEST_F(IFOFile, getTLK) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_STREQ(ifo.getTLK().c_str(), "tlk");
}

GTEST_TEST_F(IFOFile, getMinVersion) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	int major, minor;
	ifo.getMinVersion(major, minor);

	EXPECT_EQ(major,  1);
	EXPECT_EQ(minor, 60);
}

GTEST_TEST_F(IFOFile, getExpansions) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getExpansions(), 1);
}

GTEST_TEST_F(IFOFile, getHAKs) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const std::vector<Common::UString> &haks = ifo.getHAKs();
	ASSERT_EQ(haks.size(), 3);

	EXPECT_STREQ(haks[0].c_str(), "hak3");
	EXPECT_STREQ(haks[1].c_str(), "hak2");
	EXPECT_STREQ(haks[2].c_str(), "hak1");
}

GTEST_TEST_F(IFOFile, getStartMovie) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_STREQ(ifo.getStartMovie().c_str(), "movie");
}

GTEST_TEST_F(IFOFile, getEntryArea) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_STREQ(ifo.getEntryArea().c_str(), "area1");
}

GTEST_TEST_F(IFOFile, getEntryPosition) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	float x, y, z;
	ifo.getEntryPosition(x, y, z);

	EXPECT_FLOAT_EQ(x, 42.0f);
	EXPECT_FLOAT_EQ(y, 42.1f);
	EXPECT_FLOAT_EQ(z, 42.2f);
}

GTEST_TEST_F(IFOFile, getEntryDirection) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	float x, y;
	ifo.getEntryDirection(x, y);

	EXPECT_FLOAT_EQ(x, 52.0f);
	EXPECT_FLOAT_EQ(y, 52.1f);
}

GTEST_TEST_F(IFOFile, getAreas) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const std::vector<Common::UString> &areas = ifo.getAreas();
	ASSERT_EQ(areas.size(), 3);

	EXPECT_STREQ(areas[0].c_str(), "area1");
	EXPECT_STREQ(areas[1].c_str(), "area2");
	EXPECT_STREQ(areas[2].c_str(), "area3");
}

GTEST_TEST_F(IFOFile, getNSSCache) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const std::vector<Common::UString> &nss = ifo.getNSSCache();
	ASSERT_EQ(nss.size(), 3);

	EXPECT_STREQ(nss[0].c_str(), "nss1");
	EXPECT_STREQ(nss[1].c_str(), "nss2");
	EXPECT_STREQ(nss[2].c_str(), "nss3");
}

GTEST_TEST_F(IFOFile, getQuests) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const std::vector<Common::UString> &quests = ifo.getQuests();
	ASSERT_EQ(quests.size(), 3);

	EXPECT_STREQ(quests[0].c_str(), "quest1");
	EXPECT_STREQ(quests[1].c_str(), "quest2");
	EXPECT_STREQ(quests[2].c_str(), "quest3");
}

GTEST_TEST_F(IFOFile, getQuestDBs) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const std::vector<Common::UString> &questdbs = ifo.getQuestDBs();
	ASSERT_EQ(questdbs.size(), 3);

	EXPECT_STREQ(questdbs[0].c_str(), "questbase1");
	EXPECT_STREQ(questdbs[1].c_str(), "questbase2");
	EXPECT_STREQ(questdbs[2].c_str(), "questbase3");
}

GTEST_TEST_F(IFOFile, getStoryNPCs) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const std::vector<Common::UString> &storynpcs = ifo.getStoryNPCs();
	ASSERT_EQ(storynpcs.size(), 3);

	EXPECT_STREQ(storynpcs[0].c_str(), "storynpc1");
	EXPECT_STREQ(storynpcs[1].c_str(), "storynpc2");
	EXPECT_STREQ(storynpcs[2].c_str(), "storynpc3");
}

GTEST_TEST_F(IFOFile, getMonsterNPCs) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	const std::vector<Common::UString> &monsternpcs = ifo.getMonsterNPCs();
	ASSERT_EQ(monsternpcs.size(), 3);

	EXPECT_STREQ(monsternpcs[0].c_str(), "monsternpc1");
	EXPECT_STREQ(monsternpcs[1].c_str(), "monsternpc2");
	EXPECT_STREQ(monsternpcs[2].c_str(), "monsternpc3");
}

GTEST_TEST_F(IFOFile, getStartTime) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	uint8_t hour, day, month;
	uint32_t year;

	ifo.getStartTime(hour, day, month, year);

	EXPECT_EQ(hour ,  13);
	EXPECT_EQ(day  ,  1);
	EXPECT_EQ(month,  6);
	EXPECT_EQ(year ,  2342);
}

GTEST_TEST_F(IFOFile, getMinutesPerHour) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getMinutesPerHour(), 2);
}

GTEST_TEST_F(IFOFile, getMinWeatherIntensity) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getMinWeatherIntensity(), 24);
}

GTEST_TEST_F(IFOFile, getMaxWeatherIntensity) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getMaxWeatherIntensity(), 23);
}

GTEST_TEST_F(IFOFile, getRainChance) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getRainChance(), 25);
}

GTEST_TEST_F(IFOFile, getSnowChance) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_EQ(ifo.getSnowChance(), 26);
}

GTEST_TEST_F(IFOFile, getXPScale) {
	Aurora::IFOFile ifo;
	ifo.load(new Common::MemoryReadStream(kIFOFile));

	EXPECT_FLOAT_EQ(ifo.getXPScale(), 0.1f);
}
