/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Unit tests for our DLGFile class.
 */

/* NOTE: We aren't testing anything script-related here, because
 * the scripts are loaded from the ResourceManager. */

#include <vector>

#include "gtest/gtest.h"

#include "src/common/util.h"
#include "src/common/error.h"
#include "src/common/memreadstream.h"

#include "src/aurora/dlgfile.h"
#include "src/aurora/language.h"

static const byte kDLGFile[] = {
	0x44,0x4C,0x47,0x20,0x56,0x33,0x2E,0x32,0x38,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,
	0xC8,0x00,0x00,0x00,0x4B,0x00,0x00,0x00,0x4C,0x04,0x00,0x00,0x19,0x00,0x00,0x00,
	0xDC,0x05,0x00,0x00,0x07,0x01,0x00,0x00,0xE3,0x06,0x00,0x00,0x2C,0x01,0x00,0x00,
	0x0F,0x08,0x00,0x00,0x4C,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,
	0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x24,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x50,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x5C,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x68,0x00,0x00,0x00,
	0x0A,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x90,0x00,0x00,0x00,0x03,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x9C,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC0,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0xD0,0x00,0x00,0x00,
	0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF4,0x00,0x00,0x00,0x03,0x00,0x00,0x00,
	0x02,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x24,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0E,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x0F,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,
	0x07,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x08,0x00,0x00,0x00,
	0x1D,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x0C,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x26,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,
	0xFF,0xFF,0xFF,0xFF,0x0A,0x00,0x00,0x00,0x0D,0x00,0x00,0x00,0x27,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x0E,0x00,0x00,0x00,0x31,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,
	0x0F,0x00,0x00,0x00,0x3D,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x10,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0x11,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x12,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x13,0x00,0x00,0x00,0x4C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x14,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x13,0x00,0x00,0x00,0x4D,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x14,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x07,0x00,0x00,0x00,
	0x4E,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,
	0x0A,0x00,0x00,0x00,0x5F,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x7D,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,
	0x0A,0x00,0x00,0x00,0x0D,0x00,0x00,0x00,0x7E,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x0E,0x00,0x00,0x00,0x82,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x83,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0x11,0x00,0x00,0x00,0x18,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x12,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x13,0x00,0x00,0x00,0x87,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x14,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0x15,0x00,0x00,0x00,0x20,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x1D,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x09,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,
	0x88,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0xA6,0x00,0x00,0x00,
	0x04,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x0A,0x00,0x00,0x00,
	0x0D,0x00,0x00,0x00,0xA7,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x0E,0x00,0x00,0x00,
	0xAB,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0xAC,0x00,0x00,0x00,
	0x0F,0x00,0x00,0x00,0x16,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x12,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x13,0x00,0x00,0x00,
	0xB0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
	0x0A,0x00,0x00,0x00,0x17,0x00,0x00,0x00,0xB1,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x08,0x00,0x00,0x00,0x22,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0xB5,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0xD3,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x0C,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x0A,0x00,0x00,0x00,0x0D,0x00,0x00,0x00,
	0xD4,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x0E,0x00,0x00,0x00,0xD8,0x00,0x00,0x00,
	0x0A,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0xD9,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x16,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x12,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x13,0x00,0x00,0x00,0xDD,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0xDE,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0xFC,0x00,0x00,0x00,0x04,0x00,0x00,0x00,
	0x0C,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x0A,0x00,0x00,0x00,0x0D,0x00,0x00,0x00,
	0xFD,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x0E,0x00,0x00,0x00,0x01,0x01,0x00,0x00,
	0x0A,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0x02,0x01,0x00,0x00,0x0F,0x00,0x00,0x00,
	0x16,0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0x18,0x00,0x00,0x00,
	0x44,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x0B,0x00,0x00,0x00,0x13,0x00,0x00,0x00,0x06,0x01,0x00,0x00,0x44,0x65,0x6C,0x61,
	0x79,0x45,0x6E,0x74,0x72,0x79,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x65,0x6C,0x61,
	0x79,0x52,0x65,0x70,0x6C,0x79,0x00,0x00,0x00,0x00,0x00,0x00,0x4E,0x75,0x6D,0x57,
	0x6F,0x72,0x64,0x73,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x45,0x6E,0x64,0x43,
	0x6F,0x6E,0x76,0x65,0x72,0x73,0x61,0x74,0x69,0x6F,0x6E,0x00,0x45,0x6E,0x64,0x43,
	0x6F,0x6E,0x76,0x65,0x72,0x41,0x62,0x6F,0x72,0x74,0x00,0x00,0x50,0x72,0x65,0x76,
	0x65,0x6E,0x74,0x5A,0x6F,0x6F,0x6D,0x49,0x6E,0x00,0x00,0x00,0x45,0x6E,0x74,0x72,
	0x79,0x4C,0x69,0x73,0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x53,0x70,0x65,0x61,
	0x6B,0x65,0x72,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x41,0x6E,0x69,0x6D,
	0x61,0x74,0x69,0x6F,0x6E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x41,0x6E,0x69,0x6D,
	0x4C,0x6F,0x6F,0x70,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x54,0x65,0x78,0x74,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x53,0x63,0x72,0x69,
	0x70,0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x65,0x6C,0x61,
	0x79,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x43,0x6F,0x6D,0x6D,
	0x65,0x6E,0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x53,0x6F,0x75,0x6E,
	0x64,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0x75,0x65,0x73,
	0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0x75,0x65,0x73,
	0x74,0x45,0x6E,0x74,0x72,0x79,0x00,0x00,0x00,0x00,0x00,0x00,0x52,0x65,0x70,0x6C,
	0x69,0x65,0x73,0x4C,0x69,0x73,0x74,0x00,0x00,0x00,0x00,0x00,0x49,0x6E,0x64,0x65,
	0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x41,0x63,0x74,0x69,
	0x76,0x65,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x49,0x73,0x43,0x68,
	0x69,0x6C,0x64,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x52,0x65,0x70,0x6C,
	0x79,0x4C,0x69,0x73,0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x45,0x6E,0x74,0x72,
	0x69,0x65,0x73,0x4C,0x69,0x73,0x74,0x00,0x00,0x00,0x00,0x00,0x4C,0x69,0x6E,0x6B,
	0x43,0x6F,0x6D,0x6D,0x65,0x6E,0x74,0x00,0x00,0x00,0x00,0x00,0x53,0x74,0x61,0x72,
	0x74,0x69,0x6E,0x67,0x4C,0x69,0x73,0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x1C,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x01,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x0C,0x00,0x00,0x00,0x4E,0x50,0x43,0x20,0x67,0x72,0x65,0x65,0x74,0x69,
	0x6E,0x67,0x00,0x06,0x00,0x00,0x00,0x46,0x6F,0x6F,0x62,0x61,0x72,0x0B,0x61,0x6C,
	0x5F,0x61,0x6E,0x5F,0x62,0x65,0x65,0x73,0x31,0x0B,0x00,0x00,0x00,0x43,0x61,0x74,
	0x65,0x67,0x6F,0x72,0x79,0x30,0x30,0x30,0x00,0x00,0x0D,0x00,0x00,0x00,0x58,0x32,
	0x5F,0x44,0x55,0x45,0x52,0x47,0x41,0x52,0x30,0x30,0x33,0x1A,0x00,0x00,0x00,0xFF,
	0xFF,0xFF,0xFF,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x4E,
	0x50,0x43,0x20,0x6C,0x69,0x6E,0x65,0x20,0x31,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x1A,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x01,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x50,0x43,0x20,0x72,0x65,0x70,0x6C,0x79,
	0x20,0x32,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x1A,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0A,0x00,0x00,0x00,0x50,0x43,0x20,0x72,0x65,0x70,0x6C,0x79,0x20,0x31,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1A,0x00,0x00,0x00,0xFF,0xFF,
	0xFF,0xFF,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x50,0x43,
	0x20,0x72,0x65,0x70,0x6C,0x79,0x20,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x03,
	0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x25,
	0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x09,
	0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x0B,0x00,0x00,0x00,0x0C,0x00,0x00,0x00,0x0D,
	0x00,0x00,0x00,0x0E,0x00,0x00,0x00,0x0F,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x11,
	0x00,0x00,0x00,0x12,0x00,0x00,0x00,0x13,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x15,
	0x00,0x00,0x00,0x16,0x00,0x00,0x00,0x17,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x19,
	0x00,0x00,0x00,0x1A,0x00,0x00,0x00,0x1B,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x1D,
	0x00,0x00,0x00,0x1E,0x00,0x00,0x00,0x1F,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x21,
	0x00,0x00,0x00,0x22,0x00,0x00,0x00,0x23,0x00,0x00,0x00,0x24,0x00,0x00,0x00,0x26,
	0x00,0x00,0x00,0x27,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x29,0x00,0x00,0x00,0x2A,
	0x00,0x00,0x00,0x2B,0x00,0x00,0x00,0x2C,0x00,0x00,0x00,0x2D,0x00,0x00,0x00,0x2E,
	0x00,0x00,0x00,0x2F,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x31,0x00,0x00,0x00,0x32,
	0x00,0x00,0x00,0x33,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x35,0x00,0x00,0x00,0x36,
	0x00,0x00,0x00,0x37,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x39,0x00,0x00,0x00,0x3A,
	0x00,0x00,0x00,0x3B,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x3D,0x00,0x00,0x00,0x3E,
	0x00,0x00,0x00,0x3F,0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x41,0x00,0x00,0x00,0x42,
	0x00,0x00,0x00,0x43,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x45,0x00,0x00,0x00,0x46,
	0x00,0x00,0x00,0x47,0x00,0x00,0x00,0x49,0x00,0x00,0x00,0x4A,0x00,0x00,0x00,0x02,
	0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,
	0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x03,
	0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x0A,0x00,0x00,0x00,0x01,
	0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x0B,0x00,0x00,0x00
};

class DLGFile : public ::testing::Test {
protected:
	static void SetUpTestCase() {
		LangMan.clear();

		LangMan.addLanguage(Aurora::kLanguageEnglish, 0, Common::kEncodingUTF8);
	}

	static void TearDownTestCase() {
		Aurora::LanguageManager::destroy();
	}
};

GTEST_TEST_F(DLGFile, getNoZoomIn) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	EXPECT_FALSE(dlg.getNoZoomIn());
}

GTEST_TEST_F(DLGFile, getDelayEntry) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	EXPECT_EQ(dlg.getDelayEntry(), 0);
}

GTEST_TEST_F(DLGFile, getDelayReply) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	EXPECT_EQ(dlg.getDelayReply(), 0);
}

GTEST_TEST_F(DLGFile, hasEnded) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	EXPECT_TRUE(dlg.hasEnded());
}

GTEST_TEST_F(DLGFile, getCurrentEntry) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	EXPECT_EQ(dlg.getCurrentEntry(), static_cast<const Aurora::DLGFile::Line *>(0));
}

GTEST_TEST_F(DLGFile, getCurrentReplies) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	const std::vector<const Aurora::DLGFile::Line *> &replies = dlg.getCurrentReplies();

	EXPECT_EQ(replies.size(), 0);
}

GTEST_TEST_F(DLGFile, getOneLiner) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	EXPECT_EQ(dlg.getOneLiner(), static_cast<const Aurora::DLGFile::Line *>(0));
}

void checkLine(size_t n, const Aurora::DLGFile::Line *entry, uint32_t id,
               const char *speaker, const char *text, const char *sound,
               uint32_t animation, const char *quest, uint32_t questEntry, bool isEnd) {

	EXPECT_EQ(entry->id, id) << "At case " << n;

	EXPECT_STREQ(entry->speaker.c_str(), speaker) << "At case " << n;
	EXPECT_STREQ(entry->text.getString().c_str(), text) << "At case " << n;
	EXPECT_STREQ(entry->sound.c_str(), sound) << "At case " << n;

	EXPECT_EQ(entry->animation, animation) << "At case " << n;

	EXPECT_STREQ(entry->quest.c_str(), quest) << "At case " << n;
	EXPECT_EQ(entry->questEntry, questEntry) << "At case " << n;

	EXPECT_EQ(entry->isEnd, isEnd) << "At case " << n;
}

GTEST_TEST_F(DLGFile, startConversation) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	dlg.startConversation();

	EXPECT_EQ(dlg.getOneLiner(), static_cast<const Aurora::DLGFile::Line *>(0));
	EXPECT_FALSE(dlg.hasEnded());

	{
		const Aurora::DLGFile::Line *entry = dlg.getCurrentEntry();
		const std::vector<const Aurora::DLGFile::Line *> &replies = dlg.getCurrentReplies();

		ASSERT_NE(entry, static_cast<const Aurora::DLGFile::Line *>(0));
		ASSERT_EQ(replies.size(), 2);

		ASSERT_NE(replies[0], static_cast<const Aurora::DLGFile::Line *>(0));
		ASSERT_NE(replies[1], static_cast<const Aurora::DLGFile::Line *>(0));

		checkLine(0, entry, 0, "", "NPC greeting", "al_an_bees1", 29, "Category000", 1, false);

		checkLine(1, replies[0], 1, "", "PC reply 1", "", 34, "", 0xFFFFFFFF, false);
		checkLine(2, replies[1], 0, "", "PC reply 2", "", 29, "", 0xFFFFFFFF, false);
	}

	dlg.pickReply(1);

	EXPECT_EQ(dlg.getOneLiner(), static_cast<const Aurora::DLGFile::Line *>(0));
	EXPECT_FALSE(dlg.hasEnded());

	{
		const Aurora::DLGFile::Line *entry = dlg.getCurrentEntry();
		const std::vector<const Aurora::DLGFile::Line *> &replies = dlg.getCurrentReplies();

		ASSERT_NE(entry, static_cast<const Aurora::DLGFile::Line *>(0));
		ASSERT_EQ(replies.size(), 1);

		ASSERT_NE(replies[0], static_cast<const Aurora::DLGFile::Line *>(0));

		checkLine(3, entry, 1, "X2_DUERGAR003", "NPC line 1", "", 28, "", 0xFFFFFFFF, false);

		checkLine(4, replies[0], 2, "", "PC reply 3", "", 0, "", 0xFFFFFFFF, true);
	}

	dlg.pickReply(2);

	EXPECT_EQ(dlg.getOneLiner(), static_cast<const Aurora::DLGFile::Line *>(0));
	EXPECT_TRUE(dlg.hasEnded());
}

GTEST_TEST_F(DLGFile, abortConversation) {
	Aurora::DLGFile dlg(new Common::MemoryReadStream(kDLGFile));

	dlg.startConversation();

	EXPECT_FALSE(dlg.hasEnded());

	dlg.abortConversation();

	EXPECT_TRUE(dlg.hasEnded());
}
