/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Unit tests for the ActionScript interpreter.
 */

#include "gtest/gtest.h"

#include "src/common/ustring.h"
#include "src/common/memreadstream.h"

#include "src/aurora/actionscript/asbuffer.h"
#include "src/aurora/actionscript/function.h"
#include "src/aurora/actionscript/array.h"

/*
 *  class Test {
 *      private var i;
 *
 *      public function Test() {
 *          i = 1;
 *      }
 *
 *      public function inc() {
 *          i += 1;
 *  	}
 *
 *  	public function dec() {
 *  		i -= 1;
 *  	}
 *
 *  	public function getI() {
 *  		return i;
 *  	}
 *  }
 */
static const byte kTestClass[] = {
	0x88, 0x37, 0x00, 0x08, 0x00, 0x54, 0x65, 0x73, 0x74, 0x00, 0x5f, 0x67,
	0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x69, 0x00, 0x70, 0x72, 0x6f, 0x74,
	0x6f, 0x74, 0x79, 0x70, 0x65, 0x00, 0x69, 0x6e, 0x63, 0x00, 0x64, 0x65,
	0x63, 0x00, 0x67, 0x65, 0x74, 0x49, 0x00, 0x41, 0x53, 0x53, 0x65, 0x74,
	0x50, 0x72, 0x6f, 0x70, 0x46, 0x6c, 0x61, 0x67, 0x73, 0x00, 0x96, 0x02,
	0x00, 0x08, 0x00, 0x1c, 0x12, 0x12, 0x9d, 0x02, 0x00, 0xb7, 0x00, 0x96,
	0x02, 0x00, 0x08, 0x01, 0x1c, 0x96, 0x02, 0x00, 0x08, 0x00, 0x8e, 0x08,
	0x00, 0x00, 0x00, 0x00, 0x02, 0x29, 0x00, 0x0d, 0x00, 0x96, 0x09, 0x00,
	0x04, 0x01, 0x08, 0x02, 0x07, 0x01, 0x00, 0x00, 0x00, 0x4f, 0x87, 0x01,
	0x00, 0x00, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x00, 0x08, 0x03, 0x4e, 0x87,
	0x01, 0x00, 0x01, 0x17, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x04, 0x8e,
	0x08, 0x00, 0x00, 0x00, 0x00, 0x02, 0x29, 0x00, 0x16, 0x00, 0x96, 0x08,
	0x00, 0x04, 0x01, 0x08, 0x02, 0x04, 0x01, 0x08, 0x02, 0x4e, 0x96, 0x05,
	0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x47, 0x4f, 0x4f, 0x96, 0x04, 0x00,
	0x04, 0x01, 0x08, 0x05, 0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x02, 0x29,
	0x00, 0x16, 0x00, 0x96, 0x08, 0x00, 0x04, 0x01, 0x08, 0x02, 0x04, 0x01,
	0x08, 0x02, 0x4e, 0x96, 0x05, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x0b,
	0x4f, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x06, 0x8e, 0x08, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x29, 0x00, 0x09, 0x00, 0x96, 0x04, 0x00, 0x04,
	0x01, 0x08, 0x02, 0x4e, 0x3e, 0x4f, 0x96, 0x0f, 0x00, 0x07, 0x01, 0x00,
	0x00, 0x00, 0x02, 0x04, 0x01, 0x07, 0x03, 0x00, 0x00, 0x00, 0x08, 0x07,
	0x3d, 0x17, 0x00
};

/*
 *  class Test2_a {
 *      private var a, b;
 *
 *      function Test2_a(a:Boolean, b:Boolean) {
 *          this.a = a;
 *          this.b = b;
 *      }
 *
 *      function getA() {
 *          return a;
 *      }
 *
 *      function getB() {
 *          return b;
 *      }
 *  }
 */
static const byte kTestClass2a[] = {
	0x88, 0x39, 0x00, 0x08, 0x00, 0x54, 0x65, 0x73, 0x74, 0x32, 0x5f, 0x61,
	0x00, 0x5f, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x61, 0x00, 0x62,
	0x00, 0x70, 0x72, 0x6f, 0x74, 0x6f, 0x74, 0x79, 0x70, 0x65, 0x00, 0x67,
	0x65, 0x74, 0x41, 0x00, 0x67, 0x65, 0x74, 0x42, 0x00, 0x41, 0x53, 0x53,
	0x65, 0x74, 0x50, 0x72, 0x6f, 0x70, 0x46, 0x6c, 0x61, 0x67, 0x73, 0x00,
	0x96, 0x02, 0x00, 0x08, 0x00, 0x1c, 0x12, 0x12, 0x9d, 0x02, 0x00, 0x8e,
	0x00, 0x96, 0x02, 0x00, 0x08, 0x01, 0x1c, 0x96, 0x02, 0x00, 0x08, 0x00,
	0x8e, 0x0e, 0x00, 0x00, 0x02, 0x00, 0x04, 0x29, 0x00, 0x02, 0x61, 0x00,
	0x03, 0x62, 0x00, 0x14, 0x00, 0x96, 0x06, 0x00, 0x04, 0x01, 0x08, 0x02,
	0x04, 0x02, 0x4f, 0x96, 0x06, 0x00, 0x04, 0x01, 0x08, 0x03, 0x04, 0x03,
	0x4f, 0x87, 0x01, 0x00, 0x00, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x00, 0x08,
	0x04, 0x4e, 0x87, 0x01, 0x00, 0x01, 0x17, 0x96, 0x04, 0x00, 0x04, 0x01,
	0x08, 0x05, 0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x02, 0x29, 0x00, 0x09,
	0x00, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x02, 0x4e, 0x3e, 0x4f, 0x96,
	0x04, 0x00, 0x04, 0x01, 0x08, 0x06, 0x8e, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x02, 0x29, 0x00, 0x09, 0x00, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x03,
	0x4e, 0x3e, 0x4f, 0x96, 0x0f, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x02,
	0x04, 0x01, 0x07, 0x03, 0x00, 0x00, 0x00, 0x08, 0x07, 0x3d, 0x17, 0x00
};

/*
 *  class Test2_b extends Test2_a {
 *      function Test2_b(a:Boolean, b:Boolean) {
 *          super(a, b);
 *      }
 *
 *      function And() {
 *          return getA() || getB();
 *      }
 *
 *      function Or() {
 *          return getA() && getB();
 *      }
 *  }
 */
static const byte kTestClass2b[] = {
	0x88, 0x44, 0x00, 0x09, 0x00, 0x54, 0x65, 0x73, 0x74, 0x32, 0x5f, 0x62,
	0x00, 0x5f, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x54, 0x65, 0x73,
	0x74, 0x32, 0x5f, 0x61, 0x00, 0x70, 0x72, 0x6f, 0x74, 0x6f, 0x74, 0x79,
	0x70, 0x65, 0x00, 0x41, 0x6e, 0x64, 0x00, 0x67, 0x65, 0x74, 0x41, 0x00,
	0x67, 0x65, 0x74, 0x42, 0x00, 0x4f, 0x72, 0x00, 0x41, 0x53, 0x53, 0x65,
	0x74, 0x50, 0x72, 0x6f, 0x70, 0x46, 0x6c, 0x61, 0x67, 0x73, 0x00, 0x96,
	0x02, 0x00, 0x08, 0x00, 0x1c, 0x12, 0x12, 0x9d, 0x02, 0x00, 0xc7, 0x00,
	0x96, 0x02, 0x00, 0x08, 0x01, 0x1c, 0x96, 0x02, 0x00, 0x08, 0x00, 0x8e,
	0x0e, 0x00, 0x00, 0x02, 0x00, 0x05, 0x19, 0x00, 0x03, 0x61, 0x00, 0x04,
	0x62, 0x00, 0x11, 0x00, 0x96, 0x0c, 0x00, 0x04, 0x04, 0x04, 0x03, 0x07,
	0x02, 0x00, 0x00, 0x00, 0x04, 0x02, 0x03, 0x52, 0x17, 0x87, 0x01, 0x00,
	0x00, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x00, 0x08, 0x02, 0x1c, 0x69, 0x96,
	0x04, 0x00, 0x04, 0x00, 0x08, 0x03, 0x4e, 0x87, 0x01, 0x00, 0x01, 0x17,
	0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x04, 0x8e, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x02, 0x29, 0x00, 0x23, 0x00, 0x96, 0x09, 0x00, 0x07, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x01, 0x08, 0x05, 0x52, 0x4c, 0x12, 0x9d, 0x02, 0x00,
	0x0e, 0x00, 0x17, 0x96, 0x09, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x04,
	0x01, 0x08, 0x06, 0x52, 0x3e, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08,
	0x07, 0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x02, 0x29, 0x00, 0x22, 0x00,
	0x96, 0x09, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x04, 0x01, 0x08, 0x05,
	0x52, 0x4c, 0x9d, 0x02, 0x00, 0x0e, 0x00, 0x17, 0x96, 0x09, 0x00, 0x07,
	0x00, 0x00, 0x00, 0x00, 0x04, 0x01, 0x08, 0x06, 0x52, 0x3e, 0x4f, 0x96,
	0x0f, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x02, 0x04, 0x01, 0x07, 0x03,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x3d, 0x17, 0x00
};

/*
 *	class ArrayTest {
 *  	private var arr;
 *
 *		public function ArrayTest() {
 *			arr = [1, 2, 3];
 *		}
 *
 *		public function pushTwoValues(v1, v2) {
 *			arr.push(v1);
 *			arr.push(v2);
 *		}
 *
 *		public function popValue() {
 *			return arr.pop();
 *		}
 *
 *		public function setFirstZero() {
 *			arr[0] = 0;
 *		}
 *	}
 */
static const byte kArrayTestClass[] = {
	0x88, 0x5e, 0x00, 0x0a, 0x00, 0x41, 0x72, 0x72, 0x61, 0x79, 0x54, 0x65,
	0x73, 0x74, 0x00, 0x5f, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x61,
	0x72, 0x72, 0x00, 0x70, 0x72, 0x6f, 0x74, 0x6f, 0x74, 0x79, 0x70, 0x65,
	0x00, 0x70, 0x75, 0x73, 0x68, 0x54, 0x77, 0x6f, 0x56, 0x61, 0x6c, 0x75,
	0x65, 0x73, 0x00, 0x70, 0x75, 0x73, 0x68, 0x00, 0x70, 0x6f, 0x70, 0x56,
	0x61, 0x6c, 0x75, 0x65, 0x00, 0x70, 0x6f, 0x70, 0x00, 0x73, 0x65, 0x74,
	0x46, 0x69, 0x72, 0x73, 0x74, 0x5a, 0x65, 0x72, 0x6f, 0x00, 0x41, 0x53,
	0x53, 0x65, 0x74, 0x50, 0x72, 0x6f, 0x70, 0x46, 0x6c, 0x61, 0x67, 0x73,
	0x00, 0x96, 0x02, 0x00, 0x08, 0x00, 0x1c, 0x12, 0x12, 0x9d, 0x02, 0x00,
	0xf0, 0x00, 0x96, 0x02, 0x00, 0x08, 0x01, 0x1c, 0x96, 0x02, 0x00, 0x08,
	0x00, 0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x02, 0x29, 0x00, 0x1d, 0x00,
	0x96, 0x18, 0x00, 0x04, 0x01, 0x08, 0x02, 0x07, 0x03, 0x00, 0x00, 0x00,
	0x07, 0x02, 0x00, 0x00, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x07, 0x03,
	0x00, 0x00, 0x00, 0x42, 0x4f, 0x87, 0x01, 0x00, 0x00, 0x4f, 0x96, 0x04,
	0x00, 0x04, 0x00, 0x08, 0x03, 0x4e, 0x87, 0x01, 0x00, 0x01, 0x17, 0x96,
	0x04, 0x00, 0x04, 0x01, 0x08, 0x04, 0x8e, 0x10, 0x00, 0x00, 0x02, 0x00,
	0x04, 0x29, 0x00, 0x02, 0x76, 0x31, 0x00, 0x03, 0x76, 0x32, 0x00, 0x2c,
	0x00, 0x96, 0x0b, 0x00, 0x04, 0x02, 0x07, 0x01, 0x00, 0x00, 0x00, 0x04,
	0x01, 0x08, 0x02, 0x4e, 0x96, 0x02, 0x00, 0x08, 0x05, 0x52, 0x17, 0x96,
	0x0b, 0x00, 0x04, 0x03, 0x07, 0x01, 0x00, 0x00, 0x00, 0x04, 0x01, 0x08,
	0x02, 0x4e, 0x96, 0x02, 0x00, 0x08, 0x05, 0x52, 0x17, 0x4f, 0x96, 0x04,
	0x00, 0x04, 0x01, 0x08, 0x06, 0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x02,
	0x29, 0x00, 0x14, 0x00, 0x96, 0x09, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00,
	0x04, 0x01, 0x08, 0x02, 0x4e, 0x96, 0x02, 0x00, 0x08, 0x07, 0x52, 0x3e,
	0x4f, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x08, 0x8e, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x02, 0x29, 0x00, 0x16, 0x00, 0x96, 0x04, 0x00, 0x04, 0x01,
	0x08, 0x02, 0x4e, 0x96, 0x0a, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x07,
	0x00, 0x00, 0x00, 0x00, 0x4f, 0x4f, 0x96, 0x0f, 0x00, 0x07, 0x01, 0x00,
	0x00, 0x00, 0x02, 0x04, 0x01, 0x07, 0x03, 0x00, 0x00, 0x00, 0x08, 0x09,
	0x3d, 0x17, 0x00
};

/*
 *	class A {
 *		var test = 0
 *		function A() {
 *			test += 1
 *		}
 *
 *		function inc() {
 *			this.test = this.test + 1;
 *		}
 *
 *		function dec() {
 *			this.test = this.test - 1;
 *		}
 *	}
 */
static const byte kAClass[] = {
	0x88, 0x32, 0x00, 0x07, 0x00, 0x41, 0x00, 0x5f,
	0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x74,
	0x65, 0x73, 0x74, 0x00, 0x70, 0x72, 0x6f, 0x74,
	0x6f, 0x74, 0x79, 0x70, 0x65, 0x00, 0x69, 0x6e,
	0x63, 0x00, 0x64, 0x65, 0x63, 0x00, 0x41, 0x53,
	0x53, 0x65, 0x74, 0x50, 0x72, 0x6f, 0x70, 0x46,
	0x6c, 0x61, 0x67, 0x73, 0x00, 0x96, 0x02, 0x00,
	0x08, 0x00, 0x1c, 0x12, 0x12, 0x9d, 0x02, 0x00,
	0xb1, 0x00, 0x96, 0x02, 0x00, 0x08, 0x01, 0x1c,
	0x96, 0x02, 0x00, 0x08, 0x00, 0x8e, 0x08, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x29, 0x00, 0x16, 0x00,
	0x96, 0x08, 0x00, 0x04, 0x01, 0x08, 0x02, 0x04,
	0x01, 0x08, 0x02, 0x4e, 0x96, 0x05, 0x00, 0x07,
	0x01, 0x00, 0x00, 0x00, 0x47, 0x4f, 0x87, 0x01,
	0x00, 0x00, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x00,
	0x08, 0x03, 0x4e, 0x87, 0x01, 0x00, 0x01, 0x17,
	0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x04, 0x8e,
	0x08, 0x00, 0x00, 0x00, 0x00, 0x02, 0x29, 0x00,
	0x16, 0x00, 0x96, 0x08, 0x00, 0x04, 0x01, 0x08,
	0x02, 0x04, 0x01, 0x08, 0x02, 0x4e, 0x96, 0x05,
	0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x47, 0x4f,
	0x4f, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x05,
	0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x02, 0x29,
	0x00, 0x16, 0x00, 0x96, 0x08, 0x00, 0x04, 0x01,
	0x08, 0x02, 0x04, 0x01, 0x08, 0x02, 0x4e, 0x96,
	0x05, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x0b,
	0x4f, 0x4f, 0x96, 0x0f, 0x00, 0x07, 0x01, 0x00,
	0x00, 0x00, 0x02, 0x04, 0x01, 0x07, 0x03, 0x00,
	0x00, 0x00, 0x08, 0x06, 0x3d, 0x17, 0x96, 0x09,
	0x00, 0x04, 0x01, 0x08, 0x02, 0x07, 0x00, 0x00,
	0x00, 0x00, 0x4f, 0x00
};

/*
 *	class B extends A {
 *		function B() {
 *			super()
 *			test += 1
 *		}
 *	}
 */
static const byte kBClass[] = {
	0x88, 0x2c, 0x00, 0x06, 0x00, 0x42, 0x00, 0x5f,
	0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x74,
	0x65, 0x73, 0x74, 0x00, 0x41, 0x00, 0x70, 0x72,
	0x6f, 0x74, 0x6f, 0x74, 0x79, 0x70, 0x65, 0x00,
	0x41, 0x53, 0x53, 0x65, 0x74, 0x50, 0x72, 0x6f,
	0x70, 0x46, 0x6c, 0x61, 0x67, 0x73, 0x00, 0x96,
	0x02, 0x00, 0x08, 0x00, 0x1c, 0x12, 0x12, 0x9d,
	0x02, 0x00, 0x68, 0x00, 0x96, 0x02, 0x00, 0x08,
	0x01, 0x1c, 0x96, 0x02, 0x00, 0x08, 0x00, 0x8e,
	0x08, 0x00, 0x00, 0x00, 0x00, 0x03, 0x19, 0x00,
	0x23, 0x00, 0x96, 0x08, 0x00, 0x07, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x02, 0x03, 0x52, 0x17, 0x96,
	0x08, 0x00, 0x04, 0x01, 0x08, 0x02, 0x04, 0x01,
	0x08, 0x02, 0x4e, 0x96, 0x05, 0x00, 0x07, 0x01,
	0x00, 0x00, 0x00, 0x47, 0x4f, 0x87, 0x01, 0x00,
	0x00, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x00, 0x08,
	0x03, 0x1c, 0x69, 0x96, 0x04, 0x00, 0x04, 0x00,
	0x08, 0x04, 0x4e, 0x87, 0x01, 0x00, 0x01, 0x17,
	0x96, 0x0f, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00,
	0x02, 0x04, 0x01, 0x07, 0x03, 0x00, 0x00, 0x00,
	0x08, 0x05, 0x3d, 0x17, 0x00
};

/*
 *	class C extends B {
 *		function C() {
 *			super()
 *			test += 1
 *		}
 *
 *		function inc2() {
 *			super.inc()
 *			super.inc()
 *		}
 *	}
 */
static const byte kCClass[] = {
		0x88, 0x35, 0x00, 0x08, 0x00, 0x43, 0x00, 0x5f,
		0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x74,
		0x65, 0x73, 0x74, 0x00, 0x42, 0x00, 0x70, 0x72,
		0x6f, 0x74, 0x6f, 0x74, 0x79, 0x70, 0x65, 0x00,
		0x69, 0x6e, 0x63, 0x32, 0x00, 0x69, 0x6e, 0x63,
		0x00, 0x41, 0x53, 0x53, 0x65, 0x74, 0x50, 0x72,
		0x6f, 0x70, 0x46, 0x6c, 0x61, 0x67, 0x73, 0x00,
		0x96, 0x02, 0x00, 0x08, 0x00, 0x1c, 0x12, 0x12,
		0x9d, 0x02, 0x00, 0x97, 0x00, 0x96, 0x02, 0x00,
		0x08, 0x01, 0x1c, 0x96, 0x02, 0x00, 0x08, 0x00,
		0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x03, 0x19,
		0x00, 0x23, 0x00, 0x96, 0x08, 0x00, 0x07, 0x00,
		0x00, 0x00, 0x00, 0x04, 0x02, 0x03, 0x52, 0x17,
		0x96, 0x08, 0x00, 0x04, 0x01, 0x08, 0x02, 0x04,
		0x01, 0x08, 0x02, 0x4e, 0x96, 0x05, 0x00, 0x07,
		0x01, 0x00, 0x00, 0x00, 0x47, 0x4f, 0x87, 0x01,
		0x00, 0x00, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x00,
		0x08, 0x03, 0x1c, 0x69, 0x96, 0x04, 0x00, 0x04,
		0x00, 0x08, 0x04, 0x4e, 0x87, 0x01, 0x00, 0x01,
		0x17, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x05,
		0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x03, 0x19,
		0x00, 0x1c, 0x00, 0x96, 0x09, 0x00, 0x07, 0x00,
		0x00, 0x00, 0x00, 0x04, 0x02, 0x08, 0x06, 0x52,
		0x17, 0x96, 0x09, 0x00, 0x07, 0x00, 0x00, 0x00,
		0x00, 0x04, 0x02, 0x08, 0x06, 0x52, 0x17, 0x4f,
		0x96, 0x0f, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00,
		0x02, 0x04, 0x01, 0x07, 0x03, 0x00, 0x00, 0x00,
		0x08, 0x07, 0x3d, 0x17, 0x00,
};

/*
 * class D extends C {
 * 		function D() {
 * 			super()
 * 			this.test +=1
 * 		}
 *
 * 		function inc2() {
 * 			super.inc()
 * 		}
 * }
 */
static const byte kDClass[] = {
		0x88, 0x35, 0x00, 0x08, 0x00, 0x44, 0x00, 0x5f,
		0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x00, 0x74,
		0x65, 0x73, 0x74, 0x00, 0x43, 0x00, 0x70, 0x72,
		0x6f, 0x74, 0x6f, 0x74, 0x79, 0x70, 0x65, 0x00,
		0x69, 0x6e, 0x63, 0x32, 0x00, 0x69, 0x6e, 0x63,
		0x00, 0x41, 0x53, 0x53, 0x65, 0x74, 0x50, 0x72,
		0x6f, 0x70, 0x46, 0x6c, 0x61, 0x67, 0x73, 0x00,
		0x96, 0x02, 0x00, 0x08, 0x00, 0x1c, 0x12, 0x12,
		0x9d, 0x02, 0x00, 0x89, 0x00, 0x96, 0x02, 0x00,
		0x08, 0x01, 0x1c, 0x96, 0x02, 0x00, 0x08, 0x00,
		0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x03, 0x19,
		0x00, 0x23, 0x00, 0x96, 0x08, 0x00, 0x07, 0x00,
		0x00, 0x00, 0x00, 0x04, 0x02, 0x03, 0x52, 0x17,
		0x96, 0x08, 0x00, 0x04, 0x01, 0x08, 0x02, 0x04,
		0x01, 0x08, 0x02, 0x4e, 0x96, 0x05, 0x00, 0x07,
		0x01, 0x00, 0x00, 0x00, 0x47, 0x4f, 0x87, 0x01,
		0x00, 0x00, 0x4f, 0x96, 0x04, 0x00, 0x04, 0x00,
		0x08, 0x03, 0x1c, 0x69, 0x96, 0x04, 0x00, 0x04,
		0x00, 0x08, 0x04, 0x4e, 0x87, 0x01, 0x00, 0x01,
		0x17, 0x96, 0x04, 0x00, 0x04, 0x01, 0x08, 0x05,
		0x8e, 0x08, 0x00, 0x00, 0x00, 0x00, 0x03, 0x19,
		0x00, 0x0e, 0x00, 0x96, 0x09, 0x00, 0x07, 0x00,
		0x00, 0x00, 0x00, 0x04, 0x02, 0x08, 0x06, 0x52,
		0x17, 0x4f, 0x96, 0x0f, 0x00, 0x07, 0x01, 0x00,
		0x00, 0x00, 0x02, 0x04, 0x01, 0x07, 0x03, 0x00,
		0x00, 0x00, 0x08, 0x07, 0x3d, 0x17, 0x00
};

GTEST_TEST(ActionScript, TestClass) {
	Common::MemoryReadStream *stream = new Common::MemoryReadStream(kTestClass);
	Aurora::ActionScript::ASBuffer asBuffer(stream);

	Aurora::ActionScript::AVM avm;
	asBuffer.run(avm);

	EXPECT_TRUE(avm.hasVariable("Test"));

	Aurora::ActionScript::Variable test = avm.getVariable("Test");
	EXPECT_TRUE(test.isObject());
	EXPECT_TRUE(test.asObject()->hasMember("prototype"));

	Aurora::ActionScript::Variable prototype = test.asObject()->getMember("prototype");
	EXPECT_TRUE(prototype.isObject());
	EXPECT_TRUE(prototype.asObject()->hasMember("inc"));
	EXPECT_TRUE(prototype.asObject()->getMember("inc").isFunction());
	EXPECT_TRUE(prototype.asObject()->hasMember("dec"));
	EXPECT_TRUE(prototype.asObject()->getMember("dec").isFunction());
	EXPECT_TRUE(prototype.asObject()->hasMember("getI"));
	EXPECT_TRUE(prototype.asObject()->getMember("getI").isFunction());

	Aurora::ActionScript::ObjectPtr obj = avm.createNewObject("Test").asObject();
	EXPECT_EQ(obj->getMember("i").asNumber(), 1);

	obj->call("dec", avm);
	EXPECT_EQ(obj->getMember("i").asNumber(), 0);

	obj->call("inc", avm);
	obj->call("inc", avm);
	EXPECT_EQ(obj->getMember("i").asNumber(), 2);

	EXPECT_EQ(obj->call("getI", avm).asNumber(), 2);

	delete stream;
}

GTEST_TEST(ActionScript, TestClass2) {
	Common::MemoryReadStream *streama = new Common::MemoryReadStream(kTestClass2a);
	Common::MemoryReadStream *streamb = new Common::MemoryReadStream(kTestClass2b);
	Aurora::ActionScript::ASBuffer asBufferA(streama);
	Aurora::ActionScript::ASBuffer asBufferB(streamb);

	Aurora::ActionScript::AVM avm;
	asBufferA.run(avm);
	asBufferB.run(avm);

	EXPECT_TRUE(avm.hasVariable("Test2_a"));
	EXPECT_TRUE(avm.hasVariable("Test2_b"));

	Aurora::ActionScript::Variable testa = avm.getVariable("Test2_a");
	Aurora::ActionScript::Variable testb = avm.getVariable("Test2_b");
	EXPECT_TRUE(testa.isObject());
	EXPECT_TRUE(testa.asObject()->hasMember("prototype"));
	EXPECT_TRUE(testb.isObject());
	EXPECT_TRUE(testb.asObject()->hasMember("prototype"));

	std::vector<Aurora::ActionScript::Variable> arguments(2);
	arguments[0] = true;
	arguments[1] = false;

	Aurora::ActionScript::Variable test = avm.createNewObject("Test2_b", arguments);
	EXPECT_TRUE(test.isObject());
	EXPECT_TRUE(test.asObject()->hasMember("a"));
	EXPECT_TRUE(test.asObject()->hasMember("b"));
	EXPECT_TRUE(test.asObject()->hasMember("And"));
	EXPECT_TRUE(test.asObject()->hasMember("Or"));
	EXPECT_TRUE(test.asObject()->hasMember("getA"));
	EXPECT_TRUE(test.asObject()->hasMember("getB"));

	EXPECT_EQ(test.asObject()->getMember("a").asBoolean(), true);
	EXPECT_EQ(test.asObject()->getMember("b").asBoolean(), false);

	EXPECT_EQ(test.asObject()->call("getA", avm).asBoolean(), true);
	EXPECT_EQ(test.asObject()->call("getB", avm).asBoolean(), false);

	test.asObject()->call("And", avm);
	EXPECT_FALSE(avm.getReturnValue().asBoolean());
	test.asObject()->call("Or", avm);
	EXPECT_TRUE(avm.getReturnValue().asBoolean());

	delete streama;
	delete streamb;
}

GTEST_TEST(ActionScript, ArrayTestClass) {
	Common::MemoryReadStream *stream = new Common::MemoryReadStream(kArrayTestClass);
	Aurora::ActionScript::ASBuffer asBuffer(stream);

	Aurora::ActionScript::AVM avm;
	asBuffer.run(avm);

	EXPECT_TRUE(avm.hasVariable("ArrayTest"));

	Aurora::ActionScript::Variable arrayTest = avm.getVariable("ArrayTest");
	EXPECT_TRUE(arrayTest.isObject());
	EXPECT_TRUE(arrayTest.asObject()->hasMember("prototype"));
	EXPECT_TRUE(arrayTest.asObject()->getMember("prototype").isObject());

	Aurora::ActionScript::Variable test = avm.createNewObject("ArrayTest");
	EXPECT_TRUE(test.isObject());
	EXPECT_TRUE(test.asObject()->hasMember("pushTwoValues"));
	EXPECT_TRUE(test.asObject()->hasMember("popValue"));
	EXPECT_TRUE(test.asObject()->hasMember("setFirstZero"));

	EXPECT_TRUE(test.asObject()->getMember("pushTwoValues").isFunction());
	EXPECT_TRUE(test.asObject()->getMember("popValue").isFunction());
	EXPECT_TRUE(test.asObject()->getMember("setFirstZero").isFunction());

	EXPECT_TRUE(test.asObject()->hasMember("arr"));
	EXPECT_TRUE(test.asObject()->getMember("arr").as<Aurora::ActionScript::Array>());

	Aurora::ActionScript::ArrayPtr array = test.asObject()->getMember("arr").as<Aurora::ActionScript::Array>();
	EXPECT_EQ(array->length(), 3);

	Aurora::ActionScript::Variable v = test.asObject()->call("popValue", avm);
	EXPECT_EQ(array->length(), 2);

	EXPECT_TRUE(v.isNumber());
	EXPECT_EQ(v.asNumber(), 3);

	v = test.asObject()->call("popValue", avm);
	EXPECT_EQ(array->length(), 1);
	EXPECT_EQ(array->getMember("length").asNumber(), 1);

	EXPECT_TRUE(v.isNumber());
	EXPECT_EQ(v.asNumber(), 2);

	std::vector<Aurora::ActionScript::Variable> arguments(2);
	arguments[0] = 10.0;
	arguments[1] = 11.0;
	test.asObject()->call("pushTwoValues", avm, arguments);

	EXPECT_EQ(array->length(), 3);
	EXPECT_EQ(array->getMember("length").asNumber(), 3);

	Aurora::ActionScript::Variable value1 = array->getMember(0u);
	Aurora::ActionScript::Variable value2 = array->getMember(1u);
	Aurora::ActionScript::Variable value3 = array->getMember(2u);

	EXPECT_EQ(value1.asNumber(), 1);
	EXPECT_EQ(value2.asNumber(), 10);
	EXPECT_EQ(value3.asNumber(), 11);

	test.asObject()->call("setFirstZero", avm);

	value1 = array->getMember(0u);
	value2 = array->getMember(1u);
	value3 = array->getMember(2u);

	EXPECT_EQ(value1.asNumber(), 1);
	EXPECT_EQ(value2.asNumber(), 10);
	EXPECT_EQ(value3.asNumber(), 11);

	array->setMember(2u, 46u);
	value1 = array->getMember(0u);
	value2 = array->getMember(1u);
	value3 = array->getMember(2u);

	EXPECT_EQ(value1.asNumber(), 1);
	EXPECT_EQ(value2.asNumber(), 10);
	EXPECT_EQ(value3.asNumber(), 46);

	delete stream;
}

GTEST_TEST(ActionScript, CascadingInheritance) {
	Common::MemoryReadStream *streama = new Common::MemoryReadStream(kAClass);
	Common::MemoryReadStream *streamb = new Common::MemoryReadStream(kBClass);
	Common::MemoryReadStream *streamc = new Common::MemoryReadStream(kCClass);
	Common::MemoryReadStream *streamd = new Common::MemoryReadStream(kDClass);
	Aurora::ActionScript::ASBuffer asBufferA(streama);
	Aurora::ActionScript::ASBuffer asBufferB(streamb);
	Aurora::ActionScript::ASBuffer asBufferC(streamc);
	Aurora::ActionScript::ASBuffer asBufferD(streamd);

	Aurora::ActionScript::AVM avm;

	avm.pushRegisters(255);
	asBufferA.run(avm);
	asBufferB.run(avm);
	asBufferC.run(avm);
	asBufferD.run(avm);
	avm.popRegisters(255);

	EXPECT_TRUE(avm.hasVariable("A"));
	EXPECT_TRUE(avm.hasVariable("B"));
	EXPECT_TRUE(avm.hasVariable("C"));
	EXPECT_TRUE(avm.hasVariable("D"));

	Aurora::ActionScript::Variable classD = avm.createNewObject("D");

	EXPECT_TRUE(classD.asObject()->hasMember("dec"));
	EXPECT_TRUE(classD.asObject()->hasMember("inc"));
	EXPECT_TRUE(classD.asObject()->hasMember("inc2"));

	EXPECT_EQ(classD.asObject()->getMember("test").asNumber(), 4);

	classD.asObject()->call("inc", avm);
	EXPECT_EQ(classD.asObject()->getMember("test").asNumber(), 5);

	classD.asObject()->call("inc2", avm);
	EXPECT_EQ(classD.asObject()->getMember("test").asNumber(), 6);

	classD.asObject()->call("dec", avm);
	EXPECT_EQ(classD.asObject()->getMember("test").asNumber(), 5);

	delete streama;
	delete streamb;
	delete streamc;
	delete streamd;
}
