/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Static data used for decoding WMV2 videos.
 */

/* Based on the WMV2 implementation in FFmpeg (<https://ffmpeg.org/)>,
 * which is released under the terms of version 2 or later of the GNU
 * Lesser General Public License.
 *
 * The original copyright notes in the file libavcodec/wmv2dec.c reads as follows:
 *
 * Copyright (c) 2002 The FFmpeg Project
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "src/common/util.h"

#include "src/video/codecs/wmv2data.h"

namespace Video {

// I-Frame macroblock block pattern, codes
static const uint32_t wmv2HuffmanIMBCodes[64] = {
	0x0001, 0x0017, 0x0009, 0x0005, 0x0006, 0x0047, 0x0020, 0x0010,
	0x0002, 0x007C, 0x003A, 0x001D, 0x0002, 0x00EC, 0x0077, 0x0000,
	0x0003, 0x00B7, 0x002C, 0x0013, 0x0001, 0x0168, 0x0046, 0x003F,
	0x001E, 0x0712, 0x00B5, 0x0042, 0x0022, 0x01C5, 0x011E, 0x0087,
	0x0006, 0x0003, 0x001E, 0x001C, 0x0012, 0x0388, 0x0044, 0x0070,
	0x001F, 0x023E, 0x0039, 0x008E, 0x0001, 0x01C6, 0x00B6, 0x0045,
	0x0014, 0x023F, 0x007D, 0x0018, 0x0007, 0x01C7, 0x0086, 0x0019,
	0x0015, 0x01DB, 0x0002, 0x0046, 0x000D, 0x0713, 0x01DA, 0x0169
};

// I-Frame macroblock block pattern, lengths
static const uint8_t wmv2HuffmanIMBLengths[64] = {
	 1,  6,  5,  5,  5,  9,  7,  7,
	 5,  9,  7,  7,  6,  9,  8,  8,
	 5,  9,  7,  7,  6, 10,  8,  8,
	 6, 13,  9,  8,  7, 11, 10,  9,
	 4,  9,  7,  6,  7, 12,  9,  9,
	 6, 11,  8,  9,  7, 11,  9,  9,
	 6, 11,  9,  9,  7, 11,  9,  9,
	 6, 10,  9,  9,  8, 13, 10, 10
};


// DCT DC coefficients, code 0 (low motion), luma, codes
static const uint32_t wmv2HuffmanDC0LumaCodes[120] = {
	0x00001, 0x00001, 0x00001, 0x00001, 0x00005, 0x00007, 0x00008, 0x0000C,
	0x00000, 0x00002, 0x00012, 0x0001A, 0x00003, 0x00007, 0x00027, 0x00037,
	0x00005, 0x0004C, 0x0006C, 0x0006D, 0x00008, 0x00019, 0x0009B, 0x0001B,
	0x0009A, 0x00013, 0x00034, 0x00035, 0x00061, 0x00048, 0x000C4, 0x0004A,
	0x000C6, 0x000C7, 0x00092, 0x0018B, 0x00093, 0x00183, 0x00182, 0x00096,
	0x00097, 0x00180, 0x00314, 0x00315, 0x00605, 0x00604, 0x00606, 0x00C0E,
	0x303CD, 0x303C9, 0x303C8, 0x303CA, 0x303CB, 0x303CC, 0x303CE, 0x303CF,
	0x303D0, 0x303D1, 0x303D2, 0x303D3, 0x303D4, 0x303D5, 0x303D6, 0x303D7,
	0x303D8, 0x303D9, 0x303DA, 0x303DB, 0x303DC, 0x303DD, 0x303DE, 0x303DF,
	0x303E0, 0x303E1, 0x303E2, 0x303E3, 0x303E4, 0x303E5, 0x303E6, 0x303E7,
	0x303E8, 0x303E9, 0x303EA, 0x303EB, 0x303EC, 0x303ED, 0x303EE, 0x303EF,
	0x303F0, 0x303F1, 0x303F2, 0x303F3, 0x303F4, 0x303F5, 0x303F6, 0x303F7,
	0x303F8, 0x303F9, 0x303FA, 0x303FB, 0x303FC, 0x303FD, 0x303FE, 0x303FF,
	0x60780, 0x60781, 0x60782, 0x60783, 0x60784, 0x60785, 0x60786, 0x60787,
	0x60788, 0x60789, 0x6078A, 0x6078B, 0x6078C, 0x6078D, 0x6078E, 0x6078F
};

// DCT DC coefficients, code 0 (low motion), luma, lengths
static const uint8_t wmv2HuffmanDC0LumaLengths[120] = {
	 1,  2,  4,  5,  5,  5,  6,  6,
	 7,  7,  7,  7,  8,  8,  8,  8,
	 9,  9,  9,  9, 10, 10, 10, 10,
	10, 11, 11, 11, 12, 13, 13, 13,
	13, 13, 14, 14, 14, 14, 14, 14,
	14, 14, 15, 15, 16, 16, 16, 17,
	23, 23, 23, 23, 23, 23, 23, 23,
	23, 23, 23, 23, 23, 23, 23, 23,
	23, 23, 23, 23, 23, 23, 23, 23,
	23, 23, 23, 23, 23, 23, 23, 23,
	23, 23, 23, 23, 23, 23, 23, 23,
	23, 23, 23, 23, 23, 23, 23, 23,
	23, 23, 23, 23, 23, 23, 23, 23,
	24, 24, 24, 24, 24, 24, 24, 24,
	24, 24, 24, 24, 24, 24, 24, 24
};

// DCT DC coefficients, code 0 (low motion), chroma, codes
static const uint32_t wmv2HuffmanDC0ChromaCodes[120] = {
	0x000000, 0x000001, 0x000005, 0x000009, 0x00000D, 0x000011, 0x00001D, 0x00001F,
	0x000021, 0x000031, 0x000038, 0x000033, 0x000039, 0x00003D, 0x000061, 0x000079,
	0x000080, 0x0000C8, 0x0000CA, 0x0000F0, 0x000081, 0x0000C0, 0x0000C9, 0x000107,
	0x000106, 0x000196, 0x000183, 0x0001E3, 0x0001E2, 0x00020A, 0x00020B, 0x000609,
	0x000412, 0x000413, 0x00060B, 0x000411, 0x00060A, 0x00065F, 0x000410, 0x00065D,
	0x00065E, 0x000CB8, 0x000C10, 0x000CB9, 0x001823, 0x003045, 0x006089, 0x00C110,
	0x304448, 0x304449, 0x30444A, 0x30444B, 0x30444C, 0x30444D, 0x30444E, 0x30444F,
	0x304450, 0x304451, 0x304452, 0x304453, 0x304454, 0x304455, 0x304456, 0x304457,
	0x304458, 0x304459, 0x30445A, 0x30445B, 0x30445C, 0x30445D, 0x30445E, 0x30445F,
	0x304460, 0x304461, 0x304462, 0x304463, 0x304464, 0x304465, 0x304466, 0x304467,
	0x304468, 0x304469, 0x30446A, 0x30446B, 0x30446C, 0x30446D, 0x30446E, 0x30446F,
	0x304470, 0x304471, 0x304472, 0x304473, 0x304474, 0x304475, 0x304476, 0x304477,
	0x304478, 0x304479, 0x30447A, 0x30447B, 0x30447C, 0x30447D, 0x30447E, 0x30447F,
	0x608880, 0x608881, 0x608882, 0x608883, 0x608884, 0x608885, 0x608886, 0x608887,
	0x608888, 0x608889, 0x60888A, 0x60888B, 0x60888C, 0x60888D, 0x60888E, 0x60888F
};

// DCT DC coefficients, code 0 (low motion), chroma, lengths
static const uint8_t wmv2HuffmanDC0ChromaLengths[120] = {
	 2,  2,  3,  4,  4,  5,  5,  5,
	 6,  6,  6,  6,  6,  6,  7,  7,
	 8,  8,  8,  8,  8,  8,  8,  9,
	 9,  9,  9,  9,  9, 10, 10, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 12, 12, 12, 13, 14, 15, 16,
	22, 22, 22, 22, 22, 22, 22, 22,
	22, 22, 22, 22, 22, 22, 22, 22,
	22, 22, 22, 22, 22, 22, 22, 22,
	22, 22, 22, 22, 22, 22, 22, 22,
	22, 22, 22, 22, 22, 22, 22, 22,
	22, 22, 22, 22, 22, 22, 22, 22,
	22, 22, 22, 22, 22, 22, 22, 22,
	23, 23, 23, 23, 23, 23, 23, 23,
	23, 23, 23, 23, 23, 23, 23, 23
};

// DCT DC coefficients, code 1 (high motion), luma, codes
static const uint32_t wmv2HuffmanDC1LumaCodes[120] = {
	0x000002, 0x000003, 0x000003, 0x000002, 0x000005, 0x000001, 0x000003, 0x000008,
	0x000000, 0x000005, 0x00000D, 0x00000F, 0x000013, 0x000008, 0x000018, 0x00001C,
	0x000024, 0x000004, 0x000006, 0x000012, 0x000032, 0x00003B, 0x00004A, 0x00004B,
	0x00000B, 0x000026, 0x000027, 0x000066, 0x000074, 0x000075, 0x000014, 0x00001C,
	0x00001F, 0x00001D, 0x00002B, 0x00003D, 0x00019D, 0x00019F, 0x000054, 0x000339,
	0x000338, 0x00033D, 0x0000AB, 0x0000F1, 0x000678, 0x0000F2, 0x0001E0, 0x0001E1,
	0x000154, 0x000CF2, 0x0003CC, 0x0002AB, 0x0019E7, 0x0003CE, 0x0019E6, 0x000554,
	0x00079F, 0x000555, 0x000F3D, 0x000F37, 0x000F3C, 0x000F35, 0x001E6D, 0x001E68,
	0x003CD8, 0x003CD3, 0x003CD9, 0x0079A4, 0x0F34BA, 0x0F34B4, 0x0F34B5, 0x0F34B6,
	0x0F34B7, 0x0F34B8, 0x0F34B9, 0x0F34BB, 0x0F34BC, 0x0F34BD, 0x0F34BE, 0x0F34BF,
	0x1E6940, 0x1E6941, 0x1E6942, 0x1E6943, 0x1E6944, 0x1E6945, 0x1E6946, 0x1E6947,
	0x1E6948, 0x1E6949, 0x1E694A, 0x1E694B, 0x1E694C, 0x1E694D, 0x1E694E, 0x1E694F,
	0x1E6950, 0x1E6951, 0x1E6952, 0x1E6953, 0x1E6954, 0x1E6955, 0x1E6956, 0x1E6957,
	0x1E6958, 0x1E6959, 0x1E695A, 0x1E695B, 0x1E695C, 0x1E695D, 0x1E695E, 0x1E695F,
	0x1E6960, 0x1E6961, 0x1E6962, 0x1E6963, 0x1E6964, 0x1E6965, 0x1E6966, 0x1E6967
};

// DCT DC coefficients, code 1 (high motion), luma, lengths
static const uint8_t wmv2HuffmanDC1LumaLengths[120] = {
	 2,  2,  3,  4,  4,  5,  5,  5,
	 6,  6,  6,  6,  6,  7,  7,  7,
	 7,  8,  8,  8,  8,  8,  8,  8,
	 9,  9,  9,  9,  9,  9, 10, 10,
	10, 10, 11, 11, 11, 11, 12, 12,
	12, 12, 13, 13, 13, 13, 14, 14,
	14, 14, 15, 15, 15, 15, 15, 16,
	16, 16, 17, 17, 17, 17, 18, 18,
	19, 19, 19, 20, 25, 25, 25, 25,
	25, 25, 25, 25, 25, 25, 25, 25,
	26, 26, 26, 26, 26, 26, 26, 26,
	26, 26, 26, 26, 26, 26, 26, 26,
	26, 26, 26, 26, 26, 26, 26, 26,
	26, 26, 26, 26, 26, 26, 26, 26,
	26, 26, 26, 26, 26, 26, 26, 26
};

// DCT DC coefficients, code 1 (high motion), chroma, codes
static const uint32_t wmv2HuffmanDC1ChromaCodes[120] = {
	0x0000000, 0x0000001, 0x0000004, 0x0000007, 0x000000B, 0x000000D, 0x0000015, 0x0000028,
	0x0000030, 0x0000032, 0x0000052, 0x0000062, 0x0000066, 0x00000A6, 0x00000C6, 0x00000CF,
	0x000014F, 0x000018E, 0x000019C, 0x000029D, 0x000033A, 0x0000538, 0x000063C, 0x000063E,
	0x000063F, 0x0000676, 0x0000A73, 0x0000C7A, 0x0000CEF, 0x00014E5, 0x00019DD, 0x00029C8,
	0x00029C9, 0x00063DD, 0x00033B8, 0x00033B9, 0x000C7B6, 0x00063D8, 0x00063DF, 0x000C7B3,
	0x000C7B4, 0x000C7B5, 0x00063DE, 0x000C7B7, 0x000C7B8, 0x000C7B9, 0x0018F65, 0x0031EC8,
	0x0C7B248, 0x0C7B249, 0x0C7B24A, 0x0C7B24B, 0x0C7B24C, 0x0C7B24D, 0x0C7B24E, 0x0C7B24F,
	0x0C7B250, 0x0C7B251, 0x0C7B252, 0x0C7B253, 0x0C7B254, 0x0C7B255, 0x0C7B256, 0x0C7B257,
	0x0C7B258, 0x0C7B259, 0x0C7B25A, 0x0C7B25B, 0x0C7B25C, 0x0C7B25D, 0x0C7B25E, 0x0C7B25F,
	0x0C7B260, 0x0C7B261, 0x0C7B262, 0x0C7B263, 0x0C7B264, 0x0C7B265, 0x0C7B266, 0x0C7B267,
	0x0C7B268, 0x0C7B269, 0x0C7B26A, 0x0C7B26B, 0x0C7B26C, 0x0C7B26D, 0x0C7B26E, 0x0C7B26F,
	0x0C7B270, 0x0C7B271, 0x0C7B272, 0x0C7B273, 0x0C7B274, 0x0C7B275, 0x0C7B276, 0x0C7B277,
	0x0C7B278, 0x0C7B279, 0x0C7B27A, 0x0C7B27B, 0x0C7B27C, 0x0C7B27D, 0x0C7B27E, 0x0C7B27F,
	0x18F6480, 0x18F6481, 0x18F6482, 0x18F6483, 0x18F6484, 0x18F6485, 0x18F6486, 0x18F6487,
	0x18F6488, 0x18F6489, 0x18F648A, 0x18F648B, 0x18F648C, 0x18F648D, 0x18F648E, 0x18F648F
};

// DCT DC coefficients, code 1 (high motion), chroma, lengths
static const uint8_t wmv2HuffmanDC1ChromaLengths[120] = {
	 2,  2,  3,  3,  4,  4,  5,  6,
	 6,  6,  7,  7,  7,  8,  8,  8,
	 9,  9,  9, 10, 10, 11, 11, 11,
	11, 11, 12, 12, 12, 13, 13, 14,
	14, 15, 14, 14, 16, 15, 15, 16,
	16, 16, 15, 16, 16, 16, 17, 18,
	24, 24, 24, 24, 24, 24, 24, 24,
	24, 24, 24, 24, 24, 24, 24, 24,
	24, 24, 24, 24, 24, 24, 24, 24,
	24, 24, 24, 24, 24, 24, 24, 24,
	24, 24, 24, 24, 24, 24, 24, 24,
	24, 24, 24, 24, 24, 24, 24, 24,
	24, 24, 24, 24, 24, 24, 24, 24,
	25, 25, 25, 25, 25, 25, 25, 25,
	25, 25, 25, 25, 25, 25, 25, 25
};


// Motion vector code 0 (low motion), codes
static const uint32_t wmv2HuffmanMV0Codes[1100] = {
	0x0001, 0x0003, 0x0005, 0x0007, 0x0003, 0x0008, 0x000C, 0x0001,
	0x0002, 0x001B, 0x0006, 0x000B, 0x0015, 0x0002, 0x000E, 0x000F,
	0x0014, 0x0020, 0x0022, 0x0025, 0x0027, 0x0029, 0x002D, 0x004B,
	0x004D, 0x0003, 0x0022, 0x0023, 0x0025, 0x0027, 0x0042, 0x0048,
	0x0049, 0x0050, 0x005C, 0x0091, 0x009F, 0x000E, 0x0043, 0x004C,
	0x0054, 0x0056, 0x008C, 0x0098, 0x009A, 0x009B, 0x00B1, 0x00B2,
	0x0120, 0x0121, 0x0126, 0x0133, 0x0139, 0x01A1, 0x01A4, 0x01A5,
	0x01A6, 0x01A7, 0x01AE, 0x01AF, 0x000B, 0x0019, 0x0085, 0x0090,
	0x009B, 0x00AA, 0x00AF, 0x010C, 0x010E, 0x011C, 0x011E, 0x0133,
	0x0144, 0x0160, 0x0174, 0x0175, 0x0177, 0x0178, 0x0249, 0x024B,
	0x0252, 0x0261, 0x0265, 0x0270, 0x0352, 0x0353, 0x0355, 0x0359,
	0x0010, 0x0011, 0x0013, 0x0034, 0x0035, 0x0036, 0x0037, 0x003D,
	0x003E, 0x0109, 0x0126, 0x0156, 0x021A, 0x021E, 0x023A, 0x023E,
	0x028E, 0x028F, 0x02CF, 0x0491, 0x0494, 0x049F, 0x04A0, 0x04A3,
	0x04A6, 0x04A7, 0x04AD, 0x04AE, 0x04C0, 0x04C4, 0x04C6, 0x04C8,
	0x04C9, 0x04F5, 0x04F6, 0x04F7, 0x0680, 0x0682, 0x0683, 0x0688,
	0x0689, 0x068D, 0x068E, 0x068F, 0x06A2, 0x06A3, 0x06A9, 0x06B0,
	0x06B1, 0x06B4, 0x06B5, 0x0024, 0x0060, 0x0063, 0x0078, 0x0079,
	0x0211, 0x0244, 0x0245, 0x0247, 0x0248, 0x0249, 0x024A, 0x024B,
	0x026B, 0x02AF, 0x02B8, 0x02BB, 0x0436, 0x0476, 0x0477, 0x047E,
	0x04C8, 0x04C9, 0x04CA, 0x0514, 0x0586, 0x0587, 0x0598, 0x059D,
	0x05D9, 0x05DA, 0x0920, 0x0921, 0x093B, 0x093C, 0x093D, 0x0942,
	0x0943, 0x0944, 0x0945, 0x0959, 0x095E, 0x095F, 0x0982, 0x0983,
	0x098E, 0x098F, 0x09C4, 0x09E7, 0x09E8, 0x09E9, 0x0D02, 0x0D17,
	0x0D18, 0x0D19, 0x0D41, 0x0D42, 0x0D43, 0x0D50, 0x0D5F, 0x0D6D,
	0x0D6E, 0x0D6F, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x041E, 0x041F, 0x0420, 0x0421,
	0x048C, 0x048D, 0x04D3, 0x04D4, 0x04D5, 0x055C, 0x055D, 0x0572,
	0x0573, 0x0574, 0x0575, 0x08DE, 0x08DF, 0x08FE, 0x08FF, 0x0996,
	0x0A36, 0x0A37, 0x0B08, 0x0B09, 0x0B0A, 0x0B0B, 0x0B32, 0x0B33,
	0x0B34, 0x0B35, 0x0B36, 0x0B37, 0x0B38, 0x0B39, 0x0BB0, 0x0BF7,
	0x0BF8, 0x0BF9, 0x0BFA, 0x0BFB, 0x0BFC, 0x0BFD, 0x0BFE, 0x0BFF,
	0x1254, 0x1255, 0x1256, 0x1257, 0x1270, 0x1271, 0x1272, 0x1273,
	0x1274, 0x1275, 0x12AB, 0x12AC, 0x12AD, 0x12AE, 0x12AF, 0x12B0,
	0x12B1, 0x1315, 0x1316, 0x1317, 0x13BF, 0x13C0, 0x13C1, 0x13C2,
	0x13C3, 0x13C4, 0x13C5, 0x13C6, 0x13C7, 0x13C8, 0x13C9, 0x13CA,
	0x13CB, 0x13CC, 0x13CD, 0x1A06, 0x1A07, 0x1A28, 0x1A29, 0x1A2A,
	0x1A2B, 0x1A2C, 0x1A2D, 0x1A80, 0x1ABB, 0x1ABC, 0x1ABD, 0x1AD8,
	0x1AD9, 0x0094, 0x0095, 0x0096, 0x0097, 0x00A0, 0x00A1, 0x00A2,
	0x00A3, 0x0831, 0x0832, 0x0833, 0x0834, 0x0835, 0x0836, 0x0837,
	0x0838, 0x0839, 0x083A, 0x083B, 0x0939, 0x093A, 0x093B, 0x093C,
	0x093D, 0x093E, 0x093F, 0x09A0, 0x09A1, 0x09A2, 0x09A3, 0x09A4,
	0x09A5, 0x11AC, 0x11AD, 0x11AE, 0x11AF, 0x11B0, 0x11B1, 0x11B2,
	0x11B3, 0x11B4, 0x11B5, 0x11B6, 0x11B7, 0x11B8, 0x11B9, 0x11BA,
	0x11BB, 0x132F, 0x1454, 0x1455, 0x1456, 0x1457, 0x1458, 0x1459,
	0x145A, 0x145B, 0x145C, 0x145D, 0x145E, 0x145F, 0x1460, 0x1461,
	0x1462, 0x1463, 0x1464, 0x1465, 0x1466, 0x1467, 0x1468, 0x1469,
	0x146A, 0x146B, 0x17DE, 0x17DF, 0x17E0, 0x17E1, 0x17E2, 0x17E3,
	0x17E4, 0x17E5, 0x17E6, 0x17E7, 0x17E8, 0x17E9, 0x17EA, 0x17EB,
	0x17EC, 0x17ED, 0x2540, 0x2541, 0x2542, 0x2543, 0x2544, 0x2545,
	0x2546, 0x2547, 0x2548, 0x2549, 0x254A, 0x254B, 0x254C, 0x254D,
	0x254E, 0x254F, 0x2550, 0x2551, 0x2552, 0x2553, 0x2554, 0x2555,
	0x2628, 0x2766, 0x2767, 0x2768, 0x2769, 0x276A, 0x276B, 0x276C,
	0x276D, 0x276E, 0x276F, 0x2770, 0x2771, 0x2772, 0x2773, 0x2774,
	0x2775, 0x2776, 0x2777, 0x2778, 0x2779, 0x277A, 0x277B, 0x277C,
	0x277D, 0x3503, 0x3544, 0x3545, 0x3546, 0x3547, 0x3560, 0x3561,
	0x3562, 0x3563, 0x3564, 0x3565, 0x3566, 0x3567, 0x3568, 0x3569,
	0x356A, 0x356B, 0x356C, 0x356D, 0x356E, 0x356F, 0x3570, 0x3571,
	0x3572, 0x3573, 0x3574, 0x3575, 0x03F0, 0x103D, 0x103E, 0x103F,
	0x1040, 0x1041, 0x1042, 0x1043, 0x1044, 0x1045, 0x1046, 0x1047,
	0x1048, 0x1049, 0x104A, 0x104B, 0x104C, 0x104D, 0x104E, 0x104F,
	0x1050, 0x1051, 0x1052, 0x1053, 0x1054, 0x1055, 0x1056, 0x1057,
	0x1058, 0x1059, 0x105A, 0x105B, 0x105C, 0x105D, 0x105E, 0x105F,
	0x1060, 0x1061, 0x1270, 0x1271, 0x21B8, 0x21B9, 0x21BA, 0x21BB,
	0x21BC, 0x21BD, 0x21BE, 0x21BF, 0x21F0, 0x21F1, 0x21F2, 0x21F3,
	0x21F4, 0x21F5, 0x21F6, 0x21F7, 0x21F8, 0x21F9, 0x21FA, 0x21FB,
	0x21FC, 0x21FD, 0x21FE, 0x21FF, 0x2340, 0x2341, 0x2342, 0x2343,
	0x2344, 0x2345, 0x2346, 0x2347, 0x2348, 0x2349, 0x234A, 0x234B,
	0x234C, 0x234D, 0x234E, 0x234F, 0x2350, 0x2351, 0x2352, 0x2353,
	0x2354, 0x2355, 0x2356, 0x2357, 0x265C, 0x2F88, 0x2F89, 0x2F8A,
	0x2F8B, 0x2F8C, 0x2F8D, 0x2F8E, 0x2F8F, 0x2F90, 0x2F91, 0x2F92,
	0x2F93, 0x2F94, 0x2F95, 0x2F96, 0x2F97, 0x2F98, 0x2F99, 0x2F9A,
	0x2F9B, 0x2F9C, 0x2F9D, 0x2F9E, 0x2F9F, 0x2FA0, 0x2FA1, 0x2FA2,
	0x2FA3, 0x2FA4, 0x2FA5, 0x2FA6, 0x2FA7, 0x2FA8, 0x2FA9, 0x2FAA,
	0x2FAB, 0x2FAC, 0x2FAD, 0x2FAE, 0x2FAF, 0x2FB0, 0x2FB1, 0x2FB2,
	0x2FB3, 0x2FB4, 0x2FB5, 0x2FB6, 0x2FB7, 0x2FB8, 0x2FB9, 0x2FBA,
	0x2FBB, 0x4C52, 0x4C53, 0x4E28, 0x4E29, 0x4E2A, 0x4E2B, 0x4E2C,
	0x4E2D, 0x4E2E, 0x4E2F, 0x4E30, 0x4E31, 0x4E32, 0x4E33, 0x4E34,
	0x4E35, 0x4E36, 0x4E37, 0x4E38, 0x4E39, 0x4E3A, 0x4E3B, 0x4E3C,
	0x4E3D, 0x4E3E, 0x4E3F, 0x4E80, 0x4E81, 0x4E82, 0x4E83, 0x4E84,
	0x4E85, 0x4E86, 0x4E87, 0x4E88, 0x4E89, 0x4E8A, 0x4E8B, 0x4E8C,
	0x4E8D, 0x4E8E, 0x4E8F, 0x4E90, 0x4E91, 0x4E92, 0x4E93, 0x4E94,
	0x4E95, 0x4E96, 0x4E97, 0x4E98, 0x4E99, 0x4E9A, 0x4E9B, 0x4E9C,
	0x4E9D, 0x4E9E, 0x4E9F, 0x4EA0, 0x4EA1, 0x4EA2, 0x4EA3, 0x4EA4,
	0x4EA5, 0x4EA6, 0x4EA7, 0x4EA8, 0x4EA9, 0x4EAA, 0x4EAB, 0x4EAC,
	0x4EAD, 0x4EAE, 0x4EAF, 0x4EB0, 0x4EB1, 0x4EB2, 0x4EB3, 0x4EB4,
	0x4EB5, 0x4EB6, 0x4EB7, 0x4EB8, 0x4EB9, 0x4EBA, 0x4EBB, 0x4EBC,
	0x4EBD, 0x4EBE, 0x4EBF, 0x4EC0, 0x4EC1, 0x4EC2, 0x4EC3, 0x4EC4,
	0x4EC5, 0x4EC6, 0x4EC7, 0x4EC8, 0x4EC9, 0x4ECA, 0x4ECB, 0x6A04,
	0x6A05, 0x07E2, 0x07E3, 0x07E4, 0x07E5, 0x07E6, 0x07E7, 0x07E8,
	0x07E9, 0x07EA, 0x07EB, 0x07EC, 0x07ED, 0x07EE, 0x07EF, 0x07F0,
	0x07F1, 0x07F2, 0x07F3, 0x07F4, 0x07F5, 0x07F6, 0x07F7, 0x07F8,
	0x07F9, 0x07FA, 0x07FB, 0x07FC, 0x07FD, 0x07FE, 0x07FF, 0x2000,
	0x2001, 0x2002, 0x2003, 0x2004, 0x2005, 0x2006, 0x2007, 0x2008,
	0x2009, 0x200A, 0x200B, 0x200C, 0x200D, 0x200E, 0x200F, 0x2010,
	0x2011, 0x2012, 0x2013, 0x2014, 0x2015, 0x2016, 0x2017, 0x2018,
	0x2019, 0x201A, 0x201B, 0x201C, 0x201D, 0x201E, 0x201F, 0x2020,
	0x2021, 0x2022, 0x2023, 0x2024, 0x2025, 0x2026, 0x2027, 0x2028,
	0x2029, 0x202A, 0x202B, 0x202C, 0x202D, 0x202E, 0x202F, 0x2030,
	0x2031, 0x2032, 0x2033, 0x2034, 0x2035, 0x2036, 0x2037, 0x2038,
	0x2039, 0x203A, 0x203B, 0x203C, 0x203D, 0x203E, 0x203F, 0x2040,
	0x2041, 0x2042, 0x2043, 0x2044, 0x2045, 0x2046, 0x2047, 0x2048,
	0x2049, 0x204A, 0x204B, 0x204C, 0x204D, 0x204E, 0x204F, 0x2050,
	0x2051, 0x2052, 0x2053, 0x2054, 0x2055, 0x2056, 0x2057, 0x2058,
	0x2059, 0x205A, 0x205B, 0x205C, 0x205D, 0x205E, 0x205F, 0x2060,
	0x2061, 0x2062, 0x2063, 0x2064, 0x2065, 0x2066, 0x2067, 0x2068,
	0x2069, 0x206A, 0x206B, 0x206C, 0x206D, 0x206E, 0x206F, 0x2070,
	0x2071, 0x2072, 0x2073, 0x2074, 0x2075, 0x2076, 0x2077, 0x2078,
	0x2079, 0x4CBA, 0x4CBB, 0x5D88, 0x5D89, 0x5D8A, 0x5D8B, 0x5D8C,
	0x5D8D, 0x5D8E, 0x5D8F, 0x5DB0, 0x5DB1, 0x5DB2, 0x5DB3, 0x5DB4,
	0x5DB5, 0x5DB6, 0x5DB7, 0x5DB8, 0x5DB9, 0x5DBA, 0x5DBB, 0x5DBC,
	0x5DBD, 0x5DBE, 0x5DBF, 0x5E40, 0x5E41, 0x5E42, 0x5E43, 0x5E44,
	0x5E45, 0x5E46, 0x5E47, 0x5E48, 0x5E49, 0x5E4A, 0x5E4B, 0x5E4C,
	0x5E4D, 0x5E4E, 0x5E4F, 0x5E50, 0x5E51, 0x5E52, 0x5E53, 0x5E54,
	0x5E55, 0x5E56, 0x5E57, 0x5E58, 0x5E59, 0x5E5A, 0x5E5B, 0x5E5C,
	0x5E5D, 0x5E5E, 0x5E5F, 0x5E60, 0x5E61, 0x5E62, 0x5E63, 0x5E64,
	0x5E65, 0x5E66, 0x5E67, 0x5E68, 0x5E69, 0x5E6A, 0x5E6B, 0x5E6C,
	0x5E6D, 0x5E6E, 0x5E6F, 0x5E70, 0x5E71, 0x5E72, 0x5E73, 0x5E74,
	0x5E75, 0x5E76, 0x5E77, 0x5E78, 0x5E79, 0x5E7A, 0x5E7B, 0x5E7C,
	0x5E7D, 0x5E7E, 0x5E7F, 0x5E80, 0x5E81, 0x5E82, 0x5E83, 0x5E84,
	0x5E85, 0x5E86, 0x5E87, 0x5E88, 0x5E89, 0x5E8A, 0x5E8B, 0x5E8C,
	0x5E8D, 0x5E8E, 0x5E8F, 0x5E90, 0x5E91, 0x5E92, 0x5E93, 0x5E94,
	0x5E95, 0x5E96, 0x5E97, 0x5E98, 0x5E99, 0x5E9A, 0x5E9B, 0x5E9C,
	0x5E9D, 0x5E9E, 0x5E9F, 0x5EA0, 0x5EA1, 0x5EA2, 0x5EA3, 0x5EA4,
	0x5EA5, 0x5EA6, 0x5EA7, 0x5EA8, 0x5EA9, 0x5EAA, 0x5EAB, 0x5EAC,
	0x5EAD, 0x5EAE, 0x5EAF, 0x5EB0, 0x5EB1, 0x5EB2, 0x5EB3, 0x5EB4,
	0x5EB5, 0x5EB6, 0x5EB7, 0x5EB8, 0x5EB9, 0x5EBA, 0x5EBB, 0x5EBC,
	0x5EBD, 0x5EBE, 0x5EBF, 0x5EC0, 0x5EC1, 0x5EC2, 0x5EC3, 0x5EC4,
	0x5EC5, 0x5EC6, 0x5EC7, 0x5EC8, 0x5EC9, 0x5ECA, 0x5ECB, 0x5ECC,
	0x5ECD, 0x5ECE, 0x5ECF, 0x5ED0, 0x5ED1, 0x5ED2, 0x5ED3, 0x5ED4,
	0x5ED5, 0x5ED6, 0x5ED7, 0x5ED8, 0x5ED9, 0x5EDA, 0x5EDB, 0x5EDC,
	0x5EDD, 0x5EDE, 0x5EDF, 0x5EE0, 0x5EE1, 0x5EE2, 0x5EE3, 0x5EE4,
	0x5EE5, 0x5EE6, 0x5EE7, 0x5EE8, 0x5EE9, 0x5EEA, 0x5EEB, 0x5EEC,
	0x5EED, 0x5EEE, 0x5EEF, 0x5EF0, 0x5EF1, 0x5EF2, 0x5EF3, 0x5EF4,
	0x5EF5, 0x5EF6, 0x5EF7, 0x5EF8, 0x5EF9, 0x5EFA, 0x5EFB, 0x5EFC,
	0x5EFD, 0x5EFE, 0x5EFF, 0x5F00, 0x5F01, 0x5F02, 0x5F03, 0x5F04,
	0x5F05, 0x5F06, 0x5F07, 0x5F08, 0x5F09, 0x5F0A, 0x5F0B, 0x5F0C,
	0x5F0D, 0x5F0E, 0x5F0F, 0x0000
};

// Motion vector code 0 (low motion), lengths
static const uint8_t wmv2HuffmanMV0Lengths[1100] = {
	 1,  4,  4,  4,  5,  5,  5,  6,
	 6,  6,  7,  7,  7,  8,  8,  8,
	 8,  8,  8,  8,  8,  8,  8,  8,
	 8,  9,  9,  9,  9,  9,  9,  9,
	 9,  9,  9,  9,  9, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 16, 16, 16, 16, 16, 16, 16,
	16, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17, 17,
	17, 17, 17,  8
};

// Motion vector code 0 (low motion), x difference
static const uint8_t wmv2MV0DiffX[1099] = {
	32, 32, 31, 32, 33, 31, 33, 31,
	33, 32, 34, 32, 30, 32, 31, 34,
	35, 32, 34, 33, 29, 33, 30, 30,
	31, 31, 35, 29, 33, 35, 33, 34,
	31, 29, 30, 34, 30, 36, 28, 32,
	34, 37, 30, 27, 32, 25, 39, 32,
	34, 32, 35, 35, 35, 31, 35, 29,
	32, 29, 30, 29, 37, 27, 36, 38,
	37, 33, 32, 31, 29, 31, 28, 36,
	33, 30, 34, 33, 33, 28, 27, 25,
	31, 26, 39, 32, 32, 31, 33, 39,
	31, 38, 28, 36, 21, 23, 43, 36,
	34, 41, 30, 25, 28, 31, 30, 34,
	38, 35, 61, 34, 28, 30, 37, 37,
	35, 27, 36,  3, 59, 38, 37, 32,
	31, 29, 26, 33, 37, 33, 27, 27,
	35, 34, 34, 40, 42, 33, 32, 29,
	 4,  5, 28, 24, 25, 35, 39, 38,
	32, 23, 27, 32, 30, 35, 26, 34,
	60, 36, 29, 22, 26, 41,  7, 30,
	38, 30, 36, 29, 30, 41, 26, 25,
	32, 34, 24, 39,  1, 25, 39, 32,
	28, 29, 32, 38, 26, 36, 28, 63,
	28, 39, 23, 21, 26, 35, 31, 35,
	57, 31, 29, 29, 28, 30, 27, 35,
	 2, 38, 40, 34, 37, 29, 38, 43,
	26, 32, 33, 42, 24, 40, 28, 32,
	32, 32, 36, 32, 43, 25, 21, 31,
	30, 31, 41, 29, 33, 37, 26, 37,
	27, 59, 23, 33, 35, 31, 31, 37,
	38, 39, 32, 23, 32, 27, 37, 36,
	31, 40, 25, 27, 38, 31, 36, 28,
	31, 36, 25, 45,  3, 34, 38, 39,
	40, 38, 30, 32, 19, 24, 25, 26,
	45, 20, 24, 33, 33, 31, 41, 34,
	39, 47, 40, 58, 59, 41, 33,  3,
	17, 61, 42, 30, 26, 29, 36, 61,
	33, 37, 62, 28, 25, 38, 25, 38,
	17, 23, 34, 33, 21, 33, 49, 27,
	32, 23, 27, 22, 24, 22, 39, 43,
	27, 37,  6, 42, 47, 26, 30, 31,
	41, 39, 33, 22, 45, 36, 32, 45,
	19, 22, 30,  5,  5, 17, 29, 22,
	31, 31, 43, 37, 27, 32, 32, 32,
	33, 34, 43, 35, 29, 26, 22, 32,
	19, 32, 25, 31, 41, 49, 28, 34,
	28, 39, 34, 19, 37, 38, 29, 21,
	36, 42, 24, 48, 16, 28, 49, 22,
	34, 31, 38, 39, 44, 11, 35, 30,
	33, 33, 23, 28, 33, 46, 15, 13,
	24, 41, 24, 34, 34, 30, 26, 24,
	14, 60, 21, 29, 39, 23, 35, 37,
	63, 45, 33, 34, 47, 41, 22, 42,
	35, 35, 23, 32, 35, 43, 32,  7,
	31, 41, 20, 31, 16, 13, 63, 25,
	30, 32, 35, 30, 30, 31, 42, 47,
	39, 38, 40, 40, 51, 55, 56, 18,
	21, 39, 39, 33, 17, 41, 23, 24,
	43, 25, 31, 20, 19, 45,  1, 34,
	31, 22, 35, 15, 46, 46, 35, 31,
	28, 29, 29, 23, 41, 27, 14, 53,
	53, 27, 24, 32, 57, 32, 17, 42,
	37, 29, 33,  1, 25, 32, 32, 63,
	26, 40, 44, 36, 31, 39, 20, 20,
	44, 23, 33, 34, 35, 33, 33, 28,
	41, 23, 41, 41, 29, 25, 26, 49,
	29, 24, 37, 49, 50, 51, 51, 26,
	39, 25, 26, 15, 39, 18, 42, 17,
	 4, 31, 32, 32, 60,  1, 42, 32,
	 0, 12, 19, 35, 21, 41, 17, 26,
	20, 45, 46, 32, 37, 22, 47, 29,
	31, 27, 29, 30, 21, 33, 35, 18,
	25, 33, 50, 51, 42,  2, 15, 51,
	53, 33, 25, 29, 55, 37, 38, 33,
	38, 59, 38, 33, 39, 13, 32, 40,
	61, 61, 32,  9, 44,  3, 31, 29,
	25, 31, 27, 23,  9, 25,  9, 29,
	20, 30, 30, 42, 18, 28, 25, 28,
	28, 21, 29, 43, 29, 43, 26, 44,
	44, 21, 38, 21, 24, 45, 45, 35,
	39, 22, 35, 36, 34, 34, 45, 34,
	29, 31, 46, 25, 46, 16, 17, 31,
	20, 32, 47, 47, 47, 32, 49, 49,
	49, 31,  1, 27, 28, 39, 39, 21,
	36, 23, 51,  2, 40, 51, 32, 53,
	24, 30, 24, 30, 21, 40, 57, 57,
	31, 41, 58, 32, 12,  4, 32, 34,
	59, 31, 32, 13,  9, 35, 26, 35,
	37, 61, 37, 63, 26, 29, 41, 38,
	23, 20, 41, 26, 41, 42, 42, 42,
	26, 26, 26, 26,  1, 26, 37, 37,
	37, 23, 34, 42, 27, 43, 34, 27,
	31, 24, 33, 16,  3, 31, 24, 33,
	24,  4, 44, 44, 11, 44, 31, 13,
	13, 44, 45, 13, 25, 22, 38, 26,
	38, 38, 39, 32, 30, 39, 30, 22,
	32, 26, 30, 47, 47, 47, 19, 47,
	30, 31, 35,  8, 23, 47, 47, 27,
	35, 47, 31, 48, 35, 19, 36, 49,
	49, 33, 31, 39, 27, 39, 49, 49,
	50, 50, 50, 39, 31, 51, 51, 39,
	28, 33, 33, 21, 40, 31, 52, 53,
	40, 53,  9, 33, 31, 53, 54, 54,
	54, 55, 55, 34, 15, 56, 25, 56,
	21, 21, 40, 40, 25, 40, 58, 36,
	 5, 41, 41, 12, 60, 41, 41, 37,
	22, 61, 18, 29, 29, 30, 61, 30,
	61, 62, 62, 30, 30, 63, 18, 13,
	30, 23, 19, 20, 20, 41, 13,  2,
	 5,  5,  1,  5, 32,  6, 32, 35,
	20, 35, 27, 35, 35, 36, 36, 13,
	36, 41, 41, 41,  3, 30, 42, 27,
	20, 30, 27, 28, 30, 21, 33, 33,
	14, 24, 30, 42, 24, 33, 25, 42,
	43, 14, 43, 43, 14, 43,  7, 36,
	37, 37, 37, 37,  7, 14, 25, 43,
	43, 44, 15, 37,  7,  7,  3,  1,
	 8, 15, 15,  8, 44, 44, 44, 45,
	45, 45, 45,  8,  8, 45, 21, 45,
	28, 28, 28, 21, 28, 28, 22, 37,
	46, 46, 37,  8, 29, 37, 29, 22,
	46, 37, 22, 29, 47, 47, 38, 38,
	16, 38, 38, 33, 38, 22, 47, 47,
	29, 25, 16,  0, 48,  1, 34, 48,
	48, 34, 25, 26, 26, 49, 49, 26,
	 1, 49,  4, 26,  4, 49,  1,  9,
	49, 49, 49, 10, 49, 17, 38, 17,
	17, 50, 38, 50, 50, 22, 38, 51,
	38, 38, 51, 39, 39, 18, 22, 39,
	51, 22, 52, 52, 52, 39, 53, 53,
	10, 23, 18, 29, 10, 53, 29, 54,
	11, 54, 11, 11, 55,  1, 18, 55,
	55, 55, 55, 55, 55, 29, 34, 18,
	29, 56, 56, 34, 57, 34, 34, 29,
	29, 57, 57, 35, 35, 35, 35, 35,
	39, 35, 59, 59, 18, 59, 39, 30,
	18, 40, 60, 60, 61, 30, 18, 61,
	61, 19, 19
};

// Motion vector code 0 (low motion), y difference
static const uint8_t wmv2MV0DiffY[1099] = {
	32, 31, 32, 33, 32, 31, 31, 33,
	33, 34, 32, 30, 32, 35, 34, 31,
	32, 29, 33, 30, 32, 34, 33, 31,
	30, 35, 31, 31, 29, 33, 35, 30,
	29, 33, 34, 34, 30, 32, 32, 36,
	29, 32, 35, 32, 28, 32, 32, 27,
	35, 37, 34, 29, 30, 36, 35, 34,
	25, 30, 29, 35, 33, 31, 31, 32,
	31, 28, 39, 28, 29, 37, 31, 33,
	27, 36, 28, 36, 37, 33, 33, 31,
	27, 32, 31, 38, 26, 25, 25, 33,
	39, 31, 34, 30, 32, 32, 32, 34,
	36, 32, 28, 33, 30, 38, 37, 27,
	33, 28, 32, 37, 35, 38, 29, 34,
	27, 29, 29, 32, 32, 34, 35,  3,
	26, 36, 31, 38, 30, 26, 35, 34,
	37, 26, 25, 32, 32, 39, 23, 37,
	32, 32, 29, 32, 29, 36, 29, 30,
	41, 31, 30, 21, 39, 25, 34, 38,
	32, 35, 39, 32, 33, 33, 32, 27,
	29, 25, 28, 27, 26, 31, 30, 35,
	24, 24, 31, 34, 32, 30, 35, 40,
	28, 38,  5, 35, 29, 36, 36, 32,
	38, 30, 33, 31, 35, 26, 23, 38,
	32, 41, 28, 25, 37, 40, 37, 39,
	32, 36, 33, 39, 25, 26, 28, 31,
	28, 42, 23, 31, 33, 31, 39,  1,
	59, 22, 27,  4, 33, 34, 33, 24,
	41,  3, 35, 41, 41, 28, 36, 36,
	28, 33, 35, 21, 23, 21, 22, 37,
	27, 27, 43, 29, 60, 39, 27, 25,
	59, 34, 27, 27, 26, 40, 37, 27,
	61, 26, 39, 33, 31, 22, 37, 25,
	30, 25, 24, 61, 31, 34, 25, 38,
	32, 32, 30,  3, 61, 43, 29, 23,
	28, 32, 28, 32, 31, 34,  5, 33,
	32, 33, 33, 42, 37, 23, 38, 31,
	40, 26, 32, 26, 37, 38, 36, 24,
	29, 30, 20, 22, 29, 24, 32, 41,
	 2, 34, 25, 33, 29, 31, 39, 35,
	36, 24, 32, 30, 33, 27, 44, 60,
	30, 36, 19, 34, 31, 24, 16, 35,
	32, 38, 21, 33, 31, 31, 21, 35,
	 5, 17, 29, 38, 38, 18, 58, 19,
	43, 41, 30, 41, 43, 39, 29,  7,
	29, 17, 28, 19, 28, 31, 25, 19,
	40, 26, 21, 33, 39, 23, 40, 30,
	39, 34, 35, 32, 32, 24, 33, 30,
	40, 47, 39, 37, 32, 33, 24, 23,
	45, 47, 27, 23, 42, 32, 32, 33,
	36, 37, 37, 17, 18, 22, 40, 38,
	32, 31, 35, 24, 17, 25, 17, 23,
	33, 34, 51, 42, 31, 36, 36, 29,
	21, 22, 37, 44, 43, 25, 47, 33,
	45, 27, 31, 58, 31, 32, 31, 38,
	43, 20, 47, 45, 54,  1, 26, 34,
	38, 14, 22, 24, 33, 34, 32, 32,
	37, 21, 23, 49, 35, 23, 28, 39,
	39, 23, 55, 33, 30, 30, 63, 16,
	42, 28, 13, 33, 33, 35, 19, 46,
	43, 17, 19, 36, 39, 24, 31, 32,
	33, 26, 28, 62, 33, 63, 33, 39,
	19, 49, 17, 31, 43, 13, 15, 29,
	25, 35, 33, 23, 49, 41, 28, 29,
	34, 38,  7, 61, 11, 50, 13, 41,
	19, 47, 25, 26, 15, 42, 41, 29,
	45, 27, 17, 35, 32, 29, 32, 24,
	13, 26, 26, 31, 24, 33, 28, 30,
	31, 11, 45, 46, 33, 33, 35, 57,
	32, 32, 35, 45, 34, 11, 37, 42,
	39, 37, 31, 49, 21, 27, 29, 47,
	53, 40, 51, 16, 26,  1, 40, 30,
	41, 44, 34, 25, 27, 31, 35, 35,
	31, 15, 49,  1, 35, 40,  5, 58,
	21, 29, 22, 59, 45, 31,  9, 26,
	 9, 29, 11, 32, 30,  3, 13, 20,
	18, 20, 11,  3, 29, 40, 31, 53,
	30, 17, 20, 37, 31, 42, 47, 47,
	54, 38,  9, 34, 13, 37, 21, 25,
	27, 43, 42, 45, 40, 25, 27, 46,
	22, 25, 53, 20,  2, 14, 39, 15,
	22, 44, 34, 21, 38, 33, 27, 48,
	34, 52, 35, 47, 49, 54,  2, 13,
	23, 52, 29, 45, 22, 49, 54, 21,
	40, 42, 31, 30, 29, 34,  0, 25,
	23, 51, 24, 59, 28, 38, 29, 31,
	 2, 13, 31,  8, 31, 33, 12, 45,
	41,  7, 14, 30, 25, 18, 43, 20,
	43, 35, 44,  1, 49, 42, 42, 18,
	41, 38, 41, 44, 53, 11, 20, 25,
	45, 46, 47, 48, 39, 52, 46, 49,
	63, 55, 44, 38, 13, 13, 57, 22,
	51, 16, 12, 28, 35, 57, 25, 20,
	26, 28, 28, 29, 32, 31, 62, 34,
	35, 35, 19, 49, 48, 39, 40, 18,
	43, 46, 11,  6, 48, 19, 49, 41,
	10, 23, 58, 17, 21, 23, 34, 30,
	60,  0, 44, 34, 26, 37, 46, 43,
	49, 59,  4, 34, 59, 37, 22, 25,
	28, 46,  6, 40, 59, 42, 36, 61,
	28, 30, 31, 43, 10, 22, 23, 47,
	20, 52, 55, 36, 25, 16,  1, 11,
	27, 29,  5, 63, 18, 41, 31, 34,
	38,  1,  5, 13, 28, 31, 17, 38,
	39, 41, 36, 37, 22, 39, 33, 43,
	43, 15, 17, 49, 30, 21, 22, 20,
	10, 17, 25, 54, 57,  3, 34,  8,
	36, 25, 31, 14, 15, 19, 29, 25,
	18, 39, 53, 22, 27, 20, 29, 33,
	41, 42, 35, 62, 50, 29, 53, 50,
	35, 55, 42, 61, 63,  4,  7, 42,
	21, 46, 47, 49, 27, 46, 17, 55,
	41, 50, 63,  4, 56, 18,  8, 10,
	18, 51, 63, 36, 55, 18,  5, 55,
	 9, 29, 17, 21, 30, 27,  1, 59,
	 7, 11, 12, 15,  5, 42, 24, 41,
	43,  7, 27, 22, 25, 31, 30, 37,
	22, 39, 53, 29, 36, 37, 48,  0,
	 5, 13, 17, 31, 32, 26, 46, 28,
	44, 45, 46, 53, 49, 51,  3, 41,
	 3, 22, 42, 33,  5, 45,  7, 22,
	40, 53, 24, 14, 25, 27, 10, 12,
	34, 16, 17, 53, 20, 26, 39, 45,
	18, 45, 35, 33, 31, 49,  4, 39,
	42, 11, 51,  5, 13, 26, 27, 17,
	52, 30,  0, 22, 12, 34, 62, 36,
	38, 41, 47, 30, 63, 38, 41, 43,
	59, 33, 45, 37, 38, 40, 47, 24,
	48, 49, 30,  1, 10, 22, 49, 15,
	39, 59, 31, 32, 33, 18, 13, 15,
	31, 21, 27, 44, 42, 39, 46, 17,
	26, 32, 30, 31,  0, 30, 34,  9,
	12, 13, 25, 31, 32, 55, 43, 35,
	61, 33, 35, 46, 25, 47, 48, 62,
	63, 38, 61,  1,  2,  5,  7,  9,
	46, 10, 34, 35, 36, 55, 51,  7,
	40, 23, 34, 37,  5, 13, 42, 18,
	25, 27, 28
};

// Motion vector code 1 (high motion), codes
static const uint32_t wmv2HuffmanMV1Codes[1100] = {
	0x0000, 0x0007, 0x0009, 0x000F, 0x000A, 0x0011, 0x001A, 0x001C,
	0x0011, 0x0031, 0x0025, 0x002D, 0x002F, 0x006F, 0x0075, 0x0041,
	0x004C, 0x004E, 0x005C, 0x0060, 0x0062, 0x0066, 0x0068, 0x0069,
	0x006B, 0x00A6, 0x00C1, 0x00CB, 0x00CC, 0x00CE, 0x00DA, 0x00E8,
	0x00EE, 0x0087, 0x0090, 0x009E, 0x009F, 0x00BA, 0x00CA, 0x00D8,
	0x00DB, 0x00DF, 0x0104, 0x0109, 0x010C, 0x0143, 0x0145, 0x014A,
	0x0156, 0x015C, 0x01B3, 0x01D3, 0x01DA, 0x0103, 0x0109, 0x010B,
	0x0122, 0x0127, 0x0134, 0x0161, 0x0164, 0x0176, 0x0184, 0x018D,
	0x018E, 0x018F, 0x0190, 0x0193, 0x0196, 0x019D, 0x019E, 0x019F,
	0x01A9, 0x01B2, 0x01B4, 0x01BA, 0x01BB, 0x01BC, 0x0201, 0x0202,
	0x0205, 0x0207, 0x020D, 0x0210, 0x0211, 0x0215, 0x021B, 0x021F,
	0x0281, 0x0285, 0x0290, 0x029C, 0x029D, 0x02A2, 0x02A7, 0x02A8,
	0x02AA, 0x02B0, 0x02B1, 0x02B4, 0x02BC, 0x02BF, 0x0320, 0x0326,
	0x0327, 0x0329, 0x032A, 0x0336, 0x0360, 0x0362, 0x0363, 0x0372,
	0x03B2, 0x03BC, 0x03BD, 0x0203, 0x0205, 0x021A, 0x0249, 0x024A,
	0x024C, 0x02C7, 0x02CA, 0x02CE, 0x02EF, 0x030D, 0x0322, 0x0325,
	0x0338, 0x0373, 0x037A, 0x0409, 0x0415, 0x0416, 0x0418, 0x0428,
	0x042D, 0x042F, 0x0434, 0x0508, 0x0509, 0x0510, 0x0511, 0x051C,
	0x051E, 0x0524, 0x0541, 0x0543, 0x0546, 0x0547, 0x054D, 0x0557,
	0x055F, 0x056A, 0x056C, 0x056D, 0x056F, 0x0576, 0x0577, 0x057A,
	0x057B, 0x057C, 0x057D, 0x0600, 0x0601, 0x0603, 0x0614, 0x0616,
	0x0617, 0x061C, 0x061F, 0x0642, 0x0648, 0x0649, 0x064A, 0x064B,
	0x0657, 0x0668, 0x0669, 0x066B, 0x066E, 0x067F, 0x06C2, 0x06C8,
	0x06CB, 0x06DE, 0x06DF, 0x06E2, 0x06E3, 0x06EF, 0x0748, 0x074B,
	0x076E, 0x076F, 0x077C, 0x0409, 0x0423, 0x0428, 0x0429, 0x042A,
	0x042B, 0x0432, 0x0433, 0x0496, 0x049A, 0x04D5, 0x04DB, 0x0581,
	0x0582, 0x058B, 0x058C, 0x058D, 0x0598, 0x0599, 0x059A, 0x059E,
	0x05DD, 0x0619, 0x0632, 0x0633, 0x0648, 0x0672, 0x06A1, 0x06A2,
	0x06A3, 0x06AF, 0x06E2, 0x06E3, 0x06E4, 0x0800, 0x0801, 0x0802,
	0x0803, 0x081A, 0x081B, 0x0829, 0x082F, 0x0832, 0x083E, 0x083F,
	0x0852, 0x0853, 0x0858, 0x086B, 0x0877, 0x0878, 0x0879, 0x087A,
	0x087B, 0x0A00, 0x0A01, 0x0A0D, 0x0A0E, 0x0A0F, 0x0A24, 0x0A37,
	0x0A3A, 0x0A3B, 0x0A3E, 0x0A46, 0x0A47, 0x0A4A, 0x0A4B, 0x0A5F,
	0x0A79, 0x0A7A, 0x0A7B, 0x0A80, 0x0A81, 0x0A84, 0x0A85, 0x0A99,
	0x0AA5, 0x0AA6, 0x0AB8, 0x0ABA, 0x0ABB, 0x0ABC, 0x0ABD, 0x0AC8,
	0x0ACE, 0x0ACF, 0x0AD7, 0x0ADC, 0x0AEB, 0x0C04, 0x0C25, 0x0C26,
	0x0C27, 0x0C2A, 0x0C2B, 0x0C3A, 0x0C3B, 0x0C3C, 0x0C3D, 0x0CA0,
	0x0CAD, 0x0CD4, 0x0CD5, 0x0CFC, 0x0CFD, 0x0D86, 0x0D92, 0x0D93,
	0x0D94, 0x0D95, 0x0DB0, 0x0DB8, 0x0DB9, 0x0DBA, 0x0DBB, 0x0DC0,
	0x0DC2, 0x0DC3, 0x0DDA, 0x0DDB, 0x0DDC, 0x0DDD, 0x0E92, 0x0E93,
	0x0E94, 0x0E95, 0x0EC7, 0x0ECC, 0x0ECE, 0x0ECF, 0x0ED8, 0x0ED9,
	0x0EDA, 0x0EDB, 0x0808, 0x0809, 0x080A, 0x0810, 0x0811, 0x0844,
	0x0845, 0x0861, 0x0862, 0x0863, 0x086C, 0x0922, 0x0923, 0x092E,
	0x092F, 0x0936, 0x0937, 0x09B1, 0x09B2, 0x09B3, 0x09B4, 0x09B5,
	0x09B8, 0x09B9, 0x09BA, 0x09BB, 0x09BC, 0x09BD, 0x09BE, 0x09BF,
	0x0B00, 0x0B15, 0x0B2C, 0x0B2D, 0x0B2E, 0x0B2F, 0x0B36, 0x0BB9,
	0x0C28, 0x0C2A, 0x0C2B, 0x0C2C, 0x0C2D, 0x0C2E, 0x0C2F, 0x0C30,
	0x0C31, 0x0C38, 0x0C60, 0x0C61, 0x0C62, 0x0C63, 0x0C8D, 0x0C8E,
	0x0C8F, 0x0C92, 0x0CBE, 0x0CBF, 0x0CE6, 0x0CE7, 0x0D40, 0x0D41,
	0x0D57, 0x0D58, 0x0D59, 0x0D5A, 0x0D5B, 0x0D5C, 0x0D5D, 0x0D98,
	0x0D99, 0x0D9A, 0x0D9B, 0x0D9C, 0x0D9D, 0x0DAD, 0x0DAE, 0x0DAF,
	0x0DC0, 0x0DC1, 0x0DC2, 0x0DC3, 0x0DCA, 0x0DCB, 0x0DEC, 0x0DED,
	0x0DEE, 0x0DEF, 0x1018, 0x1022, 0x1023, 0x1030, 0x1031, 0x1032,
	0x1033, 0x1050, 0x1051, 0x105C, 0x1074, 0x1075, 0x1076, 0x1077,
	0x1078, 0x1079, 0x107A, 0x107B, 0x10B2, 0x10B3, 0x10B8, 0x10B9,
	0x10BA, 0x10BB, 0x10D4, 0x10EA, 0x10EB, 0x10EC, 0x10ED, 0x1404,
	0x1405, 0x1406, 0x1407, 0x1410, 0x1411, 0x1412, 0x1413, 0x1414,
	0x1415, 0x1416, 0x1417, 0x1418, 0x1419, 0x1466, 0x1467, 0x1468,
	0x1469, 0x146A, 0x146B, 0x146C, 0x146D, 0x147E, 0x147F, 0x1488,
	0x1489, 0x148A, 0x148B, 0x14B6, 0x14B7, 0x14B8, 0x14B9, 0x14BA,
	0x14BB, 0x14BC, 0x14BD, 0x14F0, 0x14F1, 0x14F8, 0x14F9, 0x14FA,
	0x14FB, 0x14FC, 0x14FD, 0x14FE, 0x14FF, 0x152A, 0x152B, 0x152C,
	0x152D, 0x152E, 0x152F, 0x1530, 0x1531, 0x1548, 0x1549, 0x154E,
	0x154F, 0x1558, 0x1559, 0x155A, 0x155B, 0x1572, 0x159A, 0x159B,
	0x15AC, 0x15BA, 0x15BB, 0x15D0, 0x15D1, 0x15D2, 0x15D3, 0x15D4,
	0x15D5, 0x181D, 0x181E, 0x181F, 0x1840, 0x1841, 0x1842, 0x1843,
	0x1844, 0x1845, 0x1846, 0x1847, 0x1848, 0x1849, 0x1861, 0x1862,
	0x1863, 0x1864, 0x1865, 0x1866, 0x1867, 0x1868, 0x1869, 0x186A,
	0x186B, 0x186C, 0x186D, 0x186E, 0x191B, 0x191C, 0x191D, 0x191E,
	0x191F, 0x1942, 0x1943, 0x1944, 0x1945, 0x1946, 0x1947, 0x1958,
	0x1959, 0x19ED, 0x19EE, 0x19EF, 0x19F0, 0x19F1, 0x19F2, 0x19F3,
	0x19F4, 0x19F5, 0x19F6, 0x19F7, 0x1B0E, 0x1B0F, 0x1B62, 0x1B63,
	0x1B64, 0x1B65, 0x1B66, 0x1B67, 0x1B68, 0x1B69, 0x1B6A, 0x1B6B,
	0x1B6C, 0x1B6D, 0x1B6E, 0x1B6F, 0x1B82, 0x1BA8, 0x1BA9, 0x1BAA,
	0x1BAB, 0x1BAC, 0x1BAD, 0x1BAE, 0x1BAF, 0x1BB0, 0x1BB1, 0x1BB2,
	0x1BB3, 0x1D80, 0x1D81, 0x1D82, 0x1D83, 0x1D84, 0x1D85, 0x1D86,
	0x1D87, 0x1D88, 0x1D89, 0x1D8A, 0x1D8B, 0x1D8C, 0x1D8D, 0x1007,
	0x1008, 0x1009, 0x100A, 0x100B, 0x100C, 0x100D, 0x100E, 0x100F,
	0x1016, 0x1080, 0x1081, 0x1082, 0x1083, 0x1084, 0x1085, 0x1086,
	0x1087, 0x10C0, 0x123A, 0x123B, 0x123C, 0x123D, 0x123E, 0x123F,
	0x1240, 0x1241, 0x1242, 0x1243, 0x1350, 0x1352, 0x1353, 0x1358,
	0x1359, 0x135A, 0x135B, 0x135C, 0x135D, 0x135E, 0x135F, 0x1360,
	0x1361, 0x1602, 0x1603, 0x160C, 0x160D, 0x160E, 0x160F, 0x1620,
	0x1621, 0x1622, 0x1623, 0x1624, 0x1625, 0x1626, 0x1627, 0x1628,
	0x1629, 0x166E, 0x166F, 0x167C, 0x167D, 0x167E, 0x167F, 0x1770,
	0x1771, 0x1852, 0x1853, 0x1872, 0x1873, 0x1874, 0x1875, 0x1876,
	0x1877, 0x1878, 0x1879, 0x187A, 0x187B, 0x187C, 0x187D, 0x187E,
	0x187F, 0x1918, 0x1919, 0x1926, 0x1927, 0x1970, 0x1971, 0x1972,
	0x1973, 0x1974, 0x1975, 0x1976, 0x1977, 0x1978, 0x1979, 0x197A,
	0x197B, 0x1AA0, 0x1AA1, 0x1AA2, 0x1AA3, 0x1AA4, 0x1AA5, 0x1AA6,
	0x1AA7, 0x1AA8, 0x1AA9, 0x1AAA, 0x1AAB, 0x1AAC, 0x1AAD, 0x1B3C,
	0x1B3D, 0x1B3E, 0x1B3F, 0x1B50, 0x1B51, 0x1B52, 0x1B53, 0x1B54,
	0x1B55, 0x1B56, 0x1B57, 0x1B58, 0x1B59, 0x2032, 0x2033, 0x2034,
	0x2035, 0x2036, 0x2037, 0x2038, 0x2039, 0x203A, 0x203B, 0x203C,
	0x203D, 0x203E, 0x203F, 0x2040, 0x2041, 0x2042, 0x2043, 0x20BA,
	0x20BB, 0x20CC, 0x20CD, 0x20CE, 0x20CF, 0x20E0, 0x20E1, 0x20E2,
	0x20E3, 0x20E4, 0x20E5, 0x20E6, 0x20E7, 0x21AA, 0x21AB, 0x21C0,
	0x21C1, 0x21C2, 0x21C3, 0x21C4, 0x21C5, 0x21C6, 0x21C7, 0x21C8,
	0x21C9, 0x21CA, 0x21CB, 0x21CC, 0x21CD, 0x21CE, 0x21CF, 0x21D0,
	0x21D1, 0x21D2, 0x21D3, 0x2894, 0x2895, 0x2896, 0x2897, 0x2898,
	0x2899, 0x289A, 0x289B, 0x289C, 0x289D, 0x289E, 0x289F, 0x28C0,
	0x28C1, 0x28C2, 0x28C3, 0x28C4, 0x28C5, 0x28C6, 0x28C7, 0x28C8,
	0x28C9, 0x28CA, 0x28CB, 0x2930, 0x2931, 0x2932, 0x2933, 0x2934,
	0x2935, 0x2936, 0x2937, 0x2938, 0x2939, 0x293A, 0x293B, 0x293C,
	0x293D, 0x293E, 0x293F, 0x2960, 0x2961, 0x2962, 0x2963, 0x2964,
	0x2965, 0x2966, 0x2967, 0x2968, 0x2969, 0x296A, 0x296B, 0x2A40,
	0x2A41, 0x2A42, 0x2A43, 0x2A44, 0x2A45, 0x2A46, 0x2A47, 0x2A48,
	0x2A49, 0x2A4A, 0x2A4B, 0x2A4C, 0x2A4D, 0x2A4E, 0x2A4F, 0x2A50,
	0x2A51, 0x2A52, 0x2A53, 0x2AE6, 0x2AE7, 0x2B24, 0x2B25, 0x2B26,
	0x2B27, 0x2B28, 0x2B29, 0x2B2A, 0x2B2B, 0x2B2C, 0x2B2D, 0x2B2E,
	0x2B2F, 0x2B30, 0x2B31, 0x2B32, 0x2B33, 0x2B5A, 0x2B5B, 0x3014,
	0x3015, 0x3016, 0x3017, 0x3020, 0x3021, 0x3022, 0x3023, 0x3024,
	0x3025, 0x3026, 0x3027, 0x3028, 0x3029, 0x302A, 0x302B, 0x302C,
	0x302D, 0x302E, 0x302F, 0x3030, 0x3031, 0x3032, 0x3033, 0x3034,
	0x3035, 0x3036, 0x3037, 0x3038, 0x3039, 0x30C0, 0x30C1, 0x30DE,
	0x30DF, 0x3218, 0x3219, 0x321A, 0x321B, 0x321C, 0x321D, 0x321E,
	0x321F, 0x3220, 0x3221, 0x3222, 0x3223, 0x3224, 0x3225, 0x3226,
	0x3227, 0x3228, 0x3229, 0x322A, 0x322B, 0x322C, 0x322D, 0x322E,
	0x322F, 0x3230, 0x3231, 0x3232, 0x3233, 0x3234, 0x3235, 0x3378,
	0x3379, 0x337A, 0x337B, 0x337C, 0x337D, 0x337E, 0x337F, 0x33C0,
	0x33C1, 0x33C2, 0x33C3, 0x33C4, 0x33C5, 0x33C6, 0x33C7, 0x33C8,
	0x33C9, 0x33CA, 0x33CB, 0x33CC, 0x33CD, 0x33CE, 0x33CF, 0x33D0,
	0x33D1, 0x33D2, 0x33D3, 0x33D4, 0x33D5, 0x33D6, 0x33D7, 0x33D8,
	0x33D9, 0x3706, 0x3707, 0x3730, 0x3731, 0x3732, 0x3733, 0x3734,
	0x3735, 0x3736, 0x3737, 0x3738, 0x3739, 0x373A, 0x373B, 0x373C,
	0x373D, 0x373E, 0x373F, 0x3740, 0x3741, 0x3742, 0x3743, 0x3744,
	0x3745, 0x3746, 0x3747, 0x3748, 0x3749, 0x374A, 0x374B, 0x374C,
	0x374D, 0x374E, 0x374F, 0x3B34, 0x3B35, 0x3B36, 0x3B37, 0x3BE8,
	0x3BE9, 0x3BEA, 0x3BEB, 0x3BEC, 0x3BED, 0x3BEE, 0x3BEF, 0x3BF0,
	0x3BF1, 0x3BF2, 0x3BF3, 0x3BF4, 0x3BF5, 0x3BF6, 0x3BF7, 0x3BF8,
	0x3BF9, 0x3BFA, 0x3BFB, 0x3BFC, 0x3BFD, 0x3BFE, 0x3BFF, 0x2000,
	0x2001, 0x2002, 0x2003, 0x2004, 0x2005, 0x2006, 0x2007, 0x2008,
	0x2009, 0x200A, 0x200B, 0x200C, 0x200D, 0x202E, 0x202F, 0x2182,
	0x2183, 0x21B4, 0x21B5, 0x21B6, 0x21B7, 0x21B8, 0x21B9, 0x21BA,
	0x21BB, 0x21BC, 0x21BD, 0x21BE, 0x21BF, 0x2460, 0x2461, 0x2462,
	0x2463, 0x2464, 0x2465, 0x2466, 0x2467, 0x2468, 0x2469, 0x246A,
	0x246B, 0x246C, 0x246D, 0x246E, 0x246F, 0x2470, 0x2471, 0x2472,
	0x2473, 0x26A2, 0x26A3, 0x000B
};

// Motion vector code 1 (high motion), lengths
static const uint8_t wmv2HuffmanMV1Lengths[1100] = {
	 2,  4,  4,  4,  5,  5,  5,  5,
	 6,  6,  7,  7,  7,  7,  7,  8,
	 8,  8,  8,  8,  8,  8,  8,  8,
	 8,  8,  8,  8,  8,  8,  8,  8,
	 8,  9,  9,  9,  9,  9,  9,  9,
	 9,  9,  9,  9,  9,  9,  9,  9,
	 9,  9,  9,  9,  9, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 11, 11, 11, 11, 11,
	11, 11, 11, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 14,
	14, 14, 14, 14, 14, 14, 14, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15, 15, 15, 15, 15, 15,
	15, 15, 15,  4
};

// Motion vector code 1 (high motion), x difference
static const uint8_t wmv2MV1DiffX[1099] = {
	32, 31, 32, 31, 33, 32, 33, 33,
	31, 34, 30, 32, 32, 34, 35, 32,
	34, 33, 29, 30, 30, 32, 31, 31,
	33, 35, 35, 33, 31, 29, 29, 33,
	34, 30, 31, 28, 36, 30, 34, 32,
	32, 37, 32, 32, 25, 27, 39, 32,
	32, 32, 38, 35, 36, 32, 37, 61,
	26, 32, 34, 35,  3, 35, 27, 28,
	29, 34, 28, 37, 31, 36, 32, 27,
	31, 30, 29, 39, 33, 29, 33, 35,
	25, 25, 29, 33, 31, 31, 31, 33,
	32, 30, 32, 32, 41, 39, 33, 36,
	32, 28, 34, 36, 38, 24, 60, 31,
	23, 28, 32, 33, 59, 32, 40, 30,
	 5, 34, 32, 38, 32, 30, 43,  4,
	32, 32, 42, 31, 31, 32, 26, 38,
	26, 22, 21, 37, 61, 63, 37, 31,
	32, 33,  2,  1, 23, 33, 41, 27,
	35, 30, 38, 23, 33,  3, 28, 34,
	34, 27, 41, 29, 39, 35, 36, 29,
	32, 27, 30, 32, 24, 61, 37, 26,
	59, 25, 35, 27, 36, 37, 30, 31,
	34, 40,  3, 28, 34, 39, 32, 31,
	32, 30, 24, 28, 35, 36, 26, 32,
	31, 33, 29, 33, 39, 25, 30, 24,
	35, 59, 29, 34, 25, 30, 21, 35,
	43, 40, 32, 29,  5, 28, 31, 62,
	33, 33, 25, 31, 21, 31, 43, 31,
	34, 33, 20, 40, 39, 31, 31, 57,
	38, 32, 42, 33, 32, 31, 32, 29,
	30, 44,  5, 31, 22, 34, 36, 17,
	38, 58, 38, 35, 32, 60, 35, 24,
	32, 38, 16, 45, 42, 32, 31, 29,
	 4, 30, 17, 40, 46, 48, 63, 32,
	42, 19, 41, 22, 28, 36, 45, 33,
	33, 32, 29,  7, 41, 42, 18, 33,
	33, 32, 22, 37,  1, 26, 22, 23,
	49, 28, 26, 27, 32, 33, 27, 23,
	28, 36, 15,  6, 34, 27, 31, 26,
	23,  2, 33, 32, 34, 41, 28, 32,
	41,  0, 36, 38, 34, 31, 47, 32,
	17, 31, 39, 33, 37, 51, 30, 47,
	32, 50, 32, 19, 63, 30, 25, 27,
	33, 62, 24, 31, 27, 30, 37, 31,
	45, 32, 39, 20, 46, 47, 35, 19,
	34,  1, 49, 21, 21, 14, 51, 26,
	23, 31, 36, 35, 58, 29, 29, 21,
	20, 42, 13, 28, 12, 40, 31, 33,
	39, 60, 32, 44, 33, 31, 28, 37,
	29, 32, 30, 49, 43, 28, 39, 25,
	32, 48,  2, 15, 20, 25, 31, 28,
	21, 24, 25, 15, 31, 17, 37, 43,
	18, 32, 33, 24, 33, 36, 13, 33,
	31, 39, 11, 31, 33, 32, 39, 37,
	32, 32, 29, 17, 44, 46, 36, 35,
	26, 37, 58, 32, 34, 38,  8, 38,
	38, 22, 29, 25, 16, 35, 32, 35,
	33, 43, 18, 46, 38, 50, 33, 18,
	53, 60, 13, 32, 36, 33, 51, 36,
	43, 45, 27, 42, 29, 24, 30, 25,
	31, 52, 31, 35, 38,  9, 22, 34,
	 4, 17, 28, 55, 42, 25, 17, 20,
	47, 34, 33, 16, 40, 25, 16, 30,
	53, 29, 10, 11, 14, 26, 33,  4,
	35, 44, 26, 16, 31, 26, 34, 38,
	29, 31, 30, 24, 22, 61, 32,  9,
	45, 34, 31, 19,  9, 31, 46, 31,
	35, 54, 29, 57, 30, 50,  3, 31,
	63, 34, 47, 41, 51, 18, 31, 14,
	37, 38, 31, 24, 32, 31, 50, 33,
	31, 54, 27,  9, 33, 23, 19, 32,
	29, 29, 33, 28, 47, 49, 30, 47,
	33, 27, 25, 54, 44, 45, 50, 58,
	51, 48, 33, 59, 33, 34, 57, 13,
	26, 33, 13, 48, 30, 11,  7, 56,
	34, 55, 26,  0, 26, 35,  1, 51,
	33, 53, 31, 45, 12, 29, 29, 51,
	31, 48,  2,  6, 34, 30, 28, 33,
	60, 40, 27, 46, 31,  9, 35, 29,
	31, 39, 55, 46, 19, 37, 62, 34,
	30, 16, 19, 49, 41, 41, 39, 37,
	14,  5, 13, 35, 55, 30, 40, 40,
	42,  8, 20, 25, 45, 35, 33, 36,
	54, 38, 27, 37, 62, 40, 15, 59,
	49, 31, 29, 34, 34, 39, 24, 29,
	25, 29, 21, 29, 10, 61, 33, 49,
	35, 34,  3, 38, 39, 29,  7, 41,
	 1, 35,  4, 23, 15, 23, 11, 37,
	28, 35, 30, 30, 24,  1, 43, 56,
	 8, 34, 42, 24, 45, 30, 20, 23,
	 8, 38, 22, 33, 17, 52, 34, 22,
	53, 43, 44,  1, 27, 31, 41, 43,
	41, 30, 31, 36, 30,  5, 55, 31,
	33, 30, 40, 23, 15, 29, 34, 34,
	59, 34, 30, 11, 13, 38,  5,  0,
	30, 42,  5, 30, 29, 34, 10, 44,
	30, 63, 35, 12,  3, 26, 15, 17,
	25, 34, 43, 39, 34, 56, 29, 23,
	30, 12, 30, 10, 35,  9, 24, 58,
	10, 12, 54, 33, 37, 20, 41, 35,
	29, 18, 61, 30, 40, 24, 39, 53,
	62, 26, 29, 33, 34, 53, 49, 21,
	27, 11, 63, 20, 26, 23,  7, 13,
	 6, 47, 29, 30,  9, 51, 22, 34,
	21, 25, 33, 56, 57, 30, 38, 51,
	51, 38, 63, 28, 40, 35, 33, 18,
	33, 33, 24, 58, 58, 34, 49, 29,
	43,  4,  1,  4, 42, 35, 35, 30,
	17,  5, 56, 61, 25, 37, 36, 55,
	28, 35, 29, 50, 48, 52,  2, 42,
	34, 40, 46, 46, 43, 35, 29, 48,
	20, 29, 31, 41,  7, 30, 35, 19,
	14, 21,  8, 39, 39, 40, 46, 55,
	34,  6, 30, 34, 37, 25, 37, 33,
	22, 44, 52, 17, 35, 29, 36, 35,
	40, 37, 28, 30, 50, 14, 28, 55,
	 6, 23, 19, 14, 30,  3, 30, 28,
	28, 61, 61, 47, 45, 48, 40, 40,
	34, 34, 25, 30, 29, 35,  4, 26,
	53, 50, 26, 41, 27, 59, 27, 38,
	39,  3, 50, 43, 47, 23, 33, 55,
	35, 21, 23, 35, 61, 33, 46, 52,
	35, 34, 24, 30, 43, 16, 37, 21,
	 2, 24, 45, 34, 30, 55, 55,  1,
	29, 29, 26, 28, 25, 31, 36, 22,
	17, 30, 52,  2, 44, 44, 57, 26,
	62, 41, 39, 57, 26, 46, 49, 11,
	16, 19,  5, 59, 38, 39, 58, 38,
	25, 49, 50, 22, 28, 59,  9, 59,
	 7, 28, 55, 17,  4, 35, 50, 21,
	29, 44, 47, 18, 24, 19, 25, 42,
	35,  3, 51, 35, 16, 35, 30, 63,
	57, 39, 39, 25, 35, 38,  9, 16,
	36, 45, 31, 60, 14, 34, 42, 24,
	 0, 37, 18, 61, 57, 37, 28, 53,
	20, 46, 14, 47, 38, 38, 38,  9,
	34, 39, 43, 17, 39, 59,  5, 27,
	 0, 12, 27
};

// Motion vector code 1 (high motion), y difference
static const uint8_t wmv2MV1DiffY[1099] = {
	32, 32, 31, 31, 32, 33, 31, 33,
	33, 32, 32, 30, 34, 31, 32, 29,
	33, 30, 32, 33, 31, 35, 34, 30,
	34, 31, 33, 29, 29, 31, 33, 35,
	30, 30, 35, 32, 32, 34, 34, 28,
	25, 32, 36, 27, 32, 32, 32, 37,
	39,  3, 32, 30, 31, 26, 31, 32,
	32, 38, 29, 29, 32, 34, 31, 31,
	34, 35, 33, 33, 28, 33,  1, 33,
	27, 29, 30, 31, 28, 29, 37, 35,
	31, 33, 35, 27, 36, 37, 25, 25,
	61, 35,  4,  5, 32, 33, 36, 30,
	23, 30, 28, 34, 31, 32, 32, 39,
	32, 34, 21, 39, 32, 59, 32, 28,
	32, 36, 60, 33, 24, 36, 32, 32,
	41,  2, 32, 38, 26, 22, 33, 30,
	31, 32, 32, 30, 31, 32, 29,  3,
	40, 38, 32, 32, 33, 26, 31, 34,
	28, 38, 34, 31,  3, 31, 35, 38,
	27, 35, 33, 28, 29, 27, 29, 27,
	43, 29, 37, 63, 31, 33, 34, 30,
	31, 30, 37, 30, 35, 35, 26, 41,
	37, 31, 33, 28, 26, 30, 42, 24,
	 7, 27, 33, 29, 36, 28, 34, 57,
	23, 41, 36, 23, 35, 34, 25, 30,
	25, 33, 25, 25, 29, 24, 33, 39,
	33, 33,  0, 37, 31, 36, 21, 32,
	61, 24, 35, 61, 31,  5, 31, 59,
	39, 21, 32, 30, 34, 22, 40, 32,
	29, 16, 31,  5, 62,  2, 20, 39,
	39, 32, 33,  1, 31, 24, 36, 32,
	36, 32, 28, 26,  6, 31, 38, 34,
	58, 35, 32, 33, 33, 17, 43, 26,
	31, 40, 31, 34, 32, 32, 31, 19,
	30, 32, 29, 33, 38, 38, 32, 59,
	40, 18, 38, 32, 35, 34, 32, 17,
	 1, 15, 30, 28, 31, 28, 34, 29,
	32, 27, 35, 27, 49, 22, 37, 34,
	37, 26, 32, 32, 22, 28, 45, 29,
	30, 31, 43, 46, 41, 30, 26, 13,
	34, 32, 27, 38, 42, 42, 33, 47,
	33, 60, 27, 42, 25, 32, 22, 32,
	48, 32, 45, 33, 33, 41, 27, 25,
	19, 31, 35, 19, 36, 42, 27, 17,
	31, 44, 28, 33, 33, 31, 23, 31,
	40, 33, 31, 34, 30, 32, 33, 36,
	35, 47, 37, 41, 31, 23, 41, 29,
	30, 35, 32, 25, 32, 28, 58,  2,
	37, 33, 14, 33, 49, 20, 39, 36,
	21,  9, 23, 33, 35, 24, 39, 37,
	11, 33, 30, 31, 31, 28, 51, 40,
	35, 29, 25, 33, 46, 35, 37, 30,
	30,  8, 63, 28, 15, 40, 33, 45,
	49, 25, 32,  4, 47, 51, 36, 39,
	53, 10, 24, 29, 30, 31, 25, 40,
	38, 38, 33, 56, 23, 27, 32, 37,
	26, 29, 43, 36, 33, 24, 55, 43,
	 9, 29, 34, 34, 24, 33, 18, 33,
	33, 30, 31, 50, 24, 60, 30, 39,
	34, 30, 39, 28, 22, 38,  2, 26,
	63, 32, 57, 21, 39, 33, 28, 18,
	30, 34, 22, 33, 29, 41, 30, 34,
	35, 21, 13, 34, 35, 39, 30, 46,
	32, 42, 32, 31, 33, 26, 11, 33,
	22, 31, 25, 31, 53, 27, 43, 25,
	40, 50, 21, 36, 38, 30, 12, 31,
	34, 20, 15, 29, 32, 62, 30, 13,
	17, 32, 19, 31, 20, 31, 30,  7,
	 1, 17, 34, 37, 31, 31, 44, 34,
	26, 40, 16, 37, 52, 48, 30, 20,
	18, 33, 38, 29,  7, 25, 30, 54,
	45, 47, 46, 41, 29, 29, 16, 30,
	14, 26, 38, 34, 34, 29, 34, 30,
	29, 30, 57, 30,  4, 46, 33, 29,
	39, 44, 30, 31, 50, 33, 31, 32,
	19, 32, 40, 31, 37, 47,  1, 35,
	16, 31,  0, 35, 33,  1, 17, 34,
	 9, 34, 33, 31, 49, 43, 42, 51,
	34, 29, 23, 29, 14, 30, 45, 49,
	11, 24, 31, 28, 35, 41, 30, 44,
	18, 29, 34, 35, 36, 25, 26, 21,
	31, 30, 34, 19, 34, 44, 36, 38,
	25, 31, 28, 23, 37,  3, 55, 41,
	30, 22, 41, 24, 33, 26, 35, 35,
	30, 55, 51, 47, 48, 38, 24, 15,
	21, 50, 25, 46, 30, 29, 10, 34,
	42, 45, 29, 42, 22,  3, 33, 27,
	34,  1, 34, 28, 34, 36, 35, 23,
	23, 13, 58,  3, 26, 63, 25, 31,
	34, 61, 38, 39, 25, 61, 29, 37,
	30, 41, 26, 48, 28, 33, 50, 35,
	30, 37, 29, 29, 40,  6, 39, 28,
	28, 19,  8, 22, 45, 34, 35, 10,
	58, 17, 37, 39, 30, 18, 54, 14,
	29, 16, 59, 30, 35, 23, 35, 30,
	47, 36, 29, 55, 20, 12, 31, 35,
	14, 29, 18, 34, 34, 24, 29, 26,
	22,  2, 27, 23,  8, 30, 55, 38,
	60, 31,  4, 34, 49, 34, 27, 34,
	33, 30, 31, 54, 42, 35, 38, 46,
	44, 26, 27,  9, 39, 25, 21, 29,
	28, 42, 13,  0,  5, 34, 37, 28,
	24, 29, 63, 26, 22, 27, 29, 25,
	33, 25, 61,  0, 35, 25, 36, 15,
	27, 40, 53, 33,  3, 10, 16, 37,
	38, 18, 30, 46, 27,  9,  6, 29,
	62,  8, 42, 28, 29,  3, 25, 16,
	26, 29, 35, 28, 27, 51, 61, 48,
	37,  9, 34,  7, 49, 45, 20, 29,
	21,  5,  5, 29, 28, 34, 29, 24,
	10, 24, 35, 36, 38, 55, 11, 36,
	38, 53, 54, 26, 30, 49, 20, 27,
	30, 39, 33, 41, 49, 22, 38, 38,
	 4, 30,  8,  9,  3, 24, 22, 50,
	37, 36, 31, 27,  2,  9, 42, 63,
	25, 19, 44,  1, 28, 28, 48, 30,
	34, 41, 41, 38, 12, 27, 15,  0,
	16, 34, 35, 38, 28, 29, 40, 42,
	51, 52, 45, 54, 59, 59, 42, 44,
	37, 26, 46, 24, 15, 39, 22, 46,
	19, 35, 38, 17, 37, 23, 52, 55,
	50, 37, 26, 11, 37, 12, 24, 30,
	16, 13, 22, 13, 36, 35, 40, 41,
	34, 41, 26, 53, 51,  5, 21, 30,
	 2, 63, 41, 20,  1, 56, 21, 24,
	25,  5, 28, 35, 26, 28, 30, 18,
	29, 23, 40, 34, 20, 42, 39, 34,
	28, 61, 38, 27, 62,  9, 36, 17,
	 9, 49, 24, 25, 54, 34, 39, 37,
	 3,  1, 25, 38, 38, 44, 35, 36,
	12, 60, 36, 38, 40, 25, 43, 39,
	53, 28, 39, 57, 46, 10, 52, 27,
	35, 42, 45, 59, 15, 60, 38, 24,
	23, 39, 12, 29, 24,  0, 20, 16,
	28, 43, 35, 28,  1, 49,  4, 21,
	42, 39, 29,  3, 44, 21, 53, 55,
	11,  5,  3, 39, 53, 28, 25, 19,
	34, 28, 21
};

// P-Frame macroblock block pattern, code 0 (high rate), codes
static const uint32_t wmv2HuffmanPMB0Codes[128] = {
	0x000A7, 0x1B2B8, 0x1B28E, 0x36575, 0x06CAC, 0x00A69, 0x02934, 0x0526B,
	0x06CA1, 0x1B2B9, 0x029AD, 0x29353, 0x06CA7, 0x06CAB, 0x1B2BB, 0x0029B,
	0x0D944, 0x00A6A, 0x149A8, 0x3651F, 0x06CAF, 0x00A4C, 0x3651E, 0x00A48,
	0x0299C, 0x0299F, 0x29352, 0x029AC, 0x00296, 0x0D946, 0x00A68, 0x00298,
	0x00527, 0x0D94D, 0x014D7, 0x36574, 0x00A5C, 0x1B299, 0x0299D, 0x0299E,
	0x00525, 0x00A66, 0x0A4D5, 0x0149B, 0x00295, 0x06CAD, 0x00A49, 0x00521,
	0x06CAA, 0x0D945, 0x1B298, 0x0052F, 0x03654, 0x06CA0, 0x00532, 0x00291,
	0x03652, 0x00520, 0x00A5D, 0x00294, 0x0009B, 0x006E2, 0x00028, 0x001B0,
	0x00001, 0x00010, 0x0002F, 0x0004C, 0x0000D, 0x00000, 0x00006, 0x00134,
	0x0000C, 0x00007, 0x00007, 0x006E1, 0x0000E, 0x000DA, 0x00022, 0x00364,
	0x0000F, 0x00006, 0x0000F, 0x00135, 0x00014, 0x000DD, 0x00004, 0x00015,
	0x0001A, 0x001B3, 0x00005, 0x006E3, 0x0000C, 0x000B9, 0x00004, 0x000DB,
	0x0000E, 0x0000B, 0x00023, 0x006CB, 0x00005, 0x001B1, 0x00001, 0x006E0,
	0x00011, 0x000DF, 0x0000E, 0x00373, 0x00003, 0x000B8, 0x00006, 0x00175,
	0x00015, 0x00174, 0x00027, 0x00372, 0x00010, 0x000BB, 0x00005, 0x000DE,
	0x0000F, 0x00001, 0x00012, 0x00004, 0x00002, 0x00016, 0x00009, 0x00001
};

// P-Frame macroblock block pattern, code 0 (high rate), lengths
static const uint8_t wmv2HuffmanPMB0Lengths[128] = {
	14, 18, 18, 19, 16, 18, 20, 21,
	16, 18, 20, 24, 16, 16, 18, 16,
	17, 18, 23, 19, 16, 18, 19, 18,
	20, 20, 24, 20, 16, 17, 18, 16,
	17, 17, 19, 19, 18, 18, 20, 20,
	17, 18, 22, 19, 16, 16, 18, 17,
	16, 17, 18, 17, 15, 16, 17, 16,
	15, 17, 18, 16, 11, 12, 12, 10,
	 3,  8,  6, 10,  4, 10,  9, 12,
	 4, 10,  9, 12,  5,  9,  9, 11,
	 4, 10,  9, 12,  5,  9,  9, 11,
	 6, 10, 10, 12,  5,  8,  8,  9,
	 4, 10,  9, 12,  6, 10, 10, 12,
	 5,  9,  9, 11,  5,  8,  8,  9,
	 5,  9,  9, 11,  5,  8,  8,  9,
	 5,  9,  8, 10,  3,  5,  4,  5
};

// P-Frame macroblock block pattern, code 1 (mid rate), codes
static const uint32_t wmv2HuffmanPMB1Codes[128] = {
	0x002A1, 0x05740, 0x1A0BF, 0x15D19, 0x01514, 0x0461E, 0x15176, 0x15177,
	0x011AD, 0x0682E, 0x682F9, 0x3417D, 0x01A36, 0x02A2D, 0x0D05E, 0x06824,
	0x01515, 0x0545C, 0x230E9, 0x11AFA, 0x015D7, 0x05747, 0x08D79, 0x06825,
	0x02BA2, 0x0A8BA, 0x235F6, 0x15D18, 0x011AE, 0x0346F, 0x08C3B, 0x0346E,
	0x00D1A, 0x0461F, 0x682F8, 0x11875, 0x02BA1, 0x08D61, 0x235F7, 0x230E8,
	0x01513, 0x08D7B, 0x11AF4, 0x11AF5, 0x01185, 0x046BF, 0x08D60, 0x08D7C,
	0x01512, 0x0461C, 0x0AE8D, 0x08D78, 0x00D0E, 0x03413, 0x046B1, 0x03416,
	0x00AEA, 0x02A2C, 0x05741, 0x02A2F, 0x00158, 0x008D2, 0x0054C, 0x00686,
	0x00000, 0x00069, 0x0006B, 0x0068C, 0x00007, 0x0015E, 0x002A3, 0x00AE9,
	0x00006, 0x00231, 0x002B8, 0x01A08, 0x00010, 0x001A9, 0x00342, 0x00A88,
	0x00004, 0x001A2, 0x002A4, 0x01184, 0x00012, 0x00232, 0x002B2, 0x00680,
	0x0001B, 0x0046A, 0x0068E, 0x02359, 0x00016, 0x0015F, 0x002A0, 0x0054D,
	0x00005, 0x00233, 0x002B9, 0x015D6, 0x00022, 0x00468, 0x00683, 0x01A0A,
	0x00013, 0x00236, 0x002BB, 0x01186, 0x00017, 0x001AB, 0x002A7, 0x008D3,
	0x00014, 0x00237, 0x00460, 0x00D0F, 0x00019, 0x001AA, 0x002B3, 0x00681,
	0x00018, 0x001A8, 0x002A5, 0x0068F, 0x00007, 0x00055, 0x00047, 0x000AD
};

// P-Frame macroblock block pattern, code 1 (mid rate), lengths
static const uint8_t wmv2HuffmanPMB1Lengths[128] = {
	10, 15, 18, 17, 13, 15, 17, 17,
	13, 16, 20, 19, 14, 14, 17, 16,
	13, 15, 18, 17, 13, 15, 16, 16,
	14, 16, 18, 17, 13, 15, 16, 15,
	13, 15, 20, 17, 14, 16, 18, 18,
	13, 16, 17, 17, 13, 15, 16, 16,
	13, 15, 16, 16, 13, 15, 15, 15,
	12, 14, 15, 14,  9, 12, 11, 12,
	 2,  8,  8, 12,  3,  9, 10, 12,
	 3, 10, 10, 14,  5, 10, 11, 12,
	 4, 10, 10, 13,  5, 10, 10, 12,
	 6, 11, 12, 14,  5,  9, 10, 11,
	 4, 10, 10, 13,  6, 11, 12, 14,
	 5, 10, 10, 13,  5, 10, 10, 12,
	 5, 10, 11, 13,  6, 10, 10, 12,
	 6, 10, 10, 12,  4,  7,  7,  8
};

// P-Frame macroblock block pattern, code 2 (low rate), codes
static const uint32_t wmv2HuffmanPMB2Codes[128] = {
	0x000D4, 0x021C5, 0x0F18A, 0x0D5BC, 0x00879, 0x0354D, 0x10E3F, 0x10F54,
	0x00866, 0x0356E, 0x10F55, 0x10E3E, 0x010CE, 0x03C84, 0x0D5BD, 0x0F18B,
	0x00868, 0x0438C, 0x087AB, 0x0790B, 0x00F10, 0x0433D, 0x06AD3, 0x0790A,
	0x01AA7, 0x043D4, 0x0871E, 0x06ADF, 0x00D7C, 0x03C94, 0x0438D, 0x06AD2,
	0x006BC, 0x021E9, 0x06ADA, 0x06A99, 0x010F7, 0x04389, 0x06ADB, 0x078C4,
	0x00D56, 0x035F7, 0x0438E, 0x06A98, 0x00D52, 0x03C95, 0x04388, 0x0433C,
	0x00D54, 0x01E4B, 0x03C63, 0x03C83, 0x00861, 0x021EB, 0x0356C, 0x035F6,
	0x00863, 0x0219F, 0x03568, 0x03C82, 0x001AE, 0x010C0, 0x00F11, 0x01AFA,
	0x00000, 0x000F0, 0x001AD, 0x010C1, 0x0000A, 0x003C5, 0x00789, 0x01AB5,
	0x00009, 0x00435, 0x00793, 0x01E40, 0x0001D, 0x003CB, 0x00878, 0x01AAF,
	0x0000B, 0x003C7, 0x00791, 0x01AAB, 0x0001F, 0x00436, 0x006BF, 0x00F19,
	0x0003D, 0x00D51, 0x010C4, 0x021E8, 0x00036, 0x00437, 0x006AF, 0x010C5,
	0x0000C, 0x00432, 0x00794, 0x01E30, 0x00042, 0x00870, 0x00F24, 0x01E43,
	0x00020, 0x0043E, 0x00795, 0x01AAA, 0x00037, 0x006AC, 0x006AE, 0x010F6,
	0x00034, 0x0043A, 0x00D50, 0x01AAE, 0x00039, 0x0043F, 0x0078D, 0x010D2,
	0x00038, 0x0043B, 0x006BD, 0x010D3, 0x00011, 0x001AC, 0x000F3, 0x00439
};

// P-Frame macroblock block pattern, code 2 (low rate), lengths
static const uint8_t wmv2HuffmanPMB2Lengths[128] = {
	 8, 14, 16, 16, 12, 14, 17, 17,
	12, 14, 17, 17, 13, 14, 16, 16,
	12, 15, 16, 15, 12, 15, 15, 15,
	13, 15, 16, 15, 12, 14, 15, 15,
	11, 14, 15, 15, 13, 15, 15, 15,
	12, 14, 15, 15, 12, 14, 15, 15,
	12, 13, 14, 14, 12, 14, 14, 14,
	12, 14, 14, 14,  9, 13, 12, 13,
	 1,  8,  9, 13,  4, 10, 11, 13,
	 4, 11, 11, 13,  5, 10, 12, 13,
	 4, 10, 11, 13,  5, 11, 11, 12,
	 6, 12, 13, 14,  6, 11, 11, 13,
	 4, 11, 11, 13,  7, 12, 12, 13,
	 6, 11, 11, 13,  6, 11, 11, 13,
	 6, 11, 12, 13,  6, 11, 11, 13,
	 6, 11, 11, 13,  5,  9,  8, 11
};


// DCT AC coefficients, code 0 (low motion), luma, codes
static const uint32_t wmv2HuffmanAC0LumaCodes[133] = {
	0x0001, 0x0006, 0x000F, 0x0016, 0x0020, 0x0018, 0x0008, 0x009A,
	0x0056, 0x013E, 0x00F0, 0x03A5, 0x0077, 0x01EF, 0x009A, 0x005D,
	0x0001, 0x0011, 0x0002, 0x000B, 0x0012, 0x01D6, 0x027E, 0x0191,
	0x00EA, 0x03DC, 0x013B, 0x0004, 0x0014, 0x009E, 0x0009, 0x01AC,
	0x01E2, 0x03CA, 0x005F, 0x0017, 0x004E, 0x005E, 0x00F3, 0x01AD,
	0x00EC, 0x05F0, 0x000E, 0x00E1, 0x03A4, 0x009C, 0x013D, 0x003B,
	0x001C, 0x0014, 0x09BE, 0x0006, 0x007A, 0x0190, 0x0137, 0x001B,
	0x0008, 0x075C, 0x0071, 0x00D7, 0x09BF, 0x0007, 0x00AF, 0x04CC,
	0x0034, 0x0265, 0x009F, 0x00E0, 0x0016, 0x0327, 0x0015, 0x017D,
	0x0EBB, 0x0014, 0x00F6, 0x01E4, 0x00CB, 0x099D, 0x00CA, 0x02FC,
	0x017F, 0x04CD, 0x02FD, 0x04FE, 0x013A, 0x000A, 0x0042, 0x01D3,
	0x04DD, 0x0012, 0x00E8, 0x004C, 0x0136, 0x0039, 0x0264, 0x0EBA,
	0x0000, 0x00AE, 0x099C, 0x001F, 0x04DE, 0x0043, 0x04DC, 0x0003,
	0x03CB, 0x0006, 0x099E, 0x002A, 0x05F1, 0x000F, 0x09FE, 0x0033,
	0x09FF, 0x0098, 0x099F, 0x00EA, 0x013C, 0x002E, 0x0192, 0x0136,
	0x006A, 0x0015, 0x03AF, 0x01E3, 0x0074, 0x00EB, 0x02F9, 0x005C,
	0x00ED, 0x03DD, 0x0326, 0x005E, 0x0016
};

// DCT AC coefficients, code 0 (low motion), luma, lengths
static const uint8_t wmv2HuffmanAC0LumaLengths[133] = {
	 2,  3,  4,  5,  6,  7,  8,  8,
	 9,  9, 10, 10, 11, 11, 12, 13,
	 4,  5,  7,  8,  9,  9, 10, 11,
	12, 12, 13,  5,  7,  8, 10, 11,
	11, 12, 13,  5,  7,  9, 10, 11,
	12, 13,  6,  8, 10, 12, 13,  6,
	 9, 11, 12,  7,  9, 11, 13,  7,
	10, 11,  7, 10, 12,  8, 10, 11,
	 8, 10, 12,  8, 11, 12,  9, 11,
	12,  9, 10, 11, 10, 12, 10, 12,
	11, 11, 12, 11, 13,  4,  7,  9,
	11,  5,  8, 11, 13,  6, 10, 12,
	 7, 10, 12,  7, 11,  7, 11,  8,
	12,  8, 12,  8, 13,  8, 12,  8,
	12,  8, 12,  8, 13,  8, 11,  9,
	 9, 11, 10, 11, 11, 12, 12, 13,
	12, 12, 12, 13,  7
};

// DCT AC coefficients, code 1 (high motion), luma, codes
static const uint32_t wmv2HuffmanAC1LumaCodes[186] = {
	0x0001, 0x0005, 0x000D, 0x0012, 0x000E, 0x0015, 0x0013, 0x003F,
	0x004B, 0x011F, 0x00B8, 0x03E3, 0x0172, 0x024D, 0x03DA, 0x02DD,
	0x1F55, 0x05B9, 0x3EAE, 0x0000, 0x0010, 0x0008, 0x0020, 0x0029,
	0x01F4, 0x0233, 0x01E0, 0x012A, 0x03DD, 0x050A, 0x1F29, 0x0A42,
	0x1272, 0x1737, 0x0003, 0x0011, 0x00C4, 0x004B, 0x00B4, 0x07D4,
	0x0345, 0x02D7, 0x07BF, 0x0938, 0x0BBB, 0x095E, 0x0013, 0x0078,
	0x0069, 0x0232, 0x0461, 0x03EC, 0x0520, 0x1F2A, 0x3E50, 0x3E51,
	0x1486, 0x000C, 0x0024, 0x0094, 0x08C0, 0x0F09, 0x1EF0, 0x003D,
	0x0053, 0x01A0, 0x02D6, 0x0F08, 0x0013, 0x007C, 0x07C1, 0x04AC,
	0x001B, 0x00A0, 0x0344, 0x0F79, 0x0079, 0x03E1, 0x02D4, 0x2306,
	0x0021, 0x023C, 0x0FAE, 0x23DE, 0x0035, 0x0175, 0x07B3, 0x00C5,
	0x0174, 0x0785, 0x0048, 0x01A3, 0x049E, 0x002C, 0x00FA, 0x07D6,
	0x0092, 0x05CC, 0x1EF1, 0x00A3, 0x03ED, 0x093E, 0x01E2, 0x1273,
	0x07C4, 0x1487, 0x0291, 0x0293, 0x0F8A, 0x0509, 0x0508, 0x078D,
	0x07BE, 0x078C, 0x04AE, 0x0BBA, 0x2307, 0x0B9A, 0x1736, 0x000E,
	0x0045, 0x01F3, 0x047A, 0x05DC, 0x23DF, 0x0019, 0x0028, 0x0176,
	0x049D, 0x23DD, 0x0030, 0x00A2, 0x02EF, 0x05B8, 0x003F, 0x00A5,
	0x03DB, 0x093F, 0x0044, 0x07CB, 0x095F, 0x0063, 0x03C3, 0x0015,
	0x08F6, 0x0017, 0x0498, 0x002C, 0x07B2, 0x002F, 0x1F54, 0x008D,
	0x07BD, 0x008E, 0x1182, 0x00FB, 0x050B, 0x002D, 0x07C0, 0x0079,
	0x1F5F, 0x007A, 0x1F56, 0x0231, 0x03E4, 0x01A1, 0x0143, 0x01F7,
	0x016F, 0x0292, 0x02E7, 0x016C, 0x016D, 0x03DC, 0x0F8B, 0x0499,
	0x03D8, 0x078E, 0x02D5, 0x1F5E, 0x1F2B, 0x078F, 0x04AD, 0x3EAF,
	0x23DC, 0x004A
};

// DCT AC coefficients, code 1 (high motion), luma, lengths
static const uint8_t wmv2HuffmanAC1LumaLengths[186] = {
	 2,  3,  4,  5,  6,  7,  8,  8,
	 9,  9, 10, 10, 11, 12, 12, 13,
	13, 14, 14,  4,  5,  7,  8,  9,
	 9, 10, 11, 12, 12, 13, 13, 14,
	15, 15,  5,  7,  8, 10, 11, 11,
	12, 13, 13, 14, 14, 15,  5,  7,
	 9, 10, 11, 12, 13, 13, 14, 14,
	15,  6,  9, 11, 12, 14, 15,  6,
	 9, 11, 13, 14,  7,  9, 11, 14,
	 7, 10, 12, 14,  7, 10, 13, 14,
	 8, 10, 12, 14,  8, 11, 13,  8,
	11, 13,  9, 11, 13,  9, 10, 11,
	10, 13, 15, 10, 12, 14, 11, 15,
	11, 15, 12, 12, 12, 13, 13, 13,
	13, 13, 14, 14, 14, 14, 15,  4,
	 7,  9, 11, 13, 14,  5,  9, 11,
	13, 14,  6, 10, 12, 14,  6, 10,
	12, 14,  7, 11, 15,  7, 12,  8,
	12,  8, 13,  8, 13,  8, 13,  8,
	13,  8, 13,  8, 13,  8, 11,  9,
	13,  9, 13, 10, 10, 11, 11, 11,
	12, 12, 12, 12, 12, 12, 12, 13,
	12, 13, 13, 13, 13, 13, 14, 14,
	14,  9
};

// DCT AC coefficients, code 2 (standard MPEG4), luma, codes
static const uint32_t wmv2HuffmanAC2LumaCodes[103] = {
	0x0002, 0x0006, 0x000F, 0x000D, 0x000C, 0x0015, 0x0013, 0x0012,
	0x0017, 0x001F, 0x001E, 0x001D, 0x0025, 0x0024, 0x0023, 0x0021,
	0x0021, 0x0020, 0x000F, 0x000E, 0x0007, 0x0006, 0x0020, 0x0021,
	0x0050, 0x0051, 0x0052, 0x000E, 0x0014, 0x0016, 0x001C, 0x0020,
	0x001F, 0x000D, 0x0022, 0x0053, 0x0055, 0x000B, 0x0015, 0x001E,
	0x000C, 0x0056, 0x0011, 0x001B, 0x001D, 0x000B, 0x0010, 0x0022,
	0x000A, 0x000D, 0x001C, 0x0008, 0x0012, 0x001B, 0x0054, 0x0014,
	0x001A, 0x0057, 0x0019, 0x0009, 0x0018, 0x0023, 0x0017, 0x0019,
	0x0018, 0x0007, 0x0058, 0x0007, 0x000C, 0x0016, 0x0017, 0x0006,
	0x0005, 0x0004, 0x0059, 0x000F, 0x0016, 0x0005, 0x000E, 0x0004,
	0x0011, 0x0024, 0x0010, 0x0025, 0x0013, 0x005A, 0x0015, 0x005B,
	0x0014, 0x0013, 0x001A, 0x0015, 0x0014, 0x0013, 0x0012, 0x0011,
	0x0026, 0x0027, 0x005C, 0x005D, 0x005E, 0x005F, 0x0003
};

// DCT AC coefficients, code 2 (standard MPEG4), luma, lengths
static const uint8_t wmv2HuffmanAC2LumaLengths[103] = {
	 2,  3,  4,  5,  5,  6,  6,  6,
	 7,  8,  8,  8,  9,  9,  9,  9,
	10, 10, 10, 10, 11, 11, 11, 11,
	12, 12, 12,  4,  6,  7,  8,  9,
	 9, 10, 11, 12, 12,  5,  7,  9,
	10, 12,  6,  8,  9, 10,  6,  9,
	10,  6,  9, 10,  7,  9, 12,  7,
	 9, 12,  8, 10,  8, 11,  8,  9,
	 9, 10, 12,  4,  6,  8,  9, 10,
	11, 11, 12,  6,  9, 10,  6, 10,
	 7, 11,  7, 11,  7, 12,  8, 12,
	 8,  8,  8,  9,  9,  9,  9,  9,
	11, 11, 12, 12, 12, 12,  7
};

// DCT AC coefficients, code 0 (low motion), chroma, codes
static const uint32_t wmv2HuffmanAC0ChromaCodes[149] = {
	0x0004, 0x0014, 0x0017, 0x007F, 0x0154, 0x01F2, 0x00BF, 0x0065,
	0x0AAA, 0x0630, 0x1597, 0x03B7, 0x2B22, 0x0BE6, 0x000B, 0x0037,
	0x0062, 0x0007, 0x0166, 0x00CE, 0x1590, 0x05F6, 0x0BE7, 0x0007,
	0x006D, 0x0003, 0x031F, 0x05F2, 0x0002, 0x0061, 0x0055, 0x01DF,
	0x001A, 0x001E, 0x0AC9, 0x2B23, 0x001E, 0x001F, 0x0AC3, 0x2B2B,
	0x0006, 0x0004, 0x02F8, 0x0019, 0x0006, 0x063D, 0x0057, 0x0182,
	0x2AA2, 0x0004, 0x0180, 0x059C, 0x007D, 0x0164, 0x076D, 0x0002,
	0x018D, 0x1581, 0x00AD, 0x0060, 0x0C67, 0x001C, 0x00EE, 0x0003,
	0x02CF, 0x00D9, 0x1580, 0x0002, 0x0183, 0x0057, 0x0061, 0x0031,
	0x0066, 0x0631, 0x0632, 0x00AC, 0x031D, 0x0076, 0x003A, 0x0165,
	0x0C66, 0x0003, 0x0054, 0x02AB, 0x0016, 0x05F7, 0x0005, 0x00F8,
	0x0AA9, 0x005F, 0x0004, 0x001C, 0x1550, 0x0004, 0x0077, 0x076C,
	0x000E, 0x000A, 0x000C, 0x0562, 0x0004, 0x031C, 0x0006, 0x00C8,
	0x000D, 0x01DA, 0x0007, 0x00C9, 0x0001, 0x002E, 0x0014, 0x1596,
	0x000A, 0x0AC2, 0x0016, 0x015B, 0x0015, 0x015A, 0x000F, 0x005E,
	0x007E, 0x00AB, 0x002D, 0x00D8, 0x000B, 0x0014, 0x02B3, 0x01F3,
	0x003A, 0x0000, 0x0058, 0x002E, 0x005E, 0x0563, 0x00EC, 0x0054,
	0x0AC1, 0x1556, 0x02FA, 0x0181, 0x1557, 0x059D, 0x2AA3, 0x2B2A,
	0x01DE, 0x063C, 0x00CF, 0x1594, 0x000D
};

// DCT AC coefficients, code 0 (low motion), chroma, lengths
static const uint8_t wmv2HuffmanAC0ChromaLengths[149] = {
	 3,  5,  7,  8,  9, 10, 11, 12,
	12, 13, 13, 14, 14, 15,  4,  7,
	 9, 11, 12, 13, 13, 14, 15,  5,
	 8, 11, 12, 14,  6,  9, 12, 14,
	 6, 10, 12, 14,  6, 10, 12, 14,
	 7, 11, 13,  7, 11, 13,  7, 11,
	14,  8, 11, 14,  8, 12, 15,  9,
	11, 13,  8, 12, 14,  9, 13,  9,
	13,  9, 13, 11, 11, 12, 12, 11,
	12, 13, 13, 13, 12, 12, 11, 12,
	14,  2,  7, 10, 13, 14,  4,  9,
	12, 15,  4, 10, 13,  5, 11, 15,
	 5, 12,  5, 11,  6, 12,  6, 13,
	 6, 13,  6, 13,  7, 14,  7, 13,
	 7, 12,  7, 14,  7, 14,  8, 15,
	 8,  8,  9,  9,  9, 10, 10, 10,
	10, 10, 10,  9, 10, 11, 12, 12,
	12, 13, 13, 11, 13, 14, 14, 14,
	14, 13, 13, 13,  9
};

// DCT AC coefficients, code 1 (high motion), chroma, codes
static const uint32_t wmv2HuffmanAC1ChromaCodes[169] = {
	0x0000, 0x0003, 0x000B, 0x0014, 0x003F, 0x005D, 0x00A2, 0x00AC,
	0x016E, 0x020A, 0x02E2, 0x0432, 0x05C9, 0x0827, 0x0B54, 0x04E6,
	0x105F, 0x172A, 0x20B2, 0x2D4E, 0x39F0, 0x4175, 0x5A9E, 0x0004,
	0x001E, 0x0042, 0x00B6, 0x0173, 0x0395, 0x072E, 0x0B94, 0x16A4,
	0x20B3, 0x2E45, 0x0005, 0x0040, 0x0049, 0x028F, 0x05CB, 0x048A,
	0x09DD, 0x73E2, 0x0018, 0x0025, 0x008A, 0x051B, 0x0E5F, 0x09C9,
	0x139C, 0x0029, 0x004F, 0x0412, 0x048D, 0x2E41, 0x0038, 0x010E,
	0x05A8, 0x105C, 0x39F2, 0x0058, 0x021F, 0x0E7E, 0x39FF, 0x0023,
	0x02E3, 0x04E5, 0x2E40, 0x00A1, 0x05BE, 0x09C8, 0x0083, 0x013A,
	0x1721, 0x0044, 0x0276, 0x39F6, 0x008B, 0x04EF, 0x5A9B, 0x0208,
	0x1CFE, 0x0399, 0x1CB4, 0x039E, 0x39F3, 0x05AB, 0x73E3, 0x0737,
	0x5A9F, 0x082D, 0x0E69, 0x0E68, 0x0433, 0x0B7B, 0x2DF8, 0x2E56,
	0x2E57, 0x39F7, 0x51A5, 0x0003, 0x002A, 0x00E4, 0x028E, 0x0735,
	0x1058, 0x1CFA, 0x2DF9, 0x4174, 0x0009, 0x0054, 0x0398, 0x048B,
	0x139D, 0x000D, 0x00AD, 0x0826, 0x2D4C, 0x0011, 0x016B, 0x0B7F,
	0x51A4, 0x0019, 0x021B, 0x16FD, 0x001D, 0x0394, 0x28D3, 0x002B,
	0x05BC, 0x5A9A, 0x002F, 0x0247, 0x0010, 0x0A35, 0x003E, 0x0B7A,
	0x0059, 0x105E, 0x0026, 0x09CF, 0x0055, 0x1CB5, 0x0057, 0x0E5B,
	0x00A0, 0x1468, 0x0170, 0x0090, 0x01CE, 0x021A, 0x0218, 0x0168,
	0x021E, 0x0244, 0x0736, 0x0138, 0x0519, 0x0E5E, 0x072C, 0x0B55,
	0x09DC, 0x20BB, 0x048C, 0x1723, 0x2E44, 0x16A5, 0x0518, 0x39FE,
	0x0169
};

// DCT AC coefficients, code 1 (high motion), chroma, lengths
static const uint8_t wmv2HuffmanAC1ChromaLengths[169] = {
	 3,  4,  5,  6,  6,  7,  8,  9,
	 9, 10, 10, 11, 11, 12, 12, 13,
	13, 13, 14, 14, 14, 15, 15,  4,
	 5,  7,  8,  9, 10, 11, 12, 13,
	14, 14,  5,  7,  9, 10, 11, 13,
	14, 15,  5,  8, 10, 11, 12, 14,
	15,  6,  9, 11, 13, 14,  6,  9,
	11, 13, 14,  7, 10, 12, 14,  8,
	10, 13, 14,  8, 11, 14,  8, 11,
	13,  9, 12, 14, 10, 13, 15, 10,
	13, 10, 13, 10, 14, 11, 15, 11,
	15, 12, 12, 12, 11, 12, 14, 14,
	14, 14, 15,  3,  6,  8, 10, 11,
	13, 13, 14, 15,  4,  8, 10, 13,
	15,  4,  9, 12, 14,  5,  9, 12,
	15,  5, 10, 13,  5, 10, 14,  6,
	11, 15,  6, 12,  7, 12,  6, 12,
	 7, 13,  8, 14,  8, 13,  8, 12,
	 8, 13,  9, 10,  9, 10, 10,  9,
	10, 12, 11, 11, 11, 12, 11, 12,
	14, 14, 13, 13, 14, 13, 11, 14,
	 9
};

// DCT AC coefficients, code 2 (standard MPEG4), chroma, codes
static const uint32_t wmv2HuffmanAC2ChromaCodes[103] = {
	0x0002, 0x000F, 0x0015, 0x0017, 0x001F, 0x0025, 0x0024, 0x0021,
	0x0020, 0x0007, 0x0006, 0x0020, 0x0006, 0x0014, 0x001E, 0x000F,
	0x0021, 0x0050, 0x000E, 0x001D, 0x000E, 0x0051, 0x000D, 0x0023,
	0x000D, 0x000C, 0x0022, 0x0052, 0x000B, 0x000C, 0x0053, 0x0013,
	0x000B, 0x0054, 0x0012, 0x000A, 0x0011, 0x0009, 0x0010, 0x0008,
	0x0016, 0x0055, 0x0015, 0x0014, 0x001C, 0x001B, 0x0021, 0x0020,
	0x001F, 0x001E, 0x001D, 0x001C, 0x001B, 0x001A, 0x0022, 0x0023,
	0x0056, 0x0057, 0x0007, 0x0019, 0x0005, 0x000F, 0x0004, 0x000E,
	0x000D, 0x000C, 0x0013, 0x0012, 0x0011, 0x0010, 0x001A, 0x0019,
	0x0018, 0x0017, 0x0016, 0x0015, 0x0014, 0x0013, 0x0018, 0x0017,
	0x0016, 0x0015, 0x0014, 0x0013, 0x0012, 0x0011, 0x0007, 0x0006,
	0x0005, 0x0004, 0x0024, 0x0025, 0x0026, 0x0027, 0x0058, 0x0059,
	0x005A, 0x005B, 0x005C, 0x005D, 0x005E, 0x005F, 0x0003
};

// DCT AC coefficients, code 2 (standard MPEG4), chroma, lengths
static const uint8_t wmv2HuffmanAC2ChromaLengths[103] = {
	 2,  4,  6,  7,  8,  9,  9, 10,
	10, 11, 11, 11,  3,  6,  8, 10,
	11, 12,  4,  8, 10, 12,  5,  9,
	10,  5,  9, 12,  5, 10, 12,  6,
	10, 12,  6, 10,  6, 10,  6, 10,
	 7, 12,  7,  7,  8,  8,  9,  9,
	 9,  9,  9,  9,  9,  9, 11, 11,
	12, 12,  4,  9, 11,  6, 11,  6,
	 6,  6,  7,  7,  7,  7,  8,  8,
	 8,  8,  8,  8,  8,  8,  9,  9,
	 9,  9,  9,  9,  9,  9, 10, 10,
	10, 10, 11, 11, 11, 11, 12, 12,
	12, 12, 12, 12, 12, 12,  7
};


// DCT AC coefficients run table, code 0 (low motion), luma
static const uint8_t wmv2RunTable0Luma[132] = {
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  2,  2,  2,
	 2,  2,  2,  2,  2,  3,  3,  3,  3,  3,
	 3,  3,  4,  4,  4,  4,  4,  5,  5,  5,
	 5,  6,  6,  6,  6,  7,  7,  7,  8,  8,
	 8,  9,  9,  9, 10, 10, 10, 11, 11, 11,
	12, 12, 12, 13, 13, 13, 14, 14, 15, 15,
	16, 17, 18, 19, 20,  0,  0,  0,  0,  1,
	 1,  1,  1,  2,  2,  2,  3,  3,  3,  4,
	 4,  5,  5,  6,  6,  7,  7,  8,  8,  9,
	 9, 10, 10, 11, 11, 12, 12, 13, 13, 14,
	15, 16, 17, 18, 19, 20, 21, 22, 23, 24,
	25, 26
};

// DCT AC coefficients run delta table, code 0 (low motion), luma
static const uint8_t wmv2RunDeltaTable0Luma[17] = {
	255, 20, 15, 13, 6, 4, 3, 3, 2, 1, 1, 1, 0, 0, 0, 0, 0
};

// DCT AC coefficients run delta table, last run, code 0 (low motion), luma
static const uint8_t wmv2RunDeltaTableLast0Luma[5] = {
	255, 26, 13, 3, 1
};

// DCT AC coefficients level table, code 0 (low motion), luma
static const uint8_t wmv2LevelTable0Luma[132] = {
	 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
	11, 12, 13, 14, 15, 16,  1,  2,  3,  4,
	 5,  6,  7,  8,  9, 10, 11,  1,  2,  3,
	 4,  5,  6,  7,  8,  1,  2,  3,  4,  5,
	 6,  7,  1,  2,  3,  4,  5,  1,  2,  3,
	 4,  1,  2,  3,  4,  1,  2,  3,  1,  2,
	 3,  1,  2,  3,  1,  2,  3,  1,  2,  3,
	 1,  2,  3,  1,  2,  3,  1,  2,  1,  2,
	 1,  1,  1,  1,  1,  1,  2,  3,  4,  1,
	 2,  3,  4,  1,  2,  3,  1,  2,  3,  1,
	 2,  1,  2,  1,  2,  1,  2,  1,  2,  1,
	 2,  1,  2,  1,  2,  1,  2,  1,  2,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1
};

// DCT AC coefficients level delta table, code 0 (low motion), luma
static const uint8_t wmv2LevelDeltaTable0Luma[21] = {
	16, 11,  8,  7,  5,  4,  4,  3,  3,  3,
   3,  3,  3,  3,  2,  2,  1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, last run, code 0 (low motion), luma
static const uint8_t wmv2LevelDeltaTableLast0Luma[27] = {
	4,  4,  3,  3,  2,  2,  2,  2,  2,  2,
	2,  2,  2,  2,  1,  1,  1,  1,  1,  1,
	1,  1,  1,  1,  1,  1
};

// DCT AC coefficients run table, code 1 (high motion), luma
static const uint8_t wmv2RunTable1Luma[185] = {
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  2,  2,  2,  2,  2,  2,
	 2,  2,  2,  2,  2,  2,  3,  3,  3,  3,
	 3,  3,  3,  3,  3,  3,  3,  4,  4,  4,
	 4,  4,  4,  5,  5,  5,  5,  5,  6,  6,
	 6,  6,  7,  7,  7,  7,  8,  8,  8,  8,
	 9,  9,  9,  9, 10, 10, 10, 11, 11, 11,
	12, 12, 12, 13, 13, 13, 14, 14, 14, 15,
	15, 15, 16, 16, 17, 17, 18, 19, 20, 21,
	22, 23, 24, 25, 26, 27, 28, 29, 30,  0,
	 0,  0,  0,  0,  0,  1,  1,  1,  1,  1,
	 2,  2,  2,  2,  3,  3,  3,  3,  4,  4,
	 4,  5,  5,  6,  6,  7,  7,  8,  8,  9,
	 9, 10, 10, 11, 11, 12, 12, 13, 13, 14,
	14, 15, 15, 16, 17, 18, 19, 20, 21, 22,
	23, 24, 25, 26, 27, 28, 29, 30, 31, 32,
	33, 34, 35, 36, 37
};

// DCT AC coefficients run delta table, code 1 (high motion), luma
static const uint8_t wmv2RunDeltaTable1Luma[20] = {
	255, 30, 17, 15,  9,  5,  4,  3,  3,  3,
	  3,  3,  2,  1,  1,  1,  0,  0,  0,  0
};

// DCT AC coefficients run delta table, last run, code 1 (high motion), luma
static const uint8_t wmv2RunDeltaTableLast1Luma[7] = {
	255, 37, 15,  4,  3,  1, 0
};

// DCT AC coefficients level table, code 1 (high motion), luma
static const uint8_t wmv2LevelTable1Luma[185] = {
	 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
	11, 12, 13, 14, 15, 16, 17, 18, 19,  1,
	 2,  3,  4,  5,  6,  7,  8,  9, 10, 11,
	12, 13, 14, 15,  1,  2,  3,  4,  5,  6,
	 7,  8,  9, 10, 11, 12,  1,  2,  3,  4,
	 5,  6,  7,  8,  9, 10, 11,  1,  2,  3,
	 4,  5,  6,  1,  2,  3,  4,  5,  1,  2,
	 3,  4,  1,  2,  3,  4,  1,  2,  3,  4,
	 1,  2,  3,  4,  1,  2,  3,  1,  2,  3,
	 1,  2,  3,  1,  2,  3,  1,  2,  3,  1,
	 2,  3,  1,  2,  1,  2,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 2,  3,  4,  5,  6,  1,  2,  3,  4,  5,
	 1,  2,  3,  4,  1,  2,  3,  4,  1,  2,
	 3,  1,  2,  1,  2,  1,  2,  1,  2,  1,
	 2,  1,  2,  1,  2,  1,  2,  1,  2,  1,
	 2,  1,  2,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, code 1 (high motion), luma
static const uint8_t wmv2LevelDeltaTable1Luma[31] = {
	19, 15, 12, 11,  6,  5,  4,  4,  4,  4,
	 3,  3,  3,  3,  3,  3,  2,  2,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1
};

// DCT AC coefficients level delta table, last run, code 1 (high motion), luma
static const uint8_t wmv2LevelDeltaTableLast1Luma[38] = {
	 6,  5,  4,  4,  3,  2,  2,  2,  2,  2,  2,
	 2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1
};

// DCT AC coefficients run table, code 2 (standard MPEG), luma
static const uint8_t wmv2RunTable2Luma[102] = {
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  2,  2,  2,
	 2,  2,  3,  3,  3,  3,  4,  4,  4,  5,
	 5,  5,  6,  6,  6,  7,  7,  7,  8,  8,
	 9,  9, 10, 11, 12, 13, 14,  0,  0,  0,
	 0,  0,  0,  0,  0,  1,  1,  1,  2,  2,
	 3,  3,  4,  4,  5,  5,  6,  6,  7,  8,
	 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
	19, 20
};

// DCT AC coefficients run delta table, code 2 (standard MPEG), luma
static const uint8_t wmv2RunDeltaTable2Luma[28] = {
	255, 14,  9,  7,  3,  2,  1,  1,  1,  1,
	  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0
};

// DCT AC coefficients run delta table, last run, code 2 (standard MPEG), luma
static const uint8_t wmv2RunDeltaTableLast2Luma[9] = {
	255, 20, 6, 1, 0, 0, 0, 0, 0
};

// DCT AC coefficients level table, code 2 (standard MPEG), luma
static const uint8_t wmv2LevelTable2Luma[102] = {
	 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
	11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
	21, 22, 23, 24, 25, 26, 27,  1,  2,  3,
	 4,  5,  6,  7,  8,  9, 10,  1,  2,  3,
	 4,  5,  1,  2,  3,  4,  1,  2,  3,  1,
	 2,  3,  1,  2,  3,  1,  2,  3,  1,  2,
	 1,  2,  1,  1,  1,  1,  1,  1,  2,  3,
	 4,  5,  6,  7,  8,  1,  2,  3,  1,  2,
	 1,  2,  1,  2,  1,  2,  1,  2,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1
};

// DCT AC coefficients level delta table, code 2 (standard MPEG), luma
static const uint8_t wmv2LevelDeltaTable2Luma[15] = {
	27, 10,  5,  4,  3,  3,  3,  3,  2,  2,
	 1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, last run, code 2 (standard MPEG), luma
static const uint8_t wmv2LevelDeltaTableLast2Luma[21] = {
	 8,  3,  2,  2,  2,  2,  2,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1
};

// DCT AC coefficients run table, code 0 (low motion), chroma
static const uint8_t wmv2RunTable0Chroma[148] = {
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  2,  2,  2,  2,  2,  3,  3,
	 3,  3,  4,  4,  4,  4,  5,  5,  5,  5,
	 6,  6,  6,  7,  7,  7,  8,  8,  8,  9,
	 9,  9, 10, 10, 10, 11, 11, 11, 12, 12,
	12, 13, 13, 14, 14, 15, 15, 16, 17, 18,
	19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
	29,  0,  0,  0,  0,  0,  1,  1,  1,  1,
	 2,  2,  2,  3,  3,  3,  4,  4,  5,  5,
	 6,  6,  7,  7,  8,  8,  9,  9, 10, 10,
	11, 11, 12, 12, 13, 13, 14, 14, 15, 15,
	16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
	26, 27, 28, 29, 30, 31, 32, 33, 34, 35,
	36, 37, 38, 39, 40, 41, 42, 43
};

// DCT AC coefficients run delta table, code 0 (low motion), chroma
static const uint8_t wmv2RunDeltaTable0Chroma[15] = {
	255, 29, 15, 12,  5,  2,  1,  1,  1,  1,
	  0,  0,  0,  0,  0
};

// DCT AC coefficients run delta table, last run, code 0 (low motion), chroma
static const uint8_t wmv2RunDeltaTableLast0Chroma[6] = {
	255, 43, 15,  3,  1,  0
};

// DCT AC coefficients level table, code 0 (low motion), chroma
static const uint8_t wmv2LevelTable0Chroma[148] = {
	 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
	11, 12, 13, 14,  1,  2,  3,  4,  5,  6,
	 7,  8,  9,  1,  2,  3,  4,  5,  1,  2,
	 3,  4,  1,  2,  3,  4,  1,  2,  3,  4,
	 1,  2,  3,  1,  2,  3,  1,  2,  3,  1,
	 2,  3,  1,  2,  3,  1,  2,  3,  1,  2,
	 3,  1,  2,  1,  2,  1,  2,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  2,  3,  4,  5,  1,  2,  3,  4,
	 1,  2,  3,  1,  2,  3,  1,  2,  1,  2,
	 1,  2,  1,  2,  1,  2,  1,  2,  1,  2,
	 1,  2,  1,  2,  1,  2,  1,  2,  1,  2,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, code 0 (low motion), chroma
static const uint8_t wmv2LevelDeltaTable0Chroma[30] = {
	14,  9,  5,  4,  4,  4,  3,  3,  3,  3,
	 3,  3,  3,  2,  2,  2,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, last run, code 0 (low motion), chroma
static const uint8_t wmv2LevelDeltaTableLast0Chroma[44] = {
	 5,  4,  3,  3,  2,  2,  2,  2,  2,  2,
	 2,  2,  2,  2,  2,  2,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1
};

// DCT AC coefficients run table, code 1 (high motion), chroma
static const uint8_t wmv2RunTable1Chroma[168] = {
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  2,  2,  2,  2,  2,  2,
	 2,  2,  3,  3,  3,  3,  3,  3,  3,  4,
	 4,  4,  4,  4,  5,  5,  5,  5,  5,  6,
	 6,  6,  6,  7,  7,  7,  7,  8,  8,  8,
	 9,  9,  9, 10, 10, 10, 11, 11, 11, 12,
	12, 13, 13, 14, 14, 15, 15, 16, 16, 17,
	18, 19, 20, 21, 22, 23, 24, 25, 26,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,  1,  1,
	 1,  1,  1,  2,  2,  2,  2,  3,  3,  3,
	 3,  4,  4,  4,  5,  5,  5,  6,  6,  6,
	 7,  7,  8,  8,  9,  9, 10, 10, 11, 11,
	12, 12, 13, 13, 14, 14, 15, 16, 17, 18,
	19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
	29, 30, 31, 32, 33, 34, 35, 36
};

// DCT AC coefficients run delta table, code 1 (high motion), chroma
static const uint8_t wmv2RunDeltaTable1Chroma[24] = {
	255, 26, 16, 11,  7,  5,  3,  3,  2,  1,
	  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0
};

// DCT AC coefficients run delta table, last run, code 1 (high motion), chroma
static const uint8_t wmv2RunDeltaTableLast1Chroma[10] = {
	255, 36, 14,  6,  3,  1,  0,  0,  0,  0
};

// DCT AC coefficients level table, code 1 (high motion), chroma
static const uint8_t wmv2LevelTable1Chroma[168] = {
	 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
	11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
	21, 22, 23,  1,  2,  3,  4,  5,  6,  7,
	 8,  9, 10, 11,  1,  2,  3,  4,  5,  6,
	 7,  8,  1,  2,  3,  4,  5,  6,  7,  1,
	 2,  3,  4,  5,  1,  2,  3,  4,  5,  1,
	 2,  3,  4,  1,  2,  3,  4,  1,  2,  3,
	 1,  2,  3,  1,  2,  3,  1,  2,  3,  1,
	 2,  1,  2,  1,  2,  1,  2,  1,  2,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 2,  3,  4,  5,  6,  7,  8,  9,  1,  2,
	 3,  4,  5,  1,  2,  3,  4,  1,  2,  3,
	 4,  1,  2,  3,  1,  2,  3,  1,  2,  3,
	 1,  2,  1,  2,  1,  2,  1,  2,  1,  2,
	 1,  2,  1,  2,  1,  2,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, code 1 (high motion), chroma
static const uint8_t wmv2LevelDeltaTable1Chroma[27] = {
	23, 11,  8,  7,  5,  5,  4,  4,  3,  3,
	 3,  3,  2,  2,  2,  2,  2,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, last run, code 1 (high motion), chroma
static const uint8_t wmv2LevelDeltaTableLast1Chroma[37] = {
	 9,  5,  4,  4,  3,  3,  3,  2,  2,  2,
	 2,  2,  2,  2,  2,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1
};

// DCT AC coefficients run table, code 2 (standard MPEG), chroma
static const uint8_t wmv2RunTable2Chroma[102] = {
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  1,  1,  1,  1,  1,  1,  2,  2,
	 2,  2,  3,  3,  3,  4,  4,  4,  5,  5,
	 5,  6,  6,  6,  7,  7,  8,  8,  9,  9,
	10, 10, 11, 12, 13, 14, 15, 16, 17, 18,
	19, 20, 21, 22, 23, 24, 25, 26,  0,  0,
	 0,  1,  1,  2,  3,  4,  5,  6,  7,  8,
	 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
	19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
	29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
	39, 40
};

// DCT AC coefficients run delta table, code 2 (standard MPEG), chroma
static const uint8_t wmv2RunDeltaTable2Chroma[13] = {
	255, 26, 10,  6,  2,  1,  1,  0,  0,  0,
	  0,  0,  0
};

// DCT AC coefficients run delta table, last run, code 2 (standard MPEG), chroma
static const uint8_t wmv2RunDeltaTableLast2Chroma[4] = {
	255, 40,  1,  0
};

// DCT AC coefficients level table, code 2 (standard MPEG), chroma
static const uint8_t wmv2LevelTable2Chroma[102] = {
	 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
	11, 12,  1,  2,  3,  4,  5,  6,  1,  2,
	 3,  4,  1,  2,  3,  1,  2,  3,  1,  2,
	 3,  1,  2,  3,  1,  2,  1,  2,  1,  2,
	 1,  2,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  2,
	 3,  1,  2,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1
};

// DCT AC coefficients level delta table, code 2 (standard MPEG), chroma
static const uint8_t wmv2LevelDeltaTable2Chroma[27] = {
	12,  6,  4,  3,  3,  3,  3,  2,  2,  2,
	 2,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1
};

// DCT AC coefficients level delta table, last run, code 2 (standard MPEG), chroma
static const uint8_t wmv2LevelDeltaTableLast2Chroma[41] = {
	 3,  2,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	 1
};


// I-Frame coefficient zig-zag scantable, horizontal
const uint8_t wmv2ZigZagHorizontal[64] = {
	0x00, 0x01, 0x08, 0x02, 0x03, 0x09, 0x10, 0x18,
	0x11, 0x0A, 0x04, 0x05, 0x0B, 0x12, 0x19, 0x20,
	0x28, 0x30, 0x21, 0x1A, 0x13, 0x0C, 0x06, 0x07,
	0x0D, 0x14, 0x1B, 0x22, 0x29, 0x38, 0x31, 0x39,
	0x2A, 0x23, 0x1C, 0x15, 0x0E, 0x0F, 0x16, 0x1D,
	0x24, 0x2B, 0x32, 0x3A, 0x33, 0x2C, 0x25, 0x1E,
	0x17, 0x1F, 0x26, 0x2D, 0x34, 0x3B, 0x3C, 0x35,
	0x2E, 0x27, 0x2F, 0x36, 0x3D, 0x3E, 0x37, 0x3F
};

// I-Frame coefficient zig-zag scantable, vertical
const uint8_t wmv2ZigZagVertical[64] = {
	0x00, 0x08, 0x10, 0x01, 0x18, 0x20, 0x28, 0x09,
	0x02, 0x03, 0x0A, 0x11, 0x19, 0x30, 0x38, 0x29,
	0x21, 0x1A, 0x12, 0x0B, 0x04, 0x05, 0x0C, 0x13,
	0x1B, 0x22, 0x31, 0x39, 0x32, 0x2A, 0x23, 0x1C,
	0x14, 0x0D, 0x06, 0x07, 0x0E, 0x15, 0x1D, 0x24,
	0x2B, 0x33, 0x3A, 0x3B, 0x34, 0x2C, 0x25, 0x1E,
	0x16, 0x0F, 0x17, 0x1F, 0x26, 0x2D, 0x3C, 0x35,
	0x2E, 0x27, 0x2F, 0x36, 0x3D, 0x3E, 0x37, 0x3F
};

// I-Frame coefficient zig-zag scantable, normal
const uint8_t wmv2ZigZagNormal[64] = {
	0x00, 0x08, 0x01, 0x02, 0x09, 0x10, 0x18, 0x11,
	0x0A, 0x03, 0x04, 0x0B, 0x12, 0x19, 0x20, 0x28,
	0x21, 0x30, 0x1A, 0x13, 0x0C, 0x05, 0x06, 0x0D,
	0x14, 0x1B, 0x22, 0x29, 0x38, 0x31, 0x39, 0x2A,
	0x23, 0x1C, 0x15, 0x0E, 0x07, 0x0F, 0x16, 0x1D,
	0x24, 0x2B, 0x32, 0x3A, 0x33, 0x3B, 0x2C, 0x25,
	0x1E, 0x17, 0x1F, 0x26, 0x2D, 0x34, 0x3C, 0x35,
	0x3D, 0x2E, 0x27, 0x2F, 0x36, 0x3E, 0x37, 0x3F
};

// P-Frame coefficient zig-zag scantable, 8x8
const uint8_t wmv2ZigZag8x8[64] = {
	 0,  1,  8, 16,  9,  2,  3, 10,
	17, 24, 32, 25, 18, 11,  4,  5,
	 6,  7, 13, 12, 19, 26, 33, 40,
	48, 41, 34, 27, 20, 14, 15, 23,
	22, 21, 28, 35, 42, 49, 56, 57,
	50, 43, 36, 29, 30, 31, 39, 38,
	37, 44, 51, 58, 59, 52, 45, 46,
	47, 55, 54, 53, 60, 61, 62, 63
};

// P-Frame coefficient zig-zag scantable, 8x4
const uint8_t wmv2ZigZag8x4[32] = {
	 0,  1,  2,  8,  3,  9, 10, 16,
	 4, 11, 17, 24, 18, 12,  5, 19,
	25, 13, 20, 26, 27,  6, 21, 28,
	14, 22, 29,  7, 30, 15, 23, 31
};

// P-Frame coefficient zig-zag scantable, 4x8
const uint8_t wmv2ZigZag4x8[32] = {
	 0,  4,  1,  8,  5, 12,  9,  2,
	16,  6, 13, 20, 10, 24, 17, 14,
	28, 21, 18,  3, 25, 29,  7, 22,
	11, 26, 15, 30, 19, 23, 27, 31
};

// I-Frame macroblock block pattern
const Common::HuffmanTable wmv2HuffmanIMB = {
	0,                              // Max code length, automatic
	ARRAYSIZE(wmv2HuffmanIMBCodes), // Number of codes
	wmv2HuffmanIMBCodes,            // Codes
	wmv2HuffmanIMBLengths,          // Lengths
	0                               // Symbols, identical to codes
};

// DCT DC coefficients
const Common::HuffmanTable wmv2HuffmanDC[2][2] = {
	// Luma
	{
		// Low motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanDC0LumaCodes),   // Number of codes
			wmv2HuffmanDC0LumaCodes,              // Codes
			wmv2HuffmanDC0LumaLengths,            // Lengths
			0                                     // Symbols, identical to codes
		},

		// High motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanDC1LumaCodes),   // Number of codes
			wmv2HuffmanDC1LumaCodes,              // Codes
			wmv2HuffmanDC1LumaLengths,            // Lengths
			0                                     // Symbols, identical to codes
		}
	},

	// Chroma
	{
		// Low motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanDC0ChromaCodes), // Number of codes
			wmv2HuffmanDC0ChromaCodes,            // Codes
			wmv2HuffmanDC0ChromaLengths,          // Lengths
			0                                     // Symbols, identical to codes
		},

		// High motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanDC1ChromaCodes), // Number of codes
			wmv2HuffmanDC1ChromaCodes,            // Codes
			wmv2HuffmanDC1ChromaLengths,          // Lengths
			0                                     // Symbols, identical to codes
		}
	}
};

const Common::HuffmanTable wmv2HuffmanPMB[3] = {
	// High rate
	{
		0,                               // Max code length, automatic
		ARRAYSIZE(wmv2HuffmanPMB0Codes), // Number of codes
		wmv2HuffmanPMB0Codes,            // Codes
		wmv2HuffmanPMB0Lengths,          // Lengths
		0                                // Symbols, identical to codes
	},

	// Mid rate
	{
		0,                               // Max code length, automatic
		ARRAYSIZE(wmv2HuffmanPMB1Codes), // Number of codes
		wmv2HuffmanPMB1Codes,            // Codes
		wmv2HuffmanPMB1Lengths,          // Lengths
		0                                // Symbols, identical to codes
	},

	// Low rate
	{
		0,                               // Max code length, automatic
		ARRAYSIZE(wmv2HuffmanPMB2Codes), // Number of codes
		wmv2HuffmanPMB2Codes,            // Codes
		wmv2HuffmanPMB2Lengths,          // Lengths
		0                                // Symbols, identical to codes
	}
};

static const Common::HuffmanTable wmv2HuffmanAC[2][3] = {
	// Luma
	{
		// Low motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanAC0LumaCodes),   // Number of codes
			wmv2HuffmanAC0LumaCodes,              // Codes
			wmv2HuffmanAC0LumaLengths,            // Lengths
			0                                     // Symbols, identical to codes
		},

		// High motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanAC1LumaCodes),   // Number of codes
			wmv2HuffmanAC1LumaCodes,              // Codes
			wmv2HuffmanAC1LumaLengths,            // Lengths
			0                                     // Symbols, identical to codes
		},

		// Standard MPEG4
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanAC2LumaCodes),   // Number of codes
			wmv2HuffmanAC2LumaCodes,              // Codes
			wmv2HuffmanAC2LumaLengths,            // Lengths
			0                                     // Symbols, identical to codes
		}
	},

	// Chroma
	{
		// Low motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanAC0ChromaCodes), // Number of codes
			wmv2HuffmanAC0ChromaCodes,            // Codes
			wmv2HuffmanAC0ChromaLengths,          // Lengths
			0                                     // Symbols, identical to codes
		},

		// High motion
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanAC1ChromaCodes), // Number of codes
			wmv2HuffmanAC1ChromaCodes,            // Codes
			wmv2HuffmanAC1ChromaLengths,          // Lengths
			0                                     // Symbols, identical to codes
		},

		// Standard MPEG4
		{
			0,                                    // Max code length, automatic
			ARRAYSIZE(wmv2HuffmanAC2ChromaCodes), // Number of codes
			wmv2HuffmanAC2ChromaCodes,            // Codes
			wmv2HuffmanAC2ChromaLengths,          // Lengths
			0                                     // Symbols, identical to codes
		}
	}
};

/** Motion vectors [low/high motion]. */
static const Common::HuffmanTable wmv2HuffmanMV[2] = {
	// Low motion
	{
		0,                              // Max code length, automatic
		ARRAYSIZE(wmv2HuffmanMV0Codes), // Number of codes
		wmv2HuffmanMV0Codes,            // Codes
		wmv2HuffmanMV0Lengths,          // Lengths
		0                               // Symbols, identical to codes
	},

	// High motion
	{
		0,                              // Max code length, automatic
		ARRAYSIZE(wmv2HuffmanMV1Codes), // Number of codes
		wmv2HuffmanMV1Codes,            // Codes
		wmv2HuffmanMV1Lengths,          // Lengths
		0                               // Symbols, identical to codes
	}
};

const WMV2ACCoefficientTable wmv2AC[2][3] = {
	// Luma
	{
		// Low motion
		{
			wmv2HuffmanAC[0][0],           // Huffman decoder
			132,                           // Escape code
			85,                            // Index of last run
			wmv2RunTable0Luma,             // Run table
			wmv2RunDeltaTable0Luma,        // Run delta tableg
			wmv2RunDeltaTableLast0Luma,    // Run delta table, last run
			wmv2LevelTable0Luma,           // Level table
			wmv2LevelDeltaTable0Luma,      // Level delta tableg
			wmv2LevelDeltaTableLast0Luma   // Level delta table, last run
		},

		// High motion
		{
			wmv2HuffmanAC[0][1],           // Huffman decoder
			185,                           // Escape code
			119,                           // Index of last run
			wmv2RunTable1Luma,             // Run table
			wmv2RunDeltaTable1Luma,        // Run delta tableg
			wmv2RunDeltaTableLast1Luma,    // Run delta table, last run
			wmv2LevelTable1Luma,           // Level table
			wmv2LevelDeltaTable1Luma,      // Level delta tableg
			wmv2LevelDeltaTableLast1Luma   // Level delta table, last run
		},

		// Standard MPEG4
		{
			wmv2HuffmanAC[0][2],           // Huffman decoder
			102,                           // Escape code
			67,                            // Index of last run
			wmv2RunTable2Luma,             // Run table
			wmv2RunDeltaTable2Luma,        // Run delta tableg
			wmv2RunDeltaTableLast2Luma,    // Run delta table, last run
			wmv2LevelTable2Luma,           // Level table
			wmv2LevelDeltaTable2Luma,      // Level delta tableg
			wmv2LevelDeltaTableLast2Luma   // Level delta table, last run
		}
	},

	// Chroma
	{
		// Low motion
		{
			wmv2HuffmanAC[1][0],           // Huffman decoder
			148,                           // Escape code
			81,                            // Index of last run
			wmv2RunTable0Chroma,           // Run table
			wmv2RunDeltaTable0Chroma,      // Run delta tableg
			wmv2RunDeltaTableLast0Chroma,  // Run delta table, last run
			wmv2LevelTable0Chroma,         // Level table
			wmv2LevelDeltaTable0Chroma,    // Level delta tableg
			wmv2LevelDeltaTableLast0Chroma // Level delta table, last run
		},

		// High motion
		{
			wmv2HuffmanAC[1][1],           // Huffman decoder
			168,                           // Escape code
			99,                            // Index of last run
			wmv2RunTable1Chroma,           // Run table
			wmv2RunDeltaTable1Chroma,      // Run delta tableg
			wmv2RunDeltaTableLast1Chroma,  // Run delta table, last run
			wmv2LevelTable1Chroma,         // Level table
			wmv2LevelDeltaTable1Chroma,    // Level delta tableg
			wmv2LevelDeltaTableLast1Chroma // Level delta table, last run
		},

		// Standard MPEG4
		{
			wmv2HuffmanAC[1][2],           // Huffman decoder
			102,                           // Escape code
			58,                            // Index of last run
			wmv2RunTable2Chroma,           // Run table
			wmv2RunDeltaTable2Chroma,      // Run delta tableg
			wmv2RunDeltaTableLast2Chroma,  // Run delta table, last run
			wmv2LevelTable2Chroma,         // Level table
			wmv2LevelDeltaTable2Chroma,    // Level delta tableg
			wmv2LevelDeltaTableLast2Chroma // Level delta table, last run
		}
	}
};

const WMV2MVTable wmv2MV[2] = {
	// Low motion
	{
		1099,             // Number of motion vectors
		wmv2HuffmanMV[0], // Huffman decoder
		wmv2MV0DiffX,     // X difference
		wmv2MV0DiffY      // Y difference
	},

	// High motion
	{
		1099,             // Number of motion vectors
		wmv2HuffmanMV[1], // Huffman decoder
		wmv2MV1DiffX,     // X difference
		wmv2MV1DiffY      // Y difference
	}
};

} // End of namespace Video
