/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Tables defining the engine functions in The Witcher.
 */

/* The functions are defined by three tables:
 * - kFunctionPointers
 * - kFunctionSignatures
 * - kFunctionDefaults
 *
 * kFunctionPointers provides the ID and name of the engine function, and a
 * pointer to Functions method doing the actual work. If the function pointer
 * is 0, a default unimplementedFunction method is used that just prints
 * the name of the function when it's called.
 *
 * kFunctionSignatures defines the signature of the function, i.e. the types
 * of the return value and its parameters.
 *
 * kFunctionDefaults pins default values to parameters. I.e. if a function
 * Foobar(int mode, int doEverything=FALSE) is called with only one parameters,
 * the default value FALSE is applied to the doEverything parameters. The
 * kFunctionDefaults entry for the function then contains a reference to a
 * false value. Because of ambiguities otherwise, the default values have to be
 * "aligned" to the right; only the last-most n parameters are allowed to have
 * default values, with no gaps.
 *
 * Please note that all three tables have to be of the same length, and that
 * the order of the functions have to be the same in all three tables.
 */

#ifndef ENGINES_WITCHER_NWSCRIPT_FUNCTION_TABLES_H
#define ENGINES_WITCHER_NWSCRIPT_FUNCTION_TABLES_H

#include "src/engines/witcher/types.h"
#include "src/engines/witcher/object.h"

namespace Engines {

namespace Witcher {

using Aurora::NWScript::kTypeVoid;
using Aurora::NWScript::kTypeInt;
using Aurora::NWScript::kTypeFloat;
using Aurora::NWScript::kTypeString;
using Aurora::NWScript::kTypeObject;
using Aurora::NWScript::kTypeVector;
using Aurora::NWScript::kTypeEngineType;
using Aurora::NWScript::kTypeScriptState;
using Aurora::NWScript::kTypeAny;

/* Default values as used by the function parameters, if no explicit value
 * is given at the time of the calling.
 */

static Witcher::Object kDefaultValueObjectInvalid(kObjectTypeInvalid);
static Witcher::Object kDefaultValueObjectSelf   (kObjectTypeSelf);

static const Aurora::NWScript::Variable kDefaultIntMinus1((int32_t) - 1);
static const Aurora::NWScript::Variable kDefaultInt0     ((int32_t)   0);
static const Aurora::NWScript::Variable kDefaultInt1     ((int32_t)   1);
static const Aurora::NWScript::Variable kDefaultInt9     ((int32_t)   9);
static const Aurora::NWScript::Variable kDefaultInt18    ((int32_t)  18);

static const Aurora::NWScript::Variable kDefaultFloatMinus1_0(- 1.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_0     (  0.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_5     (  0.5f);
static const Aurora::NWScript::Variable kDefaultFloat1_0     (  1.0f);
static const Aurora::NWScript::Variable kDefaultFloat30_0    ( 30.0f);
static const Aurora::NWScript::Variable kDefaultFloat40_0    ( 40.0f);

static const Aurora::NWScript::Variable kDefaultVector0(0.0f, 0.0f, 0.0f);

static const Aurora::NWScript::Variable kDefaultStringEmpty(Common::UString(""));

static const Aurora::NWScript::Variable kDefaultObjectInvalid(&kDefaultValueObjectInvalid);
static const Aurora::NWScript::Variable kDefaultObjectSelf   (&kDefaultValueObjectSelf);

// TODO: Add the relevant enums to types.h, and use these values.

static const Aurora::NWScript::Variable kDefaultCameraTransitionTypeSnap ((int32_t)     0);
static const Aurora::NWScript::Variable kDefaultFalse                    ((int32_t)     0);
static const Aurora::NWScript::Variable kDefaultGenderMale               ((int32_t)     0);
static const Aurora::NWScript::Variable kDefaultObjectTypeAll            ((int32_t) 32767);
static const Aurora::NWScript::Variable kDefaultObjectTypeCreature       ((int32_t)     1);
static const Aurora::NWScript::Variable kDefaultPersistentZoneActive     ((int32_t)     0);
static const Aurora::NWScript::Variable kDefaultProjectilePathTypeDefault((int32_t)     0);
static const Aurora::NWScript::Variable kDefaultTalkVolumeTalk           ((int32_t)     0);
static const Aurora::NWScript::Variable kDefaultTrue                     ((int32_t)     1);

static const Aurora::NWScript::Variable kDefaultFadeSpeedMedium          (2.0f);


/** The table defining the name and function pointer of each engine function. */
const Functions::FunctionPointer Functions::kFunctionPointers[] = {
	{   0, "Random"                            , &Functions::random                             },
	{   1, "PrintString"                       , &Functions::printString                        },
	{   2, "PrintFloat"                        , &Functions::printFloat                         },
	{   3, "FloatToString"                     , &Functions::floatToString                      },
	{   4, "PrintInteger"                      , &Functions::printInteger                       },
	{   5, "PrintObject"                       , &Functions::printObject                        },
	{   6, "AssignCommand"                     , &Functions::assignCommand                      },
	{   7, "DelayCommand"                      , &Functions::delayCommand                       },
	{   8, "ExecuteScript"                     , &Functions::executeScript                      },
	{   9, "ClearAllActions"                   , 0                                              },
	{  10, "SetFacing"                         , 0                                              },
	{  11, "SetCalendar"                       , 0                                              },
	{  12, "SetTime"                           , 0                                              },
	{  13, "GetCalendarYear"                   , 0                                              },
	{  14, "GetCalendarMonth"                  , 0                                              },
	{  15, "GetCalendarDay"                    , 0                                              },
	{  16, "GetTimeHour"                       , 0                                              },
	{  17, "GetTimeMinute"                     , 0                                              },
	{  18, "GetTimeSecond"                     , 0                                              },
	{  19, "GetTimeMillisecond"                , 0                                              },
	{  20, "ActionRandomWalk"                  , 0                                              },
	{  21, "ActionMoveToLocation"              , &Functions::actionMoveToLocation               },
	{  22, "ActionMoveToObject"                , &Functions::actionMoveToObject                 },
	{  23, "ActionMoveAwayFromObject"          , 0                                              },
	{  24, "GetArea"                           , &Functions::getArea                            },
	{  25, "GetEnteringObject"                 , &Functions::getEnteringObject                  },
	{  26, "GetExitingObject"                  , &Functions::getExitingObject                   },
	{  27, "GetPosition"                       , &Functions::getPosition                        },
	{  28, "GetFacing"                         , 0                                              },
	{  29, "GetItemPossessor"                  , 0                                              },
	{  30, "GetItemPossessedBy"                , 0                                              },
	{  31, "CreateItemOnObject"                , 0                                              },
	{  32, "ActionEquipItem"                   , 0                                              },
	{  33, "ActionUnequipItem"                 , 0                                              },
	{  34, "ActionPickUpItem"                  , 0                                              },
	{  35, "ActionPutDownItem"                 , 0                                              },
	{  36, "GetLastAttacker"                   , 0                                              },
	{  37, "ActionAttack"                      , 0                                              },
	{  38, "GetNearestCreature"                , &Functions::getNearestCreature                 },
	{  39, "ActionSpeakString"                 , &Functions::actionSpeakString                  },
	{  40, "ActionPlayAnimation"               , 0                                              },
	{  41, "GetDistanceToObject"               , &Functions::getDistanceToObject                },
	{  42, "GetIsObjectValid"                  , &Functions::getIsObjectValid                   },
	{  43, "ActionOpenDoor"                    , &Functions::actionOpenDoor                     },
	{  44, "ActionCloseDoor"                   , &Functions::actionCloseDoor                    },
	{  45, "SetCameraFacing"                   , 0                                              },
	{  46, "PlaySound"                         , &Functions::playSound                          },
	{  47, "GetSpellTargetObject"              , 0                                              },
	{  48, "ActionCastSpellAtObject"           , 0                                              },
	{  49, "GetCurrentVitalityPoints"          , 0                                              },
	{  50, "GetMaxVitalityPoints"              , 0                                              },
	{  51, "GetLocalInt"                       , &Functions::getLocalInt                        },
	{  52, "GetLocalFloat"                     , &Functions::getLocalFloat                      },
	{  53, "GetLocalString"                    , &Functions::getLocalString                     },
	{  54, "GetLocalObject"                    , &Functions::getLocalObject                     },
	{  55, "SetLocalInt"                       , &Functions::setLocalInt                        },
	{  56, "SetLocalFloat"                     , &Functions::setLocalFloat                      },
	{  57, "SetLocalString"                    , &Functions::setLocalString                     },
	{  58, "SetLocalObject"                    , &Functions::setLocalObject                     },
	{  59, "GetStringLength"                   , &Functions::getStringLength                    },
	{  60, "GetStringUpperCase"                , &Functions::getStringUpperCase                 },
	{  61, "GetStringLowerCase"                , &Functions::getStringLowerCase                 },
	{  62, "GetStringRight"                    , &Functions::getStringRight                     },
	{  63, "GetStringLeft"                     , &Functions::getStringLeft                      },
	{  64, "InsertString"                      , &Functions::insertString                       },
	{  65, "GetSubString"                      , &Functions::getSubString                       },
	{  66, "FindSubString"                     , &Functions::findSubString                      },
	{  67, "fabs"                              , &Functions::fabs                               },
	{  68, "cos"                               , &Functions::cos                                },
	{  69, "sin"                               , &Functions::sin                                },
	{  70, "tan"                               , &Functions::tan                                },
	{  71, "acos"                              , &Functions::acos                               },
	{  72, "asin"                              , &Functions::asin                               },
	{  73, "atan"                              , &Functions::atan                               },
	{  74, "log"                               , &Functions::log                                },
	{  75, "pow"                               , &Functions::pow                                },
	{  76, "sqrt"                              , &Functions::sqrt                               },
	{  77, "abs"                               , &Functions::abs                                },
	{  78, "EffectHeal"                        , 0                                              },
	{  79, "EffectDamage"                      , 0                                              },
	{  80, "GetCurrentScriptSpellID"           , 0                                              },
	{  81, "GetNumRunningEffectsApplied"       , 0                                              },
	{  82, "EffectResurrection"                , 0                                              },
	{  83, "EffectSummonCreature"              , 0                                              },
	{  84, "AddAbility"                        , 0                                              },
	{  85, "GetFirstEffect"                    , 0                                              },
	{  86, "GetNextEffect"                     , 0                                              },
	{  87, "RemoveEffect"                      , 0                                              },
	{  88, "GetIsEffectValid"                  , 0                                              },
	{  89, "GetEffectDurationType"             , 0                                              },
	{  90, "GetEffectSubType"                  , 0                                              },
	{  91, "GetEffectCreator"                  , 0                                              },
	{  92, "IntToString"                       , &Functions::intToString                        },
	{  93, "GetFirstObjectInArea"              , 0                                              },
	{  94, "GetNextObjectInArea"               , 0                                              },
	{  95, "d2"                                , &Functions::d2                                 },
	{  96, "d3"                                , &Functions::d3                                 },
	{  97, "d4"                                , &Functions::d4                                 },
	{  98, "d6"                                , &Functions::d6                                 },
	{  99, "d8"                                , &Functions::d8                                 },
	{ 100, "d10"                               , &Functions::d10                                },
	{ 101, "d12"                               , &Functions::d12                                },
	{ 102, "d20"                               , &Functions::d20                                },
	{ 103, "d100"                              , &Functions::d100                               },
	{ 104, "VectorMagnitude"                   , &Functions::vectorMagnitude                    },
	{ 105, "GetPerceptionRange"                , 0                                              },
	{ 106, "GetObjectType"                     , &Functions::getObjectType                      },
	{ 107, "GetRacialType"                     , 0                                              },
	{ 108, "AddRunningEffect"                  , 0                                              },
	{ 109, "TalentItem"                        , 0                                              },
	{ 110, "SetDoorUsable"                     , 0                                              },
	{ 111, "RemoveRunningEffect"               , 0                                              },
	{ 112, "MagicalEffect"                     , 0                                              },
	{ 113, "SupernaturalEffect"                , 0                                              },
	{ 114, "ExtraordinaryEffect"               , 0                                              },
	{ 115, "RemoveAbility"                     , 0                                              },
	{ 116, "HasAbility"                        , 0                                              },
	{ 117, "GetCurrentAttribute"               , 0                                              },
	{ 118, "AddSequenceAttack"                 , 0                                              },
	{ 119, "AddAttackSequence"                 , 0                                              },
	{ 120, "RemoveAttackSequence"              , 0                                              },
	{ 121, "RoundsToSeconds"                   , 0                                              },
	{ 122, "HoursToSeconds"                    , 0                                              },
	{ 123, "TurnsToSeconds"                    , 0                                              },
	{ 124, "ApplyForce"                        , 0                                              },
	{ 125, "TestDirectLine"                    , 0                                              },
	{ 126, "GetIsAggressive"                   , 0                                              },
	{ 127, "SetAllowAggressiveAttitude"        , 0                                              },
	{ 128, "GetFirstObjectInShape"             , 0                                              },
	{ 129, "GetNextObjectInShape"              , 0                                              },
	{ 130, "GetLastActionResult"               , 0                                              },
	{ 131, "SignalEvent"                       , 0                                              },
	{ 132, "EventUserDefined"                  , 0                                              },
	{ 133, "EffectDeath"                       , 0                                              },
	{ 134, "EffectKnockdown"                   , 0                                              },
	{ 135, "ActionGiveItem"                    , 0                                              },
	{ 136, "ActionTakeItem"                    , 0                                              },
	{ 137, "VectorNormalize"                   , &Functions::vectorNormalize                    },
	{ 138, "ReplaceSequenceAttack"             , 0                                              },
	{ 139, "GetAbilityScore"                   , 0                                              },
	{ 140, "GetIsDead"                         , 0                                              },
	{ 141, "PrintVector"                       , &Functions::printVector                        },
	{ 142, "Vector"                            , &Functions::vector                             },
	{ 143, "SetFacingPoint"                    , 0                                              },
	{ 144, "AngleToVector"                     , 0                                              },
	{ 145, "VectorToAngle"                     , 0                                              },
	{ 146, "GetCurrentEndurancePoints"         , 0                                              },
	{ 147, "GetMaxEndurancePoints"             , 0                                              },
	{ 148, "EffectParalyze"                    , 0                                              },
	{ 149, "FindSpellByTypeAndLevel"           , 0                                              },
	{ 150, "EffectDeaf"                        , 0                                              },
	{ 151, "GetDistanceBetween"                , 0                                              },
	{ 152, "SetLocalLocation"                  , 0                                              },
	{ 153, "GetLocalLocation"                  , 0                                              },
	{ 154, "EffectSleep"                       , 0                                              },
	{ 155, "GetItemInSlot"                     , 0                                              },
	{ 156, "EffectCharmed"                     , 0                                              },
	{ 157, "EffectConfused"                    , 0                                              },
	{ 158, "EffectFrightened"                  , 0                                              },
	{ 159, "EffectDominated"                   , 0                                              },
	{ 160, "EffectDazed"                       , 0                                              },
	{ 161, "EffectStunned"                     , 0                                              },
	{ 162, "SetCommandable"                    , 0                                              },
	{ 163, "GetCommandable"                    , 0                                              },
	{ 164, "EffectRegenerate"                  , 0                                              },
	{ 165, "EffectMovementSpeedIncrease"       , 0                                              },
	{ 166, "GetHitDice"                        , 0                                              },
	{ 167, "ActionForceFollowObject"           , 0                                              },
	{ 168, "GetTag"                            , &Functions::getTag                             },
	{ 169, "AddKnownSpell"                     , 0                                              },
	{ 170, "GetEffectType"                     , 0                                              },
	{ 171, "EffectAreaOfEffect"                , 0                                              },
	{ 172, "GetFactionEqual"                   , 0                                              },
	{ 173, "ChangeFaction"                     , 0                                              },
	{ 174, "GetIsListening"                    , 0                                              },
	{ 175, "SetListening"                      , 0                                              },
	{ 176, "SetListenPattern"                  , 0                                              },
	{ 177, "TestStringAgainstPattern"          , 0                                              },
	{ 178, "GetMatchedSubstring"               , 0                                              },
	{ 179, "GetMatchedSubstringsCount"         , 0                                              },
	{ 180, "EffectVisualEffect"                , 0                                              },
	{ 181, "GetFactionWeakestMember"           , 0                                              },
	{ 182, "GetFactionStrongestMember"         , 0                                              },
	{ 183, "GetFactionMostDamagedMember"       , 0                                              },
	{ 184, "GetFactionLeastDamagedMember"      , 0                                              },
	{ 185, "GetFactionGold"                    , 0                                              },
	{ 186, "GetStoryPhaseByNPC"                , 0                                              },
	{ 187, "SetPerceptionRangeType"            , 0                                              },
	{ 188, "GetCurrentNonLethalDamagePoints"   , 0                                              },
	{ 189, "GetFactionAverageLevel"            , 0                                              },
	{ 190, "GetFactionAverageXP"               , 0                                              },
	{ 191, "HasProfile"                        , 0                                              },
	{ 192, "GetFactionWorstAC"                 , 0                                              },
	{ 193, "GetFactionBestAC"                  , 0                                              },
	{ 194, "ActionSit"                         , 0                                              },
	{ 195, "GetListenPatternNumber"            , 0                                              },
	{ 196, "ActionJumpToObject"                , &Functions::actionJumpToObject                 },
	{ 197, "GetWaypointByTag"                  , &Functions::getWaypointByTag                   },
	{ 198, "GetTransitionTarget"               , 0                                              },
	{ 199, "EffectLinkEffects"                 , 0                                              },
	{ 200, "GetObjectByTag"                    , &Functions::getObjectByTag                     },
	{ 201, "ItemIsUnarmedCombatWeapon"         , 0                                              },
	{ 202, "ActionWait"                        , 0                                              },
	{ 203, "SetAreaTransitionBMP"              , 0                                              },
	{ 204, "ActionStartConversation"           , 0                                              },
	{ 205, "ActionPauseConversation"           , 0                                              },
	{ 206, "ActionResumeConversation"          , 0                                              },
	{ 207, "SetHostileToEnemiesOf"             , 0                                              },
	{ 208, "GetStoryNPCObject"                 , 0                                              },
	{ 209, "GetStoryNPC"                       , 0                                              },
	{ 210, "GetSittingCreature"                , 0                                              },
	{ 211, "GetGoingToBeAttackedBy"            , 0                                              },
	{ 212, "RemoveKnownSpell"                  , 0                                              },
	{ 213, "GetLocation"                       , &Functions::getLocation                        },
	{ 214, "ActionJumpToLocation"              , &Functions::actionJumpToLocation               },
	{ 215, "Location"                          , &Functions::location                           },
	{ 216, "ApplyEffectAtLocation"             , 0                                              },
	{ 217, "GetIsPC"                           , &Functions::getIsPC                            },
	{ 218, "FeetToMeters"                      , 0                                              },
	{ 219, "YardsToMeters"                     , 0                                              },
	{ 220, "ApplyEffectToObject"               , 0                                              },
	{ 221, "SpeakString"                       , &Functions::speakString                        },
	{ 222, "GetSpellTargetLocation"            , 0                                              },
	{ 223, "GetPositionFromLocation"           , &Functions::getPositionFromLocation            },
	{ 224, "GetAreaFromLocation"               , 0                                              },
	{ 225, "GetFacingFromLocation"             , 0                                              },
	{ 226, "GetNearestCreatureToLocation"      , 0                                              },
	{ 227, "GetNearestObject"                  , &Functions::getNearestObject                   },
	{ 228, "GetNearestObjectToLocation"        , 0                                              },
	{ 229, "GetNearestObjectByTag"             , &Functions::getNearestObjectByTag              },
	{ 230, "IntToFloat"                        , &Functions::intToFloat                         },
	{ 231, "FloatToInt"                        , &Functions::floatToInt                         },
	{ 232, "StringToInt"                       , &Functions::stringToInt                        },
	{ 233, "StringToFloat"                     , &Functions::stringToFloat                      },
	{ 234, "ActionCastSpellAtLocation"         , 0                                              },
	{ 235, "GetStoryPhase"                     , 0                                              },
	{ 236, "GetSpawnSetSpawnPhase"             , 0                                              },
	{ 237, "GetStoryNPCSpawnPhase"             , 0                                              },
	{ 238, "GetPCSpeaker"                      , 0                                              },
	{ 239, "GetStringByStrRef"                 , &Functions::getStringByStrRef                  },
	{ 240, "ActionSpeakStringByStrRef"         , 0                                              },
	{ 241, "DestroyObject"                     , 0                                              },
	{ 242, "GetModule"                         , &Functions::getModule                          },
	{ 243, "CreateObject"                      , 0                                              },
	{ 244, "EventSpellCastAt"                  , 0                                              },
	{ 245, "GetLastSpellCaster"                , 0                                              },
	{ 246, "GetLastSpell"                      , 0                                              },
	{ 247, "GetUserDefinedEventNumber"         , 0                                              },
	{ 248, "GetSpellId"                        , 0                                              },
	{ 249, "RandomName"                        , 0                                              },
	{ 250, "GetSpellIntensity"                 , 0                                              },
	{ 251, "RemoveEffectByType"                , 0                                              },
	{ 252, "EffectSilence"                     , 0                                              },
	{ 253, "GetName"                           , &Functions::getName                            },
	{ 254, "GetLastSpeaker"                    , 0                                              },
	{ 255, "BeginConversation"                 , 0                                              },
	{ 256, "GetLastPerceived"                  , 0                                              },
	{ 257, "GetLastPerceptionHeard"            , 0                                              },
	{ 258, "GetLastPerceptionInaudible"        , 0                                              },
	{ 259, "GetLastPerceptionSeen"             , 0                                              },
	{ 260, "GetLastClosedBy"                   , &Functions::getLastClosedBy                    },
	{ 261, "GetLastPerceptionVanished"         , 0                                              },
	{ 262, "GetFirstInPersistentObject"        , 0                                              },
	{ 263, "GetNextInPersistentObject"         , 0                                              },
	{ 264, "GetAreaOfEffectCreator"            , 0                                              },
	{ 265, "DeleteLocalInt"                    , 0                                              },
	{ 266, "DeleteLocalFloat"                  , 0                                              },
	{ 267, "DeleteLocalString"                 , 0                                              },
	{ 268, "DeleteLocalObject"                 , 0                                              },
	{ 269, "DeleteLocalLocation"               , 0                                              },
	{ 270, "EffectHaste"                       , 0                                              },
	{ 271, "EffectSlow"                        , 0                                              },
	{ 272, "ObjectToString"                    , &Functions::objectToString                     },
	{ 273, "FindKnownSpellByTypeAndLevel"      , 0                                              },
	{ 274, "SetCurrentScriptSpellID"           , 0                                              },
	{ 275, "PrintError"                        , 0                                              },
	{ 276, "GetEncounterActive"                , 0                                              },
	{ 277, "SetEncounterActive"                , 0                                              },
	{ 278, "GetEncounterSpawnsMax"             , 0                                              },
	{ 279, "SetEncounterSpawnsMax"             , 0                                              },
	{ 280, "GetEncounterSpawnsCurrent"         , 0                                              },
	{ 281, "SetEncounterSpawnsCurrent"         , 0                                              },
	{ 282, "GetModuleItemAcquired"             , 0                                              },
	{ 283, "GetModuleItemAcquiredFrom"         , 0                                              },
	{ 284, "SetCustomToken"                    , 0                                              },
	{ 285, "GetHasFeat"                        , 0                                              },
	{ 286, "Mod"                               , 0                                              },
	{ 287, "ActionUseFeat"                     , 0                                              },
	{ 288, "ModAdd"                            , 0                                              },
	{ 289, "GetObjectSeen"                     , 0                                              },
	{ 290, "GetObjectHeard"                    , 0                                              },
	{ 291, "GetLastPlayerDied"                 , 0                                              },
	{ 292, "GetModuleItemLost"                 , 0                                              },
	{ 293, "GetModuleItemLostBy"               , 0                                              },
	{ 294, "ActionDoCommand"                   , &Functions::actionDoCommand                    },
	{ 295, "EventConversation"                 , 0                                              },
	{ 296, "SetEncounterDifficulty"            , 0                                              },
	{ 297, "GetEncounterDifficulty"            , 0                                              },
	{ 298, "GetDistanceBetweenLocations"       , 0                                              },
	{ 299, "EffectTemporaryEndurancePoints"    , 0                                              },
	{ 300, "PlayAnimation"                     , 0                                              },
	{ 301, "TalentSpell"                       , 0                                              },
	{ 302, "TalentFeat"                        , 0                                              },
	{ 303, "ModToFloat"                        , 0                                              },
	{ 304, "GetHasSpellEffect"                 , 0                                              },
	{ 305, "GetEffectSpellId"                  , 0                                              },
	{ 306, "GetCreatureHasTalent"              , 0                                              },
	{ 307, "GetCreatureTalentRandom"           , 0                                              },
	{ 308, "GetCreatureTalentBest"             , 0                                              },
	{ 309, "ActionUseTalentOnObject"           , 0                                              },
	{ 310, "ActionUseTalentAtLocation"         , 0                                              },
	{ 311, "GetGoldPieceValue"                 , 0                                              },
	{ 312, "GetIsPlayableRacialType"           , 0                                              },
	{ 313, "JumpToLocation"                    , &Functions::jumpToLocation                     },
	{ 314, "EffectTemporaryVitalityPoints"     , 0                                              },
	{ 315, "SetCurrentVitalityPoints"          , 0                                              },
	{ 316, "GetAttackTarget"                   , 0                                              },
	{ 317, "IsCurrentActionSitting"            , 0                                              },
	{ 318, "_TriggerOnWitness"                 , 0                                              },
	{ 319, "GetMaster"                         , 0                                              },
	{ 320, "GetIsInCombat"                     , 0                                              },
	{ 321, "GetLastAssociateCommand"           , 0                                              },
	{ 322, "GiveGoldToCreature"                , 0                                              },
	{ 323, "SetIsDestroyable"                  , 0                                              },
	{ 324, "SetLocked"                         , &Functions::setLocked                          },
	{ 325, "GetLocked"                         , &Functions::getLocked                          },
	{ 326, "GetClickingObject"                 , &Functions::getClickingObject                  },
	{ 327, "SetAssociateListenPatterns"        , 0                                              },
	{ 328, "GetLastWeaponUsed"                 , 0                                              },
	{ 329, "ActionInteractObject"              , 0                                              },
	{ 330, "GetLastUsedBy"                     , &Functions::getLastUsedBy                      },
	{ 331, "GetLevel"                          , 0                                              },
	{ 332, "GetIdentified"                     , 0                                              },
	{ 333, "SetIdentified"                     , 0                                              },
	{ 334, "SummonAnimalCompanion"             , 0                                              },
	{ 335, "SummonFamiliar"                    , 0                                              },
	{ 336, "GetBlockingDoor"                   , 0                                              },
	{ 337, "GetIsDoorActionPossible"           , 0                                              },
	{ 338, "DoDoorAction"                      , 0                                              },
	{ 339, "GetFirstItemInInventory"           , 0                                              },
	{ 340, "GetNextItemInInventory"            , 0                                              },
	{ 341, "SetCurrentEndurancePoints"         , 0                                              },
	{ 342, "GetCurrentDrunkState"              , 0                                              },
	{ 343, "SetCurrentDrunkState"              , 0                                              },
	{ 344, "GetDamageDealtByMedium"            , 0                                              },
	{ 345, "GetTotalDamageDealt"               , 0                                              },
	{ 346, "GetLastDamager"                    , 0                                              },
	{ 347, "GetLastDisarmed"                   , 0                                              },
	{ 348, "GetLastDisturbed"                  , 0                                              },
	{ 349, "GetLastLocked"                     , 0                                              },
	{ 350, "GetLastUnlocked"                   , 0                                              },
	{ 351, "BroadcastUserDefinedEvent"         , 0                                              },
	{ 352, "GetInventoryDisturbType"           , 0                                              },
	{ 353, "GetInventoryDisturbItem"           , 0                                              },
	{ 354, "GetHenchman"                       , 0                                              },
	{ 355, "GetCurrentToxicity"                , 0                                              },
	{ 356, "VersusRacialTypeEffect"            , 0                                              },
	{ 357, "VersusTrapEffect"                  , 0                                              },
	{ 358, "GetGender"                         , 0                                              },
	{ 359, "GetIsTalentValid"                  , 0                                              },
	{ 360, "ActionMoveAwayFromLocation"        , 0                                              },
	{ 361, "GetAttemptedAttackTarget"          , 0                                              },
	{ 362, "GetTypeFromTalent"                 , 0                                              },
	{ 363, "GetIdFromTalent"                   , 0                                              },
	{ 364, "GetAssociate"                      , 0                                              },
	{ 365, "AddHenchman"                       , 0                                              },
	{ 366, "RemoveHenchman"                    , 0                                              },
	{ 367, "AddJournalEntry"                   , 0                                              },
	{ 368, "HasJournalEntry"                   , 0                                              },
	{ 369, "GetPCPublicCDKey"                  , 0                                              },
	{ 370, "GetPCIPAddress"                    , 0                                              },
	{ 371, "GetPCPlayerName"                   , 0                                              },
	{ 372, "SetPCLike"                         , 0                                              },
	{ 373, "SetPCDislike"                      , 0                                              },
	{ 374, "SendMessageToPC"                   , &Functions::sendMessageToPC                    },
	{ 375, "GetAttemptedSpellTarget"           , 0                                              },
	{ 376, "GetLastOpenedBy"                   , &Functions::getLastOpenedBy                    },
	{ 377, "GetHasSpell"                       , 0                                              },
	{ 378, "OpenStore"                         , 0                                              },
	{ 379, "EffectTurned"                      , 0                                              },
	{ 380, "GetFirstFactionMember"             , 0                                              },
	{ 381, "GetNextFactionMember"              , 0                                              },
	{ 382, "ActionForceMoveToLocation"         , 0                                              },
	{ 383, "ActionForceMoveToObject"           , 0                                              },
	{ 384, "GetEnemyProfile"                   , 0                                              },
	{ 385, "JumpToObject"                      , &Functions::jumpToObject                       },
	{ 386, "SetMapPinEnabled"                  , 0                                              },
	{ 387, "EffectHitPointChangeWhenDying"     , 0                                              },
	{ 388, "PopUpGUIPanel"                     , 0                                              },
	{ 389, "GetIsUsingActionPoint"             , 0                                              },
	{ 390, "GetCurrentActionName"              , 0                                              },
	{ 391, "GetMonsterNPCObject"               , 0                                              },
	{ 392, "GetMonsterNPC"                     , 0                                              },
	{ 393, "GiveXPToCreature"                  , 0                                              },
	{ 394, "SetXP"                             , 0                                              },
	{ 395, "GetXP"                             , 0                                              },
	{ 396, "IntToHexString"                    , &Functions::intToHexString                     },
	{ 397, "GetBaseItemType"                   , 0                                              },
	{ 398, "GetItemHasItemProperty"            , 0                                              },
	{ 399, "ActionEquipMostDamagingMelee"      , 0                                              },
	{ 400, "ActionEquipMostDamagingRanged"     , 0                                              },
	{ 401, "GetAnimation"                      , 0                                              },
	{ 402, "ActionRest"                        , 0                                              },
	{ 403, "ExploreAreaForPlayer"              , 0                                              },
	{ 404, "ActionEquipMostEffectiveArmor"     , 0                                              },
	{ 405, "GetIsDay"                          , 0                                              },
	{ 406, "GetIsNight"                        , 0                                              },
	{ 407, "GetIsDawn"                         , 0                                              },
	{ 408, "GetIsDusk"                         , 0                                              },
	{ 409, "GetIsEncounterCreature"            , 0                                              },
	{ 410, "GetLastPlayerDying"                , 0                                              },
	{ 411, "GetStartingLocation"               , 0                                              },
	{ 412, "ChangeToStandardFaction"           , 0                                              },
	{ 413, "SoundObjectPlay"                   , 0                                              },
	{ 414, "SoundObjectStop"                   , 0                                              },
	{ 415, "SoundObjectSetVolume"              , 0                                              },
	{ 416, "SoundObjectSetPosition"            , 0                                              },
	{ 417, "SpeakOneLinerConversation"         , &Functions::speakOneLinerConversation          },
	{ 418, "GetGold"                           , 0                                              },
	{ 419, "GetLastRespawnButtonPresser"       , 0                                              },
	{ 420, "GetIsDM"                           , 0                                              },
	{ 421, "PlayVoiceChat"                     , 0                                              },
	{ 422, "GetIsWeaponEffective"              , 0                                              },
	{ 423, "GetLastSpellHarmful"               , 0                                              },
	{ 424, "EventActivateItem"                 , 0                                              },
	{ 425, "MusicBackgroundPlay"               , &Functions::musicBackgroundPlay                },
	{ 426, "MusicBackgroundStop"               , &Functions::musicBackgroundStop                },
	{ 427, "MusicBackgroundSetDelay"           , 0                                              },
	{ 428, "MusicBackgroundChangeDay"          , &Functions::musicBackgroundChangeDay           },
	{ 429, "MusicBackgroundChangeNight"        , &Functions::musicBackgroundChangeNight         },
	{ 430, "MusicBattlePlay"                   , 0                                              },
	{ 431, "MusicBattleStop"                   , 0                                              },
	{ 432, "MusicBattleChange"                 , 0                                              },
	{ 433, "AmbientSoundPlay"                  , 0                                              },
	{ 434, "AmbientSoundStop"                  , 0                                              },
	{ 435, "AmbientSoundChangeDay"             , 0                                              },
	{ 436, "AmbientSoundChangeNight"           , 0                                              },
	{ 437, "GetLastKiller"                     , 0                                              },
	{ 438, "GetSpellCastItem"                  , 0                                              },
	{ 439, "GetItemActivated"                  , 0                                              },
	{ 440, "GetItemActivator"                  , 0                                              },
	{ 441, "GetItemActivatedTargetLocation"    , 0                                              },
	{ 442, "GetItemActivatedTarget"            , 0                                              },
	{ 443, "GetIsOpen"                         , &Functions::getIsOpen                          },
	{ 444, "TakeGoldFromCreature"              , 0                                              },
	{ 445, "IsInConversation"                  , 0                                              },
	{ 446, "EnableSequenceJumps"               , 0                                              },
	{ 447, "SetupSpellTrigger"                 , 0                                              },
	{ 448, "GetLastTriggerSpellEnergy"         , 0                                              },
	{ 449, "GetLastTriggerSpellID"             , 0                                              },
	{ 450, "GetCurrentGamePhase"               , 0                                              },
	{ 451, "EffectMovementSpeedDecrease"       , 0                                              },
	{ 452, "SetCurrentGamePhase"               , 0                                              },
	{ 453, "IsQuestCompleted"                  , 0                                              },
	{ 454, "GetQuestCurrentPhaseIndex"         , 0                                              },
	{ 455, "GetPlotFlag"                       , 0                                              },
	{ 456, "SetPlotFlag"                       , 0                                              },
	{ 457, "EffectInvisibility"                , 0                                              },
	{ 458, "GetQuestPhaseIndex"                , 0                                              },
	{ 459, "EffectDarkness"                    , 0                                              },
	{ 460, "CompareGamePhases"                 , 0                                              },
	{ 461, "SetStoryNPCStoryPhase"             , 0                                              },
	{ 462, "SetCurrentToxicity"                , 0                                              },
	{ 463, "EffectPolymorph"                   , 0                                              },
	{ 464, "SetItemModelPart"                  , 0                                              },
	{ 465, "EffectTrueSeeing"                  , 0                                              },
	{ 466, "EffectSeeInvisible"                , 0                                              },
	{ 467, "EffectTimeStop"                    , 0                                              },
	{ 468, "EffectBlindness"                   , 0                                              },
	{ 469, "_ClearPersonalAttitude"            , 0                                              },
	{ 470, "GetIsObjectVisible"                , 0                                              },
	{ 471, "CompareTags"                       , 0                                              },
	{ 472, "ActionReady"                       , 0                                              },
	{ 473, "EnableSpawnPhase"                  , 0                                              },
	{ 474, "ActivatePortal"                    , 0                                              },
	{ 475, "GetNumStackedItems"                , 0                                              },
	{ 476, "GetValueFromSettings_Float"        , 0                                              },
	{ 477, "SetConversationResponse"           , 0                                              },
	{ 478, "GetLastJoiningItem"                , 0                                              },
	{ 479, "GetCreatureSize"                   , 0                                              },
	{ 480, "EffectDisappearAppear"             , 0                                              },
	{ 481, "EffectDisappear"                   , 0                                              },
	{ 482, "EffectAppear"                      , 0                                              },
	{ 483, "GetCurrentToxinCapacity"           , 0                                              },
	{ 484, "SetCurrentToxinCapacity"           , 0                                              },
	{ 485, "GetTimeFromLastHeartbeat"          , 0                                              },
	{ 486, "GetLastTrapDetected"               , 0                                              },
	{ 487, "GetConversationResponse"           , 0                                              },
	{ 488, "GetNearestTrapToObject"            , 0                                              },
	{ 489, "SetTag"                            , 0                                              },
	{ 490, "SetName"                           , 0                                              },
	{ 491, "HasSequenceForWeapon"              , 0                                              },
	{ 492, "PlayCutscene"                      , 0                                              },
	{ 493, "SetAttackSequence"                 , 0                                              },
	{ 494, "GetCurrentWeather"                 , 0                                              },
	{ 495, "GetAge"                            , 0                                              },
	{ 496, "GetMovementRate"                   , 0                                              },
	{ 497, "GetAnimalCompanionCreatureType"    , 0                                              },
	{ 498, "GetFamiliarCreatureType"           , 0                                              },
	{ 499, "GetAnimalCompanionName"            , 0                                              },
	{ 500, "GetFamiliarName"                   , 0                                              },
	{ 501, "ActionCastFakeSpellAtObject"       , 0                                              },
	{ 502, "ActionCastFakeSpellAtLocation"     , 0                                              },
	{ 503, "RemoveSummonedAssociate"           , 0                                              },
	{ 504, "GetPlaceableState"                 , 0                                              },
	{ 505, "GetIsResting"                      , 0                                              },
	{ 506, "GetLastPCRested"                   , 0                                              },
	{ 507, "SetWeather"                        , 0                                              },
	{ 508, "GetLastRestEventType"              , 0                                              },
	{ 509, "StartNewModule"                    , &Functions::startNewModule                     },
	{ 510, "HasOppositeProfileTo"              , 0                                              },
	{ 511, "GetWeaponRanged"                   , 0                                              },
	{ 512, "DoSinglePlayerAutoSave"            , 0                                              },
	{ 513, "GetGameDifficulty"                 , 0                                              },
	{ 514, "GetDialogActionParam"              , 0                                              },
	{ 515, "PlayVoiceSetVoiceOfTag"            , 0                                              },
	{ 516, "SoundObjectPlayWithFade"           , 0                                              },
	{ 517, "SoundObjectStopWithFade"           , 0                                              },
	{ 518, "IsSphereVisibleOnClient"           , 0                                              },
	{ 519, "GetLastGiftItem"                   , 0                                              },
	{ 520, "OpenPlayerGoldPanel"               , 0                                              },
	{ 521, "SetPanelButtonFlash"               , 0                                              },
	{ 522, "GetCurrentAction"                  , 0                                              },
	{ 523, "SetStandardFactionReputation"      , 0                                              },
	{ 524, "GetStandardFactionReputation"      , 0                                              },
	{ 525, "FloatingTextStrRefOnCreature"      , 0                                              },
	{ 526, "FloatingTextStringOnCreature"      , 0                                              },
	{ 527, "GetTrapDisarmable"                 , 0                                              },
	{ 528, "GetTrapDetectable"                 , 0                                              },
	{ 529, "GetTrapDetectedBy"                 , 0                                              },
	{ 530, "GetTrapFlagged"                    , 0                                              },
	{ 531, "GetTrapBaseType"                   , 0                                              },
	{ 532, "GetTrapOneShot"                    , 0                                              },
	{ 533, "GetTrapCreator"                    , 0                                              },
	{ 534, "GetTrapKeyTag"                     , 0                                              },
	{ 535, "GetTrapDisarmDC"                   , 0                                              },
	{ 536, "GetTrapDetectDC"                   , 0                                              },
	{ 537, "GetLockKeyRequired"                , 0                                              },
	{ 538, "GetLockKeyTag"                     , 0                                              },
	{ 539, "GetLockLockable"                   , 0                                              },
	{ 540, "GetLockUnlockDC"                   , 0                                              },
	{ 541, "GetLockLockDC"                     , 0                                              },
	{ 542, "GetPCLevellingUp"                  , 0                                              },
	{ 543, "GetHasFeatEffect"                  , 0                                              },
	{ 544, "SetPlaceableIllumination"          , 0                                              },
	{ 545, "GetPlaceableIllumination"          , 0                                              },
	{ 546, "GetIsPlaceableObjectActionPossible", 0                                              },
	{ 547, "DoPlaceableObjectAction"           , 0                                              },
	{ 548, "GetFirstPC"                        , &Functions::getFirstPC                         },
	{ 549, "GetNextPC"                         , &Functions::getNextPC                          },
	{ 550, "SetTrapDetectedBy"                 , 0                                              },
	{ 551, "GetIsTrapped"                      , 0                                              },
	{ 552, "PreloadCreature"                   , 0                                              },
	{ 553, "_SetProfile"                       , 0                                              },
	{ 554, "PopUpDeathGUIPanel"                , 0                                              },
	{ 555, "SetTrapDisabled"                   , 0                                              },
	{ 556, "GetLastHostileActor"               , 0                                              },
	{ 557, "ExportAllCharacters"               , 0                                              },
	{ 558, "MusicBackgroundGetDayTrack"        , &Functions::musicBackgroundGetDayTrack         },
	{ 559, "MusicBackgroundGetNightTrack"      , &Functions::musicBackgroundGetNightTrack       },
	{ 560, "WriteTimestampedLogEntry"          , &Functions::writeTimestampedLogEntry           },
	{ 561, "GetModuleName"                     , 0                                              },
	{ 562, "GetFactionLeader"                  , 0                                              },
	{ 563, "SendMessageToAllDMs"               , 0                                              },
	{ 564, "EndGame"                           , 0                                              },
	{ 565, "BootPC"                            , 0                                              },
	{ 566, "RestartModule"                     , 0                                              },
	{ 567, "AmbientSoundSetDayVolume"          , 0                                              },
	{ 568, "AmbientSoundSetNightVolume"        , 0                                              },
	{ 569, "MusicBackgroundGetBattleTrack"     , 0                                              },
	{ 570, "GetHasInventory"                   , 0                                              },
	{ 571, "GetStrRefSoundDuration"            , 0                                              },
	{ 572, "GetAttitude"                       , 0                                              },
	{ 573, "CreateVisualEffectAtLocation"      , 0                                              },
	{ 574, "MusicIncidentalPlay"               , 0                                              },
	{ 575, "MusicIncidentalStop"               , 0                                              },
	{ 576, "SetPerceptionUpdateInterval"       , 0                                              },
	{ 577, "GetAppearanceType"                 , 0                                              },
	{ 578, "SpawnScriptDebugger"               , 0                                              },
	{ 579, "GetModuleItemAcquiredStackSize"    , 0                                              },
	{ 580, "BackupPersonalAttitudes"           , 0                                              },
	{ 581, "RestorePersonalAttitudes"          , 0                                              },
	{ 582, "GetResRef"                         , 0                                              },
	{ 583, "EffectPetrify"                     , 0                                              },
	{ 584, "CopyItem"                          , 0                                              },
	{ 585, "EffectCutsceneParalyze"            , 0                                              },
	{ 586, "GetDroppableFlag"                  , 0                                              },
	{ 587, "GetUseableFlag"                    , 0                                              },
	{ 588, "GetStolenFlag"                     , 0                                              },
	{ 589, "SetCampaignFloat"                  , 0                                              },
	{ 590, "SetCampaignInt"                    , 0                                              },
	{ 591, "SetCampaignVector"                 , 0                                              },
	{ 592, "SetCampaignLocation"               , 0                                              },
	{ 593, "SetCampaignString"                 , 0                                              },
	{ 594, "DestroyCampaignDatabase"           , 0                                              },
	{ 595, "GetCampaignFloat"                  , 0                                              },
	{ 596, "GetCampaignInt"                    , 0                                              },
	{ 597, "GetCampaignVector"                 , 0                                              },
	{ 598, "GetCampaignLocation"               , 0                                              },
	{ 599, "GetCampaignString"                 , 0                                              },
	{ 600, "CopyObject"                        , 0                                              },
	{ 601, "DeleteCampaignVariable"            , 0                                              },
	{ 602, "StoreCampaignObject"               , 0                                              },
	{ 603, "RetrieveCampaignObject"            , 0                                              },
	{ 604, "EffectCutsceneDominated"           , 0                                              },
	{ 605, "GetItemStackSize"                  , 0                                              },
	{ 606, "SetItemStackSize"                  , 0                                              },
	{ 607, "GetItemCharges"                    , 0                                              },
	{ 608, "SetItemCharges"                    , 0                                              },
	{ 609, "AddItemProperty"                   , 0                                              },
	{ 610, "RemoveItemProperty"                , 0                                              },
	{ 611, "GetIsItemPropertyValid"            , 0                                              },
	{ 612, "GetFirstItemProperty"              , 0                                              },
	{ 613, "GetNextItemProperty"               , 0                                              },
	{ 614, "GetItemPropertyType"               , 0                                              },
	{ 615, "GetItemPropertyDuration"           , 0                                              },
	{ 616, "GetLastBribeAmount"                , 0                                              },
	{ 617, "EnableStoryNPCActions"             , 0                                              },
	{ 618, "EffectPush"                        , 0                                              },
	{ 619, "AddWeaponEffect"                   , 0                                              },
	{ 620, "RemoveWeaponEffect"                , 0                                              },
	{ 621, "SetProtectionPoints"               , 0                                              },
	{ 622, "SetSpecialDistance"                , 0                                              },
	{ 623, "GetProtectionPoints"               , 0                                              },
	{ 624, "UnlockPlayerStoryBasedAbility"     , 0                                              },
	{ 625, "TerminateProtection"               , 0                                              },
	{ 626, "GetLastPerceptionEventType"        , 0                                              },
	{ 627, "GetLastPerceptionZone"             , 0                                              },
	{ 628, "GetIsItemOfType"                   , 0                                              },
	{ 629, "GetIsAmmoForWeapon"                , 0                                              },
	{ 630, "ImmediateRest"                     , 0                                              },
	{ 631, "DPrint"                            , 0                                              },
	{ 632, "CreateVisualEffectAtCreature"      , 0                                              },
	{ 633, "GetIsAISystemEnabled"              , 0                                              },
	{ 634, "GetProfileName"                    , 0                                              },
	{ 635, "GetNumPeacefulProfiles"            , 0                                              },
	{ 636, "SetPlaceableUsable"                , 0                                              },
	{ 637, "AddToCallstack"                    , 0                                              },
	{ 638, "GetCallstackLine"                  , 0                                              },
	{ 639, "GetCallstack"                      , 0                                              },
	{ 640, "ChangePlaceableEffectPhase"        , 0                                              },
	{ 641, "_SetCallstackIdent"                , 0                                              },
	{ 642, "_AdjustCallstackIdent"             , 0                                              },
	{ 643, "HasSequence"                       , 0                                              },
	{ 644, "GetCurrentSequenceId"              , 0                                              },
	{ 645, "SetCurrentSequenceId"              , 0                                              },
	{ 646, "GetIsWeaponRanged"                 , 0                                              },
	{ 647, "NoOp"                              , 0                                              },
	{ 648, "GetBattleMusicIdByCreature"        , 0                                              },
	{ 649, "GetSpecialDistance"                , 0                                              },
	{ 650, "CreateVisualEffectAtObject"        , 0                                              },
	{ 651, "DeleteVisualEffectFromObject"      , 0                                              },
	{ 652, "SetGiftMessages"                   , 0                                              },
	{ 653, "HasTheSameAffiliations"            , 0                                              },
	{ 654, "IsAfraidOf"                        , 0                                              },
	{ 655, "GetLastItemSold"                   , 0                                              },
	{ 656, "CloseStoreAndInventory"            , 0                                              },
	{ 657, "GetNPCDialogByTags"                , 0                                              },
	{ 658, "GetDialogInterlocutorByTag"        , 0                                              },
	{ 659, "IsCutscenePlaying"                 , 0                                              },
	{ 660, "SetPositionNearObject"             , 0                                              },
	{ 661, "StartMaterialEffect"               , 0                                              },
	{ 662, "StopMaterialEffect"                , 0                                              },
	{ 663, "BindEffectParamToObjectPosition"   , 0                                              },
	{ 664, "GetDialogFile"                     , 0                                              },
	{ 665, "_UpdateIsInTrigger"                , 0                                              },
	{ 666, "HasJournalEntryConcerningObject"   , 0                                              },
	{ 667, "Meditate"                          , 0                                              },
	{ 668, "GetCurrentWeatherIntensity"        , 0                                              },
	{ 669, "BackupAfraidOf"                    , 0                                              },
	{ 670, "RestoreAfraidOf"                   , 0                                              },
	{ 671, "SetAfraidOfAffiliations"           , 0                                              },
	{ 672, "GetProfileType"                    , 0                                              },
	{ 673, "GetIsExcited"                      , 0                                              },
	{ 674, "BindEffectParamToPartPosition"     , 0                                              },
	{ 675, "ForceUpdateMeleePositions"         , 0                                              },
	{ 676, "IsMeleePositionFree"               , 0                                              },
	{ 677, "CanPlotNPCRaiseFromDead"           , 0                                              },
	{ 678, "ProlongPlotNPCDeathEffect"         , 0                                              },
	{ 679, "AddTalents"                        , 0                                              },
	{ 680, "CacheSequenceDistances"            , 0                                              },
	{ 681, "RevealWorldMapRegion"              , 0                                              },
	{ 682, "GetNearestAreaTransitionTarget"    , 0                                              },
	{ 683, "AlchemyLearnMixture"               , 0                                              },
	{ 684, "AlchemyLearnItem"                  , 0                                              },
	{ 685, "AlchemyGetItemKnown"               , 0                                              },
	{ 686, "SetLockedAttackTarget"             , 0                                              },
	{ 687, "ScriptDecisionsTempOff"            , 0                                              },
	{ 688, "CanScriptDecideOfActions"          , 0                                              },
	{ 689, "IsTriggerEnabled"                  , 0                                              },
	{ 690, "EnableTrigger"                     , 0                                              },
	{ 691, "SpeakStringByStrRef"               , &Functions::speakStringByStrRef                },
	{ 692, "SetCutsceneMode"                   , 0                                              },
	{ 693, "GetLastPCToCancelCutscene"         , 0                                              },
	{ 694, "GetDialogSoundLength"              , 0                                              },
	{ 695, "FadeFromBlack"                     , 0                                              },
	{ 696, "FadeToBlack"                       , 0                                              },
	{ 697, "StopFade"                          , 0                                              },
	{ 698, "BlackScreen"                       , 0                                              },
	{ 699, "GetBaseAttackBonus"                , 0                                              },
	{ 700, "SetImmortal"                       , 0                                              },
	{ 701, "OpenPlayerInventory"               , 0                                              },
	{ 702, "StoreCameraFacing"                 , 0                                              },
	{ 703, "RestoreCameraFacing"               , 0                                              },
	{ 704, "LevelUpHenchman"                   , 0                                              },
	{ 705, "SetDroppableFlag"                  , 0                                              },
	{ 706, "SetNeutralBumpable"                , 0                                              },
	{ 707, "GetModuleItemAcquiredBy"           , 0                                              },
	{ 708, "GetImmortal"                       , 0                                              },
	{ 709, "_SetEmotionalSubstate"             , 0                                              },
	{ 710, "Get2DAString"                      , &Functions::get2DAString                       },
	{ 711, "GetMainProfile"                    , 0                                              },
	{ 712, "GetAILevel"                        , 0                                              },
	{ 713, "SetAILevel"                        , 0                                              },
	{ 714, "GetIsPossessedFamiliar"            , 0                                              },
	{ 715, "UnpossessFamiliar"                 , 0                                              },
	{ 716, "GetIsInside"                       , 0                                              },
	{ 717, "SendMessageToPCByStrRef"           , 0                                              },
	{ 718, "IncrementRemainingFeatUses"        , 0                                              },
	{ 719, "ExportSingleCharacter"             , 0                                              },
	{ 720, "PlaySoundByStrRef"                 , &Functions::playSoundByStrRef                  },
	{ 721, "RunClientMacro"                    , 0                                              },
	{ 722, "RunClientLua"                      , &Functions::runClientLua                       },
	{ 723, "EffectRunScript"                   , 0                                              },
	{ 724, "SetEmotionalState"                 , 0                                              },
	{ 725, "GetEffectScriptID"                 , 0                                              },
	{ 726, "ForceEmotionalAnim"                , 0                                              },
	{ 727, "GetObjectById"                     , 0                                              },
	{ 728, "TestDialogFlag"                    , 0                                              },
	{ 729, "SetDialogFlagValue"                , 0                                              },
	{ 730, "GetDialogFlagValue"                , 0                                              },
	{ 731, "FindDialogFlagIndex"               , 0                                              },
	{ 732, "SetStartingStyleForWeaponType"     , 0                                              },
	{ 733, "GetBestSlotForWeapon"              , 0                                              },
	{ 734, "SetWaypointDesc"                   , 0                                              },
	{ 735, "ActionJumpToModule"                , 0                                              },
	{ 736, "AddImpulse"                        , 0                                              },
	{ 737, "_SetPersonalAttitude"              , 0                                              },
	{ 738, "GetPersonalAttitude"               , 0                                              },
	{ 739, "HasPersonalAttitude"               , 0                                              },
	{ 740, "_ClearPersonalAttitudeList"        , 0                                              },
	{ 741, "GetFirstNeighbourCreature"         , 0                                              },
	{ 742, "GetNextNeighbourCreature"          , 0                                              },
	{ 743, "SetSurroundingFlag"                , 0                                              },
	{ 744, "HasProfileByName"                  , 0                                              },
	{ 745, "AddDialogActor"                    , 0                                              },
	{ 746, "DespawnCreature"                   , 0                                              },
	{ 747, "PreloadAreaModel"                  , 0                                              },
	{ 748, "PostDialogCommand"                 , 0                                              },
	{ 749, "GetCombatState"                    , 0                                              },
	{ 750, "SetCombatState"                    , 0                                              },
	{ 751, "GetCrowdActionParticipant"         , 0                                              },
	{ 752, "GetNPCDialog"                      , 0                                              },
	{ 753, "ErrorExit"                         , 0                                              },
	{ 754, "SetQueuedActionsInterruptable"     , 0                                              },
	{ 755, "_SetPauseState"                    , 0                                              },
	{ 756, "See"                               , 0                                              },
	{ 757, "_StackPopObject"                   , 0                                              },
	{ 758, "FindNearestCreatures"              , 0                                              },
	{ 759, "_GetLastInternalEventId"           , 0                                              },
	{ 760, "_StackPopInteger"                  , 0                                              },
	{ 761, "_StackPopFloat"                    , 0                                              },
	{ 762, "HasEquipedMelee"                   , 0                                              },
	{ 763, "HasEquipedRanged"                  , 0                                              },
	{ 764, "GetIsFistfighter"                  , 0                                              },
	{ 765, "SetMapPinActive"                   , 0                                              },
	{ 766, "_FastSetInt"                       , 0                                              },
	{ 767, "_FastGetInt"                       , 0                                              },
	{ 768, "_FastSetObject"                    , 0                                              },
	{ 769, "_FastGetObject"                    , 0                                              },
	{ 770, "Assert"                            , 0                                              },
	{ 771, "_FastSetFloat"                     , 0                                              },
	{ 772, "_FastGetFloat"                     , 0                                              },
	{ 773, "TriggerInternalEvent"              , 0                                              },
	{ 774, "GetNumGuardianTriggers"            , 0                                              },
	{ 775, "GetNumGuardianWaypoints"           , 0                                              },
	{ 776, "GetGuardianTrigger"                , 0                                              },
	{ 777, "GetGuardianWaypoint"               , 0                                              },
	{ 778, "SetIsInTrigger"                    , 0                                              },
	{ 779, "GetIsInMyTriggers"                 , 0                                              },
	{ 780, "SetPerceptionMultiplier"           , 0                                              },
	{ 781, "GetPerceptionMultiplier"           , 0                                              },
	{ 782, "SlotDlaTomka01"                    , 0                                              },
	{ 783, "SlotDlaTomka02"                    , 0                                              },
	{ 784, "SlotDlaTomka03"                    , 0                                              },
	{ 785, "SlotDlaTomka04"                    , 0                                              },
	{ 786, "HasAction"                         , 0                                              },
	{ 787, "ForceMapOfAnotherArea"             , 0                                              },
	{ 788, "IsMapFullyLoaded"                  , 0                                              },
	{ 789, "GetCreatureAppearance"             , 0                                              },
	{ 790, "SetCreatureAppearance"             , 0                                              },
	{ 791, "DontEnableNPCActionsAfterDialog"   , 0                                              },
	{ 792, "HasPhysics"                        , 0                                              },
	{ 793, "WillBeOneLiner"                    , 0                                              },
	{ 794, "SetCombatModeBaseDist"             , 0                                              },
	{ 795, "ActionTeleportAndTalk"             , 0                                              },
	{ 796, "PreDialogShot"                     , 0                                              },
	{ 797, "EnterFistfightMode"                , 0                                              },
	{ 798, "LeaveFistfightMode"                , 0                                              },
	{ 799, "LoadAreaMap"                       , 0                                              },
	{ 800, "SetDontEquipAnythingAfterDialog"   , 0                                              },
	{ 801, "EndFistfightKnockOutEffect"        , 0                                              },
	{ 802, "OpenPlayerBribePanel"              , 0                                              },
	{ 803, "GetBribeVariationValue"            , 0                                              },
	{ 804, "ResetXP"                           , 0                                              },
	{ 805, "AddGreaseAbility"                  , 0                                              },
	{ 806, "ClearGrease"                       , 0                                              },
	{ 807, "Get2DARow"                         , 0                                              },
	{ 808, "Get2DAInt"                         , 0                                              },
	{ 809, "SlotDlaMichala5"                   , 0                                              },
	{ 810, "SlotDlaMichala6"                   , 0                                              },
	{ 811, "SlotDlaMichala7"                   , 0                                              },
	{ 812, "SlotDlaMichala8"                   , 0                                              },
	{ 813, "SlotDlaMichala9"                   , 0                                              },
	{ 814, "Find5NearestCreatures"             , 0                                              },
	{ 815, "IsInInteractiveMode"               , 0                                              },
	{ 816, "SetMovementRate"                   , 0                                              },
	{ 817, "EquipMelee"                        , 0                                              },
	{ 818, "EquipRanged"                       , 0                                              },
	{ 819, "CanImmobileAttackMelee"            , 0                                              },
	{ 820, "HasDialog"                         , 0                                              },
	{ 821, "GetNumNearbyCreatures"             , 0                                              },
	{ 822, "_UpdatePassiveState"               , 0                                              },
	{ 823, "_SetIsPassiveAttacker"             , 0                                              },
	{ 824, "RandomFL"                          , 0                                              },
	{ 825, "GetBestRainHideout"                , 0                                              },
	{ 826, "_DespawnBecouseOfRain"             , 0                                              },
	{ 827, "_IsDespawningBecouseOfRain"        , 0                                              },
	{ 828, "_GetLastExecutedScript"            , 0                                              },
	{ 829, "ResetQuestUpdateFlag"              , 0                                              },
	{ 830, "GetQuestUpdateFlag"                , 0                                              },
	{ 831, "IsActorInDialog"                   , 0                                              },
	{ 832, "GetRandomLocationNearObject"       , 0                                              },
	{ 833, "CanPlayVoiceSetOfType"             , 0                                              },
	{ 834, "OpenPlayerBetPanel"                , 0                                              },
	{ 835, "IsAreaFullyLoadedOnClient"         , 0                                              },
	{ 836, "IsInInterior"                      , 0                                              },
	{ 837, "_CanFindPath"                      , 0                                              },
	{ 838, "_IsActionQueueEmpty"               , 0                                              },
	{ 839, "IsModuleSaveGame"                  , 0                                              },
	{ 840, "GetRealMaxActiveAttackers"         , 0                                              },
	{ 841, "GetRealActiveAttackers"            , 0                                              },
	{ 842, "_UpdatePassiveAttackersList"       , 0                                              },
	{ 843, "HasVoiceOfTag"                     , 0                                              },
	{ 844, "SetDialogOwner"                    , 0                                              },
	{ 845, "ForceCharacterDevelopment"         , 0                                              },
	{ 846, "DisableCutsceneModeAfterDialog"    , 0                                              },
	{ 847, "EnableMusician"                    , 0                                              }
};

/** The table defining the signature (return type and type of parameters) of each engine function. */
const Functions::FunctionSignature Functions::kFunctionSignatures[] = {
	{   0, kTypeInt       , { kTypeInt } },
	{   1, kTypeVoid      , { kTypeString } },
	{   2, kTypeVoid      , { kTypeFloat, kTypeInt, kTypeInt } },
	{   3, kTypeString    , { kTypeFloat, kTypeInt, kTypeInt } },
	{   4, kTypeVoid      , { kTypeInt } },
	{   5, kTypeString    , { kTypeObject } },
	{   6, kTypeVoid      , { kTypeObject, kTypeScriptState } },
	{   7, kTypeVoid      , { kTypeFloat, kTypeScriptState } },
	{   8, kTypeVoid      , { kTypeString, kTypeObject } },
	{   9, kTypeVoid      , { kTypeInt, kTypeObject } },
	{  10, kTypeVoid      , { kTypeFloat } },
	{  11, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{  12, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  13, kTypeInt       , { } },
	{  14, kTypeInt       , { } },
	{  15, kTypeInt       , { } },
	{  16, kTypeInt       , { } },
	{  17, kTypeInt       , { } },
	{  18, kTypeInt       , { } },
	{  19, kTypeInt       , { } },
	{  20, kTypeVoid      , { } },
	{  21, kTypeVoid      , { kTypeEngineType, kTypeInt } },
	{  22, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  23, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  24, kTypeObject    , { kTypeObject } },
	{  25, kTypeObject    , { } },
	{  26, kTypeObject    , { } },
	{  27, kTypeVector    , { kTypeObject } },
	{  28, kTypeFloat     , { kTypeObject } },
	{  29, kTypeObject    , { kTypeObject } },
	{  30, kTypeObject    , { kTypeObject, kTypeString } },
	{  31, kTypeObject    , { kTypeString, kTypeObject, kTypeInt } },
	{  32, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  33, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  34, kTypeVoid      , { kTypeObject } },
	{  35, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  36, kTypeObject    , { kTypeObject } },
	{  37, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  38, kTypeObject    , { kTypeInt, kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  39, kTypeVoid      , { kTypeString, kTypeInt } },
	{  40, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{  41, kTypeFloat     , { kTypeObject } },
	{  42, kTypeInt       , { kTypeObject } },
	{  43, kTypeVoid      , { kTypeObject } },
	{  44, kTypeVoid      , { kTypeObject } },
	{  45, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  46, kTypeVoid      , { kTypeString, kTypeInt, kTypeFloat } },
	{  47, kTypeObject    , { } },
	{  48, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{  49, kTypeInt       , { kTypeObject } },
	{  50, kTypeInt       , { kTypeObject } },
	{  51, kTypeInt       , { kTypeObject, kTypeString } },
	{  52, kTypeFloat     , { kTypeObject, kTypeString } },
	{  53, kTypeString    , { kTypeObject, kTypeString } },
	{  54, kTypeObject    , { kTypeObject, kTypeString } },
	{  55, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{  56, kTypeVoid      , { kTypeObject, kTypeString, kTypeFloat } },
	{  57, kTypeVoid      , { kTypeObject, kTypeString, kTypeString } },
	{  58, kTypeVoid      , { kTypeObject, kTypeString, kTypeObject } },
	{  59, kTypeInt       , { kTypeString } },
	{  60, kTypeString    , { kTypeString } },
	{  61, kTypeString    , { kTypeString } },
	{  62, kTypeString    , { kTypeString, kTypeInt } },
	{  63, kTypeString    , { kTypeString, kTypeInt } },
	{  64, kTypeString    , { kTypeString, kTypeString, kTypeInt } },
	{  65, kTypeString    , { kTypeString, kTypeInt, kTypeInt } },
	{  66, kTypeInt       , { kTypeString, kTypeString } },
	{  67, kTypeFloat     , { kTypeFloat } },
	{  68, kTypeFloat     , { kTypeFloat } },
	{  69, kTypeFloat     , { kTypeFloat } },
	{  70, kTypeFloat     , { kTypeFloat } },
	{  71, kTypeFloat     , { kTypeFloat } },
	{  72, kTypeFloat     , { kTypeFloat } },
	{  73, kTypeFloat     , { kTypeFloat } },
	{  74, kTypeFloat     , { kTypeFloat } },
	{  75, kTypeFloat     , { kTypeFloat, kTypeFloat } },
	{  76, kTypeFloat     , { kTypeFloat } },
	{  77, kTypeInt       , { kTypeInt } },
	{  78, kTypeEngineType, { kTypeInt } },
	{  79, kTypeEngineType, { kTypeFloat, kTypeFloat, kTypeFloat, kTypeString, kTypeInt, kTypeString } },
	{  80, kTypeInt       , { kTypeObject } },
	{  81, kTypeInt       , { kTypeObject, kTypeString, kTypeInt } },
	{  82, kTypeEngineType, { } },
	{  83, kTypeEngineType, { kTypeString, kTypeInt, kTypeFloat } },
	{  84, kTypeInt       , { kTypeString, kTypeObject, kTypeInt, kTypeInt } },
	{  85, kTypeEngineType, { kTypeObject } },
	{  86, kTypeEngineType, { kTypeObject } },
	{  87, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{  88, kTypeInt       , { kTypeEngineType } },
	{  89, kTypeInt       , { kTypeEngineType } },
	{  90, kTypeInt       , { kTypeEngineType } },
	{  91, kTypeObject    , { kTypeEngineType } },
	{  92, kTypeString    , { kTypeInt } },
	{  93, kTypeObject    , { kTypeObject } },
	{  94, kTypeObject    , { kTypeObject } },
	{  95, kTypeInt       , { kTypeInt } },
	{  96, kTypeInt       , { kTypeInt } },
	{  97, kTypeInt       , { kTypeInt } },
	{  98, kTypeInt       , { kTypeInt } },
	{  99, kTypeInt       , { kTypeInt } },
	{ 100, kTypeInt       , { kTypeInt } },
	{ 101, kTypeInt       , { kTypeInt } },
	{ 102, kTypeInt       , { kTypeInt } },
	{ 103, kTypeInt       , { kTypeInt } },
	{ 104, kTypeFloat     , { kTypeVector } },
	{ 105, kTypeFloat     , { kTypeInt, kTypeObject } },
	{ 106, kTypeInt       , { kTypeObject } },
	{ 107, kTypeInt       , { kTypeObject } },
	{ 108, kTypeInt       , { kTypeObject, kTypeString, kTypeInt, kTypeString, kTypeFloat, kTypeFloat, kTypeObject, kTypeString } },
	{ 109, kTypeEngineType, { kTypeObject } },
	{ 110, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 111, kTypeInt       , { kTypeObject, kTypeString, kTypeInt } },
	{ 112, kTypeEngineType, { kTypeEngineType } },
	{ 113, kTypeEngineType, { kTypeEngineType } },
	{ 114, kTypeEngineType, { kTypeEngineType } },
	{ 115, kTypeInt       , { kTypeString, kTypeObject } },
	{ 116, kTypeInt       , { kTypeString, kTypeObject } },
	{ 117, kTypeEngineType, { kTypeString, kTypeString, kTypeObject, kTypeObject } },
	{ 118, kTypeInt       , { kTypeString, kTypeInt, kTypeInt, kTypeString, kTypeObject } },
	{ 119, kTypeInt       , { kTypeString, kTypeString, kTypeString, kTypeObject } },
	{ 120, kTypeInt       , { kTypeString, kTypeObject } },
	{ 121, kTypeFloat     , { kTypeInt } },
	{ 122, kTypeFloat     , { kTypeInt } },
	{ 123, kTypeFloat     , { kTypeInt } },
	{ 124, kTypeVoid      , { kTypeObject, kTypeEngineType, kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 125, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 126, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 127, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 128, kTypeObject    , { kTypeInt, kTypeFloat, kTypeEngineType, kTypeInt, kTypeInt, kTypeVector, kTypeFloat } },
	{ 129, kTypeObject    , { kTypeInt, kTypeFloat, kTypeEngineType, kTypeInt, kTypeInt, kTypeVector, kTypeFloat } },
	{ 130, kTypeInt       , { kTypeObject } },
	{ 131, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 132, kTypeEngineType, { kTypeInt } },
	{ 133, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 134, kTypeEngineType, { kTypeObject } },
	{ 135, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 136, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 137, kTypeVector    , { kTypeVector } },
	{ 138, kTypeInt       , { kTypeString, kTypeString, kTypeString, kTypeObject } },
	{ 139, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 140, kTypeInt       , { kTypeObject } },
	{ 141, kTypeString    , { kTypeVector, kTypeInt } },
	{ 142, kTypeVector    , { kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 143, kTypeVoid      , { kTypeVector } },
	{ 144, kTypeVector    , { kTypeFloat } },
	{ 145, kTypeFloat     , { kTypeVector } },
	{ 146, kTypeInt       , { kTypeObject } },
	{ 147, kTypeInt       , { kTypeObject } },
	{ 148, kTypeEngineType, { } },
	{ 149, kTypeInt       , { kTypeString, kTypeInt, kTypeInt } },
	{ 150, kTypeEngineType, { } },
	{ 151, kTypeFloat     , { kTypeObject, kTypeObject } },
	{ 152, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{ 153, kTypeEngineType, { kTypeObject, kTypeString } },
	{ 154, kTypeEngineType, { } },
	{ 155, kTypeObject    , { kTypeInt, kTypeObject } },
	{ 156, kTypeEngineType, { } },
	{ 157, kTypeEngineType, { } },
	{ 158, kTypeEngineType, { } },
	{ 159, kTypeEngineType, { } },
	{ 160, kTypeEngineType, { } },
	{ 161, kTypeEngineType, { } },
	{ 162, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 163, kTypeInt       , { kTypeObject } },
	{ 164, kTypeEngineType, { kTypeInt, kTypeFloat } },
	{ 165, kTypeEngineType, { kTypeInt } },
	{ 166, kTypeInt       , { kTypeObject } },
	{ 167, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeInt } },
	{ 168, kTypeString    , { kTypeObject } },
	{ 169, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 170, kTypeInt       , { kTypeEngineType } },
	{ 171, kTypeEngineType, { kTypeInt, kTypeString, kTypeString, kTypeString } },
	{ 172, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 173, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 174, kTypeInt       , { kTypeObject } },
	{ 175, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 176, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 177, kTypeInt       , { kTypeString, kTypeString } },
	{ 178, kTypeString    , { kTypeInt } },
	{ 179, kTypeInt       , { } },
	{ 180, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 181, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 182, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 183, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 184, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 185, kTypeInt       , { kTypeObject } },
	{ 186, kTypeString    , { kTypeString } },
	{ 187, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 188, kTypeInt       , { kTypeObject } },
	{ 189, kTypeInt       , { kTypeObject } },
	{ 190, kTypeInt       , { kTypeObject } },
	{ 191, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 192, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 193, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 194, kTypeVoid      , { kTypeObject } },
	{ 195, kTypeInt       , { } },
	{ 196, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 197, kTypeObject    , { kTypeString } },
	{ 198, kTypeObject    , { kTypeObject } },
	{ 199, kTypeEngineType, { kTypeEngineType, kTypeEngineType } },
	{ 200, kTypeObject    , { kTypeString, kTypeInt } },
	{ 201, kTypeInt       , { kTypeObject } },
	{ 202, kTypeVoid      , { kTypeFloat } },
	{ 203, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 204, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeInt } },
	{ 205, kTypeVoid      , { } },
	{ 206, kTypeVoid      , { } },
	{ 207, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 208, kTypeObject    , { kTypeString } },
	{ 209, kTypeString    , { kTypeObject } },
	{ 210, kTypeObject    , { kTypeObject } },
	{ 211, kTypeObject    , { kTypeObject } },
	{ 212, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 213, kTypeEngineType, { kTypeObject } },
	{ 214, kTypeVoid      , { kTypeEngineType } },
	{ 215, kTypeEngineType, { kTypeObject, kTypeVector, kTypeFloat } },
	{ 216, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeEngineType, kTypeFloat } },
	{ 217, kTypeInt       , { kTypeObject } },
	{ 218, kTypeFloat     , { kTypeFloat } },
	{ 219, kTypeFloat     , { kTypeFloat } },
	{ 220, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeObject, kTypeFloat, kTypeString, kTypeInt } },
	{ 221, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 222, kTypeEngineType, { } },
	{ 223, kTypeVector    , { kTypeEngineType } },
	{ 224, kTypeObject    , { kTypeEngineType } },
	{ 225, kTypeFloat     , { kTypeEngineType } },
	{ 226, kTypeObject    , { kTypeInt, kTypeInt, kTypeEngineType, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 227, kTypeObject    , { kTypeInt, kTypeObject, kTypeInt } },
	{ 228, kTypeObject    , { kTypeInt, kTypeEngineType, kTypeInt } },
	{ 229, kTypeObject    , { kTypeString, kTypeObject, kTypeInt, kTypeInt } },
	{ 230, kTypeFloat     , { kTypeInt } },
	{ 231, kTypeInt       , { kTypeFloat } },
	{ 232, kTypeInt       , { kTypeString } },
	{ 233, kTypeFloat     , { kTypeString } },
	{ 234, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeInt, kTypeInt, kTypeInt } },
	{ 235, kTypeString    , { kTypeObject } },
	{ 236, kTypeString    , { kTypeString } },
	{ 237, kTypeString    , { kTypeString } },
	{ 238, kTypeObject    , { } },
	{ 239, kTypeString    , { kTypeInt, kTypeInt } },
	{ 240, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 241, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 242, kTypeObject    , { } },
	{ 243, kTypeObject    , { kTypeInt, kTypeString, kTypeEngineType, kTypeInt, kTypeString } },
	{ 244, kTypeEngineType, { kTypeObject, kTypeInt, kTypeInt } },
	{ 245, kTypeObject    , { } },
	{ 246, kTypeInt       , { } },
	{ 247, kTypeInt       , { } },
	{ 248, kTypeInt       , { } },
	{ 249, kTypeString    , { } },
	{ 250, kTypeFloat     , { kTypeString, kTypeInt, kTypeInt, kTypeObject, kTypeObject } },
	{ 251, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 252, kTypeEngineType, { } },
	{ 253, kTypeString    , { kTypeObject } },
	{ 254, kTypeObject    , { } },
	{ 255, kTypeInt       , { kTypeString, kTypeObject, kTypeObject } },
	{ 256, kTypeObject    , { } },
	{ 257, kTypeInt       , { } },
	{ 258, kTypeInt       , { } },
	{ 259, kTypeInt       , { } },
	{ 260, kTypeObject    , { } },
	{ 261, kTypeInt       , { } },
	{ 262, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 263, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 264, kTypeObject    , { kTypeObject } },
	{ 265, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 266, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 267, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 268, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 269, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 270, kTypeEngineType, { } },
	{ 271, kTypeEngineType, { } },
	{ 272, kTypeString    , { kTypeObject } },
	{ 273, kTypeInt       , { kTypeString, kTypeInt, kTypeInt, kTypeObject } },
	{ 274, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 275, kTypeVoid      , { kTypeString } },
	{ 276, kTypeInt       , { kTypeObject } },
	{ 277, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 278, kTypeInt       , { kTypeObject } },
	{ 279, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 280, kTypeInt       , { kTypeObject } },
	{ 281, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 282, kTypeObject    , { } },
	{ 283, kTypeObject    , { } },
	{ 284, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 285, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 286, kTypeEngineType, { kTypeFloat, kTypeFloat } },
	{ 287, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 288, kTypeEngineType, { kTypeEngineType, kTypeEngineType } },
	{ 289, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 290, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 291, kTypeObject    , { } },
	{ 292, kTypeObject    , { } },
	{ 293, kTypeObject    , { } },
	{ 294, kTypeVoid      , { kTypeScriptState } },
	{ 295, kTypeEngineType, { } },
	{ 296, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 297, kTypeInt       , { kTypeObject } },
	{ 298, kTypeFloat     , { kTypeEngineType, kTypeEngineType } },
	{ 299, kTypeEngineType, { kTypeInt } },
	{ 300, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{ 301, kTypeEngineType, { kTypeInt } },
	{ 302, kTypeEngineType, { kTypeInt } },
	{ 303, kTypeFloat     , { kTypeEngineType } },
	{ 304, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 305, kTypeInt       , { kTypeEngineType } },
	{ 306, kTypeInt       , { kTypeEngineType, kTypeObject } },
	{ 307, kTypeEngineType, { kTypeInt, kTypeObject } },
	{ 308, kTypeEngineType, { kTypeInt, kTypeInt, kTypeObject } },
	{ 309, kTypeVoid      , { kTypeEngineType, kTypeObject } },
	{ 310, kTypeVoid      , { kTypeEngineType, kTypeEngineType } },
	{ 311, kTypeInt       , { kTypeObject } },
	{ 312, kTypeInt       , { kTypeObject } },
	{ 313, kTypeVoid      , { kTypeEngineType } },
	{ 314, kTypeEngineType, { kTypeInt } },
	{ 315, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 316, kTypeObject    , { kTypeObject } },
	{ 317, kTypeInt       , { kTypeObject } },
	{ 318, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 319, kTypeObject    , { kTypeObject } },
	{ 320, kTypeInt       , { kTypeObject } },
	{ 321, kTypeInt       , { kTypeObject } },
	{ 322, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 323, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 324, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 325, kTypeInt       , { kTypeObject } },
	{ 326, kTypeObject    , { } },
	{ 327, kTypeVoid      , { kTypeObject } },
	{ 328, kTypeObject    , { kTypeObject } },
	{ 329, kTypeVoid      , { kTypeObject } },
	{ 330, kTypeObject    , { } },
	{ 331, kTypeInt       , { kTypeObject } },
	{ 332, kTypeInt       , { kTypeObject } },
	{ 333, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 334, kTypeVoid      , { kTypeObject } },
	{ 335, kTypeVoid      , { kTypeObject } },
	{ 336, kTypeObject    , { } },
	{ 337, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 338, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 339, kTypeObject    , { kTypeObject } },
	{ 340, kTypeObject    , { kTypeObject } },
	{ 341, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 342, kTypeInt       , { kTypeObject } },
	{ 343, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 344, kTypeInt       , { kTypeString } },
	{ 345, kTypeInt       , { } },
	{ 346, kTypeObject    , { } },
	{ 347, kTypeObject    , { } },
	{ 348, kTypeObject    , { } },
	{ 349, kTypeObject    , { } },
	{ 350, kTypeObject    , { } },
	{ 351, kTypeVoid      , { kTypeEngineType, kTypeFloat } },
	{ 352, kTypeInt       , { } },
	{ 353, kTypeObject    , { } },
	{ 354, kTypeObject    , { kTypeObject } },
	{ 355, kTypeInt       , { kTypeObject } },
	{ 356, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 357, kTypeEngineType, { kTypeEngineType } },
	{ 358, kTypeInt       , { kTypeObject } },
	{ 359, kTypeInt       , { kTypeEngineType } },
	{ 360, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 361, kTypeObject    , { } },
	{ 362, kTypeInt       , { kTypeEngineType } },
	{ 363, kTypeInt       , { kTypeEngineType } },
	{ 364, kTypeObject    , { kTypeInt, kTypeObject } },
	{ 365, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 366, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 367, kTypeVoid      , { kTypeString, kTypeString } },
	{ 368, kTypeInt       , { kTypeString, kTypeString } },
	{ 369, kTypeString    , { kTypeObject } },
	{ 370, kTypeString    , { kTypeObject } },
	{ 371, kTypeString    , { kTypeObject } },
	{ 372, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 373, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 374, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 375, kTypeObject    , { } },
	{ 376, kTypeObject    , { } },
	{ 377, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 378, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 379, kTypeEngineType, { } },
	{ 380, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 381, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 382, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 383, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeFloat } },
	{ 384, kTypeInt       , { kTypeInt } },
	{ 385, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 386, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 387, kTypeEngineType, { kTypeFloat } },
	{ 388, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 389, kTypeInt       , { kTypeObject } },
	{ 390, kTypeString    , { kTypeObject } },
	{ 391, kTypeObject    , { kTypeString } },
	{ 392, kTypeString    , { kTypeObject } },
	{ 393, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 394, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 395, kTypeInt       , { kTypeObject } },
	{ 396, kTypeString    , { kTypeInt } },
	{ 397, kTypeInt       , { kTypeObject } },
	{ 398, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 399, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 400, kTypeVoid      , { kTypeObject } },
	{ 401, kTypeInt       , { kTypeObject } },
	{ 402, kTypeVoid      , { } },
	{ 403, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 404, kTypeVoid      , { } },
	{ 405, kTypeInt       , { } },
	{ 406, kTypeInt       , { } },
	{ 407, kTypeInt       , { } },
	{ 408, kTypeInt       , { } },
	{ 409, kTypeInt       , { kTypeObject } },
	{ 410, kTypeObject    , { } },
	{ 411, kTypeEngineType, { } },
	{ 412, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 413, kTypeVoid      , { kTypeObject } },
	{ 414, kTypeVoid      , { kTypeObject } },
	{ 415, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 416, kTypeVoid      , { kTypeObject, kTypeVector } },
	{ 417, kTypeVoid      , { kTypeString, kTypeObject } },
	{ 418, kTypeInt       , { kTypeObject } },
	{ 419, kTypeObject    , { } },
	{ 420, kTypeInt       , { kTypeObject } },
	{ 421, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 422, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 423, kTypeInt       , { } },
	{ 424, kTypeEngineType, { kTypeObject, kTypeEngineType, kTypeObject } },
	{ 425, kTypeVoid      , { kTypeObject } },
	{ 426, kTypeVoid      , { kTypeObject } },
	{ 427, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 428, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 429, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 430, kTypeVoid      , { kTypeObject } },
	{ 431, kTypeVoid      , { kTypeObject } },
	{ 432, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 433, kTypeVoid      , { kTypeObject } },
	{ 434, kTypeVoid      , { kTypeObject } },
	{ 435, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 436, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 437, kTypeObject    , { } },
	{ 438, kTypeObject    , { } },
	{ 439, kTypeObject    , { } },
	{ 440, kTypeObject    , { } },
	{ 441, kTypeEngineType, { } },
	{ 442, kTypeObject    , { } },
	{ 443, kTypeInt       , { kTypeObject } },
	{ 444, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 445, kTypeInt       , { kTypeObject } },
	{ 446, kTypeInt       , { kTypeString, kTypeInt, kTypeObject } },
	{ 447, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat, kTypeFloat, kTypeInt, kTypeString } },
	{ 448, kTypeFloat     , { } },
	{ 449, kTypeInt       , { } },
	{ 450, kTypeString    , { } },
	{ 451, kTypeEngineType, { kTypeInt } },
	{ 452, kTypeVoid      , { kTypeString } },
	{ 453, kTypeInt       , { kTypeString } },
	{ 454, kTypeInt       , { kTypeString } },
	{ 455, kTypeInt       , { kTypeObject } },
	{ 456, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 457, kTypeEngineType, { kTypeInt } },
	{ 458, kTypeInt       , { kTypeString, kTypeString, kTypeString } },
	{ 459, kTypeEngineType, { } },
	{ 460, kTypeInt       , { kTypeString, kTypeString } },
	{ 461, kTypeVoid      , { kTypeString, kTypeString } },
	{ 462, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 463, kTypeEngineType, { kTypeInt } },
	{ 464, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeObject } },
	{ 465, kTypeEngineType, { } },
	{ 466, kTypeEngineType, { } },
	{ 467, kTypeEngineType, { } },
	{ 468, kTypeEngineType, { } },
	{ 469, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 470, kTypeInt       , { kTypeFloat, kTypeObject } },
	{ 471, kTypeInt       , { kTypeString, kTypeString, kTypeInt } },
	{ 472, kTypeVoid      , { } },
	{ 473, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 474, kTypeVoid      , { kTypeObject, kTypeString, kTypeString, kTypeString, kTypeInt } },
	{ 475, kTypeInt       , { kTypeObject } },
	{ 476, kTypeFloat     , { kTypeString } },
	{ 477, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject } },
	{ 478, kTypeObject    , { kTypeObject } },
	{ 479, kTypeInt       , { kTypeObject } },
	{ 480, kTypeEngineType, { kTypeEngineType } },
	{ 481, kTypeEngineType, { } },
	{ 482, kTypeEngineType, { } },
	{ 483, kTypeInt       , { kTypeObject } },
	{ 484, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 485, kTypeFloat     , { kTypeObject } },
	{ 486, kTypeObject    , { kTypeObject } },
	{ 487, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 488, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 489, kTypeVoid      , { kTypeString, kTypeObject } },
	{ 490, kTypeVoid      , { kTypeString, kTypeObject } },
	{ 491, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 492, kTypeVoid      , { kTypeString, kTypeString, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 493, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 494, kTypeInt       , { kTypeObject } },
	{ 495, kTypeInt       , { kTypeObject } },
	{ 496, kTypeInt       , { kTypeObject } },
	{ 497, kTypeInt       , { kTypeObject } },
	{ 498, kTypeInt       , { kTypeObject } },
	{ 499, kTypeString    , { kTypeObject } },
	{ 500, kTypeString    , { kTypeObject } },
	{ 501, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 502, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeInt } },
	{ 503, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 504, kTypeInt       , { kTypeObject } },
	{ 505, kTypeInt       , { kTypeObject } },
	{ 506, kTypeObject    , { } },
	{ 507, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 508, kTypeInt       , { } },
	{ 509, kTypeVoid      , { kTypeString } },
	{ 510, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 511, kTypeInt       , { kTypeObject } },
	{ 512, kTypeVoid      , { } },
	{ 513, kTypeInt       , { } },
	{ 514, kTypeString    , { kTypeInt, kTypeObject } },
	{ 515, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 516, kTypeVoid      , { kTypeObject } },
	{ 517, kTypeVoid      , { kTypeObject } },
	{ 518, kTypeInt       , { kTypeVector, kTypeFloat } },
	{ 519, kTypeObject    , { kTypeObject } },
	{ 520, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 521, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 522, kTypeInt       , { kTypeObject } },
	{ 523, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject } },
	{ 524, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 525, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 526, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt } },
	{ 527, kTypeInt       , { kTypeObject } },
	{ 528, kTypeInt       , { kTypeObject } },
	{ 529, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 530, kTypeInt       , { kTypeObject } },
	{ 531, kTypeInt       , { kTypeObject } },
	{ 532, kTypeInt       , { kTypeObject } },
	{ 533, kTypeObject    , { kTypeObject } },
	{ 534, kTypeString    , { kTypeObject } },
	{ 535, kTypeInt       , { kTypeObject } },
	{ 536, kTypeInt       , { kTypeObject } },
	{ 537, kTypeInt       , { kTypeObject } },
	{ 538, kTypeString    , { kTypeObject } },
	{ 539, kTypeInt       , { kTypeObject } },
	{ 540, kTypeInt       , { kTypeObject } },
	{ 541, kTypeInt       , { kTypeObject } },
	{ 542, kTypeObject    , { } },
	{ 543, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 544, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 545, kTypeInt       , { kTypeObject } },
	{ 546, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 547, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 548, kTypeObject    , { } },
	{ 549, kTypeObject    , { } },
	{ 550, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 551, kTypeInt       , { kTypeObject } },
	{ 552, kTypeInt       , { kTypeString, kTypeObject } },
	{ 553, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 554, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeString } },
	{ 555, kTypeVoid      , { kTypeObject } },
	{ 556, kTypeObject    , { kTypeObject } },
	{ 557, kTypeVoid      , { } },
	{ 558, kTypeInt       , { kTypeObject } },
	{ 559, kTypeInt       , { kTypeObject } },
	{ 560, kTypeVoid      , { kTypeString } },
	{ 561, kTypeString    , { } },
	{ 562, kTypeObject    , { kTypeObject } },
	{ 563, kTypeVoid      , { kTypeString } },
	{ 564, kTypeVoid      , { kTypeString } },
	{ 565, kTypeVoid      , { kTypeObject } },
	{ 566, kTypeVoid      , { } },
	{ 567, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 568, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 569, kTypeInt       , { kTypeObject } },
	{ 570, kTypeInt       , { kTypeObject } },
	{ 571, kTypeFloat     , { kTypeInt } },
	{ 572, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 573, kTypeVoid      , { kTypeString, kTypeEngineType } },
	{ 574, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 575, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 576, kTypeVoid      , { kTypeInt } },
	{ 577, kTypeInt       , { kTypeObject } },
	{ 578, kTypeVoid      , { } },
	{ 579, kTypeInt       , { } },
	{ 580, kTypeVoid      , { kTypeObject } },
	{ 581, kTypeVoid      , { kTypeObject } },
	{ 582, kTypeString    , { kTypeObject } },
	{ 583, kTypeEngineType, { } },
	{ 584, kTypeObject    , { kTypeObject, kTypeObject } },
	{ 585, kTypeEngineType, { } },
	{ 586, kTypeInt       , { kTypeObject } },
	{ 587, kTypeInt       , { kTypeObject } },
	{ 588, kTypeInt       , { kTypeObject } },
	{ 589, kTypeVoid      , { kTypeString, kTypeString, kTypeFloat, kTypeObject } },
	{ 590, kTypeVoid      , { kTypeString, kTypeString, kTypeInt, kTypeObject } },
	{ 591, kTypeVoid      , { kTypeString, kTypeString, kTypeVector, kTypeObject } },
	{ 592, kTypeVoid      , { kTypeString, kTypeString, kTypeEngineType, kTypeObject } },
	{ 593, kTypeVoid      , { kTypeString, kTypeString, kTypeString, kTypeObject } },
	{ 594, kTypeVoid      , { kTypeString } },
	{ 595, kTypeFloat     , { kTypeString, kTypeString, kTypeObject } },
	{ 596, kTypeInt       , { kTypeString, kTypeString, kTypeObject } },
	{ 597, kTypeVector    , { kTypeString, kTypeString, kTypeObject } },
	{ 598, kTypeEngineType, { kTypeString, kTypeString, kTypeObject } },
	{ 599, kTypeString    , { kTypeString, kTypeString, kTypeObject } },
	{ 600, kTypeObject    , { kTypeObject, kTypeEngineType, kTypeObject, kTypeString } },
	{ 601, kTypeVoid      , { kTypeString, kTypeString, kTypeObject } },
	{ 602, kTypeInt       , { kTypeString, kTypeString, kTypeObject, kTypeObject } },
	{ 603, kTypeObject    , { kTypeString, kTypeString, kTypeEngineType, kTypeObject, kTypeObject } },
	{ 604, kTypeEngineType, { } },
	{ 605, kTypeInt       , { kTypeObject } },
	{ 606, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 607, kTypeInt       , { kTypeObject } },
	{ 608, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 609, kTypeVoid      , { } },
	{ 610, kTypeVoid      , { } },
	{ 611, kTypeVoid      , { } },
	{ 612, kTypeVoid      , { } },
	{ 613, kTypeVoid      , { } },
	{ 614, kTypeVoid      , { } },
	{ 615, kTypeVoid      , { } },
	{ 616, kTypeInt       , { kTypeObject } },
	{ 617, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 618, kTypeEngineType, { kTypeFloat } },
	{ 619, kTypeInt       , { kTypeString, kTypeInt, kTypeString, kTypeFloat, kTypeObject } },
	{ 620, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 621, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeFloat, kTypeInt, kTypeString, kTypeInt } },
	{ 622, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeObject } },
	{ 623, kTypeFloat     , { kTypeObject } },
	{ 624, kTypeVoid      , { kTypeString } },
	{ 625, kTypeVoid      , { kTypeObject } },
	{ 626, kTypeInt       , { kTypeObject } },
	{ 627, kTypeInt       , { kTypeObject } },
	{ 628, kTypeInt       , { kTypeString, kTypeObject } },
	{ 629, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 630, kTypeVoid      , { kTypeInt } },
	{ 631, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 632, kTypeVoid      , { kTypeString, kTypeString, kTypeObject, kTypeInt } },
	{ 633, kTypeInt       , { kTypeObject } },
	{ 634, kTypeString    , { kTypeInt } },
	{ 635, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 636, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 637, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt } },
	{ 638, kTypeString    , { kTypeInt, kTypeObject } },
	{ 639, kTypeString    , { kTypeObject } },
	{ 640, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 641, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 642, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 643, kTypeInt       , { kTypeString, kTypeObject } },
	{ 644, kTypeString    , { kTypeObject } },
	{ 645, kTypeInt       , { kTypeString, kTypeObject } },
	{ 646, kTypeInt       , { kTypeObject } },
	{ 647, kTypeVoid      , { } },
	{ 648, kTypeInt       , { kTypeObject } },
	{ 649, kTypeFloat     , { kTypeInt, kTypeObject } },
	{ 650, kTypeVoid      , { kTypeString, kTypeString, kTypeObject } },
	{ 651, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt } },
	{ 652, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeObject } },
	{ 653, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 654, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 655, kTypeObject    , { kTypeObject } },
	{ 656, kTypeVoid      , { kTypeInt } },
	{ 657, kTypeString    , { kTypeString, kTypeString, kTypeString } },
	{ 658, kTypeString    , { kTypeString, kTypeString } },
	{ 659, kTypeInt       , { } },
	{ 660, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 661, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 662, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 663, kTypeVoid      , { kTypeObject, kTypeString, kTypeString, kTypeObject, kTypeVector } },
	{ 664, kTypeString    , { kTypeObject } },
	{ 665, kTypeVoid      , { kTypeObject } },
	{ 666, kTypeInt       , { kTypeString, kTypeString } },
	{ 667, kTypeVoid      , { kTypeObject } },
	{ 668, kTypeInt       , { kTypeObject } },
	{ 669, kTypeVoid      , { kTypeObject } },
	{ 670, kTypeVoid      , { kTypeObject } },
	{ 671, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 672, kTypeInt       , { kTypeInt } },
	{ 673, kTypeInt       , { kTypeObject } },
	{ 674, kTypeVoid      , { kTypeObject, kTypeString, kTypeString, kTypeObject, kTypeString } },
	{ 675, kTypeVoid      , { kTypeObject } },
	{ 676, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 677, kTypeInt       , { kTypeObject } },
	{ 678, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 679, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 680, kTypeVoid      , { kTypeObject } },
	{ 681, kTypeVoid      , { kTypeInt } },
	{ 682, kTypeObject    , { kTypeObject } },
	{ 683, kTypeVoid      , { kTypeString } },
	{ 684, kTypeVoid      , { kTypeString } },
	{ 685, kTypeInt       , { kTypeObject } },
	{ 686, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 687, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 688, kTypeInt       , { kTypeObject } },
	{ 689, kTypeInt       , { kTypeObject } },
	{ 690, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 691, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 692, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 693, kTypeObject    , { } },
	{ 694, kTypeFloat     , { kTypeInt } },
	{ 695, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 696, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 697, kTypeVoid      , { kTypeObject } },
	{ 698, kTypeVoid      , { kTypeObject } },
	{ 699, kTypeInt       , { kTypeObject } },
	{ 700, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 701, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 702, kTypeVoid      , { } },
	{ 703, kTypeVoid      , { } },
	{ 704, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 705, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 706, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 707, kTypeObject    , { } },
	{ 708, kTypeInt       , { kTypeObject } },
	{ 709, kTypeVoid      , { kTypeString, kTypeInt, kTypeObject } },
	{ 710, kTypeString    , { kTypeString, kTypeString, kTypeInt } },
	{ 711, kTypeInt       , { kTypeObject } },
	{ 712, kTypeInt       , { kTypeObject } },
	{ 713, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 714, kTypeInt       , { kTypeObject } },
	{ 715, kTypeVoid      , { kTypeObject } },
	{ 716, kTypeVoid      , { } },
	{ 717, kTypeVoid      , { } },
	{ 718, kTypeVoid      , { } },
	{ 719, kTypeVoid      , { kTypeObject } },
	{ 720, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 721, kTypeVoid      , { kTypeString } },
	{ 722, kTypeVoid      , { kTypeString } },
	{ 723, kTypeEngineType, { kTypeString, kTypeString, kTypeInt } },
	{ 724, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 725, kTypeInt       , { kTypeEngineType } },
	{ 726, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 727, kTypeObject    , { kTypeInt } },
	{ 728, kTypeInt       , { kTypeString, kTypeInt } },
	{ 729, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 730, kTypeInt       , { kTypeString } },
	{ 731, kTypeInt       , { kTypeString } },
	{ 732, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 733, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 734, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 735, kTypeVoid      , { kTypeObject, kTypeString, kTypeString, kTypeString } },
	{ 736, kTypeVoid      , { kTypeObject, kTypeVector } },
	{ 737, kTypeVoid      , { kTypeInt, kTypeObject, kTypeObject } },
	{ 738, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 739, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 740, kTypeVoid      , { kTypeObject } },
	{ 741, kTypeObject    , { kTypeObject } },
	{ 742, kTypeObject    , { kTypeObject } },
	{ 743, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 744, kTypeInt       , { kTypeObject, kTypeString } },
	{ 745, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 746, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 747, kTypeInt       , { kTypeString } },
	{ 748, kTypeInt       , { kTypeScriptState, kTypeObject, kTypeInt } },
	{ 749, kTypeInt       , { kTypeObject } },
	{ 750, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 751, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 752, kTypeString    , { kTypeObject, kTypeString } },
	{ 753, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 754, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 755, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 756, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 757, kTypeObject    , { } },
	{ 758, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 759, kTypeInt       , { kTypeObject } },
	{ 760, kTypeInt       , { } },
	{ 761, kTypeFloat     , { } },
	{ 762, kTypeInt       , { kTypeObject } },
	{ 763, kTypeInt       , { kTypeObject } },
	{ 764, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 765, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 766, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 767, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 768, kTypeVoid      , { kTypeObject, kTypeInt, kTypeObject } },
	{ 769, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 770, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt } },
	{ 771, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 772, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 773, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeInt, kTypeObject, kTypeInt } },
	{ 774, kTypeInt       , { kTypeObject } },
	{ 775, kTypeInt       , { kTypeObject } },
	{ 776, kTypeObject    , { kTypeInt, kTypeObject, kTypeInt, kTypeObject } },
	{ 777, kTypeObject    , { kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 778, kTypeVoid      , { kTypeObject, kTypeInt, kTypeObject } },
	{ 779, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 780, kTypeVoid      , { kTypeFloat, kTypeObject } },
	{ 781, kTypeFloat     , { kTypeObject } },
	{ 782, kTypeVoid      , { } },
	{ 783, kTypeVoid      , { } },
	{ 784, kTypeVoid      , { } },
	{ 785, kTypeVoid      , { } },
	{ 786, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 787, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeString, kTypeString, kTypeString, kTypeString, kTypeString } },
	{ 788, kTypeInt       , { } },
	{ 789, kTypeInt       , { kTypeObject } },
	{ 790, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 791, kTypeVoid      , { kTypeObject } },
	{ 792, kTypeInt       , { kTypeObject } },
	{ 793, kTypeInt       , { kTypeObject, kTypeString } },
	{ 794, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 795, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeString, kTypeString, kTypeString, kTypeObject, kTypeInt } },
	{ 796, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 797, kTypeVoid      , { kTypeObject } },
	{ 798, kTypeVoid      , { kTypeObject } },
	{ 799, kTypeVoid      , { kTypeString, kTypeString, kTypeString } },
	{ 800, kTypeVoid      , { kTypeObject } },
	{ 801, kTypeVoid      , { kTypeObject } },
	{ 802, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 803, kTypeInt       , { kTypeObject } },
	{ 804, kTypeVoid      , { kTypeObject } },
	{ 805, kTypeInt       , { kTypeString, kTypeObject, kTypeInt } },
	{ 806, kTypeVoid      , { kTypeObject } },
	{ 807, kTypeInt       , { kTypeString, kTypeString, kTypeInt } },
	{ 808, kTypeInt       , { kTypeString, kTypeString, kTypeInt } },
	{ 809, kTypeVoid      , { } },
	{ 810, kTypeVoid      , { } },
	{ 811, kTypeVoid      , { } },
	{ 812, kTypeVoid      , { } },
	{ 813, kTypeVoid      , { } },
	{ 814, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 815, kTypeInt       , { kTypeObject } },
	{ 816, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 817, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 818, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 819, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 820, kTypeInt       , { kTypeObject } },
	{ 821, kTypeInt       , { kTypeFloat, kTypeInt, kTypeInt, kTypeObject } },
	{ 822, kTypeVoid      , { kTypeObject } },
	{ 823, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 824, kTypeFloat     , { kTypeFloat, kTypeFloat } },
	{ 825, kTypeObject    , { kTypeObject } },
	{ 826, kTypeVoid      , { kTypeObject } },
	{ 827, kTypeInt       , { kTypeObject } },
	{ 828, kTypeString    , { } },
	{ 829, kTypeVoid      , { } },
	{ 830, kTypeInt       , { } },
	{ 831, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 832, kTypeEngineType, { kTypeFloat, kTypeObject, kTypeObject } },
	{ 833, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 834, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 835, kTypeInt       , { kTypeObject } },
	{ 836, kTypeInt       , { kTypeObject } },
	{ 837, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 838, kTypeInt       , { kTypeObject } },
	{ 839, kTypeInt       , { } },
	{ 840, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 841, kTypeInt       , { kTypeObject } },
	{ 842, kTypeVoid      , { kTypeObject } },
	{ 843, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 844, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 845, kTypeVoid      , { } },
	{ 846, kTypeVoid      , { kTypeInt } },
	{ 847, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject, kTypeInt } }
};

/** The table defining the default values for the parameters of each engine function. */
const Functions::FunctionDefaults Functions::kFunctionDefaults[] = {
	{   0, { } },
	{   1, { } },
	{   2, { &kDefaultInt18, &kDefaultInt9 } },
	{   3, { &kDefaultInt18, &kDefaultInt9 } },
	{   4, { } },
	{   5, { } },
	{   6, { } },
	{   7, { } },
	{   8, { &kDefaultObjectSelf } },
	{   9, { &kDefaultFalse, &kDefaultObjectInvalid } },
	{  10, { } },
	{  11, { } },
	{  12, { &kDefaultInt0, &kDefaultInt0, &kDefaultInt0 } },
	{  13, { } },
	{  14, { } },
	{  15, { } },
	{  16, { } },
	{  17, { } },
	{  18, { } },
	{  19, { } },
	{  20, { } },
	{  21, { &kDefaultFalse } },
	{  22, { &kDefaultFalse, &kDefaultFloat1_0 } },
	{  23, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{  24, { } },
	{  25, { } },
	{  26, { } },
	{  27, { } },
	{  28, { } },
	{  29, { } },
	{  30, { } },
	{  31, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{  32, { &kDefaultFalse } },
	{  33, { &kDefaultFalse } },
	{  34, { } },
	{  35, { &kDefaultTrue } },
	{  36, { &kDefaultObjectSelf } },
	{  37, { &kDefaultFalse } },
	{  38, { &kDefaultObjectSelf, &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{  39, { &kDefaultTalkVolumeTalk } },
	{  40, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{  41, { } },
	{  42, { } },
	{  43, { } },
	{  44, { } },
	{  45, { &kDefaultFloatMinus1_0, &kDefaultFloatMinus1_0, &kDefaultCameraTransitionTypeSnap, &kDefaultInt0 } },
	{  46, { &kDefaultInt0, &kDefaultFloat0_0 } },
	{  47, { } },
	{  48, { &kDefaultFalse, &kDefaultProjectilePathTypeDefault, &kDefaultFalse } },
	{  49, { &kDefaultObjectSelf } },
	{  50, { &kDefaultObjectSelf } },
	{  51, { } },
	{  52, { } },
	{  53, { } },
	{  54, { } },
	{  55, { } },
	{  56, { } },
	{  57, { } },
	{  58, { } },
	{  59, { } },
	{  60, { } },
	{  61, { } },
	{  62, { } },
	{  63, { } },
	{  64, { } },
	{  65, { } },
	{  66, { } },
	{  67, { } },
	{  68, { } },
	{  69, { } },
	{  70, { } },
	{  71, { } },
	{  72, { } },
	{  73, { } },
	{  74, { } },
	{  75, { } },
	{  76, { } },
	{  77, { } },
	{  78, { } },
	{  79, { &kDefaultInt0, &kDefaultStringEmpty } },
	{  80, { &kDefaultObjectSelf } },
	{  81, { &kDefaultIntMinus1 } },
	{  82, { } },
	{  83, { &kDefaultInt0, &kDefaultFloat0_0 } },
	{  84, { &kDefaultObjectSelf, &kDefaultInt0, &kDefaultFalse } },
	{  85, { } },
	{  86, { } },
	{  87, { } },
	{  88, { } },
	{  89, { } },
	{  90, { } },
	{  91, { } },
	{  92, { } },
	{  93, { &kDefaultObjectInvalid } },
	{  94, { &kDefaultObjectInvalid } },
	{  95, { &kDefaultInt1 } },
	{  96, { &kDefaultInt1 } },
	{  97, { &kDefaultInt1 } },
	{  98, { &kDefaultInt1 } },
	{  99, { &kDefaultInt1 } },
	{ 100, { &kDefaultInt1 } },
	{ 101, { &kDefaultInt1 } },
	{ 102, { &kDefaultInt1 } },
	{ 103, { &kDefaultInt1 } },
	{ 104, { } },
	{ 105, { &kDefaultObjectSelf } },
	{ 106, { } },
	{ 107, { } },
	{ 108, { &kDefaultFloatMinus1_0, &kDefaultFloat0_0, &kDefaultObjectInvalid, &kDefaultStringEmpty } },
	{ 109, { } },
	{ 110, { } },
	{ 111, { &kDefaultIntMinus1 } },
	{ 112, { } },
	{ 113, { } },
	{ 114, { } },
	{ 115, { &kDefaultObjectSelf } },
	{ 116, { &kDefaultObjectSelf } },
	{ 117, { &kDefaultStringEmpty, &kDefaultObjectSelf, &kDefaultObjectInvalid } },
	{ 118, { &kDefaultObjectSelf } },
	{ 119, { &kDefaultObjectSelf } },
	{ 120, { &kDefaultObjectSelf } },
	{ 121, { } },
	{ 122, { } },
	{ 123, { } },
	{ 124, { } },
	{ 125, { &kDefaultTrue } },
	{ 126, { &kDefaultObjectSelf } },
	{ 127, { &kDefaultTrue, &kDefaultObjectSelf } },
	{ 128, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0, &kDefaultFloat0_5 } },
	{ 129, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0, &kDefaultFloat0_5 } },
	{ 130, { &kDefaultObjectSelf } },
	{ 131, { } },
	{ 132, { } },
	{ 133, { &kDefaultFalse, &kDefaultTrue } },
	{ 134, { &kDefaultObjectInvalid } },
	{ 135, { } },
	{ 136, { } },
	{ 137, { } },
	{ 138, { &kDefaultObjectSelf } },
	{ 139, { } },
	{ 140, { } },
	{ 141, { } },
	{ 142, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{ 143, { } },
	{ 144, { } },
	{ 145, { } },
	{ 146, { &kDefaultObjectSelf } },
	{ 147, { &kDefaultObjectSelf } },
	{ 148, { } },
	{ 149, { } },
	{ 150, { } },
	{ 151, { } },
	{ 152, { } },
	{ 153, { } },
	{ 154, { } },
	{ 155, { &kDefaultObjectSelf } },
	{ 156, { } },
	{ 157, { } },
	{ 158, { } },
	{ 159, { } },
	{ 160, { } },
	{ 161, { } },
	{ 162, { &kDefaultObjectSelf } },
	{ 163, { &kDefaultObjectSelf } },
	{ 164, { } },
	{ 165, { } },
	{ 166, { } },
	{ 167, { &kDefaultFloat0_0, &kDefaultInt0 } },
	{ 168, { } },
	{ 169, { &kDefaultObjectSelf } },
	{ 170, { } },
	{ 171, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 172, { &kDefaultObjectSelf } },
	{ 173, { } },
	{ 174, { } },
	{ 175, { } },
	{ 176, { &kDefaultInt0 } },
	{ 177, { } },
	{ 178, { } },
	{ 179, { } },
	{ 180, { &kDefaultFalse } },
	{ 181, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 182, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 183, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 184, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 185, { } },
	{ 186, { } },
	{ 187, { &kDefaultObjectSelf } },
	{ 188, { &kDefaultObjectSelf } },
	{ 189, { } },
	{ 190, { } },
	{ 191, { } },
	{ 192, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 193, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 194, { } },
	{ 195, { } },
	{ 196, { &kDefaultTrue } },
	{ 197, { } },
	{ 198, { } },
	{ 199, { } },
	{ 200, { &kDefaultInt0 } },
	{ 201, { } },
	{ 202, { } },
	{ 203, { &kDefaultStringEmpty } },
	{ 204, { &kDefaultStringEmpty, &kDefaultFalse, &kDefaultTrue, &kDefaultFalse } },
	{ 205, { } },
	{ 206, { } },
	{ 207, { } },
	{ 208, { } },
	{ 209, { &kDefaultObjectSelf } },
	{ 210, { } },
	{ 211, { } },
	{ 212, { &kDefaultObjectSelf } },
	{ 213, { } },
	{ 214, { } },
	{ 215, { } },
	{ 216, { &kDefaultFloat0_0 } },
	{ 217, { } },
	{ 218, { } },
	{ 219, { } },
	{ 220, { &kDefaultFloat0_0, &kDefaultStringEmpty, &kDefaultIntMinus1 } },
	{ 221, { &kDefaultTalkVolumeTalk } },
	{ 222, { } },
	{ 223, { } },
	{ 224, { } },
	{ 225, { } },
	{ 226, { &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{ 227, { &kDefaultObjectTypeAll, &kDefaultObjectSelf, &kDefaultInt1 } },
	{ 228, { &kDefaultInt1 } },
	{ 229, { &kDefaultObjectSelf, &kDefaultInt1, &kDefaultObjectTypeAll } },
	{ 230, { } },
	{ 231, { } },
	{ 232, { } },
	{ 233, { } },
	{ 234, { &kDefaultFalse, &kDefaultProjectilePathTypeDefault, &kDefaultFalse } },
	{ 235, { &kDefaultObjectSelf } },
	{ 236, { } },
	{ 237, { } },
	{ 238, { } },
	{ 239, { &kDefaultGenderMale } },
	{ 240, { &kDefaultTalkVolumeTalk } },
	{ 241, { &kDefaultFloat0_0 } },
	{ 242, { } },
	{ 243, { &kDefaultFalse, &kDefaultStringEmpty } },
	{ 244, { &kDefaultTrue } },
	{ 245, { } },
	{ 246, { } },
	{ 247, { } },
	{ 248, { } },
	{ 249, { } },
	{ 250, { } },
	{ 251, { &kDefaultObjectSelf } },
	{ 252, { } },
	{ 253, { } },
	{ 254, { } },
	{ 255, { &kDefaultStringEmpty, &kDefaultObjectInvalid, &kDefaultObjectInvalid } },
	{ 256, { } },
	{ 257, { } },
	{ 258, { } },
	{ 259, { } },
	{ 260, { } },
	{ 261, { } },
	{ 262, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{ 263, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{ 264, { &kDefaultObjectSelf } },
	{ 265, { } },
	{ 266, { } },
	{ 267, { } },
	{ 268, { } },
	{ 269, { } },
	{ 270, { } },
	{ 271, { } },
	{ 272, { } },
	{ 273, { &kDefaultIntMinus1, &kDefaultObjectSelf } },
	{ 274, { &kDefaultObjectSelf } },
	{ 275, { } },
	{ 276, { &kDefaultObjectSelf } },
	{ 277, { &kDefaultObjectSelf } },
	{ 278, { &kDefaultObjectSelf } },
	{ 279, { &kDefaultObjectSelf } },
	{ 280, { &kDefaultObjectSelf } },
	{ 281, { &kDefaultObjectSelf } },
	{ 282, { } },
	{ 283, { } },
	{ 284, { } },
	{ 285, { &kDefaultObjectSelf } },
	{ 286, { &kDefaultFloat1_0 } },
	{ 287, { } },
	{ 288, { } },
	{ 289, { &kDefaultObjectSelf } },
	{ 290, { &kDefaultObjectSelf } },
	{ 291, { } },
	{ 292, { } },
	{ 293, { } },
	{ 294, { } },
	{ 295, { } },
	{ 296, { &kDefaultObjectSelf } },
	{ 297, { &kDefaultObjectSelf } },
	{ 298, { } },
	{ 299, { } },
	{ 300, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{ 301, { } },
	{ 302, { } },
	{ 303, { } },
	{ 304, { &kDefaultObjectSelf } },
	{ 305, { } },
	{ 306, { &kDefaultObjectSelf } },
	{ 307, { &kDefaultObjectSelf } },
	{ 308, { &kDefaultObjectSelf } },
	{ 309, { } },
	{ 310, { } },
	{ 311, { } },
	{ 312, { } },
	{ 313, { } },
	{ 314, { } },
	{ 315, { &kDefaultObjectSelf } },
	{ 316, { &kDefaultObjectSelf } },
	{ 317, { &kDefaultObjectSelf } },
	{ 318, { } },
	{ 319, { &kDefaultObjectSelf } },
	{ 320, { &kDefaultObjectSelf } },
	{ 321, { &kDefaultObjectSelf } },
	{ 322, { } },
	{ 323, { &kDefaultTrue, &kDefaultFalse } },
	{ 324, { } },
	{ 325, { } },
	{ 326, { } },
	{ 327, { &kDefaultObjectSelf } },
	{ 328, { } },
	{ 329, { } },
	{ 330, { } },
	{ 331, { &kDefaultObjectSelf } },
	{ 332, { } },
	{ 333, { } },
	{ 334, { &kDefaultObjectSelf } },
	{ 335, { &kDefaultObjectSelf } },
	{ 336, { } },
	{ 337, { } },
	{ 338, { } },
	{ 339, { &kDefaultObjectSelf } },
	{ 340, { &kDefaultObjectSelf } },
	{ 341, { &kDefaultObjectSelf } },
	{ 342, { &kDefaultObjectSelf } },
	{ 343, { &kDefaultObjectSelf } },
	{ 344, { } },
	{ 345, { } },
	{ 346, { } },
	{ 347, { } },
	{ 348, { } },
	{ 349, { } },
	{ 350, { } },
	{ 351, { } },
	{ 352, { } },
	{ 353, { } },
	{ 354, { &kDefaultObjectSelf } },
	{ 355, { &kDefaultObjectSelf } },
	{ 356, { } },
	{ 357, { } },
	{ 358, { } },
	{ 359, { } },
	{ 360, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{ 361, { } },
	{ 362, { } },
	{ 363, { } },
	{ 364, { &kDefaultObjectSelf } },
	{ 365, { &kDefaultObjectSelf } },
	{ 366, { &kDefaultObjectSelf } },
	{ 367, { } },
	{ 368, { } },
	{ 369, { } },
	{ 370, { } },
	{ 371, { } },
	{ 372, { } },
	{ 373, { } },
	{ 374, { } },
	{ 375, { } },
	{ 376, { } },
	{ 377, { &kDefaultObjectSelf } },
	{ 378, { &kDefaultIntMinus1 } },
	{ 379, { } },
	{ 380, { &kDefaultTrue } },
	{ 381, { &kDefaultTrue } },
	{ 382, { &kDefaultFalse, &kDefaultFloat30_0 } },
	{ 383, { &kDefaultFalse, &kDefaultFloat1_0, &kDefaultFloat30_0 } },
	{ 384, { } },
	{ 385, { &kDefaultInt1 } },
	{ 386, { } },
	{ 387, { } },
	{ 388, { } },
	{ 389, { &kDefaultObjectSelf } },
	{ 390, { &kDefaultObjectSelf } },
	{ 391, { } },
	{ 392, { &kDefaultObjectSelf } },
	{ 393, { &kDefaultTrue } },
	{ 394, { &kDefaultTrue } },
	{ 395, { } },
	{ 396, { } },
	{ 397, { } },
	{ 398, { } },
	{ 399, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{ 400, { &kDefaultObjectInvalid } },
	{ 401, { &kDefaultObjectSelf } },
	{ 402, { } },
	{ 403, { } },
	{ 404, { } },
	{ 405, { } },
	{ 406, { } },
	{ 407, { } },
	{ 408, { } },
	{ 409, { &kDefaultObjectSelf } },
	{ 410, { } },
	{ 411, { } },
	{ 412, { } },
	{ 413, { } },
	{ 414, { } },
	{ 415, { } },
	{ 416, { } },
	{ 417, { &kDefaultStringEmpty, &kDefaultObjectInvalid } },
	{ 418, { &kDefaultObjectSelf } },
	{ 419, { } },
	{ 420, { } },
	{ 421, { &kDefaultObjectSelf } },
	{ 422, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{ 423, { } },
	{ 424, { &kDefaultObjectInvalid } },
	{ 425, { } },
	{ 426, { } },
	{ 427, { } },
	{ 428, { } },
	{ 429, { } },
	{ 430, { } },
	{ 431, { } },
	{ 432, { } },
	{ 433, { } },
	{ 434, { } },
	{ 435, { } },
	{ 436, { } },
	{ 437, { } },
	{ 438, { } },
	{ 439, { } },
	{ 440, { } },
	{ 441, { } },
	{ 442, { } },
	{ 443, { } },
	{ 444, { &kDefaultFalse } },
	{ 445, { } },
	{ 446, { &kDefaultInt1, &kDefaultObjectSelf } },
	{ 447, { } },
	{ 448, { } },
	{ 449, { } },
	{ 450, { } },
	{ 451, { } },
	{ 452, { } },
	{ 453, { } },
	{ 454, { } },
	{ 455, { &kDefaultObjectSelf } },
	{ 456, { } },
	{ 457, { } },
	{ 458, { &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 459, { } },
	{ 460, { } },
	{ 461, { } },
	{ 462, { &kDefaultObjectSelf } },
	{ 463, { } },
	{ 464, { &kDefaultInt0, &kDefaultObjectSelf } },
	{ 465, { } },
	{ 466, { } },
	{ 467, { } },
	{ 468, { } },
	{ 469, { } },
	{ 470, { &kDefaultObjectSelf } },
	{ 471, { &kDefaultFalse } },
	{ 472, { } },
	{ 473, { &kDefaultTrue } },
	{ 474, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultFalse } },
	{ 475, { } },
	{ 476, { } },
	{ 477, { &kDefaultObjectSelf } },
	{ 478, { &kDefaultObjectSelf } },
	{ 479, { } },
	{ 480, { } },
	{ 481, { } },
	{ 482, { } },
	{ 483, { &kDefaultObjectSelf } },
	{ 484, { &kDefaultObjectSelf } },
	{ 485, { &kDefaultObjectSelf } },
	{ 486, { &kDefaultObjectSelf } },
	{ 487, { &kDefaultObjectSelf } },
	{ 488, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 489, { &kDefaultObjectSelf } },
	{ 490, { &kDefaultObjectSelf } },
	{ 491, { } },
	{ 492, { &kDefaultStringEmpty, &kDefaultInt1, &kDefaultInt1, &kDefaultInt1, &kDefaultInt0, &kDefaultInt0 } },
	{ 493, { &kDefaultObjectSelf } },
	{ 494, { } },
	{ 495, { } },
	{ 496, { } },
	{ 497, { } },
	{ 498, { } },
	{ 499, { } },
	{ 500, { } },
	{ 501, { &kDefaultProjectilePathTypeDefault } },
	{ 502, { &kDefaultProjectilePathTypeDefault } },
	{ 503, { &kDefaultObjectSelf } },
	{ 504, { &kDefaultObjectSelf } },
	{ 505, { &kDefaultObjectSelf } },
	{ 506, { } },
	{ 507, { } },
	{ 508, { } },
	{ 509, { } },
	{ 510, { &kDefaultObjectSelf } },
	{ 511, { } },
	{ 512, { } },
	{ 513, { } },
	{ 514, { &kDefaultObjectSelf } },
	{ 515, { } },
	{ 516, { } },
	{ 517, { } },
	{ 518, { } },
	{ 519, { } },
	{ 520, { } },
	{ 521, { } },
	{ 522, { &kDefaultObjectSelf } },
	{ 523, { &kDefaultObjectSelf } },
	{ 524, { &kDefaultObjectSelf } },
	{ 525, { &kDefaultTrue } },
	{ 526, { &kDefaultTrue } },
	{ 527, { } },
	{ 528, { } },
	{ 529, { } },
	{ 530, { } },
	{ 531, { } },
	{ 532, { } },
	{ 533, { } },
	{ 534, { } },
	{ 535, { } },
	{ 536, { } },
	{ 537, { } },
	{ 538, { } },
	{ 539, { } },
	{ 540, { } },
	{ 541, { } },
	{ 542, { } },
	{ 543, { &kDefaultObjectSelf } },
	{ 544, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 545, { &kDefaultObjectSelf } },
	{ 546, { } },
	{ 547, { } },
	{ 548, { } },
	{ 549, { } },
	{ 550, { } },
	{ 551, { } },
	{ 552, { } },
	{ 553, { } },
	{ 554, { &kDefaultTrue, &kDefaultTrue, &kDefaultInt0, &kDefaultStringEmpty } },
	{ 555, { } },
	{ 556, { &kDefaultObjectSelf } },
	{ 557, { } },
	{ 558, { } },
	{ 559, { } },
	{ 560, { } },
	{ 561, { } },
	{ 562, { } },
	{ 563, { } },
	{ 564, { } },
	{ 565, { } },
	{ 566, { } },
	{ 567, { } },
	{ 568, { } },
	{ 569, { } },
	{ 570, { } },
	{ 571, { } },
	{ 572, { &kDefaultObjectSelf } },
	{ 573, { } },
	{ 574, { &kDefaultInt0, &kDefaultIntMinus1, &kDefaultFalse } },
	{ 575, { &kDefaultInt0, &kDefaultIntMinus1, &kDefaultFalse } },
	{ 576, { } },
	{ 577, { } },
	{ 578, { } },
	{ 579, { } },
	{ 580, { } },
	{ 581, { } },
	{ 582, { } },
	{ 583, { } },
	{ 584, { &kDefaultObjectInvalid } },
	{ 585, { } },
	{ 586, { } },
	{ 587, { &kDefaultObjectSelf } },
	{ 588, { } },
	{ 589, { &kDefaultObjectInvalid } },
	{ 590, { &kDefaultObjectInvalid } },
	{ 591, { &kDefaultObjectInvalid } },
	{ 592, { &kDefaultObjectInvalid } },
	{ 593, { &kDefaultObjectInvalid } },
	{ 594, { } },
	{ 595, { &kDefaultObjectInvalid } },
	{ 596, { &kDefaultObjectInvalid } },
	{ 597, { &kDefaultObjectInvalid } },
	{ 598, { &kDefaultObjectInvalid } },
	{ 599, { &kDefaultObjectInvalid } },
	{ 600, { &kDefaultObjectInvalid, &kDefaultStringEmpty } },
	{ 601, { &kDefaultObjectInvalid } },
	{ 602, { &kDefaultObjectInvalid } },
	{ 603, { &kDefaultObjectInvalid, &kDefaultObjectInvalid } },
	{ 604, { } },
	{ 605, { } },
	{ 606, { } },
	{ 607, { } },
	{ 608, { } },
	{ 609, { } },
	{ 610, { } },
	{ 611, { } },
	{ 612, { } },
	{ 613, { } },
	{ 614, { } },
	{ 615, { } },
	{ 616, { } },
	{ 617, { &kDefaultObjectSelf } },
	{ 618, { } },
	{ 619, { &kDefaultObjectSelf } },
	{ 620, { &kDefaultObjectSelf } },
	{ 621, { } },
	{ 622, { &kDefaultFloat0_0, &kDefaultObjectSelf } },
	{ 623, { &kDefaultObjectSelf } },
	{ 624, { } },
	{ 625, { &kDefaultObjectSelf } },
	{ 626, { &kDefaultObjectSelf } },
	{ 627, { &kDefaultObjectSelf } },
	{ 628, { &kDefaultObjectSelf } },
	{ 629, { &kDefaultObjectSelf } },
	{ 630, { } },
	{ 631, { &kDefaultInt0 } },
	{ 632, { &kDefaultFalse } },
	{ 633, { } },
	{ 634, { } },
	{ 635, { } },
	{ 636, { } },
	{ 637, { &kDefaultInt0 } },
	{ 638, { } },
	{ 639, { } },
	{ 640, { } },
	{ 641, { } },
	{ 642, { } },
	{ 643, { } },
	{ 644, { } },
	{ 645, { } },
	{ 646, { } },
	{ 647, { } },
	{ 648, { } },
	{ 649, { } },
	{ 650, { } },
	{ 651, { } },
	{ 652, { } },
	{ 653, { } },
	{ 654, { } },
	{ 655, { &kDefaultObjectSelf } },
	{ 656, { &kDefaultTrue } },
	{ 657, { } },
	{ 658, { } },
	{ 659, { } },
	{ 660, { } },
	{ 661, { } },
	{ 662, { } },
	{ 663, { &kDefaultVector0 } },
	{ 664, { } },
	{ 665, { } },
	{ 666, { } },
	{ 667, { } },
	{ 668, { } },
	{ 669, { } },
	{ 670, { } },
	{ 671, { } },
	{ 672, { } },
	{ 673, { } },
	{ 674, { } },
	{ 675, { } },
	{ 676, { } },
	{ 677, { } },
	{ 678, { } },
	{ 679, { } },
	{ 680, { } },
	{ 681, { } },
	{ 682, { } },
	{ 683, { } },
	{ 684, { } },
	{ 685, { } },
	{ 686, { } },
	{ 687, { } },
	{ 688, { } },
	{ 689, { } },
	{ 690, { } },
	{ 691, { &kDefaultTalkVolumeTalk } },
	{ 692, { &kDefaultTrue, &kDefaultFalse } },
	{ 693, { } },
	{ 694, { } },
	{ 695, { &kDefaultFadeSpeedMedium } },
	{ 696, { &kDefaultFadeSpeedMedium } },
	{ 697, { } },
	{ 698, { } },
	{ 699, { } },
	{ 700, { } },
	{ 701, { &kDefaultIntMinus1 } },
	{ 702, { } },
	{ 703, { } },
	{ 704, { &kDefaultFalse } },
	{ 705, { } },
	{ 706, { } },
	{ 707, { } },
	{ 708, { &kDefaultObjectSelf } },
	{ 709, { } },
	{ 710, { } },
	{ 711, { &kDefaultObjectSelf } },
	{ 712, { &kDefaultObjectSelf } },
	{ 713, { } },
	{ 714, { } },
	{ 715, { } },
	{ 716, { } },
	{ 717, { } },
	{ 718, { } },
	{ 719, { } },
	{ 720, { &kDefaultTrue } },
	{ 721, { } },
	{ 722, { } },
	{ 723, { } },
	{ 724, { } },
	{ 725, { } },
	{ 726, { &kDefaultIntMinus1 } },
	{ 727, { } },
	{ 728, { } },
	{ 729, { } },
	{ 730, { } },
	{ 731, { } },
	{ 732, { } },
	{ 733, { } },
	{ 734, { } },
	{ 735, { } },
	{ 736, { } },
	{ 737, { } },
	{ 738, { } },
	{ 739, { } },
	{ 740, { } },
	{ 741, { } },
	{ 742, { } },
	{ 743, { } },
	{ 744, { } },
	{ 745, { } },
	{ 746, { &kDefaultTrue } },
	{ 747, { } },
	{ 748, { &kDefaultObjectSelf, &kDefaultFalse } },
	{ 749, { } },
	{ 750, { } },
	{ 751, { } },
	{ 752, { } },
	{ 753, { } },
	{ 754, { &kDefaultObjectSelf } },
	{ 755, { } },
	{ 756, { } },
	{ 757, { } },
	{ 758, { &kDefaultInt1 } },
	{ 759, { } },
	{ 760, { } },
	{ 761, { } },
	{ 762, { } },
	{ 763, { } },
	{ 764, { &kDefaultFalse } },
	{ 765, { } },
	{ 766, { } },
	{ 767, { } },
	{ 768, { } },
	{ 769, { } },
	{ 770, { &kDefaultFalse } },
	{ 771, { } },
	{ 772, { } },
	{ 773, { &kDefaultObjectInvalid, &kDefaultInt0 } },
	{ 774, { } },
	{ 775, { } },
	{ 776, { &kDefaultFalse, &kDefaultObjectInvalid } },
	{ 777, { &kDefaultFalse, &kDefaultFalse, &kDefaultObjectInvalid } },
	{ 778, { } },
	{ 779, { } },
	{ 780, { } },
	{ 781, { } },
	{ 782, { } },
	{ 783, { } },
	{ 784, { } },
	{ 785, { } },
	{ 786, { } },
	{ 787, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 788, { } },
	{ 789, { } },
	{ 790, { } },
	{ 791, { } },
	{ 792, { } },
	{ 793, { &kDefaultStringEmpty } },
	{ 794, { } },
	{ 795, { &kDefaultInt0, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultObjectSelf, &kDefaultInt0 } },
	{ 796, { &kDefaultInt0 } },
	{ 797, { } },
	{ 798, { } },
	{ 799, { &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 800, { } },
	{ 801, { } },
	{ 802, { } },
	{ 803, { } },
	{ 804, { } },
	{ 805, { &kDefaultObjectSelf, &kDefaultInt0 } },
	{ 806, { } },
	{ 807, { } },
	{ 808, { } },
	{ 809, { } },
	{ 810, { } },
	{ 811, { } },
	{ 812, { } },
	{ 813, { } },
	{ 814, { } },
	{ 815, { } },
	{ 816, { } },
	{ 817, { &kDefaultFalse } },
	{ 818, { &kDefaultFalse } },
	{ 819, { } },
	{ 820, { } },
	{ 821, { } },
	{ 822, { } },
	{ 823, { } },
	{ 824, { } },
	{ 825, { } },
	{ 826, { } },
	{ 827, { } },
	{ 828, { } },
	{ 829, { } },
	{ 830, { } },
	{ 831, { } },
	{ 832, { } },
	{ 833, { } },
	{ 834, { } },
	{ 835, { } },
	{ 836, { } },
	{ 837, { } },
	{ 838, { } },
	{ 839, { } },
	{ 840, { } },
	{ 841, { } },
	{ 842, { } },
	{ 843, { } },
	{ 844, { } },
	{ 845, { } },
	{ 846, { &kDefaultTrue } },
	{ 847, { &kDefaultFalse } }
};

} // End of namespace Witcher

} // End of namespace Engines

#endif // ENGINES_WITCHER_NWSCRIPT_FUNCTION_TABLES_H
