/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Tables defining the engine functions in Neverwinter Nights 2
 */

/* The functions are defined by three tables:
 * - kFunctionPointers
 * - kFunctionSignatures
 * - kFunctionDefaults
 *
 * kFunctionPointers provides the ID and name of the engine function, and a
 * pointer to Functions method doing the actual work. If the function pointer
 * is 0, a default unimplementedFunction method is used that just prints
 * the name of the function when it's called.
 *
 * kFunctionSignatures defines the signature of the function, i.e. the types
 * of the return value and its parameters.
 *
 * kFunctionDefaults pins default values to parameters. I.e. if a function
 * Foobar(int mode, int doEverything=FALSE) is called with only one parameters,
 * the default value FALSE is applied to the doEverything parameters. The
 * kFunctionDefaults entry for the function then contains a reference to a
 * false value. Because of ambiguities otherwise, the default values have to be
 * "aligned" to the right; only the last-most n parameters are allowed to have
 * default values, with no gaps.
 *
 * Please note that all three tables have to be of the same length, and that
 * the order of the functions have to be the same in all three tables.
 */

#ifndef ENGINES_NWN2_SCRIPT_FUNCTION_TABLES_H
#define ENGINES_NWN2_SCRIPT_FUNCTION_TABLES_H

#include "src/engines/nwn2/types.h"
#include "src/engines/nwn2/object.h"

namespace Engines {

namespace NWN2 {

using Aurora::NWScript::kTypeVoid;
using Aurora::NWScript::kTypeInt;
using Aurora::NWScript::kTypeFloat;
using Aurora::NWScript::kTypeString;
using Aurora::NWScript::kTypeObject;
using Aurora::NWScript::kTypeVector;
using Aurora::NWScript::kTypeEngineType;
using Aurora::NWScript::kTypeScriptState;
using Aurora::NWScript::kTypeAny;

/* Default values as used by the function parameters, if no explicit value
 * is given at the time of the calling.
 */

static NWN2::Object kDefaultValueObjectInvalid(kObjectTypeInvalid);
static NWN2::Object kDefaultValueObjectSelf   (kObjectTypeSelf);

static const Aurora::NWScript::Variable kDefaultIntMinus1    ((int32_t) -         1);
static const Aurora::NWScript::Variable kDefaultInt0         ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultInt1         ((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultInt9         ((int32_t)           9);
static const Aurora::NWScript::Variable kDefaultInt18        ((int32_t)          18);
static const Aurora::NWScript::Variable kDefaultInt50        ((int32_t)          50);
static const Aurora::NWScript::Variable kDefaultInt100       ((int32_t)         100);
static const Aurora::NWScript::Variable kDefaultInt4294901760((int32_t)  4294901760U);
static const Aurora::NWScript::Variable kDefaultInt4294967295((int32_t)  4294967295U);

static const Aurora::NWScript::Variable kDefaultFloatMinus1_0(-  1.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_0     (   0.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_5     (   0.5f);
static const Aurora::NWScript::Variable kDefaultFloat1_0     (   1.0f);
static const Aurora::NWScript::Variable kDefaultFloat2_0     (   2.0f);
static const Aurora::NWScript::Variable kDefaultFloat5_0     (   5.0f);
static const Aurora::NWScript::Variable kDefaultFloat30_0    (  30.0f);
static const Aurora::NWScript::Variable kDefaultFloat40_0    (  40.0f);
static const Aurora::NWScript::Variable kDefaultFloat180_0   ( 180.0f);

static const Aurora::NWScript::Variable kDefaultVector0(0.0f, 0.0f, 0.0f);

static const Aurora::NWScript::Variable kDefaultStringEmpty(Common::UString(""));

static const Aurora::NWScript::Variable kDefaultObjectInvalid(&kDefaultValueObjectInvalid);
static const Aurora::NWScript::Variable kDefaultObjectSelf   (&kDefaultValueObjectSelf);

// TODO: Add the relevant enums to types.h, and use these values.

static const Aurora::NWScript::Variable kDefaultACDodgeBonus             ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultACVsDamageTypeAll        ((int32_t)        4103);
static const Aurora::NWScript::Variable kDefaultAlignmentAll             ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultAttackBonusMisc          ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultCameraTransitionTypeSnap ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultClassTypeInvalid         ((int32_t)         255);
static const Aurora::NWScript::Variable kDefaultDamagePowerNormal        ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultDamageTypeMagical        ((int32_t)           8);
static const Aurora::NWScript::Variable kDefaultDRTypeMagicBonus         ((int32_t)           2);
static const Aurora::NWScript::Variable kDefaultFalse                    ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultGenderMale               ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultIPConstUnlimitedAmmoBasic((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultMetaMagicAny             ((int32_t)  4294967295U);
static const Aurora::NWScript::Variable kDefaultMissChanceTypeNormal     ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultObjectTypeAll            ((int32_t)       32767);
static const Aurora::NWScript::Variable kDefaultObjectTypeCreature       ((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultPackageInvalid           ((int32_t)         255);
static const Aurora::NWScript::Variable kDefaultPersistentZoneActive     ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultProjectilePathTypeDefault((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultSavingThrowTypeAll       ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultSavingThrowTypeNone      ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultSavingThrowWill          ((int32_t)           3);
static const Aurora::NWScript::Variable kDefaultSpellAllSpells           ((int32_t) -         1);
static const Aurora::NWScript::Variable kDefaultSpellSchoolGeneral       ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultStandardFactionHostile   ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultTalkVolumeTalk           ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultTrue                     ((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultVFXNone                  ((int32_t) -         1);
static const Aurora::NWScript::Variable kDefaultWeatherPowerMedium       ((int32_t)           3);

static const Aurora::NWScript::Variable kDefaultFadeSpeedMedium          (1.5f);


/** The table defining the name and function pointer of each engine function. */
const Functions::FunctionPointer Functions::kFunctionPointers[] = {
	{    0, "Random"                              , &Functions::random                               },
	{    1, "PrintString"                         , &Functions::printString                          },
	{    2, "PrintFloat"                          , &Functions::printFloat                           },
	{    3, "FloatToString"                       , &Functions::floatToString                        },
	{    4, "PrintInteger"                        , &Functions::printInteger                         },
	{    5, "PrintObject"                         , &Functions::printObject                          },
	{    6, "AssignCommand"                       , &Functions::assignCommand                        },
	{    7, "DelayCommand"                        , &Functions::delayCommand                         },
	{    8, "ExecuteScript"                       , &Functions::executeScript                        },
	{    9, "ClearAllActions"                     , 0                                                },
	{   10, "SetFacing"                           , 0                                                },
	{   11, "SetCalendar"                         , 0                                                },
	{   12, "SetTime"                             , 0                                                },
	{   13, "GetCalendarYear"                     , 0                                                },
	{   14, "GetCalendarMonth"                    , 0                                                },
	{   15, "GetCalendarDay"                      , 0                                                },
	{   16, "GetTimeHour"                         , 0                                                },
	{   17, "GetTimeMinute"                       , 0                                                },
	{   18, "GetTimeSecond"                       , 0                                                },
	{   19, "GetTimeMillisecond"                  , 0                                                },
	{   20, "ActionRandomWalk"                    , 0                                                },
	{   21, "ActionMoveToLocation"                , &Functions::actionMoveToLocation                 },
	{   22, "ActionMoveToObject"                  , &Functions::actionMoveToObject                   },
	{   23, "ActionMoveAwayFromObject"            , 0                                                },
	{   24, "GetArea"                             , &Functions::getArea                              },
	{   25, "GetEnteringObject"                   , &Functions::getEnteringObject                    },
	{   26, "GetExitingObject"                    , &Functions::getExitingObject                     },
	{   27, "GetPosition"                         , &Functions::getPosition                          },
	{   28, "GetFacing"                           , 0                                                },
	{   29, "GetItemPossessor"                    , 0                                                },
	{   30, "GetItemPossessedBy"                  , 0                                                },
	{   31, "CreateItemOnObject"                  , &Functions::createItemOnObject                   },
	{   32, "ActionEquipItem"                     , 0                                                },
	{   33, "ActionUnequipItem"                   , 0                                                },
	{   34, "ActionPickUpItem"                    , 0                                                },
	{   35, "ActionPutDownItem"                   , 0                                                },
	{   36, "GetLastAttacker"                     , 0                                                },
	{   37, "ActionAttack"                        , 0                                                },
	{   38, "GetNearestCreature"                  , &Functions::getNearestCreature                   },
	{   39, "ActionSpeakString"                   , &Functions::actionSpeakString                    },
	{   40, "ActionPlayAnimation"                 , 0                                                },
	{   41, "GetDistanceToObject"                 , &Functions::getDistanceToObject                  },
	{   42, "GetIsObjectValid"                    , &Functions::getIsObjectValid                     },
	{   43, "ActionOpenDoor"                      , &Functions::actionOpenDoor                       },
	{   44, "ActionCloseDoor"                     , &Functions::actionCloseDoor                      },
	{   45, "SetCameraFacing"                     , 0                                                },
	{   46, "PlaySound"                           , &Functions::playSound                            },
	{   47, "GetSpellTargetObject"                , 0                                                },
	{   48, "ActionCastSpellAtObject"             , 0                                                },
	{   49, "GetCurrentHitPoints"                 , &Functions::getCurrentHitPoints                  },
	{   50, "GetMaxHitPoints"                     , &Functions::getMaxHitPoints                      },
	{   51, "GetLocalInt"                         , &Functions::getLocalInt                          },
	{   52, "GetLocalFloat"                       , &Functions::getLocalFloat                        },
	{   53, "GetLocalString"                      , &Functions::getLocalString                       },
	{   54, "GetLocalObject"                      , &Functions::getLocalObject                       },
	{   55, "SetLocalInt"                         , &Functions::setLocalInt                          },
	{   56, "SetLocalFloat"                       , &Functions::setLocalFloat                        },
	{   57, "SetLocalString"                      , &Functions::setLocalString                       },
	{   58, "SetLocalObject"                      , &Functions::setLocalObject                       },
	{   59, "GetStringLength"                     , &Functions::getStringLength                      },
	{   60, "GetStringUpperCase"                  , &Functions::getStringUpperCase                   },
	{   61, "GetStringLowerCase"                  , &Functions::getStringLowerCase                   },
	{   62, "GetStringRight"                      , &Functions::getStringRight                       },
	{   63, "GetStringLeft"                       , &Functions::getStringLeft                        },
	{   64, "InsertString"                        , &Functions::insertString                         },
	{   65, "GetSubString"                        , &Functions::getSubString                         },
	{   66, "FindSubString"                       , &Functions::findSubString                        },
	{   67, "fabs"                                , &Functions::fabs                                 },
	{   68, "cos"                                 , &Functions::cos                                  },
	{   69, "sin"                                 , &Functions::sin                                  },
	{   70, "tan"                                 , &Functions::tan                                  },
	{   71, "acos"                                , &Functions::acos                                 },
	{   72, "asin"                                , &Functions::asin                                 },
	{   73, "atan"                                , &Functions::atan                                 },
	{   74, "log"                                 , &Functions::log                                  },
	{   75, "pow"                                 , &Functions::pow                                  },
	{   76, "sqrt"                                , &Functions::sqrt                                 },
	{   77, "abs"                                 , &Functions::abs                                  },
	{   78, "EffectHeal"                          , 0                                                },
	{   79, "EffectDamage"                        , 0                                                },
	{   80, "EffectAbilityIncrease"               , 0                                                },
	{   81, "EffectDamageResistance"              , 0                                                },
	{   82, "EffectResurrection"                  , 0                                                },
	{   83, "EffectSummonCreature"                , 0                                                },
	{   84, "GetCasterLevel"                      , 0                                                },
	{   85, "GetFirstEffect"                      , 0                                                },
	{   86, "GetNextEffect"                       , 0                                                },
	{   87, "RemoveEffect"                        , 0                                                },
	{   88, "GetIsEffectValid"                    , 0                                                },
	{   89, "GetEffectDurationType"               , 0                                                },
	{   90, "GetEffectSubType"                    , 0                                                },
	{   91, "GetEffectCreator"                    , 0                                                },
	{   92, "IntToString"                         , &Functions::intToString                          },
	{   93, "GetFirstObjectInArea"                , 0                                                },
	{   94, "GetNextObjectInArea"                 , 0                                                },
	{   95, "d2"                                  , &Functions::d2                                   },
	{   96, "d3"                                  , &Functions::d3                                   },
	{   97, "d4"                                  , &Functions::d4                                   },
	{   98, "d6"                                  , &Functions::d6                                   },
	{   99, "d8"                                  , &Functions::d8                                   },
	{  100, "d10"                                 , &Functions::d10                                  },
	{  101, "d12"                                 , &Functions::d12                                  },
	{  102, "d20"                                 , &Functions::d20                                  },
	{  103, "d100"                                , &Functions::d100                                 },
	{  104, "VectorMagnitude"                     , &Functions::vectorMagnitude                      },
	{  105, "GetMetaMagicFeat"                    , 0                                                },
	{  106, "GetObjectType"                       , &Functions::getObjectType                        },
	{  107, "GetRacialType"                       , &Functions::getRacialType                        },
	{  108, "FortitudeSave"                       , 0                                                },
	{  109, "ReflexSave"                          , 0                                                },
	{  110, "WillSave"                            , 0                                                },
	{  111, "GetSpellSaveDC"                      , 0                                                },
	{  112, "MagicalEffect"                       , 0                                                },
	{  113, "SupernaturalEffect"                  , 0                                                },
	{  114, "ExtraordinaryEffect"                 , 0                                                },
	{  115, "EffectACIncrease"                    , 0                                                },
	{  116, "GetAC"                               , 0                                                },
	{  117, "EffectSavingThrowIncrease"           , 0                                                },
	{  118, "EffectAttackIncrease"                , 0                                                },
	{  119, "EffectDamageReduction"               , 0                                                },
	{  120, "EffectDamageIncrease"                , 0                                                },
	{  121, "RoundsToSeconds"                     , 0                                                },
	{  122, "HoursToSeconds"                      , 0                                                },
	{  123, "TurnsToSeconds"                      , 0                                                },
	{  124, "GetLawChaosValue"                    , &Functions::getLawChaosValue                     },
	{  125, "GetGoodEvilValue"                    , &Functions::getGoodEvilValue                     },
	{  126, "GetAlignmentLawChaos"                , &Functions::getAlignmentLawChaos                 },
	{  127, "GetAlignmentGoodEvil"                , &Functions::getAlignmentGoodEvil                 },
	{  128, "GetFirstObjectInShape"               , 0                                                },
	{  129, "GetNextObjectInShape"                , 0                                                },
	{  130, "EffectEntangle"                      , 0                                                },
	{  131, "SignalEvent"                         , 0                                                },
	{  132, "EventUserDefined"                    , 0                                                },
	{  133, "EffectDeath"                         , 0                                                },
	{  134, "EffectKnockdown"                     , 0                                                },
	{  135, "ActionGiveItem"                      , 0                                                },
	{  136, "ActionTakeItem"                      , 0                                                },
	{  137, "VectorNormalize"                     , &Functions::vectorNormalize                      },
	{  138, "EffectCurse"                         , 0                                                },
	{  139, "GetAbilityScore"                     , &Functions::getAbilityScore                      },
	{  140, "GetIsDead"                           , &Functions::getIsDead                            },
	{  141, "PrintVector"                         , &Functions::printVector                          },
	{  142, "Vector"                              , &Functions::vector                               },
	{  143, "SetFacingPoint"                      , 0                                                },
	{  144, "AngleToVector"                       , 0                                                },
	{  145, "VectorToAngle"                       , 0                                                },
	{  146, "TouchAttackMelee"                    , 0                                                },
	{  147, "TouchAttackRanged"                   , 0                                                },
	{  148, "EffectParalyze"                      , 0                                                },
	{  149, "EffectSpellImmunity"                 , 0                                                },
	{  150, "EffectDeaf"                          , 0                                                },
	{  151, "GetDistanceBetween"                  , 0                                                },
	{  152, "SetLocalLocation"                    , 0                                                },
	{  153, "GetLocalLocation"                    , 0                                                },
	{  154, "EffectSleep"                         , 0                                                },
	{  155, "GetItemInSlot"                       , &Functions::getItemInSlot                        },
	{  156, "EffectCharmed"                       , 0                                                },
	{  157, "EffectConfused"                      , 0                                                },
	{  158, "EffectFrightened"                    , 0                                                },
	{  159, "EffectDominated"                     , 0                                                },
	{  160, "EffectDazed"                         , 0                                                },
	{  161, "EffectStunned"                       , 0                                                },
	{  162, "SetCommandable"                      , 0                                                },
	{  163, "GetCommandable"                      , 0                                                },
	{  164, "EffectRegenerate"                    , 0                                                },
	{  165, "EffectMovementSpeedIncrease"         , 0                                                },
	{  166, "GetHitDice"                          , &Functions::getHitDice                           },
	{  167, "ActionForceFollowObject"             , 0                                                },
	{  168, "GetTag"                              , &Functions::getTag                               },
	{  169, "ResistSpell"                         , 0                                                },
	{  170, "GetEffectType"                       , 0                                                },
	{  171, "EffectAreaOfEffect"                  , 0                                                },
	{  172, "GetFactionEqual"                     , 0                                                },
	{  173, "ChangeFaction"                       , 0                                                },
	{  174, "GetIsListening"                      , &Functions::getIsListening                       },
	{  175, "SetListening"                        , &Functions::setListening                         },
	{  176, "SetListenPattern"                    , &Functions::setListenPattern                     },
	{  177, "TestStringAgainstPattern"            , 0                                                },
	{  178, "GetMatchedSubstring"                 , 0                                                },
	{  179, "GetMatchedSubstringsCount"           , 0                                                },
	{  180, "EffectVisualEffect"                  , 0                                                },
	{  181, "GetFactionWeakestMember"             , 0                                                },
	{  182, "GetFactionStrongestMember"           , 0                                                },
	{  183, "GetFactionMostDamagedMember"         , 0                                                },
	{  184, "GetFactionLeastDamagedMember"        , 0                                                },
	{  185, "GetFactionGold"                      , 0                                                },
	{  186, "GetFactionAverageReputation"         , 0                                                },
	{  187, "GetFactionAverageGoodEvilAlignment"  , 0                                                },
	{  188, "GetFactionAverageLawChaosAlignment"  , 0                                                },
	{  189, "GetFactionAverageLevel"              , 0                                                },
	{  190, "GetFactionAverageXP"                 , 0                                                },
	{  191, "GetFactionMostFrequentClass"         , 0                                                },
	{  192, "GetFactionWorstAC"                   , 0                                                },
	{  193, "GetFactionBestAC"                    , 0                                                },
	{  194, "ActionSit"                           , 0                                                },
	{  195, "GetListenPatternNumber"              , 0                                                },
	{  196, "ActionJumpToObject"                  , &Functions::actionJumpToObject                   },
	{  197, "GetWaypointByTag"                    , &Functions::getWaypointByTag                     },
	{  198, "GetTransitionTarget"                 , 0                                                },
	{  199, "EffectLinkEffects"                   , 0                                                },
	{  200, "GetObjectByTag"                      , &Functions::getObjectByTag                       },
	{  201, "AdjustAlignment"                     , 0                                                },
	{  202, "ActionWait"                          , 0                                                },
	{  203, "SetAreaTransitionBMP"                , 0                                                },
	{  204, "ActionStartConversation"             , 0                                                },
	{  205, "ActionPauseConversation"             , 0                                                },
	{  206, "ActionResumeConversation"            , 0                                                },
	{  207, "EffectBeam"                          , 0                                                },
	{  208, "GetReputation"                       , 0                                                },
	{  209, "AdjustReputation"                    , 0                                                },
	{  210, "GetSittingCreature"                  , 0                                                },
	{  211, "GetGoingToBeAttackedBy"              , 0                                                },
	{  212, "EffectSpellResistanceIncrease"       , 0                                                },
	{  213, "GetLocation"                         , &Functions::getLocation                          },
	{  214, "ActionJumpToLocation"                , &Functions::actionJumpToLocation                 },
	{  215, "Location"                            , &Functions::location                             },
	{  216, "ApplyEffectAtLocation"               , 0                                                },
	{  217, "GetIsPC"                             , &Functions::getIsPC                              },
	{  218, "FeetToMeters"                        , &Functions::feetToMeters                         },
	{  219, "YardsToMeters"                       , &Functions::yardsToMeters                        },
	{  220, "ApplyEffectToObject"                 , 0                                                },
	{  221, "SpeakString"                         , &Functions::speakString                          },
	{  222, "GetSpellTargetLocation"              , 0                                                },
	{  223, "GetPositionFromLocation"             , &Functions::getPositionFromLocation              },
	{  224, "GetAreaFromLocation"                 , 0                                                },
	{  225, "GetFacingFromLocation"               , 0                                                },
	{  226, "GetNearestCreatureToLocation"        , 0                                                },
	{  227, "GetNearestObject"                    , &Functions::getNearestObject                     },
	{  228, "GetNearestObjectToLocation"          , 0                                                },
	{  229, "GetNearestObjectByTag"               , &Functions::getNearestObjectByTag                },
	{  230, "IntToFloat"                          , &Functions::intToFloat                           },
	{  231, "FloatToInt"                          , &Functions::floatToInt                           },
	{  232, "StringToInt"                         , &Functions::stringToInt                          },
	{  233, "StringToFloat"                       , &Functions::stringToFloat                        },
	{  234, "ActionCastSpellAtLocation"           , 0                                                },
	{  235, "GetIsEnemy"                          , 0                                                },
	{  236, "GetIsFriend"                         , 0                                                },
	{  237, "GetIsNeutral"                        , 0                                                },
	{  238, "GetPCSpeaker"                        , 0                                                },
	{  239, "GetStringByStrRef"                   , &Functions::getStringByStrRef                    },
	{  240, "ActionSpeakStringByStrRef"           , 0                                                },
	{  241, "DestroyObject"                       , 0                                                },
	{  242, "GetModule"                           , &Functions::getModule                            },
	{  243, "CreateObject"                        , 0                                                },
	{  244, "EventSpellCastAt"                    , 0                                                },
	{  245, "GetLastSpellCaster"                  , 0                                                },
	{  246, "GetLastSpell"                        , 0                                                },
	{  247, "GetUserDefinedEventNumber"           , 0                                                },
	{  248, "GetSpellId"                          , 0                                                },
	{  249, "RandomName"                          , &Functions::randomName                           },
	{  250, "EffectPoison"                        , 0                                                },
	{  251, "EffectDisease"                       , 0                                                },
	{  252, "EffectSilence"                       , 0                                                },
	{  253, "GetName"                             , &Functions::getName                              },
	{  254, "GetLastSpeaker"                      , 0                                                },
	{  255, "BeginConversation"                   , 0                                                },
	{  256, "GetLastPerceived"                    , 0                                                },
	{  257, "GetLastPerceptionHeard"              , 0                                                },
	{  258, "GetLastPerceptionInaudible"          , 0                                                },
	{  259, "GetLastPerceptionSeen"               , 0                                                },
	{  260, "GetLastClosedBy"                     , &Functions::getLastClosedBy                      },
	{  261, "GetLastPerceptionVanished"           , 0                                                },
	{  262, "GetFirstInPersistentObject"          , 0                                                },
	{  263, "GetNextInPersistentObject"           , 0                                                },
	{  264, "GetAreaOfEffectCreator"              , 0                                                },
	{  265, "DeleteLocalInt"                      , 0                                                },
	{  266, "DeleteLocalFloat"                    , 0                                                },
	{  267, "DeleteLocalString"                   , 0                                                },
	{  268, "DeleteLocalObject"                   , 0                                                },
	{  269, "DeleteLocalLocation"                 , 0                                                },
	{  270, "EffectHaste"                         , 0                                                },
	{  271, "EffectSlow"                          , 0                                                },
	{  272, "ObjectToString"                      , &Functions::objectToString                       },
	{  273, "EffectImmunity"                      , 0                                                },
	{  274, "GetIsImmune"                         , 0                                                },
	{  275, "EffectDamageImmunityIncrease"        , 0                                                },
	{  276, "GetEncounterActive"                  , 0                                                },
	{  277, "SetEncounterActive"                  , 0                                                },
	{  278, "GetEncounterSpawnsMax"               , 0                                                },
	{  279, "SetEncounterSpawnsMax"               , 0                                                },
	{  280, "GetEncounterSpawnsCurrent"           , 0                                                },
	{  281, "SetEncounterSpawnsCurrent"           , 0                                                },
	{  282, "GetModuleItemAcquired"               , 0                                                },
	{  283, "GetModuleItemAcquiredFrom"           , 0                                                },
	{  284, "SetCustomToken"                      , 0                                                },
	{  285, "GetHasFeat"                          , &Functions::getHasFeat                           },
	{  286, "GetHasSkill"                         , 0                                                },
	{  287, "ActionUseFeat"                       , 0                                                },
	{  288, "ActionUseSkill"                      , 0                                                },
	{  289, "GetObjectSeen"                       , 0                                                },
	{  290, "GetObjectHeard"                      , 0                                                },
	{  291, "GetLastPlayerDied"                   , 0                                                },
	{  292, "GetModuleItemLost"                   , 0                                                },
	{  293, "GetModuleItemLostBy"                 , 0                                                },
	{  294, "ActionDoCommand"                     , &Functions::actionDoCommand                      },
	{  295, "EventConversation"                   , 0                                                },
	{  296, "SetEncounterDifficulty"              , 0                                                },
	{  297, "GetEncounterDifficulty"              , 0                                                },
	{  298, "GetDistanceBetweenLocations"         , 0                                                },
	{  299, "GetReflexAdjustedDamage"             , 0                                                },
	{  300, "PlayAnimation"                       , 0                                                },
	{  301, "TalentSpell"                         , 0                                                },
	{  302, "TalentFeat"                          , 0                                                },
	{  303, "TalentSkill"                         , 0                                                },
	{  304, "GetHasSpellEffect"                   , 0                                                },
	{  305, "GetEffectSpellId"                    , 0                                                },
	{  306, "GetCreatureHasTalent"                , 0                                                },
	{  307, "GetCreatureTalentRandom"             , 0                                                },
	{  308, "GetCreatureTalentBest"               , 0                                                },
	{  309, "ActionUseTalentOnObject"             , 0                                                },
	{  310, "ActionUseTalentAtLocation"           , 0                                                },
	{  311, "GetGoldPieceValue"                   , 0                                                },
	{  312, "GetIsPlayableRacialType"             , 0                                                },
	{  313, "JumpToLocation"                      , &Functions::jumpToLocation                       },
	{  314, "EffectTemporaryHitpoints"            , 0                                                },
	{  315, "GetSkillRank"                        , &Functions::getSkillRank                         },
	{  316, "GetAttackTarget"                     , 0                                                },
	{  317, "GetLastAttackType"                   , 0                                                },
	{  318, "GetLastAttackMode"                   , 0                                                },
	{  319, "GetMaster"                           , 0                                                },
	{  320, "GetIsInCombat"                       , 0                                                },
	{  321, "GetLastAssociateCommand"             , 0                                                },
	{  322, "GiveGoldToCreature"                  , 0                                                },
	{  323, "SetIsDestroyable"                    , 0                                                },
	{  324, "SetLocked"                           , &Functions::setLocked                            },
	{  325, "GetLocked"                           , &Functions::getLocked                            },
	{  326, "GetClickingObject"                   , &Functions::getClickingObject                    },
	{  327, "SetAssociateListenPatterns"          , &Functions::setAssociateListenPatterns           },
	{  328, "GetLastWeaponUsed"                   , 0                                                },
	{  329, "ActionInteractObject"                , 0                                                },
	{  330, "GetLastUsedBy"                       , &Functions::getLastUsedBy                        },
	{  331, "GetAbilityModifier"                  , 0                                                },
	{  332, "GetIdentified"                       , &Functions::getIdentified                        },
	{  333, "SetIdentified"                       , &Functions::setIdentified                        },
	{  334, "SummonAnimalCompanion"               , 0                                                },
	{  335, "SummonFamiliar"                      , 0                                                },
	{  336, "GetBlockingDoor"                     , 0                                                },
	{  337, "GetIsDoorActionPossible"             , 0                                                },
	{  338, "DoDoorAction"                        , 0                                                },
	{  339, "GetFirstItemInInventory"             , &Functions::getFirstItemInInventory              },
	{  340, "GetNextItemInInventory"              , &Functions::getNextItemInInventory               },
	{  341, "GetClassByPosition"                  , &Functions::getClassByPosition                   },
	{  342, "GetLevelByPosition"                  , &Functions::getLevelByPosition                   },
	{  343, "GetLevelByClass"                     , &Functions::getLevelByClass                      },
	{  344, "GetDamageDealtByType"                , 0                                                },
	{  345, "GetTotalDamageDealt"                 , 0                                                },
	{  346, "GetLastDamager"                      , 0                                                },
	{  347, "GetLastDisarmed"                     , 0                                                },
	{  348, "GetLastDisturbed"                    , 0                                                },
	{  349, "GetLastLocked"                       , 0                                                },
	{  350, "GetLastUnlocked"                     , 0                                                },
	{  351, "EffectSkillIncrease"                 , 0                                                },
	{  352, "GetInventoryDisturbType"             , 0                                                },
	{  353, "GetInventoryDisturbItem"             , 0                                                },
	{  354, "GetHenchman"                         , 0                                                },
	{  355, "VersusAlignmentEffect"               , 0                                                },
	{  356, "VersusRacialTypeEffect"              , 0                                                },
	{  357, "VersusTrapEffect"                    , 0                                                },
	{  358, "GetGender"                           , &Functions::getGender                            },
	{  359, "GetIsTalentValid"                    , 0                                                },
	{  360, "ActionMoveAwayFromLocation"          , 0                                                },
	{  361, "GetAttemptedAttackTarget"            , 0                                                },
	{  362, "GetTypeFromTalent"                   , 0                                                },
	{  363, "GetIdFromTalent"                     , 0                                                },
	{  364, "GetAssociate"                        , 0                                                },
	{  365, "AddHenchman"                         , 0                                                },
	{  366, "RemoveHenchman"                      , 0                                                },
	{  367, "AddJournalQuestEntry"                , &Functions::addJournalQuestEntry                 },
	{  368, "RemoveJournalQuestEntry"             , &Functions::removeJournalQuestEntry              },
	{  369, "GetPCPublicCDKey"                    , 0                                                },
	{  370, "GetPCIPAddress"                      , 0                                                },
	{  371, "GetPCPlayerName"                     , 0                                                },
	{  372, "SetPCLike"                           , 0                                                },
	{  373, "SetPCDislike"                        , 0                                                },
	{  374, "SendMessageToPC"                     , &Functions::sendMessageToPC                      },
	{  375, "GetAttemptedSpellTarget"             , 0                                                },
	{  376, "GetLastOpenedBy"                     , &Functions::getLastOpenedBy                      },
	{  377, "GetHasSpell"                         , 0                                                },
	{  378, "OpenStore"                           , 0                                                },
	{  379, "EffectTurned"                        , 0                                                },
	{  380, "GetFirstFactionMember"               , 0                                                },
	{  381, "GetNextFactionMember"                , 0                                                },
	{  382, "ActionForceMoveToLocation"           , 0                                                },
	{  383, "ActionForceMoveToObject"             , 0                                                },
	{  384, "GetJournalQuestExperience"           , &Functions::getJournalQuestExperience            },
	{  385, "JumpToObject"                        , &Functions::jumpToObject                         },
	{  386, "SetMapPinEnabled"                    , 0                                                },
	{  387, "EffectHitPointChangeWhenDying"       , 0                                                },
	{  388, "PopUpGUIPanel"                       , 0                                                },
	{  389, "ClearPersonalReputation"             , 0                                                },
	{  390, "SetIsTemporaryFriend"                , 0                                                },
	{  391, "SetIsTemporaryEnemy"                 , 0                                                },
	{  392, "SetIsTemporaryNeutral"               , 0                                                },
	{  393, "GiveXPToCreature"                    , 0                                                },
	{  394, "SetXP"                               , 0                                                },
	{  395, "GetXP"                               , &Functions::getXP                                },
	{  396, "IntToHexString"                      , &Functions::intToHexString                       },
	{  397, "GetBaseItemType"                     , &Functions::getBaseItemType                      },
	{  398, "GetItemHasItemProperty"              , &Functions::getItemHasItemProperty               },
	{  399, "ActionEquipMostDamagingMelee"        , 0                                                },
	{  400, "ActionEquipMostDamagingRanged"       , 0                                                },
	{  401, "GetItemACValue"                      , 0                                                },
	{  402, "ActionRest"                          , 0                                                },
	{  403, "ExploreAreaForPlayer"                , 0                                                },
	{  404, "ActionEquipMostEffectiveArmor"       , 0                                                },
	{  405, "GetIsDay"                            , 0                                                },
	{  406, "GetIsNight"                          , 0                                                },
	{  407, "GetIsDawn"                           , 0                                                },
	{  408, "GetIsDusk"                           , 0                                                },
	{  409, "GetIsEncounterCreature"              , 0                                                },
	{  410, "GetLastPlayerDying"                  , 0                                                },
	{  411, "GetStartingLocation"                 , 0                                                },
	{  412, "ChangeToStandardFaction"             , 0                                                },
	{  413, "SoundObjectPlay"                     , 0                                                },
	{  414, "SoundObjectStop"                     , 0                                                },
	{  415, "SoundObjectSetVolume"                , 0                                                },
	{  416, "SoundObjectSetPosition"              , 0                                                },
	{  417, "SpeakOneLinerConversation"           , &Functions::speakOneLinerConversation            },
	{  418, "GetGold"                             , 0                                                },
	{  419, "GetLastRespawnButtonPresser"         , 0                                                },
	{  420, "GetIsDM"                             , &Functions::getIsDM                              },
	{  421, "PlayVoiceChat"                       , &Functions::playVoiceChat                        },
	{  422, "GetIsWeaponEffective"                , 0                                                },
	{  423, "GetLastSpellHarmful"                 , 0                                                },
	{  424, "EventActivateItem"                   , 0                                                },
	{  425, "MusicBackgroundPlay"                 , &Functions::musicBackgroundPlay                  },
	{  426, "MusicBackgroundStop"                 , &Functions::musicBackgroundStop                  },
	{  427, "MusicBackgroundSetDelay"             , 0                                                },
	{  428, "MusicBackgroundChangeDay"            , &Functions::musicBackgroundChangeDay             },
	{  429, "MusicBackgroundChangeNight"          , &Functions::musicBackgroundChangeNight           },
	{  430, "MusicBattlePlay"                     , 0                                                },
	{  431, "MusicBattleStop"                     , 0                                                },
	{  432, "MusicBattleChange"                   , 0                                                },
	{  433, "AmbientSoundPlay"                    , 0                                                },
	{  434, "AmbientSoundStop"                    , 0                                                },
	{  435, "AmbientSoundChangeDay"               , 0                                                },
	{  436, "AmbientSoundChangeNight"             , 0                                                },
	{  437, "GetLastKiller"                       , 0                                                },
	{  438, "GetSpellCastItem"                    , 0                                                },
	{  439, "GetItemActivated"                    , 0                                                },
	{  440, "GetItemActivator"                    , 0                                                },
	{  441, "GetItemActivatedTargetLocation"      , 0                                                },
	{  442, "GetItemActivatedTarget"              , 0                                                },
	{  443, "GetIsOpen"                           , &Functions::getIsOpen                            },
	{  444, "TakeGoldFromCreature"                , 0                                                },
	{  445, "IsInConversation"                    , 0                                                },
	{  446, "EffectAbilityDecrease"               , 0                                                },
	{  447, "EffectAttackDecrease"                , 0                                                },
	{  448, "EffectDamageDecrease"                , 0                                                },
	{  449, "EffectDamageImmunityDecrease"        , 0                                                },
	{  450, "EffectACDecrease"                    , 0                                                },
	{  451, "EffectMovementSpeedDecrease"         , 0                                                },
	{  452, "EffectSavingThrowDecrease"           , 0                                                },
	{  453, "EffectSkillDecrease"                 , 0                                                },
	{  454, "EffectSpellResistanceDecrease"       , 0                                                },
	{  455, "GetPlotFlag"                         , &Functions::getPlotFlag                          },
	{  456, "SetPlotFlag"                         , &Functions::setPlotFlag                          },
	{  457, "EffectInvisibility"                  , 0                                                },
	{  458, "EffectConcealment"                   , 0                                                },
	{  459, "EffectDarkness"                      , 0                                                },
	{  460, "EffectDispelMagicAll"                , 0                                                },
	{  461, "EffectUltravision"                   , 0                                                },
	{  462, "EffectNegativeLevel"                 , 0                                                },
	{  463, "EffectPolymorph"                     , 0                                                },
	{  464, "EffectSanctuary"                     , 0                                                },
	{  465, "EffectTrueSeeing"                    , 0                                                },
	{  466, "EffectSeeInvisible"                  , 0                                                },
	{  467, "EffectTimeStop"                      , 0                                                },
	{  468, "EffectBlindness"                     , 0                                                },
	{  469, "GetIsReactionTypeFriendly"           , 0                                                },
	{  470, "GetIsReactionTypeNeutral"            , 0                                                },
	{  471, "GetIsReactionTypeHostile"            , 0                                                },
	{  472, "EffectSpellLevelAbsorption"          , 0                                                },
	{  473, "EffectDispelMagicBest"               , 0                                                },
	{  474, "ActivatePortal"                      , 0                                                },
	{  475, "GetNumStackedItems"                  , &Functions::getNumStackedItems                   },
	{  476, "SurrenderToEnemies"                  , 0                                                },
	{  477, "EffectMissChance"                    , 0                                                },
	{  478, "GetTurnResistanceHD"                 , 0                                                },
	{  479, "GetCreatureSize"                     , 0                                                },
	{  480, "EffectDisappearAppear"               , 0                                                },
	{  481, "EffectDisappear"                     , 0                                                },
	{  482, "EffectAppear"                        , 0                                                },
	{  483, "ActionUnlockObject"                  , 0                                                },
	{  484, "ActionLockObject"                    , 0                                                },
	{  485, "EffectModifyAttacks"                 , 0                                                },
	{  486, "GetLastTrapDetected"                 , 0                                                },
	{  487, "EffectDamageShield"                  , 0                                                },
	{  488, "GetNearestTrapToObject"              , 0                                                },
	{  489, "GetDeity"                            , 0                                                },
	{  490, "GetSubRace"                          , 0                                                },
	{  491, "GetFortitudeSavingThrow"             , 0                                                },
	{  492, "GetWillSavingThrow"                  , 0                                                },
	{  493, "GetReflexSavingThrow"                , 0                                                },
	{  494, "GetChallengeRating"                  , 0                                                },
	{  495, "GetAge"                              , 0                                                },
	{  496, "GetMovementRate"                     , 0                                                },
	{  497, "GetAnimalCompanionCreatureType"      , 0                                                },
	{  498, "GetFamiliarCreatureType"             , 0                                                },
	{  499, "GetAnimalCompanionName"              , 0                                                },
	{  500, "GetFamiliarName"                     , 0                                                },
	{  501, "ActionCastFakeSpellAtObject"         , 0                                                },
	{  502, "ActionCastFakeSpellAtLocation"       , 0                                                },
	{  503, "RemoveSummonedAssociate"             , 0                                                },
	{  504, "SetCameraMode"                       , 0                                                },
	{  505, "GetIsResting"                        , 0                                                },
	{  506, "GetLastPCRested"                     , 0                                                },
	{  507, "SetWeather"                          , 0                                                },
	{  508, "GetLastRestEventType"                , 0                                                },
	{  509, "StartNewModule"                      , &Functions::startNewModule                       },
	{  510, "EffectSwarm"                         , 0                                                },
	{  511, "GetWeaponRanged"                     , 0                                                },
	{  512, "DoSinglePlayerAutoSave"              , 0                                                },
	{  513, "GetGameDifficulty"                   , &Functions::getGameDifficulty                    },
	{  514, "SetTileMainLightColor"               , 0                                                },
	{  515, "SetTileSourceLightColor"             , 0                                                },
	{  516, "RecomputeStaticLighting"             , 0                                                },
	{  517, "GetTileMainLight1Color"              , 0                                                },
	{  518, "GetTileMainLight2Color"              , 0                                                },
	{  519, "GetTileSourceLight1Color"            , 0                                                },
	{  520, "GetTileSourceLight2Color"            , 0                                                },
	{  521, "SetPanelButtonFlash"                 , 0                                                },
	{  522, "GetCurrentAction"                    , 0                                                },
	{  523, "SetStandardFactionReputation"        , 0                                                },
	{  524, "GetStandardFactionReputation"        , 0                                                },
	{  525, "FloatingTextStrRefOnCreature"        , 0                                                },
	{  526, "FloatingTextStringOnCreature"        , 0                                                },
	{  527, "GetTrapDisarmable"                   , &Functions::getTrapDisarmable                    },
	{  528, "GetTrapDetectable"                   , &Functions::getTrapDetectable                    },
	{  529, "GetTrapDetectedBy"                   , &Functions::getTrapDetectedBy                    },
	{  530, "GetTrapFlagged"                      , &Functions::getTrapFlagged                       },
	{  531, "GetTrapBaseType"                     , &Functions::getTrapBaseType                      },
	{  532, "GetTrapOneShot"                      , &Functions::getTrapOneShot                       },
	{  533, "GetTrapCreator"                      , &Functions::getTrapCreator                       },
	{  534, "GetTrapKeyTag"                       , &Functions::getTrapKeyTag                        },
	{  535, "GetTrapDisarmDC"                     , &Functions::getTrapDisarmDC                      },
	{  536, "GetTrapDetectDC"                     , &Functions::getTrapDetectDC                      },
	{  537, "GetLockKeyRequired"                  , &Functions::getLockKeyRequired                   },
	{  538, "GetLockKeyTag"                       , &Functions::getLockKeyTag                        },
	{  539, "GetLockLockable"                     , &Functions::getLockLockable                      },
	{  540, "GetLockUnlockDC"                     , &Functions::getLockUnlockDC                      },
	{  541, "GetLockLockDC"                       , &Functions::getLockLockDC                        },
	{  542, "GetPCLevellingUp"                    , 0                                                },
	{  543, "GetHasFeatEffect"                    , 0                                                },
	{  544, "SetPlaceableIllumination"            , 0                                                },
	{  545, "GetPlaceableIllumination"            , 0                                                },
	{  546, "GetIsPlaceableObjectActionPossible"  , 0                                                },
	{  547, "DoPlaceableObjectAction"             , 0                                                },
	{  548, "GetFirstPC"                          , &Functions::getFirstPC                           },
	{  549, "GetNextPC"                           , &Functions::getNextPC                            },
	{  550, "SetTrapDetectedBy"                   , &Functions::setTrapDetectedBy                    },
	{  551, "GetIsTrapped"                        , &Functions::getIsTrapped                         },
	{  552, "EffectTurnResistanceDecrease"        , 0                                                },
	{  553, "EffectTurnResistanceIncrease"        , 0                                                },
	{  554, "PopUpDeathGUIPanel"                  , 0                                                },
	{  555, "SetTrapDisabled"                     , &Functions::setTrapDisabled                      },
	{  556, "GetLastHostileActor"                 , 0                                                },
	{  557, "ExportAllCharacters"                 , 0                                                },
	{  558, "MusicBackgroundGetDayTrack"          , &Functions::musicBackgroundGetDayTrack           },
	{  559, "MusicBackgroundGetNightTrack"        , &Functions::musicBackgroundGetNightTrack         },
	{  560, "WriteTimestampedLogEntry"            , &Functions::writeTimestampedLogEntry             },
	{  561, "GetModuleName"                       , 0                                                },
	{  562, "GetFactionLeader"                    , &Functions::getFactionLeader                     },
	{  563, "SendMessageToAllDMs"                 , 0                                                },
	{  564, "EndGame"                             , 0                                                },
	{  565, "BootPC"                              , 0                                                },
	{  566, "ActionCounterSpell"                  , 0                                                },
	{  567, "AmbientSoundSetDayVolume"            , 0                                                },
	{  568, "AmbientSoundSetNightVolume"          , 0                                                },
	{  569, "MusicBackgroundGetBattleTrack"       , 0                                                },
	{  570, "GetHasInventory"                     , &Functions::getHasInventory                      },
	{  571, "GetStrRefSoundDuration"              , 0                                                },
	{  572, "AddToParty"                          , 0                                                },
	{  573, "RemoveFromParty"                     , 0                                                },
	{  574, "GetStealthMode"                      , 0                                                },
	{  575, "GetDetectMode"                       , 0                                                },
	{  576, "GetDefensiveCastingMode"             , 0                                                },
	{  577, "GetAppearanceType"                   , 0                                                },
	{  578, "SpawnScriptDebugger"                 , 0                                                },
	{  579, "GetModuleItemAcquiredStackSize"      , 0                                                },
	{  580, "DecrementRemainingFeatUses"          , 0                                                },
	{  581, "DecrementRemainingSpellUses"         , 0                                                },
	{  582, "GetResRef"                           , 0                                                },
	{  583, "EffectPetrify"                       , 0                                                },
	{  584, "CopyItem"                            , 0                                                },
	{  585, "EffectCutsceneParalyze"              , 0                                                },
	{  586, "GetDroppableFlag"                    , &Functions::getDroppableFlag                     },
	{  587, "GetUseableFlag"                      , &Functions::getUseableFlag                       },
	{  588, "GetStolenFlag"                       , &Functions::getStolenFlag                        },
	{  589, "SetCampaignFloat"                    , 0                                                },
	{  590, "SetCampaignInt"                      , 0                                                },
	{  591, "SetCampaignVector"                   , 0                                                },
	{  592, "SetCampaignLocation"                 , 0                                                },
	{  593, "SetCampaignString"                   , 0                                                },
	{  594, "DestroyCampaignDatabase"             , 0                                                },
	{  595, "GetCampaignFloat"                    , 0                                                },
	{  596, "GetCampaignInt"                      , 0                                                },
	{  597, "GetCampaignVector"                   , 0                                                },
	{  598, "GetCampaignLocation"                 , 0                                                },
	{  599, "GetCampaignString"                   , 0                                                },
	{  600, "CopyObject"                          , 0                                                },
	{  601, "DeleteCampaignVariable"              , 0                                                },
	{  602, "StoreCampaignObject"                 , 0                                                },
	{  603, "RetrieveCampaignObject"              , 0                                                },
	{  604, "EffectCutsceneDominated"             , 0                                                },
	{  605, "GetItemStackSize"                    , &Functions::getItemStackSize                     },
	{  606, "SetItemStackSize"                    , &Functions::setItemStackSize                     },
	{  607, "GetItemCharges"                      , &Functions::getItemCharges                       },
	{  608, "SetItemCharges"                      , &Functions::setItemCharges                       },
	{  609, "AddItemProperty"                     , 0                                                },
	{  610, "RemoveItemProperty"                  , 0                                                },
	{  611, "GetIsItemPropertyValid"              , &Functions::getIsItemPropertyValid               },
	{  612, "GetFirstItemProperty"                , &Functions::getFirstItemProperty                 },
	{  613, "GetNextItemProperty"                 , &Functions::getNextItemProperty                  },
	{  614, "GetItemPropertyType"                 , &Functions::getItemPropertyType                  },
	{  615, "GetItemPropertyDurationType"         , 0                                                },
	{  616, "ItemPropertyAbilityBonus"            , &Functions::itemPropertyAbilityBonus             },
	{  617, "ItemPropertyACBonus"                 , &Functions::itemPropertyACBonus                  },
	{  618, "ItemPropertyACBonusVsAlign"          , &Functions::itemPropertyACBonusVsAlign           },
	{  619, "ItemPropertyACBonusVsDmgType"        , &Functions::itemPropertyACBonusVsDmgType         },
	{  620, "ItemPropertyACBonusVsRace"           , &Functions::itemPropertyACBonusVsRace            },
	{  621, "ItemPropertyACBonusVsSAlign"         , &Functions::itemPropertyACBonusVsSAlign          },
	{  622, "ItemPropertyEnhancementBonus"        , &Functions::itemPropertyEnhancementBonus         },
	{  623, "ItemPropertyEnhancementBonusVsAlign" , &Functions::itemPropertyEnhancementBonusVsAlign  },
	{  624, "ItemPropertyEnhancementBonusVsRace"  , &Functions::itemPropertyEnhancementBonusVsRace   },
	{  625, "ItemPropertyEnhancementBonusVsSAlign", &Functions::itemPropertyEnhancementBonusVsSAlign },
	{  626, "ItemPropertyEnhancementPenalty"      , &Functions::itemPropertyEnhancementPenalty       },
	{  627, "ItemPropertyWeightReduction"         , &Functions::itemPropertyWeightReduction          },
	{  628, "ItemPropertyBonusFeat"               , &Functions::itemPropertyBonusFeat                },
	{  629, "ItemPropertyBonusLevelSpell"         , &Functions::itemPropertyBonusLevelSpell          },
	{  630, "ItemPropertyCastSpell"               , &Functions::itemPropertyCastSpell                },
	{  631, "ItemPropertyDamageBonus"             , &Functions::itemPropertyDamageBonus              },
	{  632, "ItemPropertyDamageBonusVsAlign"      , &Functions::itemPropertyDamageBonusVsAlign       },
	{  633, "ItemPropertyDamageBonusVsRace"       , &Functions::itemPropertyDamageBonusVsRace        },
	{  634, "ItemPropertyDamageBonusVsSAlign"     , &Functions::itemPropertyDamageBonusVsSAlign      },
	{  635, "ItemPropertyDamageImmunity"          , &Functions::itemPropertyDamageImmunity           },
	{  636, "ItemPropertyDamagePenalty"           , &Functions::itemPropertyDamagePenalty            },
	{  637, "ItemPropertyDamageReduction"         , &Functions::itemPropertyDamageReduction          },
	{  638, "ItemPropertyDamageResistance"        , &Functions::itemPropertyDamageResistance         },
	{  639, "ItemPropertyDamageVulnerability"     , &Functions::itemPropertyDamageVulnerability      },
	{  640, "ItemPropertyDarkvision"              , &Functions::itemPropertyDarkvision               },
	{  641, "ItemPropertyDecreaseAbility"         , &Functions::itemPropertyDecreaseAbility          },
	{  642, "ItemPropertyDecreaseAC"              , &Functions::itemPropertyDecreaseAC               },
	{  643, "ItemPropertyDecreaseSkill"           , &Functions::itemPropertyDecreaseSkill            },
	{  644, "ItemPropertyContainerReducedWeight"  , &Functions::itemPropertyContainerReducedWeight   },
	{  645, "ItemPropertyExtraMeleeDamageType"    , &Functions::itemPropertyExtraMeleeDamageType     },
	{  646, "ItemPropertyExtraRangeDamageType"    , &Functions::itemPropertyExtraRangeDamageType     },
	{  647, "ItemPropertyHaste"                   , &Functions::itemPropertyHaste                    },
	{  648, "ItemPropertyHolyAvenger"             , &Functions::itemPropertyHolyAvenger              },
	{  649, "ItemPropertyImmunityMisc"            , &Functions::itemPropertyImmunityMisc             },
	{  650, "ItemPropertyImprovedEvasion"         , &Functions::itemPropertyImprovedEvasion          },
	{  651, "ItemPropertyBonusSpellResistance"    , &Functions::itemPropertyBonusSpellResistance     },
	{  652, "ItemPropertyBonusSavingThrowVsX"     , &Functions::itemPropertyBonusSavingThrowVsX      },
	{  653, "ItemPropertyBonusSavingThrow"        , &Functions::itemPropertyBonusSavingThrow         },
	{  654, "ItemPropertyKeen"                    , &Functions::itemPropertyKeen                     },
	{  655, "ItemPropertyLight"                   , &Functions::itemPropertyLight                    },
	{  656, "ItemPropertyMaxRangeStrengthMod"     , &Functions::itemPropertyMaxRangeStrengthMod      },
	{  657, "ItemPropertyNoDamage"                , &Functions::itemPropertyNoDamage                 },
	{  658, "ItemPropertyOnHitProps"              , &Functions::itemPropertyOnHitProps               },
	{  659, "ItemPropertyReducedSavingThrowVsX"   , &Functions::itemPropertyReducedSavingThrowVsX    },
	{  660, "ItemPropertyReducedSavingThrow"      , &Functions::itemPropertyReducedSavingThrow       },
	{  661, "ItemPropertyRegeneration"            , &Functions::itemPropertyRegeneration             },
	{  662, "ItemPropertySkillBonus"              , &Functions::itemPropertySkillBonus               },
	{  663, "ItemPropertySpellImmunitySpecific"   , &Functions::itemPropertySpellImmunitySpecific    },
	{  664, "ItemPropertySpellImmunitySchool"     , &Functions::itemPropertySpellImmunitySchool      },
	{  665, "ItemPropertyThievesTools"            , &Functions::itemPropertyThievesTools             },
	{  666, "ItemPropertyAttackBonus"             , &Functions::itemPropertyAttackBonus              },
	{  667, "ItemPropertyAttackBonusVsAlign"      , &Functions::itemPropertyAttackBonusVsAlign       },
	{  668, "ItemPropertyAttackBonusVsRace"       , &Functions::itemPropertyAttackBonusVsRace        },
	{  669, "ItemPropertyAttackBonusVsSAlign"     , &Functions::itemPropertyAttackBonusVsSAlign      },
	{  670, "ItemPropertyAttackPenalty"           , &Functions::itemPropertyAttackPenalty            },
	{  671, "ItemPropertyUnlimitedAmmo"           , &Functions::itemPropertyUnlimitedAmmo            },
	{  672, "ItemPropertyLimitUseByAlign"         , &Functions::itemPropertyLimitUseByAlign          },
	{  673, "ItemPropertyLimitUseByClass"         , &Functions::itemPropertyLimitUseByClass          },
	{  674, "ItemPropertyLimitUseByRace"          , &Functions::itemPropertyLimitUseByRace           },
	{  675, "ItemPropertyLimitUseBySAlign"        , &Functions::itemPropertyLimitUseBySAlign         },
	{  676, "ItemPropertyBonusHitpoints"          , &Functions::itemPropertyBonusHitpoints           },
	{  677, "ItemPropertyVampiricRegeneration"    , &Functions::itemPropertyVampiricRegeneration     },
	{  678, "ItemPropertyTrap"                    , &Functions::itemPropertyTrap                     },
	{  679, "ItemPropertyTrueSeeing"              , &Functions::itemPropertyTrueSeeing               },
	{  680, "ItemPropertyOnMonsterHitProperties"  , &Functions::itemPropertyOnMonsterHitProperties   },
	{  681, "ItemPropertyTurnResistance"          , &Functions::itemPropertyTurnResistance           },
	{  682, "ItemPropertyMassiveCritical"         , &Functions::itemPropertyMassiveCritical          },
	{  683, "ItemPropertyFreeAction"              , &Functions::itemPropertyFreeAction               },
	{  684, "ItemPropertyMonsterDamage"           , &Functions::itemPropertyMonsterDamage            },
	{  685, "ItemPropertyImmunityToSpellLevel"    , &Functions::itemPropertyImmunityToSpellLevel     },
	{  686, "ItemPropertySpecialWalk"             , &Functions::itemPropertySpecialWalk              },
	{  687, "ItemPropertyHealersKit"              , &Functions::itemPropertyHealersKit               },
	{  688, "ItemPropertyWeightIncrease"          , &Functions::itemPropertyWeightIncrease           },
	{  689, "GetIsSkillSuccessful"                , 0                                                },
	{  690, "EffectSpellFailure"                  , 0                                                },
	{  691, "SpeakStringByStrRef"                 , &Functions::speakStringByStrRef                  },
	{  692, "SetCutsceneMode"                     , 0                                                },
	{  693, "GetLastPCToCancelCutscene"           , 0                                                },
	{  694, "GetDialogSoundLength"                , 0                                                },
	{  695, "FadeFromBlack"                       , 0                                                },
	{  696, "FadeToBlack"                         , 0                                                },
	{  697, "StopFade"                            , 0                                                },
	{  698, "BlackScreen"                         , 0                                                },
	{  699, "GetBaseAttackBonus"                  , 0                                                },
	{  700, "SetImmortal"                         , &Functions::setImmortal                          },
	{  701, "OpenInventory"                       , 0                                                },
	{  702, "StoreCameraFacing"                   , 0                                                },
	{  703, "RestoreCameraFacing"                 , 0                                                },
	{  704, "LevelUpHenchman"                     , 0                                                },
	{  705, "SetDroppableFlag"                    , &Functions::setDroppableFlag                     },
	{  706, "GetWeight"                           , 0                                                },
	{  707, "GetModuleItemAcquiredBy"             , 0                                                },
	{  708, "GetImmortal"                         , &Functions::getImmortal                          },
	{  709, "DoWhirlwindAttack"                   , 0                                                },
	{  710, "Get2DAString"                        , &Functions::get2DAString                         },
	{  711, "EffectEthereal"                      , 0                                                },
	{  712, "GetAILevel"                          , 0                                                },
	{  713, "SetAILevel"                          , 0                                                },
	{  714, "GetIsPossessedFamiliar"              , 0                                                },
	{  715, "UnpossessFamiliar"                   , 0                                                },
	{  716, "GetIsAreaInterior"                   , 0                                                },
	{  717, "SendMessageToPCByStrRef"             , 0                                                },
	{  718, "IncrementRemainingFeatUses"          , 0                                                },
	{  719, "ExportSingleCharacter"               , 0                                                },
	{  720, "PlaySoundByStrRef"                   , &Functions::playSoundByStrRef                    },
	{  721, "SetSubRace"                          , 0                                                },
	{  722, "SetDeity"                            , 0                                                },
	{  723, "GetIsDMPossessed"                    , 0                                                },
	{  724, "GetWeather"                          , 0                                                },
	{  725, "GetIsAreaNatural"                    , 0                                                },
	{  726, "GetIsAreaAboveGround"                , 0                                                },
	{  727, "GetPCItemLastEquipped"               , 0                                                },
	{  728, "GetPCItemLastEquippedBy"             , 0                                                },
	{  729, "GetPCItemLastUnequipped"             , 0                                                },
	{  730, "GetPCItemLastUnequippedBy"           , 0                                                },
	{  731, "CopyItemAndModify"                   , 0                                                },
	{  732, "GetItemAppearance"                   , 0                                                },
	{  733, "ItemPropertyOnHitCastSpell"          , &Functions::itemPropertyOnHitCastSpell           },
	{  734, "GetItemPropertySubType"              , &Functions::getItemPropertySubType               },
	{  735, "GetActionMode"                       , 0                                                },
	{  736, "SetActionMode"                       , 0                                                },
	{  737, "GetArcaneSpellFailure"               , 0                                                },
	{  738, "ActionExamine"                       , 0                                                },
	{  739, "ItemPropertyVisualEffect"            , &Functions::itemPropertyVisualEffect             },
	{  740, "SetLootable"                         , &Functions::setLootable                          },
	{  741, "GetLootable"                         , &Functions::getLootable                          },
	{  742, "GetCutsceneCameraMoveRate"           , 0                                                },
	{  743, "SetCutsceneCameraMoveRate"           , 0                                                },
	{  744, "GetItemCursedFlag"                   , &Functions::getItemCursedFlag                    },
	{  745, "SetItemCursedFlag"                   , &Functions::setItemCursedFlag                    },
	{  746, "SetMaxHenchmen"                      , 0                                                },
	{  747, "GetMaxHenchmen"                      , 0                                                },
	{  748, "GetAssociateType"                    , 0                                                },
	{  749, "GetSpellResistance"                  , 0                                                },
	{  750, "DayToNight"                          , 0                                                },
	{  751, "NightToDay"                          , 0                                                },
	{  752, "LineOfSightObject"                   , 0                                                },
	{  753, "LineOfSightVector"                   , 0                                                },
	{  754, "GetLastSpellCastClass"               , 0                                                },
	{  755, "SetBaseAttackBonus"                  , 0                                                },
	{  756, "RestoreBaseAttackBonus"              , 0                                                },
	{  757, "EffectCutsceneGhost"                 , 0                                                },
	{  758, "ItemPropertyArcaneSpellFailure"      , &Functions::itemPropertyArcaneSpellFailure       },
	{  759, "GetStoreGold"                        , &Functions::getStoreGold                         },
	{  760, "SetStoreGold"                        , &Functions::setStoreGold                         },
	{  761, "GetStoreMaxBuyPrice"                 , &Functions::getStoreMaxBuyPrice                  },
	{  762, "SetStoreMaxBuyPrice"                 , &Functions::setStoreMaxBuyPrice                  },
	{  763, "GetStoreIdentifyCost"                , &Functions::getStoreIdentifyCost                 },
	{  764, "SetStoreIdentifyCost"                , &Functions::setStoreIdentifyCost                 },
	{  765, "SetCreatureAppearanceType"           , 0                                                },
	{  766, "GetCreatureStartingPackage"          , 0                                                },
	{  767, "EffectCutsceneImmobilize"            , 0                                                },
	{  768, "GetIsInSubArea"                      , 0                                                },
	{  769, "GetItemPropertyCostTable"            , &Functions::getItemPropertyCostTable             },
	{  770, "GetItemPropertyCostTableValue"       , &Functions::getItemPropertyCostTableValue        },
	{  771, "GetItemPropertyParam1"               , &Functions::getItemPropertyParam1                },
	{  772, "GetItemPropertyParam1Value"          , &Functions::getItemPropertyParam1Value           },
	{  773, "GetIsCreatureDisarmable"             , &Functions::getIsCreatureDisarmable              },
	{  774, "SetStolenFlag"                       , &Functions::setStolenFlag                        },
	{  775, "ForceRest"                           , 0                                                },
	{  776, "SetCameraHeight"                     , 0                                                },
	{  777, "SetGlobalInt"                        , &Functions::setGlobalInt                         },
	{  778, "SetGlobalBool"                       , &Functions::setGlobalBool                        },
	{  779, "SetGlobalString"                     , &Functions::setGlobalString                      },
	{  780, "SetGlobalFloat"                      , &Functions::setGlobalFloat                       },
	{  781, "GetGlobalInt"                        , &Functions::getGlobalInt                         },
	{  782, "GetGlobalBool"                       , &Functions::getGlobalBool                        },
	{  783, "GetGlobalString"                     , &Functions::getGlobalString                      },
	{  784, "GetGlobalFloat"                      , &Functions::getGlobalFloat                       },
	{  785, "SaveGlobalVariables"                 , 0                                                },
	{  786, "LoadGlobalVariables"                 , 0                                                },
	{  787, "MountObject"                         , 0                                                },
	{  788, "DismountObject"                      , 0                                                },
	{  789, "GetJournalEntry"                     , &Functions::getJournalEntry                      },
	{  790, "EffectNWN2ParticleEffect"            , 0                                                },
	{  791, "EffectNWN2ParticleEffectFile"        , 0                                                },
	{  792, "EffectNWN2SpecialEffectFile"         , 0                                                },
	{  793, "GetSpellLevel"                       , 0                                                },
	{  794, "RemoveSEFFromObject"                 , 0                                                },
	{  795, "ActionPauseCutscene"                 , 0                                                },
	{  796, "AssignCutsceneActionToObject"        , 0                                                },
	{  797, "GetCharBackground"                   , 0                                                },
	{  798, "SetTrapActive"                       , &Functions::setTrapActive                        },
	{  799, "SetOrientOnDialog"                   , 0                                                },
	{  800, "EffectDetectUndead"                  , 0                                                },
	{  801, "EffectLowLightVision"                , 0                                                },
	{  802, "EffectSetScale"                      , 0                                                },
	{  803, "EffectShareDamage"                   , 0                                                },
	{  804, "EffectAssayResistance"               , 0                                                },
	{  805, "EffectSeeTrueHPs"                    , 0                                                },
	{  806, "GetNumCutsceneActionsPending"        , 0                                                },
	{  807, "EffectDamageOverTime"                , 0                                                },
	{  808, "EffectAbsorbDamage"                  , 0                                                },
	{  809, "EffectHideousBlow"                   , 0                                                },
	{  810, "EffectMesmerize"                     , 0                                                },
	{  811, "GetSpellFeatId"                      , 0                                                },
	{  812, "SetFog"                              , 0                                                },
	{  813, "EffectDarkVision"                    , 0                                                },
	{  814, "DebugPostString"                     , 0                                                },
	{  815, "GetHasAnySpellEffect"                , 0                                                },
	{  816, "EffectArmorCheckPenaltyIncrease"     , 0                                                },
	{  817, "EffectDisintegrate"                  , 0                                                },
	{  818, "EffectHealOnZeroHP"                  , 0                                                },
	{  819, "EffectBreakEnchantment"              , 0                                                },
	{  820, "GetFirstEnteringPC"                  , 0                                                },
	{  821, "GetNextEnteringPC"                   , 0                                                },
	{  822, "AddRosterMemberByTemplate"           , &Functions::addRosterMemberByTemplate            },
	{  823, "AddRosterMemberByCharacter"          , 0                                                },
	{  824, "RemoveRosterMember"                  , 0                                                },
	{  825, "GetIsRosterMemberAvailable"          , &Functions::getIsRosterMemberAvailable           },
	{  826, "GetIsRosterMemberSelectable"         , &Functions::getIsRosterMemberSelectable          },
	{  827, "SetIsRosterMemberSelectable"         , &Functions::setIsRosterMemberSelectable          },
	{  828, "GetObjectFromRosterName"             , 0                                                },
	{  829, "GetRosterNameFromObject"             , 0                                                },
	{  830, "SpawnRosterMember"                   , 0                                                },
	{  831, "DespawnRosterMember"                 , 0                                                },
	{  832, "AddRosterMemberToParty"              , 0                                                },
	{  833, "RemoveRosterMemberFromParty"         , 0                                                },
	{  834, "GetFirstRosterMember"                , &Functions::getFirstRosterMember                 },
	{  835, "GetNextRosterMember"                 , &Functions::getNextRosterMember                  },
	{  836, "SpawnSpellProjectile"                , 0                                                },
	{  837, "SpawnItemProjectile"                 , 0                                                },
	{  838, "GetIsOwnedByPlayer"                  , 0                                                },
	{  839, "SetOwnersControlledCompanion"        , 0                                                },
	{  840, "SetCreatureScriptsToSet"             , 0                                                },
	{  841, "GetProjectileTravelTime"             , 0                                                },
	{  842, "SetRosterNPCPartyLimit"              , &Functions::setRosterNPCPartyLimit               },
	{  843, "GetRosterNPCPartyLimit"              , &Functions::getRosterNPCPartyLimit               },
	{  844, "SetIsRosterMemberCampaignNPC"        , &Functions::setIsRosterMemberCampaignNPC         },
	{  845, "GetIsRosterMemberCampaignNPC"        , &Functions::getIsRosterMemberCampaignNPC         },
	{  846, "GetIsRosterMember"                   , &Functions::getIsRosterMember                    },
	{  847, "ShowWorldMap"                        , 0                                                },
	{  848, "TriggerEncounter"                    , 0                                                },
	{  849, "GetIsSinglePlayer"                   , &Functions::getIsSinglePlayer                    },
	{  850, "DisplayGuiScreen"                    , 0                                                },
	{  851, "EffectOnDispel"                      , 0                                                },
	{  852, "LoadNewModule"                       , 0                                                },
	{  853, "SetScriptHidden"                     , 0                                                },
	{  854, "SetIsCompanionPossessionBlocked"     , 0                                                },
	{  855, "SetEventHandler"                     , 0                                                },
	{  856, "GetEventHandler"                     , 0                                                },
	{  857, "GetIsPartyTransition"                , 0                                                },
	{  858, "JumpPartyToArea"                     , 0                                                },
	{  859, "GetNumActions"                       , 0                                                },
	{  860, "DisplayMessageBox"                   , 0                                                },
	{  861, "StringCompare"                       , 0                                                },
	{  862, "CharToASCII"                         , 0                                                },
	{  863, "GetOwnedCharacter"                   , 0                                                },
	{  864, "GetControlledCharacter"              , 0                                                },
	{  865, "FeatAdd"                             , &Functions::featAdd                              },
	{  866, "FeatRemove"                          , 0                                                },
	{  867, "SetCanTalkToNonPlayerOwnedCreatures" , 0                                                },
	{  868, "GetCanTalkToNonPlayerOwnedCreatures" , 0                                                },
	{  869, "SetLevelUpPackage"                   , 0                                                },
	{  870, "GetLevelUpPackage"                   , 0                                                },
	{  871, "SetCombatOverrides"                  , 0                                                },
	{  872, "ClearCombatOverrides"                , 0                                                },
	{  873, "ResetCreatureLevelForXP"             , 0                                                },
	{  874, "CalcPointAwayFromPoint"              , 0                                                },
	{  875, "CalcSafeLocation"                    , 0                                                },
	{  876, "GetTotalLevels"                      , 0                                                },
	{  877, "ResetFeatUses"                       , 0                                                },
	{  878, "SetNWN2Fog"                          , 0                                                },
	{  879, "ResetNWN2Fog"                        , 0                                                },
	{  880, "EffectBonusHitpoints"                , 0                                                },
	{  881, "SetGUIObjectHidden"                  , 0                                                },
	{  882, "CloseGUIScreen"                      , 0                                                },
	{  883, "FiredFromPartyTransition"            , 0                                                },
	{  884, "GetScriptHidden"                     , 0                                                },
	{  885, "GetNodeSpeaker"                      , 0                                                },
	{  886, "SetLocalGUIVariable"                 , 0                                                },
	{  887, "SetGUIObjectDisabled"                , 0                                                },
	{  888, "SetGUIObjectText"                    , 0                                                },
	{  889, "GetIsCompanionPossessionBlocked"     , 0                                                },
	{  890, "EffectBardSongSinging"               , 0                                                },
	{  891, "EffectJarring"                       , 0                                                },
	{  892, "GetEffectInteger"                    , 0                                                },
	{  893, "RefreshSpellEffectDurations"         , 0                                                },
	{  894, "SetEffectSpellId"                    , 0                                                },
	{  895, "EffectBABMinimum"                    , 0                                                },
	{  896, "GetTRUEBaseAttackBonus"              , 0                                                },
	{  897, "SetFirstName"                        , 0                                                },
	{  898, "SetLastName"                         , 0                                                },
	{  899, "SetDescription"                      , 0                                                },
	{  900, "GetFirstName"                        , 0                                                },
	{  901, "GetLastName"                         , 0                                                },
	{  902, "GetDescription"                      , 0                                                },
	{  903, "IsInMultiplayerConversation"         , 0                                                },
	{  904, "PlayCustomAnimation"                 , 0                                                },
	{  905, "EffectMaxDamage"                     , 0                                                },
	{  906, "DisplayInputBox"                     , 0                                                },
	{  907, "SetWeaponVisibility"                 , 0                                                },
	{  908, "SetLookAtTarget"                     , 0                                                },
	{  909, "GetBumpState"                        , 0                                                },
	{  910, "SetBumpState"                        , 0                                                },
	{  911, "GetOnePartyMode"                     , &Functions::getOnePartyMode                      },
	{  912, "GetPrimaryPlayer"                    , 0                                                },
	{  913, "EffectArcaneSpellFailure"            , 0                                                },
	{  914, "SpawnBloodHit"                       , 0                                                },
	{  915, "GetFirstArea"                        , 0                                                },
	{  916, "GetNextArea"                         , 0                                                },
	{  917, "GetArmorRank"                        , 0                                                },
	{  918, "GetWeaponType"                       , 0                                                },
	{  919, "GetPlayerCurrentTarget"              , 0                                                },
	{  920, "EffectWildshape"                     , 0                                                },
	{  921, "GetEncumbranceState"                 , 0                                                },
	{  922, "PackCampaignDatabase"                , 0                                                },
	{  923, "UnlinkDoor"                          , 0                                                },
	{  924, "GetPlayerCreatureExamineTarget"      , 0                                                },
	{  925, "Clear2DACache"                       , 0                                                },
	{  926, "NWNXGetInt"                          , 0                                                },
	{  927, "NWNXGetFloat"                        , 0                                                },
	{  928, "NWNXGetString"                       , 0                                                },
	{  929, "NWNXSetInt"                          , 0                                                },
	{  930, "NWNXSetFloat"                        , 0                                                },
	{  931, "NWNXSetString"                       , 0                                                },
	{  932, "EffectEffectIcon"                    , 0                                                },
	{  933, "SetGUIProgressBarPosition"           , 0                                                },
	{  934, "SetGUITexture"                       , 0                                                },
	{  935, "EffectRescue"                        , 0                                                },
	{  936, "IntToObject"                         , 0                                                },
	{  937, "ObjectToInt"                         , 0                                                },
	{  938, "StringToObject"                      , 0                                                },
	{  939, "GetIsSpirit"                         , 0                                                },
	{  940, "EffectDetectSpirits"                 , 0                                                },
	{  941, "EffectDamageReductionNegated"        , 0                                                },
	{  942, "EffectConcealmentNegated"            , 0                                                },
	{  943, "GetInfiniteFlag"                     , &Functions::getInfiniteFlag                      },
	{  944, "GetKeyRequiredFeedbackMessage"       , &Functions::getKeyRequiredFeedbackMessage        },
	{  945, "SetKeyRequiredFeedbackMessage"       , &Functions::setKeyRequiredFeedbackMessage        },
	{  946, "SetInfiniteFlag"                     , &Functions::setInfiniteFlag                      },
	{  947, "GetPickpocketableFlag"               , &Functions::getPickpocketableFlag                },
	{  948, "SetPickpocketableFlag"               , &Functions::setPickpocketableFlag                },
	{  949, "GetTrapActive"                       , &Functions::getTrapActive                        },
	{  950, "SetWillSavingThrow"                  , 0                                                },
	{  951, "SetReflexSavingThrow"                , 0                                                },
	{  952, "SetFortitudeSavingThrow"             , 0                                                },
	{  953, "SetLockKeyRequired"                  , &Functions::setLockKeyRequired                   },
	{  954, "SetLockKeyTag"                       , &Functions::setLockKeyTag                        },
	{  955, "SetLockLockDC"                       , &Functions::setLockLockDC                        },
	{  956, "SetLockUnlockDC"                     , &Functions::setLockUnlockDC                      },
	{  957, "SetLockLockable"                     , &Functions::setLockLockable                      },
	{  958, "SetHardness"                         , 0                                                },
	{  959, "GetHardness"                         , 0                                                },
	{  960, "GetModuleXPScale"                    , 0                                                },
	{  961, "SetModuleXPScale"                    , 0                                                },
	{  962, "SetTrapDetectable"                   , &Functions::setTrapDetectable                    },
	{  963, "SetTrapDetectDC"                     , &Functions::setTrapDetectDC                      },
	{  964, "SetTrapDisarmable"                   , &Functions::setTrapDisarmable                    },
	{  965, "SetTrapDisarmDC"                     , &Functions::setTrapDisarmDC                      },
	{  966, "SetTrapKeyTag"                       , &Functions::setTrapKeyTag                        },
	{  967, "SetTrapOneShot"                      , &Functions::setTrapOneShot                       },
	{  968, "CreateTrapAtLocation"                , 0                                                },
	{  969, "CreateTrapOnObject"                  , &Functions::createTrapOnObject                   },
	{  970, "GetAreaSize"                         , 0                                                },
	{  971, "GetTrapRecoverable"                  , &Functions::getTrapRecoverable                   },
	{  972, "SetTrapRecoverable"                  , &Functions::setTrapRecoverable                   },
	{  973, "SetUseableFlag"                      , &Functions::setUseableFlag                       },
	{  974, "GetPlaceableLastClickedBy"           , 0                                                },
	{  975, "SetRenderWaterInArea"                , 0                                                },
	{  976, "EffectInsane"                        , 0                                                },
	{  977, "SetPlayerGUIHidden"                  , 0                                                },
	{  978, "GetSelectedMapPointTag"              , 0                                                },
	{  979, "SetNoticeText"                       , 0                                                },
	{  980, "SetLightActive"                      , 0                                                },
	{  981, "EffectSummonCopy"                    , 0                                                },
	{  982, "GetPolymorphLocked"                  , 0                                                },
	{  983, "SetSoundSet"                         , 0                                                },
	{  984, "SetScale"                            , 0                                                },
	{  985, "GetScale"                            , 0                                                },
	{  986, "GetNum2DARows"                       , 0                                                },
	{  987, "GetNum2DAColumns"                    , 0                                                },
	{  988, "SetCustomHeartbeat"                  , 0                                                },
	{  989, "GetCustomHeartbeat"                  , 0                                                },
	{  990, "SetScrollBarRanges"                  , 0                                                },
	{  991, "ClearListBox"                        , 0                                                },
	{  992, "AddListBoxRow"                       , 0                                                },
	{  993, "RemoveListBoxRow"                    , 0                                                },
	{  994, "GetItemPropActivation"               , 0                                                },
	{  995, "SetItemPropActivation"               , 0                                                },
	{  996, "ModifyListBoxRow"                    , 0                                                },
	{  997, "SetFactionLeader"                    , 0                                                },
	{  998, "GetFirstSubArea"                     , 0                                                },
	{  999, "GetNextSubArea"                      , 0                                                },
	{ 1000, "GetMovementRateFactor"               , 0                                                },
	{ 1001, "SetMovementRateFactor"               , 0                                                },
	{ 1002, "GetBicFileName"                      , 0                                                },
	{ 1003, "GetCollision"                        , 0                                                },
	{ 1004, "SetCollision"                        , 0                                                },
	{ 1005, "GetItemIcon"                         , &Functions::getItemIcon                          },
	{ 1006, "GetVariableName"                     , 0                                                },
	{ 1007, "GetVariableType"                     , 0                                                },
	{ 1008, "GetAreaOfEffectDuration"             , 0                                                },
	{ 1009, "GetIsPlayerCreated"                  , 0                                                },
	{ 1010, "GetPartyName"                        , &Functions::getPartyName                         },
	{ 1011, "GetPartyMotto"                       , &Functions::getPartyMotto                        },
	{ 1012, "GetIsOverlandMap"                    , 0                                                },
	{ 1013, "SetUnrestrictedLevelUp"              , 0                                                },
	{ 1014, "GetSoundFileDuration"                , 0                                                },
	{ 1015, "GetPartyMembersDyingFlag"            , 0                                                },
	{ 1016, "SetListBoxRowSelected"               , 0                                                },
	{ 1017, "GetTalkTableLanguage"                , 0                                                },
	{ 1018, "SetScrollBarValue"                   , 0                                                },
	{ 1019, "SetPause"                            , 0                                                },
	{ 1020, "GetPause"                            , 0                                                },
	{ 1021, "GetAreaOfEffectSpellId"              , 0                                                },
	{ 1022, "SetGlobalGUIVariable"                , 0                                                },
	{ 1023, "CreateInstancedAreaFromSource"       , 0                                                },
	{ 1024, "GetVariableValueInt"                 , 0                                                },
	{ 1025, "GetVariableValueString"              , 0                                                },
	{ 1026, "GetVariableValueFloat"               , 0                                                },
	{ 1027, "GetVariableValueLocation"            , 0                                                },
	{ 1028, "GetVariableValueObject"              , 0                                                },
	{ 1029, "GetVariableCount"                    , 0                                                },
	{ 1030, "SetBaseAbilityScore"                 , 0                                                },
	{ 1031, "SetBaseSkillRank"                    , 0                                                },
	{ 1032, "SendChatMessage"                     , 0                                                },
	{ 1033, "GetIsLocationValid"                  , 0                                                },
	{ 1034, "GetSurfaceMaterialsAtLocation"       , 0                                                },
	{ 1035, "GetSpellKnown"                       , 0                                                },
	{ 1036, "GetItemBaseMaterialType"             , 0                                                },
	{ 1037, "SetItemBaseMaterialType"             , 0                                                },
	{ 1038, "SetSpellKnown"                       , 0                                                },
	{ 1039, "GetLimboCreatureCount"               , 0                                                },
	{ 1040, "GetCreatureInLimbo"                  , 0                                                },
	{ 1041, "SendCreatureToLimbo"                 , 0                                                },
	{ 1042, "AddScriptParameterInt"               , 0                                                },
	{ 1043, "AddScriptParameterString"            , 0                                                },
	{ 1044, "AddScriptParameterFloat"             , 0                                                },
	{ 1045, "AddScriptParameterObject"            , 0                                                },
	{ 1046, "ExecuteScriptEnhanced"               , 0                                                },
	{ 1047, "ClearScriptParams"                   , 0                                                },
	{ 1048, "SetSkillPointsRemaining"             , 0                                                },
	{ 1049, "GetSkillPointsRemaining"             , 0                                                },
	{ 1050, "GetCasterClassSpellSchool"           , 0                                                },
	{ 1051, "SetGender"                           , 0                                                },
	{ 1052, "SetTag"                              , 0                                                },
	{ 1053, "GetArmorRulesType"                   , 0                                                },
	{ 1054, "SetArmorRulesType"                   , 0                                                },
	{ 1055, "SetItemIcon"                         , &Functions::setItemIcon                          },
	{ 1056, "GetObjectByTagAndType"               , 0                                                },
	{ 1057, "RecallCreatureFromLimboToLocation"   , 0                                                },
	{ 1058, "SetIsRosterMemberAvailable"          , &Functions::setIsRosterMemberAvailable           }
};

/** The table defining the signature (return type and type of parameters) of each engine function. */
const Functions::FunctionSignature Functions::kFunctionSignatures[] = {
	{    0, kTypeInt       , { kTypeInt } },
	{    1, kTypeVoid      , { kTypeString } },
	{    2, kTypeVoid      , { kTypeFloat,kTypeInt,kTypeInt } },
	{    3, kTypeString    , { kTypeFloat,kTypeInt,kTypeInt } },
	{    4, kTypeVoid      , { kTypeInt } },
	{    5, kTypeVoid      , { kTypeObject } },
	{    6, kTypeVoid      , { kTypeObject,kTypeScriptState } },
	{    7, kTypeVoid      , { kTypeFloat,kTypeScriptState } },
	{    8, kTypeVoid      , { kTypeString,kTypeObject } },
	{    9, kTypeVoid      , { kTypeInt } },
	{   10, kTypeVoid      , { kTypeFloat,kTypeInt } },
	{   11, kTypeVoid      , { kTypeInt,kTypeInt,kTypeInt } },
	{   12, kTypeVoid      , { kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{   13, kTypeInt       , { } },
	{   14, kTypeInt       , { } },
	{   15, kTypeInt       , { } },
	{   16, kTypeInt       , { } },
	{   17, kTypeInt       , { } },
	{   18, kTypeInt       , { } },
	{   19, kTypeInt       , { } },
	{   20, kTypeVoid      , { } },
	{   21, kTypeVoid      , { kTypeEngineType,kTypeInt } },
	{   22, kTypeVoid      , { kTypeObject,kTypeInt,kTypeFloat } },
	{   23, kTypeVoid      , { kTypeObject,kTypeInt,kTypeFloat } },
	{   24, kTypeObject    , { kTypeObject } },
	{   25, kTypeObject    , { } },
	{   26, kTypeObject    , { } },
	{   27, kTypeVector    , { kTypeObject } },
	{   28, kTypeFloat     , { kTypeObject } },
	{   29, kTypeObject    , { kTypeObject } },
	{   30, kTypeObject    , { kTypeObject,kTypeString } },
	{   31, kTypeObject    , { kTypeString,kTypeObject,kTypeInt,kTypeString,kTypeInt } },
	{   32, kTypeVoid      , { kTypeObject,kTypeInt } },
	{   33, kTypeVoid      , { kTypeObject } },
	{   34, kTypeVoid      , { kTypeObject } },
	{   35, kTypeVoid      , { kTypeObject } },
	{   36, kTypeObject    , { kTypeObject } },
	{   37, kTypeVoid      , { kTypeObject,kTypeInt } },
	{   38, kTypeObject    , { kTypeInt,kTypeInt,kTypeObject,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{   39, kTypeVoid      , { kTypeString,kTypeInt } },
	{   40, kTypeVoid      , { kTypeInt,kTypeFloat,kTypeFloat } },
	{   41, kTypeFloat     , { kTypeObject } },
	{   42, kTypeInt       , { kTypeObject } },
	{   43, kTypeVoid      , { kTypeObject } },
	{   44, kTypeVoid      , { kTypeObject } },
	{   45, kTypeVoid      , { kTypeFloat,kTypeFloat,kTypeFloat,kTypeInt } },
	{   46, kTypeVoid      , { kTypeString,kTypeInt } },
	{   47, kTypeObject    , { } },
	{   48, kTypeVoid      , { kTypeInt,kTypeObject,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{   49, kTypeInt       , { kTypeObject } },
	{   50, kTypeInt       , { kTypeObject } },
	{   51, kTypeInt       , { kTypeObject,kTypeString } },
	{   52, kTypeFloat     , { kTypeObject,kTypeString } },
	{   53, kTypeString    , { kTypeObject,kTypeString } },
	{   54, kTypeObject    , { kTypeObject,kTypeString } },
	{   55, kTypeVoid      , { kTypeObject,kTypeString,kTypeInt } },
	{   56, kTypeVoid      , { kTypeObject,kTypeString,kTypeFloat } },
	{   57, kTypeVoid      , { kTypeObject,kTypeString,kTypeString } },
	{   58, kTypeVoid      , { kTypeObject,kTypeString,kTypeObject } },
	{   59, kTypeInt       , { kTypeString } },
	{   60, kTypeString    , { kTypeString } },
	{   61, kTypeString    , { kTypeString } },
	{   62, kTypeString    , { kTypeString,kTypeInt } },
	{   63, kTypeString    , { kTypeString,kTypeInt } },
	{   64, kTypeString    , { kTypeString,kTypeString,kTypeInt } },
	{   65, kTypeString    , { kTypeString,kTypeInt,kTypeInt } },
	{   66, kTypeInt       , { kTypeString,kTypeString,kTypeInt } },
	{   67, kTypeFloat     , { kTypeFloat } },
	{   68, kTypeFloat     , { kTypeFloat } },
	{   69, kTypeFloat     , { kTypeFloat } },
	{   70, kTypeFloat     , { kTypeFloat } },
	{   71, kTypeFloat     , { kTypeFloat } },
	{   72, kTypeFloat     , { kTypeFloat } },
	{   73, kTypeFloat     , { kTypeFloat } },
	{   74, kTypeFloat     , { kTypeFloat } },
	{   75, kTypeFloat     , { kTypeFloat,kTypeFloat } },
	{   76, kTypeFloat     , { kTypeFloat } },
	{   77, kTypeInt       , { kTypeInt } },
	{   78, kTypeEngineType, { kTypeInt } },
	{   79, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{   80, kTypeEngineType, { kTypeInt,kTypeInt } },
	{   81, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{   82, kTypeEngineType, { } },
	{   83, kTypeEngineType, { kTypeString,kTypeInt,kTypeFloat,kTypeInt } },
	{   84, kTypeInt       , { kTypeObject } },
	{   85, kTypeEngineType, { kTypeObject } },
	{   86, kTypeEngineType, { kTypeObject } },
	{   87, kTypeVoid      , { kTypeObject,kTypeEngineType } },
	{   88, kTypeInt       , { kTypeEngineType } },
	{   89, kTypeInt       , { kTypeEngineType } },
	{   90, kTypeInt       , { kTypeEngineType } },
	{   91, kTypeObject    , { kTypeEngineType } },
	{   92, kTypeString    , { kTypeInt } },
	{   93, kTypeObject    , { kTypeObject } },
	{   94, kTypeObject    , { kTypeObject } },
	{   95, kTypeInt       , { kTypeInt } },
	{   96, kTypeInt       , { kTypeInt } },
	{   97, kTypeInt       , { kTypeInt } },
	{   98, kTypeInt       , { kTypeInt } },
	{   99, kTypeInt       , { kTypeInt } },
	{  100, kTypeInt       , { kTypeInt } },
	{  101, kTypeInt       , { kTypeInt } },
	{  102, kTypeInt       , { kTypeInt } },
	{  103, kTypeInt       , { kTypeInt } },
	{  104, kTypeFloat     , { kTypeVector } },
	{  105, kTypeInt       , { } },
	{  106, kTypeInt       , { kTypeObject } },
	{  107, kTypeInt       , { kTypeObject } },
	{  108, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt,kTypeObject } },
	{  109, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt,kTypeObject } },
	{  110, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt,kTypeObject } },
	{  111, kTypeInt       , { } },
	{  112, kTypeEngineType, { kTypeEngineType } },
	{  113, kTypeEngineType, { kTypeEngineType } },
	{  114, kTypeEngineType, { kTypeEngineType } },
	{  115, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  116, kTypeInt       , { kTypeObject,kTypeInt } },
	{  117, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  118, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  119, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  120, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  121, kTypeFloat     , { kTypeInt } },
	{  122, kTypeFloat     , { kTypeInt } },
	{  123, kTypeFloat     , { kTypeInt } },
	{  124, kTypeInt       , { kTypeObject } },
	{  125, kTypeInt       , { kTypeObject } },
	{  126, kTypeInt       , { kTypeObject } },
	{  127, kTypeInt       , { kTypeObject } },
	{  128, kTypeObject    , { kTypeInt,kTypeFloat,kTypeEngineType,kTypeInt,kTypeInt,kTypeVector } },
	{  129, kTypeObject    , { kTypeInt,kTypeFloat,kTypeEngineType,kTypeInt,kTypeInt,kTypeVector } },
	{  130, kTypeEngineType, { } },
	{  131, kTypeVoid      , { kTypeObject,kTypeEngineType } },
	{  132, kTypeEngineType, { kTypeInt } },
	{  133, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  134, kTypeEngineType, { } },
	{  135, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt } },
	{  136, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt } },
	{  137, kTypeVector    , { kTypeVector } },
	{  138, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  139, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt } },
	{  140, kTypeInt       , { kTypeObject,kTypeInt } },
	{  141, kTypeVoid      , { kTypeVector,kTypeInt } },
	{  142, kTypeVector    , { kTypeFloat,kTypeFloat,kTypeFloat } },
	{  143, kTypeVoid      , { kTypeVector,kTypeInt } },
	{  144, kTypeVector    , { kTypeFloat } },
	{  145, kTypeFloat     , { kTypeVector } },
	{  146, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt } },
	{  147, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt } },
	{  148, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  149, kTypeEngineType, { kTypeInt } },
	{  150, kTypeEngineType, { } },
	{  151, kTypeFloat     , { kTypeObject,kTypeObject } },
	{  152, kTypeVoid      , { kTypeObject,kTypeString,kTypeEngineType } },
	{  153, kTypeEngineType, { kTypeObject,kTypeString } },
	{  154, kTypeEngineType, { } },
	{  155, kTypeObject    , { kTypeInt,kTypeObject } },
	{  156, kTypeEngineType, { } },
	{  157, kTypeEngineType, { } },
	{  158, kTypeEngineType, { } },
	{  159, kTypeEngineType, { } },
	{  160, kTypeEngineType, { } },
	{  161, kTypeEngineType, { } },
	{  162, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  163, kTypeInt       , { kTypeObject } },
	{  164, kTypeEngineType, { kTypeInt,kTypeFloat } },
	{  165, kTypeEngineType, { kTypeInt } },
	{  166, kTypeInt       , { kTypeObject } },
	{  167, kTypeVoid      , { kTypeObject,kTypeFloat,kTypeInt } },
	{  168, kTypeString    , { kTypeObject } },
	{  169, kTypeInt       , { kTypeObject,kTypeObject } },
	{  170, kTypeInt       , { kTypeEngineType } },
	{  171, kTypeEngineType, { kTypeInt,kTypeString,kTypeString,kTypeString,kTypeString } },
	{  172, kTypeInt       , { kTypeObject,kTypeObject } },
	{  173, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  174, kTypeInt       , { kTypeObject } },
	{  175, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  176, kTypeVoid      , { kTypeObject,kTypeString,kTypeInt } },
	{  177, kTypeInt       , { kTypeString,kTypeString } },
	{  178, kTypeString    , { kTypeInt } },
	{  179, kTypeInt       , { } },
	{  180, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  181, kTypeObject    , { kTypeObject,kTypeInt } },
	{  182, kTypeObject    , { kTypeObject,kTypeInt } },
	{  183, kTypeObject    , { kTypeObject,kTypeInt } },
	{  184, kTypeObject    , { kTypeObject,kTypeInt } },
	{  185, kTypeInt       , { kTypeObject } },
	{  186, kTypeInt       , { kTypeObject,kTypeObject } },
	{  187, kTypeInt       , { kTypeObject } },
	{  188, kTypeInt       , { kTypeObject } },
	{  189, kTypeInt       , { kTypeObject } },
	{  190, kTypeInt       , { kTypeObject } },
	{  191, kTypeInt       , { kTypeObject } },
	{  192, kTypeObject    , { kTypeObject,kTypeInt } },
	{  193, kTypeObject    , { kTypeObject,kTypeInt } },
	{  194, kTypeVoid      , { kTypeObject } },
	{  195, kTypeInt       , { } },
	{  196, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  197, kTypeObject    , { kTypeString } },
	{  198, kTypeObject    , { kTypeObject } },
	{  199, kTypeEngineType, { kTypeEngineType,kTypeEngineType } },
	{  200, kTypeObject    , { kTypeString,kTypeInt } },
	{  201, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  202, kTypeVoid      , { kTypeFloat } },
	{  203, kTypeVoid      , { kTypeInt,kTypeString } },
	{  204, kTypeVoid      , { kTypeObject,kTypeString,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  205, kTypeVoid      , { } },
	{  206, kTypeVoid      , { } },
	{  207, kTypeEngineType, { kTypeInt,kTypeObject,kTypeInt,kTypeInt } },
	{  208, kTypeInt       , { kTypeObject,kTypeObject } },
	{  209, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt } },
	{  210, kTypeObject    , { kTypeObject } },
	{  211, kTypeObject    , { kTypeObject } },
	{  212, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  213, kTypeEngineType, { kTypeObject } },
	{  214, kTypeVoid      , { kTypeEngineType } },
	{  215, kTypeEngineType, { kTypeObject,kTypeVector,kTypeFloat } },
	{  216, kTypeVoid      , { kTypeInt,kTypeEngineType,kTypeEngineType,kTypeFloat } },
	{  217, kTypeInt       , { kTypeObject } },
	{  218, kTypeFloat     , { kTypeFloat } },
	{  219, kTypeFloat     , { kTypeFloat } },
	{  220, kTypeVoid      , { kTypeInt,kTypeEngineType,kTypeObject,kTypeFloat } },
	{  221, kTypeVoid      , { kTypeString,kTypeInt } },
	{  222, kTypeEngineType, { } },
	{  223, kTypeVector    , { kTypeEngineType } },
	{  224, kTypeObject    , { kTypeEngineType } },
	{  225, kTypeFloat     , { kTypeEngineType } },
	{  226, kTypeObject    , { kTypeInt,kTypeInt,kTypeEngineType,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  227, kTypeObject    , { kTypeInt,kTypeObject,kTypeInt } },
	{  228, kTypeObject    , { kTypeInt,kTypeEngineType,kTypeInt } },
	{  229, kTypeObject    , { kTypeString,kTypeObject,kTypeInt } },
	{  230, kTypeFloat     , { kTypeInt } },
	{  231, kTypeInt       , { kTypeFloat } },
	{  232, kTypeInt       , { kTypeString } },
	{  233, kTypeFloat     , { kTypeString } },
	{  234, kTypeVoid      , { kTypeInt,kTypeEngineType,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  235, kTypeInt       , { kTypeObject,kTypeObject } },
	{  236, kTypeInt       , { kTypeObject,kTypeObject } },
	{  237, kTypeInt       , { kTypeObject,kTypeObject } },
	{  238, kTypeObject    , { } },
	{  239, kTypeString    , { kTypeInt,kTypeInt } },
	{  240, kTypeVoid      , { kTypeInt,kTypeInt } },
	{  241, kTypeVoid      , { kTypeObject,kTypeFloat,kTypeInt } },
	{  242, kTypeObject    , { } },
	{  243, kTypeObject    , { kTypeInt,kTypeString,kTypeEngineType,kTypeInt,kTypeString } },
	{  244, kTypeEngineType, { kTypeObject,kTypeInt,kTypeInt } },
	{  245, kTypeObject    , { } },
	{  246, kTypeInt       , { } },
	{  247, kTypeInt       , { } },
	{  248, kTypeInt       , { } },
	{  249, kTypeString    , { } },
	{  250, kTypeEngineType, { kTypeInt } },
	{  251, kTypeEngineType, { kTypeInt } },
	{  252, kTypeEngineType, { } },
	{  253, kTypeString    , { kTypeObject } },
	{  254, kTypeObject    , { } },
	{  255, kTypeInt       , { kTypeString,kTypeObject,kTypeInt } },
	{  256, kTypeObject    , { } },
	{  257, kTypeInt       , { } },
	{  258, kTypeInt       , { } },
	{  259, kTypeInt       , { } },
	{  260, kTypeObject    , { } },
	{  261, kTypeInt       , { } },
	{  262, kTypeObject    , { kTypeObject,kTypeInt,kTypeInt } },
	{  263, kTypeObject    , { kTypeObject,kTypeInt,kTypeInt } },
	{  264, kTypeObject    , { kTypeObject } },
	{  265, kTypeVoid      , { kTypeObject,kTypeString } },
	{  266, kTypeVoid      , { kTypeObject,kTypeString } },
	{  267, kTypeVoid      , { kTypeObject,kTypeString } },
	{  268, kTypeVoid      , { kTypeObject,kTypeString } },
	{  269, kTypeVoid      , { kTypeObject,kTypeString } },
	{  270, kTypeEngineType, { } },
	{  271, kTypeEngineType, { } },
	{  272, kTypeString    , { kTypeObject } },
	{  273, kTypeEngineType, { kTypeInt } },
	{  274, kTypeInt       , { kTypeObject,kTypeInt,kTypeObject } },
	{  275, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  276, kTypeInt       , { kTypeObject } },
	{  277, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  278, kTypeInt       , { kTypeObject } },
	{  279, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  280, kTypeInt       , { kTypeObject } },
	{  281, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  282, kTypeObject    , { } },
	{  283, kTypeObject    , { } },
	{  284, kTypeVoid      , { kTypeInt,kTypeString } },
	{  285, kTypeInt       , { kTypeInt,kTypeObject,kTypeInt } },
	{  286, kTypeInt       , { kTypeInt,kTypeObject } },
	{  287, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  288, kTypeInt       , { kTypeInt,kTypeObject,kTypeInt,kTypeObject } },
	{  289, kTypeInt       , { kTypeObject,kTypeObject } },
	{  290, kTypeInt       , { kTypeObject,kTypeObject } },
	{  291, kTypeObject    , { } },
	{  292, kTypeObject    , { } },
	{  293, kTypeObject    , { } },
	{  294, kTypeVoid      , { kTypeScriptState } },
	{  295, kTypeEngineType, { } },
	{  296, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  297, kTypeInt       , { kTypeObject } },
	{  298, kTypeFloat     , { kTypeEngineType,kTypeEngineType } },
	{  299, kTypeInt       , { kTypeInt,kTypeObject,kTypeInt,kTypeInt,kTypeObject } },
	{  300, kTypeVoid      , { kTypeInt,kTypeFloat,kTypeFloat } },
	{  301, kTypeEngineType, { kTypeInt } },
	{  302, kTypeEngineType, { kTypeInt } },
	{  303, kTypeEngineType, { kTypeInt } },
	{  304, kTypeInt       , { kTypeInt,kTypeObject } },
	{  305, kTypeInt       , { kTypeEngineType } },
	{  306, kTypeInt       , { kTypeEngineType,kTypeObject } },
	{  307, kTypeEngineType, { kTypeInt,kTypeObject,kTypeInt } },
	{  308, kTypeEngineType, { kTypeInt,kTypeInt,kTypeObject,kTypeInt } },
	{  309, kTypeVoid      , { kTypeEngineType,kTypeObject } },
	{  310, kTypeVoid      , { kTypeEngineType,kTypeEngineType } },
	{  311, kTypeInt       , { kTypeObject } },
	{  312, kTypeInt       , { kTypeObject } },
	{  313, kTypeVoid      , { kTypeEngineType } },
	{  314, kTypeEngineType, { kTypeInt } },
	{  315, kTypeInt       , { kTypeInt,kTypeObject,kTypeInt } },
	{  316, kTypeObject    , { kTypeObject } },
	{  317, kTypeInt       , { kTypeObject } },
	{  318, kTypeInt       , { kTypeObject } },
	{  319, kTypeObject    , { kTypeObject } },
	{  320, kTypeInt       , { kTypeObject } },
	{  321, kTypeInt       , { kTypeObject } },
	{  322, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  323, kTypeVoid      , { kTypeInt,kTypeInt,kTypeInt } },
	{  324, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  325, kTypeInt       , { kTypeObject } },
	{  326, kTypeObject    , { } },
	{  327, kTypeVoid      , { kTypeObject } },
	{  328, kTypeObject    , { kTypeObject } },
	{  329, kTypeVoid      , { kTypeObject } },
	{  330, kTypeObject    , { } },
	{  331, kTypeInt       , { kTypeInt,kTypeObject } },
	{  332, kTypeInt       , { kTypeObject } },
	{  333, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  334, kTypeVoid      , { kTypeObject,kTypeString } },
	{  335, kTypeVoid      , { kTypeObject,kTypeString } },
	{  336, kTypeObject    , { } },
	{  337, kTypeInt       , { kTypeObject,kTypeInt } },
	{  338, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  339, kTypeObject    , { kTypeObject } },
	{  340, kTypeObject    , { kTypeObject } },
	{  341, kTypeInt       , { kTypeInt,kTypeObject } },
	{  342, kTypeInt       , { kTypeInt,kTypeObject } },
	{  343, kTypeInt       , { kTypeInt,kTypeObject } },
	{  344, kTypeInt       , { kTypeInt } },
	{  345, kTypeInt       , { } },
	{  346, kTypeObject    , { kTypeObject } },
	{  347, kTypeObject    , { } },
	{  348, kTypeObject    , { } },
	{  349, kTypeObject    , { } },
	{  350, kTypeObject    , { } },
	{  351, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  352, kTypeInt       , { } },
	{  353, kTypeObject    , { } },
	{  354, kTypeObject    , { kTypeObject,kTypeInt } },
	{  355, kTypeEngineType, { kTypeEngineType,kTypeInt,kTypeInt } },
	{  356, kTypeEngineType, { kTypeEngineType,kTypeInt } },
	{  357, kTypeEngineType, { kTypeEngineType } },
	{  358, kTypeInt       , { kTypeObject } },
	{  359, kTypeInt       , { kTypeEngineType } },
	{  360, kTypeVoid      , { kTypeEngineType,kTypeInt,kTypeFloat } },
	{  361, kTypeObject    , { } },
	{  362, kTypeInt       , { kTypeEngineType } },
	{  363, kTypeInt       , { kTypeEngineType } },
	{  364, kTypeObject    , { kTypeInt,kTypeObject,kTypeInt } },
	{  365, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  366, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  367, kTypeVoid      , { kTypeString,kTypeInt,kTypeObject,kTypeInt,kTypeInt,kTypeInt } },
	{  368, kTypeVoid      , { kTypeString,kTypeObject,kTypeInt,kTypeInt } },
	{  369, kTypeString    , { kTypeObject } },
	{  370, kTypeString    , { kTypeObject } },
	{  371, kTypeString    , { kTypeObject } },
	{  372, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  373, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  374, kTypeVoid      , { kTypeObject,kTypeString } },
	{  375, kTypeObject    , { } },
	{  376, kTypeObject    , { } },
	{  377, kTypeInt       , { kTypeInt,kTypeObject } },
	{  378, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt,kTypeInt } },
	{  379, kTypeEngineType, { } },
	{  380, kTypeObject    , { kTypeObject,kTypeInt } },
	{  381, kTypeObject    , { kTypeObject,kTypeInt } },
	{  382, kTypeVoid      , { kTypeEngineType,kTypeInt,kTypeFloat } },
	{  383, kTypeVoid      , { kTypeObject,kTypeInt,kTypeFloat,kTypeFloat } },
	{  384, kTypeInt       , { kTypeString } },
	{  385, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  386, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  387, kTypeEngineType, { kTypeFloat } },
	{  388, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  389, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  390, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt,kTypeFloat } },
	{  391, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt,kTypeFloat } },
	{  392, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt,kTypeFloat } },
	{  393, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  394, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  395, kTypeInt       , { kTypeObject } },
	{  396, kTypeString    , { kTypeInt } },
	{  397, kTypeInt       , { kTypeObject } },
	{  398, kTypeInt       , { kTypeObject,kTypeInt } },
	{  399, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  400, kTypeVoid      , { kTypeObject } },
	{  401, kTypeInt       , { kTypeObject } },
	{  402, kTypeVoid      , { kTypeInt } },
	{  403, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt } },
	{  404, kTypeVoid      , { } },
	{  405, kTypeInt       , { } },
	{  406, kTypeInt       , { } },
	{  407, kTypeInt       , { } },
	{  408, kTypeInt       , { } },
	{  409, kTypeInt       , { kTypeObject } },
	{  410, kTypeObject    , { } },
	{  411, kTypeEngineType, { } },
	{  412, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  413, kTypeVoid      , { kTypeObject } },
	{  414, kTypeVoid      , { kTypeObject } },
	{  415, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  416, kTypeVoid      , { kTypeObject,kTypeVector } },
	{  417, kTypeVoid      , { kTypeString,kTypeObject,kTypeInt } },
	{  418, kTypeInt       , { kTypeObject } },
	{  419, kTypeObject    , { } },
	{  420, kTypeInt       , { kTypeObject } },
	{  421, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  422, kTypeInt       , { kTypeObject,kTypeInt } },
	{  423, kTypeInt       , { } },
	{  424, kTypeEngineType, { kTypeObject,kTypeEngineType,kTypeObject } },
	{  425, kTypeVoid      , { kTypeObject } },
	{  426, kTypeVoid      , { kTypeObject } },
	{  427, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  428, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  429, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  430, kTypeVoid      , { kTypeObject } },
	{  431, kTypeVoid      , { kTypeObject } },
	{  432, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  433, kTypeVoid      , { kTypeObject } },
	{  434, kTypeVoid      , { kTypeObject } },
	{  435, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  436, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  437, kTypeObject    , { } },
	{  438, kTypeObject    , { } },
	{  439, kTypeObject    , { } },
	{  440, kTypeObject    , { } },
	{  441, kTypeEngineType, { } },
	{  442, kTypeObject    , { } },
	{  443, kTypeInt       , { kTypeObject } },
	{  444, kTypeVoid      , { kTypeInt,kTypeObject,kTypeInt,kTypeInt } },
	{  445, kTypeInt       , { kTypeObject } },
	{  446, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  447, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  448, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  449, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  450, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  451, kTypeEngineType, { kTypeInt } },
	{  452, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  453, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  454, kTypeEngineType, { kTypeInt } },
	{  455, kTypeInt       , { kTypeObject } },
	{  456, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  457, kTypeEngineType, { kTypeInt } },
	{  458, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  459, kTypeEngineType, { } },
	{  460, kTypeEngineType, { kTypeInt,kTypeScriptState } },
	{  461, kTypeEngineType, { } },
	{  462, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  463, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  464, kTypeEngineType, { kTypeInt } },
	{  465, kTypeEngineType, { } },
	{  466, kTypeEngineType, { } },
	{  467, kTypeEngineType, { } },
	{  468, kTypeEngineType, { } },
	{  469, kTypeInt       , { kTypeObject,kTypeObject } },
	{  470, kTypeInt       , { kTypeObject,kTypeObject } },
	{  471, kTypeInt       , { kTypeObject,kTypeObject } },
	{  472, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  473, kTypeEngineType, { kTypeInt,kTypeScriptState } },
	{  474, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeString,kTypeInt } },
	{  475, kTypeInt       , { kTypeObject } },
	{  476, kTypeVoid      , { } },
	{  477, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  478, kTypeInt       , { kTypeObject } },
	{  479, kTypeInt       , { kTypeObject } },
	{  480, kTypeEngineType, { kTypeEngineType,kTypeInt } },
	{  481, kTypeEngineType, { kTypeInt } },
	{  482, kTypeEngineType, { kTypeInt } },
	{  483, kTypeVoid      , { kTypeObject } },
	{  484, kTypeVoid      , { kTypeObject } },
	{  485, kTypeEngineType, { kTypeInt } },
	{  486, kTypeObject    , { kTypeObject } },
	{  487, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  488, kTypeObject    , { kTypeObject,kTypeInt } },
	{  489, kTypeString    , { kTypeObject } },
	{  490, kTypeInt       , { kTypeObject } },
	{  491, kTypeInt       , { kTypeObject } },
	{  492, kTypeInt       , { kTypeObject } },
	{  493, kTypeInt       , { kTypeObject } },
	{  494, kTypeFloat     , { kTypeObject } },
	{  495, kTypeInt       , { kTypeObject } },
	{  496, kTypeInt       , { kTypeObject } },
	{  497, kTypeInt       , { kTypeObject } },
	{  498, kTypeInt       , { kTypeObject } },
	{  499, kTypeString    , { kTypeObject } },
	{  500, kTypeString    , { kTypeObject } },
	{  501, kTypeVoid      , { kTypeInt,kTypeObject,kTypeInt } },
	{  502, kTypeVoid      , { kTypeInt,kTypeEngineType,kTypeInt } },
	{  503, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  504, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  505, kTypeInt       , { kTypeObject } },
	{  506, kTypeObject    , { } },
	{  507, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  508, kTypeInt       , { } },
	{  509, kTypeVoid      , { kTypeString,kTypeString } },
	{  510, kTypeEngineType, { kTypeInt,kTypeString,kTypeString,kTypeString,kTypeString } },
	{  511, kTypeInt       , { kTypeObject } },
	{  512, kTypeVoid      , { } },
	{  513, kTypeInt       , { } },
	{  514, kTypeVoid      , { kTypeEngineType,kTypeInt,kTypeInt } },
	{  515, kTypeVoid      , { kTypeEngineType,kTypeInt,kTypeInt } },
	{  516, kTypeVoid      , { kTypeObject } },
	{  517, kTypeInt       , { kTypeEngineType } },
	{  518, kTypeInt       , { kTypeEngineType } },
	{  519, kTypeInt       , { kTypeEngineType } },
	{  520, kTypeInt       , { kTypeEngineType } },
	{  521, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  522, kTypeInt       , { kTypeObject } },
	{  523, kTypeVoid      , { kTypeInt,kTypeInt,kTypeObject } },
	{  524, kTypeInt       , { kTypeInt,kTypeObject } },
	{  525, kTypeVoid      , { kTypeInt,kTypeObject,kTypeInt,kTypeFloat,kTypeInt,kTypeInt,kTypeFloat,kTypeVector } },
	{  526, kTypeVoid      , { kTypeString,kTypeObject,kTypeInt,kTypeFloat,kTypeInt,kTypeInt,kTypeFloat,kTypeVector } },
	{  527, kTypeInt       , { kTypeObject } },
	{  528, kTypeInt       , { kTypeObject } },
	{  529, kTypeInt       , { kTypeObject,kTypeObject } },
	{  530, kTypeInt       , { kTypeObject } },
	{  531, kTypeInt       , { kTypeObject } },
	{  532, kTypeInt       , { kTypeObject } },
	{  533, kTypeObject    , { kTypeObject } },
	{  534, kTypeString    , { kTypeObject } },
	{  535, kTypeInt       , { kTypeObject } },
	{  536, kTypeInt       , { kTypeObject } },
	{  537, kTypeInt       , { kTypeObject } },
	{  538, kTypeString    , { kTypeObject } },
	{  539, kTypeInt       , { kTypeObject } },
	{  540, kTypeInt       , { kTypeObject } },
	{  541, kTypeInt       , { kTypeObject } },
	{  542, kTypeObject    , { } },
	{  543, kTypeInt       , { kTypeInt,kTypeObject } },
	{  544, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  545, kTypeInt       , { kTypeObject } },
	{  546, kTypeInt       , { kTypeObject,kTypeInt } },
	{  547, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  548, kTypeObject    , { kTypeInt } },
	{  549, kTypeObject    , { kTypeInt } },
	{  550, kTypeInt       , { kTypeObject,kTypeObject } },
	{  551, kTypeInt       , { kTypeObject } },
	{  552, kTypeEngineType, { kTypeInt } },
	{  553, kTypeEngineType, { kTypeInt } },
	{  554, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt,kTypeInt,kTypeString } },
	{  555, kTypeVoid      , { kTypeObject } },
	{  556, kTypeObject    , { kTypeObject } },
	{  557, kTypeVoid      , { } },
	{  558, kTypeInt       , { kTypeObject } },
	{  559, kTypeInt       , { kTypeObject } },
	{  560, kTypeVoid      , { kTypeString } },
	{  561, kTypeString    , { } },
	{  562, kTypeObject    , { kTypeObject } },
	{  563, kTypeVoid      , { kTypeString } },
	{  564, kTypeVoid      , { kTypeString } },
	{  565, kTypeVoid      , { kTypeObject } },
	{  566, kTypeVoid      , { kTypeObject } },
	{  567, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  568, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  569, kTypeInt       , { kTypeObject } },
	{  570, kTypeInt       , { kTypeObject } },
	{  571, kTypeFloat     , { kTypeInt } },
	{  572, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  573, kTypeVoid      , { kTypeObject } },
	{  574, kTypeInt       , { kTypeObject } },
	{  575, kTypeInt       , { kTypeObject } },
	{  576, kTypeInt       , { kTypeObject } },
	{  577, kTypeInt       , { kTypeObject } },
	{  578, kTypeVoid      , { } },
	{  579, kTypeInt       , { } },
	{  580, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  581, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  582, kTypeString    , { kTypeObject } },
	{  583, kTypeEngineType, { } },
	{  584, kTypeObject    , { kTypeObject,kTypeObject,kTypeInt } },
	{  585, kTypeEngineType, { } },
	{  586, kTypeInt       , { kTypeObject } },
	{  587, kTypeInt       , { kTypeObject } },
	{  588, kTypeInt       , { kTypeObject } },
	{  589, kTypeVoid      , { kTypeString,kTypeString,kTypeFloat,kTypeObject } },
	{  590, kTypeVoid      , { kTypeString,kTypeString,kTypeInt,kTypeObject } },
	{  591, kTypeVoid      , { kTypeString,kTypeString,kTypeVector,kTypeObject } },
	{  592, kTypeVoid      , { kTypeString,kTypeString,kTypeEngineType,kTypeObject } },
	{  593, kTypeVoid      , { kTypeString,kTypeString,kTypeString,kTypeObject } },
	{  594, kTypeVoid      , { kTypeString } },
	{  595, kTypeFloat     , { kTypeString,kTypeString,kTypeObject } },
	{  596, kTypeInt       , { kTypeString,kTypeString,kTypeObject } },
	{  597, kTypeVector    , { kTypeString,kTypeString,kTypeObject } },
	{  598, kTypeEngineType, { kTypeString,kTypeString,kTypeObject } },
	{  599, kTypeString    , { kTypeString,kTypeString,kTypeObject } },
	{  600, kTypeObject    , { kTypeObject,kTypeEngineType,kTypeObject,kTypeString } },
	{  601, kTypeVoid      , { kTypeString,kTypeString,kTypeObject } },
	{  602, kTypeInt       , { kTypeString,kTypeString,kTypeObject,kTypeObject } },
	{  603, kTypeObject    , { kTypeString,kTypeString,kTypeEngineType,kTypeObject,kTypeObject } },
	{  604, kTypeEngineType, { } },
	{  605, kTypeInt       , { kTypeObject } },
	{  606, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  607, kTypeInt       , { kTypeObject } },
	{  608, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  609, kTypeVoid      , { kTypeInt,kTypeEngineType,kTypeObject,kTypeFloat } },
	{  610, kTypeVoid      , { kTypeObject,kTypeEngineType } },
	{  611, kTypeInt       , { kTypeEngineType } },
	{  612, kTypeEngineType, { kTypeObject } },
	{  613, kTypeEngineType, { kTypeObject } },
	{  614, kTypeInt       , { kTypeEngineType } },
	{  615, kTypeInt       , { kTypeEngineType } },
	{  616, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  617, kTypeEngineType, { kTypeInt } },
	{  618, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  619, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  620, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  621, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  622, kTypeEngineType, { kTypeInt } },
	{  623, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  624, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  625, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  626, kTypeEngineType, { kTypeInt } },
	{  627, kTypeEngineType, { kTypeInt } },
	{  628, kTypeEngineType, { kTypeInt } },
	{  629, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  630, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  631, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  632, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  633, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  634, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  635, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  636, kTypeEngineType, { kTypeInt } },
	{  637, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  638, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  639, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  640, kTypeEngineType, { } },
	{  641, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  642, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  643, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  644, kTypeEngineType, { kTypeInt } },
	{  645, kTypeEngineType, { kTypeInt } },
	{  646, kTypeEngineType, { kTypeInt } },
	{  647, kTypeEngineType, { } },
	{  648, kTypeEngineType, { } },
	{  649, kTypeEngineType, { kTypeInt } },
	{  650, kTypeEngineType, { } },
	{  651, kTypeEngineType, { kTypeInt } },
	{  652, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  653, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  654, kTypeEngineType, { } },
	{  655, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  656, kTypeEngineType, { kTypeInt } },
	{  657, kTypeEngineType, { } },
	{  658, kTypeEngineType, { kTypeInt,kTypeInt,kTypeInt } },
	{  659, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  660, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  661, kTypeEngineType, { kTypeInt } },
	{  662, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  663, kTypeEngineType, { kTypeInt } },
	{  664, kTypeEngineType, { kTypeInt } },
	{  665, kTypeEngineType, { kTypeInt } },
	{  666, kTypeEngineType, { kTypeInt } },
	{  667, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  668, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  669, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  670, kTypeEngineType, { kTypeInt } },
	{  671, kTypeEngineType, { kTypeInt } },
	{  672, kTypeEngineType, { kTypeInt } },
	{  673, kTypeEngineType, { kTypeInt } },
	{  674, kTypeEngineType, { kTypeInt } },
	{  675, kTypeEngineType, { kTypeInt } },
	{  676, kTypeEngineType, { kTypeInt } },
	{  677, kTypeEngineType, { kTypeInt } },
	{  678, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  679, kTypeEngineType, { } },
	{  680, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  681, kTypeEngineType, { kTypeInt } },
	{  682, kTypeEngineType, { kTypeInt } },
	{  683, kTypeEngineType, { } },
	{  684, kTypeEngineType, { kTypeInt } },
	{  685, kTypeEngineType, { kTypeInt } },
	{  686, kTypeEngineType, { kTypeInt } },
	{  687, kTypeEngineType, { kTypeInt } },
	{  688, kTypeEngineType, { kTypeInt } },
	{  689, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt,kTypeInt } },
	{  690, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  691, kTypeVoid      , { kTypeInt,kTypeInt } },
	{  692, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  693, kTypeObject    , { } },
	{  694, kTypeFloat     , { kTypeInt } },
	{  695, kTypeVoid      , { kTypeObject,kTypeFloat } },
	{  696, kTypeVoid      , { kTypeObject,kTypeFloat,kTypeFloat,kTypeInt } },
	{  697, kTypeVoid      , { kTypeObject } },
	{  698, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  699, kTypeInt       , { kTypeObject } },
	{  700, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  701, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  702, kTypeVoid      , { } },
	{  703, kTypeVoid      , { } },
	{  704, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt,kTypeInt } },
	{  705, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  706, kTypeInt       , { kTypeObject } },
	{  707, kTypeObject    , { } },
	{  708, kTypeInt       , { kTypeObject } },
	{  709, kTypeVoid      , { kTypeInt,kTypeInt } },
	{  710, kTypeString    , { kTypeString,kTypeString,kTypeInt } },
	{  711, kTypeEngineType, { } },
	{  712, kTypeInt       , { kTypeObject } },
	{  713, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  714, kTypeInt       , { kTypeObject } },
	{  715, kTypeVoid      , { kTypeObject } },
	{  716, kTypeInt       , { kTypeObject } },
	{  717, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  718, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  719, kTypeVoid      , { kTypeObject } },
	{  720, kTypeVoid      , { kTypeInt,kTypeInt } },
	{  721, kTypeVoid      , { kTypeObject,kTypeString } },
	{  722, kTypeVoid      , { kTypeObject,kTypeString } },
	{  723, kTypeInt       , { kTypeObject } },
	{  724, kTypeInt       , { kTypeObject,kTypeInt } },
	{  725, kTypeInt       , { kTypeObject } },
	{  726, kTypeInt       , { kTypeObject } },
	{  727, kTypeObject    , { } },
	{  728, kTypeObject    , { } },
	{  729, kTypeObject    , { } },
	{  730, kTypeObject    , { } },
	{  731, kTypeObject    , { kTypeObject,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  732, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt } },
	{  733, kTypeEngineType, { kTypeInt,kTypeInt } },
	{  734, kTypeInt       , { kTypeEngineType } },
	{  735, kTypeInt       , { kTypeObject,kTypeInt } },
	{  736, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  737, kTypeInt       , { kTypeObject } },
	{  738, kTypeVoid      , { kTypeObject } },
	{  739, kTypeEngineType, { kTypeInt } },
	{  740, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  741, kTypeInt       , { kTypeObject } },
	{  742, kTypeFloat     , { kTypeObject } },
	{  743, kTypeVoid      , { kTypeObject,kTypeFloat } },
	{  744, kTypeInt       , { kTypeObject } },
	{  745, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  746, kTypeVoid      , { kTypeInt } },
	{  747, kTypeInt       , { } },
	{  748, kTypeInt       , { kTypeObject } },
	{  749, kTypeInt       , { kTypeObject } },
	{  750, kTypeVoid      , { kTypeObject,kTypeFloat } },
	{  751, kTypeVoid      , { kTypeObject,kTypeFloat } },
	{  752, kTypeInt       , { kTypeObject,kTypeObject } },
	{  753, kTypeInt       , { kTypeVector,kTypeVector } },
	{  754, kTypeInt       , { } },
	{  755, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  756, kTypeVoid      , { kTypeObject } },
	{  757, kTypeEngineType, { } },
	{  758, kTypeEngineType, { kTypeInt } },
	{  759, kTypeInt       , { kTypeObject } },
	{  760, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  761, kTypeInt       , { kTypeObject } },
	{  762, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  763, kTypeInt       , { kTypeObject } },
	{  764, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  765, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  766, kTypeInt       , { kTypeObject } },
	{  767, kTypeEngineType, { } },
	{  768, kTypeInt       , { kTypeObject,kTypeObject } },
	{  769, kTypeInt       , { kTypeEngineType } },
	{  770, kTypeInt       , { kTypeEngineType } },
	{  771, kTypeInt       , { kTypeEngineType } },
	{  772, kTypeInt       , { kTypeEngineType } },
	{  773, kTypeInt       , { kTypeObject } },
	{  774, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  775, kTypeVoid      , { kTypeObject } },
	{  776, kTypeVoid      , { kTypeObject,kTypeFloat } },
	{  777, kTypeInt       , { kTypeString,kTypeInt } },
	{  778, kTypeInt       , { kTypeString,kTypeInt } },
	{  779, kTypeInt       , { kTypeString,kTypeString } },
	{  780, kTypeInt       , { kTypeString,kTypeFloat } },
	{  781, kTypeInt       , { kTypeString } },
	{  782, kTypeInt       , { kTypeString } },
	{  783, kTypeString    , { kTypeString } },
	{  784, kTypeFloat     , { kTypeString } },
	{  785, kTypeInt       , { kTypeString } },
	{  786, kTypeInt       , { kTypeString } },
	{  787, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  788, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  789, kTypeInt       , { kTypeString,kTypeObject } },
	{  790, kTypeEngineType, { } },
	{  791, kTypeEngineType, { kTypeString } },
	{  792, kTypeEngineType, { kTypeString,kTypeObject,kTypeVector } },
	{  793, kTypeInt       , { kTypeInt } },
	{  794, kTypeVoid      , { kTypeObject,kTypeString } },
	{  795, kTypeVoid      , { kTypeInt,kTypeInt } },
	{  796, kTypeVoid      , { kTypeObject,kTypeScriptState } },
	{  797, kTypeInt       , { kTypeObject } },
	{  798, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  799, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  800, kTypeEngineType, { } },
	{  801, kTypeEngineType, { } },
	{  802, kTypeEngineType, { kTypeFloat,kTypeFloat,kTypeFloat } },
	{  803, kTypeEngineType, { kTypeObject,kTypeInt,kTypeInt } },
	{  804, kTypeEngineType, { kTypeObject } },
	{  805, kTypeEngineType, { } },
	{  806, kTypeInt       , { } },
	{  807, kTypeEngineType, { kTypeInt,kTypeFloat,kTypeInt,kTypeInt } },
	{  808, kTypeEngineType, { kTypeInt } },
	{  809, kTypeEngineType, { kTypeInt } },
	{  810, kTypeEngineType, { kTypeInt,kTypeFloat } },
	{  811, kTypeInt       , { } },
	{  812, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt,kTypeFloat,kTypeFloat,kTypeFloat } },
	{  813, kTypeEngineType, { } },
	{  814, kTypeVoid      , { kTypeObject,kTypeString,kTypeInt,kTypeInt,kTypeFloat,kTypeInt } },
	{  815, kTypeInt       , { kTypeObject } },
	{  816, kTypeEngineType, { kTypeObject,kTypeInt } },
	{  817, kTypeEngineType, { kTypeObject } },
	{  818, kTypeEngineType, { kTypeObject,kTypeInt } },
	{  819, kTypeEngineType, { kTypeInt } },
	{  820, kTypeObject    , { } },
	{  821, kTypeObject    , { } },
	{  822, kTypeInt       , { kTypeString,kTypeString } },
	{  823, kTypeInt       , { kTypeString,kTypeObject } },
	{  824, kTypeInt       , { kTypeString } },
	{  825, kTypeInt       , { kTypeString } },
	{  826, kTypeInt       , { kTypeString } },
	{  827, kTypeInt       , { kTypeString,kTypeInt } },
	{  828, kTypeObject    , { kTypeString } },
	{  829, kTypeString    , { kTypeObject } },
	{  830, kTypeObject    , { kTypeString,kTypeEngineType } },
	{  831, kTypeInt       , { kTypeString } },
	{  832, kTypeInt       , { kTypeString,kTypeObject } },
	{  833, kTypeVoid      , { kTypeString,kTypeObject,kTypeInt } },
	{  834, kTypeString    , { } },
	{  835, kTypeString    , { } },
	{  836, kTypeVoid      , { kTypeObject,kTypeObject,kTypeEngineType,kTypeEngineType,kTypeInt,kTypeInt } },
	{  837, kTypeVoid      , { kTypeObject,kTypeObject,kTypeEngineType,kTypeEngineType,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  838, kTypeInt       , { kTypeObject } },
	{  839, kTypeObject    , { kTypeObject,kTypeObject } },
	{  840, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  841, kTypeFloat     , { kTypeEngineType,kTypeEngineType,kTypeInt,kTypeInt } },
	{  842, kTypeVoid      , { kTypeInt } },
	{  843, kTypeInt       , { } },
	{  844, kTypeInt       , { kTypeString,kTypeInt } },
	{  845, kTypeInt       , { kTypeString } },
	{  846, kTypeInt       , { kTypeObject } },
	{  847, kTypeVoid      , { kTypeString,kTypeObject,kTypeString } },
	{  848, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt,kTypeFloat } },
	{  849, kTypeInt       , { } },
	{  850, kTypeVoid      , { kTypeObject,kTypeString,kTypeInt,kTypeString,kTypeInt } },
	{  851, kTypeEngineType, { kTypeFloat,kTypeScriptState } },
	{  852, kTypeVoid      , { kTypeString,kTypeString } },
	{  853, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  854, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  855, kTypeVoid      , { kTypeObject,kTypeInt,kTypeString } },
	{  856, kTypeString    , { kTypeObject,kTypeInt } },
	{  857, kTypeInt       , { kTypeObject } },
	{  858, kTypeVoid      , { kTypeObject,kTypeObject } },
	{  859, kTypeInt       , { kTypeObject } },
	{  860, kTypeVoid      , { kTypeObject,kTypeInt,kTypeString,kTypeString,kTypeString,kTypeInt,kTypeString,kTypeInt,kTypeString,kTypeInt,kTypeString } },
	{  861, kTypeInt       , { kTypeString,kTypeString,kTypeInt } },
	{  862, kTypeInt       , { kTypeString } },
	{  863, kTypeObject    , { kTypeObject } },
	{  864, kTypeObject    , { kTypeObject } },
	{  865, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  866, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  867, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  868, kTypeInt       , { kTypeObject } },
	{  869, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  870, kTypeInt       , { kTypeObject } },
	{  871, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  872, kTypeVoid      , { kTypeObject } },
	{  873, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{  874, kTypeEngineType, { kTypeEngineType,kTypeEngineType,kTypeFloat,kTypeFloat,kTypeInt } },
	{  875, kTypeEngineType, { kTypeObject,kTypeEngineType,kTypeFloat,kTypeInt,kTypeInt } },
	{  876, kTypeInt       , { kTypeObject,kTypeInt } },
	{  877, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt,kTypeInt } },
	{  878, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt,kTypeFloat,kTypeFloat } },
	{  879, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  880, kTypeEngineType, { kTypeInt } },
	{  881, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeInt } },
	{  882, kTypeVoid      , { kTypeObject,kTypeString } },
	{  883, kTypeInt       , { } },
	{  884, kTypeInt       , { kTypeObject } },
	{  885, kTypeString    , { } },
	{  886, kTypeVoid      , { kTypeObject,kTypeString,kTypeInt,kTypeString } },
	{  887, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeInt } },
	{  888, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeInt,kTypeString } },
	{  889, kTypeInt       , { kTypeObject } },
	{  890, kTypeEngineType, { kTypeInt } },
	{  891, kTypeEngineType, { } },
	{  892, kTypeInt       , { kTypeEngineType,kTypeInt } },
	{  893, kTypeVoid      , { kTypeObject,kTypeInt,kTypeFloat } },
	{  894, kTypeEngineType, { kTypeEngineType,kTypeInt } },
	{  895, kTypeEngineType, { kTypeInt } },
	{  896, kTypeInt       , { kTypeObject } },
	{  897, kTypeVoid      , { kTypeObject,kTypeString } },
	{  898, kTypeVoid      , { kTypeObject,kTypeString } },
	{  899, kTypeVoid      , { kTypeObject,kTypeString } },
	{  900, kTypeString    , { kTypeObject } },
	{  901, kTypeString    , { kTypeObject } },
	{  902, kTypeString    , { kTypeObject } },
	{  903, kTypeInt       , { kTypeObject } },
	{  904, kTypeInt       , { kTypeObject,kTypeString,kTypeInt,kTypeFloat } },
	{  905, kTypeEngineType, { } },
	{  906, kTypeVoid      , { kTypeObject,kTypeInt,kTypeString,kTypeString,kTypeString,kTypeInt,kTypeString,kTypeInt,kTypeString,kTypeInt,kTypeString,kTypeString,kTypeString } },
	{  907, kTypeInt       , { kTypeObject,kTypeInt,kTypeInt } },
	{  908, kTypeVoid      , { kTypeObject,kTypeVector,kTypeInt } },
	{  909, kTypeInt       , { kTypeObject } },
	{  910, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  911, kTypeInt       , { } },
	{  912, kTypeObject    , { } },
	{  913, kTypeEngineType, { kTypeInt } },
	{  914, kTypeVoid      , { kTypeObject,kTypeInt,kTypeObject } },
	{  915, kTypeObject    , { } },
	{  916, kTypeObject    , { } },
	{  917, kTypeInt       , { kTypeObject } },
	{  918, kTypeInt       , { kTypeObject } },
	{  919, kTypeObject    , { kTypeObject } },
	{  920, kTypeEngineType, { } },
	{  921, kTypeInt       , { kTypeObject } },
	{  922, kTypeVoid      , { kTypeString } },
	{  923, kTypeVoid      , { kTypeObject } },
	{  924, kTypeObject    , { kTypeObject } },
	{  925, kTypeVoid      , { kTypeString } },
	{  926, kTypeInt       , { kTypeString,kTypeString,kTypeString,kTypeInt } },
	{  927, kTypeFloat     , { kTypeString,kTypeString,kTypeString,kTypeInt } },
	{  928, kTypeString    , { kTypeString,kTypeString,kTypeString,kTypeInt } },
	{  929, kTypeVoid      , { kTypeString,kTypeString,kTypeString,kTypeInt,kTypeInt } },
	{  930, kTypeVoid      , { kTypeString,kTypeString,kTypeString,kTypeInt,kTypeFloat } },
	{  931, kTypeVoid      , { kTypeString,kTypeString,kTypeString,kTypeInt,kTypeString } },
	{  932, kTypeEngineType, { kTypeInt } },
	{  933, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeFloat } },
	{  934, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeString } },
	{  935, kTypeEngineType, { kTypeInt } },
	{  936, kTypeObject    , { kTypeInt } },
	{  937, kTypeInt       , { kTypeObject } },
	{  938, kTypeObject    , { kTypeString } },
	{  939, kTypeInt       , { kTypeObject } },
	{  940, kTypeEngineType, { } },
	{  941, kTypeEngineType, { } },
	{  942, kTypeEngineType, { } },
	{  943, kTypeInt       , { kTypeObject } },
	{  944, kTypeString    , { kTypeObject } },
	{  945, kTypeVoid      , { kTypeObject,kTypeString } },
	{  946, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  947, kTypeInt       , { kTypeObject } },
	{  948, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  949, kTypeInt       , { kTypeObject } },
	{  950, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  951, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  952, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  953, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  954, kTypeVoid      , { kTypeObject,kTypeString } },
	{  955, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  956, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  957, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  958, kTypeVoid      , { kTypeInt,kTypeObject } },
	{  959, kTypeInt       , { kTypeObject } },
	{  960, kTypeInt       , { } },
	{  961, kTypeVoid      , { kTypeInt } },
	{  962, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  963, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  964, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  965, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  966, kTypeVoid      , { kTypeObject,kTypeString } },
	{  967, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  968, kTypeObject    , { kTypeInt,kTypeEngineType,kTypeFloat,kTypeString,kTypeInt,kTypeString,kTypeString } },
	{  969, kTypeVoid      , { kTypeInt,kTypeObject,kTypeInt,kTypeString,kTypeString } },
	{  970, kTypeInt       , { kTypeInt,kTypeObject } },
	{  971, kTypeInt       , { kTypeObject } },
	{  972, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  973, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  974, kTypeObject    , { } },
	{  975, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  976, kTypeEngineType, { } },
	{  977, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  978, kTypeString    , { } },
	{  979, kTypeVoid      , { kTypeObject,kTypeString } },
	{  980, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  981, kTypeEngineType, { kTypeObject,kTypeInt,kTypeFloat,kTypeString,kTypeInt,kTypeString } },
	{  982, kTypeInt       , { kTypeObject } },
	{  983, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  984, kTypeVoid      , { kTypeObject,kTypeFloat,kTypeFloat,kTypeFloat } },
	{  985, kTypeFloat     , { kTypeObject,kTypeInt } },
	{  986, kTypeInt       , { kTypeString } },
	{  987, kTypeInt       , { kTypeString } },
	{  988, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  989, kTypeInt       , { kTypeObject } },
	{  990, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{  991, kTypeVoid      , { kTypeObject,kTypeString,kTypeString } },
	{  992, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeString,kTypeString,kTypeString,kTypeString,kTypeString } },
	{  993, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeString } },
	{  994, kTypeInt       , { kTypeObject } },
	{  995, kTypeVoid      , { kTypeObject,kTypeInt } },
	{  996, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeString,kTypeString,kTypeString,kTypeString,kTypeString } },
	{  997, kTypeVoid      , { kTypeObject } },
	{  998, kTypeObject    , { kTypeObject,kTypeVector } },
	{  999, kTypeObject    , { kTypeObject } },
	{ 1000, kTypeFloat     , { kTypeObject } },
	{ 1001, kTypeVoid      , { kTypeObject,kTypeFloat } },
	{ 1002, kTypeString    , { kTypeObject } },
	{ 1003, kTypeInt       , { kTypeObject } },
	{ 1004, kTypeVoid      , { kTypeObject,kTypeInt } },
	{ 1005, kTypeInt       , { kTypeObject } },
	{ 1006, kTypeString    , { kTypeObject,kTypeInt } },
	{ 1007, kTypeInt       , { kTypeObject,kTypeInt } },
	{ 1008, kTypeInt       , { kTypeObject } },
	{ 1009, kTypeInt       , { kTypeObject } },
	{ 1010, kTypeString    , { } },
	{ 1011, kTypeString    , { } },
	{ 1012, kTypeInt       , { kTypeObject } },
	{ 1013, kTypeInt       , { kTypeObject } },
	{ 1014, kTypeInt       , { kTypeString } },
	{ 1015, kTypeInt       , { } },
	{ 1016, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeString } },
	{ 1017, kTypeInt       , { } },
	{ 1018, kTypeVoid      , { kTypeObject,kTypeString,kTypeString,kTypeInt } },
	{ 1019, kTypeVoid      , { kTypeInt } },
	{ 1020, kTypeInt       , { } },
	{ 1021, kTypeInt       , { kTypeObject } },
	{ 1022, kTypeVoid      , { kTypeObject,kTypeInt,kTypeString } },
	{ 1023, kTypeObject    , { kTypeObject } },
	{ 1024, kTypeInt       , { kTypeObject,kTypeInt } },
	{ 1025, kTypeString    , { kTypeObject,kTypeInt } },
	{ 1026, kTypeFloat     , { kTypeObject,kTypeInt } },
	{ 1027, kTypeEngineType, { kTypeObject,kTypeInt } },
	{ 1028, kTypeObject    , { kTypeObject,kTypeInt } },
	{ 1029, kTypeInt       , { kTypeObject } },
	{ 1030, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt } },
	{ 1031, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt,kTypeInt } },
	{ 1032, kTypeVoid      , { kTypeObject,kTypeObject,kTypeInt,kTypeString,kTypeInt } },
	{ 1033, kTypeInt       , { kTypeEngineType } },
	{ 1034, kTypeInt       , { kTypeEngineType } },
	{ 1035, kTypeInt       , { kTypeObject,kTypeInt } },
	{ 1036, kTypeInt       , { kTypeObject } },
	{ 1037, kTypeVoid      , { kTypeObject,kTypeInt } },
	{ 1038, kTypeVoid      , { kTypeObject,kTypeInt,kTypeInt,kTypeInt,kTypeInt } },
	{ 1039, kTypeInt       , { } },
	{ 1040, kTypeObject    , { kTypeInt } },
	{ 1041, kTypeVoid      , { kTypeObject } },
	{ 1042, kTypeVoid      , { kTypeInt } },
	{ 1043, kTypeVoid      , { kTypeString } },
	{ 1044, kTypeVoid      , { kTypeFloat } },
	{ 1045, kTypeVoid      , { kTypeObject } },
	{ 1046, kTypeInt       , { kTypeString,kTypeObject,kTypeInt } },
	{ 1047, kTypeVoid      , { } },
	{ 1048, kTypeVoid      , { kTypeObject,kTypeInt } },
	{ 1049, kTypeInt       , { kTypeObject } },
	{ 1050, kTypeInt       , { kTypeObject,kTypeInt } },
	{ 1051, kTypeVoid      , { kTypeObject,kTypeInt } },
	{ 1052, kTypeVoid      , { kTypeObject,kTypeString } },
	{ 1053, kTypeInt       , { kTypeObject } },
	{ 1054, kTypeVoid      , { kTypeObject,kTypeInt } },
	{ 1055, kTypeVoid      , { kTypeObject,kTypeInt } },
	{ 1056, kTypeObject    , { kTypeString,kTypeInt,kTypeInt } },
	{ 1057, kTypeVoid      , { kTypeObject,kTypeEngineType } },
	{ 1058, kTypeInt       , { kTypeString,kTypeInt } }
};

/** The table defining the default values for the parameters of each engine function. */
const Functions::FunctionDefaults Functions::kFunctionDefaults[] = {
	{    0, { } },
	{    1, { } },
	{    2, { &kDefaultInt18, &kDefaultInt9 } },
	{    3, { &kDefaultInt18, &kDefaultInt9 } },
	{    4, { } },
	{    5, { } },
	{    6, { } },
	{    7, { } },
	{    8, { } },
	{    9, { &kDefaultFalse } },
	{   10, { &kDefaultFalse } },
	{   11, { } },
	{   12, { } },
	{   13, { } },
	{   14, { } },
	{   15, { } },
	{   16, { } },
	{   17, { } },
	{   18, { } },
	{   19, { } },
	{   20, { } },
	{   21, { &kDefaultFalse } },
	{   22, { &kDefaultFalse, &kDefaultFloat1_0 } },
	{   23, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{   24, { } },
	{   25, { } },
	{   26, { } },
	{   27, { } },
	{   28, { } },
	{   29, { } },
	{   30, { } },
	{   31, { &kDefaultObjectSelf, &kDefaultInt1, &kDefaultStringEmpty, &kDefaultInt1 } },
	{   32, { } },
	{   33, { } },
	{   34, { } },
	{   35, { } },
	{   36, { &kDefaultObjectSelf } },
	{   37, { &kDefaultFalse } },
	{   38, { &kDefaultObjectSelf, &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{   39, { &kDefaultTalkVolumeTalk } },
	{   40, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{   41, { } },
	{   42, { } },
	{   43, { } },
	{   44, { } },
	{   45, { &kDefaultFloatMinus1_0, &kDefaultFloatMinus1_0, &kDefaultCameraTransitionTypeSnap } },
	{   46, { &kDefaultFalse } },
	{   47, { } },
	{   48, { &kDefaultMetaMagicAny, &kDefaultFalse, &kDefaultInt0, &kDefaultProjectilePathTypeDefault, &kDefaultFalse } },
	{   49, { &kDefaultObjectSelf } },
	{   50, { &kDefaultObjectSelf } },
	{   51, { } },
	{   52, { } },
	{   53, { } },
	{   54, { } },
	{   55, { } },
	{   56, { } },
	{   57, { } },
	{   58, { } },
	{   59, { } },
	{   60, { } },
	{   61, { } },
	{   62, { } },
	{   63, { } },
	{   64, { } },
	{   65, { } },
	{   66, { &kDefaultInt0 } },
	{   67, { } },
	{   68, { } },
	{   69, { } },
	{   70, { } },
	{   71, { } },
	{   72, { } },
	{   73, { } },
	{   74, { } },
	{   75, { } },
	{   76, { } },
	{   77, { } },
	{   78, { } },
	{   79, { &kDefaultDamageTypeMagical, &kDefaultDamagePowerNormal, &kDefaultFalse } },
	{   80, { } },
	{   81, { &kDefaultInt0 } },
	{   82, { } },
	{   83, { &kDefaultVFXNone, &kDefaultFloat0_0, &kDefaultInt0 } },
	{   84, { } },
	{   85, { } },
	{   86, { } },
	{   87, { } },
	{   88, { } },
	{   89, { } },
	{   90, { } },
	{   91, { } },
	{   92, { } },
	{   93, { &kDefaultObjectInvalid } },
	{   94, { &kDefaultObjectInvalid } },
	{   95, { &kDefaultInt1 } },
	{   96, { &kDefaultInt1 } },
	{   97, { &kDefaultInt1 } },
	{   98, { &kDefaultInt1 } },
	{   99, { &kDefaultInt1 } },
	{  100, { &kDefaultInt1 } },
	{  101, { &kDefaultInt1 } },
	{  102, { &kDefaultInt1 } },
	{  103, { &kDefaultInt1 } },
	{  104, { } },
	{  105, { } },
	{  106, { } },
	{  107, { } },
	{  108, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{  109, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{  110, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{  111, { } },
	{  112, { } },
	{  113, { } },
	{  114, { } },
	{  115, { &kDefaultACDodgeBonus, &kDefaultACVsDamageTypeAll, &kDefaultFalse } },
	{  116, { &kDefaultInt0 } },
	{  117, { &kDefaultSavingThrowTypeAll, &kDefaultFalse } },
	{  118, { &kDefaultAttackBonusMisc } },
	{  119, { &kDefaultDamagePowerNormal, &kDefaultInt0, &kDefaultDRTypeMagicBonus } },
	{  120, { &kDefaultDamageTypeMagical, &kDefaultIntMinus1 } },
	{  121, { } },
	{  122, { } },
	{  123, { } },
	{  124, { } },
	{  125, { } },
	{  126, { } },
	{  127, { } },
	{  128, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{  129, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{  130, { } },
	{  131, { } },
	{  132, { } },
	{  133, { &kDefaultFalse, &kDefaultTrue, &kDefaultFalse, &kDefaultTrue } },
	{  134, { } },
	{  135, { &kDefaultTrue } },
	{  136, { &kDefaultTrue } },
	{  137, { } },
	{  138, { &kDefaultInt1, &kDefaultInt1, &kDefaultInt1, &kDefaultInt1, &kDefaultInt1, &kDefaultInt1 } },
	{  139, { &kDefaultFalse } },
	{  140, { &kDefaultFalse } },
	{  141, { } },
	{  142, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{  143, { &kDefaultFalse } },
	{  144, { } },
	{  145, { } },
	{  146, { &kDefaultTrue, &kDefaultInt0 } },
	{  147, { &kDefaultTrue, &kDefaultInt0 } },
	{  148, { &kDefaultIntMinus1, &kDefaultSavingThrowWill, &kDefaultTrue } },
	{  149, { &kDefaultSpellAllSpells } },
	{  150, { } },
	{  151, { } },
	{  152, { } },
	{  153, { } },
	{  154, { } },
	{  155, { &kDefaultObjectSelf } },
	{  156, { } },
	{  157, { } },
	{  158, { } },
	{  159, { } },
	{  160, { } },
	{  161, { } },
	{  162, { &kDefaultObjectSelf } },
	{  163, { &kDefaultObjectSelf } },
	{  164, { } },
	{  165, { } },
	{  166, { } },
	{  167, { &kDefaultFloat0_5, &kDefaultInt0 } },
	{  168, { } },
	{  169, { } },
	{  170, { } },
	{  171, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{  172, { &kDefaultObjectSelf } },
	{  173, { } },
	{  174, { } },
	{  175, { } },
	{  176, { &kDefaultInt0 } },
	{  177, { } },
	{  178, { } },
	{  179, { } },
	{  180, { &kDefaultFalse } },
	{  181, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  182, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  183, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  184, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  185, { } },
	{  186, { } },
	{  187, { } },
	{  188, { } },
	{  189, { } },
	{  190, { } },
	{  191, { } },
	{  192, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  193, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  194, { } },
	{  195, { } },
	{  196, { &kDefaultTrue } },
	{  197, { } },
	{  198, { } },
	{  199, { } },
	{  200, { &kDefaultInt0 } },
	{  201, { } },
	{  202, { } },
	{  203, { &kDefaultStringEmpty } },
	{  204, { &kDefaultStringEmpty, &kDefaultFalse, &kDefaultTrue, &kDefaultFalse, &kDefaultFalse } },
	{  205, { } },
	{  206, { } },
	{  207, { &kDefaultFalse } },
	{  208, { } },
	{  209, { } },
	{  210, { } },
	{  211, { } },
	{  212, { &kDefaultIntMinus1 } },
	{  213, { } },
	{  214, { } },
	{  215, { } },
	{  216, { &kDefaultFloat0_0 } },
	{  217, { } },
	{  218, { } },
	{  219, { } },
	{  220, { &kDefaultFloat0_0 } },
	{  221, { &kDefaultTalkVolumeTalk } },
	{  222, { } },
	{  223, { } },
	{  224, { } },
	{  225, { } },
	{  226, { &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{  227, { &kDefaultObjectTypeAll, &kDefaultObjectSelf, &kDefaultInt1 } },
	{  228, { &kDefaultInt1 } },
	{  229, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{  230, { } },
	{  231, { } },
	{  232, { } },
	{  233, { } },
	{  234, { &kDefaultMetaMagicAny, &kDefaultFalse, &kDefaultProjectilePathTypeDefault, &kDefaultFalse, &kDefaultInt0 } },
	{  235, { &kDefaultObjectSelf } },
	{  236, { &kDefaultObjectSelf } },
	{  237, { &kDefaultObjectSelf } },
	{  238, { } },
	{  239, { &kDefaultGenderMale } },
	{  240, { &kDefaultTalkVolumeTalk } },
	{  241, { &kDefaultFloat0_0, &kDefaultTrue } },
	{  242, { } },
	{  243, { &kDefaultFalse, &kDefaultStringEmpty } },
	{  244, { &kDefaultTrue } },
	{  245, { } },
	{  246, { } },
	{  247, { } },
	{  248, { } },
	{  249, { } },
	{  250, { } },
	{  251, { } },
	{  252, { } },
	{  253, { } },
	{  254, { } },
	{  255, { &kDefaultStringEmpty, &kDefaultObjectInvalid, &kDefaultFalse } },
	{  256, { } },
	{  257, { } },
	{  258, { } },
	{  259, { } },
	{  260, { } },
	{  261, { } },
	{  262, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{  263, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{  264, { &kDefaultObjectSelf } },
	{  265, { } },
	{  266, { } },
	{  267, { } },
	{  268, { } },
	{  269, { } },
	{  270, { } },
	{  271, { } },
	{  272, { } },
	{  273, { } },
	{  274, { &kDefaultObjectInvalid } },
	{  275, { } },
	{  276, { &kDefaultObjectSelf } },
	{  277, { &kDefaultObjectSelf } },
	{  278, { &kDefaultObjectSelf } },
	{  279, { &kDefaultObjectSelf } },
	{  280, { &kDefaultObjectSelf } },
	{  281, { &kDefaultObjectSelf } },
	{  282, { } },
	{  283, { } },
	{  284, { } },
	{  285, { &kDefaultObjectSelf, &kDefaultFalse } },
	{  286, { &kDefaultObjectSelf } },
	{  287, { } },
	{  288, { &kDefaultInt0, &kDefaultObjectInvalid } },
	{  289, { &kDefaultObjectSelf } },
	{  290, { &kDefaultObjectSelf } },
	{  291, { } },
	{  292, { } },
	{  293, { } },
	{  294, { } },
	{  295, { } },
	{  296, { &kDefaultObjectSelf } },
	{  297, { &kDefaultObjectSelf } },
	{  298, { } },
	{  299, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{  300, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{  301, { } },
	{  302, { } },
	{  303, { } },
	{  304, { &kDefaultObjectSelf } },
	{  305, { } },
	{  306, { &kDefaultObjectSelf } },
	{  307, { &kDefaultObjectSelf, &kDefaultInt0 } },
	{  308, { &kDefaultObjectSelf, &kDefaultInt0 } },
	{  309, { } },
	{  310, { } },
	{  311, { } },
	{  312, { } },
	{  313, { } },
	{  314, { } },
	{  315, { &kDefaultObjectSelf, &kDefaultFalse } },
	{  316, { &kDefaultObjectSelf } },
	{  317, { &kDefaultObjectSelf } },
	{  318, { &kDefaultObjectSelf } },
	{  319, { &kDefaultObjectSelf } },
	{  320, { &kDefaultObjectSelf } },
	{  321, { &kDefaultObjectSelf } },
	{  322, { &kDefaultTrue } },
	{  323, { &kDefaultTrue, &kDefaultFalse } },
	{  324, { } },
	{  325, { } },
	{  326, { } },
	{  327, { &kDefaultObjectSelf } },
	{  328, { } },
	{  329, { } },
	{  330, { } },
	{  331, { &kDefaultObjectSelf } },
	{  332, { } },
	{  333, { } },
	{  334, { &kDefaultObjectSelf, &kDefaultStringEmpty } },
	{  335, { &kDefaultObjectSelf, &kDefaultStringEmpty } },
	{  336, { } },
	{  337, { } },
	{  338, { } },
	{  339, { &kDefaultObjectSelf } },
	{  340, { &kDefaultObjectSelf } },
	{  341, { &kDefaultObjectSelf } },
	{  342, { &kDefaultObjectSelf } },
	{  343, { &kDefaultObjectSelf } },
	{  344, { } },
	{  345, { } },
	{  346, { &kDefaultObjectSelf } },
	{  347, { } },
	{  348, { } },
	{  349, { } },
	{  350, { } },
	{  351, { } },
	{  352, { } },
	{  353, { } },
	{  354, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{  355, { &kDefaultAlignmentAll, &kDefaultAlignmentAll } },
	{  356, { } },
	{  357, { } },
	{  358, { } },
	{  359, { } },
	{  360, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{  361, { } },
	{  362, { } },
	{  363, { } },
	{  364, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{  365, { &kDefaultObjectSelf } },
	{  366, { &kDefaultObjectSelf } },
	{  367, { &kDefaultTrue, &kDefaultFalse, &kDefaultFalse } },
	{  368, { &kDefaultTrue, &kDefaultFalse } },
	{  369, { } },
	{  370, { } },
	{  371, { } },
	{  372, { } },
	{  373, { } },
	{  374, { } },
	{  375, { } },
	{  376, { } },
	{  377, { &kDefaultObjectSelf } },
	{  378, { &kDefaultInt0, &kDefaultInt0 } },
	{  379, { } },
	{  380, { &kDefaultTrue } },
	{  381, { &kDefaultTrue } },
	{  382, { &kDefaultFalse, &kDefaultFloat30_0 } },
	{  383, { &kDefaultFalse, &kDefaultFloat1_0, &kDefaultFloat30_0 } },
	{  384, { } },
	{  385, { &kDefaultInt1 } },
	{  386, { } },
	{  387, { } },
	{  388, { } },
	{  389, { &kDefaultObjectSelf } },
	{  390, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{  391, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{  392, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{  393, { } },
	{  394, { } },
	{  395, { } },
	{  396, { } },
	{  397, { } },
	{  398, { } },
	{  399, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{  400, { &kDefaultObjectInvalid } },
	{  401, { } },
	{  402, { &kDefaultInt0 } },
	{  403, { &kDefaultTrue } },
	{  404, { } },
	{  405, { } },
	{  406, { } },
	{  407, { } },
	{  408, { } },
	{  409, { &kDefaultObjectSelf } },
	{  410, { } },
	{  411, { } },
	{  412, { } },
	{  413, { } },
	{  414, { } },
	{  415, { } },
	{  416, { } },
	{  417, { &kDefaultStringEmpty, &kDefaultObjectInvalid, &kDefaultTalkVolumeTalk } },
	{  418, { &kDefaultObjectSelf } },
	{  419, { } },
	{  420, { } },
	{  421, { &kDefaultObjectSelf } },
	{  422, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{  423, { } },
	{  424, { &kDefaultObjectInvalid } },
	{  425, { } },
	{  426, { } },
	{  427, { } },
	{  428, { } },
	{  429, { } },
	{  430, { } },
	{  431, { } },
	{  432, { } },
	{  433, { } },
	{  434, { } },
	{  435, { } },
	{  436, { } },
	{  437, { } },
	{  438, { } },
	{  439, { } },
	{  440, { } },
	{  441, { } },
	{  442, { } },
	{  443, { } },
	{  444, { &kDefaultFalse, &kDefaultTrue } },
	{  445, { } },
	{  446, { } },
	{  447, { &kDefaultAttackBonusMisc } },
	{  448, { &kDefaultDamageTypeMagical } },
	{  449, { } },
	{  450, { &kDefaultACDodgeBonus, &kDefaultACVsDamageTypeAll } },
	{  451, { } },
	{  452, { &kDefaultSavingThrowTypeAll } },
	{  453, { } },
	{  454, { } },
	{  455, { &kDefaultObjectSelf } },
	{  456, { } },
	{  457, { } },
	{  458, { &kDefaultMissChanceTypeNormal } },
	{  459, { } },
	{  460, { } },
	{  461, { } },
	{  462, { &kDefaultFalse } },
	{  463, { &kDefaultFalse, &kDefaultFalse } },
	{  464, { } },
	{  465, { } },
	{  466, { } },
	{  467, { } },
	{  468, { } },
	{  469, { &kDefaultObjectSelf } },
	{  470, { &kDefaultObjectSelf } },
	{  471, { &kDefaultObjectSelf } },
	{  472, { &kDefaultInt0, &kDefaultSpellSchoolGeneral } },
	{  473, { } },
	{  474, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultFalse } },
	{  475, { } },
	{  476, { } },
	{  477, { &kDefaultMissChanceTypeNormal } },
	{  478, { &kDefaultObjectSelf } },
	{  479, { } },
	{  480, { &kDefaultInt1 } },
	{  481, { &kDefaultInt1 } },
	{  482, { &kDefaultInt1 } },
	{  483, { } },
	{  484, { } },
	{  485, { } },
	{  486, { &kDefaultObjectSelf } },
	{  487, { } },
	{  488, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  489, { } },
	{  490, { } },
	{  491, { } },
	{  492, { } },
	{  493, { } },
	{  494, { } },
	{  495, { } },
	{  496, { } },
	{  497, { } },
	{  498, { } },
	{  499, { } },
	{  500, { } },
	{  501, { &kDefaultProjectilePathTypeDefault } },
	{  502, { &kDefaultProjectilePathTypeDefault } },
	{  503, { &kDefaultObjectSelf } },
	{  504, { } },
	{  505, { &kDefaultObjectSelf } },
	{  506, { } },
	{  507, { &kDefaultWeatherPowerMedium } },
	{  508, { } },
	{  509, { &kDefaultStringEmpty } },
	{  510, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{  511, { } },
	{  512, { } },
	{  513, { } },
	{  514, { } },
	{  515, { } },
	{  516, { } },
	{  517, { } },
	{  518, { } },
	{  519, { } },
	{  520, { } },
	{  521, { } },
	{  522, { &kDefaultObjectSelf } },
	{  523, { &kDefaultObjectSelf } },
	{  524, { &kDefaultObjectSelf } },
	{  525, { &kDefaultTrue, &kDefaultFloat5_0, &kDefaultInt4294967295, &kDefaultInt4294967295, &kDefaultFloat0_0, &kDefaultVector0 } },
	{  526, { &kDefaultTrue, &kDefaultFloat5_0, &kDefaultInt4294967295, &kDefaultInt4294967295, &kDefaultFloat0_0, &kDefaultVector0 } },
	{  527, { } },
	{  528, { } },
	{  529, { } },
	{  530, { } },
	{  531, { } },
	{  532, { } },
	{  533, { } },
	{  534, { } },
	{  535, { } },
	{  536, { } },
	{  537, { } },
	{  538, { } },
	{  539, { } },
	{  540, { } },
	{  541, { } },
	{  542, { } },
	{  543, { &kDefaultObjectSelf } },
	{  544, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  545, { &kDefaultObjectSelf } },
	{  546, { } },
	{  547, { } },
	{  548, { &kDefaultTrue } },
	{  549, { &kDefaultTrue } },
	{  550, { } },
	{  551, { } },
	{  552, { } },
	{  553, { } },
	{  554, { &kDefaultTrue, &kDefaultTrue, &kDefaultInt0, &kDefaultStringEmpty } },
	{  555, { } },
	{  556, { &kDefaultObjectSelf } },
	{  557, { } },
	{  558, { } },
	{  559, { } },
	{  560, { } },
	{  561, { } },
	{  562, { } },
	{  563, { } },
	{  564, { } },
	{  565, { } },
	{  566, { } },
	{  567, { } },
	{  568, { } },
	{  569, { } },
	{  570, { } },
	{  571, { } },
	{  572, { } },
	{  573, { } },
	{  574, { } },
	{  575, { } },
	{  576, { } },
	{  577, { } },
	{  578, { } },
	{  579, { } },
	{  580, { } },
	{  581, { } },
	{  582, { } },
	{  583, { } },
	{  584, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{  585, { } },
	{  586, { } },
	{  587, { &kDefaultObjectSelf } },
	{  588, { } },
	{  589, { &kDefaultObjectInvalid } },
	{  590, { &kDefaultObjectInvalid } },
	{  591, { &kDefaultObjectInvalid } },
	{  592, { &kDefaultObjectInvalid } },
	{  593, { &kDefaultObjectInvalid } },
	{  594, { } },
	{  595, { &kDefaultObjectInvalid } },
	{  596, { &kDefaultObjectInvalid } },
	{  597, { &kDefaultObjectInvalid } },
	{  598, { &kDefaultObjectInvalid } },
	{  599, { &kDefaultObjectInvalid } },
	{  600, { &kDefaultObjectInvalid, &kDefaultStringEmpty } },
	{  601, { &kDefaultObjectInvalid } },
	{  602, { &kDefaultObjectInvalid } },
	{  603, { &kDefaultObjectInvalid, &kDefaultObjectInvalid } },
	{  604, { } },
	{  605, { } },
	{  606, { &kDefaultTrue } },
	{  607, { } },
	{  608, { } },
	{  609, { &kDefaultFloat0_0 } },
	{  610, { } },
	{  611, { } },
	{  612, { } },
	{  613, { } },
	{  614, { } },
	{  615, { } },
	{  616, { } },
	{  617, { } },
	{  618, { } },
	{  619, { } },
	{  620, { } },
	{  621, { } },
	{  622, { } },
	{  623, { } },
	{  624, { } },
	{  625, { } },
	{  626, { } },
	{  627, { } },
	{  628, { } },
	{  629, { } },
	{  630, { } },
	{  631, { } },
	{  632, { } },
	{  633, { } },
	{  634, { } },
	{  635, { } },
	{  636, { } },
	{  637, { &kDefaultInt0, &kDefaultDRTypeMagicBonus } },
	{  638, { } },
	{  639, { } },
	{  640, { } },
	{  641, { } },
	{  642, { } },
	{  643, { } },
	{  644, { } },
	{  645, { } },
	{  646, { } },
	{  647, { } },
	{  648, { } },
	{  649, { } },
	{  650, { } },
	{  651, { } },
	{  652, { } },
	{  653, { } },
	{  654, { } },
	{  655, { } },
	{  656, { } },
	{  657, { } },
	{  658, { &kDefaultInt0 } },
	{  659, { } },
	{  660, { } },
	{  661, { } },
	{  662, { } },
	{  663, { } },
	{  664, { } },
	{  665, { } },
	{  666, { } },
	{  667, { } },
	{  668, { } },
	{  669, { } },
	{  670, { } },
	{  671, { &kDefaultIPConstUnlimitedAmmoBasic } },
	{  672, { } },
	{  673, { } },
	{  674, { } },
	{  675, { } },
	{  676, { } },
	{  677, { } },
	{  678, { } },
	{  679, { } },
	{  680, { &kDefaultInt0 } },
	{  681, { } },
	{  682, { } },
	{  683, { } },
	{  684, { } },
	{  685, { } },
	{  686, { &kDefaultInt0 } },
	{  687, { } },
	{  688, { } },
	{  689, { &kDefaultTrue } },
	{  690, { &kDefaultInt100, &kDefaultSpellSchoolGeneral } },
	{  691, { &kDefaultTalkVolumeTalk } },
	{  692, { &kDefaultTrue } },
	{  693, { } },
	{  694, { } },
	{  695, { &kDefaultFadeSpeedMedium } },
	{  696, { &kDefaultFadeSpeedMedium, &kDefaultFloat5_0, &kDefaultInt0 } },
	{  697, { } },
	{  698, { &kDefaultInt0 } },
	{  699, { } },
	{  700, { } },
	{  701, { } },
	{  702, { } },
	{  703, { } },
	{  704, { &kDefaultClassTypeInvalid, &kDefaultFalse, &kDefaultPackageInvalid } },
	{  705, { } },
	{  706, { &kDefaultObjectSelf } },
	{  707, { } },
	{  708, { &kDefaultObjectSelf } },
	{  709, { &kDefaultTrue, &kDefaultFalse } },
	{  710, { } },
	{  711, { } },
	{  712, { &kDefaultObjectSelf } },
	{  713, { } },
	{  714, { } },
	{  715, { } },
	{  716, { &kDefaultObjectInvalid } },
	{  717, { } },
	{  718, { } },
	{  719, { } },
	{  720, { &kDefaultTrue } },
	{  721, { } },
	{  722, { } },
	{  723, { } },
	{  724, { } },
	{  725, { } },
	{  726, { } },
	{  727, { } },
	{  728, { } },
	{  729, { } },
	{  730, { } },
	{  731, { &kDefaultFalse } },
	{  732, { } },
	{  733, { } },
	{  734, { } },
	{  735, { } },
	{  736, { } },
	{  737, { } },
	{  738, { } },
	{  739, { } },
	{  740, { } },
	{  741, { } },
	{  742, { } },
	{  743, { } },
	{  744, { } },
	{  745, { } },
	{  746, { } },
	{  747, { } },
	{  748, { } },
	{  749, { } },
	{  750, { &kDefaultFloat0_0 } },
	{  751, { &kDefaultFloat0_0 } },
	{  752, { } },
	{  753, { } },
	{  754, { } },
	{  755, { &kDefaultObjectSelf } },
	{  756, { &kDefaultObjectSelf } },
	{  757, { } },
	{  758, { } },
	{  759, { } },
	{  760, { } },
	{  761, { } },
	{  762, { } },
	{  763, { } },
	{  764, { } },
	{  765, { } },
	{  766, { } },
	{  767, { } },
	{  768, { &kDefaultObjectSelf } },
	{  769, { } },
	{  770, { } },
	{  771, { } },
	{  772, { } },
	{  773, { } },
	{  774, { } },
	{  775, { } },
	{  776, { &kDefaultFloat0_0 } },
	{  777, { } },
	{  778, { } },
	{  779, { } },
	{  780, { } },
	{  781, { } },
	{  782, { } },
	{  783, { } },
	{  784, { } },
	{  785, { &kDefaultStringEmpty } },
	{  786, { &kDefaultStringEmpty } },
	{  787, { } },
	{  788, { } },
	{  789, { } },
	{  790, { } },
	{  791, { } },
	{  792, { &kDefaultObjectInvalid, &kDefaultVector0 } },
	{  793, { } },
	{  794, { } },
	{  795, { &kDefaultFalse } },
	{  796, { } },
	{  797, { } },
	{  798, { } },
	{  799, { } },
	{  800, { } },
	{  801, { } },
	{  802, { &kDefaultFloatMinus1_0, &kDefaultFloatMinus1_0 } },
	{  803, { &kDefaultInt50, &kDefaultInt50 } },
	{  804, { } },
	{  805, { } },
	{  806, { } },
	{  807, { &kDefaultDamageTypeMagical, &kDefaultFalse } },
	{  808, { } },
	{  809, { } },
	{  810, { &kDefaultFloat0_0 } },
	{  811, { } },
	{  812, { } },
	{  813, { } },
	{  814, { &kDefaultInt4294901760 } },
	{  815, { } },
	{  816, { } },
	{  817, { } },
	{  818, { } },
	{  819, { } },
	{  820, { } },
	{  821, { } },
	{  822, { } },
	{  823, { } },
	{  824, { } },
	{  825, { } },
	{  826, { } },
	{  827, { } },
	{  828, { } },
	{  829, { } },
	{  830, { } },
	{  831, { } },
	{  832, { } },
	{  833, { &kDefaultTrue } },
	{  834, { } },
	{  835, { } },
	{  836, { } },
	{  837, { } },
	{  838, { } },
	{  839, { &kDefaultObjectInvalid } },
	{  840, { } },
	{  841, { &kDefaultIntMinus1 } },
	{  842, { } },
	{  843, { } },
	{  844, { } },
	{  845, { } },
	{  846, { } },
	{  847, { } },
	{  848, { } },
	{  849, { } },
	{  850, { &kDefaultStringEmpty, &kDefaultFalse } },
	{  851, { } },
	{  852, { &kDefaultStringEmpty } },
	{  853, { &kDefaultTrue } },
	{  854, { } },
	{  855, { } },
	{  856, { } },
	{  857, { } },
	{  858, { } },
	{  859, { } },
	{  860, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultFalse, &kDefaultStringEmpty, &kDefaultInt0, &kDefaultStringEmpty, &kDefaultInt0, &kDefaultStringEmpty } },
	{  861, { &kDefaultFalse } },
	{  862, { } },
	{  863, { } },
	{  864, { } },
	{  865, { &kDefaultFalse, &kDefaultFalse } },
	{  866, { } },
	{  867, { } },
	{  868, { } },
	{  869, { } },
	{  870, { } },
	{  871, { } },
	{  872, { } },
	{  873, { } },
	{  874, { } },
	{  875, { } },
	{  876, { } },
	{  877, { } },
	{  878, { } },
	{  879, { } },
	{  880, { } },
	{  881, { } },
	{  882, { } },
	{  883, { } },
	{  884, { } },
	{  885, { } },
	{  886, { } },
	{  887, { } },
	{  888, { } },
	{  889, { } },
	{  890, { } },
	{  891, { } },
	{  892, { } },
	{  893, { } },
	{  894, { } },
	{  895, { } },
	{  896, { } },
	{  897, { } },
	{  898, { } },
	{  899, { } },
	{  900, { } },
	{  901, { } },
	{  902, { } },
	{  903, { } },
	{  904, { &kDefaultFloat1_0 } },
	{  905, { } },
	{  906, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultFalse, &kDefaultStringEmpty, &kDefaultInt0, &kDefaultStringEmpty, &kDefaultInt0, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{  907, { &kDefaultInt0 } },
	{  908, { &kDefaultInt0 } },
	{  909, { } },
	{  910, { } },
	{  911, { } },
	{  912, { } },
	{  913, { } },
	{  914, { } },
	{  915, { } },
	{  916, { } },
	{  917, { } },
	{  918, { } },
	{  919, { } },
	{  920, { } },
	{  921, { } },
	{  922, { } },
	{  923, { } },
	{  924, { } },
	{  925, { &kDefaultStringEmpty } },
	{  926, { } },
	{  927, { } },
	{  928, { } },
	{  929, { } },
	{  930, { } },
	{  931, { } },
	{  932, { } },
	{  933, { } },
	{  934, { } },
	{  935, { } },
	{  936, { } },
	{  937, { } },
	{  938, { } },
	{  939, { } },
	{  940, { } },
	{  941, { } },
	{  942, { } },
	{  943, { } },
	{  944, { } },
	{  945, { } },
	{  946, { &kDefaultTrue } },
	{  947, { } },
	{  948, { } },
	{  949, { } },
	{  950, { } },
	{  951, { } },
	{  952, { } },
	{  953, { &kDefaultTrue } },
	{  954, { } },
	{  955, { } },
	{  956, { } },
	{  957, { &kDefaultTrue } },
	{  958, { } },
	{  959, { } },
	{  960, { } },
	{  961, { } },
	{  962, { &kDefaultTrue } },
	{  963, { } },
	{  964, { &kDefaultTrue } },
	{  965, { } },
	{  966, { } },
	{  967, { &kDefaultTrue } },
	{  968, { &kDefaultFloat2_0, &kDefaultStringEmpty, &kDefaultStandardFactionHostile, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{  969, { &kDefaultStandardFactionHostile, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{  970, { &kDefaultObjectInvalid } },
	{  971, { } },
	{  972, { &kDefaultTrue } },
	{  973, { } },
	{  974, { } },
	{  975, { } },
	{  976, { } },
	{  977, { } },
	{  978, { } },
	{  979, { } },
	{  980, { } },
	{  981, { &kDefaultVFXNone, &kDefaultFloat0_0, &kDefaultStringEmpty, &kDefaultInt0, &kDefaultStringEmpty } },
	{  982, { } },
	{  983, { } },
	{  984, { } },
	{  985, { } },
	{  986, { } },
	{  987, { } },
	{  988, { } },
	{  989, { } },
	{  990, { } },
	{  991, { } },
	{  992, { } },
	{  993, { } },
	{  994, { } },
	{  995, { } },
	{  996, { } },
	{  997, { } },
	{  998, { } },
	{  999, { } },
	{ 1000, { } },
	{ 1001, { } },
	{ 1002, { } },
	{ 1003, { } },
	{ 1004, { } },
	{ 1005, { } },
	{ 1006, { } },
	{ 1007, { } },
	{ 1008, { &kDefaultObjectSelf } },
	{ 1009, { } },
	{ 1010, { } },
	{ 1011, { } },
	{ 1012, { } },
	{ 1013, { } },
	{ 1014, { } },
	{ 1015, { } },
	{ 1016, { } },
	{ 1017, { } },
	{ 1018, { } },
	{ 1019, { } },
	{ 1020, { } },
	{ 1021, { &kDefaultObjectSelf } },
	{ 1022, { } },
	{ 1023, { } },
	{ 1024, { } },
	{ 1025, { } },
	{ 1026, { } },
	{ 1027, { } },
	{ 1028, { } },
	{ 1029, { } },
	{ 1030, { } },
	{ 1031, { &kDefaultTrue } },
	{ 1032, { &kDefaultFalse } },
	{ 1033, { } },
	{ 1034, { } },
	{ 1035, { } },
	{ 1036, { } },
	{ 1037, { } },
	{ 1038, { &kDefaultTrue, &kDefaultTrue } },
	{ 1039, { } },
	{ 1040, { &kDefaultInt0 } },
	{ 1041, { } },
	{ 1042, { } },
	{ 1043, { } },
	{ 1044, { } },
	{ 1045, { } },
	{ 1046, { &kDefaultTrue } },
	{ 1047, { } },
	{ 1048, { } },
	{ 1049, { } },
	{ 1050, { } },
	{ 1051, { } },
	{ 1052, { } },
	{ 1053, { } },
	{ 1054, { } },
	{ 1055, { } },
	{ 1056, { } },
	{ 1057, { } },
	{ 1058, { } }
};

} // End of namespace NWN2

} // End of namespace Engines

#endif // ENGINES_NWN2_SCRIPT_TABLES_H
