/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Tables defining the engine functions in Neverwinter Nights.
 */

/* The functions are defined by three tables:
 * - kFunctionPointers
 * - kFunctionSignatures
 * - kFunctionDefaults
 *
 * kFunctionPointers provides the ID and name of the engine function, and a
 * pointer to Functions method doing the actual work. If the function pointer
 * is 0, a default unimplementedFunction method is used that just prints
 * the name of the function when it's called.
 *
 * kFunctionSignatures defines the signature of the function, i.e. the types
 * of the return value and its parameters.
 *
 * kFunctionDefaults pins default values to parameters. I.e. if a function
 * Foobar(int mode, int doEverything=FALSE) is called with only one parameters,
 * the default value FALSE is applied to the doEverything parameters. The
 * kFunctionDefaults entry for the function then contains a reference to a
 * false value. Because of ambiguities otherwise, the default values have to be
 * "aligned" to the right; only the last-most n parameters are allowed to have
 * default values, with no gaps.
 *
 * Please note that all three tables have to be of the same length, and that
 * the order of the functions have to be the same in all three tables.
 */

#ifndef ENGINES_NWN_SCRIPT_FUNCTION_TABLES_H
#define ENGINES_NWN_SCRIPT_FUNCTION_TABLES_H

#include "src/engines/nwn/types.h"
#include "src/engines/nwn/object.h"

namespace Engines {

namespace NWN {

using Aurora::NWScript::kTypeVoid;
using Aurora::NWScript::kTypeInt;
using Aurora::NWScript::kTypeFloat;
using Aurora::NWScript::kTypeString;
using Aurora::NWScript::kTypeObject;
using Aurora::NWScript::kTypeVector;
using Aurora::NWScript::kTypeEngineType;
using Aurora::NWScript::kTypeScriptState;
using Aurora::NWScript::kTypeAny;

/* Default values as used by the function parameters, if no explicit value
 * is given at the time of the calling.
 */

static NWN::Object kDefaultValueObjectInvalid(kObjectTypeInvalid);
static NWN::Object kDefaultValueObjectSelf   (kObjectTypeSelf);

static const Aurora::NWScript::Variable kDefaultIntMinus1((int32_t) -  1);
static const Aurora::NWScript::Variable kDefaultInt0     ((int32_t)    0);
static const Aurora::NWScript::Variable kDefaultInt1     ((int32_t)    1);
static const Aurora::NWScript::Variable kDefaultInt9     ((int32_t)    9);
static const Aurora::NWScript::Variable kDefaultInt18    ((int32_t)   18);
static const Aurora::NWScript::Variable kDefaultInt100   ((int32_t)  100);

static const Aurora::NWScript::Variable kDefaultFloatMinus1_0(-  1.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_0     (   0.0f);
static const Aurora::NWScript::Variable kDefaultFloat1_0     (   1.0f);
static const Aurora::NWScript::Variable kDefaultFloat2_0     (   2.0f);
static const Aurora::NWScript::Variable kDefaultFloat30_0    (  30.0f);
static const Aurora::NWScript::Variable kDefaultFloat40_0    (  40.0f);
static const Aurora::NWScript::Variable kDefaultFloat180_0   ( 180.0f);

static const Aurora::NWScript::Variable kDefaultVector0    (0.0f, 0.0f, 0.0f);

static const Aurora::NWScript::Variable kDefaultStringEmpty(Common::UString(""));

static const Aurora::NWScript::Variable kDefaultObjectInvalid(&kDefaultValueObjectInvalid);
static const Aurora::NWScript::Variable kDefaultObjectSelf   (&kDefaultValueObjectSelf);

// TODO: Add the relevant enums to types.h, and use these values.

static const Aurora::NWScript::Variable kDefaultACDodgeBonus             ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultACVsDamageTypeAll        ((int32_t)   4103);
static const Aurora::NWScript::Variable kDefaultAlignmentAll             ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultAttackBonusMisc          ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultCameraTransitionTypeSnap ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultClassTypeInvalid         ((int32_t)    255);
static const Aurora::NWScript::Variable kDefaultDamagePowerNormal        ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultDamageTypeMagical        ((int32_t)      8);
static const Aurora::NWScript::Variable kDefaultFalse                    ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultGenderMale               ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultIPConstUnlimitedAmmoBasic((int32_t)      1);
static const Aurora::NWScript::Variable kDefaultMetaMagicAny             ((int32_t)    255);
static const Aurora::NWScript::Variable kDefaultMissChanceTypeNormal     ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultNameFirstGenericMale     ((int32_t) -    1);
static const Aurora::NWScript::Variable kDefaultObjectTypeAll            ((int32_t)  32767);
static const Aurora::NWScript::Variable kDefaultObjectTypeCreature       ((int32_t)      1);
static const Aurora::NWScript::Variable kDefaultObjectTypeInvalid        ((int32_t)  32767);
static const Aurora::NWScript::Variable kDefaultPackageInvalid           ((int32_t)    255);
static const Aurora::NWScript::Variable kDefaultPersistentZoneActive     ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultProjectilePathTypeDefault((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultSavingThrowTypeAll       ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultSavingThrowTypeNone      ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultSpellAllSpells           ((int32_t) -    1);
static const Aurora::NWScript::Variable kDefaultSpellSchoolGeneral       ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultStandardFactionHostile   ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultTalkVolumeTalk           ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultTrue                     ((int32_t)      1);
static const Aurora::NWScript::Variable kDefaultUseCreatureLevel         ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultVFXNone                  ((int32_t) -    1);

static const Aurora::NWScript::Variable kDefaultFadeSpeedMedium          (0.01f);


/** The table defining the name and function pointer of each engine function. */
const Functions::FunctionPointer Functions::kFunctionPointers[] = {
	{   0, "Random"                              , &Functions::random                               },
	{   1, "PrintString"                         , &Functions::printString                          },
	{   2, "PrintFloat"                          , &Functions::printFloat                           },
	{   3, "FloatToString"                       , &Functions::floatToString                        },
	{   4, "PrintInteger"                        , &Functions::printInteger                         },
	{   5, "PrintObject"                         , &Functions::printObject                          },
	{   6, "AssignCommand"                       , &Functions::assignCommand                        },
	{   7, "DelayCommand"                        , &Functions::delayCommand                         },
	{   8, "ExecuteScript"                       , &Functions::executeScript                        },
	{   9, "ClearAllActions"                     , 0                                                },
	{  10, "SetFacing"                           , 0                                                },
	{  11, "SetCalendar"                         , 0                                                },
	{  12, "SetTime"                             , 0                                                },
	{  13, "GetCalendarYear"                     , 0                                                },
	{  14, "GetCalendarMonth"                    , 0                                                },
	{  15, "GetCalendarDay"                      , 0                                                },
	{  16, "GetTimeHour"                         , 0                                                },
	{  17, "GetTimeMinute"                       , 0                                                },
	{  18, "GetTimeSecond"                       , 0                                                },
	{  19, "GetTimeMillisecond"                  , 0                                                },
	{  20, "ActionRandomWalk"                    , 0                                                },
	{  21, "ActionMoveToLocation"                , &Functions::actionMoveToLocation                 },
	{  22, "ActionMoveToObject"                  , &Functions::actionMoveToObject                   },
	{  23, "ActionMoveAwayFromObject"            , 0                                                },
	{  24, "GetArea"                             , &Functions::getArea                              },
	{  25, "GetEnteringObject"                   , &Functions::getEnteringObject                    },
	{  26, "GetExitingObject"                    , &Functions::getExitingObject                     },
	{  27, "GetPosition"                         , &Functions::getPosition                          },
	{  28, "GetFacing"                           , 0                                                },
	{  29, "GetItemPossessor"                    , 0                                                },
	{  30, "GetItemPossessedBy"                  , 0                                                },
	{  31, "CreateItemOnObject"                  , 0                                                },
	{  32, "ActionEquipItem"                     , 0                                                },
	{  33, "ActionUnequipItem"                   , 0                                                },
	{  34, "ActionPickUpItem"                    , 0                                                },
	{  35, "ActionPutDownItem"                   , 0                                                },
	{  36, "GetLastAttacker"                     , 0                                                },
	{  37, "ActionAttack"                        , 0                                                },
	{  38, "GetNearestCreature"                  , &Functions::getNearestCreature                   },
	{  39, "ActionSpeakString"                   , &Functions::actionSpeakString                    },
	{  40, "ActionPlayAnimation"                 , &Functions::actionPlayAnimation                  },
	{  41, "GetDistanceToObject"                 , &Functions::getDistanceToObject                  },
	{  42, "GetIsObjectValid"                    , &Functions::getIsObjectValid                     },
	{  43, "ActionOpenDoor"                      , &Functions::actionOpenDoor                       },
	{  44, "ActionCloseDoor"                     , &Functions::actionCloseDoor                      },
	{  45, "SetCameraFacing"                     , 0                                                },
	{  46, "PlaySound"                           , &Functions::playSound                            },
	{  47, "GetSpellTargetObject"                , 0                                                },
	{  48, "ActionCastSpellAtObject"             , 0                                                },
	{  49, "GetCurrentHitPoints"                 , 0                                                },
	{  50, "GetMaxHitPoints"                     , 0                                                },
	{  51, "GetLocalInt"                         , &Functions::getLocalInt                          },
	{  52, "GetLocalFloat"                       , &Functions::getLocalFloat                        },
	{  53, "GetLocalString"                      , &Functions::getLocalString                       },
	{  54, "GetLocalObject"                      , &Functions::getLocalObject                       },
	{  55, "SetLocalInt"                         , &Functions::setLocalInt                          },
	{  56, "SetLocalFloat"                       , &Functions::setLocalFloat                        },
	{  57, "SetLocalString"                      , &Functions::setLocalString                       },
	{  58, "SetLocalObject"                      , &Functions::setLocalObject                       },
	{  59, "GetStringLength"                     , &Functions::getStringLength                      },
	{  60, "GetStringUpperCase"                  , &Functions::getStringUpperCase                   },
	{  61, "GetStringLowerCase"                  , &Functions::getStringLowerCase                   },
	{  62, "GetStringRight"                      , &Functions::getStringRight                       },
	{  63, "GetStringLeft"                       , &Functions::getStringLeft                        },
	{  64, "InsertString"                        , &Functions::insertString                         },
	{  65, "GetSubString"                        , &Functions::getSubString                         },
	{  66, "FindSubString"                       , &Functions::findSubString                        },
	{  67, "fabs"                                , &Functions::fabs                                 },
	{  68, "cos"                                 , &Functions::cos                                  },
	{  69, "sin"                                 , &Functions::sin                                  },
	{  70, "tan"                                 , &Functions::tan                                  },
	{  71, "acos"                                , &Functions::acos                                 },
	{  72, "asin"                                , &Functions::asin                                 },
	{  73, "atan"                                , &Functions::atan                                 },
	{  74, "log"                                 , &Functions::log                                  },
	{  75, "pow"                                 , &Functions::pow                                  },
	{  76, "sqrt"                                , &Functions::sqrt                                 },
	{  77, "abs"                                 , &Functions::abs                                  },
	{  78, "EffectHeal"                          , 0                                                },
	{  79, "EffectDamage"                        , 0                                                },
	{  80, "EffectAbilityIncrease"               , 0                                                },
	{  81, "EffectDamageResistance"              , 0                                                },
	{  82, "EffectResurrection"                  , 0                                                },
	{  83, "EffectSummonCreature"                , 0                                                },
	{  84, "GetCasterLevel"                      , 0                                                },
	{  85, "GetFirstEffect"                      , 0                                                },
	{  86, "GetNextEffect"                       , 0                                                },
	{  87, "RemoveEffect"                        , 0                                                },
	{  88, "GetIsEffectValid"                    , 0                                                },
	{  89, "GetEffectDurationType"               , 0                                                },
	{  90, "GetEffectSubType"                    , 0                                                },
	{  91, "GetEffectCreator"                    , 0                                                },
	{  92, "IntToString"                         , &Functions::intToString                          },
	{  93, "GetFirstObjectInArea"                , 0                                                },
	{  94, "GetNextObjectInArea"                 , 0                                                },
	{  95, "d2"                                  , &Functions::d2                                   },
	{  96, "d3"                                  , &Functions::d3                                   },
	{  97, "d4"                                  , &Functions::d4                                   },
	{  98, "d6"                                  , &Functions::d6                                   },
	{  99, "d8"                                  , &Functions::d8                                   },
	{ 100, "d10"                                 , &Functions::d10                                  },
	{ 101, "d12"                                 , &Functions::d12                                  },
	{ 102, "d20"                                 , &Functions::d20                                  },
	{ 103, "d100"                                , &Functions::d100                                 },
	{ 104, "VectorMagnitude"                     , &Functions::vectorMagnitude                      },
	{ 105, "GetMetaMagicFeat"                    , 0                                                },
	{ 106, "GetObjectType"                       , &Functions::getObjectType                        },
	{ 107, "GetRacialType"                       , &Functions::getRacialType                        },
	{ 108, "FortitudeSave"                       , 0                                                },
	{ 109, "ReflexSave"                          , 0                                                },
	{ 110, "WillSave"                            , 0                                                },
	{ 111, "GetSpellSaveDC"                      , 0                                                },
	{ 112, "MagicalEffect"                       , 0                                                },
	{ 113, "SupernaturalEffect"                  , 0                                                },
	{ 114, "ExtraordinaryEffect"                 , 0                                                },
	{ 115, "EffectACIncrease"                    , 0                                                },
	{ 116, "GetAC"                               , 0                                                },
	{ 117, "EffectSavingThrowIncrease"           , 0                                                },
	{ 118, "EffectAttackIncrease"                , 0                                                },
	{ 119, "EffectDamageReduction"               , 0                                                },
	{ 120, "EffectDamageIncrease"                , 0                                                },
	{ 121, "RoundsToSeconds"                     , 0                                                },
	{ 122, "HoursToSeconds"                      , 0                                                },
	{ 123, "TurnsToSeconds"                      , 0                                                },
	{ 124, "GetLawChaosValue"                    , &Functions::getLawChaosValue                     },
	{ 125, "GetGoodEvilValue"                    , &Functions::getGoodEvilValue                     },
	{ 126, "GetAlignmentLawChaos"                , &Functions::getAlignmentLawChaos                 },
	{ 127, "GetAlignmentGoodEvil"                , &Functions::getAlignmentGoodEvil                 },
	{ 128, "GetFirstObjectInShape"               , 0                                                },
	{ 129, "GetNextObjectInShape"                , 0                                                },
	{ 130, "EffectEntangle"                      , 0                                                },
	{ 131, "SignalEvent"                         , 0                                                },
	{ 132, "EventUserDefined"                    , 0                                                },
	{ 133, "EffectDeath"                         , 0                                                },
	{ 134, "EffectKnockdown"                     , 0                                                },
	{ 135, "ActionGiveItem"                      , 0                                                },
	{ 136, "ActionTakeItem"                      , 0                                                },
	{ 137, "VectorNormalize"                     , &Functions::vectorNormalize                      },
	{ 138, "EffectCurse"                         , 0                                                },
	{ 139, "GetAbilityScore"                     , &Functions::getAbilityScore                      },
	{ 140, "GetIsDead"                           , &Functions::getIsDead                            },
	{ 141, "PrintVector"                         , &Functions::printVector                          },
	{ 142, "Vector"                              , &Functions::vector                               },
	{ 143, "SetFacingPoint"                      , 0                                                },
	{ 144, "AngleToVector"                       , 0                                                },
	{ 145, "VectorToAngle"                       , 0                                                },
	{ 146, "TouchAttackMelee"                    , 0                                                },
	{ 147, "TouchAttackRanged"                   , 0                                                },
	{ 148, "EffectParalyze"                      , 0                                                },
	{ 149, "EffectSpellImmunity"                 , 0                                                },
	{ 150, "EffectDeaf"                          , 0                                                },
	{ 151, "GetDistanceBetween"                  , 0                                                },
	{ 152, "SetLocalLocation"                    , 0                                                },
	{ 153, "GetLocalLocation"                    , 0                                                },
	{ 154, "EffectSleep"                         , 0                                                },
	{ 155, "GetItemInSlot"                       , 0                                                },
	{ 156, "EffectCharmed"                       , 0                                                },
	{ 157, "EffectConfused"                      , 0                                                },
	{ 158, "EffectFrightened"                    , 0                                                },
	{ 159, "EffectDominated"                     , 0                                                },
	{ 160, "EffectDazed"                         , 0                                                },
	{ 161, "EffectStunned"                       , 0                                                },
	{ 162, "SetCommandable"                      , &Functions::setCommandable                       },
	{ 163, "GetCommandable"                      , &Functions::getCommandable                       },
	{ 164, "EffectRegenerate"                    , 0                                                },
	{ 165, "EffectMovementSpeedIncrease"         , 0                                                },
	{ 166, "GetHitDice"                          , &Functions::getHitDice                           },
	{ 167, "ActionForceFollowObject"             , 0                                                },
	{ 168, "GetTag"                              , &Functions::getTag                               },
	{ 169, "ResistSpell"                         , 0                                                },
	{ 170, "GetEffectType"                       , 0                                                },
	{ 171, "EffectAreaOfEffect"                  , 0                                                },
	{ 172, "GetFactionEqual"                     , 0                                                },
	{ 173, "ChangeFaction"                       , 0                                                },
	{ 174, "GetIsListening"                      , 0                                                },
	{ 175, "SetListening"                        , 0                                                },
	{ 176, "SetListenPattern"                    , 0                                                },
	{ 177, "TestStringAgainstPattern"            , 0                                                },
	{ 178, "GetMatchedSubstring"                 , 0                                                },
	{ 179, "GetMatchedSubstringsCount"           , 0                                                },
	{ 180, "EffectVisualEffect"                  , 0                                                },
	{ 181, "GetFactionWeakestMember"             , 0                                                },
	{ 182, "GetFactionStrongestMember"           , 0                                                },
	{ 183, "GetFactionMostDamagedMember"         , 0                                                },
	{ 184, "GetFactionLeastDamagedMember"        , 0                                                },
	{ 185, "GetFactionGold"                      , 0                                                },
	{ 186, "GetFactionAverageReputation"         , 0                                                },
	{ 187, "GetFactionAverageGoodEvilAlignment"  , 0                                                },
	{ 188, "GetFactionAverageLawChaosAlignment"  , 0                                                },
	{ 189, "GetFactionAverageLevel"              , 0                                                },
	{ 190, "GetFactionAverageXP"                 , 0                                                },
	{ 191, "GetFactionMostFrequentClass"         , 0                                                },
	{ 192, "GetFactionWorstAC"                   , 0                                                },
	{ 193, "GetFactionBestAC"                    , 0                                                },
	{ 194, "ActionSit"                           , 0                                                },
	{ 195, "GetListenPatternNumber"              , &Functions::getListenPatternNumber               },
	{ 196, "ActionJumpToObject"                  , &Functions::actionJumpToObject                   },
	{ 197, "GetWaypointByTag"                    , &Functions::getWaypointByTag                     },
	{ 198, "GetTransitionTarget"                 , 0                                                },
	{ 199, "EffectLinkEffects"                   , 0                                                },
	{ 200, "GetObjectByTag"                      , &Functions::getObjectByTag                       },
	{ 201, "AdjustAlignment"                     , 0                                                },
	{ 202, "ActionWait"                          , 0                                                },
	{ 203, "SetAreaTransitionBMP"                , 0                                                },
	{ 204, "ActionStartConversation"             , &Functions::actionStartConversation              },
	{ 205, "ActionPauseConversation"             , 0                                                },
	{ 206, "ActionResumeConversation"            , 0                                                },
	{ 207, "EffectBeam"                          , 0                                                },
	{ 208, "GetReputation"                       , 0                                                },
	{ 209, "AdjustReputation"                    , 0                                                },
	{ 210, "GetSittingCreature"                  , 0                                                },
	{ 211, "GetGoingToBeAttackedBy"              , 0                                                },
	{ 212, "EffectSpellResistanceIncrease"       , 0                                                },
	{ 213, "GetLocation"                         , &Functions::getLocation                          },
	{ 214, "ActionJumpToLocation"                , &Functions::actionJumpToLocation                 },
	{ 215, "Location"                            , &Functions::location                             },
	{ 216, "ApplyEffectAtLocation"               , 0                                                },
	{ 217, "GetIsPC"                             , &Functions::getIsPC                              },
	{ 218, "FeetToMeters"                        , 0                                                },
	{ 219, "YardsToMeters"                       , 0                                                },
	{ 220, "ApplyEffectToObject"                 , 0                                                },
	{ 221, "SpeakString"                         , &Functions::speakString                          },
	{ 222, "GetSpellTargetLocation"              , 0                                                },
	{ 223, "GetPositionFromLocation"             , &Functions::getPositionFromLocation              },
	{ 224, "GetAreaFromLocation"                 , 0                                                },
	{ 225, "GetFacingFromLocation"               , 0                                                },
	{ 226, "GetNearestCreatureToLocation"        , 0                                                },
	{ 227, "GetNearestObject"                    , &Functions::getNearestObject                     },
	{ 228, "GetNearestObjectToLocation"          , 0                                                },
	{ 229, "GetNearestObjectByTag"               , &Functions::getNearestObjectByTag                },
	{ 230, "IntToFloat"                          , &Functions::intToFloat                           },
	{ 231, "FloatToInt"                          , &Functions::floatToInt                           },
	{ 232, "StringToInt"                         , &Functions::stringToInt                          },
	{ 233, "StringToFloat"                       , &Functions::stringToFloat                        },
	{ 234, "ActionCastSpellAtLocation"           , 0                                                },
	{ 235, "GetIsEnemy"                          , 0                                                },
	{ 236, "GetIsFriend"                         , 0                                                },
	{ 237, "GetIsNeutral"                        , 0                                                },
	{ 238, "GetPCSpeaker"                        , &Functions::getPCSpeaker                         },
	{ 239, "GetStringByStrRef"                   , &Functions::getStringByStrRef                    },
	{ 240, "ActionSpeakStringByStrRef"           , 0                                                },
	{ 241, "DestroyObject"                       , 0                                                },
	{ 242, "GetModule"                           , &Functions::getModule                            },
	{ 243, "CreateObject"                        , 0                                                },
	{ 244, "EventSpellCastAt"                    , 0                                                },
	{ 245, "GetLastSpellCaster"                  , 0                                                },
	{ 246, "GetLastSpell"                        , 0                                                },
	{ 247, "GetUserDefinedEventNumber"           , 0                                                },
	{ 248, "GetSpellId"                          , 0                                                },
	{ 249, "RandomName"                          , 0                                                },
	{ 250, "EffectPoison"                        , 0                                                },
	{ 251, "EffectDisease"                       , 0                                                },
	{ 252, "EffectSilence"                       , 0                                                },
	{ 253, "GetName"                             , &Functions::getName                              },
	{ 254, "GetLastSpeaker"                      , &Functions::getLastSpeaker                       },
	{ 255, "BeginConversation"                   , &Functions::beginConversation                    },
	{ 256, "GetLastPerceived"                    , 0                                                },
	{ 257, "GetLastPerceptionHeard"              , 0                                                },
	{ 258, "GetLastPerceptionInaudible"          , 0                                                },
	{ 259, "GetLastPerceptionSeen"               , 0                                                },
	{ 260, "GetLastClosedBy"                     , &Functions::getLastClosedBy                      },
	{ 261, "GetLastPerceptionVanished"           , 0                                                },
	{ 262, "GetFirstInPersistentObject"          , 0                                                },
	{ 263, "GetNextInPersistentObject"           , 0                                                },
	{ 264, "GetAreaOfEffectCreator"              , 0                                                },
	{ 265, "DeleteLocalInt"                      , 0                                                },
	{ 266, "DeleteLocalFloat"                    , 0                                                },
	{ 267, "DeleteLocalString"                   , 0                                                },
	{ 268, "DeleteLocalObject"                   , 0                                                },
	{ 269, "DeleteLocalLocation"                 , 0                                                },
	{ 270, "EffectHaste"                         , 0                                                },
	{ 271, "EffectSlow"                          , 0                                                },
	{ 272, "ObjectToString"                      , &Functions::objectToString                       },
	{ 273, "EffectImmunity"                      , 0                                                },
	{ 274, "GetIsImmune"                         , 0                                                },
	{ 275, "EffectDamageImmunityIncrease"        , 0                                                },
	{ 276, "GetEncounterActive"                  , 0                                                },
	{ 277, "SetEncounterActive"                  , 0                                                },
	{ 278, "GetEncounterSpawnsMax"               , 0                                                },
	{ 279, "SetEncounterSpawnsMax"               , 0                                                },
	{ 280, "GetEncounterSpawnsCurrent"           , 0                                                },
	{ 281, "SetEncounterSpawnsCurrent"           , 0                                                },
	{ 282, "GetModuleItemAcquired"               , 0                                                },
	{ 283, "GetModuleItemAcquiredFrom"           , 0                                                },
	{ 284, "SetCustomToken"                      , &Functions::setCustomToken                       },
	{ 285, "GetHasFeat"                          , 0                                                },
	{ 286, "GetHasSkill"                         , 0                                                },
	{ 287, "ActionUseFeat"                       , 0                                                },
	{ 288, "ActionUseSkill"                      , 0                                                },
	{ 289, "GetObjectSeen"                       , 0                                                },
	{ 290, "GetObjectHeard"                      , 0                                                },
	{ 291, "GetLastPlayerDied"                   , 0                                                },
	{ 292, "GetModuleItemLost"                   , 0                                                },
	{ 293, "GetModuleItemLostBy"                 , 0                                                },
	{ 294, "ActionDoCommand"                     , &Functions::actionDoCommand                      },
	{ 295, "EventConversation"                   , 0                                                },
	{ 296, "SetEncounterDifficulty"              , 0                                                },
	{ 297, "GetEncounterDifficulty"              , 0                                                },
	{ 298, "GetDistanceBetweenLocations"         , 0                                                },
	{ 299, "GetReflexAdjustedDamage"             , 0                                                },
	{ 300, "PlayAnimation"                       , &Functions::playAnimation                        },
	{ 301, "TalentSpell"                         , 0                                                },
	{ 302, "TalentFeat"                          , 0                                                },
	{ 303, "TalentSkill"                         , 0                                                },
	{ 304, "GetHasSpellEffect"                   , 0                                                },
	{ 305, "GetEffectSpellId"                    , 0                                                },
	{ 306, "GetCreatureHasTalent"                , 0                                                },
	{ 307, "GetCreatureTalentRandom"             , 0                                                },
	{ 308, "GetCreatureTalentBest"               , 0                                                },
	{ 309, "ActionUseTalentOnObject"             , 0                                                },
	{ 310, "ActionUseTalentAtLocation"           , 0                                                },
	{ 311, "GetGoldPieceValue"                   , 0                                                },
	{ 312, "GetIsPlayableRacialType"             , 0                                                },
	{ 313, "JumpToLocation"                      , &Functions::jumpToLocation                       },
	{ 314, "EffectTemporaryHitpoints"            , 0                                                },
	{ 315, "GetSkillRank"                        , &Functions::getSkillRank                         },
	{ 316, "GetAttackTarget"                     , 0                                                },
	{ 317, "GetLastAttackType"                   , 0                                                },
	{ 318, "GetLastAttackMode"                   , 0                                                },
	{ 319, "GetMaster"                           , &Functions::getMaster                            },
	{ 320, "GetIsInCombat"                       , 0                                                },
	{ 321, "GetLastAssociateCommand"             , 0                                                },
	{ 322, "GiveGoldToCreature"                  , 0                                                },
	{ 323, "SetIsDestroyable"                    , 0                                                },
	{ 324, "SetLocked"                           , &Functions::setLocked                            },
	{ 325, "GetLocked"                           , &Functions::getLocked                            },
	{ 326, "GetClickingObject"                   , &Functions::getClickingObject                    },
	{ 327, "SetAssociateListenPatterns"          , 0                                                },
	{ 328, "GetLastWeaponUsed"                   , 0                                                },
	{ 329, "ActionInteractObject"                , 0                                                },
	{ 330, "GetLastUsedBy"                       , &Functions::getLastUsedBy                        },
	{ 331, "GetAbilityModifier"                  , 0                                                },
	{ 332, "GetIdentified"                       , 0                                                },
	{ 333, "SetIdentified"                       , 0                                                },
	{ 334, "SummonAnimalCompanion"               , 0                                                },
	{ 335, "SummonFamiliar"                      , 0                                                },
	{ 336, "GetBlockingDoor"                     , 0                                                },
	{ 337, "GetIsDoorActionPossible"             , 0                                                },
	{ 338, "DoDoorAction"                        , 0                                                },
	{ 339, "GetFirstItemInInventory"             , 0                                                },
	{ 340, "GetNextItemInInventory"              , 0                                                },
	{ 341, "GetClassByPosition"                  , &Functions::getClassByPosition                   },
	{ 342, "GetLevelByPosition"                  , &Functions::getLevelByPosition                   },
	{ 343, "GetLevelByClass"                     , &Functions::getLevelByClass                      },
	{ 344, "GetDamageDealtByType"                , 0                                                },
	{ 345, "GetTotalDamageDealt"                 , 0                                                },
	{ 346, "GetLastDamager"                      , 0                                                },
	{ 347, "GetLastDisarmed"                     , 0                                                },
	{ 348, "GetLastDisturbed"                    , 0                                                },
	{ 349, "GetLastLocked"                       , 0                                                },
	{ 350, "GetLastUnlocked"                     , 0                                                },
	{ 351, "EffectSkillIncrease"                 , 0                                                },
	{ 352, "GetInventoryDisturbType"             , 0                                                },
	{ 353, "GetInventoryDisturbItem"             , 0                                                },
	{ 354, "GetHenchman"                         , &Functions::getHenchman                          },
	{ 355, "VersusAlignmentEffect"               , 0                                                },
	{ 356, "VersusRacialTypeEffect"              , 0                                                },
	{ 357, "VersusTrapEffect"                    , 0                                                },
	{ 358, "GetGender"                           , &Functions::getGender                            },
	{ 359, "GetIsTalentValid"                    , 0                                                },
	{ 360, "ActionMoveAwayFromLocation"          , 0                                                },
	{ 361, "GetAttemptedAttackTarget"            , 0                                                },
	{ 362, "GetTypeFromTalent"                   , 0                                                },
	{ 363, "GetIdFromTalent"                     , 0                                                },
	{ 364, "GetAssociate"                        , &Functions::getAssociate                         },
	{ 365, "AddHenchman"                         , &Functions::addHenchman                          },
	{ 366, "RemoveHenchman"                      , &Functions::removeHenchman                       },
	{ 367, "AddJournalQuestEntry"                , 0                                                },
	{ 368, "RemoveJournalQuestEntry"             , 0                                                },
	{ 369, "GetPCPublicCDKey"                    , 0                                                },
	{ 370, "GetPCIPAddress"                      , 0                                                },
	{ 371, "GetPCPlayerName"                     , 0                                                },
	{ 372, "SetPCLike"                           , 0                                                },
	{ 373, "SetPCDislike"                        , 0                                                },
	{ 374, "SendMessageToPC"                     , &Functions::sendMessageToPC                      },
	{ 375, "GetAttemptedSpellTarget"             , 0                                                },
	{ 376, "GetLastOpenedBy"                     , &Functions::getLastOpenedBy                      },
	{ 377, "GetHasSpell"                         , 0                                                },
	{ 378, "OpenStore"                           , 0                                                },
	{ 379, "EffectTurned"                        , 0                                                },
	{ 380, "GetFirstFactionMember"               , 0                                                },
	{ 381, "GetNextFactionMember"                , 0                                                },
	{ 382, "ActionForceMoveToLocation"           , 0                                                },
	{ 383, "ActionForceMoveToObject"             , 0                                                },
	{ 384, "GetJournalQuestExperience"           , 0                                                },
	{ 385, "JumpToObject"                        , &Functions::jumpToObject                         },
	{ 386, "SetMapPinEnabled"                    , 0                                                },
	{ 387, "EffectHitPointChangeWhenDying"       , 0                                                },
	{ 388, "PopUpGUIPanel"                       , 0                                                },
	{ 389, "ClearPersonalReputation"             , 0                                                },
	{ 390, "SetIsTemporaryFriend"                , 0                                                },
	{ 391, "SetIsTemporaryEnemy"                 , 0                                                },
	{ 392, "SetIsTemporaryNeutral"               , 0                                                },
	{ 393, "GiveXPToCreature"                    , 0                                                },
	{ 394, "SetXP"                               , 0                                                },
	{ 395, "GetXP"                               , &Functions::getXP                                },
	{ 396, "IntToHexString"                      , &Functions::intToHexString                       },
	{ 397, "GetBaseItemType"                     , 0                                                },
	{ 398, "GetItemHasItemProperty"              , 0                                                },
	{ 399, "ActionEquipMostDamagingMelee"        , 0                                                },
	{ 400, "ActionEquipMostDamagingRanged"       , 0                                                },
	{ 401, "GetItemACValue"                      , 0                                                },
	{ 402, "ActionRest"                          , 0                                                },
	{ 403, "ExploreAreaForPlayer"                , 0                                                },
	{ 404, "ActionEquipMostEffectiveArmor"       , 0                                                },
	{ 405, "GetIsDay"                            , 0                                                },
	{ 406, "GetIsNight"                          , 0                                                },
	{ 407, "GetIsDawn"                           , 0                                                },
	{ 408, "GetIsDusk"                           , 0                                                },
	{ 409, "GetIsEncounterCreature"              , 0                                                },
	{ 410, "GetLastPlayerDying"                  , 0                                                },
	{ 411, "GetStartingLocation"                 , 0                                                },
	{ 412, "ChangeToStandardFaction"             , 0                                                },
	{ 413, "SoundObjectPlay"                     , 0                                                },
	{ 414, "SoundObjectStop"                     , 0                                                },
	{ 415, "SoundObjectSetVolume"                , 0                                                },
	{ 416, "SoundObjectSetPosition"              , 0                                                },
	{ 417, "SpeakOneLinerConversation"           , &Functions::speakOneLinerConversation            },
	{ 418, "GetGold"                             , 0                                                },
	{ 419, "GetLastRespawnButtonPresser"         , 0                                                },
	{ 420, "GetIsDM"                             , &Functions::getIsDM                              },
	{ 421, "PlayVoiceChat"                       , &Functions::playVoiceChat                        },
	{ 422, "GetIsWeaponEffective"                , 0                                                },
	{ 423, "GetLastSpellHarmful"                 , 0                                                },
	{ 424, "EventActivateItem"                   , 0                                                },
	{ 425, "MusicBackgroundPlay"                 , &Functions::musicBackgroundPlay                  },
	{ 426, "MusicBackgroundStop"                 , &Functions::musicBackgroundStop                  },
	{ 427, "MusicBackgroundSetDelay"             , 0                                                },
	{ 428, "MusicBackgroundChangeDay"            , &Functions::musicBackgroundChangeDay             },
	{ 429, "MusicBackgroundChangeNight"          , &Functions::musicBackgroundChangeNight           },
	{ 430, "MusicBattlePlay"                     , 0                                                },
	{ 431, "MusicBattleStop"                     , 0                                                },
	{ 432, "MusicBattleChange"                   , 0                                                },
	{ 433, "AmbientSoundPlay"                    , 0                                                },
	{ 434, "AmbientSoundStop"                    , 0                                                },
	{ 435, "AmbientSoundChangeDay"               , 0                                                },
	{ 436, "AmbientSoundChangeNight"             , 0                                                },
	{ 437, "GetLastKiller"                       , 0                                                },
	{ 438, "GetSpellCastItem"                    , 0                                                },
	{ 439, "GetItemActivated"                    , 0                                                },
	{ 440, "GetItemActivator"                    , 0                                                },
	{ 441, "GetItemActivatedTargetLocation"      , 0                                                },
	{ 442, "GetItemActivatedTarget"              , 0                                                },
	{ 443, "GetIsOpen"                           , &Functions::getIsOpen                            },
	{ 444, "TakeGoldFromCreature"                , 0                                                },
	{ 445, "IsInConversation"                    , &Functions::isInConversation                     },
	{ 446, "EffectAbilityDecrease"               , 0                                                },
	{ 447, "EffectAttackDecrease"                , 0                                                },
	{ 448, "EffectDamageDecrease"                , 0                                                },
	{ 449, "EffectDamageImmunityDecrease"        , 0                                                },
	{ 450, "EffectACDecrease"                    , 0                                                },
	{ 451, "EffectMovementSpeedDecrease"         , 0                                                },
	{ 452, "EffectSavingThrowDecrease"           , 0                                                },
	{ 453, "EffectSkillDecrease"                 , 0                                                },
	{ 454, "EffectSpellResistanceDecrease"       , 0                                                },
	{ 455, "GetPlotFlag"                         , 0                                                },
	{ 456, "SetPlotFlag"                         , 0                                                },
	{ 457, "EffectInvisibility"                  , 0                                                },
	{ 458, "EffectConcealment"                   , 0                                                },
	{ 459, "EffectDarkness"                      , 0                                                },
	{ 460, "EffectDispelMagicAll"                , 0                                                },
	{ 461, "EffectUltravision"                   , 0                                                },
	{ 462, "EffectNegativeLevel"                 , 0                                                },
	{ 463, "EffectPolymorph"                     , 0                                                },
	{ 464, "EffectSanctuary"                     , 0                                                },
	{ 465, "EffectTrueSeeing"                    , 0                                                },
	{ 466, "EffectSeeInvisible"                  , 0                                                },
	{ 467, "EffectTimeStop"                      , 0                                                },
	{ 468, "EffectBlindness"                     , 0                                                },
	{ 469, "GetIsReactionTypeFriendly"           , 0                                                },
	{ 470, "GetIsReactionTypeNeutral"            , 0                                                },
	{ 471, "GetIsReactionTypeHostile"            , 0                                                },
	{ 472, "EffectSpellLevelAbsorption"          , 0                                                },
	{ 473, "EffectDispelMagicBest"               , 0                                                },
	{ 474, "ActivatePortal"                      , 0                                                },
	{ 475, "GetNumStackedItems"                  , 0                                                },
	{ 476, "SurrenderToEnemies"                  , 0                                                },
	{ 477, "EffectMissChance"                    , 0                                                },
	{ 478, "GetTurnResistanceHD"                 , 0                                                },
	{ 479, "GetCreatureSize"                     , 0                                                },
	{ 480, "EffectDisappearAppear"               , 0                                                },
	{ 481, "EffectDisappear"                     , 0                                                },
	{ 482, "EffectAppear"                        , 0                                                },
	{ 483, "ActionUnlockObject"                  , 0                                                },
	{ 484, "ActionLockObject"                    , 0                                                },
	{ 485, "EffectModifyAttacks"                 , 0                                                },
	{ 486, "GetLastTrapDetected"                 , 0                                                },
	{ 487, "EffectDamageShield"                  , 0                                                },
	{ 488, "GetNearestTrapToObject"              , 0                                                },
	{ 489, "GetDeity"                            , 0                                                },
	{ 490, "GetSubRace"                          , 0                                                },
	{ 491, "GetFortitudeSavingThrow"             , 0                                                },
	{ 492, "GetWillSavingThrow"                  , 0                                                },
	{ 493, "GetReflexSavingThrow"                , 0                                                },
	{ 494, "GetChallengeRating"                  , 0                                                },
	{ 495, "GetAge"                              , 0                                                },
	{ 496, "GetMovementRate"                     , 0                                                },
	{ 497, "GetFamiliarCreatureType"             , 0                                                },
	{ 498, "GetAnimalCompanionCreatureType"      , 0                                                },
	{ 499, "GetFamiliarName"                     , 0                                                },
	{ 500, "GetAnimalCompanionName"              , 0                                                },
	{ 501, "ActionCastFakeSpellAtObject"         , 0                                                },
	{ 502, "ActionCastFakeSpellAtLocation"       , 0                                                },
	{ 503, "RemoveSummonedAssociate"             , 0                                                },
	{ 504, "SetCameraMode"                       , 0                                                },
	{ 505, "GetIsResting"                        , 0                                                },
	{ 506, "GetLastPCRested"                     , 0                                                },
	{ 507, "SetWeather"                          , 0                                                },
	{ 508, "GetLastRestEventType"                , 0                                                },
	{ 509, "StartNewModule"                      , &Functions::startNewModule                       },
	{ 510, "EffectSwarm"                         , 0                                                },
	{ 511, "GetWeaponRanged"                     , 0                                                },
	{ 512, "DoSinglePlayerAutoSave"              , 0                                                },
	{ 513, "GetGameDifficulty"                   , 0                                                },
	{ 514, "SetTileMainLightColor"               , 0                                                },
	{ 515, "SetTileSourceLightColor"             , 0                                                },
	{ 516, "RecomputeStaticLighting"             , 0                                                },
	{ 517, "GetTileMainLight1Color"              , 0                                                },
	{ 518, "GetTileMainLight2Color"              , 0                                                },
	{ 519, "GetTileSourceLight1Color"            , 0                                                },
	{ 520, "GetTileSourceLight2Color"            , 0                                                },
	{ 521, "SetPanelButtonFlash"                 , 0                                                },
	{ 522, "GetCurrentAction"                    , 0                                                },
	{ 523, "SetStandardFactionReputation"        , 0                                                },
	{ 524, "GetStandardFactionReputation"        , 0                                                },
	{ 525, "FloatingTextStrRefOnCreature"        , 0                                                },
	{ 526, "FloatingTextStringOnCreature"        , 0                                                },
	{ 527, "GetTrapDisarmable"                   , 0                                                },
	{ 528, "GetTrapDetectable"                   , 0                                                },
	{ 529, "GetTrapDetectedBy"                   , 0                                                },
	{ 530, "GetTrapFlagged"                      , 0                                                },
	{ 531, "GetTrapBaseType"                     , 0                                                },
	{ 532, "GetTrapOneShot"                      , 0                                                },
	{ 533, "GetTrapCreator"                      , 0                                                },
	{ 534, "GetTrapKeyTag"                       , 0                                                },
	{ 535, "GetTrapDisarmDC"                     , 0                                                },
	{ 536, "GetTrapDetectDC"                     , 0                                                },
	{ 537, "GetLockKeyRequired"                  , 0                                                },
	{ 538, "GetLockKeyTag"                       , 0                                                },
	{ 539, "GetLockLockable"                     , 0                                                },
	{ 540, "GetLockUnlockDC"                     , 0                                                },
	{ 541, "GetLockLockDC"                       , 0                                                },
	{ 542, "GetPCLevellingUp"                    , 0                                                },
	{ 543, "GetHasFeatEffect"                    , 0                                                },
	{ 544, "SetPlaceableIllumination"            , 0                                                },
	{ 545, "GetPlaceableIllumination"            , 0                                                },
	{ 546, "GetIsPlaceableObjectActionPossible"  , 0                                                },
	{ 547, "DoPlaceableObjectAction"             , 0                                                },
	{ 548, "GetFirstPC"                          , &Functions::getFirstPC                           },
	{ 549, "GetNextPC"                           , &Functions::getNextPC                            },
	{ 550, "SetTrapDetectedBy"                   , 0                                                },
	{ 551, "GetIsTrapped"                        , 0                                                },
	{ 552, "EffectTurnResistanceDecrease"        , 0                                                },
	{ 553, "EffectTurnResistanceIncrease"        , 0                                                },
	{ 554, "PopUpDeathGUIPanel"                  , 0                                                },
	{ 555, "SetTrapDisabled"                     , 0                                                },
	{ 556, "GetLastHostileActor"                 , 0                                                },
	{ 557, "ExportAllCharacters"                 , 0                                                },
	{ 558, "MusicBackgroundGetDayTrack"          , &Functions::musicBackgroundGetDayTrack           },
	{ 559, "MusicBackgroundGetNightTrack"        , &Functions::musicBackgroundGetNightTrack         },
	{ 560, "WriteTimestampedLogEntry"            , &Functions::writeTimestampedLogEntry             },
	{ 561, "GetModuleName"                       , 0                                                },
	{ 562, "GetFactionLeader"                    , 0                                                },
	{ 563, "SendMessageToAllDMs"                 , 0                                                },
	{ 564, "EndGame"                             , &Functions::endGame                              },
	{ 565, "BootPC"                              , 0                                                },
	{ 566, "ActionCounterSpell"                  , 0                                                },
	{ 567, "AmbientSoundSetDayVolume"            , 0                                                },
	{ 568, "AmbientSoundSetNightVolume"          , 0                                                },
	{ 569, "MusicBackgroundGetBattleTrack"       , 0                                                },
	{ 570, "GetHasInventory"                     , 0                                                },
	{ 571, "GetStrRefSoundDuration"              , 0                                                },
	{ 572, "AddToParty"                          , 0                                                },
	{ 573, "RemoveFromParty"                     , 0                                                },
	{ 574, "GetStealthMode"                      , 0                                                },
	{ 575, "GetDetectMode"                       , 0                                                },
	{ 576, "GetDefensiveCastingMode"             , 0                                                },
	{ 577, "GetAppearanceType"                   , 0                                                },
	{ 578, "SpawnScriptDebugger"                 , 0                                                },
	{ 579, "GetModuleItemAcquiredStackSize"      , 0                                                },
	{ 580, "DecrementRemainingFeatUses"          , 0                                                },
	{ 581, "DecrementRemainingSpellUses"         , 0                                                },
	{ 582, "GetResRef"                           , 0                                                },
	{ 583, "EffectPetrify"                       , 0                                                },
	{ 584, "CopyItem"                            , 0                                                },
	{ 585, "EffectCutsceneParalyze"              , 0                                                },
	{ 586, "GetDroppableFlag"                    , 0                                                },
	{ 587, "GetUseableFlag"                      , 0                                                },
	{ 588, "GetStolenFlag"                       , 0                                                },
	{ 589, "SetCampaignFloat"                    , 0                                                },
	{ 590, "SetCampaignInt"                      , 0                                                },
	{ 591, "SetCampaignVector"                   , 0                                                },
	{ 592, "SetCampaignLocation"                 , 0                                                },
	{ 593, "SetCampaignString"                   , 0                                                },
	{ 594, "DestroyCampaignDatabase"             , 0                                                },
	{ 595, "GetCampaignFloat"                    , 0                                                },
	{ 596, "GetCampaignInt"                      , 0                                                },
	{ 597, "GetCampaignVector"                   , 0                                                },
	{ 598, "GetCampaignLocation"                 , 0                                                },
	{ 599, "GetCampaignString"                   , 0                                                },
	{ 600, "CopyObject"                          , 0                                                },
	{ 601, "DeleteCampaignVariable"              , 0                                                },
	{ 602, "StoreCampaignObject"                 , 0                                                },
	{ 603, "RetrieveCampaignObject"              , 0                                                },
	{ 604, "EffectCutsceneDominated"             , 0                                                },
	{ 605, "GetItemStackSize"                    , 0                                                },
	{ 606, "SetItemStackSize"                    , 0                                                },
	{ 607, "GetItemCharges"                      , 0                                                },
	{ 608, "SetItemCharges"                      , 0                                                },
	{ 609, "AddItemProperty"                     , 0                                                },
	{ 610, "RemoveItemProperty"                  , 0                                                },
	{ 611, "GetIsItemPropertyValid"              , 0                                                },
	{ 612, "GetFirstItemProperty"                , 0                                                },
	{ 613, "GetNextItemProperty"                 , 0                                                },
	{ 614, "GetItemPropertyType"                 , 0                                                },
	{ 615, "GetItemPropertyDurationType"         , 0                                                },
	{ 616, "ItemPropertyAbilityBonus"            , 0                                                },
	{ 617, "ItemPropertyACBonus"                 , 0                                                },
	{ 618, "ItemPropertyACBonusVsAlign"          , 0                                                },
	{ 619, "ItemPropertyACBonusVsDmgType"        , 0                                                },
	{ 620, "ItemPropertyACBonusVsRace"           , 0                                                },
	{ 621, "ItemPropertyACBonusVsSAlign"         , 0                                                },
	{ 622, "ItemPropertyEnhancementBonus"        , 0                                                },
	{ 623, "ItemPropertyEnhancementBonusVsAlign" , 0                                                },
	{ 624, "ItemPropertyEnhancementBonusVsRace"  , 0                                                },
	{ 625, "ItemPropertyEnhancementBonusVsSAlign", 0                                                },
	{ 626, "ItemPropertyEnhancementPenalty"      , 0                                                },
	{ 627, "ItemPropertyWeightReduction"         , 0                                                },
	{ 628, "ItemPropertyBonusFeat"               , 0                                                },
	{ 629, "ItemPropertyBonusLevelSpell"         , 0                                                },
	{ 630, "ItemPropertyCastSpell"               , 0                                                },
	{ 631, "ItemPropertyDamageBonus"             , 0                                                },
	{ 632, "ItemPropertyDamageBonusVsAlign"      , 0                                                },
	{ 633, "ItemPropertyDamageBonusVsRace"       , 0                                                },
	{ 634, "ItemPropertyDamageBonusVsSAlign"     , 0                                                },
	{ 635, "ItemPropertyDamageImmunity"          , 0                                                },
	{ 636, "ItemPropertyDamagePenalty"           , 0                                                },
	{ 637, "ItemPropertyDamageReduction"         , 0                                                },
	{ 638, "ItemPropertyDamageResistance"        , 0                                                },
	{ 639, "ItemPropertyDamageVulnerability"     , 0                                                },
	{ 640, "ItemPropertyDarkvision"              , 0                                                },
	{ 641, "ItemPropertyDecreaseAbility"         , 0                                                },
	{ 642, "ItemPropertyDecreaseAC"              , 0                                                },
	{ 643, "ItemPropertyDecreaseSkill"           , 0                                                },
	{ 644, "ItemPropertyContainerReducedWeight"  , 0                                                },
	{ 645, "ItemPropertyExtraMeleeDamageType"    , 0                                                },
	{ 646, "ItemPropertyExtraRangeDamageType"    , 0                                                },
	{ 647, "ItemPropertyHaste"                   , 0                                                },
	{ 648, "ItemPropertyHolyAvenger"             , 0                                                },
	{ 649, "ItemPropertyImmunityMisc"            , 0                                                },
	{ 650, "ItemPropertyImprovedEvasion"         , 0                                                },
	{ 651, "ItemPropertyBonusSpellResistance"    , 0                                                },
	{ 652, "ItemPropertyBonusSavingThrowVsX"     , 0                                                },
	{ 653, "ItemPropertyBonusSavingThrow"        , 0                                                },
	{ 654, "ItemPropertyKeen"                    , 0                                                },
	{ 655, "ItemPropertyLight"                   , 0                                                },
	{ 656, "ItemPropertyMaxRangeStrengthMod"     , 0                                                },
	{ 657, "ItemPropertyNoDamage"                , 0                                                },
	{ 658, "ItemPropertyOnHitProps"              , 0                                                },
	{ 659, "ItemPropertyReducedSavingThrowVsX"   , 0                                                },
	{ 660, "ItemPropertyReducedSavingThrow"      , 0                                                },
	{ 661, "ItemPropertyRegeneration"            , 0                                                },
	{ 662, "ItemPropertySkillBonus"              , 0                                                },
	{ 663, "ItemPropertySpellImmunitySpecific"   , 0                                                },
	{ 664, "ItemPropertySpellImmunitySchool"     , 0                                                },
	{ 665, "ItemPropertyThievesTools"            , 0                                                },
	{ 666, "ItemPropertyAttackBonus"             , 0                                                },
	{ 667, "ItemPropertyAttackBonusVsAlign"      , 0                                                },
	{ 668, "ItemPropertyAttackBonusVsRace"       , 0                                                },
	{ 669, "ItemPropertyAttackBonusVsSAlign"     , 0                                                },
	{ 670, "ItemPropertyAttackPenalty"           , 0                                                },
	{ 671, "ItemPropertyUnlimitedAmmo"           , 0                                                },
	{ 672, "ItemPropertyLimitUseByAlign"         , 0                                                },
	{ 673, "ItemPropertyLimitUseByClass"         , 0                                                },
	{ 674, "ItemPropertyLimitUseByRace"          , 0                                                },
	{ 675, "ItemPropertyLimitUseBySAlign"        , 0                                                },
	{ 676, "BadBadReplaceMeThisDoesNothing"      , 0                                                },
	{ 677, "ItemPropertyVampiricRegeneration"    , 0                                                },
	{ 678, "ItemPropertyTrap"                    , 0                                                },
	{ 679, "ItemPropertyTrueSeeing"              , 0                                                },
	{ 680, "ItemPropertyOnMonsterHitProperties"  , 0                                                },
	{ 681, "ItemPropertyTurnResistance"          , 0                                                },
	{ 682, "ItemPropertyMassiveCritical"         , 0                                                },
	{ 683, "ItemPropertyFreeAction"              , 0                                                },
	{ 684, "ItemPropertyMonsterDamage"           , 0                                                },
	{ 685, "ItemPropertyImmunityToSpellLevel"    , 0                                                },
	{ 686, "ItemPropertySpecialWalk"             , 0                                                },
	{ 687, "ItemPropertyHealersKit"              , 0                                                },
	{ 688, "ItemPropertyWeightIncrease"          , 0                                                },
	{ 689, "GetIsSkillSuccessful"                , 0                                                },
	{ 690, "EffectSpellFailure"                  , 0                                                },
	{ 691, "SpeakStringByStrRef"                 , &Functions::speakStringByStrRef                  },
	{ 692, "SetCutsceneMode"                     , 0                                                },
	{ 693, "GetLastPCToCancelCutscene"           , 0                                                },
	{ 694, "GetDialogSoundLength"                , 0                                                },
	{ 695, "FadeFromBlack"                       , 0                                                },
	{ 696, "FadeToBlack"                         , 0                                                },
	{ 697, "StopFade"                            , 0                                                },
	{ 698, "BlackScreen"                         , 0                                                },
	{ 699, "GetBaseAttackBonus"                  , 0                                                },
	{ 700, "SetImmortal"                         , 0                                                },
	{ 701, "OpenInventory"                       , 0                                                },
	{ 702, "StoreCameraFacing"                   , 0                                                },
	{ 703, "RestoreCameraFacing"                 , 0                                                },
	{ 704, "LevelUpHenchman"                     , 0                                                },
	{ 705, "SetDroppableFlag"                    , 0                                                },
	{ 706, "GetWeight"                           , 0                                                },
	{ 707, "GetModuleItemAcquiredBy"             , 0                                                },
	{ 708, "GetImmortal"                         , 0                                                },
	{ 709, "DoWhirlwindAttack"                   , 0                                                },
	{ 710, "Get2DAString"                        , &Functions::get2DAString                         },
	{ 711, "EffectEthereal"                      , 0                                                },
	{ 712, "GetAILevel"                          , 0                                                },
	{ 713, "SetAILevel"                          , 0                                                },
	{ 714, "GetIsPossessedFamiliar"              , 0                                                },
	{ 715, "UnpossessFamiliar"                   , 0                                                },
	{ 716, "GetIsAreaInterior"                   , 0                                                },
	{ 717, "SendMessageToPCByStrRef"             , 0                                                },
	{ 718, "IncrementRemainingFeatUses"          , 0                                                },
	{ 719, "ExportSingleCharacter"               , 0                                                },
	{ 720, "PlaySoundByStrRef"                   , &Functions::playSoundByStrRef                    },
	{ 721, "SetSubRace"                          , 0                                                },
	{ 722, "SetDeity"                            , 0                                                },
	{ 723, "GetIsDMPossessed"                    , 0                                                },
	{ 724, "GetWeather"                          , 0                                                },
	{ 725, "GetIsAreaNatural"                    , 0                                                },
	{ 726, "GetIsAreaAboveGround"                , 0                                                },
	{ 727, "GetPCItemLastEquipped"               , 0                                                },
	{ 728, "GetPCItemLastEquippedBy"             , 0                                                },
	{ 729, "GetPCItemLastUnequipped"             , 0                                                },
	{ 730, "GetPCItemLastUnequippedBy"           , 0                                                },
	{ 731, "CopyItemAndModify"                   , 0                                                },
	{ 732, "GetItemAppearance"                   , 0                                                },
	{ 733, "ItemPropertyOnHitCastSpell"          , 0                                                },
	{ 734, "GetItemPropertySubType"              , 0                                                },
	{ 735, "GetActionMode"                       , 0                                                },
	{ 736, "SetActionMode"                       , 0                                                },
	{ 737, "GetArcaneSpellFailure"               , 0                                                },
	{ 738, "ActionExamine"                       , 0                                                },
	{ 739, "ItemPropertyVisualEffect"            , 0                                                },
	{ 740, "SetLootable"                         , 0                                                },
	{ 741, "GetLootable"                         , 0                                                },
	{ 742, "GetCutsceneCameraMoveRate"           , 0                                                },
	{ 743, "SetCutsceneCameraMoveRate"           , 0                                                },
	{ 744, "GetItemCursedFlag"                   , 0                                                },
	{ 745, "SetItemCursedFlag"                   , 0                                                },
	{ 746, "SetMaxHenchmen"                      , 0                                                },
	{ 747, "GetMaxHenchmen"                      , 0                                                },
	{ 748, "GetAssociateType"                    , 0                                                },
	{ 749, "GetSpellResistance"                  , 0                                                },
	{ 750, "DayToNight"                          , 0                                                },
	{ 751, "NightToDay"                          , 0                                                },
	{ 752, "LineOfSightObject"                   , 0                                                },
	{ 753, "LineOfSightVector"                   , 0                                                },
	{ 754, "GetLastSpellCastClass"               , 0                                                },
	{ 755, "SetBaseAttackBonus"                  , 0                                                },
	{ 756, "RestoreBaseAttackBonus"              , 0                                                },
	{ 757, "EffectCutsceneGhost"                 , 0                                                },
	{ 758, "ItemPropertyArcaneSpellFailure"      , 0                                                },
	{ 759, "GetStoreGold"                        , 0                                                },
	{ 760, "SetStoreGold"                        , 0                                                },
	{ 761, "GetStoreMaxBuyPrice"                 , 0                                                },
	{ 762, "SetStoreMaxBuyPrice"                 , 0                                                },
	{ 763, "GetStoreIdentifyCost"                , 0                                                },
	{ 764, "SetStoreIdentifyCost"                , 0                                                },
	{ 765, "SetCreatureAppearanceType"           , 0                                                },
	{ 766, "GetCreatureStartingPackage"          , 0                                                },
	{ 767, "EffectCutsceneImmobilize"            , 0                                                },
	{ 768, "GetIsInSubArea"                      , 0                                                },
	{ 769, "GetItemPropertyCostTable"            , 0                                                },
	{ 770, "GetItemPropertyCostTableValue"       , 0                                                },
	{ 771, "GetItemPropertyParam1"               , 0                                                },
	{ 772, "GetItemPropertyParam1Value"          , 0                                                },
	{ 773, "GetIsCreatureDisarmable"             , 0                                                },
	{ 774, "SetStolenFlag"                       , 0                                                },
	{ 775, "ForceRest"                           , 0                                                },
	{ 776, "SetCameraHeight"                     , 0                                                },
	{ 777, "SetSkyBox"                           , 0                                                },
	{ 778, "GetPhenoType"                        , 0                                                },
	{ 779, "SetPhenoType"                        , 0                                                },
	{ 780, "SetFogColor"                         , 0                                                },
	{ 781, "GetCutsceneMode"                     , 0                                                },
	{ 782, "GetSkyBox"                           , 0                                                },
	{ 783, "GetFogColor"                         , 0                                                },
	{ 784, "SetFogAmount"                        , 0                                                },
	{ 785, "GetFogAmount"                        , 0                                                },
	{ 786, "GetPickpocketableFlag"               , 0                                                },
	{ 787, "SetPickpocketableFlag"               , 0                                                },
	{ 788, "GetFootstepType"                     , 0                                                },
	{ 789, "SetFootstepType"                     , 0                                                },
	{ 790, "GetCreatureWingType"                 , 0                                                },
	{ 791, "SetCreatureWingType"                 , 0                                                },
	{ 792, "GetCreatureBodyPart"                 , 0                                                },
	{ 793, "SetCreatureBodyPart"                 , 0                                                },
	{ 794, "GetCreatureTailType"                 , 0                                                },
	{ 795, "SetCreatureTailType"                 , 0                                                },
	{ 796, "GetHardness"                         , 0                                                },
	{ 797, "SetHardness"                         , 0                                                },
	{ 798, "SetLockKeyRequired"                  , 0                                                },
	{ 799, "SetLockKeyTag"                       , 0                                                },
	{ 800, "SetLockLockable"                     , 0                                                },
	{ 801, "SetLockUnlockDC"                     , 0                                                },
	{ 802, "SetLockLockDC"                       , 0                                                },
	{ 803, "SetTrapDisarmable"                   , 0                                                },
	{ 804, "SetTrapDetectable"                   , 0                                                },
	{ 805, "SetTrapOneShot"                      , 0                                                },
	{ 806, "SetTrapKeyTag"                       , 0                                                },
	{ 807, "SetTrapDisarmDC"                     , 0                                                },
	{ 808, "SetTrapDetectDC"                     , 0                                                },
	{ 809, "CreateTrapAtLocation"                , 0                                                },
	{ 810, "CreateTrapOnObject"                  , 0                                                },
	{ 811, "SetWillSavingThrow"                  , 0                                                },
	{ 812, "SetReflexSavingThrow"                , 0                                                },
	{ 813, "SetFortitudeSavingThrow"             , 0                                                },
	{ 814, "GetTilesetResRef"                    , 0                                                },
	{ 815, "GetTrapRecoverable"                  , 0                                                },
	{ 816, "SetTrapRecoverable"                  , 0                                                },
	{ 817, "GetModuleXPScale"                    , 0                                                },
	{ 818, "SetModuleXPScale"                    , 0                                                },
	{ 819, "GetKeyRequiredFeedback"              , 0                                                },
	{ 820, "SetKeyRequiredFeedback"              , 0                                                },
	{ 821, "GetTrapActive"                       , 0                                                },
	{ 822, "SetTrapActive"                       , 0                                                },
	{ 823, "LockCameraPitch"                     , 0                                                },
	{ 824, "LockCameraDistance"                  , 0                                                },
	{ 825, "LockCameraDirection"                 , 0                                                },
	{ 826, "GetPlaceableLastClickedBy"           , 0                                                },
	{ 827, "GetInfiniteFlag"                     , 0                                                },
	{ 828, "SetInfiniteFlag"                     , 0                                                },
	{ 829, "GetAreaSize"                         , 0                                                },
	{ 830, "SetName"                             , 0                                                },
	{ 831, "GetPortraitId"                       , 0                                                },
	{ 832, "SetPortraitId"                       , 0                                                },
	{ 833, "GetPortraitResRef"                   , 0                                                },
	{ 834, "SetPortraitResRef"                   , 0                                                },
	{ 835, "SetUseableFlag"                      , 0                                                },
	{ 836, "GetDescription"                      , 0                                                },
	{ 837, "SetDescription"                      , 0                                                },
	{ 838, "GetPCChatSpeaker"                    , 0                                                },
	{ 839, "GetPCChatMessage"                    , 0                                                },
	{ 840, "GetPCChatVolume"                     , 0                                                },
	{ 841, "SetPCChatMessage"                    , 0                                                },
	{ 842, "SetPCChatVolume"                     , 0                                                },
	{ 843, "GetColor"                            , 0                                                },
	{ 844, "SetColor"                            , 0                                                },
	{ 845, "ItemPropertyMaterial"                , 0                                                },
	{ 846, "ItemPropertyQuality"                 , 0                                                },
	{ 847, "ItemPropertyAdditional"              , 0                                                }
};

/** The table defining the signature (return type and type of parameters) of each engine function. */
const Functions::FunctionSignature Functions::kFunctionSignatures[] = {
	{   0, kTypeInt       , { kTypeInt } },
	{   1, kTypeVoid      , { kTypeString } },
	{   2, kTypeVoid      , { kTypeFloat, kTypeInt, kTypeInt } },
	{   3, kTypeString    , { kTypeFloat, kTypeInt, kTypeInt } },
	{   4, kTypeVoid      , { kTypeInt } },
	{   5, kTypeVoid      , { kTypeObject } },
	{   6, kTypeVoid      , { kTypeObject, kTypeScriptState } },
	{   7, kTypeVoid      , { kTypeFloat, kTypeScriptState } },
	{   8, kTypeVoid      , { kTypeString, kTypeObject } },
	{   9, kTypeVoid      , { kTypeInt } },
	{  10, kTypeVoid      , { kTypeFloat } },
	{  11, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{  12, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  13, kTypeInt       , { } },
	{  14, kTypeInt       , { } },
	{  15, kTypeInt       , { } },
	{  16, kTypeInt       , { } },
	{  17, kTypeInt       , { } },
	{  18, kTypeInt       , { } },
	{  19, kTypeInt       , { } },
	{  20, kTypeVoid      , { } },
	{  21, kTypeVoid      , { kTypeEngineType, kTypeInt } },
	{  22, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  23, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  24, kTypeObject    , { kTypeObject } },
	{  25, kTypeObject    , { } },
	{  26, kTypeObject    , { } },
	{  27, kTypeVector    , { kTypeObject } },
	{  28, kTypeFloat     , { kTypeObject } },
	{  29, kTypeObject    , { kTypeObject } },
	{  30, kTypeObject    , { kTypeObject, kTypeString } },
	{  31, kTypeObject    , { kTypeString, kTypeObject, kTypeInt, kTypeString } },
	{  32, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  33, kTypeVoid      , { kTypeObject } },
	{  34, kTypeVoid      , { kTypeObject } },
	{  35, kTypeVoid      , { kTypeObject } },
	{  36, kTypeObject    , { kTypeObject } },
	{  37, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  38, kTypeObject    , { kTypeInt, kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  39, kTypeVoid      , { kTypeString, kTypeInt } },
	{  40, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{  41, kTypeFloat     , { kTypeObject } },
	{  42, kTypeInt       , { kTypeObject } },
	{  43, kTypeVoid      , { kTypeObject } },
	{  44, kTypeVoid      , { kTypeObject } },
	{  45, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  46, kTypeVoid      , { kTypeString } },
	{  47, kTypeObject    , { } },
	{  48, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  49, kTypeInt       , { kTypeObject } },
	{  50, kTypeInt       , { kTypeObject } },
	{  51, kTypeInt       , { kTypeObject, kTypeString } },
	{  52, kTypeFloat     , { kTypeObject, kTypeString } },
	{  53, kTypeString    , { kTypeObject, kTypeString } },
	{  54, kTypeObject    , { kTypeObject, kTypeString } },
	{  55, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{  56, kTypeVoid      , { kTypeObject, kTypeString, kTypeFloat } },
	{  57, kTypeVoid      , { kTypeObject, kTypeString, kTypeString } },
	{  58, kTypeVoid      , { kTypeObject, kTypeString, kTypeObject } },
	{  59, kTypeInt       , { kTypeString } },
	{  60, kTypeString    , { kTypeString } },
	{  61, kTypeString    , { kTypeString } },
	{  62, kTypeString    , { kTypeString, kTypeInt } },
	{  63, kTypeString    , { kTypeString, kTypeInt } },
	{  64, kTypeString    , { kTypeString, kTypeString, kTypeInt } },
	{  65, kTypeString    , { kTypeString, kTypeInt, kTypeInt } },
	{  66, kTypeInt       , { kTypeString, kTypeString, kTypeInt } },
	{  67, kTypeFloat     , { kTypeFloat } },
	{  68, kTypeFloat     , { kTypeFloat } },
	{  69, kTypeFloat     , { kTypeFloat } },
	{  70, kTypeFloat     , { kTypeFloat } },
	{  71, kTypeFloat     , { kTypeFloat } },
	{  72, kTypeFloat     , { kTypeFloat } },
	{  73, kTypeFloat     , { kTypeFloat } },
	{  74, kTypeFloat     , { kTypeFloat } },
	{  75, kTypeFloat     , { kTypeFloat, kTypeFloat } },
	{  76, kTypeFloat     , { kTypeFloat } },
	{  77, kTypeInt       , { kTypeInt } },
	{  78, kTypeEngineType, { kTypeInt } },
	{  79, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  80, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  81, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  82, kTypeEngineType, { } },
	{  83, kTypeEngineType, { kTypeString, kTypeInt, kTypeFloat, kTypeInt } },
	{  84, kTypeInt       , { kTypeObject } },
	{  85, kTypeEngineType, { kTypeObject } },
	{  86, kTypeEngineType, { kTypeObject } },
	{  87, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{  88, kTypeInt       , { kTypeEngineType } },
	{  89, kTypeInt       , { kTypeEngineType } },
	{  90, kTypeInt       , { kTypeEngineType } },
	{  91, kTypeObject    , { kTypeEngineType } },
	{  92, kTypeString    , { kTypeInt } },
	{  93, kTypeObject    , { kTypeObject } },
	{  94, kTypeObject    , { kTypeObject } },
	{  95, kTypeInt       , { kTypeInt } },
	{  96, kTypeInt       , { kTypeInt } },
	{  97, kTypeInt       , { kTypeInt } },
	{  98, kTypeInt       , { kTypeInt } },
	{  99, kTypeInt       , { kTypeInt } },
	{ 100, kTypeInt       , { kTypeInt } },
	{ 101, kTypeInt       , { kTypeInt } },
	{ 102, kTypeInt       , { kTypeInt } },
	{ 103, kTypeInt       , { kTypeInt } },
	{ 104, kTypeFloat     , { kTypeVector } },
	{ 105, kTypeInt       , { } },
	{ 106, kTypeInt       , { kTypeObject } },
	{ 107, kTypeInt       , { kTypeObject } },
	{ 108, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 109, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 110, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 111, kTypeInt       , { } },
	{ 112, kTypeEngineType, { kTypeEngineType } },
	{ 113, kTypeEngineType, { kTypeEngineType } },
	{ 114, kTypeEngineType, { kTypeEngineType } },
	{ 115, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 116, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 117, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 118, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 119, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 120, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 121, kTypeFloat     , { kTypeInt } },
	{ 122, kTypeFloat     , { kTypeInt } },
	{ 123, kTypeFloat     , { kTypeInt } },
	{ 124, kTypeInt       , { kTypeObject } },
	{ 125, kTypeInt       , { kTypeObject } },
	{ 126, kTypeInt       , { kTypeObject } },
	{ 127, kTypeInt       , { kTypeObject } },
	{ 128, kTypeObject    , { kTypeInt, kTypeFloat, kTypeEngineType, kTypeInt, kTypeInt, kTypeVector } },
	{ 129, kTypeObject    , { kTypeInt, kTypeFloat, kTypeEngineType, kTypeInt, kTypeInt, kTypeVector } },
	{ 130, kTypeEngineType, { } },
	{ 131, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 132, kTypeEngineType, { kTypeInt } },
	{ 133, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 134, kTypeEngineType, { } },
	{ 135, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 136, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 137, kTypeVector    , { kTypeVector } },
	{ 138, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 139, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 140, kTypeInt       , { kTypeObject } },
	{ 141, kTypeVoid      , { kTypeVector, kTypeInt } },
	{ 142, kTypeVector    , { kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 143, kTypeVoid      , { kTypeVector } },
	{ 144, kTypeVector    , { kTypeFloat } },
	{ 145, kTypeFloat     , { kTypeVector } },
	{ 146, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 147, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 148, kTypeEngineType, { } },
	{ 149, kTypeEngineType, { kTypeInt } },
	{ 150, kTypeEngineType, { } },
	{ 151, kTypeFloat     , { kTypeObject, kTypeObject } },
	{ 152, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{ 153, kTypeEngineType, { kTypeObject, kTypeString } },
	{ 154, kTypeEngineType, { } },
	{ 155, kTypeObject    , { kTypeInt, kTypeObject } },
	{ 156, kTypeEngineType, { } },
	{ 157, kTypeEngineType, { } },
	{ 158, kTypeEngineType, { } },
	{ 159, kTypeEngineType, { } },
	{ 160, kTypeEngineType, { } },
	{ 161, kTypeEngineType, { } },
	{ 162, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 163, kTypeInt       , { kTypeObject } },
	{ 164, kTypeEngineType, { kTypeInt, kTypeFloat } },
	{ 165, kTypeEngineType, { kTypeInt } },
	{ 166, kTypeInt       , { kTypeObject } },
	{ 167, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 168, kTypeString    , { kTypeObject } },
	{ 169, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 170, kTypeInt       , { kTypeEngineType } },
	{ 171, kTypeEngineType, { kTypeInt, kTypeString, kTypeString, kTypeString } },
	{ 172, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 173, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 174, kTypeInt       , { kTypeObject } },
	{ 175, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 176, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 177, kTypeInt       , { kTypeString, kTypeString } },
	{ 178, kTypeString    , { kTypeInt } },
	{ 179, kTypeInt       , { } },
	{ 180, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 181, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 182, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 183, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 184, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 185, kTypeInt       , { kTypeObject } },
	{ 186, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 187, kTypeInt       , { kTypeObject } },
	{ 188, kTypeInt       , { kTypeObject } },
	{ 189, kTypeInt       , { kTypeObject } },
	{ 190, kTypeInt       , { kTypeObject } },
	{ 191, kTypeInt       , { kTypeObject } },
	{ 192, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 193, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 194, kTypeVoid      , { kTypeObject } },
	{ 195, kTypeInt       , { } },
	{ 196, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 197, kTypeObject    , { kTypeString } },
	{ 198, kTypeObject    , { kTypeObject } },
	{ 199, kTypeEngineType, { kTypeEngineType, kTypeEngineType } },
	{ 200, kTypeObject    , { kTypeString, kTypeInt } },
	{ 201, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 202, kTypeVoid      , { kTypeFloat } },
	{ 203, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 204, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt } },
	{ 205, kTypeVoid      , { } },
	{ 206, kTypeVoid      , { } },
	{ 207, kTypeEngineType, { kTypeInt, kTypeObject, kTypeInt, kTypeInt } },
	{ 208, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 209, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 210, kTypeObject    , { kTypeObject } },
	{ 211, kTypeObject    , { kTypeObject } },
	{ 212, kTypeEngineType, { kTypeInt } },
	{ 213, kTypeEngineType, { kTypeObject } },
	{ 214, kTypeVoid      , { kTypeEngineType } },
	{ 215, kTypeEngineType, { kTypeObject, kTypeVector, kTypeFloat } },
	{ 216, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeEngineType, kTypeFloat } },
	{ 217, kTypeInt       , { kTypeObject } },
	{ 218, kTypeFloat     , { kTypeFloat } },
	{ 219, kTypeFloat     , { kTypeFloat } },
	{ 220, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeObject, kTypeFloat } },
	{ 221, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 222, kTypeEngineType, { } },
	{ 223, kTypeVector    , { kTypeEngineType } },
	{ 224, kTypeObject    , { kTypeEngineType } },
	{ 225, kTypeFloat     , { kTypeEngineType } },
	{ 226, kTypeObject    , { kTypeInt, kTypeInt, kTypeEngineType, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 227, kTypeObject    , { kTypeInt, kTypeObject, kTypeInt } },
	{ 228, kTypeObject    , { kTypeInt, kTypeEngineType, kTypeInt } },
	{ 229, kTypeObject    , { kTypeString, kTypeObject, kTypeInt } },
	{ 230, kTypeFloat     , { kTypeInt } },
	{ 231, kTypeInt       , { kTypeFloat } },
	{ 232, kTypeInt       , { kTypeString } },
	{ 233, kTypeFloat     , { kTypeString } },
	{ 234, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 235, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 236, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 237, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 238, kTypeObject    , { } },
	{ 239, kTypeString    , { kTypeInt, kTypeInt } },
	{ 240, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 241, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 242, kTypeObject    , { } },
	{ 243, kTypeObject    , { kTypeInt, kTypeString, kTypeEngineType, kTypeInt, kTypeString } },
	{ 244, kTypeEngineType, { kTypeObject, kTypeInt, kTypeInt } },
	{ 245, kTypeObject    , { } },
	{ 246, kTypeInt       , { } },
	{ 247, kTypeInt       , { } },
	{ 248, kTypeInt       , { } },
	{ 249, kTypeString    , { kTypeInt } },
	{ 250, kTypeEngineType, { kTypeInt } },
	{ 251, kTypeEngineType, { kTypeInt } },
	{ 252, kTypeEngineType, { } },
	{ 253, kTypeString    , { kTypeObject, kTypeInt } },
	{ 254, kTypeObject    , { } },
	{ 255, kTypeInt       , { kTypeString, kTypeObject } },
	{ 256, kTypeObject    , { } },
	{ 257, kTypeInt       , { } },
	{ 258, kTypeInt       , { } },
	{ 259, kTypeInt       , { } },
	{ 260, kTypeObject    , { } },
	{ 261, kTypeInt       , { } },
	{ 262, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 263, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 264, kTypeObject    , { kTypeObject } },
	{ 265, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 266, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 267, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 268, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 269, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 270, kTypeEngineType, { } },
	{ 271, kTypeEngineType, { } },
	{ 272, kTypeString    , { kTypeObject } },
	{ 273, kTypeEngineType, { kTypeInt } },
	{ 274, kTypeInt       , { kTypeObject, kTypeInt, kTypeObject } },
	{ 275, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 276, kTypeInt       , { kTypeObject } },
	{ 277, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 278, kTypeInt       , { kTypeObject } },
	{ 279, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 280, kTypeInt       , { kTypeObject } },
	{ 281, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 282, kTypeObject    , { } },
	{ 283, kTypeObject    , { } },
	{ 284, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 285, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 286, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 287, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 288, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt, kTypeObject } },
	{ 289, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 290, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 291, kTypeObject    , { } },
	{ 292, kTypeObject    , { } },
	{ 293, kTypeObject    , { } },
	{ 294, kTypeVoid      , { kTypeScriptState } },
	{ 295, kTypeEngineType, { } },
	{ 296, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 297, kTypeInt       , { kTypeObject } },
	{ 298, kTypeFloat     , { kTypeEngineType, kTypeEngineType } },
	{ 299, kTypeInt       , { kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 300, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{ 301, kTypeEngineType, { kTypeInt } },
	{ 302, kTypeEngineType, { kTypeInt } },
	{ 303, kTypeEngineType, { kTypeInt } },
	{ 304, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 305, kTypeInt       , { kTypeEngineType } },
	{ 306, kTypeInt       , { kTypeEngineType, kTypeObject } },
	{ 307, kTypeEngineType, { kTypeInt, kTypeObject } },
	{ 308, kTypeEngineType, { kTypeInt, kTypeInt, kTypeObject } },
	{ 309, kTypeVoid      , { kTypeEngineType, kTypeObject } },
	{ 310, kTypeVoid      , { kTypeEngineType, kTypeEngineType } },
	{ 311, kTypeInt       , { kTypeObject } },
	{ 312, kTypeInt       , { kTypeObject } },
	{ 313, kTypeVoid      , { kTypeEngineType } },
	{ 314, kTypeEngineType, { kTypeInt } },
	{ 315, kTypeInt       , { kTypeInt, kTypeObject, kTypeInt } },
	{ 316, kTypeObject    , { kTypeObject } },
	{ 317, kTypeInt       , { kTypeObject } },
	{ 318, kTypeInt       , { kTypeObject } },
	{ 319, kTypeObject    , { kTypeObject } },
	{ 320, kTypeInt       , { kTypeObject } },
	{ 321, kTypeInt       , { kTypeObject } },
	{ 322, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 323, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 324, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 325, kTypeInt       , { kTypeObject } },
	{ 326, kTypeObject    , { } },
	{ 327, kTypeVoid      , { kTypeObject } },
	{ 328, kTypeObject    , { kTypeObject } },
	{ 329, kTypeVoid      , { kTypeObject } },
	{ 330, kTypeObject    , { } },
	{ 331, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 332, kTypeInt       , { kTypeObject } },
	{ 333, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 334, kTypeVoid      , { kTypeObject } },
	{ 335, kTypeVoid      , { kTypeObject } },
	{ 336, kTypeObject    , { } },
	{ 337, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 338, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 339, kTypeObject    , { kTypeObject } },
	{ 340, kTypeObject    , { kTypeObject } },
	{ 341, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 342, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 343, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 344, kTypeInt       , { kTypeInt } },
	{ 345, kTypeInt       , { } },
	{ 346, kTypeObject    , { kTypeObject } },
	{ 347, kTypeObject    , { } },
	{ 348, kTypeObject    , { } },
	{ 349, kTypeObject    , { } },
	{ 350, kTypeObject    , { } },
	{ 351, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 352, kTypeInt       , { } },
	{ 353, kTypeObject    , { } },
	{ 354, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 355, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 356, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 357, kTypeEngineType, { kTypeEngineType } },
	{ 358, kTypeInt       , { kTypeObject } },
	{ 359, kTypeInt       , { kTypeEngineType } },
	{ 360, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 361, kTypeObject    , { } },
	{ 362, kTypeInt       , { kTypeEngineType } },
	{ 363, kTypeInt       , { kTypeEngineType } },
	{ 364, kTypeObject    , { kTypeInt, kTypeObject, kTypeInt } },
	{ 365, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 366, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 367, kTypeVoid      , { kTypeString, kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 368, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt, kTypeInt } },
	{ 369, kTypeString    , { kTypeObject, kTypeInt } },
	{ 370, kTypeString    , { kTypeObject } },
	{ 371, kTypeString    , { kTypeObject } },
	{ 372, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 373, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 374, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 375, kTypeObject    , { } },
	{ 376, kTypeObject    , { } },
	{ 377, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 378, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{ 379, kTypeEngineType, { } },
	{ 380, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 381, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 382, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 383, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeFloat } },
	{ 384, kTypeInt       , { kTypeString } },
	{ 385, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 386, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 387, kTypeEngineType, { kTypeFloat } },
	{ 388, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 389, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 390, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat } },
	{ 391, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat } },
	{ 392, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat } },
	{ 393, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 394, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 395, kTypeInt       , { kTypeObject } },
	{ 396, kTypeString    , { kTypeInt } },
	{ 397, kTypeInt       , { kTypeObject } },
	{ 398, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 399, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 400, kTypeVoid      , { kTypeObject } },
	{ 401, kTypeInt       , { kTypeObject } },
	{ 402, kTypeVoid      , { kTypeInt } },
	{ 403, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 404, kTypeVoid      , { } },
	{ 405, kTypeInt       , { } },
	{ 406, kTypeInt       , { } },
	{ 407, kTypeInt       , { } },
	{ 408, kTypeInt       , { } },
	{ 409, kTypeInt       , { kTypeObject } },
	{ 410, kTypeObject    , { } },
	{ 411, kTypeEngineType, { } },
	{ 412, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 413, kTypeVoid      , { kTypeObject } },
	{ 414, kTypeVoid      , { kTypeObject } },
	{ 415, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 416, kTypeVoid      , { kTypeObject, kTypeVector } },
	{ 417, kTypeVoid      , { kTypeString, kTypeObject } },
	{ 418, kTypeInt       , { kTypeObject } },
	{ 419, kTypeObject    , { } },
	{ 420, kTypeInt       , { kTypeObject } },
	{ 421, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 422, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 423, kTypeInt       , { } },
	{ 424, kTypeEngineType, { kTypeObject, kTypeEngineType, kTypeObject } },
	{ 425, kTypeVoid      , { kTypeObject } },
	{ 426, kTypeVoid      , { kTypeObject } },
	{ 427, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 428, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 429, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 430, kTypeVoid      , { kTypeObject } },
	{ 431, kTypeVoid      , { kTypeObject } },
	{ 432, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 433, kTypeVoid      , { kTypeObject } },
	{ 434, kTypeVoid      , { kTypeObject } },
	{ 435, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 436, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 437, kTypeObject    , { } },
	{ 438, kTypeObject    , { } },
	{ 439, kTypeObject    , { } },
	{ 440, kTypeObject    , { } },
	{ 441, kTypeEngineType, { } },
	{ 442, kTypeObject    , { } },
	{ 443, kTypeInt       , { kTypeObject } },
	{ 444, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 445, kTypeInt       , { kTypeObject } },
	{ 446, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 447, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 448, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 449, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 450, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 451, kTypeEngineType, { kTypeInt } },
	{ 452, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 453, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 454, kTypeEngineType, { kTypeInt } },
	{ 455, kTypeInt       , { kTypeObject } },
	{ 456, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 457, kTypeEngineType, { kTypeInt } },
	{ 458, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 459, kTypeEngineType, { } },
	{ 460, kTypeEngineType, { kTypeInt } },
	{ 461, kTypeEngineType, { } },
	{ 462, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 463, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 464, kTypeEngineType, { kTypeInt } },
	{ 465, kTypeEngineType, { } },
	{ 466, kTypeEngineType, { } },
	{ 467, kTypeEngineType, { } },
	{ 468, kTypeEngineType, { } },
	{ 469, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 470, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 471, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 472, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 473, kTypeEngineType, { kTypeInt } },
	{ 474, kTypeVoid      , { kTypeObject, kTypeString, kTypeString, kTypeString, kTypeInt } },
	{ 475, kTypeInt       , { kTypeObject } },
	{ 476, kTypeVoid      , { } },
	{ 477, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 478, kTypeInt       , { kTypeObject } },
	{ 479, kTypeInt       , { kTypeObject } },
	{ 480, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 481, kTypeEngineType, { kTypeInt } },
	{ 482, kTypeEngineType, { kTypeInt } },
	{ 483, kTypeVoid      , { kTypeObject } },
	{ 484, kTypeVoid      , { kTypeObject } },
	{ 485, kTypeEngineType, { kTypeInt } },
	{ 486, kTypeObject    , { kTypeObject } },
	{ 487, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 488, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 489, kTypeString    , { kTypeObject } },
	{ 490, kTypeString    , { kTypeObject } },
	{ 491, kTypeInt       , { kTypeObject } },
	{ 492, kTypeInt       , { kTypeObject } },
	{ 493, kTypeInt       , { kTypeObject } },
	{ 494, kTypeFloat     , { kTypeObject } },
	{ 495, kTypeInt       , { kTypeObject } },
	{ 496, kTypeInt       , { kTypeObject } },
	{ 497, kTypeInt       , { kTypeObject } },
	{ 498, kTypeInt       , { kTypeObject } },
	{ 499, kTypeString    , { kTypeObject } },
	{ 500, kTypeString    , { kTypeObject } },
	{ 501, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 502, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeInt } },
	{ 503, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 504, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 505, kTypeInt       , { kTypeObject } },
	{ 506, kTypeObject    , { } },
	{ 507, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 508, kTypeInt       , { } },
	{ 509, kTypeVoid      , { kTypeString } },
	{ 510, kTypeEngineType, { kTypeInt, kTypeString, kTypeString, kTypeString, kTypeString } },
	{ 511, kTypeInt       , { kTypeObject } },
	{ 512, kTypeVoid      , { } },
	{ 513, kTypeInt       , { } },
	{ 514, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 515, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 516, kTypeVoid      , { kTypeObject } },
	{ 517, kTypeInt       , { kTypeEngineType } },
	{ 518, kTypeInt       , { kTypeEngineType } },
	{ 519, kTypeInt       , { kTypeEngineType } },
	{ 520, kTypeInt       , { kTypeEngineType } },
	{ 521, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 522, kTypeInt       , { kTypeObject } },
	{ 523, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject } },
	{ 524, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 525, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 526, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt } },
	{ 527, kTypeInt       , { kTypeObject } },
	{ 528, kTypeInt       , { kTypeObject } },
	{ 529, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 530, kTypeInt       , { kTypeObject } },
	{ 531, kTypeInt       , { kTypeObject } },
	{ 532, kTypeInt       , { kTypeObject } },
	{ 533, kTypeObject    , { kTypeObject } },
	{ 534, kTypeString    , { kTypeObject } },
	{ 535, kTypeInt       , { kTypeObject } },
	{ 536, kTypeInt       , { kTypeObject } },
	{ 537, kTypeInt       , { kTypeObject } },
	{ 538, kTypeString    , { kTypeObject } },
	{ 539, kTypeInt       , { kTypeObject } },
	{ 540, kTypeInt       , { kTypeObject } },
	{ 541, kTypeInt       , { kTypeObject } },
	{ 542, kTypeObject    , { } },
	{ 543, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 544, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 545, kTypeInt       , { kTypeObject } },
	{ 546, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 547, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 548, kTypeObject    , { } },
	{ 549, kTypeObject    , { } },
	{ 550, kTypeInt       , { kTypeObject, kTypeObject, kTypeInt } },
	{ 551, kTypeInt       , { kTypeObject } },
	{ 552, kTypeEngineType, { kTypeInt } },
	{ 553, kTypeEngineType, { kTypeInt } },
	{ 554, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeString } },
	{ 555, kTypeVoid      , { kTypeObject } },
	{ 556, kTypeObject    , { kTypeObject } },
	{ 557, kTypeVoid      , { } },
	{ 558, kTypeInt       , { kTypeObject } },
	{ 559, kTypeInt       , { kTypeObject } },
	{ 560, kTypeVoid      , { kTypeString } },
	{ 561, kTypeString    , { } },
	{ 562, kTypeObject    , { kTypeObject } },
	{ 563, kTypeVoid      , { kTypeString } },
	{ 564, kTypeVoid      , { kTypeString } },
	{ 565, kTypeVoid      , { kTypeObject } },
	{ 566, kTypeVoid      , { kTypeObject } },
	{ 567, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 568, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 569, kTypeInt       , { kTypeObject } },
	{ 570, kTypeInt       , { kTypeObject } },
	{ 571, kTypeFloat     , { kTypeInt } },
	{ 572, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 573, kTypeVoid      , { kTypeObject } },
	{ 574, kTypeInt       , { kTypeObject } },
	{ 575, kTypeInt       , { kTypeObject } },
	{ 576, kTypeInt       , { kTypeObject } },
	{ 577, kTypeInt       , { kTypeObject } },
	{ 578, kTypeVoid      , { } },
	{ 579, kTypeInt       , { } },
	{ 580, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 581, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 582, kTypeString    , { kTypeObject } },
	{ 583, kTypeEngineType, { } },
	{ 584, kTypeObject    , { kTypeObject, kTypeObject, kTypeInt } },
	{ 585, kTypeEngineType, { } },
	{ 586, kTypeInt       , { kTypeObject } },
	{ 587, kTypeInt       , { kTypeObject } },
	{ 588, kTypeInt       , { kTypeObject } },
	{ 589, kTypeVoid      , { kTypeString, kTypeString, kTypeFloat, kTypeObject } },
	{ 590, kTypeVoid      , { kTypeString, kTypeString, kTypeInt, kTypeObject } },
	{ 591, kTypeVoid      , { kTypeString, kTypeString, kTypeVector, kTypeObject } },
	{ 592, kTypeVoid      , { kTypeString, kTypeString, kTypeEngineType, kTypeObject } },
	{ 593, kTypeVoid      , { kTypeString, kTypeString, kTypeString, kTypeObject } },
	{ 594, kTypeVoid      , { kTypeString } },
	{ 595, kTypeFloat     , { kTypeString, kTypeString, kTypeObject } },
	{ 596, kTypeInt       , { kTypeString, kTypeString, kTypeObject } },
	{ 597, kTypeVector    , { kTypeString, kTypeString, kTypeObject } },
	{ 598, kTypeEngineType, { kTypeString, kTypeString, kTypeObject } },
	{ 599, kTypeString    , { kTypeString, kTypeString, kTypeObject } },
	{ 600, kTypeObject    , { kTypeObject, kTypeEngineType, kTypeObject, kTypeString } },
	{ 601, kTypeVoid      , { kTypeString, kTypeString, kTypeObject } },
	{ 602, kTypeInt       , { kTypeString, kTypeString, kTypeObject, kTypeObject } },
	{ 603, kTypeObject    , { kTypeString, kTypeString, kTypeEngineType, kTypeObject, kTypeObject } },
	{ 604, kTypeEngineType, { } },
	{ 605, kTypeInt       , { kTypeObject } },
	{ 606, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 607, kTypeInt       , { kTypeObject } },
	{ 608, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 609, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeObject, kTypeFloat } },
	{ 610, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 611, kTypeInt       , { kTypeEngineType } },
	{ 612, kTypeEngineType, { kTypeObject } },
	{ 613, kTypeEngineType, { kTypeObject } },
	{ 614, kTypeInt       , { kTypeEngineType } },
	{ 615, kTypeInt       , { kTypeEngineType } },
	{ 616, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 617, kTypeEngineType, { kTypeInt } },
	{ 618, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 619, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 620, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 621, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 622, kTypeEngineType, { kTypeInt } },
	{ 623, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 624, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 625, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 626, kTypeEngineType, { kTypeInt } },
	{ 627, kTypeEngineType, { kTypeInt } },
	{ 628, kTypeEngineType, { kTypeInt } },
	{ 629, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 630, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 631, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 632, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 633, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 634, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 635, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 636, kTypeEngineType, { kTypeInt } },
	{ 637, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 638, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 639, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 640, kTypeEngineType, { } },
	{ 641, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 642, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 643, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 644, kTypeEngineType, { kTypeInt } },
	{ 645, kTypeEngineType, { kTypeInt } },
	{ 646, kTypeEngineType, { kTypeInt } },
	{ 647, kTypeEngineType, { } },
	{ 648, kTypeEngineType, { } },
	{ 649, kTypeEngineType, { kTypeInt } },
	{ 650, kTypeEngineType, { } },
	{ 651, kTypeEngineType, { kTypeInt } },
	{ 652, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 653, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 654, kTypeEngineType, { } },
	{ 655, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 656, kTypeEngineType, { kTypeInt } },
	{ 657, kTypeEngineType, { } },
	{ 658, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 659, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 660, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 661, kTypeEngineType, { kTypeInt } },
	{ 662, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 663, kTypeEngineType, { kTypeInt } },
	{ 664, kTypeEngineType, { kTypeInt } },
	{ 665, kTypeEngineType, { kTypeInt } },
	{ 666, kTypeEngineType, { kTypeInt } },
	{ 667, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 668, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 669, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 670, kTypeEngineType, { kTypeInt } },
	{ 671, kTypeEngineType, { kTypeInt } },
	{ 672, kTypeEngineType, { kTypeInt } },
	{ 673, kTypeEngineType, { kTypeInt } },
	{ 674, kTypeEngineType, { kTypeInt } },
	{ 675, kTypeEngineType, { kTypeInt } },
	{ 676, kTypeEngineType, { } },
	{ 677, kTypeEngineType, { kTypeInt } },
	{ 678, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 679, kTypeEngineType, { } },
	{ 680, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 681, kTypeEngineType, { kTypeInt } },
	{ 682, kTypeEngineType, { kTypeInt } },
	{ 683, kTypeEngineType, { } },
	{ 684, kTypeEngineType, { kTypeInt } },
	{ 685, kTypeEngineType, { kTypeInt } },
	{ 686, kTypeEngineType, { kTypeInt } },
	{ 687, kTypeEngineType, { kTypeInt } },
	{ 688, kTypeEngineType, { kTypeInt } },
	{ 689, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 690, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 691, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 692, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 693, kTypeObject    , { } },
	{ 694, kTypeFloat     , { kTypeInt } },
	{ 695, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 696, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 697, kTypeVoid      , { kTypeObject } },
	{ 698, kTypeVoid      , { kTypeObject } },
	{ 699, kTypeInt       , { kTypeObject } },
	{ 700, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 701, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 702, kTypeVoid      , { } },
	{ 703, kTypeVoid      , { } },
	{ 704, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 705, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 706, kTypeInt       , { kTypeObject } },
	{ 707, kTypeObject    , { } },
	{ 708, kTypeInt       , { kTypeObject } },
	{ 709, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 710, kTypeString    , { kTypeString, kTypeString, kTypeInt } },
	{ 711, kTypeEngineType, { } },
	{ 712, kTypeInt       , { kTypeObject } },
	{ 713, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 714, kTypeInt       , { kTypeObject } },
	{ 715, kTypeVoid      , { kTypeObject } },
	{ 716, kTypeInt       , { kTypeObject } },
	{ 717, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 718, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 719, kTypeVoid      , { kTypeObject } },
	{ 720, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 721, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 722, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 723, kTypeInt       , { kTypeObject } },
	{ 724, kTypeInt       , { kTypeObject } },
	{ 725, kTypeInt       , { kTypeObject } },
	{ 726, kTypeInt       , { kTypeObject } },
	{ 727, kTypeObject    , { } },
	{ 728, kTypeObject    , { } },
	{ 729, kTypeObject    , { } },
	{ 730, kTypeObject    , { } },
	{ 731, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 732, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 733, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 734, kTypeInt       , { kTypeEngineType } },
	{ 735, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 736, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 737, kTypeInt       , { kTypeObject } },
	{ 738, kTypeVoid      , { kTypeObject } },
	{ 739, kTypeEngineType, { kTypeInt } },
	{ 740, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 741, kTypeInt       , { kTypeObject } },
	{ 742, kTypeFloat     , { kTypeObject } },
	{ 743, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 744, kTypeInt       , { kTypeObject } },
	{ 745, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 746, kTypeVoid      , { kTypeInt } },
	{ 747, kTypeInt       , { } },
	{ 748, kTypeInt       , { kTypeObject } },
	{ 749, kTypeInt       , { kTypeObject } },
	{ 750, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 751, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 752, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 753, kTypeInt       , { kTypeVector, kTypeVector } },
	{ 754, kTypeInt       , { } },
	{ 755, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 756, kTypeVoid      , { kTypeObject } },
	{ 757, kTypeEngineType, { } },
	{ 758, kTypeEngineType, { kTypeInt } },
	{ 759, kTypeInt       , { kTypeObject } },
	{ 760, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 761, kTypeInt       , { kTypeObject } },
	{ 762, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 763, kTypeInt       , { kTypeObject } },
	{ 764, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 765, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 766, kTypeInt       , { kTypeObject } },
	{ 767, kTypeEngineType, { } },
	{ 768, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 769, kTypeInt       , { kTypeEngineType } },
	{ 770, kTypeInt       , { kTypeEngineType } },
	{ 771, kTypeInt       , { kTypeEngineType } },
	{ 772, kTypeInt       , { kTypeEngineType } },
	{ 773, kTypeInt       , { kTypeObject } },
	{ 774, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 775, kTypeVoid      , { kTypeObject } },
	{ 776, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 777, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 778, kTypeInt       , { kTypeObject } },
	{ 779, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 780, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject } },
	{ 781, kTypeInt       , { kTypeObject } },
	{ 782, kTypeInt       , { kTypeObject } },
	{ 783, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 784, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject } },
	{ 785, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 786, kTypeInt       , { kTypeObject } },
	{ 787, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 788, kTypeInt       , { kTypeObject } },
	{ 789, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 790, kTypeInt       , { kTypeObject } },
	{ 791, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 792, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 793, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject } },
	{ 794, kTypeInt       , { kTypeObject } },
	{ 795, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 796, kTypeInt       , { kTypeObject } },
	{ 797, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 798, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 799, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 800, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 801, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 802, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 803, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 804, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 805, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 806, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 807, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 808, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 809, kTypeObject    , { kTypeInt, kTypeEngineType, kTypeFloat, kTypeString, kTypeInt, kTypeString, kTypeString } },
	{ 810, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt, kTypeString, kTypeString } },
	{ 811, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 812, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 813, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 814, kTypeString    , { kTypeObject } },
	{ 815, kTypeInt       , { kTypeObject } },
	{ 816, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 817, kTypeInt       , { } },
	{ 818, kTypeVoid      , { kTypeInt } },
	{ 819, kTypeString    , { kTypeObject } },
	{ 820, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 821, kTypeInt       , { kTypeObject } },
	{ 822, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 823, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 824, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 825, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 826, kTypeObject    , { } },
	{ 827, kTypeInt       , { kTypeObject } },
	{ 828, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 829, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 830, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 831, kTypeInt       , { kTypeObject } },
	{ 832, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 833, kTypeString    , { kTypeObject } },
	{ 834, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 835, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 836, kTypeString    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 837, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 838, kTypeObject    , { } },
	{ 839, kTypeString    , { } },
	{ 840, kTypeInt       , { } },
	{ 841, kTypeVoid      , { kTypeString } },
	{ 842, kTypeVoid      , { kTypeInt } },
	{ 843, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 844, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 845, kTypeEngineType, { kTypeInt } },
	{ 846, kTypeEngineType, { kTypeInt } },
	{ 847, kTypeEngineType, { kTypeInt } }
};

/** The table defining the default values for the parameters of each engine function. */
const Functions::FunctionDefaults Functions::kFunctionDefaults[] = {
	{   0, { } },
	{   1, { } },
	{   2, { &kDefaultInt18, &kDefaultInt9 } },
	{   3, { &kDefaultInt18, &kDefaultInt9 } },
	{   4, { } },
	{   5, { } },
	{   6, { } },
	{   7, { } },
	{   8, { } },
	{   9, { &kDefaultFalse } },
	{  10, { } },
	{  11, { } },
	{  12, { } },
	{  13, { } },
	{  14, { } },
	{  15, { } },
	{  16, { } },
	{  17, { } },
	{  18, { } },
	{  19, { } },
	{  20, { } },
	{  21, { &kDefaultFalse } },
	{  22, { &kDefaultFalse, &kDefaultFloat1_0 } },
	{  23, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{  24, { } },
	{  25, { } },
	{  26, { } },
	{  27, { } },
	{  28, { } },
	{  29, { } },
	{  30, { } },
	{  31, { &kDefaultObjectSelf, &kDefaultInt1, &kDefaultStringEmpty } },
	{  32, { } },
	{  33, { } },
	{  34, { } },
	{  35, { } },
	{  36, { &kDefaultObjectSelf } },
	{  37, { &kDefaultFalse } },
	{  38, { &kDefaultObjectSelf, &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{  39, { &kDefaultTalkVolumeTalk } },
	{  40, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{  41, { } },
	{  42, { } },
	{  43, { } },
	{  44, { } },
	{  45, { &kDefaultFloatMinus1_0, &kDefaultFloatMinus1_0, &kDefaultCameraTransitionTypeSnap } },
	{  46, { } },
	{  47, { } },
	{  48, { &kDefaultMetaMagicAny, &kDefaultFalse, &kDefaultInt0, &kDefaultProjectilePathTypeDefault, &kDefaultFalse } },
	{  49, { &kDefaultObjectSelf } },
	{  50, { &kDefaultObjectSelf } },
	{  51, { } },
	{  52, { } },
	{  53, { } },
	{  54, { } },
	{  55, { } },
	{  56, { } },
	{  57, { } },
	{  58, { } },
	{  59, { } },
	{  60, { } },
	{  61, { } },
	{  62, { } },
	{  63, { } },
	{  64, { } },
	{  65, { } },
	{  66, { &kDefaultInt0 } },
	{  67, { } },
	{  68, { } },
	{  69, { } },
	{  70, { } },
	{  71, { } },
	{  72, { } },
	{  73, { } },
	{  74, { } },
	{  75, { } },
	{  76, { } },
	{  77, { } },
	{  78, { } },
	{  79, { &kDefaultDamageTypeMagical, &kDefaultDamagePowerNormal } },
	{  80, { } },
	{  81, { &kDefaultInt0 } },
	{  82, { } },
	{  83, { &kDefaultVFXNone, &kDefaultFloat0_0, &kDefaultInt0 } },
	{  84, { } },
	{  85, { } },
	{  86, { } },
	{  87, { } },
	{  88, { } },
	{  89, { } },
	{  90, { } },
	{  91, { } },
	{  92, { } },
	{  93, { &kDefaultObjectInvalid } },
	{  94, { &kDefaultObjectInvalid } },
	{  95, { &kDefaultInt1 } },
	{  96, { &kDefaultInt1 } },
	{  97, { &kDefaultInt1 } },
	{  98, { &kDefaultInt1 } },
	{  99, { &kDefaultInt1 } },
	{ 100, { &kDefaultInt1 } },
	{ 101, { &kDefaultInt1 } },
	{ 102, { &kDefaultInt1 } },
	{ 103, { &kDefaultInt1 } },
	{ 104, { } },
	{ 105, { } },
	{ 106, { } },
	{ 107, { } },
	{ 108, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 109, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 110, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 111, { } },
	{ 112, { } },
	{ 113, { } },
	{ 114, { } },
	{ 115, { &kDefaultACDodgeBonus, &kDefaultACVsDamageTypeAll } },
	{ 116, { &kDefaultInt0 } },
	{ 117, { &kDefaultSavingThrowTypeAll } },
	{ 118, { &kDefaultAttackBonusMisc } },
	{ 119, { &kDefaultInt0 } },
	{ 120, { &kDefaultDamageTypeMagical } },
	{ 121, { } },
	{ 122, { } },
	{ 123, { } },
	{ 124, { } },
	{ 125, { } },
	{ 126, { } },
	{ 127, { } },
	{ 128, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{ 129, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{ 130, { } },
	{ 131, { } },
	{ 132, { } },
	{ 133, { &kDefaultFalse, &kDefaultTrue } },
	{ 134, { } },
	{ 135, { } },
	{ 136, { } },
	{ 137, { } },
	{ 138, { &kDefaultInt1, &kDefaultInt1, &kDefaultInt1, &kDefaultInt1, &kDefaultInt1, &kDefaultInt1 } },
	{ 139, { &kDefaultFalse } },
	{ 140, { } },
	{ 141, { } },
	{ 142, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{ 143, { } },
	{ 144, { } },
	{ 145, { } },
	{ 146, { &kDefaultTrue } },
	{ 147, { &kDefaultTrue } },
	{ 148, { } },
	{ 149, { &kDefaultSpellAllSpells } },
	{ 150, { } },
	{ 151, { } },
	{ 152, { } },
	{ 153, { } },
	{ 154, { } },
	{ 155, { &kDefaultObjectSelf } },
	{ 156, { } },
	{ 157, { } },
	{ 158, { } },
	{ 159, { } },
	{ 160, { } },
	{ 161, { } },
	{ 162, { &kDefaultObjectSelf } },
	{ 163, { &kDefaultObjectSelf } },
	{ 164, { } },
	{ 165, { } },
	{ 166, { } },
	{ 167, { &kDefaultFloat0_0 } },
	{ 168, { } },
	{ 169, { } },
	{ 170, { } },
	{ 171, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 172, { &kDefaultObjectSelf } },
	{ 173, { } },
	{ 174, { } },
	{ 175, { } },
	{ 176, { &kDefaultInt0 } },
	{ 177, { } },
	{ 178, { } },
	{ 179, { } },
	{ 180, { &kDefaultFalse } },
	{ 181, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 182, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 183, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 184, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 185, { } },
	{ 186, { } },
	{ 187, { } },
	{ 188, { } },
	{ 189, { } },
	{ 190, { } },
	{ 191, { } },
	{ 192, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 193, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 194, { } },
	{ 195, { } },
	{ 196, { &kDefaultTrue } },
	{ 197, { } },
	{ 198, { } },
	{ 199, { } },
	{ 200, { &kDefaultInt0 } },
	{ 201, { &kDefaultTrue } },
	{ 202, { } },
	{ 203, { &kDefaultStringEmpty } },
	{ 204, { &kDefaultStringEmpty, &kDefaultFalse, &kDefaultTrue } },
	{ 205, { } },
	{ 206, { } },
	{ 207, { &kDefaultFalse } },
	{ 208, { } },
	{ 209, { } },
	{ 210, { } },
	{ 211, { } },
	{ 212, { } },
	{ 213, { } },
	{ 214, { } },
	{ 215, { } },
	{ 216, { &kDefaultFloat0_0 } },
	{ 217, { } },
	{ 218, { } },
	{ 219, { } },
	{ 220, { &kDefaultFloat0_0 } },
	{ 221, { &kDefaultTalkVolumeTalk } },
	{ 222, { } },
	{ 223, { } },
	{ 224, { } },
	{ 225, { } },
	{ 226, { &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{ 227, { &kDefaultObjectTypeAll, &kDefaultObjectSelf, &kDefaultInt1 } },
	{ 228, { &kDefaultInt1 } },
	{ 229, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{ 230, { } },
	{ 231, { } },
	{ 232, { } },
	{ 233, { } },
	{ 234, { &kDefaultMetaMagicAny, &kDefaultFalse, &kDefaultProjectilePathTypeDefault, &kDefaultFalse } },
	{ 235, { &kDefaultObjectSelf } },
	{ 236, { &kDefaultObjectSelf } },
	{ 237, { &kDefaultObjectSelf } },
	{ 238, { } },
	{ 239, { &kDefaultGenderMale } },
	{ 240, { &kDefaultTalkVolumeTalk } },
	{ 241, { &kDefaultFloat0_0 } },
	{ 242, { } },
	{ 243, { &kDefaultFalse, &kDefaultStringEmpty } },
	{ 244, { &kDefaultTrue } },
	{ 245, { } },
	{ 246, { } },
	{ 247, { } },
	{ 248, { } },
	{ 249, { &kDefaultNameFirstGenericMale } },
	{ 250, { } },
	{ 251, { } },
	{ 252, { } },
	{ 253, { &kDefaultFalse } },
	{ 254, { } },
	{ 255, { &kDefaultStringEmpty, &kDefaultObjectInvalid } },
	{ 256, { } },
	{ 257, { } },
	{ 258, { } },
	{ 259, { } },
	{ 260, { } },
	{ 261, { } },
	{ 262, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{ 263, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{ 264, { &kDefaultObjectSelf } },
	{ 265, { } },
	{ 266, { } },
	{ 267, { } },
	{ 268, { } },
	{ 269, { } },
	{ 270, { } },
	{ 271, { } },
	{ 272, { } },
	{ 273, { } },
	{ 274, { &kDefaultObjectInvalid } },
	{ 275, { } },
	{ 276, { &kDefaultObjectSelf } },
	{ 277, { &kDefaultObjectSelf } },
	{ 278, { &kDefaultObjectSelf } },
	{ 279, { &kDefaultObjectSelf } },
	{ 280, { &kDefaultObjectSelf } },
	{ 281, { &kDefaultObjectSelf } },
	{ 282, { } },
	{ 283, { } },
	{ 284, { } },
	{ 285, { &kDefaultObjectSelf } },
	{ 286, { &kDefaultObjectSelf } },
	{ 287, { } },
	{ 288, { &kDefaultInt0, &kDefaultObjectInvalid } },
	{ 289, { &kDefaultObjectSelf } },
	{ 290, { &kDefaultObjectSelf } },
	{ 291, { } },
	{ 292, { } },
	{ 293, { } },
	{ 294, { } },
	{ 295, { } },
	{ 296, { &kDefaultObjectSelf } },
	{ 297, { &kDefaultObjectSelf } },
	{ 298, { } },
	{ 299, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 300, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{ 301, { } },
	{ 302, { } },
	{ 303, { } },
	{ 304, { &kDefaultObjectSelf } },
	{ 305, { } },
	{ 306, { &kDefaultObjectSelf } },
	{ 307, { &kDefaultObjectSelf } },
	{ 308, { &kDefaultObjectSelf } },
	{ 309, { } },
	{ 310, { } },
	{ 311, { } },
	{ 312, { } },
	{ 313, { } },
	{ 314, { } },
	{ 315, { &kDefaultObjectSelf, &kDefaultFalse } },
	{ 316, { &kDefaultObjectSelf } },
	{ 317, { &kDefaultObjectSelf } },
	{ 318, { &kDefaultObjectSelf } },
	{ 319, { &kDefaultObjectSelf } },
	{ 320, { &kDefaultObjectSelf } },
	{ 321, { &kDefaultObjectSelf } },
	{ 322, { } },
	{ 323, { &kDefaultTrue, &kDefaultFalse } },
	{ 324, { } },
	{ 325, { } },
	{ 326, { } },
	{ 327, { &kDefaultObjectSelf } },
	{ 328, { } },
	{ 329, { } },
	{ 330, { } },
	{ 331, { &kDefaultObjectSelf } },
	{ 332, { } },
	{ 333, { } },
	{ 334, { &kDefaultObjectSelf } },
	{ 335, { &kDefaultObjectSelf } },
	{ 336, { } },
	{ 337, { } },
	{ 338, { } },
	{ 339, { &kDefaultObjectSelf } },
	{ 340, { &kDefaultObjectSelf } },
	{ 341, { &kDefaultObjectSelf } },
	{ 342, { &kDefaultObjectSelf } },
	{ 343, { &kDefaultObjectSelf } },
	{ 344, { } },
	{ 345, { } },
	{ 346, { &kDefaultObjectSelf } },
	{ 347, { } },
	{ 348, { } },
	{ 349, { } },
	{ 350, { } },
	{ 351, { } },
	{ 352, { } },
	{ 353, { } },
	{ 354, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{ 355, { &kDefaultAlignmentAll, &kDefaultAlignmentAll } },
	{ 356, { } },
	{ 357, { } },
	{ 358, { } },
	{ 359, { } },
	{ 360, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{ 361, { } },
	{ 362, { } },
	{ 363, { } },
	{ 364, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{ 365, { &kDefaultObjectSelf } },
	{ 366, { &kDefaultObjectSelf } },
	{ 367, { &kDefaultTrue, &kDefaultFalse, &kDefaultFalse } },
	{ 368, { &kDefaultTrue, &kDefaultFalse } },
	{ 369, { &kDefaultFalse } },
	{ 370, { } },
	{ 371, { } },
	{ 372, { } },
	{ 373, { } },
	{ 374, { } },
	{ 375, { } },
	{ 376, { } },
	{ 377, { &kDefaultObjectSelf } },
	{ 378, { &kDefaultInt0, &kDefaultInt0 } },
	{ 379, { } },
	{ 380, { &kDefaultTrue } },
	{ 381, { &kDefaultTrue } },
	{ 382, { &kDefaultFalse, &kDefaultFloat30_0 } },
	{ 383, { &kDefaultFalse, &kDefaultFloat1_0, &kDefaultFloat30_0 } },
	{ 384, { } },
	{ 385, { &kDefaultInt1 } },
	{ 386, { } },
	{ 387, { } },
	{ 388, { } },
	{ 389, { &kDefaultObjectSelf } },
	{ 390, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{ 391, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{ 392, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{ 393, { } },
	{ 394, { } },
	{ 395, { } },
	{ 396, { } },
	{ 397, { } },
	{ 398, { } },
	{ 399, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{ 400, { &kDefaultObjectInvalid } },
	{ 401, { } },
	{ 402, { &kDefaultFalse } },
	{ 403, { &kDefaultTrue } },
	{ 404, { } },
	{ 405, { } },
	{ 406, { } },
	{ 407, { } },
	{ 408, { } },
	{ 409, { &kDefaultObjectSelf } },
	{ 410, { } },
	{ 411, { } },
	{ 412, { } },
	{ 413, { } },
	{ 414, { } },
	{ 415, { } },
	{ 416, { } },
	{ 417, { &kDefaultStringEmpty, &kDefaultObjectSelf } },
	{ 418, { &kDefaultObjectSelf } },
	{ 419, { } },
	{ 420, { } },
	{ 421, { &kDefaultObjectSelf } },
	{ 422, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{ 423, { } },
	{ 424, { &kDefaultObjectInvalid } },
	{ 425, { } },
	{ 426, { } },
	{ 427, { } },
	{ 428, { } },
	{ 429, { } },
	{ 430, { } },
	{ 431, { } },
	{ 432, { } },
	{ 433, { } },
	{ 434, { } },
	{ 435, { } },
	{ 436, { } },
	{ 437, { } },
	{ 438, { } },
	{ 439, { } },
	{ 440, { } },
	{ 441, { } },
	{ 442, { } },
	{ 443, { } },
	{ 444, { &kDefaultFalse } },
	{ 445, { } },
	{ 446, { } },
	{ 447, { &kDefaultAttackBonusMisc } },
	{ 448, { &kDefaultDamageTypeMagical } },
	{ 449, { } },
	{ 450, { &kDefaultACDodgeBonus, &kDefaultACVsDamageTypeAll } },
	{ 451, { } },
	{ 452, { &kDefaultSavingThrowTypeAll } },
	{ 453, { } },
	{ 454, { } },
	{ 455, { &kDefaultObjectSelf } },
	{ 456, { } },
	{ 457, { } },
	{ 458, { &kDefaultMissChanceTypeNormal } },
	{ 459, { } },
	{ 460, { &kDefaultUseCreatureLevel } },
	{ 461, { } },
	{ 462, { &kDefaultFalse } },
	{ 463, { &kDefaultFalse } },
	{ 464, { } },
	{ 465, { } },
	{ 466, { } },
	{ 467, { } },
	{ 468, { } },
	{ 469, { &kDefaultObjectSelf } },
	{ 470, { &kDefaultObjectSelf } },
	{ 471, { &kDefaultObjectSelf } },
	{ 472, { &kDefaultInt0, &kDefaultSpellSchoolGeneral } },
	{ 473, { &kDefaultUseCreatureLevel } },
	{ 474, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultFalse } },
	{ 475, { } },
	{ 476, { } },
	{ 477, { &kDefaultMissChanceTypeNormal } },
	{ 478, { &kDefaultObjectSelf } },
	{ 479, { } },
	{ 480, { &kDefaultInt1 } },
	{ 481, { &kDefaultInt1 } },
	{ 482, { &kDefaultInt1 } },
	{ 483, { } },
	{ 484, { } },
	{ 485, { } },
	{ 486, { &kDefaultObjectSelf } },
	{ 487, { } },
	{ 488, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 489, { } },
	{ 490, { } },
	{ 491, { } },
	{ 492, { } },
	{ 493, { } },
	{ 494, { } },
	{ 495, { } },
	{ 496, { } },
	{ 497, { } },
	{ 498, { } },
	{ 499, { } },
	{ 500, { } },
	{ 501, { &kDefaultProjectilePathTypeDefault } },
	{ 502, { &kDefaultProjectilePathTypeDefault } },
	{ 503, { &kDefaultObjectSelf } },
	{ 504, { } },
	{ 505, { &kDefaultObjectSelf } },
	{ 506, { } },
	{ 507, { } },
	{ 508, { } },
	{ 509, { } },
	{ 510, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 511, { } },
	{ 512, { } },
	{ 513, { } },
	{ 514, { } },
	{ 515, { } },
	{ 516, { } },
	{ 517, { } },
	{ 518, { } },
	{ 519, { } },
	{ 520, { } },
	{ 521, { } },
	{ 522, { &kDefaultObjectSelf } },
	{ 523, { &kDefaultObjectSelf } },
	{ 524, { &kDefaultObjectSelf } },
	{ 525, { &kDefaultTrue } },
	{ 526, { &kDefaultTrue } },
	{ 527, { } },
	{ 528, { } },
	{ 529, { } },
	{ 530, { } },
	{ 531, { } },
	{ 532, { } },
	{ 533, { } },
	{ 534, { } },
	{ 535, { } },
	{ 536, { } },
	{ 537, { } },
	{ 538, { } },
	{ 539, { } },
	{ 540, { } },
	{ 541, { } },
	{ 542, { } },
	{ 543, { &kDefaultObjectSelf } },
	{ 544, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 545, { &kDefaultObjectSelf } },
	{ 546, { } },
	{ 547, { } },
	{ 548, { } },
	{ 549, { } },
	{ 550, { &kDefaultTrue } },
	{ 551, { } },
	{ 552, { } },
	{ 553, { } },
	{ 554, { &kDefaultTrue, &kDefaultTrue, &kDefaultInt0, &kDefaultStringEmpty } },
	{ 555, { } },
	{ 556, { &kDefaultObjectSelf } },
	{ 557, { } },
	{ 558, { } },
	{ 559, { } },
	{ 560, { } },
	{ 561, { } },
	{ 562, { } },
	{ 563, { } },
	{ 564, { } },
	{ 565, { } },
	{ 566, { } },
	{ 567, { } },
	{ 568, { } },
	{ 569, { } },
	{ 570, { } },
	{ 571, { } },
	{ 572, { } },
	{ 573, { } },
	{ 574, { } },
	{ 575, { } },
	{ 576, { } },
	{ 577, { } },
	{ 578, { } },
	{ 579, { } },
	{ 580, { } },
	{ 581, { } },
	{ 582, { } },
	{ 583, { } },
	{ 584, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{ 585, { } },
	{ 586, { } },
	{ 587, { &kDefaultObjectSelf } },
	{ 588, { } },
	{ 589, { &kDefaultObjectInvalid } },
	{ 590, { &kDefaultObjectInvalid } },
	{ 591, { &kDefaultObjectInvalid } },
	{ 592, { &kDefaultObjectInvalid } },
	{ 593, { &kDefaultObjectInvalid } },
	{ 594, { } },
	{ 595, { &kDefaultObjectInvalid } },
	{ 596, { &kDefaultObjectInvalid } },
	{ 597, { &kDefaultObjectInvalid } },
	{ 598, { &kDefaultObjectInvalid } },
	{ 599, { &kDefaultObjectInvalid } },
	{ 600, { &kDefaultObjectInvalid, &kDefaultStringEmpty } },
	{ 601, { &kDefaultObjectInvalid } },
	{ 602, { &kDefaultObjectInvalid } },
	{ 603, { &kDefaultObjectInvalid, &kDefaultObjectInvalid } },
	{ 604, { } },
	{ 605, { } },
	{ 606, { } },
	{ 607, { } },
	{ 608, { } },
	{ 609, { &kDefaultFloat0_0 } },
	{ 610, { } },
	{ 611, { } },
	{ 612, { } },
	{ 613, { } },
	{ 614, { } },
	{ 615, { } },
	{ 616, { } },
	{ 617, { } },
	{ 618, { } },
	{ 619, { } },
	{ 620, { } },
	{ 621, { } },
	{ 622, { } },
	{ 623, { } },
	{ 624, { } },
	{ 625, { } },
	{ 626, { } },
	{ 627, { } },
	{ 628, { } },
	{ 629, { } },
	{ 630, { } },
	{ 631, { } },
	{ 632, { } },
	{ 633, { } },
	{ 634, { } },
	{ 635, { } },
	{ 636, { } },
	{ 637, { } },
	{ 638, { } },
	{ 639, { } },
	{ 640, { } },
	{ 641, { } },
	{ 642, { } },
	{ 643, { } },
	{ 644, { } },
	{ 645, { } },
	{ 646, { } },
	{ 647, { } },
	{ 648, { } },
	{ 649, { } },
	{ 650, { } },
	{ 651, { } },
	{ 652, { } },
	{ 653, { } },
	{ 654, { } },
	{ 655, { } },
	{ 656, { } },
	{ 657, { } },
	{ 658, { &kDefaultInt0 } },
	{ 659, { } },
	{ 660, { } },
	{ 661, { } },
	{ 662, { } },
	{ 663, { } },
	{ 664, { } },
	{ 665, { } },
	{ 666, { } },
	{ 667, { } },
	{ 668, { } },
	{ 669, { } },
	{ 670, { } },
	{ 671, { &kDefaultIPConstUnlimitedAmmoBasic } },
	{ 672, { } },
	{ 673, { } },
	{ 674, { } },
	{ 675, { } },
	{ 676, { } },
	{ 677, { } },
	{ 678, { } },
	{ 679, { } },
	{ 680, { &kDefaultInt0 } },
	{ 681, { } },
	{ 682, { } },
	{ 683, { } },
	{ 684, { } },
	{ 685, { } },
	{ 686, { &kDefaultInt0 } },
	{ 687, { } },
	{ 688, { } },
	{ 689, { } },
	{ 690, { &kDefaultInt100, &kDefaultSpellSchoolGeneral } },
	{ 691, { &kDefaultTalkVolumeTalk } },
	{ 692, { &kDefaultTrue, &kDefaultFalse } },
	{ 693, { } },
	{ 694, { } },
	{ 695, { &kDefaultFadeSpeedMedium } },
	{ 696, { &kDefaultFadeSpeedMedium } },
	{ 697, { } },
	{ 698, { } },
	{ 699, { } },
	{ 700, { } },
	{ 701, { } },
	{ 702, { } },
	{ 703, { } },
	{ 704, { &kDefaultClassTypeInvalid, &kDefaultFalse, &kDefaultPackageInvalid } },
	{ 705, { } },
	{ 706, { &kDefaultObjectSelf } },
	{ 707, { } },
	{ 708, { &kDefaultObjectSelf } },
	{ 709, { &kDefaultTrue, &kDefaultFalse } },
	{ 710, { } },
	{ 711, { } },
	{ 712, { &kDefaultObjectSelf } },
	{ 713, { } },
	{ 714, { } },
	{ 715, { } },
	{ 716, { &kDefaultObjectInvalid } },
	{ 717, { } },
	{ 718, { } },
	{ 719, { } },
	{ 720, { &kDefaultTrue } },
	{ 721, { } },
	{ 722, { } },
	{ 723, { } },
	{ 724, { } },
	{ 725, { } },
	{ 726, { } },
	{ 727, { } },
	{ 728, { } },
	{ 729, { } },
	{ 730, { } },
	{ 731, { &kDefaultFalse } },
	{ 732, { } },
	{ 733, { } },
	{ 734, { } },
	{ 735, { } },
	{ 736, { } },
	{ 737, { } },
	{ 738, { } },
	{ 739, { } },
	{ 740, { } },
	{ 741, { } },
	{ 742, { } },
	{ 743, { } },
	{ 744, { } },
	{ 745, { } },
	{ 746, { } },
	{ 747, { } },
	{ 748, { } },
	{ 749, { } },
	{ 750, { &kDefaultFloat0_0 } },
	{ 751, { &kDefaultFloat0_0 } },
	{ 752, { } },
	{ 753, { } },
	{ 754, { } },
	{ 755, { &kDefaultObjectSelf } },
	{ 756, { &kDefaultObjectSelf } },
	{ 757, { } },
	{ 758, { } },
	{ 759, { } },
	{ 760, { } },
	{ 761, { } },
	{ 762, { } },
	{ 763, { } },
	{ 764, { } },
	{ 765, { } },
	{ 766, { } },
	{ 767, { } },
	{ 768, { &kDefaultObjectSelf } },
	{ 769, { } },
	{ 770, { } },
	{ 771, { } },
	{ 772, { } },
	{ 773, { } },
	{ 774, { } },
	{ 775, { } },
	{ 776, { &kDefaultFloat0_0 } },
	{ 777, { &kDefaultObjectInvalid } },
	{ 778, { } },
	{ 779, { &kDefaultObjectSelf } },
	{ 780, { &kDefaultObjectInvalid } },
	{ 781, { &kDefaultObjectSelf } },
	{ 782, { &kDefaultObjectInvalid } },
	{ 783, { &kDefaultObjectInvalid } },
	{ 784, { &kDefaultObjectInvalid } },
	{ 785, { &kDefaultObjectInvalid } },
	{ 786, { } },
	{ 787, { } },
	{ 788, { &kDefaultObjectSelf } },
	{ 789, { &kDefaultObjectSelf } },
	{ 790, { &kDefaultObjectSelf } },
	{ 791, { &kDefaultObjectSelf } },
	{ 792, { &kDefaultObjectSelf } },
	{ 793, { &kDefaultObjectSelf } },
	{ 794, { &kDefaultObjectSelf } },
	{ 795, { &kDefaultObjectSelf } },
	{ 796, { &kDefaultObjectSelf } },
	{ 797, { &kDefaultObjectSelf } },
	{ 798, { &kDefaultTrue } },
	{ 799, { } },
	{ 800, { &kDefaultTrue } },
	{ 801, { } },
	{ 802, { } },
	{ 803, { &kDefaultTrue } },
	{ 804, { &kDefaultTrue } },
	{ 805, { &kDefaultTrue } },
	{ 806, { } },
	{ 807, { } },
	{ 808, { } },
	{ 809, { &kDefaultFloat2_0, &kDefaultStringEmpty, &kDefaultStandardFactionHostile, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 810, { &kDefaultStandardFactionHostile, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 811, { } },
	{ 812, { } },
	{ 813, { } },
	{ 814, { } },
	{ 815, { } },
	{ 816, { &kDefaultTrue } },
	{ 817, { } },
	{ 818, { } },
	{ 819, { } },
	{ 820, { } },
	{ 821, { } },
	{ 822, { &kDefaultTrue } },
	{ 823, { &kDefaultTrue } },
	{ 824, { &kDefaultTrue } },
	{ 825, { &kDefaultTrue } },
	{ 826, { } },
	{ 827, { } },
	{ 828, { &kDefaultTrue } },
	{ 829, { &kDefaultObjectInvalid } },
	{ 830, { &kDefaultStringEmpty } },
	{ 831, { &kDefaultObjectSelf } },
	{ 832, { } },
	{ 833, { &kDefaultObjectSelf } },
	{ 834, { } },
	{ 835, { } },
	{ 836, { &kDefaultFalse, &kDefaultTrue } },
	{ 837, { &kDefaultStringEmpty, &kDefaultTrue } },
	{ 838, { } },
	{ 839, { } },
	{ 840, { } },
	{ 841, { &kDefaultStringEmpty } },
	{ 842, { &kDefaultTalkVolumeTalk } },
	{ 843, { } },
	{ 844, { } },
	{ 845, { } },
	{ 846, { } },
	{ 847, { } }
};

} // End of namespace NWN

} // End of namespace Engines

#endif // ENGINES_NWN_SCRIPT_TABLES_H
