/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Selection circle for KotOR games.
 */

#include "src/engines/kotorbase/objectcontainer.h"
#include "src/engines/kotorbase/object.h"

#include "src/engines/kotorbase/gui/selectioncircle.h"

namespace Engines {

namespace KotORBase {

SelectionCircle::SelectionCircle() :
		_hoveredQuad(nullptr),
		_targetQuad(nullptr),
		_hoveredQuadFriendly(new Graphics::Aurora::GUIQuad("friendlyreticle", 0.0f, 0.0f, kSelectionCircleSize, kSelectionCircleSize)),
		_hoveredQuadHostile(new Graphics::Aurora::GUIQuad("hostilereticle", 0.0f, 0.0f, kSelectionCircleSize, kSelectionCircleSize)),
		_targetQuadFriendly(new Graphics::Aurora::GUIQuad("friendlyreticle2", 0.0f, 0.0f, kSelectionCircleSize, kSelectionCircleSize)),
		_targetQuadHostile(new Graphics::Aurora::GUIQuad("hostilereticle2", 0.0f, 0.0f, kSelectionCircleSize, kSelectionCircleSize)) {
}

void SelectionCircle::show() {
	if (_target)
		_targetQuad->show();

	if (_hovered)
		_hoveredQuad->show();

	_visible = true;
}

void SelectionCircle::hide() {
	if (_hoveredQuad)
		_hoveredQuad->hide();
	if (_targetQuad)
		_targetQuad->hide();
	_visible = false;
}

void SelectionCircle::setPosition(float x, float y) {
	float halfSize = kSelectionCircleSize / 2.0f;
	_hoveredQuad->setPosition(x - halfSize, y - halfSize, -FLT_MIN);
	_targetQuad->setPosition(x - halfSize, y - halfSize, -100.0f);
}

void SelectionCircle::setHovered(bool hovered) {
	if (_hovered == hovered)
		return;

	_hovered = hovered;

	if (_visible) {
		if (_hovered)
			_hoveredQuad->show();
		else
			_hoveredQuad->hide();
	}
}

void SelectionCircle::setTarget(bool target) {
	if (_target == target)
		return;

	_target = target;

	if (_visible) {
		if (_target)
			_targetQuad->show();
		else
			_targetQuad->hide();
	}
}

bool SelectionCircle::moveTo(Object *object, float &sX, float &sY) {
	Graphics::Aurora::GUIQuad *lastQuad = _hoveredQuad;
	if (object->isEnemy())
		_hoveredQuad = _hoveredQuadHostile.get();
	else
		_hoveredQuad = _hoveredQuadFriendly.get();

	if (lastQuad && lastQuad != _hoveredQuad) {
		lastQuad->hide();
		_hoveredQuad->show();
	}

	lastQuad = _targetQuad;
	if (object->isEnemy())
		_targetQuad = _targetQuadHostile.get();
	else
		_targetQuad = _targetQuadFriendly.get();

	if (lastQuad && lastQuad != _targetQuad && lastQuad->isVisible()) {
		lastQuad->hide();
		_hoveredQuad->show();
	}

	float x, y, z;
	object->getTooltipAnchor(x, y, z);

	float sZ;
	GfxMan.project(x, y, z, sX, sY, sZ);
	setPosition(sX, sY);

	return (sZ >= 0.0f) && (sZ <= 1.0f);
}

} // End of namespace KotORBase

} // End of namespace Engines
