/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Tables defining the engine functions in Star Wars: Knights of the Old Republic.
 */

/* The functions are defined by three tables:
 * - kFunctionPointers
 * - kFunctionSignatures
 * - kFunctionDefaults
 *
 * kFunctionPointers provides the ID and name of the engine function, and a
 * pointer to Functions method doing the actual work. If the function pointer
 * is 0, a default unimplementedFunction method is used that just prints
 * the name of the function when it's called.
 *
 * kFunctionSignatures defines the signature of the function, i.e. the types
 * of the return value and its parameters.
 *
 * kFunctionDefaults pins default values to parameters. I.e. if a function
 * Foobar(int mode, int doEverything=FALSE) is called with only one parameters,
 * the default value FALSE is applied to the doEverything parameters. The
 * kFunctionDefaults entry for the function then contains a reference to a
 * false value. Because of ambiguities otherwise, the default values have to be
 * "aligned" to the right; only the last-most n parameters are allowed to have
 * default values, with no gaps.
 *
 * Please note that all three tables have to be of the same length, and that
 * the order of the functions have to be the same in all three tables.
 */

#ifndef ENGINES_KOTOR_SCRIPT_FUNCTION_TABLES_H
#define ENGINES_KOTOR_SCRIPT_FUNCTION_TABLES_H

#include "src/engines/kotorbase/types.h"
#include "src/engines/kotorbase/object.h"

namespace Engines {

namespace KotOR {

using Aurora::NWScript::kTypeVoid;
using Aurora::NWScript::kTypeInt;
using Aurora::NWScript::kTypeFloat;
using Aurora::NWScript::kTypeString;
using Aurora::NWScript::kTypeObject;
using Aurora::NWScript::kTypeVector;
using Aurora::NWScript::kTypeEngineType;
using Aurora::NWScript::kTypeScriptState;
using Aurora::NWScript::kTypeAny;

/* Default values as used by the function parameters, if no explicit value
 * is given at the time of the calling.
 */

static KotORBase::Object kDefaultValueObjectInvalid(KotORBase::kObjectTypeInvalid);
static KotORBase::Object kDefaultValueObjectSelf   (KotORBase::kObjectTypeSelf);

static const Aurora::NWScript::Variable kDefaultIntMinus1((int32_t) - 1);
static const Aurora::NWScript::Variable kDefaultInt0     ((int32_t)   0);
static const Aurora::NWScript::Variable kDefaultInt1     ((int32_t)   1);
static const Aurora::NWScript::Variable kDefaultInt9     ((int32_t)   9);
static const Aurora::NWScript::Variable kDefaultInt18    ((int32_t)  18);

static const Aurora::NWScript::Variable kDefaultFloat0_0 ( 0.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_75( 0.75f);
static const Aurora::NWScript::Variable kDefaultFloat1_0 ( 1.0f);
static const Aurora::NWScript::Variable kDefaultFloat30_0(30.0f);
static const Aurora::NWScript::Variable kDefaultFloat40_0(40.0f);

static const Aurora::NWScript::Variable kDefaultVector0(0.0f, 0.0f, 0.0f);

static const Aurora::NWScript::Variable kDefaultStringEmpty     (Common::UString(""));
static const Aurora::NWScript::Variable kDefaultStringWorldEntry(Common::UString("World Entry"));

static const Aurora::NWScript::Variable kDefaultObjectInvalid(&kDefaultValueObjectInvalid);
static const Aurora::NWScript::Variable kDefaultObjectSelf   (&kDefaultValueObjectSelf);

// TODO: Add the relevant enums to types.h, and use these values.

static const Aurora::NWScript::Variable kDefaultACDodgeBonus              ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultACVsDamageTypeAll         ((int32_t)   8199);
static const Aurora::NWScript::Variable kDefaultAlignmentAll              ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultAttackBonusMisc           ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultConversationTypeCinematic ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultDamagePowerNormal         ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultDamageTypeUniversal       ((int32_t)      8);
static const Aurora::NWScript::Variable kDefaultFalse                     ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultForcePowerAllForcePowers  ((int32_t) -    1);
static const Aurora::NWScript::Variable kDefaultObjectTypeAll             ((int32_t)  32767);
static const Aurora::NWScript::Variable kDefaultObjectTypeCreature        ((int32_t)      1);
static const Aurora::NWScript::Variable kDefaultObjectTypeInvalid         ((int32_t)  32767);
static const Aurora::NWScript::Variable kDefaultPersistentZoneActive      ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultProjectilePathTypeDefault ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultSavingThrowTypeAll        ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultSavingThrowTypeNone       ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultTalkVolumeTalk            ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultTrue                      ((int32_t)      1);


/** The table defining the name and function pointer of each engine function. */
const Functions::FunctionPointer Functions::kFunctionPointers[] = {
	{   0, "Random"                              , &Functions::random                               },
	{   1, "PrintString"                         , &Functions::printString                          },
	{   2, "PrintFloat"                          , &Functions::printFloat                           },
	{   3, "FloatToString"                       , &Functions::floatToString                        },
	{   4, "PrintInteger"                        , &Functions::printInteger                         },
	{   5, "PrintObject"                         , &Functions::printObject                          },
	{   6, "AssignCommand"                       , &Functions::assignCommand                        },
	{   7, "DelayCommand"                        , &Functions::delayCommand                         },
	{   8, "ExecuteScript"                       , &Functions::executeScript                        },
	{   9, "ClearAllActions"                     , &Functions::clearAllActions                      },
	{  10, "SetFacing"                           , 0                                                },
	{  11, "SwitchPlayerCharacter"               , 0                                                },
	{  12, "SetTime"                             , 0                                                },
	{  13, "SetPartyLeader"                      , &Functions::setPartyLeader                       },
	{  14, "SetAreaUnescapable"                  , 0                                                },
	{  15, "GetAreaUnescapable"                  , 0                                                },
	{  16, "GetTimeHour"                         , &Functions::getTimeHour                          },
	{  17, "GetTimeMinute"                       , &Functions::getTimeMinute                        },
	{  18, "GetTimeSecond"                       , &Functions::getTimeSecond                        },
	{  19, "GetTimeMillisecond"                  , &Functions::getTimeMillisecond                   },
	{  20, "ActionRandomWalk"                    , 0                                                },
	{  21, "ActionMoveToLocation"                , 0                                                },
	{  22, "ActionMoveToObject"                  , &Functions::actionMoveToObject                   },
	{  23, "ActionMoveAwayFromObject"            , 0                                                },
	{  24, "GetArea"                             , &Functions::getArea                              },
	{  25, "GetEnteringObject"                   , &Functions::getEnteringObject                    },
	{  26, "GetExitingObject"                    , &Functions::getExitingObject                     },
	{  27, "GetPosition"                         , 0                                                },
	{  28, "GetFacing"                           , 0                                                },
	{  29, "GetItemPossessor"                    , 0                                                },
	{  30, "GetItemPossessedBy"                  , 0                                                },
	{  31, "CreateItemOnObject"                  , &Functions::createItemOnObject                   },
	{  32, "ActionEquipItem"                     , 0                                                },
	{  33, "ActionUnequipItem"                   , 0                                                },
	{  34, "ActionPickUpItem"                    , 0                                                },
	{  35, "ActionPutDownItem"                   , 0                                                },
	{  36, "GetLastAttacker"                     , 0                                                },
	{  37, "ActionAttack"                        , 0                                                },
	{  38, "GetNearestCreature"                  , &Functions::getNearestCreature                   },
	{  39, "ActionSpeakString"                   , 0                                                },
	{  40, "ActionPlayAnimation"                 , 0                                                },
	{  41, "GetDistanceToObject"                 , 0                                                },
	{  42, "GetIsObjectValid"                    , &Functions::getIsObjectValid                     },
	{  43, "ActionOpenDoor"                      , &Functions::actionOpenDoor                       },
	{  44, "ActionCloseDoor"                     , &Functions::actionCloseDoor                      },
	{  45, "SetCameraFacing"                     , 0                                                },
	{  46, "PlaySound"                           , 0                                                },
	{  47, "GetSpellTargetObject"                , 0                                                },
	{  48, "ActionCastSpellAtObject"             , 0                                                },
	{  49, "GetCurrentHitPoints"                 , &Functions::getCurrentHitPoints                  },
	{  50, "GetMaxHitPoints"                     , &Functions::getMaxHitPoints                      },
	{  51, "EffectAssuredHit"                    , 0                                                },
	{  52, "GetLastItemEquipped"                 , 0                                                },
	{  53, "GetSubScreenID"                      , 0                                                },
	{  54, "CancelCombat"                        , 0                                                },
	{  55, "GetCurrentForcePoints"               , 0                                                },
	{  56, "GetMaxForcePoints"                   , 0                                                },
	{  57, "PauseGame"                           , 0                                                },
	{  58, "SetPlayerRestrictMode"               , 0                                                },
	{  59, "GetStringLength"                     , &Functions::getStringLength                      },
	{  60, "GetStringUpperCase"                  , &Functions::getStringUpperCase                   },
	{  61, "GetStringLowerCase"                  , &Functions::getStringLowerCase                   },
	{  62, "GetStringRight"                      , &Functions::getStringRight                       },
	{  63, "GetStringLeft"                       , &Functions::getStringLeft                        },
	{  64, "InsertString"                        , &Functions::insertString                         },
	{  65, "GetSubString"                        , &Functions::getSubString                         },
	{  66, "FindSubString"                       , &Functions::findSubString                        },
	{  67, "fabs"                                , &Functions::fabs                                 },
	{  68, "cos"                                 , &Functions::cos                                  },
	{  69, "sin"                                 , &Functions::sin                                  },
	{  70, "tan"                                 , &Functions::tan                                  },
	{  71, "acos"                                , &Functions::acos                                 },
	{  72, "asin"                                , &Functions::asin                                 },
	{  73, "atan"                                , &Functions::atan                                 },
	{  74, "log"                                 , &Functions::log                                  },
	{  75, "pow"                                 , &Functions::pow                                  },
	{  76, "sqrt"                                , &Functions::sqrt                                 },
	{  77, "abs"                                 , &Functions::abs                                  },
	{  78, "EffectHeal"                          , 0                                                },
	{  79, "EffectDamage"                        , 0                                                },
	{  80, "EffectAbilityIncrease"               , 0                                                },
	{  81, "EffectDamageResistance"              , 0                                                },
	{  82, "EffectResurrection"                  , 0                                                },
	{  83, "GetPlayerRestrictMode"               , 0                                                },
	{  84, "GetCasterLevel"                      , 0                                                },
	{  85, "GetFirstEffect"                      , 0                                                },
	{  86, "GetNextEffect"                       , 0                                                },
	{  87, "RemoveEffect"                        , 0                                                },
	{  88, "GetIsEffectValid"                    , 0                                                },
	{  89, "GetEffectDurationType"               , 0                                                },
	{  90, "GetEffectSubType"                    , 0                                                },
	{  91, "GetEffectCreator"                    , 0                                                },
	{  92, "IntToString"                         , &Functions::intToString                          },
	{  93, "GetFirstObjectInArea"                , 0                                                },
	{  94, "GetNextObjectInArea"                 , 0                                                },
	{  95, "d2"                                  , &Functions::d2                                   },
	{  96, "d3"                                  , &Functions::d3                                   },
	{  97, "d4"                                  , &Functions::d4                                   },
	{  98, "d6"                                  , &Functions::d6                                   },
	{  99, "d8"                                  , &Functions::d8                                   },
	{ 100, "d10"                                 , &Functions::d10                                  },
	{ 101, "d12"                                 , &Functions::d12                                  },
	{ 102, "d20"                                 , &Functions::d20                                  },
	{ 103, "d100"                                , &Functions::d100                                 },
	{ 104, "VectorMagnitude"                     , &Functions::vectorMagnitude                      },
	{ 105, "GetMetaMagicFeat"                    , 0                                                },
	{ 106, "GetObjectType"                       , 0                                                },
	{ 107, "GetRacialType"                       , &Functions::getRacialType                        },
	{ 108, "FortitudeSave"                       , 0                                                },
	{ 109, "ReflexSave"                          , 0                                                },
	{ 110, "WillSave"                            , 0                                                },
	{ 111, "GetSpellSaveDC"                      , 0                                                },
	{ 112, "MagicalEffect"                       , 0                                                },
	{ 113, "SupernaturalEffect"                  , 0                                                },
	{ 114, "ExtraordinaryEffect"                 , 0                                                },
	{ 115, "EffectACIncrease"                    , 0                                                },
	{ 116, "GetAC"                               , 0                                                },
	{ 117, "EffectSavingThrowIncrease"           , 0                                                },
	{ 118, "EffectAttackIncrease"                , 0                                                },
	{ 119, "EffectDamageReduction"               , 0                                                },
	{ 120, "EffectDamageIncrease"                , 0                                                },
	{ 121, "RoundsToSeconds"                     , 0                                                },
	{ 122, "HoursToSeconds"                      , 0                                                },
	{ 123, "TurnsToSeconds"                      , 0                                                },
	{ 124, "SoundObjectSetFixedVariance"         , 0                                                },
	{ 125, "GetGoodEvilValue"                    , 0                                                },
	{ 126, "GetPartyMemberCount"                 , 0                                                },
	{ 127, "GetAlignmentGoodEvil"                , 0                                                },
	{ 128, "GetFirstObjectInShape"               , 0                                                },
	{ 129, "GetNextObjectInShape"                , 0                                                },
	{ 130, "EffectEntangle"                      , 0                                                },
	{ 131, "SignalEvent"                         , &Functions::signalEvent                          },
	{ 132, "EventUserDefined"                    , &Functions::eventUserDefined                     },
	{ 133, "EffectDeath"                         , 0                                                },
	{ 134, "EffectKnockdown"                     , 0                                                },
	{ 135, "ActionGiveItem"                      , 0                                                },
	{ 136, "ActionTakeItem"                      , 0                                                },
	{ 137, "VectorNormalize"                     , &Functions::vectorNormalize                      },
	{ 138, "GetItemStackSize"                    , 0                                                },
	{ 139, "GetAbilityScore"                     , &Functions::getAbilityScore                      },
	{ 140, "GetIsDead"                           , 0                                                },
	{ 141, "PrintVector"                         , &Functions::printVector                          },
	{ 142, "Vector"                              , &Functions::vector                               },
	{ 143, "SetFacingPoint"                      , 0                                                },
	{ 144, "AngleToVector"                       , 0                                                },
	{ 145, "VectorToAngle"                       , 0                                                },
	{ 146, "TouchAttackMelee"                    , 0                                                },
	{ 147, "TouchAttackRanged"                   , 0                                                },
	{ 148, "EffectParalyze"                      , 0                                                },
	{ 149, "EffectSpellImmunity"                 , 0                                                },
	{ 150, "SetItemStackSize"                    , 0                                                },
	{ 151, "GetDistanceBetween"                  , 0                                                },
	{ 152, "SetReturnStrref"                     , &Functions::setReturnStrref                      },
	{ 153, "EffectForceJump"                     , 0                                                },
	{ 154, "EffectSleep"                         , 0                                                },
	{ 155, "GetItemInSlot"                       , &Functions::getItemInSlot                        },
	{ 156, "EffectTemporaryForcePoints"          , 0                                                },
	{ 157, "EffectConfused"                      , 0                                                },
	{ 158, "EffectFrightened"                    , 0                                                },
	{ 159, "EffectChoke"                         , 0                                                },
	{ 160, "SetGlobalString"                     , 0                                                },
	{ 161, "EffectStunned"                       , 0                                                },
	{ 162, "SetCommandable"                      , &Functions::setCommandable                       },
	{ 163, "GetCommandable"                      , &Functions::getCommandable                       },
	{ 164, "EffectRegenerate"                    , 0                                                },
	{ 165, "EffectMovementSpeedIncrease"         , 0                                                },
	{ 166, "GetHitDice"                          , 0                                                },
	{ 167, "ActionForceFollowObject"             , 0                                                },
	{ 168, "GetTag"                              , &Functions::getTag                               },
	{ 169, "ResistForce"                         , 0                                                },
	{ 170, "GetEffectType"                       , 0                                                },
	{ 171, "EffectAreaOfEffect"                  , 0                                                },
	{ 172, "GetFactionEqual"                     , 0                                                },
	{ 173, "ChangeFaction"                       , 0                                                },
	{ 174, "GetIsListening"                      , 0                                                },
	{ 175, "SetListening"                        , 0                                                },
	{ 176, "SetListenPattern"                    , 0                                                },
	{ 177, "TestStringAgainstPattern"            , 0                                                },
	{ 178, "GetMatchedSubstring"                 , 0                                                },
	{ 179, "GetMatchedSubstringsCount"           , 0                                                },
	{ 180, "EffectVisualEffect"                  , 0                                                },
	{ 181, "GetFactionWeakestMember"             , 0                                                },
	{ 182, "GetFactionStrongestMember"           , 0                                                },
	{ 183, "GetFactionMostDamagedMember"         , 0                                                },
	{ 184, "GetFactionLeastDamagedMember"        , 0                                                },
	{ 185, "GetFactionGold"                      , 0                                                },
	{ 186, "GetFactionAverageReputation"         , 0                                                },
	{ 187, "GetFactionAverageGoodEvilAlignment"  , 0                                                },
	{ 188, "SoundObjectGetFixedVariance"         , 0                                                },
	{ 189, "GetFactionAverageLevel"              , 0                                                },
	{ 190, "GetFactionAverageXP"                 , 0                                                },
	{ 191, "GetFactionMostFrequentClass"         , 0                                                },
	{ 192, "GetFactionWorstAC"                   , 0                                                },
	{ 193, "GetFactionBestAC"                    , 0                                                },
	{ 194, "GetGlobalString"                     , 0                                                },
	{ 195, "GetListenPatternNumber"              , 0                                                },
	{ 196, "ActionJumpToObject"                  , 0                                                },
	{ 197, "GetWaypointByTag"                    , 0                                                },
	{ 198, "GetTransitionTarget"                 , 0                                                },
	{ 199, "EffectLinkEffects"                   , 0                                                },
	{ 200, "GetObjectByTag"                      , &Functions::getObjectByTag                       },
	{ 201, "AdjustAlignment"                     , 0                                                },
	{ 202, "ActionWait"                          , 0                                                },
	{ 203, "SetAreaTransitionBMP"                , 0                                                },
	{ 204, "ActionStartConversation"             , &Functions::actionStartConversation              },
	{ 205, "ActionPauseConversation"             , 0                                                },
	{ 206, "ActionResumeConversation"            , 0                                                },
	{ 207, "EffectBeam"                          , 0                                                },
	{ 208, "GetReputation"                       , 0                                                },
	{ 209, "AdjustReputation"                    , 0                                                },
	{ 210, "GetModuleFileName"                   , 0                                                },
	{ 211, "GetGoingToBeAttackedBy"              , 0                                                },
	{ 212, "EffectForceResistanceIncrease"       , 0                                                },
	{ 213, "GetLocation"                         , &Functions::getLocation                          },
	{ 214, "ActionJumpToLocation"                , 0                                                },
	{ 215, "Location"                            , 0                                                },
	{ 216, "ApplyEffectAtLocation"               , 0                                                },
	{ 217, "GetIsPC"                             , &Functions::getIsPC                              },
	{ 218, "FeetToMeters"                        , 0                                                },
	{ 219, "YardsToMeters"                       , 0                                                },
	{ 220, "ApplyEffectToObject"                 , 0                                                },
	{ 221, "SpeakString"                         , 0                                                },
	{ 222, "GetSpellTargetLocation"              , 0                                                },
	{ 223, "GetPositionFromLocation"             , 0                                                },
	{ 224, "EffectBodyFuel"                      , 0                                                },
	{ 225, "GetFacingFromLocation"               , 0                                                },
	{ 226, "GetNearestCreatureToLocation"        , 0                                                },
	{ 227, "GetNearestObject"                    , 0                                                },
	{ 228, "GetNearestObjectToLocation"          , 0                                                },
	{ 229, "GetNearestObjectByTag"               , 0                                                },
	{ 230, "IntToFloat"                          , &Functions::intToFloat                           },
	{ 231, "FloatToInt"                          , &Functions::floatToInt                           },
	{ 232, "StringToInt"                         , &Functions::stringToInt                          },
	{ 233, "StringToFloat"                       , &Functions::stringToFloat                        },
	{ 234, "ActionCastSpellAtLocation"           , 0                                                },
	{ 235, "GetIsEnemy"                          , 0                                                },
	{ 236, "GetIsFriend"                         , 0                                                },
	{ 237, "GetIsNeutral"                        , 0                                                },
	{ 238, "GetPCSpeaker"                        , &Functions::getPCSpeaker                         },
	{ 239, "GetStringByStrRef"                   , &Functions::getStringByStrRef                    },
	{ 240, "ActionSpeakStringByStrRef"           , 0                                                },
	{ 241, "DestroyObject"                       , &Functions::destroyObject                        },
	{ 242, "GetModule"                           , &Functions::getModule                            },
	{ 243, "CreateObject"                        , 0                                                },
	{ 244, "EventSpellCastAt"                    , 0                                                },
	{ 245, "GetLastSpellCaster"                  , 0                                                },
	{ 246, "GetLastSpell"                        , 0                                                },
	{ 247, "GetUserDefinedEventNumber"           , &Functions::getUserDefinedEventNumber            },
	{ 248, "GetSpellId"                          , 0                                                },
	{ 249, "RandomName"                          , 0                                                },
	{ 250, "EffectPoison"                        , 0                                                },
	{ 251, "GetLoadFromSaveGame"                 , 0                                                },
	{ 252, "EffectAssuredDeflection"             , 0                                                },
	{ 253, "GetName"                             , 0                                                },
	{ 254, "GetLastSpeaker"                      , 0                                                },
	{ 255, "BeginConversation"                   , 0                                                },
	{ 256, "GetLastPerceived"                    , 0                                                },
	{ 257, "GetLastPerceptionHeard"              , 0                                                },
	{ 258, "GetLastPerceptionInaudible"          , 0                                                },
	{ 259, "GetLastPerceptionSeen"               , 0                                                },
	{ 260, "GetLastClosedBy"                     , &Functions::getLastClosedBy                      },
	{ 261, "GetLastPerceptionVanished"           , 0                                                },
	{ 262, "GetFirstInPersistentObject"          , 0                                                },
	{ 263, "GetNextInPersistentObject"           , 0                                                },
	{ 264, "GetAreaOfEffectCreator"              , 0                                                },
	{ 265, "ShowLevelUpGUI"                      , 0                                                },
	{ 266, "SetItemNonEquippable"                , 0                                                },
	{ 267, "GetButtonMashCheck"                  , 0                                                },
	{ 268, "SetButtonMashCheck"                  , 0                                                },
	{ 269, "EffectForcePushTargeted"             , 0                                                },
	{ 270, "EffectHaste"                         , 0                                                },
	{ 271, "GiveItem"                            , 0                                                },
	{ 272, "ObjectToString"                      , &Functions::objectToString                       },
	{ 273, "EffectImmunity"                      , 0                                                },
	{ 274, "GetIsImmune"                         , 0                                                },
	{ 275, "EffectDamageImmunityIncrease"        , 0                                                },
	{ 276, "GetEncounterActive"                  , 0                                                },
	{ 277, "SetEncounterActive"                  , 0                                                },
	{ 278, "GetEncounterSpawnsMax"               , 0                                                },
	{ 279, "SetEncounterSpawnsMax"               , 0                                                },
	{ 280, "GetEncounterSpawnsCurrent"           , 0                                                },
	{ 281, "SetEncounterSpawnsCurrent"           , 0                                                },
	{ 282, "GetModuleItemAcquired"               , 0                                                },
	{ 283, "GetModuleItemAcquiredFrom"           , 0                                                },
	{ 284, "SetCustomToken"                      , 0                                                },
	{ 285, "GetHasFeat"                          , 0                                                },
	{ 286, "GetHasSkill"                         , &Functions::getHasSkill                          },
	{ 287, "ActionUseFeat"                       , 0                                                },
	{ 288, "ActionUseSkill"                      , 0                                                },
	{ 289, "GetObjectSeen"                       , 0                                                },
	{ 290, "GetObjectHeard"                      , 0                                                },
	{ 291, "GetLastPlayerDied"                   , 0                                                },
	{ 292, "GetModuleItemLost"                   , 0                                                },
	{ 293, "GetModuleItemLostBy"                 , 0                                                },
	{ 294, "ActionDoCommand"                     , 0                                                },
	{ 295, "EventConversation"                   , 0                                                },
	{ 296, "SetEncounterDifficulty"              , 0                                                },
	{ 297, "GetEncounterDifficulty"              , 0                                                },
	{ 298, "GetDistanceBetweenLocations"         , 0                                                },
	{ 299, "GetReflexAdjustedDamage"             , 0                                                },
	{ 300, "PlayAnimation"                       , 0                                                },
	{ 301, "TalentSpell"                         , 0                                                },
	{ 302, "TalentFeat"                          , 0                                                },
	{ 303, "TalentSkill"                         , 0                                                },
	{ 304, "GetHasSpellEffect"                   , 0                                                },
	{ 305, "GetEffectSpellId"                    , 0                                                },
	{ 306, "GetCreatureHasTalent"                , 0                                                },
	{ 307, "GetCreatureTalentRandom"             , 0                                                },
	{ 308, "GetCreatureTalentBest"               , 0                                                },
	{ 309, "ActionUseTalentOnObject"             , 0                                                },
	{ 310, "ActionUseTalentAtLocation"           , 0                                                },
	{ 311, "GetGoldPieceValue"                   , 0                                                },
	{ 312, "GetIsPlayableRacialType"             , 0                                                },
	{ 313, "JumpToLocation"                      , &Functions::jumpToLocation                       },
	{ 314, "EffectTemporaryHitpoints"            , 0                                                },
	{ 315, "GetSkillRank"                        , &Functions::getSkillRank                         },
	{ 316, "GetAttackTarget"                     , 0                                                },
	{ 317, "GetLastAttackType"                   , 0                                                },
	{ 318, "GetLastAttackMode"                   , 0                                                },
	{ 319, "GetDistanceBetween2D"                , 0                                                },
	{ 320, "GetIsInCombat"                       , 0                                                },
	{ 321, "GetLastAssociateCommand"             , 0                                                },
	{ 322, "GiveGoldToCreature"                  , 0                                                },
	{ 323, "SetIsDestroyable"                    , 0                                                },
	{ 324, "SetLocked"                           , &Functions::setLocked                            },
	{ 325, "GetLocked"                           , &Functions::getLocked                            },
	{ 326, "GetClickingObject"                   , &Functions::getClickingObject                    },
	{ 327, "SetAssociateListenPatterns"          , 0                                                },
	{ 328, "GetLastWeaponUsed"                   , 0                                                },
	{ 329, "ActionInteractObject"                , 0                                                },
	{ 330, "GetLastUsedBy"                       , &Functions::getLastUsedBy                        },
	{ 331, "GetAbilityModifier"                  , 0                                                },
	{ 332, "GetIdentified"                       , 0                                                },
	{ 333, "SetIdentified"                       , 0                                                },
	{ 334, "GetDistanceBetweenLocations2D"       , 0                                                },
	{ 335, "GetDistanceToObject2D"               , 0                                                },
	{ 336, "GetBlockingDoor"                     , 0                                                },
	{ 337, "GetIsDoorActionPossible"             , 0                                                },
	{ 338, "DoDoorAction"                        , 0                                                },
	{ 339, "GetFirstItemInInventory"             , 0                                                },
	{ 340, "GetNextItemInInventory"              , 0                                                },
	{ 341, "GetClassByPosition"                  , &Functions::getClassByPosition                   },
	{ 342, "GetLevelByPosition"                  , &Functions::getLevelByPosition                   },
	{ 343, "GetLevelByClass"                     , &Functions::getLevelByClass                      },
	{ 344, "GetDamageDealtByType"                , 0                                                },
	{ 345, "GetTotalDamageDealt"                 , 0                                                },
	{ 346, "GetLastDamager"                      , 0                                                },
	{ 347, "GetLastDisarmed"                     , 0                                                },
	{ 348, "GetLastDisturbed"                    , 0                                                },
	{ 349, "GetLastLocked"                       , 0                                                },
	{ 350, "GetLastUnlocked"                     , 0                                                },
	{ 351, "EffectSkillIncrease"                 , 0                                                },
	{ 352, "GetInventoryDisturbType"             , 0                                                },
	{ 353, "GetInventoryDisturbItem"             , 0                                                },
	{ 354, "ShowUpgradeScreen"                   , 0                                                },
	{ 355, "VersusAlignmentEffect"               , 0                                                },
	{ 356, "VersusRacialTypeEffect"              , 0                                                },
	{ 357, "VersusTrapEffect"                    , 0                                                },
	{ 358, "GetGender"                           , &Functions::getGender                            },
	{ 359, "GetIsTalentValid"                    , 0                                                },
	{ 360, "ActionMoveAwayFromLocation"          , 0                                                },
	{ 361, "GetAttemptedAttackTarget"            , 0                                                },
	{ 362, "GetTypeFromTalent"                   , 0                                                },
	{ 363, "GetIdFromTalent"                     , 0                                                },
	{ 364, "PlayPazaak"                          , 0                                                },
	{ 365, "GetLastPazaakResult"                 , 0                                                },
	{ 366, "DisplayFeedBackText"                 , 0                                                },
	{ 367, "AddJournalQuestEntry"                , 0                                                },
	{ 368, "RemoveJournalQuestEntry"             , 0                                                },
	{ 369, "GetJournalEntry"                     , 0                                                },
	{ 370, "PlayRumblePattern"                   , 0                                                },
	{ 371, "StopRumblePattern"                   , 0                                                },
	{ 372, "EffectDamageForcePoints"             , 0                                                },
	{ 373, "EffectHealForcePoints"               , 0                                                },
	{ 374, "SendMessageToPC"                     , &Functions::sendMessageToPC                      },
	{ 375, "GetAttemptedSpellTarget"             , 0                                                },
	{ 376, "GetLastOpenedBy"                     , &Functions::getLastOpenedBy                      },
	{ 377, "GetHasSpell"                         , 0                                                },
	{ 378, "OpenStore"                           , 0                                                },
	{ 379, "ActionSurrenderToEnemies"            , 0                                                },
	{ 380, "GetFirstFactionMember"               , 0                                                },
	{ 381, "GetNextFactionMember"                , 0                                                },
	{ 382, "ActionForceMoveToLocation"           , 0                                                },
	{ 383, "ActionForceMoveToObject"             , 0                                                },
	{ 384, "GetJournalQuestExperience"           , 0                                                },
	{ 385, "JumpToObject"                        , 0                                                },
	{ 386, "SetMapPinEnabled"                    , 0                                                },
	{ 387, "EffectHitPointChangeWhenDying"       , 0                                                },
	{ 388, "PopUpGUIPanel"                       , 0                                                },
	{ 389, "AddMultiClass"                       , 0                                                },
	{ 390, "GetIsLinkImmune"                     , 0                                                },
	{ 391, "EffectDroidStun"                     , 0                                                },
	{ 392, "EffectForcePushed"                   , 0                                                },
	{ 393, "GiveXPToCreature"                    , 0                                                },
	{ 394, "SetXP"                               , 0                                                },
	{ 395, "GetXP"                               , 0                                                },
	{ 396, "IntToHexString"                      , &Functions::intToHexString                       },
	{ 397, "GetBaseItemType"                     , 0                                                },
	{ 398, "GetItemHasItemProperty"              , 0                                                },
	{ 399, "ActionEquipMostDamagingMelee"        , 0                                                },
	{ 400, "ActionEquipMostDamagingRanged"       , 0                                                },
	{ 401, "GetItemACValue"                      , 0                                                },
	{ 402, "EffectForceResisted"                 , 0                                                },
	{ 403, "ExploreAreaForPlayer"                , 0                                                },
	{ 404, "ActionEquipMostEffectiveArmor"       , 0                                                },
	{ 405, "GetIsDay"                            , 0                                                },
	{ 406, "GetIsNight"                          , 0                                                },
	{ 407, "GetIsDawn"                           , 0                                                },
	{ 408, "GetIsDusk"                           , 0                                                },
	{ 409, "GetIsEncounterCreature"              , 0                                                },
	{ 410, "GetLastPlayerDying"                  , 0                                                },
	{ 411, "GetStartingLocation"                 , 0                                                },
	{ 412, "ChangeToStandardFaction"             , &Functions::changeToStandardFaction              },
	{ 413, "SoundObjectPlay"                     , &Functions::soundObjectPlay                      },
	{ 414, "SoundObjectStop"                     , &Functions::soundObjectStop                      },
	{ 415, "SoundObjectSetVolume"                , 0                                                },
	{ 416, "SoundObjectSetPosition"              , 0                                                },
	{ 417, "SpeakOneLinerConversation"           , 0                                                },
	{ 418, "GetGold"                             , 0                                                },
	{ 419, "GetLastRespawnButtonPresser"         , 0                                                },
	{ 420, "EffectForceFizzle"                   , 0                                                },
	{ 421, "SetLightsaberPowered"                , 0                                                },
	{ 422, "GetIsWeaponEffective"                , 0                                                },
	{ 423, "GetLastSpellHarmful"                 , 0                                                },
	{ 424, "EventActivateItem"                   , 0                                                },
	{ 425, "MusicBackgroundPlay"                 , &Functions::musicBackgroundPlay                  },
	{ 426, "MusicBackgroundStop"                 , &Functions::musicBackgroundStop                  },
	{ 427, "MusicBackgroundSetDelay"             , 0                                                },
	{ 428, "MusicBackgroundChangeDay"            , &Functions::musicBackgroundChangeDay             },
	{ 429, "MusicBackgroundChangeNight"          , &Functions::musicBackgroundChangeNight           },
	{ 430, "MusicBattlePlay"                     , 0                                                },
	{ 431, "MusicBattleStop"                     , 0                                                },
	{ 432, "MusicBattleChange"                   , 0                                                },
	{ 433, "AmbientSoundPlay"                    , 0                                                },
	{ 434, "AmbientSoundStop"                    , 0                                                },
	{ 435, "AmbientSoundChangeDay"               , 0                                                },
	{ 436, "AmbientSoundChangeNight"             , 0                                                },
	{ 437, "GetLastKiller"                       , 0                                                },
	{ 438, "GetSpellCastItem"                    , 0                                                },
	{ 439, "GetItemActivated"                    , 0                                                },
	{ 440, "GetItemActivator"                    , 0                                                },
	{ 441, "GetItemActivatedTargetLocation"      , 0                                                },
	{ 442, "GetItemActivatedTarget"              , 0                                                },
	{ 443, "GetIsOpen"                           , &Functions::getIsOpen                            },
	{ 444, "TakeGoldFromCreature"                , 0                                                },
	{ 445, "GetIsInConversation"                 , 0                                                },
	{ 446, "EffectAbilityDecrease"               , 0                                                },
	{ 447, "EffectAttackDecrease"                , 0                                                },
	{ 448, "EffectDamageDecrease"                , 0                                                },
	{ 449, "EffectDamageImmunityDecrease"        , 0                                                },
	{ 450, "EffectACDecrease"                    , 0                                                },
	{ 451, "EffectMovementSpeedDecrease"         , 0                                                },
	{ 452, "EffectSavingThrowDecrease"           , 0                                                },
	{ 453, "EffectSkillDecrease"                 , 0                                                },
	{ 454, "EffectForceResistanceDecrease"       , 0                                                },
	{ 455, "GetPlotFlag"                         , 0                                                },
	{ 456, "SetPlotFlag"                         , 0                                                },
	{ 457, "EffectInvisibility"                  , 0                                                },
	{ 458, "EffectConcealment"                   , 0                                                },
	{ 459, "EffectForceShield"                   , 0                                                },
	{ 460, "EffectDispelMagicAll"                , 0                                                },
	{ 461, "SetDialogPlaceableCamera"            , 0                                                },
	{ 462, "GetSoloMode"                         , &Functions::getSoloMode                          },
	{ 463, "EffectDisguise"                      , 0                                                },
	{ 464, "GetMaxStealthXP"                     , 0                                                },
	{ 465, "EffectTrueSeeing"                    , 0                                                },
	{ 466, "EffectSeeInvisible"                  , 0                                                },
	{ 467, "EffectTimeStop"                      , 0                                                },
	{ 468, "SetMaxStealthXP"                     , 0                                                },
	{ 469, "EffectBlasterDeflectionIncrease"     , 0                                                },
	{ 470, "EffectBlasterDeflectionDecrease"     , 0                                                },
	{ 471, "EffectHorrified"                     , 0                                                },
	{ 472, "EffectSpellLevelAbsorption"          , 0                                                },
	{ 473, "EffectDispelMagicBest"               , 0                                                },
	{ 474, "GetCurrentStealthXP"                 , 0                                                },
	{ 475, "GetNumStackedItems"                  , 0                                                },
	{ 476, "SurrenderToEnemies"                  , 0                                                },
	{ 477, "EffectMissChance"                    , 0                                                },
	{ 478, "SetCurrentStealthXP"                 , 0                                                },
	{ 479, "GetCreatureSize"                     , 0                                                },
	{ 480, "AwardStealthXP"                      , 0                                                },
	{ 481, "GetStealthXPEnabled"                 , 0                                                },
	{ 482, "SetStealthXPEnabled"                 , 0                                                },
	{ 483, "ActionUnlockObject"                  , 0                                                },
	{ 484, "ActionLockObject"                    , 0                                                },
	{ 485, "EffectModifyAttacks"                 , 0                                                },
	{ 486, "GetLastTrapDetected"                 , 0                                                },
	{ 487, "EffectDamageShield"                  , 0                                                },
	{ 488, "GetNearestTrapToObject"              , 0                                                },
	{ 489, "GetAttemptedMovementTarget"          , 0                                                },
	{ 490, "GetBlockingCreature"                 , 0                                                },
	{ 491, "GetFortitudeSavingThrow"             , 0                                                },
	{ 492, "GetWillSavingThrow"                  , 0                                                },
	{ 493, "GetReflexSavingThrow"                , 0                                                },
	{ 494, "GetChallengeRating"                  , 0                                                },
	{ 495, "GetFoundEnemyCreature"               , 0                                                },
	{ 496, "GetMovementRate"                     , 0                                                },
	{ 497, "GetSubRace"                          , &Functions::getSubRace                           },
	{ 498, "GetStealthXPDecrement"               , 0                                                },
	{ 499, "SetStealthXPDecrement"               , 0                                                },
	{ 500, "DuplicateHeadAppearance"             , 0                                                },
	{ 501, "ActionCastFakeSpellAtObject"         , 0                                                },
	{ 502, "ActionCastFakeSpellAtLocation"       , 0                                                },
	{ 503, "CutsceneAttack"                      , 0                                                },
	{ 504, "SetCameraMode"                       , 0                                                },
	{ 505, "SetLockOrientationInDialog"          , 0                                                },
	{ 506, "SetLockHeadFollowInDialog"           , 0                                                },
	{ 507, "CutsceneMove"                        , 0                                                },
	{ 508, "EnableVideoEffect"                   , 0                                                },
	{ 509, "StartNewModule"                      , &Functions::startNewModule                       },
	{ 510, "DisableVideoEffect"                  , 0                                                },
	{ 511, "GetWeaponRanged"                     , 0                                                },
	{ 512, "DoSinglePlayerAutoSave"              , 0                                                },
	{ 513, "GetGameDifficulty"                   , 0                                                },
	{ 514, "GetUserActionsPending"               , 0                                                },
	{ 515, "RevealMap"                           , 0                                                },
	{ 516, "SetTutorialWindowsEnabled"           , 0                                                },
	{ 517, "ShowTutorialWindow"                  , 0                                                },
	{ 518, "StartCreditSequence"                 , 0                                                },
	{ 519, "IsCreditSequenceInProgress"          , 0                                                },
	{ 520, "SWMG_SetLateralAccelerationPerSecond", 0                                                },
	{ 521, "SWMG_GetLateralAccelerationPerSecond", 0                                                },
	{ 522, "GetCurrentAction"                    , &Functions::getCurrentAction                     },
	{ 523, "GetDifficultyModifier"               , 0                                                },
	{ 524, "GetAppearanceType"                   , 0                                                },
	{ 525, "FloatingTextStrRefOnCreature"        , 0                                                },
	{ 526, "FloatingTextStringOnCreature"        , 0                                                },
	{ 527, "GetTrapDisarmable"                   , 0                                                },
	{ 528, "GetTrapDetectable"                   , 0                                                },
	{ 529, "GetTrapDetectedBy"                   , 0                                                },
	{ 530, "GetTrapFlagged"                      , 0                                                },
	{ 531, "GetTrapBaseType"                     , 0                                                },
	{ 532, "GetTrapOneShot"                      , 0                                                },
	{ 533, "GetTrapCreator"                      , 0                                                },
	{ 534, "GetTrapKeyTag"                       , 0                                                },
	{ 535, "GetTrapDisarmDC"                     , 0                                                },
	{ 536, "GetTrapDetectDC"                     , 0                                                },
	{ 537, "GetLockKeyRequired"                  , 0                                                },
	{ 538, "GetLockKeyTag"                       , 0                                                },
	{ 539, "GetLockLockable"                     , 0                                                },
	{ 540, "GetLockUnlockDC"                     , 0                                                },
	{ 541, "GetLockLockDC"                       , 0                                                },
	{ 542, "GetPCLevellingUp"                    , 0                                                },
	{ 543, "GetHasFeatEffect"                    , 0                                                },
	{ 544, "SetPlaceableIllumination"            , 0                                                },
	{ 545, "GetPlaceableIllumination"            , 0                                                },
	{ 546, "GetIsPlaceableObjectActionPossible"  , 0                                                },
	{ 547, "DoPlaceableObjectAction"             , 0                                                },
	{ 548, "GetFirstPC"                          , &Functions::getFirstPC                           },
	{ 549, "GetNextPC"                           , &Functions::getNextPC                            },
	{ 550, "SetTrapDetectedBy"                   , 0                                                },
	{ 551, "GetIsTrapped"                        , 0                                                },
	{ 552, "SetEffectIcon"                       , 0                                                },
	{ 553, "FaceObjectAwayFromObject"            , 0                                                },
	{ 554, "PopUpDeathGUIPanel"                  , 0                                                },
	{ 555, "SetTrapDisabled"                     , 0                                                },
	{ 556, "GetLastHostileActor"                 , 0                                                },
	{ 557, "ExportAllCharacters"                 , 0                                                },
	{ 558, "MusicBackgroundGetDayTrack"          , &Functions::musicBackgroundGetDayTrack           },
	{ 559, "MusicBackgroundGetNightTrack"        , &Functions::musicBackgroundGetNightTrack         },
	{ 560, "WriteTimestampedLogEntry"            , &Functions::writeTimestampedLogEntry             },
	{ 561, "GetModuleName"                       , 0                                                },
	{ 562, "GetFactionLeader"                    , 0                                                },
	{ 563, "SWMG_SetSpeedBlurEffect"             , 0                                                },
	{ 564, "EndGame"                             , 0                                                },
	{ 565, "GetRunScriptVar"                     , &Functions::getRunScriptVar                      },
	{ 566, "GetCreatureMovmentType"              , 0                                                },
	{ 567, "AmbientSoundSetDayVolume"            , 0                                                },
	{ 568, "AmbientSoundSetNightVolume"          , 0                                                },
	{ 569, "MusicBackgroundGetBattleTrack"       , 0                                                },
	{ 570, "GetHasInventory"                     , 0                                                },
	{ 571, "GetStrRefSoundDuration"              , 0                                                },
	{ 572, "AddToParty"                          , 0                                                },
	{ 573, "RemoveFromParty"                     , 0                                                },
	{ 574, "AddPartyMember"                      , 0                                                },
	{ 575, "RemovePartyMember"                   , 0                                                },
	{ 576, "IsObjectPartyMember"                 , &Functions::isObjectPartyMember                  },
	{ 577, "GetPartyMemberByIndex"               , &Functions::getPartyMemberByIndex                },
	{ 578, "GetGlobalBoolean"                    , &Functions::getGlobalBoolean                     },
	{ 579, "SetGlobalBoolean"                    , &Functions::setGlobalBoolean                     },
	{ 580, "GetGlobalNumber"                     , &Functions::getGlobalNumber                      },
	{ 581, "SetGlobalNumber"                     , &Functions::setGlobalNumber                      },
	{ 582, "AurPostString"                       , 0                                                },
	{ 583, "SWMG_GetLastEvent"                   , 0                                                },
	{ 584, "SWMG_GetLastEventModelName"          , 0                                                },
	{ 585, "SWMG_GetObjectByName"                , 0                                                },
	{ 586, "SWMG_PlayAnimation"                  , 0                                                },
	{ 587, "SWMG_GetLastBulletHitDamage"         , 0                                                },
	{ 588, "SWMG_GetLastBulletHitTarget"         , 0                                                },
	{ 589, "SWMG_GetLastBulletHitShooter"        , 0                                                },
	{ 590, "SWMG_AdjustFollowerHitPoints"        , 0                                                },
	{ 591, "SWMG_OnBulletHit"                    , 0                                                },
	{ 592, "SWMG_OnObstacleHit"                  , 0                                                },
	{ 593, "SWMG_GetLastFollowerHit"             , 0                                                },
	{ 594, "SWMG_GetLastObstacleHit"             , 0                                                },
	{ 595, "SWMG_GetLastBulletFiredDamage"       , 0                                                },
	{ 596, "SWMG_GetLastBulletFiredTarget"       , 0                                                },
	{ 597, "SWMG_GetObjectName"                  , 0                                                },
	{ 598, "SWMG_OnDeath"                        , 0                                                },
	{ 599, "SWMG_IsFollower"                     , 0                                                },
	{ 600, "SWMG_IsPlayer"                       , 0                                                },
	{ 601, "SWMG_IsEnemy"                        , 0                                                },
	{ 602, "SWMG_IsTrigger"                      , 0                                                },
	{ 603, "SWMG_IsObstacle"                     , 0                                                },
	{ 604, "SWMG_SetFollowerHitPoints"           , 0                                                },
	{ 605, "SWMG_OnDamage"                       , 0                                                },
	{ 606, "SWMG_GetLastHPChange"                , 0                                                },
	{ 607, "SWMG_RemoveAnimation"                , 0                                                },
	{ 608, "SWMG_GetCameraNearClip"              , 0                                                },
	{ 609, "SWMG_GetCameraFarClip"               , 0                                                },
	{ 610, "SWMG_SetCameraClip"                  , 0                                                },
	{ 611, "SWMG_GetPlayer"                      , 0                                                },
	{ 612, "SWMG_GetEnemyCount"                  , 0                                                },
	{ 613, "SWMG_GetEnemy"                       , 0                                                },
	{ 614, "SWMG_GetObstacleCount"               , 0                                                },
	{ 615, "SWMG_GetObstacle"                    , 0                                                },
	{ 616, "SWMG_GetHitPoints"                   , 0                                                },
	{ 617, "SWMG_GetMaxHitPoints"                , 0                                                },
	{ 618, "SWMG_SetMaxHitPoints"                , 0                                                },
	{ 619, "SWMG_GetSphereRadius"                , 0                                                },
	{ 620, "SWMG_SetSphereRadius"                , 0                                                },
	{ 621, "SWMG_GetNumLoops"                    , 0                                                },
	{ 622, "SWMG_SetNumLoops"                    , 0                                                },
	{ 623, "SWMG_GetPosition"                    , 0                                                },
	{ 624, "SWMG_GetGunBankCount"                , 0                                                },
	{ 625, "SWMG_GetGunBankBulletModel"          , 0                                                },
	{ 626, "SWMG_GetGunBankGunModel"             , 0                                                },
	{ 627, "SWMG_GetGunBankDamage"               , 0                                                },
	{ 628, "SWMG_GetGunBankTimeBetweenShots"     , 0                                                },
	{ 629, "SWMG_GetGunBankLifespan"             , 0                                                },
	{ 630, "SWMG_GetGunBankSpeed"                , 0                                                },
	{ 631, "SWMG_GetGunBankTarget"               , 0                                                },
	{ 632, "SWMG_SetGunBankBulletModel"          , 0                                                },
	{ 633, "SWMG_SetGunBankGunModel"             , 0                                                },
	{ 634, "SWMG_SetGunBankDamage"               , 0                                                },
	{ 635, "SWMG_SetGunBankTimeBetweenShots"     , 0                                                },
	{ 636, "SWMG_SetGunBankLifespan"             , 0                                                },
	{ 637, "SWMG_SetGunBankSpeed"                , 0                                                },
	{ 638, "SWMG_SetGunBankTarget"               , 0                                                },
	{ 639, "SWMG_GetLastBulletHitPart"           , 0                                                },
	{ 640, "SWMG_IsGunBankTargetting"            , 0                                                },
	{ 641, "SWMG_GetPlayerOffset"                , 0                                                },
	{ 642, "SWMG_GetPlayerInvincibility"         , 0                                                },
	{ 643, "SWMG_GetPlayerSpeed"                 , 0                                                },
	{ 644, "SWMG_GetPlayerMinSpeed"              , 0                                                },
	{ 645, "SWMG_GetPlayerAccelerationPerSecond" , 0                                                },
	{ 646, "SWMG_GetPlayerTunnelPos"             , 0                                                },
	{ 647, "SWMG_SetPlayerOffset"                , 0                                                },
	{ 648, "SWMG_SetPlayerInvincibility"         , 0                                                },
	{ 649, "SWMG_SetPlayerSpeed"                 , 0                                                },
	{ 650, "SWMG_SetPlayerMinSpeed"              , 0                                                },
	{ 651, "SWMG_SetPlayerAccelerationPerSecond" , 0                                                },
	{ 652, "SWMG_SetPlayerTunnelPos"             , 0                                                },
	{ 653, "SWMG_GetPlayerTunnelNeg"             , 0                                                },
	{ 654, "SWMG_SetPlayerTunnelNeg"             , 0                                                },
	{ 655, "SWMG_GetPlayerOrigin"                , 0                                                },
	{ 656, "SWMG_SetPlayerOrigin"                , 0                                                },
	{ 657, "SWMG_GetGunBankHorizontalSpread"     , 0                                                },
	{ 658, "SWMG_GetGunBankVerticalSpread"       , 0                                                },
	{ 659, "SWMG_GetGunBankSensingRadius"        , 0                                                },
	{ 660, "SWMG_GetGunBankInaccuracy"           , 0                                                },
	{ 661, "SWMG_SetGunBankHorizontalSpread"     , 0                                                },
	{ 662, "SWMG_SetGunBankVerticalSpread"       , 0                                                },
	{ 663, "SWMG_SetGunBankSensingRadius"        , 0                                                },
	{ 664, "SWMG_SetGunBankInaccuracy"           , 0                                                },
	{ 665, "SWMG_GetIsInvulnerable"              , 0                                                },
	{ 666, "SWMG_StartInvulnerability"           , 0                                                },
	{ 667, "SWMG_GetPlayerMaxSpeed"              , 0                                                },
	{ 668, "SWMG_SetPlayerMaxSpeed"              , 0                                                },
	{ 669, "AddJournalWorldEntry"                , 0                                                },
	{ 670, "AddJournalWorldEntryStrref"          , 0                                                },
	{ 671, "BarkString"                          , 0                                                },
	{ 672, "DeleteJournalWorldAllEntries"        , 0                                                },
	{ 673, "DeleteJournalWorldEntry"             , 0                                                },
	{ 674, "DeleteJournalWorldEntryStrref"       , 0                                                },
	{ 675, "EffectForceDrain"                    , 0                                                },
	{ 676, "EffectPsychicStatic"                 , 0                                                },
	{ 677, "PlayVisualAreaEffect"                , 0                                                },
	{ 678, "SetJournalQuestEntryPicture"         , 0                                                },
	{ 679, "GetLocalBoolean"                     , &Functions::getLocalBoolean                      },
	{ 680, "SetLocalBoolean"                     , &Functions::setLocalBoolean                      },
	{ 681, "GetLocalNumber"                      , &Functions::getLocalNumber                       },
	{ 682, "SetLocalNumber"                      , &Functions::setLocalNumber                       },
	{ 683, "SWMG_GetSoundFrequency"              , 0                                                },
	{ 684, "SWMG_SetSoundFrequency"              , 0                                                },
	{ 685, "SWMG_GetSoundFrequencyIsRandom"      , 0                                                },
	{ 686, "SWMG_SetSoundFrequencyIsRandom"      , 0                                                },
	{ 687, "SWMG_GetSoundVolume"                 , 0                                                },
	{ 688, "SWMG_SetSoundVolume"                 , 0                                                },
	{ 689, "SoundObjectGetPitchVariance"         , 0                                                },
	{ 690, "SoundObjectSetPitchVariance"         , 0                                                },
	{ 691, "SoundObjectGetVolume"                , 0                                                },
	{ 692, "GetGlobalLocation"                   , 0                                                },
	{ 693, "SetGlobalLocation"                   , 0                                                },
	{ 694, "AddAvailableNPCByObject"             , 0                                                },
	{ 695, "RemoveAvailableNPC"                  , 0                                                },
	{ 696, "IsAvailableCreature"                 , &Functions::isAvailableCreature                  },
	{ 697, "AddAvailableNPCByTemplate"           , &Functions::addAvailableNPCByTemplate            },
	{ 698, "SpawnAvailableNPC"                   , 0                                                },
	{ 699, "IsNPCPartyMember"                    , 0                                                },
	{ 700, "ActionBarkString"                    , 0                                                },
	{ 701, "GetIsConversationActive"             , &Functions::getIsConversationActive              },
	{ 702, "EffectLightsaberThrow"               , 0                                                },
	{ 703, "EffectWhirlWind"                     , 0                                                },
	{ 704, "GetPartyAIStyle"                     , 0                                                },
	{ 705, "GetNPCAIStyle"                       , 0                                                },
	{ 706, "SetPartyAIStyle"                     , 0                                                },
	{ 707, "SetNPCAIStyle"                       , 0                                                },
	{ 708, "SetNPCSelectability"                 , 0                                                },
	{ 709, "GetNPCSelectability"                 , 0                                                },
	{ 710, "ClearAllEffects"                     , 0                                                },
	{ 711, "GetLastConversation"                 , 0                                                },
	{ 712, "ShowPartySelectionGUI"               , &Functions::showPartySelectionGUI                },
	{ 713, "GetStandardFaction"                  , &Functions::getStandardFaction                   },
	{ 714, "GivePlotXP"                          , 0                                                },
	{ 715, "GetMinOneHP"                         , &Functions::getMinOneHP                          },
	{ 716, "SetMinOneHP"                         , &Functions::setMinOneHP                          },
	{ 717, "SWMG_GetPlayerTunnelInfinite"        , 0                                                },
	{ 718, "SWMG_SetPlayerTunnelInfinite"        , 0                                                },
	{ 719, "SetGlobalFadeIn"                     , &Functions::setGlobalFadeIn                      },
	{ 720, "SetGlobalFadeOut"                    , &Functions::setGlobalFadeOut                     },
	{ 721, "GetLastHostileTarget"                , 0                                                },
	{ 722, "GetLastAttackAction"                 , 0                                                },
	{ 723, "GetLastForcePowerUsed"               , 0                                                },
	{ 724, "GetLastCombatFeatUsed"               , 0                                                },
	{ 725, "GetLastAttackResult"                 , 0                                                },
	{ 726, "GetWasForcePowerSuccessful"          , 0                                                },
	{ 727, "GetFirstAttacker"                    , 0                                                },
	{ 728, "GetNextAttacker"                     , 0                                                },
	{ 729, "SetFormation"                        , 0                                                },
	{ 730, "ActionFollowLeader"                  , &Functions::actionFollowLeader                   },
	{ 731, "SetForcePowerUnsuccessful"           , 0                                                },
	{ 732, "GetIsDebilitated"                    , 0                                                },
	{ 733, "PlayMovie"                           , &Functions::playMovie                            },
	{ 734, "SaveNPCState"                        , 0                                                },
	{ 735, "GetCategoryFromTalent"               , 0                                                },
	{ 736, "SurrenderByFaction"                  , 0                                                },
	{ 737, "ChangeFactionByFaction"              , 0                                                },
	{ 738, "PlayRoomAnimation"                   , 0                                                },
	{ 739, "ShowGalaxyMap"                       , 0                                                },
	{ 740, "SetPlanetSelectable"                 , 0                                                },
	{ 741, "GetPlanetSelectable"                 , 0                                                },
	{ 742, "SetPlanetAvailable"                  , 0                                                },
	{ 743, "GetPlanetAvailable"                  , 0                                                },
	{ 744, "GetSelectedPlanet"                   , 0                                                },
	{ 745, "SoundObjectFadeAndStop"              , 0                                                },
	{ 746, "SetAreaFogColor"                     , 0                                                },
	{ 747, "ChangeItemCost"                      , 0                                                },
	{ 748, "GetIsLiveContentAvailable"           , 0                                                },
	{ 749, "ResetDialogState"                    , 0                                                },
	{ 750, "SetGoodEvilValue"                    , 0                                                },
	{ 751, "GetIsPoisoned"                       , 0                                                },
	{ 752, "GetSpellTarget"                      , 0                                                },
	{ 753, "SetSoloMode"                         , 0                                                },
	{ 754, "EffectCutSceneHorrified"             , 0                                                },
	{ 755, "EffectCutSceneParalyze"              , 0                                                },
	{ 756, "EffectCutSceneStunned"               , 0                                                },
	{ 757, "CancelPostDialogCharacterSwitch"     , 0                                                },
	{ 758, "SetMaxHitPoints"                     , &Functions::setMaxHitPoints                      },
	{ 759, "NoClicksFor"                         , 0                                                },
	{ 760, "HoldWorldFadeInForDialog"            , 0                                                },
	{ 761, "ShipBuild"                           , 0                                                },
	{ 762, "SurrenderRetainBuffs"                , 0                                                },
	{ 763, "SuppressStatusSummaryEntry"          , 0                                                },
	{ 764, "GetCheatCode"                        , 0                                                },
	{ 765, "SetMusicVolume"                      , 0                                                },
	{ 766, "CreateItemOnFloor"                   , 0                                                },
	{ 767, "SetAvailableNPCId"                   , 0                                                },
	{ 768, "IsMoviePlaying"                      , 0                                                },
	{ 769, "QueueMovie"                          , 0                                                },
	{ 770, "PlayMovieQueue"                      , 0                                                },
	{ 771, "YavinHackCloseDoor"                  , 0                                                }
};

/** The table defining the signature (return type and type of parameters) of each engine function. */
const Functions::FunctionSignature Functions::kFunctionSignatures[] = {
	{   0, kTypeInt       , { kTypeInt } },
	{   1, kTypeVoid      , { kTypeString } },
	{   2, kTypeVoid      , { kTypeFloat, kTypeInt, kTypeInt } },
	{   3, kTypeString    , { kTypeFloat, kTypeInt, kTypeInt } },
	{   4, kTypeVoid      , { kTypeInt } },
	{   5, kTypeVoid      , { kTypeObject } },
	{   6, kTypeVoid      , { kTypeObject, kTypeScriptState } },
	{   7, kTypeVoid      , { kTypeFloat, kTypeScriptState } },
	{   8, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt } },
	{   9, kTypeVoid      , { } },
	{  10, kTypeVoid      , { kTypeFloat } },
	{  11, kTypeInt       , { kTypeInt } },
	{  12, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  13, kTypeInt       , { kTypeInt } },
	{  14, kTypeVoid      , { kTypeInt } },
	{  15, kTypeInt       , { } },
	{  16, kTypeInt       , { } },
	{  17, kTypeInt       , { } },
	{  18, kTypeInt       , { } },
	{  19, kTypeInt       , { } },
	{  20, kTypeVoid      , { } },
	{  21, kTypeVoid      , { kTypeEngineType, kTypeInt } },
	{  22, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  23, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  24, kTypeObject    , { kTypeObject } },
	{  25, kTypeObject    , { } },
	{  26, kTypeObject    , { } },
	{  27, kTypeVector    , { kTypeObject } },
	{  28, kTypeFloat     , { kTypeObject } },
	{  29, kTypeObject    , { kTypeObject } },
	{  30, kTypeObject    , { kTypeObject, kTypeString } },
	{  31, kTypeObject    , { kTypeString, kTypeObject, kTypeInt } },
	{  32, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  33, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  34, kTypeVoid      , { kTypeObject } },
	{  35, kTypeVoid      , { kTypeObject } },
	{  36, kTypeObject    , { kTypeObject } },
	{  37, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  38, kTypeObject    , { kTypeInt, kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  39, kTypeVoid      , { kTypeString, kTypeInt } },
	{  40, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{  41, kTypeFloat     , { kTypeObject } },
	{  42, kTypeInt       , { kTypeObject } },
	{  43, kTypeVoid      , { kTypeObject } },
	{  44, kTypeVoid      , { kTypeObject } },
	{  45, kTypeVoid      , { kTypeFloat } },
	{  46, kTypeVoid      , { kTypeString } },
	{  47, kTypeObject    , { } },
	{  48, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  49, kTypeInt       , { kTypeObject } },
	{  50, kTypeInt       , { kTypeObject } },
	{  51, kTypeEngineType, { } },
	{  52, kTypeObject    , { } },
	{  53, kTypeInt       , { } },
	{  54, kTypeVoid      , { kTypeObject } },
	{  55, kTypeInt       , { kTypeObject } },
	{  56, kTypeInt       , { kTypeObject } },
	{  57, kTypeVoid      , { kTypeInt } },
	{  58, kTypeVoid      , { kTypeInt } },
	{  59, kTypeInt       , { kTypeString } },
	{  60, kTypeString    , { kTypeString } },
	{  61, kTypeString    , { kTypeString } },
	{  62, kTypeString    , { kTypeString, kTypeInt } },
	{  63, kTypeString    , { kTypeString, kTypeInt } },
	{  64, kTypeString    , { kTypeString, kTypeString, kTypeInt } },
	{  65, kTypeString    , { kTypeString, kTypeInt, kTypeInt } },
	{  66, kTypeInt       , { kTypeString, kTypeString } },
	{  67, kTypeFloat     , { kTypeFloat } },
	{  68, kTypeFloat     , { kTypeFloat } },
	{  69, kTypeFloat     , { kTypeFloat } },
	{  70, kTypeFloat     , { kTypeFloat } },
	{  71, kTypeFloat     , { kTypeFloat } },
	{  72, kTypeFloat     , { kTypeFloat } },
	{  73, kTypeFloat     , { kTypeFloat } },
	{  74, kTypeFloat     , { kTypeFloat } },
	{  75, kTypeFloat     , { kTypeFloat, kTypeFloat } },
	{  76, kTypeFloat     , { kTypeFloat } },
	{  77, kTypeInt       , { kTypeInt } },
	{  78, kTypeEngineType, { kTypeInt } },
	{  79, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  80, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  81, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  82, kTypeEngineType, { } },
	{  83, kTypeInt       , { kTypeObject } },
	{  84, kTypeInt       , { kTypeObject } },
	{  85, kTypeEngineType, { kTypeObject } },
	{  86, kTypeEngineType, { kTypeObject } },
	{  87, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{  88, kTypeInt       , { kTypeEngineType } },
	{  89, kTypeInt       , { kTypeEngineType } },
	{  90, kTypeInt       , { kTypeEngineType } },
	{  91, kTypeObject    , { kTypeEngineType } },
	{  92, kTypeString    , { kTypeInt } },
	{  93, kTypeObject    , { kTypeObject, kTypeInt } },
	{  94, kTypeObject    , { kTypeObject, kTypeInt } },
	{  95, kTypeInt       , { kTypeInt } },
	{  96, kTypeInt       , { kTypeInt } },
	{  97, kTypeInt       , { kTypeInt } },
	{  98, kTypeInt       , { kTypeInt } },
	{  99, kTypeInt       , { kTypeInt } },
	{ 100, kTypeInt       , { kTypeInt } },
	{ 101, kTypeInt       , { kTypeInt } },
	{ 102, kTypeInt       , { kTypeInt } },
	{ 103, kTypeInt       , { kTypeInt } },
	{ 104, kTypeFloat     , { kTypeVector } },
	{ 105, kTypeInt       , { } },
	{ 106, kTypeInt       , { kTypeObject } },
	{ 107, kTypeInt       , { kTypeObject } },
	{ 108, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 109, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 110, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 111, kTypeInt       , { } },
	{ 112, kTypeEngineType, { kTypeEngineType } },
	{ 113, kTypeEngineType, { kTypeEngineType } },
	{ 114, kTypeEngineType, { kTypeEngineType } },
	{ 115, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 116, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 117, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 118, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 119, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 120, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 121, kTypeFloat     , { kTypeInt } },
	{ 122, kTypeFloat     , { kTypeInt } },
	{ 123, kTypeFloat     , { kTypeInt } },
	{ 124, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 125, kTypeInt       , { kTypeObject } },
	{ 126, kTypeInt       , { } },
	{ 127, kTypeInt       , { kTypeObject } },
	{ 128, kTypeObject    , { kTypeInt, kTypeFloat, kTypeEngineType, kTypeInt, kTypeInt, kTypeVector } },
	{ 129, kTypeObject    , { kTypeInt, kTypeFloat, kTypeEngineType, kTypeInt, kTypeInt, kTypeVector } },
	{ 130, kTypeEngineType, { } },
	{ 131, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 132, kTypeEngineType, { kTypeInt } },
	{ 133, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 134, kTypeEngineType, { } },
	{ 135, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 136, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 137, kTypeVector    , { kTypeVector } },
	{ 138, kTypeInt       , { kTypeObject } },
	{ 139, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 140, kTypeInt       , { kTypeObject } },
	{ 141, kTypeVoid      , { kTypeVector, kTypeInt } },
	{ 142, kTypeVector    , { kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 143, kTypeVoid      , { kTypeVector } },
	{ 144, kTypeVector    , { kTypeFloat } },
	{ 145, kTypeFloat     , { kTypeVector } },
	{ 146, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 147, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 148, kTypeEngineType, { } },
	{ 149, kTypeEngineType, { kTypeInt } },
	{ 150, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 151, kTypeFloat     , { kTypeObject, kTypeObject } },
	{ 152, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 153, kTypeEngineType, { kTypeObject, kTypeInt } },
	{ 154, kTypeEngineType, { } },
	{ 155, kTypeObject    , { kTypeInt, kTypeObject } },
	{ 156, kTypeEngineType, { kTypeInt } },
	{ 157, kTypeEngineType, { } },
	{ 158, kTypeEngineType, { } },
	{ 159, kTypeEngineType, { } },
	{ 160, kTypeVoid      , { kTypeString, kTypeString } },
	{ 161, kTypeEngineType, { } },
	{ 162, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 163, kTypeInt       , { kTypeObject } },
	{ 164, kTypeEngineType, { kTypeInt, kTypeFloat } },
	{ 165, kTypeEngineType, { kTypeInt } },
	{ 166, kTypeInt       , { kTypeObject } },
	{ 167, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 168, kTypeString    , { kTypeObject } },
	{ 169, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 170, kTypeInt       , { kTypeEngineType } },
	{ 171, kTypeEngineType, { kTypeInt, kTypeString, kTypeString, kTypeString } },
	{ 172, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 173, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 174, kTypeInt       , { kTypeObject } },
	{ 175, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 176, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 177, kTypeInt       , { kTypeString, kTypeString } },
	{ 178, kTypeString    , { kTypeInt } },
	{ 179, kTypeInt       , { } },
	{ 180, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 181, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 182, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 183, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 184, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 185, kTypeInt       , { kTypeObject } },
	{ 186, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 187, kTypeInt       , { kTypeObject } },
	{ 188, kTypeFloat     , { kTypeObject } },
	{ 189, kTypeInt       , { kTypeObject } },
	{ 190, kTypeInt       , { kTypeObject } },
	{ 191, kTypeInt       , { kTypeObject } },
	{ 192, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 193, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 194, kTypeString    , { kTypeString } },
	{ 195, kTypeInt       , { } },
	{ 196, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 197, kTypeObject    , { kTypeString } },
	{ 198, kTypeObject    , { kTypeObject } },
	{ 199, kTypeEngineType, { kTypeEngineType, kTypeEngineType } },
	{ 200, kTypeObject    , { kTypeString, kTypeInt } },
	{ 201, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 202, kTypeVoid      , { kTypeFloat } },
	{ 203, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 204, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeInt, kTypeString, kTypeString, kTypeString, kTypeString, kTypeString, kTypeString, kTypeInt } },
	{ 205, kTypeVoid      , { } },
	{ 206, kTypeVoid      , { } },
	{ 207, kTypeEngineType, { kTypeInt, kTypeObject, kTypeInt, kTypeInt } },
	{ 208, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 209, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 210, kTypeString    , { } },
	{ 211, kTypeObject    , { kTypeObject } },
	{ 212, kTypeEngineType, { kTypeInt } },
	{ 213, kTypeEngineType, { kTypeObject } },
	{ 214, kTypeVoid      , { kTypeEngineType } },
	{ 215, kTypeEngineType, { kTypeVector, kTypeFloat } },
	{ 216, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeEngineType, kTypeFloat } },
	{ 217, kTypeInt       , { kTypeObject } },
	{ 218, kTypeFloat     , { kTypeFloat } },
	{ 219, kTypeFloat     , { kTypeFloat } },
	{ 220, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeObject, kTypeFloat } },
	{ 221, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 222, kTypeEngineType, { } },
	{ 223, kTypeVector    , { kTypeEngineType } },
	{ 224, kTypeEngineType, { } },
	{ 225, kTypeFloat     , { kTypeEngineType } },
	{ 226, kTypeObject    , { kTypeInt, kTypeInt, kTypeEngineType, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 227, kTypeObject    , { kTypeInt, kTypeObject, kTypeInt } },
	{ 228, kTypeObject    , { kTypeInt, kTypeEngineType, kTypeInt } },
	{ 229, kTypeObject    , { kTypeString, kTypeObject, kTypeInt } },
	{ 230, kTypeFloat     , { kTypeInt } },
	{ 231, kTypeInt       , { kTypeFloat } },
	{ 232, kTypeInt       , { kTypeString } },
	{ 233, kTypeFloat     , { kTypeString } },
	{ 234, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 235, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 236, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 237, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 238, kTypeObject    , { } },
	{ 239, kTypeString    , { kTypeInt } },
	{ 240, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 241, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeInt, kTypeFloat } },
	{ 242, kTypeObject    , { } },
	{ 243, kTypeObject    , { kTypeInt, kTypeString, kTypeEngineType, kTypeInt } },
	{ 244, kTypeEngineType, { kTypeObject, kTypeInt, kTypeInt } },
	{ 245, kTypeObject    , { } },
	{ 246, kTypeInt       , { } },
	{ 247, kTypeInt       , { } },
	{ 248, kTypeInt       , { } },
	{ 249, kTypeString    , { } },
	{ 250, kTypeEngineType, { kTypeInt } },
	{ 251, kTypeInt       , { } },
	{ 252, kTypeEngineType, { kTypeInt } },
	{ 253, kTypeString    , { kTypeObject } },
	{ 254, kTypeObject    , { } },
	{ 255, kTypeInt       , { kTypeString, kTypeObject } },
	{ 256, kTypeObject    , { } },
	{ 257, kTypeInt       , { } },
	{ 258, kTypeInt       , { } },
	{ 259, kTypeInt       , { } },
	{ 260, kTypeObject    , { } },
	{ 261, kTypeInt       , { } },
	{ 262, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 263, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{ 264, kTypeObject    , { kTypeObject } },
	{ 265, kTypeInt       , { } },
	{ 266, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 267, kTypeInt       , { } },
	{ 268, kTypeVoid      , { kTypeInt } },
	{ 269, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 270, kTypeEngineType, { } },
	{ 271, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 272, kTypeString    , { kTypeObject } },
	{ 273, kTypeEngineType, { kTypeInt } },
	{ 274, kTypeInt       , { kTypeObject, kTypeInt, kTypeObject } },
	{ 275, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 276, kTypeInt       , { kTypeObject } },
	{ 277, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 278, kTypeInt       , { kTypeObject } },
	{ 279, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 280, kTypeInt       , { kTypeObject } },
	{ 281, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 282, kTypeObject    , { } },
	{ 283, kTypeObject    , { } },
	{ 284, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 285, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 286, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 287, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 288, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt, kTypeObject } },
	{ 289, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 290, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 291, kTypeObject    , { } },
	{ 292, kTypeObject    , { } },
	{ 293, kTypeObject    , { } },
	{ 294, kTypeVoid      , { kTypeScriptState } },
	{ 295, kTypeEngineType, { } },
	{ 296, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 297, kTypeInt       , { kTypeObject } },
	{ 298, kTypeFloat     , { kTypeEngineType, kTypeEngineType } },
	{ 299, kTypeInt       , { kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 300, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{ 301, kTypeEngineType, { kTypeInt } },
	{ 302, kTypeEngineType, { kTypeInt } },
	{ 303, kTypeEngineType, { kTypeInt } },
	{ 304, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 305, kTypeInt       , { kTypeEngineType } },
	{ 306, kTypeInt       , { kTypeEngineType, kTypeObject } },
	{ 307, kTypeEngineType, { kTypeInt, kTypeObject, kTypeInt } },
	{ 308, kTypeEngineType, { kTypeInt, kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 309, kTypeVoid      , { kTypeEngineType, kTypeObject } },
	{ 310, kTypeVoid      , { kTypeEngineType, kTypeEngineType } },
	{ 311, kTypeInt       , { kTypeObject } },
	{ 312, kTypeInt       , { kTypeObject } },
	{ 313, kTypeVoid      , { kTypeEngineType } },
	{ 314, kTypeEngineType, { kTypeInt } },
	{ 315, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 316, kTypeObject    , { kTypeObject } },
	{ 317, kTypeInt       , { kTypeObject } },
	{ 318, kTypeInt       , { kTypeObject } },
	{ 319, kTypeFloat     , { kTypeObject, kTypeObject } },
	{ 320, kTypeInt       , { kTypeObject } },
	{ 321, kTypeInt       , { kTypeObject } },
	{ 322, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 323, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 324, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 325, kTypeInt       , { kTypeObject } },
	{ 326, kTypeObject    , { } },
	{ 327, kTypeVoid      , { kTypeObject } },
	{ 328, kTypeObject    , { kTypeObject } },
	{ 329, kTypeVoid      , { kTypeObject } },
	{ 330, kTypeObject    , { } },
	{ 331, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 332, kTypeInt       , { kTypeObject } },
	{ 333, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 334, kTypeFloat     , { kTypeEngineType, kTypeEngineType } },
	{ 335, kTypeFloat     , { kTypeObject } },
	{ 336, kTypeObject    , { } },
	{ 337, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 338, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 339, kTypeObject    , { kTypeObject } },
	{ 340, kTypeObject    , { kTypeObject } },
	{ 341, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 342, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 343, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 344, kTypeInt       , { kTypeInt } },
	{ 345, kTypeInt       , { } },
	{ 346, kTypeObject    , { } },
	{ 347, kTypeObject    , { } },
	{ 348, kTypeObject    , { } },
	{ 349, kTypeObject    , { } },
	{ 350, kTypeObject    , { } },
	{ 351, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 352, kTypeInt       , { } },
	{ 353, kTypeObject    , { } },
	{ 354, kTypeVoid      , { kTypeObject } },
	{ 355, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 356, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 357, kTypeEngineType, { kTypeEngineType } },
	{ 358, kTypeInt       , { kTypeObject } },
	{ 359, kTypeInt       , { kTypeEngineType } },
	{ 360, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 361, kTypeObject    , { } },
	{ 362, kTypeInt       , { kTypeEngineType } },
	{ 363, kTypeInt       , { kTypeEngineType } },
	{ 364, kTypeVoid      , { kTypeInt, kTypeString, kTypeInt, kTypeInt, kTypeObject } },
	{ 365, kTypeInt       , { } },
	{ 366, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 367, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt } },
	{ 368, kTypeVoid      , { kTypeString } },
	{ 369, kTypeInt       , { kTypeString } },
	{ 370, kTypeInt       , { kTypeInt } },
	{ 371, kTypeInt       , { kTypeInt } },
	{ 372, kTypeEngineType, { kTypeInt } },
	{ 373, kTypeEngineType, { kTypeInt } },
	{ 374, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 375, kTypeObject    , { } },
	{ 376, kTypeObject    , { } },
	{ 377, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 378, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{ 379, kTypeVoid      , { } },
	{ 380, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 381, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 382, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 383, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeFloat } },
	{ 384, kTypeInt       , { kTypeString } },
	{ 385, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 386, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 387, kTypeEngineType, { kTypeFloat } },
	{ 388, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 389, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 390, kTypeInt       , { kTypeObject, kTypeEngineType } },
	{ 391, kTypeEngineType, { } },
	{ 392, kTypeEngineType, { } },
	{ 393, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 394, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 395, kTypeInt       , { kTypeObject } },
	{ 396, kTypeString    , { kTypeInt } },
	{ 397, kTypeInt       , { kTypeObject } },
	{ 398, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 399, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 400, kTypeVoid      , { kTypeObject } },
	{ 401, kTypeInt       , { kTypeObject } },
	{ 402, kTypeEngineType, { kTypeObject } },
	{ 403, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 404, kTypeVoid      , { } },
	{ 405, kTypeInt       , { } },
	{ 406, kTypeInt       , { } },
	{ 407, kTypeInt       , { } },
	{ 408, kTypeInt       , { } },
	{ 409, kTypeInt       , { kTypeObject } },
	{ 410, kTypeObject    , { } },
	{ 411, kTypeEngineType, { } },
	{ 412, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 413, kTypeVoid      , { kTypeObject } },
	{ 414, kTypeVoid      , { kTypeObject } },
	{ 415, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 416, kTypeVoid      , { kTypeObject, kTypeVector } },
	{ 417, kTypeVoid      , { kTypeString, kTypeObject } },
	{ 418, kTypeInt       , { kTypeObject } },
	{ 419, kTypeObject    , { } },
	{ 420, kTypeEngineType, { } },
	{ 421, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 422, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 423, kTypeInt       , { } },
	{ 424, kTypeEngineType, { kTypeObject, kTypeEngineType, kTypeObject } },
	{ 425, kTypeVoid      , { kTypeObject } },
	{ 426, kTypeVoid      , { kTypeObject } },
	{ 427, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 428, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 429, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 430, kTypeVoid      , { kTypeObject } },
	{ 431, kTypeVoid      , { kTypeObject } },
	{ 432, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 433, kTypeVoid      , { kTypeObject } },
	{ 434, kTypeVoid      , { kTypeObject } },
	{ 435, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 436, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 437, kTypeObject    , { } },
	{ 438, kTypeObject    , { } },
	{ 439, kTypeObject    , { } },
	{ 440, kTypeObject    , { } },
	{ 441, kTypeEngineType, { } },
	{ 442, kTypeObject    , { } },
	{ 443, kTypeInt       , { kTypeObject } },
	{ 444, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 445, kTypeInt       , { kTypeObject } },
	{ 446, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 447, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 448, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 449, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 450, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 451, kTypeEngineType, { kTypeInt } },
	{ 452, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 453, kTypeEngineType, { kTypeInt, kTypeInt } },
	{ 454, kTypeEngineType, { kTypeInt } },
	{ 455, kTypeInt       , { kTypeObject } },
	{ 456, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 457, kTypeEngineType, { kTypeInt } },
	{ 458, kTypeEngineType, { kTypeInt } },
	{ 459, kTypeEngineType, { kTypeInt } },
	{ 460, kTypeEngineType, { kTypeInt } },
	{ 461, kTypeVoid      , { kTypeInt } },
	{ 462, kTypeInt       , { } },
	{ 463, kTypeEngineType, { kTypeInt } },
	{ 464, kTypeInt       , { } },
	{ 465, kTypeEngineType, { } },
	{ 466, kTypeEngineType, { } },
	{ 467, kTypeEngineType, { } },
	{ 468, kTypeVoid      , { kTypeInt } },
	{ 469, kTypeEngineType, { kTypeInt } },
	{ 470, kTypeEngineType, { kTypeInt } },
	{ 471, kTypeEngineType, { } },
	{ 472, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 473, kTypeEngineType, { kTypeInt } },
	{ 474, kTypeInt       , { } },
	{ 475, kTypeInt       , { kTypeObject } },
	{ 476, kTypeVoid      , { } },
	{ 477, kTypeEngineType, { kTypeInt } },
	{ 478, kTypeVoid      , { kTypeInt } },
	{ 479, kTypeInt       , { kTypeObject } },
	{ 480, kTypeVoid      , { kTypeObject } },
	{ 481, kTypeInt       , { } },
	{ 482, kTypeVoid      , { kTypeInt } },
	{ 483, kTypeVoid      , { kTypeObject } },
	{ 484, kTypeVoid      , { kTypeObject } },
	{ 485, kTypeEngineType, { kTypeInt } },
	{ 486, kTypeObject    , { kTypeObject } },
	{ 487, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{ 488, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 489, kTypeObject    , { } },
	{ 490, kTypeObject    , { kTypeObject } },
	{ 491, kTypeInt       , { kTypeObject } },
	{ 492, kTypeInt       , { kTypeObject } },
	{ 493, kTypeInt       , { kTypeObject } },
	{ 494, kTypeFloat     , { kTypeObject } },
	{ 495, kTypeObject    , { kTypeObject } },
	{ 496, kTypeInt       , { kTypeObject } },
	{ 497, kTypeInt       , { kTypeObject } },
	{ 498, kTypeInt       , { } },
	{ 499, kTypeVoid      , { kTypeInt } },
	{ 500, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 501, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 502, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeInt } },
	{ 503, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 504, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 505, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 506, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 507, kTypeVoid      , { kTypeObject, kTypeVector, kTypeInt } },
	{ 508, kTypeVoid      , { kTypeInt } },
	{ 509, kTypeVoid      , { kTypeString, kTypeString, kTypeString, kTypeString, kTypeString, kTypeString, kTypeString, kTypeString } },
	{ 510, kTypeVoid      , { } },
	{ 511, kTypeInt       , { kTypeObject } },
	{ 512, kTypeVoid      , { } },
	{ 513, kTypeInt       , { } },
	{ 514, kTypeInt       , { } },
	{ 515, kTypeVoid      , { kTypeVector, kTypeInt } },
	{ 516, kTypeVoid      , { kTypeInt } },
	{ 517, kTypeVoid      , { kTypeInt } },
	{ 518, kTypeVoid      , { kTypeInt } },
	{ 519, kTypeInt       , { } },
	{ 520, kTypeVoid      , { kTypeFloat } },
	{ 521, kTypeFloat     , { } },
	{ 522, kTypeInt       , { kTypeObject } },
	{ 523, kTypeFloat     , { } },
	{ 524, kTypeInt       , { kTypeObject } },
	{ 525, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 526, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt } },
	{ 527, kTypeInt       , { kTypeObject } },
	{ 528, kTypeInt       , { kTypeObject } },
	{ 529, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 530, kTypeInt       , { kTypeObject } },
	{ 531, kTypeInt       , { kTypeObject } },
	{ 532, kTypeInt       , { kTypeObject } },
	{ 533, kTypeObject    , { kTypeObject } },
	{ 534, kTypeString    , { kTypeObject } },
	{ 535, kTypeInt       , { kTypeObject } },
	{ 536, kTypeInt       , { kTypeObject } },
	{ 537, kTypeInt       , { kTypeObject } },
	{ 538, kTypeInt       , { kTypeObject } },
	{ 539, kTypeInt       , { kTypeObject } },
	{ 540, kTypeInt       , { kTypeObject } },
	{ 541, kTypeInt       , { kTypeObject } },
	{ 542, kTypeObject    , { } },
	{ 543, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 544, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 545, kTypeInt       , { kTypeObject } },
	{ 546, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 547, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 548, kTypeObject    , { } },
	{ 549, kTypeObject    , { } },
	{ 550, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 551, kTypeInt       , { kTypeObject } },
	{ 552, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 553, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 554, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeString } },
	{ 555, kTypeVoid      , { kTypeObject } },
	{ 556, kTypeObject    , { kTypeObject } },
	{ 557, kTypeVoid      , { } },
	{ 558, kTypeInt       , { kTypeObject } },
	{ 559, kTypeInt       , { kTypeObject } },
	{ 560, kTypeVoid      , { kTypeString } },
	{ 561, kTypeString    , { } },
	{ 562, kTypeObject    , { kTypeObject } },
	{ 563, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{ 564, kTypeVoid      , { kTypeInt } },
	{ 565, kTypeInt       , { } },
	{ 566, kTypeInt       , { kTypeObject } },
	{ 567, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 568, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 569, kTypeInt       , { kTypeObject } },
	{ 570, kTypeInt       , { kTypeObject } },
	{ 571, kTypeFloat     , { kTypeInt } },
	{ 572, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 573, kTypeVoid      , { kTypeObject } },
	{ 574, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 575, kTypeInt       , { kTypeInt } },
	{ 576, kTypeInt       , { kTypeObject } },
	{ 577, kTypeObject    , { kTypeInt } },
	{ 578, kTypeInt       , { kTypeString } },
	{ 579, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 580, kTypeInt       , { kTypeString } },
	{ 581, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 582, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt, kTypeFloat } },
	{ 583, kTypeString    , { } },
	{ 584, kTypeString    , { } },
	{ 585, kTypeObject    , { kTypeString } },
	{ 586, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeInt } },
	{ 587, kTypeInt       , { } },
	{ 588, kTypeInt       , { } },
	{ 589, kTypeObject    , { } },
	{ 590, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 591, kTypeVoid      , { } },
	{ 592, kTypeVoid      , { } },
	{ 593, kTypeObject    , { } },
	{ 594, kTypeObject    , { } },
	{ 595, kTypeInt       , { } },
	{ 596, kTypeInt       , { } },
	{ 597, kTypeString    , { kTypeObject } },
	{ 598, kTypeVoid      , { } },
	{ 599, kTypeInt       , { kTypeObject } },
	{ 600, kTypeInt       , { kTypeObject } },
	{ 601, kTypeInt       , { kTypeObject } },
	{ 602, kTypeInt       , { kTypeObject } },
	{ 603, kTypeInt       , { kTypeObject } },
	{ 604, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 605, kTypeVoid      , { } },
	{ 606, kTypeInt       , { } },
	{ 607, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 608, kTypeFloat     , { } },
	{ 609, kTypeFloat     , { } },
	{ 610, kTypeVoid      , { kTypeFloat, kTypeFloat } },
	{ 611, kTypeObject    , { } },
	{ 612, kTypeInt       , { } },
	{ 613, kTypeObject    , { kTypeInt } },
	{ 614, kTypeInt       , { } },
	{ 615, kTypeObject    , { kTypeInt } },
	{ 616, kTypeInt       , { kTypeObject } },
	{ 617, kTypeInt       , { kTypeObject } },
	{ 618, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 619, kTypeFloat     , { kTypeObject } },
	{ 620, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 621, kTypeInt       , { kTypeObject } },
	{ 622, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 623, kTypeVector    , { kTypeObject } },
	{ 624, kTypeInt       , { kTypeObject } },
	{ 625, kTypeString    , { kTypeObject, kTypeInt } },
	{ 626, kTypeString    , { kTypeObject, kTypeInt } },
	{ 627, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 628, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 629, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 630, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 631, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 632, kTypeVoid      , { kTypeObject, kTypeInt, kTypeString } },
	{ 633, kTypeVoid      , { kTypeObject, kTypeInt, kTypeString } },
	{ 634, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 635, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 636, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 637, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 638, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 639, kTypeString    , { } },
	{ 640, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 641, kTypeVector    , { } },
	{ 642, kTypeFloat     , { } },
	{ 643, kTypeFloat     , { } },
	{ 644, kTypeFloat     , { } },
	{ 645, kTypeFloat     , { } },
	{ 646, kTypeVector    , { } },
	{ 647, kTypeVoid      , { kTypeVector } },
	{ 648, kTypeVoid      , { kTypeFloat } },
	{ 649, kTypeVoid      , { kTypeFloat } },
	{ 650, kTypeVoid      , { kTypeFloat } },
	{ 651, kTypeVoid      , { kTypeFloat } },
	{ 652, kTypeVoid      , { kTypeVector } },
	{ 653, kTypeVector    , { } },
	{ 654, kTypeVoid      , { kTypeVector } },
	{ 655, kTypeVector    , { } },
	{ 656, kTypeVoid      , { kTypeVector } },
	{ 657, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 658, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 659, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 660, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 661, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 662, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 663, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 664, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 665, kTypeInt       , { kTypeObject } },
	{ 666, kTypeVoid      , { kTypeObject } },
	{ 667, kTypeFloat     , { } },
	{ 668, kTypeVoid      , { kTypeFloat } },
	{ 669, kTypeVoid      , { kTypeInt, kTypeString, kTypeString } },
	{ 670, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 671, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 672, kTypeVoid      , { } },
	{ 673, kTypeVoid      , { kTypeInt } },
	{ 674, kTypeVoid      , { kTypeInt } },
	{ 675, kTypeEngineType, { kTypeInt } },
	{ 676, kTypeEngineType, { } },
	{ 677, kTypeVoid      , { kTypeInt, kTypeEngineType } },
	{ 678, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 679, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 680, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 681, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 682, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 683, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 684, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 685, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 686, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 687, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 688, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 689, kTypeFloat     , { kTypeObject } },
	{ 690, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 691, kTypeInt       , { kTypeObject } },
	{ 692, kTypeEngineType, { kTypeString } },
	{ 693, kTypeVoid      , { kTypeString, kTypeEngineType } },
	{ 694, kTypeInt       , { kTypeInt, kTypeObject } },
	{ 695, kTypeInt       , { kTypeInt } },
	{ 696, kTypeInt       , { kTypeInt } },
	{ 697, kTypeInt       , { kTypeInt, kTypeString } },
	{ 698, kTypeObject    , { kTypeInt, kTypeEngineType } },
	{ 699, kTypeInt       , { kTypeInt } },
	{ 700, kTypeVoid      , { kTypeInt } },
	{ 701, kTypeInt       , { } },
	{ 702, kTypeEngineType, { kTypeObject, kTypeObject, kTypeObject, kTypeInt } },
	{ 703, kTypeEngineType, { } },
	{ 704, kTypeInt       , { } },
	{ 705, kTypeInt       , { kTypeObject } },
	{ 706, kTypeVoid      , { kTypeInt } },
	{ 707, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 708, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 709, kTypeInt       , { kTypeInt } },
	{ 710, kTypeVoid      , { } },
	{ 711, kTypeString    , { } },
	{ 712, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt } },
	{ 713, kTypeInt       , { kTypeObject } },
	{ 714, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 715, kTypeInt       , { kTypeObject } },
	{ 716, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 717, kTypeVector    , { } },
	{ 718, kTypeVoid      , { kTypeVector } },
	{ 719, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 720, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 721, kTypeObject    , { kTypeObject } },
	{ 722, kTypeInt       , { kTypeObject } },
	{ 723, kTypeInt       , { kTypeObject } },
	{ 724, kTypeInt       , { kTypeObject } },
	{ 725, kTypeInt       , { kTypeObject } },
	{ 726, kTypeInt       , { kTypeObject } },
	{ 727, kTypeObject    , { kTypeObject } },
	{ 728, kTypeObject    , { kTypeObject } },
	{ 729, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{ 730, kTypeVoid      , { } },
	{ 731, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 732, kTypeInt       , { kTypeObject } },
	{ 733, kTypeVoid      , { kTypeString } },
	{ 734, kTypeVoid      , { kTypeInt } },
	{ 735, kTypeInt       , { kTypeEngineType } },
	{ 736, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 737, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 738, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 739, kTypeVoid      , { kTypeInt } },
	{ 740, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 741, kTypeInt       , { kTypeInt } },
	{ 742, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 743, kTypeInt       , { kTypeInt } },
	{ 744, kTypeInt       , { } },
	{ 745, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 746, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 747, kTypeVoid      , { kTypeString, kTypeFloat } },
	{ 748, kTypeInt       , { kTypeInt } },
	{ 749, kTypeVoid      , { } },
	{ 750, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 751, kTypeInt       , { kTypeObject } },
	{ 752, kTypeObject    , { kTypeObject } },
	{ 753, kTypeVoid      , { kTypeInt } },
	{ 754, kTypeEngineType, { } },
	{ 755, kTypeEngineType, { } },
	{ 756, kTypeEngineType, { } },
	{ 757, kTypeVoid      , { } },
	{ 758, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 759, kTypeVoid      , { kTypeFloat } },
	{ 760, kTypeVoid      , { } },
	{ 761, kTypeInt       , { } },
	{ 762, kTypeVoid      , { } },
	{ 763, kTypeVoid      , { kTypeInt } },
	{ 764, kTypeInt       , { kTypeInt } },
	{ 765, kTypeVoid      , { kTypeFloat } },
	{ 766, kTypeObject    , { kTypeString, kTypeEngineType, kTypeInt } },
	{ 767, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 768, kTypeInt       , { } },
	{ 769, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 770, kTypeVoid      , { kTypeInt } },
	{ 771, kTypeVoid      , { kTypeObject } }
};

/** The table defining the default values for the parameters of each engine function. */
const Functions::FunctionDefaults Functions::kFunctionDefaults[] = {
	{   0, { } },
	{   1, { } },
	{   2, { &kDefaultInt18, &kDefaultInt9 } },
	{   3, { &kDefaultInt18, &kDefaultInt9 } },
	{   4, { } },
	{   5, { } },
	{   6, { } },
	{   7, { } },
	{   8, { &kDefaultIntMinus1 } },
	{   9, { } },
	{  10, { } },
	{  11, { } },
	{  12, { } },
	{  13, { } },
	{  14, { } },
	{  15, { } },
	{  16, { } },
	{  17, { } },
	{  18, { } },
	{  19, { } },
	{  20, { } },
	{  21, { &kDefaultFalse } },
	{  22, { &kDefaultFalse, &kDefaultFloat1_0 } },
	{  23, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{  24, { } },
	{  25, { } },
	{  26, { } },
	{  27, { } },
	{  28, { } },
	{  29, { } },
	{  30, { } },
	{  31, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{  32, { &kDefaultFalse } },
	{  33, { &kDefaultFalse } },
	{  34, { } },
	{  35, { } },
	{  36, { &kDefaultObjectSelf } },
	{  37, { &kDefaultFalse } },
	{  38, { &kDefaultObjectSelf, &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{  39, { &kDefaultTalkVolumeTalk } },
	{  40, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{  41, { } },
	{  42, { } },
	{  43, { } },
	{  44, { } },
	{  45, { } },
	{  46, { } },
	{  47, { } },
	{  48, { &kDefaultInt0 , &kDefaultFalse, &kDefaultInt0, &kDefaultProjectilePathTypeDefault, &kDefaultFalse } },
	{  49, { &kDefaultObjectSelf } },
	{  50, { &kDefaultObjectSelf } },
	{  51, { } },
	{  52, { } },
	{  53, { } },
	{  54, { } },
	{  55, { &kDefaultObjectSelf } },
	{  56, { &kDefaultObjectSelf } },
	{  57, { } },
	{  58, { } },
	{  59, { } },
	{  60, { } },
	{  61, { } },
	{  62, { } },
	{  63, { } },
	{  64, { } },
	{  65, { } },
	{  66, { } },
	{  67, { } },
	{  68, { } },
	{  69, { } },
	{  70, { } },
	{  71, { } },
	{  72, { } },
	{  73, { } },
	{  74, { } },
	{  75, { } },
	{  76, { } },
	{  77, { } },
	{  78, { } },
	{  79, { &kDefaultDamageTypeUniversal, &kDefaultDamagePowerNormal } },
	{  80, { } },
	{  81, { &kDefaultInt0 } },
	{  82, { } },
	{  83, { &kDefaultObjectSelf } },
	{  84, { } },
	{  85, { } },
	{  86, { } },
	{  87, { } },
	{  88, { } },
	{  89, { } },
	{  90, { } },
	{  91, { } },
	{  92, { } },
	{  93, { &kDefaultObjectInvalid, &kDefaultObjectTypeCreature } },
	{  94, { &kDefaultObjectInvalid, &kDefaultObjectTypeCreature } },
	{  95, { &kDefaultInt1 } },
	{  96, { &kDefaultInt1 } },
	{  97, { &kDefaultInt1 } },
	{  98, { &kDefaultInt1 } },
	{  99, { &kDefaultInt1 } },
	{ 100, { &kDefaultInt1 } },
	{ 101, { &kDefaultInt1 } },
	{ 102, { &kDefaultInt1 } },
	{ 103, { &kDefaultInt1 } },
	{ 104, { } },
	{ 105, { } },
	{ 106, { } },
	{ 107, { } },
	{ 108, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 109, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 110, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 111, { } },
	{ 112, { } },
	{ 113, { } },
	{ 114, { } },
	{ 115, { &kDefaultACDodgeBonus, &kDefaultACVsDamageTypeAll } },
	{ 116, { &kDefaultInt0 } },
	{ 117, { &kDefaultSavingThrowTypeAll } },
	{ 118, { &kDefaultAttackBonusMisc } },
	{ 119, { &kDefaultInt0 } },
	{ 120, { &kDefaultDamageTypeUniversal } },
	{ 121, { } },
	{ 122, { } },
	{ 123, { } },
	{ 124, { } },
	{ 125, { } },
	{ 126, { } },
	{ 127, { } },
	{ 128, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{ 129, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{ 130, { } },
	{ 131, { } },
	{ 132, { } },
	{ 133, { &kDefaultFalse, &kDefaultTrue } },
	{ 134, { } },
	{ 135, { } },
	{ 136, { } },
	{ 137, { } },
	{ 138, { } },
	{ 139, { } },
	{ 140, { } },
	{ 141, { } },
	{ 142, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{ 143, { } },
	{ 144, { } },
	{ 145, { } },
	{ 146, { &kDefaultTrue } },
	{ 147, { &kDefaultTrue } },
	{ 148, { } },
	{ 149, { &kDefaultForcePowerAllForcePowers } },
	{ 150, { } },
	{ 151, { } },
	{ 152, { &kDefaultInt0, &kDefaultInt0 } },
	{ 153, { &kDefaultInt0 } },
	{ 154, { } },
	{ 155, { &kDefaultObjectSelf } },
	{ 156, { } },
	{ 157, { } },
	{ 158, { } },
	{ 159, { } },
	{ 160, { } },
	{ 161, { } },
	{ 162, { &kDefaultObjectSelf } },
	{ 163, { &kDefaultObjectSelf } },
	{ 164, { } },
	{ 165, { } },
	{ 166, { } },
	{ 167, { &kDefaultFloat0_0 } },
	{ 168, { } },
	{ 169, { } },
	{ 170, { } },
	{ 171, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 172, { &kDefaultObjectSelf } },
	{ 173, { } },
	{ 174, { } },
	{ 175, { } },
	{ 176, { &kDefaultInt0 } },
	{ 177, { } },
	{ 178, { } },
	{ 179, { } },
	{ 180, { &kDefaultFalse } },
	{ 181, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 182, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 183, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 184, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 185, { } },
	{ 186, { } },
	{ 187, { } },
	{ 188, { } },
	{ 189, { } },
	{ 190, { } },
	{ 191, { } },
	{ 192, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 193, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 194, { } },
	{ 195, { } },
	{ 196, { &kDefaultTrue } },
	{ 197, { } },
	{ 198, { } },
	{ 199, { } },
	{ 200, { &kDefaultInt0 } },
	{ 201, { } },
	{ 202, { } },
	{ 203, { &kDefaultStringEmpty } },
	{ 204, { &kDefaultStringEmpty, &kDefaultFalse, &kDefaultConversationTypeCinematic, &kDefaultFalse, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultFalse } },
	{ 205, { } },
	{ 206, { } },
	{ 207, { &kDefaultFalse } },
	{ 208, { } },
	{ 209, { } },
	{ 210, { } },
	{ 211, { } },
	{ 212, { } },
	{ 213, { } },
	{ 214, { } },
	{ 215, { } },
	{ 216, { &kDefaultFloat0_0 } },
	{ 217, { } },
	{ 218, { } },
	{ 219, { } },
	{ 220, { &kDefaultFloat0_0 } },
	{ 221, { &kDefaultTalkVolumeTalk } },
	{ 222, { } },
	{ 223, { } },
	{ 224, { } },
	{ 225, { } },
	{ 226, { &kDefaultInt1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{ 227, { &kDefaultObjectTypeAll, &kDefaultObjectSelf, &kDefaultInt1 } },
	{ 228, { &kDefaultInt1 } },
	{ 229, { &kDefaultObjectSelf, &kDefaultInt1 } },
	{ 230, { } },
	{ 231, { } },
	{ 232, { } },
	{ 233, { } },
	{ 234, { &kDefaultInt0, &kDefaultFalse, &kDefaultProjectilePathTypeDefault, &kDefaultFalse } },
	{ 235, { &kDefaultObjectSelf } },
	{ 236, { &kDefaultObjectSelf } },
	{ 237, { &kDefaultObjectSelf } },
	{ 238, { } },
	{ 239, { } },
	{ 240, { &kDefaultTalkVolumeTalk } },
	{ 241, { &kDefaultFloat0_0, &kDefaultFalse, &kDefaultFloat0_0 } },
	{ 242, { } },
	{ 243, { &kDefaultFalse } },
	{ 244, { &kDefaultTrue } },
	{ 245, { } },
	{ 246, { } },
	{ 247, { } },
	{ 248, { } },
	{ 249, { } },
	{ 250, { } },
	{ 251, { } },
	{ 252, { &kDefaultInt0 } },
	{ 253, { } },
	{ 254, { } },
	{ 255, { &kDefaultStringEmpty, &kDefaultObjectInvalid } },
	{ 256, { } },
	{ 257, { } },
	{ 258, { } },
	{ 259, { } },
	{ 260, { } },
	{ 261, { } },
	{ 262, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{ 263, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultPersistentZoneActive } },
	{ 264, { &kDefaultObjectSelf } },
	{ 265, { } },
	{ 266, { } },
	{ 267, { } },
	{ 268, { } },
	{ 269, { &kDefaultInt0 } },
	{ 270, { } },
	{ 271, { } },
	{ 272, { } },
	{ 273, { } },
	{ 274, { &kDefaultObjectInvalid } },
	{ 275, { } },
	{ 276, { &kDefaultObjectSelf } },
	{ 277, { &kDefaultObjectSelf } },
	{ 278, { &kDefaultObjectSelf } },
	{ 279, { &kDefaultObjectSelf } },
	{ 280, { &kDefaultObjectSelf } },
	{ 281, { &kDefaultObjectSelf } },
	{ 282, { } },
	{ 283, { } },
	{ 284, { } },
	{ 285, { &kDefaultObjectSelf } },
	{ 286, { &kDefaultObjectSelf } },
	{ 287, { } },
	{ 288, { &kDefaultInt0, &kDefaultObjectInvalid } },
	{ 289, { &kDefaultObjectSelf } },
	{ 290, { &kDefaultObjectSelf } },
	{ 291, { } },
	{ 292, { } },
	{ 293, { } },
	{ 294, { } },
	{ 295, { } },
	{ 296, { &kDefaultObjectSelf } },
	{ 297, { &kDefaultObjectSelf } },
	{ 298, { } },
	{ 299, { &kDefaultSavingThrowTypeNone, &kDefaultObjectSelf } },
	{ 300, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{ 301, { } },
	{ 302, { } },
	{ 303, { } },
	{ 304, { &kDefaultObjectSelf } },
	{ 305, { } },
	{ 306, { &kDefaultObjectSelf } },
	{ 307, { &kDefaultObjectSelf, &kDefaultInt0 } },
	{ 308, { &kDefaultObjectSelf, &kDefaultInt0, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{ 309, { } },
	{ 310, { } },
	{ 311, { } },
	{ 312, { } },
	{ 313, { } },
	{ 314, { } },
	{ 315, { &kDefaultObjectSelf } },
	{ 316, { &kDefaultObjectSelf } },
	{ 317, { &kDefaultObjectSelf } },
	{ 318, { &kDefaultObjectSelf } },
	{ 319, { } },
	{ 320, { &kDefaultObjectSelf } },
	{ 321, { &kDefaultObjectSelf } },
	{ 322, { } },
	{ 323, { &kDefaultTrue, &kDefaultFalse } },
	{ 324, { } },
	{ 325, { } },
	{ 326, { } },
	{ 327, { &kDefaultObjectSelf } },
	{ 328, { } },
	{ 329, { } },
	{ 330, { } },
	{ 331, { &kDefaultObjectSelf } },
	{ 332, { } },
	{ 333, { } },
	{ 334, { } },
	{ 335, { } },
	{ 336, { } },
	{ 337, { } },
	{ 338, { } },
	{ 339, { &kDefaultObjectSelf } },
	{ 340, { &kDefaultObjectSelf } },
	{ 341, { &kDefaultObjectSelf } },
	{ 342, { &kDefaultObjectSelf } },
	{ 343, { &kDefaultObjectSelf } },
	{ 344, { } },
	{ 345, { } },
	{ 346, { } },
	{ 347, { } },
	{ 348, { } },
	{ 349, { } },
	{ 350, { } },
	{ 351, { } },
	{ 352, { } },
	{ 353, { } },
	{ 354, { &kDefaultObjectInvalid } },
	{ 355, { &kDefaultAlignmentAll, &kDefaultAlignmentAll } },
	{ 356, { } },
	{ 357, { } },
	{ 358, { } },
	{ 359, { } },
	{ 360, { &kDefaultFalse, &kDefaultFloat40_0 } },
	{ 361, { } },
	{ 362, { } },
	{ 363, { } },
	{ 364, { &kDefaultFalse, &kDefaultObjectInvalid } },
	{ 365, { } },
	{ 366, { } },
	{ 367, { &kDefaultFalse } },
	{ 368, { } },
	{ 369, { } },
	{ 370, { } },
	{ 371, { } },
	{ 372, { } },
	{ 373, { } },
	{ 374, { } },
	{ 375, { } },
	{ 376, { } },
	{ 377, { &kDefaultObjectSelf } },
	{ 378, { &kDefaultInt0, &kDefaultInt0 } },
	{ 379, { } },
	{ 380, { &kDefaultTrue } },
	{ 381, { &kDefaultTrue } },
	{ 382, { &kDefaultFalse, &kDefaultFloat30_0 } },
	{ 383, { &kDefaultFalse, &kDefaultFloat1_0, &kDefaultFloat30_0 } },
	{ 384, { } },
	{ 385, { &kDefaultInt1 } },
	{ 386, { } },
	{ 387, { } },
	{ 388, { } },
	{ 389, { } },
	{ 390, { } },
	{ 391, { } },
	{ 392, { } },
	{ 393, { } },
	{ 394, { } },
	{ 395, { } },
	{ 396, { } },
	{ 397, { } },
	{ 398, { } },
	{ 399, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{ 400, { &kDefaultObjectInvalid } },
	{ 401, { } },
	{ 402, { } },
	{ 403, { } },
	{ 404, { } },
	{ 405, { } },
	{ 406, { } },
	{ 407, { } },
	{ 408, { } },
	{ 409, { &kDefaultObjectSelf } },
	{ 410, { } },
	{ 411, { } },
	{ 412, { } },
	{ 413, { } },
	{ 414, { } },
	{ 415, { } },
	{ 416, { } },
	{ 417, { &kDefaultStringEmpty, &kDefaultObjectInvalid } },
	{ 418, { &kDefaultObjectSelf } },
	{ 419, { } },
	{ 420, { } },
	{ 421, { &kDefaultTrue, &kDefaultFalse } },
	{ 422, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{ 423, { } },
	{ 424, { &kDefaultObjectInvalid } },
	{ 425, { } },
	{ 426, { } },
	{ 427, { } },
	{ 428, { } },
	{ 429, { } },
	{ 430, { } },
	{ 431, { } },
	{ 432, { } },
	{ 433, { } },
	{ 434, { } },
	{ 435, { } },
	{ 436, { } },
	{ 437, { } },
	{ 438, { } },
	{ 439, { } },
	{ 440, { } },
	{ 441, { } },
	{ 442, { } },
	{ 443, { } },
	{ 444, { &kDefaultFalse } },
	{ 445, { } },
	{ 446, { } },
	{ 447, { &kDefaultAttackBonusMisc } },
	{ 448, { &kDefaultDamageTypeUniversal } },
	{ 449, { } },
	{ 450, { &kDefaultACDodgeBonus, &kDefaultACVsDamageTypeAll } },
	{ 451, { } },
	{ 452, { &kDefaultSavingThrowTypeAll } },
	{ 453, { } },
	{ 454, { } },
	{ 455, { &kDefaultObjectSelf } },
	{ 456, { } },
	{ 457, { } },
	{ 458, { } },
	{ 459, { } },
	{ 460, { } },
	{ 461, { } },
	{ 462, { } },
	{ 463, { } },
	{ 464, { } },
	{ 465, { } },
	{ 466, { } },
	{ 467, { } },
	{ 468, { } },
	{ 469, { } },
	{ 470, { } },
	{ 471, { } },
	{ 472, { &kDefaultInt0, &kDefaultInt0 } },
	{ 473, { } },
	{ 474, { } },
	{ 475, { } },
	{ 476, { } },
	{ 477, { } },
	{ 478, { } },
	{ 479, { } },
	{ 480, { } },
	{ 481, { } },
	{ 482, { } },
	{ 483, { } },
	{ 484, { } },
	{ 485, { } },
	{ 486, { &kDefaultObjectSelf } },
	{ 487, { } },
	{ 488, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 489, { } },
	{ 490, { &kDefaultObjectSelf } },
	{ 491, { } },
	{ 492, { } },
	{ 493, { } },
	{ 494, { } },
	{ 495, { &kDefaultObjectSelf } },
	{ 496, { } },
	{ 497, { } },
	{ 498, { } },
	{ 499, { } },
	{ 500, { } },
	{ 501, { &kDefaultProjectilePathTypeDefault } },
	{ 502, { &kDefaultProjectilePathTypeDefault } },
	{ 503, { } },
	{ 504, { } },
	{ 505, { } },
	{ 506, { } },
	{ 507, { } },
	{ 508, { } },
	{ 509, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringEmpty } },
	{ 510, { } },
	{ 511, { } },
	{ 512, { } },
	{ 513, { } },
	{ 514, { } },
	{ 515, { &kDefaultVector0, &kDefaultIntMinus1 } },
	{ 516, { } },
	{ 517, { } },
	{ 518, { } },
	{ 519, { } },
	{ 520, { } },
	{ 521, { } },
	{ 522, { &kDefaultObjectSelf } },
	{ 523, { } },
	{ 524, { } },
	{ 525, { &kDefaultTrue } },
	{ 526, { &kDefaultTrue } },
	{ 527, { } },
	{ 528, { } },
	{ 529, { } },
	{ 530, { } },
	{ 531, { } },
	{ 532, { } },
	{ 533, { } },
	{ 534, { } },
	{ 535, { } },
	{ 536, { } },
	{ 537, { } },
	{ 538, { } },
	{ 539, { } },
	{ 540, { } },
	{ 541, { } },
	{ 542, { } },
	{ 543, { &kDefaultObjectSelf } },
	{ 544, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 545, { &kDefaultObjectSelf } },
	{ 546, { } },
	{ 547, { } },
	{ 548, { } },
	{ 549, { } },
	{ 550, { } },
	{ 551, { } },
	{ 552, { } },
	{ 553, { } },
	{ 554, { &kDefaultTrue, &kDefaultTrue, &kDefaultInt0, &kDefaultStringEmpty } },
	{ 555, { } },
	{ 556, { &kDefaultObjectSelf } },
	{ 557, { } },
	{ 558, { } },
	{ 559, { } },
	{ 560, { } },
	{ 561, { } },
	{ 562, { } },
	{ 563, { &kDefaultFloat0_75 } },
	{ 564, { &kDefaultTrue } },
	{ 565, { } },
	{ 566, { } },
	{ 567, { } },
	{ 568, { } },
	{ 569, { } },
	{ 570, { } },
	{ 571, { } },
	{ 572, { } },
	{ 573, { } },
	{ 574, { } },
	{ 575, { } },
	{ 576, { } },
	{ 577, { } },
	{ 578, { } },
	{ 579, { } },
	{ 580, { } },
	{ 581, { } },
	{ 582, { } },
	{ 583, { } },
	{ 584, { } },
	{ 585, { } },
	{ 586, { &kDefaultInt1, &kDefaultInt0, &kDefaultInt0 } },
	{ 587, { } },
	{ 588, { } },
	{ 589, { } },
	{ 590, { &kDefaultInt0 } },
	{ 591, { } },
	{ 592, { } },
	{ 593, { } },
	{ 594, { } },
	{ 595, { } },
	{ 596, { } },
	{ 597, { &kDefaultObjectSelf } },
	{ 598, { } },
	{ 599, { &kDefaultObjectSelf } },
	{ 600, { &kDefaultObjectSelf } },
	{ 601, { &kDefaultObjectSelf } },
	{ 602, { &kDefaultObjectSelf } },
	{ 603, { &kDefaultObjectSelf } },
	{ 604, { } },
	{ 605, { } },
	{ 606, { } },
	{ 607, { } },
	{ 608, { } },
	{ 609, { } },
	{ 610, { } },
	{ 611, { } },
	{ 612, { } },
	{ 613, { } },
	{ 614, { } },
	{ 615, { } },
	{ 616, { } },
	{ 617, { } },
	{ 618, { } },
	{ 619, { } },
	{ 620, { } },
	{ 621, { } },
	{ 622, { } },
	{ 623, { } },
	{ 624, { } },
	{ 625, { } },
	{ 626, { } },
	{ 627, { } },
	{ 628, { } },
	{ 629, { } },
	{ 630, { } },
	{ 631, { } },
	{ 632, { } },
	{ 633, { } },
	{ 634, { } },
	{ 635, { } },
	{ 636, { } },
	{ 637, { } },
	{ 638, { } },
	{ 639, { } },
	{ 640, { } },
	{ 641, { } },
	{ 642, { } },
	{ 643, { } },
	{ 644, { } },
	{ 645, { } },
	{ 646, { } },
	{ 647, { } },
	{ 648, { } },
	{ 649, { } },
	{ 650, { } },
	{ 651, { } },
	{ 652, { } },
	{ 653, { } },
	{ 654, { } },
	{ 655, { } },
	{ 656, { } },
	{ 657, { } },
	{ 658, { } },
	{ 659, { } },
	{ 660, { } },
	{ 661, { } },
	{ 662, { } },
	{ 663, { } },
	{ 664, { } },
	{ 665, { } },
	{ 666, { } },
	{ 667, { } },
	{ 668, { } },
	{ 669, { &kDefaultStringWorldEntry } },
	{ 670, { } },
	{ 671, { } },
	{ 672, { } },
	{ 673, { } },
	{ 674, { } },
	{ 675, { } },
	{ 676, { } },
	{ 677, { } },
	{ 678, { &kDefaultTrue, &kDefaultFalse } },
	{ 679, { } },
	{ 680, { } },
	{ 681, { } },
	{ 682, { } },
	{ 683, { } },
	{ 684, { } },
	{ 685, { } },
	{ 686, { } },
	{ 687, { } },
	{ 688, { } },
	{ 689, { } },
	{ 690, { } },
	{ 691, { } },
	{ 692, { } },
	{ 693, { } },
	{ 694, { } },
	{ 695, { } },
	{ 696, { } },
	{ 697, { } },
	{ 698, { } },
	{ 699, { } },
	{ 700, { } },
	{ 701, { } },
	{ 702, { &kDefaultObjectInvalid, &kDefaultObjectInvalid, &kDefaultInt0 } },
	{ 703, { } },
	{ 704, { } },
	{ 705, { } },
	{ 706, { } },
	{ 707, { } },
	{ 708, { } },
	{ 709, { } },
	{ 710, { } },
	{ 711, { } },
	{ 712, { &kDefaultStringEmpty, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{ 713, { } },
	{ 714, { } },
	{ 715, { } },
	{ 716, { } },
	{ 717, { } },
	{ 718, { } },
	{ 719, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{ 720, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{ 721, { &kDefaultObjectSelf } },
	{ 722, { &kDefaultObjectSelf } },
	{ 723, { &kDefaultObjectSelf } },
	{ 724, { &kDefaultObjectSelf } },
	{ 725, { &kDefaultObjectSelf } },
	{ 726, { &kDefaultObjectSelf } },
	{ 727, { &kDefaultObjectSelf } },
	{ 728, { &kDefaultObjectSelf } },
	{ 729, { } },
	{ 730, { } },
	{ 731, { &kDefaultObjectSelf } },
	{ 732, { &kDefaultObjectSelf } },
	{ 733, { } },
	{ 734, { } },
	{ 735, { } },
	{ 736, { } },
	{ 737, { } },
	{ 738, { } },
	{ 739, { } },
	{ 740, { } },
	{ 741, { } },
	{ 742, { } },
	{ 743, { } },
	{ 744, { } },
	{ 745, { } },
	{ 746, { } },
	{ 747, { } },
	{ 748, { } },
	{ 749, { } },
	{ 750, { } },
	{ 751, { } },
	{ 752, { &kDefaultObjectSelf } },
	{ 753, { } },
	{ 754, { } },
	{ 755, { } },
	{ 756, { } },
	{ 757, { } },
	{ 758, { } },
	{ 759, { } },
	{ 760, { } },
	{ 761, { } },
	{ 762, { } },
	{ 763, { &kDefaultInt1 } },
	{ 764, { } },
	{ 765, { &kDefaultFloat1_0 } },
	{ 766, { &kDefaultFalse } },
	{ 767, { } },
	{ 768, { } },
	{ 769, { } },
	{ 770, { } },
	{ 771, { } }
};

} // End of namespace KotOR

} // End of namespace Engines

#endif // ENGINES_KOTOR_SCRIPT_TABLES_H
