/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Tables defining the engine functions in Jade Empire.
 */

/* The functions are defined by three tables:
 * - kFunctionPointers
 * - kFunctionSignatures
 * - kFunctionDefaults
 *
 * kFunctionPointers provides the ID and name of the engine function, and a
 * pointer to Functions method doing the actual work. If the function pointer
 * is 0, a default unimplementedFunction method is used that just prints
 * the name of the function when it's called.
 *
 * kFunctionSignatures defines the signature of the function, i.e. the types
 * of the return value and its parameters.
 *
 * kFunctionDefaults pins default values to parameters. I.e. if a function
 * Foobar(int mode, int doEverything=kBooleanFalse) is called with only one parameters,
 * the default value kBooleanFalse is applied to the doEverything parameters. The
 * kFunctionDefaults entry for the function then contains a reference to a
 * false value. Because of ambiguities otherwise, the default values have to be
 * "aligned" to the right; only the last-most n parameters are allowed to have
 * default values, with no gaps.
 *
 * Please note that all three tables have to be of the same length, and that
 * the order of the functions have to be the same in all three tables.
 */

#ifndef ENGINES_JADE_SCRIPT_FUNCTION_TABLES_H
#define ENGINES_JADE_SCRIPT_FUNCTION_TABLES_H

#include "src/aurora/nwscript/types.h"

#include "src/engines/jade/types.h"
#include "src/engines/jade/object.h"

namespace Engines {

namespace Jade {

using Aurora::NWScript::kTypeVoid;
using Aurora::NWScript::kTypeInt;
using Aurora::NWScript::kTypeFloat;
using Aurora::NWScript::kTypeString;
using Aurora::NWScript::kTypeObject;
using Aurora::NWScript::kTypeVector;
using Aurora::NWScript::kTypeEngineType;
using Aurora::NWScript::kTypeScriptState;
using Aurora::NWScript::kTypeAny;

/* Default values as used by the function parameters, if no explicit value
 * is given at the time of the calling.
 */

static Jade::Object kDefaultValueObjectInvalid(kObjectTypeInvalid);
static Jade::Object kDefaultValueObjectCreature(kObjectTypeCreature);
static Jade::Object kDefaultValueObjectSelf(kObjectTypeSelf);

static const Aurora::NWScript::Variable kDefaultIntMinus1((int32_t)  -1);
static const Aurora::NWScript::Variable kDefaultInt0     ((int32_t)   0);
static const Aurora::NWScript::Variable kDefaultInt1     ((int32_t)   1);
static const Aurora::NWScript::Variable kDefaultInt7     ((int32_t)   7);
static const Aurora::NWScript::Variable kDefaultInt9     ((int32_t)   9);
static const Aurora::NWScript::Variable kDefaultInt18    ((int32_t)  18);

static const Aurora::NWScript::Variable kDefaultFloatMinus1_0  (-1.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_0       ( 0.0f);
static const Aurora::NWScript::Variable kDefaultFloat1_0       ( 1.0f);
static const Aurora::NWScript::Variable kDefaultFloat30_0      (30.0f);
static const Aurora::NWScript::Variable kDefaultFloat40_0      (40.0f);
static const Aurora::NWScript::Variable kDefaultFloat100_0    (100.0f);
static const Aurora::NWScript::Variable kDefaultFloat180_0    (180.0f);

static const Aurora::NWScript::Variable kDefaultVector0(0.0f, 0.0f, 0.0f);

static const Aurora::NWScript::Variable kDefaultStringEmpty     (Common::UString(""));
static const Aurora::NWScript::Variable kDefaultStringMinus1    (Common::UString("-1"));
static const Aurora::NWScript::Variable kDefaultStringWorldEntry(Common::UString("World Entry"));

static const Aurora::NWScript::Variable kDefaultObjectTypeCreature ((int32_t) kObjectTypeCreature);

static const Aurora::NWScript::Variable kDefaultObjectInvalid (&kDefaultValueObjectInvalid);
static const Aurora::NWScript::Variable kDefaultObjectCreature(&kDefaultValueObjectCreature);
static const Aurora::NWScript::Variable kDefaultObjectSelf    (&kDefaultValueObjectSelf);

// TODO: Add the relevant enums to types.h, and use these values.

static const Aurora::NWScript::Variable kDefaultACDodgeBonus                   ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultACVsDamageTypeAll              ((int32_t)   4103);
static const Aurora::NWScript::Variable kDefaultAttackBonusMisc                ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultProjectilePathTypeDefault      ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultEngineAnimationBlendtypeLinear ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultCombatAreaActive               ((int32_t)      1);
static const Aurora::NWScript::Variable kDefaultDamagetypeHealth               ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultDialogshotSideAuto             ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultEvadeForward                   ((int32_t)      1);
static const Aurora::NWScript::Variable kDefaultFacingTurnshortest             ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultMovespeedDefault               ((int32_t)     -1);
static const Aurora::NWScript::Variable kDefaultStylemapNone                   ((int32_t)      0);

static const Aurora::NWScript::Variable kDefaultFalse                          ((int32_t)      0);
static const Aurora::NWScript::Variable kDefaultTrue                           ((int32_t)      1);

/** The table defining the name and function pointer of each engine function. */
const Functions::FunctionPointer Functions::kFunctionPointers[] = {
	{    0, "Random"                                 , &Functions::random                                  },
	{    1, "PrintString"                            , &Functions::printString                             },
	{    2, "PrintFloat"                             , &Functions::printFloat                              },
	{    3, "FloatToString"                          , &Functions::floatToString                           },
	{    4, "PrintInteger"                           , &Functions::printInteger                            },
	{    5, "PrintObject"                            , &Functions::printObject                             },
	{    6, "AssignCommand"                          , &Functions::assignCommand                           },
	{    7, "DelayCommand"                           , &Functions::delayCommand                            },
	{    8, "ExecuteScript"                          , &Functions::executeScript                           },
	{    9, "ClearAllActions"                        , 0                                                   },
	{   10, "SetFacing"                              , 0                                                   },
	{   11, "SetCalendar"                            , 0                                                   },
	{   12, "SetTime"                                , 0                                                   },
	{   13, "DesignerAssert"                         , 0                                                   },
	{   14, "ActionPlayerCombat"                     , 0                                                   },
	{   15, "CanUse"                                 , 0                                                   },
	{   16, "GetTimeHour"                            , 0                                                   },
	{   17, "GetTimeMinute"                          , 0                                                   },
	{   18, "GetTimeSecond"                          , 0                                                   },
	{   19, "GetTimeMillisecond"                     , 0                                                   },
	{   20, "ActionRandomWalk"                       , 0                                                   },
	{   21, "ActionMoveToLocation"                   , &Functions::actionMoveToLocation                    },
	{   22, "ActionMoveToObject"                     , &Functions::actionMoveToObject                      },
	{   23, "ActionMoveAwayFromObject"               , 0                                                   },
	{   24, "GetArea"                                , &Functions::getArea                                 },
	{   25, "GetEnteringObject"                      , &Functions::getEnteringObject                       },
	{   26, "GetExitingObject"                       , &Functions::getExitingObject                        },
	{   27, "GetPosition"                            , &Functions::getPosition                             },
	{   28, "GetFacing"                              , 0                                                   },
	{   29, "Use"                                    , 0                                                   },
	{   30, "GetIsItemPossessedBy"                   , 0                                                   },
	{   31, "GetFirstUse"                            , 0                                                   },
	{   32, "GetNextUse"                             , 0                                                   },
	{   33, "GetState"                               , 0                                                   },
	{   34, "GetCameraTagEntry"                      , 0                                                   },
	{   35, "GetCameraTagReplies"                    , 0                                                   },
	{   36, "PauseConversation"                      , 0                                                   },
	{   37, "LockCombatTarget"                       , 0                                                   },
	{   38, "UnlockCombatTarget"                     , 0                                                   },
	{   39, "SetConversationAutoFace"                , 0                                                   },
	{   40, "ActionPlayAnimation"                    , &Functions::actionPlayAnimation                     },
	{   41, "GetDistanceToObject"                    , &Functions::getDistanceToObject                     },
	{   42, "GetIsObjectValid"                       , &Functions::getIsObjectValid                        },
	{   43, "ActionOpenDoor"                         , &Functions::actionOpenDoor                          },
	{   44, "ActionCloseDoor"                        , &Functions::actionCloseDoor                         },
	{   45, "SetCameraFacing"                        , 0                                                   },
	{   46, "PlaySound2D"                            , 0                                                   },
	{   47, "PlaySound3D"                            , 0                                                   },
	{   48, "PlaySoundAtPosition"                    , 0                                                   },
	{   49, "GetHealth"                              , 0                                                   },
	{   50, "GetMaxHealth"                           , 0                                                   },
	{   51, "GetLocalInt"                            , &Functions::getLocalInt                             },
	{   52, "GetLocalFloat"                          , &Functions::getLocalFloat                           },
	{   53, "ZZ_HIDDEN_GetLocalString"               , &Functions::getLocalString                          },
	{   54, "GetLocalObject"                         , &Functions::getLocalObject                          },
	{   55, "SetLocalInt"                            , &Functions::setLocalInt                             },
	{   56, "SetLocalFloat"                          , &Functions::setLocalFloat                           },
	{   57, "ZZ_HIDDEN_SetLocalString"               , &Functions::setLocalString                          },
	{   58, "SetLocalObject"                         , &Functions::setLocalObject                          },
	{   59, "GetStringLength"                        , &Functions::getStringLength                         },
	{   60, "GetStringUpperCase"                     , &Functions::getStringUpperCase                      },
	{   61, "GetStringLowerCase"                     , &Functions::getStringLowerCase                      },
	{   62, "GetStringRight"                         , &Functions::getStringRight                          },
	{   63, "GetStringLeft"                          , &Functions::getStringLeft                           },
	{   64, "InsertString"                           , &Functions::insertString                            },
	{   65, "GetSubString"                           , &Functions::getSubString                            },
	{   66, "FindSubString"                          , &Functions::findSubString                           },
	{   67, "fabs"                                   , &Functions::fabs                                    },
	{   68, "cos"                                    , &Functions::cos                                     },
	{   69, "sin"                                    , &Functions::sin                                     },
	{   70, "tan"                                    , &Functions::tan                                     },
	{   71, "acos"                                   , &Functions::acos                                    },
	{   72, "asin"                                   , &Functions::asin                                    },
	{   73, "atan"                                   , &Functions::atan                                    },
	{   74, "log"                                    , &Functions::log                                     },
	{   75, "pow"                                    , &Functions::pow                                     },
	{   76, "sqrt"                                   , &Functions::sqrt                                    },
	{   77, "abs"                                    , &Functions::abs                                     },
	{   78, "RemoveCombatStyle"                      , 0                                                   },
	{   79, "PlaySoundSet"                           , 0                                                   },
	{   80, "EffectAbilityIncrease"                  , 0                                                   },
	{   81, "GetIsSpeedLoadMode"                     , 0                                                   },
	{   82, "EffectResurrection"                     , 0                                                   },
	{   83, "GetGoreSetting"                         , 0                                                   },
	{   84, "SetCreatureMaterial"                    , 0                                                   },
	{   85, "GetFirstEffect"                         , 0                                                   },
	{   86, "GetNextEffect"                          , 0                                                   },
	{   87, "RemoveEffect"                           , 0                                                   },
	{   88, "GetIsEffectValid"                       , 0                                                   },
	{   89, "GetEffectDurationType"                  , 0                                                   },
	{   90, "GetEffectSubType"                       , 0                                                   },
	{   91, "GetEffectCreator"                       , 0                                                   },
	{   92, "IntToString"                            , &Functions::intToString                             },
	{   93, "RestoreCreatureMaterial"                , 0                                                   },
	{   94, "ClearDelayCommands"                     , 0                                                   },
	{   95, "d2"                                     , &Functions::d2                                      },
	{   96, "d3"                                     , &Functions::d3                                      },
	{   97, "d4"                                     , &Functions::d4                                      },
	{   98, "d6"                                     , &Functions::d6                                      },
	{   99, "d8"                                     , &Functions::d8                                      },
	{  100, "d10"                                    , &Functions::d10                                     },
	{  101, "d12"                                    , &Functions::d12                                     },
	{  102, "d20"                                    , &Functions::d20                                     },
	{  103, "d100"                                   , &Functions::d100                                    },
	{  104, "VectorMagnitude"                        , &Functions::vectorMagnitude                         },
	{  105, "ConsoleCommand"                         , 0                                                   },
	{  106, "GetObjectType"                          , &Functions::getObjectType                           },
	{  107, "StoreFXStates"                          , 0                                                   },
	{  108, "RestoreFXStates"                        , 0                                                   },
	{  109, "GetAppearance"                          , 0                                                   },
	{  110, "SignalWait"                             , 0                                                   },
	{  111, "SetUnselectable"                        , 0                                                   },
	{  112, "SetConversationOwnerForCamera"          , 0                                                   },
	{  113, "GetConversationNumber"                  , 0                                                   },
	{  114, "GetConversationOwnerForCamera"          , 0                                                   },
	{  115, "GetCutsceneFromScriptCommand"           , 0                                                   },
	{  116, "GetPlayerConversationOwner"             , 0                                                   },
	{  117, "GetObjectConversationResref"            , &Functions::getObjectConversationResref             },
	{  118, "SetAppearance"                          , 0                                                   },
	{  119, "GetHeadType"                            , 0                                                   },
	{  120, "GetInCutsceneMode"                      , 0                                                   },
	{  121, "RoundsToSeconds"                        , 0                                                   },
	{  122, "HoursToSeconds"                         , 0                                                   },
	{  123, "TurnsToSeconds"                         , 0                                                   },
	{  124, "SetGoodEvilValue"                       , 0                                                   },
	{  125, "GetGoodEvilValue"                       , 0                                                   },
	{  126, "ArtPlaceableGetPreviousState"           , 0                                                   },
	{  127, "ZZ_CUT_GetAlignmentGoodEvil"            , 0                                                   },
	{  128, "GetFirstObjectInShape"                  , 0                                                   },
	{  129, "GetNextObjectInShape"                   , 0                                                   },
	{  130, "SetOnClickedDeadTime"                   , 0                                                   },
	{  131, "SignalEvent"                            , &Functions::signalEvent                             },
	{  132, "EventUserDefined"                       , &Functions::eventUserDefined                        },
	{  133, "StopConversation"                       , 0                                                   },
	{  134, "GetTagOfAreaFrom"                       , 0                                                   },
	{  135, "ActionGiveItem"                         , 0                                                   },
	{  136, "ActionTakeItem"                         , 0                                                   },
	{  137, "VectorNormalize"                        , &Functions::vectorNormalize                         },
	{  138, "GetCombatStyleLevel"                    , 0                                                   },
	{  139, "GetAbilityScore"                        , 0                                                   },
	{  140, "GetIsDead"                              , 0                                                   },
	{  141, "PrintVector"                            , &Functions::printVector                             },
	{  142, "Vector"                                 , &Functions::vector                                  },
	{  143, "SetFacingPoint"                         , 0                                                   },
	{  144, "AngleToVector"                          , 0                                                   },
	{  145, "VectorToAngle"                          , 0                                                   },
	{  146, "SetLookAtObject"                        , 0                                                   },
	{  147, "StartMinigame"                          , 0                                                   },
	{  148, "ClearCombatAnimations"                  , 0                                                   },
	{  149, "CreateGroundPile"                       , 0                                                   },
	{  150, "DisableHexgrid"                         , 0                                                   },
	{  151, "GetDistanceBetween"                     , 0                                                   },
	{  152, "ZZ_HIDDEN_SetLocalLocation"             , 0                                                   },
	{  153, "ZZ_HIDDEN_GetLocalLocation"             , 0                                                   },
	{  154, "IsCreatureInViewVolume"                 , 0                                                   },
	{  155, "GetMGStats"                             , 0                                                   },
	{  156, "SetHealth"                              , 0                                                   },
	{  157, "GetGemInAmulet"                         , 0                                                   },
	{  158, "SetFloatingTextAlpha"                   , 0                                                   },
	{  159, "GetPriorActivation"                     , &Functions::getPriorActivation                      },
	{  160, "ActionSetFacing"                        , 0                                                   },
	{  161, "GetBuildType"                           , 0                                                   },
	{  162, "SetCommandable"                         , 0                                                   },
	{  163, "GetCommandable"                         , 0                                                   },
	{  164, "EnableOrientationChanges"               , 0                                                   },
	{  165, "EffectMovementSpeedIncrease"            , 0                                                   },
	{  166, "Cutscene_PropLoad"                      , 0                                                   },
	{  167, "ActionFollowCreature"                   , 0                                                   },
	{  168, "GetTag"                                 , &Functions::getTag                                  },
	{  169, "Cutscene_PropUnload"                    , 0                                                   },
	{  170, "GetEffectType"                          , 0                                                   },
	{  171, "GetTimePlayed"                          , 0                                                   },
	{  172, "Cutscene_PropAttach"                    , 0                                                   },
	{  173, "ChangeFaction"                          , 0                                                   },
	{  174, "CreateItem"                             , 0                                                   },
	{  175, "DestroyItem"                            , 0                                                   },
	{  176, "TransferItem"                           , 0                                                   },
	{  177, "Cutscene_PropDetach"                    , 0                                                   },
	{  178, "Cutscene_PropSetPosition"               , 0                                                   },
	{  179, "Cutscene_PropSetOrientation"            , 0                                                   },
	{  180, "Cutscene_PropPlayAnimation"             , 0                                                   },
	{  181, "Cutscene_Start"                         , 0                                                   },
	{  182, "Cutscene_End"                           , 0                                                   },
	{  183, "Cutscene_TargetPosition"                , 0                                                   },
	{  184, "Cutscene_TargetObject"                  , 0                                                   },
	{  185, "Cutscene_MoveToPosition"                , 0                                                   },
	{  186, "Cutscene_MoveToObject"                  , 0                                                   },
	{  187, "Cutscene_OrbitPosition"                 , 0                                                   },
	{  188, "Cutscene_OrbitObject"                   , 0                                                   },
	{  189, "Cutscene_PlayAnimation"                 , 0                                                   },
	{  190, "Cutscene_Truck"                         , 0                                                   },
	{  191, "Cutscene_Dolly"                         , 0                                                   },
	{  192, "Cutscene_Boom"                          , 0                                                   },
	{  193, "Cutscene_Pan"                           , 0                                                   },
	{  194, "Cutscene_Tilt"                          , 0                                                   },
	{  195, "GetLoadedFromAreaTransition"            , 0                                                   },
	{  196, "ActionJumpToObject"                     , &Functions::actionJumpToObject                      },
	{  197, "GetWaypointByTag"                       , &Functions::getWaypointByTag                        },
	{  198, "GetTransitionTarget"                    , 0                                                   },
	{  199, "EffectLinkEffects"                      , 0                                                   },
	{  200, "GetObjectByTag"                         , &Functions::getObjectByTag                          },
	{  201, "Cutscene_Roll"                          , 0                                                   },
	{  202, "ActionWait"                             , 0                                                   },
	{  203, "SetAreaTransitionMovie"                 , 0                                                   },
	{  204, "ActionStartConversation"                , &Functions::actionStartConversation                 },
	{  205, "ActionPauseConversation"                , 0                                                   },
	{  206, "ActionResumeConversation"               , 0                                                   },
	{  207, "GetObjectScript"                        , 0                                                   },
	{  208, "ShowAreaTransGui"                       , 0                                                   },
	{  209, "ApplyVisualEffectToObject"              , 0                                                   },
	{  210, "Cutscene_Zoom"                          , 0                                                   },
	{  211, "Cutscene_ClearTarget"                   , 0                                                   },
	{  212, "SetUserDefinedEventNumber"              , &Functions::setUserDefinedEventNumber               },
	{  213, "GetLocation"                            , &Functions::getLocation                             },
	{  214, "ActionJumpToLocation"                   , &Functions::actionJumpToLocation                    },
	{  215, "Location"                               , &Functions::location                                },
	{  216, "ApplyVisualEffectAtLocation"            , 0                                                   },
	{  217, "GetIsPC"                                , &Functions::getIsPC                                 },
	{  218, "FeetToMeters"                           , 0                                                   },
	{  219, "YardsToMeters"                          , 0                                                   },
	{  220, "ApplyEffectToObject"                    , 0                                                   },
	{  221, "Cutscene_StopOrbit"                     , 0                                                   },
	{  222, "GetSpellTargetLocation"                 , 0                                                   },
	{  223, "GetPositionFromLocation"                , &Functions::getPositionFromLocation                 },
	{  224, "ZZ_CUT_GetAreaFromLocation"             , 0                                                   },
	{  225, "GetFacingFromLocation"                  , 0                                                   },
	{  226, "MGConfirmation"                         , 0                                                   },
	{  227, "GetNearestObject"                       , &Functions::getNearestObject                        },
	{  228, "GetNearestObjectToLocation"             , 0                                                   },
	{  229, "UnlockMGGroup"                          , 0                                                   },
	{  230, "IntToFloat"                             , &Functions::intToFloat                              },
	{  231, "FloatToInt"                             , &Functions::floatToInt                              },
	{  232, "StringToInt"                            , &Functions::stringToInt                             },
	{  233, "StringToFloat"                          , &Functions::stringToFloat                           },
	{  234, "ActionCastSpell"                        , 0                                                   },
	{  235, "GetIsEnemy"                             , 0                                                   },
	{  236, "GetIsFriend"                            , 0                                                   },
	{  237, "GetIsNeutral"                           , 0                                                   },
	{  238, "Cutscene_Stop"                          , 0                                                   },
	{  239, "GetStringByStrRef"                      , &Functions::getStringByStrRef                       },
	{  240, "ActionSpeakStringByStrRef"              , &Functions::actionSpeakStringByStrRef               },
	{  241, "DestroyObject"                          , 0                                                   },
	{  242, "Cutscene_SetZoom"                       , 0                                                   },
	{  243, "CreateObject"                           , 0                                                   },
	{  244, "EventSpellCastAt"                       , 0                                                   },
	{  245, "Cutscene_SetOrientation"                , 0                                                   },
	{  246, "Cutscene_LoadCamera"                    , 0                                                   },
	{  247, "GetUserDefinedEventNumber"              , &Functions::getUserDefinedEventNumber               },
	{  248, "GetSpellId"                             , 0                                                   },
	{  249, "RandomName"                             , 0                                                   },
	{  250, "EffectPoison"                           , 0                                                   },
	{  251, "GetBuildNumber"                         , 0                                                   },
	{  252, "GetOneLinerSpeaker"                     , 0                                                   },
	{  253, "Cutscene_CameraAnimation"               , 0                                                   },
	{  254, "GetLastSpeaker"                         , 0                                                   },
	{  255, "BeginConversation"                      , 0                                                   },
	{  256, "Cutscene_StuntAnimation"                , 0                                                   },
	{  257, "Cutscene_DialogShot"                    , 0                                                   },
	{  258, "Cutscene_StuntSetPosition"              , 0                                                   },
	{  259, "Cutscene_StuntSetOrientation"           , 0                                                   },
	{  260, "GetOwner"                               , 0                                                   },
	{  261, "AcquireHenchmanStore"                   , 0                                                   },
	{  262, "GetFirstInObject"                       , 0                                                   },
	{  263, "GetNextInObject"                        , 0                                                   },
	{  264, "ZZ_CUT_GetAreaOfEffectCreator"          , 0                                                   },
	{  265, "DeleteLocalInt"                         , 0                                                   },
	{  266, "DeleteLocalFloat"                       , 0                                                   },
	{  267, "DeleteLocalString"                      , 0                                                   },
	{  268, "DeleteLocalObject"                      , 0                                                   },
	{  269, "DeleteLocalLocation"                    , 0                                                   },
	{  270, "ZY_EFFECT_EffectHaste"                  , 0                                                   },
	{  271, "ZY_EFFECT_EffectSlow"                   , 0                                                   },
	{  272, "ObjectToString"                         , &Functions::objectToString                          },
	{  273, "ZY_EFFECT_EffectImmunity"               , 0                                                   },
	{  274, "ZY_EFFECT_GetIsImmune"                  , 0                                                   },
	{  275, "ZY_EFFECT_EffectDamageImmunityIncrease" , 0                                                   },
	{  276, "GetCharm"                               , 0                                                   },
	{  277, "SetCharm"                               , 0                                                   },
	{  278, "GetIntimidate"                          , 0                                                   },
	{  279, "SetIntimidate"                          , 0                                                   },
	{  280, "GetIntuition"                           , 0                                                   },
	{  281, "SetIntuition"                           , 0                                                   },
	{  282, "SetCreatureEmotionalState"              , 0                                                   },
	{  283, "GetCreatureEmotionalState"              , 0                                                   },
	{  284, "SetCustomToken"                         , &Functions::setCustomToken                          },
	{  285, "GetAILevel"                             , 0                                                   },
	{  286, "SetAILevel"                             , 0                                                   },
	{  287, "SetObjectNoCollide"                     , &Functions::setObjectNoCollide                      },
	{  288, "ToggleHexGridPatch"                     , 0                                                   },
	{  289, "GetPlayerSeenByObject"                  , 0                                                   },
	{  290, "EnableDepthOfField"                     , 0                                                   },
	{  291, "SetDepthOfField"                        , 0                                                   },
	{  292, "SetHenchmanLockStrRef"                  , 0                                                   },
	{  293, "Cutscene_ForceAnimation"                , 0                                                   },
	{  294, "ActionDoCommand"                        , &Functions::actionDoCommand                         },
	{  295, "EventConversation"                      , 0                                                   },
	{  296, "GetLastWorldMapDestination"             , 0                                                   },
	{  297, "PolymorphPlayerToHenchman"              , 0                                                   },
	{  298, "GetDistanceBetweenLocations"            , 0                                                   },
	{  299, "SetHenchmanAppearance"                  , 0                                                   },
	{  300, "PlayAnimation"                          , &Functions::playAnimation                           },
	{  301, "Get2DAEntryIntByString"                 , &Functions::get2DAEntryIntByString                  },
	{  302, "Get2DAEntryFloatByString"               , &Functions::get2DAEntryFloatByString                },
	{  303, "Get2DAEntryStringByString"              , &Functions::get2DAEntryStringByString               },
	{  304, "GetHasSpellEffect"                      , 0                                                   },
	{  305, "GetEffectSpellId"                       , 0                                                   },
	{  306, "GetHenchmanAvailable"                   , 0                                                   },
	{  307, "SetHenchmanState"                       , 0                                                   },
	{  308, "GetHenchmanState"                       , 0                                                   },
	{  309, "ActivateHenchman"                       , 0                                                   },
	{  310, "SetLockCamera"                          , 0                                                   },
	{  311, "DeactivateHenchman"                     , 0                                                   },
	{  312, "SetHenchmanLevel"                       , 0                                                   },
	{  313, "JumpToLocation"                         , &Functions::jumpToLocation                          },
	{  314, "ZY_EFFECT_EffectTemporaryHitpoints"     , 0                                                   },
	{  315, "GetHenchmanLevel"                       , 0                                                   },
	{  316, "SetHenchmanCombatMode"                  , 0                                                   },
	{  317, "GetHenchmanCombatMode"                  , 0                                                   },
	{  318, "GetStyleOfHarmonicCombo"                , 0                                                   },
	{  319, "SaveCombatRetry"                        , 0                                                   },
	{  320, "GetIsInCombat"                          , 0                                                   },
	{  321, "LockCurrentHenchman"                    , 0                                                   },
	{  322, "GiveGoldToPlayer"                       , 0                                                   },
	{  323, "Cutscene_TargetMidpoint"                , 0                                                   },
	{  324, "SetLocked"                              , 0                                                   },
	{  325, "GetLocked"                              , 0                                                   },
	{  326, "GetSelectedHenchman"                    , 0                                                   },
	{  327, "SetSelectedHenchman"                    , 0                                                   },
	{  328, "DeactivateAllHenchmen"                  , 0                                                   },
	{  329, "ActionInteractObject"                   , 0                                                   },
	{  330, "GetLastUsedBy"                          , 0                                                   },
	{  331, "GetSelectedHenchmanSubtype"             , 0                                                   },
	{  332, "Cutscene_EnableAllTriggers"             , 0                                                   },
	{  333, "Cutscene_LoadRim"                       , 0                                                   },
	{  334, "Cutscene_UnloadRim"                     , 0                                                   },
	{  335, "SetTravelGiver"                         , 0                                                   },
	{  336, "GetBlockingDoor"                        , 0                                                   },
	{  337, "GetIsDoorActionPossible"                , 0                                                   },
	{  338, "DoDoorAction"                           , 0                                                   },
	{  339, "GetFirstItemInInventory"                , 0                                                   },
	{  340, "GetNextItemInInventory"                 , 0                                                   },
	{  341, "GetClass"                               , 0                                                   },
	{  342, "Cutscene_DialogShotVerified"            , 0                                                   },
	{  343, "GetLevel"                               , 0                                                   },
	{  344, "GetDamageDealtByType"                   , 0                                                   },
	{  345, "GetTotalDamageDealt"                    , 0                                                   },
	{  346, "GetLastDamager"                         , 0                                                   },
	{  347, "GetLastDisarmed"                        , 0                                                   },
	{  348, "Cutscene_SetDialogShotConstraint"       , 0                                                   },
	{  349, "GetLastLockedBy"                        , 0                                                   },
	{  350, "GetLastUnlockedBy"                      , 0                                                   },
	{  351, "Cutscene_GetObjectHeight"               , 0                                                   },
	{  352, "Cutscene_GetDialogSide"                 , 0                                                   },
	{  353, "Cutscene_GetObjectAnim"                 , 0                                                   },
	{  354, "Cutscene_GetSpeaker"                    , 0                                                   },
	{  355, "Cutscene_GetListener"                   , 0                                                   },
	{  356, "Cutscene_GetPosition"                   , 0                                                   },
	{  357, "Cutscene_GetOrientation"                , 0                                                   },
	{  358, "GetGender"                              , 0                                                   },
	{  359, "Cutscene_GetWaypointOrientation"        , 0                                                   },
	{  360, "ActionMoveAwayFromLocation"             , 0                                                   },
	{  361, "ZZ_COMBAT_GetAttemptedAttackTarget"     , 0                                                   },
	{  362, "Cutscene_SetAnimScale"                  , 0                                                   },
	{  363, "Cutscene_WaitOnSoundReady"              , 0                                                   },
	{  364, "ZZ_CUT_GetAssociate"                    , 0                                                   },
	{  365, "ZZ_CUT_AddHenchman"                     , 0                                                   },
	{  266, "ZZ_CUT_RemoveHenchman"                  , 0                                                   },
	{  367, "ZZ_CUT_AddJournalQuestEntry"            , 0                                                   },
	{  368, "ZZ_CUT_RemoveJournalQuestEntry"         , 0                                                   },
	{  369, "ZZ_CUT_GetPCPublicCDKey"                , 0                                                   },
	{  370, "ZZ_CUT_GetPCIPAddress"                  , 0                                                   },
	{  371, "ZZ_CUT_GetPCPlayerName"                 , 0                                                   },
	{  372, "ZZ_CUT_SetPCLike"                       , 0                                                   },
	{  373, "ZZ_CUT_SetPCDislike"                    , 0                                                   },
	{  374, "ZZ_CUT_SendMessageToPC"                 , &Functions::sendMessageToPC                         },
	{  375, "ZZ_COMBAT_GetAttemptedSpellTarget"      , 0                                                   },
	{  376, "GetLastOpenedBy"                        , 0                                                   },
	{  377, "DisplayFloatyFeedback"                  , 0                                                   },
	{  378, "OpenStore"                              , 0                                                   },
	{  379, "SetItemBuyPrice"                        , 0                                                   },
	{  380, "SetItemSellPrice"                       , 0                                                   },
	{  381, "GetItemBuyPrice"                        , 0                                                   },
	{  382, "ActionForceMoveToLocation"              , 0                                                   },
	{  383, "ActionForceMoveToObject"                , 0                                                   },
	{  384, "GetJournalQuestExperience"              , 0                                                   },
	{  385, "JumpToObject"                           , &Functions::jumpToObject                            },
	{  386, "SetMapPinEnabled"                       , 0                                                   },
	{  387, "GetItemSellPrice"                       , 0                                                   },
	{  388, "PopUpGUIPanel"                          , 0                                                   },
	{  389, "ZZ_CUT_ClearPersonalReputation"         , 0                                                   },
	{  390, "ZZ_COMBAT_SetIsTemporaryFriend"         , 0                                                   },
	{  391, "ZZ_COMBAT_SetIsTemporaryEnemy"          , 0                                                   },
	{  392, "ZZ_COMBAT_SetIsTemporaryNeutral"        , 0                                                   },
	{  393, "GiveXPToPlayer"                         , 0                                                   },
	{  394, "SetXP"                                  , 0                                                   },
	{  395, "GetXP"                                  , 0                                                   },
	{  396, "IntToHexString"                         , &Functions::intToHexString                          },
	{  397, "ZZ_CUT_GetBaseItemType"                 , 0                                                   },
	{  398, "ZZ_CUT_GetItemHasItemProperty"          , 0                                                   },
	{  399, "ZZ_CUT_ActionEquipMostDamagingMelee"    , 0                                                   },
	{  400, "ZZ_CUT_ActionEquipMostDamagingRanged"   , 0                                                   },
	{  401, "ZZ_CUT_GetItemACValue"                  , 0                                                   },
	{  402, "ZZ_CUT_ActionRest"                      , 0                                                   },
	{  403, "ExploreArea"                            , 0                                                   },
	{  404, "ZZ_CUT_ActionEquipMostEffectiveArmor"   , 0                                                   },
	{  405, "GetIsDay"                               , 0                                                   },
	{  406, "GetIsNight"                             , 0                                                   },
	{  407, "GetIsDawn"                              , 0                                                   },
	{  408, "GetIsDusk"                              , 0                                                   },
	{  409, "GetIsEncounterCreature"                 , 0                                                   },
	{  410, "ZZ_CUT_GetLastPlayerDying"              , 0                                                   },
	{  411, "ZZ_CUT_GetStartingLocation"             , 0                                                   },
	{  412, "ChangeToStandardFaction"                , 0                                                   },
	{  413, "SoundObjectPlay"                        , 0                                                   },
	{  414, "SoundObjectStop"                        , 0                                                   },
	{  415, "SoundObjectSetVolume"                   , 0                                                   },
	{  416, "SoundObjectSetPosition"                 , 0                                                   },
	{  417, "SpeakOneLinerConversation"              , 0                                                   },
	{  418, "GetGold"                                , 0                                                   },
	{  419, "GetItemAquired"                         , 0                                                   },
	{  420, "ZZ_CUT_GetIsDM"                         , 0                                                   },
	{  421, "ZZ_CUT_PlayVoiceChat"                   , 0                                                   },
	{  422, "ZZ_CUT_GetIsWeaponEffective"            , 0                                                   },
	{  423, "ZZ_CUT_GetLastSpellHarmful"             , 0                                                   },
	{  424, "ZZ_CUT_EventActivateItem"               , 0                                                   },
	{  425, "PauseSound"                             , 0                                                   },
	{  426, "DisableSoundSet"                        , 0                                                   },
	{  427, "DisableMusicStateChanges"               , 0                                                   },
	{  428, "SetMusicState"                          , 0                                                   },
	{  429, "GetMusicState"                          , 0                                                   },
	{  430, "SetAreaMusicState"                      , 0                                                   },
	{  431, "StartCutsceneAudio"                     , 0                                                   },
	{  432, "ZZ_CUT_MusicBattleChange"               , 0                                                   },
	{  433, "ZZ_CUT_AmbientSoundPlay"                , 0                                                   },
	{  434, "ZZ_CUT_AmbientSoundStop"                , 0                                                   },
	{  435, "ZZ_CUT_AmbientSoundChange"              , 0                                                   },
	{  436, "ZZ_CUT_AmbientSoundChangeNight"         , 0                                                   },
	{  437, "GetLastKiller"                          , 0                                                   },
	{  438, "ZZ_CUT_GetSpellCastItem"                , 0                                                   },
	{  439, "ZZ_CUT_GetItemActivated"                , 0                                                   },
	{  440, "ZZ_CUT_GetItemActivator"                , 0                                                   },
	{  441, "ZZ_CUT_GetItemActivatedTargetLocation"  , 0                                                   },
	{  442, "ZZ_CUT_GetItemActivatedTarget"          , 0                                                   },
	{  443, "GetIsOpen"                              , 0                                                   },
	{  444, "TakeGoldFromPlayer"                     , 0                                                   },
	{  445, "GetIsInConversation"                    , 0                                                   },
	{  446, "EffectAbilityDecrease"                  , 0                                                   },
	{  447, "ZY_EFFECT_EffectAttackDecrease"         , 0                                                   },
	{  448, "ZY_EFFECT_EffectDamageDecrease"         , 0                                                   },
	{  449, "ZY_EFFECT_EffectDamageImmunityDecrease" , 0                                                   },
	{  450, "ZY_EFFECT_EffectACDecrease"             , 0                                                   },
	{  451, "EffectMovementSpeedDecrease"            , 0                                                   },
	{  452, "ZY_EFFECT_EffectSavingThrowDecrease"    , 0                                                   },
	{  453, "ZY_EFFECT_EffectSkillDecrease"          , 0                                                   },
	{  454, "ZY_EFFECT_EffectForceResistanceDecrease", 0                                                   },
	{  455, "GetPlotFlag"                            , 0                                                   },
	{  456, "SetPlotFlag"                            , 0                                                   },
	{  457, "ZY_EFFECT_EffectInvisibility"           , 0                                                   },
	{  458, "ZY_EFFECT_EffectConcealment"            , 0                                                   },
	{  459, "ZY_EFFECT_EffectDarkness"               , 0                                                   },
	{  460, "ZY_EFFECT_EffectDispelMagicAll"         , 0                                                   },
	{  461, "ZY_EFFECT_EffectElementalShield"        , 0                                                   },
	{  462, "ZY_EFFECT_EffectLevelDrain"             , 0                                                   },
	{  463, "ZY_EFFECT_EffectPolymorph"              , 0                                                   },
	{  464, "ZY_EFFECT_EffectSanctuary"              , 0                                                   },
	{  465, "ZY_EFFECT_EffectTrueSeeing"             , 0                                                   },
	{  466, "ZY_EFFECT_EffectSeeInvisible"           , 0                                                   },
	{  467, "ZY_EFFECT_EffectTimeStop"               , 0                                                   },
	{  468, "ZY_EFFECT_EffectBlindness"              , 0                                                   },
	{  469, "ZZ_CUT_GetIsReactionTypeFriendly"       , 0                                                   },
	{  470, "ZZ_CUT_GetIsReactionTypeNeutral"        , 0                                                   },
	{  471, "ZZ_CUT_GetIsReactionTypeHostile"        , 0                                                   },
	{  472, "ZY_EFFECT_EffectSpellLevelAbsorption"   , 0                                                   },
	{  473, "ZY_EFFECT_EffectDispelMagicBest"        , 0                                                   },
	{  474, "ZZ_CUT_ActivatePortal"                  , 0                                                   },
	{  475, "GetItemCount"                           , 0                                                   },
	{  476, "SurrenderToEnemies"                     , 0                                                   },
	{  477, "ZY_EFFECT_EffectMissChance"             , 0                                                   },
	{  478, "ZZ_CUT_GetTurnResistanceHD"             , 0                                                   },
	{  479, "GetCreatureSize"                        , 0                                                   },
	{  480, "ZY_EFFECT_EffectDisappearAppear"        , 0                                                   },
	{  481, "ZY_EFFECT_EffectDisappear"              , 0                                                   },
	{  482, "ZY_EFFECT_EffectAppear"                 , 0                                                   },
	{  483, "ActionUnlockObject"                     , 0                                                   },
	{  484, "ActionLockObject"                       , 0                                                   },
	{  485, "ZY_EFFECT_EffectModifyAttacks"          , 0                                                   },
	{  486, "ZZ_CUT_GetLastTrapDetected"             , 0                                                   },
	{  487, "ZY_EFFECT_EffectDamageShield"           , 0                                                   },
	{  488, "ZZ_CUT_GetNearestTrapToObject"          , 0                                                   },
	{  489, "ZZ_DELETED_44"                          , 0                                                   },
	{  490, "ZZ_DELETED_45"                          , 0                                                   },
	{  491, "ZZ_CUT_GetFortitudeSavingThrow"         , 0                                                   },
	{  492, "ZZ_CUT_GetWillSavingThrow"              , 0                                                   },
	{  493, "ZZ_CUT_GetReflexSavingThrow"            , 0                                                   },
	{  494, "ZZ_CUT_GetChallengeRating"              , 0                                                   },
	{  495, "ZZ_DELETED_46"                          , 0                                                   },
	{  496, "GetMovementRate"                        , 0                                                   },
	{  497, "SetRegisteredScript"                    , 0                                                   },
	{  498, "GetRunningRegisteredScript"             , 0                                                   },
	{  499, "GetHasStyle"                            , 0                                                   },
	{  500, "GetNumberOfStyles"                      , 0                                                   },
	{  501, "GetStyle"                               , 0                                                   },
	{  502, "GetRoom"                                , 0                                                   },
	{  503, "GetConversationResref"                  , 0                                                   },
	{  504, "SetCutSceneCameraMode"                  , 0                                                   },
	{  505, "ZZ_DELETED_51"                          , 0                                                   },
	{  506, "ZZ_DELETED_52"                          , 0                                                   },
	{  507, "ZZ_DELETED_53"                          , 0                                                   },
	{  508, "ZZ_DELETED_54"                          , 0                                                   },
	{  509, "StartNewArea"                           , 0                                                   },
	{  510, "ZZ_DELETED_55"                          , 0                                                   },
	{  511, "ZZ_CUT_GetWeaponRanged"                 , 0                                                   },
	{  512, "DoAutoSavePlotPoint"                    , 0                                                   },
	{  513, "GetGameDifficulty"                      , 0                                                   },
	{  514, "DoAutoSave"                             , 0                                                   },
	{  515, "ZZ_DELETED_57"                          , 0                                                   },
	{  516, "ZZ_DELETED_58"                          , 0                                                   },
	{  517, "ZZ_DELETED_59"                          , 0                                                   },
	{  518, "ZZ_DELETED_60"                          , 0                                                   },
	{  519, "ZZ_DELETED_61"                          , 0                                                   },
	{  520, "ZZ_DELETED_62"                          , 0                                                   },
	{  521, "ZZ_DELETED_63"                          , 0                                                   },
	{  522, "GetCurrentAction"                       , 0                                                   },
	{  523, "ZZ_DELETED_64"                          , 0                                                   },
	{  524, "ZZ_DELETED_65"                          , 0                                                   },
	{  525, "ZZ_CUT_FloatingTextStrRefOnCreature"    , 0                                                   },
	{  526, "ZZ_CUT_FloatingTextStringOnCreature"    , 0                                                   },
	{  527, "ZZ_CUT_GetTrapDisarmable"               , 0                                                   },
	{  528, "ZZ_CUT_GetTrapDetectable"               , 0                                                   },
	{  529, "ZZ_CUT_GetTrapDetectedBy"               , 0                                                   },
	{  530, "ZZ_CUT_GetTrapFlagged"                  , 0                                                   },
	{  531, "ZZ_CUT_GetTrapBaseType"                 , 0                                                   },
	{  532, "ZZ_CUT_GetTrapOneShot"                  , 0                                                   },
	{  533, "ZZ_CUT_GetTrapCreator"                  , 0                                                   },
	{  534, "ZZ_CUT_GetTrapKeyTag"                   , 0                                                   },
	{  535, "ZZ_CUT_GetTrapDisarmDC"                 , 0                                                   },
	{  536, "ZZ_CUT_GetTrapDetectDC"                 , 0                                                   },
	{  537, "GetLockKeyRequired"                     , 0                                                   },
	{  538, "GetLockKeyItem"                         , 0                                                   },
	{  539, "ZZ_CUT_GetLockLockable"                 , 0                                                   },
	{  540, "ZZ_CUT_GetLockUnlockDC"                 , 0                                                   },
	{  541, "ZZ_CUT_GetLockLockDC"                   , 0                                                   },
	{  542, "ZZ_CUT_GetPCLevellingUp"                , 0                                                   },
	{  543, "ZZ_CUT_GetHasFeatEffect"                , 0                                                   },
	{  544, "SetPlaceableIllumination"               , 0                                                   },
	{  545, "GetPlaceableIllumination"               , 0                                                   },
	{  546, "GetIsPlaceableObjectActionPossible"     , 0                                                   },
	{  547, "DoPlaceableObjectAction"                , 0                                                   },
	{  548, "GetPlayer"                              , &Functions::getPlayer                               },
	{  549, "ZZ_CUT_GetNextPC"                       , &Functions::getNextPC                               },
	{  550, "ZZ_CUT_SetTrapDetectedBy"               , 0                                                   },
	{  551, "ZZ_CUT_GetIsTrapped"                    , 0                                                   },
	{  552, "ZZ_DELETED_66"                          , 0                                                   },
	{  553, "ZZ_DELETED_67"                          , 0                                                   },
	{  554, "PopUpDeathGUIPanel"                     , 0                                                   },
	{  555, "ZZ_CUT_SetTrapDisabled"                 , 0                                                   },
	{  556, "ZZ_COMBAT_GetLastHostileActor"          , 0                                                   },
	{  557, "ZZ_CUT_ExportAllCharacters"             , 0                                                   },
	{  558, "ZZ_CUT_MusicBackgroundGetTrack"         , 0                                                   },
	{  559, "ZZ_CUT_MusicBackgroundGetNightTrack"    , 0                                                   },
	{  560, "WriteTimestampedLogEntry"               , &Functions::writeTimestampedLogEntry                },
	{  561, "ZZ_CUT_GetModuleName"                   , 0                                                   },
	{  562, "ZZ_CUT_GetFactionLeader"                , 0                                                   },
	{  563, "ZZ_DELETED_68"                          , 0                                                   },
	{  564, "EndGame"                                , 0                                                   },
	{  565, "GetRunScriptVar"                        , 0                                                   },
	{  566, "GetIsObjectInCombatMode"                , 0                                                   },
	{  567, "IgnoreCombatEndTransition"              , 0                                                   },
	{  568, "SetNoReactionAnimations"                , 0                                                   },
	{  569, "ResetCombatInput"                       , 0                                                   },
	{  570, "ZZ_CUT_GetHasInventory"                 , 0                                                   },
	{  571, "ZZ_CUT_GetStrRefSoundDuration"          , 0                                                   },
	{  572, "ZZ_CUT_AddToParty"                      , 0                                                   },
	{  573, "ZZ_CUT_RemoveFromParty"                 , 0                                                   },
	{  574, "GetGlobalObject"                        , 0                                                   },
	{  575, "SetGlobalObject"                        , 0                                                   },
	{  576, "ZZ_HIDDEN_GetGlobalString"              , &Functions::getGlobalString                         },
	{  577, "ZZ_HIDDEN_SetGlobalString"              , &Functions::setGlobalString                         },
	{  578, "GetGlobalFloat"                         , &Functions::getGlobalFloat                          },
	{  579, "SetGlobalFloat"                         , &Functions::setGlobalFloat                          },
	{  580, "GetGlobalInt"                           , &Functions::getGlobalInt                            },
	{  581, "SetGlobalInt"                           , &Functions::setGlobalInt                            },
	{  582, "AurPostString"                          , 0                                                   },
	{  583, "MG_GetLastEvent"                        , 0                                                   },
	{  584, "MG_GetLastEventModelName"               , 0                                                   },
	{  585, "MG_GetObjectByName"                     , 0                                                   },
	{  586, "MG_PlayAnimation"                       , 0                                                   },
	{  587, "MG_GetLastBulletHitDamage"              , 0                                                   },
	{  588, "MG_GetLastBulletHitTarget"              , 0                                                   },
	{  589, "MG_GetLastBulletHitShooter"             , 0                                                   },
	{  590, "MG_AdjustFollowerHitPoints"             , 0                                                   },
	{  591, "MG_OnBulletHit"                         , 0                                                   },
	{  592, "MG_OnObstacleHit"                       , 0                                                   },
	{  593, "MG_GetLastFollowerHit"                  , 0                                                   },
	{  594, "MG_GetLastObstacleHit"                  , 0                                                   },
	{  595, "MG_GetLastBulletFiredDamage"            , 0                                                   },
	{  596, "MG_GetLastBulletFiredTarget"            , 0                                                   },
	{  597, "MG_GetObjectName"                       , 0                                                   },
	{  598, "MG_OnDeath"                             , 0                                                   },
	{  599, "MG_GetIsFollower"                       , 0                                                   },
	{  600, "MG_GetIsPlayer"                         , 0                                                   },
	{  601, "MG_GetIsEnemy"                          , 0                                                   },
	{  602, "MG_GetIsTrigger"                        , 0                                                   },
	{  603, "MG_GetIsObstacle"                       , 0                                                   },
	{  604, "MG_SetFollowerHitPoints"                , 0                                                   },
	{  605, "MG_OnDamage"                            , 0                                                   },
	{  606, "MG_GetLastHPChange"                     , 0                                                   },
	{  607, "MG_RemoveAnimation"                     , 0                                                   },
	{  608, "MG_GetCameraNearClip"                   , 0                                                   },
	{  609, "MG_GetCameraFarClip"                    , 0                                                   },
	{  610, "MG_SetCameraClip"                       , 0                                                   },
	{  611, "MG_GetPlayer"                           , 0                                                   },
	{  612, "MG_GetEnemyCount"                       , 0                                                   },
	{  613, "MG_GetEnemy"                            , 0                                                   },
	{  614, "MG_GetObstacleCount"                    , 0                                                   },
	{  615, "MG_GetObstacle"                         , 0                                                   },
	{  616, "MG_GetHitPoints"                        , 0                                                   },
	{  617, "MG_GetMaxHitPoints"                     , 0                                                   },
	{  618, "MG_SetMaxHitPoints"                     , 0                                                   },
	{  619, "MG_GetSphereRadius"                     , 0                                                   },
	{  620, "MG_SetSphereRadius"                     , 0                                                   },
	{  621, "MG_GetNumLoops"                         , 0                                                   },
	{  622, "MG_SetNumLoops"                         , 0                                                   },
	{  623, "MG_GetPosition"                         , 0                                                   },
	{  624, "MG_GetGunBankCount"                     , 0                                                   },
	{  625, "MG_GetGunBankBulletModel"               , 0                                                   },
	{  626, "MG_GetGunBankGunModel"                  , 0                                                   },
	{  627, "MG_GetGunBankDamage"                    , 0                                                   },
	{  628, "MG_GetGunBankTimeBetweenShots"          , 0                                                   },
	{  629, "MG_GetGunBankLifespan"                  , 0                                                   },
	{  630, "MG_GetGunBankSpeed"                     , 0                                                   },
	{  631, "MG_GetGunBankTarget"                    , 0                                                   },
	{  632, "MG_SetGunBankBulletModel"               , 0                                                   },
	{  633, "MG_SetGunBankGunModel"                  , 0                                                   },
	{  634, "MG_SetGunBankDamage"                    , 0                                                   },
	{  635, "MG_SetGunBankTimeBetweenShots"          , 0                                                   },
	{  636, "MG_SetGunBankLifespan"                  , 0                                                   },
	{  637, "MG_SetGunBankSpeed"                     , 0                                                   },
	{  638, "MG_SetGunBankTarget"                    , 0                                                   },
	{  639, "MG_GetLastBulletHitPart"                , 0                                                   },
	{  640, "MG_GetIsGunBankTargetting"              , 0                                                   },
	{  641, "MG_GetPlayerOffset"                     , 0                                                   },
	{  642, "MG_GetPlayerInvincibility"              , 0                                                   },
	{  643, "MG_GetPlayerSpeed"                      , 0                                                   },
	{  644, "MG_GetPlayerMinSpeed"                   , 0                                                   },
	{  645, "MG_GetPlayerAccelerationPerSecond"      , 0                                                   },
	{  646, "MG_GetPlayerTunnelPos"                  , 0                                                   },
	{  647, "MG_SetPlayerOffset"                     , 0                                                   },
	{  648, "MG_SetPlayerInvincibility"              , 0                                                   },
	{  649, "MG_SetPlayerSpeed"                      , 0                                                   },
	{  650, "MG_SetPlayerMinSpeed"                   , 0                                                   },
	{  651, "MG_SetPlayerAccelerationPerSecond"      , 0                                                   },
	{  652, "MG_SetPlayerTunnelPos"                  , 0                                                   },
	{  653, "MG_GetPlayerTunnelNeg"                  , 0                                                   },
	{  654, "MG_SetPlayerTunnelNeg"                  , 0                                                   },
	{  655, "MG_GetPlayerOrigin"                     , 0                                                   },
	{  656, "MG_SetPlayerOrigin"                     , 0                                                   },
	{  657, "MG_GetGunBankHorizontalSpread"          , 0                                                   },
	{  658, "MG_GetGunBankVerticalSpread"            , 0                                                   },
	{  659, "MG_GetGunBankSensingRadius"             , 0                                                   },
	{  660, "MG_GetGunBankInaccuracy"                , 0                                                   },
	{  661, "MG_SetGunBankHorizontalSpread"          , 0                                                   },
	{  662, "MG_SetGunBankVerticalSpread"            , 0                                                   },
	{  663, "MG_SetGunBankSensingRadius"             , 0                                                   },
	{  664, "MG_SetGunBankInaccuracy"                , 0                                                   },
	{  665, "MG_GetIsInvulnerable"                   , 0                                                   },
	{  666, "MG_StartInvulnerability"                , 0                                                   },
	{  667, "MG_GetPlayerMaxSpeed"                   , 0                                                   },
	{  668, "MG_SetPlayerMaxSpeed"                   , 0                                                   },
	{  669, "ZZ_CUT_AddJournalWorldEntry"            , 0                                                   },
	{  670, "ZZ_CUT_AddJournalWorldEntryStrref"      , 0                                                   },
	{  671, "BarkString"                             , 0                                                   },
	{  672, "BarkStringPlaying"                      , 0                                                   },
	{  673, "ZZ_CUT_DeleteJournalWorldEntry"         , 0                                                   },
	{  674, "ZZ_CUT_DeleteJournalWorldEntryStrref"   , 0                                                   },
	{  675, "ZZ_CUT_EffectForceDrain"                , 0                                                   },
	{  676, "ZZ_CUT_EffectPsychicStatic"             , 0                                                   },
	{  677, "ZZ_CUT_PlayVisualAreaEffect"            , 0                                                   },
	{  678, "ZZ_CUT_SetJournalQuestEntryPicture"     , 0                                                   },
	{  679, "ZZ_CUT_GetLocalBoolean"                 , 0                                                   },
	{  680, "ZZ_CUT_SetLocalBoolean"                 , 0                                                   },
	{  681, "ZZ_CUT_GetLocalNumber"                  , 0                                                   },
	{  682, "ZZ_CUT_SetLocalNumber"                  , 0                                                   },
	{  683, "ZZ_CUT_MG_GetSoundFrequency"            , 0                                                   },
	{  684, "ZZ_CUT_MG_SetSoundFrequency"            , 0                                                   },
	{  685, "ZZ_CUT_MG_GetSoundFrequencyIsRandom"    , 0                                                   },
	{  686, "ZZ_CUT_MG_SetSoundFrequencyIsRandom"    , 0                                                   },
	{  687, "ZZ_CUT_MG_GetSoundVolume"               , 0                                                   },
	{  688, "ZZ_CUT_MG_SetSoundVolume"               , 0                                                   },
	{  689, "ZZ_CUT_SoundObjectGetPitchVariance"     , 0                                                   },
	{  690, "ZZ_CUT_SoundObjectSetPitchVariance"     , 0                                                   },
	{  691, "ZZ_CUT_SoundObjectGetVolume"            , 0                                                   },
	{  692, "ZZ_HIDDEN_GetGlobalLocation"            , 0                                                   },
	{  693, "ZZ_HIDDEN_SetGlobalLocation"            , 0                                                   },
	{  694, "ZZ_CUT_AddAvailableNPCByObject"         , 0                                                   },
	{  695, "ZZ_CUT_RemoveAvailableNPC"              , 0                                                   },
	{  696, "ZZ_CUT_GetIsAvailableCreature"          , 0                                                   },
	{  697, "ZZ_CUT_AddAvailableNPCByTemplate"       , 0                                                   },
	{  698, "ZZ_CUT_SpawnAvailableNPC"               , 0                                                   },
	{  699, "ZZ_CUT_GetIsNPCPartyMember"             , 0                                                   },
	{  700, "ZZ_CUT_ActionBarkString"                , 0                                                   },
	{  701, "ZZ_CUT_GetIsConversationActive"         , 0                                                   },
	{  702, "ZZ_CUT_EffectLightsaberThrow"           , 0                                                   },
	{  703, "ZZ_CUT_EffectWhirlWind"                 , 0                                                   },
	{  704, "ZZ_CUT_SetPartyAIStyle"                 , 0                                                   },
	{  705, "ZZ_CUT_SetNPCAIStyle"                   , 0                                                   },
	{  706, "ZZ_CUT_SetNPCSelectability"             , 0                                                   },
	{  707, "ZZ_CUT_GetNPCSelectability"             , 0                                                   },
	{  708, "GetCurrentStyle"                        , 0                                                   },
	{  709, "SetCurrentStyle"                        , 0                                                   },
	{  710, "ClearAllEffects"                        , 0                                                   },
	{  711, "GetLastConversation"                    , 0                                                   },
	{  712, "GetFaction"                             , 0                                                   },
	{  713, "GetStandardFaction"                     , 0                                                   },
	{  714, "ZZ_CUT_GivePlotXP"                      , 0                                                   },
	{  715, "GetMinimumHealth"                       , 0                                                   },
	{  716, "SetMinimumHealth"                       , 0                                                   },
	{  717, "MG_GetPlayerTunnelInfinite"             , 0                                                   },
	{  718, "MG_SetPlayerTunnelInfinite"             , 0                                                   },
	{  719, "ZZ_CUT_SetGlobalFadeIn"                 , 0                                                   },
	{  720, "ZZ_CUT_SetGlobalFadeOut"                , 0                                                   },
	{  721, "ZZ_COMBAT_GetLastHostileTarget"         , 0                                                   },
	{  722, "ZZ_COMBAT_GetLastAttackAction"          , 0                                                   },
	{  723, "ZZ_CUT_GetLastForcePowerUsed"           , 0                                                   },
	{  724, "ZZ_CUT_GetLastCombatFeatUsed"           , 0                                                   },
	{  725, "ZZ_COMBAT_GetLastAttackResult"          , 0                                                   },
	{  726, "ZZ_CUT_GetWasForcePowerSuccessful"      , 0                                                   },
	{  727, "ZZ_COMBAT_GetFirstAttacker"             , 0                                                   },
	{  728, "ZZ_COMBAT_GetNextAttacker"              , 0                                                   },
	{  729, "ZZ_CUT_SetFormation"                    , 0                                                   },
	{  730, "ZZ_CUT_ActionFollowLeader"              , 0                                                   },
	{  731, "ZZ_CUT_SetForcePowerUnsuccessful"       , 0                                                   },
	{  732, "ZZ_CUT_GetIsDebilitated"                , 0                                                   },
	{  733, "PlayMovie"                              , 0                                                   },
	{  734, "ZZ_CUT_SaveNPCState"                    , 0                                                   },
	{  735, "ZZ_CUT_GetCategoryFromTalent"           , 0                                                   },
	{  736, "FadeScreen"                             , 0                                                   },
	{  737, "PauseMovie"                             , 0                                                   },
	{  738, "StopMovie"                              , 0                                                   },
	{  739, "PlayMovieBlocking"                      , 0                                                   },
	{  740, "ActionPlaySound2D"                      , 0                                                   },
	{  741, "ActionPlaySound3D"                      , 0                                                   },
	{  742, "ActionPlaySoundAtPosition"              , 0                                                   },
	{  743, "LoadTaskGroup"                          , 0                                                   },
	{  744, "GetTaskComplete"                        , 0                                                   },
	{  745, "GetTaskGroupComplete"                   , 0                                                   },
	{  746, "EffectModifyHealth"                     , 0                                                   },
	{  747, "EffectModifyFocus"                      , 0                                                   },
	{  748, "EffectModifyChi"                        , 0                                                   },
	{  749, "SetTaskComplete"                        , 0                                                   },
	{  750, "SetAreaFogColor"                        , 0                                                   },
	{  751, "GetIsLiveContentAvailable"              , 0                                                   },
	{  752, "ShowWorldMap"                           , 0                                                   },
	{  753, "GetAreaState"                           , 0                                                   },
	{  754, "SetAreaState"                           , 0                                                   },
	{  755, "GetObjectActive"                        , 0                                                   },
	{  756, "SetObjectActive"                        , 0                                                   },
	{  757, "FailQuest"                              , 0                                                   },
	{  758, "GetPlayerCreatureCombatTarget"          , 0                                                   },
	{  759, "SetCombatCounter"                       , 0                                                   },
	{  760, "GetCombatCounter"                       , 0                                                   },
	{  761, "SetCombatTimerCurrent"                  , 0                                                   },
	{  762, "SetCombatTimerFinish"                   , 0                                                   },
	{  763, "SetCombatTimerOn"                       , 0                                                   },
	{  764, "GetCombatTimerCurrent"                  , 0                                                   },
	{  765, "GetCombatTimerOn"                       , 0                                                   },
	{  766, "SetPosition"                            , 0                                                   },
	{  767, "GetCombatTeam"                          , 0                                                   },
	{  768, "SetCombatTeam"                          , 0                                                   },
	{  769, "GetAggressivenessValue"                 , 0                                                   },
	{  770, "SetAggressivenessValue"                 , 0                                                   },
	{  771, "EffectDamageHealth"                     , 0                                                   },
	{  772, "EffectDamageChi"                        , 0                                                   },
	{  773, "EffectDamageFocus"                      , 0                                                   },
	{  774, "EffectDamageDrunk"                      , 0                                                   },
	{  775, "SetFactionReputation"                   , 0                                                   },
	{  776, "GetGlobalBoolean"                       , &Functions::getGlobalBool                           },
	{  777, "SetGlobalBoolean"                       , &Functions::setGlobalBool                           },
	{  778, "GetLocalBoolean"                        , &Functions::getLocalBool                            },
	{  779, "SetLocalBoolean"                        , &Functions::setLocalBool                            },
	{  780, "EffectSlowed"                           , 0                                                   },
	{  781, "EffectParalyzed"                        , 0                                                   },
	{  782, "EffectDazed"                            , 0                                                   },
	{  783, "EffectHealing"                          , 0                                                   },
	{  784, "GetRecoveryTime"                        , 0                                                   },
	{  785, "SetRecoveryTime"                        , 0                                                   },
	{  786, "SetQuestActive"                         , 0                                                   },
	{  787, "GetQuestActive"                         , 0                                                   },
	{  788, "ZZ_CUT_CompleteQuest"                   , 0                                                   },
	{  789, "GetQuestComplete"                       , 0                                                   },
	{  790, "GetQuestFailed"                         , 0                                                   },
	{  791, "GetFavoredMove"                         , 0                                                   },
	{  792, "SetFavoredMove"                         , 0                                                   },
	{  793, "GetCombatProbAttack"                    , 0                                                   },
	{  794, "GetCombatProbBlock"                     , 0                                                   },
	{  795, "GetCombatProbHeal"                      , 0                                                   },
	{  796, "GetCombatProbMove"                      , 0                                                   },
	{  797, "GetCombatProbSwitchStyle"               , 0                                                   },
	{  798, "GetCombatProbIdle"                      , 0                                                   },
	{  799, "SetCombatProbabilities"                 , 0                                                   },
	{  800, "GetAccumulationRatio"                   , 0                                                   },
	{  801, "GetFogColour"                           , 0                                                   },
	{  802, "GetFogNear"                             , 0                                                   },
	{  803, "GetFogFar"                              , 0                                                   },
	{  804, "GetForceDistortion"                     , 0                                                   },
	{  805, "GetNoise"                               , 0                                                   },
	{  806, "GetOceanFresnel"                        , 0                                                   },
	{  807, "GetOceanHeight"                         , 0                                                   },
	{  808, "GetOceanWaveClip"                       , 0                                                   },
	{  809, "GetOceanWaveHeight"                     , 0                                                   },
	{  810, "GetOceanWaveOffset"                     , 0                                                   },
	{  811, "GetOceanWaveFreq"                       , 0                                                   },
	{  812, "GetOceanWaveDirX"                       , 0                                                   },
	{  813, "GetOceanWaveDirY"                       , 0                                                   },
	{  814, "GetPixelFlareThresholdColour"           , 0                                                   },
	{  815, "GetPixelFlareThresholdMaxScale"         , 0                                                   },
	{  816, "GetSaturation"                          , 0                                                   },
	{  817, "GetWindDirection"                       , 0                                                   },
	{  818, "GetWindVariation"                       , 0                                                   },
	{  819, "SetAccumulationRatio"                   , 0                                                   },
	{  820, "SetFogColour"                           , 0                                                   },
	{  821, "SetFogNear"                             , 0                                                   },
	{  822, "SetFogFar"                              , 0                                                   },
	{  823, "SetForceDistortion"                     , 0                                                   },
	{  824, "SetNoise"                               , 0                                                   },
	{  825, "SetOceanFresnel"                        , 0                                                   },
	{  826, "SetOceanHeight"                         , 0                                                   },
	{  827, "SetOceanWaveClip"                       , 0                                                   },
	{  828, "SetOceanWaveHeight"                     , 0                                                   },
	{  829, "SetOceanWaveOffset"                     , 0                                                   },
	{  830, "SetOceanWaveFreq"                       , 0                                                   },
	{  831, "SetOceanWaveDirX"                       , 0                                                   },
	{  832, "SetOceanWaveDirY"                       , 0                                                   },
	{  833, "SetPixelFlareThresholdColour"           , 0                                                   },
	{  834, "SetPixelFlareThresholdMaxScale"         , 0                                                   },
	{  835, "SetSaturation"                          , 0                                                   },
	{  836, "SetWindDirection"                       , 0                                                   },
	{  837, "SetWindVariation"                       , 0                                                   },
	{  838, "AnimateAccumulationRatio"               , 0                                                   },
	{  839, "AnimateFogColour"                       , 0                                                   },
	{  840, "AnimateFogNear"                         , 0                                                   },
	{  841, "AnimateFogFar"                          , 0                                                   },
	{  842, "AnimateForceDistortion"                 , 0                                                   },
	{  843, "AnimateNoise"                           , 0                                                   },
	{  844, "AnimateOceanFresnel"                    , 0                                                   },
	{  845, "AnimateOceanHeight"                     , 0                                                   },
	{  846, "AnimateOceanWaveClip"                   , 0                                                   },
	{  847, "AnimateOceanWaveHeights"                , 0                                                   },
	{  848, "AnimateOceanWaveOffsets"                , 0                                                   },
	{  849, "AnimateOceanWaveFreqs"                  , 0                                                   },
	{  850, "AnimateOceanWaveDirX"                   , 0                                                   },
	{  851, "AnimateOceanWaveDirY"                   , 0                                                   },
	{  852, "AnimatePixelFlareThreshold"             , 0                                                   },
	{  853, "AnimateSaturation"                      , 0                                                   },
	{  854, "AnimateWindDirection"                   , 0                                                   },
	{  855, "AnimateWindVariation"                   , 0                                                   },
	{  856, "AnimateRestoreRoomDefaults"             , 0                                                   },
	{  857, "Create2DAEffect"                        , 0                                                   },
	{  858, "Create2DAEffectByLabel"                 , 0                                                   },
	{  859, "Get2DAEntryString"                      , &Functions::get2DAEntryString                       },
	{  860, "Get2DAEntryFloat"                       , &Functions::get2DAEntryFloat                        },
	{  861, "Get2DAEntryInt"                         , &Functions::get2DAEntryInt                          },
	{  862, "GetUseDefaultStyleAI"                   , 0                                                   },
	{  863, "SetUseDefaultStyleAI"                   , 0                                                   },
	{  864, "GetCombatTemplate"                      , 0                                                   },
	{  865, "SetCombatTemplate"                      , 0                                                   },
	{  866, "GetObtainPowerUp"                       , 0                                                   },
	{  867, "SetObtainPowerUp"                       , 0                                                   },
	{  868, "ApplyEffectInRadius"                    , 0                                                   },
	{  869, "SetCombatCounterOn"                     , 0                                                   },
	{  870, "AddCombatStyle"                         , 0                                                   },
	{  871, "AddCombatStylePoints"                   , 0                                                   },
	{  872, "SetCombatAreaActive"                    , 0                                                   },
	{  873, "SetCombatAreaEnterable"                 , 0                                                   },
	{  874, "SetCombatAreaExitable"                  , 0                                                   },
	{  875, "AddCombatAreaParticipant"               , 0                                                   },
	{  876, "RemoveCombatAreaParticipant"            , 0                                                   },
	{  877, "AddCombatAreaTeam"                      , 0                                                   },
	{  878, "RemoveCombatAreaTeam"                   , 0                                                   },
	{  879, "SetCombatAreaInvulnerable"              , 0                                                   },
	{  880, "SetCombatAreaCutsceneMode"              , 0                                                   },
	{  881, "GetCombatAreaParticipantByIndex"        , 0                                                   },
	{  882, "GetCombatAreaTotalParticipants"         , 0                                                   },
	{  883, "GetCombatAreaActive"                    , 0                                                   },
	{  884, "GetIsObjectInCombatArea"                , 0                                                   },
	{  885, "GetCombatAreaParticipantByTag"          , 0                                                   },
	{  886, "EffectModifyHealthBase"                 , 0                                                   },
	{  887, "EffectModifyChiBase"                    , 0                                                   },
	{  888, "EffectModifyFocusBase"                  , 0                                                   },
	{  889, "EffectModifyDrunkBase"                  , 0                                                   },
	{  890, "SetCombatAreaCombatOverride"            , 0                                                   },
	{  891, "SetAmuletState"                         , 0                                                   },
	{  892, "GetIsLastParticipantInCombatArea"       , 0                                                   },
	{  893, "GetCombatAreaNoQuestXP"                 , 0                                                   },
	{  894, "SetCombatAreaNoQuestXP"                 , 0                                                   },
	{  895, "SetCombatAreaNoRewards"                 , 0                                                   },
	{  896, "SetObjectUntargettable"                 , 0                                                   },
	{  897, "ShowScoreBoard"                         , 0                                                   },
	{  898, "GetScoreBoardPosition"                  , 0                                                   },
	{  899, "WriteToScoreBoard"                      , 0                                                   },
	{  900, "ShowChapterScreen"                      , 0                                                   },
	{  901, "GetTaskGroupActive"                     , 0                                                   },
	{  902, "SetCreatureHalfShadow"                  , 0                                                   },
	{  903, "SetCreatureFullShadow"                  , 0                                                   },
	{  904, "SetAllCreaturesHalfShadow"              , 0                                                   },
	{  905, "LoadAMPFile"                            , 0                                                   },
	{  906, "ClearAMPSettings"                       , 0                                                   },
	{  907, "GetMaxChi"                              , 0                                                   },
	{  908, "GetMaxFocus"                            , 0                                                   },
	{  909, "GetChi"                                 , 0                                                   },
	{  910, "GetFocus"                               , 0                                                   },
	{  911, "GetFocusTime"                           , 0                                                   },
	{  912, "GetFocusPackets"                        , 0                                                   },
	{  913, "GetMaxFocusTime"                        , 0                                                   },
	{  914, "GetMaxFocusPackets"                     , 0                                                   },
	{  915, "GetInitialCombatCameraAngle"            , 0                                                   },
	{  916, "SetInitialCombatCameraAngle"            , 0                                                   },
	{  917, "SetLoadHintOverride"                    , 0                                                   },
	{  918, "SplashScreens"                          , 0                                                   },
	{  919, "SetInvulnerable"                        , 0                                                   },
	{  920, "SetDeathHintOverride"                   , 0                                                   },
	{  921, "RemoveFromAllCombatAreas"               , 0                                                   },
	{  922, "StartSpeedBlur"                         , 0                                                   },
	{  923, "EndSpeedBlur"                           , 0                                                   },
	{  924, "MapStyleToQuickSlot"                    , 0                                                   },
	{  925, "GetAvoidCombatState"                    , 0                                                   },
	{  926, "SetAvoidCombatState"                    , 0                                                   },
	{  927, "EffectDeath"                            , 0                                                   },
	{  928, "GetArtPlaceableTransparent"             , 0                                                   },
	{  929, "SetArtPlaceableTransparent"             , 0                                                   },
	{  930, "SetDisableCombatAI"                     , 0                                                   },
	{  931, "SetGameSpeed"                           , 0                                                   },
	{  932, "SetDecalTint"                           , 0                                                   },
	{  933, "ShowFloaties"                           , 0                                                   },
	{  934, "ClearScoreBoard"                        , 0                                                   },
	{  935, "SetDisableCreatureCombatAI"             , 0                                                   },
	{  936, "UpdateScriptedProgressString"           , 0                                                   },
	{  937, "RemoveScriptedProgressString"           , 0                                                   },
	{  938, "SetRomanceIndex"                        , 0                                                   },
	{  939, "GetCurrentCombatArea"                   , 0                                                   },
	{  940, "PlayCombatAnimation"                    , 0                                                   },
	{  941, "GetAttackPattern"                       , 0                                                   },
	{  942, "SetAttackPattern"                       , 0                                                   },
	{  943, "ShowAmuletGui"                          , 0                                                   },
	{  944, "ShowDialogBox"                          , 0                                                   },
	{  945, "ShowHint"                               , 0                                                   },
	{  946, "EffectEvade"                            , 0                                                   },
	{  947, "ShowTutorial"                           , 0                                                   },
	{  948, "ShowRewardScreen"                       , 0                                                   },
	{  949, "CreateCrustEffect"                      , 0                                                   },
	{  950, "FinishCrustEffect"                      , 0                                                   },
	{  951, "DestroyCrustEffect"                     , 0                                                   },
	{  952, "ShowGroundPiles"                        , 0                                                   },
	{  953, "GetE3Num"                               , 0                                                   },
	{  954, "SetCombatStyleShoutOverride"            , 0                                                   },
	{  955, "ResetCamera"                            , 0                                                   },
	{  956, "OverrideBodyMaterial"                   , 0                                                   },
	{  957, "OverrideHeadMaterial"                   , 0                                                   },
	{  958, "AnimateBrightening"                     , 0                                                   },
	{  959, "GetRandomRewardXP"                      , 0                                                   },
	{  960, "GetRandomRewardSilver"                  , 0                                                   },
	{  961, "GetRandomRewardTreasure"                , 0                                                   },
	{  962, "GetRandomRewardMaxItem"                 , 0                                                   },
	{  963, "GetRandomRewardPowerUpClass"            , 0                                                   },
	{  964, "GetRandomRewardPowerUpChance"           , 0                                                   },
	{  965, "GetRandomRewardWeight"                  , 0                                                   },
	{  966, "GetRandomRewardRefreshOnActivate"       , 0                                                   },
	{  967, "GetRandomRewardWeightOverload"          , 0                                                   },
	{  968, "GetRandomRewardActive"                  , 0                                                   },
	{  969, "SetRandomRewardXP"                      , 0                                                   },
	{  970, "SetRandomRewardSilver"                  , 0                                                   },
	{  971, "SetRandomRewardTreasure"                , 0                                                   },
	{  972, "SetRandomRewardMaxItem"                 , 0                                                   },
	{  973, "SetRandomRewardPowerUpClass"            , 0                                                   },
	{  974, "SetRandomRewardPowerUpChance"           , 0                                                   },
	{  975, "SetRandomRewardWeight"                  , 0                                                   },
	{  976, "SetRandomRewardRefreshOnActivate"       , 0                                                   },
	{  977, "SetRandomRewardWeightOverload"          , 0                                                   },
	{  978, "SetRandomRewardActive"                  , 0                                                   },
	{  979, "FadeObjectAlpha"                        , 0                                                   },
	{  980, "PerformStyleMove"                       , 0                                                   },
	{  981, "CreateRandomReward"                     , 0                                                   },
	{  982, "CreateRandomRewardTag"                  , 0                                                   },
	{  983, "GetCombatNeutralBehavior"               , 0                                                   },
	{  984, "SetCombatNeutralBehavior"               , 0                                                   },
	{  985, "GetLastDamagerStyle"                    , 0                                                   },
	{  986, "DisableAllCombatButtons"                , 0                                                   },
	{  987, "EnableAllCombatButtons"                 , 0                                                   },
	{  988, "ChangeCombatButtonState"                , 0                                                   },
	{  989, "ApplyStyleAdvanceLine"                  , 0                                                   },
	{  990, "CleanCutsceneMoveQueue"                 , 0                                                   },
	{  991, "PerformStyleMoveNoQueue"                , 0                                                   },
	{  992, "ZZ_GetPlayerCreatureAlignment"          , 0                                                   },
	{  993, "ZZ_SetPlayerCreatureAlignment"          , 0                                                   },
	{  994, "SetTutorialMode"                        , 0                                                   },
	{  995, "SetMonitoredButton"                     , 0                                                   },
	{  996, "GetChapter"                             , 0                                                   },
	{  997, "SetChapter"                             , 0                                                   },
	{  998, "GetTutorialMode"                        , 0                                                   },
	{  999, "SetNoiseTexture"                        , 0                                                   },
	{ 1000, "ToggleNoise"                            , 0                                                   },
	{ 1001, "CombatAreaSubCombatEnd"                 , 0                                                   },
	{ 1002, "OceanEnable"                            , 0                                                   },
	{ 1003, "LockCurrentStyle"                       , 0                                                   },
	{ 1004, "GetOverrideCombatStance"                , 0                                                   },
	{ 1005, "SetOverrideCombatStance"                , 0                                                   },
	{ 1006, "SetCreatureIllumination"                , 0                                                   },
	{ 1007, "RestoreCreatureIllumination"            , 0                                                   },
	{ 1008, "GetUseDefaultDeathType"                 , 0                                                   },
	{ 1009, "SetUseDefaultDeathType"                 , 0                                                   },
	{ 1010, "CombatMoveCreature"                     , 0                                                   },
	{ 1011, "SetCreatureSpeed"                       , 0                                                   },
	{ 1012, "SetMiniMapMode"                         , 0                                                   },
	{ 1013, "SetOverrideEffectDuration"              , 0                                                   },
	{ 1014, "GetCreatureRank"                        , 0                                                   },
	{ 1015, "Get2DANumRows"                          , &Functions::get2DANumRows                           },
	{ 1016, "Get2DANumColumn"                        , &Functions::get2DANumColumn                         },
	{ 1017, "SetCombatCollision"                     , 0                                                   },
	{ 1018, "LogMemory"                              , 0                                                   },
	{ 1019, "SetHenchmanPortraitType"                , 0                                                   },
	{ 1020, "SetResistanceOverride"                  , 0                                                   },
	{ 1021, "GetResistanceOverride"                  , 0                                                   },
	{ 1022, "GetPlotGiver"                           , 0                                                   },
	{ 1023, "SetPlotGiver"                           , 0                                                   },
	{ 1024, "ToggleDistortion"                       , 0                                                   },
	{ 1025, "ShowGuiRing"                            , 0                                                   },
	{ 1026, "SetSepiaEffect"                         , 0                                                   },
	{ 1027, "SetFogOfWarExplored"                    , 0                                                   },
	{ 1028, "ResetTaskGroup"                         , 0                                                   },
	{ 1029, "AddKnownRecipe"                         , 0                                                   },
	{ 1030, "AttachObjectToObject"                   , 0                                                   },
	{ 1031, "DetachObject"                           , 0                                                   },
	{ 1032, "RemoveFogOfWarFromCurrentArea"          , 0                                                   },
	{ 1033, "PlayAnimationAdvanced"                  , 0                                                   },
	{ 1034, "GetFileExists"                          , 0                                                   },
	{ 1035, "AllowQuickTravel"                       , 0                                                   },
	{ 1036, "GetStyleType"                           , 0                                                   },
	{ 1037, "GetAutoBalance"                         , &Functions::getAutoBalance                          },
	{ 1038, "RemoveQuest"                            , 0                                                   },
	{ 1039, "GetLastHostileAttacker"                 , 0                                                   },
	{ 1040, "SaveVisibilityGraph"                    , 0                                                   },
	{ 1041, "RestoreVisibiltyGraph"                  , 0                                                   },
	{ 1042, "PreloadRoom"                            , 0                                                   },
	{ 1043, "UnloadRoom"                             , 0                                                   },
	{ 1044, "PlaceableCastSpell"                     , 0                                                   },
	{ 1045, "SetHenchmanFollowPlayerInCombat"        , 0                                                   },
	{ 1046, "SetChi"                                 , 0                                                   },
	{ 1047, "GetSystemTime"                          , 0                                                   },
	{ 1048, "HaveSaveGames"                          , 0                                                   },
	{ 1049, "GetUnspentStylePoints"                  , 0                                                   },
	{ 1050, "GetUnspentAttributePoints"              , 0                                                   },
	{ 1051, "SetStayTransformed"                     , 0                                                   },
	{ 1052, "SetDoNotEnterCombatArea"                , 0                                                   },
	{ 1053, "SetCombatVFXOverride"                   , 0                                                   },
	{ 1054, "TransformHenchman"                      , 0                                                   },
	{ 1055, "DisableEvade"                           , 0                                                   },
	{ 1056, "GetIsInFocus"                           , 0                                                   },
	{ 1057, "SetIsInPOPBattle2"                      , 0                                                   },
	{ 1058, "CreateJadeMasterSave"                   , 0                                                   }
};

/** The table defining the signature (return type and type of parameters) of each engine function. */
const Functions::FunctionSignature Functions::kFunctionSignatures[] = {
	{    0, kTypeInt       , { kTypeInt } },
	{    1, kTypeVoid      , { kTypeString, kTypeInt } },
	{    2, kTypeVoid      , { kTypeFloat, kTypeInt, kTypeInt, kTypeInt } },
	{    3, kTypeString    , { kTypeFloat, kTypeInt, kTypeInt } },
	{    4, kTypeVoid      , { kTypeInt, kTypeInt } },
	{    5, kTypeVoid      , { kTypeObject, kTypeInt } },
	{    6, kTypeVoid      , { kTypeObject, kTypeScriptState } },
	{    7, kTypeVoid      , { kTypeFloat, kTypeScriptState } },
	{    8, kTypeVoid      , { kTypeObject, kTypeString } },
	{    9, kTypeVoid      , { } },
	{   10, kTypeVoid      , { kTypeFloat, kTypeInt, kTypeObject, kTypeInt } },
	{   11, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{   12, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{   13, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt } },
	{   14, kTypeVoid      , { } },
	{   15, kTypeInt       , { kTypeObject, kTypeString } },
	{   16, kTypeInt       , { } },
	{   17, kTypeInt       , { } },
	{   18, kTypeInt       , { } },
	{   19, kTypeInt       , { } },
	{   20, kTypeVoid      , { kTypeInt } },
	{   21, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeInt } },
	{   22, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeInt } },
	{   23, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeInt } },
	{   24, kTypeObject    , { kTypeObject } },
	{   25, kTypeObject    , { } },
	{   26, kTypeObject    , { } },
	{   27, kTypeVector    , { kTypeObject } },
	{   28, kTypeFloat     , { kTypeObject } },
	{   29, kTypeInt       , { kTypeObject, kTypeString, kTypeObject } },
	{   30, kTypeInt       , { kTypeObject, kTypeInt } },
	{   31, kTypeString    , { kTypeObject } },
	{   32, kTypeString    , { kTypeObject } },
	{   33, kTypeString    , { kTypeObject } },
	{   34, kTypeString    , { } },
	{   35, kTypeString    , { } },
	{   36, kTypeVoid      , { kTypeInt, kTypeObject } },
	{   37, kTypeInt       , { kTypeObject, kTypeObject } },
	{   38, kTypeVoid      , { kTypeObject } },
	{   39, kTypeVoid      , { kTypeInt } },
	{   40, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{   41, kTypeFloat     , { kTypeObject } },
	{   42, kTypeInt       , { kTypeObject } },
	{   43, kTypeVoid      , { kTypeObject } },
	{   44, kTypeVoid      , { kTypeObject } },
	{   45, kTypeVoid      , { kTypeFloat } },
	{   46, kTypeVoid      , { kTypeString } },
	{   47, kTypeVoid      , { kTypeString } },
	{   48, kTypeVoid      , { kTypeString, kTypeVector } },
	{   49, kTypeInt       , { kTypeObject } },
	{   50, kTypeInt       , { kTypeObject } },
	{   51, kTypeInt       , { kTypeObject, kTypeInt } },
	{   52, kTypeFloat     , { kTypeObject, kTypeInt } },
	{   53, kTypeString    , { kTypeObject, kTypeInt } },
	{   54, kTypeObject    , { kTypeObject, kTypeInt } },
	{   55, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{   56, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{   57, kTypeVoid      , { kTypeObject, kTypeInt, kTypeString } },
	{   58, kTypeVoid      , { kTypeObject, kTypeInt, kTypeObject } },
	{   59, kTypeInt       , { kTypeString } },
	{   60, kTypeString    , { kTypeString } },
	{   61, kTypeString    , { kTypeString } },
	{   62, kTypeString    , { kTypeString, kTypeInt } },
	{   63, kTypeString    , { kTypeString, kTypeInt } },
	{   64, kTypeString    , { kTypeString, kTypeString, kTypeInt } },
	{   65, kTypeString    , { kTypeString, kTypeInt, kTypeInt } },
	{   66, kTypeInt       , { kTypeString, kTypeString } },
	{   67, kTypeFloat     , { kTypeFloat } },
	{   68, kTypeFloat     , { kTypeFloat } },
	{   69, kTypeFloat     , { kTypeFloat } },
	{   70, kTypeFloat     , { kTypeFloat } },
	{   71, kTypeFloat     , { kTypeFloat } },
	{   72, kTypeFloat     , { kTypeFloat } },
	{   73, kTypeFloat     , { kTypeFloat } },
	{   74, kTypeFloat     , { kTypeFloat } },
	{   75, kTypeFloat     , { kTypeFloat, kTypeFloat } },
	{   76, kTypeFloat     , { kTypeFloat } },
	{   77, kTypeInt       , { kTypeInt } },
	{   78, kTypeVoid      , { kTypeObject, kTypeInt } },
	{   79, kTypeVoid      , { kTypeString } },
	{   80, kTypeEngineType, { kTypeInt, kTypeInt } },
	{   81, kTypeInt       , { } },
	{   82, kTypeEngineType, { kTypeInt } },
	{   83, kTypeInt       , { } },
	{   84, kTypeInt       , { kTypeObject, kTypeString } },
	{   85, kTypeEngineType, { kTypeObject } },
	{   86, kTypeEngineType, { kTypeObject } },
	{   87, kTypeVoid      , { kTypeObject, kTypeEngineType, kTypeInt, kTypeString } },
	{   88, kTypeInt       , { kTypeEngineType } },
	{   89, kTypeInt       , { kTypeEngineType } },
	{   90, kTypeInt       , { kTypeEngineType } },
	{   91, kTypeObject    , { kTypeEngineType } },
	{   92, kTypeString    , { kTypeInt } },
	{   93, kTypeVoid      , { kTypeObject } },
	{   94, kTypeVoid      , { kTypeObject } },
	{   95, kTypeInt       , { kTypeInt } },
	{   96, kTypeInt       , { kTypeInt } },
	{   97, kTypeInt       , { kTypeInt } },
	{   98, kTypeInt       , { kTypeInt } },
	{   99, kTypeInt       , { kTypeInt } },
	{  100, kTypeInt       , { kTypeInt } },
	{  101, kTypeInt       , { kTypeInt } },
	{  102, kTypeInt       , { kTypeInt } },
	{  103, kTypeInt       , { kTypeInt } },
	{  104, kTypeFloat     , { kTypeVector } },
	{  105, kTypeVoid      , { kTypeString } },
	{  106, kTypeInt       , { kTypeObject } },
	{  107, kTypeInt       , { } },
	{  108, kTypeInt       , { kTypeFloat, kTypeInt } },
	{  109, kTypeInt       , { kTypeObject } },
	{  110, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  111, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  112, kTypeVoid      , { kTypeObject } },
	{  113, kTypeInt       , { kTypeObject } },
	{  114, kTypeObject    , { } },
	{  115, kTypeInt       , { } },
	{  116, kTypeObject    , { } },
	{  117, kTypeString    , { kTypeObject } },
	{  118, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  119, kTypeInt       , { kTypeObject } },
	{  120, kTypeInt       , { } },
	{  121, kTypeFloat     , { kTypeInt } },
	{  122, kTypeFloat     , { kTypeInt } },
	{  123, kTypeFloat     , { kTypeInt } },
	{  124, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  125, kTypeInt       , { kTypeObject } },
	{  126, kTypeString    , { kTypeObject } },
	{  127, kTypeInt       , { kTypeObject } },
	{  128, kTypeObject    , { kTypeEngineType, kTypeInt, kTypeFloat, kTypeInt, kTypeInt, kTypeVector } },
	{  129, kTypeObject    , { kTypeEngineType, kTypeInt, kTypeFloat, kTypeInt, kTypeInt, kTypeVector } },
	{  130, kTypeVoid      , { kTypeInt } },
	{  131, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{  132, kTypeEngineType, { kTypeInt } },
	{  133, kTypeVoid      , { } },
	{  134, kTypeString    , { } },
	{  135, kTypeVoid      , { kTypeObject, kTypeString } },
	{  136, kTypeVoid      , { kTypeObject, kTypeString } },
	{  137, kTypeVector    , { kTypeVector } },
	{  138, kTypeInt       , { kTypeObject, kTypeInt } },
	{  139, kTypeInt       , { kTypeObject, kTypeInt } },
	{  140, kTypeInt       , { kTypeObject } },
	{  141, kTypeVoid      , { kTypeVector, kTypeInt } },
	{  142, kTypeVector    , { kTypeFloat, kTypeFloat, kTypeFloat } },
	{  143, kTypeVoid      , { kTypeVector, kTypeInt, kTypeObject, kTypeInt } },
	{  144, kTypeVector    , { kTypeFloat } },
	{  145, kTypeFloat     , { kTypeVector } },
	{  146, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  147, kTypeVoid      , { kTypeInt, kTypeString, kTypeString } },
	{  148, kTypeVoid      , { kTypeObject } },
	{  149, kTypeVoid      , { kTypeInt, kTypeEngineType } },
	{  150, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  151, kTypeFloat     , { kTypeObject, kTypeObject } },
	{  152, kTypeVoid      , { kTypeObject, kTypeInt, kTypeEngineType } },
	{  153, kTypeEngineType, { kTypeObject, kTypeInt } },
	{  154, kTypeInt       , { kTypeEngineType, kTypeObject } },
	{  155, kTypeInt       , { kTypeInt } },
	{  156, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  157, kTypeInt       , { kTypeInt } },
	{  158, kTypeVoid      , { kTypeFloat } },
	{  159, kTypeInt       , { kTypeObject } },
	{  160, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  161, kTypeInt       , { } },
	{  162, kTypeVoid      , { kTypeInt, kTypeObject } },
	{  163, kTypeInt       , { kTypeObject } },
	{  164, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  165, kTypeEngineType, { kTypeInt } },
	{  166, kTypeVoid      , { kTypeString, kTypeString } },
	{  167, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{  168, kTypeString    , { kTypeObject } },
	{  169, kTypeVoid      , { kTypeString } },
	{  170, kTypeInt       , { kTypeEngineType } },
	{  171, kTypeInt       , { } },
	{  172, kTypeVoid      , { kTypeString, kTypeString, kTypeString } },
	{  173, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  174, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  175, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  176, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{  177, kTypeVoid      , { kTypeString } },
	{  178, kTypeVoid      , { kTypeString, kTypeVector } },
	{  179, kTypeVoid      , { kTypeString, kTypeVector } },
	{  180, kTypeVoid      , { kTypeString, kTypeString, kTypeInt } },
	{  181, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeFloat, kTypeInt } },
	{  182, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  183, kTypeVoid      , { kTypeVector, kTypeFloat, kTypeInt } },
	{  184, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  185, kTypeVoid      , { kTypeVector, kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  186, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  187, kTypeVoid      , { kTypeVector, kTypeFloat } },
	{  188, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{  189, kTypeVoid      , { kTypeString, kTypeString } },
	{  190, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  191, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  192, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  193, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  194, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  195, kTypeInt       , { } },
	{  196, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  197, kTypeObject    , { kTypeString } },
	{  198, kTypeObject    , { kTypeObject } },
	{  199, kTypeEngineType, { kTypeEngineType, kTypeEngineType } },
	{  200, kTypeObject    , { kTypeString, kTypeInt } },
	{  201, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  202, kTypeVoid      , { kTypeFloat, kTypeInt } },
	{  203, kTypeVoid      , { kTypeInt } },
	{  204, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  205, kTypeVoid      , { kTypeInt, kTypeObject } },
	{  206, kTypeVoid      , { kTypeInt } },
	{  207, kTypeString    , { kTypeObject, kTypeInt } },
	{  208, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  209, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  210, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  211, kTypeVoid      , { } },
	{  212, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  213, kTypeEngineType, { kTypeObject } },
	{  214, kTypeVoid      , { kTypeEngineType, kTypeInt } },
	{  215, kTypeEngineType, { kTypeObject, kTypeVector, kTypeFloat } },
	{  216, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeInt } },
	{  217, kTypeInt       , { kTypeObject } },
	{  218, kTypeFloat     , { kTypeFloat } },
	{  219, kTypeFloat     , { kTypeFloat } },
	{  220, kTypeVoid      , { kTypeObject, kTypeInt, kTypeEngineType, kTypeFloat } },
	{  221, kTypeVoid      , { } },
	{  222, kTypeEngineType, { } },
	{  223, kTypeVector    , { kTypeEngineType } },
	{  224, kTypeObject    , { kTypeEngineType } },
	{  225, kTypeFloat     , { kTypeEngineType } },
	{  226, kTypeVoid      , { kTypeInt, kTypeString, kTypeString, kTypeString } },
	{  227, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{  228, kTypeObject    , { kTypeEngineType, kTypeInt, kTypeInt } },
	{  229, kTypeVoid      , { kTypeInt } },
	{  230, kTypeFloat     , { kTypeInt } },
	{  231, kTypeInt       , { kTypeFloat } },
	{  232, kTypeInt       , { kTypeString } },
	{  233, kTypeFloat     , { kTypeString } },
	{  234, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  235, kTypeInt       , { kTypeObject, kTypeObject } },
	{  236, kTypeInt       , { kTypeObject, kTypeObject } },
	{  237, kTypeInt       , { kTypeObject, kTypeObject } },
	{  238, kTypeVoid      , { } },
	{  239, kTypeString    , { kTypeInt } },
	{  240, kTypeVoid      , { kTypeInt } },
	{  241, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{  242, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{  243, kTypeObject    , { kTypeInt, kTypeString, kTypeEngineType } },
	{  244, kTypeEngineType, { kTypeObject, kTypeInt, kTypeInt } },
	{  245, kTypeVoid      , { kTypeVector, kTypeFloat, kTypeInt } },
	{  246, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeInt, kTypeInt } },
	{  247, kTypeInt       , { } },
	{  248, kTypeInt       , { } },
	{  249, kTypeString    , { } },
	{  250, kTypeEngineType, { kTypeInt } },
	{  251, kTypeInt       , { } },
	{  252, kTypeObject    , { } },
	{  253, kTypeVoid      , { kTypeString, kTypeString, kTypeFloat, kTypeInt, kTypeInt } },
	{  254, kTypeObject    , { } },
	{  255, kTypeInt       , { kTypeObject, kTypeString } },
	{  256, kTypeVoid      , { kTypeObject, kTypeString, kTypeString, kTypeInt, kTypeInt, kTypeInt, kTypeFloat } },
	{  257, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeFloat, kTypeInt, kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{  258, kTypeVoid      , { kTypeObject, kTypeVector } },
	{  259, kTypeVoid      , { kTypeObject, kTypeVector } },
	{  260, kTypeObject    , { kTypeObject } },
	{  261, kTypeVoid      , { } },
	{  262, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{  263, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{  264, kTypeObject    , { kTypeObject } },
	{  265, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  266, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  267, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  268, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  269, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  270, kTypeEngineType, { } },
	{  271, kTypeEngineType, { } },
	{  272, kTypeString    , { kTypeObject } },
	{  273, kTypeEngineType, { kTypeInt } },
	{  274, kTypeInt       , { kTypeObject, kTypeInt, kTypeObject } },
	{  275, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  276, kTypeInt       , { kTypeObject } },
	{  277, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  278, kTypeInt       , { kTypeObject } },
	{  279, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  280, kTypeInt       , { kTypeObject } },
	{  281, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  282, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  283, kTypeInt       , { kTypeObject } },
	{  284, kTypeVoid      , { kTypeInt, kTypeString } },
	{  285, kTypeInt       , { kTypeObject } },
	{  286, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  287, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  288, kTypeInt       , { kTypeObject, kTypeInt } },
	{  289, kTypeInt       , { kTypeObject } },
	{  290, kTypeVoid      , { kTypeInt } },
	{  291, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeObject } },
	{  292, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  293, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  294, kTypeVoid      , { kTypeScriptState } },
	{  295, kTypeEngineType, { } },
	{  296, kTypeString    , { } },
	{  297, kTypeVoid      , { kTypeInt } },
	{  298, kTypeFloat     , { kTypeEngineType, kTypeEngineType } },
	{  299, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  300, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat } },
	{  301, kTypeInt       , { kTypeInt, kTypeInt, kTypeString } },
	{  302, kTypeFloat     , { kTypeInt, kTypeInt, kTypeString } },
	{  303, kTypeString    , { kTypeInt, kTypeInt, kTypeString } },
	{  304, kTypeInt       , { kTypeInt, kTypeObject } },
	{  305, kTypeInt       , { kTypeEngineType } },
	{  306, kTypeInt       , { kTypeInt, kTypeInt, kTypeInt } },
	{  307, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{  308, kTypeInt       , { kTypeInt, kTypeInt } },
	{  309, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt, kTypeInt } },
	{  310, kTypeVoid      , { kTypeInt } },
	{  311, kTypeVoid      , { kTypeInt } },
	{  312, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  313, kTypeVoid      , { kTypeEngineType, kTypeInt } },
	{  314, kTypeEngineType, { kTypeInt } },
	{  315, kTypeInt       , { kTypeInt } },
	{  316, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  317, kTypeInt       , { kTypeInt } },
	{  318, kTypeInt       , { kTypeObject } },
	{  319, kTypeVoid      , { } },
	{  320, kTypeInt       , { } },
	{  321, kTypeVoid      , { kTypeInt } },
	{  322, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  323, kTypeVoid      , { kTypeObject, kTypeObject, kTypeFloat, kTypeInt } },
	{  324, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  325, kTypeInt       , { kTypeObject } },
	{  326, kTypeInt       , { } },
	{  327, kTypeVoid      , { kTypeInt, kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{  328, kTypeVoid      , { } },
	{  329, kTypeVoid      , { kTypeObject } },
	{  330, kTypeObject    , { } },
	{  331, kTypeInt       , { } },
	{  332, kTypeVoid      , { kTypeInt } },
	{  333, kTypeVoid      , { kTypeString } },
	{  334, kTypeVoid      , { kTypeString } },
	{  335, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  336, kTypeObject    , { } },
	{  337, kTypeInt       , { kTypeObject, kTypeInt } },
	{  338, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  339, kTypeInt       , { kTypeObject } },
	{  340, kTypeInt       , { } },
	{  341, kTypeInt       , { kTypeObject } },
	{  342, kTypeInt       , { kTypeInt, kTypeInt, kTypeInt, kTypeFloat, kTypeInt, kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{  343, kTypeInt       , { kTypeObject } },
	{  344, kTypeInt       , { kTypeInt } },
	{  345, kTypeInt       , { } },
	{  346, kTypeObject    , { } },
	{  347, kTypeObject    , { } },
	{  348, kTypeVoid      , { kTypeInt } },
	{  349, kTypeObject    , { } },
	{  350, kTypeObject    , { } },
	{  351, kTypeFloat     , { kTypeObject } },
	{  352, kTypeInt       , { } },
	{  353, kTypeInt       , { kTypeObject } },
	{  354, kTypeObject    , { } },
	{  355, kTypeObject    , { } },
	{  356, kTypeVector    , { } },
	{  357, kTypeVector    , { } },
	{  358, kTypeInt       , { kTypeObject } },
	{  359, kTypeVector    , { kTypeObject } },
	{  360, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat, kTypeInt } },
	{  361, kTypeObject    , { } },
	{  362, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{  363, kTypeVoid      , { kTypeInt } },
	{  364, kTypeObject    , { kTypeInt, kTypeObject } },
	{  365, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  266, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  367, kTypeVoid      , { kTypeString, kTypeInt, kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{  368, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt, kTypeInt } },
	{  369, kTypeString    , { kTypeObject } },
	{  370, kTypeString    , { kTypeObject } },
	{  371, kTypeString    , { kTypeObject } },
	{  372, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  373, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  374, kTypeVoid      , { kTypeObject, kTypeString } },
	{  375, kTypeObject    , { } },
	{  376, kTypeObject    , { } },
	{  377, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  378, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  379, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  380, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  381, kTypeInt       , { kTypeObject, kTypeInt } },
	{  382, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeFloat, kTypeInt } },
	{  383, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeFloat, kTypeInt } },
	{  384, kTypeInt       , { kTypeString } },
	{  385, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  386, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  387, kTypeInt       , { kTypeObject, kTypeInt } },
	{  388, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  389, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  390, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat } },
	{  391, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat } },
	{  392, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat } },
	{  393, kTypeVoid      , { kTypeInt } },
	{  394, kTypeVoid      , { kTypeInt } },
	{  395, kTypeInt       , { } },
	{  396, kTypeString    , { kTypeInt } },
	{  397, kTypeInt       , { kTypeObject } },
	{  398, kTypeInt       , { kTypeObject, kTypeInt } },
	{  399, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  400, kTypeVoid      , { kTypeObject } },
	{  401, kTypeInt       , { kTypeObject } },
	{  402, kTypeVoid      , { } },
	{  403, kTypeVoid      , { } },
	{  404, kTypeVoid      , { } },
	{  405, kTypeInt       , { } },
	{  406, kTypeInt       , { } },
	{  407, kTypeInt       , { } },
	{  408, kTypeInt       , { } },
	{  409, kTypeInt       , { kTypeObject } },
	{  410, kTypeObject    , { } },
	{  411, kTypeEngineType, { } },
	{  412, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  413, kTypeVoid      , { kTypeObject } },
	{  414, kTypeVoid      , { kTypeObject } },
	{  415, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  416, kTypeVoid      , { kTypeObject, kTypeVector } },
	{  417, kTypeVoid      , { kTypeString, kTypeInt } },
	{  418, kTypeInt       , { kTypeObject } },
	{  419, kTypeInt       , { } },
	{  420, kTypeVoid      , { } },
	{  421, kTypeVoid      , { } },
	{  422, kTypeInt       , { kTypeObject, kTypeInt } },
	{  423, kTypeInt       , { } },
	{  424, kTypeEngineType, { kTypeObject, kTypeEngineType, kTypeObject } },
	{  425, kTypeVoid      , { kTypeInt } },
	{  426, kTypeVoid      , { kTypeInt, kTypeObject } },
	{  427, kTypeVoid      , { kTypeInt } },
	{  428, kTypeVoid      , { kTypeInt } },
	{  429, kTypeInt       , { } },
	{  430, kTypeVoid      , { kTypeInt } },
	{  431, kTypeVoid      , { kTypeString } },
	{  432, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  433, kTypeVoid      , { } },
	{  434, kTypeVoid      , { } },
	{  435, kTypeVoid      , { kTypeInt } },
	{  436, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  437, kTypeObject    , { } },
	{  438, kTypeObject    , { } },
	{  439, kTypeObject    , { } },
	{  440, kTypeObject    , { } },
	{  441, kTypeEngineType, { } },
	{  442, kTypeObject    , { } },
	{  443, kTypeInt       , { kTypeObject } },
	{  444, kTypeVoid      , { kTypeInt } },
	{  445, kTypeInt       , { kTypeObject } },
	{  446, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  447, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  448, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  449, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  450, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  451, kTypeEngineType, { kTypeInt } },
	{  452, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  453, kTypeEngineType, { kTypeInt, kTypeInt } },
	{  454, kTypeEngineType, { kTypeInt } },
	{  455, kTypeInt       , { kTypeObject } },
	{  456, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  457, kTypeEngineType, { kTypeInt } },
	{  458, kTypeEngineType, { kTypeInt } },
	{  459, kTypeEngineType, { } },
	{  460, kTypeEngineType, { kTypeInt } },
	{  461, kTypeEngineType, { } },
	{  462, kTypeEngineType, { } },
	{  463, kTypeEngineType, { kTypeInt } },
	{  464, kTypeEngineType, { } },
	{  465, kTypeEngineType, { } },
	{  466, kTypeEngineType, { } },
	{  467, kTypeEngineType, { } },
	{  468, kTypeEngineType, { } },
	{  469, kTypeInt       , { } },
	{  470, kTypeInt       , { } },
	{  471, kTypeInt       , { } },
	{  472, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  473, kTypeEngineType, { kTypeInt } },
	{  474, kTypeVoid      , { } },
	{  475, kTypeInt       , { kTypeObject, kTypeInt } },
	{  476, kTypeVoid      , { } },
	{  477, kTypeEngineType, { kTypeInt } },
	{  478, kTypeInt       , { } },
	{  479, kTypeInt       , { kTypeObject } },
	{  480, kTypeEngineType, { } },
	{  481, kTypeEngineType, { } },
	{  482, kTypeEngineType, { } },
	{  483, kTypeVoid      , { kTypeObject } },
	{  484, kTypeVoid      , { kTypeObject } },
	{  485, kTypeEngineType, { kTypeInt } },
	{  486, kTypeObject    , { kTypeObject } },
	{  487, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  488, kTypeObject    , { kTypeObject, kTypeInt } },
	{  489, kTypeVoid      , { } },
	{  490, kTypeVoid      , { } },
	{  491, kTypeInt       , { kTypeObject } },
	{  492, kTypeInt       , { kTypeObject } },
	{  493, kTypeInt       , { kTypeObject } },
	{  494, kTypeFloat     , { kTypeObject } },
	{  495, kTypeVoid      , { } },
	{  496, kTypeInt       , { kTypeObject } },
	{  497, kTypeVoid      , { kTypeString } },
	{  498, kTypeInt       , { } },
	{  499, kTypeInt       , { kTypeObject, kTypeInt } },
	{  500, kTypeInt       , { kTypeObject } },
	{  501, kTypeInt       , { kTypeObject, kTypeInt } },
	{  502, kTypeInt       , { kTypeObject } },
	{  503, kTypeString    , { kTypeObject } },
	{  504, kTypeVoid      , { kTypeInt } },
	{  505, kTypeVoid      , { } },
	{  506, kTypeVoid      , { } },
	{  507, kTypeVoid      , { } },
	{  508, kTypeVoid      , { } },
	{  509, kTypeVoid      , { kTypeString, kTypeString } },
	{  510, kTypeVoid      , { } },
	{  511, kTypeInt       , { kTypeObject } },
	{  512, kTypeVoid      , { kTypeString, kTypeString, kTypeInt } },
	{  513, kTypeInt       , { kTypeInt } },
	{  514, kTypeVoid      , { } },
	{  515, kTypeVoid      , { } },
	{  516, kTypeVoid      , { } },
	{  517, kTypeVoid      , { } },
	{  518, kTypeVoid      , { } },
	{  519, kTypeVoid      , { } },
	{  520, kTypeVoid      , { } },
	{  521, kTypeVoid      , { } },
	{  522, kTypeInt       , { kTypeObject } },
	{  523, kTypeVoid      , { } },
	{  524, kTypeVoid      , { } },
	{  525, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{  526, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt } },
	{  527, kTypeInt       , { kTypeObject } },
	{  528, kTypeInt       , { kTypeObject } },
	{  529, kTypeInt       , { kTypeObject, kTypeObject } },
	{  530, kTypeInt       , { kTypeObject } },
	{  531, kTypeInt       , { kTypeObject } },
	{  532, kTypeInt       , { kTypeObject } },
	{  533, kTypeObject    , { kTypeObject } },
	{  534, kTypeString    , { kTypeObject } },
	{  535, kTypeInt       , { kTypeObject } },
	{  536, kTypeInt       , { kTypeObject } },
	{  537, kTypeInt       , { kTypeObject } },
	{  538, kTypeInt       , { kTypeObject } },
	{  539, kTypeInt       , { kTypeObject } },
	{  540, kTypeInt       , { kTypeObject } },
	{  541, kTypeInt       , { kTypeObject } },
	{  542, kTypeObject    , { } },
	{  543, kTypeInt       , { kTypeInt, kTypeObject } },
	{  544, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  545, kTypeInt       , { kTypeObject } },
	{  546, kTypeInt       , { kTypeObject, kTypeInt } },
	{  547, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  548, kTypeObject    , { } },
	{  549, kTypeObject    , { } },
	{  550, kTypeInt       , { kTypeObject, kTypeObject } },
	{  551, kTypeInt       , { kTypeObject } },
	{  552, kTypeVoid      , { } },
	{  553, kTypeVoid      , { } },
	{  554, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeString } },
	{  555, kTypeVoid      , { kTypeObject } },
	{  556, kTypeObject    , { kTypeObject } },
	{  557, kTypeVoid      , { } },
	{  558, kTypeInt       , { kTypeObject } },
	{  559, kTypeInt       , { kTypeObject } },
	{  560, kTypeVoid      , { kTypeString } },
	{  561, kTypeString    , { } },
	{  562, kTypeObject    , { kTypeObject } },
	{  563, kTypeVoid      , { } },
	{  564, kTypeVoid      , { kTypeString } },
	{  565, kTypeInt       , { } },
	{  566, kTypeInt       , { kTypeObject } },
	{  567, kTypeVoid      , { kTypeInt } },
	{  568, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  569, kTypeVoid      , { } },
	{  570, kTypeInt       , { kTypeObject } },
	{  571, kTypeFloat     , { kTypeInt } },
	{  572, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  573, kTypeVoid      , { kTypeObject } },
	{  574, kTypeObject    , { kTypeString } },
	{  575, kTypeVoid      , { kTypeString, kTypeObject } },
	{  576, kTypeString    , { kTypeString } },
	{  577, kTypeVoid      , { kTypeString, kTypeString } },
	{  578, kTypeFloat     , { kTypeString } },
	{  579, kTypeVoid      , { kTypeString, kTypeFloat } },
	{  580, kTypeInt       , { kTypeString } },
	{  581, kTypeVoid      , { kTypeString, kTypeInt } },
	{  582, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt, kTypeFloat } },
	{  583, kTypeString    , { } },
	{  584, kTypeString    , { } },
	{  585, kTypeObject    , { kTypeString } },
	{  586, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeInt } },
	{  587, kTypeInt       , { } },
	{  588, kTypeInt       , { } },
	{  589, kTypeObject    , { } },
	{  590, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{  591, kTypeVoid      , { } },
	{  592, kTypeVoid      , { } },
	{  593, kTypeObject    , { } },
	{  594, kTypeObject    , { } },
	{  595, kTypeInt       , { } },
	{  596, kTypeInt       , { } },
	{  597, kTypeString    , { kTypeObject } },
	{  598, kTypeVoid      , { } },
	{  599, kTypeInt       , { kTypeObject } },
	{  600, kTypeInt       , { kTypeObject } },
	{  601, kTypeInt       , { kTypeObject } },
	{  602, kTypeInt       , { kTypeObject } },
	{  603, kTypeInt       , { kTypeObject } },
	{  604, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  605, kTypeVoid      , { } },
	{  606, kTypeInt       , { } },
	{  607, kTypeVoid      , { kTypeObject, kTypeString } },
	{  608, kTypeFloat     , { } },
	{  609, kTypeFloat     , { } },
	{  610, kTypeVoid      , { kTypeFloat, kTypeFloat } },
	{  611, kTypeObject    , { } },
	{  612, kTypeInt       , { } },
	{  613, kTypeObject    , { kTypeInt } },
	{  614, kTypeInt       , { } },
	{  615, kTypeObject    , { kTypeInt } },
	{  616, kTypeInt       , { kTypeObject } },
	{  617, kTypeInt       , { kTypeObject } },
	{  618, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  619, kTypeFloat     , { kTypeObject } },
	{  620, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{  621, kTypeInt       , { kTypeObject } },
	{  622, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  623, kTypeVector    , { kTypeObject } },
	{  624, kTypeInt       , { kTypeObject } },
	{  625, kTypeString    , { kTypeObject, kTypeInt } },
	{  626, kTypeString    , { kTypeObject, kTypeInt } },
	{  627, kTypeInt       , { kTypeObject, kTypeInt } },
	{  628, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  629, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  630, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  631, kTypeInt       , { kTypeObject, kTypeInt } },
	{  632, kTypeVoid      , { kTypeObject, kTypeInt, kTypeString } },
	{  633, kTypeVoid      , { kTypeObject, kTypeInt, kTypeString } },
	{  634, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  635, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  636, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  637, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  638, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  639, kTypeString    , { } },
	{  640, kTypeInt       , { kTypeObject, kTypeInt } },
	{  641, kTypeVector    , { } },
	{  642, kTypeFloat     , { } },
	{  643, kTypeFloat     , { } },
	{  644, kTypeFloat     , { } },
	{  645, kTypeFloat     , { } },
	{  646, kTypeVector    , { } },
	{  647, kTypeVoid      , { kTypeVector } },
	{  648, kTypeVoid      , { kTypeFloat } },
	{  649, kTypeVoid      , { kTypeFloat } },
	{  650, kTypeVoid      , { kTypeFloat } },
	{  651, kTypeVoid      , { kTypeFloat } },
	{  652, kTypeVoid      , { kTypeVector } },
	{  653, kTypeVector    , { } },
	{  654, kTypeVoid      , { kTypeVector } },
	{  655, kTypeVector    , { } },
	{  656, kTypeVoid      , { kTypeVector } },
	{  657, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  658, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  659, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  660, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  661, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  662, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  663, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  664, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{  665, kTypeInt       , { kTypeObject } },
	{  666, kTypeVoid      , { kTypeObject } },
	{  667, kTypeFloat     , { } },
	{  668, kTypeVoid      , { kTypeFloat } },
	{  669, kTypeVoid      , { kTypeInt, kTypeString, kTypeString } },
	{  670, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  671, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  672, kTypeInt       , { kTypeObject } },
	{  673, kTypeVoid      , { kTypeInt } },
	{  674, kTypeVoid      , { kTypeInt } },
	{  675, kTypeEngineType, { kTypeInt } },
	{  676, kTypeEngineType, { } },
	{  677, kTypeVoid      , { kTypeInt, kTypeEngineType } },
	{  678, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{  679, kTypeInt       , { kTypeObject, kTypeInt } },
	{  680, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  681, kTypeInt       , { kTypeObject, kTypeInt } },
	{  682, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  683, kTypeInt       , { kTypeObject, kTypeInt } },
	{  684, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  685, kTypeInt       , { kTypeObject, kTypeInt } },
	{  686, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  687, kTypeInt       , { kTypeObject, kTypeInt } },
	{  688, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  689, kTypeFloat     , { kTypeObject } },
	{  690, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{  691, kTypeInt       , { kTypeObject } },
	{  692, kTypeEngineType, { kTypeString } },
	{  693, kTypeVoid      , { kTypeString, kTypeEngineType } },
	{  694, kTypeInt       , { kTypeInt, kTypeObject } },
	{  695, kTypeInt       , { kTypeInt } },
	{  696, kTypeInt       , { kTypeInt } },
	{  697, kTypeInt       , { kTypeInt, kTypeString } },
	{  698, kTypeObject    , { kTypeInt, kTypeEngineType } },
	{  699, kTypeInt       , { kTypeInt } },
	{  700, kTypeVoid      , { kTypeInt } },
	{  701, kTypeInt       , { } },
	{  702, kTypeEngineType, { kTypeObject, kTypeObject, kTypeObject } },
	{  703, kTypeEngineType, { } },
	{  704, kTypeVoid      , { kTypeInt } },
	{  705, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  706, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  707, kTypeInt       , { kTypeInt } },
	{  708, kTypeInt       , { kTypeObject } },
	{  709, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  710, kTypeVoid      , { } },
	{  711, kTypeString    , { } },
	{  712, kTypeInt       , { kTypeObject } },
	{  713, kTypeInt       , { kTypeObject } },
	{  714, kTypeVoid      , { kTypeString, kTypeInt } },
	{  715, kTypeInt       , { kTypeObject } },
	{  716, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  717, kTypeVector    , { } },
	{  718, kTypeVoid      , { kTypeVector } },
	{  719, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat } },
	{  720, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat } },
	{  721, kTypeObject    , { kTypeObject } },
	{  722, kTypeInt       , { kTypeObject } },
	{  723, kTypeInt       , { kTypeObject } },
	{  724, kTypeInt       , { kTypeObject } },
	{  725, kTypeInt       , { kTypeObject } },
	{  726, kTypeInt       , { kTypeObject } },
	{  727, kTypeObject    , { kTypeObject } },
	{  728, kTypeObject    , { kTypeObject } },
	{  729, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{  730, kTypeVoid      , { } },
	{  731, kTypeVoid      , { kTypeInt, kTypeObject } },
	{  732, kTypeInt       , { kTypeObject } },
	{  733, kTypeVoid      , { kTypeString, kTypeString } },
	{  734, kTypeVoid      , { kTypeInt } },
	{  735, kTypeInt       , { kTypeEngineType } },
	{  736, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat, kTypeVector } },
	{  737, kTypeVoid      , { } },
	{  738, kTypeVoid      , { } },
	{  739, kTypeVoid      , { kTypeString, kTypeString } },
	{  740, kTypeVoid      , { kTypeString } },
	{  741, kTypeVoid      , { kTypeString } },
	{  742, kTypeVoid      , { kTypeString, kTypeVector } },
	{  743, kTypeInt       , { kTypeString, kTypeInt } },
	{  744, kTypeInt       , { kTypeString, kTypeInt, kTypeInt } },
	{  745, kTypeInt       , { kTypeString, kTypeInt } },
	{  746, kTypeEngineType, { kTypeInt } },
	{  747, kTypeEngineType, { kTypeInt } },
	{  748, kTypeEngineType, { kTypeInt } },
	{  749, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt } },
	{  750, kTypeVoid      , { } },
	{  751, kTypeVoid      , { kTypeInt } },
	{  752, kTypeVoid      , { kTypeString } },
	{  753, kTypeInt       , { kTypeString } },
	{  754, kTypeVoid      , { kTypeString, kTypeInt } },
	{  755, kTypeInt       , { kTypeObject } },
	{  756, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  757, kTypeVoid      , { kTypeString } },
	{  758, kTypeObject    , { } },
	{  759, kTypeVoid      , { kTypeInt } },
	{  760, kTypeInt       , { } },
	{  761, kTypeVoid      , { kTypeInt } },
	{  762, kTypeVoid      , { kTypeInt } },
	{  763, kTypeVoid      , { kTypeInt, kTypeString } },
	{  764, kTypeInt       , { } },
	{  765, kTypeInt       , { } },
	{  766, kTypeInt       , { kTypeObject, kTypeVector } },
	{  767, kTypeInt       , { kTypeObject } },
	{  768, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  769, kTypeInt       , { kTypeObject } },
	{  770, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  771, kTypeEngineType, { kTypeInt } },
	{  772, kTypeEngineType, { kTypeInt } },
	{  773, kTypeEngineType, { kTypeInt } },
	{  774, kTypeEngineType, { kTypeInt } },
	{  775, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt } },
	{  776, kTypeInt       , { kTypeString } },
	{  777, kTypeVoid      , { kTypeString, kTypeInt } },
	{  778, kTypeInt       , { kTypeObject, kTypeInt } },
	{  779, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  780, kTypeEngineType, { kTypeInt } },
	{  781, kTypeEngineType, { } },
	{  782, kTypeEngineType, { } },
	{  783, kTypeVoid      , { kTypeObject } },
	{  784, kTypeInt       , { kTypeObject } },
	{  785, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  786, kTypeVoid      , { kTypeString, kTypeInt } },
	{  787, kTypeInt       , { kTypeString } },
	{  788, kTypeVoid      , { kTypeString } },
	{  789, kTypeInt       , { kTypeString } },
	{  790, kTypeInt       , { kTypeString } },
	{  791, kTypeInt       , { kTypeObject } },
	{  792, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  793, kTypeFloat     , { kTypeObject } },
	{  794, kTypeFloat     , { kTypeObject } },
	{  795, kTypeFloat     , { kTypeObject } },
	{  796, kTypeFloat     , { kTypeObject } },
	{  797, kTypeFloat     , { kTypeObject } },
	{  798, kTypeFloat     , { kTypeObject } },
	{  799, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat } },
	{  800, kTypeFloat     , { } },
	{  801, kTypeVector    , { } },
	{  802, kTypeFloat     , { } },
	{  803, kTypeFloat     , { } },
	{  804, kTypeFloat     , { } },
	{  805, kTypeFloat     , { } },
	{  806, kTypeFloat     , { } },
	{  807, kTypeFloat     , { } },
	{  808, kTypeVector    , { } },
	{  809, kTypeFloat     , { kTypeInt } },
	{  810, kTypeFloat     , { kTypeInt } },
	{  811, kTypeFloat     , { kTypeInt } },
	{  812, kTypeFloat     , { kTypeInt } },
	{  813, kTypeFloat     , { kTypeInt } },
	{  814, kTypeVector    , { } },
	{  815, kTypeFloat     , { } },
	{  816, kTypeFloat     , { } },
	{  817, kTypeVector    , { } },
	{  818, kTypeVector    , { } },
	{  819, kTypeVoid      , { kTypeFloat } },
	{  820, kTypeVoid      , { kTypeVector } },
	{  821, kTypeVoid      , { kTypeFloat } },
	{  822, kTypeVoid      , { kTypeFloat } },
	{  823, kTypeVoid      , { kTypeFloat } },
	{  824, kTypeVoid      , { kTypeFloat } },
	{  825, kTypeVoid      , { kTypeFloat } },
	{  826, kTypeVoid      , { kTypeFloat } },
	{  827, kTypeVoid      , { kTypeFloat, kTypeFloat } },
	{  828, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{  829, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{  830, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{  831, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{  832, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{  833, kTypeVoid      , { kTypeVector } },
	{  834, kTypeVoid      , { kTypeFloat } },
	{  835, kTypeVoid      , { kTypeVector, kTypeFloat } },
	{  836, kTypeVoid      , { kTypeVector } },
	{  837, kTypeVoid      , { kTypeFloat, kTypeFloat } },
	{  838, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt } },
	{  839, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  840, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt } },
	{  841, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt } },
	{  842, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt } },
	{  843, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt } },
	{  844, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt } },
	{  845, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeInt } },
	{  846, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  847, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  848, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  849, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  850, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  851, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  852, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  853, kTypeVoid      , { kTypeFloat, kTypeFloat } },
	{  854, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  855, kTypeVoid      , { kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{  856, kTypeVoid      , { kTypeFloat, kTypeInt } },
	{  857, kTypeEngineType, { kTypeInt, kTypeString, kTypeInt } },
	{  858, kTypeEngineType, { kTypeString, kTypeString } },
	{  859, kTypeString    , { kTypeInt, kTypeInt, kTypeInt } },
	{  860, kTypeFloat     , { kTypeInt, kTypeInt, kTypeInt } },
	{  861, kTypeInt       , { kTypeInt, kTypeInt, kTypeInt } },
	{  862, kTypeInt       , { kTypeObject } },
	{  863, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  864, kTypeInt       , { kTypeObject } },
	{  865, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  866, kTypeInt       , { kTypeObject } },
	{  867, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  868, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeEngineType, kTypeFloat, kTypeFloat, kTypeInt } },
	{  869, kTypeVoid      , { kTypeInt } },
	{  870, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  871, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  872, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  873, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  874, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  875, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  876, kTypeVoid      , { kTypeObject, kTypeObject } },
	{  877, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  878, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  879, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  880, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  881, kTypeObject    , { kTypeObject, kTypeInt } },
	{  882, kTypeInt       , { kTypeObject } },
	{  883, kTypeInt       , { kTypeObject } },
	{  884, kTypeInt       , { kTypeObject, kTypeObject } },
	{  885, kTypeObject    , { kTypeObject, kTypeString } },
	{  886, kTypeEngineType, { kTypeInt } },
	{  887, kTypeEngineType, { kTypeInt } },
	{  888, kTypeEngineType, { kTypeInt } },
	{  889, kTypeEngineType, { kTypeInt } },
	{  890, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  891, kTypeVoid      , { kTypeInt } },
	{  892, kTypeInt       , { kTypeObject } },
	{  893, kTypeInt       , { kTypeObject } },
	{  894, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  895, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  896, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  897, kTypeVoid      , { kTypeInt } },
	{  898, kTypeInt       , { kTypeInt, kTypeInt } },
	{  899, kTypeVoid      , { kTypeInt, kTypeString, kTypeInt } },
	{  900, kTypeVoid      , { kTypeInt, kTypeInt, kTypeString, kTypeFloat, kTypeInt, kTypeString, kTypeInt, kTypeVector, kTypeInt, kTypeVector, kTypeInt, kTypeString } },
	{  901, kTypeInt       , { kTypeString, kTypeInt } },
	{  902, kTypeVoid      , { kTypeObject } },
	{  903, kTypeVoid      , { kTypeObject } },
	{  904, kTypeVoid      , { } },
	{  905, kTypeInt       , { kTypeString } },
	{  906, kTypeVoid      , { } },
	{  907, kTypeInt       , { kTypeObject } },
	{  908, kTypeInt       , { kTypeObject } },
	{  909, kTypeInt       , { kTypeObject } },
	{  910, kTypeInt       , { kTypeObject } },
	{  911, kTypeInt       , { kTypeObject } },
	{  912, kTypeFloat     , { kTypeObject } },
	{  913, kTypeInt       , { kTypeObject } },
	{  914, kTypeFloat     , { kTypeObject } },
	{  915, kTypeFloat     , { kTypeObject } },
	{  916, kTypeInt       , { kTypeObject, kTypeFloat } },
	{  917, kTypeVoid      , { kTypeInt } },
	{  918, kTypeVoid      , { kTypeString } },
	{  919, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  920, kTypeVoid      , { kTypeInt } },
	{  921, kTypeVoid      , { kTypeObject } },
	{  922, kTypeVoid      , { } },
	{  923, kTypeVoid      , { } },
	{  924, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  925, kTypeInt       , { kTypeObject, kTypeInt } },
	{  926, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{  927, kTypeEngineType, { kTypeInt } },
	{  928, kTypeInt       , { kTypeObject } },
	{  929, kTypeInt       , { kTypeObject, kTypeInt } },
	{  930, kTypeVoid      , { kTypeInt } },
	{  931, kTypeVoid      , { kTypeFloat } },
	{  932, kTypeVoid      , { kTypeObject, kTypeVector } },
	{  933, kTypeVoid      , { kTypeInt } },
	{  934, kTypeVoid      , { kTypeInt } },
	{  935, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  936, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  937, kTypeVoid      , { kTypeInt } },
	{  938, kTypeVoid      , { kTypeInt } },
	{  939, kTypeObject    , { kTypeObject, kTypeInt, kTypeInt } },
	{  940, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{  941, kTypeInt       , { kTypeObject } },
	{  942, kTypeInt       , { kTypeObject, kTypeInt } },
	{  943, kTypeVoid      , { kTypeString } },
	{  944, kTypeVoid      , { kTypeInt, kTypeInt, kTypeString, kTypeString } },
	{  945, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{  946, kTypeEngineType, { kTypeObject, kTypeInt } },
	{  947, kTypeVoid      , { kTypeInt, kTypeInt, kTypeString } },
	{  948, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeString } },
	{  949, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  950, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  951, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  952, kTypeVoid      , { kTypeInt } },
	{  953, kTypeInt       , { } },
	{  954, kTypeVoid      , { kTypeInt } },
	{  955, kTypeVoid      , { } },
	{  956, kTypeInt       , { kTypeObject, kTypeInt } },
	{  957, kTypeInt       , { kTypeObject, kTypeInt } },
	{  958, kTypeInt       , { kTypeString, kTypeFloat } },
	{  959, kTypeInt       , { kTypeObject } },
	{  960, kTypeInt       , { kTypeObject } },
	{  961, kTypeInt       , { kTypeObject } },
	{  962, kTypeInt       , { kTypeObject } },
	{  963, kTypeString    , { kTypeObject } },
	{  964, kTypeInt       , { kTypeObject } },
	{  965, kTypeInt       , { kTypeObject } },
	{  966, kTypeInt       , { kTypeObject } },
	{  967, kTypeInt       , { kTypeObject } },
	{  968, kTypeInt       , { kTypeObject } },
	{  969, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  970, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  971, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  972, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  973, kTypeVoid      , { kTypeObject, kTypeString } },
	{  974, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  975, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  976, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  977, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  978, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  979, kTypeInt       , { kTypeObject, kTypeFloat, kTypeFloat } },
	{  980, kTypeInt       , { kTypeObject, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeFloat, kTypeInt } },
	{  981, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  982, kTypeVoid      , { kTypeObject, kTypeInt, kTypeString } },
	{  983, kTypeInt       , { kTypeObject } },
	{  984, kTypeVoid      , { kTypeObject, kTypeInt } },
	{  985, kTypeInt       , { kTypeObject } },
	{  986, kTypeVoid      , { } },
	{  987, kTypeVoid      , { } },
	{  988, kTypeVoid      , { kTypeInt, kTypeInt } },
	{  989, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{  990, kTypeVoid      , { } },
	{  991, kTypeInt       , { kTypeObject, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeFloat, kTypeInt } },
	{  992, kTypeInt       , { } },
	{  993, kTypeVoid      , { kTypeInt } },
	{  994, kTypeVoid      , { kTypeInt } },
	{  995, kTypeVoid      , { kTypeInt } },
	{  996, kTypeInt       , { } },
	{  997, kTypeVoid      , { kTypeInt } },
	{  998, kTypeInt       , { } },
	{  999, kTypeVoid      , { kTypeString } },
	{ 1000, kTypeVoid      , { kTypeInt } },
	{ 1001, kTypeVoid      , { kTypeObject } },
	{ 1002, kTypeVoid      , { kTypeInt } },
	{ 1003, kTypeVoid      , { kTypeInt } },
	{ 1004, kTypeInt       , { kTypeObject } },
	{ 1005, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1006, kTypeVoid      , { kTypeObject, kTypeVector, kTypeFloat } },
	{ 1007, kTypeVoid      , { kTypeObject } },
	{ 1008, kTypeInt       , { kTypeObject } },
	{ 1009, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1010, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeFloat, kTypeInt, kTypeVector } },
	{ 1011, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeInt } },
	{ 1012, kTypeVoid      , { kTypeInt } },
	{ 1013, kTypeVoid      , { kTypeEngineType, kTypeFloat } },
	{ 1014, kTypeInt       , { kTypeObject } },
	{ 1015, kTypeInt       , { kTypeInt } },
	{ 1016, kTypeInt       , { kTypeInt } },
	{ 1017, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1018, kTypeVoid      , { } },
	{ 1019, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 1020, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1021, kTypeInt       , { kTypeObject } },
	{ 1022, kTypeInt       , { kTypeObject } },
	{ 1023, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1024, kTypeVoid      , { kTypeInt } },
	{ 1025, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 1026, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 1027, kTypeVoid      , { kTypeVector, kTypeInt } },
	{ 1028, kTypeInt       , { kTypeString, kTypeInt } },
	{ 1029, kTypeVoid      , { kTypeInt } },
	{ 1030, kTypeInt       , { kTypeObject, kTypeObject, kTypeString } },
	{ 1031, kTypeVoid      , { kTypeObject } },
	{ 1032, kTypeVoid      , { } },
	{ 1033, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{ 1034, kTypeInt       , { kTypeString, kTypeInt } },
	{ 1035, kTypeVoid      , { kTypeInt } },
	{ 1036, kTypeInt       , { kTypeInt } },
	{ 1037, kTypeInt       , { kTypeObject } },
	{ 1038, kTypeInt       , { kTypeString } },
	{ 1039, kTypeObject    , { } },
	{ 1040, kTypeInt       , { } },
	{ 1041, kTypeInt       , { } },
	{ 1042, kTypeInt       , { kTypeString } },
	{ 1043, kTypeInt       , { kTypeString } },
	{ 1044, kTypeInt       , { kTypeObject, kTypeObject, kTypeVector, kTypeInt, kTypeInt, kTypeInt } },
	{ 1045, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 1046, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1047, kTypeInt       , { } },
	{ 1048, kTypeInt       , { } },
	{ 1049, kTypeInt       , { } },
	{ 1050, kTypeInt       , { } },
	{ 1051, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1052, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1053, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 1054, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 1055, kTypeVoid      , { kTypeInt } },
	{ 1056, kTypeInt       , { kTypeObject } },
	{ 1057, kTypeInt       , { kTypeInt } },
	{ 1058, kTypeVoid      , { } }
};

/** The table defining the default values for the parameters of each engine function. */
const Functions::FunctionDefaults Functions::kFunctionDefaults[] = {
	{    0, { } },
	{    1, { &kDefaultFalse } },
	{    2, { &kDefaultInt18, &kDefaultInt9, &kDefaultFalse } },
	{    3, { &kDefaultInt18, &kDefaultInt9 } },
	{    4, { &kDefaultFalse } },
	{    5, { &kDefaultFalse } },
	{    6, { } },
	{    7, { } },
	{    8, { } },
	{    9, { } },
	{   10, { &kDefaultInt0, &kDefaultObjectSelf, &kDefaultFacingTurnshortest } },
	{   11, { } },
	{   12, { } },
	{   13, { &kDefaultInt1, &kDefaultInt1 } },
	{   14, { } },
	{   15, { } },
	{   16, { } },
	{   17, { } },
	{   18, { } },
	{   19, { } },
	{   20, { &kDefaultInt7 } },
	{   21, { &kDefaultInt0, &kDefaultIntMinus1 } },
	{   22, { &kDefaultInt0, &kDefaultFloat1_0, &kDefaultIntMinus1 } },
	{   23, { &kDefaultInt0, &kDefaultFloat40_0, &kDefaultIntMinus1 } },
	{   24, { } },
	{   25, { } },
	{   26, { } },
	{   27, { } },
	{   28, { } },
	{   29, { &kDefaultObjectInvalid } },
	{   30, { } },
	{   31, { } },
	{   32, { } },
	{   33, { } },
	{   34, { } },
	{   35, { } },
	{   36, { &kDefaultInt1, &kDefaultObjectSelf } },
	{   37, { } },
	{   38, { } },
	{   39, { } },
	{   40, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{   41, { } },
	{   42, { } },
	{   43, { } },
	{   44, { } },
	{   45, { } },
	{   46, { } },
	{   47, { } },
	{   48, { } },
	{   49, { &kDefaultObjectSelf } },
	{   50, { &kDefaultObjectSelf } },
	{   51, { } },
	{   52, { } },
	{   53, { } },
	{   54, { } },
	{   55, { } },
	{   56, { } },
	{   57, { } },
	{   58, { } },
	{   59, { } },
	{   60, { } },
	{   61, { } },
	{   62, { } },
	{   63, { } },
	{   64, { } },
	{   65, { } },
	{   66, { } },
	{   67, { } },
	{   68, { } },
	{   69, { } },
	{   70, { } },
	{   71, { } },
	{   72, { } },
	{   73, { } },
	{   74, { } },
	{   75, { } },
	{   76, { } },
	{   77, { } },
	{   78, { } },
	{   79, { } },
	{   80, { } },
	{   81, { } },
	{   82, { &kDefaultFalse } },
	{   83, { } },
	{   84, { } },
	{   85, { } },
	{   86, { } },
	{   87, { &kDefaultIntMinus1, &kDefaultStringMinus1 } },
	{   88, { } },
	{   89, { } },
	{   90, { } },
	{   91, { } },
	{   92, { } },
	{   93, { } },
	{   94, { } },
	{   95, { &kDefaultInt1 } },
	{   96, { &kDefaultInt1 } },
	{   97, { &kDefaultInt1 } },
	{   98, { &kDefaultInt1 } },
	{   99, { &kDefaultInt1 } },
	{  100, { &kDefaultInt1 } },
	{  101, { &kDefaultInt1 } },
	{  102, { &kDefaultInt1 } },
	{  103, { &kDefaultInt1 } },
	{  104, { } },
	{  105, { } },
	{  106, { } },
	{  107, { } },
	{  108, { &kDefaultFloat1_0, &kDefaultEngineAnimationBlendtypeLinear } },
	{  109, { } },
	{  110, { &kDefaultInt1 } },
	{  111, { } },
	{  112, { } },
	{  113, { &kDefaultObjectSelf } },
	{  114, { } },
	{  115, { } },
	{  116, { } },
	{  117, { } },
	{  118, { &kDefaultIntMinus1 } },
	{  119, { } },
	{  120, { } },
	{  121, { } },
	{  122, { } },
	{  123, { } },
	{  124, { } },
	{  125, { } },
	{  126, { } },
	{  127, { } },
	{  128, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{  129, { &kDefaultFalse, &kDefaultObjectTypeCreature, &kDefaultVector0 } },
	{  130, { } },
	{  131, { } },
	{  132, { } },
	{  133, { } },
	{  134, { } },
	{  135, { } },
	{  136, { } },
	{  137, { } },
	{  138, { } },
	{  139, { } },
	{  140, { } },
	{  141, { } },
	{  142, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{  143, { &kDefaultInt0, &kDefaultObjectSelf, &kDefaultFacingTurnshortest } },
	{  144, { } },
	{  145, { } },
	{  146, { } },
	{  147, { } },
	{  148, { } },
	{  149, { } },
	{  150, { } },
	{  151, { } },
	{  152, { } },
	{  153, { } },
	{  154, { &kDefaultObjectInvalid } },
	{  155, { } },
	{  156, { } },
	{  157, { } },
	{  158, { } },
	{  159, { } },
	{  160, { &kDefaultFacingTurnshortest } },
	{  161, { } },
	{  162, { &kDefaultObjectSelf } },
	{  163, { &kDefaultObjectSelf } },
	{  164, { } },
	{  165, { } },
	{  166, { } },
	{  167, { &kDefaultFloat0_0 } },
	{  168, { } },
	{  169, { } },
	{  170, { } },
	{  171, { } },
	{  172, { } },
	{  173, { } },
	{  174, { } },
	{  175, { } },
	{  176, { } },
	{  177, { } },
	{  178, { } },
	{  179, { } },
	{  180, { &kDefaultFalse } },
	{  181, { &kDefaultObjectInvalid, &kDefaultTrue, &kDefaultFloatMinus1_0, &kDefaultFloatMinus1_0, &kDefaultFalse } },
	{  182, { &kDefaultTrue, &kDefaultTrue } },
	{  183, { &kDefaultFalse } },
	{  184, { &kDefaultFloat0_0, &kDefaultInt0, &kDefaultFalse } },
	{  185, { &kDefaultFalse, &kDefaultFalse } },
	{  186, { &kDefaultFalse, &kDefaultFalse } },
	{  187, { } },
	{  188, { } },
	{  189, { } },
	{  190, { &kDefaultFalse, &kDefaultFalse } },
	{  191, { &kDefaultFalse, &kDefaultFalse } },
	{  192, { &kDefaultFalse, &kDefaultFalse } },
	{  193, { &kDefaultFalse, &kDefaultFalse } },
	{  194, { &kDefaultFalse, &kDefaultFalse } },
	{  195, { } },
	{  196, { &kDefaultTrue } },
	{  197, { } },
	{  198, { } },
	{  199, { } },
	{  200, { &kDefaultInt0 } },
	{  201, { &kDefaultFalse, &kDefaultFalse } },
	{  202, { &kDefaultInt0 } },
	{  203, { } },
	{  204, { &kDefaultStringEmpty, &kDefaultInt0, &kDefaultInt0, &kDefaultInt1, &kDefaultInt0, &kDefaultInt1 } },
	{  205, { &kDefaultInt1, &kDefaultObjectSelf } },
	{  206, { &kDefaultInt1 } },
	{  207, { } },
	{  208, { } },
	{  209, { &kDefaultInt0 } },
	{  210, { &kDefaultFalse, &kDefaultFalse } },
	{  211, { } },
	{  212, { } },
	{  213, { } },
	{  214, { &kDefaultTrue } },
	{  215, { } },
	{  216, { &kDefaultInt0 } },
	{  217, { } },
	{  218, { } },
	{  219, { } },
	{  220, { &kDefaultFloat0_0 } },
	{  221, { } },
	{  222, { } },
	{  223, { } },
	{  224, { } },
	{  225, { } },
	{  226, { } },
	{  227, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultInt0 } },
	{  228, { &kDefaultInt0 } },
	{  229, { } },
	{  230, { } },
	{  231, { } },
	{  232, { } },
	{  233, { } },
	{  234, { &kDefaultProjectilePathTypeDefault } },
	{  235, { &kDefaultObjectSelf } },
	{  236, { &kDefaultObjectSelf } },
	{  237, { &kDefaultObjectSelf } },
	{  238, { } },
	{  239, { } },
	{  240, { } },
	{  241, { &kDefaultFloat0_0 } },
	{  242, { &kDefaultFalse, &kDefaultFalse } },
	{  243, { } },
	{  244, { &kDefaultTrue } },
	{  245, { &kDefaultFalse } },
	{  246, { &kDefaultFloat0_0, &kDefaultFalse, &kDefaultFalse } },
	{  247, { } },
	{  248, { } },
	{  249, { } },
	{  250, { } },
	{  251, { } },
	{  252, { } },
	{  253, { &kDefaultFloat0_0, &kDefaultFalse, &kDefaultFalse } },
	{  254, { } },
	{  255, { &kDefaultObjectInvalid, &kDefaultStringEmpty } },
	{  256, { &kDefaultFalse, &kDefaultFalse, &kDefaultFalse, &kDefaultFloat1_0 } },
	{  257, { &kDefaultFloat0_0, &kDefaultFalse, &kDefaultObjectInvalid, &kDefaultObjectInvalid, &kDefaultInt0, &kDefaultDialogshotSideAuto } },
	{  258, { } },
	{  259, { } },
	{  260, { &kDefaultObjectSelf } },
	{  261, { } },
	{  262, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultFalse } },
	{  263, { &kDefaultObjectSelf, &kDefaultObjectTypeCreature, &kDefaultFalse } },
	{  264, { &kDefaultObjectSelf } },
	{  265, { } },
	{  266, { } },
	{  267, { } },
	{  268, { } },
	{  269, { } },
	{  270, { } },
	{  271, { } },
	{  272, { } },
	{  273, { } },
	{  274, { &kDefaultObjectInvalid } },
	{  275, { } },
	{  276, { } },
	{  277, { } },
	{  278, { } },
	{  279, { } },
	{  280, { } },
	{  281, { } },
	{  282, { } },
	{  283, { } },
	{  284, { } },
	{  285, { } },
	{  286, { } },
	{  287, { } },
	{  288, { } },
	{  289, { &kDefaultObjectSelf } },
	{  290, { } },
	{  291, { &kDefaultObjectInvalid } },
	{  292, { } },
	{  293, { } },
	{  294, { } },
	{  295, { } },
	{  296, { } },
	{  297, { } },
	{  298, { } },
	{  299, { } },
	{  300, { &kDefaultFloat1_0, &kDefaultFloat0_0 } },
	{  301, { } },
	{  302, { } },
	{  303, { } },
	{  304, { &kDefaultObjectSelf } },
	{  305, { } },
	{  306, { &kDefaultFalse, &kDefaultInt0 } },
	{  307, { &kDefaultInt0 } },
	{  308, { &kDefaultInt0 } },
	{  309, { &kDefaultFalse, &kDefaultInt0 } },
	{  310, { } },
	{  311, { } },
	{  312, { } },
	{  313, { &kDefaultInt1 } },
	{  314, { } },
	{  315, { } },
	{  316, { } },
	{  317, { } },
	{  318, { &kDefaultObjectSelf } },
	{  319, { } },
	{  320, { } },
	{  321, { } },
	{  322, { &kDefaultTrue } },
	{  323, { &kDefaultFalse } },
	{  324, { } },
	{  325, { } },
	{  326, { } },
	{  327, { &kDefaultObjectInvalid, &kDefaultObjectInvalid, &kDefaultFalse, &kDefaultInt0 } },
	{  328, { } },
	{  329, { } },
	{  330, { } },
	{  331, { } },
	{  332, { } },
	{  333, { } },
	{  334, { } },
	{  335, { } },
	{  336, { } },
	{  337, { } },
	{  338, { } },
	{  339, { &kDefaultObjectSelf } },
	{  340, { } },
	{  341, { &kDefaultObjectSelf } },
	{  342, { &kDefaultFloat0_0, &kDefaultFalse, &kDefaultObjectInvalid, &kDefaultObjectInvalid, &kDefaultInt0, &kDefaultDialogshotSideAuto } },
	{  343, { &kDefaultObjectSelf } },
	{  344, { } },
	{  345, { } },
	{  346, { } },
	{  347, { } },
	{  348, { } },
	{  349, { } },
	{  350, { } },
	{  351, { } },
	{  352, { } },
	{  353, { } },
	{  354, { } },
	{  355, { } },
	{  356, { } },
	{  357, { } },
	{  358, { } },
	{  359, { } },
	{  360, { &kDefaultInt0, &kDefaultFloat40_0, &kDefaultIntMinus1 } },
	{  361, { } },
	{  362, { } },
	{  363, { } },
	{  364, { &kDefaultObjectSelf } },
	{  365, { &kDefaultObjectSelf } },
	{  266, { &kDefaultObjectSelf } },
	{  367, { &kDefaultTrue, &kDefaultFalse, &kDefaultFalse } },
	{  368, { &kDefaultTrue, &kDefaultFalse } },
	{  369, { } },
	{  370, { } },
	{  371, { } },
	{  372, { } },
	{  373, { } },
	{  374, { } },
	{  375, { } },
	{  376, { } },
	{  377, { } },
	{  378, { &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{  379, { } },
	{  380, { } },
	{  381, { } },
	{  382, { &kDefaultInt0, &kDefaultFloat30_0, &kDefaultIntMinus1 } },
	{  383, { &kDefaultInt0, &kDefaultFloat1_0, &kDefaultFloat30_0, &kDefaultIntMinus1 } },
	{  384, { } },
	{  385, { &kDefaultInt1 } },
	{  386, { } },
	{  387, { } },
	{  388, { } },
	{  389, { &kDefaultObjectSelf } },
	{  390, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{  391, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{  392, { &kDefaultObjectSelf, &kDefaultFalse, &kDefaultFloat180_0 } },
	{  393, { } },
	{  394, { } },
	{  395, { } },
	{  396, { } },
	{  397, { } },
	{  398, { } },
	{  399, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{  400, { &kDefaultObjectInvalid } },
	{  401, { } },
	{  402, { } },
	{  403, { } },
	{  404, { } },
	{  405, { } },
	{  406, { } },
	{  407, { } },
	{  408, { } },
	{  409, { &kDefaultObjectSelf } },
	{  410, { } },
	{  411, { } },
	{  412, { } },
	{  413, { } },
	{  414, { } },
	{  415, { } },
	{  416, { } },
	{  417, { &kDefaultStringEmpty, &kDefaultTrue } },
	{  418, { &kDefaultObjectSelf } },
	{  419, { } },
	{  420, { } },
	{  421, { } },
	{  422, { &kDefaultObjectInvalid, &kDefaultFalse } },
	{  423, { } },
	{  424, { &kDefaultObjectInvalid } },
	{  425, { &kDefaultTrue } },
	{  426, { &kDefaultObjectInvalid } },
	{  427, { &kDefaultTrue } },
	{  428, { } },
	{  429, { } },
	{  430, { } },
	{  431, { } },
	{  432, { } },
	{  433, { } },
	{  434, { } },
	{  435, { } },
	{  436, { } },
	{  437, { } },
	{  438, { } },
	{  439, { } },
	{  440, { } },
	{  441, { } },
	{  442, { } },
	{  443, { } },
	{  444, { } },
	{  445, { } },
	{  446, { } },
	{  447, { &kDefaultAttackBonusMisc } },
	{  448, { &kDefaultDamagetypeHealth } },
	{  449, { } },
	{  450, { &kDefaultACDodgeBonus, &kDefaultACVsDamageTypeAll } },
	{  451, { } },
	{  452, { } },
	{  453, { } },
	{  454, { } },
	{  455, { &kDefaultObjectSelf } },
	{  456, { } },
	{  457, { } },
	{  458, { } },
	{  459, { } },
	{  460, { } },
	{  461, { } },
	{  462, { } },
	{  463, { } },
	{  464, { } },
	{  465, { } },
	{  466, { } },
	{  467, { } },
	{  468, { } },
	{  469, { } },
	{  470, { } },
	{  471, { } },
	{  472, { &kDefaultInt0, &kDefaultInt0 } },
	{  473, { } },
	{  474, { } },
	{  475, { } },
	{  476, { } },
	{  477, { } },
	{  478, { } },
	{  479, { } },
	{  480, { } },
	{  481, { } },
	{  482, { } },
	{  483, { } },
	{  484, { } },
	{  485, { } },
	{  486, { &kDefaultObjectSelf } },
	{  487, { } },
	{  488, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  489, { } },
	{  490, { } },
	{  491, { } },
	{  492, { } },
	{  493, { } },
	{  494, { } },
	{  495, { } },
	{  496, { } },
	{  497, { } },
	{  498, { } },
	{  499, { } },
	{  500, { } },
	{  501, { } },
	{  502, { } },
	{  503, { } },
	{  504, { } },
	{  505, { } },
	{  506, { } },
	{  507, { } },
	{  508, { } },
	{  509, { } },
	{  510, { } },
	{  511, { } },
	{  512, { } },
	{  513, { } },
	{  514, { } },
	{  515, { } },
	{  516, { } },
	{  517, { } },
	{  518, { } },
	{  519, { } },
	{  520, { } },
	{  521, { } },
	{  522, { &kDefaultObjectSelf } },
	{  523, { } },
	{  524, { } },
	{  525, { &kDefaultTrue } },
	{  526, { &kDefaultTrue } },
	{  527, { } },
	{  528, { } },
	{  529, { } },
	{  530, { } },
	{  531, { } },
	{  532, { } },
	{  533, { } },
	{  534, { } },
	{  535, { } },
	{  536, { } },
	{  537, { } },
	{  538, { } },
	{  539, { } },
	{  540, { } },
	{  541, { } },
	{  542, { } },
	{  543, { &kDefaultObjectSelf } },
	{  544, { &kDefaultObjectSelf, &kDefaultTrue } },
	{  545, { &kDefaultObjectSelf } },
	{  546, { } },
	{  547, { } },
	{  548, { } },
	{  549, { } },
	{  550, { } },
	{  551, { } },
	{  552, { } },
	{  553, { } },
	{  554, { &kDefaultTrue, &kDefaultTrue, &kDefaultInt0, &kDefaultStringEmpty } },
	{  555, { } },
	{  556, { &kDefaultObjectSelf } },
	{  557, { } },
	{  558, { } },
	{  559, { } },
	{  560, { } },
	{  561, { } },
	{  562, { } },
	{  563, { } },
	{  564, { &kDefaultStringEmpty } },
	{  565, { } },
	{  566, { } },
	{  567, { } },
	{  568, { } },
	{  569, { } },
	{  570, { } },
	{  571, { } },
	{  572, { } },
	{  573, { } },
	{  574, { } },
	{  575, { } },
	{  576, { } },
	{  577, { } },
	{  578, { } },
	{  579, { } },
	{  580, { } },
	{  581, { } },
	{  582, { } },
	{  583, { } },
	{  584, { } },
	{  585, { } },
	{  586, { &kDefaultInt1, &kDefaultInt0, &kDefaultInt0 } },
	{  587, { } },
	{  588, { } },
	{  589, { } },
	{  590, { &kDefaultInt0 } },
	{  591, { } },
	{  592, { } },
	{  593, { } },
	{  594, { } },
	{  595, { } },
	{  596, { } },
	{  597, { &kDefaultObjectSelf } },
	{  598, { } },
	{  599, { &kDefaultObjectSelf } },
	{  600, { &kDefaultObjectSelf } },
	{  601, { &kDefaultObjectSelf } },
	{  602, { &kDefaultObjectSelf } },
	{  603, { &kDefaultObjectSelf } },
	{  604, { } },
	{  605, { } },
	{  606, { } },
	{  607, { } },
	{  608, { } },
	{  609, { } },
	{  610, { } },
	{  611, { } },
	{  612, { } },
	{  613, { } },
	{  614, { } },
	{  615, { } },
	{  616, { } },
	{  617, { } },
	{  618, { } },
	{  619, { } },
	{  620, { } },
	{  621, { } },
	{  622, { } },
	{  623, { } },
	{  624, { } },
	{  625, { } },
	{  626, { } },
	{  627, { } },
	{  628, { } },
	{  629, { } },
	{  630, { } },
	{  631, { } },
	{  632, { } },
	{  633, { } },
	{  634, { } },
	{  635, { } },
	{  636, { } },
	{  637, { } },
	{  638, { } },
	{  639, { } },
	{  640, { } },
	{  641, { } },
	{  642, { } },
	{  643, { } },
	{  644, { } },
	{  645, { } },
	{  646, { } },
	{  647, { } },
	{  648, { } },
	{  649, { } },
	{  650, { } },
	{  651, { } },
	{  652, { } },
	{  653, { } },
	{  654, { } },
	{  655, { } },
	{  656, { } },
	{  657, { } },
	{  658, { } },
	{  659, { } },
	{  660, { } },
	{  661, { } },
	{  662, { } },
	{  663, { } },
	{  664, { } },
	{  665, { } },
	{  666, { } },
	{  667, { } },
	{  668, { } },
	{  669, { &kDefaultStringWorldEntry } },
	{  670, { } },
	{  671, { } },
	{  672, { } },
	{  673, { } },
	{  674, { } },
	{  675, { } },
	{  676, { } },
	{  677, { } },
	{  678, { &kDefaultTrue, &kDefaultFalse } },
	{  679, { } },
	{  680, { } },
	{  681, { } },
	{  682, { } },
	{  683, { } },
	{  684, { } },
	{  685, { } },
	{  686, { } },
	{  687, { } },
	{  688, { } },
	{  689, { } },
	{  690, { } },
	{  691, { } },
	{  692, { } },
	{  693, { } },
	{  694, { } },
	{  695, { } },
	{  696, { } },
	{  697, { } },
	{  698, { } },
	{  699, { } },
	{  700, { } },
	{  701, { } },
	{  702, { &kDefaultObjectInvalid, &kDefaultObjectInvalid } },
	{  703, { } },
	{  704, { } },
	{  705, { } },
	{  706, { } },
	{  707, { } },
	{  708, { } },
	{  709, { } },
	{  710, { } },
	{  711, { } },
	{  712, { } },
	{  713, { } },
	{  714, { } },
	{  715, { } },
	{  716, { } },
	{  717, { } },
	{  718, { } },
	{  719, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{  720, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{  721, { &kDefaultObjectSelf } },
	{  722, { &kDefaultObjectSelf } },
	{  723, { &kDefaultObjectSelf } },
	{  724, { &kDefaultObjectSelf } },
	{  725, { &kDefaultObjectSelf } },
	{  726, { &kDefaultObjectSelf } },
	{  727, { &kDefaultObjectSelf } },
	{  728, { &kDefaultObjectSelf } },
	{  729, { } },
	{  730, { } },
	{  731, { &kDefaultObjectSelf } },
	{  732, { &kDefaultObjectSelf } },
	{  733, { &kDefaultStringEmpty } },
	{  734, { } },
	{  735, { } },
	{  736, { } },
	{  737, { } },
	{  738, { } },
	{  739, { &kDefaultStringEmpty } },
	{  740, { } },
	{  741, { } },
	{  742, { } },
	{  743, { } },
	{  744, { } },
	{  745, { } },
	{  746, { } },
	{  747, { } },
	{  748, { } },
	{  749, { } },
	{  750, { } },
	{  751, { } },
	{  752, { } },
	{  753, { } },
	{  754, { } },
	{  755, { } },
	{  756, { &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultIntMinus1 } },
	{  757, { } },
	{  758, { } },
	{  759, { } },
	{  760, { } },
	{  761, { } },
	{  762, { } },
	{  763, { } },
	{  764, { } },
	{  765, { } },
	{  766, { } },
	{  767, { } },
	{  768, { } },
	{  769, { } },
	{  770, { } },
	{  771, { } },
	{  772, { } },
	{  773, { } },
	{  774, { } },
	{  775, { } },
	{  776, { } },
	{  777, { } },
	{  778, { } },
	{  779, { } },
	{  780, { } },
	{  781, { } },
	{  782, { } },
	{  783, { } },
	{  784, { } },
	{  785, { } },
	{  786, { } },
	{  787, { } },
	{  788, { } },
	{  789, { } },
	{  790, { } },
	{  791, { } },
	{  792, { } },
	{  793, { } },
	{  794, { } },
	{  795, { } },
	{  796, { } },
	{  797, { } },
	{  798, { } },
	{  799, { } },
	{  800, { } },
	{  801, { } },
	{  802, { } },
	{  803, { } },
	{  804, { } },
	{  805, { } },
	{  806, { } },
	{  807, { } },
	{  808, { } },
	{  809, { } },
	{  810, { } },
	{  811, { } },
	{  812, { } },
	{  813, { } },
	{  814, { } },
	{  815, { } },
	{  816, { } },
	{  817, { } },
	{  818, { } },
	{  819, { } },
	{  820, { } },
	{  821, { } },
	{  822, { } },
	{  823, { } },
	{  824, { } },
	{  825, { } },
	{  826, { } },
	{  827, { } },
	{  828, { } },
	{  829, { } },
	{  830, { } },
	{  831, { } },
	{  832, { } },
	{  833, { } },
	{  834, { } },
	{  835, { } },
	{  836, { } },
	{  837, { } },
	{  838, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  839, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  840, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  841, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  842, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  843, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  844, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  845, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  846, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  847, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  848, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  849, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  850, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  851, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  852, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  853, { } },
	{  854, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  855, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  856, { &kDefaultEngineAnimationBlendtypeLinear } },
	{  857, { &kDefaultInt0 } },
	{  858, { } },
	{  859, { } },
	{  860, { } },
	{  861, { } },
	{  862, { } },
	{  863, { } },
	{  864, { } },
	{  865, { } },
	{  866, { } },
	{  867, { } },
	{  868, { &kDefaultInt0 } },
	{  869, { } },
	{  870, { &kDefaultInt1, &kDefaultStylemapNone, &kDefaultTrue } },
	{  871, { } },
	{  872, { &kDefaultCombatAreaActive } },
	{  873, { &kDefaultTrue } },
	{  874, { &kDefaultTrue } },
	{  875, { } },
	{  876, { } },
	{  877, { } },
	{  878, { } },
	{  879, { &kDefaultTrue } },
	{  880, { &kDefaultTrue } },
	{  881, { &kDefaultInt0 } },
	{  882, { } },
	{  883, { } },
	{  884, { } },
	{  885, { } },
	{  886, { } },
	{  887, { } },
	{  888, { } },
	{  889, { } },
	{  890, { } },
	{  891, { } },
	{  892, { } },
	{  893, { } },
	{  894, { } },
	{  895, { } },
	{  896, { } },
	{  897, { } },
	{  898, { } },
	{  899, { } },
	{  900, { &kDefaultFalse, &kDefaultVector0, &kDefaultFalse, &kDefaultVector0, &kDefaultFalse, &kDefaultStringEmpty } },
	{  901, { } },
	{  902, { } },
	{  903, { } },
	{  904, { } },
	{  905, { } },
	{  906, { } },
	{  907, { &kDefaultObjectSelf } },
	{  908, { &kDefaultObjectSelf } },
	{  909, { &kDefaultObjectSelf } },
	{  910, { &kDefaultObjectSelf } },
	{  911, { &kDefaultObjectSelf } },
	{  912, { &kDefaultObjectSelf } },
	{  913, { &kDefaultObjectSelf } },
	{  914, { &kDefaultObjectSelf } },
	{  915, { &kDefaultObjectSelf } },
	{  916, { /*TODO &kDefaultObjectSelf missing camera angle default */ } },
	{  917, { } },
	{  918, { } },
	{  919, { } },
	{  920, { } },
	{  921, { } },
	{  922, { } },
	{  923, { } },
	{  924, { } },
	{  925, { } },
	{  926, { } },
	{  927, { &kDefaultInt0 } },
	{  928, { } },
	{  929, { } },
	{  930, { } },
	{  931, { } },
	{  932, { } },
	{  933, { } },
	{  934, { } },
	{  935, { } },
	{  936, { } },
	{  937, { } },
	{  938, { } },
	{  939, { &kDefaultInt1 } },
	{  940, { &kDefaultInt1, &kDefaultInt0 } },
	{  941, { } },
	{  942, { } },
	{  943, { &kDefaultStringEmpty } },
	{  944, { } },
	{  945, { } },
	{  946, { &kDefaultEvadeForward } },
	{  947, { &kDefaultStringEmpty } },
	{  948, { &kDefaultInt0, &kDefaultInt0, &kDefaultStringEmpty } },
	{  949, { } },
	{  950, { } },
	{  951, { } },
	{  952, { } },
	{  953, { } },
	{  954, { } },
	{  955, { } },
	{  956, { } },
	{  957, { } },
	{  958, { } },
	{  959, { } },
	{  960, { } },
	{  961, { } },
	{  962, { } },
	{  963, { } },
	{  964, { } },
	{  965, { } },
	{  966, { } },
	{  967, { } },
	{  968, { } },
	{  969, { } },
	{  970, { } },
	{  971, { } },
	{  972, { } },
	{  973, { } },
	{  974, { } },
	{  975, { } },
	{  976, { } },
	{  977, { } },
	{  978, { } },
	{  979, { } },
	{  980, { &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultInt0, &kDefaultFloat100_0, &kDefaultInt1 } },
	{  981, { } },
	{  982, { } },
	{  983, { } },
	{  984, { } },
	{  985, { } },
	{  986, { } },
	{  987, { } },
	{  988, { } },
	{  989, { } },
	{  990, { } },
	{  991, { &kDefaultIntMinus1, &kDefaultIntMinus1, &kDefaultInt0, &kDefaultFloat100_0, &kDefaultInt0 } },
	{  992, { } },
	{  993, { } },
	{  994, { } },
	{  995, { } },
	{  996, { } },
	{  997, { } },
	{  998, { } },
	{  999, { } },
	{ 1000, { } },
	{ 1001, { } },
	{ 1002, { } },
	{ 1003, { } },
	{ 1004, { } },
	{ 1005, { } },
	{ 1006, { } },
	{ 1007, { } },
	{ 1008, { } },
	{ 1009, { } },
	{ 1010, { &kDefaultMovespeedDefault, &kDefaultVector0 } },
	{ 1011, { &kDefaultTrue } },
	{ 1012, { } },
	{ 1013, { } },
	{ 1014, { } },
	{ 1015, { } },
	{ 1016, { } },
	{ 1017, { } },
	{ 1018, { } },
	{ 1019, { } },
	{ 1020, { } },
	{ 1021, { } },
	{ 1022, { } },
	{ 1023, { } },
	{ 1024, { } },
	{ 1025, { } },
	{ 1026, { } },
	{ 1027, { } },
	{ 1028, { } },
	{ 1029, { } },
	{ 1030, { } },
	{ 1031, { } },
	{ 1032, { } },
	{ 1033, { &kDefaultFloat1_0, &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultInt0 } },
	{ 1034, { } },
	{ 1035, { } },
	{ 1036, { } },
	{ 1037, { } },
	{ 1038, { } },
	{ 1039, { } },
	{ 1040, { } },
	{ 1041, { } },
	{ 1042, { } },
	{ 1043, { } },
	{ 1044, { } },
	{ 1045, { } },
	{ 1046, { } },
	{ 1047, { } },
	{ 1048, { } },
	{ 1049, { } },
	{ 1050, { } },
	{ 1051, { } },
	{ 1052, { } },
	{ 1053, { } },
	{ 1054, { } },
	{ 1055, { } },
	{ 1056, { &kDefaultObjectSelf } },
	{ 1057, { } },
	{ 1058, { } }
};

} // End of namespace Jade

} // End of namespace Engines

#endif // ENGINES_JADE_SCRIPT_FUNCTION_TABLES_H
