/* xoreos - A reimplementation of BioWare's Aurora engine
 *
 * xoreos is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Tables defining the engine functions in Dragon Age: Origins.
 */

/* The functions are defined by three tables:
 * - kFunctionPointers
 * - kFunctionSignatures
 * - kFunctionDefaults
 *
 * kFunctionPointers provides the ID and name of the engine function, and a
 * pointer to Functions method doing the actual work. If the function pointer
 * is 0, a default unimplementedFunction method is used that just prints
 * the name of the function when it's called.
 *
 * kFunctionSignatures defines the signature of the function, i.e. the types
 * of the return value and its parameters.
 *
 * kFunctionDefaults pins default values to parameters. I.e. if a function
 * Foobar(int mode, int doEverything=FALSE) is called with only one parameters,
 * the default value FALSE is applied to the doEverything parameters. The
 * kFunctionDefaults entry for the function then contains a reference to a
 * false value. Because of ambiguities otherwise, the default values have to be
 * "aligned" to the right; only the last-most n parameters are allowed to have
 * default values, with no gaps.
 *
 * Please note that all three tables have to be of the same length, and that
 * the order of the functions have to be the same in all three tables.
 */

#ifndef ENGINES_DRAGONAGE_SCRIPT_FUNCTION_TABLES_H
#define ENGINES_DRAGONAGE_SCRIPT_FUNCTION_TABLES_H

#include "src/engines/dragonage/types.h"
#include "src/engines/dragonage/object.h"

namespace Engines {

namespace DragonAge {

using Aurora::NWScript::kTypeVoid;
using Aurora::NWScript::kTypeInt;
using Aurora::NWScript::kTypeFloat;
using Aurora::NWScript::kTypeString;
using Aurora::NWScript::kTypeObject;
using Aurora::NWScript::kTypeVector;
using Aurora::NWScript::kTypeEngineType;
using Aurora::NWScript::kTypeArray;
using Aurora::NWScript::kTypeScriptState;
using Aurora::NWScript::kTypeAny;

/* Default values as used by the function parameters, if no explicit value
 * is given at the time of the calling.
 */

static DragonAge::Object kDefaultValueObjectInvalid(kObjectTypeInvalid);
static DragonAge::Object kDefaultValueObjectSelf   (kObjectTypeSelf);

static const Aurora::NWScript::Variable kDefaultIntMinus1    ((int32_t) -         1);
static const Aurora::NWScript::Variable kDefaultInt0         ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultInt1         ((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultInt9         ((int32_t)           9);
static const Aurora::NWScript::Variable kDefaultInt10        ((int32_t)          10);
static const Aurora::NWScript::Variable kDefaultInt18        ((int32_t)          18);
static const Aurora::NWScript::Variable kDefaultInt205       ((int32_t)         205);
static const Aurora::NWScript::Variable kDefaultInt206       ((int32_t)         206);
static const Aurora::NWScript::Variable kDefaultInt16777215  ((int32_t)    16777215);
static const Aurora::NWScript::Variable kDefaultInt4294967295((int32_t)  4294967295U);

static const Aurora::NWScript::Variable kDefaultFloatMinus1_0(-  1.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_0     (   0.0f);
static const Aurora::NWScript::Variable kDefaultFloat0_5     (   0.5f);
static const Aurora::NWScript::Variable kDefaultFloat10_0    (  10.0f);

static const Aurora::NWScript::Variable kDefaultVector0(0.0f, 0.0f, 0.0f);

static const Aurora::NWScript::Variable kDefaultStringEmpty    (Common::UString(""));
static const Aurora::NWScript::Variable kDefaultStringLogWindow(Common::UString("LogWindow"));
static const Aurora::NWScript::Variable kDefaultStringxE       (Common::UString("xE"));

static const Aurora::NWScript::Variable kDefaultObjectInvalid(&kDefaultValueObjectInvalid);
static const Aurora::NWScript::Variable kDefaultObjectSelf   (&kDefaultValueObjectSelf);

// TODO: Add the relevant enums to types.h, and use these values.

static const Aurora::NWScript::Variable kDefaultAbilityInvalid           ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultCombatResultInvalid      ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultEffectDurationTypeInvalid((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultEffectTypeInvalid        ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultFalse                    ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultFloatyMessage            ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultGenderMale               ((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultGetItemsOptionAll        ((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultInvalidWeaponSet         ((int32_t)  4294967295U);
static const Aurora::NWScript::Variable kDefaultInventorySlotInvalid     ((int32_t)         255);
static const Aurora::NWScript::Variable kDefaultLogChannelGeneral        ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultObjectTypeAll            ((int32_t)  4294967295U);
static const Aurora::NWScript::Variable kDefaultObjectTypeCreature       ((int32_t)          32);
static const Aurora::NWScript::Variable kDefaultProjectileTargetInvalid  ((int32_t)           0);
static const Aurora::NWScript::Variable kDefaultPropertyValueBase        ((int32_t)           2);
static const Aurora::NWScript::Variable kDefaultPropertyValueTotal       ((int32_t)           1);
static const Aurora::NWScript::Variable kDefaultResistanceCheckInvalid   ((int32_t) -         1);
static const Aurora::NWScript::Variable kDefaultTrue                     ((int32_t)           1);


/** The table defining the name and function pointer of each engine function. */
const Functions::FunctionPointer Functions::kFunctionPointers[] = {
	{   0, "PrintInteger"                          , &Functions::printInteger                           },
	{   1, "PrintFloat"                            , &Functions::printFloat                             },
	{   2, "PrintString"                           , &Functions::printString                            },
	{   3, "PrintObject"                           , &Functions::printObject                            },
	{   4, "PrintVector"                           , &Functions::printVector                            },
	{   5, "PrintToLog"                            , &Functions::printToLog                             },
	{   6, "InvalidFunction6"                      , 0                                                  },
	{   7, "InvalidFunction7"                      , 0                                                  },
	{   8, "Deprecated_ExecuteScript"              , 0                                                  },
	{   9, "GetM2DAString"                         , 0                                                  },
	{  10, "InvalidFunction10"                     , 0                                                  },
	{  11, "GetM2DARows"                           , 0                                                  },
	{  12, "GetM2DAColumns"                        , 0                                                  },
	{  13, "GetM2DAResource"                       , 0                                                  },
	{  14, "GetTlkTableString"                     , &Functions::getTlkTableString                      },
	{  15, "GetGameDifficulty"                     , 0                                                  },
	{  16, "DebugSpawnScriptDebugger"              , 0                                                  },
	{  17, "GetFacing"                             , 0                                                  },
	{  18, "SetFacing"                             , 0                                                  },
	{  19, "Effect"                                , 0                                                  },
	{  20, "GetPreviousCommand"                    , 0                                                  },
	{  21, "GetPosition"                           , &Functions::getPosition                            },
	{  22, "SetPosition"                           , 0                                                  },
	{  23, "SetFacingPosition"                     , 0                                                  },
	{  24, "SetFacingObject"                       , 0                                                  },
	{  25, "Random"                                , &Functions::random                                 },
	{  26, "abs"                                   , &Functions::abs                                    },
	{  27, "fabs"                                  , &Functions::fabs                                   },
	{  28, "cos"                                   , &Functions::cos                                    },
	{  29, "sin"                                   , &Functions::sin                                    },
	{  30, "tan"                                   , &Functions::tan                                    },
	{  31, "acos"                                  , &Functions::acos                                   },
	{  32, "asin"                                  , &Functions::asin                                   },
	{  33, "atan"                                  , &Functions::atan                                   },
	{  34, "log"                                   , &Functions::log                                    },
	{  35, "pow"                                   , &Functions::pow                                    },
	{  36, "sqrt"                                  , &Functions::sqrt                                   },
	{  37, "GetDistanceBetween"                    , &Functions::getDistanceBetween                     },
	{  38, "GetDistanceBetweenLocations"           , 0                                                  },
	{  39, "IsInTrigger"                           , 0                                                  },
	{  40, "IsVectorEmpty"                         , &Functions::isVectorEmpty                          },
	{  41, "Vector"                                , &Functions::vector                                 },
	{  42, "GetVectorMagnitude"                    , &Functions::getVectorMagnitude                     },
	{  43, "GetVectorNormalize"                    , &Functions::getVectorNormalize                     },
	{  44, "AngleToVector"                         , 0                                                  },
	{  45, "VectorToAngle"                         , 0                                                  },
	{  46, "CheckLineOfSightObject"                , 0                                                  },
	{  47, "CheckLineOfSightVector"                , 0                                                  },
	{  48, "IntToFloat"                            , &Functions::intToFloat                             },
	{  49, "FloatToInt"                            , &Functions::floatToInt                             },
	{  50, "IntToString"                           , &Functions::intToString                            },
	{  51, "StringToInt"                           , &Functions::stringToInt                            },
	{  52, "FloatToString"                         , &Functions::floatToString                          },
	{  53, "StringToFloat"                         , &Functions::stringToFloat                          },
	{  54, "ObjectToString"                        , &Functions::objectToString                         },
	{  55, "VectorToString"                        , &Functions::vectorToString                         },
	{  56, "StringToVector"                        , &Functions::stringToVector                         },
	{  57, "IntToHexString"                        , &Functions::intToHexString                         },
	{  58, "HexStringToInt"                        , 0                                                  },
	{  59, "CharToInt"                             , &Functions::charToInt                              },
	{  60, "IntToChar"                             , &Functions::intToChar                              },
	{  61, "FeetToMeters"                          , 0                                                  },
	{  62, "YardsToMeters"                         , 0                                                  },
	{  63, "GetLocalInt"                           , &Functions::getLocalInt                            },
	{  64, "SetLocalInt"                           , &Functions::setLocalInt                            },
	{  65, "InvalidFunction65"                     , 0                                                  },
	{  66, "GetLocalFloat"                         , &Functions::getLocalFloat                          },
	{  67, "SetLocalFloat"                         , &Functions::setLocalFloat                          },
	{  68, "InvalidFunction68"                     , 0                                                  },
	{  69, "GetLocalString"                        , &Functions::getLocalString                         },
	{  70, "SetLocalString"                        , &Functions::setLocalString                         },
	{  71, "InvalidFunction71"                     , 0                                                  },
	{  72, "GetLocalObject"                        , &Functions::getLocalObject                         },
	{  73, "SetLocalObject"                        , &Functions::setLocalObject                         },
	{  74, "InvalidFunction74"                     , 0                                                  },
	{  75, "GetLocalLocation"                      , &Functions::getLocalLocation                       },
	{  76, "SetLocalLocation"                      , &Functions::setLocalLocation                       },
	{  77, "InvalidFunction77"                     , 0                                                  },
	{  78, "GetLocalPlayer"                        , &Functions::getLocalPlayer                         },
	{  79, "SetLocalPlayer"                        , &Functions::setLocalPlayer                         },
	{  80, "InvalidFunction80"                     , 0                                                  },
	{  81, "DEBUG_PrintToScreen"                   , &Functions::DEBUG_printToScreen                    },
	{  82, "PrintToLogWindow"                      , &Functions::printToLogWindow                       },
	{  83, "GetParty"                              , 0                                                  },
	{  84, "GetLocalEvent"                         , &Functions::getLocalEvent                          },
	{  85, "SetLocalEvent"                         , &Functions::setLocalEvent                          },
	{  86, "InvalidFunction86"                     , 0                                                  },
	{  87, "GetLocalCommand"                       , &Functions::getLocalCommand                        },
	{  88, "SetLocalCommand"                       , &Functions::setLocalCommand                        },
	{  89, "InvalidFunction89"                     , 0                                                  },
	{  90, "GetLocalEffect"                        , &Functions::getLocalEffect                         },
	{  91, "SetLocalEffect"                        , &Functions::setLocalEffect                         },
	{  92, "InvalidFunction92"                     , 0                                                  },
	{  93, "GetLocalItemProperty"                  , &Functions::getLocalItemProperty                   },
	{  94, "SetLocalItemProperty"                  , &Functions::setLocalItemProperty                   },
	{  95, "InvalidFunction95"                     , 0                                                  },
	{  96, "GetStringLength"                       , &Functions::getStringLength                        },
	{  97, "StringUpperCase"                       , &Functions::stringUpperCase                        },
	{  98, "StringLowerCase"                       , &Functions::stringLowerCase                        },
	{  99, "StringRight"                           , &Functions::stringRight                            },
	{ 100, "StringLeft"                            , &Functions::stringLeft                             },
	{ 101, "InsertString"                          , &Functions::insertString                           },
	{ 102, "SubString"                             , &Functions::subString                              },
	{ 103, "FindSubString"                         , &Functions::findSubString                          },
	{ 104, "IsStringEmpty"                         , &Functions::isStringEmpty                          },
	{ 105, "SignalEvent"                           , 0                                                  },
	{ 106, "Event"                                 , 0                                                  },
	{ 107, "IsEventValid"                          , &Functions::isEventValid                           },
	{ 108, "GetCurrentEvent"                       , &Functions::getCurrentEvent                        },
	{ 109, "GetEventCreator"                       , &Functions::getEventCreator                        },
	{ 110, "SetEventCreator"                       , &Functions::setEventCreator                        },
	{ 111, "GetEventType"                          , &Functions::getEventType                           },
	{ 112, "SetEventType"                          , &Functions::setEventType                           },
	{ 113, "GetEventInteger"                       , 0                                                  },
	{ 114, "SetEventInteger"                       , 0                                                  },
	{ 115, "GetEventFloat"                         , 0                                                  },
	{ 116, "SetEventFloat"                         , 0                                                  },
	{ 117, "GetEventObject"                        , 0                                                  },
	{ 118, "SetEventObject"                        , 0                                                  },
	{ 119, "GetEventString"                        , 0                                                  },
	{ 120, "SetEventString"                        , 0                                                  },
	{ 121, "HandleEvent"                           , &Functions::handleEvent                            },
	{ 122, "SetEventScript"                        , 0                                                  },
	{ 123, "EnablevEvent"                          , 0                                                  },
	{ 124, "RegisterEventListener"                 , 0                                                  },
	{ 125, "UnregisterEventListener"               , 0                                                  },
	{ 126, "GetEventTarget"                        , &Functions::getEventTarget                         },
	{ 127, "GetNearestObjectByHostility"           , 0                                                  },
	{ 128, "DelayEvent"                            , 0                                                  },
	{ 129, "IsFollower"                            , 0                                                  },
	{ 130, "PrintToLogAndFlush"                    , &Functions::printToLogAndFlush                     },
	{ 131, "GetEventLocation"                      , 0                                                  },
	{ 132, "SetEventLocation"                      , 0                                                  },
	{ 133, "HandleEvent_String"                    , 0                                                  },
	{ 134, "GetItemAttackRating"                   , 0                                                  },
	{ 135, "GetBaseItemType"                       , 0                                                  },
	{ 136, "GetItemMaterialType"                   , 0                                                  },
	{ 137, "SetItemMaterialType"                   , 0                                                  },
	{ 138, "GetM2DAFloat"                          , 0                                                  },
	{ 139, "RemoveEffectsByCreator"                , 0                                                  },
	{ 140, "SetObjectActive"                       , 0                                                  },
	{ 141, "GetObjectActive"                       , 0                                                  },
	{ 142, "InvalidFunction142"                    , 0                                                  },
	{ 143, "LogTrace"                              , &Functions::logTrace                               },
	{ 144, "SetFollowerSubState"                   , 0                                                  },
	{ 145, "GetCurrentScriptName"                  , &Functions::getCurrentScriptName                   },
	{ 146, "IsFollowerLocked"                      , 0                                                  },
	{ 147, "SetFollowerLocked"                     , 0                                                  },
	{ 148, "Engine_ApplyEffectOnParty"             , 0                                                  },
	{ 149, "ToString"                              , &Functions::toString                               },
	{ 150, "GetAppearanceType"                     , 0                                                  },
	{ 151, "SetPlaceableActionResult"              , 0                                                  },
	{ 152, "GetPlaceableAction"                    , 0                                                  },
	{ 153, "GetPlaceableBaseType"                  , 0                                                  },
	{ 154, "GetPlaceableAutoRemoveKey"             , 0                                                  },
	{ 155, "GetPlaceableKeyRequired"               , 0                                                  },
	{ 156, "GetPlaceableKeyTag"                    , 0                                                  },
	{ 157, "GetPlaceablePickLockLevel"             , 0                                                  },
	{ 158, "CommandSheatheWeapons"                 , 0                                                  },
	{ 159, "CommandUnsheatheWeapons"               , 0                                                  },
	{ 160, "GetWeaponsUnsheathedStatus"            , 0                                                  },
	{ 161, "GetAngleBetweenObjects"                , 0                                                  },
	{ 162, "CommandSwitchWeaponSet"                , 0                                                  },
	{ 163, "InvalidFunction163"                    , 0                                                  },
	{ 164, "IsPerceivingHostiles"                  , 0                                                  },
	{ 165, "IsPartyPerceivingHostiles"             , 0                                                  },
	{ 166, "TriggerPerception"                     , 0                                                  },
	{ 167, "GetActiveWeaponSet"                    , 0                                                  },
	{ 168, "GetMainControlled"                     , 0                                                  },
	{ 169, "GetPerceivedCreatureList"              , 0                                                  },
	{ 170, "IsPerceiving"                          , 0                                                  },
	{ 171, "GetItemHeraldry"                       , 0                                                  },
	{ 172, "SetItemHeraldry"                       , 0                                                  },
	{ 173, "PlayMovie"                             , 0                                                  },
	{ 174, "SendToRunDatabase"                     , 0                                                  },
	{ 175, "SendGameEventRunDatabase"              , 0                                                  },
	{ 176, "GetLowResTimer"                        , 0                                                  },
	{ 177, "GetHasEffects"                         , 0                                                  },
	{ 178, "SetAppearanceType"                     , 0                                                  },
	{ 179, "GetStringByStringId"                   , &Functions::getStringByStringId                    },
	{ 180, "GetItemsInInventory"                   , 0                                                  },
	{ 181, "GetPlotFlagRewardId"                   , 0                                                  },
	{ 182, "SetSoundSet"                           , 0                                                  },
	{ 183, "GetCreatureRank"                       , 0                                                  },
	{ 184, "GetCanDiePermanently"                  , 0                                                  },
	{ 185, "EnableWeaponTrail"                     , 0                                                  },
	{ 186, "GetVisualEffectLocation"               , 0                                                  },
	{ 187, "SetObjectAlwaysVisible"                , 0                                                  },
	{ 188, "ToggleGamePause"                       , 0                                                  },
	{ 189, "GetAutoPauseCombatStatus"              , 0                                                  },
	{ 190, "GetItemAbilityId"                      , 0                                                  },
	{ 191, "SetItemAbilityId"                      , 0                                                  },
	{ 192, "GetEffectsFlags"                       , 0                                                  },
	{ 193, "FireProjectile"                        , 0                                                  },
	{ 194, "SetCooldown"                           , 0                                                  },
	{ 195, "GetRemainingCooldown"                  , 0                                                  },
	{ 196, "PlayAdditiveAnimation"                 , 0                                                  },
	{ 197, "FilterObjectsInShape"                  , 0                                                  },
	{ 198, "GetThreatValueByObjectID"              , 0                                                  },
	{ 199, "GetThreatValueByIndex"                 , 0                                                  },
	{ 200, "GetThreatEnemy"                        , 0                                                  },
	{ 201, "GetThreatTableSize"                    , 0                                                  },
	{ 202, "ClearThreatTable"                      , 0                                                  },
	{ 203, "ClearEnemyThreat"                      , 0                                                  },
	{ 204, "UpdateThreatTable"                     , 0                                                  },
	{ 205, "GetCreaturesInMeleeRing"               , 0                                                  },
	{ 206, "IsObjectValid"                         , &Functions::isObjectValid                          },
	{ 207, "GetObjectType"                         , &Functions::getObjectType                          },
	{ 208, "GetResRef"                             , &Functions::getResRef                              },
	{ 209, "GetTag"                                , &Functions::getTag                                 },
	{ 210, "SetTag"                                , 0                                                  },
	{ 211, "GetObjectByTag"                        , &Functions::getObjectByTag                         },
	{ 212, "CreateItemOnObject"                    , 0                                                  },
	{ 213, "CreatePool"                            , 0                                                  },
	{ 214, "DestroyObject"                         , 0                                                  },
	{ 215, "IsDestroyable"                         , 0                                                  },
	{ 216, "SetDestroyable"                        , 0                                                  },
	{ 217, "IsPlot"                                , 0                                                  },
	{ 218, "SetPlot"                               , 0                                                  },
	{ 219, "IsImmortal"                            , 0                                                  },
	{ 220, "Engine_SetImmortal"                    , 0                                                  },
	{ 221, "GetAILevel"                            , 0                                                  },
	{ 222, "SetAILevel"                            , 0                                                  },
	{ 223, "GetArea"                               , &Functions::getArea                                },
	{ 224, "GetModule"                             , &Functions::getModule                              },
	{ 225, "InvalidFunction225"                    , 0                                                  },
	{ 226, "InvalidFunction226"                    , 0                                                  },
	{ 227, "InvalidFunction227"                    , 0                                                  },
	{ 228, "InvalidFunction228"                    , 0                                                  },
	{ 229, "InvalidFunction229"                    , 0                                                  },
	{ 230, "InvalidFunction230"                    , 0                                                  },
	{ 231, "InvalidFunction231"                    , 0                                                  },
	{ 232, "InvalidFunction232"                    , 0                                                  },
	{ 233, "GetHealth"                             , 0                                                  },
	{ 234, "SetPlaceableHealth"                    , 0                                                  },
	{ 235, "Deprecated_GetMaxHealth"               , 0                                                  },
	{ 236, "SetMaxHealth"                          , 0                                                  },
	{ 237, "IsDead"                                , 0                                                  },
	{ 238, "GetCreatureStamina"                    , 0                                                  },
	{ 239, "SetCreatureStamina"                    , 0                                                  },
	{ 240, "GetCreatureMaxStamina"                 , 0                                                  },
	{ 241, "SetCreatureMaxStamina"                 , 0                                                  },
	{ 242, "GetCreatureMana"                       , 0                                                  },
	{ 243, "SetCreatureMana"                       , 0                                                  },
	{ 244, "GetCreatureMaxMana"                    , 0                                                  },
	{ 245, "SetCreatureMaxMana"                    , 0                                                  },
	{ 246, "CommandDeathBlow"                      , 0                                                  },
	{ 247, "InvalidFunction247"                    , 0                                                  },
	{ 248, "GetCreatureAttribute"                  , 0                                                  },
	{ 249, "InvalidFunction249"                    , 0                                                  },
	{ 250, "InvalidFunction250"                    , 0                                                  },
	{ 251, "InvalidFunction251"                    , 0                                                  },
	{ 252, "HasAbility"                            , 0                                                  },
	{ 253, "AddAbility"                            , 0                                                  },
	{ 254, "RemoveAbility"                         , 0                                                  },
	{ 255, "InvalidFunction255"                    , 0                                                  },
	{ 256, "InvalidFunction256"                    , 0                                                  },
	{ 257, "InvalidFunction257"                    , 0                                                  },
	{ 258, "InvalidFunction258"                    , 0                                                  },
	{ 259, "InvalidFunction259"                    , 0                                                  },
	{ 260, "InvalidFunction260"                    , 0                                                  },
	{ 261, "InvalidFunction261"                    , 0                                                  },
	{ 262, "InvalidFunction262"                    , 0                                                  },
	{ 263, "GetCreatureGender"                     , 0                                                  },
	{ 264, "SetCreatureGender"                     , 0                                                  },
	{ 265, "InvalidFunction265"                    , 0                                                  },
	{ 266, "InvalidFunction266"                    , 0                                                  },
	{ 267, "InvalidFunction267"                    , 0                                                  },
	{ 268, "InvalidFunction268"                    , 0                                                  },
	{ 269, "GetCreatureRacialType"                 , 0                                                  },
	{ 270, "SetCreatureRacialType"                 , 0                                                  },
	{ 271, "InvalidFunction271"                    , 0                                                  },
	{ 272, "InvalidFunction272"                    , 0                                                  },
	{ 273, "SwitchWeaponSet"                       , 0                                                  },
	{ 274, "InvalidFunction274"                    , 0                                                  },
	{ 275, "GetName"                               , &Functions::getName                                },
	{ 276, "SetName"                               , &Functions::setName                                },
	{ 277, "InvalidFunction277"                    , 0                                                  },
	{ 278, "InvalidFunction278"                    , 0                                                  },
	{ 279, "InvalidFunction279"                    , 0                                                  },
	{ 280, "InvalidFunction280"                    , 0                                                  },
	{ 281, "InvalidFunction281"                    , 0                                                  },
	{ 282, "InvalidFunction282"                    , 0                                                  },
	{ 283, "InvalidFunction283"                    , 0                                                  },
	{ 284, "InvalidFunction284"                    , 0                                                  },
	{ 285, "InvalidFunction285"                    , 0                                                  },
	{ 286, "GetPlaceableState"                     , 0                                                  },
	{ 287, "SetPlaceableState"                     , 0                                                  },
	{ 288, "InvalidFunction288"                    , 0                                                  },
	{ 289, "InvalidFunction289"                    , 0                                                  },
	{ 290, "InvalidFunction290"                    , 0                                                  },
	{ 291, "GetItemValue"                          , 0                                                  },
	{ 292, "InvalidFunction292"                    , 0                                                  },
	{ 293, "InvalidFunction293"                    , 0                                                  },
	{ 294, "InvalidFunction294"                    , 0                                                  },
	{ 295, "IsItemDroppable"                       , 0                                                  },
	{ 296, "SetItemDroppable"                      , 0                                                  },
	{ 297, "InvalidFunction297"                    , 0                                                  },
	{ 298, "InvalidFunction298"                    , 0                                                  },
	{ 299, "GetMaxItemStackSize"                   , 0                                                  },
	{ 300, "GetItemStackSize"                      , 0                                                  },
	{ 301, "SetItemStackSize"                      , 0                                                  },
	{ 302, "InvalidFunction302"                    , 0                                                  },
	{ 303, "AddItemProperty"                       , 0                                                  },
	{ 304, "RemoveItemProperty"                    , 0                                                  },
	{ 305, "GetItemStat"                           , 0                                                  },
	{ 306, "CreateObject"                          , 0                                                  },
	{ 307, "SetLocName"                            , 0                                                  },
	{ 308, "GetCurrentScriptResource"              , &Functions::getCurrentScriptResource               },
	{ 309, "OpenFadeMap"                           , 0                                                  },
	{ 310, "ShowTutorial"                          , 0                                                  },
	{ 311, "SetPartyPickerGUIStatus"               , 0                                                  },
	{ 312, "GetPartyPickerGUIStatus"               , 0                                                  },
	{ 313, "GetStoreMarkDown"                      , 0                                                  },
	{ 314, "SetStoreMarkDown"                      , 0                                                  },
	{ 315, "GetCreatureTreasureCategory"           , 0                                                  },
	{ 316, "GetItemProperties"                     , 0                                                  },
	{ 317, "GetPackage"                            , 0                                                  },
	{ 318, "GetPackageAI"                          , 0                                                  },
	{ 319, "SetTacticEntry"                        , 0                                                  },
	{ 320, "CharGen_ClearAbilityList"              , 0                                                  },
	{ 321, "SetDeathHint"                          , 0                                                  },
	{ 322, "SetLoadHint"                           , 0                                                  },
	{ 323, "ShowPopup"                             , 0                                                  },
	{ 324, "AdjustFollowerApproval"                , 0                                                  },
	{ 325, "SetClassRank"                          , 0                                                  },
	{ 326, "HasDeathEffect"                        , 0                                                  },
	{ 327, "CommandTurn"                           , 0                                                  },
	{ 328, "SetQuickslotBar"                       , 0                                                  },
	{ 329, "LockQuickslotBar"                      , 0                                                  },
	{ 330, "GetQuickslot"                          , 0                                                  },
	{ 331, "GetDebugHelpersEnabled"                , 0                                                  },
	{ 332, "RevealCurrentMap"                      , 0                                                  },
	{ 333, "ClearCreatureProperties"               , 0                                                  },
	{ 334, "GetAOEFlags"                           , 0                                                  },
	{ 335, "SetAOEFlags"                           , 0                                                  },
	{ 336, "IsInAOE"                               , 0                                                  },
	{ 337, "GetCreaturesInAOE"                     , 0                                                  },
	{ 338, "DEBUG_SendRawInputEvent"               , 0                                                  },
	{ 339, "DEBUG_SetCursorPosition"               , 0                                                  },
	{ 340, "LoadItemsFromTemplate"                 , 0                                                  },
	{ 341, "GetAutoLevelFollowers"                 , 0                                                  },
	{ 342, "InitHeartbeat"                         , 0                                                  },
	{ 343, "EndHeartbeat"                          , 0                                                  },
	{ 344, "DialogIsAmbient"                       , 0                                                  },
	{ 345, "StorePartyInventory"                   , 0                                                  },
	{ 346, "StoreFollowerInventory"                , 0                                                  },
	{ 347, "RestorePartyInventory"                 , 0                                                  },
	{ 348, "RestoreFollowerInventory"              , 0                                                  },
	{ 349, "MoveItem"                              , 0                                                  },
	{ 350, "MoveAllItems"                          , 0                                                  },
	{ 351, "AddNonPartyFollower"                   , 0                                                  },
	{ 352, "RemoveNonPartyFollower"                , 0                                                  },
	{ 353, "SetAttackDuration"                     , 0                                                  },
	{ 354, "GetWeaponStyle"                        , 0                                                  },
	{ 355, "ShowCraftingGUI"                       , 0                                                  },
	{ 356, "GetPlaceablePopupText"                 , 0                                                  },
	{ 357, "LinkAreaMiniMaps"                      , 0                                                  },
	{ 358, "GetItemEquipSlot"                      , 0                                                  },
	{ 359, "SignalDamage"                          , 0                                                  },
	{ 360, "UT_GetNearestObjectByTag"              , &Functions::UT_getNearestObjectByTag               },
	{ 361, "SetLoadImage"                          , 0                                                  },
	{ 362, "ShowStartMenu"                         , 0                                                  },
	{ 363, "IsHumanoid"                            , 0                                                  },
	{ 364, "ShowAsAllyOnMap"                       , 0                                                  },
	{ 365, "GetTacticPresetID"                     , 0                                                  },
	{ 366, "SetTacticPresetID"                     , 0                                                  },
	{ 367, "AddTacticPresetID"                     , 0                                                  },
	{ 368, "GetItemUnique"                         , 0                                                  },
	{ 369, "GetItemMaterialProgression"            , 0                                                  },
	{ 370, "FireHomingProjectile"                  , 0                                                  },
	{ 371, "SetProjectileImpactEvent"              , 0                                                  },
	{ 372, "GetTacticTargetObject"                 , 0                                                  },
	{ 373, "GetTacticConditionObject"              , 0                                                  },
	{ 374, "GetTacticCommandItemTag"               , 0                                                  },
	{ 375, "SetItemIrremovable"                    , 0                                                  },
	{ 376, "IsItemIrremovable"                     , 0                                                  },
	{ 377, "SetItemIndestructible"                 , 0                                                  },
	{ 378, "IsItemIndestructible"                  , 0                                                  },
	{ 379, "RemoveItemsByTag"                      , 0                                                  },
	{ 380, "CountItemsByTag"                       , 0                                                  },
	{ 381, "GetPlotSummary"                        , 0                                                  },
	{ 382, "SetFollowerApprovalEnabled"            , 0                                                  },
	{ 383, "ECSplitString"                         , 0                                                  },
	{ 384, "GetAutoLevelUp"                        , 0                                                  },
	{ 385, "SetAutoLevelUp"                        , 0                                                  },
	{ 386, "ScaleEquippedItems"                    , 0                                                  },
	{ 387, "ClosePrimaryWorldMap"                  , 0                                                  },
	{ 388, "IsMoving"                              , 0                                                  },
	{ 389, "GetAbilitiesDueToAOEs"                 , 0                                                  },
	{ 390, "PreloadCharGen"                        , 0                                                  },
	{ 391, "SetCanChangeEquipment"                 , 0                                                  },
	{ 392, "EquipItem"                             , 0                                                  },
	{ 393, "UnequipItem"                           , 0                                                  },
	{ 394, "SetAimLoopDuration"                    , 0                                                  },
	{ 395, "GetPartyPoolList"                      , 0                                                  },
	{ 396, "InvalidFunction396"                    , 0                                                  },
	{ 397, "InvalidFunction397"                    , 0                                                  },
	{ 398, "InvalidFunction398"                    , 0                                                  },
	{ 399, "GetFollowerState"                      , 0                                                  },
	{ 400, "GetFollowerSubState"                   , 0                                                  },
	{ 401, "GetItemInEquipSlot"                    , 0                                                  },
	{ 402, "InvalidFunction402"                    , 0                                                  },
	{ 403, "RemoveItem"                            , 0                                                  },
	{ 404, "GetMaxInventorySize"                   , 0                                                  },
	{ 405, "SetMaxInventorySize"                   , 0                                                  },
	{ 406, "AddCreatureMoney"                      , 0                                                  },
	{ 407, "InvalidFunction407"                    , 0                                                  },
	{ 408, "GetCreatureMoney"                      , 0                                                  },
	{ 409, "SetCreatureMoney"                      , 0                                                  },
	{ 410, "OpenInventory"                         , 0                                                  },
	{ 411, "GetTrapDetectionDifficulty"            , 0                                                  },
	{ 412, "InvalidFunction412"                    , 0                                                  },
	{ 413, "GetTrapDisarmDifficulty"               , 0                                                  },
	{ 414, "InvalidFunction414"                    , 0                                                  },
	{ 415, "InvalidFunction415"                    , 0                                                  },
	{ 416, "SetTrapDetected"                       , 0                                                  },
	{ 417, "InvalidFunction417"                    , 0                                                  },
	{ 418, "InvalidFunction418"                    , 0                                                  },
	{ 419, "InvalidFunction419"                    , 0                                                  },
	{ 420, "InvalidFunction420"                    , 0                                                  },
	{ 421, "InvalidFunction421"                    , 0                                                  },
	{ 422, "InvalidFunction422"                    , 0                                                  },
	{ 423, "InvalidFunction423"                    , 0                                                  },
	{ 424, "InvalidFunction424"                    , 0                                                  },
	{ 425, "InvalidFunction425"                    , 0                                                  },
	{ 426, "InvalidFunction426"                    , 0                                                  },
	{ 427, "InvalidFunction427"                    , 0                                                  },
	{ 428, "InvalidFunction428"                    , 0                                                  },
	{ 429, "InvalidFunction429"                    , 0                                                  },
	{ 430, "InvalidFunction430"                    , 0                                                  },
	{ 431, "InvalidFunction431"                    , 0                                                  },
	{ 432, "InvalidFunction432"                    , 0                                                  },
	{ 433, "GetNearestObject"                      , &Functions::getNearestObject                       },
	{ 434, "GetNearestObjectToLocation"            , 0                                                  },
	{ 435, "GetNearestObjectByTag"                 , &Functions::getNearestObjectByTag                  },
	{ 436, "GetNearestObjectByGroup"               , 0                                                  },
	{ 437, "InvalidFunction437"                    , 0                                                  },
	{ 438, "GetPCSpeaker"                          , 0                                                  },
	{ 439, "InvalidFunction439"                    , 0                                                  },
	{ 440, "BeginConversation"                     , 0                                                  },
	{ 441, "InvalidFunction441"                    , 0                                                  },
	{ 442, "InvalidFunction442"                    , 0                                                  },
	{ 443, "GetConversationEntryParameter"         , 0                                                  },
	{ 444, "HasConversation"                       , 0                                                  },
	{ 445, "GetClusterCenter"                      , 0                                                  },
	{ 446, "ClearPerceptionList"                   , 0                                                  },
	{ 447, "OpenStore"                             , 0                                                  },
	{ 448, "GetStoreMoney"                         , 0                                                  },
	{ 449, "SetStoreMoney"                         , 0                                                  },
	{ 450, "GetStoreMaxBuyPrice"                   , 0                                                  },
	{ 451, "SetStoreMaxBuyPrice"                   , 0                                                  },
	{ 452, "GetStoreMarkUp"                        , 0                                                  },
	{ 453, "SetStoreMarkUp"                        , 0                                                  },
	{ 454, "Location"                              , 0                                                  },
	{ 455, "IsLocationValid"                       , 0                                                  },
	{ 456, "IsLocationSafe"                        , 0                                                  },
	{ 457, "GetLocation"                           , 0                                                  },
	{ 458, "SetLocation"                           , 0                                                  },
	{ 459, "GetPositionFromLocation"               , 0                                                  },
	{ 460, "SetLocationPosition"                   , 0                                                  },
	{ 461, "GetAreaFromLocation"                   , 0                                                  },
	{ 462, "SetLocationArea"                       , 0                                                  },
	{ 463, "GetFacingFromLocation"                 , 0                                                  },
	{ 464, "GetOrientationFromLocation"            , 0                                                  },
	{ 465, "SetLocationOrientation"                , 0                                                  },
	{ 466, "GetGroupHostility"                     , 0                                                  },
	{ 467, "SetGroupHostility"                     , 0                                                  },
	{ 468, "IsObjectHostile"                       , 0                                                  },
	{ 469, "GetGroupId"                            , 0                                                  },
	{ 470, "SetGroupId"                            , 0                                                  },
	{ 471, "SetReactionOverride"                   , 0                                                  },
	{ 472, "InvalidFunction472"                    , 0                                                  },
	{ 473, "InvalidFunction473"                    , 0                                                  },
	{ 474, "InvalidFunction474"                    , 0                                                  },
	{ 475, "InvalidFunction475"                    , 0                                                  },
	{ 476, "InvalidFunction476"                    , 0                                                  },
	{ 477, "PlaySound"                             , 0                                                  },
	{ 478, "StopSound"                             , 0                                                  },
	{ 479, "PlaySoundSet"                          , 0                                                  },
	{ 480, "PlaySoundObject"                       , 0                                                  },
	{ 481, "StopSoundObject"                       , 0                                                  },
	{ 482, "InvalidFunction482"                    , 0                                                  },
	{ 483, "InvalidFunction483"                    , 0                                                  },
	{ 484, "PlayMusic"                             , 0                                                  },
	{ 485, "StopMusic"                             , 0                                                  },
	{ 486, "SetMusicIntensity"                     , 0                                                  },
	{ 487, "ActivateSoundByTag"                    , 0                                                  },
	{ 488, "SetMusicVolumeStateByTag"              , 0                                                  },
	{ 489, "InvalidFunction489"                    , 0                                                  },
	{ 490, "InvalidFunction490"                    , 0                                                  },
	{ 491, "InvalidFunction491"                    , 0                                                  },
	{ 492, "InvalidFunction492"                    , 0                                                  },
	{ 493, "InvalidFunction493"                    , 0                                                  },
	{ 494, "InvalidFunction494"                    , 0                                                  },
	{ 495, "InvalidFunction495"                    , 0                                                  },
	{ 496, "InvalidFunction496"                    , 0                                                  },
	{ 497, "InvalidFunction497"                    , 0                                                  },
	{ 498, "InvalidFunction498"                    , 0                                                  },
	{ 499, "InvalidFunction499"                    , 0                                                  },
	{ 500, "IsHero"                                , &Functions::isHero                                 },
	{ 501, "InvalidFunction501"                    , 0                                                  },
	{ 502, "GetHero"                               , &Functions::getHero                                },
	{ 503, "InvalidFunction503"                    , 0                                                  },
	{ 504, "GetPartyList"                          , 0                                                  },
	{ 505, "InvalidFunction505"                    , 0                                                  },
	{ 506, "InvalidFunction506"                    , 0                                                  },
	{ 507, "InvalidFunction507"                    , 0                                                  },
	{ 508, "InvalidFunction508"                    , 0                                                  },
	{ 509, "InvalidFunction509"                    , 0                                                  },
	{ 510, "InvalidFunction510"                    , 0                                                  },
	{ 511, "InvalidFunction511"                    , 0                                                  },
	{ 512, "InvalidFunction512"                    , 0                                                  },
	{ 513, "DoAutoSave"                            , 0                                                  },
	{ 514, "InvalidFunction514"                    , 0                                                  },
	{ 515, "InvalidFunction515"                    , 0                                                  },
	{ 516, "InvalidFunction516"                    , 0                                                  },
	{ 517, "InvalidFunction517"                    , 0                                                  },
	{ 518, "InvalidFunction518"                    , 0                                                  },
	{ 519, "InvalidFunction519"                    , 0                                                  },
	{ 520, "InvalidFunction520"                    , 0                                                  },
	{ 521, "InvalidFunction521"                    , 0                                                  },
	{ 522, "GetAttackTarget"                       , 0                                                  },
	{ 523, "InvalidFunction523"                    , 0                                                  },
	{ 524, "InvalidFunction524"                    , 0                                                  },
	{ 525, "SetCombatState"                        , 0                                                  },
	{ 526, "GetCombatState"                        , 0                                                  },
	{ 527, "InvalidFunction527"                    , 0                                                  },
	{ 528, "InvalidFunction528"                    , 0                                                  },
	{ 529, "GetGameMode"                           , 0                                                  },
	{ 530, "SetGameMode"                           , 0                                                  },
	{ 531, "InvalidFunction531"                    , 0                                                  },
	{ 532, "InvalidFunction532"                    , 0                                                  },
	{ 533, "AddCommand"                            , 0                                                  },
	{ 534, "RemoveCommandByIndex"                  , 0                                                  },
	{ 535, "RemoveCommand"                         , 0                                                  },
	{ 536, "GetCommandByIndex"                     , 0                                                  },
	{ 537, "InvalidFunction537"                    , 0                                                  },
	{ 538, "ClearAllCommands"                      , 0                                                  },
	{ 539, "GetCurrentCommand"                     , 0                                                  },
	{ 540, "ClearCurrentCommand"                   , 0                                                  },
	{ 541, "GetCommandType"                        , 0                                                  },
	{ 542, "InvalidFunction542"                    , 0                                                  },
	{ 543, "InvalidFunction543"                    , 0                                                  },
	{ 544, "GetCommandInt"                         , 0                                                  },
	{ 545, "SetCommandInt"                         , 0                                                  },
	{ 546, "GetCommandFloat"                       , 0                                                  },
	{ 547, "SetCommandFloat"                       , 0                                                  },
	{ 548, "InvalidFunction548"                    , 0                                                  },
	{ 549, "InvalidFunction549"                    , 0                                                  },
	{ 550, "GetCommandObject"                      , 0                                                  },
	{ 551, "SetCommandObject"                      , 0                                                  },
	{ 552, "InvalidFunction552"                    , 0                                                  },
	{ 553, "InvalidFunction553"                    , 0                                                  },
	{ 554, "InvalidFunction554"                    , 0                                                  },
	{ 555, "InvalidFunction555"                    , 0                                                  },
	{ 556, "InvalidFunction556"                    , 0                                                  },
	{ 557, "InvalidFunction557"                    , 0                                                  },
	{ 558, "InvalidFunction558"                    , 0                                                  },
	{ 559, "CommandMoveToLocation"                 , 0                                                  },
	{ 560, "CommandMoveToObject"                   , 0                                                  },
	{ 561, "CommandMoveAwayFromObject"             , 0                                                  },
	{ 562, "CommandEquipItem"                      , 0                                                  },
	{ 563, "CommandUnequipItem"                    , 0                                                  },
	{ 564, "InvalidFunction564"                    , 0                                                  },
	{ 565, "InvalidFunction565"                    , 0                                                  },
	{ 566, "CommandAttack"                         , 0                                                  },
	{ 567, "InvalidFunction567"                    , 0                                                  },
	{ 568, "InvalidFunction568"                    , 0                                                  },
	{ 569, "CommandPlayAnimation"                  , 0                                                  },
	{ 570, "InvalidFunction570"                    , 0                                                  },
	{ 571, "InvalidFunction571"                    , 0                                                  },
	{ 572, "InvalidFunction572"                    , 0                                                  },
	{ 573, "InvalidFunction573"                    , 0                                                  },
	{ 574, "InvalidFunction574"                    , 0                                                  },
	{ 575, "InvalidFunction575"                    , 0                                                  },
	{ 576, "InvalidFunction576"                    , 0                                                  },
	{ 577, "InvalidFunction577"                    , 0                                                  },
	{ 578, "CommandWait"                           , 0                                                  },
	{ 579, "CommandStartConversation"              , 0                                                  },
	{ 580, "CommandJumpToObject"                   , 0                                                  },
	{ 581, "CommandJumpToLocation"                 , 0                                                  },
	{ 582, "CommandUseAbility"                     , 0                                                  },
	{ 583, "InvalidFunction583"                    , 0                                                  },
	{ 584, "InvalidFunction584"                    , 0                                                  },
	{ 585, "InvalidFunction585"                    , 0                                                  },
	{ 586, "CommandUseObject"                      , 0                                                  },
	{ 587, "InteractWithObject"                    , 0                                                  },
	{ 588, "InvalidFunction588"                    , 0                                                  },
	{ 589, "InvalidFunction589"                    , 0                                                  },
	{ 590, "InvalidFunction590"                    , 0                                                  },
	{ 591, "InvalidFunction591"                    , 0                                                  },
	{ 592, "InvalidFunction592"                    , 0                                                  },
	{ 593, "RemoveEffectsByParameters"             , 0                                                  },
	{ 594, "RemoveAllEffects"                      , 0                                                  },
	{ 595, "IsEffectValid"                         , 0                                                  },
	{ 596, "Engine_ApplyEffectAtLocation"          , 0                                                  },
	{ 597, "Engine_ApplyEffectOnObject"            , 0                                                  },
	{ 598, "RemoveEffect"                          , 0                                                  },
	{ 599, "TakeScreenshot"                        , 0                                                  },
	{ 600, "SetWorldMapLocationLocName"            , 0                                                  },
	{ 601, "GetShowSpecialMoveFloaties"            , 0                                                  },
	{ 602, "GetGamePlotAssist"                     , 0                                                  },
	{ 603, "GetHasAchievementByID"                 , 0                                                  },
	{ 604, "GetEffectType"                         , 0                                                  },
	{ 605, "GetEffectFlags"                        , 0                                                  },
	{ 606, "GetMeleeRingPosition"                  , 0                                                  },
	{ 607, "GetEffectAnimation"                    , 0                                                  },
	{ 608, "GetEffectDurationType"                 , 0                                                  },
	{ 609, "SetEffectFlags"                        , 0                                                  },
	{ 610, "IsAbilityAvailable"                    , 0                                                  },
	{ 611, "SetEffectAnimation"                    , 0                                                  },
	{ 612, "GetEffectCreator"                      , 0                                                  },
	{ 613, "SetEffectCreator"                      , 0                                                  },
	{ 614, "GetEffectID"                           , 0                                                  },
	{ 615, "GetAbilityCount"                       , 0                                                  },
	{ 616, "UnlockAchievementByID"                 , 0                                                  },
	{ 617, "IncrementAchievementCountByID"         , 0                                                  },
	{ 618, "GetAchievementCountByID"               , 0                                                  },
	{ 619, "ShowAllCodexEntries"                   , 0                                                  },
	{ 620, "GetEffectAbilityID"                    , 0                                                  },
	{ 621, "SetEffectAbilityID"                    , 0                                                  },
	{ 622, "GetFollowerApproval"                   , 0                                                  },
	{ 623, "GetClassRank"                          , 0                                                  },
	{ 624, "GetPartyLeader"                        , 0                                                  },
	{ 625, "SetPartyLeader"                        , 0                                                  },
	{ 626, "ClearAmbientDialogs"                   , 0                                                  },
	{ 627, "SetCheatUsedFlag"                      , 0                                                  },
	{ 628, "GetCommandIsPlayerIssued"              , 0                                                  },
	{ 629, "GetHasAchievement"                     , 0                                                  },
	{ 630, "SendOnlineTelemetry"                   , 0                                                  },
	{ 631, "SendOnlineScreenShot"                  , 0                                                  },
	{ 632, "BeginSlideshow"                        , 0                                                  },
	{ 633, "GetCanLevelUp"                         , 0                                                  },
	{ 634, "EffectAreaOfEffect"                    , 0                                                  },
	{ 635, "ApplyEffectVisualEffect"               , 0                                                  },
	{ 636, "OpenItemUpgradeGUI"                    , 0                                                  },
	{ 637, "GetBackgroundDefaultsIndex"            , 0                                                  },
	{ 638, "UnlockAchievement"                     , 0                                                  },
	{ 639, "UpdateOnlineStatistic"                 , 0                                                  },
	{ 640, "GetOnlineStatistic"                    , 0                                                  },
	{ 641, "IsItemStealable"                       , 0                                                  },
	{ 642, "SetCreatureIsStatue"                   , 0                                                  },
	{ 643, "SetCreatureIsGhost"                    , 0                                                  },
	{ 644, "CommandMoveToMultiLocations"           , 0                                                  },
	{ 645, "SetItemDamaged"                        , 0                                                  },
	{ 646, "GetItemDamaged"                        , 0                                                  },
	{ 647, "LoadCutscene"                          , 0                                                  },
	{ 648, "PlayCutscene"                          , 0                                                  },
	{ 649, "SetMapPatchState"                      , 0                                                  },
	{ 650, "GetMapPatchState"                      , 0                                                  },
	{ 651, "GetPlayerMapPatch"                     , 0                                                  },
	{ 652, "SetMapPinState"                        , 0                                                  },
	{ 653, "GetMapPinState"                        , 0                                                  },
	{ 654, "GetPlayerMapPin"                       , 0                                                  },
	{ 655, "SetEffectEngineInteger"                , 0                                                  },
	{ 656, "GetOrientation"                        , 0                                                  },
	{ 657, "SetOrientation"                        , 0                                                  },
	{ 658, "SetLocationFacing"                     , 0                                                  },
	{ 659, "GetSafeLocation"                       , 0                                                  },
	{ 660, "GetPartyPlotFlag"                      , 0                                                  },
	{ 661, "SetPartyPlotFlag"                      , 0                                                  },
	{ 662, "InvalidFunction662"                    , 0                                                  },
	{ 663, "InvalidFunction663"                    , 0                                                  },
	{ 664, "InvalidFunction664"                    , 0                                                  },
	{ 665, "GetPlotEntryName"                      , 0                                                  },
	{ 666, "InvalidFunction666"                    , 0                                                  },
	{ 667, "GetM2DAInt"                            , 0                                                  },
	{ 668, "InvalidFunction668"                    , 0                                                  },
	{ 669, "ResetReaction"                         , 0                                                  },
	{ 670, "InvalidFunction670"                    , 0                                                  },
	{ 671, "InvalidFunction671"                    , 0                                                  },
	{ 672, "SpawnBodyBag"                          , 0                                                  },
	{ 673, "GetObjectsInShape"                     , 0                                                  },
	{ 674, "Warning"                               , &Functions::printWarning                           },
	{ 675, "GetItemPossessedBy"                    , 0                                                  },
	{ 676, "SetObjectInteractive"                  , 0                                                  },
	{ 677, "GetObjectInteractive"                  , 0                                                  },
	{ 678, "InvalidFunction678"                    , 0                                                  },
	{ 679, "InvalidFunction679"                    , 0                                                  },
	{ 680, "Probe"                                 , 0                                                  },
	{ 681, "GetCurrentEffect"                      , 0                                                  },
	{ 682, "SetIsCurrentEffectValid"               , 0                                                  },
	{ 683, "GetEffectInteger"                      , 0                                                  },
	{ 684, "SetEffectInteger"                      , 0                                                  },
	{ 685, "GetEffectFloat"                        , 0                                                  },
	{ 686, "SetEffectFloat"                        , 0                                                  },
	{ 687, "GetEffectObject"                       , 0                                                  },
	{ 688, "SetEffectObject"                       , 0                                                  },
	{ 689, "GetEffectString"                       , 0                                                  },
	{ 690, "SetEffectString"                       , 0                                                  },
	{ 691, "SetAttackResult"                       , 0                                                  },
	{ 692, "InvalidFunction692"                    , 0                                                  },
	{ 693, "GetWeaponAttackRating"                 , 0                                                  },
	{ 694, "GetPlatform"                           , 0                                                  },
	{ 695, "GetAttackImpactDamageEffect"           , 0                                                  },
	{ 696, "WorldMapStartTravelling"               , 0                                                  },
	{ 697, "WorldMapCompleteRandomEncounter"       , 0                                                  },
	{ 698, "SetBodybagDecayDelay"                  , 0                                                  },
	{ 699, "GetWeaponBaseDamage"                   , 0                                                  },
	{ 700, "DEPRECATED_GetCreatureStrengthModifier", 0                                                  },
	{ 701, "GetWeaponStrengthModifier"             , 0                                                  },
	{ 702, "SetCreaturesGlobalMaxTimeBeforeDecay"  , 0                                                  },
	{ 703, "GetWeaponArmorPenetration"             , 0                                                  },
	{ 704, "DEBUG_ConsoleCommand"                  , 0                                                  },
	{ 705, "GetTime"                               , 0                                                  },
	{ 706, "SetFollowerState"                      , 0                                                  },
	{ 707, "GetObjectsInArea"                      , 0                                                  },
	{ 708, "GetArraySize"                          , &Functions::getArraySize                           },
	{ 709, "InvalidFunction709"                    , 0                                                  },
	{ 710, "InvalidFunction710"                    , 0                                                  },
	{ 711, "InvalidFunction711"                    , 0                                                  },
	{ 712, "GetEffects"                            , 0                                                  },
	{ 713, "GetEffectsByCreator"                   , 0                                                  },
	{ 714, "IsControlled"                          , 0                                                  },
	{ 715, "SetAbilityResult"                      , 0                                                  },
	{ 716, "SetCommandResult"                      , 0                                                  },
	{ 717, "InvalidFunction717"                    , 0                                                  },
	{ 718, "InvalidFunction718"                    , 0                                                  },
	{ 719, "GetCreatureGoreLevel"                  , 0                                                  },
	{ 720, "SetCreatureGoreLevel"                  , 0                                                  },
	{ 721, "InvalidFunction721"                    , 0                                                  },
	{ 722, "ShowSpecUnlockedNotification"          , 0                                                  },
	{ 723, "ShowAreaUnlockedNotification"          , 0                                                  },
	{ 724, "GetCommandQueueSize"                   , 0                                                  },
	{ 725, "Engine_SetModalAbilityGUI"             , 0                                                  },
	{ 726, "IsModalAbilityActive"                  , 0                                                  },
	{ 727, "SetEffectEngineVector"                 , 0                                                  },
	{ 728, "GetFollowerWouldBeLocation"            , 0                                                  },
	{ 729, "SetDead"                               , 0                                                  },
	{ 730, "GetCommandPriority"                    , 0                                                  },
	{ 731, "IsConjuring"                           , 0                                                  },
	{ 732, "ResourceToString"                      , &Functions::resourceToString                       },
	{ 733, "PrintResource"                         , &Functions::printResource                          },
	{ 734, "SetEventResource"                      , 0                                                  },
	{ 735, "GetEventResource"                      , 0                                                  },
	{ 736, "GetLocalResource"                      , &Functions::getLocalResource                       },
	{ 737, "SetLocalResource"                      , &Functions::setLocalResource                       },
	{ 738, "GetCreatureProperty"                   , 0                                                  },
	{ 739, "GetCreaturePropertyType"               , 0                                                  },
	{ 740, "SetCreatureProperty"                   , 0                                                  },
	{ 741, "UpdateCreatureProperty"                , 0                                                  },
	{ 742, "GetPlotResRef"                         , 0                                                  },
	{ 743, "GetPlotFlagName"                       , 0                                                  },
	{ 744, "DisplayFloatyMessage"                  , 0                                                  },
	{ 745, "DoAreaTransition"                      , 0                                                  },
	{ 746, "StartCharGen"                          , 0                                                  },
	{ 747, "DisplayStatusMessage"                  , 0                                                  },
	{ 748, "SetQuickslot"                          , 0                                                  },
	{ 749, "ShowFloatyIcon"                        , 0                                                  },
	{ 750, "SetStoryPlot"                          , 0                                                  },
	{ 751, "SetPlotGiver"                          , 0                                                  },
	{ 752, "SetWorldMapGuiStatus"                  , 0                                                  },
	{ 753, "SetWorldMapLocationStatus"             , 0                                                  },
	{ 754, "SetWorldMapPlayerLocation"             , 0                                                  },
	{ 755, "SetWorldMapPrimary"                    , 0                                                  },
	{ 756, "SetWorldMapSecondary"                  , 0                                                  },
	{ 757, "OpenPrimaryWorldMap"                   , 0                                                  },
	{ 758, "SetStealthEnabled"                     , 0                                                  },
	{ 759, "GetStealthEnabled"                     , 0                                                  },
	{ 760, "SetCanLevelUp"                         , 0                                                  },
	{ 761, "SetEffectEngineFloat"                  , 0                                                  },
	{ 762, "SetEffectEngineObject"                 , 0                                                  },
	{ 763, "GetCombatantType"                      , 0                                                  },
	{ 764, "SetCreatureCanBeTracked"               , 0                                                  },
	{ 765, "GetCreatureCanBeTracked"               , 0                                                  },
	{ 766, "SetCreatureRank"                       , 0                                                  },
	{ 767, "RandomFloat"                           , &Functions::randomFloat                            },
	{ 768, "GetFollowPartyLeader"                  , 0                                                  },
	{ 769, "SetFollowPartyLeader"                  , 0                                                  },
	{ 770, "DisplayPortraitMessage"                , 0                                                  },
	{ 771, "ShowDeathScreen"                       , 0                                                  },
	{ 772, "GetNumTactics"                         , 0                                                  },
	{ 773, "SetNumTactics"                         , 0                                                  },
	{ 774, "IsTacticEnabled"                       , 0                                                  },
	{ 775, "GetTacticTargetType"                   , 0                                                  },
	{ 776, "GetTacticCondition"                    , 0                                                  },
	{ 777, "GetTacticCommand"                      , 0                                                  },
	{ 778, "GetTacticCommandParam"                 , 0                                                  },
	{ 779, "ShowChantersGUI"                       , 0                                                  },
	{ 780, "AddItemVisualEffect"                   , 0                                                  },
	{ 781, "GetItemVisualEffectsIDs"               , 0                                                  },
	{ 782, "RemoveItemVisualEffect"                , 0                                                  },
	{ 783, "RemoveAllItemVisualEffects"            , 0                                                  },
	{ 784, "GetItemPropertyPower"                  , 0                                                  },
	{ 785, "InvalidFunction785"                    , 0                                                  },
	{ 786, "InvalidFunction786"                    , 0                                                  },
	{ 787, "IsPartyAIEnabled"                      , 0                                                  },
	{ 788, "ShowGeneralScoreboardGUI"              , 0                                                  },
	{ 789, "ShowChampionshipScoreboardGUI"         , 0                                                  },
	{ 790, "ShowTournamentScoreboardGUI"           , 0                                                  },
	{ 791, "AddGeneralScoreboardEntry"             , 0                                                  },
	{ 792, "SetChampionshipScoreboardEntry"        , 0                                                  },
	{ 793, "SetTournamentScoreboardEntry"          , 0                                                  },
	{ 794, "ShowPartyPickerGUI"                    , 0                                                  },
	{ 795, "SetPlotActionsEnabled"                 , 0                                                  },
	{ 796, "SetPlotActionState"                    , 0                                                  },
	{ 797, "GetPlotActionState"                    , 0                                                  },
	{ 798, "SetPlotActionCount"                    , 0                                                  },
	{ 799, "GetPlotActionCount"                    , 0                                                  },
	{ 800, "SetAtmosphere"                         , 0                                                  },
	{ 801, "SetAtmosphereRGB"                      , 0                                                  },
	{ 802, "ResetAtmosphere"                       , 0                                                  },
	{ 803, "FB_SetEffectResource"                  , 0                                                  },
	{ 804, "FB_SetEffectEnabled"                   , 0                                                  },
	{ 805, "SetLookAtEnabled"                      , 0                                                  },
	{ 806, "SetTeamId"                             , 0                                                  },
	{ 807, "GetTeamId"                             , 0                                                  },
	{ 808, "GetTeam"                               , 0                                                  },
	{ 809, "ReadIniEntry"                          , 0                                                  },
	{ 810, "SetPlotActionSet"                      , 0                                                  },
	{ 811, "CommandFly"                            , 0                                                  },
	{ 812, "CanUseItem"                            , 0                                                  },
	{ 813, "SetCanUseItem"                         , 0                                                  },
	{ 814, "SetControllable"                       , 0                                                  },
	{ 815, "SetPrimaryControlled"                  , 0                                                  },
	{ 816, "GetConditionedAbilities"               , 0                                                  },
	{ 817, "GetPlaceableTreasureCategory"          , 0                                                  },
	{ 818, "GetPlaceableTreasureRank"              , 0                                                  },
	{ 819, "CanUseConditionedAbility"              , 0                                                  },
	{ 820, "Screenshot"                            , 0                                                  },
	{ 821, "GetItemSubItems"                       , 0                                                  },
	{ 822, "InvalidFunction822"                    , 0                                                  },
	{ 823, "InvalidFunction823"                    , 0                                                  },
	{ 824, "SetRoamLocation"                       , 0                                                  },
	{ 825, "SetRoamRadius"                         , 0                                                  },
	{ 826, "GetRoamLocation"                       , 0                                                  },
	{ 827, "GetCreatureBodyBag"                    , 0                                                  },
	{ 828, "GetPRCEnabled"                         , 0                                                  },
	{ 829, "SetFollowerApprovalDescription"        , 0                                                  },
	{ 830, "GetPlayTime"                           , 0                                                  },
	{ 831, "BuyDownload"                           , 0                                                  },
	{ 832, "SaveGamePostCampaign"                  , 0                                                  },
	{ 833, "EffectDamage"                          , 0                                                  },
	{ 834, "EffectImpact"                          , 0                                                  },
	{ 835, "EffectModifyProperty"                  , 0                                                  },
	{ 836, "IsPRCAvailable"                        , 0                                                  },
	{ 837, "LogStoryEvent"                         , 0                                                  },
	{ 838, "GetNameStrref"                         , 0                                                  },
	{ 839, "SetPhysicsController"                  , 0                                                  },
	{ 840, "GetMaxLevel"                           , 0                                                  },
	{ 841, "GetHashedM2DAInt"                      , 0                                                  },
	{ 842, "SetTrainingMode"                       , 0                                                  },
	{ 843, "BeginTrainingStep"                     , 0                                                  },
	{ 844, "SetGameCompletionPercentage"           , 0                                                  },
	{ 845, "SetIndividualImpactAOEEvent"           , 0                                                  },
	{ 846, "CloseQuestGroup"                       , 0                                                  },
	{ 847, "EffectModifyPropertyHostile"           , 0                                                  },
	{ 848, "HighlightGuiElement"                   , 0                                                  },
	{ 849, "SetZoomLevel"                          , 0                                                  },
	{ 850, "RemoveQuest"                           , 0                                                  },
	{ 851, "GetGraphicsDetailLevel"                , 0                                                  },
	{ 852, "OpenJournalPRCTab"                     , 0                                                  },
	{ 853, "GetPlotGUID"                           , 0                                                  },
	{ 854, "ApplyEffectModifyProperty"             , 0                                                  },
	{ 855, "RemoveEffectModifyProperty"            , 0                                                  },
	{ 856, "GetM2DARowIdFromRowIndex"              , 0                                                  },
	{ 857, "IsPRC"                                 , 0                                                  },
	{ 858, "SetPartyPickerStage"                   , 0                                                  },
	{ 859, "GetAbilityList"                        , 0                                                  },
	{ 860, "RequestTarget"                         , 0                                                  },
	{ 861, "SetPlotActionCooldown"                 , 0                                                  },
	{ 862, "IsPackageLoaded"                       , 0                                                  }
};

/** The table defining the signature (return type and type of parameters) of each engine function. */
const Functions::FunctionSignature Functions::kFunctionSignatures[] = {
	{   0, kTypeVoid      , { kTypeInt, kTypeInt } },
	{   1, kTypeVoid      , { kTypeFloat, kTypeInt, kTypeInt, kTypeInt } },
	{   2, kTypeVoid      , { kTypeString, kTypeInt } },
	{   3, kTypeVoid      , { kTypeObject, kTypeInt } },
	{   4, kTypeVoid      , { kTypeVector, kTypeInt } },
	{   5, kTypeVoid      , { kTypeString } },
	{   6, kTypeVoid      , { } },
	{   7, kTypeVoid      , { } },
	{   8, kTypeVoid      , { kTypeString, kTypeObject } },
	{   9, kTypeString    , { kTypeInt, kTypeString, kTypeInt, kTypeString } },
	{  10, kTypeVoid      , { } },
	{  11, kTypeInt       , { kTypeInt, kTypeString } },
	{  12, kTypeInt       , { kTypeInt } },
	{  13, kTypeString    , { kTypeInt, kTypeString, kTypeInt, kTypeString } },
	{  14, kTypeString    , { kTypeInt, kTypeInt } },
	{  15, kTypeInt       , { } },
	{  16, kTypeVoid      , { } },
	{  17, kTypeFloat     , { kTypeObject, kTypeInt } },
	{  18, kTypeVoid      , { kTypeObject, kTypeFloat, kTypeInt } },
	{  19, kTypeEngineType, { kTypeInt } },
	{  20, kTypeEngineType, { kTypeObject } },
	{  21, kTypeVector    , { kTypeObject } },
	{  22, kTypeVoid      , { kTypeObject, kTypeVector, kTypeInt } },
	{  23, kTypeVoid      , { kTypeObject, kTypeVector } },
	{  24, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{  25, kTypeInt       , { kTypeInt } },
	{  26, kTypeInt       , { kTypeInt } },
	{  27, kTypeFloat     , { kTypeFloat } },
	{  28, kTypeFloat     , { kTypeFloat } },
	{  29, kTypeFloat     , { kTypeFloat } },
	{  30, kTypeFloat     , { kTypeFloat } },
	{  31, kTypeFloat     , { kTypeFloat } },
	{  32, kTypeFloat     , { kTypeFloat } },
	{  33, kTypeFloat     , { kTypeFloat } },
	{  34, kTypeFloat     , { kTypeFloat } },
	{  35, kTypeFloat     , { kTypeFloat, kTypeFloat } },
	{  36, kTypeFloat     , { kTypeFloat } },
	{  37, kTypeFloat     , { kTypeObject, kTypeObject } },
	{  38, kTypeFloat     , { kTypeEngineType, kTypeEngineType } },
	{  39, kTypeInt       , { kTypeObject, kTypeObject } },
	{  40, kTypeInt       , { kTypeVector } },
	{  41, kTypeVector    , { kTypeFloat, kTypeFloat, kTypeFloat } },
	{  42, kTypeFloat     , { kTypeVector } },
	{  43, kTypeVector    , { kTypeVector } },
	{  44, kTypeVector    , { kTypeFloat } },
	{  45, kTypeFloat     , { kTypeVector } },
	{  46, kTypeInt       , { kTypeObject, kTypeObject } },
	{  47, kTypeInt       , { kTypeVector, kTypeVector } },
	{  48, kTypeFloat     , { kTypeInt } },
	{  49, kTypeInt       , { kTypeFloat } },
	{  50, kTypeString    , { kTypeInt } },
	{  51, kTypeInt       , { kTypeString } },
	{  52, kTypeString    , { kTypeFloat, kTypeInt, kTypeInt } },
	{  53, kTypeFloat     , { kTypeString } },
	{  54, kTypeString    , { kTypeObject } },
	{  55, kTypeString    , { kTypeVector } },
	{  56, kTypeVector    , { kTypeString } },
	{  57, kTypeString    , { kTypeInt } },
	{  58, kTypeInt       , { kTypeString } },
	{  59, kTypeInt       , { kTypeString } },
	{  60, kTypeString    , { kTypeInt } },
	{  61, kTypeFloat     , { kTypeFloat } },
	{  62, kTypeFloat     , { kTypeFloat } },
	{  63, kTypeInt       , { kTypeObject, kTypeString } },
	{  64, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{  65, kTypeVoid      , { } },
	{  66, kTypeFloat     , { kTypeObject, kTypeString } },
	{  67, kTypeVoid      , { kTypeObject, kTypeString, kTypeFloat } },
	{  68, kTypeVoid      , { } },
	{  69, kTypeString    , { kTypeObject, kTypeString } },
	{  70, kTypeVoid      , { kTypeObject, kTypeString, kTypeString } },
	{  71, kTypeVoid      , { } },
	{  72, kTypeObject    , { kTypeObject, kTypeString } },
	{  73, kTypeVoid      , { kTypeObject, kTypeString, kTypeObject } },
	{  74, kTypeVoid      , { } },
	{  75, kTypeEngineType, { kTypeObject, kTypeString } },
	{  76, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{  77, kTypeVoid      , { } },
	{  78, kTypeEngineType, { kTypeObject, kTypeString } },
	{  79, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{  80, kTypeVoid      , { } },
	{  81, kTypeVoid      , { kTypeString, kTypeInt, kTypeFloat } },
	{  82, kTypeVoid      , { kTypeString, kTypeString } },
	{  83, kTypeObject    , { kTypeObject } },
	{  84, kTypeEngineType, { kTypeObject, kTypeString } },
	{  85, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{  86, kTypeVoid      , { } },
	{  87, kTypeEngineType, { kTypeObject, kTypeString } },
	{  88, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{  89, kTypeVoid      , { } },
	{  90, kTypeEngineType, { kTypeObject, kTypeString } },
	{  91, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{  92, kTypeVoid      , { } },
	{  93, kTypeEngineType, { kTypeObject, kTypeString } },
	{  94, kTypeVoid      , { kTypeObject, kTypeString, kTypeEngineType } },
	{  95, kTypeVoid      , { } },
	{  96, kTypeInt       , { kTypeString } },
	{  97, kTypeString    , { kTypeString } },
	{  98, kTypeString    , { kTypeString } },
	{  99, kTypeString    , { kTypeString, kTypeInt } },
	{ 100, kTypeString    , { kTypeString, kTypeInt } },
	{ 101, kTypeString    , { kTypeString, kTypeString, kTypeInt } },
	{ 102, kTypeString    , { kTypeString, kTypeInt, kTypeInt } },
	{ 103, kTypeInt       , { kTypeString, kTypeString, kTypeInt } },
	{ 104, kTypeInt       , { kTypeString } },
	{ 105, kTypeVoid      , { kTypeObject, kTypeEngineType, kTypeInt } },
	{ 106, kTypeEngineType, { kTypeInt } },
	{ 107, kTypeInt       , { kTypeEngineType } },
	{ 108, kTypeEngineType, { } },
	{ 109, kTypeObject    , { kTypeEngineType } },
	{ 110, kTypeEngineType, { kTypeEngineType, kTypeObject } },
	{ 111, kTypeInt       , { kTypeEngineType } },
	{ 112, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 113, kTypeInt       , { kTypeEngineType, kTypeInt } },
	{ 114, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 115, kTypeFloat     , { kTypeEngineType, kTypeInt } },
	{ 116, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 117, kTypeObject    , { kTypeEngineType, kTypeInt } },
	{ 118, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeObject } },
	{ 119, kTypeString    , { kTypeEngineType, kTypeInt } },
	{ 120, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeString } },
	{ 121, kTypeVoid      , { kTypeEngineType, kTypeString } },
	{ 122, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 123, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 124, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 125, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 126, kTypeObject    , { kTypeEngineType } },
	{ 127, kTypeArray     , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 128, kTypeVoid      , { kTypeFloat, kTypeObject, kTypeEngineType, kTypeString } },
	{ 129, kTypeInt       , { kTypeObject } },
	{ 130, kTypeVoid      , { kTypeString } },
	{ 131, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 132, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeEngineType } },
	{ 133, kTypeVoid      , { kTypeEngineType, kTypeString } },
	{ 134, kTypeFloat     , { kTypeObject } },
	{ 135, kTypeInt       , { kTypeObject } },
	{ 136, kTypeInt       , { kTypeObject } },
	{ 137, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 138, kTypeFloat     , { kTypeInt, kTypeString, kTypeInt, kTypeString } },
	{ 139, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 140, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 141, kTypeInt       , { kTypeObject } },
	{ 142, kTypeVoid      , { } },
	{ 143, kTypeVoid      , { kTypeInt, kTypeString, kTypeObject } },
	{ 144, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 145, kTypeString    , { } },
	{ 146, kTypeInt       , { kTypeObject } },
	{ 147, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 148, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeFloat, kTypeObject, kTypeInt, kTypeInt } },
	{ 149, kTypeString    , { kTypeAny } },
	{ 150, kTypeInt       , { kTypeObject } },
	{ 151, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 152, kTypeInt       , { kTypeObject } },
	{ 153, kTypeInt       , { kTypeObject } },
	{ 154, kTypeInt       , { kTypeObject } },
	{ 155, kTypeInt       , { kTypeObject } },
	{ 156, kTypeString    , { kTypeObject } },
	{ 157, kTypeInt       , { kTypeObject } },
	{ 158, kTypeEngineType, { } },
	{ 159, kTypeEngineType, { } },
	{ 160, kTypeInt       , { kTypeObject } },
	{ 161, kTypeFloat     , { kTypeObject, kTypeObject } },
	{ 162, kTypeEngineType, { kTypeInt } },
	{ 163, kTypeVoid      , { } },
	{ 164, kTypeInt       , { kTypeObject } },
	{ 165, kTypeInt       , { kTypeObject } },
	{ 166, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 167, kTypeInt       , { kTypeObject } },
	{ 168, kTypeObject    , { } },
	{ 169, kTypeArray     , { kTypeObject, kTypeInt } },
	{ 170, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 171, kTypeInt       , { kTypeObject } },
	{ 172, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 173, kTypeVoid      , { kTypeString } },
	{ 174, kTypeVoid      , { kTypeString } },
	{ 175, kTypeVoid      , { kTypeInt, kTypeObject, kTypeObject, kTypeInt, kTypeInt, kTypeString, kTypeString, kTypeString } },
	{ 176, kTypeInt       , { } },
	{ 177, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 178, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 179, kTypeString    , { kTypeInt } },
	{ 180, kTypeArray     , { kTypeObject, kTypeInt, kTypeInt, kTypeString, kTypeInt } },
	{ 181, kTypeInt       , { kTypeString, kTypeInt } },
	{ 182, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 183, kTypeInt       , { kTypeObject } },
	{ 184, kTypeInt       , { kTypeObject } },
	{ 185, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 186, kTypeEngineType, { kTypeEngineType } },
	{ 187, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 188, kTypeVoid      , { kTypeInt } },
	{ 189, kTypeInt       , { } },
	{ 190, kTypeInt       , { kTypeObject } },
	{ 191, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 192, kTypeInt       , { kTypeObject } },
	{ 193, kTypeObject    , { kTypeInt, kTypeVector, kTypeVector, kTypeInt, kTypeInt, kTypeObject } },
	{ 194, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeString } },
	{ 195, kTypeFloat     , { kTypeObject, kTypeInt, kTypeString } },
	{ 196, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 197, kTypeArray     , { kTypeArray, kTypeInt, kTypeInt, kTypeEngineType, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{ 198, kTypeFloat     , { kTypeObject, kTypeObject } },
	{ 199, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 200, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 201, kTypeInt       , { kTypeObject } },
	{ 202, kTypeVoid      , { kTypeObject } },
	{ 203, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 204, kTypeVoid      , { kTypeObject, kTypeObject, kTypeFloat } },
	{ 205, kTypeArray     , { kTypeObject, kTypeFloat, kTypeFloat, kTypeInt, kTypeInt } },
	{ 206, kTypeInt       , { kTypeObject } },
	{ 207, kTypeInt       , { kTypeObject } },
	{ 208, kTypeString    , { kTypeObject } },
	{ 209, kTypeString    , { kTypeObject } },
	{ 210, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 211, kTypeObject    , { kTypeString, kTypeInt } },
	{ 212, kTypeObject    , { kTypeString, kTypeObject, kTypeInt, kTypeString, kTypeInt, kTypeInt } },
	{ 213, kTypeInt       , { kTypeString, kTypeInt } },
	{ 214, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 215, kTypeInt       , { kTypeObject } },
	{ 216, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 217, kTypeInt       , { kTypeObject } },
	{ 218, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 219, kTypeInt       , { kTypeObject } },
	{ 220, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 221, kTypeInt       , { kTypeObject } },
	{ 222, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 223, kTypeObject    , { kTypeObject } },
	{ 224, kTypeObject    , { } },
	{ 225, kTypeVoid      , { } },
	{ 226, kTypeVoid      , { } },
	{ 227, kTypeVoid      , { } },
	{ 228, kTypeVoid      , { } },
	{ 229, kTypeVoid      , { } },
	{ 230, kTypeVoid      , { } },
	{ 231, kTypeVoid      , { } },
	{ 232, kTypeVoid      , { } },
	{ 233, kTypeInt       , { kTypeObject } },
	{ 234, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 235, kTypeInt       , { kTypeObject } },
	{ 236, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 237, kTypeInt       , { kTypeObject } },
	{ 238, kTypeInt       , { kTypeObject } },
	{ 239, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 240, kTypeInt       , { kTypeObject } },
	{ 241, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 242, kTypeInt       , { kTypeObject } },
	{ 243, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 244, kTypeInt       , { kTypeObject } },
	{ 245, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 246, kTypeEngineType, { kTypeObject, kTypeInt } },
	{ 247, kTypeVoid      , { } },
	{ 248, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 249, kTypeVoid      , { } },
	{ 250, kTypeVoid      , { } },
	{ 251, kTypeVoid      , { } },
	{ 252, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 253, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 254, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 255, kTypeVoid      , { } },
	{ 256, kTypeVoid      , { } },
	{ 257, kTypeVoid      , { } },
	{ 258, kTypeVoid      , { } },
	{ 259, kTypeVoid      , { } },
	{ 260, kTypeVoid      , { } },
	{ 261, kTypeVoid      , { } },
	{ 262, kTypeVoid      , { } },
	{ 263, kTypeInt       , { kTypeObject } },
	{ 264, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 265, kTypeVoid      , { } },
	{ 266, kTypeVoid      , { } },
	{ 267, kTypeVoid      , { } },
	{ 268, kTypeVoid      , { } },
	{ 269, kTypeInt       , { kTypeObject } },
	{ 270, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 271, kTypeVoid      , { } },
	{ 272, kTypeVoid      , { } },
	{ 273, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 274, kTypeVoid      , { } },
	{ 275, kTypeString    , { kTypeObject } },
	{ 276, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 277, kTypeVoid      , { } },
	{ 278, kTypeVoid      , { } },
	{ 279, kTypeVoid      , { } },
	{ 280, kTypeVoid      , { } },
	{ 281, kTypeVoid      , { } },
	{ 282, kTypeVoid      , { } },
	{ 283, kTypeVoid      , { } },
	{ 284, kTypeVoid      , { } },
	{ 285, kTypeVoid      , { } },
	{ 286, kTypeInt       , { kTypeObject } },
	{ 287, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 288, kTypeVoid      , { } },
	{ 289, kTypeVoid      , { } },
	{ 290, kTypeVoid      , { } },
	{ 291, kTypeInt       , { kTypeObject } },
	{ 292, kTypeVoid      , { } },
	{ 293, kTypeVoid      , { } },
	{ 294, kTypeVoid      , { } },
	{ 295, kTypeInt       , { kTypeObject } },
	{ 296, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 297, kTypeVoid      , { } },
	{ 298, kTypeVoid      , { } },
	{ 299, kTypeInt       , { kTypeObject } },
	{ 300, kTypeInt       , { kTypeObject } },
	{ 301, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 302, kTypeVoid      , { } },
	{ 303, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 304, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 305, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 306, kTypeObject    , { kTypeInt, kTypeString, kTypeEngineType, kTypeString, kTypeInt, kTypeInt } },
	{ 307, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 308, kTypeString    , { } },
	{ 309, kTypeVoid      , { } },
	{ 310, kTypeVoid      , { kTypeInt } },
	{ 311, kTypeVoid      , { kTypeInt } },
	{ 312, kTypeInt       , { } },
	{ 313, kTypeInt       , { kTypeObject } },
	{ 314, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 315, kTypeInt       , { kTypeObject } },
	{ 316, kTypeArray     , { kTypeObject, kTypeInt } },
	{ 317, kTypeInt       , { kTypeObject } },
	{ 318, kTypeInt       , { kTypeObject } },
	{ 319, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 320, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 321, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 322, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 323, kTypeVoid      , { kTypeInt, kTypeInt, kTypeObject, kTypeInt, kTypeInt } },
	{ 324, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 325, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 326, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 327, kTypeEngineType, { kTypeFloat } },
	{ 328, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 329, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 330, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 331, kTypeInt       , { } },
	{ 332, kTypeVoid      , { } },
	{ 333, kTypeVoid      , { kTypeObject } },
	{ 334, kTypeInt       , { kTypeObject } },
	{ 335, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 336, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 337, kTypeArray     , { kTypeObject } },
	{ 338, kTypeVoid      , { kTypeInt, kTypeString } },
	{ 339, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 340, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 341, kTypeInt       , { } },
	{ 342, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 343, kTypeVoid      , { kTypeObject } },
	{ 344, kTypeInt       , { } },
	{ 345, kTypeVoid      , { kTypeObject } },
	{ 346, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 347, kTypeVoid      , { kTypeObject } },
	{ 348, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 349, kTypeVoid      , { kTypeObject, kTypeObject, kTypeObject } },
	{ 350, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 351, kTypeInt       , { kTypeObject } },
	{ 352, kTypeVoid      , { kTypeObject } },
	{ 353, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 354, kTypeInt       , { kTypeObject } },
	{ 355, kTypeVoid      , { kTypeInt } },
	{ 356, kTypeString    , { kTypeObject } },
	{ 357, kTypeInt       , { kTypeString, kTypeString } },
	{ 358, kTypeInt       , { kTypeObject } },
	{ 359, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 360, kTypeObject    , { kTypeObject, kTypeString, kTypeInt } },
	{ 361, kTypeVoid      , { kTypeString } },
	{ 362, kTypeVoid      , { kTypeInt } },
	{ 363, kTypeInt       , { kTypeObject } },
	{ 364, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 365, kTypeInt       , { kTypeObject } },
	{ 366, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 367, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 368, kTypeInt       , { kTypeObject } },
	{ 369, kTypeInt       , { kTypeObject } },
	{ 370, kTypeObject    , { kTypeInt, kTypeVector, kTypeObject, kTypeInt, kTypeObject } },
	{ 371, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 372, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 373, kTypeObject    , { kTypeObject, kTypeInt } },
	{ 374, kTypeString    , { kTypeObject, kTypeInt } },
	{ 375, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 376, kTypeInt       , { kTypeObject } },
	{ 377, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 378, kTypeInt       , { kTypeObject } },
	{ 379, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 380, kTypeInt       , { kTypeObject, kTypeString } },
	{ 381, kTypeString    , { kTypeString, kTypeInt } },
	{ 382, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 383, kTypeArray     , { kTypeString } },
	{ 384, kTypeInt       , { kTypeObject } },
	{ 385, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 386, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 387, kTypeVoid      , { } },
	{ 388, kTypeInt       , { kTypeObject } },
	{ 389, kTypeArray     , { kTypeObject } },
	{ 390, kTypeVoid      , { } },
	{ 391, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 392, kTypeInt       , { kTypeObject, kTypeObject, kTypeInt, kTypeInt } },
	{ 393, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 394, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 395, kTypeArray     , { } },
	{ 396, kTypeVoid      , { } },
	{ 397, kTypeVoid      , { } },
	{ 398, kTypeVoid      , { } },
	{ 399, kTypeInt       , { kTypeObject } },
	{ 400, kTypeInt       , { kTypeObject } },
	{ 401, kTypeObject    , { kTypeInt, kTypeObject, kTypeInt } },
	{ 402, kTypeVoid      , { } },
	{ 403, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 404, kTypeInt       , { kTypeObject } },
	{ 405, kTypeVoid      , { kTypeInt, kTypeObject } },
	{ 406, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 407, kTypeVoid      , { } },
	{ 408, kTypeInt       , { kTypeObject } },
	{ 409, kTypeVoid      , { kTypeInt, kTypeObject, kTypeInt } },
	{ 410, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt } },
	{ 411, kTypeInt       , { kTypeObject } },
	{ 412, kTypeVoid      , { } },
	{ 413, kTypeInt       , { kTypeObject } },
	{ 414, kTypeVoid      , { } },
	{ 415, kTypeVoid      , { } },
	{ 416, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 417, kTypeVoid      , { } },
	{ 418, kTypeVoid      , { } },
	{ 419, kTypeVoid      , { } },
	{ 420, kTypeVoid      , { } },
	{ 421, kTypeVoid      , { } },
	{ 422, kTypeVoid      , { } },
	{ 423, kTypeVoid      , { } },
	{ 424, kTypeVoid      , { } },
	{ 425, kTypeVoid      , { } },
	{ 426, kTypeVoid      , { } },
	{ 427, kTypeVoid      , { } },
	{ 428, kTypeVoid      , { } },
	{ 429, kTypeVoid      , { } },
	{ 430, kTypeVoid      , { } },
	{ 431, kTypeVoid      , { } },
	{ 432, kTypeVoid      , { } },
	{ 433, kTypeArray     , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 434, kTypeArray     , { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 435, kTypeArray     , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 436, kTypeArray     , { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 437, kTypeVoid      , { } },
	{ 438, kTypeObject    , { } },
	{ 439, kTypeVoid      , { } },
	{ 440, kTypeInt       , { kTypeObject, kTypeObject, kTypeString } },
	{ 441, kTypeVoid      , { } },
	{ 442, kTypeVoid      , { } },
	{ 443, kTypeInt       , { } },
	{ 444, kTypeInt       , { kTypeObject } },
	{ 445, kTypeEngineType, { kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 446, kTypeVoid      , { kTypeObject } },
	{ 447, kTypeVoid      , { kTypeObject } },
	{ 448, kTypeInt       , { kTypeObject } },
	{ 449, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 450, kTypeInt       , { kTypeObject } },
	{ 451, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 452, kTypeInt       , { kTypeObject } },
	{ 453, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 454, kTypeEngineType, { kTypeObject, kTypeVector, kTypeFloat } },
	{ 455, kTypeInt       , { kTypeEngineType } },
	{ 456, kTypeInt       , { kTypeEngineType } },
	{ 457, kTypeEngineType, { kTypeObject } },
	{ 458, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 459, kTypeVector    , { kTypeEngineType } },
	{ 460, kTypeEngineType, { kTypeEngineType, kTypeVector } },
	{ 461, kTypeObject    , { kTypeEngineType } },
	{ 462, kTypeEngineType, { kTypeEngineType, kTypeObject } },
	{ 463, kTypeFloat     , { kTypeEngineType } },
	{ 464, kTypeVector    , { kTypeEngineType } },
	{ 465, kTypeEngineType, { kTypeEngineType, kTypeVector } },
	{ 466, kTypeInt       , { kTypeInt, kTypeInt } },
	{ 467, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 468, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 469, kTypeInt       , { kTypeObject } },
	{ 470, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 471, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 472, kTypeVoid      , { } },
	{ 473, kTypeVoid      , { } },
	{ 474, kTypeVoid      , { } },
	{ 475, kTypeVoid      , { } },
	{ 476, kTypeVoid      , { } },
	{ 477, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 478, kTypeVoid      , { kTypeString } },
	{ 479, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat } },
	{ 480, kTypeVoid      , { kTypeObject } },
	{ 481, kTypeVoid      , { kTypeObject } },
	{ 482, kTypeVoid      , { } },
	{ 483, kTypeVoid      , { } },
	{ 484, kTypeVoid      , { kTypeString } },
	{ 485, kTypeVoid      , { kTypeString } },
	{ 486, kTypeVoid      , { kTypeInt } },
	{ 487, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 488, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 489, kTypeVoid      , { } },
	{ 490, kTypeVoid      , { } },
	{ 491, kTypeVoid      , { } },
	{ 492, kTypeVoid      , { } },
	{ 493, kTypeVoid      , { } },
	{ 494, kTypeVoid      , { } },
	{ 495, kTypeVoid      , { } },
	{ 496, kTypeVoid      , { } },
	{ 497, kTypeVoid      , { } },
	{ 498, kTypeVoid      , { } },
	{ 499, kTypeVoid      , { } },
	{ 500, kTypeInt       , { kTypeObject } },
	{ 501, kTypeVoid      , { } },
	{ 502, kTypeObject    , { } },
	{ 503, kTypeVoid      , { } },
	{ 504, kTypeArray     , { kTypeObject } },
	{ 505, kTypeVoid      , { } },
	{ 506, kTypeVoid      , { } },
	{ 507, kTypeVoid      , { } },
	{ 508, kTypeVoid      , { } },
	{ 509, kTypeVoid      , { } },
	{ 510, kTypeVoid      , { } },
	{ 511, kTypeVoid      , { } },
	{ 512, kTypeVoid      , { } },
	{ 513, kTypeVoid      , { } },
	{ 514, kTypeVoid      , { } },
	{ 515, kTypeVoid      , { } },
	{ 516, kTypeVoid      , { } },
	{ 517, kTypeVoid      , { } },
	{ 518, kTypeVoid      , { } },
	{ 519, kTypeVoid      , { } },
	{ 520, kTypeVoid      , { } },
	{ 521, kTypeVoid      , { } },
	{ 522, kTypeObject    , { kTypeObject } },
	{ 523, kTypeVoid      , { } },
	{ 524, kTypeVoid      , { } },
	{ 525, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 526, kTypeInt       , { kTypeObject } },
	{ 527, kTypeVoid      , { } },
	{ 528, kTypeVoid      , { } },
	{ 529, kTypeInt       , { } },
	{ 530, kTypeVoid      , { kTypeInt } },
	{ 531, kTypeVoid      , { } },
	{ 532, kTypeVoid      , { } },
	{ 533, kTypeInt       , { kTypeObject, kTypeEngineType, kTypeInt, kTypeInt, kTypeInt } },
	{ 534, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 535, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 536, kTypeEngineType, { kTypeObject, kTypeInt } },
	{ 537, kTypeVoid      , { } },
	{ 538, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 539, kTypeEngineType, { kTypeObject } },
	{ 540, kTypeVoid      , { kTypeObject } },
	{ 541, kTypeInt       , { kTypeEngineType } },
	{ 542, kTypeVoid      , { } },
	{ 543, kTypeVoid      , { } },
	{ 544, kTypeInt       , { kTypeEngineType, kTypeInt } },
	{ 545, kTypeVoid      , { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 546, kTypeFloat     , { kTypeEngineType, kTypeInt } },
	{ 547, kTypeEngineType, { kTypeEngineType, kTypeFloat, kTypeInt } },
	{ 548, kTypeVoid      , { } },
	{ 549, kTypeVoid      , { } },
	{ 550, kTypeObject    , { kTypeEngineType, kTypeInt } },
	{ 551, kTypeEngineType, { kTypeEngineType, kTypeObject, kTypeInt } },
	{ 552, kTypeVoid      , { } },
	{ 553, kTypeVoid      , { } },
	{ 554, kTypeVoid      , { } },
	{ 555, kTypeVoid      , { } },
	{ 556, kTypeVoid      , { } },
	{ 557, kTypeVoid      , { } },
	{ 558, kTypeVoid      , { } },
	{ 559, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 560, kTypeEngineType, { kTypeObject, kTypeInt, kTypeFloat, kTypeInt, kTypeFloat } },
	{ 561, kTypeEngineType, { kTypeObject, kTypeFloat, kTypeInt } },
	{ 562, kTypeEngineType, { kTypeObject, kTypeInt, kTypeInt } },
	{ 563, kTypeEngineType, { kTypeObject } },
	{ 564, kTypeVoid      , { } },
	{ 565, kTypeVoid      , { } },
	{ 566, kTypeEngineType, { kTypeObject, kTypeInt } },
	{ 567, kTypeVoid      , { } },
	{ 568, kTypeVoid      , { } },
	{ 569, kTypeEngineType, { kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 570, kTypeVoid      , { } },
	{ 571, kTypeVoid      , { } },
	{ 572, kTypeVoid      , { } },
	{ 573, kTypeVoid      , { } },
	{ 574, kTypeVoid      , { } },
	{ 575, kTypeVoid      , { } },
	{ 576, kTypeVoid      , { } },
	{ 577, kTypeVoid      , { } },
	{ 578, kTypeEngineType, { kTypeFloat } },
	{ 579, kTypeEngineType, { kTypeObject, kTypeString } },
	{ 580, kTypeEngineType, { kTypeObject } },
	{ 581, kTypeEngineType, { kTypeEngineType } },
	{ 582, kTypeEngineType, { kTypeInt, kTypeObject, kTypeVector, kTypeFloat, kTypeString } },
	{ 583, kTypeVoid      , { } },
	{ 584, kTypeVoid      , { } },
	{ 585, kTypeVoid      , { } },
	{ 586, kTypeEngineType, { kTypeObject, kTypeInt } },
	{ 587, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 588, kTypeVoid      , { } },
	{ 589, kTypeVoid      , { } },
	{ 590, kTypeVoid      , { } },
	{ 591, kTypeVoid      , { } },
	{ 592, kTypeVoid      , { } },
	{ 593, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeObject } },
	{ 594, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 595, kTypeInt       , { kTypeEngineType } },
	{ 596, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeEngineType, kTypeFloat, kTypeObject, kTypeInt } },
	{ 597, kTypeVoid      , { kTypeInt, kTypeEngineType, kTypeObject, kTypeFloat, kTypeObject, kTypeInt } },
	{ 598, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 599, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeString } },
	{ 600, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 601, kTypeInt       , { } },
	{ 602, kTypeInt       , { } },
	{ 603, kTypeInt       , { kTypeInt } },
	{ 604, kTypeInt       , { kTypeEngineType } },
	{ 605, kTypeInt       , { kTypeEngineType } },
	{ 606, kTypeVector    , { kTypeObject, kTypeInt } },
	{ 607, kTypeInt       , { kTypeEngineType } },
	{ 608, kTypeInt       , { kTypeEngineType } },
	{ 609, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 610, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 611, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 612, kTypeObject    , { kTypeEngineType } },
	{ 613, kTypeEngineType, { kTypeEngineType, kTypeObject } },
	{ 614, kTypeInt       , { kTypeEngineType } },
	{ 615, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 616, kTypeVoid      , { kTypeInt } },
	{ 617, kTypeInt       , { kTypeInt, kTypeInt } },
	{ 618, kTypeInt       , { kTypeInt } },
	{ 619, kTypeVoid      , { } },
	{ 620, kTypeInt       , { kTypeEngineType } },
	{ 621, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 622, kTypeInt       , { kTypeObject } },
	{ 623, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 624, kTypeObject    , { } },
	{ 625, kTypeVoid      , { kTypeObject } },
	{ 626, kTypeInt       , { kTypeObject } },
	{ 627, kTypeVoid      , { } },
	{ 628, kTypeInt       , { kTypeEngineType } },
	{ 629, kTypeInt       , { kTypeString } },
	{ 630, kTypeVoid      , { kTypeInt, kTypeInt, kTypeString } },
	{ 631, kTypeVoid      , { kTypeString, kTypeString, kTypeString } },
	{ 632, kTypeVoid      , { kTypeString } },
	{ 633, kTypeInt       , { kTypeObject } },
	{ 634, kTypeEngineType, { kTypeInt, kTypeString, kTypeInt } },
	{ 635, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeInt, kTypeFloat, kTypeInt } },
	{ 636, kTypeVoid      , { } },
	{ 637, kTypeInt       , { kTypeObject } },
	{ 638, kTypeVoid      , { kTypeString } },
	{ 639, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt, kTypeFloat, kTypeString } },
	{ 640, kTypeFloat     , { kTypeString, kTypeInt } },
	{ 641, kTypeInt       , { kTypeObject } },
	{ 642, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 643, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 644, kTypeEngineType, { kTypeArray, kTypeInt, kTypeInt, kTypeInt } },
	{ 645, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 646, kTypeInt       , { kTypeObject } },
	{ 647, kTypeVoid      , { kTypeString, kTypeObject, kTypeInt, kTypeArray, kTypeArray, kTypeInt } },
	{ 648, kTypeVoid      , { kTypeObject } },
	{ 649, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 650, kTypeInt       , { kTypeObject } },
	{ 651, kTypeObject    , { kTypeObject, kTypeString, kTypeInt } },
	{ 652, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 653, kTypeInt       , { kTypeObject } },
	{ 654, kTypeObject    , { kTypeObject, kTypeString, kTypeInt } },
	{ 655, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 656, kTypeVector    , { kTypeObject } },
	{ 657, kTypeVoid      , { kTypeObject, kTypeVector } },
	{ 658, kTypeEngineType, { kTypeEngineType, kTypeFloat } },
	{ 659, kTypeEngineType, { kTypeEngineType } },
	{ 660, kTypeInt       , { kTypeObject, kTypeString, kTypeInt, kTypeInt } },
	{ 661, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeInt } },
	{ 662, kTypeVoid      , { } },
	{ 663, kTypeVoid      , { } },
	{ 664, kTypeVoid      , { } },
	{ 665, kTypeInt       , { kTypeString } },
	{ 666, kTypeVoid      , { } },
	{ 667, kTypeInt       , { kTypeInt, kTypeString, kTypeInt, kTypeString } },
	{ 668, kTypeVoid      , { } },
	{ 669, kTypeVoid      , { kTypeObject } },
	{ 670, kTypeVoid      , { } },
	{ 671, kTypeVoid      , { } },
	{ 672, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 673, kTypeArray     , { kTypeInt, kTypeInt, kTypeEngineType, kTypeFloat, kTypeFloat, kTypeFloat, kTypeInt } },
	{ 674, kTypeVoid      , { kTypeString } },
	{ 675, kTypeObject    , { kTypeObject, kTypeString } },
	{ 676, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 677, kTypeInt       , { kTypeObject } },
	{ 678, kTypeVoid      , { } },
	{ 679, kTypeVoid      , { } },
	{ 680, kTypeVoid      , { kTypeString, kTypeFloat } },
	{ 681, kTypeEngineType, { } },
	{ 682, kTypeVoid      , { kTypeInt } },
	{ 683, kTypeInt       , { kTypeEngineType, kTypeInt } },
	{ 684, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeInt } },
	{ 685, kTypeFloat     , { kTypeEngineType, kTypeInt } },
	{ 686, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 687, kTypeObject    , { kTypeEngineType, kTypeInt } },
	{ 688, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeObject } },
	{ 689, kTypeString    , { kTypeEngineType, kTypeInt } },
	{ 690, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeString } },
	{ 691, kTypeVoid      , { kTypeObject, kTypeInt, kTypeEngineType, kTypeInt, kTypeEngineType } },
	{ 692, kTypeVoid      , { } },
	{ 693, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 694, kTypeInt       , { } },
	{ 695, kTypeEngineType, { kTypeObject, kTypeInt } },
	{ 696, kTypeVoid      , { kTypeString, kTypeString, kTypeObject } },
	{ 697, kTypeVoid      , { } },
	{ 698, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 699, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 700, kTypeInt       , { kTypeObject } },
	{ 701, kTypeFloat     , { kTypeObject, kTypeInt } },
	{ 702, kTypeVoid      , { kTypeInt } },
	{ 703, kTypeFloat     , { kTypeObject, kTypeObject, kTypeInt } },
	{ 704, kTypeString    , { kTypeString } },
	{ 705, kTypeInt       , { } },
	{ 706, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 707, kTypeArray     , { kTypeObject, kTypeString } },
	{ 708, kTypeInt       , { kTypeAny } },
	{ 709, kTypeVoid      , { } },
	{ 710, kTypeVoid      , { } },
	{ 711, kTypeVoid      , { } },
	{ 712, kTypeArray     , { kTypeObject, kTypeInt, kTypeInt, kTypeObject, kTypeInt, kTypeInt } },
	{ 713, kTypeArray     , { kTypeObject, kTypeInt, kTypeInt } },
	{ 714, kTypeInt       , { kTypeObject } },
	{ 715, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 716, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 717, kTypeVoid      , { } },
	{ 718, kTypeVoid      , { } },
	{ 719, kTypeFloat     , { kTypeObject } },
	{ 720, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 721, kTypeVoid      , { } },
	{ 722, kTypeVoid      , { kTypeInt } },
	{ 723, kTypeVoid      , { kTypeString, kTypeString } },
	{ 724, kTypeInt       , { kTypeObject } },
	{ 725, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 726, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 727, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeVector } },
	{ 728, kTypeEngineType, { kTypeObject } },
	{ 729, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 730, kTypeInt       , { kTypeEngineType } },
	{ 731, kTypeInt       , { kTypeObject } },
	{ 732, kTypeString    , { kTypeString } },
	{ 733, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 734, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeString } },
	{ 735, kTypeString    , { kTypeEngineType, kTypeInt } },
	{ 736, kTypeString    , { kTypeObject, kTypeString } },
	{ 737, kTypeVoid      , { kTypeObject, kTypeString, kTypeString } },
	{ 738, kTypeFloat     , { kTypeObject, kTypeInt, kTypeInt } },
	{ 739, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 740, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeInt } },
	{ 741, kTypeVoid      , { kTypeObject, kTypeInt, kTypeFloat, kTypeInt } },
	{ 742, kTypeString    , { kTypeString } },
	{ 743, kTypeString    , { kTypeString, kTypeInt } },
	{ 744, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt, kTypeInt, kTypeFloat } },
	{ 745, kTypeInt       , { kTypeString, kTypeString } },
	{ 746, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 747, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 748, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt, kTypeString } },
	{ 749, kTypeVoid      , { kTypeObject, kTypeString } },
	{ 750, kTypeVoid      , { kTypeString } },
	{ 751, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 752, kTypeVoid      , { kTypeInt } },
	{ 753, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 754, kTypeVoid      , { kTypeObject, kTypeObject } },
	{ 755, kTypeVoid      , { kTypeObject } },
	{ 756, kTypeVoid      , { kTypeObject } },
	{ 757, kTypeVoid      , { } },
	{ 758, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 759, kTypeInt       , { kTypeObject } },
	{ 760, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 761, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeFloat } },
	{ 762, kTypeEngineType, { kTypeEngineType, kTypeInt, kTypeObject } },
	{ 763, kTypeInt       , { kTypeObject } },
	{ 764, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 765, kTypeInt       , { kTypeObject } },
	{ 766, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 767, kTypeFloat     , { } },
	{ 768, kTypeInt       , { kTypeObject } },
	{ 769, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 770, kTypeVoid      , { kTypeObject, kTypeString, kTypeInt } },
	{ 771, kTypeVoid      , { kTypeInt } },
	{ 772, kTypeInt       , { kTypeObject } },
	{ 773, kTypeVoid      , { kTypeObject, kTypeInt, kTypeInt } },
	{ 774, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 775, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 776, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 777, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 778, kTypeInt       , { kTypeObject, kTypeInt } },
	{ 779, kTypeVoid      , { kTypeInt } },
	{ 780, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 781, kTypeArray     , { kTypeObject } },
	{ 782, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 783, kTypeVoid      , { kTypeObject } },
	{ 784, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 785, kTypeVoid      , { } },
	{ 786, kTypeVoid      , { } },
	{ 787, kTypeInt       , { kTypeObject } },
	{ 788, kTypeVoid      , { } },
	{ 789, kTypeVoid      , { } },
	{ 790, kTypeVoid      , { } },
	{ 791, kTypeVoid      , { kTypeString, kTypeInt, kTypeInt, kTypeInt, kTypeInt, kTypeInt } },
	{ 792, kTypeVoid      , { kTypeInt, kTypeString, kTypeInt, kTypeInt } },
	{ 793, kTypeVoid      , { kTypeInt, kTypeString, kTypeInt, kTypeInt } },
	{ 794, kTypeVoid      , { } },
	{ 795, kTypeVoid      , { kTypeInt } },
	{ 796, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 797, kTypeInt       , { kTypeInt } },
	{ 798, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 799, kTypeInt       , { kTypeInt } },
	{ 800, kTypeVoid      , { kTypeInt, kTypeFloat } },
	{ 801, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat, kTypeFloat } },
	{ 802, kTypeVoid      , { } },
	{ 803, kTypeVoid      , { kTypeString, kTypeString, kTypeFloat } },
	{ 804, kTypeVoid      , { kTypeString, kTypeInt } },
	{ 805, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 806, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 807, kTypeInt       , { kTypeObject } },
	{ 808, kTypeArray     , { kTypeInt, kTypeInt } },
	{ 809, kTypeString    , { kTypeString, kTypeString } },
	{ 810, kTypeVoid      , { kTypeInt } },
	{ 811, kTypeEngineType, { kTypeEngineType, kTypeInt } },
	{ 812, kTypeInt       , { kTypeObject, kTypeObject } },
	{ 813, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 814, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 815, kTypeVoid      , { kTypeObject } },
	{ 816, kTypeArray     , { kTypeObject, kTypeInt } },
	{ 817, kTypeInt       , { kTypeObject } },
	{ 818, kTypeInt       , { kTypeObject } },
	{ 819, kTypeInt       , { kTypeObject, kTypeInt, kTypeInt } },
	{ 820, kTypeVoid      , { kTypeString, kTypeString, kTypeInt } },
	{ 821, kTypeArray     , { kTypeObject } },
	{ 822, kTypeVoid      , { } },
	{ 823, kTypeVoid      , { } },
	{ 824, kTypeVoid      , { kTypeObject, kTypeEngineType } },
	{ 825, kTypeVoid      , { kTypeObject, kTypeFloat } },
	{ 826, kTypeEngineType, { kTypeObject } },
	{ 827, kTypeObject    , { kTypeObject } },
	{ 828, kTypeInt       , { kTypeString } },
	{ 829, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 830, kTypeInt       , { } },
	{ 831, kTypeInt       , { kTypeString } },
	{ 832, kTypeVoid      , { } },
	{ 833, kTypeEngineType, { kTypeFloat, kTypeInt, kTypeInt, kTypeInt } },
	{ 834, kTypeEngineType, { kTypeFloat, kTypeObject, kTypeInt, kTypeInt, kTypeInt } },
	{ 835, kTypeEngineType, { kTypeInt, kTypeFloat, kTypeInt, kTypeFloat, kTypeInt, kTypeFloat } },
	{ 836, kTypeInt       , { } },
	{ 837, kTypeVoid      , { kTypeInt } },
	{ 838, kTypeInt       , { kTypeObject } },
	{ 839, kTypeVoid      , { kTypeObject, kTypeInt } },
	{ 840, kTypeInt       , { } },
	{ 841, kTypeInt       , { kTypeInt, kTypeInt, kTypeInt } },
	{ 842, kTypeVoid      , { kTypeInt } },
	{ 843, kTypeVoid      , { kTypeInt, kTypeInt, kTypeInt } },
	{ 844, kTypeVoid      , { kTypeFloat } },
	{ 845, kTypeVoid      , { kTypeObject, kTypeObject, kTypeInt, kTypeEngineType, kTypeInt } },
	{ 846, kTypeVoid      , { kTypeString } },
	{ 847, kTypeEngineType, { kTypeInt, kTypeFloat, kTypeInt, kTypeFloat, kTypeInt, kTypeFloat } },
	{ 848, kTypeVoid      , { kTypeInt, kTypeInt } },
	{ 849, kTypeVoid      , { kTypeFloat } },
	{ 850, kTypeVoid      , { kTypeString } },
	{ 851, kTypeInt       , { } },
	{ 852, kTypeVoid      , { kTypeString } },
	{ 853, kTypeString    , { kTypeString } },
	{ 854, kTypeVoid      , { kTypeEngineType } },
	{ 855, kTypeVoid      , { kTypeEngineType } },
	{ 856, kTypeInt       , { kTypeInt, kTypeInt, kTypeString } },
	{ 857, kTypeInt       , { kTypeString, kTypeInt } },
	{ 858, kTypeVoid      , { kTypeString, kTypeString } },
	{ 859, kTypeArray     , { kTypeObject, kTypeInt, kTypeInt } },
	{ 860, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeFloat, kTypeInt, kTypeObject, kTypeString } },
	{ 861, kTypeVoid      , { kTypeInt, kTypeFloat, kTypeInt, kTypeObject, kTypeString } },
	{ 862, kTypeInt       , { kTypeString } },
};

/** The table defining the default values for the parameters of each engine function. */
const Functions::FunctionDefaults Functions::kFunctionDefaults[] = {
	{   0, { &kDefaultFalse } },
	{   1, { &kDefaultInt18, &kDefaultInt9, &kDefaultFalse } },
	{   2, { &kDefaultFalse } },
	{   3, { &kDefaultFalse } },
	{   4, { &kDefaultFalse } },
	{   5, { } },
	{   6, { } },
	{   7, { } },
	{   8, { } },
	{   9, { &kDefaultStringEmpty } },
	{  10, { } },
	{  11, { &kDefaultStringEmpty } },
	{  12, { } },
	{  13, { &kDefaultStringEmpty } },
	{  14, { &kDefaultGenderMale } },
	{  15, { } },
	{  16, { } },
	{  17, { &kDefaultTrue } },
	{  18, { &kDefaultTrue } },
	{  19, { &kDefaultEffectTypeInvalid } },
	{  20, { } },
	{  21, { } },
	{  22, { &kDefaultTrue } },
	{  23, { } },
	{  24, { &kDefaultTrue } },
	{  25, { } },
	{  26, { } },
	{  27, { } },
	{  28, { } },
	{  29, { } },
	{  30, { } },
	{  31, { } },
	{  32, { } },
	{  33, { } },
	{  34, { } },
	{  35, { } },
	{  36, { } },
	{  37, { } },
	{  38, { } },
	{  39, { } },
	{  40, { } },
	{  41, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFloat0_0 } },
	{  42, { } },
	{  43, { } },
	{  44, { } },
	{  45, { } },
	{  46, { } },
	{  47, { } },
	{  48, { } },
	{  49, { } },
	{  50, { } },
	{  51, { } },
	{  52, { &kDefaultInt18, &kDefaultInt9 } },
	{  53, { } },
	{  54, { } },
	{  55, { } },
	{  56, { } },
	{  57, { } },
	{  58, { } },
	{  59, { } },
	{  60, { } },
	{  61, { } },
	{  62, { } },
	{  63, { } },
	{  64, { } },
	{  65, { } },
	{  66, { } },
	{  67, { } },
	{  68, { } },
	{  69, { } },
	{  70, { } },
	{  71, { } },
	{  72, { } },
	{  73, { } },
	{  74, { } },
	{  75, { } },
	{  76, { } },
	{  77, { } },
	{  78, { } },
	{  79, { } },
	{  80, { } },
	{  81, { &kDefaultInt10, &kDefaultFloat10_0 } },
	{  82, { &kDefaultStringLogWindow } },
	{  83, { } },
	{  84, { } },
	{  85, { } },
	{  86, { } },
	{  87, { } },
	{  88, { } },
	{  89, { } },
	{  90, { } },
	{  91, { } },
	{  92, { } },
	{  93, { } },
	{  94, { } },
	{  95, { } },
	{  96, { } },
	{  97, { } },
	{  98, { } },
	{  99, { } },
	{ 100, { } },
	{ 101, { } },
	{ 102, { } },
	{ 103, { &kDefaultInt0 } },
	{ 104, { } },
	{ 105, { &kDefaultInt0 } },
	{ 106, { } },
	{ 107, { } },
	{ 108, { } },
	{ 109, { } },
	{ 110, { } },
	{ 111, { } },
	{ 112, { } },
	{ 113, { } },
	{ 114, { } },
	{ 115, { } },
	{ 116, { } },
	{ 117, { } },
	{ 118, { } },
	{ 119, { } },
	{ 120, { } },
	{ 121, { &kDefaultStringEmpty } },
	{ 122, { } },
	{ 123, { } },
	{ 124, { } },
	{ 125, { } },
	{ 126, { } },
	{ 127, { &kDefaultObjectTypeAll, &kDefaultInt1, &kDefaultInt0, &kDefaultInt0, &kDefaultInt0 } },
	{ 128, { &kDefaultStringEmpty } },
	{ 129, { } },
	{ 130, { } },
	{ 131, { } },
	{ 132, { } },
	{ 133, { &kDefaultStringEmpty } },
	{ 134, { } },
	{ 135, { } },
	{ 136, { } },
	{ 137, { } },
	{ 138, { &kDefaultStringEmpty } },
	{ 139, { &kDefaultAbilityInvalid } },
	{ 140, { &kDefaultInt0, &kDefaultInt0 } },
	{ 141, { } },
	{ 142, { } },
	{ 143, { &kDefaultLogChannelGeneral, &kDefaultStringEmpty, &kDefaultObjectInvalid } },
	{ 144, { } },
	{ 145, { } },
	{ 146, { } },
	{ 147, { } },
	{ 148, { &kDefaultFloat0_0, &kDefaultObjectSelf, &kDefaultInt0, &kDefaultFalse } },
	{ 149, { } },
	{ 150, { } },
	{ 151, { &kDefaultFalse } },
	{ 152, { } },
	{ 153, { } },
	{ 154, { } },
	{ 155, { } },
	{ 156, { } },
	{ 157, { } },
	{ 158, { } },
	{ 159, { } },
	{ 160, { } },
	{ 161, { } },
	{ 162, { &kDefaultInvalidWeaponSet } },
	{ 163, { } },
	{ 164, { } },
	{ 165, { } },
	{ 166, { } },
	{ 167, { } },
	{ 168, { } },
	{ 169, { &kDefaultInt0 } },
	{ 170, { } },
	{ 171, { } },
	{ 172, { } },
	{ 173, { } },
	{ 174, { } },
	{ 175, { &kDefaultObjectSelf, &kDefaultObjectInvalid, &kDefaultInt0, &kDefaultInt0, &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultStringxE } },
	{ 176, { } },
	{ 177, { &kDefaultEffectTypeInvalid, &kDefaultIntMinus1 } },
	{ 178, { &kDefaultFalse } },
	{ 179, { } },
	{ 180, { &kDefaultGetItemsOptionAll, &kDefaultInt0, &kDefaultStringEmpty, &kDefaultFalse } },
	{ 181, { } },
	{ 182, { } },
	{ 183, { } },
	{ 184, { } },
	{ 185, { } },
	{ 186, { } },
	{ 187, { } },
	{ 188, { &kDefaultTrue } },
	{ 189, { } },
	{ 190, { } },
	{ 191, { &kDefaultInt1 } },
	{ 192, { } },
	{ 193, { &kDefaultInt0, &kDefaultFalse, &kDefaultObjectInvalid } },
	{ 194, { &kDefaultStringEmpty } },
	{ 195, { &kDefaultStringEmpty } },
	{ 196, { &kDefaultFalse } },
	{ 197, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFalse } },
	{ 198, { } },
	{ 199, { } },
	{ 200, { } },
	{ 201, { } },
	{ 202, { } },
	{ 203, { } },
	{ 204, { } },
	{ 205, { &kDefaultFalse, &kDefaultInt0 } },
	{ 206, { } },
	{ 207, { } },
	{ 208, { } },
	{ 209, { } },
	{ 210, { } },
	{ 211, { &kDefaultInt0 } },
	{ 212, { &kDefaultInt1, &kDefaultStringEmpty, &kDefaultFalse, &kDefaultTrue } },
	{ 213, { } },
	{ 214, { &kDefaultInt0 } },
	{ 215, { } },
	{ 216, { } },
	{ 217, { } },
	{ 218, { } },
	{ 219, { } },
	{ 220, { } },
	{ 221, { } },
	{ 222, { } },
	{ 223, { } },
	{ 224, { } },
	{ 225, { } },
	{ 226, { } },
	{ 227, { } },
	{ 228, { } },
	{ 229, { } },
	{ 230, { } },
	{ 231, { } },
	{ 232, { } },
	{ 233, { } },
	{ 234, { } },
	{ 235, { } },
	{ 236, { } },
	{ 237, { &kDefaultObjectSelf } },
	{ 238, { } },
	{ 239, { } },
	{ 240, { } },
	{ 241, { } },
	{ 242, { } },
	{ 243, { } },
	{ 244, { } },
	{ 245, { } },
	{ 246, { &kDefaultInt0 } },
	{ 247, { } },
	{ 248, { &kDefaultPropertyValueTotal } },
	{ 249, { } },
	{ 250, { } },
	{ 251, { } },
	{ 252, { } },
	{ 253, { &kDefaultFalse } },
	{ 254, { } },
	{ 255, { } },
	{ 256, { } },
	{ 257, { } },
	{ 258, { } },
	{ 259, { } },
	{ 260, { } },
	{ 261, { } },
	{ 262, { } },
	{ 263, { } },
	{ 264, { } },
	{ 265, { } },
	{ 266, { } },
	{ 267, { } },
	{ 268, { } },
	{ 269, { } },
	{ 270, { } },
	{ 271, { } },
	{ 272, { } },
	{ 273, { &kDefaultInvalidWeaponSet } },
	{ 274, { } },
	{ 275, { } },
	{ 276, { } },
	{ 277, { } },
	{ 278, { } },
	{ 279, { } },
	{ 280, { } },
	{ 281, { } },
	{ 282, { } },
	{ 283, { } },
	{ 284, { } },
	{ 285, { } },
	{ 286, { } },
	{ 287, { } },
	{ 288, { } },
	{ 289, { } },
	{ 290, { } },
	{ 291, { } },
	{ 292, { } },
	{ 293, { } },
	{ 294, { } },
	{ 295, { } },
	{ 296, { } },
	{ 297, { } },
	{ 298, { } },
	{ 299, { } },
	{ 300, { } },
	{ 301, { } },
	{ 302, { } },
	{ 303, { } },
	{ 304, { } },
	{ 305, { } },
	{ 306, { &kDefaultStringEmpty, &kDefaultTrue, &kDefaultFalse } },
	{ 307, { } },
	{ 308, { } },
	{ 309, { } },
	{ 310, { } },
	{ 311, { } },
	{ 312, { } },
	{ 313, { } },
	{ 314, { } },
	{ 315, { } },
	{ 316, { } },
	{ 317, { } },
	{ 318, { } },
	{ 319, { &kDefaultInt0 } },
	{ 320, { &kDefaultInt0 } },
	{ 321, { &kDefaultInt205 } },
	{ 322, { &kDefaultInt206 } },
	{ 323, { &kDefaultObjectInvalid, &kDefaultFalse, &kDefaultInt0 } },
	{ 324, { &kDefaultFalse } },
	{ 325, { } },
	{ 326, { &kDefaultObjectSelf, &kDefaultFalse } },
	{ 327, { } },
	{ 328, { } },
	{ 329, { } },
	{ 330, { } },
	{ 331, { } },
	{ 332, { } },
	{ 333, { } },
	{ 334, { } },
	{ 335, { } },
	{ 336, { } },
	{ 337, { } },
	{ 338, { } },
	{ 339, { } },
	{ 340, { &kDefaultFalse } },
	{ 341, { } },
	{ 342, { } },
	{ 343, { } },
	{ 344, { } },
	{ 345, { } },
	{ 346, { } },
	{ 347, { } },
	{ 348, { } },
	{ 349, { } },
	{ 350, { } },
	{ 351, { } },
	{ 352, { } },
	{ 353, { } },
	{ 354, { } },
	{ 355, { } },
	{ 356, { } },
	{ 357, { } },
	{ 358, { } },
	{ 359, { } },
	{ 360, { &kDefaultInt0 } },
	{ 361, { } },
	{ 362, { &kDefaultFalse } },
	{ 363, { } },
	{ 364, { } },
	{ 365, { } },
	{ 366, { } },
	{ 367, { } },
	{ 368, { } },
	{ 369, { } },
	{ 370, { &kDefaultInt0, &kDefaultObjectInvalid } },
	{ 371, { } },
	{ 372, { } },
	{ 373, { } },
	{ 374, { } },
	{ 375, { } },
	{ 376, { } },
	{ 377, { } },
	{ 378, { } },
	{ 379, { &kDefaultIntMinus1 } },
	{ 380, { } },
	{ 381, { } },
	{ 382, { } },
	{ 383, { } },
	{ 384, { } },
	{ 385, { } },
	{ 386, { } },
	{ 387, { } },
	{ 388, { } },
	{ 389, { } },
	{ 390, { } },
	{ 391, { } },
	{ 392, { &kDefaultInventorySlotInvalid, &kDefaultInvalidWeaponSet } },
	{ 393, { } },
	{ 394, { } },
	{ 395, { } },
	{ 396, { } },
	{ 397, { } },
	{ 398, { } },
	{ 399, { } },
	{ 400, { } },
	{ 401, { &kDefaultObjectSelf, &kDefaultInvalidWeaponSet } },
	{ 402, { } },
	{ 403, { &kDefaultIntMinus1 } },
	{ 404, { &kDefaultObjectInvalid } },
	{ 405, { &kDefaultObjectInvalid } },
	{ 406, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 407, { } },
	{ 408, { &kDefaultObjectSelf } },
	{ 409, { &kDefaultObjectSelf, &kDefaultTrue } },
	{ 410, { &kDefaultFalse } },
	{ 411, { } },
	{ 412, { } },
	{ 413, { } },
	{ 414, { } },
	{ 415, { } },
	{ 416, { } },
	{ 417, { } },
	{ 418, { } },
	{ 419, { } },
	{ 420, { } },
	{ 421, { } },
	{ 422, { } },
	{ 423, { } },
	{ 424, { } },
	{ 425, { } },
	{ 426, { } },
	{ 427, { } },
	{ 428, { } },
	{ 429, { } },
	{ 430, { } },
	{ 431, { } },
	{ 432, { } },
	{ 433, { &kDefaultObjectTypeAll, &kDefaultInt1, &kDefaultInt0, &kDefaultInt0, &kDefaultInt0 } },
	{ 434, { &kDefaultObjectTypeAll, &kDefaultInt1 } },
	{ 435, { &kDefaultObjectTypeAll, &kDefaultInt1, &kDefaultInt0, &kDefaultInt0, &kDefaultInt0 } },
	{ 436, { &kDefaultObjectTypeAll, &kDefaultInt1, &kDefaultInt0, &kDefaultInt0, &kDefaultInt0 } },
	{ 437, { } },
	{ 438, { } },
	{ 439, { } },
	{ 440, { &kDefaultStringEmpty } },
	{ 441, { } },
	{ 442, { } },
	{ 443, { } },
	{ 444, { } },
	{ 445, { } },
	{ 446, { } },
	{ 447, { } },
	{ 448, { } },
	{ 449, { } },
	{ 450, { } },
	{ 451, { } },
	{ 452, { } },
	{ 453, { } },
	{ 454, { } },
	{ 455, { } },
	{ 456, { } },
	{ 457, { } },
	{ 458, { } },
	{ 459, { } },
	{ 460, { } },
	{ 461, { } },
	{ 462, { } },
	{ 463, { } },
	{ 464, { } },
	{ 465, { } },
	{ 466, { } },
	{ 467, { } },
	{ 468, { } },
	{ 469, { } },
	{ 470, { } },
	{ 471, { } },
	{ 472, { } },
	{ 473, { } },
	{ 474, { } },
	{ 475, { } },
	{ 476, { } },
	{ 477, { } },
	{ 478, { } },
	{ 479, { &kDefaultFloat0_0 } },
	{ 480, { } },
	{ 481, { } },
	{ 482, { } },
	{ 483, { } },
	{ 484, { } },
	{ 485, { } },
	{ 486, { } },
	{ 487, { } },
	{ 488, { } },
	{ 489, { } },
	{ 490, { } },
	{ 491, { } },
	{ 492, { } },
	{ 493, { } },
	{ 494, { } },
	{ 495, { } },
	{ 496, { } },
	{ 497, { } },
	{ 498, { } },
	{ 499, { } },
	{ 500, { } },
	{ 501, { } },
	{ 502, { } },
	{ 503, { } },
	{ 504, { &kDefaultObjectInvalid } },
	{ 505, { } },
	{ 506, { } },
	{ 507, { } },
	{ 508, { } },
	{ 509, { } },
	{ 510, { } },
	{ 511, { } },
	{ 512, { } },
	{ 513, { } },
	{ 514, { } },
	{ 515, { } },
	{ 516, { } },
	{ 517, { } },
	{ 518, { } },
	{ 519, { } },
	{ 520, { } },
	{ 521, { } },
	{ 522, { } },
	{ 523, { } },
	{ 524, { } },
	{ 525, { } },
	{ 526, { } },
	{ 527, { } },
	{ 528, { } },
	{ 529, { } },
	{ 530, { } },
	{ 531, { } },
	{ 532, { } },
	{ 533, { &kDefaultFalse, &kDefaultFalse, &kDefaultIntMinus1 } },
	{ 534, { } },
	{ 535, { } },
	{ 536, { } },
	{ 537, { } },
	{ 538, { &kDefaultTrue } },
	{ 539, { } },
	{ 540, { } },
	{ 541, { } },
	{ 542, { } },
	{ 543, { } },
	{ 544, { &kDefaultInt0 } },
	{ 545, { &kDefaultInt0 } },
	{ 546, { &kDefaultInt0 } },
	{ 547, { &kDefaultInt0 } },
	{ 548, { } },
	{ 549, { } },
	{ 550, { &kDefaultInt0 } },
	{ 551, { &kDefaultInt0 } },
	{ 552, { } },
	{ 553, { } },
	{ 554, { } },
	{ 555, { } },
	{ 556, { } },
	{ 557, { } },
	{ 558, { } },
	{ 559, { &kDefaultTrue, &kDefaultFalse } },
	{ 560, { &kDefaultTrue, &kDefaultFloat0_0, &kDefaultFalse, &kDefaultFloat0_0 } },
	{ 561, { &kDefaultTrue } },
	{ 562, { &kDefaultInventorySlotInvalid, &kDefaultInvalidWeaponSet } },
	{ 563, { } },
	{ 564, { } },
	{ 565, { } },
	{ 566, { &kDefaultCombatResultInvalid } },
	{ 567, { } },
	{ 568, { } },
	{ 569, { &kDefaultInt0, &kDefaultInt0, &kDefaultInt1 } },
	{ 570, { } },
	{ 571, { } },
	{ 572, { } },
	{ 573, { } },
	{ 574, { } },
	{ 575, { } },
	{ 576, { } },
	{ 577, { } },
	{ 578, { } },
	{ 579, { &kDefaultStringEmpty } },
	{ 580, { } },
	{ 581, { } },
	{ 582, { &kDefaultObjectInvalid, &kDefaultVector0, &kDefaultFloatMinus1_0, &kDefaultStringEmpty } },
	{ 583, { } },
	{ 584, { } },
	{ 585, { } },
	{ 586, { } },
	{ 587, { &kDefaultInt1, &kDefaultInt0, &kDefaultInt1, &kDefaultInt0 } },
	{ 588, { } },
	{ 589, { } },
	{ 590, { } },
	{ 591, { } },
	{ 592, { } },
	{ 593, { &kDefaultEffectTypeInvalid, &kDefaultAbilityInvalid, &kDefaultObjectInvalid } },
	{ 594, { &kDefaultTrue, &kDefaultFalse } },
	{ 595, { } },
	{ 596, { &kDefaultFloat0_0, &kDefaultObjectSelf, &kDefaultInt0 } },
	{ 597, { &kDefaultFloat0_0, &kDefaultObjectSelf, &kDefaultInt0 } },
	{ 598, { } },
	{ 599, { &kDefaultInt0, &kDefaultInt0, &kDefaultInt0, &kDefaultInt0, &kDefaultStringEmpty } },
	{ 600, { } },
	{ 601, { } },
	{ 602, { } },
	{ 603, { } },
	{ 604, { } },
	{ 605, { } },
	{ 606, { &kDefaultInt0 } },
	{ 607, { } },
	{ 608, { } },
	{ 609, { } },
	{ 610, { } },
	{ 611, { } },
	{ 612, { } },
	{ 613, { } },
	{ 614, { } },
	{ 615, { &kDefaultInt0, &kDefaultInt0 } },
	{ 616, { } },
	{ 617, { &kDefaultInt1 } },
	{ 618, { } },
	{ 619, { } },
	{ 620, { } },
	{ 621, { } },
	{ 622, { } },
	{ 623, { } },
	{ 624, { } },
	{ 625, { } },
	{ 626, { &kDefaultObjectInvalid } },
	{ 627, { } },
	{ 628, { } },
	{ 629, { } },
	{ 630, { } },
	{ 631, { } },
	{ 632, { } },
	{ 633, { } },
	{ 634, { &kDefaultInt0 } },
	{ 635, { &kDefaultAbilityInvalid } },
	{ 636, { } },
	{ 637, { } },
	{ 638, { } },
	{ 639, { } },
	{ 640, { } },
	{ 641, { } },
	{ 642, { } },
	{ 643, { } },
	{ 644, { &kDefaultTrue, &kDefaultInt0, &kDefaultFalse } },
	{ 645, { } },
	{ 646, { } },
	{ 647, { } },
	{ 648, { &kDefaultObjectInvalid } },
	{ 649, { } },
	{ 650, { } },
	{ 651, { &kDefaultInt0 } },
	{ 652, { } },
	{ 653, { } },
	{ 654, { &kDefaultInt0 } },
	{ 655, { } },
	{ 656, { } },
	{ 657, { } },
	{ 658, { } },
	{ 659, { } },
	{ 660, { &kDefaultFalse } },
	{ 661, { &kDefaultFalse } },
	{ 662, { } },
	{ 663, { } },
	{ 664, { } },
	{ 665, { } },
	{ 666, { } },
	{ 667, { &kDefaultStringEmpty } },
	{ 668, { } },
	{ 669, { } },
	{ 670, { } },
	{ 671, { } },
	{ 672, { &kDefaultFalse } },
	{ 673, { &kDefaultFloat0_0, &kDefaultFloat0_0, &kDefaultFalse } },
	{ 674, { } },
	{ 675, { } },
	{ 676, { } },
	{ 677, { } },
	{ 678, { } },
	{ 679, { } },
	{ 680, { &kDefaultFloat0_0 } },
	{ 681, { } },
	{ 682, { &kDefaultTrue } },
	{ 683, { } },
	{ 684, { } },
	{ 685, { } },
	{ 686, { } },
	{ 687, { } },
	{ 688, { } },
	{ 689, { } },
	{ 690, { } },
	{ 691, { } },
	{ 692, { } },
	{ 693, { &kDefaultTrue } },
	{ 694, { } },
	{ 695, { } },
	{ 696, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultObjectInvalid } },
	{ 697, { } },
	{ 698, { } },
	{ 699, { &kDefaultTrue } },
	{ 700, { } },
	{ 701, { &kDefaultTrue } },
	{ 702, { } },
	{ 703, { &kDefaultTrue } },
	{ 704, { } },
	{ 705, { } },
	{ 706, { } },
	{ 707, { &kDefaultStringEmpty } },
	{ 708, { } },
	{ 709, { } },
	{ 710, { } },
	{ 711, { } },
	{ 712, { &kDefaultEffectTypeInvalid, &kDefaultInt0, &kDefaultObjectInvalid, &kDefaultEffectDurationTypeInvalid, &kDefaultIntMinus1 } },
	{ 713, { &kDefaultAbilityInvalid, &kDefaultInt0 } },
	{ 714, { } },
	{ 715, { &kDefaultResistanceCheckInvalid, &kDefaultProjectileTargetInvalid } },
	{ 716, { } },
	{ 717, { } },
	{ 718, { } },
	{ 719, { } },
	{ 720, { } },
	{ 721, { } },
	{ 722, { } },
	{ 723, { &kDefaultStringEmpty } },
	{ 724, { } },
	{ 725, { } },
	{ 726, { } },
	{ 727, { } },
	{ 728, { } },
	{ 729, { } },
	{ 730, { } },
	{ 731, { } },
	{ 732, { } },
	{ 733, { &kDefaultFalse } },
	{ 734, { } },
	{ 735, { } },
	{ 736, { } },
	{ 737, { } },
	{ 738, { &kDefaultPropertyValueTotal } },
	{ 739, { } },
	{ 740, { &kDefaultPropertyValueBase } },
	{ 741, { } },
	{ 742, { } },
	{ 743, { } },
	{ 744, { &kDefaultFloatyMessage, &kDefaultInt16777215, &kDefaultFloat0_5 } },
	{ 745, { } },
	{ 746, { &kDefaultInt0, &kDefaultFalse } },
	{ 747, { &kDefaultInt16777215 } },
	{ 748, { &kDefaultStringEmpty } },
	{ 749, { } },
	{ 750, { } },
	{ 751, { } },
	{ 752, { } },
	{ 753, { &kDefaultFalse } },
	{ 754, { } },
	{ 755, { } },
	{ 756, { } },
	{ 757, { } },
	{ 758, { } },
	{ 759, { } },
	{ 760, { &kDefaultInt0 } },
	{ 761, { } },
	{ 762, { } },
	{ 763, { } },
	{ 764, { } },
	{ 765, { } },
	{ 766, { } },
	{ 767, { } },
	{ 768, { } },
	{ 769, { } },
	{ 770, { &kDefaultInt16777215 } },
	{ 771, { } },
	{ 772, { } },
	{ 773, { &kDefaultFalse } },
	{ 774, { } },
	{ 775, { } },
	{ 776, { } },
	{ 777, { } },
	{ 778, { } },
	{ 779, { } },
	{ 780, { } },
	{ 781, { } },
	{ 782, { } },
	{ 783, { } },
	{ 784, { } },
	{ 785, { } },
	{ 786, { } },
	{ 787, { } },
	{ 788, { } },
	{ 789, { } },
	{ 790, { } },
	{ 791, { } },
	{ 792, { } },
	{ 793, { } },
	{ 794, { } },
	{ 795, { } },
	{ 796, { } },
	{ 797, { } },
	{ 798, { } },
	{ 799, { } },
	{ 800, { } },
	{ 801, { } },
	{ 802, { } },
	{ 803, { } },
	{ 804, { } },
	{ 805, { } },
	{ 806, { } },
	{ 807, { } },
	{ 808, { &kDefaultObjectTypeCreature } },
	{ 809, { } },
	{ 810, { } },
	{ 811, { &kDefaultFalse } },
	{ 812, { } },
	{ 813, { } },
	{ 814, { } },
	{ 815, { } },
	{ 816, { } },
	{ 817, { } },
	{ 818, { } },
	{ 819, { &kDefaultInt4294967295 } },
	{ 820, { &kDefaultStringEmpty, &kDefaultStringEmpty, &kDefaultFalse } },
	{ 821, { } },
	{ 822, { } },
	{ 823, { } },
	{ 824, { } },
	{ 825, { } },
	{ 826, { } },
	{ 827, { } },
	{ 828, { &kDefaultStringEmpty } },
	{ 829, { } },
	{ 830, { } },
	{ 831, { &kDefaultStringEmpty } },
	{ 832, { } },
	{ 833, { &kDefaultInt1, &kDefaultInt0, &kDefaultInt0 } },
	{ 834, { &kDefaultInt0, &kDefaultInt0, &kDefaultInt1 } },
	{ 835, { &kDefaultInt0, &kDefaultFloat0_0, &kDefaultInt0, &kDefaultFloat0_0 } },
	{ 836, { } },
	{ 837, { } },
	{ 838, { } },
	{ 839, { } },
	{ 840, { } },
	{ 841, { } },
	{ 842, { } },
	{ 843, { &kDefaultInt0 } },
	{ 844, { } },
	{ 845, { &kDefaultInt0 } },
	{ 846, { } },
	{ 847, { &kDefaultInt0, &kDefaultFloat0_0, &kDefaultInt0, &kDefaultFloat0_0 } },
	{ 848, { &kDefaultInt0 } },
	{ 849, { } },
	{ 850, { } },
	{ 851, { } },
	{ 852, { &kDefaultStringEmpty } },
	{ 853, { } },
	{ 854, { } },
	{ 855, { } },
	{ 856, { &kDefaultStringEmpty } },
	{ 857, { } },
	{ 858, { } },
	{ 859, { &kDefaultAbilityInvalid, &kDefaultFalse } },
	{ 860, { &kDefaultStringEmpty } },
	{ 861, { &kDefaultStringEmpty } },
	{ 862, { } }
};

} // End of namespace DragonAge

} // End of namespace Engines

#endif // ENGINES_DRAGONAGE_SCRIPT_TABLES_H
