import("private.core.base.select_script")

function _match_patterns(patterns, opt)
    local scripts = {}
    for _, pattern in ipairs(patterns) do
        pattern = pattern:gsub("([%+%.%-%^%$%%])", "%%%1")
        pattern = pattern:gsub("%*", "\001")
        pattern = pattern:gsub("\001", ".*")
        scripts[pattern] = true
    end
    return select_script(scripts, opt) == true
end

function test_plat_only(t)
    t:require(_match_patterns("*", {plat = "macosx"}))
    t:require(_match_patterns("macosx", {plat = "macosx"}))
    t:require(_match_patterns("macosx,linux", {plat = "macosx"}))
    t:require(_match_patterns("mac*", {plat = "macosx"}))
    t:require_not(_match_patterns("macosx", {plat = "linux"}))
    t:require_not(_match_patterns("linux", {plat = "macosx"}))
    t:require_not(_match_patterns("!macosx", {plat = "macosx"}))
    t:require_not(_match_patterns("!mac*", {plat = "macosx"}))
    t:require(_match_patterns("!macosx", {plat = "linux"}))
    t:require(_match_patterns("!macosx,!android", {plat = "linux"}))
    t:require(_match_patterns("!mac*", {plat = "linux"}))
end

function test_plat_arch(t)
    t:require(_match_patterns("!wasm|!arm*", {plat = "linux", arch = "x86_64"}))
    t:require_not(_match_patterns("!wasm|!arm*", {plat = "linux", arch = "arm64"}))
    t:require(_match_patterns("*|x86_64", {plat = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("macosx|x86_64", {plat = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("macosx|x86_64,linux|x86_64", {plat = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("macosx|x86_*", {plat = "macosx", arch = "x86_64"}))
    t:require_not(_match_patterns("macosx|x86_64", {plat = "linux", arch = "x86_64"}))
    t:require_not(_match_patterns("macosx|i386", {plat = "macosx", arch = "x86_64"}))
    t:require_not(_match_patterns("!macosx|x86_64", {plat = "macosx", arch = "x86_64"}))
    t:require_not(_match_patterns("!mac*|x86_64", {plat = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("!macosx|x86_64", {plat = "linux", arch = "x86_64"}))
    t:require(_match_patterns("!mac*|x86_64", {plat = "linux", arch = "x86_64"}))
    t:require(_match_patterns("macosx|!i386", {plat = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("!macosx|!i386", {plat = "linux", arch = "x86_64"}))
    t:require(_match_patterns("windows|!x86", {plat = "windows", arch = "x64"}))
    t:require_not(_match_patterns("windows|!x86", {plat = "android", arch = "arm64-v8a"}))
    t:require(_match_patterns("macosx|native", {plat = "macosx", arch = "x86_64", subarch = "x86_64"}))
    t:require(_match_patterns("macosx|!native", {plat = "macosx", arch = "arm64", subarch = "x86_64"}))
    t:require_not(_match_patterns("macosx|!native", {plat = "macosx", arch = "x86_64", subarch = "x86_64"}))
    t:require_not(_match_patterns("windows|native", {plat = "macosx", arch = "x86_64", subarch = "x86_64"}))
end

function test_subhost_only(t)
    t:require(_match_patterns("@*", {subhost = "macosx"}))
    t:require(_match_patterns("@macosx", {subhost = "macosx"}))
    t:require(_match_patterns("@mac*", {subhost = "macosx"}))
    t:require_not(_match_patterns("@macosx", {subhost = "linux"}))
    t:require_not(_match_patterns("@linux", {subhost = "macosx"}))
    t:require_not(_match_patterns("@!macosx", {subhost = "macosx"}))
    t:require_not(_match_patterns("@!mac*", {subhost = "macosx"}))
    t:require(_match_patterns("@!macosx", {subhost = "linux"}))
    t:require(_match_patterns("@!mac*", {subhost = "linux"}))
end

function test_subhost_subarch(t)
    t:require(_match_patterns("@*|x86_64", {subhost = "macosx", subarch = "x86_64"}))
    t:require(_match_patterns("@macosx|x86_64", {subhost = "macosx", subarch = "x86_64"}))
    t:require(_match_patterns("@macosx|x86_*", {subhost = "macosx", subarch = "x86_64"}))
    t:require_not(_match_patterns("@macosx|x86_64", {subhost = "linux", subarch = "x86_64"}))
    t:require_not(_match_patterns("@macosx|i386", {subhost = "macosx", subarch = "x86_64"}))
    t:require_not(_match_patterns("@!macosx|x86_64", {subhost = "macosx", subarch = "x86_64"}))
    t:require_not(_match_patterns("@!mac*|x86_64", {subhost = "macosx", subarch = "x86_64"}))
    t:require(_match_patterns("@!macosx|x86_64", {subhost = "linux", subarch = "x86_64"}))
    t:require(_match_patterns("@!mac*|x86_64", {subhost = "linux", subarch = "x86_64"}))
    t:require(_match_patterns("@macosx|!i386", {subhost = "macosx", subarch = "x86_64"}))
    t:require(_match_patterns("@!macosx|!i386", {subhost = "linux", subarch = "x86_64"}))
    t:require(_match_patterns("@windows|!x86", {subhost = "windows", subarch = "x64"}))
    t:require_not(_match_patterns("@windows|!x86", {subhost = "android", subarch = "arm64-v8a"}))
    t:require(_match_patterns("@macosx|native", {subhost = "macosx", subarch = "x86_64"}))
    t:require(_match_patterns("@macosx|native", {subhost = "macosx", subarch = "arm64"}))
    t:require_not(_match_patterns("@macosx|!native", {subhost = "macosx", subarch = "x86_64"}))
    t:require_not(_match_patterns("@windows|native", {subhost = "macosx", subarch = "x86_64"}))
end

function test_plat_subhost(t)
    t:require(_match_patterns("*@macosx", {plat = "macosx", subhost = "macosx"}))
    t:require(_match_patterns("android@macosx", {plat = "android", subhost = "macosx"}))
    t:require(_match_patterns("android@macosx,linux", {plat = "android", subhost = "linux"}))
    t:require(_match_patterns("android@mac*", {plat = "android", subhost = "macosx"}))
    t:require(_match_patterns("android@!macosx", {plat = "android", subhost = "linux"}))
    t:require_not(_match_patterns("!android@macosx", {plat = "android", subhost = "macosx"}))
    t:require(_match_patterns("!iphon*@macosx", {plat = "linux", subhost = "macosx"}))
end

function test_plat_arch_subhost(t)
    t:require(_match_patterns("*|x86_64@macosx", {plat = "macosx", subhost = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("android|arm64-v8a@macosx", {plat = "android", subhost = "macosx", arch = "arm64-v8a"}))
    t:require(_match_patterns("android|x86_64@macosx,linux", {plat = "android", subhost = "linux", arch = "x86_64"}))
    t:require(_match_patterns("android|x86_64@mac*", {plat = "android", subhost = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("android|x86_64@!macosx", {plat = "android", subhost = "linux", arch = "x86_64"}))
    t:require_not(_match_patterns("!android|x86_64@macosx", {plat = "android", subhost = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("!iphon*|x86_64@macosx", {plat = "linux", subhost = "macosx", arch = "x86_64"}))
    t:require(_match_patterns("iphon*|arm64@macosx", {plat = "iphoneos", subhost = "macosx", arch = "arm64"}))
    t:require_not(_match_patterns("iphon*|arm64@macosx", {plat = "iphoneos", subhost = "linux", arch = "arm64"}))
end

function test_plat_arch_subhost_subarch(t)
    t:require(_match_patterns("*|x86_64@macosx|x86_64", {plat = "macosx", subhost = "macosx", arch = "x86_64", subarch = "x86_64"}))
    t:require(_match_patterns("android|arm64-v8a@macosx|x86_64", {plat = "android", subhost = "macosx", arch = "arm64-v8a", subarch = "x86_64"}))
    t:require(_match_patterns("android|x86_64@macosx|x86_64,linux|x86_64", {plat = "android", subhost = "linux", arch = "x86_64", subarch = "x86_64"}))
    t:require(_match_patterns("android|x86_64@mac*|x86_64", {plat = "android", subhost = "macosx", arch = "x86_64", subarch = "x86_64"}))
    t:require(_match_patterns("android|x86_64@!macosx|x86_64", {plat = "android", subhost = "linux", arch = "x86_64", subarch = "x86_64"}))
    t:require_not(_match_patterns("!android|x86_64@macosx|x86_64", {plat = "android", subhost = "macosx", arch = "x86_64", subarch = "x86_64"}))
    t:require(_match_patterns("!iphon*|x86_64@macosx|x86_64", {plat = "linux", subhost = "macosx", arch = "x86_64", subarch = "x86_64"}))
    t:require(_match_patterns("iphon*|arm64@macosx|x86_64", {plat = "iphoneos", subhost = "macosx", arch = "arm64", subarch = "x86_64"}))
    t:require_not(_match_patterns("iphon*|arm64@macosx|x86_64", {plat = "iphoneos", subhost = "linux", arch = "arm64", subarch = "x86_64"}))
    t:require(_match_patterns("android|native@macosx|x86_64", {plat = "android", subhost = "macosx", arch = "x86_64", subarch = "x86_64"}))
end

function test_logical_expr(t)
    t:require(_match_patterns("!macosx|x86_64,!iphoneos|arm64", {plat = "linux", arch = "x86_64"}))
    t:require(_match_patterns("!wasm|!arm* and !cross|!arm*", {plat = "linux", arch = "x86_64"}))
    t:require_not(_match_patterns("!wasm|!arm* and !cross|!arm*", {plat = "linux", arch = "arm64"}))
    t:require_not(_match_patterns("!wasm|!arm* and !cross|!arm*", {plat = "wasm", arch = "x86_64"}))
    t:require(_match_patterns("!macosx|x86_64 or !iphoneos|arm64", {plat = "linux", arch = "x86_64"}))
    t:require_not(_match_patterns("!android@macosx or !android@linux", {plat = "android", subhost = "macosx"}))
    t:require(_match_patterns("@macosx|x86_64 or @linux|x86_64", {subhost = "macosx", subarch = "x86_64"}))
    t:require(_match_patterns("@macosx or @linux", {subhost = "macosx"}))
    t:require(_match_patterns("!wasm|!arm* or (!cross|!arm* or linux)", {plat = "linux", arch = "arm64"}))
    t:require_not(_match_patterns("!wasm|!arm* or (!cross|!arm* and !linux)", {plat = "linux", arch = "arm64"}))
    t:require_not(_match_patterns("!macosx|x86_64 and !linux|x86_64", {plat = "linux", arch = "x86_64"}))
    t:require_not(_match_patterns("!macosx and !android", {plat = "android"}))
    t:require_not(_match_patterns("!macosx and !android", {plat = "macosx"}))
end

