import { NextResponse } from 'next/server'

export async function GET(req: Request) {
  try {
    if (!process.env.GITHUB_ACCESS_TOKEN || process.env.GITHUB_ACCESS_TOKEN === '') {
      return new Response('Missing GITHUB_ACCESS_TOKEN – make sure to add it to your .env file.', {
        status: 400
      })
    }
    const params = new URL(req.url)
    const repoName = params.searchParams.get('repo')
    const owner = params.searchParams.get('owner')
    const response = await fetch(`https://api.github.com/repos/${owner}/${repoName}`, {
      headers: {
        Accept: 'application/vnd.github+json',
        Authorization: `Bearer ${process.env.GITHUB_ACCESS_TOKEN}`,
        'X-GitHub-Api-Version': '2022-11-28'
      }
    })
    if (!response.ok) return NextResponse.json({ error: 'Repository not found' }, { status: 404 })

    const res = await response.json()
    const data = {
      repoName: res.name,
      owner: res.owner.login,
      description: res.description,
      language: res.language,
      branch: res.default_branch,
      urlRepository: res.html_url
    }
    return NextResponse.json({ data })
  } catch (error) {
    console.error(error)
    return NextResponse.json(
      {
        error: 'An error has ocurred'
      },
      {
        status: 500
      }
    )
  }
}
