﻿import { h, Component, Prop, Emit, WuComponent, Fragment, OnConnected } from '@wu-component/web-core-plus';
import css from './index.scss';
type TypeEnums = 'success' | 'warning' | 'info' | 'error';
const TYPE_CLASSES_MAP = {
    'success': 'el-icon-success',
    'warning': 'el-icon-warning',
    'error': 'el-icon-error'
};

const TAG = 'wu-plus-alert';
interface WuAlertProps {
    visible?: boolean;
    tip?: string;
    description?: string;
    type?: TypeEnums;
    closable?: boolean;
    closeText?: string;
    showIcon?: boolean;
    center?: boolean;
    effect?: string;
    onConfirm: (e: any) => void;
    onCancel: (e: any) => void;
    onClose: (e: any) => void;
    [key: string]: any;
}
declare global {
    namespace JSX {
        interface IntrinsicElements {
           [TAG]: WuAlertProps
        }
    }
}
@Component({
    name: TAG,
    css: css,
})
export class WuAlert extends WuComponent implements OnConnected {
    constructor() {
        super();
    }

    // 是否显示
    @Prop({ default: true, type: Boolean })
    public visible: boolean;

    public show = true;

    @Prop({ default: '', type: String })
    public tip: string;

    @Prop({ default: '', type: String })
    public description: string;

    @Prop({ default: 'info', type: String })
    public type: TypeEnums;

    @Prop({ default: true, type: Boolean })
    public closable: boolean;

    @Prop({ default: '', type: String })
    public closeText: string;

    @Prop({ default: false, type: Boolean })
    public showIcon: boolean;

    @Prop({ default: false, type: Boolean })
    public center: boolean;

    @Prop({ default: 'light', type: String })
    public effect: string;

    public override connected(shadowRoot: ShadowRoot) {
        this.show = this.visible;
    }

    @Emit("confirm")
    public confirm() {
        this.visible = false;
        this.show = false;
        this.update();
        return this;
    }

    @Emit("cancel")
    public cancel() {
        this.visible = false;
        this.show = false;
        this.update();
        return this;
    }

    @Emit("close")
    public close() {
        this.visible = false;
        this.show = false;
        this.update();
        return this;
    }

    get typeClass() {
        return `wu-alert-${ this.type }`;
    }

    get iconClass() {
        return TYPE_CLASSES_MAP[this.type] || 'wu-icon-info';
    }

    get isBoldTitle() {
        return this.description ? 'is-bold' : '';
    }

    public override render(_renderProps = {}, _store = {}) {
        if ((!this.visible) || (!this.show)) {
            return <Fragment></Fragment>;
        }
        return (
            <div class={`wu-alert ${this.typeClass} ${this.center ? "is-center" : ''} ${'is-' + this.effect}`} role="alert">
                {
                    this.showIcon ? (
                        <slot name="icon" />
                    ) : null
                }
                <div class="wu-alert_content">
                    <span class={`wu-alert_title ${this.isBoldTitle} `}>
                        {this.tip ? this.tip : <slot name="title" />}
                    </span>
                    <p class="wu-alert_description">
                        {this.description ? this.description : <slot name="description" />}
                    </p>
                    {
                        this.closable && !this.closeText ? (
                            <svg onClick={this.close.bind(this)} class="wu-tag_close wu-icon-close wu-alert_closebtn" fill="currentColor" width="1em" height="1em" focusable="false" viewBox="0 0 24 24" aria-hidden="true">
                                <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z" />
                            </svg>
                        ) : null
                    }
                    {
                        this.closable && this.closeText ? (
                            <i class="wu-alert_closebtn is-customed" onClick={this.close.bind(this)}>{this.closeText}</i>
                        ) : null
                    }
                </div>
           </div>
        );
    }
}
