/* packet-e212.c
 * Routines for output and filtering of E.212 numbers common
 * to many dissectors.
 * Copyright 2006, Anders Broman <anders.broman@ericsson.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "config.h"

#include <stdlib.h>

#include <epan/packet.h>

#include <wsutil/strtoi.h>
#include <wsutil/array.h>

#include "packet-e212.h"
#include "expert.h"

void proto_register_e212(void);

/*
 * Annexes to ITU Operational Bulletin
 * No. 1117 - 1.II.2017 (List of Mobile Country Codes)
 * No. 1280 - 15.XI.2023 (Mobile Network Codes)
 *
 * With Amendment No. 1 (T-SP-OB 1164 - 15.I.2019) up to and
 * including Amendment No. 34 (T-SP-OB 1198 - 15.VI.2020)
 *
 * COMPLEMENT TO ITU-T RECOMMENDATION E.212 (09/2016)
 * WITH Rec. ITU-T E.212 (2016)/Amd.2
 *
 * Find the bulletins here: http://www.itu.int/pub/T-SP-OB
 *
 * Amended with information from http://www.mcc-mnc.com
 * - Chinese PLMN codes
 * - Iranian PLMN codes
 * - Kazakhstani PLMN codes (see also [1] and [2])
 *
 * [1] https://ru.wikipedia.org/wiki/MNC
 * [2] https://htmlweb.ru/geo/oper.php?country=KZ
 */
static const value_string E212_codes[] = {
    /* 001 is used by some telecoms for internal TEST networks. */
    {  202, "Greece" },
    {  203, "Unassigned" },
    {  204, "Netherlands" },
    {  205, "Unassigned" },
    {  206, "Belgium" },
    {  207, "Unassigned" },
    {  208, "France" },
    {  209, "Unassigned" },
    {  210, "Unassigned" },
    {  211, "Unassigned" },
    {  212, "Monaco" },
    {  213, "Andorra" },
    {  214, "Spain" },
    {  215, "Unassigned" },
    {  216, "Hungary" },
    {  217, "Unassigned" },
    {  218, "Bosnia and Herzegovina" },
    {  219, "Croatia" },
    {  220, "Serbia" },
    {  221, "Kosovo" },
    {  222, "Italy" },
    {  223, "Unassigned" },
    {  224, "Unassigned" },
    {  225, "Vatican City" },
    {  226, "Romania" },
    {  227, "Unassigned" },
    {  228, "Switzerland" },
    {  229, "Unassigned" },
    {  230, "Czech Rep." },
    {  231, "Slovakia" },
    {  232, "Austria" },
    {  233, "Unassigned" },
    {  234, "United Kingdom" },
    {  235, "United Kingdom" },
    {  236, "Unassigned" },
    {  237, "Unassigned" },
    {  238, "Denmark" },
    {  239, "Unassigned" },
    {  240, "Sweden" },
    {  241, "Unassigned" },
    {  242, "Norway" },
    {  243, "Unassigned" },
    {  244, "Finland" },
    {  245, "Unassigned" },
    {  246, "Lithuania" },
    {  247, "Latvia" },
    {  248, "Estonia" },
    {  249, "Unassigned" },
    {  250, "Russian Federation" },
    {  251, "Unassigned" },
    {  252, "Unassigned" },
    {  253, "Unassigned" },
    {  254, "Unassigned" },
    {  255, "Ukraine" },
    {  256, "Unassigned" },
    {  257, "Belarus" },
    {  258, "Unassigned" },
    {  259, "Moldova (Republic of)" },
    {  260, "Poland" },
    {  261, "Unassigned" },
    {  262, "Germany" },
    {  263, "Unassigned" },
    {  264, "Unassigned" },
    {  265, "Unassigned" },
    {  266, "Gibraltar" },
    {  267, "Unassigned" },
    {  268, "Portugal" },
    {  269, "Unassigned" },
    {  270, "Luxembourg" },
    {  271, "Unassigned" },
    {  272, "Ireland" },
    {  273, "Unassigned" },
    {  274, "Iceland" },
    {  275, "Unassigned" },
    {  276, "Albania" },
    {  277, "Unassigned" },
    {  278, "Malta" },
    {  279, "Unassigned" },
    {  280, "Cyprus" },
    {  281, "Unassigned" },
    {  282, "Georgia" },
    {  283, "Armenia" },
    {  284, "Bulgaria" },
    {  285, "Unassigned" },
    {  286, "Turkey" },
    {  287, "Unassigned" },
    {  288, "Faroe Islands" },
    {  289, "Unassigned" },
    /* Abkhazia uses MCC 289; ITU does not list the number due to status
     * as disputed territory. */
    {  290, "Greenland" },
    {  291, "Unassigned" },
    {  292, "San Marino" },
    {  293, "Slovenia" },
    {  294, "North Macedonia" },
    {  295, "Liechtenstein" },
    {  296, "Unassigned" },
    {  297, "Montenegro" },
    {  298, "Unassigned" },
    {  299, "Unassigned" },
    {  300, "Unassigned" },
    {  301, "Unassigned" },
    {  302, "Canada" },
    {  303, "Unassigned" },
    {  304, "Unassigned" },
    {  305, "Unassigned" },
    {  306, "Unassigned" },
    {  307, "Unassigned" },
    {  308, "Saint Pierre and Miquelon" },
    {  309, "Unassigned" },
    {  310, "United States" },
    {  311, "United States" },
    {  312, "United States" },
    {  313, "United States" },
    {  314, "United States" },
    {  315, "United States" },
    {  316, "United States" },
    {  317, "Unassigned" },
    {  318, "Unassigned" },
    {  319, "Unassigned" },
    {  320, "Unassigned" },
    {  321, "Unassigned" },
    {  322, "Unassigned" },
    {  323, "Unassigned" },
    {  324, "Unassigned" },
    {  325, "Unassigned" },
    {  326, "Unassigned" },
    {  327, "Unassigned" },
    {  328, "Unassigned" },
    {  329, "Unassigned" },
    {  330, "Puerto Rico" },
    /* Some local operators use USA MCCs, some use assigned MCC 330. */
    {  331, "Unassigned" },
    {  332, "United States Virgin Islands" },
    /* MCC 332 assigned to USVI but local operators use the USA MCCs */
    {  333, "Unassigned" },
    {  334, "Mexico" },
    {  335, "Unassigned" },
    {  336, "Unassigned" },
    {  337, "Unassigned" },
    {  338, "Jamaica" },
    {  339, "Unassigned" },
    {  340, "French Guiana / Guadeloupe / Martinique" },
    /* French Guiana has been assigned MCC 742, but some networks
     * still operate on MCC 340. */
    {  341, "Unassigned" },
    {  342, "Barbados" },
    {  343, "Unassigned" },
    {  344, "Antigua and Barbuda" },
    {  345, "Unassigned" },
    {  346, "Cayman Islands" },
    {  347, "Unassigned" },
    {  348, "British Virgin Islands" },
    {  349, "Unassigned" },
    {  350, "Bermuda" },
    {  351, "Unassigned" },
    {  352, "Grenada" },
    {  353, "Unassigned" },
    {  354, "Montserrat" },
    {  355, "Unassigned" },
    {  356, "Saint Kitts and Nevis" },
    {  357, "Unassigned" },
    {  358, "Saint Lucia" },
    {  359, "Unassigned" },
    {  360, "Saint Vincent and the Grenadines" },
    {  361, "Unassigned" },
    {  362, "Curaçao / Sint Maarten / Bonaire, Sint Eustatius and Saba" },
    {  363, "Aruba" },
    {  364, "Bahamas" },
    {  365, "Anguilla" },
    {  366, "Dominica" },
    {  367, "Unassigned" },
    {  368, "Cuba" },
    {  369, "Unassigned" },
    {  370, "Dominican Rep." },
    {  371, "Unassigned" },
    {  372, "Haiti" },
    {  373, "Unassigned" },
    {  374, "Trinidad and Tobago" },
    {  375, "Unassigned" },
    {  376, "Turks and Caicos Islands" },
    {  377, "Unassigned" },
    {  378, "Unassigned" },
    {  379, "Unassigned" },
    {  380, "Unassigned" },
    {  381, "Unassigned" },
    {  382, "Unassigned" },
    {  383, "Unassigned" },
    {  384, "Unassigned" },
    {  385, "Unassigned" },
    {  386, "Unassigned" },
    {  387, "Unassigned" },
    {  388, "Unassigned" },
    {  389, "Unassigned" },
    {  390, "Unassigned" },
    {  391, "Unassigned" },
    {  392, "Unassigned" },
    {  393, "Unassigned" },
    {  394, "Unassigned" },
    {  395, "Unassigned" },
    {  396, "Unassigned" },
    {  397, "Unassigned" },
    {  398, "Unassigned" },
    {  399, "Unassigned" },
    {  400, "Azerbaijan" },
    {  401, "Kazakhstan" },
    {  402, "Bhutan" },
    {  403, "Unassigned" },
    {  404, "India" },
    {  405, "India" },
    {  406, "India" },
    {  407, "Unassigned" },
    {  408, "Unassigned" },
    {  409, "Unassigned" },
    {  410, "Pakistan" },
    {  411, "Unassigned" },
    {  412, "Afghanistan" },
    {  413, "Sri Lanka" },
    {  414, "Myanmar" },
    {  415, "Lebanon" },
    {  416, "Jordan" },
    {  417, "Syrian Arab Republic" },
    {  418, "Iraq" },
    {  419, "Kuwait" },
    {  420, "Saudi Arabia" },
    {  421, "Yemen" },
    {  422, "Oman" },
    {  423, "Unassigned" },
    {  424, "United Arab Emirates" },
    {  425, "Israel" },
    {  426, "Bahrain" },
    {  427, "Qatar" },
    {  428, "Mongolia" },
    {  429, "Nepal" },
    {  430, "United Arab Emirates (Abu Dhabi)" },
    {  431, "United Arab Emirates (Dubai)" },
    {  432, "Iran (Islamic Republic of)" },
    {  433, "Unassigned" },
    {  434, "Uzbekistan" },
    {  435, "Unassigned" },
    {  436, "Tajikistan" },
    {  437, "Kyrgyzstan" },
    {  438, "Turkmenistan" },
    {  439, "Unassigned" },
    {  440, "Japan" },
    {  441, "Japan" },
    {  442, "Unassigned" },
    {  443, "Unassigned" },
    {  444, "Unassigned" },
    {  445, "Unassigned" },
    {  446, "Unassigned" },
    {  447, "Unassigned" },
    {  448, "Unassigned" },
    {  449, "Unassigned" },
    {  450, "Korea (Rep. of)" },
    {  451, "Unassigned" },
    {  452, "Viet Nam" },
    {  453, "Unassigned" },
    {  454, "Hong Kong, China" },
    {  455, "Macao, China" },
    {  456, "Cambodia" },
    {  457, "Lao P.D.R." },
    {  458, "Unassigned" },
    {  459, "Unassigned" },
    {  460, "China" },
    {  461, "China" },
    {  462, "Unassigned" },
    {  463, "Unassigned" },
    {  464, "Unassigned" },
    {  465, "Unassigned" },
    {  466, "Taiwan" },
    {  467, "Dem. People's Rep. of Korea" },
    {  468, "Unassigned" },
    {  469, "Unassigned" },
    {  470, "Bangladesh" },
    {  471, "Unassigned" },
    {  472, "Maldives" },
    {  473, "Unassigned" },
    {  474, "Unassigned" },
    {  475, "Unassigned" },
    {  476, "Unassigned" },
    {  477, "Unassigned" },
    {  478, "Unassigned" },
    {  479, "Unassigned" },
    {  480, "Unassigned" },
    {  481, "Unassigned" },
    {  482, "Unassigned" },
    {  483, "Unassigned" },
    {  484, "Unassigned" },
    {  485, "Unassigned" },
    {  486, "Unassigned" },
    {  487, "Unassigned" },
    {  488, "Unassigned" },
    {  489, "Unassigned" },
    {  490, "Unassigned" },
    {  491, "Unassigned" },
    {  492, "Unassigned" },
    {  493, "Unassigned" },
    {  494, "Unassigned" },
    {  495, "Unassigned" },
    {  496, "Unassigned" },
    {  497, "Unassigned" },
    {  498, "Unassigned" },
    {  499, "Unassigned" },
    {  500, "Unassigned" },
    {  501, "Unassigned" },
    {  502, "Malaysia" },
    {  503, "Unassigned" },
    {  504, "Unassigned" },
    {  505, "Australia" },
    {  506, "Unassigned" },
    {  507, "Unassigned" },
    {  508, "Unassigned" },
    {  509, "Unassigned" },
    {  510, "Indonesia" },
    {  511, "Unassigned" },
    {  512, "Unassigned" },
    {  513, "Unassigned" },
    {  514, "Timor-Leste" },
    {  515, "Philippines" },
    {  516, "Unassigned" },
    {  517, "Unassigned" },
    {  518, "Unassigned" },
    {  519, "Unassigned" },
    {  520, "Thailand" },
    {  521, "Unassigned" },
    {  522, "Unassigned" },
    {  523, "Unassigned" },
    {  524, "Unassigned" },
    {  525, "Singapore" },
    {  526, "Unassigned" },
    {  527, "Unassigned" },
    {  528, "Brunei Darussalam" },
    {  529, "Unassigned" },
    {  530, "New Zealand" },
    {  531, "Unassigned" },
    {  532, "Unassigned" },
    {  533, "Unassigned" },
    {  534, "Unassigned" },
    {  535, "Unassigned" },
    {  536, "Nauru" },
    {  537, "Papua New Guinea" },
    {  538, "Unassigned" },
    {  539, "Tonga" },
    {  540, "Solomon Islands" },
    {  541, "Vanuatu" },
    {  542, "Fiji" },
    {  543, "Wallis and Futuna" },
    {  544, "American Samoa" },
    /* Some operators use a USA MCC, some use the assigned */
    {  545, "Kiribati" },
    {  546, "New Caledonia" },
    {  547, "French Polynesia" },
    {  548, "Cook Islands" },
    {  549, "Samoa" },
    {  550, "Micronesia" },
    {  551, "Marshall Islands" },
    {  552, "Palau" },
    {  553, "Tuvalu" },
    {  554, "Tokelau" },
    {  555, "Niue" },
    {  556, "Unassigned" },
    {  557, "Unassigned" },
    {  558, "Unassigned" },
    {  559, "Unassigned" },
    {  560, "Unassigned" },
    {  561, "Unassigned" },
    {  562, "Unassigned" },
    {  563, "Unassigned" },
    {  564, "Unassigned" },
    {  565, "Unassigned" },
    {  566, "Unassigned" },
    {  567, "Unassigned" },
    {  568, "Unassigned" },
    {  569, "Unassigned" },
    {  570, "Unassigned" },
    {  571, "Unassigned" },
    {  572, "Unassigned" },
    {  573, "Unassigned" },
    {  574, "Unassigned" },
    {  575, "Unassigned" },
    {  576, "Unassigned" },
    {  577, "Unassigned" },
    {  578, "Unassigned" },
    {  579, "Unassigned" },
    {  580, "Unassigned" },
    {  581, "Unassigned" },
    {  582, "Unassigned" },
    {  583, "Unassigned" },
    {  584, "Unassigned" },
    {  585, "Unassigned" },
    {  586, "Unassigned" },
    {  587, "Unassigned" },
    {  588, "Unassigned" },
    {  589, "Unassigned" },
    {  590, "Unassigned" },
    {  591, "Unassigned" },
    {  592, "Unassigned" },
    {  593, "Unassigned" },
    {  594, "Unassigned" },
    {  595, "Unassigned" },
    {  596, "Unassigned" },
    {  597, "Unassigned" },
    {  598, "Unassigned" },
    {  599, "Unassigned" },
    {  600, "Unassigned" },
    {  601, "Unassigned" },
    {  602, "Egypt" },
    {  603, "Algeria" },
    {  604, "Morocco" },
    {  605, "Tunisia" },
    {  606, "Libya" },
    {  607, "Gambia" },
    {  608, "Senegal" },
    {  609, "Mauritania" },
    {  610, "Mali" },
    {  611, "Guinea" },
    {  612, "Côte d'Ivoire" },
    {  613, "Burkina Faso" },
    {  614, "Niger" },
    {  615, "Togo" },
    {  616, "Benin" },
    {  617, "Mauritius" },
    {  618, "Liberia" },
    {  619, "Sierra Leone" },
    {  620, "Ghana" },
    {  621, "Nigeria" },
    {  622, "Chad" },
    {  623, "Central African Rep." },
    {  624, "Cameroon" },
    {  625, "Cabo Verde" },
    {  626, "Sao Tome and Principe" },
    {  627, "Equatorial Guinea" },
    {  628, "Gabon" },
    {  629, "Congo" },
    {  630, "Dem. Rep. of the Congo" },
    {  631, "Angola" },
    {  632, "Guinea-Bissau" },
    {  633, "Seychelles" },
    {  634, "Sudan" },
    {  635, "Rwanda" },
    {  636, "Ethiopia" },
    {  637, "Somalia" },
    {  638, "Djibouti" },
    {  639, "Kenya" },
    {  640, "Tanzania" },
    {  641, "Uganda" },
    {  642, "Burundi" },
    {  643, "Mozambique" },
    {  644, "Unassigned" },
    {  645, "Zambia" },
    {  646, "Madagascar" },
    {  647, "French Departments and Territories in the Indian Ocean" },
    {  648, "Zimbabwe" },
    {  649, "Namibia" },
    {  650, "Malawi" },
    {  651, "Lesotho" },
    {  652, "Botswana" },
    {  653, "Eswatini" },
    {  654, "Comoros" },
    {  655, "South Africa" },
    {  656, "Unassigned" },
    {  657, "Eritrea" },
    {  658, "Saint Helena, Ascension and Tristan da Cunha" },
    {  659, "South Sudan" },
    {  660, "Unassigned" },
    {  661, "Unassigned" },
    {  662, "Unassigned" },
    {  663, "Unassigned" },
    {  664, "Unassigned" },
    {  665, "Unassigned" },
    {  666, "Unassigned" },
    {  667, "Unassigned" },
    {  668, "Unassigned" },
    {  669, "Unassigned" },
    {  670, "Unassigned" },
    {  671, "Unassigned" },
    {  672, "Unassigned" },
    {  673, "Unassigned" },
    {  674, "Unassigned" },
    {  675, "Unassigned" },
    {  676, "Unassigned" },
    {  677, "Unassigned" },
    {  678, "Unassigned" },
    {  679, "Unassigned" },
    {  680, "Unassigned" },
    {  681, "Unassigned" },
    {  682, "Unassigned" },
    {  683, "Unassigned" },
    {  684, "Unassigned" },
    {  685, "Unassigned" },
    {  686, "Unassigned" },
    {  687, "Unassigned" },
    {  688, "Unassigned" },
    {  689, "Unassigned" },
    {  690, "Unassigned" },
    {  691, "Unassigned" },
    {  692, "Unassigned" },
    {  693, "Unassigned" },
    {  694, "Unassigned" },
    {  695, "Unassigned" },
    {  696, "Unassigned" },
    {  697, "Unassigned" },
    {  698, "Unassigned" },
    {  699, "Unassigned" },
    {  700, "Unassigned" },
    {  701, "Unassigned" },
    {  702, "Belize" },
    {  703, "Unassigned" },
    {  704, "Guatemala" },
    {  705, "Unassigned" },
    {  706, "El Salvador" },
    {  707, "Unassigned" },
    {  708, "Honduras" },
    {  709, "Unassigned" },
    {  710, "Nicaragua" },
    {  711, "Unassigned" },
    {  712, "Costa Rica" },
    {  713, "Unassigned" },
    {  714, "Panama" },
    {  715, "Unassigned" },
    {  716, "Peru" },
    {  717, "Unassigned" },
    {  718, "Unassigned" },
    {  719, "Unassigned" },
    {  720, "Unassigned" },
    {  721, "Unassigned" },
    {  722, "Argentina" },
    {  723, "Unassigned" },
    {  724, "Brazil" },
    {  725, "Unassigned" },
    {  726, "Unassigned" },
    {  727, "Unassigned" },
    {  728, "Unassigned" },
    {  729, "Unassigned" },
    {  730, "Chile" },
    {  731, "Unassigned" },
    {  732, "Colombia" },
    {  733, "Unassigned" },
    {  734, "Venezuela" },
    {  735, "Unassigned" },
    {  736, "Bolivia" },
    {  737, "Unassigned" },
    {  738, "Guyana" },
    {  739, "Unassigned" },
    {  740, "Ecuador" },
    {  741, "Unassigned" },
    {  742, "French Guiana" },
    /* French Guiana has been assigned MCC 742, but some networks
     * still operate on MCC 340. */
    {  743, "Unassigned" },
    {  744, "Paraguay" },
    {  745, "Unassigned" },
    {  746, "Suriname" },
    {  747, "Unassigned" },
    {  748, "Uruguay" },
    {  749, "Unassigned" },
    {  750, "Falkland Islands (Malvinas)" },
    {  751, "Unassigned" },
    {  752, "Unassigned" },
    {  753, "Unassigned" },
    {  754, "Unassigned" },
    {  755, "Unassigned" },
    {  756, "Unassigned" },
    {  757, "Unassigned" },
    {  758, "Unassigned" },
    {  759, "Unassigned" },
    {  760, "Unassigned" },
    {  761, "Unassigned" },
    {  762, "Unassigned" },
    {  763, "Unassigned" },
    {  764, "Unassigned" },
    {  765, "Unassigned" },
    {  766, "Unassigned" },
    {  767, "Unassigned" },
    {  768, "Unassigned" },
    {  769, "Unassigned" },
    {  770, "Unassigned" },
    {  771, "Unassigned" },
    {  772, "Unassigned" },
    {  773, "Unassigned" },
    {  774, "Unassigned" },
    {  775, "Unassigned" },
    {  776, "Unassigned" },
    {  777, "Unassigned" },
    {  778, "Unassigned" },
    {  779, "Unassigned" },
    {  780, "Unassigned" },
    {  781, "Unassigned" },
    {  782, "Unassigned" },
    {  783, "Unassigned" },
    {  784, "Unassigned" },
    {  785, "Unassigned" },
    {  786, "Unassigned" },
    {  787, "Unassigned" },
    {  788, "Unassigned" },
    {  789, "Unassigned" },
    {  790, "Unassigned" },
    {  791, "Unassigned" },
    {  792, "Unassigned" },
    {  793, "Unassigned" },
    {  794, "Unassigned" },
    {  795, "Unassigned" },
    {  796, "Unassigned" },
    {  797, "Unassigned" },
    {  798, "Unassigned" },
    {  799, "Unassigned" },
    {  800, "Unassigned" },
    {  801, "Unassigned" },
    {  802, "Unassigned" },
    {  803, "Unassigned" },
    {  804, "Unassigned" },
    {  805, "Unassigned" },
    {  806, "Unassigned" },
    {  807, "Unassigned" },
    {  808, "Unassigned" },
    {  809, "Unassigned" },
    {  810, "Unassigned" },
    {  811, "Unassigned" },
    {  812, "Unassigned" },
    {  813, "Unassigned" },
    {  814, "Unassigned" },
    {  815, "Unassigned" },
    {  816, "Unassigned" },
    {  817, "Unassigned" },
    {  818, "Unassigned" },
    {  819, "Unassigned" },
    {  820, "Unassigned" },
    {  821, "Unassigned" },
    {  822, "Unassigned" },
    {  823, "Unassigned" },
    {  824, "Unassigned" },
    {  825, "Unassigned" },
    {  826, "Unassigned" },
    {  827, "Unassigned" },
    {  828, "Unassigned" },
    {  829, "Unassigned" },
    {  830, "Unassigned" },
    {  831, "Unassigned" },
    {  832, "Unassigned" },
    {  833, "Unassigned" },
    {  834, "Unassigned" },
    {  835, "Unassigned" },
    {  836, "Unassigned" },
    {  837, "Unassigned" },
    {  838, "Unassigned" },
    {  839, "Unassigned" },
    {  840, "Unassigned" },
    {  841, "Unassigned" },
    {  842, "Unassigned" },
    {  843, "Unassigned" },
    {  844, "Unassigned" },
    {  845, "Unassigned" },
    {  846, "Unassigned" },
    {  847, "Unassigned" },
    {  848, "Unassigned" },
    {  849, "Unassigned" },
    {  850, "Unassigned" },
    {  851, "Unassigned" },
    {  852, "Unassigned" },
    {  853, "Unassigned" },
    {  854, "Unassigned" },
    {  855, "Unassigned" },
    {  856, "Unassigned" },
    {  857, "Unassigned" },
    {  858, "Unassigned" },
    {  859, "Unassigned" },
    {  860, "Unassigned" },
    {  861, "Unassigned" },
    {  862, "Unassigned" },
    {  863, "Unassigned" },
    {  864, "Unassigned" },
    {  865, "Unassigned" },
    {  866, "Unassigned" },
    {  867, "Unassigned" },
    {  868, "Unassigned" },
    {  869, "Unassigned" },
    {  870, "Unassigned" },
    {  871, "Unassigned" },
    {  872, "Unassigned" },
    {  873, "Unassigned" },
    {  874, "Unassigned" },
    {  875, "Unassigned" },
    {  876, "Unassigned" },
    {  877, "Unassigned" },
    {  878, "Unassigned" },
    {  879, "Unassigned" },
    {  880, "Unassigned" },
    {  881, "Unassigned" },
    {  882, "Unassigned" },
    {  883, "Unassigned" },
    {  884, "Unassigned" },
    {  885, "Unassigned" },
    {  886, "Unassigned" },
    {  887, "Unassigned" },
    {  888, "Unassigned" },
    {  889, "Unassigned" },
    {  890, "Unassigned" },
    {  891, "Unassigned" },
    {  892, "Unassigned" },
    {  893, "Unassigned" },
    {  894, "Unassigned" },
    {  895, "Unassigned" },
    {  896, "Unassigned" },
    {  897, "Unassigned" },
    {  898, "Unassigned" },
    {  899, "Unassigned" },
    {  900, "Unassigned" },
    {  901, "International Mobile, shared code" },
    {  902, "Other purposes, shared code" },
    {  903, "Unassigned" },
    {  904, "Unassigned" },
    {  905, "Unassigned" },
    {  906, "Unassigned" },
    {  907, "Unassigned" },
    {  908, "Unassigned" },
    {  909, "Unassigned" },
    {  910, "Unassigned" },
    {  911, "Unassigned" },
    {  912, "Unassigned" },
    {  913, "Unassigned" },
    {  914, "Unassigned" },
    {  915, "Unassigned" },
    {  916, "Unassigned" },
    {  917, "Unassigned" },
    {  918, "Unassigned" },
    {  919, "Unassigned" },
    {  920, "Unassigned" },
    {  921, "Unassigned" },
    {  922, "Unassigned" },
    {  923, "Unassigned" },
    {  924, "Unassigned" },
    {  925, "Unassigned" },
    {  926, "Unassigned" },
    {  927, "Unassigned" },
    {  928, "Unassigned" },
    {  929, "Unassigned" },
    {  930, "Unassigned" },
    {  931, "Unassigned" },
    {  932, "Unassigned" },
    {  933, "Unassigned" },
    {  934, "Unassigned" },
    {  935, "Unassigned" },
    {  936, "Unassigned" },
    {  937, "Unassigned" },
    {  938, "Unassigned" },
    {  939, "Unassigned" },
    {  940, "Unassigned" },
    {  941, "Unassigned" },
    {  942, "Unassigned" },
    {  943, "Unassigned" },
    {  944, "Unassigned" },
    {  945, "Unassigned" },
    {  946, "Unassigned" },
    {  947, "Unassigned" },
    {  948, "Unassigned" },
    {  949, "Unassigned" },
    {  950, "Unassigned" },
    {  951, "Unassigned" },
    {  952, "Unassigned" },
    {  953, "Unassigned" },
    {  954, "Unassigned" },
    {  955, "Unassigned" },
    {  956, "Unassigned" },
    {  957, "Unassigned" },
    {  958, "Unassigned" },
    {  959, "Unassigned" },
    {  960, "Unassigned" },
    {  961, "Unassigned" },
    {  962, "Unassigned" },
    {  963, "Unassigned" },
    {  964, "Unassigned" },
    {  965, "Unassigned" },
    {  966, "Unassigned" },
    {  967, "Unassigned" },
    {  968, "Unassigned" },
    {  969, "Unassigned" },
    {  970, "Unassigned" },
    {  971, "Unassigned" },
    {  972, "Unassigned" },
    {  973, "Unassigned" },
    {  974, "Unassigned" },
    {  975, "Unassigned" },
    {  976, "Unassigned" },
    {  977, "Unassigned" },
    {  978, "Unassigned" },
    {  979, "Unassigned" },
    {  980, "Unassigned" },
    {  981, "Unassigned" },
    {  982, "Unassigned" },
    {  983, "Unassigned" },
    {  984, "Unassigned" },
    {  985, "Unassigned" },
    {  986, "Unassigned" },
    {  987, "Unassigned" },
    {  988, "Unassigned" },
    {  989, "Unassigned" },
    {  990, "Unassigned" },
    {  991, "Trial of a proposed new international telecommunication service, shared code" },
    {  992, "Unassigned" },
    {  993, "Unassigned" },
    {  994, "Unassigned" },
    {  995, "Unassigned" },
    {  996, "Unassigned" },
    {  997, "Unassigned" },
    {  998, "Unassigned" },
    {  999, "Private network" },
    { 1665, "Unset" },
    { 0, NULL }
};

value_string_ext E212_codes_ext = VALUE_STRING_EXT_INIT(E212_codes);

static const value_string mcc_mnc_2digits_codes[] = {
    { 20201, "Cosmote AE" },
    { 20202, "Cosmote AE" },
    { 20203, "OTE AE" },
    { 20204, "OTE AE" },
    { 20205, "Vodafone - Panafon" },
    { 20207, "AMD TELECOM AE" },
    { 20209, "WIND HELLAS TELECOMMUNICATIONS" },
    { 20210, "WIND HELLAS TELECOMMUNICATIONS" },
    { 20211, "INTERCONNECT" },
    { 20212, "YUBOTO" },
    { 20213, "COMPATEL LIMITED" },
    { 20214, "CYTA (HELLAS)" },
    { 20215, "BWS" },
    { 20216, "INTER TELECOM" },
    { 20400, "Intovoice B.V." },
    { 20401, "RadioAccess Network Services B.V." },
    { 20402, "T-Mobile Netherlands B.V." },
    { 20403, "Voiceworks B.V." },
    { 20404, "Vodafone Libertel B.V." },
    { 20405, "Elephant Talk Communications Premium Rate Services" },
    { 20406, "Private Mobility Nederland B.V." },
    { 20407, "Tata Communications MOVE B.V." },
    { 20408, "KPN B.V." },
    { 20409, "Lycamobile Netherlands Limited" },
    { 20410, "KPN B.V." },
    { 20411, "Greenet Netwerk B.V." },
    { 20412, "KPN B.V." },
    { 20413, "Unica Installatietechniek B.V." },
    { 20414, "Venus & Mercury Telecom" },
    { 20415, "Ziggo B.V." },
    { 20416, "T-Mobile Netherlands B.V." },
    { 20417, "Lebara Ltd" },
    { 20418, "Ziggo Services B.V." },
    { 20419, "Mixe Communication Solutions B.V." },
    { 20420, "T-Mobile Netherlands B.V." },
    { 20421, "ProRail B.V." },
    { 20422, "Ministerie van Defensie" },
    { 20423, "KORE Wireless Nederland B.V." },
    { 20424, "PM Factory B.V." },
    { 20425, "CapX Nederland" },
    { 20426, "SpeakUp B.V." },
    { 20427, "L-Mobi Mobile B.V." },
    { 20428, "Lancelot B.V." },
    { 20429, "Tismi B.V." },
    { 20430, "ASpider Solutions Nederland B.V." },
    { 20432, "Cubic Telecom Limited" },
    { 20433, "Truphone B.V." },
    { 20460, "Nextgen Mobile Ltd" },
    { 20461, "Alcadis B.V." },
    { 20462, "RGTN Wholesale Netherlands B.V." },
    { 20463, "Messagebird BV" },
    { 20464, "Zetacom B.V." },
    { 20465, "AGMS Nederland B.V." },
    { 20466, "Utility Connect B.V." },
    { 20467, "RadioAccess B.V." },
    { 20468, "Roamware (Netherlands) B.V." },
    { 20469, "KPN B.V." },
    { 20491, "Enexis Netbeheer B.V." },
    { 20601, "Proximus" },
    { 20602, "N.M.B.S" },
    { 20603, "Citymesh" },
    { 20604, "MWINGZ (Proximus/Orange Belgium)" },
    { 20605, "Telenet" },
    { 20606, "Lycamobile sprl" },
    { 20607, "Mundio Mobile Belgium nv" },
    { 20608, "Nethys" },
    { 20610, "Orange Belgium" },
    { 20611, "L-Mobi Mobile" },
    { 20620, "Telenet Group" },
    { 20622, "FEBO Telecom" },
    { 20625, "Voyacom" },
    { 20628, "BICS SA" },
    { 20629, "TISMI" },
    { 20630, "Unleashed" },
    { 20633, "Ericsson *test use only*" },
    { 20634, "ONOFFAPP" },
    { 20650, "IP Nexia" },
    { 20699, "e-BO Enterprises" },
    { 20801, "Orange" }, /* Annex E: MSIN range 0100000000-0199999999 are Orange/Monoco Télécom */
    { 20802, "Orange" },
    { 20803, "MobiquiThings" },
    { 20804, "Sisteer" },
    { 20805, "Globalstar Europe" },
    { 20806, "Globalstar Europe" },
    { 20807, "Globalstar Europe" },
    { 20808, "Société Française du Radiotéléphone" },
    { 20809, "Société Française du Radiotéléphone" },
    { 20810, "Société Française du Radiotéléphone" },
    { 20811, "Société Française du Radiotéléphone" },
    { 20812, "Truphone France" },
    { 20813, "Société Française du Radiotéléphone" },
    { 20814, "RFF" },
    { 20815, "Free Mobile" },
    { 20816, "Free Mobile" },
    { 20817, "Legos" },
    { 20819, "Haute-Garonne numérique" },
    { 20820, "Bouygues Telecom" },
    { 20821, "Bouygues Telecom" },
    { 20822, "Transatel" },
    { 20823, "Syndicat mixte ouvert Charente Numérique" },
    { 20824, "MobiquiThings" },
    { 20825, "Lycamobile" },
    { 20826, "Bouygues Telecom Business - Distribution" },
    { 20827, "Coriolis Telecom" },
    { 20828, "Airmob Infra Full" },
    { 20829, "Cubic télécom France" },
    { 20830, "Syma Mobile" },
    { 20831, "Vectone Mobile" },
    { 20832, "Orange" },
    { 20833, "Syndicat mixte La Fibre64" },
    { 20834, "Cellhire (France)" },
    { 20835, "Free mobile" },
    { 20836, "Free mobile" },
    { 20837, "IP Directions" },
    { 20838, "Lebara France Limited" },
    { 20839, "Netwo" },
    { 20886, "SEM@FOR77" },
    { 20887, "AIRBUS DEFENCE AND SPACE SAS" },
    { 20888, "Bouygues Telecom" },
    { 20889, "Hub One" },
    { 20891, "Orange" },
    { 20893, "Thales communications & Security" },
    { 20894, "Halys" },
    { 20895, "Orange" },
    { 20896, "Région Bourgogne-Franche-Comté" },
    { 20897, "Thales communications & Security" },
    { 20898, "Société Air France" },
    { 21210, "Monaco Telecom" },
    { 21303, "Mobiland" },
    { 21401, "Vodafone España, SAU" },
    { 21402, "Alta Tecnologia en Comunicacions, S.L." },
    { 21403, "France Telecom España, SA" },
    { 21404, "Xfera Móviles, S.A." },
    { 21405, "Telefónica Móviles España, SAU" },
    { 21406, "Vodafone España, SAU" },
    { 21407, "Telefónica Móviles España, SAU" },
    { 21408, "Euskaltel, SA" },
    { 21409, "France Telecom España, SA" },
    { 21410, "ZINNIA TELECOMUNICACIONES, S.L.U." },
    { 21411, "TELECOM CASTILLA-LA MANCHA, S.A." },
    { 21412, "VENUS MOVIL, S.L. UNIPERSONAL" },
    { 21413, "SYMA MOBILE ESPAÑA, S.L." },
    { 21414, "AVATEL MÓVIL, S.L.U." },
    { 21416, "R CABLE Y TELECOMUNICACIONES GALICIA, S.A." },
    { 21417, "R Cable y Telecomunicaciones Galicia, SA" },
    { 21419, "E-Plus Móviles, SL" },
    { 21421, "ORANGE ESPAGNE, S.A. UNIPERSONAL" },
    { 21422, "Best Spain Telecom, SL" },
    { 21423, "Xfera Móviles, S.A.U." },
    { 21424, "VODAFONE ESPAÑA, S.A.U." },
    { 21425, "XFERA MÓVILES, S.A. UNIPERSONAL" },
    { 21426, "Lleida Networks Serveis Telemátics, SL" },
    { 21427, "SCN Truphone SL" },
    { 21428, "Consorcio de Telecomunicaciones Avanzadas, S.A." },
    { 21429, "XFERA MÓVILES, S.A.U." },
    { 21430, "Compatel Limited" },
    { 21431, "Red Digital De Telecomunicaciones de las Islas Baleares, S.L." },
    { 21433, "XFERA MÓVILES, S.A.U." },
    { 21434, "AIRE NETWORKS DEL MEDITERRÁNEO, S.L. UNIPERSONAL" },
    { 21435, "INGENIUM OUTSOURCING SERVICES, S.L." },
    { 21436, "ALAI OPERADOR DE TELECOMUNICACIONES, S.L." },
    { 21437, "VODAFONE ESPAÑA, S.A.U." },
    { 21438, "Telefónica Móviles España, SAU" },
    { 21451, "ENTIDAD PÚBLICA EMPRESARIAL ADMINISTRADOR DE INFRAESTRUCTURAS FERROVIARIAS" },
    { 21601, "Telenor Hungary Ltd" },
    { 21602, "MVM NET Ltd." },
    { 21603, "DIGI Telecommunication Ltd." },
    { 21604, "Pro-M PrCo. Ltd." },
    { 21620, "Yettel Hungary Ltd." },
    { 21630, "Magyar Telecom Plc" },
    { 21670, "Vodafone" },
    { 21671, "Vodafone Hungary Ltd" },
    { 21699, "MÁV Co." },
    { 21803, "Eronet Mobile Communications Ltd." },
    { 21805, "MOBI'S (Mobilina Srpske)" },
    { 21890, "GSMBIH" },
    { 21901, "T-Mobile Hrvatska d.o.o./T-Mobile Croatia LLC" },
    { 21902, "Tele2/Tele2 d.o.o." },
    { 21910, "VIPnet/VIPnet d.o.o." },
    { 22001, "Telenor d.o.o." },
    { 22003, "Telekom Srbija a.d." },
    { 22005, "Vip mobile d.o.o." },
    { 22007, "Orion telekom d.o.o." },
    { 22009, "MUNDIO MOBILE d.o.o." },
    { 22011, "GLOBALTEL d.o.o." },
    { 22101, "Telecom of Kosovo J.S.C." },
    { 22102, "IPKO Telecommunications LLC" },
    { 22106, "Dardafon.Net LLC" },
    { 22107, "Dukagjini Telecommunications LLC" },
    { 22201, "Telecom Italia Mobile (TIM)" },
    { 22202, "Elsacom" },
    { 22210, "Omnitel Pronto Italia (OPI)" },
    { 22277, "IPSE 2000" },
    { 22288, "Wind" },
    { 22298, "Blu" },
    { 22299, "H3G" },
    { 22601, "Vodafone" },
    { 22602, "Romtelecom" },
    { 22603, "Cosmote" },
    { 22604, "Cosmote" },
    { 22605, "Digi.Mobil" },
    { 22606, "Cosmote" },
    { 22610, "Orange" },
    { 22611, "Enigma-System" },
    { 22801, "Swisscom Schweiz AG" },
    { 22802, "Sunrise Communications AG" },
    { 22803, "Salt Mobile SA" },
    { 22805, "Comfone AG" },
    { 22806, "SBB AG" },
    { 22808, "Sunrise Communications AG" },
    { 22809, "Comfone AG" },
    { 22811, "Swisscom Broadcast AG" },
    { 22812, "Sunrise Communications AG" },
    { 22851, "Bebbicell AG" },
    { 22853, "upc Cablecom GmbH" },
    { 22854, "Lycamobile AG" },
    { 22855, "WeMobile SA" },
    { 22857, "Mitto AG" },
    { 22858, "Beeone Communications SA" },
    { 22859, "Vectone Mobile Limited, London" },
    { 22860, "Sunrise Communications AG" },
    { 22862, "Telecom26 AG" },
    { 22863, "Fink Telecom Services" },
    { 22864, "NTH AG" },
    { 22866, "Inovia Services SA" },
    { 22867, "Datatrade Managed AG" },
    { 22868, "Intellico AG" },
    { 22869, "MTEL Schweiz GmbH" },
    { 22870, "Tismi BV" },
    { 22871, "Spusu AG" },
    { 23001, "T-Mobile Czech Republic a.s." },
    { 23002, "O2 Czech Republic a.s." },
    { 23003, "Vodafone Czech Republic a.s." },
    { 23004, "Nordic Telecom Regional s.r.o." },
    { 23005, "PODA a.s." },
    { 23006, "Nordic Telecom 5G a.s." },
    { 23007, "T-Mobile Czech Republic a.s." },
    { 23008, "Compatel s.r.o" },
    { 23009, "Uniphone, s.r.o" },
    { 23011, "incrate s.r.o" },
    { 23098, "Sprava zeleznic, statni organizace" },
    { 23101, "Orange, GSM" },
    { 23102, "Eurotel, GSM & NMT" },
    { 23104, "Eurotel, UMTS" },
    { 23105, "Orange, UMTS" },
    { 23201, "A1 Telekom Austria AG" },
    { 23202, "A1 Telekom Austria AG" },
    { 23203, "T-Mobile Austria GmbH" },
    { 23204, "T-Mobile Austria GmbH" },
    { 23205, "Hutchison Drei Austria GmbH" },
    { 23207, "T-Mobile Austria GmbH" },
    { 23208, "Lycamobile Austria Ltd" },
    { 23209, "A1 Telekom Austria AG" },
    { 23210, "Hutchison Drei Austria GmbH" },
    { 23211, "A1 Telekom Austria AG" },
    { 23212, "A1 Telekom Austria AG" },
    { 23213, "UPC Austria Services GmbH" },
    { 23214, "Hutchison Drei Austria GmbH" },
    { 23215, "Mundio Mobile (Austria) Ltd" },
    { 23216, "Hutchison Drei Austria GmbH" },
    { 23217, "MASS Response Service GmbH" },
    { 23218, "smartspace GmbH" },
    { 23219, "Tele2 Telecommunication GmbH" },
    { 23220, "Mtel Austrija GmbH" },
    { 23291, "ÖBB - Infrastruktur AG" },
    { 23400, "British Telecom" },
    { 23401, "Vectone Mobile Limited" },
    { 23402, "Telefonica UK Limited" },
    { 23403, "Jersey Airtel Limited" },
    { 23404, "FMS Solutions Limited" },
    { 23405, "COLT Mobile Telecommunications Limited" },
    { 23406, "Internet Computer Bureau Limited" },
    { 23407, "Vodafone Ltd (C&W)" },
    { 23408, "BT OnePhone Limited" },
    { 23409, "Tismi BV" },
    { 23410, "Telefonica UK Limited" },
    { 23411, "Telefonica UK Limited" },
    { 23412, "Network Rail Infrastructure Limited" },
    { 23413, "Network Rail Infrastructure Limited" },
    { 23414, "LINK MOBILITY UK LTD" },
    { 23415, "Vodafone Limited" },
    { 23416, "TalkTalk Communications Limited" },
    { 23417, "FleXtel Limited" },
    { 23418, "Cloud9 Communications Limited" },
    { 23419, "TeleWare Group PLC" },
    { 23420, "Hutchison 3G UK Limited" },
    { 23422, "Telesign Mobile Limited" },
    { 23423, "Icron Network Limited" },
    { 23424, "Stour Marine Limited" },
    { 23425, "Truphone Limited" },
    { 23426, "Lycamobile UK Limited" },
    { 23427, "Tata Communications Move UK Ltd" },
    { 23428, "Marathon Telecom Limited" },
    { 23429, "(AQ) LIMITED" },
    { 23430, "EE Limited ( TM)" },
    { 23431, "EE Limited ( TM)" },
    { 23432, "EE Limited ( TM)" },
    { 23433, "EE Limited (Orange)" },
    { 23434, "EE Limited (Orange)" },
    { 23436, "Sure (Isle of Man) Limited" },
    { 23437, "Synectiv Ltd" },
    { 23438, "Virgin Mobile Telecoms Limited" },
    { 23439, "Gamma Telecom Holdings Ltd" },
    { 23440, "Mass Response Service GmbH" },
    { 23450, "JT (Jersey) Limited" },
    { 23451, "UK Broadband Limited" },
    { 23452, "Shyam Telecom UK Ltd" },
    { 23453, "Tango Networks UK Ltd" },
    { 23454, "The Carphone Warehouse Limited" },
    { 23455, "Sure (Guernsey) Limited" },
    { 23456, "The National Cyber Security Centre" },
    { 23457, "Sky UK Limited" },
    { 23458, "MANX TELECOM TRADING LIMITED" },
    { 23459, "Limitless Mobile Ltd" },
    { 23470, "AMSUK Limited" },
    { 23471, "Home Office" },
    { 23472, "Hanhaa Limited" },
    { 23473, "BlueWave Communications" },
    { 23474, "Pareteum Europe B.V." },
    { 23476, "British Telecom" },
    { 23477, "Vodafone Limited" },
    { 23478, "Airwave Solutions Ltd" },
    { 23486, "EE Limited ( TM)" },
    { 23488, "Telet Research (N.I.) Limited" },
    { 23500, "Vectone Mobile Limited" },
    { 23501, "EE Limited ( TM)" },
    { 23502, "EE Limited ( TM)" },
    { 23503, "UK Broadband Limited" },
    { 23504, "University Of Strathclyde" },
    { 23506, "University Of Strathclyde" },
    { 23507, "University Of Strathclyde" },
    { 23508, "Spitfire Network Services Limited" },
    { 23577, "British Telecom" },
    { 23588, "Telet Research (N.I.) Limited" },
    { 23591, "Vodafone Limited" },
    { 23592, "Vodafone Ltd (C&W)" },
    { 23594, "Hutchison 3G UK Limited" },
    { 23595, "Network Rail Infrastructure Limited" },
    { 23801, "TDC A/S" },
    { 23802, "Telenor" },
    { 23803, "Syniverse Technologies" },
    { 23805, "Dansk Beredskabskommunikation" },
    { 23806, "Hi3G" },
    { 23808, "Voxbone" },
    { 23809, "Dansk Beredskabskommunikation" },
    { 23810, "TDC A/S" },
    { 23811, "Dansk Beredskabskommunikation" },
    { 23812, "Lycamobile Denmark Ltd." },
    { 23813, "Compatel Limited" },
    { 23814, "Monty UK Global Limited" },
    { 23815, "Ice Danmark ApS" },
    { 23816, "Tismi B.V." },
    { 23817, "Gotanet AB" },
    { 23818, "Cubic Telecom" },
    { 23820, "Telia" },
    { 23823, "Banedanmark" },
    { 23825, "Viahub (SMS Provider Corp.)" },
    { 23828, "LINK Mobility A/S" },
    { 23830, "Interactive Digital Media GmbH" },
    { 23842, "Greenwave Mobile IoT ApS" },
    { 23866, "TT-Netvaerket P/S" },
    { 23873, "Onomondo ApS" },
    { 23877, "Telenor" },
    { 23888, "Cobira ApS" },
    { 23896, "Telia Danmark" },
    { 24001, "Telia Sverige AB" },
    { 24002, "Hi3G Access AB" },
    { 24003, "Netett Sverige AB" },
    { 24004, "3G Infrastructure Services AB" },
    { 24005, "Svenska UMTS-Nät AB" },
    { 24006, "Telenor Sverige AB" },
    { 24007, "Tele2 Sverige AB" },
    { 24008, "Telenor Sverige AB" },
    { 24009, "Com4 Sweden AB" },
    { 24010, "Tele2 Sverige AB" },
    { 24011, "ComHem AB" },
    { 24012, "Lycamobile Sweden Limited" },
    { 24013, "Bredband2 Allmänna IT AB" },
    { 24014, "Tele2 Business AB" },
    { 24015, "Sierra Wireless Sweden AB" },
    { 24016, "42 Telecom AB" },
    { 24017, "Götalandsnätet AB" },
    { 24018, "Generic Mobile Systems Sweden AB" },
    { 24019, "Vecton Mobile (Sweden) Ltd" },
    { 24020, "Sierra Wireless Messaging AB" },
    { 24021, "Trafikverket centralfunktion IT" },
    { 24022, "EuTel AB" },
    { 24023, "Infobip LTD (UK)" },
    { 24024, "Net4Mobility HB" },
    { 24025, "Monty UK Global Limited" },
    { 24026, "Twilio Ireland Ltd." },
    { 24027, "GlobeTouch AB" },
    { 24028, "LINK Mobile A/S" },
    { 24029, "MI Carrier Services AB" },
    { 24031, "RebTel Networks AB" },
    { 24032, "Compatel Limited" },
    { 24033, "Mobile Arts AB" },
    { 24034, "Trafikverket centralfunktion IT" },
    { 24035, "42 Telecom LTD" },
    { 24036, "interactive digital media GmbH" },
    { 24037, "Sinch Sweden AB" },
    { 24038, "Voxbone SA" },
    { 24039, "Primlight AB" },
    { 24040, "Netmore Group AB" },
    { 24041, "Telenor Sverige AB" },
    { 24042, "Telenor Connexion AB" },
    { 24043, "MobiWeb Ltd." },
    { 24044, "Telenabler AB" },
    { 24045, "Spirius AB" },
    { 24046, "SMS Provider Corp." },
    { 24047, "Viatel Sweden AB" },
    { 24048, "Tismi BV" },
    { 24049, "Telia Sverige AB" },
    { 24060, "Västra Götalandsregionen (temporary assigned until 2026-12-31)" },
    { 24061, "MessageBird B.V." },
    { 24063, "Fink Telecom Services" },
    { 24065, "shared use for closed networks" },
    { 24066, "shared use for closed networks" },
    { 24067, "shared use for test purpose" },
    { 24068, "shared use for test purpose" },
    { 24069, "crisis management after determination by the Swedish Post- and Telecom Authority" },
    { 24099, "Unassigned (may be used for test purposes)" },
    { 24201, "Telenor Norge AS" },
    { 24202, "Telia Norge AS" },
    { 24203, "Teletopia Gruppen AS" },
    { 24205, "Telia Norge AS" },
    { 24206, "ICE Norge AS" },
    { 24207, "Phonero AS" },
    { 24208, "TDC AS" },
    { 24209, "Com4 AS" },
    { 24210, "Norwegian Communications Authority" },
    { 24211, "Systemnet AS" },
    { 24212, "Telenor Norge AS" },
    { 24214, "ICE Communication Norge AS" },
    { 24220, "Jernbaneverket" },
    { 24221, "Jernbaneverket" },
    { 24223, "Lycamobile Norway Ltd" },
    { 24299, "Tampnet AS" },
    { 24403, "DNA Oy" },
    { 24404, "DNA Oy" },
    { 24405, "Elisa Oyj" },
    { 24406, "Elisa Oyj" },
    { 24407, "Nokia Solutions and Networks Oy" },
    { 24408, "Nokia Solutions and Networks Oy" },
    { 24409, "Nokia Solutions and Networks Oy" },
    { 24410, "Viestintävirasto" },
    { 24411, "Viestintävirasto" },
    { 24412, "DNA Oy" },
    { 24413, "DNA Oy" },
    { 24414, "Ålands Telekommunikation Ab" },
    { 24415, "Satakunnan ammattikorkeakoulu Oy" },
    { 24417, "Liikennevirasto" },
    { 24421, "Elisa Oyj" },
    { 24422, "EXFO Oy" },
    { 24423, "EXFO Oy" },
    { 24424, "TTY-säätiö" },
    { 24426, "Compatel Limited" },
    { 24427, "Teknologian tutkimuskeskus VTT Oy" },
    { 24431, "Kuiri Mobile Oy" },
    { 24432, "Voxbone S.A." },
    { 24433, "Virve Tuotteet ja Palvelut Oy" },
    { 24434, "Bittium Wireless Oy" },
    { 24435, "Ukkoverkot Oy" },
    { 24436, "TeliaSonera Finland Oyj" },
    { 24437, "Tismi BV" },
    { 24438, "Nokia Solutions and Networks Oy" },
    { 24439, "Nokia Solutions and Networks Oy" },
    { 24440, "Nokia Solutions and Networks Oy" },
    { 24441, "Nokia Solutions and Networks Oy" },
    { 24491, "TeliaSonera Finland Oyj" },
    { 24492, "TeliaSonera Finland Oyj" },
    { 24601, "Omnitel" },
    { 24602, "Bité GSM" },
    { 24603, "Tele2" },
    { 24701, "Latvijas Mobilais Telefons SIA" },
    { 24702, "Tele2" },
    { 24703, "Telekom Baltija" },
    { 24704, "Beta Telecom" },
    { 24705, "Bite Mobile" },
    { 24706, "Rigatta" },
    { 24707, "Master Telecom" },
    { 24708, "IZZI" },
    { 24709, "SIA \"Camel Mobile\"" },
    { 24801, "AS Eesti Telekom" },
    { 24802, "RLE" },
    { 24803, "Tele2" },
    { 24804, "OY Top Connect" },
    { 24805, "CSC Telecom Estonia OÜ" },
    { 24807, "Televõrgu AS" },
    { 24811, "UAB Raystorm Eesti filiaal" },
    { 24812, "Ntel Solutions OÜ" },
    { 24813, "Telia Eesti AS" },
    { 24814, "Estonian Crafts OÜ" },
    { 24816, "SmartTel Plus OÜ" },
    { 24817, "Baltergo OÜ" },
    { 24818, "CLOUD COMMUNICATIONS OÜ" },
    { 24819, "OkTelecom OÜ" },
    { 24820, "DOTT Telecom OÜ" },
    { 24821, "Tismi B.V." },
    { 24822, "M2MConnect OÜ" },
    { 24824, "Novametro OÜ" },
    { 24826, "It-Decision Telecom OÜ" },
    { 24828, "Nord Connect OÜ" },
    { 24829, "SkyTel OÜ" },
    { 24830, "Mediafon Carrier Services OÜ" },
    { 24831, "YATECO OÜ" },
    { 24832, "Narayana OÜ" },
    { 24871, "Siseministeerium (Ministry of Interior)" },
    { 25001, "Mobile Telesystems" },
    { 25002, "Megafon" },
    { 25003, "Nizhegorodskaya Cellular Communications" },
    { 25004, "Sibchallenge" },
    { 25005, "Mobile Comms System" },
    { 25007, "BM Telecom" },
    { 25010, "Don Telecom" },
    { 25011, "Orensot" },
    { 25012, "Baykal Westcom" },
    { 25013, "Kuban GSM" },
    { 25016, "New Telephone Company" },
    { 25017, "Ermak RMS" },
    { 25019, "Volgograd Mobile" },
    { 25020, "ECC" },
    { 25028, "Extel" },
    { 25039, "Uralsvyazinform" },
    { 25044, "Stuvtelesot" },
    { 25092, "Printelefone" },
    { 25093, "Telecom XXI" },
    { 25099, "Beeline" },
    { 25501, "\"VF UKRAINE\" PrJSC" },
    { 25502, "\"Kyivstar\" PrJSC" },
    { 25503, "\"Kyivstar\" PrJSC" },
    { 25504, "\"Intertelecom\" LLC" },
    { 25506, "\"lifecell\" LLC" },
    { 25507, "\"TriMob\" LLC" },
    { 25508, "\"Ukrtelecom\" JSC" },
    { 25509, "\"Farlep-Invest\", PrJSC" },
    { 25510, "\"Atlantis Telecom\", LLC" },
    { 25521, "\"Telesystems of Ukraine\" PrJSC" },
    { 25701, "MDC Velcom" },
    { 25702, "MTS" },
    { 25703, "BelCel Joint Venture (JV)" },
    { 25704, "Closed joint-stock company \"Belarusian telecommunication network\"" },
    { 25705, "Republican Unitary Telecommunication Enterprise (RUE) Beltelecom (National Telecommunications Operator of the Republic of Belarus)" },
    { 25706, "Belorussian Cloud Technologies" },
    { 25901, "Orange Moldova GSM" },
    { 25902, "Moldcell GSM" },
    { 25905, "J.S.C. Moldtelecom/3G UMTS (W-CDMA)" },
    { 25999, "J.S.C. Moldtelecom" },
    { 26001, "Plus / Polkomtel S.A." },
    { 26002, "T-Mobile / PTC S.A." },
    { 26003, "Orange / PTK Centertel Sp. z o.o." },
    { 26004, "LTE / CenterNet S.A." },
    { 26005, "Orange(UMTS) / PTK Centertel Sp. z o.o." },
    { 26006, "Play / P4 Sp. z o.o." },
    { 26007, "Netia / Netia S.A." },
    { 26008, "E-Telko / E-Telko Sp. z o.o." },
    { 26009, "Lycamobile / Lycamobile Sp. z o.o." },
    { 26010, "Sferia / Sferia S.A." },
    { 26011, "Nordisk Polska / Nordisk Polska Sp. z o.o." },
    { 26012, "Cyfrowy Polsat / Cyfrowy Polsat S.A." },
    { 26013, "Sferia / Sferia S.A." },
    { 26014, "Sferia / Sferia S.A." },
    { 26015, "CenterNet / CenterNet S.A." },
    { 26016, "Mobyland / Mobyland Sp. z o.o." },
    { 26017, "Aero 2 / Aero 2 Sp. z o.o." },
    { 26018, "AMD Telecom / AMD Telecom S.A." },
    { 26019, "Teleena / Teleena Holding BV" },
    { 26020, "Mobile.Net / Mobile.Net Sp. z o.o." },
    { 26021, "Exteri / Exteri Sp. z o.o." },
    { 26022, "Arcomm / Arcomm Sp. z o.o." },
    { 26023, "Amicomm / Amicomm Sp. z o.o." },
    { 26024, "WideNet / WideNet Sp. z o.o." },
    { 26025, "BS&T / Best Solutions & Technology Sp. z o.o." },
    { 26026, "ATE / ATE-Advanced Technology & Experience Sp. z o.o." },
    { 26027, "Intertelcom / Intertelcom Sp. z o.o." },
    { 26028, "PhoneNet / PhoneNet Sp. z o.o." },
    { 26029, "Interfonica / Interfonica Sp. z o.o." },
    { 26030, "GrandTel / GrandTel Sp. z o.o." },
    { 26031, "Phone IT / Phone IT Sp. z o.o." },
    { 26032, "Compatel Ltd / COMPATEL LIMITED" },
    { 26033, "Truphone Poland / Truphone Poland Sp. Z o.o." },
    { 26034, "T-Mobile / PTC S.A." },
    { 26098, "Play (testowy) / P4 Sp. z o.o." },
    { 26201, "Telekom Deutschland GmbH" },
    { 26202, "Vodafone GmbH" },
    { 26203, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26204, "Vodafone GmbH" },
    { 26205, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26206, "Telekom Deutschland GmbH" },
    { 26207, "Telefónica Germany GmbH & Co. OHG" },
    { 26208, "Telefónica Germany GmbH & Co. OHG" },
    { 26209, "Vodafone GmbH" },
    { 26210, "DB Netz AG" },
    { 26211, "Telefónica Germany GmbH & Co. OHG" },
    { 26212, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26215, "AirData AG" },
    { 26216, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26217, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26218, "NetCologne Gesellschaft für Telekommunikation mbH" },
    { 26219, "Inquam Deutschland GmbH" },
    { 26220, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26221, "Multiconnect GmbH" },
    { 26222, "Sipgate Wireless GmbH" },
    { 26223, "Drillisch Online AG" },
    { 26242, "Vodafone GmbH" },
    { 26243, "Vodafone GmbH" },
    { 26272, "Ericsson GmbH" },
    { 26273, "Xantaro Deutschland GmbH" },
    { 26274, "Qualcomm CDMA Technologies GmbH" },
    { 26275, "Core Network Dynamics GmbH" },
    { 26277, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26278, "Telekom Deutschland GmbH" },
    { 26601, "Gibtel" },
    { 26603, "GibFibre Ltd (trading as “Gibfibrespeed”)" },
    { 26609, "Eazi Telecom Ltd (trading as “Limba”)" },
    { 26801, "Vodafone Portugal - Comunicações Pessoais, S.A." },
    { 26803, "NOS Comunicações, S.A." },
    { 26804, "Lycamobile Portugal, Lda" },
    { 26806, "MEO - Serviços de Comunicações e Multimédia, S.A." },
    { 26811, "Compatel, Limited" },
    { 26812, "Infraestruturas de Portugal, S.A." },
    { 26813, "G9Telecom, S.A." },
    { 26880, "MEO - Serviços de Comunicações e Multimédia, S.A." },
    { 27001, "POST Luxembourg" },
    { 27002, "MTX Connect S.à r.l." },
    { 27007, "Bouygues Telecom S.A." },
    { 27010, "Join Experience S.A." },
    { 27077, "Proximus Luxembourgh S.A." },
    { 27078, "Interactive Digital Media GmbH" },
    { 27079, "Mitto A.G." },
    { 27080, "Syniverse Technologies S.à r.l." },
    { 27081, "E-Lux Mobile Telecommunication S.A." },
    { 27099, "Orange Communications Luxembourg S.A." },
    { 27201, "Vodafone Ireland Plc" },
    { 27202, "Three Ireland Services (Hutchison) Ltd" },
    { 27203, "Eircom Ltd" },
    { 27205, "Three Ireland (Hutchison) Ltd" },
    { 27207, "Eircom Ltd" },
    { 27208, "Eircom Ltd" },
    { 27211, "Liffey Telecom Ltd" },
    { 27213, "Lycamobile Ireland Ltd" },
    { 27215, "Virgin Media Ireland Ltd" },
    { 27216, "Carphone Warehouse Ireland Mobile Ltd" },
    { 27217, "Three Ireland (Hutchison) Ltd" },
    { 27218, "Cubic Telecom Limited" },
    { 27221, "Net Feasa Limited" },
    { 27268, "Office of the Government Chief Information Officer" },
    { 27401, "Iceland Telecom Ltd." },
    { 27402, "Og fjarskipti hf (Vodafone Iceland)" }, /* Annex E: MSIN beginning with 9 are P/F Kall, reg. No 2868 (Vodafone FO) in Faroe Islands */
    { 27403, "Og fjarskipti hf (Vodafone Iceland)" },
    { 27404, "IMC Islande ehf" },
    { 27407, "IceCell ehf" },
    { 27601, "Albanian Mobile Communications (AMC)" },
    { 27602, "Vodafone Albania" },
    { 27603, "Eagle Mobile" },
    { 27604, "Mobile 4 AL" },
    { 27801, "Epic Communications Ltd" },
    { 27811, "YOM" },
    { 27821, "GO mobile" },
    { 27830, "GO Mobile" },
    { 27877, "Melita Ltd" },
    { 28001, "CYTA" },
    { 28002, "CYTA" },
    { 28010, "Scancom (Cyprus) Ltd." },
    { 28020, "PrimeTel PLC" },
    { 28022, "Lemontel Ltd" },
    { 28201, "Geocell Ltd." },
    { 28202, "Magti GSM Ltd." },
    { 28203, "Iberiatel Ltd." },
    { 28204, "Mobitel Ltd." },
    { 28205, "Silknet JSC" },
    { 28206, "JSC Compatel" },
    { 28207, "GLOBALCELL LTD" },
    { 28208, "Silknet GSC" },
    { 28209, "Gmobile LTD" },
    { 28210, "Premium Net International SRL LTD" },
    { 28211, "Mobilive LTD" },
    { 28212, "\"Telecom1\" LTD" },
    { 28213, "\"Asanet\" LTD" },
    { 28214, "\"Datahouseglobal” LTD" },
    { 28215, "\"Servicebox\" LTD" },
    { 28222, "\"Myphone\" LTD" },
    { 28401, "Mobiltel EAD" },
    { 28405, "Globul" },
    { 28601, "Turkcell" },
    { 28602, "Telsim GSM" },
    { 28603, "Aria" },
    { 28604, "Aycell" },
    { 28801, "Faroese Telecom - GSM" },
    { 28802, "Kall GSM" },
    { 28803, "Sp/f Tosa" },
    { 28810, "Faroese Telecom" },
    { 29001, "Tele Greenland" },
    { 29002, "inu:it a/s" },
    { 29003, "GTV" },
    { 29201, "Prima San Marino / San Marino Telecom" },
    { 29310, "Slovenske železnice – Infrastruktura d.o.o." },
    { 29311, "BeeIN d.o.o." },
    { 29320, "Compatel Limited" },
    { 29321, "Novatel d.o.o." },
    { 29340, "A1 Slovenija d.d." },
    { 29341, "Telekom Slovenije d.d." },
    { 29364, "T-2 d.o.o." },
    { 29370, "Telemach d.o.o." },
    { 29401, "T-Mobile" },
    { 29402, "Cosmofon" },
    { 29403, "Nov Operator" },
    { 29404, "Company for telecommunications LYCAMOBILE LLC-Skopje" },
    { 29410, "WTI Macedonia" },
    { 29411, "MOBIK TELEKOMUNIKACII DOOEL- Skopje" },
    { 29412, "MTEL DOOEL Skopje" },
    { 29501, "Swisscom Schweiz AG" },
    { 29502, "Salt (Liechtenstein) AG" },
    { 29505, "Telecom Liechtenstein AG" },
    { 29506, "Cubic AG" },
    { 29507, "First Mobile AG" },
    { 29509, "Emnify GmbH" },
    { 29510, "SORACOM CORPORATION, LTD." },
    { 29511, "DIMOCO Messaging AG" },
    { 29701, "Telenor Montenegro" },
    { 29702, "Crnogorski Telekom" },
    { 29703, "Mtel Montenegro" },
    { 30801, "SAS SPM Telecom" },
    { 30802, "Globaltel" },
    { 30803, "SAS SPM Telecom" },
    { 34001, "Orange Caraïbe" },
    { 34002, "Outremer Telecom" },
    { 34003, "United telecommunications services Caraïbe" },
    { 34008, "Dauphin Telecom" },
    { 34009, "Free Caraïbe" },
    { 34010, "Guadeloupe Téléphone Mobile" },
    { 34011, "Guyane Téléphone Mobile" },
    { 34012, "Martinique Téléphone Mobile" },
    { 34020, "Digicel Antilles Françaises Guyane" },
    { 35005, "Telecom Networks" },
    { 35011, "Deltronics" },
    { 35015, "FKB Net Ltd." },
    { 36251, "TELCELL GSM" },
    { 36269, "CT GSM" },
    { 36291, "SETEL GSM" },
    { 36301, "SETAR N.V." },
    { 36302, "DIGICEL ARUBA" },
    { 36439, "Bahamas Telecommunications Company Limited" },
    { 36449, "NewCo2015 Limited" },
    { 36801, "ETECSA" },
    { 37001, "Orange Dominicana, S.A." },
    { 37002, "Verizon Dominicana S.A." },
    { 37003, "Tricom S.A." },
    { 37004, "CentennialDominicana" },
    { 37201, "Comcel" },
    { 37202, "Digicel" },
    { 37203, "Rectel" },
    { 37412, "TSTT Mobile" },
    { 40001, "\"Azercell Telecom\" LLC" },
    { 40002, "\"Bakcell\" LLC" },
    { 40003, "\"Catel\" LLC" },
    { 40004, "\"Azerfon\" LLC" },
    { 40005, "Special State Protection Service of the Republic of Azerbaijan" },
    { 40006, "\"Nakhtel\" LLC" },
    { 40101, "Kar-Tel llc" },
    { 40102, "TSC Kazak Telecom" },
    { 40107, "Altel / Dalacom" },
    { 40108, "Kazakhtelecom CDMA2000" },
    { 40177, "Tele2 / NEO" },
    { 40211, "Bhutan Telecom Limited (Bmobile)" },
    { 40217, "Bhutan Telecom Limited (Bmobile)" },
    { 40277, "Tashi InfoComm Limited (Tashi Cell)" },
    { 40400, "Dishnet Wireless Ltd, Madhya Pradesh" },
    { 40401, "Aircell Digilink India Ltd., Haryana" },
    { 40402, "Bharti Airtel Ltd., Punjab" },
    { 40403, "Bharti Airtel Ltd., H.P." },
    { 40404, "Idea Cellular Ltd., Delhi" },
    { 40405, "Fascel Ltd., Gujarat" },
    { 40406, "Bharti Airtel Ltd., Karnataka" },
    { 40407, "Idea Cellular Ltd., Andhra Pradesh" },
    { 40409, "Reliance Telecom Ltd., Assam" },
    { 40410, "Bharti Airtel Ltd., Delhi" },
    { 40411, "Hutchison Essar Mobile Services Ltd, Delhi" },
    { 40412, "Idea Mobile Communications Ltd., Haryana" },
    { 40413, "Hutchison Essar South Ltd., Andhra Pradesh" },
    { 40414, "Spice Communications PVT Ltd., Punjab" },
    { 40415, "Aircell Digilink India Ltd., UP (East)" },
    { 40416, "Bharti Airtel Ltd, North East" },
    { 40417, "Dishnet Wireless Ltd, West Bengal" },
    { 40418, "Reliance Telecom Ltd., H.P." },
    { 40419, "Idea Mobile Communications Ltd., Kerala" },
    { 40420, "Hutchison Essar Ltd, Mumbai" },
    { 40421, "BPL Mobile Communications Ltd., Mumbai" },
    { 40422, "Idea Cellular Ltd., Maharashtra" },
    { 40423, "Idea Cellular Ltd, Maharashtra" },
    { 40424, "Idea Cellular Ltd., Gujarat" },
    { 40425, "Dishnet Wireless Ltd, Bihar" },
    { 40427, "Hutchison Essar Cellular Ltd., Maharashtra" },
    { 40429, "Dishnet Wireless Ltd, Assam" },
    { 40430, "Hutchison Telecom East Ltd, Kolkata" },
    { 40431, "Bharti Airtel Ltd., Kolkata" },
    { 40433, "Dishnet Wireless Ltd, North East" },
    { 40434, "BSNL, Haryana" },
    { 40435, "Dishnet Wireless Ltd, Himachal Pradesh" },
    { 40436, "Reliance Telecom Ltd., Bihar" },
    { 40437, "Dishnet Wireless Ltd, J&K" },
    { 40438, "BSNL, Assam" },
    { 40440, "Bharti Airtel Ltd., Chennai" },
    { 40441, "Aircell Cellular Ltd, Chennai" },
    { 40442, "Aircel Ltd., Tamil Nadu" },
    { 40443, "Hutchison Essar Cellular Ltd., Tamil Nadu" },
    { 40444, "Spice Communications PVT Ltd., Karnataka" },
    { 40446, "Hutchison Essar Cellular Ltd., Kerala" },
    { 40448, "Dishnet Wireless Ltd, UP (West)" },
    { 40449, "Bharti Airtel Ltd., Andra Pradesh" },
    { 40450, "Reliance Telecom Ltd., North East" },
    { 40451, "BSNL, H.P." },
    { 40452, "Reliance Telecom Ltd., Orissa" },
    { 40453, "BSNL, Punjab" },
    { 40454, "BSNL, UP (West)" },
    { 40455, "BSNL, UP (East)" },
    { 40456, "Idea Mobile Communications Ltd., UP (West)" },
    { 40457, "BSNL, Gujarat" },
    { 40458, "BSNL, Madhya Pradesh" },
    { 40459, "BSNL, Rajasthan" },
    { 40460, "Aircell Digilink India Ltd., Rajasthan" },
    { 40461, "Dishnet Wireless Ltd, Punjab" },
    { 40462, "BSNL, J&K" },
    { 40463, "Dishnet Wireless Ltd, Haryana" },
    { 40464, "BSNL, Chennai" },
    { 40465, "Dishnet Wireless Ltd, UP (East)" },
    { 40466, "BSNL, Maharashtra" },
    { 40467, "Reliance Telecom Ltd., Madhya Pradesh" },
    { 40468, "MTNL, Delhi" },
    { 40469, "MTNL, Mumbai" },
    { 40470, "Bharti Hexacom Ltd, Rajasthan" },
    { 40471, "BSNL, Karnataka" },
    { 40472, "BSNL, Kerala" },
    { 40473, "BSNL, Andhra Pradesh" },
    { 40474, "BSNL, West Bengal" },
    { 40475, "BSNL, Bihar" },
    { 40476, "BSNL, Orissa" },
    { 40477, "BSNL, North East" },
    { 40478, "BTA Cellcom Ltd., Madhya Pradesh" },
    { 40479, "BSNL, Andaman & Nicobar" },
    { 40480, "BSNL, Tamil Nadu" },
    { 40481, "BSNL, Kolkata" },
    { 40482, "Idea Telecommunications Ltd, H.P." },
    { 40483, "Reliable Internet Services Ltd., Kolkata" },
    { 40484, "Hutchison Essar South Ltd., Chennai" },
    { 40485, "Reliance Telecom Ltd., W.B. & A.N." },
    { 40486, "Hutchison Essar South Ltd., Karnataka" },
    { 40487, "Idea Telecommunications Ltd, Rajasthan" },
    { 40488, "Hutchison Essar South Ltd, Punjab" },
    { 40489, "Idea Telecommunications Ltd, UP (East)" },
    { 40490, "Bharti Airtel Ltd., Maharashtra" },
    { 40491, "Dishnet Wireless Ltd, Kolkata" },
    { 40492, "Bharti Airtel Ltd., Mumbai" },
    { 40493, "Bharti Airtel Ltd., Madhya Pradesh" },
    { 40494, "Bharti Airtel Ltd., Tamil Nadu" },
    { 40495, "Bharti Airtel Ltd., Kerala" },
    { 40496, "Bharti Airtel Ltd., Haryana" },
    { 40497, "Bharti Airtel Ltd., UP (West)" },
    { 40498, "Bharti Airtel Ltd., Gujarat" },
    { 40499, "Dishnet Wireless Ltd, Kerala" },
    { 40508, "Reliance Infocomm Ltd, Himachal Pradesh" },
    { 40512, "Reliance Infocomm Ltd, Kolkata" },
    { 40515, "Reliance Infocomm Ltd, Mumbai" },
    { 40517, "Reliance Infocomm Ltd, Orissa" },
    { 40523, "Reliance Infocomm Ltd, West bengal" },
    { 40528, "Tata Teleservices Ltd, Chennai" },
    { 40552, "Bharti Airtel Ltd, Bihar" },
    { 40553, "Bharti Airtel Ltd, Orissa" },
    { 40554, "Bharti Airtel Ltd, UP (East)" },
    { 40555, "Bharti Airtel Ltd, J&K" },
    { 40556, "Bharti Airtel Ltd, Assam" },
    { 40566, "Hutchison Essar South Ltd, UP (West)" },
    { 40567, "Hutchison Essar South Ltd, Orissa" },
    { 40568, "Vodaphone/Hutchison, Madhya Pradesh" },
    { 40570, "Aditya Birla Telecom Ltd, Bihar" },
    { 40571, "Essar Spacetel Ltd, Himachal Pradesh" },
    { 40572, "Essar Spacetel Ltd, North East" },
    { 40573, "Essar Spacetel Ltd, Assam" },
    { 40574, "Essar Spacetel Ltd, J&K" },
    { 40576, "Essar Spacetel Ltd, Orissa" },
    { 40577, "Essar Spacetel Ltd, Maharashtra" },
    { 40581, "Aircell Ltd, Delhi" },
    { 40582, "Aircell Ltd, Andhra Pradesh" },
    { 40583, "Aircell Ltd, Gujarat" },
    { 40584, "Aircell Ltd, Maharashtra" },
    { 40585, "Aircell Ltd, Mumbai" },
    { 40586, "Aircell Ltd, Rajasthan" },
    { 41001, "Mobilink" },
    { 41003, "PAK Telecom Mobile Ltd. (UFONE)" },
    { 41004, "CMPak" },
    { 41006, "Telenor Pakistan" },
    { 41007, "Warid Telecom" },
    { 41201, "AWCC" },
    { 41220, "Roshan" },
    { 41240, "Areeba Afghanistan" },
    { 41250, "Etisalat" },
    { 41280, "Afghan Telecom" },
    { 41288, "Afghan Telecom" },
    { 41302, "MTN Network Ltd." },
    { 41303, "Hutchison Telecommunications Lanka (Pvt) Ltd. (Valid until 12 September 2021)" },
    { 41309, "Hutchison Telecommunications Lanka (Pvt) Ltd. (Valid until 12 September 2021)" },
    { 41400, "Myanmar Posts and Telecommunications" },
    { 41401, "Myanmar Posts and Telecommunications" },
    { 41402, "Myanmar Posts and Telecommunications" },
    { 41403, "Myanmar Economic Corporation" },
    { 41404, "Myanmar Posts and Telecommunications" },
    { 41405, "Ooredoo Myanmar Limited" },
    { 41406, "Telenor Myanmar Limited" },
    { 41409, "Myanmar National Tele & Communication Co.,Ltd" },
    { 41420, "Amara Communication Co.,Ltd" },
    { 41421, "Amara Communication Co.,Ltd" },
    { 41422, "Fortune Telecom Co., Ltd" },
    { 41423, "Global Technology Co., Ltd" },
    { 41505, "Ogero Telecom" },
    { 41532, "Cellis" },
    { 41533, "Cellis" },
    { 41534, "Cellis" },
    { 41535, "Cellis" },
    { 41536, "Libancell" },
    { 41537, "Libancell" },
    { 41538, "Libancell" },
    { 41539, "Libancell" },
    { 41601, "Fastlink" },
    { 41602, "Xpress" },
    { 41603, "Umniah" },
    { 41677, "MobileCom" },
    { 41701, "Syriatel" },
    { 41702, "MTN Syria" },
    { 41703, "WAFA Telecom" },
    { 41709, "Syrian Telecom" },
    { 41805, "Asia Cell" },
    { 41820, "Zain Iraq (previously Atheer)" },
    { 41830, "Zain Iraq (previously Iraqna)" },
    { 41840, "Korek Telecom" },
    { 41847, "Iraq Central Cooperative Association for Communication and Transportation" },
    { 41848, "ITC Fanoos" },
    { 41849, "Iraqtel" },
    { 41862, "Itisaluna" },
    { 41870, "Kalimat" },
    { 41880, "Iraqi Telecommunications & Post Company (ITPC)" },
    { 41881, "ITPC (Al-Mazaya)" },
    { 41883, "ITPC (Sader Al-Iraq)" },
    { 41884, "ITPC (Eaamar Albasrah)" },
    { 41885, "ITPC (Anwar Yagotat Alkhalee)" },
    { 41886, "ITPC (Furatfone)" },
    { 41887, "ITPC (Al-Seraj)" },
    { 41888, "ITPC (High Link)" },
    { 41889, "ITPC (Al-Shams)" },
    { 41891, "ITPC (Belad Babel)" },
    { 41892, "ITPC (Al Nakheel)" },
    { 41893, "ITPC (Iraqcell)" },
    { 41894, "ITPC (Shaly)" },
    { 41902, "ZAIN" },
    { 41903, "Wataniya Telecom" },
    { 41904, "Viva" },
    { 42001, "Saudi Telecom" },
    { 42003, "Etihad Etisalat Company (Mobily)" },
    { 42101, "Yemen Mobile Phone Company" },
    { 42102, "Spacetel Yemen" },
    { 42104, "Y-Telecom" },
    { 42202, "Oman Mobile Telecommunications Company (Oman Mobile)" },
    { 42203, "Oman Qatari Telecommunications Company (Nawras)" },
    { 42204, "Oman Telecommunications Company (Omantel)" },
    { 42206, "Vodafone Oman" },
    { 42402, "Etisalat" },
    { 42501, "Partner Communications Co. Ltd." },
    { 42502, "Cellcom Israel Ltd" },
    { 42503, "Pelephone Communications Ltd" },
    { 42504, "Globalsim Ltd" },
    { 42506, "Wataniya" },
    { 42507, "Mirs Ltd" },
    { 42508, "Golan Telecom Ltd" },
    { 42509, "Marathon 018 Xphone Ltd." },
    { 42511, "365 Telecom (MVNO)" },
    { 42512, "Free Telecom (MVNO)" },
    { 42513, "Ituran Cellular Communications" },
    { 42514, "Alon Cellular Ltd." },
    { 42515, "Home Cellular (MVNO)" },
    { 42516, "Rami Levi (MVNO)" },
    { 42517, "Gale Phone (MVNO)" },
    { 42518, "Cellact Communications Ltd (MVNO)" },
    { 42519, "Azi Communications Ltd" },
    { 42520, "Bezeq Ltd" },
    { 42521, "B.I.P Communications Ltd." },
    { 42522, "Maskyoo Telephonia Ltd." },
    { 42523, "Beezz Communication Solutions Ltd." },
    { 42524, "012 Telecom Ltd." },
    { 42525, "IMOD" },
    { 42526, "LB Annatel Ltd." },
    { 42527, "BITIT Ltd." },
    { 42528, "PHI Networks" },
    { 42529, "CG Networks" },
    { 42601, "Bahrain Telecommunications Company (BATELCO)" },
    { 42602, "Zain Bahrain" },
    { 42603, "Civil Aviation Authority" },
    { 42604, "STC Bahrain" },
    { 42605, "Royal Court" },
    { 42606, "STC Bahrain" },
    { 42607, "TAIF" },
    { 42701, "QATARNET" },
    { 42706, "Ooredoo Q.S.C./MOI LTE" },
    { 42899, "Mobicom" },
    { 42901, "Nepal Telecommunications" },
    { 43201, "KISH CELL PARS" },
    { 43202, "NEGIN ERTEBATAT AVA" },
    { 43203, "PARSIAN HAMRAH LOTUS" },
    { 43204, "TOSE E FANAVARI ERTEBATAT NOVIN HAMRAH" },
    { 43205, "HAMRAH HOSHMAND AYANDEH" },
    { 43206, "ERTEBATAT ARYANTEL" },
    { 43207, "HOOSHMAND AMIN MOBILE" },
    { 43208, "TOSE-E ERTEBATAT HAMRAH SHATEL" },
    { 43209, "HIWEB" },
    { 43211, "MCI (Mobile Communications of Iran)" },
    { 43212, "HIWEB" },
    { 43213, "HIWEB" },
    { 43214, "Kish Free Zone Organization" },
    { 43219, "Telecommunication Company of Iran (TCI) - Isfahan Celcom GSM" },
    { 43220, "Rightel" },
    { 43232, "TCI (Telecommunication Company of Iran)" },
    { 43235, "IRANCELL" },
    { 43244, "ERTEBATAT MOBIN NET" },
    { 43245, "FARABORD DADEHAYE IRANIAN" },
    { 43246, "HIWEB" },
    { 43249, "GOSTARESH ERTEBATAT MABNA" },
    { 43250, "SHATEL" },
    { 43251, "PISHGAMAN TOSE-E ERTEBATAT" },
    { 43252, "ASIATECH" },
    { 43270, "TCI (Telecommunication Company of Iran)" },
    { 43271, "ERTEBATAT KOOHE NOOR" },
    { 43293, "ERTEBATAT FARZANEGAN PARS" },
    { 43401, "Buztel" },
    { 43402, "Uzmacom" },
    { 43404, "Daewoo Unitel" },
    { 43405, "Coscom" },
    { 43407, "Uzdunrobita" },
    { 43601, "JC Somoncom" },
    { 43602, "CJSC Indigo Tajikistan" },
    { 43603, "TT mobile" },
    { 43604, "Josa Babilon-T" },
    { 43605, "CTJTHSC Tajik-tel" },
    { 43701, "Sky Mobile" },
    { 43703, "7 Mobile" },
    { 43705, "Alfa Telecom" },
    { 43706, "Kyrgyztelecom" },
    { 43709, "Nur Telecom" },
    { 43710, "Saima Telecom" },
    { 43711, "iTel" },
    { 43801, "Barash Communication Technologies (BCTI)" },
    { 43802, "TM-Cell" },
    { 44000, "SoftBank Corp." },
    { 44001, "KDDI Corporation" },
    { 44002, "Hanshin Cable Engineering Co. Ltd." },
    { 44003, "Internet Initiative Japan Inc." },
    { 44004, "Japan Radio Co., Ltd." },
    { 44005, "Wireless City Planning Inc." },
    { 44006, "SAKURA Internet Inc." },
    { 44007, "closip, Inc." },
    { 44008, "Panasonic Connect Co., Ltd" },
    { 44009, "Marubeni Network Solutions Inc." },
    { 44010, "NTT DOCOMO, INC." },
    { 44011, "Rakuten Mobile Network, Inc." },
    { 44012, "CABLE MEDIA WAIWAI CORPORATION" },
    { 44013, "NTT Communications Corporation" },
    { 44014, "GRAPE ONE LTD." },
    { 44015, "BB Backbone Corp." },
    { 44016, "Nokia Innovations Japan G.K." },
    { 44017, "OSAKA GAS BUSINESS CREATE CORPORATION" },
    { 44018, "Kintetsu Cable Network, Ltd" },
    { 44019, "NEC Networks & System Integration Corporation" },
    { 44020, "SoftBank Corp." },
    { 44021, "SoftBank Corp." },
    { 44022, "JTOWER Inc." },
    { 44023, "FUJITSU LIMITED" },
    { 44050, "KDDI Corporation" },
    { 44051, "KDDI Corporation" },
    { 44052, "KDDI Corporation" },
    { 44053, "KDDI Corporation" },
    { 44054, "KDDI Corporation" },
    { 44070, "KDDI Corporation" },
    { 44071, "KDDI Corporation" },
    { 44072, "KDDI Corporation" },
    { 44073, "KDDI Corporation" },
    { 44074, "KDDI Corporation" },
    { 44075, "KDDI Corporation" },
    { 44076, "KDDI Corporation" },
    { 44078, "Okinawa Cellular Telephone Company" },
    { 44100, "Wireless City Planning Inc." },
    { 44101, "SoftBank Corp." },
    { 44110, "UQ Communications Inc." },
    { 44191, "The Tokyo Organising Committee of the Olympic and Paralympic Games" },
    { 45001, "Globalstar Asia Pacific / Satellite network" },
    { 45002, "KT / 5G test bed" },
    { 45004, "KT / IoT network" },
    { 45005, "SK Telecom / 3G, 4G network" },
    { 45006, "LGU+ / 3G, 4G network" },
    { 45007, "KT Powertel / 3G network" },
    { 45008, "KT / 3G, 4G network" },
    { 45011, "SK Telecom / 3G, 4G network" },
    { 45012, "SK Telecom / IoT network" },
    { 45201, "Mobifone" },
    { 45202, "Vinaphone" },
    { 45204, "Viettel" },
    { 45205, "Vietnamobile" },
    { 45207, "Gmobile" },
    { 45208, "I-Telecom" },
    { 45209, "REDDI" },
    { 45400, "Hong Kong Telecommunications (HKT) Limited" },
    { 45401, "CITIC Telecom International Ltd" },
    { 45402, "Hong Kong Telecommunications (HKT) Limited" },
    { 45403, "Hutchison Telephone Company Limited" },
    { 45404, "Hutchison Telephone Company Limited" },
    { 45405, "Hutchison Telephone Company Limited" },
    { 45406, "SmarTone Mobile Communications Limited" },
    { 45407, "China Unicom (Hong Kong) Operations Limited" },
    { 45408, "Truphone (Hong Kong) Ltd" },
    { 45409, "CMMobile Global Communications Ltd" },
    { 45410, "Hong Kong Telecommunications (HKT) Limited" },
    { 45411, "China-Hongkong Telecom Limited" },
    { 45412, "China Mobile Hong Kong Co Ltd" },
    { 45413, "China Mobile Hong Kong Co Ltd" },
    { 45414, "Hutchison Telephone Company Limited" },
    { 45415, "SmarTone Mobile Communications Limited" },
    { 45416, "Hong Kong Telecommunications (HKT) Limited" },
    { 45417, "SmarTone Mobile Communications Limited" },
    { 45418, "Hong Kong Telecommunications (HKT) Limited" },
    { 45419, "Hong Kong Telecommunications (HKT) Limited" },
    { 45420, "Hong Kong Telecommunications (HKT) Limited" },
    { 45421, "21Vianet Mobile Limited" },
    { 45422, "263 Mobile Communications (HongKong) Limited" },
    { 45423, "Lycamobile Hong Kong Ltd" },
    { 45424, "Multibyte Info Technology Limited" },
    { 45425, "Government use" },
    { 45426, "Government use" },
    { 45427, "Government use" },
    { 45428, "Government use" },
    { 45429, "Hong Kong Telecommunications (HKT) Limited" },
    { 45430, "China Data Enterprises Ltd" },
    { 45431, "China Telecom Global Limited" },
    { 45432, "Hong Kong Broadband Network Ltd" },
    { 45435, "Webbing Hong Kong Limited" },
    { 45436, "Easco Telecommunications Limited" },
    { 45500, "SmarTone – Comunicações Móveis, S.A." },
    { 45501, "Companhia de Telecomunicações de Macau, S.A.R.L." },
    { 45502, "China Telecom (Macau) Limitada" },
    { 45503, "Hutchison – Telefone (Macau), Limitada" },
    { 45504, "Companhia de Telecomunicações de Macau, S.A.R.L." },
    { 45505, "Hutchison – Telefone (Macau), Limitada" },
    { 45506, "SmarTone – Comunicações Móveis, S.A." },
    { 45507, "China Telecom (Macau) Limitada" },
    { 45601, "Mobitel (Cam GSM)" },
    { 45602, "Hello" },
    { 45603, "S Telecom (CDMA)" },
    { 45604, "Cadcomms" },
    { 45605, "Starcell" },
    { 45606, "Smart" },
    { 45608, "Viettel" },
    { 45618, "Mfone" },
    { 45701, "Lao Telecommunication Public Company" },
    { 45702, "ETL Company Limited" },
    { 45703, "Star Telecom Co., Ltd" },
    { 45707, "Best Telecom Co., Ltd" },
    { 45708, "TPLUS Digital Sole Company Limited" },
    { 46000, "China Mobile" },
    { 46001, "China Unicom" },
    { 46002, "China Mobile" },
    { 46003, "China Telecom" },
    { 46004, "China Satellite Global Star Network" },
    { 46005, "China Telecom" },
    { 46006, "China Unicom" },
    { 46007, "China Mobile" },
    { 47001, "GramenPhone" },
    { 47002, "Aktel" },
    { 47003, "Mobile 2000" },
    { 47201, "DhiMobile" },
    { 50210, "DIGI Telecommunications" },
    { 50212, "Malaysian Mobile Services Sdn Bhd" },
    { 50213, "Celcom (Malaysia) Berhad" },
    { 50214, "Telekom Malaysia Berhad" },
    { 50216, "DIGI Telecommunications" },
    { 50217, "Malaysian Mobile Services Sdn Bhd" },
    { 50218, "U Mobile Sdn. Bhd." },
    { 50219, "Celcom (Malaysia) Berhad" },
    { 50220, "Electcoms Wireless Sdn Bhd" },
    { 50501, "Telstra Corporation Ltd." },
    { 50502, "Optus Mobile Pty. Ltd." },
    { 50503, "Vodafone Network Pty. Ltd." },
    { 50504, "Department of Defence" },
    { 50505, "The Ozitel Network Pty. Ltd." },
    { 50506, "Hutchison 3G Australia Pty. Ltd." },
    { 50507, "Vodafone Network Pty. Ltd." },
    { 50508, "One.Tel GSM 1800 Pty. Ltd." },
    { 50509, "Airnet Commercial Australia Ltd." },
    { 50510, "Norfolk Telecom" },
    { 50511, "Telstra Corporation Ltd." },
    { 50512, "Vodafone Hutchison Australia Pty Ltd" },
    { 50513, "Railcorp" },
    { 50514, "AAPT Ltd" },
    { 50515, "3GIS Pty Ltd. (Telstra & Hutchison 3G)" },
    { 50516, "Victorian Rail Track" },
    { 50517, "Vivid Wireless Pty Ltd" },
    { 50518, "Pactel International Pty Ltd" },
    { 50519, "Lycamobile Pty Ltd" },
    { 50520, "Ausgrid Corporation" },
    { 50521, "Queensland Rail Limited" },
    { 50522, "iiNet Ltd" },
    { 50523, "Challenge Networks Pty Ltd" },
    { 50524, "Advanced Communications Technologies Pty. Ltd." },
    { 50525, "Pilbara Iron Company Services Pty Ltd" },
    { 50526, "Dialogue Communications Pty Ltd" },
    { 50527, "Nexium Telecommunications" },
    { 50528, "RCOM International Pty Ltd" },
    { 50530, "Compatel Limited" },
    { 50531, "BHP Billiton" },
    { 50532, "Thales Australia" },
    { 50533, "CLX Networks Pty Ltd" },
    { 50534, "Santos Ltd" },
    { 50535, "MessageBird Pty Ltd" },
    { 50536, "Optus Mobile Pty. Ltd." },
    { 50537, "Yancoal Australia Ltd" },
    { 50538, "Truphone Pty Ltd" },
    { 50539, "Telstra Corporation Ltd." },
    { 50540, "CITIC PACIFIC MINING" },
    { 50541, "Aqura Technologies Pty" },
    { 50542, "Groote Eylandt Mining Company Pty Ltd" },
    { 50543, "Arrow Energy Pty Ltd" },
    { 50544, "ROY HILL IRON ORE PTY LTD" },
    { 50545, "CLERMONT COAL OPERATIONS PTY Limited" },
    { 50546, "ANGLOGOLD ASHANTI AUSTRALIA LTD" },
    { 50547, "Woodside Energy Limited" },
    { 50548, "Titan ICT Pty Ltd" },
    { 50549, "Field Solutions Group Pty Ltd" },
    { 50550, "Pivotel Group Pty Limited" },
    { 50551, "Fortescue Metals Group Ltd" },
    { 50552, "Optitel Pty Ltd" },
    { 50553, "Shell Australia Pty Ltd" },
    { 50555, "New South Wales Government Telecommunications Authority" },
    { 50556, "Nokia Solutions and Networks Pty Ltd" },
    { 50557, "CiFi Pty Ltd" },
    { 50561, "Commtel Network Solutions Pty Ltd" },
    { 50562, "NBNCo Limited" },
    { 50568, "NBNCo Limited" },
    { 50571, "Telstra Corporation Ltd." },
    { 50572, "Telstra Corporation Ltd." },
    { 50588, "Pivotel Group Pty Limited" },
    { 50590, "UE Access Pty Ltd" },
    { 50599, "One.Tel GSM 1800 Pty. Ltd." },
    { 51000, "PSN" },
    { 51001, "Satelindo" },
    { 51008, "Natrindo (Lippo Telecom)" },
    { 51010, "Telkomsel" },
    { 51011, "Excelcomindo" },
    { 51021, "Indosat - M3" },
    { 51028, "Komselindo" },
    { 51401, "Telin Timor-Leste" },
    { 51402, "Timor Telecom" },
    { 51403, "Viettel Timor-Leste" },
    { 51501, "Islacom" },
    { 51502, "Globe Telecom" },
    { 51503, "Smart Communications" },
    { 51505, "Digitel" },
    { 52000, "CAT CDMA" },
    { 52001, "AIS GSM" },
    { 52002, "CAT CDMA" },
    { 52003, "Advanced Wireless Network Company Limited" },
    { 52004, "Real Future Company Limited" },
    { 52005, "DTAC Network Company Limite" },
    { 52015, "TOT Public Company Limited" },
    { 52018, "Total Access Communications Public Company Limited" },
    { 52020, "ACes Regional Services Company Limited" },
    { 52023, "Digital Phone Company Limited" },
    { 52047, "TOT Public Company Limited" },
    { 52099, "True Move Company Limited" },
    { 52501, "Singtel ST GSM900" },
    { 52502, "Singtel ST GSM1800" },
    { 52503, "M1" },
    { 52505, "StarHub" },
    { 52508, "StarHub" },
    { 52509, "Liberty Wireless Pte Ltd" },
    { 52510, "TPG Telecom Pte Ltd" },
    { 52512, "Digital Trunked Radio Network" },
    { 52801, "Telekom Brunei Berhad (TelBru)" },
    { 52802, "Progresif Cellular Sdn Bhd (PCSB)" },
    { 52803, "Unified National Networks Sdn Bhd (UNN)" },
    { 52811, "DST Com" },
    { 53000, "Reserved for AMPS MIN based IMSI's" },
    { 53001, "One New Zealand Group Limited" },
    { 53002, "Teleom New Zealand CDMA Network" },
    { 53003, "Woosh Wireless - CDMA Network" },
    { 53004, "One New Zealand Group Limited" },
    { 53005, "Telecom New Zealand - UMTS Ntework" },
    { 53006, "FX Networks Ltd" },
    { 53007, "Dense Air New Zealand Ltd" },
    { 53011, "Interim Māori Spectrum Commission" },
    { 53024, "NZ Communications - UMTS Network" },
    { 53701, "Bmobile" },
    { 53702, "Telikom PNG Ltd" },
    { 53703, "Digicel Ltd" },
    { 53704, "Digitec Communication Limited" },
    { 53901, "Tonga Communications Corporation" },
    { 53943, "Digicel" },
    { 53988, "Digicel (Tonga) Ltd" },
    { 54002, "Bemobile (BMobile (SI) Ltd)" },
    { 54101, "SMILE" },
    { 54105, "Digicel Vanuatu" },
    { 54107, "WANTOK" },
    { 54201, "Vodafone (Fiji) Ltd" },
    { 54202, "Digicel (Fiji) Ltd" },
    { 54203, "Telecom Fiji Ltd (CDMA)" },
    { 54301, "Manuia" },
    { 54501, "ATHKL" },
    { 54502, "OceanLink" },
    { 54601, "OPT Mobilis" },
    { 54705, "VITI" },
    { 54710, "Mara Telecom" },
    { 54715, "Pacific Mobile Telecom" },
    { 54720, "Tikiphone" },
    { 54801, "Telecom Cook" },
    { 54901, "Telecom Samoa Cellular Ltd." },
    { 54927, "GoMobile SamoaTel Ltd" },
    { 55001, "FSM Telecom" },
    { 55201, "Palau National Communications Corp. (a.k.a. PNCC)" },
    { 55202, "PECI / PalauTel" },
    { 55299, "Palau Mobile Communications Inc. (PMCI)" },
    { 55301, "Tuvalu Telecommunications Corporation" },
    { 55401, "Teletok/LTE 4G" },
    { 55501, "Telecom Niue" },
    { 60201, "Mobinil" },
    { 60202, "Vodafone" },
    { 60203, "Etisalat" },
    { 60301, "Algérie Télécom Mobile « ATM » / GSM/public" },
    { 60302, "Optimum Télécom Algérie « OTA » / GSM/public" },
    { 60303, "Wataniya Télécom Algérie « WTA » / GSM/public" },
    { 60307, "Algérie Télécom « AT » / WLL / public" },
    { 60309, "Algérie Télécom « AT » / LTE Fixe / public" },
    { 60400, "Médi Télécom" },
    { 60401, "Itissalat Al-Maghrib" },
    { 60402, "Wana Corporate" },
    { 60404, "Al Houria Telecom" },
    { 60405, "Wana Corporate" },
    { 60406, "Itissalat Al-Maghrib" },
    { 60499, "Al Houria Telecom" },
    { 60502, "Tunisie Telecom" },
    { 60503, "Orascom Telecom" },
    { 60701, "Gamcel" },
    { 60702, "Africell" },
    { 60703, "COMIUM" },
    { 60704, "Qcell" },
    { 60705, "GAMTEL-Ecowan" },
    { 60706, "NETPAGE" },
    { 60801, "Sonatel (Orange)" },
    { 60802, "Sentel GSM (Tigo)" },
    { 60803, "Expresso Sénégal" },
    { 60804, "CSU" },
    { 60901, "Mattel S.A." },
    { 60902, "Chinguitel S.A." },
    { 60910, "Mauritel Mobiles" },
    { 61001, "Malitel" },
    { 61002, "Orange Mali Sa" },
    { 61003, "ATEL-SA" },
    { 61101, "Orange Guinée" },
    { 61102, "Sotelgui" },
    { 61105, "Cellcom Guinée SA" },
    { 61202, "Atlantique Cellulaire" },
    { 61203, "Orange Côte d'Ivoire" },
    { 61204, "Comium Côte d'Ivoire" },
    { 61205, "Loteny Telecom" },
    { 61206, "Oricel Côte d'Ivoire" },
    { 61207, "Aircomm Côte d'Ivoire" },
    { 61302, "Celtel" },
    { 61303, "Telecel" },
    { 61401, "Sahel.Com" },
    { 61402, "Celtel" },
    { 61403, "Telecel" },
    { 61501, "Togo Telecom" },
    { 61601, "Libercom" },
    { 61602, "Telecel" },
    { 61603, "Spacetel Benin" },
    { 61701, "Cellplus" },
    { 61702, "Mahanagar Telephone (Mauritius) Ltd" },
    { 61703, "Mahanagar Telephone (Mauritius) Ltd" },
    { 61710, "Emtel" },
    { 61804, "Comium Liberia" },
    { 61901, "Celtel" },
    { 61902, "Millicom" },
    { 61903, "Africell" },
    { 61904, "Comium (Sierra Leone) Ltd" },
    { 61905, "Lintel (Sierra Leone) Ltd." },
    { 61907, "QCELL SIERRA LEONE" },
    { 61909, "INTERGROUP TELECOM" },
    { 61925, "Mobitel" },
    { 61940, "Datatel (SL) Ltd GSM" },
    { 61950, "Datatel (SL) Ltd CDMA" },
    { 62001, "Spacefon" },
    { 62002, "Ghana Telecom Mobile" },
    { 62003, "Mobitel" },
    { 62004, "Kasapa Telecom Ltd." },
    { 62011, "Netafriques Dot Com Ltd" },
    { 62120, "Econet Wireless Nigeria Ltd." },
    { 62130, "MTN Nigeria Communications" },
    { 62140, "MTEL" },
    { 62150, "Globacom" },
    { 62160, "EMTS" },
    { 62201, "Celtel" },
    { 62202, "Tchad Mobile" },
    { 62301, "Centrafrique Telecom Plus (CTP)" },
    { 62302, "Telecel Centrafrique (TC)" },
    { 62303, "Celca (Socatel)" },
    { 62401, "Mobile Telephone Networks Cameroon" },
    { 62402, "Orange Cameroun" },
    { 62404, "NEXTTEL (ex VIETTEL CAMEROON)" },
    { 62501, "Cabo Verde Telecom" },
    { 62502, "T+Telecomunicações" },
    { 62601, "Companhia Santomese de Telecomunicações" },
    { 62701, "Guinea Ecuatorial de Telecomunicaciones Sociedad Anónima (GETESA)" },
    { 62801, "LIBERTIS" },
    { 62802, "MOOV" },
    { 62803, "CELTEL" },
    { 62804, "USAN GABON" },
    { 62805, "Réseau de l’Administration Gabonaise (RAG)" },
    { 62901, "Celtel" },
    { 62910, "Libertis Telecom" },
    { 63001, "Vodacom Congo RDC sprl" },
    { 63002, "AIRTEL sprl" },
    { 63005, "Supercell Sprl" },
    { 63086, "Congo-Chine Telecom s.a.r.l." },
    { 63088, "YOZMA TIMETURNS sprl" },
    { 63089, "OASIS sprl" },
    { 63090, "Africell RDC" },
    { 63102, "Unitel" },
    { 63104, "Movicel" },
    { 63105, "AFRICELL" },
    { 63201, "Guinétel S.A." },
    { 63202, "Spacetel Guinea-Bissau S.A." },
    { 63301, "Cable and wireless (Seychelles) Ltd" },
    { 63305, "Intelvision Ltd" },
    { 63310, "Airtel (Seychelles) Ltd" },
    { 63401, "SD Mobitel" },
    { 63402, "Areeba-Sudan" },
    { 63403, "MTN Sudan" },
    { 63405, "Network of the World Ltd (NOW)" },
    { 63406, "Zain Sudan" },
    { 63407, "Sudanese Telecommunication Co. LTD (SUDATEL)" },
    { 63499, "MTN Sudan" },
    { 63510, "MTN Rwandacell" },
    { 63513, "AIRTEL RWANDA Ltd" },
    { 63517, "Olleh Rwanda Networks (ORN)" },
    { 63601, "ETH MTN" },
    { 63801, "Evatis" },
    { 63901, "Safaricom PLC" },
    { 63902, "Safaricom PLC" },
    { 63903, "Airtel Networks Kenya Limited" },
    { 63904, "Mobile Pay Kenya Limited" },
    { 63905, "Airtel Networks Kenya Limited" },
    { 63906, "Finserve Africa Limited" },
    { 63907, "Telkom Kenya Limited" },
    { 63909, "Homeland Media Group Limited" },
    { 63910, "Jamii Telecommunications Limited" },
    { 63911, "Jambo Telecoms Limited" },
    { 63912, "Infura Limited" },
    { 64002, "MIC Tanzania Limited (Tigo)" },
    { 64003, "Zanzibar Telecom Limited (Zantel)" },
    { 64004, "Vodacom Tanzania Limited" },
    { 64005, "Airtel Tanzania Limited" },
    { 64006, "WIA Company Limited" },
    { 64007, "Tanzania Telecommunications Company Limited" },
    { 64009, "Viettel Tanzania Limited (Halotel)" },
    { 64011, "Smile Communications Tanzania Ltd" },
    { 64013, "Wiafrica Tanzania Limited (Cootel)" },
    { 64014, "MO Mobile Holding Limited" },
    { 64101, "Airtel Uganda Limited" },
    { 64104, "Tangerine Uganda Limited" },
    { 64108, "Talkio Mobile Limited" },
    { 64110, "MTN Uganda Limited" },
    { 64111, "Uganda Telecom Limited" },
    { 64116, "SimbaNET Uganda Limited" },
    { 64118, "Suretelecom Uganda Limited" },
    { 64120, "Hamilton Telecom Limited" },
    { 64122, "Airtel Uganda Limited" },
    { 64133, "Smile Communications Uganda Limited" },
    { 64140, "Civil Aviation Authority (CAA)" },
    { 64144, "K2 Telecom Limited" },
    { 64201, "Econet" },
    { 64202, "Africell" },
    { 64203, "ONAMOB" },
    { 64207, "LACELL" },
    { 64282, "U.COM" },
    { 64301, "T.D.M. GSM" },
    { 64303, "Movitel" },
    { 64304, "VM Sarl" },
    { 64501, "Celtel Zambia Ltd." },
    { 64502, "Telecel Zambia Ltd." },
    { 64503, "Zamtel" },
    { 64507, "Liquid Telecom Zambia Limited" },
    { 64601, "Celtel Madagascar (Zain), GSM" },
    { 64602, "Orange Madagascar, GSM" },
    { 64604, "Telecom Malagasy Mobile, GSM" },
    { 64700, "Orange" },
    { 64701, "BJT Partners" },
    { 64702, "Telco OI" },
    { 64703, "Telco OI" },
    { 64704, "Zeop Mobile" },
    { 64710, "Société Réunionnaise du Radiotéléphone" },
    { 64801, "Net One" },
    { 64803, "Telecel" },
    { 64804, "Econet" },
    { 64901, "Mobile Telecommunications Ltd." },
    { 64902, "Telecom Namibia" },
    { 64903, "Powercom Pty Ltd (leo)" },
    { 64904, "Paratus Telecommunications (Pty)" },
    { 64905, "Demshi Investments CC" },
    { 64906, "MTN Namibia" },
    { 64907, "Capricorn Connect" },
    { 65001, "Telekom Network Ltd." },
    { 65010, "Celtel ltd." },
    { 65101, "Vodacom Lesotho (pty) Ltd." },
    { 65102, "Econet Ezin-cel" },
    { 65110, "VODACOM LESOTHO" },
    { 65201, "Mascom Wireless (Pty) Ltd" },
    { 65202, "Orange Botswana (Pty) Ltd" },
    { 65204, "Botswana Telecommunications Corporation (BTC)" },
    { 65301, "SPTC" },
    { 65302, "Swazi Mobile Limited" },
    { 65310, "Swazi MTN" },
    { 65401, "HURI / Comores Telecom" },
    { 65402, "TELMA / TELCO SA" },
    { 65501, "Vodacom (Pty) Ltd." },
    { 65502, "Telkom SA Ltd" },
    { 65503, "Telkom SA SOC Ltd" },
    { 65505, "Telkom SA Ltd" },
    { 65506, "Sentech (Pty) Ltd." },
    { 65507, "Cell C (Pty) Ltd." },
    { 65510, "Mobile Telephone Networks (MTN) Pty Ltd" },
    { 65512, "Mobile Telephone Networks (MTN) Pty Ltd" },
    { 65513, "Neotel Pty Ltd" },
    { 65514, "Neotel Pty Ltd" },
    { 65519, "Wireless Business Solutions (iBurst)" },
    { 65524, "SMS Portal (Pty) Ltd" },
    { 65525, "Wirels Connect" },
    { 65527, "A to Z Vaal Industrial Supplies Pty Ltd" },
    { 65528, "Hymax Talking Solutions (Pty) Ltd" },
    { 65530, "Bokamoso Consortium Pty Ltd" },
    { 65531, "Karabo Telecoms (Pty) Ltd." },
    { 65532, "Ilizwi Telecommunications Pty Ltd" },
    { 65533, "Thinta Thinta Telecommunications Pty Ltd" },
    { 65534, "Bokone Telecoms Pty Ltd" },
    { 65535, "Kingdom Communications Pty Ltd" },
    { 65536, "Amatole Telecommunication Pty Ltd" },
    { 65538, "Wireless Business Solutions (Pty) Ltd" },
    { 65546, "SMS Cellular Services (Pty) Ltd" },
    { 65550, "Ericsson South Africa (Pty) Ltd" },
    { 65551, "Integrat (Pty) Ltd" },
    { 65553, "Lycamobile (Pty) Ltd" },
    { 65565, "Vodacom Pty Ltd" },
    { 65573, "Wireless Business Solutions (Pty) Ltd" },
    { 65574, "Wireless Business Solutions (Pty) Ltd" },
    { 65576, "Comsol Networks (Pty) Ltd" },
    { 65577, "K2015315513 (Pty) Ltd t\\a One Telecom (Pty) Ltd" },
    { 65801, "Sure South Atlantic Ltd. (Ascension)" },
    { 65912, "Sudani/Sudatel" },
    { 65991, "Zain-South Sudan" },
    { 65992, "MTN-South Sudan" },
    { 65995, "Vivacel/NOW" },
    { 65997, "Gemtel" },
    { 70267, "Belize Telecommunications Ltd., GSM 1900" },
    { 70269, "SMART/Speednet Communications Ltd." },
    { 70401, "Servicios de Comunicaciones Personales Inalámbricas, S.A. (SERCOM, S.A" },
    { 70402, "Comunicaciones Celulares S.A." },
    { 70403, "Telefónica Centroamérica Guatemala S.A." },
    { 70601, "CTE Telecom Personal, S.A. de C.V." },
    { 70602, "Digicel, S.A. de C.V." },
    { 70603, "Telemóvil El Salvador, S.A." },
    { 71021, "Empresa Nicaragüense de Telecomunicaciones, S.A. (ENITEL)" },
    { 71073, "Servicios de Comunicaciones, S.A. (SERCOM)" },
    { 71201, "Instituto Costarricense de Electricidad - ICE" },
    { 71202, "Instituto Costarricense de Electricidad - ICE" },
    { 71203, "CLARO CR Telecomunicaciones S.A." },
    { 71204, "Telefónica de Costa Rica TC, S.A." },
    { 71220, "Virtualis" },
    { 71401, "Cable & Wireless Panama S.A." },
    { 71402, "Grupo de Comunicaciones Digitales, S.A. (TIGO)" },
    { 71403, "Claro Panamá, S.A." },
    { 71404, "Digicel (Panamá), S.A." },
    { 71405, "Cable & Wireless Panamá, S.A." },
    { 71610, "TIM Peru" },
    { 72400, "NEXTEL" },
    { 72401, "SISTEER DO BRASIL TELECOMUNICAÇÔES (MVNO)" },
    { 72402, "TIM REGIÂO I" },
    { 72403, "TIM REGIÂO III" },
    { 72404, "TIM REGIÂO III" },
    { 72405, "CLARO" },
    { 72406, "Telefônica Brasil S.A." },
    { 72410, "Telefônica Brasil S.A." },
    { 72411, "Telefônica Brasil S.A." },
    { 72415, "SERCOMTEL" },
    { 72416, "BRT CELULAR" },
    { 72417, "SURF TELECOM S.A." },
    { 72418, "DATORA (MVNO)" },
    { 72421, "LIGUE" },
    { 72423, "Telefônica Brasil S.A." },
    { 72424, "AMAZONIA CELULAR" },
    { 72430, "TNL PCS Oi" },
    { 72431, "TNL PCS Oi" },
    { 72432, "CTBC CELULAR R III" },
    { 72433, "CTBC CELULAR R II" },
    { 72434, "CTBC CELULAR R I" },
    { 72435, "TELCOM" },
    { 72436, "OPTIONS" },
    { 72437, "UNICEL" },
    { 72438, "CLARO" },
    { 72439, "NEXTEL (SMP)" },
    { 72454, "TIM S.A." },
    { 72499, "LOCAL (STFC)" },
    { 73001, "Entel Telefónica Móvil" },
    { 73002, "Telefónica Móvil" },
    { 73003, "Smartcom" },
    { 73004, "Centennial Cayman Corp. Chile S.A." },
    { 73005, "Multikom S.A." },
    { 73006, "Blue Two Chile SA" },
    { 73007, "Telefónica Móviles Chile S.A." },
    { 73008, "VTR Móvil S.A." },
    { 73009, "Centennial Cayman Corp. Chile S.A." },
    { 73010, "Entel" },
    { 73011, "Celupago S.A." },
    { 73012, "Telestar Móvil S.A." },
    { 73013, "TRIBE Mobile Chile SPA" },
    { 73014, "Netline Telefónica Móvil Ltda" },
    { 73015, "CIBELES TELECOM S.A." },
    { 73016, "Nomade Telecomunicaciones S.A." },
    { 73017, "COMPATEL Chile Limitada" },
    { 73018, "Empresas Bunker S.A." },
    { 73019, "Sociedad Falabella Móvil SPA" },
    { 73020, "Inversiones Santa Fe Limitada" },
    { 73021, "WILL S.A." },
    { 73022, "CELLPLUS SPA" },
    { 73023, "CLARO SERVICIOS EMPRESARIALES S.A." },
    { 73026, "WILL S.A." },
    { 73027, "Cibeles Telecom S.A." },
    { 73402, "Corporación Digitel" },
    { 73403, "GALAXY ENTERTAINMENT DE VENEZUELA C.A." },
    { 73404, "Telcel, C.A." },
    { 73406, "Telecomunicaciones Movilnet, C.A." },
    { 73408, "PATRIACELL C.A." },
    { 73601, "Nuevatel S.A." },
    { 73602, "ENTEL S.A." },
    { 73603, "Telecel S.A." },
    { 73800, "E-Networks Inc." },
    { 73801, "U-Mobile (Cellular) Inc." },
    { 73805, "eGovernment Unit, Ministry of the Presidency" },
    { 74000, "Otecel S.A. - Bellsouth" },
    { 74001, "Porta GSM" },
    { 74002, "Telecsa S.A." },
    { 74204, "Free Caraïbe" },
    { 74401, "Hóla Paraguay S.A." },
    { 74402, "Hutchison Telecom S.A." },
    { 74403, "Compañia Privada de Comunicaciones S.A." },
    { 74602, "Telesur" },
    { 74603, "Digicel" },
    { 74605, "Telesur (CDMA)" },
    { 74801, "Administración Nacional de Telecomunicaciones (ANTEL)" },
    { 74807, "Telefónica Móviles del Uruguay S.A. (Movistar)" },
    { 74810, "AM Wireless Uruguay S.A. (Claro)" },
    { 74815, "ENALUR S.A." },
    { 90101, "Webbing" },
    { 90102, "GlobalmatiX AG" },
    { 90103, "Iridium Communications Inc" },
    { 90104, "BBIX Singapore Pte. Ltd." },
    { 90105, "Thuraya RMSS Network" },
    { 90106, "Thuraya Satellite Telecommunications Company" },
    { 90111, "Inmarsat Ltd." },
    { 90112, "Maritime Communications Partner AS (MCP network)" },
    { 90113, "BebbiCell AG (Formerly Global Networks Switzerland AG)" },
    { 90114, "AeroMobile AS" },
    { 90115, "OnAir N.V. (Formerly SITA on behalf of Onair)" },
    { 90116, "Cisco Systems, Inc." },
    { 90117, "Jersey Telecom" },
    { 90118, "Cingular Wireless" },
    { 90119, "Epic Communications Limited (Monaco Telecom)" },
    { 90120, "Intermatica" },
    { 90121, "Wins Limited (Formerly Seanet Maritime Communications AB)" },
    { 90122, "MediaLincc Ltd" },
    { 90124, "Voxbone SA" },
    { 90126, "Telecom Italia" },
    { 90127, "Monaco Telecom" },
    { 90128, "Vodafone Group" },
    { 90129, "Telenor Connexion AB" },
    { 90131, "Orange" },
    { 90132, "MegaFon" },
    { 90133, "Smart Communications , Inc" },
    { 90134, "Tyntec Limited" },
    { 90135, "Globecomm Network Services" },
    { 90136, "Azerfon LLC" },
    { 90137, "TRANSATEL" },
    { 90138, "Multiregional TransitTelecom (MTT)" },
    { 90139, "MTX Connect Ltd" },
    { 90140, "Deutsche Telekom AG" },
    { 90141, "One Network B.V." },
    { 90142, "DCN Hub ehf" },
    { 90143, "EMnify GmbH" },
    { 90144, "AT&T, Inc." },
    { 90145, "Advanced Wireless Network Company Limited" },
    { 90146, "Telecom26 AG" },
    { 90147, "Ooredoo" },
    { 90148, "Com4 Sweden AB" },
    { 90149, "Mobile Telecommunications Company K.S.C.P. (Formerly Zain Kuwait)" },
    { 90150, "Sawatch Limited/EchoStar Mobile Limited" },
    { 90151, "VisionNG" },
    { 90152, "Manx Telecom Trading Ltd." },
    { 90153, "Inmarsat Ltd. (replacing Deutsche Telekom AG)" },
    { 90154, "Teleena Holding B.V." },
    { 90155, "Beezz Communication Solutions Ltd." },
    { 90156, "European Telecommunications Standards Institute (ETSI)" },
    { 90157, "SAP" },
    { 90158, "BICS SA" },
    { 90159, "MessageBird B.V." },
    { 90160, "OneWeb" },
    { 90161, "MTN Management Services" },
    { 90162, "Twilio Inc." },
    { 90163, "GloTell B.V." },
    { 90164, "Syniverse Technologies, LLC" },
    { 90165, "Plintron Global Technology Solutions Private Limited" },
    { 90166, "Limitless Mobile, LLC" },
    { 90167, "1NCE GmbH" },
    { 90168, "Maersk Line A/S" },
    { 90169, "Legos" },
    { 90170, "Clementvale Baltic OÜ" },
    { 90171, "Tampnet AS" },
    { 90172, "Tele2 Sverige Aktiebolag (Formerly Tele2 IoT)" },
    { 90173, "Cubic Telecom Limited" },
    { 90174, "Etisalat" },
    { 90175, "Podsystem Ltd." },
    { 90176, "A1 Telekom Austria AG" },
    { 90177, "Bouygues Telecom" },
    { 90178, "Telecom Italia Sparkle S.p.A." },
    { 90179, "Nokia Corporation" },
    { 90180, "Flo Live Limited" },
    { 90181, "Airnity SAS" },
    { 90182, "Eseye Limited" },
    { 90183, "iBasis Netherlands BV" },
    { 90184, "TELEFONICA MOVILES ESPAÑA, S.A. UNIPERSONAL (TME)" },
    { 90185, "Telefónica Germany GmbH & Co. OHG" },
    { 90186, "BJT PARTNERS SAS" },
    { 90187, "Cisco Systems, Inc." },
    { 90188, "UN Office for the Coordination of Humanitarian Affairs (OCHA)" },
    { 90189, "DIDWW Ireland Limited" },
    { 90190, "Truphone Limited" },
    { 90191, "World Mobile Group Limited" },
    { 90192, "Phonegroup SA" },
    { 90193, "SkyFive AG" },
    { 90194, "Intelsat US LLC" },
    { 90195, "HMD Global Oy" },
    { 90196, "KORE Wireless" },
    { 90197, "Satelio IoT Services S.L." },
    { 90198, "Skylo Technologies, Inc." },
    { 90199, "Athalos Global Services BV" },
    { 90201, "MulteFire Alliance" },
    { 99101, "World's Global Telecom (temporary assignment for trial until 14.I.2021)" },
    { 99999, "Internal use, example, testing" },
    { 0, NULL }
};

value_string_ext mcc_mnc_2digits_codes_ext = VALUE_STRING_EXT_INIT(mcc_mnc_2digits_codes);

static const value_string mcc_mnc_3digits_codes[] = {
    {  208500, "EDF" },
    {  208700, "Weaccess group" },
    {  208701, "GIP Vendée numérique" },
    {  208702, "17-Numérique" },
    {  208703, "Nivertel" },
    {  208704, "Axione Limousin" },
    {  208705, "Hautes-Pyrénées Numérique" },
    {  208706, "Tours Métropole Numérique" },
    {  208707, "Sartel THD" },
    {  208708, "Melis@ territoires ruraux" },
    {  208709, "Quimper communauté télécom" },
    {  208710, "Losange" },
    {  208711, "Nomotech" },
    {  208712, "Syndicat Audois d'énergies et du Numérique" },
    {  208713, "SD NUM SAS" },
    {  208714, "Département de l'Isère" },
    {  214700, "IBERDROLA ESPAÑA, S.A.UNIPERSONAL" },
    {  214701, "ENDESA DISTRIBUCIÓN ELÉCTRICA, S.L." },
    {  302100, "Data on Tap Inc." },
    {  302130, "Xplornet Communications" },
    {  302131, "Xplornet Communications" },
    {  302140, "Fibernetics Corporation" },
    {  302150, "Cogeco Connexion Inc." },
    {  302151, "Cogeco Connexion Inc." },
    {  302152, "Cogeco Connexion Inc." },
    {  302220, "Telus Mobility" },
    {  302221, "Telus Mobility" },
    {  302222, "Telus Mobility" },
    {  302230, "ISP Telecom" },
    {  302250, "ALO Mobile Inc" },
    {  302270, "Bragg Communications" },
    {  302290, "Airtel Wireless" },
    {  302300, "ECOTEL inc." },
    {  302320, "Dave Wireless" },
    {  302330, "Blue Canada Wireless Inc" },
    {  302340, "Execulink" },
    {  302360, "Telus Mobility" },
    {  302370, "Microcell" },
    {  302380, "Dryden Mobility" },
    {  302390, "Dryden Mobility" },
    {  302420, "A.B.C. Allen Business Communications Ltd." },
    {  302490, "Globalive Wireless" },
    {  302491, "Freedom Mobile Inc." },
    {  302500, "Videotron Ltd" },
    {  302510, "Videotron Ltd" },
    {  302530, "Keewatinook Okimacinac" },
    {  302550, "Star Solutions International Inc." },
    {  302560, "Lynx Mobility" },
    {  302570, "Ligado Networks Corp." },
    {  302590, "Quadro Communication" },
    {  302600, "Iristel Inc." },
    {  302610, "Bell Mobility" },
    {  302620, "Ice Wireless" },
    {  302630, "Aliant Mobility" },
    {  302640, "Bell Mobility" },
    {  302650, "Tbaytel" },
    {  302660, "MTS Mobility" },
    {  302670, "CityTel Mobility" },
    {  302680, "Sask Tel Mobility" },
    {  302681, "SaskTel Mobility" },
    {  302690, "Bell Mobility" },
    {  302710, "Globalstar" },
    {  302720, "Rogers Wireless" },
    {  302721, "Rogers Communications Canada Inc. (Wireless)" },
    {  302730, "TerreStar Solutions" },
    {  302740, "Rogers Communications Canada Inc." },
    {  302741, "Rogers Communications Canada Inc." },
    {  302760, "Public Mobile Inc" },
    {  302770, "Rural Com" },
    {  302780, "Sask Tel Mobility" },
    {  302781, "SaskTel Mobility" },
    {  302848, "Vocom International Telecommunications, Inc" },
    {  302860, "Telus Mobility" },
    {  302880, "Telus/Bell shared" },
    {  302940, "Wightman Telecom" },
    {  302990, "Test" },
    {  302996, "Powertech Labs (experimental)" },
    {  302998, "Institu de Recherche d’Hydro-Québec (experimental)" },
    {  310010, "Verizon Wireless" },
    {  310012, "Verizon Wireless" },
    {  310013, "Verizon Wireless" },
    {  310014, "TEST IMSI HNI" },
    {  310016, "AT&T Mobility" },
    {  310020, "Union Telephone Company" },
    {  310030, "AT&T Mobility" },
    {  310035, "ETEX Communications, LP (d/b/a) ETEX Wireless" },
    {  310040, "Mobi" },
    {  310050, "Alaska Wireless Networks" },
    {  310070, "AT&T Mobility" },
    {  310080, "AT&T Mobility" },
    {  310090, "AT&T Mobility" },
    {  310100, "New Mexico RSA 4 East Limited Partnership" },
    {  310110, "PTI Pacifica, Inc." },
    {  310120, "T-Mobile USA" },
    {  310130, "Carolina West Wireless" },
    {  310140, "GTA Wireless LLC" },
    {  310150, "AT&T Mobility" },
    {  310160, "T-Mobile USA" },
    {  310170, "AT&T Mobility" },
    {  310180, "West Central Wireless" },
    {  310190, "Alaska Wireless Networks" },
    {  310200, "T-Mobile USA" },
    {  310210, "T-Mobile USA" },
    {  310220, "T-Mobile USA" },
    {  310230, "T-Mobile USA" },
    {  310240, "T-Mobile USA" },
    {  310250, "T-Mobile USA" },
    {  310260, "T-Mobile USA" },
    {  310270, "T-Mobile USA" },
    {  310280, "AT&T Mobility" },
    {  310290, "NEP Cellcorp Inc." },
    {  310310, "T-Mobile USA" },
    {  310320, "Smith Bagley Inc, dba Cellular One" },
    {  310330, "Wireless Partners LLC" },
    {  310340, "Limitless Mobile, LLC" },
    {  310350, "Verizon Wireless" },
    {  310360, "Cellular Network Partnership dba Pioneer Cellular" },
    {  310370, "Docomo Pacific Inc" },
    {  310380, "AT&T Mobility" },
    {  310390, "TX-11 Acquistion LLC" },
    {  310400, "Wave Runner LLC" },
    {  310410, "AT&T Mobility" },
    {  310420, "World Mobile Networks, Inc" },
    {  310430, "GCI Communications Corp" },
    {  310440, "Numerex Corp" },
    {  310450, "North East Cellular Inc." },
    {  310460, "Eseye" },
    {  310470, "Docomo Pacific Inc" },
    {  310480, "PTI Pacifica, Inc." },
    {  310490, "T-Mobile USA" },
    {  310500, "Public Service Cellular, Inc." },
    {  310510, "Nsight" },
    {  310520, "Transactions Network Services (TNS)" },
    {  310530, "T-Mobile" },
    {  310540, "Oklahoma Western Telephone Company" },
    {  310550, "Syniverse Technologies" },
    {  310570, "Broadpoint, LLC (former PetroCom, LLC) c/o MTPCS, LLC dba CellularOne" },
    {  310580, "Inland Cellular Telephone Company" },
    {  310590, "Verizon Wireless" },
    {  310591, "Verizon Wireless" },
    {  310592, "Verizon Wireless" },
    {  310593, "Verizon Wireless" },
    {  310594, "Verizon Wireless" },
    {  310595, "Verizon Wireless" },
    {  310596, "Verizon Wireless" },
    {  310597, "Verizon Wireless" },
    {  310598, "Verizon Wireless" },
    {  310599, "Verizon Wireless" },
    {  310600, "NewCell dba Cellcom" },
    {  310620, "Nsighttel Wireless, LLC" },
    {  310630, "Choice Wireless" },
    {  310640, "Numerex Corp" },
    {  310650, "JASPER TECHNOLOGIES INC." },
    {  310660, "T-Mobile USA" },
    {  310670, "AT&T Mobility" },
    {  310680, "AT&T Mobility" },
    {  310690, "Limitless Mobile, LLC" },
    {  310700, "Cross Valiant Cellular Partnership" },
    {  310710, "Arctic Slope Telephone Association Cooperative" },
    {  310720, "Syniverse Technologies" },
    {  310730, "U.S. Cellular" },
    {  310740, "Viaero Wireless" },
    {  310750, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  310760, "Lynch 3G Communications Corporation" },
    {  310770, "T-Mobile" },
    {  310790, "PinPoint Communications Inc." },
    {  310800, "T-Mobile USA" },
    {  310810, "Pacific Lightwave Inc." },
    {  310820, "Verizon Wireless" },
    {  310830, "T-Mobile USA" },
    {  310840, "Telecom North America Mobile Inc" },
    {  310850, "Aeris Communications, Inc." },
    {  310860, "West Central Wireless" },
    {  310880, "Advantage Cellular Systems, Inc." },
    {  310890, "Verizon Wireless" },
    {  310891, "Verizon Wireless" },
    {  310892, "Verizon Wireless" },
    {  310893, "Verizon Wireless" },
    {  310894, "Verizon Wireless" },
    {  310895, "Verizon Wireless" },
    {  310896, "Verizon Wireless" },
    {  310897, "Verizon Wireless" },
    {  310898, "Verizon Wireless" },
    {  310899, "Verizon Wireless" },
    {  310900, "Mid-Rivers Telephone Cooperative" },
    {  310910, "Verizon Wireless" },
    {  310920, "James Valley Wireless LLC" },
    {  310930, "Copper Valley Wireless" },
    {  310940, "Tyntec Limited" },
    {  310950, "AT&T Mobility" },
    {  310960, "UBET Wireless" },
    {  310970, "Globalstar USA" },
    {  310990, "Evolve Cellular Inc." },
    {  311000, "West Central Wireless" },
    {  311010, "Chariton Valley Communications Corporation" },
    {  311020, "Missouri RSA No. 5 Partnership" },
    {  311030, "Indigo Wireless, Inc." },
    {  311040, "Commnet Wireless LLC" },
    {  311050, "Thumb Cellular Limited Partnership" },
    {  311060, "Space Data Corporation" },
    {  311070, "AT&T Mobility" },
    {  311080, "Pine Telephone Company dba Pine Cellular" },
    {  311090, "AT&T Mobility" },
    {  311100, "Nex-Tech Wireless LLC" },
    {  311110, "Verizon Wireless" },
    {  311120, "PTI Pacifica, Inc." },
    {  311140, "Cross Telephone Company" },
    {  311150, "Wilkes Cellular Inc." },
    {  311170, "Tampnet (formerly Broadpoint, LLC (former PetroCom, LLC) c/o MTPCS, LL" },
    {  311180, "AT&T Mobility" },
    {  311190, "AT&T Mobility" },
    {  311210, "Telnyx LLC" },
    {  311220, "United States Cellular" },
    {  311225, "U.S. Cellular" },
    {  311228, "U.S. Cellular" },
    {  311229, "U.S. Cellular" },
    {  311230, "Cellular South Inc." },
    {  311240, "Cordova Wireless Communications Inc" },
    {  311250, "Wave Runner LLC" },
    {  311260, "Sprint" },
    {  311270, "Verizon Wireless" },
    {  311271, "Verizon Wireless" },
    {  311272, "Verizon Wireless" },
    {  311273, "Verizon Wireless" },
    {  311274, "Verizon Wireless" },
    {  311275, "Verizon Wireless" },
    {  311276, "Verizon Wireless" },
    {  311277, "Verizon Wireless" },
    {  311278, "Verizon Wireless" },
    {  311279, "Verizon Wireless" },
    {  311280, "Verizon Wireless" },
    {  311281, "Verizon Wireless" },
    {  311282, "Verizon Wireless" },
    {  311283, "Verizon Wireless" },
    {  311284, "Verizon Wireless" },
    {  311285, "Verizon Wireless" },
    {  311286, "Verizon Wireless" },
    {  311287, "Verizon Wireless" },
    {  311288, "Verizon Wireless" },
    {  311289, "Verizon Wireless" },
    {  311290, "Pinpoint Wireless Inc." },
    {  311310, "Leaco Rural Telephone Company Inc" },
    {  311320, "Commnet Wireless LLC" },
    {  311330, "Bug Tussel Wireless LLC" },
    {  311340, "Illinois Valley Cellular" },
    {  311350, "Sagebrush Cellular Inc dba Nemont" },
    {  311360, "Stelera Wireless LLC" },
    {  311370, "GCI Communications Corp" },
    {  311380, "New Dimension Wireless Ltd" },
    {  311390, "Verizon Wireless" },
    {  311400, "TEST IMSI HNI" },
    {  311410, "Iowa RSA No. 2 Limited Partnership" },
    {  311420, "Northwest Cell" },
    {  311430, "RSA 1 Limited Partnership dba Cellular 29 Plus" },
    {  311440, "Verizon Wireless" },
    {  311450, "Panhandle Telecommunication Systems Inc." },
    {  311460, "ELECTRIC IMP INCORPORATED" },
    {  311470, "Vitelcom Cellular D/B/A Innovative Wireless" },
    {  311480, "Verizon Wireless" },
    {  311481, "Verizon Wireless" },
    {  311482, "Verizon Wireless" },
    {  311483, "Verizon Wireless" },
    {  311484, "Verizon Wireless" },
    {  311485, "Verizon Wireless" },
    {  311486, "Verizon Wireless" },
    {  311487, "Verizon Wireless" },
    {  311488, "Verizon Wireless" },
    {  311489, "Verizon Wireless" },
    {  311490, "T-Mobile USA" },
    {  311500, "Mobi" },
    {  311510, "Ligado Networks" },
    {  311530, "WorldCell Solutions LLC" },
    {  311540, "Coeur Rochester, Inc" },
    {  311550, "Commnet Midwest, LLC" },
    {  311560, "OTZ Communications Inc" },
    {  311570, "Medicom" },
    {  311580, "U.S. Cellular" },
    {  311588, "U.S. Cellular" },
    {  311589, "U.S. Cellular" },
    {  311590, "Verizon Wireless" },
    {  311600, "Limitless Mobile, LLC" },
    {  311610, "North Dakota Network Co." },
    {  311620, "TerreStar Networks Inc." },
    {  311630, "Cellular South Inc." },
    {  311640, "Standing Rock Telecommunications" },
    {  311650, "United Wireless Inc" },
    {  311660, "Metro PCS Wireless Inc" },
    {  311670, "Pine Belt Cellular Inc dba Pine Belt Wireless" },
    {  311680, "GreenFly LLC" },
    {  311690, "TeleBeeper of New Mexico Inc" },
    {  311700, "Midwest Network Solutions Hub, LLC. (MNS)" },
    {  311710, "Northeast Wireless Networks LLC" },
    {  311720, "Maine PCS LLC" },
    {  311730, "Proximiti Mobility Inc" },
    {  311740, "Telalaska Cellular" },
    {  311750, "Flat Wireless, LLC" },
    {  311760, "Reclaimed 06/21/2016" },
    {  311770, "Altiostar Networks, Inc." },
    {  311780, "The American Samoa Telecommunications Authority" },
    {  311790, "Coleman County Telephone Cooperative, Inc." },
    {  311800, "Verizon Wireless" },
    {  311810, "Verizon Wireless" },
    {  311820, "Ribbon Communications" },
    {  311830, "Thumb Cellular LLC" },
    {  311840, "Nsight" },
    {  311850, "Nsight" },
    {  311860, "Uintah Basin Electronic Telecommunications" },
    {  311870, "T-Mobile USA" },
    {  311880, "T-Mobile USA" },
    {  311882, "T-Mobile" },
    {  311890, "Globecomm Network Services Corporation" },
    {  311900, "Gigsky Inc." },
    {  311910, "SI Wireless LLC" },
    {  311920, "Missouri RSA No 5 Partnership" },
    {  311930, "Cox Communications" },
    {  311940, "T-Mobile USA" },
    {  311950, "Sunman Telecommunications corp." },
    {  311970, "Big River Broadband LLC" },
    {  311980, "LigTel Communications" },
    {  311990, "VTel Wireless" },
    {  312010, "Chariton Valley Communications Corporation" },
    {  312020, "Infrastructure Networks LLC" },
    {  312030, "Cross Wireless" },
    {  312040, "Custer Telephone Cooperative Inc" },
    {  312060, "CoverageCo" },
    {  312080, "South Georgia Regional Information Technology Authority" },
    {  312090, "AT&T Mobility" },
    {  312100, "ClearSky Technologies Inc" },
    {  312120, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312130, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312150, "Northwest Cell" },
    {  312160, "RSA1 Limited Partnership dba Chat Mobility" },
    {  312170, "Iowa RSA No. 2 Limited Partnership" },
    {  312180, "Limitless Mobile, LLC" },
    {  312190, "T-Mobile USA" },
    {  312210, "ASPENTA, LLC" },
    {  312220, "Chariton Valley Communications Corporation" },
    {  312230, "SRT Communications Inc." },
    {  312240, "Sprint" },
    {  312250, "T-Mobile USA" },
    {  312260, "WorldCell Solutions LLC" },
    {  312270, "Cellular Network Partnership dba Pioneer Cellular" },
    {  312280, "Cellular Network Partnership dba Pioneer Cellular" },
    {  312290, "Uintah Basin Electronic Telecommunications" },
    {  312300, "Telecom North America Mobile Inc." },
    {  312310, "Clear Stream Communications, LLC" },
    {  312320, "S and R Communications LLC" },
    {  312330, "Nemont Communications, Inc." },
    {  312350, "Triangle Communication System Inc." },
    {  312360, "Wes-Tex Telecommunications, LTD" },
    {  312370, "Commnet Wireless" },
    {  312380, "Copper Valley Wireless" },
    {  312390, "FTC Communications LLC" },
    {  312400, "Mid-Rivers Telephone Cooperative" },
    {  312410, "Eltopia Communications, LLC" },
    {  312420, "Nex-Tech Wireless, LLC" },
    {  312430, "Silver Star Communications" },
    {  312450, "Cable & Communications Corporation" },
    {  312460, "KPU Telecommunications Division" },
    {  312470, "Carolina West Wireless, Inc." },
    {  312480, "Sagebrush Cellular, Inc." },
    {  312490, "TrustComm, Inc" },
    {  312510, "Wue" },
    {  312530, "T-Mobile USA" },
    {  312550, "Great Plains Communications, Inc" },
    {  312570, "Buffalo-Lake Erie Wireless Systems Co., LLC" },
    {  312580, "Google LLC" },
    {  312590, "Northern Michigan University" },
    {  312600, "Sagebrush Cellular, Inc." },
    {  312620, "GlobeTouch Inc." },
    {  312630, "NetGenuity, Inc." },
    {  312650, "Brightlink" },
    {  312670, "AT&T Mobility" },
    {  312680, "AT&T Mobility" },
    {  312690, "TGS, LLC" },
    {  312700, "Wireless Partners,LLC" },
    {  312710, "Great North Woods Wireless LLC" },
    {  312720, "Southern Communications Services, Inc. D/B/A SouthernLINC Wireless" },
    {  312730, "Triangle Communication System Inc." },
    {  312750, "Artemis" },
    {  312760, "ARCTIC SLOPE TELEPHONE ASSOCIATION COOPERATIVE" },
    {  312770, "Verizon Wireless" },
    {  312780, "RedZone Wireless LLC" },
    {  312790, "Gila Electronics" },
    {  312800, "Cirrus Core Networks" },
    {  312810, "Bristol Bay Telephone Cooperative" },
    {  312820, "Santel Communications Cooperative, Inc." },
    {  312830, "Kings County Office of Education" },
    {  312840, "South Georgia Regional Information Technology" },
    {  312850, "Onvoy Spectrum, LLC" },
    {  312860, "Flat Wireless, LLC" },
    {  312870, "GigSky Mobile, LLC" },
    {  312880, "Albemarle County Public Schools" },
    {  312890, "Circle Gx" },
    {  312900, "Flat West Wireless, LLC" },
    {  312910, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312920, "Northeast Wireless Networks" },
    {  312930, "Hewlett-Packard Communication Services, LLC" },
    {  312940, "Webformix" },
    {  312950, "Custer Telephone Cooperative, Inc" },
    {  312960, "M&A Technology, Inc." },
    {  312970, "IOSAZ Intellectual Property LLC" },
    {  312980, "Mark Twain Communications Company" },
    {  312990, "PREMIER HOLDINGS, LLC" },
    {  313000, "Tennessee Wireless" },
    {  313010, "Cross Wireless LLC dba Bravado Wireless" },
    {  313020, "CTC Telecom, INC. dba CTC Wireless" },
    {  313030, "AT&T Mobility" },
    {  313040, "Nucla-Naturita Telephone Company" },
    {  313050, "Manti Tele Communications Company, Inc. dba Breakaway Wireless" },
    {  313060, "Country Wireless" },
    {  313061, "Country Wireless" },
    {  313070, "Midwest Network Solutions Hub LLC" },
    {  313080, "Speedwavz LLP" },
    {  313090, "Vivint Wireless, Inc" },
    {  313100, "AT&T FirstNet" },
    {  313110, "AT&T FirstNet" },
    {  313120, "AT&T FirstNet" },
    {  313130, "AT&T FirstNet" },
    {  313140, "AT&T FirstNet" },
    {  313150, "RESERVED FOR PUBLIC SAFETY" },
    {  313160, "RESERVED FOR PUBLIC SAFETY" },
    {  313170, "RESERVED FOR PUBLIC SAFETY" },
    {  313180, "RESERVED FOR PUBLIC SAFETY" },
    {  313190, "RESERVED FOR PUBLIC SAFETY" },
    {  313200, "Mercury Network Corporation" },
    {  313210, "AT&T Mobility" },
    {  313220, "Custer Telephone Cooperative,Inc." },
    {  313230, "Velocity Communications Inc" },
    {  313240, "Fundamental Holdings, Corp" },
    {  313250, "Imperial County Office of Education" },
    {  313260, "Expeto Wireless Inc." },
    {  313280, "King Street Wireless, LP" },
    {  313290, "Gulf Coast Broadband LLC" },
    {  313300, "Southern California Edison" },
    {  313310, "CAL.NET, INC" },
    {  313320, "Paladin Wireless" },
    {  313330, "CenturyTel Broadband Services LLC" },
    {  313340, "Dish Network" },
    {  313350, "Dish Network" },
    {  313360, "Dish Network" },
    {  313370, "Red Truck Wireless, LLC" },
    {  313380, "OptimERA Inc." },
    {  313390, "Altice USA Wireless, Inc." },
    {  313400, "Texoma Communications, LLC" },
    {  313410, "Anterix Inc." },
    {  313420, "Hudson Valley Wireless" },
    {  313440, "Arvig Enterprises INC" },
    {  313450, "Spectrum Wireless Holdings, LLC" },
    {  313460, "Mobi" },
    {  313470, "San Diego Gas & Electric Company" },
    {  313480, "Ready Wireless, LLC" },
    {  313490, "Puloli, Inc" },
    {  313500, "Shelcomm, Inc" },
    {  313510, "Puerto Rico Telecom Company" },
    {  313520, "Florida Broadband, Inc" },
    {  313540, "Nokia Innovations US LLC" },
    {  313550, "Mile High Networks LLC" },
    {  313560, "Boldyn Networks Transit US LLC" },
    {  313570, "Celular Network Partnership" },
    {  313580, "Telecall Telecommuncations Corp." },
    {  313590, "Southern Communications Services, Inc. D/B/A Southern Linc" },
    {  313600, "ST Engineering iDirect" },
    {  313610, "Point Broadband Fiber Holding, LLC" },
    {  313620, "Omniprophis Corporation" },
    {  313630, "LICT Corporation" },
    {  313640, "Geoverse" },
    {  313650, "Chevron USA INC" },
    {  313660, "Hudson Valley Wireless" },
    {  313670, "Hudson Valley Wireless" },
    {  313680, "Hudson Valley Wireless" },
    {  313690, "Shenandoah Cable Television, LLC" },
    {  313700, "Ameren Services Company" },
    {  313710, "Extent Systems" },
    {  313720, "1st Point Communications, LLC" },
    {  313730, "TruAccess Networks" },
    {  313740, "RTO Wireless" },
    {  313750, "CellTex Networks, LLC" },
    {  313760, "Hologram" },
    {  313770, "Tango Networks" },
    {  313780, "Windstream Services LLC" },
    {  313790, "Liberty Cablevision of Puerto Rico LLC" },
    {  313810, "W.A.T.C.H. TV Co. dba Watch Communications" },
    {  313820, "Inland Cellular Telephone Company" },
    {  313830, "360 communications INC" },
    {  313840, "Celblox Acquisitions" },
    {  313850, "Softcom Internet Communications, Inc." },
    {  313860, "AMG Technology Investment Group dba Nextlink Internet" },
    {  313870, "Elektrafi LLC" },
    {  313880, "Shuttle Wireless Solutions Inc." },
    {  313890, "Tulare County Office of Education" },
    {  313900, "All Tribal Networks" },
    {  313910, "San Diego Gas and Electric" },
    {  313920, "JCI US INC" },
    {  313930, "Standing Rock Telecom" },
    {  313940, "Motorola Solutions, Inc" },
    {  313950, "Cheyenne and Arapaho Development Group" },
    {  313960, "Townes 5G, LLC" },
    {  313970, "Tychron Corporation" },
    {  313990, "Ericsson US" },
    {  314010, "Boingo Wireless Inc" },
    {  314020, "Spectrum Wireless Holdings, LLC" },
    {  314030, "Baicells Technologies North America Inc." },
    {  314060, "Texas A&M University System – RELLIS Campus" },
    {  314070, "Texas A&M University System – RELLIS Campus" },
    {  314080, "Texas A&M University System – RELLIS Campus" },
    {  314090, "Southern Communications Services, Inc. D/B/A Southern Linc" },
    {  314100, "RESERVED FOR PUBLIC SAFETY" },
    {  314110, "RESERVED FOR PUBLIC SAFETY" },
    {  314120, "RESERVED FOR PUBLIC SAFETY" },
    {  314130, "RESERVED FOR PUBLIC SAFETY" },
    {  314140, "RESERVED FOR PUBLIC SAFETY" },
    {  314150, "RESERVED FOR PUBLIC SAFETY" },
    {  314160, "RESERVED FOR PUBLIC SAFETY" },
    {  314170, "RESERVED FOR PUBLIC SAFETY" },
    {  314180, "RESERVED FOR PUBLIC SAFETY" },
    {  314190, "RESERVED FOR PUBLIC SAFETY" },
    {  314200, "XF Wireless Investments" },
    {  314210, "Telecom Resource Center" },
    {  314220, "Secrus Technologies" },
    {  314230, "Trace-Tek" },
    {  314240, "XF Wireless Investments" },
    {  314260, "AT&T Mobility" },
    {  314270, "AT&T Mobility" },
    {  314280, "Pollen Mobile LLC" },
    {  314290, "Wave" },
    {  314310, "Terranet" },
    {  314320, "Agri-Valley Communications, Inc" },
    {  314330, "Nova Labs Inc." },
    {  314340, "E-MARCONI LLC" },
    {  314350, "Evergy" },
    {  314360, "Oceus Networks, LLC" },
    {  314370, "Texas A&M University – ITEC" },
    {  314380, "Circle Computer Resources, Inc." },
    {  314390, "AT&T" },
    {  314400, "Cellular South Inc. dba C Spire" },
    {  314410, "Peeringhub Inc" },
    {  314420, "Cox Communications, Inc" },
    {  314430, "Highway9 Networks, Inc." },
    {  314440, "Tecore Global Services, LLC" },
    {  314450, "NUWAVE Communications, Inc." },
    {  314460, "Texas A&M University" },
    {  314470, "Manhattan Telecommunications Corporation LLC" },
    {  314480, "Xcel Energy Services Inc." },
    {  314490, "Utah Education and Telehealth Network (UETN)" },
    {  314500, "Aetheros Inc" },
    {  314510, "SI Wireless LLC" },
    {  314520, "Oklahoma Gas & Electric Company (OG&E)" },
    {  314530, "Agile Networks" },
    {  314540, "RGTN USA, Inc." },
    {  316010, "Sprint" },
    {  316011, "Southern Communications Services Inc." },
    {  316700, "Mile High Networks LLC" },
    {  334001, "COMUNICACIONES DIGITALES DEL NORTE, S.A. DE C.V" },
    {  334010, "AT&T COMUNICACIONES DIGITALES, S. DE R.L. DE C.V." },
    {  334020, "RADIOMOVIL DIPSA, S.A. DE C.V." },
    {  334030, "PEGASO PCS, S.A. DE C.V." },
    {  334040, "AT&T NORTE, S. DE R.L. DE C.V. Y AT&T DESARROLLO EN COMUNICACIONES DE MÉXICO, S. DE R.L. DE C.V." },
    {  334050, "GRUPO AT&T CELULLAR, S. DE R.L. DE C.V." },
    {  334060, "SERVICIOS DE ACCESO INALAMBRICO, S.A DE C.V." },
    {  334066, "TELÉFONOS DE MÉXICO, S.A.B. DE C.V" },
    {  334070, "AT&T COMERCIALIZACIÓN MÓVIL, S. DE R.L. DE C.V." },
    {  334080, "AT&T COMERCIALIZACIÓN MÓVIL, S. DE R.L. DE C.V." },
    {  334090, "AT&T COMUNICACIONES DIGITALES, S. DE R.L. DE C.V." },
    {  334100, "TELECOMUNICACIONES DE MÉXICO" },
    {  334110, "MAXCOM TELECOMUNICACIONES, S.A.B. DE C.V." },
    {  334120, "QUICKLY PHONE, S.A. DE C.V." },
    {  334130, "AXTEL, S.A.B. DE C.V." },
    {  334140, "ALTÁN REDES, S.A.P.I. DE C.V." },
    {  334150, "ULTRAVISIÓN, S.A. DE C.V." },
    {  334160, "CABLEVISIÓN RED, S.A. DE C.V." },
    {  334170, "OXIO MOBILE, S.A. DE C.V." },
    {  334180, "FREEDOMPOP MÉXICO, S.A. DE C.V." },
    {  334190, "VIASAT TECNOLOGÍA, S.A. DE C.V." },
    {  338040, "Symbiote Investment Limited" },
    {  338050, "Digicel (Jamaica) Ltd" },
    {  338080, "Rock Mobile Limited" },
    {  338110, "Cable and Wireless Jamaica Ltd" },
    {  342600, "Cable & Wireless (Barbados) Ltd. trading as Lime" },
    {  342646, "KW Telecommunications Inc." },
    {  342800, "Ozone" },
    {  342820, "Neptune Communications Inc." },
    {  344030, "APUA PCS" },
    {  344920, "Cable & Wireless (Antigua) trading as Lime" },
    {  344930, "AT&T Wireless (Antigua)" },
    {  346001, "WestTel Ltd., trading as Logic" },
    {  346140, "Cable & Wireless (Cayman) trading as Lime" },
    {  348170, "Cable & Wireless (BVI) Ltd trading as lime" },
    {  348370, "BVI Cable TV Ltd" },
    {  348570, "Caribbean Cellular Telephone Ltd." },
    {  348770, "Digicel (BVI) Ltd" },
    {  350000, "Bermuda Digital Communications Ltd (CellOne)" },
    {  350007, "Paradise Mobile" },
    {  352110, "Cable & Wireless Grenada ltd trading as lime" },
    {  354860, "Cable & Wireless (West Indies) Ltd trading as Lime" },
    {  356110, "Cable & Wireless St Kitts & Nevis Ltd trading as Lime" },
    {  358110, "Cable & Wireless (St Lucia) Ltd trading as Lime" },
    {  360110, "Cable & Wireless St Vincent and the Grenadines Ltd trading as lime" },
    {  365010, "Weblinks Limited" },
    {  365840, "Cable and Wireless (Anguilla) Ltd trading as Lime" },
    {  366110, "Cable & Wireless Dominica Ltd trading as Lime" },
    {  374130, "Digicel Trinidad and Tobago Ltd." },
    {  374140, "LaqTel Ltd." },
    {  376350, "Cable & Wireless (TCI) Ltd trading asLime" },
    {  376351, "Digicel TCI Ltd" },
    {  405000, "Shyam Telelink Ltd, Rajasthan" },
    {  405005, "Reliance Communications Ltd/GSM, Delhi" },
    {  405006, "Reliance Communications Ltd/GSM, Gujarat" },
    {  405007, "Reliance Communications Ltd/GSM, Haryana" },
    {  405009, "Reliance Communications Ltd/GSM, J&K" },
    {  405010, "Reliance Communications Ltd,/GSM Karnataka" },
    {  405011, "Reliance Communications Ltd/GSM, Kerala" },
    {  405012, "Reliance Infocomm Ltd, Andhra Pradesh" },
    {  405013, "Reliance Communications Ltd/GSM, Maharashtra" },
    {  405014, "Reliance Communications Ltd/GSM, Madhya Pradesh" },
    {  405018, "Reliance Communications Ltd/GSM, Punjab" },
    {  405020, "Reliance Communications Ltd/GSM, Tamilnadu" },
    {  405021, "Reliance Communications Ltd/GSM, UP (East)" },
    {  405022, "Reliance Communications Ltd/GSM, UP (West)" },
    {  405025, "Tata Teleservices Ltd/GSM, Andhra Pradesh" },
    {  405027, "Tata Teleservices Ltd,/GSM Bihar" },
    {  405029, "Tata Teleservices Ltd/GSM, Delhi" },
    {  405030, "Tata Teleservices Ltd/GSM, Gujarat" },
    {  405031, "Tata Teleservices Ltd/GSM, Haryana" },
    {  405032, "Tata Teleservices Ltd/GSM, Himachal Pradesh" },
    {  405033, "Reliance Infocomm Ltd, Bihar" },
    {  405034, "Tata Teleservices Ltd/GSM, Kamataka" },
    {  405035, "Tata Teleservices Ltd/GSM, Kerala" },
    {  405036, "Tata Teleservices Ltd/GSM, Kolkata" },
    {  405037, "Tata Teleservices Ltd/GSM, Maharashtra" },
    {  405038, "Tata Teleservices Ltd/GSM, Madhya Pradesh" },
    {  405039, "Tata Teleservices Ltd/GSM, Mumbai" },
    {  405040, "Reliance Infocomm Ltd, Chennai" },
    {  405041, "Tata Teleservices Ltd/GSM, Orissa" },
    {  405042, "Tata Teleservices Ltd/GSM, Punjab" },
    {  405043, "Tata Teleservices Ltd/GSM, Rajasthan" },
    {  405044, "Tata Teleservices Ltd/GSM, Tamilnadu" },
    {  405045, "Tata Teleservices Ltd/GSM, UP (East)" },
    {  405046, "Tata Teleservices Ltd/GSM, UP (West)" },
    {  405047, "Tata Teleservices Ltd/GSM, West Bengal" },
    {  405750, "Vodafone Essar Spacetel Ltd, J&K" },
    {  405751, "Vodafone Essar Spacetel Ltd, Assam" },
    {  405752, "Vodafone Essar Spacetel Ltd, Bihar" },
    {  405753, "Vodafone Essar Spacetel Ltd, Orissa" },
    {  405754, "Vodafone Essar Spacetel Ltd, Himachal Pradesh" },
    {  405755, "Vodafone Essar Spacetel Ltd, North East" },
    {  405799, "Idea Cellular Ltd, MUMBAI" },
    {  405800, "Aircell Ltd, Delhi" },
    {  405801, "Aircell Ltd, Andhra Pradesh" },
    {  405802, "Aircell Ltd, Gujarat" },
    {  405803, "Aircell Ltd, Kamataka" },
    {  405804, "Aircell Ltd, Maharashtra" },
    {  405805, "Aircell Ltd, Mumbai" },
    {  405806, "Aircell Ltd, Rajasthan" },
    {  405807, "Dishnet Wireless Ltd, Haryana" },
    {  405808, "Dishnet Wireless Ltd, Madhya Pradesh" },
    {  405809, "Dishnet Wireless Ltd, Kerala" },
    {  441200, "SORACOM, Inc." },
    {  441201, "Aurens Co.,Ltd." },
    {  441202, "Sony Wireless Communications Inc." },
    {  441203, "GujoCity" },
    {  441204, "Wicom Inc." },
    {  441205, "KATCH NETWORK INC." },
    {  441206, "MITSUBISHI ELECTRIC CORPORATION" },
    {  441207, "Mitsui Knowledge Industry Co., Ltd." },
    {  441208, "CHUDENKO CORPORATION" },
    {  441209, "Cable Television TOYAMA Inc." },
    {  441210, "NIPPON TELEGRAPH AND TELEPHONE EAST CORPORATION" },
    {  441211, "STARCAT CABLE NETWORK Co., LTD." },
    {  441212, "I-TEC Solutions Co., Ltd." },
    {  441213, "Hokkaido Telecommunication Network Co., Inc." },
    {  708001, "Megatel" },
    {  708002, "Celtel" },
    {  708040, "Digicel Honduras" },
    {  714020, "Grupo de Comunicaciones Digitales, S.A. (TIGO)" },
    {  722010, "Compañia de Radiocomunicaciones Moviles S.A." },
    {  722020, "Nextel Argentina srl" },
    {  722070, "Telefónica Comunicaciones Personales S.A." },
    {  722310, "CTI PCS S.A." },
    {  722320, "Compañia de Telefonos del Interior Norte S.A." },
    {  722330, "Compañia de Telefonos del Interior S.A." },
    {  722341, "Telecom Personal S.A." },
    {  732001, "Colombia Telecomunicaciones S.A. - Telecom" },
    {  732002, "Edatel S.A." },
    {  732020, "Emtelsa" },
    {  732099, "Emcali" },
    {  732101, "Comcel S.A. Occel S.A./Celcaribe" },
    {  732102, "Bellsouth Colombia S.A." },
    {  732103, "Colombia Móvil S.A." },
    {  732111, "Colombia Móvil S.A." },
    {  732123, "Telefónica Móviles Colombia S.A." },
    {  732130, "Avantel" },
    {  738002, "Guyana Telephone & Telegraph Company Limited (Cellink)" },
    {  738003, "Quark Communications Inc." },
    {  738040, "E-Networks Inc." },
    {  750001, "Touch" },
    {  999999, "Internal use, example, testing" },
    { 1666665, "Unset" },
    {  0, NULL }
};

value_string_ext mcc_mnc_3digits_codes_ext = VALUE_STRING_EXT_INIT(mcc_mnc_3digits_codes);

static int proto_e212;
static int hf_E212_imsi;
static int hf_e212_assoc_imsi;
static int hf_E212_mcc;
static int hf_E212_mcc_lai;
static int hf_E212_mcc_sai;
static int hf_E212_mcc_rai;
static int hf_E212_mcc_cgi;
static int hf_E212_mcc_ecgi;
static int hf_E212_mcc_tai;
static int hf_E212_mcc_nrcgi;
static int hf_E212_mcc_5gstai;
static int hf_E212_mcc_gummei;
static int hf_E212_mcc_guami;
static int hf_E212_mcc_serv_net;
static int hf_E212_mnc;
static int hf_E212_mnc_lai;
static int hf_E212_mnc_sai;
static int hf_E212_mnc_rai;
static int hf_E212_mnc_cgi;
static int hf_E212_mnc_ecgi;
static int hf_E212_mnc_tai;
static int hf_E212_mnc_nrcgi;
static int hf_E212_mnc_5gstai;
static int hf_E212_mnc_gummei;
static int hf_E212_mnc_guami;
static int hf_E212_mnc_serv_net;

static int ett_e212_imsi;

static expert_field ei_E212_mcc_non_decimal;
static expert_field ei_E212_mnc_non_decimal;
static expert_field ei_E212_imsi_malformed;

/* static int hf_E212_msin; */

/*
 * MCC/MNC dissection - little endian MNC encoding
 *
 * MNC of length 2:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 3  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 *
 * MCC/MNC dissection - big endian MNC encoding
 *
 * MNC of length 2:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 1  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 3  |  MNC digit 2  |  octet x+2
 * +---------------+---------------+
 */

/*
 * Return MCC MNC in a packet scope allocated string that can be used in labels.
 */
char *
dissect_e212_mcc_mnc_wmem_packet_str(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, e212_number_type_t number_type, bool little_endian)
{

    int         start_offset, mcc_mnc;
    uint8_t     octet;
    uint16_t    mcc, mnc;
    uint8_t     mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    proto_item *item;
    char       *mcc_str, *mnc_str, *mcc_mnc_str;
    bool        long_mnc = false;
    int         hf_E212_mcc_id, hf_E212_mnc_id;

    switch(number_type){
    case E212_LAI:
        hf_E212_mcc_id = hf_E212_mcc_lai;
        hf_E212_mnc_id = hf_E212_mnc_lai;
        break;
    case E212_RAI:
        hf_E212_mcc_id = hf_E212_mcc_rai;
        hf_E212_mnc_id = hf_E212_mnc_rai;
        break;
    case E212_SAI:
        hf_E212_mcc_id = hf_E212_mcc_sai;
        hf_E212_mnc_id = hf_E212_mnc_sai;
        break;
    case E212_CGI:
        hf_E212_mcc_id = hf_E212_mcc_cgi;
        hf_E212_mnc_id = hf_E212_mnc_cgi;
        break;
    case E212_ECGI:
        hf_E212_mcc_id = hf_E212_mcc_ecgi;
        hf_E212_mnc_id = hf_E212_mnc_ecgi;
        break;
    case E212_TAI:
        hf_E212_mcc_id = hf_E212_mcc_tai;
        hf_E212_mnc_id = hf_E212_mnc_tai;
        break;
    case E212_NRCGI:
        hf_E212_mcc_id = hf_E212_mcc_nrcgi;
        hf_E212_mnc_id = hf_E212_mnc_nrcgi;
        break;
    case E212_5GSTAI:
        hf_E212_mcc_id = hf_E212_mcc_5gstai;
        hf_E212_mnc_id = hf_E212_mnc_5gstai;
        break;
    case E212_GUMMEI:
        hf_E212_mcc_id = hf_E212_mcc_gummei;
        hf_E212_mnc_id = hf_E212_mnc_gummei;
        break;
    case E212_GUAMI:
        hf_E212_mcc_id = hf_E212_mcc_guami;
        hf_E212_mnc_id = hf_E212_mnc_guami;
        break;
    case E212_SERV_NET:
        hf_E212_mcc_id = hf_E212_mcc_serv_net;
        hf_E212_mnc_id = hf_E212_mnc_serv_net;
        break;
    default:
        hf_E212_mcc_id = hf_E212_mcc;
        hf_E212_mnc_id = hf_E212_mnc;
    }
    start_offset = offset;
    /* MCC + MNC */
    mcc_mnc = tvb_get_ntoh24(tvb,offset);
    /* Mobile country code MCC */
    octet = tvb_get_uint8(tvb,offset);
    mcc1 = octet & 0x0f;
    mcc2 = octet >> 4;
    offset++;
    octet = tvb_get_uint8(tvb,offset);
    mcc3 = octet & 0x0f;
    /* MNC, Mobile network code (octet 3 bits 5 to 8, octet 4)  */
    mnc3 = octet >> 4;
    offset++;
    octet = tvb_get_uint8(tvb,offset);
    mnc1 = octet & 0x0f;
    mnc2 = octet >> 4;

    mcc = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc = 10 * mnc1 + mnc2;
    if ((mnc3 != 0xf) || (mcc_mnc == 0xffffff)) {
        long_mnc = true;
        if(little_endian)
            mnc = 10 * mnc + mnc3;
        else
            mnc = 100 * mnc3 + mnc;
    }
    mcc_str = wmem_strdup_printf(pinfo->pool, "%03u", mcc);
    item = proto_tree_add_string_format_value(tree, hf_E212_mcc_id, tvb,
                                              start_offset, 2, mcc_str, "%s (%s)",
                                              val_to_str_ext_const(mcc, &E212_codes_ext, "Unknown"),
                                              mcc_str);
    if (((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9)) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mcc_non_decimal);

    if (long_mnc) {
        mnc_str = wmem_strdup_printf(pinfo->pool, "%03u", mnc);
        item = proto_tree_add_string_format_value(tree, hf_E212_mnc_id , tvb, start_offset + 1, 2, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown"),
                   mnc_str);
        /* Prepare a string with the MCC and MNC including the country and Operator if
         * known, do NOT print unknown.
         */
        mcc_mnc_str = wmem_strdup_printf(pinfo->pool, "MCC %u %s, MNC %03u %s",
            mcc,
            val_to_str_ext_const(mcc,&E212_codes_ext,""),
            mnc,
            val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, ""));
    } else {
        mnc_str = wmem_strdup_printf(pinfo->pool, "%02u", mnc);
        item = proto_tree_add_string_format_value(tree, hf_E212_mnc_id , tvb, start_offset + 1, 2, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown"),
                   mnc_str);
        /* Prepare a string with the MCC and MNC including the country and Operator if
         * known, do NOT print unknown.
         */
        mcc_mnc_str = wmem_strdup_printf(pinfo->pool, "MCC %u %s, MNC %02u %s",
            mcc,
            val_to_str_ext_const(mcc,&E212_codes_ext,""),
            mnc,
            val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, ""));
    }

    if (((mnc1 > 9) || (mnc2 > 9) || ((mnc3 > 9) && (mnc3 != 0x0f))) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mnc_non_decimal);

    return mcc_mnc_str;
}

int
dissect_e212_mcc_mnc(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, e212_number_type_t number_type, bool little_endian)
{
    dissect_e212_mcc_mnc_wmem_packet_str(tvb, pinfo, tree, offset, number_type, little_endian);
    return offset +3;
}

/*
 * When we want to decode the MCC/MNC pair in an address that is encoded according to E.212
 * the main problem is that we do not know whether we are dealing with a length 2 or length 3
 * MNC. Initially, it was possible to find the length of the MNC by checking the MCC code.
 * Originally each country employed a pure 2 digit or 3 digit MNC scheme. However, it is possible
 * to find countries now that employ both lengths for the MNC (e.g. Canada, India).
 * Since in these cases we can be sure that an MNC cannot possible be a prefix for another MNC, we
 * initially assume that the MNC is 2 digits long. If the MCC/MNC combination gives a match in our
 * mcc_mnc_codes list then we can be sure that we deal with an MNC that is 2 digits long. Otherwise,
 * assume that the MNC is 3 digits long.
 *
 * MNC of length 2:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   | addr digit 1  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * MNC of length 3:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   |  MNC digit 3  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * This function will consume either 2.5 or 3 octets. For this reason it returns
 * the number of nibbles consumed, i.e. 5 or 6 (2 or 3 digits long MNC respectively)
 */
int
dissect_e212_mcc_mnc_in_address(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    uint32_t    start_offset, mcc_mnc;
    uint8_t     octet;
    uint16_t    mcc, mnc;
    char       *mcc_str, *mnc_str;
    uint8_t     mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    proto_item *item;
    bool        long_mnc;

    long_mnc = false;
    start_offset = offset;

    /* MCC + MNC */
    mcc_mnc = tvb_get_ntoh24(tvb,offset);

    /* MCC digits 1 and 2 */
    octet = tvb_get_uint8(tvb,offset);
    mcc1  = octet & 0x0f;
    mcc2  = octet >> 4;
    offset++;

    /* MCC digit 3 and MNC digit 1 */
    octet = tvb_get_uint8(tvb,offset);
    mcc3  = octet & 0x0f;
    mnc1  = octet >> 4;
    offset++;

    /* MNC digits 2 and 3 */
    octet = tvb_get_uint8(tvb,offset);
    mnc2  = octet & 0x0f;
    mnc3  = octet >> 4;

    mcc   = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc   = 10 * mnc1 + mnc2;

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext) && mnc3 != 0xf) {
        mnc = 10 * mnc + mnc3;
        long_mnc = true;
    }

    mcc_str = wmem_strdup_printf(pinfo->pool, "%03u", mcc);
    item = proto_tree_add_string_format_value(tree, hf_E212_mcc, tvb,
                                              start_offset, 2, mcc_str, "%s (%s)",
                                              val_to_str_ext_const(mcc, &E212_codes_ext, "Unknown"),
                                              mcc_str);
    if (((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9)) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mcc_non_decimal);

    if (long_mnc){
        mnc_str = wmem_strdup_printf(pinfo->pool, "%03u", mnc);
        item = proto_tree_add_string_format_value(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown"),
                   mnc_str);
    }else{
        mnc_str = wmem_strdup_printf(pinfo->pool, "%02u", mnc);
        item = proto_tree_add_string_format_value(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown"),
                   mnc_str);
    }

    if (((mnc1 > 9) || (mnc2 > 9) || (long_mnc && (mnc3 > 9))) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mnc_non_decimal);

    if (long_mnc)
        return 6;
    else
        return 5;
}

/*
 * MNC of length 2:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 1  |  Other data   |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 2  |  octet x+1
 *   +---------------+---------------+
 *   | MNC digit  3  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * MNC of length 3:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 1  |  Other data   |  octet x
 *   +---------------+---------------+
 *   |  MCC digit 3  |  MCC digit 2  |  octet x+1
 *   +---------------+---------------+
 *   |  MNC digit 2  |  MNC digit 1  |  octet x+2
 *   +---------------+---------------+
 *   |  .....        |  MNC digit 3  |  octet x+3
 *   +---------------+---------------+
 */
static int
dissect_e212_mcc_mnc_high_nibble(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{

    uint32_t    start_offset;
    uint8_t     octet;
    uint16_t    mcc, mnc;
    char       *mcc_str, *mnc_str;
    uint8_t     mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    bool        long_mnc;

    long_mnc = false;
    start_offset = offset;

    /* MCC digits 1 */
    octet = tvb_get_uint8(tvb,offset);
    mcc1  = octet >> 4;
    offset++;

    /* MCC digits 1 and 2 */
    octet = tvb_get_uint8(tvb,offset);
    mcc2  = octet & 0x0f;
    mcc3  = octet >> 4;
    offset++;

    /* MNC digit 1 and MNC digit 2 */
    octet = tvb_get_uint8(tvb,offset);
    mnc1  = octet & 0x0f;
    mnc2  = octet >> 4;
    offset++;

    /* MNC digits 3 */
    octet = tvb_get_uint8(tvb,offset);
    mnc3  = octet & 0x0f;

    mcc   = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc   = 10 * mnc1 + mnc2;

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext) && mnc3 != 0xf) {
        mnc = 10 * mnc + mnc3;
        long_mnc = true;
    }

    mcc_str = wmem_strdup_printf(pinfo->pool, "%03u", mcc);
    proto_tree_add_string_format_value(tree, hf_E212_mcc, tvb,
                                       start_offset, 2, mcc_str, "%s (%s)",
                                       val_to_str_ext_const(mcc, &E212_codes_ext, "Unknown"),
                                       mcc_str);

    if (long_mnc){
        mnc_str = wmem_strdup_printf(pinfo->pool, "%03u", mnc);
        proto_tree_add_string_format_value(tree, hf_E212_mnc , tvb, start_offset + 2, 2, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown"),
                   mnc_str);
    }else{
        mnc_str = wmem_strdup_printf(pinfo->pool, "%02u", mnc);
        proto_tree_add_string_format_value(tree, hf_E212_mnc , tvb, start_offset + 2, 1, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown"),
                   mnc_str);
    }

    if (long_mnc)
        return 7;
    else
        return 5;
}

int
dissect_e212_mcc_mnc_in_utf8_address(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{
    uint16_t mcc = 0, mnc = 0;
    char   *mcc_str, *mnc_str;
    bool        long_mnc = false;

    ws_strtou16(tvb_get_string_enc(pinfo->pool, tvb, offset, 3, ENC_UTF_8),
        NULL, &mcc);
    ws_strtou16(tvb_get_string_enc(pinfo->pool, tvb, offset + 3, 2, ENC_UTF_8),
        NULL, &mnc);

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext)) {
        if (tvb_reported_length_remaining(tvb, offset + 3) > 2) {
            ws_strtou16(tvb_get_string_enc(pinfo->pool, tvb, offset + 3, 3, ENC_UTF_8),
                NULL, &mnc);
            long_mnc = true;
        }
    }

    mcc_str = wmem_strdup_printf(pinfo->pool, "%03u", mcc);
    proto_tree_add_string_format_value(tree, hf_E212_mcc, tvb,
                                       offset, 3, mcc_str, "%s (%s)",
                                       val_to_str_ext_const(mcc, &E212_codes_ext, "Unknown"),
                                       mcc_str);

    if (long_mnc){
        mnc_str = wmem_strdup_printf(pinfo->pool, "%03u", mnc);
        proto_tree_add_string_format_value(tree, hf_E212_mnc, tvb, offset + 3, 3, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown1"),
                   mnc_str);
    }else{
        mnc_str = wmem_strdup_printf(pinfo->pool, "%02u", mnc);
        proto_tree_add_string_format_value(tree, hf_E212_mnc, tvb, offset + 3, 2, mnc_str,
                   "%s (%s)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown2"),
                   mnc_str);
    }

    if (long_mnc)
        return 6;
    else
        return 5;
}

static bool
is_imsi_string_valid(const char *imsi_str)
{
    size_t len;

    if (imsi_str == NULL)
        return false;
    len = strlen(imsi_str);
    /* According to TS 23.003 2.2 and 2.3, the number of digits in IMSI shall not exceed 15.
     * Even if in the reality imsis are always 14 or 15 digits long, the standard doesn't say
     * anything about minimum length, except for the fact that they shall have a valid MCC
     * (3 digits long), a valid MNC (2 or 3 digits long) and a MSIN (at least 1 digit).
     * As the dissector actually only decodes the MCC and MNC, allow to decode a
       short IMSI (without MSIN) as used by MAP messages */
    if (len < 5 || len > 15 || strchr(imsi_str, '?')) {
        return false;
    }
    return true;
}

const char *
dissect_e212_imsi(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int length, bool skip_first)
{
    proto_item *item;
    proto_tree *subtree;
    const uint8_t *imsi_str;
    uint8_t   oct;

    /* Fetch the BCD encoded digits from tvb indicated half byte, formating the digits according to
     * a default digit set of 0-9 returning "?" for overdecadic digits a pointer to the wmem
     * allocated string will be returned.
     */
    if (skip_first) {
        oct = tvb_get_uint8(tvb, offset);

        /* Check whether IMSI has even or odd number of digits.
         * The format is specified in Figure 10.5.4/3GPP TS 24.008 Mobile Identity information element
         */
        if (oct & 0x08) {
            item = proto_tree_add_item_ret_string(tree, hf_E212_imsi, tvb, offset, length, ENC_BCD_DIGITS_0_9 | ENC_LITTLE_ENDIAN | ENC_BCD_SKIP_FIRST, pinfo->pool, &imsi_str);
        } else {
            /* See tvb_get_bcd_string(...) documentation.
             *     Note that if both skip_first and odd are true, then both the first and last semi-octet are skipped,
             *     i.e. an even number of nibbles are considered.
             */
            item = proto_tree_add_item_ret_string(tree, hf_E212_imsi, tvb, offset, length, ENC_BCD_DIGITS_0_9 | ENC_LITTLE_ENDIAN | ENC_BCD_SKIP_FIRST | ENC_BCD_ODD_NUM_DIG, pinfo->pool, &imsi_str);
        }
    } else {
        item = proto_tree_add_item_ret_string(tree, hf_E212_imsi, tvb, offset, length, ENC_BCD_DIGITS_0_9 | ENC_LITTLE_ENDIAN, pinfo->pool, &imsi_str);
    }
    if (!is_imsi_string_valid(imsi_str)) {
        expert_add_info(pinfo, item, &ei_E212_imsi_malformed);
    }
    item = proto_tree_add_string(tree, hf_e212_assoc_imsi, tvb, offset, length, imsi_str);
    proto_item_set_generated(item);
    subtree = proto_item_add_subtree(item, ett_e212_imsi);

    if(skip_first) {
        dissect_e212_mcc_mnc_high_nibble(tvb, pinfo, subtree, offset);
    } else {
        dissect_e212_mcc_mnc_in_address(tvb, pinfo, subtree, offset);
    }

    return imsi_str;
}

const char *
dissect_e212_utf8_imsi(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int length)
{
    proto_item *item;
    proto_tree *subtree;
    const char *imsi_str;

    /* Fetch the UTF8-encoded IMSI */
    imsi_str = tvb_get_string_enc(pinfo->pool, tvb, offset, length, ENC_UTF_8);
    item = proto_tree_add_string(tree, hf_E212_imsi, tvb, offset, length, imsi_str);
    if (!is_imsi_string_valid(imsi_str)) {
        expert_add_info(pinfo, item, &ei_E212_imsi_malformed);
    }
    item = proto_tree_add_string(tree, hf_e212_assoc_imsi, tvb, offset, length, imsi_str);
    proto_item_set_generated(item);

    subtree = proto_item_add_subtree(item, ett_e212_imsi);

    dissect_e212_mcc_mnc_in_utf8_address(tvb, pinfo, subtree, offset);

    return imsi_str;
}

/*
 * Register the protocol with Wireshark.
 *
 * This format is required because a script is used to build the C function
 * that calls all the protocol registration.
 */


void
proto_register_e212(void)
{

/* Setup list of header fields  See Section 1.6.1 for details */
    static hf_register_info hf[] = {
    { &hf_E212_imsi,
        { "IMSI","e212.imsi",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "International mobile subscriber identity(IMSI)", HFILL }
    },
    { &hf_e212_assoc_imsi,
      { "Association IMSI", "e212.assoc.imsi",
        FT_STRING, BASE_NONE, NULL, 0,
        "Used to tie MSGs related to the same IMSI", HFILL}
    },

    { &hf_E212_mcc,
        { "Mobile Country Code (MCC)","e212.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_lai,
        { "Mobile Country Code (MCC)","e212.lai.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_rai,
        { "Mobile Country Code (MCC)","e212.rai.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_sai,
        { "Mobile Country Code (MCC)","e212.sai.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_cgi,
        { "Mobile Country Code (MCC)","e212.cgi.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_ecgi,
        { "Mobile Country Code (MCC)","e212.ecgi.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_tai,
        { "Mobile Country Code (MCC)","e212.tai.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_nrcgi,
        { "Mobile Country Code (MCC)","e212.nrcgi.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_5gstai,
        { "Mobile Country Code (MCC)","e212.5gstai.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_gummei,
        { "Mobile Country Code (MCC)","e212.gummei.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_guami,
        { "Mobile Country Code (MCC)","e212.guami.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_serv_net,
        { "Mobile Country Code (MCC)","e212.serv_net.mcc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mnc,
        { "Mobile Network Code (MNC)","e212.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_lai,
        { "Mobile Network Code (MNC)","e212.lai.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_rai,
        { "Mobile Network Code (MNC)","e212.rai.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_sai,
        { "Mobile Network Code (MNC)","e212.sai.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_cgi,
        { "Mobile Network Code (MNC)","e212.cgi.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_ecgi,
        { "Mobile Network Code (MNC)","e212.ecgi.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_tai,
        { "Mobile Network Code (MNC)","e212.tai.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_nrcgi,
        { "Mobile Network Code (MNC)","e212.nrcgi.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_5gstai,
        { "Mobile Network Code (MNC)","e212.5gstai.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_gummei,
        { "Mobile Network Code (MNC)","e212.gummei.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_guami,
        { "Mobile Network Code (MNC)","e212.guami.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
    { &hf_E212_mnc_serv_net,
        { "Mobile Network Code (MNC)","e212.serv_net.mnc",
        FT_STRING, BASE_NONE, NULL, 0x0,
        NULL, HFILL }
    },
#if 0
    { &hf_E212_msin,
        { "Mobile Subscriber Identification Number (MSIN)", "e212.msin",
        FT_STRING, BASE_NONE, NULL, 0,
        "Mobile Subscriber Identification Number(MSIN)", HFILL }},
#endif
    };


    static int *ett_e212_array[] = {
        &ett_e212_imsi,
    };

    static ei_register_info ei[] = {
        { &ei_E212_mcc_non_decimal, { "e212.mcc.non_decimal", PI_MALFORMED, PI_WARN, "MCC contains non-decimal digits", EXPFILL }},
        { &ei_E212_mnc_non_decimal, { "e212.mnc.non_decimal", PI_MALFORMED, PI_WARN, "MNC contains non-decimal digits", EXPFILL }},
        { &ei_E212_imsi_malformed, { "e212.imsi.malformed", PI_MALFORMED, PI_WARN, "Malformed IMSI", EXPFILL }},
    };

    expert_module_t* expert_e212;

    /*
     * Register the protocol name and description
     */
    proto_e212 = proto_register_protocol("ITU-T E.212 number", "E.212", "e212");

    /*
     * Required function calls to register
     * the header fields and subtrees used.
     */
    proto_register_field_array(proto_e212, hf, array_length(hf));
    proto_register_subtree_array(ett_e212_array, array_length(ett_e212_array));
    expert_e212 = expert_register_protocol(proto_e212);
    expert_register_field_array(expert_e212, ei, array_length(ei));

}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
