// Item.cpp - Implementation of CItem
//
// WinDirStat - Directory Statistics
// Copyright (C) 2003-2005 Bernhard Seifert
// Copyright (C) 2004-2024 WinDirStat Team (windirstat.net)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#include "stdafx.h"
#include "aclapi.h"
#include "sddl.h"

#include "WinDirStat.h"
#include "DirStatDoc.h"
#include "MainFrame.h"
#include "CommonHelpers.h"
#include "GlobalHelpers.h"
#include "SelectObject.h"
#include "Item.h"
#include "BlockingQueue.h"
#include "Localization.h"
#include "SmartPointer.h"

#include <string>
#include <algorithm>
#include <unordered_set>
#include <functional>
#include <queue>
#include <shared_mutex>
#include <stack>
#include <array>

#pragma comment(lib, "crypt32.lib")
#pragma comment(lib, "bcrypt.lib")

CItem::CItem(const ITEMTYPE type, const std::wstring & name) : m_Name(name), m_Type(type)
{
    if (IsType(IT_DRIVE))
    {
        // The name string on the drive is two parts separated by a pipe.  For example,
        // C:\|Local Disk (C:) is the true path following by the name description
        m_Name = std::format(L"{:.2}|{}", m_Name, FormatVolumeNameOfRootPath(m_Name));
    }

    if (IsType(IT_FILE))
    {
        if (const LPCWSTR ext = wcsrchr(name.c_str(), L'.'); ext != nullptr)
        {
            std::wstring extToAdd(&ext[0]);
            _wcslwr_s(extToAdd.data(), extToAdd.size() + 1);

            static std::shared_mutex extLock;
            static std::unordered_set<std::wstring> extcache;
            std::lock_guard lock(extLock);
            const auto cached = extcache.insert(std::move(extToAdd));
            m_Extension = cached.first->c_str();
        }
        else
        {
            static LPCWSTR noext = L"";
            m_Extension = noext;
        }
    }
    else
    {
        m_FolderInfo = new CHILDINFO;
        m_Extension = m_Name.c_str();
    }
}

CItem::CItem(const ITEMTYPE type, const std::wstring& name, const FILETIME lastChange,
             const ULONGLONG sizePhysical, const ULONGLONG sizeLogical,
             const DWORD attributes, const ULONG files, const ULONG subdirs) : CItem(type, name)
{
    m_LastChange = lastChange;
    m_SizePhysical = sizePhysical;
    m_SizeLogical = sizeLogical;
    m_Attributes = attributes;
    if (m_FolderInfo != nullptr)
    {
        m_FolderInfo->m_Subdirs = subdirs;
        m_FolderInfo->m_Files = files;
    }
}

CItem::~CItem()
{
    if (m_FolderInfo != nullptr)
    {
        for (const auto& m_Child : m_FolderInfo->m_Children)
        {
            delete m_Child;
        }
        delete m_FolderInfo;
    }
}

CRect CItem::TmiGetRectangle() const
{
    return m_Rect;
}

void CItem::TmiSetRectangle(const CRect& rc)
{
    m_Rect = rc;
}

bool CItem::DrawSubitem(const int subitem, CDC* pdc, CRect rc, const UINT state, int* width, int* focusLeft) const
{
    if (subitem == COL_NAME)
    {
        return CTreeListItem::DrawSubitem(subitem, pdc, rc, state, width, focusLeft);
    }
    if (subitem != COL_SUBTREEPERCENTAGE)
    {
        return false;
    }

    const bool showReadJobs = MustShowReadJobs();

    if (showReadJobs && !COptions::PacmanAnimation)
    {
        return false;
    }

    if (showReadJobs && IsDone())
    {
        return false;
    }

    if (width != nullptr)
    {
        *width = GetSubtreePercentageWidth();
        return true;
    }

    DrawSelection(CFileTreeControl::Get(), pdc, rc, state);

    if (showReadJobs)
    {
        constexpr SIZE sizeDeflatePacman = { 1, 2 };
        rc.DeflateRect(sizeDeflatePacman);
        DrawPacman(pdc, rc, CFileTreeControl::Get()->GetItemSelectionBackgroundColor(this));
    }
    else
    {
        rc.DeflateRect(2, 5);
        for (int i = 0; i < GetIndent(); i++)
        {
            rc.left += rc.Width() / 10;
        }

        DrawPercentage(pdc, rc, GetFraction(), GetPercentageColor());
    }
    return true;
}

std::wstring CItem::GetText(const int subitem) const
{
    switch (subitem)
    {
    case COL_SIZE_PHYSICAL: return FormatBytes(GetSizePhysical());
    case COL_SIZE_LOGICAL: return FormatBytes(GetSizeLogical());

    case COL_NAME:
        if (IsType(IT_DRIVE))
        {
            return m_Name.substr(_countof(L"?:"));
        }
        return m_Name;

    case COL_OWNER:
        if (IsType(IT_FILE | IT_DIRECTORY))
        {
            return GetOwner();
        }
        break;

    case COL_SUBTREEPERCENTAGE:
        if (!IsDone())
        {
            if (GetReadJobs() == 1)
            {
                return Localization::Lookup(IDS_ONEREADJOB);
            }

            return Localization::Format(IDS_sREADJOBS, FormatCount(GetReadJobs()));
        }
        break;

    case COL_PERCENTAGE:
        if (COptions::ShowTimeSpent && MustShowReadJobs() || IsRootItem())
        {
            return L"[" + FormatMilliseconds(GetTicksWorked() * 1000) + L"]";
        }
        return FormatDouble(GetFraction() * 100) + L"%";

    case COL_ITEMS:
        if (!IsType(IT_FILE | IT_FREESPACE | IT_UNKNOWN))
        {
            return FormatCount(GetItemsCount());
        }
        break;

    case COL_FILES:
        if (!IsType(IT_FILE | IT_FREESPACE | IT_UNKNOWN))
        {
            return FormatCount(GetFilesCount());
        }
        break;

    case COL_FOLDERS:
        if (!IsType(IT_FILE | IT_FREESPACE | IT_UNKNOWN))
        {
            return FormatCount(GetFoldersCount());
        }
        break;

    case COL_LASTCHANGE:
        if (!IsType(IT_FREESPACE | IT_UNKNOWN))
        {
            return FormatFileTime(m_LastChange);
        }
        break;

    case COL_ATTRIBUTES:
        if (!IsType(IT_FREESPACE | IT_UNKNOWN | IT_MYCOMPUTER))
        {
            return FormatAttributes(GetAttributes());
        }
        break;

    default: ASSERT(FALSE);
    }

    return {};
}

COLORREF CItem::GetItemTextColor() const
{
    // Get the file/folder attributes
    const DWORD attr = GetAttributes();

    // This happens e.g. on a Unicode-capable FS when using ANSI APIs
    // to list files with ("real") Unicode names
    if (attr == INVALID_FILE_ATTRIBUTES)
    {
        return CTreeListItem::GetItemTextColor();
    }

    // Check for compressed flag
    if (attr & FILE_ATTRIBUTE_COMPRESSED)
    {
        return CDirStatApp::Get()->AltColor();
    }

    if (attr & FILE_ATTRIBUTE_ENCRYPTED)
    {
        return CDirStatApp::Get()->AltEncryptionColor();
    }

    // The rest is not colored
    return CTreeListItem::GetItemTextColor();
}

int CItem::CompareSibling(const CTreeListItem* tlib, const int subitem) const
{
    const CItem* other = reinterpret_cast<const CItem*>(tlib);

    switch (subitem)
    {
        case COL_NAME:
        {
            if (IsType(IT_DRIVE))
            {
                return signum(_wcsicmp(m_Name.c_str(),other->m_Name.c_str()));
            }
        }

        case COL_SUBTREEPERCENTAGE:
        {
            if (MustShowReadJobs())
            {
                return usignum(GetReadJobs(), other->GetReadJobs());
            }
            else
            {
                return signum(GetFraction() - other->GetFraction());
            }
        }

        case COL_PERCENTAGE:
        {
            return signum(GetFraction() - other->GetFraction());
        }

        case COL_SIZE_PHYSICAL:
        {
            return usignum(GetSizePhysical(), other->GetSizePhysical());
        }

        case COL_SIZE_LOGICAL:
        {
            return usignum(GetSizeLogical(), other->GetSizeLogical());
        }

        case COL_ITEMS:
        {
            return usignum(GetItemsCount(), other->GetItemsCount());
        }

        case COL_FILES:
        {
            return usignum(GetFilesCount(), other->GetFilesCount());
        }

        case COL_FOLDERS:
        {
            return usignum(GetFoldersCount(), other->GetFoldersCount());
        }

        case COL_LASTCHANGE:
        {
            if (m_LastChange < other->m_LastChange)
            {
                return -1;
            }
            if (m_LastChange == other->m_LastChange)
            {
                return 0;
            }

            return 1;
        }

        case COL_ATTRIBUTES:
        {
            return signum(GetSortAttributes() - other->GetSortAttributes());
        }

        case COL_OWNER:
        {
            return signum(_wcsicmp(GetOwner().c_str(), other->GetOwner().c_str()));
        }

        default:
        {
            return 0;
        }
    }
}

int CItem::GetTreeListChildCount() const
{
    if (m_FolderInfo == nullptr) return 0;
    return static_cast<int>(GetChildren().size());
}

CTreeListItem* CItem::GetTreeListChild(const int i) const
{
    return GetChildren()[i];
}

short CItem::GetImageToCache() const
{
    // (Caching is done in CTreeListItem)

    if (IsType(IT_MYCOMPUTER))
    {
        return GetIconImageList()->GetMyComputerImage();
    }
    if (IsType(IT_FREESPACE))
    {
        return GetIconImageList()->GetFreeSpaceImage();
    }
    if (IsType(IT_UNKNOWN))
    {
        return GetIconImageList()->GetUnknownImage();
    }

    const std::wstring longpath = GetPathLong();
    if (CDirStatApp::Get()->GetReparseInfo()->IsVolumeMountPoint(longpath, m_Attributes))
    {
        return GetIconImageList()->GetMountPointImage();
    }
    if ((CReparsePoints::IsSymbolicLink(longpath, m_Attributes) ||
        CDirStatApp::Get()->GetReparseInfo()->IsJunction(longpath, m_Attributes)) &&
        !CReparsePoints::IsCloudLink(longpath, m_Attributes))
    {
        constexpr DWORD mask = FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM;
        const bool osFile = (GetAttributes() & mask) == mask;
        return osFile ? GetIconImageList()->GetJunctionProtectedImage() : GetIconImageList()->GetJunctionImage();
    }

    return GetIconImageList()->GetFileImage(GetPath(), GetAttributes());
}

void CItem::DrawAdditionalState(CDC* pdc, const CRect& rcLabel) const
{
    if (!IsRootItem() && this == CDirStatDoc::GetDocument()->GetZoomItem())
    {
        CRect rc = rcLabel;
        rc.InflateRect(1, 0);
        rc.bottom++;

        CSelectStockObject sobrush(pdc, NULL_BRUSH);
        CPen pen(PS_SOLID, 2, CDirStatDoc::GetDocument()->GetZoomColor());
        CSelectObject sopen(pdc, &pen);

        pdc->Rectangle(rc);
    }
}

int CItem::GetSubtreePercentageWidth()
{
    return 105;
}

CItem* CItem::FindCommonAncestor(const CItem* item1, const CItem* item2)
{
    for (auto parent = item1; parent != nullptr; parent = parent->GetParent())
    {
        if (parent->IsAncestorOf(item2)) return const_cast<CItem*>(parent);
    }

    ASSERT(FALSE);
    return nullptr;
}

ULONGLONG CItem::GetProgressRange() const
{
    if (IsType(IT_MYCOMPUTER))
    {
        return GetProgressRangeMyComputer();
    }
    if (IsType(IT_DRIVE))
    {
        return GetProgressRangeDrive();
    }
    if (IsType(IT_FILE | IT_DIRECTORY))
    {
        return 0;
    }

    ASSERT(FALSE);
    return 0;
}

ULONGLONG CItem::GetProgressPos() const
{
    if (IsType(IT_MYCOMPUTER))
    {
        ULONGLONG pos = 0;
        for (const auto& child : GetChildren())
        {
            pos += child->GetProgressPos();
        }
        return pos;
    }
    if (IsType(IT_DRIVE))
    {
        ULONGLONG pos = GetSizePhysical();
        const CItem* fs = FindFreeSpaceItem();
        pos -= (fs != nullptr) ? fs->GetSizePhysical() : 0;
        return pos;
    }

    return 0;
}

void CItem::UpdateStatsFromDisk()
{
    if (IsType(IT_DIRECTORY | IT_FILE))
    {
        FileFindEnhanced finder;
        if (finder.FindFile(GetFolderPath(),IsType(ITF_ROOTITEM) ? std::wstring() : GetName()))
        {
            SetLastChange(finder.GetLastWriteTime());
            SetAttributes(finder.GetAttributes());

            if (IsType(IT_FILE))
            {
                UpwardSubtractSizePhysical(m_SizePhysical);
                UpwardAddSizePhysical(finder.GetFileSizePhysical());
                UpwardSubtractSizeLogical(m_SizeLogical);
                UpwardAddSizeLogical(finder.GetFileSizeLogical());
            }
        }
    }
    else if (IsType(IT_DRIVE))
    {
        SmartPointer<HANDLE> handle(CloseHandle, CreateFile(GetPathLong().c_str(), FILE_READ_ATTRIBUTES, 
            FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE, nullptr,
            OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, nullptr));
        if (handle != INVALID_HANDLE_VALUE)
        {
            GetFileTime(handle, nullptr, nullptr, &m_LastChange);
        }
    }
}

const std::vector<CItem*>& CItem::GetChildren() const
{
    return m_FolderInfo->m_Children;
}

CItem* CItem::GetParent() const
{
    return reinterpret_cast<CItem*>(CTreeListItem::GetParent());
}

void CItem::AddChild(CItem* child, const bool addOnly)
{
    if (!addOnly)
    {
        UpwardAddSizePhysical(child->m_SizePhysical);
        UpwardAddSizeLogical(child->m_SizeLogical);
        UpwardUpdateLastChange(child->m_LastChange);
    }

    child->SetParent(this);

    std::lock_guard guard(m_FolderInfo->m_Protect);
    m_FolderInfo->m_Children.push_back(child);

    if (IsVisible() && IsExpanded())
    {
        (void)GetImage();
        CMainFrame::Get()->InvokeInMessageThread([this, child]
        {
            CFileTreeControl::Get()->OnChildAdded(this, child);
        });
    }
}

void CItem::RemoveChild(CItem* child)
{
    std::lock_guard guard(m_FolderInfo->m_Protect);
    std::erase(m_FolderInfo->m_Children, child);

    if (IsVisible())
    {
        CMainFrame::Get()->InvokeInMessageThread([this, child]
        {
            CFileTreeControl::Get()->OnChildRemoved(this, child);
        });
    }

    delete child;
}

void CItem::RemoveAllChildren()
{
    if (m_FolderInfo == nullptr) return;
    CMainFrame::Get()->InvokeInMessageThread([this]
    {
        CFileTreeControl::Get()->OnRemovingAllChildren(this);
    });

    std::lock_guard guard(m_FolderInfo->m_Protect);
    for (const auto& child : m_FolderInfo->m_Children)
    {
        delete child;
    }
    m_FolderInfo->m_Children.clear();
}

void CItem::UpwardAddFolders(const ULONG dirCount)
{
    if (dirCount == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (p->IsType(IT_FILE)) continue;
        p->m_FolderInfo->m_Subdirs += dirCount;
    }
}

void CItem::UpwardSubtractFolders(const ULONG dirCount)
{
    if (dirCount == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        ASSERT(p->m_FolderInfo->m_Subdirs - dirCount >= 0);
        if (p->IsType(IT_FILE)) continue;
        p->m_FolderInfo->m_Subdirs -= dirCount;
    }
}

void CItem::UpwardAddFiles(const ULONG fileCount)
{
    if (fileCount == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (p->IsType(IT_FILE)) continue;
        p->m_FolderInfo->m_Files += fileCount;
    }
}

void CItem::UpwardSubtractFiles(const ULONG fileCount)
{
    if (fileCount == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (p->IsType(IT_FILE)) continue;
        ASSERT(p->m_FolderInfo->m_Files - fileCount >= 0);
        p->m_FolderInfo->m_Files -= fileCount;
    }
}

void CItem::UpwardAddSizePhysical(const ULONGLONG bytes)
{
    if (bytes == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        p->m_SizePhysical += bytes;
    }
}

void CItem::UpwardSubtractSizePhysical(const ULONGLONG bytes)
{
    if (bytes == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        ASSERT(p->m_SizePhysical - bytes >= 0);
        p->m_SizePhysical -= bytes;
    }
}

void CItem::UpwardAddSizeLogical(const ULONGLONG bytes)
{
    if (bytes == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        p->m_SizeLogical += bytes;
    }
}

void CItem::UpwardSubtractSizeLogical(const ULONGLONG bytes)
{
    if (bytes == 0) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        ASSERT(p->m_SizeLogical - bytes >= 0);
        p->m_SizeLogical -= bytes;
    }
}

void CItem::UpwardAddReadJobs(const ULONG count)
{
    if (m_FolderInfo == nullptr || count == 0) return;
    if (m_FolderInfo->m_Jobs == 0) m_FolderInfo->m_Tstart = static_cast<ULONG>(GetTickCount64() / 1000ull);
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (p->IsType(IT_FILE)) continue;
        p->m_FolderInfo->m_Jobs += count;
    }
}

void CItem::UpwardSubtractReadJobs(const ULONG count)
{
    if (count == 0 || IsType(IT_FILE)) return;
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        ASSERT(p->m_FolderInfo->m_Jobs - count >= 0);
        p->m_FolderInfo->m_Jobs -= count;
        if (p->m_FolderInfo->m_Jobs == 0) p->SetDone();
    }
}

// This method increases the last change
void CItem::UpwardUpdateLastChange(const FILETIME& t)
{
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (CompareFileTime(&t, &p->m_LastChange) == 1) p->m_LastChange = t;
    }
}

void CItem::UpwardRecalcLastChange(const bool withoutItem)
{
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        p->UpdateStatsFromDisk();

        if (p->m_FolderInfo == nullptr) continue;
        for (const auto& child : p->GetChildren())
        {
            if (withoutItem && child == this) continue;
            if (CompareFileTime(&child->m_LastChange, &p->m_LastChange) == 1)
                p->m_LastChange = child->m_LastChange;
        }
    }
}

ULONGLONG CItem::GetSizePhysical() const
{
    return m_SizePhysical;
}

ULONGLONG CItem::GetSizeLogical() const
{
    return m_SizeLogical;
}

void CItem::SetSizePhysical(const ULONGLONG size)
{
    ASSERT(size >= 0);
    m_SizePhysical = size;
}

void CItem::SetSizeLogical(const ULONGLONG size)
{
    ASSERT(size >= 0);
    m_SizeLogical = size;
}

ULONG CItem::GetReadJobs() const
{
    if (m_FolderInfo == nullptr) return 0;
    return m_FolderInfo->m_Jobs;
}

FILETIME CItem::GetLastChange() const
{
    return m_LastChange;
}

void CItem::SetLastChange(const FILETIME& t)
{
    m_LastChange = t;
}

void CItem::SetAttributes(const DWORD attr)
{
    m_Attributes = attr;
}

// Decode the attributes encoded by SetAttributes()
DWORD CItem::GetAttributes() const
{
    return m_Attributes;
}

// Returns a value which resembles sorting of RHSACE considering gaps
unsigned short CItem::GetSortAttributes() const
{
    unsigned short ret = 0;

    // We want to enforce the order RHSACE with R being the highest priority
    // attribute and E being the lowest priority attribute.
    ret |= m_Attributes & FILE_ATTRIBUTE_READONLY    ? 1 << 6 : 0; // R
    ret |= m_Attributes & FILE_ATTRIBUTE_HIDDEN      ? 1 << 5 : 0; // H
    ret |= m_Attributes & FILE_ATTRIBUTE_SYSTEM      ? 1 << 4 : 0; // S
    ret |= m_Attributes & FILE_ATTRIBUTE_ARCHIVE     ? 1 << 3 : 0; // A
    ret |= m_Attributes & FILE_ATTRIBUTE_COMPRESSED  ? 1 << 2 : 0; // C
    ret |= m_Attributes & FILE_ATTRIBUTE_ENCRYPTED   ? 1 << 1 : 0; // E
    ret |= m_Attributes & FILE_ATTRIBUTE_SPARSE_FILE ? 1 << 0 : 0; // Z

    return ret;
}

double CItem::GetFraction() const
{
    if (!GetParent() || GetParent()->GetSizePhysical() == 0)
    {
        return 1.0;
    }
    return static_cast<double>(GetSizePhysical()) /
        static_cast<double>(GetParent()->GetSizePhysical());
}

bool CItem::IsRootItem() const
{
    return (m_Type & ITF_ROOTITEM) != 0;
}

std::wstring CItem::GetPath() const
{
    std::wstring path = UpwardGetPathWithoutBackslash();
    if (IsType(IT_DRIVE))
    {
        path += L"\\";
    }
    return path;
}

std::wstring CItem::GetPathLong() const
{
    return FileFindEnhanced::MakeLongPathCompatible(GetPath());
}

std::wstring CItem::GetOwner(const bool force) const
{
    if (!IsVisible() && !force)
    {
        return {};
    }

    // If visible, use cached variable
    std::wstring tmp;
    std::wstring & ret = (force) ? tmp : m_VisualInfo->owner;
    if (!ret.empty()) return ret;

    // Fetch owner information from drive
    SmartPointer<PSECURITY_DESCRIPTOR> ps(LocalFree);
    PSID sid = nullptr;
    GetNamedSecurityInfo(GetPathLong().c_str(), SE_FILE_OBJECT, OWNER_SECURITY_INFORMATION,
        &sid, nullptr, nullptr, nullptr, &ps);
    ret = GetNameFromSid(sid);
    return ret;
}

bool CItem::HasUncPath() const
{
    const std::wstring path = GetPath();
    return path.size() >= 2 && path.substr(0, 2) == L"\\\\";
}

// Returns the path for "Explorer here" or "Command Prompt here"
std::wstring CItem::GetFolderPath() const
{
    std::wstring path = GetPath();
    if (IsType(IT_FILE))
    {
        const auto i = path.find_last_of(wds::chrBackslash);
        ASSERT(i != std::wstring::npos);
        path = path.substr(0, i + 1);
    }

    return path;
}

std::wstring CItem::GetName() const
{
    return m_Name;
}

std::wstring CItem::GetExtension() const
{
    return m_Extension;
}

ULONG CItem::GetFilesCount() const
{
    if (m_FolderInfo == nullptr) return 0;
    return m_FolderInfo->m_Files;
}

ULONG CItem::GetFoldersCount() const
{
    if (m_FolderInfo == nullptr) return 0;
    return m_FolderInfo->m_Subdirs;
}

ULONGLONG CItem::GetItemsCount() const
{
    if (m_FolderInfo == nullptr) return 0;
    return static_cast<ULONGLONG>(m_FolderInfo->m_Files) + static_cast<ULONGLONG>(m_FolderInfo->m_Subdirs);
}

void CItem::SetDone()
{
    if (IsDone())
    {
        return;
    }

    if (IsType(IT_DRIVE))
    {
        UpdateFreeSpaceItem();
        UpdateUnknownItem();
    }

    // Sort and set finish time
    if (m_FolderInfo != nullptr)
    {
        SortItemsBySizePhysical();
        m_FolderInfo->m_Tfinish = static_cast<ULONG>(GetTickCount64() / 1000ull);
    }

    m_Rect = { 0,0,0,0 };
    SetType(ITF_DONE, true);
}

void CItem::SortItemsBySizePhysical() const
{
    if (m_FolderInfo == nullptr) return;
    
    // sort by size for proper treemap rendering
    std::lock_guard guard(m_FolderInfo->m_Protect);
    m_FolderInfo->m_Children.shrink_to_fit();
    std::ranges::sort(m_FolderInfo->m_Children, [](auto item1, auto item2)
    {
        return item1->GetSizePhysical() > item2->GetSizePhysical(); // biggest first
    });
}

ULONGLONG CItem::GetTicksWorked() const
{
    if (m_FolderInfo == nullptr) return 0;
    return m_FolderInfo->m_Tfinish > 0 ? (m_FolderInfo->m_Tfinish - m_FolderInfo->m_Tstart) :
        (m_FolderInfo->m_Tstart > 0) ? ((GetTickCount64() / 1000ull) - m_FolderInfo->m_Tstart) : 0;
}

void CItem::ResetScanStartTime() const
{
    if (m_FolderInfo != nullptr)
    {
        m_FolderInfo->m_Tfinish = 0;
        m_FolderInfo->m_Tstart = static_cast<ULONG>(GetTickCount64() / 1000ull);
    }
}

void CItem::ScanItemsFinalize(CItem* item)
{
    if (item == nullptr) return;
    std::stack<CItem*> queue({item});
    while (!queue.empty())
    {
        const auto & qitem = queue.top();
        queue.pop();
        qitem->SetDone();
        if (qitem->IsType(IT_FILE)) continue;
        for (const auto& child : qitem->GetChildren())
        {
            if (!child->IsDone()) queue.push(child);
        }
    }
}

void CItem::ScanItems(BlockingQueue<CItem*> * queue)
{
    while (CItem * item = queue->Pop())
    {
        // Mark the time we started evaluating this node
        item->ResetScanStartTime();

        if (item->IsType(IT_DRIVE | IT_DIRECTORY))
        {
            FileFindEnhanced finder;
            for (BOOL b = finder.FindFile(item->GetPath()); b; b = finder.FindNextFile())
            {
                if (finder.IsDots())
                {
                    continue;
                }

                if (finder.IsDirectory())
                {
                    if (COptions::ExcludeHiddenDirectory && finder.IsHidden() ||
                        COptions::ExcludeProtectedDirectory && finder.IsHiddenSystem())
                    {
                        continue;
                    }
  
                    // Exclude directories matching path filter
                    if (!COptions::FilteringExcludeDirsRegex.empty() && std::ranges::any_of(COptions::FilteringExcludeDirsRegex,
                        [&finder](const auto& pattern) { return std::regex_match(finder.GetFilePath(), pattern); }))
                    {
                        continue;
                    }

                    item->UpwardAddFolders(1);
                    if (CItem* newitem = item->AddDirectory(finder); newitem->GetReadJobs() > 0)
                    {
                        queue->Push(newitem);
                    }
                }
                else
                {
                    if (COptions::ExcludeHiddenFile && finder.IsHidden() ||
                        COptions::ExcludeProtectedFile && finder.IsHiddenSystem() ||
                        COptions::ExcludeSymbolicLinksFile && CReparsePoints::IsReparsePoint(finder.GetAttributes()) &&
                            CReparsePoints::IsSymbolicLink(finder.GetFilePathLong(), finder.GetAttributes()))
                    {
                        continue;
                    }

                    // Exclude files matching name filter
                    if (!COptions::FilteringExcludeFilesRegex.empty() && std::ranges::any_of(COptions::FilteringExcludeFilesRegex,
                        [&finder](const auto& pattern) { return std::regex_match(finder.GetFileName(), pattern); }))
                    {
                        continue;
                    }

                    // Exclude files matching size filter
                    if (COptions::FilteringSizeMinimumCalculated > 0 && finder.GetFileSizeLogical() < COptions::FilteringSizeMinimumCalculated)
                    {
                        continue;
                    }

                    item->UpwardAddFiles(1);
                    CItem* newitem = item->AddFile(finder);
                    CFileDupeControl::Get()->ProcessDuplicate(newitem, queue);
                    queue->WaitIfSuspended();
                }

                // Update pacman position
                item->UpwardDrivePacman();
            }
        }
        else if (item->IsType(IT_FILE))
        {
            // Only used for refreshes
            item->UpdateStatsFromDisk();
            item->SetDone();
        }
        else if (item->IsType(IT_MYCOMPUTER))
        {
            for (const auto & child : item->GetChildren())
            {
                child->UpwardAddReadJobs(1);
                queue->Push(child);
            }
        }
        item->UpwardSubtractReadJobs(1);
        item->UpwardDrivePacman();
    }
}

void CItem::UpwardSetDone()
{
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        p->SetDone();
    }
}

void CItem::UpwardSetUndone()
{
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (p->IsType(IT_DRIVE) && p->IsDone())
        {
            if (CItem* unknown = p->FindUnknownItem(); unknown != nullptr)
            {
                p->UpwardSubtractSizePhysical(unknown->GetSizePhysical());
                unknown->SetSizePhysical(0);
            }
        }

        p->SetType(ITF_DONE, false);
    }
}

CItem* CItem::FindRecyclerItem() const
{
    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (!p->IsType(IT_DRIVE)) continue;

        // There are no cross-platform way to consistently identify the recycle bin so attempt
        // to find an item with the most probable to least probable values
        for (const std::wstring& possible : { L"$RECYCLE.BIN", L"RECYCLER", L"RECYCLED" })
        {
            for (const auto& child : p->GetChildren())
            {
                if (child->IsType(IT_DIRECTORY) && _wcsicmp(child->GetName().c_str(), possible.c_str()) == 0)
                {
                    return child;
                }
            }
        }
    }

    return nullptr;
}

void CItem::CreateFreeSpaceItem()
{
    ASSERT(IsType(IT_DRIVE));

    UpwardSetUndone();

    auto [total, free] = CDirStatApp::GetFreeDiskSpace(GetPath());

    const auto freespace = new CItem(IT_FREESPACE, Localization::Lookup(IDS_FREESPACE_ITEM));
    freespace->SetSizePhysical(free);
    freespace->SetDone();

    AddChild(freespace);
}

CItem* CItem::FindFreeSpaceItem() const
{
    for (const auto& child : GetChildren())
    {
        if (child->IsType(IT_FREESPACE))
        {
            return child;
        }
    }
    return nullptr;
}

void CItem::UpdateFreeSpaceItem() const
{
    ASSERT(IsType(IT_DRIVE));

    CItem* freeSpaceItem = FindFreeSpaceItem();
    if (freeSpaceItem == nullptr)
    {
        return;
    }

    // Rebaseline as if freespace was not shown
    freeSpaceItem->UpwardSubtractSizePhysical(freeSpaceItem->GetSizePhysical());

    auto [total, free] = CDirStatApp::GetFreeDiskSpace(GetPath());
    freeSpaceItem->UpwardAddSizePhysical(free);
}

void CItem::UpdateUnknownItem() const
{
    ASSERT(IsType(IT_DRIVE));

    CItem* unknown = FindUnknownItem();
    if (unknown == nullptr)
    {
        return;
    }

    // Rebaseline as if unknown size was not shown
    unknown->UpwardSubtractSizePhysical(unknown->GetSizePhysical());

    // Get the tallied size, account for whether the freespace item is part of it
    const CItem* freeSpaceItem = FindFreeSpaceItem();
    const ULONGLONG tallied = GetSizePhysical() - (freeSpaceItem ? freeSpaceItem->GetSizePhysical() : 0);

    auto [total, free] = CDirStatApp::GetFreeDiskSpace(GetPath());
    unknown->UpwardAddSizePhysical((tallied > total - free) ? 0 : total - free - tallied);
}

void CItem::RemoveFreeSpaceItem()
{
    ASSERT(IsType(IT_DRIVE));

    if (const auto freespace = FindFreeSpaceItem(); freespace != nullptr)
    {
        UpwardSetUndone();
        UpwardSubtractSizePhysical(freespace->GetSizePhysical());
        RemoveChild(freespace);
    }
}

void CItem::CreateUnknownItem()
{
    ASSERT(IsType(IT_DRIVE));

    UpwardSetUndone();

    const auto unknown = new CItem(IT_UNKNOWN, Localization::Lookup(IDS_UNKNOWN_ITEM));
    unknown->SetDone();

    AddChild(unknown);
}

CItem* CItem::FindUnknownItem() const
{
    for (const auto& child : GetChildren())
    {
        if (child->IsType(IT_UNKNOWN))
        {
            return child;
        }
    }
    return nullptr;
}

void CItem::RemoveUnknownItem()
{
    ASSERT(IsType(IT_DRIVE));

    if (const auto unknown = FindUnknownItem(); unknown != nullptr)
    {
        UpwardSetUndone();
        UpwardSubtractSizePhysical(unknown->GetSizePhysical());
        RemoveChild(unknown);
    }
} 

void CItem::CollectExtensionData(CExtensionData* ed) const
{
    std::stack<const CItem*> queue({this});
    while (!queue.empty())
    {
        const auto& qitem = queue.top();
        queue.pop();
        if (qitem->IsType(IT_FILE))
        {
            const std::wstring& ext = qitem->GetExtension();
            const auto & record = ed->find(ext);
            if (record != ed->end())
            {
                record->second.bytes += qitem->GetSizePhysical();
                record->second.files++;
            }
            else
            {
                SExtensionRecord new_record;
                new_record.bytes = qitem->GetSizePhysical();
                new_record.files = 1;
                ed->emplace(ext, new_record);
            }
        }
        else for (const auto& child : qitem->m_FolderInfo->m_Children)
        {
            queue.push(child);
        }
    }
}

ULONGLONG CItem::GetProgressRangeMyComputer() const
{
    ASSERT(IsType(IT_MYCOMPUTER));

    ULONGLONG range = 0;
    for (const auto& child : GetChildren())
    {
        range += child->GetProgressRangeDrive();
    }
    return range;
}

ULONGLONG CItem::GetProgressRangeDrive() const
{
    auto [total, free] = CDirStatApp::GetFreeDiskSpace(GetPath());

    total -= free;

    ASSERT(total >= 0);
    return total;
}

COLORREF CItem::GetGraphColor() const
{
    if (IsType(IT_UNKNOWN))
    {
        return RGB(255, 255, 0) | CTreeMap::COLORFLAG_LIGHTER;
    }

    if (IsType(IT_FREESPACE))
    {
        return RGB(100, 100, 100) | CTreeMap::COLORFLAG_DARKER;
    }

    if (IsType(IT_FILE))
    {
        return CDirStatDoc::GetDocument()->GetCushionColor(GetExtension());
    }

    return RGB(0, 0, 0);
 }

bool CItem::MustShowReadJobs() const
{
    if (GetParent() != nullptr)
    {
        return !GetParent()->IsDone();
    }

    return !IsDone();
}

COLORREF CItem::GetPercentageColor() const
{
    const int i = GetIndent() % COptions::FileTreeColorCount;
    return std::vector<COLORREF>
    {
        COptions::FileTreeColor0,
        COptions::FileTreeColor1,
        COptions::FileTreeColor2,
        COptions::FileTreeColor3,
        COptions::FileTreeColor4,
        COptions::FileTreeColor5,
        COptions::FileTreeColor6,
        COptions::FileTreeColor7
    } [i] ;
}

std::wstring CItem::UpwardGetPathWithoutBackslash() const
{
    // allow persistent storage to prevent constant reallocation
    std::wstring path = L"\\";

    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (p->IsType(IT_DIRECTORY))
        {
            path.insert(0, p->m_Name + L"\\");
        }
        else if (p->IsType(IT_FILE))
        {
            path = p->m_Name;
        }
        else if (p->IsType(IT_DRIVE))
        {
            path.insert(0, p->m_Name.substr(0, 2) + L"\\");
        }
    }

    while (!path.empty() && path.back() == L'\\') path.pop_back();
    return path;
}

CItem* CItem::AddDirectory(const FileFindEnhanced& finder)
{
    const bool follow = !finder.IsProtectedReparsePoint() &&
        CDirStatApp::Get()->IsFollowingAllowed(finder.GetFilePathLong(), finder.GetAttributes());

    const auto & child = new CItem(IT_DIRECTORY, finder.GetFileName());
    child->SetLastChange(finder.GetLastWriteTime());
    child->SetAttributes(finder.GetAttributes());
    AddChild(child);
    child->UpwardAddReadJobs(follow ? 1 : 0);
    return child;
}

CItem* CItem::AddFile(const FileFindEnhanced& finder)
{
    const auto & child = new CItem(IT_FILE, finder.GetFileName());
    child->SetSizePhysical(finder.GetFileSizePhysical());
    child->SetSizeLogical(finder.GetFileSizeLogical());
    child->SetLastChange(finder.GetLastWriteTime());
    child->SetAttributes(finder.GetAttributes());
    AddChild(child);
    child->SetDone();
    return child;
}

void CItem::UpwardDrivePacman()
{
    if (!COptions::PacmanAnimation)
    {
        return;
    }

    for (auto p = this; p != nullptr; p = p->GetParent())
    {
        if (p->IsType(IT_FILE) || !p->IsVisible()) continue;
        if (p->GetReadJobs() == 0) p->StopPacman();
        else p->DrivePacman();
    }
}

std::wstring CItem::GetFileHash(ULONGLONG hashSizeLimit, BlockingQueue<CItem*>* queue)
{
    // Initialize hash for this thread
    thread_local std::vector<BYTE> FileBuffer(1024ull * 1024ull);
    thread_local std::vector<BYTE> Hash;
    thread_local SmartPointer<BCRYPT_HASH_HANDLE> HashHandle(BCryptDestroyHash);
    thread_local DWORD HashLength = 0;

    if (HashLength == 0)
    {
        BCRYPT_ALG_HANDLE AlgHandle = nullptr;
        DWORD ResultLength = 0;
        if (BCryptOpenAlgorithmProvider(&AlgHandle, BCRYPT_SHA512_ALGORITHM, MS_PRIMITIVE_PROVIDER, BCRYPT_HASH_REUSABLE_FLAG) != 0 ||
            BCryptGetProperty(AlgHandle, BCRYPT_HASH_LENGTH, reinterpret_cast<PBYTE>(&HashLength), sizeof(HashLength), &ResultLength, 0) != 0 ||
            BCryptCreateHash(AlgHandle, &HashHandle, nullptr, 0, nullptr, 0, BCRYPT_HASH_REUSABLE_FLAG) != 0)
        {
            return {};
        }
    }

    // Open file for reading
    SmartPointer<HANDLE> hFile(CloseHandle, CreateFile(GetPathLong().c_str(), GENERIC_READ,
        FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE, nullptr, OPEN_EXISTING,
        FILE_FLAG_BACKUP_SEMANTICS | FILE_FLAG_SEQUENTIAL_SCAN, nullptr));
    if (hFile == INVALID_HANDLE_VALUE)
    {
        return {};
    }
    
    // Hash data one read at a time
    DWORD iReadResult = 0;
    DWORD iHashResult = 0;
    DWORD iReadBytes = 0;
    Hash.resize(HashLength);
    while ((iReadResult = ReadFile(hFile, FileBuffer.data(), static_cast<DWORD>(
        hashSizeLimit > 0 ? min(hashSizeLimit, FileBuffer.size()) : FileBuffer.size()),
        &iReadBytes, nullptr)) != 0 && iReadBytes > 0)
    {
        UpwardDrivePacman();
        iHashResult = BCryptHashData(HashHandle, FileBuffer.data(), iReadBytes, 0);
        if (iHashResult != 0 || hashSizeLimit > 0) break;
        queue->WaitIfSuspended();
    }

    // Complete hash data
    if (iHashResult != 0 || iReadResult == 0 ||
        BCryptFinishHash(HashHandle, Hash.data(), HashLength, 0) != 0)
    {
        return {};
    }

    // Convert to hex string
    std::wstring sHash;
    sHash.resize(2ull * HashLength);
    DWORD iHashStringLength = static_cast<DWORD>(sHash.size() + 1ull);
    CryptBinaryToStringW(Hash.data(), HashLength, CRYPT_STRING_HEXRAW | CRYPT_STRING_NOCRLF,
        sHash.data(), &iHashStringLength);
    return sHash;
}
