'use strict'

const casing = require('../../../lib/utils/casing')
const assert = require('assert')

describe('getConverter()', () => {
  it('should convert string to camelCase', () => {
    const converter = casing.getConverter('camelCase')

    assert.equal(converter('foo'), 'foo')
    assert.equal(converter('fooBar'), 'fooBar')
    assert.equal(converter('foo-bar'), 'fooBar')
    assert.equal(converter('foo_bar'), 'fooBar')
    assert.equal(converter('FooBar'), 'fooBar')
    assert.equal(converter('Foo1Bar'), 'foo1Bar')
    assert.equal(converter('FooBAR'), 'fooBAR')
    assert.equal(converter('Foo1BAZ'), 'foo1BAZ')
    assert.equal(converter('foo1b_a_z'), 'foo1bAZ')
    assert.equal(converter('darИībaÊÊw'), 'darИībaÊÊw')
    assert.equal(converter('klâwen-ûf'), 'klâwen-ûf')
    assert.equal(converter('пустынныхИвдалП'), 'пустынныхИвдалП')
    assert.equal(converter('kpłĄżć'), 'kpłĄżć')
    assert.equal(converter('ÊtreSîne'), 'êtreSîne')
    assert.equal(converter('    foo       Bar   '), '    foo       Bar   ')
  })

  it('should convert string to PascalCase', () => {
    const converter = casing.getConverter('PascalCase')

    assert.equal(converter('foo'), 'Foo')
    assert.equal(converter('fooBar'), 'FooBar')
    assert.equal(converter('foo-bar'), 'FooBar')
    assert.equal(converter('foo_bar'), 'FooBar')
    assert.equal(converter('FooBar'), 'FooBar')
    assert.equal(converter('Foo1Bar'), 'Foo1Bar')
    assert.equal(converter('FooBAR'), 'FooBAR')
    assert.equal(converter('Foo1BAZ'), 'Foo1BAZ')
    assert.equal(converter('foo1b_a_z'), 'Foo1bAZ')
    assert.equal(converter('darИībaÊÊw'), 'DarИībaÊÊw')
    assert.equal(converter('klâwen-ûf'), 'Klâwen-ûf')
    assert.equal(converter('пустынныхИвдалП'), 'ПустынныхИвдалП')
    assert.equal(converter('kpłĄżć'), 'KpłĄżć')
    assert.equal(converter('ÊtreSîne'), 'ÊtreSîne')
    assert.equal(converter('    foo       Bar   '), '    foo       Bar   ')
  })

  it('should convert string to kebab-case', () => {
    const converter = casing.getConverter('kebab-case')

    assert.equal(converter('foo'), 'foo')
    assert.equal(converter('fooBar'), 'foo-bar')
    assert.equal(converter('foo-bar'), 'foo-bar')
    assert.equal(converter('foo_bar'), 'foo-bar')
    assert.equal(converter('FooBar'), 'foo-bar')
    assert.equal(converter('Foo1Bar'), 'foo1-bar')
    assert.equal(converter('FooBAR'), 'foo-b-a-r')
    assert.equal(converter('Foo1BAZ'), 'foo1-b-a-z')
    assert.equal(converter('foo1b_a_z'), 'foo1b-a-z')
    assert.equal(converter('darИībaÊÊw'), 'darиībaêêw')
    assert.equal(converter('klâwen-ûf'), 'klâwen-ûf')
    assert.equal(converter('пустынныхИвдалП'), 'пустынныхивдалп')
    assert.equal(converter('kpłĄżć'), 'kpłążć')
    assert.equal(converter('ÊtreSîne'), 'être-sîne')
    assert.equal(converter('    foo       Bar   '), '    foo       bar   ')
  })

  it('should convert string to snake_case', () => {
    const converter = casing.getConverter('snake_case')

    assert.equal(converter('a'), 'a')
    assert.equal(converter('fooBar'), 'foo_bar')
    assert.equal(converter('foo-bar'), 'foo_bar')
    assert.equal(converter('FooBar'), 'foo_bar')
    assert.equal(converter('Foo1Bar'), 'foo1_bar')
    assert.equal(converter('FooBAR'), 'foo_b_a_r')
    assert.equal(converter('Foo1BAZ'), 'foo1_b_a_z')
    assert.equal(converter('foo1b_a_z'), 'foo1b_a_z')
    assert.equal(converter('darИībaÊÊw'), 'darиībaêêw')
    assert.equal(converter('klâwen-ûf'), 'klâwen_ûf')
    assert.equal(converter('пустынныхИвдалП'), 'пустынныхивдалп')
    assert.equal(converter('kpłĄżć'), 'kpłążć')
    assert.equal(converter('ÊtreSîne'), 'être_sîne')
    assert.equal(converter('    foo       Bar   '), '    foo       bar   ')
  })
})

describe('getChecker()', () => {
  it('should check string to camelCase', () => {
    const checker = casing.getChecker('camelCase')

    assert.equal(checker('foo'), true)
    assert.equal(checker('fooBar'), true)
    assert.equal(checker('fooBar'), true)
    assert.equal(checker('fooBar'), true)
    assert.equal(checker('fooBar'), true)
    assert.equal(checker('foo1Bar'), true)
    assert.equal(checker('fooBAR'), true)
    assert.equal(checker('foo1BAZ'), true)
    assert.equal(checker('foo1bAZ'), true)
    assert.equal(checker('darИībaÊÊw'), true)
    assert.equal(checker('klâwen-ûf'), false)
    assert.equal(checker('пустынныхИвдалП'), true)
    assert.equal(checker('kpłĄżć'), true)
    assert.equal(checker('êtreSîne'), true)
    assert.equal(checker('    foo       Bar   '), false)

    assert.equal(checker('camelCase'), true)
    assert.equal(checker('PascalCase'), false)
    assert.equal(checker('kebab-case'), false)
    assert.equal(checker('snake_case'), false)

    assert.equal(checker('camel-Kebab-Case'), false)
    assert.equal(checker('camel_Snake_Case'), false)
    assert.equal(checker('Pascal-Kebab-Case'), false)
    assert.equal(checker('Pascal_Snake_Case'), false)
    assert.equal(checker('snake_kebab-case'), false)
  })

  it('should check string to PascalCase', () => {
    const checker = casing.getChecker('PascalCase')

    assert.equal(checker('Foo'), true)
    assert.equal(checker('FooBar'), true)
    assert.equal(checker('FooBar'), true)
    assert.equal(checker('FooBar'), true)
    assert.equal(checker('FooBar'), true)
    assert.equal(checker('Foo1Bar'), true)
    assert.equal(checker('FooBAR'), true)
    assert.equal(checker('Foo1BAZ'), true)
    assert.equal(checker('Foo1bAZ'), true)
    assert.equal(checker('DarИībaÊÊw'), true)
    assert.equal(checker('Klâwen-ûf'), false)
    assert.equal(checker('ПустынныхИвдалП'), true)
    assert.equal(checker('KpłĄżć'), true)
    assert.equal(checker('ÊtreSîne'), true)
    assert.equal(checker('    foo       Bar   '), false)

    assert.equal(checker('DarbībaShow'), true)

    assert.equal(checker('camelCase'), false)
    assert.equal(checker('PascalCase'), true)
    assert.equal(checker('kebab-case'), false)
    assert.equal(checker('snake_case'), false)

    assert.equal(checker('camel-Kebab-Case'), false)
    assert.equal(checker('camel_Snake_Case'), false)
    assert.equal(checker('Pascal-Kebab-Case'), false)
    assert.equal(checker('Pascal_Snake_Case'), false)
    assert.equal(checker('snake_kebab-case'), false)
  })

  it('should convert string to kebab-case', () => {
    const checker = casing.getChecker('kebab-case')

    assert.equal(checker('foo'), true)
    assert.equal(checker('foo-bar'), true)
    assert.equal(checker('foo-bar'), true)
    assert.equal(checker('foo-bar'), true)
    assert.equal(checker('foo-bar'), true)
    assert.equal(checker('foo1-bar'), true)
    assert.equal(checker('foo-b-a-r'), true)
    assert.equal(checker('foo1-b-a-z'), true)
    assert.equal(checker('foo1b-a-z'), true)
    assert.equal(checker('darиībaêêw'), true)
    assert.equal(checker('klâwen-ûf'), true)
    assert.equal(checker('пустынныхивдалп'), true)
    assert.equal(checker('kpłążć'), true)
    assert.equal(checker('être-sîne'), true)
    assert.equal(checker('    foo       bar   '), false)

    assert.equal(checker('camelCase'), false)
    assert.equal(checker('PascalCase'), false)
    assert.equal(checker('kebab-case'), true)
    assert.equal(checker('snake_case'), false)

    assert.equal(checker('camel-Kebab-Case'), false)
    assert.equal(checker('camel_Snake_Case'), false)
    assert.equal(checker('Pascal-Kebab-Case'), false)
    assert.equal(checker('Pascal_Snake_Case'), false)
    assert.equal(checker('snake_kebab-case'), false)

    assert.equal(checker('valid-kebab-case'), true)
    assert.equal(checker('-invalid-kebab-case'), false)
    assert.equal(checker('invalid--kebab-case'), false)
  })

  it('should check string to snake_case', () => {
    const checker = casing.getChecker('snake_case')

    assert.equal(checker('a'), true)
    assert.equal(checker('foo_bar'), true)
    assert.equal(checker('foo_bar'), true)
    assert.equal(checker('foo_bar'), true)
    assert.equal(checker('foo1_bar'), true)
    assert.equal(checker('foo_b_a_r'), true)
    assert.equal(checker('foo1_b_a_z'), true)
    assert.equal(checker('foo1b_a_z'), true)
    assert.equal(checker('darиībaêêw'), true)
    assert.equal(checker('klâwen_ûf'), true)
    assert.equal(checker('пустынныхивдалп'), true)
    assert.equal(checker('kpłążć'), true)
    assert.equal(checker('être_sîne'), true)
    assert.equal(checker('    foo       bar   '), false)

    assert.equal(checker('camelCase'), false)
    assert.equal(checker('PascalCase'), false)
    assert.equal(checker('kebab-case'), false)
    assert.equal(checker('snake_case'), true)

    assert.equal(checker('camel-Kebab-Case'), false)
    assert.equal(checker('camel_Snake_Case'), false)
    assert.equal(checker('Pascal-Kebab-Case'), false)
    assert.equal(checker('Pascal_Snake_Case'), false)
    assert.equal(checker('snake_kebab-case'), false)

    assert.equal(checker('_valid_snake_case'), true)
    assert.equal(checker('invalid__snake_case'), false)
  })
})
