//
// AcknowLocalization.swift
//
// Copyright (c) 2015-2024 Vincent Tourraine (https://www.vtourraine.net)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import Foundation

/// Manages the localization for the main acknowledgements labels and strings.
open class AcknowLocalization {

    /**
     The localized version of “Acknowledgements”. You can use this value for the button presenting the `AcknowListViewController`, for instance.
     - Returns: The localized title.
     */
    open class func localizedTitle() -> String {
        return localizedString(forKey: "VTAckAcknowledgements", defaultString: "Acknowledgements")
    }

    /**
     Returns the resources bundle (for localizations).
     - Returns: The resources bundle.
     */
    class func resourcesBundle() -> Bundle? {
#if SWIFT_PACKAGE
        // Preprocessor definition documented here:
        // https://github.com/apple/swift-package-manager/blob/main/Documentation/Usage.md#packaging-legacy-code
        return Bundle.module
#else
        let rootBundle = Bundle(for: AcknowLocalization.self)

        let CocoaPodsBundleName = "AcknowListBundle"
        if let bundleURL = rootBundle.url(forResource: CocoaPodsBundleName, withExtension: "bundle") {
            return Bundle(url: bundleURL)
        }
        else {
            return rootBundle
        }
#endif
    }

    /**
     Returns a localized string.
     - Parameters:
       - key: The key for the string to localize.
       - defaultString: The default non-localized string.
     - Returns: The localized string.
     */
    class func localizedString(forKey key: String, defaultString: String) -> String {
        if let bundle = resourcesBundle() {
            return bundle.localizedString(forKey: key, value: defaultString, table: nil)
        }
        else {
            return defaultString
        }
    }

    /**
     Returns the URL for the CocoaPods main website.
     - Returns: The CocoaPods website URL.
     */
    class func CocoaPodsURLString() -> String {
        return "https://cocoapods.org"
    }

    /**
     Returns the default localized footer text.
     - Returns: The localized footer text.
     */
    class func localizedCocoaPodsFooterText() -> String {
        return localizedString(forKey: "VTAckGeneratedByCocoaPods", defaultString: "Generated by CocoaPods") + "\n" + self.CocoaPodsURLString()
    }

    /**
     Returns an array of `Acknow`, sorted using localized comparison.
     - Parameter acknowledgements: An array of `Acknow` to sort.
     - Returns: A sorted array of `Acknow`.
     */
    open class func sorted(_ acknowledgements: [Acknow]) -> [Acknow] {
        return acknowledgements.sorted { ack1, ack2 in
            let result = ack1.title.localizedCompare(ack2.title)
            return (result == ComparisonResult.orderedAscending)
        }
    }
}
