This document explains how to build, test, and install the SDK.

## Building the Code

To get started, clone the code from the repository and also download dependent libraries by running

```
git clone https://github.com/utilForever/RosettaStone.git
cd RosettaStone.git
git submodule init
git submodule update
```

Or, you can simply clone the clode by adding --recursive option.
```
git clone https://github.com/utilForever/RosettaStone.git --recursive
```

To build the code, a compiler that supports C++17 is required. Platform-specific build instructions are described below.

### Building from macOS

RosettaStone supports OS X 10.14 Mojave or higher. Also, Xcode 10.2 or higher and the command line tools are required for building RosettaStone. Once ready, install [Homebrew](http://brew.sh) and run the following command line to setup [CMake](https://cmake.org/):

```
brew install cmake
```

Once CMake and Python is installed, build the code by running

```
mkdir build
cd build
cmake ..
make
```

> Of course, use `make -j <num_threads>` flag to boost up the build performance by using multithreads.

This will build entire codebase. To run the unit test, execute

```
bin/UnitTests
```

It should show all the tests are passing.

### Building from Ubuntu and Windows Subsystem for Linux (WSL)

RosettaStone supports Ubuntu 18.04 or higher. Using `apt-get`, install required tools and libraries by running,

```
sudo apt-get install build-essential python3-dev python3-pip cmake
```

This will install GNU compilers and CMake. Once installed, build the code by running

```
mkdir build
cd build
cmake ..
make
```

> Again, use `make -j <num_threads>` flag to boost up the build performance by using multithreads.

This will build entire codebase. To run the unit test, execute

```
bin/UnitTests
```

It should show all the tests are passing.

### Building from Windows

To build the code on Windows, CMake and Visual Studio 2017 (or higher) is required. Windows' version of CMake is available from [this website](https://cmake.org/). To install Visual Studio, the community edition of the tool can be downloaded from [Visual Studio Community 2017](https://www.Visualstudio.com/en-us/products/Visual-studio-community-vs.aspx). You can also use Visual Studio 2019.

Once everything is installed, run the following commands:

```
md build
cd build
cmake .. -G"Visual Studio 15 2017 Win64"
```

This will generate 64-bit version of VS 2017 solution and projects. Once executed, you can find `RosettaStone.sln` solution file in the `build` directory. Open the solution file and hit `Ctrl + Shift + B` to build the entire solution. Set `UnitTests` as a start-up project and hit `Ctrl + F5` to run the test.

Alternatively, you can use MSBuild to build the solution from the command prompt. In such case, simply run:

```
MSBuild RosettaStone.sln /p:Configuration=Release
```

This will build the whole solution in release mode. Once built, run the following command to execute unit tests:

```
bin\Release\UnitTests.exe
```

### Running Tests

There are two different tests in the codebase including the unit test and manual test. For the detailed instruction on how to run those tests, please checkout the documentation page from [the project website](https://utilforever.github.io/RosettaStone/Documentation/).

### Code Coverage

RosettaStone uses `lcov` for the code coverage. For macOS and Ubuntu platforms, the code coverage report can be generated by running

```
mkdir build
cd build
cmake .. -DCMAKE_BUILD_TYPE=Debug -DBUILD_COVERAGE=ON
make -j 8
lcov -c -i -d Tests/UnitTests -o base.info
bin/UnitTests
lcov -c -d Tests/UnitTests -o test.info
lcov -a base.info -a test.info -o coverage.info
lcov -r coverage.info '/usr/*' -o coverage.info
lcov -r coverage.info '*/Libraries/*' -o coverage.info
lcov -r coverage.info '*/Programs/*' -o coverage.info
lcov -r coverage.info '*/Tests/*' -o coverage.info
lcov -r coverage.info '*/Tools/*' -o coverage.info
lcov -l coverage.info
genhtml coverage.info -o out
```

This will exports the code coverage report ```index.html``` under `out` folder.

### Installing C++ SDK

For macOS, Ubuntu and WSL platforms, the library can be installed by running

```
cmake .. -DCMAKE_INSTALL_PREFIX=_INSTALL_PATH_
make
make install
```

This will install the header files and the static library `libRosettaStone.a` under `_INSTALL_PATH_`.

For Windows, run:

```
cmake .. -G"Visual Studio 15 2017 Win64" -DCMAKE_INSTALL_PREFIX=_INSTALL_PATH_
```

Then, build `INSTALL` project under `RosettaStone.sln`. This will install the header files and the static library `RosettaStone.lib` under `_INSTALL_PATH_`.

### Installing Python SDK

To install the Python SDK, `pyRosetta`, run the following command from the project root directory (where `setup.py` lives):

```
pip install -U .
```

> You can also use `virtualenv` to isolate the SDK installation. Check out [the virtualenv documentation](https://virtualenv.pypa.io/en/stable/) for more details.

To run the test/example scripts, install other Python dependencies as follows:

```
pip install -r requirements.txt
```

Once installed, try running the unit test to see if the module is installed correctly:

```
pytest ./Tests/PythonTests
```

The tests should pass.

### Using Docker

You can also use pre-built docker image by pulling the latest version from Docker Hub:

```
docker pull utilforever/rosettastone
```

Run a container and see if it can import `pyRosetta` module and the unit test passes:

```
docker run -it utilforever/rosettastone
python import -c "pyRosetta"

docker run utilforever/rosettastone /app/build/bin/UnitTests
```

You can also build the image from the source as well. From the root directory of this codebase, run:

```
docker build -t utilforever/rosettastone .
```

### Coding Style

RosettaStone uses clang-format. Checkout [`.clang-format`](./../.clang-format) file for the style guideline.

### Continuous Integration

The build quality is tracked by [Travis CI](https://travis-ci.org/utilForever/RosettaStone) for Linux and Mac. For Windows, [AppVeyor](https://ci.appveyor.com/project/utilForever/RosettaStone) is used. Recently, we added [Azure Pipelines](https://utilforever.visualstudio.com/RosettaStone/_build) for Linux, Mac and Windows. Any pull requests must pass all the builds.

### Code Quality

The code quality is tracked by [Codacy](https://app.codacy.com/project/utilForever/RosettaStone/dashboard) and [CodeFactor](https://www.codefactor.io/repository/github/utilforever/RosettaStone). Any pull requests must pass all the code quality tools.