/*
 * Copyright (C) 2022 LEIDOS.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

#include <gtest/gtest.h>
#include <iostream>
#include <autoware_lanelet2_ros2_interface/utility/message_conversion.hpp>
#include <../src/WMListenerWorker.hpp>
#include <carma_wm/CARMAWorldModel.hpp>
#include <lanelet2_core/geometry/LineString.h>
#include <lanelet2_traffic_rules/TrafficRulesFactory.h>
#include <lanelet2_core/Attribute.h>
#include <boost/archive/binary_oarchive.hpp>
#include <sstream>
#include <string>
#include "TestHelpers.hpp"
#include <rclcpp/rclcpp.hpp>
#include <carma_wm/MapConformer.hpp>
#include <lanelet2_io/Io.h>
#include <lanelet2_io/io_handlers/Factory.h>
#include <lanelet2_io/io_handlers/Writer.h>
#include <lanelet2_extension/traffic_rules/CarmaUSTrafficRules.h>
#include <autoware_lanelet2_ros2_interface/utility/query.hpp>
#include <autoware_lanelet2_ros2_interface/utility/utilities.hpp>
#include <lanelet2_extension/projection/local_frame_projector.h>
#include <lanelet2_extension/io/autoware_osm_parser.h>

namespace carma_wm
{
// Remove DISABLED and update file directory to test segfault.
TEST(SegFaultTest, DISABLED_updateSegFault) {
  ///////////
  // UNIT TEST ARGUMENTS
  ///////////

  // File to process. Path is relative to test folder
  std::string file = "/workspaces/carma_ws/carma/src/carma-platform/carma_wm/test/resource/tfhrc.osm";

  ///////////
  // START OF LOGIC
  ///////////

  // Write new map to file
  int projector_type = 0;
  std::string target_frame;
  lanelet::ErrorMessages load_errors;
  // Parse geo reference info from the original lanelet map (.osm)
  lanelet::io_handlers::AutowareOsmParser::parseMapParams(file, &projector_type, &target_frame);

  lanelet::projection::LocalFrameProjector local_projector(target_frame.c_str());

  lanelet::LaneletMapPtr map = lanelet::load(file, local_projector, &load_errors);

  if (map->laneletLayer.size() == 0)
  {
    FAIL() << "Input map does not contain any lanelets";
  }

  lanelet::Velocity con_lim = 80_mph;

  lanelet::MapConformer::ensureCompliance(map, con_lim);

  WMListenerWorker wmlw;
  autoware_lanelet2_msgs::msg::MapBin msg;
  lanelet::utils::conversion::toBinMsg(map, &msg);

  ///// Test map not set
  ASSERT_FALSE((bool)(wmlw.getWorldModel()->getMap()));

  ///// Test Map callback without user callback
  wmlw.mapCallback(std::make_unique<autoware_lanelet2_msgs::msg::MapBin>(msg));
  ASSERT_TRUE((bool)(wmlw.getWorldModel()->getMap()));

  autoware_lanelet2_msgs::msg::MapBin update_msg;
  update_msg.data = {22, 0, 0, 0, 0, 0, 0, 0, 115, 101, 114, 105, 97, 108, 105, 122, 97, 116, 105, 111, 110, 58, 58, 97, 114, 99, 104, 105, 118, 101, 12, 0, 4, 8, 4, 8, 1, 0, 0, 0, 0, 0, 0, 0, 0, 36, 0, 0, 0, 0, 0, 0, 0, 52, 53, 101, 48, 99, 55, 51, 55, 45, 53, 99, 53, 101, 45, 52, 98, 99, 98, 45, 57, 56, 98, 54, 45, 48, 99, 48, 101, 49, 102, 98, 49, 100, 49, 56, 56, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 12, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 19, 12, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 4, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 19, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0, 108, 105, 109, 105, 116, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0, 0, 50, 53, 32, 109, 112, 104, 19, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 19, 0, 0, 0, 0, 0, 0, 0, 100, 105, 103, 105, 116, 97, 108, 95, 115, 112, 101, 101, 100, 95, 108, 105, 109, 105, 116, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 18, 0, 0, 0, 0, 0, 0, 0, 114, 101, 103, 117, 108, 97, 116, 111, 114, 121, 95, 101, 108, 101, 109, 101, 110, 116, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0, 0, 114, 101, 102, 101, 114, 115, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 15, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 18, 12, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 108, 111, 99, 97, 116, 105, 111, 110, 5, 0, 0, 0, 0, 0, 0, 0, 117, 114, 98, 97, 110, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 4, 0, 0, 0, 0, 0, 0, 0, 114, 111, 97, 100, 14, 0, 0, 0, 0, 0, 0, 0, 116, 117, 114, 110, 95, 100, 105, 114, 101, 99, 116, 105, 111, 110, 8, 0, 0, 0, 0, 0, 0, 0, 115, 116, 114, 97, 105, 103, 104, 116, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 7, 0, 0, 0, 0, 0, 0, 0, 108, 97, 110, 101, 108, 101, 116, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 18, 0, 1, 0, 0, 0, 0, 2, 0, 0, 0, 16, 12, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 5, 0, 0, 0, 0, 0, 0, 0, 115, 111, 108, 105, 100, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 9, 0, 0, 0, 0, 0, 0, 0, 108, 105, 110, 101, 95, 116, 104, 105, 110, 0, 0, 0, 0, 0, -87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 22, 0, 1, 0, 0, 0, 0, 3, 0, 0, 0, -66, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97, -69, -102, -91, 19, -56, 99, -64, -39, 119, -108, 51, 45, 45, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 4, 0, 0, 0, -65, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -61, -98, -49, -116, 101, -53, 99, -64, 125, 76, 112, -1, -23, 40, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 5, 0, 0, 0, -64, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 36, 3, 116, -73, -50, 99, -64, 33, 33, 76, -53, -90, 36, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 6, 0, 0, 0, -63, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -2, 64, 56, 91, 9, -46, 99, -64, 27, 14, 40, -105, 99, 32, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 7, 0, 0, 0, -62, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -45, -1, 107, 66, 91, -43, 99, -64, -65, -30, 3, 99, 32, 28, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 8, 0, 0, 0, -61, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -53, -66, -96, 41, -83, -40, 99, -64, 15, -97, -33, 46, -35, 23, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 9, 0, 0, 0, -60, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -62, 78, -43, 16, -1, -37, 99, -64, -75, 42, -69, -6, -103, 19, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 10, 0, 0, 0, -59, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -72, -81, 9, -8, 80, -33, 99, -64, 3, 48, -105, -58, 86, 15, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 11, 0, 0, 0, -58, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -80, -31, 61, -33, -94, -30, 99, -64, -4, 28, 115, -110, 19, 11, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 12, 0, 0, 0, -57, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 55, 124, 114, -58, -12, -27, 99, -64, 76, -39, 78, 94, -48, 6, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 13, 0, 0, 0, -56, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -67, -25, -90, -83, 70, -23, 99, -64, 69, -58, 42, 42, -115, 2, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 14, 0, 0, 0, -55, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -44, -69, -37, -108, -104, -20, 99, -64, -45, 53, 13, -20, -109, -4, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 15, 0, 0, 0, -54, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 63, -1, -88, 43, -81, -17, 99, -64, -102, -29, -68, -46, -91, -12, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 16, 0, 0, 0, -53, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -127, -58, -9, -85, -60, -14, 99, -64, -17, 125, -113, -124, -70, -20, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 17, 0, 0, 0, -52, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -85, -18, 69, 44, -38, -11, 99, -64, -20, 72, 98, 54, -49, -28, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 18, 0, 0, 0, -51, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 75, 15, -108, -84, -17, -8, 99, -64, -18, -127, 52, -24, -29, -36, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 19, 0, 0, 0, -50, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -19, -65, -30, 44, 5, -4, 99, -64, -107, 125, 7, -102, -8, -44, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 20, 0, 0, 0, -49, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 119, -47, 48, -83, 26, -1, 99, -64, -23, 23, -38, 75, 13, -51, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 21, 0, 0, 0, -48, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 115, 127, 45, 48, 2, 100, -64, 61, -78, -84, -3, 33, -59, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 22, 0, 0, 0, -47, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 117, -51, -83, 69, 5, 100, -64, -111, 76, 127, -81, 54, -67, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 23, 0, 0, 0, -46, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -14, 7, 28, 46, 91, 8, 100, -64, 56, 72, 82, 97, 75, -75, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 24, 0, 0, 0, -45, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -33, -110, 106, -82, 112, 11, 100, -64, -116, -30, 36, 19, 96, -83, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 25, 0, 0, 0, -44, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -79, 126, -72, 46, -122, 14, 100, -64, -32, 124, -9, -60, 116, -91, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 26, 0, 0, 0, -43, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -74, 82, -65, 27, -98, 17, 100, -64, -106, 123, -102, -77, -119, -99, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 27, 0, 0, 0, -42, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -34, -127, 29, 78, -41, 20, 100, -64, 119, 21, 127, -10, -95, -107, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 28, 0, 0, 0, -41, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -36, -39, 82, 16, 59, 24, 100, -64, 119, -39, 1, -82, -66, -115, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 29, 0, 0, 0, -40, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -119, 6, -63, 73, -55, 27, 100, -64, 4, 22, 108, 19, -32, -123, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 30, 0, 0, 0, -39, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -104, -89, -108, -32, -127, 31, 100, -64, 42, -91, -27, 95, 6, 126, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 31, 0, 0, 0, -38, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -20, -35, -56, -71, 100, 35, 100, -64, -7, -31, 113, -52, 49, 118, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 32, 0, 0, 0, -37, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -94, 38, 35, -71, 113, 39, 100, -64, 55, 71, -17, -111, 98, 110, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 33, 0, 0, 0, -36, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 23, 56, -63, -88, 43, 100, -64, 18, -22, 21, -23, -104, 102, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 34, 0, 0, 0, -35, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 90, -10, 107, -77, 9, 48, 100, -64, -121, -109, 117, 10, -43, 94, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 35, 0, 0, 0, -34, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 91, -16, 111, -108, 52, 100, -64, -56, -75, 114, 46, 23, 87, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 36, 0, 0, 0, -33, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, -78, -54, -43, 72, 57, 100, -64, 61, -2, 70, -115, 95, 79, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 37, 0, 0, 0, -32, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 11, 35, -51, -62, 38, 62, 100, -64, -115, -97, -1, 94, -82, 71, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 38, 0, 0, 0, -31, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -113, 54, -95, 19, 46, 67, 100, -64, 18, 35, 121, -37, 3, 64, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 39, 0, 0, 0, -30, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -126, -67, -64, -93, 94, 72, 100, -64, -53, -80, 97, 58, 96, 56, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 40, 0, 0, 0, -29, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 36, 123, 77, -72, 77, 100, -64, -39, 42, 51, -77, -61, 48, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 41, 0, 0, 0, -28, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -88, 19, -13, -23, 58, 83, 100, -64, 105, 7, 54, 125, 46, 41, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 42, 0, 0, 0, -27, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 37, 38, 81, -26, 88, 100, -64, -37, -64, 124, -49, -96, 33, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 43, 0, 0, 0, -26, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 93, -26, 89, -70, 94, 100, -64, 30, -17, -31, -32, 26, 26, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 44, 0, 0, 0, -25, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -63, -127, -33, -39, -74, 100, 100, -64, -88, -3, 9, -24, -100, 18, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 45, 0, 0, 0, -24, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 111, -81, -105, -91, -37, 106, 100, -64, -105, -101, 94, 27, 39, 11, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 46, 0, 0, 0, -23, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -85, -14, 111, -112, 40, 113, 100, -64, 13, 103, 13, -79, -71, 3, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 47, 0, 0, 0, -22, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -67, 13, -90, 108, -99, 119, 100, -64, -125, 78, 8, -33, 84, -4, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 48, 0, 0, 0, -21, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 127, 63, 86, 11, 58, 126, 100, -64, -27, 36, 2, -37, -8, -12, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 49, 0, 0, 0, -20, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 67, 123, 60, -2, -124, 100, -64, 61, 64, 110, -38, -91, -19, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 50, 0, 0, 0, -19, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 90, -81, -16, -50, -23, -117, 100, -64, 30, 111, 125, 18, 92, -26, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 51, 0, 0, 0, -18, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 35, -69, 116, -112, -4, -110, 100, -64, -10, 89, 30, -72, 27, -33, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 52, 0, 0, 0, -17, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -88, -89, 77, 54, -102, 100, -64, 29, -51, -5, -1, -28, -41, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 53, 0, 0, 0, -16, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 116, 78, 15, -46, -106, -95, 100, -64, -109, 125, 121, 30, -72, -48, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 54, 0, 0, 0, -15, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -58, -28, 24, -24, 29, -87, 100, -64, -81, 57, -76, 71, -107, -55, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 55, 0, 0, 0, -14, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -71, 18, -86, 84, -53, -80, 100, -64, -28, -41, 72, -82, 124, -62, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 56, 0, 0, 0, -13, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -112, 19, -116, -70, -118, -72, 100, -64, -79, -15, 10, -10, 104, -69, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 57, 0, 0, 0, -12, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 59, 62, 109, 32, 74, -64, 100, -64, 126, 11, -51, 61, 85, -76, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 58, 0, 0, 0, -11, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 109, 89, 79, -122, 9, -56, 100, -64, -12, 85, -113, -123, 65, -83, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 59, 0, 0, 0, -10, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 118, -98, 48, -20, -56, -49, 100, -64, 23, 63, 81, -51, 45, -90, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 60, 0, 0, 0, -9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, -44, 18, 82, -120, -41, 100, -64, 58, 40, 19, 21, 26, -97, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 61, 0, 0, 0, -8, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 106, 51, -12, -73, 71, -33, 100, -64, -76, -32, -44, 92, 6, -104, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 62, 0, 0, 0, -7, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -60, -21, -43, 29, 7, -25, 100, -64, 125, -116, -105, -92, -14, -112, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 63, 0, 0, 0, -6, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 94, 43, 109, -104, -41, -18, 100, -64, -127, -119, -104, -89, -29, -119, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 64, 0, 0, 0, -5, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 93, -69, -15, -7, -10, 100, -64, -26, -73, -14, -9, -21, -126, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 65, 0, 0, 0, -4, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 29, -104, -23, 86, 115, -1, 100, -64, 109, 122, 0, 105, 14, 124, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 66, 0, 0, 0, -3, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 13, 91, 117, 120, 66, 8, 101, -64, 105, -126, -109, 10, 76, 117, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 67, 0, 0, 0, -2, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -29, 38, -101, -7, 101, 17, 101, -64, 126, 41, 71, -24, -91, 110, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 68, 0, 0, 0, -1, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 40, -46, -120, 112, -36, 26, 101, -64, 93, -37, 89, 9, 29, 104, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 69, 0, 0, 0, 0, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 62, -20, -103, 102, -92, 36, 101, -64, -14, 6, -124, 112, -78, 97, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 70, 0, 0, 0, 1, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -82, -99, -115, 88, -68, 46, 101, -64, 55, 28, -50, 27, 103, 91, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 71, 0, 0, 0, 2, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -52, 105, -59, -74, 34, 57, 101, -64, -107, -36, 107, 4, 60, 85, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 72, 0, 0, 0, 3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 111, -63, -126, -27, -43, 67, 101, -64, 5, -34, -109, 30, 50, 79, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 73, 0, 0, 0, 4, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -52, -50, 34, 61, -44, 78, 101, -64, 26, 12, 91, 89, 74, 73, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 74, 0, 0, 0, 5, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 107, -80, 102, 10, 28, 90, 101, -64, -71, -93, -114, -98, -123, 67, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 75, 0, 0, 0, 6, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -50, 11, -79, -114, -85, 101, 101, -64, 22, 106, -111, -46, -28, 61, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 76, 0, 0, 0, 7, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -65, -127, 76, 0, -127, 113, 101, -64, 21, 107, 54, -44, 104, 56, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 77, 0, 0, 0, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -62, -49, 44, 112, -102, 125, 101, -64, -104, 95, 61, 109, 18, 51, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 78, 0, 0, 0, 9, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 117, -71, 98, -77, -47, -119, 101, -64, 66, -29, 125, 9, -51, 45, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 79, 0, 0, 0, 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -3, 1, -28, 13, -106, 101, -64, 10, 52, 23, -123, -118, 40, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 80, 0, 0, 0, 11, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 121, -109, 33, -118, 123, -94, 101, -64, -29, 47, 14, 94, 101, 35, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 81, 0, 0, 0, 12, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -116, -102, -87, -6, 35, -81, 101, -64, 104, -88, 6, -116, 100, 30, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 82, 0, 0, 0, 13, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -122, -55, -41, -116, 5, -68, 101, -64, 123, -57, -29, -74, -120, 25, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 83, 0, 0, 0, 14, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 25, -10, 112, -112, 30, -55, 101, -64, -120, 10, -80, -127, -46, 20, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 84, 0, 0, 0, 15, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -46, -96, -15, 77, 109, -42, 101, -64, 38, 5, -121, -118, 66, 16, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 85, 0, 0, 0, 16, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, -73, -52, 6, -16, -29, 101, -64, -84, -39, -128, 106, -39, 11, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 86, 0, 0, 0, 17, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 118, -103, -91, -11, -92, -15, 101, -64, -71, -43, -98, -75, -105, 7, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 87, 0, 0, 0, 18, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -113, 31, -118, 78, -118, -1, 101, -64, 115, 27, -73, -6, 125, 3, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 88, 0, 0, 0, 19, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -88, -3, 46, 63, -98, 13, 102, -64, 10, 98, 98, -61, -116, -1, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 89, 0, 0, 0, 20, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -13, -123, 46, -17, -34, 27, 102, -64, -117, -87, -22, -109, -60, -5, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 90, 0, 0, 0, 21, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 47, -22, 114, 74, 42, 102, -64, -84, 53, -112, -35, 37, -8, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 91, 0, 0, 0, 22, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 119, 31, -62, -56, 56, 102, -64, -73, -2, 82, 3, -102, -12, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 92, 0, 0, 0, 23, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -4, 127, -60, -70, 78, 71, 102, -64, 92, 76, 75, 18, 22, -15, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 93, 0, 0, 0, 24, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 63, -69, 11, -13, 85, 102, -64, -127, -43, -55, 49, -78, -19, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 94, 0, 0, 0, 25, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 78, 36, 69, 26, -74, 100, 102, -64, -18, 27, -95, 79, 112, -22, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 95, 0, 0, 0, 26, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -122, -60, -69, -86, -106, 115, 102, -64, -66, -91, 120, -79, 80, -25, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 96, 0, 0, 0, 27, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 79, 12, -2, 126, -109, -126, 102, -64, -1, 112, 26, -102, 83, -28, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 97, 0, 0, 0, 28, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -62, -107, -110, 86, -85, -111, 102, -64, -128, 2, 110, 73, 121, -31, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 98, 0, 0, 0, 29, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 44, -54, -69, -18, -36, -96, 102, -64, -20, -7, 116, -4, -63, -34, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 99, 0, 0, 0, 30, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 22, 20, -106, 2, 39, -80, 102, -64, 95, -63, 65, -19, 45, -36, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 100, 0, 0, 0, 31, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -52, -42, -88, 73, -120, -65, 102, -64, -66, -118, 119, 80, -67, -39, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 101, 0, 0, 0, 32, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 105, -85, -90, -62, -22, -50, 102, -64, -89, 78, 84, -113, 78, -41, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 102, 0, 0, 0, 33, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -9, -59, -111, 14, 58, -34, 102, -64, -19, 69, -24, 66, -62, -44, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 103, 0, 0, 0, 34, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -108, 31, 125, 74, 117, -19, 102, -64, 108, -55, -8, -112, 24, -46, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 104, 0, 0, 0, 35, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 44, 11, -90, -108, -101, -4, 102, -64, -50, -71, -3, -96, 81, -49, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 105, 0, 0, 0, 36, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 60, -82, 125, 12, -84, 11, 103, -64, 47, 30, 33, -100, 109, -52, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 106, 0, 0, 0, 37, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -109, -61, -70, -46, -91, 26, 103, -64, 68, -108, 58, -83, 108, -55, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 107, 0, 0, 0, 38, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 39, 20, 99, 9, -120, 41, 103, -64, -96, 103, -47, 0, 79, -58, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 108, 0, 0, 0, 39, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -112, -37, -38, -45, 81, 56, 103, -64, -14, 3, 20, -59, 20, -61, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 109, 0, 0, 0, 40, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -69, 81, -49, -36, 3, 71, 103, -64, -32, 5, 32, -46, -65, -65, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 110, 0, 0, 0, 41, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 70, 23, -95, 46, -74, 85, 103, -64, 114, 92, -108, 109, 99, -68, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 111, 0, 0, 0, 42, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 43, -90, -96, 28, 103, 100, 103, -64, -48, 42, 103, 90, 7, -71, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 112, 0, 0, 0, 43, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -76, 120, -95, 10, 24, 115, 103, -64, -40, 41, 58, 71, -85, -75, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 113, 0, 0, 0, 44, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 44, -56, -95, -8, -56, -127, 103, -64, 55, -8, 12, 52, 79, -78, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 114, 0, 0, 0, 45, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 35, 44, -94, -26, 121, -112, 103, -64, -110, 88, -32, 32, -13, -82, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 115, 0, 0, 0, 46, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -101, -92, -94, -44, 42, -97, 103, -64, 67, -120, -77, 13, -105, -85, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 116, 0, 0, 0, 47, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, -55, -93, -62, -37, -83, 103, -64, -8, 37, -122, -6, 58, -88, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 117, 0, 0, 0, 48, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, -45, -93, -80, -116, -68, 103, -64, 83, -122, 89, -25, -34, -92, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 118, 0, 0, 0, 49, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, -15, -93, -98, 61, -53, 103, -64, -82, -26, 44, -44, -126, -95, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 119, 0, 0, 0, 50, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -110, -52, -122, -18, -19, -39, 103, -64, -25, -104, 84, 20, 38, -98, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 120, 0, 0, 0, 51, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -5, -49, -19, -2, -102, -24, 103, -64, -45, -4, -66, -54, -59, -102, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 121, 0, 0, 0, 52, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 8, 38, -128, 68, -9, 103, -64, -122, 44, -101, -91, 97, -105, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 122, 0, 0, 0, 53, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -99, 20, 33, 110, -22, 5, 104, -64, -105, 10, -39, -91, -7, -109, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 123, 0, 0, 0, 54, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -38, -62, -47, -60, -116, 20, 104, -64, -106, -99, 105, -52, -115, -112, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 124, 0, 0, 0, 55, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -122, 120, 43, -128, 43, 35, 104, -64, -82, -46, 63, 26, 30, -115, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 125, 0, 0, 0, 56, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -83, 97, 35, -100, -58, 49, 104, -64, -72, 53, 78, -112, -86, -119, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 126, 0, 0, 0, 57, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 51, -86, -82, 20, 94, 64, 104, -64, -43, 105, -119, 47, 51, -122, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 127, 0, 0, 0, 58, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -94, 68, -60, -27, -15, 78, 104, -64, -60, 102, -25, -8, -73, -126, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -128, 0, 0, 0, 59, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -25, -70, 91, 11, -126, 93, 104, -64, -17, 84, 94, -19, 56, 127, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -127, 0, 0, 0, 60, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -125, 93, 110, -127, 14, 108, 104, -64, -74, 18, -26, 13, -74, 123, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -126, 0, 0, 0, 61, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 52, -74, -13, 67, -105, 122, 104, -64, 116, 126, 118, 91, 47, 120, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -125, 0, 0, 0, 62, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -72, -47, -23, 78, 28, -119, 104, -64, 119, -66, 9, -41, -92, 116, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -124, 0, 0, 0, 63, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 65, -87, -9, 28, -97, -105, 104, -64, 101, -125, 12, 8, 24, 113, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -123, 0, 0, 0, 64, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 100, 110, -83, -27, 33, -90, 104, -64, 4, 33, -105, 51, -117, 109, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -122, 0, 0, 0, 65, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -117, -85, 98, -82, -92, -76, 104, -64, -11, 31, 34, 95, -2, 105, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -121, 0, 0, 0, 66, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 78, -8, 23, 119, 39, -61, 104, -64, -109, -67, -84, -118, 113, 102, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -120, 0, 0, 0, 67, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 25, -67, -52, 63, -86, -47, 104, -64, 50, 91, 55, -74, -28, 98, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -119, 0, 0, 0, 68, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 41, -126, 8, 45, -32, 104, -64, 39, -56, -63, -31, 87, 95, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -118, 0, 0, 0, 69, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -99, -92, 55, -47, -81, -18, 104, -64, 111, -106, 76, 13, -53, 91, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -117, 0, 0, 0, 70, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49, -104, -20, -103, 50, -3, 104, -64, 96, -107, -41, 56, 62, 88, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -116, 0, 0, 0, 71, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -9, 50, -94, 98, -75, 11, 105, -64, -1, 50, 98, 100, -79, 84, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -115, 0, 0, 0, 72, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -64, 69, 87, 43, 56, 26, 105, -64, -103, 98, -19, -113, 36, 81, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -114, 0, 0, 0, 73, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 39, 104, 12, -12, -70, 40, 105, -64, -27, -98, 119, -69, -105, 77, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -113, 0, 0, 0, 74, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 35, -102, -63, -68, 61, 55, 105, -64, -42, -99, 2, -25, 10, 74, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -112, 0, 0, 0, 75, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -67, -37, 118, -123, -64, 69, 105, -64, -53, 10, -115, 18, 126, 70, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -111, 0, 0, 0, 76, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -13, 44, 44, 78, 67, 84, 105, -64, 19, -39, 23, 62, -15, 66, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -110, 0, 0, 0, 77, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -67, -115, -31, 22, -58, 98, 105, -64, 91, -89, -94, 105, 100, 63, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -109, 0, 0, 0, 78, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -78, -107, -105, -33, 72, 113, 105, -64, -6, 68, 45, -107, -41, 59, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -108, 0, 0, 0, 79, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -110, -26, 75, -88, -53, 127, 105, -64, 66, 19, -72, -64, 74, 56, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -107, 0, 0, 0, 80, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 45, 118, 1, 113, 78, -114, 105, -64, 51, 18, 67, -20, -67, 52, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -106, 0, 0, 0, 81, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -47, 125, -74, 57, -47, -100, 105, -64, -47, -81, -51, 23, 49, 49, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -105, 0, 0, 0, 82, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -99, 44, 108, 2, 84, -85, 105, -64, 25, 126, 88, 67, -92, 45, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -104, 0, 0, 0, 83, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -23, -69, 32, -53, -42, -71, 105, -64, 14, -21, -30, 110, 23, 42, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -103, 0, 0, 0, 84, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -23, -119, -42, -109, 89, -56, 105, -64, 86, -71, 109, -102, -118, 38, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -102, 0, 0, 0, 85, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -123, 103, -116, 92, -36, -42, 105, -64, -98, -121, -8, -59, -3, 34, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -101, 0, 0, 0, 86, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, -114, 64, 37, 95, -27, 105, -64, -26, 85, -125, -15, 112, 31, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -100, 0, 0, 0, 87, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, -13, -11, -19, -31, -13, 105, -64, -124, -13, 13, 29, -28, 27, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -99, 0, 0, 0, 88, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49, 104, -85, -74, 100, 2, 106, -64, 35, -111, -104, 72, 87, 24, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -98, 0, 0, 0, 89, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 22, 85, 96, 127, -25, 16, 106, -64, 107, 95, 35, 116, -54, 20, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -97, 0, 0, 0, 90, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 39, -23, 21, 72, 106, 31, 106, -64, 92, 94, -82, -97, 61, 17, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -96, 0, 0, 0, 91, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, -11, -54, 16, -19, 45, 106, -64, 81, -53, 56, -53, -80, 13, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -95, 0, 0, 0, 92, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -6, 16, -128, -39, 111, 60, 106, -64, 66, -54, -61, -10, 35, 10, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -94, 0, 0, 0, 93, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 110, 107, 54, -94, -14, 74, 106, -64, -114, 6, 78, 34, -105, 6, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -93, 0, 0, 0, 94, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, -39, -62, 125, 113, 89, 106, -64, -120, -70, -23, 78, 6, 3, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -92, 0, 0, 0, 95, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -22, 16, -57, 97, -26, 103, 106, -64, -97, 57, 35, 108, 107, -1, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -91, 0, 0, 0, 96, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 9, -8, 33, 81, 118, 106, -64, 44, -82, -86, 112, -58, -5, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -90, 0, 0, 0, 97, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -110, -72, 32, -94, -79, -124, 106, -64, -117, 21, -95, 99, 23, -8, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -89, 0, 0, 0, 98, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -70, -52, 32, -58, 7, -109, 106, -64, -124, -122, 60, 76, 94, -12, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -88, 0, 0, 0, 99, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -23, 19, -20, 113, 83, -95, 106, -64, 97, 51, -60, 49, -101, -16, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -87, 0, 0, 0, 100, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 116, -85, -117, -119, -108, -81, 106, -64, 123, 42, -107, 27, -50, -20, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -86, 0, 0, 0, 101, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, 10, 27, -15, -54, -67, 106, -64, 85, 88, 30, 17, -9, -24, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -85, 0, 0, 0, 102, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -23, 83, -51, -116, -10, -53, 106, -64, -126, -13, -29, 25, 22, -27, 124, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 0, -84, 0, 0, 0, 17, 12, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 5, 0, 0, 0, 0, 0, 0, 0, 115, 111, 108, 105, 100, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 9, 0, 0, 0, 0, 0, 0, 0, 108, 105, 110, 101, 95, 116, 104, 105, 110, -87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -83, 0, 0, 0, 103, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -17, -112, 95, 80, 93, 65, 100, -64, 23, -83, -43, -84, 20, 51, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -82, 0, 0, 0, 104, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -125, -126, -29, 111, -41, 66, 100, -64, -108, 61, 109, -127, -70, 46, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -81, 0, 0, 0, 105, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, -39, -38, 52, 117, 68, 100, -64, 119, -28, 60, 18, 98, 42, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -80, 0, 0, 0, 106, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 90, 28, -108, 56, 70, 100, -64, -9, 125, -90, 119, 11, 38, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -79, 0, 0, 0, 107, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -76, -46, 126, -126, 35, 72, 100, -64, 80, 11, 11, -54, -74, 33, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -78, 0, 0, 0, 108, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 25, -39, -12, 55, 74, 100, -64, -68, 31, -52, 33, 100, 29, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -77, 0, 0, 0, 109, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -86, -36, 0, -32, 119, 76, 100, -64, 31, -19, 74, -105, 19, 25, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -76, 0, 0, 0, 110, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -4, -102, -51, 56, -27, 78, 100, -64, 6, 31, -23, 66, -59, 20, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -75, 0, 0, 0, 111, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -77, 66, 22, -12, -127, 81, 100, -64, 3, -49, 7, 61, 121, 16, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -74, 0, 0, 0, 112, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -62, 50, -79, 6, 80, 84, 100, -64, -9, -64, 8, -98, 47, 12, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -73, 0, 0, 0, 113, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -33, -94, 115, 101, 81, 87, 100, -64, 117, -59, 76, 126, -24, 7, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -72, 0, 0, 0, 114, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -126, -9, 54, 5, -120, 90, 100, -64, 8, -47, 53, -10, -93, 3, -128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -71, 0, 0, 0, 115, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 43, 79, -108, -112, -99, 93, 100, -64, 54, -87, 17, -71, 89, -1, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -70, 0, 0, 0, 116, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -118, 27, -16, -11, -81, 96, 100, -64, -50, 79, 127, 29, 110, -9, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -69, 0, 0, 0, 117, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, -81, -43, 86, -64, 99, 100, -64, -29, -26, -92, 79, -126, -17, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -68, 0, 0, 0, 118, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, -111, 80, -66, -50, 102, 100, -64, 19, 26, -107, 80, -106, -25, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -67, 0, 0, 0, 119, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49, -31, 108, 55, -37, 105, 100, -64, -17, 74, 100, 33, -86, -33, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -66, 0, 0, 0, 120, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 86, 55, -51, -27, 108, 100, -64, 105, -12, 36, -61, -67, -41, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -65, 0, 0, 0, 121, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -23, -32, -70, -118, -18, 111, 100, -64, -64, 22, -21, 54, -47, -49, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -64, 0, 0, 0, 122, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -52, 113, 2, 123, -11, 114, 100, -64, -25, 44, -55, 125, -28, -57, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -63, 0, 0, 0, 123, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -97, -17, 27, -87, -6, 117, 100, -64, 28, 55, -45, -104, -9, -65, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -62, 0, 0, 0, 124, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -3, -30, 18, 32, -2, 120, 100, -64, -95, -93, 28, -119, 10, -72, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -61, 0, 0, 0, 125, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, 108, -13, -22, -1, 123, 100, -64, 106, -19, -73, 79, 29, -80, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -60, 0, 0, 0, 126, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 47, 73, -125, -6, -18, 126, 100, -64, 7, -61, -56, -42, 97, -88, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -59, 0, 0, 0, 127, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 62, 78, 74, 81, -43, -127, 100, -64, 97, 118, -22, -39, 10, -95, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -58, 0, 0, 0, -128, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -120, 56, 49, -79, -31, -124, 100, -64, -1, -46, -26, -107, -73, -103, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -57, 0, 0, 0, -127, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 23, 22, 19, 20, -120, 100, -64, -112, 96, 37, 59, 104, -110, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -56, 0, 0, 0, -126, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 41, 13, -29, 110, 108, -117, 100, -64, -98, 35, 54, -6, 28, -117, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -55, 0, 0, 0, -125, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -113, 16, -125, -69, -22, -114, 100, -64, 22, 39, -53, 3, -42, -125, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -54, 0, 0, 0, -124, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 75, -31, -28, -18, -114, -110, 100, -64, 23, -62, -66, -120, -109, 124, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -53, 0, 0, 0, -123, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -28, -19, -13, -3, 88, -106, 100, -64, 45, -100, 11, -70, 85, 117, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -52, 0, 0, 0, -122, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 26, -102, -36, 72, -102, 100, -64, -11, 111, -51, -56, 28, 110, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -51, 0, 0, 0, -121, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -29, -45, -71, 125, 94, -98, 100, -64, -51, -87, 64, -26, -24, 102, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -50, 0, 0, 0, -120, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -59, -77, 43, -45, -103, -94, 100, -64, 53, -127, -64, 67, -70, 95, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -49, 0, 0, 0, -119, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -98, -26, -67, -51, -6, -90, 100, -64, -25, -116, -61, 18, -111, 88, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -48, 0, 0, 0, -118, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -4, 7, 48, 93, -127, -85, 100, -64, 35, 72, -35, -124, 109, 81, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -47, 0, 0, 0, -117, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -41, -108, 47, 112, 45, -80, 100, -64, 113, -41, -70, -53, 79, 74, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -46, 0, 0, 0, -116, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -96, 26, 89, -12, -2, -76, 100, -64, 11, -2, 31, 25, 56, 67, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -45, 0, 0, 0, -115, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -87, 75, 50, -42, -11, -71, 100, -64, 119, -106, -23, -98, 38, 60, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -44, 0, 0, 0, -114, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -55, -96, 39, 1, 18, -65, 100, -64, 87, 15, 8, -113, 27, 53, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -43, 0, 0, 0, -113, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -13, -16, -116, 95, 83, -60, 100, -64, -55, -88, 126, 27, 23, 46, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -42, 0, 0, 0, -112, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -102, -76, -104, -38, -71, -55, 100, -64, 95, 116, 99, 118, 25, 39, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -41, 0, 0, 0, -111, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -14, 15, 97, 90, 69, -49, 100, -64, -27, -85, -36, -47, 34, 32, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -40, 0, 0, 0, -110, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -8, 1, -35, -59, -11, -44, 100, -64, 125, 33, 28, 96, 51, 25, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -39, 0, 0, 0, -109, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -57, 120, -34, 2, -53, -38, 100, -64, -37, 12, 99, 83, 75, 18, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -38, 0, 0, 0, -108, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -71, -128, 19, -10, -60, -32, 100, -64, 25, -10, -5, -35, 106, 11, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -37, 0, 0, 0, -107, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -70, 88, 0, -125, -29, -26, 100, -64, 24, 97, 57, 50, -110, 4, 127, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -36, 0, 0, 0, -106, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -14, 56, 1, -116, 38, -19, 100, -64, 43, -2, 117, -126, -63, -3, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -35, 0, 0, 0, -105, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 57, -106, 69, -14, -115, -13, 100, -64, 50, 26, 16, 1, -7, -10, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -34, 0, 0, 0, -104, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 34, -48, -107, 25, -6, 100, -64, -94, -98, 105, -32, 56, -16, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -33, 0, 0, 0, -103, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 108, 116, 85, -55, 0, 101, -64, 54, -6, -27, 82, -127, -23, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -32, 0, 0, 0, -102, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -66, -66, -46, 14, -99, 7, 101, -64, 97, 22, -25, -118, -46, -30, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -31, 0, 0, 0, -101, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 117, -46, 52, -59, -104, 14, 101, -64, 122, -77, 12, -21, 40, -36, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -30, 0, 0, 0, -100, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 26, 94, 42, 91, 43, 22, 101, -64, -109, -89, 14, -17, 8, -43, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -29, 0, 0, 0, -99, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, -65, 118, -82, -67, 29, 101, -64, -99, 115, -46, -32, -24, -51, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -28, 0, 0, 0, -98, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 28, -66, -8, -56, 79, 37, 101, -64, -40, -38, 10, -61, -56, -58, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -27, 0, 0, 0, -97, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -34, -69, -113, -76, -31, 44, 101, -64, 120, 86, 108, -104, -88, -65, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -26, 0, 0, 0, -96, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -20, 24, 27, 123, 115, 52, 101, -64, 101, -38, -87, 99, -120, -72, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -25, 0, 0, 0, -95, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 54, 122, 38, 5, 60, 101, -64, 43, -81, 119, 39, 104, -79, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -24, 0, 0, 0, -94, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -11, 115, -116, -64, -106, 67, 101, -64, -82, 90, -119, -26, 71, -86, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -23, 0, 0, 0, -93, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -35, -4, -112, 82, 113, 74, 101, -64, 33, 87, 114, -33, -33, -93, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -22, 0, 0, 0, -92, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -99, -54, -22, -46, 24, 81, 101, -64, -90, 64, -17, -27, -8, -99, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -21, 0, 0, 0, -91, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, -77, 62, -51, 10, 88, 101, -64, 26, 40, 41, -56, 38, -104, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -20, 0, 0, 0, -90, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -127, 95, 31, 69, 70, 95, 101, -64, -15, -80, 99, 116, 106, -110, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -19, 0, 0, 0, -89, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -39, 52, -31, 49, -54, 102, 101, -64, 40, -109, 31, -42, -60, -116, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -18, 0, 0, 0, -88, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 67, -121, -72, 126, -107, 110, 101, -64, 10, 115, -13, -43, 54, -121, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -17, 0, 0, 0, -87, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -108, 14, -45, 10, -89, 118, 101, -64, 106, -118, 96, 89, -63, -127, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -16, 0, 0, 0, -86, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 71, 119, -87, -3, 126, 101, -64, -67, -67, -86, 66, 101, 124, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -15, 0, 0, 0, -85, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, -112, 40, 34, -104, -121, 101, -64, 91, -77, -84, 112, 35, 119, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -14, 0, 0, 0, -84, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -57, -13, -59, 48, 117, -112, 101, -64, -38, 35, -77, -66, -4, 113, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -13, 0, 0, 0, -83, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 99, -103, -77, -123, -109, -103, 101, -64, -89, 46, 80, 4, -14, 108, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -12, 0, 0, 0, -82, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -2, 122, -1, -59, -15, -94, 101, -64, 21, 37, 53, 21, 4, 104, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -11, 0, 0, 0, -81, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 12, -62, -112, -117, -114, -84, 101, -64, 127, -97, 10, -63, 51, 99, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -10, 0, 0, 0, -80, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 27, 10, 80, 101, 104, -74, 101, -64, 107, 0, 72, -45, -127, 94, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -9, 0, 0, 0, -79, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -61, 75, 17, -103, -64, 101, -64, -71, -1, -20, 60, -29, 89, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -8, 0, 0, 0, -78, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 47, 103, 68, 82, -79, -52, 101, -64, -31, -97, -30, -31, -117, 84, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -7, 0, 0, 0, -77, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -88, 107, -126, -23, 25, -40, 101, -64, 66, -27, -117, 12, -123, 79, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -6, 0, 0, 0, -76, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 13, 49, -17, -80, -94, -30, 101, -64, 117, -5, 86, -73, 10, 75, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -5, 0, 0, 0, -75, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -113, 93, -85, -17, 95, -19, 101, -64, -26, -116, -74, 41, -81, 70, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -4, 0, 0, 0, -74, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 43, 53, 15, 65, 80, -8, 101, -64, 127, 100, 42, 3, 115, 66, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -3, 0, 0, 0, -73, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 44, 72, 103, 56, 114, 3, 102, -64, 112, 97, 68, -33, 86, 62, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -2, 0, 0, 0, -72, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 89, -58, 38, 97, -60, 14, 102, -64, -4, -13, -117, 85, 91, 58, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, -1, 0, 0, 0, -71, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -79, -100, 11, 63, 69, 26, 102, -64, -110, 31, 99, -7, -128, 54, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 0, 1, 0, 0, -70, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -117, -56, 80, 78, -13, 37, 102, -64, 55, -34, -21, 89, -56, 50, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 1, 1, 0, 0, -69, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 75, -65, -38, 3, -51, 49, 102, -64, -108, 70, -18, 1, 50, 47, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 2, 1, 0, 0, -68, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 42, 105, -51, -48, 61, 102, -64, -72, -66, -67, 119, -66, 43, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 3, 1, 0, 0, -67, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 115, -48, -55, 17, -3, 73, 102, -64, -51, -28, 31, 61, 110, 40, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 4, 1, 0, 0, -66, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -88, 90, -9, 121, 106, 86, 102, -64, 1, -75, 51, 82, 59, 37, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 5, 1, 0, 0, -65, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -12, -124, -45, 66, -39, 100, 102, -64, 121, -15, 67, -101, -97, 33, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 6, 1, 0, 0, -64, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -36, -23, 94, -13, 49, 114, 102, -64, 42, -117, 70, 81, 81, 30, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 7, 1, 0, 0, -63, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 127, -18, 113, -86, 36, 127, 102, -64, -107, -33, -110, -60, 63, 27, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 8, 1, 0, 0, -62, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 34, 48, 73, 52, -116, 102, -64, -113, -54, 3, -58, 76, 24, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 9, 1, 0, 0, -61, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 93, 30, -44, -83, 95, -103, 102, -64, -68, 18, -1, -97, 120, 21, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 10, 1, 0, 0, -60, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -72, 86, 54, -77, -91, -90, 102, -64, -113, 60, 65, -102, -61, 18, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 11, 1, 0, 0, -59, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, -66, -25, 48, 5, -76, 102, -64, -23, -89, -45, -7, 45, 16, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 12, 1, 0, 0, -58, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -96, 71, 72, -5, 124, -63, 102, -64, 112, -106, 0, 1, -72, 13, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 13, 1, 0, 0, -57, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 58, 21, -91, -29, 11, -49, 102, -64, -44, -25, 72, -17, 97, 11, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 14, 1, 0, 0, -56, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 71, -11, 118, -101, -66, -36, 102, -64, -66, 88, -77, -37, 41, 9, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 15, 1, 0, 0, -55, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -27, 75, -78, -19, -122, -19, 102, -64, 73, 66, 117, -116, 119, 6, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 16, 1, 0, 0, -54, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 33, -6, -21, 62, 58, -2, 102, -64, 90, 93, 118, -37, -91, 3, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 17, 1, 0, 0, -53, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -25, 9, -121, -82, -41, 14, 103, -64, -21, 122, -55, -11, -76, 0, 126, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 18, 1, 0, 0, -52, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -20, -52, -86, 93, 94, 31, 103, -64, -44, 44, -10, 9, -91, -3, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 19, 1, 0, 0, -51, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 118, 122, 80, 111, -51, 47, 103, -64, -103, 66, -13, 71, 118, -6, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 20, 1, 0, 0, -50, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 52, 84, 71, 8, 36, 64, 103, -64, -40, -66, 35, -31, 40, -9, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 21, 1, 0, 0, -49, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 119, 88, 60, 79, 97, 80, 103, -64, 48, -24, 77, 8, -67, -13, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 22, 1, 0, 0, -48, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -68, -127, -59, 108, -124, 96, 103, -64, -9, 85, -102, -15, 50, -16, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 23, 1, 0, 0, -47, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -36, -104, 125, -74, 23, 112, 103, -64, -68, -24, 2, -69, -89, -20, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 24, 1, 0, 0, -46, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 63, -100, 16, 8, -54, 126, 103, -64, 66, -83, 49, 86, 75, -23, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 25, 1, 0, 0, -45, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -54, -76, 85, -11, 122, -115, 103, -64, 97, -61, 55, 66, -17, -27, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 26, 1, 0, 0, -44, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -72, -19, 32, -30, 43, -100, 103, -64, -112, 92, -71, 45, -109, -30, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 27, 1, 0, 0, -43, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 101, -44, 117, -50, -36, -86, 103, -64, 15, 34, -71, 24, 55, -33, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 28, 1, 0, 0, -42, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, -57, 86, -70, -115, -71, 103, -64, -49, 91, 57, 3, -37, -37, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 29, 1, 0, 0, -41, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -43, 35, -58, -91, 62, -56, 103, -64, -77, 7, 62, -19, 126, -40, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 30, 1, 0, 0, -40, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 42, 120, -57, -112, -17, -42, 103, -64, 75, -62, -54, -42, 34, -43, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 31, 1, 0, 0, -39, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 56, 34, 93, 123, -96, -27, 103, -64, -30, 16, -31, -65, -58, -47, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 32, 1, 0, 0, -38, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, -128, -119, 101, 81, -12, 103, -64, -78, -64, -124, -88, 106, -50, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 33, 1, 0, 0, -37, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, -84, 100, -103, 29, 3, 104, -64, 105, -116, -91, 127, 7, -53, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 34, 1, 0, 0, -36, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 75, -5, 73, -43, -7, 17, 104, -64, 66, 5, 18, 86, -100, -57, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 35, 1, 0, 0, -35, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 98, 56, 34, 118, -46, 32, 104, -64, 9, -20, 33, 68, 45, -60, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 36, 1, 0, 0, -34, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 127, -61, -45, 119, -89, 47, 104, -64, -46, -103, -53, 74, -70, -64, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 37, 1, 0, 0, -33, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 105, 43, 70, -42, 120, 62, 104, -64, -92, 29, 7, 107, 67, -67, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 38, 1, 0, 0, -32, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 91, -106, 97, -115, 70, 77, 104, -64, 50, 37, -52, -91, -56, -71, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 39, 1, 0, 0, -31, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -99, 89, 15, -103, 16, 92, 104, -64, 116, 7, 21, -4, 73, -74, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 40, 1, 0, 0, -30, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 62, -54, 56, -11, -42, 106, 104, -64, 92, -83, -36, 110, -57, -78, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 41, 1, 0, 0, -29, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 60, 98, -53, -99, -103, 121, 104, -64, 43, -13, 30, -1, 64, -81, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 42, 1, 0, 0, -28, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -109, -91, -79, -114, 88, -120, 104, -64, 30, 71, -40, -83, -74, -85, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 43, 1, 0, 0, -27, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -84, -44, -38, -61, 19, -105, 104, -64, -73, 46, 7, 124, 40, -88, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 44, 1, 0, 0, -26, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -91, 0, 53, 57, -53, -91, 104, -64, -47, -2, -87, 106, -106, -92, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 45, 1, 0, 0, -25, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 59, 1, -80, -22, 126, -76, 104, -64, 50, -26, -63, 122, 0, -95, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 46, 1, 0, 0, -24, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -65, -14, -99, -97, 6, -61, 104, -64, -30, -100, -111, 109, 114, -99, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 47, 1, 0, 0, -23, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 36, -114, 40, 95, -119, -47, 104, -64, -24, 83, -66, -113, -27, -103, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 48, 1, 0, 0, -22, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 104, 73, 115, 30, 12, -32, 104, -64, 24, -40, -87, -79, 88, -106, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 49, 1, 0, 0, -21, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 26, -127, -35, -114, -18, 104, -64, 8, 52, 87, -45, -53, -110, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 50, 1, 0, 0, -20, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -106, -10, 84, -100, 17, -3, 104, -64, 80, -32, -56, -12, 62, -113, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 51, 1, 0, 0, -19, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -125, 60, -15, 90, -108, 11, 105, -64, 42, -86, 2, 22, -78, -117, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 52, 1, 0, 0, -18, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 107, 121, 89, 25, 23, 26, 105, -64, 51, 120, 6, 55, 37, -120, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 53, 1, 0, 0, -17, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, -93, -112, -41, -103, 40, 105, -64, -91, 23, -40, 87, -104, -124, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 54, 1, 0, 0, -16, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -101, 23, -103, -107, 28, 55, 105, -64, 21, -109, 122, 120, 11, -127, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 55, 1, 0, 0, -15, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 100, 118, 83, -97, 69, 105, -64, -59, -109, -16, -104, 126, 125, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 56, 1, 0, 0, -14, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -41, -25, 42, 17, 34, 84, 105, -64, -95, -13, 60, -71, -15, 121, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 57, 1, 0, 0, -13, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 47, -70, -50, -92, 98, 105, -64, -112, 30, 99, -39, 100, 118, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 58, 1, 0, 0, -12, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 70, -104, 38, -116, 39, 113, 105, -64, 38, 31, 102, -7, -41, 114, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 59, 1, 0, 0, -11, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 85, 25, 115, 73, -86, 127, 105, -64, 83, 61, 72, 25, 75, 111, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 60, 1, 0, 0, -10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -117, 63, -93, 6, 45, -114, 105, -64, -87, 21, 13, 57, -66, 107, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 61, 1, 0, 0, -9, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32, 105, -71, -61, -81, -100, 105, -64, 18, -126, -73, 88, 49, 104, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 62, 1, 0, 0, -8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, -12, -73, -128, 50, -85, 105, -64, 36, -115, 74, 120, -92, 100, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 63, 1, 0, 0, -7, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -29, 5, -93, 61, -75, -71, 105, -64, -55, 16, -55, -105, 23, 97, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 64, 1, 0, 0, -6, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 46, -4, 124, -6, 55, -56, 105, -64, 63, 72, 54, -73, -118, 93, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 65, 1, 0, 0, -5, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -19, -52, 72, -73, -70, -42, 105, -64, -54, -36, -108, -42, -3, 89, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 66, 1, 0, 0, -4, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -28, 109, 9, 116, 61, -27, 105, -64, -2, -40, -25, -11, 112, 86, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 67, 1, 0, 0, -3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -33, -44, -63, 48, -64, -13, 105, -64, -62, -88, 50, 21, -28, 82, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 68, 1, 0, 0, -2, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -99, -9, 116, -19, 66, 2, 106, -64, 1, 38, 120, 52, 87, 79, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 69, 1, 0, 0, -1, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 76, 52, 37, -86, -59, 16, 106, -64, 85, -55, -70, 83, -54, 75, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 70, 1, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -34, -81, -42, 102, 72, 31, 106, -64, 78, 47, -2, 114, 61, 72, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 71, 1, 0, 0, 1, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -14, 48, -117, 35, -53, 45, 106, -64, -40, 49, 69, -110, -80, 68, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 72, 1, 0, 0, 2, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -33, 68, 70, -32, 77, 60, 106, -64, -117, 73, -110, -79, 35, 65, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 73, 1, 0, 0, 3, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -36, 73, 10, -99, -48, 74, 106, -64, -94, 67, -23, -48, -106, 61, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 74, 1, 0, 0, 4, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 54, -51, -38, 89, 83, 89, 106, -64, -74, -104, 76, -16, 9, 58, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 75, 1, 0, 0, 5, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 39, 45, -70, 22, -42, 103, 106, -64, 3, -124, -65, 15, 125, 54, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 76, 1, 0, 0, 6, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -109, -114, -84, -45, 88, 118, 106, -64, 119, -33, 68, 47, -16, 50, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 77, 1, 0, 0, 7, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -70, 8, 113, -92, 69, -123, 106, -64, 82, 111, -16, 59, 68, 47, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 78, 1, 0, 0, 8, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 34, 96, -13, 20, 54, -108, 106, -64, -2, -22, 71, 123, -118, 43, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 79, 1, 0, 0, 9, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 98, -42, -55, 10, 28, -93, 106, -64, -76, -76, -57, 75, -58, 39, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 80, 1, 0, 0, 10, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -60, -15, -44, 104, -9, -79, 106, -64, 2, 58, -47, -76, -9, 35, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 81, 1, 0, 0, 11, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -114, 41, 8, 18, -56, -64, 106, -64, 61, -83, -39, -67, 30, 32, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 82, 1, 0, 0, 12, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -21, 20, 107, -23, -115, -49, 106, -64, -48, -8, 106, 110, 59, 28, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 83, 1, 0, 0, 13, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -19, 96, 28, -46, 72, -34, 106, -64, 65, 45, 35, -50, 77, 24, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 84, 1, 0, 0, 14, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -84, -38, 78, -81, -8, -20, 106, -64, 45, -127, -76, -28, 85, 20, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 22, 0, 85, 1, 0, 0, 15, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 115, -51, 75, 100, -99, -5, 106, -64, 77, 81, -27, -71, 83, 16, 125, 64, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0, 30, -29, 19, 0, 0, 0, 0, 0, 4, 0, 86, 1, 0, 0, 30, -29, 19, 0, 0, 0, 0, 0, 13, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 98, 105, 99, 121, 99, 108, 101, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 19, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 23, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 98, 117, 115, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 23, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 99, 97, 114, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 34, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 99, 97, 114, 58, 99, 111, 109, 98, 117, 115, 116, 105, 111, 110, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 32, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 99, 97, 114, 58, 101, 108, 101, 99, 116, 114, 105, 99, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 29, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 101, 109, 101, 114, 103, 101, 110, 99, 121, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 30, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 109, 111, 116, 111, 114, 99, 121, 99, 108, 101, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 24, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 116, 97, 120, 105, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 25, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 116, 114, 117, 99, 107, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 6, 0, 0, 0, 0, 0, 0, 0, 114, 101, 97, 115, 111, 110, 2, 0, 0, 0, 0, 0, 0, 0, 78, 65, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 18, 0, 0, 0, 0, 0, 0, 0, 114, 101, 103, 105, 111, 110, 95, 97, 99, 99, 101, 115, 115, 95, 114, 117, 108, 101, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 18, 0, 0, 0, 0, 0, 0, 0, 114, 101, 103, 117, 108, 97, 116, 111, 114, 121, 95, 101, 108, 101, 109, 101, 110, 116, 1, 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0, 0, 114, 101, 102, 101, 114, 115, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 15, 0, 1, 0, 0, 0, -5, -30, 19, 0, 0, 0, 0, 0, 4, 0, 87, 1, 0, 0, -5, -30, 19, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 20, 0, 0, 0, 0, 0, 0, 0, 112, 97, 115, 115, 105, 110, 103, 95, 99, 111, 110, 116, 114, 111, 108, 95, 108, 105, 110, 101, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 18, 0, 0, 0, 0, 0, 0, 0, 114, 101, 103, 117, 108, 97, 116, 111, 114, 121, 95, 101, 108, 101, 109, 101, 110, 116, 1, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 114, 101, 102, 95, 108, 105, 110, 101, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 18, 0, 2, 0, 0, 0, 31, -29, 19, 0, 0, 0, 0, 0, 4, 0, 88, 1, 0, 0, 31, -29, 19, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 20, 0, 0, 0, 0, 0, 0, 0, 112, 97, 115, 115, 105, 110, 103, 95, 99, 111, 110, 116, 114, 111, 108, 95, 108, 105, 110, 101, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 18, 0, 0, 0, 0, 0, 0, 0, 114, 101, 103, 117, 108, 97, 116, 111, 114, 121, 95, 101, 108, 101, 109, 101, 110, 116, 1, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 114, 101, 102, 95, 108, 105, 110, 101, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 18, 0, -84, 0, 0, 0, 32, -29, 19, 0, 0, 0, 0, 0, 4, 0, 89, 1, 0, 0, 32, -29, 19, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 0, 0, 100, 105, 114, 101, 99, 116, 105, 111, 110, 14, 0, 0, 0, 0, 0, 0, 0, 98, 105, 95, 100, 105, 114, 101, 99, 116, 105, 111, 110, 97, 108, 22, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 112, 101, 100, 101, 115, 116, 114, 105, 97, 110, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 19, 0, 0, 0, 0, 0, 0, 0, 100, 105, 114, 101, 99, 116, 105, 111, 110, 95, 111, 102, 95, 116, 114, 97, 118, 101, 108, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 18, 0, 0, 0, 0, 0, 0, 0, 114, 101, 103, 117, 108, 97, 116, 111, 114, 121, 95, 101, 108, 101, 109, 101, 110, 116, 1, 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0, 0, 114, 101, 102, 101, 114, 115, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 15, 0, 1, 0, 0, 0, -46, -29, 19, 0, 0, 0, 0, 0, 4, 0, 90, 1, 0, 0, -46, -29, 19, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0, 108, 105, 109, 105, 116, 9, 0, 0, 0, 0, 0, 0, 0, 55, 50, 46, 52, 50, 48, 52, 56, 48, 23, 0, 0, 0, 0, 0, 0, 0, 112, 97, 114, 116, 105, 99, 105, 112, 97, 110, 116, 58, 118, 101, 104, 105, 99, 108, 101, 58, 99, 97, 114, 3, 0, 0, 0, 0, 0, 0, 0, 121, 101, 115, 7, 0, 0, 0, 0, 0, 0, 0, 115, 117, 98, 116, 121, 112, 101, 19, 0, 0, 0, 0, 0, 0, 0, 100, 105, 103, 105, 116, 97, 108, 95, 115, 112, 101, 101, 100, 95, 108, 105, 109, 105, 116, 4, 0, 0, 0, 0, 0, 0, 0, 116, 121, 112, 101, 18, 0, 0, 0, 0, 0, 0, 0, 114, 101, 103, 117, 108, 97, 116, 111, 114, 121, 95, 101, 108, 101, 109, 101, 110, 116, 1, 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0, 0, 114, 101, 102, 101, 114, 115, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 15, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 18, 12, 0, 0, 0, 0, 0, 0, -46, -29, 19, 0, 0, 0, 0, 0}
  ;

  wmlw.mapUpdateCallback(std::make_unique<autoware_lanelet2_msgs::msg::MapBin>(update_msg));

}
}