/* Copyright (c) 2020 The Neuropod Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

#include "neuropod/multiprocess/shm/shm_allocator.hh"

#include "fmt/ranges.h"
#include "neuropod/internal/error_utils.hh"
#include "neuropod/internal/memory_utils.hh"

#include <boost/interprocess/sync/interprocess_mutex.hpp>
#include <boost/interprocess/sync/scoped_lock.hpp>

#include <list>
#include <mutex>
#include <unordered_map>

namespace neuropod
{

namespace
{

namespace ipc = boost::interprocess;

// A struct stored in shared memory that contains the data and a cross process refcount
struct __attribute__((__packed__)) SHMBlockInternal
{
    // This mutex is used to synchronize operations on the metadata below
    ipc::interprocess_mutex mutex;

    // The number of times this block is used across processes
    size_t refcount = 0;

    // These blocks of memory can be reused
    // This is incremented on each reuse to invalidate stale blocks
    uint64_t reuse_count = 0;

    // The data in this block
    uint8_t data[];

    // Delete the copy and move constructors
    // (if the copy constructors are deleted, no move constructors will be autogenerated)
    SHMBlockInternal()                         = default;
    ~SHMBlockInternal()                        = default;
    SHMBlockInternal(const SHMBlockInternal &) = delete;
    SHMBlockInternal &operator=(const SHMBlockInternal &) = delete;
};

// A unique ID for a SHM block
// Since blocks of memory can be reused, the UUID used to load
// the block isn't enough. An additional `reuse_count` field is
// used to detect when stale IDs are used and throw an error.
struct __attribute__((__packed__)) SHMBlockIDInternal
{
    RawSHMHandle block_handle;
    uint64_t     reuse_count = 0;
};

// Make sure the size of the ID struct matches the size of the user facing version
static_assert(sizeof(SHMBlockIDInternal) == std::tuple_size<SHMBlockID>::value,
              "The size of SHMBlockIDInternal must match the size of SHMBlockID");

} // namespace

// A cache for raw blocks we've loaded before
class LoadCache
{
private:
    std::map<RawSHMHandle, std::shared_ptr<void>> loaded_cache_;
    std::mutex                                    loaded_cache_mutex_;

public:
    LoadCache()  = default;
    ~LoadCache() = default;

    // Try to get a raw block from the loaded cache
    // If this returns a block, it removes it from the cache
    void maybe_get_and_pop(const RawSHMHandle &handle, std::shared_ptr<void> &raw_block)
    {
        std::lock_guard<std::mutex> lock(loaded_cache_mutex_);
        auto                        item = loaded_cache_.find(handle);
        if (item != loaded_cache_.end())
        {
            raw_block = item->second;
            loaded_cache_.erase(item);
        }
    }

    void insert(RawSHMHandle handle, std::shared_ptr<void> item)
    {
        std::lock_guard<std::mutex> lock(loaded_cache_mutex_);
        loaded_cache_[handle] = std::move(item);
    }

    void clear()
    {
        std::lock_guard<std::mutex> lock(loaded_cache_mutex_);
        loaded_cache_.clear();
    }
};

// A cache for raw blocks we've created
class AllocationCache
{
private:
    // A struct that wraps a raw block and its handle
    struct RawBlockWrapper
    {
        std::shared_ptr<void> block;
        RawSHMHandle          block_handle;
    };

    // In our cache, the main operations we care about are the following:
    //  - equal_range (give me all raw blocks of a certain size)
    //  - erase       (remove a specific raw block from the cache)
    //  - insert      (add a raw block of a specific size to the cache)
    //
    // By using an unordered_map with a list, all the operations we care about can be implemented
    // in an O(1) way. This is better than using an `std::unordered_multimap` or a `std::multimap`
    // directly
    //
    // More importantly, using this structure has a significant improvement on benchmarks over
    // using a multimap or an unordered_multimap
    std::unordered_map<size_t, std::list<RawBlockWrapper>> created_cache_;
    std::mutex                                             created_cache_mutex_;

public:
    AllocationCache()  = default;
    ~AllocationCache() = default;

    // Maybe get an unused raw block from the created cache
    // Increments the refcount and reuse count of the returned block (if any)
    // If this returns a block, it removes it from the cache
    void maybe_get_and_pop(size_t requested_size, std::shared_ptr<void> &raw_block, SHMBlockIDInternal &id)
    {
        std::lock_guard<std::mutex> lock(created_cache_mutex_);
        auto &                      range = created_cache_[requested_size];
        for (auto it = range.begin(); it != range.end(); it++)
        {
            auto &cache_item   = *it;
            auto *cached_block = static_cast<SHMBlockInternal *>(cache_item.block.get());

            ipc::scoped_lock<ipc::interprocess_mutex> lock(cached_block->mutex);

            // Check whether or not this block is used in another process
            if (cached_block->refcount == 0)
            {
                // The block is unused so we can use it!
                // Increment the refcount
                cached_block->refcount++;

                // Increase the reuse_count so that stale IDs will no longer work
                cached_block->reuse_count++;

                raw_block       = cache_item.block;
                id.reuse_count  = cached_block->reuse_count;
                id.block_handle = cache_item.block_handle;

                range.erase(it);
                return;
            }
        }
    }

    void insert(size_t size_bytes, RawSHMHandle handle, std::shared_ptr<void> item)
    {
        std::lock_guard<std::mutex> lock(created_cache_mutex_);
        RawBlockWrapper             wrapper = {std::move(item), handle};
        created_cache_[size_bytes].emplace_back(std::move(wrapper));
    }

    void clear()
    {
        std::lock_guard<std::mutex> lock(created_cache_mutex_);
        created_cache_.clear();
    }
};

SHMAllocator::SHMAllocator()
    : allocation_cache_(stdx::make_unique<AllocationCache>()), load_cache_(stdx::make_unique<LoadCache>())
{
}

SHMAllocator::~SHMAllocator() = default;

std::shared_ptr<void> SHMAllocator::allocate_shm(size_t size_bytes, SHMBlockID &block_id)
{
    // The id to return to the caller
    SHMBlockIDInternal id;

    // The underlying raw block
    std::shared_ptr<void> raw_block;

    // Include the size of our metadata
    auto requested_size = size_bytes + sizeof(SHMBlockInternal);

    // Maybe get a raw block of the requested size from the cache
    allocation_cache_->maybe_get_and_pop(requested_size, raw_block, id);

    // If we didn't get anything from the cache
    if (raw_block == nullptr)
    {
        // Create a block of the requested size
        raw_block = allocator_.allocate_shm(requested_size, id.block_handle);

        // Get a pointer to the struct and initialize it
        auto *block = new (raw_block.get()) SHMBlockInternal;

        // Increment the refcount
        block->refcount++;
    }

    // Return the ID of this block to the caller
    memcpy(block_id.data(), &id, sizeof(id));

    // Get the block from the raw block
    auto *block = static_cast<SHMBlockInternal *>(raw_block.get());

    // Create a shared pointer to the underlying data with a custom deleter
    // that keeps the block alive. Add the block to the cache on destruction.
    return std::shared_ptr<void>(
        block->data, [this, block, raw_block = std::move(raw_block), requested_size, id](void *unused) mutable {
            {
                // Lock the block's mutex
                ipc::scoped_lock<ipc::interprocess_mutex> lock(block->mutex);

                // Decrement the refcount
                block->refcount--;
            }

            // Add it to the created cache
            allocation_cache_->insert(requested_size, id.block_handle, std::move(raw_block));
        });
}

std::shared_ptr<void> SHMAllocator::load_shm(const SHMBlockID &block_id)
{
    // Load an existing block of shared memory by ID
    auto id     = reinterpret_cast<const SHMBlockIDInternal *>(block_id.data());
    auto handle = id->block_handle;

    // The underlying raw block
    std::shared_ptr<void> raw_block;

    // Check if we can get it from the cache
    load_cache_->maybe_get_and_pop(handle, raw_block);

    // If not, load it from scratch
    if (raw_block == nullptr)
    {
        raw_block = allocator_.load_shm(handle);
    }

    // Get the block from the raw block
    auto *block = static_cast<SHMBlockInternal *>(raw_block.get());

    // Sanity checks to avoid race conditions + increment the refcount
    {
        // Lock the block's mutex
        ipc::scoped_lock<ipc::interprocess_mutex> lock(block->mutex);

        // Sanity check
        if (block->refcount == 0)
        {
            // This means that the other process isn't keeping references to data long enough for this
            // process to load the data.
            // This can lead to some hard to debug race conditions so we always throw an error.
            NEUROPOD_ERROR("Tried getting a pointer to an existing chunk of memory that has a refcount of zero: {}",
                           handle);
        }

        // Make sure the `reuse_count` matches what we expect
        if (id->reuse_count != block->reuse_count)
        {
            // This means that the other process isn't keeping references to data long enough for this
            // process to load the data.
            // (A block of memory went out of scope and was reused before it could be loaded by the other process)
            NEUROPOD_ERROR("Tried loading a block of memory that went out of scope in the creating process. "
                           "Ensure that blocks of SHM stay in scope until they are loaded by the receiving process. "
                           "UUID: {}. Reuse count: {}. Expected reuse count: {}",
                           handle,
                           block->reuse_count,
                           id->reuse_count);
        }

        // Increment the refcount
        block->refcount++;
    }

    // Create a shared pointer to the underlying data with a custom deleter
    // that keeps the block alive. Add the block to the cache on destruction.
    return std::shared_ptr<void>(block->data,
                                 [this, block, raw_block = std::move(raw_block), handle](void *unused) mutable {
                                     {
                                         // Lock the block's mutex
                                         ipc::scoped_lock<ipc::interprocess_mutex> lock(block->mutex);

                                         // Decrement the refcount
                                         block->refcount--;
                                     }

                                     // Add it to the load cache
                                     load_cache_->insert(handle, std::move(raw_block));
                                 });
}

void SHMAllocator::free_unused_shm_blocks()
{
    // Free all currently unused blocks in the caches
    allocation_cache_->clear();
    load_cache_->clear();
}

// A shared memory allocator that is used by the WireFormat and by SHMNeuropodTensor
// TODO(vip): Remove global allocator instance
SHMAllocator shm_allocator;

} // namespace neuropod
