/* eslint-disable jsx-a11y/anchor-is-valid */
import { useState, useEffect } from 'react'
import { Auth } from 'aws-amplify'
import { useRouter } from 'next/router'

import Link from 'next/link'
import { Formik, Form, Field, useFormikContext } from 'formik'

import { AuthNav } from '../components/layout/AuthNav'

import { serverErrorOptions } from '../components/errors/serverErrorOptions'

import {
  SignUpValues,
  SignUpStepOneSchema,
  SignUpStepTwoSchema,
} from '../components/formik/SignUpValidation'

import { AiOutlineEyeInvisible } from 'react-icons/ai'
import { AiOutlineEye } from 'react-icons/ai'

function SignUp() {
  const router = useRouter()

  const [data, setData] = useState(SignUpValues)
  const [currentStep, setCurrentStep] = useState(0)

  const handleNextStep = (newData, final = false) => {
    setData((prev) => ({ ...prev, ...newData }))

    if (final) {
      router.push('/signin')
      return
    }

    setCurrentStep((prev) => prev + 1)
  }

  const steps = [
    <StepOne next={handleNextStep} data={data} />,
    <StepTwo next={handleNextStep} data={data} />,
  ]

  return (
    <>
      <AuthNav />
      <div className="flex flex-col w-11/12 m-auto justify-center mt-9 md:mt-10 py-6 md:max-w-4xl">
        {steps[currentStep]}
      </div>
    </>
  )
}

const StepOne = (props) => {
  const [signingIn, setSigningIn] = useState(false)
  const [serverError, setServerError] = useState(null)
  const [isPasswordVisible, setIsPasswordVisible] = useState(false)

  const toggle = () => {
    const passwordInput = document.getElementById('password')
    if (passwordInput.type === 'password') {
      passwordInput.type = 'text'
      setIsPasswordVisible(true)
    } else {
      passwordInput.type = 'password'
      setIsPasswordVisible(false)
    }
  }

  const handleSubmit = async ({ username, password, email }) => {
    setSigningIn(true)
    try {
      await Auth.signUp({
        username,
        password,
        attributes: { email },
      })
      props.next({ username, password, email })
    } catch (err) {
      serverErrorOptions(err.code, setServerError)
      setSigningIn(false)

      console.error('error confirming account..', err)
    }
  }

  return (
    <div className="flex flex-col gap-10 md:flex-row-reverse">
      <section className="md:w-full flex gap-4 flex-col">
        <Formik
          validationSchema={SignUpStepOneSchema}
          initialValues={props.data}
          onSubmit={handleSubmit}
        >
          {({ errors, touched }) => (
            <Form className="flex gap-4 flex-col">
              <h1 className="font-bold text-white text-3xl md:text-left">
                Create an Account
              </h1>
              <div className="flex gap-3 flex-col">
                <label className="text-white text-xs" htmlFor="username">
                  Family/ Friend Group Name
                </label>
                <Field
                  name="username"
                  type="text"
                  className="focus-tertiary-ring transition-all rounded-md py-3 pl-3 border-2 "
                  placeholder="ahslandboys2000"
                />
                {serverError && (
                  <span className="text-error">{serverError}</span>
                )}
                {errors.username && touched.username ? (
                  <span className="text-sm text-error">{errors.username}</span>
                ) : null}
              </div>

              <div className="flex gap-3 flex-col">
                <label className="text-white text-xs" htmlFor="email">
                  Email Addresss
                </label>
                <Field
                  type="email"
                  name="email"
                  className="transition-all rounded-md py-3 pl-3 border-2 focus-tertiary-ring"
                  placeholder="anjay@gmail.com"
                />
                {errors.email && touched.email ? (
                  <span className="text-sm text-error">{errors.email}</span>
                ) : null}
              </div>

              <div className="flex gap-3 flex-col">
                <label className="text-white text-xs" htmlFor="password">
                  Password
                </label>
                <div className="relative w-full flex items-center">
                  <Field
                    name="password"
                    type="password"
                    placeholder="password"
                    id="password"
                    className="transition-all rounded-md py-3 pl-3 border-2 focus-tertiary-ring w-full"
                  />
                  {isPasswordVisible ? (
                    <button
                      type="button"
                      onClick={toggle}
                      className="absolute right-3 cursor-pointer p-1 transition-all"
                    >
                      <AiOutlineEye size="1.5em" color="grey" />
                    </button>
                  ) : (
                    <button
                      type="button"
                      onClick={toggle}
                      className="absolute right-3 cursor-pointer p-1 transition-all"
                    >
                      <AiOutlineEyeInvisible size="1.5em" color="grey" />
                    </button>
                  )}
                </div>
                {errors.password && touched.password ? (
                  <span className="text-sm text-error">{errors.password}</span>
                ) : null}
              </div>

              <button
                type="submit"
                disabled={signingIn ? true : false}
                className={`transition-all transform hover:translate-y-1 w-full rounded-md bg-tertiary py-3 mt-6 cursor-pointer border-2 border-transparent focus-tertiary-ring ${
                  signingIn ? 'opacity-50 cursor-wait' : 'opacity-100'
                }`}
              >
                {signingIn ? 'Loading...' : 'Next'}
              </button>
            </Form>
          )}
        </Formik>

        {/* Already have an account */}
        <div className="mt-6">
          <p className="font-body text-white">
            <span className="mr-2">Already have an account?</span>
            <Link href="/signin">
              <a className="text-quad cursor-pointer">Sign In</a>
            </Link>
          </p>
        </div>
      </section>

      {/* Marketing Section */}
      <section className="md:w-full ">
        <h2 className="text-white text-2xl mb-7">
          Track and view every family game night.
        </h2>
        <ul className="grid gap-5">
          <li className="grid gap-2 grid-cols-list">
            <span>✍️</span>
            <span className="text-white font-body font-bold">
              Document EVERY game. Easily track each game your family plays.
            </span>
          </li>
          <li className="grid gap-2 grid-cols-list">
            <span>📊</span>
            <span className="text-white font-body font-bold">
              Discover new insights. Uncover who has the most wins overall wins.
            </span>
          </li>
          <li className="grid gap-2 grid-cols-list">
            <span>📚</span>
            <span className="text-white font-body font-bold">
              Digitize your boardgame catalogue. Catagorize & store your board
              games.
            </span>
          </li>
        </ul>
      </section>
    </div>
  )
}

const StepTwo = (props) => {
  const [signingIn, setSigningIn] = useState(false)
  const [serverError, setServerError] = useState(null)
  const AutoSubmitToken = () => {
    // Grab values and submitForm from context
    const { values, submitForm } = useFormikContext()
    useEffect(() => {
      // Submit the form imperatively as an effect as soon as form values. confirmationCode is 6 digits long
      if (values.confirmationCode.length === 6) {
        submitForm()
      }
    }, [values, submitForm])
    return null
  }

  const handleSubmit = async ({ username, confirmationCode }) => {
    setSigningIn(true)
    try {
      await Auth.confirmSignUp(username, confirmationCode)
      props.next(username, true)
    } catch (err) {
      setSigningIn(false)
      setServerError(err.message)
      console.error('error confirming account..', err)
    }
  }

  return (
    <section className="w-11/12 m-auto max-w-md">
      <Formik
        validationSchema={SignUpStepTwoSchema}
        initialValues={props.data}
        onSubmit={handleSubmit}
      >
        {({ errors, touched }) => (
          <Form className="md:w-full flex gap-2 flex-col">
            <h1 className="font-bold text-center text-white text-3xl">
              Enter your Validation Code
            </h1>
            <p className="text-white text-sm text-center mt-4 mb-8 font-body">
              You were just emailed a validation code. Please enter it below to
              confrim you account
            </p>
            <div className="mt-4 flex gap-4 flex-col">
              <label className="text-white text-xs" htmlFor="authCode">
                Authentication Code
              </label>
              <Field
                name="confirmationCode"
                type="text"
                className="transition-all rounded-md py-3 pl-3 border-2 focus-tertiary-ring"
                placeholder="123456"
              />
              <AutoSubmitToken />
              {serverError && <span className="text-error">{serverError}</span>}
              {errors.confirmationCode && touched.confirmationCode ? (
                <span className="text-sm text-error">
                  {errors.confirmationCode}
                </span>
              ) : null}
            </div>
            <button
              type="submit"
              disabled={signingIn ? true : false}
              className={`transition-all transform hover:translate-y-1 rounded-md bg-tertiary py-3 mt-6 cursor-pointer border-2 border-transparent focus-tertiary-ring ${
                signingIn ? 'opacity-50 cursor-wait' : 'opacity-100'
              }`}
            >
              {signingIn ? 'Confirming...' : 'Submit'}
            </button>
          </Form>
        )}
      </Formik>
    </section>
  )
}

export default SignUp
