/* eslint-disable jsx-a11y/anchor-is-valid */
import { useState } from 'react'
import { Auth } from 'aws-amplify'
import { useRouter } from 'next/router'

import Link from 'next/link'
import { Formik, Form, Field } from 'formik'

import { AuthNav } from '../components/layout/AuthNav'

import { serverErrorOptions } from '../components/errors/serverErrorOptions'

import {
  ForgotPasswordValues,
  ForgotPasswordStepOneSchema,
  ForgotPasswordStepTwoSchema,
} from '../components/formik/ForgotPasswordValidation'

function ForgotPassword() {
  const router = useRouter()

  const [data, setData] = useState(ForgotPasswordValues)
  const [currentStep, setCurrentStep] = useState(0)

  const makeRequest = (formData) => {
    router.push('/signin')
  }

  const handleNextStep = (newData, final = false) => {
    setData((prev) => ({ ...prev, ...newData }))

    if (final) {
      makeRequest(newData)
      return
    }

    setCurrentStep((prev) => prev + 1)
  }

  const steps = [
    <StepOne next={handleNextStep} data={data} />,
    <StepTwo next={handleNextStep} data={data} />,
  ]

  return (
    <>
      <AuthNav />
      <div className="flex flex-col w-11/12 m-auto justify-center mt-9 pb-6 md:max-w-4xl">
        {steps[currentStep]}
      </div>
    </>
  )
}

const StepOne = (props) => {
  const [signingIn, setSigningIn] = useState(false)
  const [serverError, setServerError] = useState()

  const handleSubmit = async ({ username, password, email }) => {
    setSigningIn(true)
    try {
      await Auth.forgotPassword(username)
      props.next({ username, password, email })
    } catch (err) {
      serverErrorOptions(err.code, setServerError)
      setSigningIn(false)

      console.error('error confirming account..', err)
    }
  }

  return (
    <div className="flex flex-col w-11/12 m-auto justify-center mt-9 md:mt-14 py-6 md:max-w-md">
      <section className="md:w-full flex gap-4 flex-col">
        <Formik
          validationSchema={ForgotPasswordStepOneSchema}
          initialValues={props.data}
          onSubmit={handleSubmit}
        >
          {({ errors, touched }) => (
            <Form className="flex gap-4 flex-col">
              <h1 className="font-bold text-white text-3xl md:text-left">
                Reset Password
              </h1>
              <div className="flex gap-3 flex-col">
                <label className="text-white text-xs" htmlFor="username">
                  Account name
                </label>
                <Field
                  name="username"
                  type="text"
                  className="transition-all rounded-md py-3 pl-3 border-2 focus-tertiary-ring"
                  placeholder="username"
                />

                {serverError && (
                  <span className="text-error">{serverError}</span>
                )}
                {errors.username && touched.username ? (
                  <span className="text-sm text-error">{errors.username}</span>
                ) : null}
              </div>

              <button
                type="submit"
                disabled={signingIn ? true : false}
                className={`transition-all transform hover:translate-y-1 w-full rounded-md bg-tertiary py-3 mt-6 cursor-pointer border-2 border-transparent focus-tertiary-ring ${
                  signingIn ? 'opacity-50 cursor-wait' : 'opacity-100'
                }`}
              >
                {signingIn ? 'Loading...' : 'Next'}
              </button>
            </Form>
          )}
        </Formik>

        {/* Already have an account */}
        <div className="mt-6">
          <p className="font-body text-white">
            <span className="mr-2">Remembered your password?</span>
            <Link href="/signin">
              <a className="text-quad cursor-pointer">Sign In</a>
            </Link>
          </p>
        </div>
      </section>
    </div>
  )
}

const StepTwo = (props) => {
  const router = useRouter()

  const [signingIn, setSigningIn] = useState(false)
  const [serverError, setServerError] = useState(null)

  const handleSubmit = async ({ username, confirmationCode, newPassword }) => {
    setSigningIn(true)
    console.log(username, confirmationCode, newPassword)
    try {
      await Auth.forgotPasswordSubmit(username, confirmationCode, newPassword)
      router.push('/signin')
    } catch (err) {
      setSigningIn(false)
      setServerError(err.message)
      console.error('error confirming account..', err)
    }
  }

  return (
    <section className="w-11/12 m-auto max-w-md">
      <Formik
        validationSchema={ForgotPasswordStepTwoSchema}
        initialValues={props.data}
        onSubmit={handleSubmit}
      >
        {({ errors, touched }) => (
          <Form className="md:w-full flex gap-2 flex-col">
            <h1 className="font-bold text-center text-white text-3xl">
              Reset your Password
            </h1>
            <p className="text-white text-sm text-center mt-4 mb-8 font-body">
              You were just emailed a validation code. Please enter it below to
              confrim you account
            </p>
            {/* Confirmation Code */}
            <div className="mt-4 flex gap-4 flex-col">
              <label className="text-white text-xs" htmlFor="confirmationCode">
                Confirmation Code
              </label>
              <Field
                name="confirmationCode"
                type="text"
                className="focus:ring-tertiary transition-all rounded-md py-3 pl-3 border-2 focus-tertiary-ring"
                placeholder="123456"
              />
              {serverError && <span className="text-error">{serverError}</span>}
              {errors.confirmationCode && touched.confirmationCode ? (
                <span className="text-sm text-error">
                  {errors.confirmationCode}
                </span>
              ) : null}
            </div>
            {/* New password */}
            <div className="mt-4 flex gap-4 flex-col">
              <label className="text-white text-xs" htmlFor="newPassword">
                New Password
              </label>
              <Field
                name="newPassword"
                type="text"
                className="focus:ring-tertiary transition-all rounded-md py-3 pl-3 border-2 focus-tertiary-ring"
                placeholder="qwerty123"
              />
              {serverError && <span className="text-error">{serverError}</span>}
              {errors.newPassword && touched.newPassword ? (
                <span className="text-sm text-error">{errors.newPassword}</span>
              ) : null}
            </div>
            <button
              type="submit"
              disabled={signingIn ? true : false}
              className={`transition-all transform hover:translate-y-1 rounded-md bg-tertiary py-3 mt-6 cursor-pointer border-2 border-transparent focus:outline-none focus:ring-2 focus:ring-tertiary focus:border-transparent ${
                signingIn ? 'opacity-50 cursor-wait' : 'opacity-100'
              }`}
            >
              {signingIn ? 'Loading...' : 'Submit'}
            </button>
          </Form>
        )}
      </Formik>
    </section>
  )
}

export default ForgotPassword
