import Foundation

/// The status of one running task.
///
/// A `TaskStatus` can be updated over time in the context of the task it
/// represents to notify subscribers that the status of the task has changed.
///
/// You must tell the application about the task status for it to be reported in the
/// UI. To do this, the task must be registered in the application's task status
/// reporter using ``TaskStatusReporter/register(taskStatus:)``.
final class TaskStatus: Identifiable, ObservableObject {
    typealias ID = String

    let id: ID
    let displayName: String

    @Published private(set) var state: TaskState

    init(displayName: String, initialState state: TaskState) {
        id = UUID().uuidString
        self.displayName = displayName
        self.state = state
    }

    /// Updates the state of the task.
    /// - Parameter state: The new state of the task.
    @MainActor func update(state: TaskState) {
        self.state = state
    }

    /// Convenience for setting the state of the task to ``TaskState/done``.
    @MainActor func markAsDone(
        message: String? = nil
    ) {
        update(state: .done(message: message))
    }
}

// MARK: - Hashable

extension TaskStatus: Hashable {
    static func == (lhs: TaskStatus, rhs: TaskStatus) -> Bool {
        lhs.id == rhs.id
    }

    func hash(into hasher: inout Hasher) {
        hasher.combine(id)
    }
}
