# frozen_string_literal: true

require_relative 'base_command'
require_relative '../../examiner'
require_relative '../../configuration/app_configuration'

module Reek
  module CLI
    module Command
      #
      # A command to collect smells from a set of sources and writes a configuration
      # file that can serve as a todo list.
      #
      class TodoListCommand < BaseCommand
        HEADER = "# Auto generated by Reeks --todo flag\n"
        EXISTING_FILE_MESSAGE = "\nExisting '#{DEFAULT_CONFIGURATION_FILE_NAME}' detected - aborting.\n".freeze
        NO_SMELLS_FOUND_MESSAGE = "\nNo smells found - nothing to do, exiting.\n"

        def execute
          if smells.empty?
            puts NO_SMELLS_FOUND_MESSAGE
          elsif File.exist?(DEFAULT_CONFIGURATION_FILE_NAME)
            puts EXISTING_FILE_MESSAGE
          else
            write_to_file
            puts "\n'#{DEFAULT_CONFIGURATION_FILE_NAME}' generated! " \
                 'You can now use this as a starting point.'
          end
          options.success_exit_code
        end

        private

        def smells
          @smells ||= sources.map do |source|
            Examiner.new(source, filter_by_smells: smell_names)
          end.map(&:smells).flatten
        end

        def groups
          @groups ||=
            begin
              todos = DetectorRepository.smell_types.map do |smell_class|
                smells_for_class = grouped_smells[smell_class.smell_type] or next
                smell_class.todo_configuration_for(smells_for_class)
              end
              todos.compact.inject(&:merge)
            end
        end

        def grouped_smells
          @grouped_smells ||= smells.group_by(&:smell_type)
        end

        # :reek:FeatureEnvy
        def write_to_file
          File.open(DEFAULT_CONFIGURATION_FILE_NAME, 'w') do |configuration_file|
            configuration_file.write HEADER
            configuration_file.write({ DETECTORS_KEY => groups }.to_yaml)
          end
        end
      end
    end
  end
end
