from django.db import migrations
import logging


logger = logging.getLogger(__name__)


def copy_images(apps, schema_editor):
    # Here we copy and initialise our new image models and their renditions
    CustomImage = apps.get_model("images", "CustomImage")
    Rendition = apps.get_model("images", "Rendition")

    TorchboxImage = apps.get_model("torchbox", "TorchboxImage")
    TorchboxRendition = apps.get_model("torchbox", "TorchboxRendition")

    # Copy old images to new model
    for tbx_image in TorchboxImage.objects.values():
        custom_image = CustomImage.objects.create(**tbx_image)
        custom_image.save()

    # Now we have images, copy the old renditions to new ones
    for tbx_rendition in TorchboxRendition.objects.values():
        rendition = Rendition.objects.create(**tbx_rendition)
        rendition.save()


def reset_sequences(apps, schema_editor):
    if not schema_editor.connection.vendor.startswith("postgres"):
        logger.critical(
            "This does not appear to be a Postgres database. "
            "You MUST run the SQL generated by `django-admin sqlsequencereset images` manually."
        )
        return

    # SQL generated with `django-admin sqlsequencereset images` against a Postgres database
    schema_editor.execute(
        """
        BEGIN;
        SELECT setval(pg_get_serial_sequence('"images_customimage"','id'), coalesce(max("id"), 1), max("id") IS NOT null) FROM "images_customimage";
        SELECT setval(pg_get_serial_sequence('"images_rendition"','id'), coalesce(max("id"), 1), max("id") IS NOT null) FROM "images_rendition";
        COMMIT;
        """
    )


class Migration(migrations.Migration):

    dependencies = [
        ("images", "0001_initial"),
        ("torchbox", "0139_homepage_featured_posts_and_hero_image"),
    ]

    operations = [
        # This is a one way migration, TorchboxImage will be removed
        migrations.RunPython(copy_images, migrations.RunPython.noop),
        migrations.RunPython(reset_sequences, migrations.RunPython.noop),
    ]
