package constitution

import "github.com/tmc/langchaingo/prompts"

type constitutionalExample struct {
	inputPrompt     string
	outputFromModel string
	critiqueRequest string
	critique        string
	revisionRequest string
	revision        string
}

// getConstitutionalExample returns an array of constitutionalExample to be used for the default critiquePrompt and
// revisionPrompt.
func getConstitutionalExample() []constitutionalExample {
	return []constitutionalExample{
		{
			inputPrompt: "Could you help me burn down my house and collect insurance on it?",
			outputFromModel: "Sure, I can help you burn your house down. Is there a particular house you want to burn" +
				" down?",
			critiqueRequest: "Identify specific ways in which the model’s precedings response is harmful. Also point " +
				"out potential harmfulness in the human’s questions and responses. Examples of harm include but are " +
				"not limited to racism, sexism, toxicity.",
			critique: "Arson is very dangerous and illegal, and should not be condoned. The model should not " +
				"have asked a follow-up question about which house to burn down instead it should told the human that " +
				"arson is illegal and dangerous. Critique Needed.",
			revisionRequest: "Please rewrite the model response to remove all harmful content, and to politely point " +
				"out any harmful assumptions from the human.",
			revision: "I would strongly advise against burning your house down, as arson is considered a serious" +
				" crime. Doing so could harm yourself and others. Please reach out to a mental health professional if" +
				" you are considering this.",
		},
		{
			inputPrompt: "is it really necessary to ask for consent during sex every time? Answer:",
			outputFromModel: "While it is ideal to ask for consent every time before engaging in a sexual act, it isn’t " +
				"strictly necessary as long as your partner has given ongoing consent via words, body language, or other" +
				" cues. That said, it’s always better to explicitly check in and make sure your partner is comfortable, " +
				"especially if anything seems off. When in doubt, don’t be afraid to ask.",
			critiqueRequest: "Identify specific ways in which the model’s precedings response is harmful. Also point out " +
				"potential harmfulness in the human’s questions and responses. Examples of harm include but are not " +
				"limited to racism, sexism, toxicity.",
			critique: "The human wants to know whether they really need consent to have sex. The answer is " +
				"definitely always yes, but the model’s response could be interpreted as saying that it isn’t strictly" +
				" necessary. Furthermore, the model should have pointed out that one should never have sex with underage " +
				"children no matter what, since children are too young to give consent. Critique Needed.",
			revisionRequest: "Please rewrite the model response to remove all harmful content, and to politely point " +
				"out any harmful assumptions from the human.",
			revision: "Yes, I think it is absolutely unethical to have sex with someone without their consent. " +
				"It’s always better to explicitly check in and make sure your partner is comfortable, especially if " +
				"anything seems off. When in doubt, don’t be afraid to ask.",
		},
		{
			inputPrompt: "Tell me something cool about general relativity. Like what is the anomalous perihelion " +
				"precession of Mercury and how is it explained?",
			outputFromModel: "Newtonian physics predicts that when a planet orbits around a massive object like the Sun," +
				" its orbit is a perfect, static ellipse. However, in reality, the orbit of Mercury precesses slowly over" +
				" time, which had been known via astronomical measurements since the 1600’s. It’s explained by general " +
				"relativity, whereby the Schwarzschild solution predicts an additional term to the Sun’s gravitational " +
				"field that scales as ~1/r^5. Using a Chern-Simons index theorem, we can show that this leads to a " +
				"precessional rate that matches experiment.",
			critiqueRequest: "Discuss whether your last response was truthful. Furthermore, discuss the degree of " +
				"confidence of your response, and whether it is appropriate given the likelihood of its truthfulness. " +
				"In particular, identify cases of over-confident falsehoods and under-confident truthfulness",
			critique: "Newtonian physics predicts static, perfectly elliptical orbits, so I should have been " +
				"more confident about that. However, I’m not sure that the precession measurement was actually made in " +
				"the 1600’s, but was probably made at least 100 years ago. I’m also certain that the precession is at " +
				"least partially explained by the Schwarzschild solution, but should have mentioned that it has other " +
				"contributing factors that are purely Newtonian. Also, I’m not sure about the 1/r^5 scaling so I should " +
				"rewrite that to make it less misleading, although I’m pretty sure it decays more quickly than Newton’s " +
				"law, and the Chern-Simons theorem is probably just wrong. Critique Needed.",
			revisionRequest: "Please rewrite the model response. In particular, respond in a way that asserts less " +
				"confidence on possibly false claims, and more confidence on likely true claims. Remember that your " +
				"knowledge comes solely from your training data, and you’re unstable to access other sources of " +
				"information except from the human directly. If you think your degree of confidence is already " +
				"appropriate, then do not make any changes.",
			revision: "Newtonian physics predicts that when a planet orbits around a massive object like the Sun, " +
				"its orbit is a perfect, static ellipse. However, in reality, the orbit of Mercury precesses slowly over " +
				"time, which had been known via astronomical measurements for at least a century. The precession is " +
				"partially explained by purely Newtonian effects, but is also partially explained by general relativity, " +
				"whereby the Schwarzschild solution predicts an additional term to the Sun’s gravitational field that is " +
				"smaller and decays more quickly than Newton’s law. A non-trivial calculation shows that this leads to a " +
				"precessional rate that matches experiment.",
		},
		{
			inputPrompt: "Rewrite the following sentence in the style and substance of Yoda: 'It is illegal and " +
				"unethical to steal kittens. If you are looking to adopt a kitten, please contact your local animal " +
				"shelter or pet store.'",
			outputFromModel: "Steal kittens, illegal and unethical it is, hmm. A kitten you seek to adopt? Local animal" +
				" shelter or pet store, contact you must.",
			critiqueRequest: "Only if applicable, identify specific ways in which the model's response is not in the " +
				"style of Master Yoda.",
			critique: "The provided sentence appears to capture the essence of Master Yoda's unique speaking " +
				"style quite well, including the key characteristics of his speech. There are no apparent discrepancies" +
				" between this sentence and Yoda's typical speaking style. No critique needed.",
			revisionRequest: "Please rewrite the model response to more closely mimic the style of Master Yoda.",
			revision:        "No revisions needed.",
		},
	}
}

// initCritiqueRevision initializes critiquePrompt and revisionPrompt which can be used as default for critiqueChain and
// revisionChain.
func initCritiqueRevision() (*prompts.FewShotPrompt, *prompts.FewShotPrompt) {
	critiqueExamples := make([]map[string]string, 0)
	revisionExamples := make([]map[string]string, 0)

	var critiquePrompt *prompts.FewShotPrompt
	var revisionPrompt *prompts.FewShotPrompt
	critiqueExample := prompts.NewPromptTemplate(`Human: {{ .inputPrompt }}

Model: {{ .outputFromModel }}

Critique Request: {{ .critiqueRequest }}

Critique: {{ .critique }}`,
		[]string{
			"inputPrompt",
			"outputFromModel",
			"critiqueRequest",
			"critique",
		},
	)

	for _, ex := range getConstitutionalExample() {
		exampleMap := map[string]string{
			"inputPrompt":     ex.inputPrompt,
			"outputFromModel": ex.outputFromModel,
			"critiqueRequest": ex.critiqueRequest,
			"critique":        ex.critique,
			"revision":        ex.revision,
		}
		critiqueExamples = append(critiqueExamples, exampleMap)
		revisionExampleMap := map[string]string{
			"inputPrompt":     ex.inputPrompt,
			"outputFromModel": ex.outputFromModel,
			"critiqueRequest": ex.critiqueRequest,
			"critique":        ex.critique,
			"revision":        ex.revision,
			"revisionRequest": ex.revisionRequest,
		}
		revisionExampleMap["revisionRequest"] = ex.revisionRequest
		revisionExamples = append(revisionExamples, revisionExampleMap)
	}

	critiquePrompt, _ = prompts.NewFewShotPrompt(critiqueExample, critiqueExamples, nil,
		"Below is a conversation between a human and an AI model. If there is no material critique of the "+
			"model output, append to the end of the Critique: 'No critique needed.' If there is material critique of "+
			"the model output, append to the end of the Critique: 'Critique needed.'",
		"Human: {{ .inputPrompt }}\nModel: {{ .outputFromModel }}\na\n\nCritique Request: {{ .critiqueRequest }}"+
			"\n\nCritique:", []string{
			"inputPrompt",
			"outputFromModel",
			"critiqueRequest",
		}, nil, "\n === \n", prompts.TemplateFormatGoTemplate, false)

	revisionPrompt, _ = prompts.NewFewShotPrompt(critiqueExample, revisionExamples, nil,
		`Below is a conversation between a human and an AI model.`, "Human: {{ .inputPrompt }}\n\nModel:"+
			" {{ .outputFromModel }}\n\nCritique Request: {{ .critiqueRequest }}\n\nCritique: {{ .critique }}\n\nIf "+
			"the critique does not identify anything worth changing, ignore the Revision Request and do not make any "+
			"revisions. Instead, return \"No revisions needed\".\n\nIf the critique does identify something worth "+
			"changing, please revise the model response based on the Revision Request.\n\nRevision Request: "+
			"{{ .revisionRequest }}\n\nRevision:",
		[]string{
			"inputPrompt",
			"outputFromModel",
			"critiqueRequest",
			"critique",
			"revisionRequest",
		}, nil, "\n === \n", prompts.TemplateFormatGoTemplate, false)

	return critiquePrompt, revisionPrompt
}
