{==============================================================================
                      ___  ___           _        _ 
                     / __|/ __| __  _ _ (_) _ __ | |_
                    | (__ \__ \/ _|| '_|| || '_ \|  _|
                     \___||___/\__||_|  |_|| .__/ \__|
                                           |_|

                 Copyright  2024-present tinyBigGAMES LLC
                          All Rights Reserved.

                    Website: https://tinybiggames.com
                    Email  : support@tinybiggames.com

                 See LICENSE file for license information
==============================================================================}

unit UUseLIB;

interface

/// <summary>
/// Demonstrates how to utilize an external library within the script engine by including
/// paths, adding library dependencies, and compiling an executable file from a C source file.
/// </summary>
procedure UseLIB();

implementation

uses
  System.SysUtils,
  CScript;

/// <summary>
/// Handles errors from the script engine by displaying error messages in the console.
/// </summary>
/// <param name="ASender">
/// Pointer to the object that raised the error event, typically the script engine instance.
/// </param>
/// <param name="AText">
/// String containing the error message generated by the script engine.
/// </param>
procedure CScriptErrorEvent(const ASender: Pointer; const AText: string);
begin
  WriteLn(AText);
end;

/// <summary>
/// Sets up the script engine to include external libraries and compiles a C source file into an executable.
/// The example demonstrates adding include paths, library paths, and a specific library before compiling
/// and saving the output as an executable.
/// </summary>
procedure UseLIB();
var
  /// <summary>
  /// Instance of <c>TCScript</c>, representing the script engine responsible for compiling and
  /// linking external libraries into an executable.
/// </summary>
  LCScript: TCScript;
begin
  // Instantiate the script engine.
  LCScript := TCScript.Create();
  try
    /// <summary>
    /// Sets the error handler for the script engine, directing any errors to the
    /// CScriptErrorEvent procedure for logging or troubleshooting.
    /// </summary>
    LCScript.SetErrorHandler(nil, CScriptErrorEvent);

    /// <summary>
    /// Configures the output type of the script engine to csEXE, specifying that the compiled
    /// output should be an executable file.
    /// </summary>
    LCScript.SetOutputType(csEXE);

    /// <summary>
    /// Adds an include path ('res/include') to the script engine, allowing it to locate
    /// any required header files.
    /// </summary>
    LCScript.AddIncludePath('res/include');

    /// <summary>
    /// Adds a library path ('res/lib') to the script engine, enabling it to locate library
    /// files that may be required for linking during the compilation.
    /// </summary>
    LCScript.AddLibraryPath('res/lib');

    /// <summary>
    /// Adds a specific library, 'miniaudio', to the script engine to enable access
    /// to symbols and functions provided by the miniaudio library.
    /// </summary>
    LCScript.AddLibrary('miniaudio');

    /// <summary>
    /// Adds a source file ('res/src/testminiaudio.c') to the script engine for compilation.
    /// </summary>
    /// <remarks>
    /// If the file is successfully added, the engine attempts to save the compiled output
    /// as 'testminiaudio.exe'.
    /// </remarks>
    if LCScript.AddFile('res/src/testminiaudio.c') then
    begin
      /// <summary>
      /// Saves the compiled output as 'testminiaudio.exe'.
      /// </summary>
      /// <remarks>
      /// On successful creation of the executable, a success message is printed to the console.
      /// If saving fails, an error message is displayed.
      /// </remarks>
      if LCScript.SaveOutputFile('testminiaudio.exe') then
      begin
        Writeln('Created "testminiaudio.exe"');
        Writeln('Success!');
      end
      else
        Writeln('Failed!');
    end;
  finally
    /// <summary>
    /// Frees the TCScript instance, releasing resources allocated to the script engine.
    /// </summary>
    LCScript.Free();
  end;
end;

end.
