{==============================================================================
                      ___  ___           _        _ 
                     / __|/ __| __  _ _ (_) _ __ | |_
                    | (__ \__ \/ _|| '_|| || '_ \|  _|
                     \___||___/\__||_|  |_|| .__/ \__|
                                           |_|

                 Copyright  2024-present tinyBigGAMES LLC
                          All Rights Reserved.

                    Website: https://tinybiggames.com
                    Email  : support@tinybiggames.com

                 See LICENSE file for license information
==============================================================================}

unit UAddSymbol;

interface

/// <summary>
/// Demonstrates adding custom symbols, such as functions and constants, to the script interpreter.
/// This procedure initializes a <c>TCScript</c> instance, configures error handling, and
/// registers both a function and a constant string to the script environment.
/// </summary>
procedure AddSymbol();

implementation

uses
  System.SysUtils,
  CScript;

/// <summary>
/// Handles errors from the CScript interpreter by displaying error messages in the console.
/// </summary>
/// <param name="ASender">
/// A pointer to the sender object that raised the error event, generally the script engine.
/// </param>
/// <param name="AText">
/// The error message text generated by the script engine.
/// </param>
procedure CScriptErrorEvent(const ASender: Pointer; const AText: string);
begin
  WriteLn(AText);
end;

/// <summary>
/// Adds two integer values and returns the result.
/// </summary>
/// <param name="a">
/// The first integer to add.
/// </param>
/// <param name="b">
/// The second integer to add.
/// </param>
/// <returns>
/// The sum of <paramref name="a"/> and <paramref name="b"/>.
/// </returns>
/// <remarks>
/// This function uses the cdecl calling convention to match the expected symbol convention
/// for the script engine.
/// </remarks>
function Add(a: integer; b: integer): integer; cdecl;
begin
  Result := a + b;
end;

/// <summary>
/// Demonstrates how to add symbols, such as a function and a constant, to the script environment.
/// Configures the script engine with an error handler, sets up memory-based output,
/// and registers custom symbols for use within the script.
/// </summary>
procedure AddSymbol();
const
  /// <summary>
  /// A constant string symbol, representing a static message to be used within the script.
  /// </summary>
  CHello = 'Hello World!';
var
  /// <summary>
  /// Instance of <c>TCScript</c>, representing the script engine. This instance is used
  /// to add files, register symbols, and execute script-related functionality.
/// </summary>
  LCScript: TCScript;

  /// <summary>
  /// Function pointer to a function symbol named 'foo', defined within the script.
  /// This function takes an integer argument and returns an integer result using cdecl calling convention.
/// </summary>
  LFoo: function(n: integer): integer; cdecl;
begin
  // Create a new instance of the script engine.
  LCScript := TCScript.Create();
  try
    /// <summary>
    /// Sets the error handler for the script engine, assigning CScriptErrorEvent to handle any
    /// errors and display messages through the console.
    /// </summary>
    LCScript.SetErrorHandler(nil, CScriptErrorEvent);

    /// <summary>
    /// Configures the output type of the script engine to csMEMORY, ensuring all output
    /// is stored in memory for efficient symbol retrieval and execution.
    /// </summary>
    LCScript.SetOutputType(csMEMORY);

    /// <summary>
    /// Adds an external source file ('res/src/test02.c') to the script engine. This source file
    /// can contain additional code and definitions to be utilized within the script.
    /// </summary>
    LCScript.AddFile('res/src/test02.c');

    /// <summary>
    /// Registers the 'add' function symbol in the script environment, enabling it to be accessed
    /// from within the script. The symbol is linked to the Add function defined in Delphi.
    /// </summary>
    LCScript.AddSymbol('add', @Add);

    /// <summary>
    /// Adds a string constant symbol named 'hello' to the script engine, making the 'Hello World!'
    /// message accessible within the script as a symbol.
    /// </summary>
    LCScript.AddSymbol('hello', PUTF8Char(CHello));

    /// <summary>
    /// Relocates all symbols in memory, resolving addresses to prepare for script execution.
    /// </summary>
    LCScript.Relocate();

    /// <summary>
    /// Retrieves the 'foo' function symbol from the script. If successfully assigned,
    /// calls 'foo' with the integer argument 32.
    /// </summary>
    LFoo := LCScript.GetSymbol('foo');
    if Assigned(LFoo) then
      LFoo(32);
  finally
    /// <summary>
    /// Frees the script engine instance, releasing resources and ensuring proper memory management.
    /// </summary>
    LCScript.Free();
  end;
end;

end.
