
"use strict"


/*!md
# Project Control
A control server is one which provides the business logic as to how to handle a phone call(s). When a call comes into our SIP server, the SIP server passes this off to the control server and then goes of to do 'other things'. When we have instructions as to what to do with the call in question, we then make a HTTP request on the SIP server with relevent information.

This package provides utilities to handle requests from our SIP server and then provides functions to be able to respond to them.

```
         SIP Server
         |       |
         |       |
  RTP Server----Control Server (this)
```

When the SIP server requires a call to be handled it passes the call off to the control server. This processes any business logic. Then will pass back any information back to the SIP server (via a fresh HTTP request).


This file provides two classes. projectcontrol and call.

## projectcontrol

### onnewcall

Setter to set a callback to indicate a new call.

Example

```js
projectcontrol.onnewcall = ( call ) =>
{
  call.ring()
}
```

As the '=>' syntax alters the this pointer we pass in the call object. We also support:

```js
projectcontrol.onnewcall = function()
{
  this.ring()
}
```

### sipserver
Make a request to the SIP server (i.e. pass a HTTP request to the SIP server).

### invite
Request the SIP server to perform an INVITE.

### run
After setting up our control server listen for new connections.

```
projectcontrol.run()
```

### directory
The SIP server stores user information in memory. This information is pushed to it by the control server (or other server).

Information is passed to the SIP server via a HTTP request (like all events/traffic) in the following format.
```
HTTP PUT /dir/bling.babblevoice.com
{
  "control": "http://127.0.0.1:9001",
  "users":
  [
    { "username": "1003", "secret": "mysecret"}
  ]
}
```

To pass this information to the SIP server use this function:
```
projectcontrol.directory( "bling.babblevoice.com", [{ "username": "1003", "secret": "1123654789" } ] )
```

## class call
Represents a call which can be controlled.

### onringing
Setter to set a callback to be called when a ringing signal is received in regards to this call.

### onanswer

Setter to set a callback to be called when an answer signal (a call is answered) is received in regards to this call.

### onhangup
Setter to set a callback to be called when a hangup signal is received in regards to this call.

### ringing
Getter to see if the call is ringing.

Example
```
if( call.ringing )
{
  call.answer()
}
```

### answered

### hungup

### error
If an error occures then an error object will be set on the call object

## TODO

- [] Finish off onhangup

*/

const call = require( "./call.js" )
const http = require( "http" )
const url = require( "url" )

class projectcontrol
{
  constructor()
  {
    this.onregcallbacks = []
    this.onderegcallbacks = []
    this.onnewcallcallbacks = []
    this.onhangupcallbacks = []
    this.onreportingcallbacks = []

    this.handlers = { 'PUT': {}, 'POST': {}, 'DELETE': {}, 'GET': {} }

    /* Store by call-id */
    this.calls = {}

    /* Reverse channel uuid to call id */
    this.channels = {}

    /* TODO - work out if we need more and how to add. */
    this.sip = {}
    this.sip.host = "127.0.0.1"
    this.sip.port = 9000

    this.us = {}
    this.us.host = "127.0.0.1"
    this.us.port = 9001

    this.rtp = {}
    this.rtp.host = "127.0.0.1"
    this.rtp.port = 9002

    this.codecs = [ "pcma", "pcmu", "722", "ilbc",  "2833" ]
    this.sessionidcounter = 1

    this.gateways = []
    this.mohsoup = {}

    this.handlers.GET.calls = (  pathparts, req, res, body  ) =>
    {
      var callstr = JSON.stringify( Object.keys( this.calls ) )
      res.writeHead( 200, { "Content-Length": callstr.length } )
      res.write( callstr )
    }

    this.handlers.PUT.dialog = ( pathparts, req, res, body ) =>
    {
      /*
        We add a call to our call dictionary when we get a call we do not know about (i.e new),
        we remove it when the call has been marked as hungup (we should not hear anything
        else from it).
      */
      try
      {
        var callinfo = JSON.parse( body )
        var c

        var laststate = { "ring": false, "answered": false, "hold": false, "hangup": false }
        if( !( callinfo.callid in this.calls ) )
        {
          c = new call( this, callinfo )
          c.history = "New call " + callinfo.callid
          this.calls[ callinfo.callid ] = c
          this.runnewcalls( c )
        }
        else
        {
          c = this.calls[ callinfo.callid ]
          laststate = c.info
          c.info = callinfo
        }

        if( c.ringing && c.onringingcallback.length > 0 )
        {
          c.history = "Client confirmed ringing"
          for( let cb of c.onringingcallback )
          {
            cb.call( c, c )
          }
          c.onringingcallback = []
        }

        if( !laststate.answered && c.answered && c.onanswercallback.length > 0 )
        {
          c.history = "Client confirmed answer"
          for( let cb of c.onanswercallback )
          {
            cb.call( c, c )
          }
          c.onanswercallback = []
        }
        else if( laststate.answered && c.answered && !c.hungup )
        {
          /* re-invite or refer */
          if( laststate.hold != c.hold )
          {
            if( c.hold )
            {
              c.history = "Hold"
              for( let cb of c.onholdcallback )
              {
                cb.call( c, c )
              }
              c.info.sdp.us.m[ 0 ].direction = "inactive"
            }
            if( !c.hold )
            {
              c.history = "Off hold"
              for( let cb of c.onoffholdcallback )
              {
                cb.call( c, c )
              }
              c.info.sdp.us.m[ 0 ].direction = "sendrecv"
            }
            this.sipserver( { "callid": c.info.callid, "sdp": c.info.sdp.us }, "/dialog/ok" )
            return
          }

          if( "refer" in c.info )
          {
            if( "replaces" in c.info.refer )
            {
              c.history = "Attended xfer - 4 leg"
              /*
                leg1 calls leg2
                leg2 puts leg1 onhold
                leg3 calls application
                leg3 refers leg1

                OR

                leg1 calls leg2
                leg2 puts leg1 on hold
                leg3 calls leg4
                leg4 refers leg1
              */
              var leg1 = false
              var leg2 = false
              var leg3 = c
              var leg4 = leg3.other

              if( c.info.refer.replaces in this.calls )
              {
                leg2 = this.calls[ c.info.refer.replaces ]
                leg1 = leg2.other
              }

              if( false !== leg4 )
              {
                /*
                  4 legged call. The call we are xfering is 2 legged. So likely a media/linked calls.
                  leg1 needs to have media from the xfered call.
                */

                leg1.unmix( () =>
                {
                  leg3.unmix( () =>
                  {
                    leg1.unlink()
                    leg3.unlink()
    
                    leg2.link( leg3 )
                    leg1.link( leg4 )
    
                    leg3.onhangup = () =>
                    {
                      leg1.mix()
                    }
    
                    this.sipserver( { "callid": leg3.info.callid }, "/dialog/notify" )
                  } )
                } )
              }
              else
              {

                c.history = "Attended xfer - 3 leg"

                /*
                  The call we are xfering is 1 legged - possible a conference or playing audio.
                  c is call 3 ( or leg 3 ) - this contains the sdp and channel we need
                  the channel we are replacing is leg 1 ( and that channel will hangup )
                  so we place that channel on leg 3 as leg 3 will be ended.
                */
                var oldchannel = leg1.metadata.channel
                var oldsdp = leg1.info.sdp.us

                leg1.info.sdp.us = leg3.info.sdp.us
                leg1.info.sdp.us.m[ 0 ].direction = "sendrecv"
                leg1.metadata.channel = leg3.metadata.channel

                leg3.info.sdp.us = oldsdp
                leg3.metadata.channel = oldchannel

                leg1.unlink()
                leg1.unmix( () =>
                {
                  /* re-invite leg1 to update the RTP stream */
                  this.sipserver( { "callid": leg1.info.callid, "sdp": leg1.info.sdp.us }, "/dialog/invite", "PUT", () =>
                  {
                    /* Link leg2 to leg3, so that when the client hangs up leg3, leg2 will follow. */
                    leg3.link( leg2 )

                    /* this link will only happen if leg4 exists */
                    leg1.link( leg4 )

                    /* Let leg3 know the xfer is complete. */
                    this.sipserver( { "callid": leg3.info.callid }, "/dialog/notify" )
                  } )
                } )
              }
            }
            else
            {
              c.history = "Blind xfer"
              /* Blnd xfer */
              var other = c.other
              if( false !== other )
              {
                other.clearcallbacks()
                other.info.refer = c.info.refer
                other.info.refer.referedby = c
                c.unmix( () =>
                {
                  for( let cb of this.onnewcallcallbacks )
                  {
                    cb.call( other, other )
                  }

                  c.unlink()
                  this.sipserver( { "callid": c.info.callid }, "/dialog/notify" )
                } )
              }
            }
          }
        }

        if( c.hungup )
        {
          c.history = "Client hungup: " + c.info.error.status + "(" + c.info.error.code + ")"
          this.runhangups( c )
        }
      }
      catch( e )
      {
        console.log( e )
        console.log( "Body: " + body )
      }

      res.writeHead( 200, { "Content-Length": "0" } )
    }

    this.handlers.PUT.reg = ( pathparts, req, res, body ) =>
    {
      try
      {
        for( let cb of this.onregcallbacks )
        {
          cb( body )
        }
      }
      catch( e )
      {
        console.log( e )
      }
      res.writeHead( 200, { "Content-Length": "0" } )
    }

    this.handlers.DELETE.reg = ( pathparts, req, res, body ) =>
    {
      try
      {
        for( let cb of this.onderegcallbacks )
        {
          cb( body )
        }
      }
      catch( e )
      {
        console.log( e )
      }
      res.writeHead( 200, { "Content-Length": "0" } )
    }
  }

  runhangups( call )
  {
    for( let cb of this.onhangupcallbacks )
    {
      cb.call( call, call )
    }

    for( let cb of call.onhangupcallback )
    {
      cb.call( call, call )
    }
    call.onhangupcallback = []

    for( let cb of this.onreportingcallbacks )
    {
      cb.call( call, call )
    }

    for( let cb of call.onreportingcallbacks )
    {
      cb.call( call, call )
    }

    if( "callinfo" in call && "callid" in call.callinfo )
    {
      delete this.calls[ call.callinfo.callid ]
    }
  }

  runnewcalls( call )
  {
    if( call.originator )
    {
      return
    }

    /* global new call handlers */
    for( let cb of this.onnewcallcallbacks )
    {
      cb.call( call, call )
    }

    /* call specific handlers */
    for( let cb of call.onnewcallcallback )
    {
      cb.call( call, call )
    }
    call.onnewcallcallback = []
  }

  set onreg( callback )
  {
    this.onregcallbacks.push( callback )
  }

  set ondereg( callback )
  {
    this.onderegcallbacks.push( callback )
  }

  set onnewcall( callback )
  {
    this.onnewcallcallbacks.push( callback )
  }

  set onhangup( callback )
  {
    this.onhangupcallbacks.push( callback )
  }

  set onreporting( callback )
  {
    this.onreportingcallbacks.push( callback )
  }

/*!md
# sipserver
Send a request to our SIP server.
*/
  sipserver( request, path, method = "PUT", callback )
  {
    return this.server( request, path, method, this.sip, callback )
  }
/*!md
# rtpserver
Send a request to our rtp server. This is work in progress. Simple for now but the goal is to add intelligence to send it out to 1 of many servers and choose which one.
*/
  rtpserver( request, path, method = "PUT", callback )
  {
    return this.server( request, path, method, this.rtp, callback )
  }

  server( request, path, method, server, callback )
  {
    var data = JSON.stringify( request )
    var post_options = {
      "host": server.host,
      "port": server.port,
      "path": path,
      "method": method,
      "headers": {
        "Content-Type": "text/json",
        "Content-Length": Buffer.byteLength( data )
      }
    }

    var post_req = http.request( post_options )

    post_req.on( "error", ( e ) =>
    {
      if( callback )
      {
        callback( { code: 500, message: `Problem with request: ${e.message}, for ${method} ${path}` } )
      }
    } )


    post_req.on( "response", ( req ) =>
    {
      req.on( "data", ( chunk ) =>
      {
        if( !( "collatedbody" in this ) ) this.collatedbody = []
        this.collatedbody.push( chunk )
      } )

      req.on( "end", () =>
      {
        var body = {}
        if( Array.isArray( this.collatedbody ) )
        {
          try
          {
            if( this.collatedbody.length > 0 )
            {
              body = JSON.parse( Buffer.concat( this.collatedbody ).toString() )
            }
          }
          catch( e )
          {
            console.log( e )
          }
        }
        this.collatedbody = []
        if( callback )
        {
          callback( { code: req.statusCode, message: req.statusMessage, json: body } )
        }
      } )
    } )

    post_req.write( data )
    post_req.end()
  }

/*!md
# newcall

request object
```json
{
  from: {
    domain: "",
    user: ""
  },
  to: {
    domain: "optional",
    user: ""
  },
  maxforwards: 70,
  cid: {
    name: "",
    number: "",
    private: false
  },
  codecs[ 'pcmu' ],
  forward = false
}
```

A gateway object which we will take settings from if the forward param is set looks like
```json
{
  from: {
    user: "omniis",
    domain: "omniis.babblevoice.com",
    authsecret: "thecatsatonthemat"
  },
  to: {
    domain: "omniis.babblevoice.com",
  }
}
```

The follow list is what we will be taken from a gateway (if it exsists)

* cid
* codecs
* from ( user, domain and authsecret )
* to ( domain )
*/
  newcall( request )
  {
    var c = new call( this, { originator: true } )

    if( !( "maxforwards" in request ) ) request.maxforwards = 70

    if( "forward" in request && request.forward )
    {
      var gw = this.gw
      if( "from" in gw )
      {
        if( !( "from" in request ) ) request.from = {}
        if( "user" in gw.from ) request.from.user = gw.from.user
        if( "domain" in gw.from ) request.from.domain = gw.from.domain
        if( "authsecret" in gw.from ) request.from.authsecret = gw.from.authsecret
      }
      if( "to" in gw )
      {
        if( !( "to" in request ) ) request.to = {}
        if( "domain" in gw.to ) request.to.domain = gw.to.domain
      }
      if( "cid" in gw )
      {
        if( !( "cid" in request ) ) request.cid = {}
        if( "name" in gw.cid ) request.cid.name = gw.cid.name
        if( "number" in gw.cid ) request.cid.name = gw.cid.number
        if( "private" in gw.cid ) request.cid.name = gw.cid.private
      }
      if( "codecs" in gw ) request.codecs = gw.codecs
    }

    this.createchannel( c, request, ( channel ) =>
    {
      if( !( "sdp" in request ) )
      {
        c.error = { code: 480, message: "Unable to create channel" }
        c.history = "We failed to create a new channel for call"
        for( let cb of c.onnewcallcallback )
        {
          cb.call( c, c )
        }
        return
      }

      c.metadata.channel = channel

      this.sipserver( request, "/dialog/invite", "POST", ( response ) =>
      {
        if( 200 == response.code )
        {
          c.history = "We created new call " + response.json.callid
          c.callinfo.callid = response.json.callid

          this.calls[ c.callinfo.callid ] = c
          this.runnewcalls( c )
        }
        else
        {
          c.history = "We failed to create a new call"
          this.runhangups( c )
        }
      } )
    } )

    return c
  }

/*!md
# createchannel
Negotiates a channel with an RTP server then creates the corrosponding SDP object with CODECS requested. The SDP object is added to the request (as it is required in the SIP INVITE) and then we pass in the channel object to the callback. Currently our RTP servers return:
{
  ip: "",
  port: 10000,
}
*/
  createchannel( call, request, callback )
  {
    request.control = "http://" + this.us.host
    if( 80 != this.us.port )
    {
      request.control += ":" + this.us.port
    }

    this.server( request, "/channel/", "POST", this.rtp, ( response ) =>
    {
      if( 200 == response.code )
      {
        var ch = {}
        ch.ip = response.json.ip
        ch.port = response.json.port
        ch.control = this.rtp
        ch.uuid = response.json.uuid

        /* reverse map from channel uuid to call */
        this.channels[ response.json.uuid ] = call

        request.sdp = {
          v: 0,
          t: { start: 0, stop: 0 },
          o: {
            username: "-",
            sessionid: this.sessionidcounter,
            sessionversion: 0,
            nettype: "IN",
            ipver: 4,
            address: response.json.ip
          },
          s: " ",
          c: [
            {
              nettype: "IN",
              ipver: 4,
              address: response.json.ip
            } ],
          m: [
              {
                media: "audio",
                port: response.json.port,
                proto: "RTP/AVP",
                direction: "sendrecv",
                payloads: [],
                rtpmap: {},
              }
            ]
        }

        if( !( "codecs" in request ) ) request.codecs = this.codecs

        ch.codecs = request.codecs

        if( Array.isArray( request.codecs ) )
        {
          for( let cod of request.codecs )
          {
            this.addmedia( request.sdp, cod )
          }
        }

        this.sessionidcounter = ( this.sessionidcounter + 1 ) % 4294967296
      }

      callback( ch )
    } )
  }

/*!md
# addmedia
Add a CODEC to the SDP object.
*/
  addmedia( sdp, codec )
  {
    switch( codec )
    {
      case "pcmu":
      {
        sdp.m[ 0 ].payloads.push( 0 )
        sdp.m[ 0 ].rtpmap[ "0" ] = { encoding: "PCMU", clock: "8000" }
        break
      }
      case "pcma":
      {
        sdp.m[ 0 ].payloads.push( 8 )
        sdp.m[ 0 ].rtpmap[ "8" ] = { encoding: "PCMA", clock: "8000" }
        break
      }
      case "722":
      {
        sdp.m[ 0 ].payloads.push( 9 )
        break
      }
      case "ilbc":
      {
        sdp.m[ 0 ].payloads.push( 97 )
        sdp.m[ 0 ].rtpmap[ "97" ] = { encoding: "iLBC", clock: "8000" }
        if( !( "fmtp" in sdp.m[ 0 ] ) ) sdp.m[ 0 ].fmtp = {}
        sdp.m[ 0 ].fmtp[ "97" ] = "mode=20"
        break
      }
      /* rfc 2833 - DTMF*/
      case "2833":
      {
        sdp.m[ 0 ].payloads.push( 101 )
        sdp.m[ 0 ].rtpmap[ "101" ] = { encoding: "telephone-event", clock: "8000" }
        if( !( "fmtp" in sdp.m[ 0 ] ) ) sdp.m[ 0 ].fmtp = {}
        sdp.m[ 0 ].fmtp[ "101" ] = "0-16"
        break
      }
    }
  }

  directory( domain, users )
  {
    var request = {}
    request.control = "http://" + this.us.host + ":" + this.us.port
    request.users = users

    this.sipserver( request, "/dir/" + domain )
  }

  /* We have a list of gateways. This is for when we indicate we wish to forward a call.  */
  set gw( gw )
  {
    this.gateways.push( gw )
  }

  get gw()
  {
    return this.gateways[ 0 ]
  }

  set moh( m )
  {
    this.mohsoup = m
  }

  get moh()
  {
    return this.mohsoup
  }



/*!md
# run
Our main event loop. Listen for HTTP control events.
*/
  run()
  {
    this.httpserver = http.createServer( ( req, res ) =>
    {
      /*
       Gather our body.
      */
      req.on( "data", ( chunk ) =>
      {
        if( !( "collatedbody" in this ) )
        {
          this.collatedbody = []
        }
        this.collatedbody.push( chunk )
      } )

      req.on( "end", () =>
      {
        var urlparts = url.parse( req.url )
        /* Remove the leading '/' */
        var path = urlparts.path.substr( 1 )
        var pathparts = path.split( '/' )

        if( req.method in this.handlers && pathparts[ 0 ] in this.handlers[ req.method ] )
        {
          this.handlers[ req.method ][ pathparts[ 0 ] ]( pathparts, req, res, Buffer.concat( this.collatedbody ).toString() )
          this.collatedbody = []
        }
        else
        {
          console.log( "Unknown method " + req.method + ":" + url )
          res.writeHead( 404, { "Content-Length": "0" } )
        }
        res.end()
      } )

    } )

    this.httpserver.listen( this.us.port, this.us.host, () =>
    {
      console.log( `Project Control Server is running on ${this.us.host} port ${this.us.port}` )
    } )
  }
}


module.exports = new projectcontrol()
