

"use strict"

module.exports = class call
{
  constructor( control, callinfo )
  {
    this.control = control
    this.callinfo = callinfo
    this.metadata = {}
    this.metadata.history = []

    /* A call's family. It can only have 1 parent but can have multiple children */
    this.metadata.family = {}
    this.metadata.family.children = []
    this.clearcallbacks()


    /* Some default handlers */
    this.onhangup = () =>
    {
      for ( let child of this.metadata.family.children )
      {
        child.hangup()
      }

      if( "parent" in this.metadata.family )
      {
        /* we only hangup the parent, if the parent only has us as a child */
        var filtered = this.metadata.family.parent.metadata.family.children.filter( ( value ) =>
        {
          return value != this
        } )
        this.metadata.family.parent.metadata.family.children = filtered

        /* We may want to alter this. There may be occasions where we wish further processing of the originating call after any child may hang up. The reason could be passed back into hanup of the parent who then makes the decision. */
        if( 0 == filtered.length ) this.metadata.family.parent.hangup()
      }

      if( "channel" in this.metadata )
      {
        this.control.server( {}, "/channel/" + this.metadata.channel.uuid, "DELETE", this.metadata.channel.control, ( response ) =>
        {
          delete this.control.channels[ this.metadata.channel.uuid ]
        } )
      }
    }

    this.onanswer = () =>
    {
      /*
        them = their sdp - so information like where we send RTP to.
        us = our sdp - where rtp should be sent to. this was issued from information from our RTP server so the RTP server does not need updating.
      */
      if( !( "sdp" in this.callinfo ) || !( "them" in this.callinfo.sdp ) )
      {
        return
      }

      var request = {}

      if( "c" in this.callinfo.sdp.them && Array.isArray( this.callinfo.sdp.them.c ) )
      {
        request.ip = this.callinfo.sdp.them.c[ 0 ].address
      }

      for( let m of this.callinfo.sdp.them.m )
      {
        if( "audio" == m.media )
        {
          request.port = m.port
          request.audio = {}
          request.audio.payloads = m.payloads
          break
        }
      }

      /* Only include payloads we find acceptable ourself */
      var ourpayloads = []
      for( let m of this.callinfo.sdp.us.m )
      {
        if( "audio" == m.media )
        {
          ourpayloads = m.payloads
          break
        }
      }

      request.audio.payloads = request.audio.payloads.filter( payload => ourpayloads.includes( payload ) )

      if( "channel" in this.metadata )
      {
        this.control.server( request, "/channel/" + this.metadata.channel.uuid, "PUT", this.metadata.channel.control, ( response ) =>
        {
          if( "parent" in this.metadata.family )
          {
            this.metadata.family.parent.onanswer = () =>
            {
              if( "channel" in this.metadata.family.parent.metadata )
              {
                var url = "/channel/"
                url += this.metadata.channel.uuid
                url += "/mix/"
                url += this.metadata.family.parent.metadata.channel.uuid

                this.control.server( {}, url , "PUT", this.metadata.channel.control, ( response ) =>
                {

                } )
              }
            }
            this.metadata.family.parent.answer()
          }
        } )
      }
    }

    this.onringing = () =>
    {
      if( "parent" in this.metadata.family )
      {
        this.metadata.family.parent.ring()
      }
    }

    this.onhold = () =>
    {
      this.unmix( () =>
      {
        var other = this.other
        if( false !== other )
        {
          other.play( this.moh )
        }
      } )
    }

    this.onoffhold = () =>
    {
      var other = this.other
      if( false !== other && "channel" in this.metadata )
      {
        this.control.server( {}, "/channel/" + this.metadata.channel.uuid + "/mix/" + other.metadata.channel.uuid , "PUT", this.metadata.channel.control, ( response ) =>
        {
        } )
      }
    }
  }

  clearcallbacks()
  {
    /* Our callback handlers */
    this.onnewcallcallback = []
    this.onringingcallback = []
    this.onanswercallback = []
    this.onhangupcallback = []
    this.onreportingcallbacks = []
    this.onholdcallback = []
    this.onoffholdcallback = []
  }

  set history( what )
  {
    this.metadata.history.push( { "when": Date.now(), "what": what } )
  }

  get history()
  {
    return this.metadata.history
  }

  set info( callinf )
  {
    this.callinfo = callinf
  }

  get info()
  {
    return this.callinfo
  }

  set onnewcall( cb )
  {
    this.onnewcallcallback.push( cb )
    return this
  }

  set onringing( cb )
  {
    this.onringingcallback.push( cb )
    return this
  }

  set onanswer( cb )
  {
    this.onanswercallback.push( cb )
    return this
  }

  set onhold( cb )
  {
    this.onholdcallback.push( cb )
    return this
  }

  set onoffhold( cb )
  {
    this.onoffholdcallback.push( cb )
    return this
  }

  set onhangup( cb )
  {
    this.onhangupcallback.push( cb )
    return this
  }

  set onreporting( cb )
  {
    this.onreportingcallbacks.push( cb )
    return this
  }

  set moh( m )
  {
    this.mohsoup = m
  }

  get moh()
  {
    if( "mohsoup" in this )
    {
      return this.mohsoup
    }

    return this.control.moh
  }

  get ringing()
  {
    if( "ring" in this.callinfo )
    {
      return true == this.callinfo.ring
    }
    return false
  }

  get answered()
  {
    if( "answered" in this.callinfo )
    {
      return true == this.callinfo.answered
    }
    return false
  }

  get hungup()
  {
    if( "hangup" in this.callinfo )
    {
      return true == this.callinfo.hangup
    }
    return false
  }

  get hold()
  {
    if( "hold" in this.callinfo )
    {
      return true == this.callinfo.hold
    }
    return false
  }

/*!md
### destination
Get the destination the user has dialled.
*/
  get destination()
  {
    if( "refer" in this.callinfo )
    {
      return this.callinfo.refer.to
    }
    return this.callinfo.to
  }

/*!md
### originator
Did we originate the call?
*/
  get originator()
  {
    if( "originator" in this.callinfo )
    {
      return true == this.callinfo.originator
    }
    return false
  }

  get haserror()
  {
    return "error" in this
  }

/*!md
### answer
Request is similar to new call. As the call is already (partly) established all we can have is optionally the list of codecs.
request = { codecs: [ 'pcmu' ] }
*/
  answer( request )
  {
    this.history = "We answer call"
    if( this.answered )
    {
      /* If a call has already been answered, we schedule running these callbacks as they will have been recently added. */
      for( let cb of this.onanswercallback )
      {
        setTimeout( () => { cb.call( this, this ) }, 0 )
      }
      this.onanswercallback = []
      return
    }

    if( undefined == request ) request = {}

    this.control.createchannel( this, request, ( channel ) =>
    {
      if ( undefined == channel )
      {
        this.hangup()
        return
      }

      this.metadata.channel = channel

      this.postrequest( "answer", { "sdp": request.sdp } )
    } )
  }

  ring( alertinfo )
  {
    this.history = "We ring"
    if( this.ringing || this.answered )
    {
      for( let cb of this.onringingcallback )
      {
        setTimeout( () => { cb.call( this, this ) }, 0 )
      }
      this.onringingcallback = []

      for( let cb of this.onanswercallback )
      {
        setTimeout( () => { cb.call( this, this ) }, 0 )
      }
      this.onanswercallback = []
      return
    }

    var postdata = {}

    if( undefined != alertinfo )
    {
      postdata.alertinfo = alertinfo
    }

    this.postrequest( "ring", postdata )
  }

  notfound()
  {
    if( this.hungup ) return

    this.postrequest( "hangup", { "reason": "Not found", "code": 404 } )
  }

  paymentrequired()
  {
    if( this.hungup ) return

    this.postrequest( "hangup", { "reason": "Payment required", "code": 402 } )
  }

  busy()
  {
    this.history = "We busy"
    if( this.hungup ) return

    this.postrequest( "hangup", { "reason": "Busy here", "code": 486 } )
  }

  hangup()
  {
    this.history = "We hangup"
    if( this.hungup ) return

    this.postrequest( "hangup", {} )
  }

/*!md
### newcall

Similar to the newcall method on the control object except this method calls 'from' an exsisting call. The caller only needs to set the desitnation. This method sets up the call, then if sucsessful will bridge RTP.

forward = false means it will pass this call off to our SIP server (local), forward = true we will still use SIP but use one of our Gateways.

request is a request object as defined in projectsipcontrol.newcall.

TODO
- [] Improve CID
*/
  newcall( request )
  {
    /* A simple group call */
    if( Array.isArray( request ) )
    {
      var newchildren = []
      for( let thisrequest of request )
      {
        var newcall = this.newcall( thisrequest )
        newcall.onanswer = ( call ) =>
        {
          for( let eachcall of this.metadata.family.children )
          {
            if( eachcall != call )
            {
              // TODO - need to indicate lose race
              eachcall.hangup()
            }
          }
        }
        newchildren.push( newcall )
      }
      return newchildren
    }

    if( !( "to" in request ) || !( "user" in request.to ) )
    {
      return
    }

    if( !( "maxforwards" in request ) )
    {
      if( "maxforwards" in this.callinfo )
      {
        request.maxforwards = this.callinfo.maxforwards - 1
      }
      else
      {
        request.maxforwards = 70
      }
    }

    if( !( "from" in request ) )
    {
      request.from = {}
      request.from.domain = this.callinfo.domain
      request.from.user = this.callinfo.from
    }

    if( !( "cid" in request ) ) request.cid = {}
    if( !( "number" in request.cid ) ) request.cid.number = request.from.user
    if( !( "name" in request.cid ) ) request.cid.name = request.from.user

    var dom = request.from.domain
    if( "domain" in request.to ) dom = request.to.domain
    this.history = "Creating child call to " + request.to.user + "@" + dom

    var call = this.control.newcall( request )
    call.metadata.family.parent = this
    this.metadata.family.children.push( call )

    return call
  }

/*!md
### unmix
*/
  unmix( oncomplete )
  {
    var other = this.other

    if( "channel" in this.metadata && false !== other )
    {
      this.history = "Unmix"
      this.control.server( {}, "/channel/" + this.metadata.channel.uuid + "/unmix", "PUT", this.metadata.channel.control, ( response ) =>
      {
        if( undefined != oncomplete ) oncomplete()
      } )
    }
    else
    {
      if( undefined != oncomplete ) oncomplete()
    }
  }

  mix( oncomplete )
  {
    this.history = "Mix"
    if( "channel" in this.metadata )
    {
      var other = this.other
      if( false !== other )
      {
        this.control.server( {}, "/channel/" + this.metadata.channel.uuid + "/mix/" + other.metadata.channel.uuid, "PUT", this.metadata.channel.control, ( response ) =>
        {
          if( undefined != oncomplete ) oncomplete()
        } )
        return
      }
    }

    if( undefined != oncomplete ) oncomplete()
  }

/*!md
### unlink
Removes the 2 way links between this call and children.
*/
  unlink()
  {
    this.history = "Unlink"
    if( "parent" in this.metadata.family )
    {
      this.metadata.family.parent.metadata.family.children =
        this.metadata.family.parent.metadata.family.children.filter( ( value, index, arr ) =>
        {
          return value == this
        } )

      delete this.metadata.family.parent
    }

    for( let child of this.metadata.family.children )
    {
      delete child.metadata.family.parent
    }
    this.metadata.family.children = []
  }

  link( other )
  {
    this.history = "Link"
    if( false === other ) return
    other.metadata.family.parent = this
    this.metadata.family.children.push( other )
  }

  get other()
  {
    if( "parent" in this.metadata.family ) return this.metadata.family.parent
    if( this.metadata.family.children.length > 0 ) return this.metadata.family.children[ 0 ]

    return false
  }

  get others()
  {
    if( "parent" in this.metadata.family ) return [ this.metadata.family.parent ]
    if( this.metadata.family.children.length > 0 ) return this.metadata.family.children

    return []
  }

/*!md
### play
Request the RTP server to play sound on the channel. Soup and structure documented elsewhere.
*/
  play( soup )
  {
    this.history = "Play soup: " + JSON.stringify( soup )

    if( "channel" in this.metadata )
    {
      this.control.server( soup, "/channel/" + this.metadata.channel.uuid + "/play", "PUT", this.metadata.channel.control, ( response ) =>
      {
      } )
    }
  }

/*!md
### postrequest
Generic purpose fuction to post data to a SIP server.
*/
  postrequest( action, data )
  {
    data.callid = this.callinfo.callid
    this.control.sipserver( data, "/dialog/" + action )
  }
}
