
#import "MaterialDesignSymbol.h"
#import <CoreText/CoreText.h>

@interface MaterialDesignSymbol ()

@property (strong, nonatomic) NSMutableAttributedString *symbolAttributedString;
@property (nonatomic) CGFloat fontSize;

@end

@implementation MaterialDesignSymbol

+ (void)initialize {
    
    if (self == MaterialDesignSymbol.class) {
        
        NSURL *url = [[NSBundle bundleForClass:self] URLForResource:@"material-design-icons" withExtension:@"ttf"];
        
        CGDataProviderRef fontDataProvider = CGDataProviderCreateWithURL((__bridge CFURLRef)url);
        CGFontRef newFont = CGFontCreateWithDataProvider(fontDataProvider);
        CGDataProviderRelease(fontDataProvider);
        
        CFErrorRef error;
        CTFontManagerRegisterGraphicsFont(newFont, &error);
        CGFontRelease(newFont);
    }
}

+ (instancetype)iconWithCode:(NSString *)code fontSize:(CGFloat)fontSize {
    
    MaterialDesignSymbol *symbol = [MaterialDesignSymbol new];
    symbol.fontSize = fontSize;
    symbol.symbolAttributedString = [NSMutableAttributedString.alloc initWithString:code attributes:@{NSFontAttributeName:[self iconFontWithSize:fontSize]}];
    
    return symbol;
}

+ (UIFont *)iconFontWithSize:(CGFloat)size {
    UIFont *font = [UIFont fontWithName:@"material-design-icons" size:size];
    return font;
}

- (UIImage *)imageWithSize:(CGSize)imageSize {
    UIGraphicsBeginImageContextWithOptions(imageSize, NO, [UIScreen.mainScreen scale]);
    
    [self.symbolAttributedString drawInRect:({
        CGSize iconSize = [self.symbolAttributedString size];
        CGFloat xOffset = (imageSize.width - iconSize.width) / 2.0;
        CGFloat yOffset = (imageSize.height - iconSize.height) / 2.0;
        CGRectMake(xOffset, yOffset, iconSize.width, iconSize.height);
    })];
    
    UIImage *iconImage = UIGraphicsGetImageFromCurrentImageContext();
    
    UIGraphicsEndImageContext();

    return iconImage;
}

- (UIImage *)image {
    return [self imageWithSize:CGSizeMake(self.fontSize, self.fontSize)];
}

- (void)addAttributes:(NSDictionary *)attributes {
    [self.symbolAttributedString addAttributes:attributes range:[self symbolAttributedStringRange]];
}

- (void)addAttribute:(NSString *)name value:(id)value {
    [self.symbolAttributedString addAttribute:name value:value range:[self symbolAttributedStringRange]];
}

- (id)attribute:(NSString *)attributeName {
    return [self.symbolAttributedString attributesAtIndex:0 effectiveRange:NULL];
}

- (NSRange)symbolAttributedStringRange {
    return NSMakeRange(0, [self.symbolAttributedString length]);
}

const struct MaterialDesignIconCode MaterialDesignIconCode = {
    .threeDRotation24px = @"\ue600",
    .threeDRotation48px = @"\ue601",
    .accessibility24px = @"\ue602",
    .accessibility48px = @"\ue603",
    .accountBalance24px = @"\ue604",
    .accountBalance48px = @"\ue605",
    .accountBalanceWallet24px = @"\ue606",
    .accountBalanceWallet48px = @"\ue607",
    .accountBox18px = @"\ue608",
    .accountBox24px = @"\ue609",
    .accountBox48px = @"\ue60a",
    .accountChild24px = @"\ue60b",
    .accountChild48px = @"\ue60c",
    .accountCircle18px = @"\ue60d",
    .accountCircle24px = @"\ue60e",
    .accountCircle48px = @"\ue60f",
    .addShoppingCart24px = @"\ue610",
    .addShoppingCart48px = @"\ue611",
    .alarm24px = @"\ue612",
    .alarm48px = @"\ue613",
    .alarmAdd24px = @"\ue614",
    .alarmAdd48px = @"\ue615",
    .alarmOff24px = @"\ue616",
    .alarmOff48px = @"\ue617",
    .alarmOn24px = @"\ue618",
    .alarmOn48px = @"\ue619",
    .android24px = @"\ue61a",
    .android48px = @"\ue61b",
    .announcement24px = @"\ue61c",
    .announcement48px = @"\ue61d",
    .aspectRatio24px = @"\ue61e",
    .aspectRatio48px = @"\ue61f",
    .assessment24px = @"\ue620",
    .assessment48px = @"\ue621",
    .assignment24px = @"\ue622",
    .assignment48px = @"\ue623",
    .assignmentInd24px = @"\ue624",
    .assignmentInd48px = @"\ue625",
    .assignmentLate24px = @"\ue626",
    .assignmentLate48px = @"\ue627",
    .assignmentReturn24px = @"\ue628",
    .assignmentReturn48px = @"\ue629",
    .assignmentReturned24px = @"\ue62a",
    .assignmentReturned48px = @"\ue62b",
    .assignmentTurnedIn24px = @"\ue62c",
    .assignmentTurnedIn48px = @"\ue62d",
    .autorenew24px = @"\ue62e",
    .autorenew48px = @"\ue62f",
    .backup24px = @"\ue630",
    .backup48px = @"\ue631",
    .book24px = @"\ue632",
    .book48px = @"\ue633",
    .bookmark24px = @"\ue634",
    .bookmark48px = @"\ue635",
    .bookmarkOutline24px = @"\ue636",
    .bookmarkOutline48px = @"\ue637",
    .bugReport24px = @"\ue638",
    .bugReport48px = @"\ue639",
    .cached24px = @"\ue63a",
    .cached48px = @"\ue63b",
    .class24px = @"\ue63c",
    .class48px = @"\ue63d",
    .creditCard24px = @"\ue63e",
    .creditCard48px = @"\ue63f",
    .dashboard24px = @"\ue640",
    .dashboard48px = @"\ue641",
    .delete24px = @"\ue642",
    .delete48px = @"\ue643",
    .description24px = @"\ue644",
    .description48px = @"\ue645",
    .dns24px = @"\ue646",
    .dns48px = @"\ue647",
    .done24px = @"\ue648",
    .done48px = @"\ue649",
    .doneAll24px = @"\ue64a",
    .doneAll48px = @"\ue64b",
    .event18px = @"\ue64c",
    .event24px = @"\ue64d",
    .event48px = @"\ue64e",
    .exitToApp24px = @"\ue64f",
    .exitToApp48px = @"\ue650",
    .explore24px = @"\ue651",
    .explore48px = @"\ue652",
    .extension24px = @"\ue653",
    .extension48px = @"\ue654",
    .faceUnlock24px = @"\ue655",
    .faceUnlock48px = @"\ue656",
    .favorite24px = @"\ue657",
    .favorite48px = @"\ue658",
    .favoriteOutline24px = @"\ue659",
    .favoriteOutline48px = @"\ue65a",
    .findInPage24px = @"\ue65b",
    .findInPage48px = @"\ue65c",
    .findReplace24px = @"\ue65d",
    .findReplace48px = @"\ue65e",
    .flipToBack24px = @"\ue65f",
    .flipToBack48px = @"\ue660",
    .flipToFront24px = @"\ue661",
    .flipToFront48px = @"\ue662",
    .getApp24px = @"\ue663",
    .getApp48px = @"\ue664",
    .grade24px = @"\ue665",
    .grade48px = @"\ue666",
    .groupWork24px = @"\ue667",
    .groupWork48px = @"\ue668",
    .help24px = @"\ue669",
    .help48px = @"\ue66a",
    .highlightRemove24px = @"\ue66b",
    .highlightRemove48px = @"\ue66c",
    .history24px = @"\ue66d",
    .history48px = @"\ue66e",
    .home24px = @"\ue66f",
    .home48px = @"\ue670",
    .https24px = @"\ue671",
    .https48px = @"\ue672",
    .info24px = @"\ue673",
    .info48px = @"\ue674",
    .infoOutline24px = @"\ue675",
    .infoOutline48px = @"\ue676",
    .input24px = @"\ue677",
    .input48px = @"\ue678",
    .invertColors24px = @"\ue679",
    .invertColors48px = @"\ue67a",
    .label24px = @"\ue67b",
    .label48px = @"\ue67c",
    .labelOutline24px = @"\ue67d",
    .labelOutline48px = @"\ue67e",
    .language24px = @"\ue67f",
    .language48px = @"\ue680",
    .launch24px = @"\ue681",
    .launch48px = @"\ue682",
    .list24px = @"\ue683",
    .list48px = @"\ue684",
    .lock24px = @"\ue685",
    .lock48px = @"\ue686",
    .lockOpen24px = @"\ue687",
    .lockOpen48px = @"\ue688",
    .lockOutline24px = @"\ue689",
    .lockOutline48px = @"\ue68a",
    .loyalty24px = @"\ue68b",
    .loyalty48px = @"\ue68c",
    .markunreadMailbox24px = @"\ue68d",
    .markunreadMailbox48px = @"\ue68e",
    .noteAdd24px = @"\ue68f",
    .noteAdd48px = @"\ue690",
    .openInBrowser24px = @"\ue691",
    .openInBrowser48px = @"\ue692",
    .openInNew24px = @"\ue693",
    .openInNew48px = @"\ue694",
    .openWith24px = @"\ue695",
    .openWith48px = @"\ue696",
    .pageview24px = @"\ue697",
    .pageview48px = @"\ue698",
    .payment24px = @"\ue699",
    .payment48px = @"\ue69a",
    .permCameraMic24px = @"\ue69b",
    .permCameraMic48px = @"\ue69c",
    .permContactCal24px = @"\ue69d",
    .permContactCal48px = @"\ue69e",
    .permDataSetting24px = @"\ue69f",
    .permDataSetting48px = @"\ue6a0",
    .permDeviceInfo24px = @"\ue6a1",
    .permDeviceInfo48px = @"\ue6a2",
    .permIdentity24px = @"\ue6a3",
    .permIdentity48px = @"\ue6a4",
    .permMedia24px = @"\ue6a5",
    .permMedia48px = @"\ue6a6",
    .permPhoneMsg24px = @"\ue6a7",
    .permPhoneMsg48px = @"\ue6a8",
    .permScanWifi24px = @"\ue6a9",
    .permScanWifi48px = @"\ue6aa",
    .pictureInPicture24px = @"\ue6ab",
    .pictureInPicture48px = @"\ue6ac",
    .polymer24px = @"\ue6ad",
    .polymer48px = @"\ue6ae",
    .print24px = @"\ue6af",
    .print48px = @"\ue6b0",
    .queryBuilder24px = @"\ue6b1",
    .queryBuilder48px = @"\ue6b2",
    .questionAnswer24px = @"\ue6b3",
    .questionAnswer48px = @"\ue6b4",
    .receipt24px = @"\ue6b5",
    .receipt48px = @"\ue6b6",
    .redeem24px = @"\ue6b7",
    .redeem48px = @"\ue6b8",
    .reorder24px = @"\ue6b9",
    .reportProblem24px = @"\ue6ba",
    .reportProblem48px = @"\ue6bb",
    .restore24px = @"\ue6bc",
    .restore48px = @"\ue6bd",
    .room24px = @"\ue6be",
    .room48px = @"\ue6bf",
    .schedule24px = @"\ue6c0",
    .schedule48px = @"\ue6c1",
    .search24px = @"\ue6c2",
    .search48px = @"\ue6c3",
    .settings24px = @"\ue6c4",
    .settings48px = @"\ue6c5",
    .settingsApplications24px = @"\ue6c6",
    .settingsApplications48px = @"\ue6c7",
    .settingsBackupRestore24px = @"\ue6c8",
    .settingsBackupRestore48px = @"\ue6c9",
    .settingsBluetooth24px = @"\ue6ca",
    .settingsBluetooth48px = @"\ue6cb",
    .settingsCell24px = @"\ue6cc",
    .settingsCell48px = @"\ue6cd",
    .settingsDisplay24px = @"\ue6ce",
    .settingsDisplay48px = @"\ue6cf",
    .settingsEthernet24px = @"\ue6d0",
    .settingsEthernet48px = @"\ue6d1",
    .settingsInputAntenna24px = @"\ue6d2",
    .settingsInputAntenna48px = @"\ue6d3",
    .settingsInputComponent24px = @"\ue6d4",
    .settingsInputComponent48px = @"\ue6d5",
    .settingsInputComposite24px = @"\ue6d6",
    .settingsInputComposite48px = @"\ue6d7",
    .settingsInputHdmi24px = @"\ue6d8",
    .settingsInputHdmi48px = @"\ue6d9",
    .settingsInputSvideo24px = @"\ue6da",
    .settingsInputSvideo48px = @"\ue6db",
    .settingsOverscan24px = @"\ue6dc",
    .settingsOverscan48px = @"\ue6dd",
    .settingsPhone24px = @"\ue6de",
    .settingsPhone48px = @"\ue6df",
    .settingsPower24px = @"\ue6e0",
    .settingsPower48px = @"\ue6e1",
    .settingsRemote24px = @"\ue6e2",
    .settingsRemote48px = @"\ue6e3",
    .settingsVoice24px = @"\ue6e4",
    .settingsVoice48px = @"\ue6e5",
    .shop24px = @"\ue6e6",
    .shop48px = @"\ue6e7",
    .shopTwo24px = @"\ue6e8",
    .shopTwo48px = @"\ue6e9",
    .shoppingBasket24px = @"\ue6ea",
    .shoppingBasket48px = @"\ue6eb",
    .shoppingCart24px = @"\ue6ec",
    .shoppingCart48px = @"\ue6ed",
    .speakerNotes24px = @"\ue6ee",
    .speakerNotes48px = @"\ue6ef",
    .spellcheck24px = @"\ue6f0",
    .spellcheck48px = @"\ue6f1",
    .starRate24px = @"\ue6f2",
    .starRate48px = @"\ue6f3",
    .stars24px = @"\ue6f4",
    .stars48px = @"\ue6f5",
    .store24px = @"\ue6f6",
    .store48px = @"\ue6f7",
    .subject24px = @"\ue6f8",
    .subject48px = @"\ue6f9",
    .supervisorAccount24px = @"\ue6fa",
    .swapHoriz24px = @"\ue6fb",
    .swapHoriz48px = @"\ue6fc",
    .swapVert24px = @"\ue6fd",
    .swapVert48px = @"\ue6fe",
    .swapVertCircle24px = @"\ue6ff",
    .swapVertCircle48px = @"\ue700",
    .systemUpdateTv24px = @"\ue701",
    .systemUpdateTv48px = @"\ue702",
    .tab24px = @"\ue703",
    .tab48px = @"\ue704",
    .tabUnselected24px = @"\ue705",
    .tabUnselected48px = @"\ue706",
    .theaters24px = @"\ue707",
    .theaters48px = @"\ue708",
    .thumbDown24px = @"\ue709",
    .thumbDown48px = @"\ue70a",
    .thumbUp24px = @"\ue70b",
    .thumbUp48px = @"\ue70c",
    .thumbsUpDown24px = @"\ue70d",
    .thumbsUpDown48px = @"\ue70e",
    .toc24px = @"\ue70f",
    .toc48px = @"\ue710",
    .today24px = @"\ue711",
    .today48px = @"\ue712",
    .trackChanges24px = @"\ue713",
    .trackChanges48px = @"\ue714",
    .translate24px = @"\ue715",
    .translate48px = @"\ue716",
    .trendingDown24px = @"\ue717",
    .trendingDown48px = @"\ue718",
    .trendingNeutral24px = @"\ue719",
    .trendingNeutral48px = @"\ue71a",
    .trendingUp24px = @"\ue71b",
    .trendingUp48px = @"\ue71c",
    .turnedIn24px = @"\ue71d",
    .turnedIn48px = @"\ue71e",
    .turnedInNot24px = @"\ue71f",
    .turnedInNot48px = @"\ue720",
    .verifiedUser24px = @"\ue721",
    .verifiedUser48px = @"\ue722",
    .viewAgenda24px = @"\ue723",
    .viewAgenda48px = @"\ue724",
    .viewArray24px = @"\ue725",
    .viewArray48px = @"\ue726",
    .viewCarousel24px = @"\ue727",
    .viewCarousel48px = @"\ue728",
    .viewColumn24px = @"\ue729",
    .viewColumn48px = @"\ue72a",
    .viewDay24px = @"\ue72b",
    .viewDay48px = @"\ue72c",
    .viewHeadline24px = @"\ue72d",
    .viewHeadline48px = @"\ue72e",
    .viewList24px = @"\ue72f",
    .viewList48px = @"\ue730",
    .viewModule24px = @"\ue731",
    .viewModule48px = @"\ue732",
    .viewQuilt24px = @"\ue733",
    .viewQuilt48px = @"\ue734",
    .viewStream24px = @"\ue735",
    .viewStream48px = @"\ue736",
    .viewWeek24px = @"\ue737",
    .viewWeek48px = @"\ue738",
    .visibility24px = @"\ue739",
    .visibility48px = @"\ue73a",
    .visibilityOff24px = @"\ue73b",
    .visibilityOff48px = @"\ue73c",
    .walletGiftcard24px = @"\ue73d",
    .walletGiftcard48px = @"\ue73e",
    .walletMembership24px = @"\ue73f",
    .walletMembership48px = @"\ue740",
    .walletTravel24px = @"\ue741",
    .walletTravel48px = @"\ue742",
    .work24px = @"\ue743",
    .work48px = @"\ue744",
    .error18px = @"\ue745",
    .error24px = @"\ue746",
    .error36px = @"\ue747",
    .error48px = @"\ue748",
    .warning18px = @"\ue749",
    .warning24px = @"\ue74a",
    .warning36px = @"\ue74b",
    .warning48px = @"\ue74c",
    .album24px = @"\ue74d",
    .album48px = @"\ue74e",
    .avTimer24px = @"\ue74f",
    .avTimer48px = @"\ue750",
    .closedCaption24px = @"\ue751",
    .closedCaption48px = @"\ue752",
    .equalizer24px = @"\ue753",
    .equalizer48px = @"\ue754",
    .explicit24px = @"\ue755",
    .explicit48px = @"\ue756",
    .fastForward24px = @"\ue757",
    .fastForward48px = @"\ue758",
    .fastRewind24px = @"\ue759",
    .fastRewind48px = @"\ue75a",
    .games24px = @"\ue75b",
    .games48px = @"\ue75c",
    .hearing24px = @"\ue75d",
    .hearing48px = @"\ue75e",
    .highQuality24px = @"\ue75f",
    .highQuality48px = @"\ue760",
    .loop24px = @"\ue761",
    .loop48px = @"\ue762",
    .mic24px = @"\ue763",
    .mic48px = @"\ue764",
    .micNone24px = @"\ue765",
    .micNone48px = @"\ue766",
    .micOff24px = @"\ue767",
    .micOff48px = @"\ue768",
    .movie24px = @"\ue769",
    .movie48px = @"\ue76a",
    .myLibraryAdd24px = @"\ue76b",
    .myLibraryAdd48px = @"\ue76c",
    .myLibraryBooks24px = @"\ue76d",
    .myLibraryBooks48px = @"\ue76e",
    .myLibraryMusic24px = @"\ue76f",
    .myLibraryMusic48px = @"\ue770",
    .newReleases24px = @"\ue771",
    .newReleases48px = @"\ue772",
    .notInterested24px = @"\ue773",
    .notInterested48px = @"\ue774",
    .pause24px = @"\ue775",
    .pause48px = @"\ue776",
    .pauseCircleFill24px = @"\ue777",
    .pauseCircleFill48px = @"\ue778",
    .pauseCircleOutline24px = @"\ue779",
    .pauseCircleOutline48px = @"\ue77a",
    .playArrow24px = @"\ue77b",
    .playArrow48px = @"\ue77c",
    .playCircleFill24px = @"\ue77d",
    .playCircleFill48px = @"\ue77e",
    .playCircleOutline24px = @"\ue77f",
    .playCircleOutline48px = @"\ue780",
    .playShoppingBag24px = @"\ue781",
    .playShoppingBag48px = @"\ue782",
    .playlistAdd24px = @"\ue783",
    .playlistAdd48px = @"\ue784",
    .queue24px = @"\ue785",
    .queue48px = @"\ue786",
    .queueMusic24px = @"\ue787",
    .queueMusic48px = @"\ue788",
    .radio24px = @"\ue789",
    .radio48px = @"\ue78a",
    .recentActors24px = @"\ue78b",
    .recentActors48px = @"\ue78c",
    .repeat24px = @"\ue78d",
    .repeat48px = @"\ue78e",
    .repeatOne24px = @"\ue78f",
    .repeatOne48px = @"\ue790",
    .replay24px = @"\ue791",
    .replay48px = @"\ue792",
    .shuffle24px = @"\ue793",
    .shuffle48px = @"\ue794",
    .skipNext24px = @"\ue795",
    .skipNext48px = @"\ue796",
    .skipPrevious24px = @"\ue797",
    .skipPrevious48px = @"\ue798",
    .snooze24px = @"\ue799",
    .snooze48px = @"\ue79a",
    .stop24px = @"\ue79b",
    .stop48px = @"\ue79c",
    .subtitles24px = @"\ue79d",
    .subtitles48px = @"\ue79e",
    .surroundSound24px = @"\ue79f",
    .surroundSound48px = @"\ue7a0",
    .videoCollection24px = @"\ue7a1",
    .videoCollection48px = @"\ue7a2",
    .videocam24px = @"\ue7a3",
    .videocam48px = @"\ue7a4",
    .videocamOff24px = @"\ue7a5",
    .videocamOff48px = @"\ue7a6",
    .volumeDown18px = @"\ue7a7",
    .volumeDown24px = @"\ue7a8",
    .volumeDown48px = @"\ue7a9",
    .volumeMute18px = @"\ue7aa",
    .volumeMute24px = @"\ue7ab",
    .volumeMute48px = @"\ue7ac",
    .volumeOff24px = @"\ue7ad",
    .volumeOff48px = @"\ue7ae",
    .volumeUp18px = @"\ue7af",
    .volumeUp24px = @"\ue7b0",
    .volumeUp48px = @"\ue7b1",
    .web24px = @"\ue7b2",
    .web48px = @"\ue7b3",
    .business24px = @"\ue7b4",
    .business48px = @"\ue7b5",
    .call24px = @"\ue7b6",
    .call48px = @"\ue7b7",
    .callEnd24px = @"\ue7b8",
    .callEnd48px = @"\ue7b9",
    .callMade24px = @"\ue7ba",
    .callMade48px = @"\ue7bb",
    .callMerge24px = @"\ue7bc",
    .callMerge48px = @"\ue7bd",
    .callMissed24px = @"\ue7be",
    .callMissed48px = @"\ue7bf",
    .callReceived24px = @"\ue7c0",
    .callReceived48px = @"\ue7c1",
    .callSplit24px = @"\ue7c2",
    .callSplit48px = @"\ue7c3",
    .chat24px = @"\ue7c4",
    .chat48px = @"\ue7c5",
    .clearAll24px = @"\ue7c6",
    .clearAll48px = @"\ue7c7",
    .comment24px = @"\ue7c8",
    .comment48px = @"\ue7c9",
    .contacts24px = @"\ue7ca",
    .contacts48px = @"\ue7cb",
    .dialerSip24px = @"\ue7cc",
    .dialerSip48px = @"\ue7cd",
    .dialpad24px = @"\ue7ce",
    .dialpad48px = @"\ue7cf",
    .dndOn24px = @"\ue7d0",
    .dndOn48px = @"\ue7d1",
    .email24px = @"\ue7d2",
    .email48px = @"\ue7d3",
    .forum24px = @"\ue7d4",
    .forum48px = @"\ue7d5",
    .importExport24px = @"\ue7d6",
    .importExport48px = @"\ue7d7",
    .invertColorsOff24px = @"\ue7d8",
    .invertColorsOff48px = @"\ue7d9",
    .invertColorsOn24px = @"\ue7da",
    .invertColorsOn48px = @"\ue7db",
    .liveHelp24px = @"\ue7dc",
    .liveHelp48px = @"\ue7dd",
    .locationOff24px = @"\ue7de",
    .locationOff48px = @"\ue7df",
    .locationOn24px = @"\ue7e0",
    .locationOn48px = @"\ue7e1",
    .message24px = @"\ue7e2",
    .message48px = @"\ue7e3",
    .messenger24px = @"\ue7e4",
    .messenger48px = @"\ue7e5",
    .noSim24px = @"\ue7e6",
    .noSim48px = @"\ue7e7",
    .phone24px = @"\ue7e8",
    .phone48px = @"\ue7e9",
    .portableWifiOff24px = @"\ue7ea",
    .portableWifiOff48px = @"\ue7eb",
    .quickContactsDialer24px = @"\ue7ec",
    .quickContactsDialer48px = @"\ue7ed",
    .quickContactsMail24px = @"\ue7ee",
    .quickContactsMail48px = @"\ue7ef",
    .ringVolume24px = @"\ue7f0",
    .ringVolume48px = @"\ue7f1",
    .stayCurrentLandscape24px = @"\ue7f2",
    .stayCurrentLandscape48px = @"\ue7f3",
    .stayCurrentPortrait24px = @"\ue7f4",
    .stayCurrentPortrait48px = @"\ue7f5",
    .stayPrimaryLandscape24px = @"\ue7f6",
    .stayPrimaryLandscape48px = @"\ue7f7",
    .stayPrimaryPortrait24px = @"\ue7f8",
    .stayPrimaryPortrait48px = @"\ue7f9",
    .swapCalls24px = @"\ue7fa",
    .swapCalls48px = @"\ue7fb",
    .textsms24px = @"\ue7fc",
    .textsms48px = @"\ue7fd",
    .voicemail24px = @"\ue7fe",
    .voicemail48px = @"\ue7ff",
    .vpnKey24px = @"\ue800",
    .vpnKey48px = @"\ue801",
    .add24px = @"\ue802",
    .add48px = @"\ue803",
    .addBox24px = @"\ue804",
    .addBox48px = @"\ue805",
    .addCircle24px = @"\ue806",
    .addCircle48px = @"\ue807",
    .addCircleOutline24px = @"\ue808",
    .addCircleOutline48px = @"\ue809",
    .archive24px = @"\ue80a",
    .archive48px = @"\ue80b",
    .backspace24px = @"\ue80c",
    .backspace48px = @"\ue80d",
    .block24px = @"\ue80e",
    .block48px = @"\ue80f",
    .clear24px = @"\ue810",
    .clear48px = @"\ue811",
    .contentCopy24px = @"\ue812",
    .contentCopy48px = @"\ue813",
    .contentCut24px = @"\ue814",
    .contentCut48px = @"\ue815",
    .contentPaste24px = @"\ue816",
    .contentPaste48px = @"\ue817",
    .create24px = @"\ue818",
    .create48px = @"\ue819",
    .drafts24px = @"\ue81a",
    .drafts48px = @"\ue81b",
    .filterList24px = @"\ue81c",
    .filterList48px = @"\ue81d",
    .flag24px = @"\ue81e",
    .flag48px = @"\ue81f",
    .forward24px = @"\ue820",
    .forward48px = @"\ue821",
    .gesture24px = @"\ue822",
    .gesture48px = @"\ue823",
    .inbox24px = @"\ue824",
    .inbox48px = @"\ue825",
    .link24px = @"\ue826",
    .link48px = @"\ue827",
    .mail24px = @"\ue828",
    .mail48px = @"\ue829",
    .markunread24px = @"\ue82a",
    .markunread48px = @"\ue82b",
    .redo24px = @"\ue82c",
    .redo48px = @"\ue82d",
    .remove24px = @"\ue82e",
    .remove48px = @"\ue82f",
    .removeCircle24px = @"\ue830",
    .removeCircle48px = @"\ue831",
    .removeCircleOutline24px = @"\ue832",
    .removeCircleOutline48px = @"\ue833",
    .reply24px = @"\ue834",
    .reply48px = @"\ue835",
    .replyAll24px = @"\ue836",
    .replyAll48px = @"\ue837",
    .report24px = @"\ue838",
    .report48px = @"\ue839",
    .save24px = @"\ue83a",
    .save48px = @"\ue83b",
    .selectAll24px = @"\ue83c",
    .selectAll48px = @"\ue83d",
    .send24px = @"\ue83e",
    .send48px = @"\ue83f",
    .sort24px = @"\ue840",
    .sort48px = @"\ue841",
    .textFormat24px = @"\ue842",
    .textFormat48px = @"\ue843",
    .undo24px = @"\ue844",
    .undo48px = @"\ue845",
    .accessAlarm24px = @"\ue846",
    .accessAlarm48px = @"\ue847",
    .accessAlarms24px = @"\ue848",
    .accessAlarms48px = @"\ue849",
    .accessTime24px = @"\ue84a",
    .accessTime48px = @"\ue84b",
    .addAlarm24px = @"\ue84c",
    .addAlarm48px = @"\ue84d",
    .airplanemodeOff24px = @"\ue84e",
    .airplanemodeOff48px = @"\ue84f",
    .airplanemodeOn24px = @"\ue850",
    .airplanemodeOn48px = @"\ue851",
    .battery2018px = @"\ue852",
    .battery2024px = @"\ue853",
    .battery2048px = @"\ue854",
    .battery3018px = @"\ue855",
    .battery3024px = @"\ue856",
    .battery3048px = @"\ue857",
    .battery5018px = @"\ue858",
    .battery5024px = @"\ue859",
    .battery5048px = @"\ue85a",
    .battery6018px = @"\ue85b",
    .battery6024px = @"\ue85c",
    .battery6048px = @"\ue85d",
    .battery8018px = @"\ue85e",
    .battery8024px = @"\ue85f",
    .battery8048px = @"\ue860",
    .battery9018px = @"\ue861",
    .battery9024px = @"\ue862",
    .battery9048px = @"\ue863",
    .batteryAlert18px = @"\ue864",
    .batteryAlert24px = @"\ue865",
    .batteryAlert48px = @"\ue866",
    .batteryCharging2018px = @"\ue867",
    .batteryCharging2024px = @"\ue868",
    .batteryCharging2048px = @"\ue869",
    .batteryCharging3018px = @"\ue86a",
    .batteryCharging3024px = @"\ue86b",
    .batteryCharging3048px = @"\ue86c",
    .batteryCharging5018px = @"\ue86d",
    .batteryCharging5024px = @"\ue86e",
    .batteryCharging5048px = @"\ue86f",
    .batteryCharging6018px = @"\ue870",
    .batteryCharging6024px = @"\ue871",
    .batteryCharging6048px = @"\ue872",
    .batteryCharging8018px = @"\ue873",
    .batteryCharging8024px = @"\ue874",
    .batteryCharging8048px = @"\ue875",
    .batteryCharging9018px = @"\ue876",
    .batteryCharging9024px = @"\ue877",
    .batteryCharging9048px = @"\ue878",
    .batteryChargingFull18px = @"\ue879",
    .batteryChargingFull24px = @"\ue87a",
    .batteryChargingFull48px = @"\ue87b",
    .batteryFull18px = @"\ue87c",
    .batteryFull24px = @"\ue87d",
    .batteryFull48px = @"\ue87e",
    .batteryStd18px = @"\ue87f",
    .batteryStd24px = @"\ue880",
    .batteryStd48px = @"\ue881",
    .batteryUnknown18px = @"\ue882",
    .batteryUnknown24px = @"\ue883",
    .batteryUnknown48px = @"\ue884",
    .bluetooth24px = @"\ue885",
    .bluetooth48px = @"\ue886",
    .bluetoothConnected24px = @"\ue887",
    .bluetoothConnected48px = @"\ue888",
    .bluetoothDisabled24px = @"\ue889",
    .bluetoothDisabled48px = @"\ue88a",
    .bluetoothSearching24px = @"\ue88b",
    .bluetoothSearching48px = @"\ue88c",
    .brightnessAuto24px = @"\ue88d",
    .brightnessAuto48px = @"\ue88e",
    .brightnessHigh24px = @"\ue88f",
    .brightnessHigh48px = @"\ue890",
    .brightnessLow24px = @"\ue891",
    .brightnessLow48px = @"\ue892",
    .brightnessMedium24px = @"\ue893",
    .brightnessMedium48px = @"\ue894",
    .dataUsage24px = @"\ue895",
    .dataUsage48px = @"\ue896",
    .developerMode24px = @"\ue897",
    .developerMode48px = @"\ue898",
    .devices24px = @"\ue899",
    .devices48px = @"\ue89a",
    .dvr24px = @"\ue89b",
    .dvr48px = @"\ue89c",
    .gpsFixed24px = @"\ue89d",
    .gpsFixed48px = @"\ue89e",
    .gpsNotFixed24px = @"\ue89f",
    .gpsNotFixed48px = @"\ue8a0",
    .gpsOff24px = @"\ue8a1",
    .gpsOff48px = @"\ue8a2",
    .locationDisabled24px = @"\ue8a3",
    .locationDisabled48px = @"\ue8a4",
    .locationSearching24px = @"\ue8a5",
    .locationSearching48px = @"\ue8a6",
    .multitrackAudio24px = @"\ue8a7",
    .multitrackAudio48px = @"\ue8a8",
    .networkCell18px = @"\ue8a9",
    .networkCell24px = @"\ue8aa",
    .networkCell48px = @"\ue8ab",
    .networkWifi18px = @"\ue8ac",
    .networkWifi24px = @"\ue8ad",
    .networkWifi48px = @"\ue8ae",
    .nfc24px = @"\ue8af",
    .nfc48px = @"\ue8b0",
    .nowWallpaper18px = @"\ue8b1",
    .nowWallpaper24px = @"\ue8b2",
    .nowWallpaper48px = @"\ue8b3",
    .nowWidgets18px = @"\ue8b4",
    .nowWidgets24px = @"\ue8b5",
    .nowWidgets48px = @"\ue8b6",
    .screenLockLandscape24px = @"\ue8b7",
    .screenLockLandscape48px = @"\ue8b8",
    .screenLockPortrait24px = @"\ue8b9",
    .screenLockPortrait48px = @"\ue8ba",
    .screenLockRotation24px = @"\ue8bb",
    .screenLockRotation48px = @"\ue8bc",
    .screenRotation24px = @"\ue8bd",
    .screenRotation48px = @"\ue8be",
    .sdStorage24px = @"\ue8bf",
    .sdStorage48px = @"\ue8c0",
    .settingsSystemDaydream24px = @"\ue8c1",
    .settingsSystemDaydream48px = @"\ue8c2",
    .signalCellular0Bar18px = @"\ue8c3",
    .signalCellular0Bar24px = @"\ue8c4",
    .signalCellular0Bar48px = @"\ue8c5",
    .signalCellular1Bar18px = @"\ue8c6",
    .signalCellular1Bar24px = @"\ue8c7",
    .signalCellular1Bar48px = @"\ue8c8",
    .signalCellular2Bar18px = @"\ue8c9",
    .signalCellular2Bar24px = @"\ue8ca",
    .signalCellular2Bar48px = @"\ue8cb",
    .signalCellular3Bar18px = @"\ue8cc",
    .signalCellular3Bar24px = @"\ue8cd",
    .signalCellular3Bar48px = @"\ue8ce",
    .signalCellular4Bar18px = @"\ue8cf",
    .signalCellular4Bar24px = @"\ue8d0",
    .signalCellular4Bar48px = @"\ue8d1",
    .signalCellularConnectedNoInternet0Bar18px = @"\ue8d2",
    .signalCellularConnectedNoInternet0Bar24px = @"\ue8d3",
    .signalCellularConnectedNoInternet0Bar48px = @"\ue8d4",
    .signalCellularConnectedNoInternet1Bar18px = @"\ue8d5",
    .signalCellularConnectedNoInternet1Bar24px = @"\ue8d6",
    .signalCellularConnectedNoInternet1Bar48px = @"\ue8d7",
    .signalCellularConnectedNoInternet2Bar18px = @"\ue8d8",
    .signalCellularConnectedNoInternet2Bar24px = @"\ue8d9",
    .signalCellularConnectedNoInternet2Bar48px = @"\ue8da",
    .signalCellularConnectedNoInternet3Bar18px = @"\ue8db",
    .signalCellularConnectedNoInternet3Bar24px = @"\ue8dc",
    .signalCellularConnectedNoInternet3Bar48px = @"\ue8dd",
    .signalCellularConnectedNoInternet4Bar18px = @"\ue8de",
    .signalCellularConnectedNoInternet4Bar24px = @"\ue8df",
    .signalCellularConnectedNoInternet4Bar48px = @"\ue8e0",
    .signalCellularNoSim24px = @"\ue8e1",
    .signalCellularNoSim48px = @"\ue8e2",
    .signalCellularNull18px = @"\ue8e3",
    .signalCellularNull24px = @"\ue8e4",
    .signalCellularNull48px = @"\ue8e5",
    .signalCellularOff18px = @"\ue8e6",
    .signalCellularOff24px = @"\ue8e7",
    .signalCellularOff48px = @"\ue8e8",
    .signalWifi0Bar18px = @"\ue8e9",
    .signalWifi0Bar24px = @"\ue8ea",
    .signalWifi0Bar48px = @"\ue8eb",
    .signalWifi1Bar18px = @"\ue8ec",
    .signalWifi1Bar24px = @"\ue8ed",
    .signalWifi1Bar48px = @"\ue8ee",
    .signalWifi2Bar18px = @"\ue8ef",
    .signalWifi2Bar24px = @"\ue8f0",
    .signalWifi2Bar48px = @"\ue8f1",
    .signalWifi3Bar18px = @"\ue8f2",
    .signalWifi3Bar24px = @"\ue8f3",
    .signalWifi3Bar48px = @"\ue8f4",
    .signalWifi4Bar18px = @"\ue8f5",
    .signalWifi4Bar24px = @"\ue8f6",
    .signalWifi4Bar48px = @"\ue8f7",
    .signalWifiOff18px = @"\ue8f8",
    .signalWifiOff24px = @"\ue8f9",
    .signalWifiOff48px = @"\ue8fa",
    .signalWifiStatusbar1Bar26x24px = @"\ue8fb",
    .signalWifiStatusbar2Bar26x24px = @"\ue8fc",
    .signalWifiStatusbar3Bar26x24px = @"\ue8fd",
    .signalWifiStatusbar4Bar26x24px = @"\ue8fe",
    .signalWifiStatusbarConnectedNoInternet126x24px = @"\ue8ff",
    .signalWifiStatusbarConnectedNoInternet226x24px = @"\ue900",
    .signalWifiStatusbarConnectedNoInternet326x24px = @"\ue901",
    .signalWifiStatusbarConnectedNoInternet426x24px = @"\ue902",
    .signalWifiStatusbarConnectedNoInternet26x24px = @"\ue903",
    .signalWifiStatusbarNotConnected26x24px = @"\ue904",
    .signalWifiStatusbarNull26x24px = @"\ue905",
    .storage24px = @"\ue906",
    .storage48px = @"\ue907",
    .usb18px = @"\ue908",
    .usb24px = @"\ue909",
    .usb48px = @"\ue90a",
    .wifiLock24px = @"\ue90b",
    .wifiLock48px = @"\ue90c",
    .wifiTethering24px = @"\ue90d",
    .wifiTethering48px = @"\ue90e",
    .attachFile18px = @"\ue90f",
    .attachFile24px = @"\ue910",
    .attachFile48px = @"\ue911",
    .attachMoney18px = @"\ue912",
    .attachMoney24px = @"\ue913",
    .attachMoney48px = @"\ue914",
    .borderAll18px = @"\ue915",
    .borderAll24px = @"\ue916",
    .borderAll48px = @"\ue917",
    .borderBottom18px = @"\ue918",
    .borderBottom24px = @"\ue919",
    .borderBottom48px = @"\ue91a",
    .borderClear18px = @"\ue91b",
    .borderClear24px = @"\ue91c",
    .borderClear48px = @"\ue91d",
    .borderColor18px = @"\ue91e",
    .borderColor24px = @"\ue91f",
    .borderColor48px = @"\ue920",
    .borderHorizontal18px = @"\ue921",
    .borderHorizontal24px = @"\ue922",
    .borderHorizontal48px = @"\ue923",
    .borderInner18px = @"\ue924",
    .borderInner24px = @"\ue925",
    .borderInner48px = @"\ue926",
    .borderLeft18px = @"\ue927",
    .borderLeft24px = @"\ue928",
    .borderLeft48px = @"\ue929",
    .borderOuter18px = @"\ue92a",
    .borderOuter24px = @"\ue92b",
    .borderOuter48px = @"\ue92c",
    .borderRight18px = @"\ue92d",
    .borderRight24px = @"\ue92e",
    .borderRight48px = @"\ue92f",
    .borderStyle18px = @"\ue930",
    .borderStyle24px = @"\ue931",
    .borderStyle48px = @"\ue932",
    .borderTop18px = @"\ue933",
    .borderTop24px = @"\ue934",
    .borderTop48px = @"\ue935",
    .borderVertical18px = @"\ue936",
    .borderVertical24px = @"\ue937",
    .borderVertical48px = @"\ue938",
    .formatAlignCenter18px = @"\ue939",
    .formatAlignCenter24px = @"\ue93a",
    .formatAlignCenter48px = @"\ue93b",
    .formatAlignJustify18px = @"\ue93c",
    .formatAlignJustify24px = @"\ue93d",
    .formatAlignJustify48px = @"\ue93e",
    .formatAlignLeft18px = @"\ue93f",
    .formatAlignLeft24px = @"\ue940",
    .formatAlignLeft48px = @"\ue941",
    .formatAlignRight18px = @"\ue942",
    .formatAlignRight24px = @"\ue943",
    .formatAlignRight48px = @"\ue944",
    .formatBold18px = @"\ue945",
    .formatBold24px = @"\ue946",
    .formatBold48px = @"\ue947",
    .formatClear18px = @"\ue948",
    .formatClear24px = @"\ue949",
    .formatClear48px = @"\ue94a",
    .formatColorFill18px = @"\ue94b",
    .formatColorFill24px = @"\ue94c",
    .formatColorFill48px = @"\ue94d",
    .formatColorReset18px = @"\ue94e",
    .formatColorReset24px = @"\ue94f",
    .formatColorReset48px = @"\ue950",
    .formatColorText18px = @"\ue951",
    .formatColorText24px = @"\ue952",
    .formatColorText48px = @"\ue953",
    .formatIndentDecrease18px = @"\ue954",
    .formatIndentDecrease24px = @"\ue955",
    .formatIndentDecrease48px = @"\ue956",
    .formatIndentIncrease18px = @"\ue957",
    .formatIndentIncrease24px = @"\ue958",
    .formatIndentIncrease48px = @"\ue959",
    .formatItalic18px = @"\ue95a",
    .formatItalic24px = @"\ue95b",
    .formatItalic48px = @"\ue95c",
    .formatLineSpacing18px = @"\ue95d",
    .formatLineSpacing24px = @"\ue95e",
    .formatLineSpacing48px = @"\ue95f",
    .formatListBulleted18px = @"\ue960",
    .formatListBulleted24px = @"\ue961",
    .formatListBulleted48px = @"\ue962",
    .formatListNumbered18px = @"\ue963",
    .formatListNumbered24px = @"\ue964",
    .formatListNumbered48px = @"\ue965",
    .formatPaint18px = @"\ue966",
    .formatPaint24px = @"\ue967",
    .formatPaint48px = @"\ue968",
    .formatQuote18px = @"\ue969",
    .formatQuote24px = @"\ue96a",
    .formatQuote48px = @"\ue96b",
    .formatSize18px = @"\ue96c",
    .formatSize24px = @"\ue96d",
    .formatSize48px = @"\ue96e",
    .formatStrikethrough18px = @"\ue96f",
    .formatStrikethrough24px = @"\ue970",
    .formatStrikethrough48px = @"\ue971",
    .formatTextdirectionLToR18px = @"\ue972",
    .formatTextdirectionLToR24px = @"\ue973",
    .formatTextdirectionLToR48px = @"\ue974",
    .formatTextdirectionRToL18px = @"\ue975",
    .formatTextdirectionRToL24px = @"\ue976",
    .formatTextdirectionRToL48px = @"\ue977",
    .formatUnderline18px = @"\ue978",
    .formatUnderline24px = @"\ue979",
    .formatUnderline48px = @"\ue97a",
    .functions18px = @"\ue97b",
    .functions24px = @"\ue97c",
    .functions48px = @"\ue97d",
    .insertChart18px = @"\ue97e",
    .insertChart24px = @"\ue97f",
    .insertChart48px = @"\ue980",
    .insertComment18px = @"\ue981",
    .insertComment24px = @"\ue982",
    .insertComment48px = @"\ue983",
    .insertDriveFile18px = @"\ue984",
    .insertDriveFile24px = @"\ue985",
    .insertDriveFile48px = @"\ue986",
    .insertEmoticon18px = @"\ue987",
    .insertEmoticon24px = @"\ue988",
    .insertEmoticon48px = @"\ue989",
    .insertInvitation18px = @"\ue98a",
    .insertInvitation24px = @"\ue98b",
    .insertInvitation48px = @"\ue98c",
    .insertLink18px = @"\ue98d",
    .insertLink24px = @"\ue98e",
    .insertLink48px = @"\ue98f",
    .insertPhoto18px = @"\ue990",
    .insertPhoto24px = @"\ue991",
    .insertPhoto48px = @"\ue992",
    .mergeType18px = @"\ue993",
    .mergeType24px = @"\ue994",
    .mergeType48px = @"\ue995",
    .modeComment18px = @"\ue996",
    .modeComment24px = @"\ue997",
    .modeComment48px = @"\ue998",
    .modeEdit18px = @"\ue999",
    .modeEdit24px = @"\ue99a",
    .modeEdit48px = @"\ue99b",
    .publish18px = @"\ue99c",
    .publish24px = @"\ue99d",
    .publish48px = @"\ue99e",
    .verticalAlignBottom18px = @"\ue99f",
    .verticalAlignBottom24px = @"\ue9a0",
    .verticalAlignBottom48px = @"\ue9a1",
    .verticalAlignCenter18px = @"\ue9a2",
    .verticalAlignCenter24px = @"\ue9a3",
    .verticalAlignCenter48px = @"\ue9a4",
    .verticalAlignTop18px = @"\ue9a5",
    .verticalAlignTop24px = @"\ue9a6",
    .verticalAlignTop48px = @"\ue9a7",
    .wrapText18px = @"\ue9a8",
    .wrapText24px = @"\ue9a9",
    .wrapText48px = @"\ue9aa",
    .attachment18px = @"\ue9ab",
    .attachment24px = @"\ue9ac",
    .attachment48px = @"\ue9ad",
    .cloud24px = @"\ue9ae",
    .cloud48px = @"\ue9af",
    .cloudCircle18px = @"\ue9b0",
    .cloudCircle24px = @"\ue9b1",
    .cloudCircle48px = @"\ue9b2",
    .cloudDone24px = @"\ue9b3",
    .cloudDone48px = @"\ue9b4",
    .cloudDownload24px = @"\ue9b5",
    .cloudDownload48px = @"\ue9b6",
    .cloudOff24px = @"\ue9b7",
    .cloudOff48px = @"\ue9b8",
    .cloudQueue24px = @"\ue9b9",
    .cloudQueue48px = @"\ue9ba",
    .cloudUpload24px = @"\ue9bb",
    .cloudUpload48px = @"\ue9bc",
    .fileDownload24px = @"\ue9bd",
    .fileDownload48px = @"\ue9be",
    .fileUpload24px = @"\ue9bf",
    .fileUpload48px = @"\ue9c0",
    .folder18px = @"\ue9c1",
    .folder24px = @"\ue9c2",
    .folder48px = @"\ue9c3",
    .folderOpen18px = @"\ue9c4",
    .folderOpen24px = @"\ue9c5",
    .folderOpen48px = @"\ue9c6",
    .folderShared18px = @"\ue9c7",
    .folderShared24px = @"\ue9c8",
    .folderShared48px = @"\ue9c9",
    .cast24px = @"\ue9ca",
    .cast48px = @"\ue9cb",
    .castConnected24px = @"\ue9cc",
    .castConnected48px = @"\ue9cd",
    .computer24px = @"\ue9ce",
    .computer48px = @"\ue9cf",
    .desktopMac24px = @"\ue9d0",
    .desktopMac48px = @"\ue9d1",
    .desktopWindows24px = @"\ue9d2",
    .desktopWindows48px = @"\ue9d3",
    .dock24px = @"\ue9d4",
    .dock48px = @"\ue9d5",
    .gamepad24px = @"\ue9d6",
    .gamepad48px = @"\ue9d7",
    .headset24px = @"\ue9d8",
    .headset48px = @"\ue9d9",
    .headsetMic24px = @"\ue9da",
    .headsetMic48px = @"\ue9db",
    .keyboard24px = @"\ue9dc",
    .keyboard48px = @"\ue9dd",
    .keyboardAlt24px = @"\ue9de",
    .keyboardAlt48px = @"\ue9df",
    .keyboardArrowDown24px = @"\ue9e0",
    .keyboardArrowDown48px = @"\ue9e1",
    .keyboardArrowLeft24px = @"\ue9e2",
    .keyboardArrowLeft48px = @"\ue9e3",
    .keyboardArrowRight24px = @"\ue9e4",
    .keyboardArrowRight48px = @"\ue9e5",
    .keyboardArrowUp24px = @"\ue9e6",
    .keyboardArrowUp48px = @"\ue9e7",
    .keyboardBackspace24px = @"\ue9e8",
    .keyboardBackspace48px = @"\ue9e9",
    .keyboardCapslock24px = @"\ue9ea",
    .keyboardCapslock48px = @"\ue9eb",
    .keyboardControl24px = @"\ue9ec",
    .keyboardControl48px = @"\ue9ed",
    .keyboardHide24px = @"\ue9ee",
    .keyboardHide48px = @"\ue9ef",
    .keyboardReturn24px = @"\ue9f0",
    .keyboardReturn48px = @"\ue9f1",
    .keyboardTab24px = @"\ue9f2",
    .keyboardTab48px = @"\ue9f3",
    .keyboardVoice24px = @"\ue9f4",
    .keyboardVoice48px = @"\ue9f5",
    .laptop24px = @"\ue9f6",
    .laptop48px = @"\ue9f7",
    .laptopChromebook24px = @"\ue9f8",
    .laptopChromebook48px = @"\ue9f9",
    .laptopMac24px = @"\ue9fa",
    .laptopMac48px = @"\ue9fb",
    .laptopWindows24px = @"\ue9fc",
    .laptopWindows48px = @"\ue9fd",
    .memory24px = @"\ue9fe",
    .memory48px = @"\ue9ff",
    .mouse24px = @"\uea00",
    .mouse48px = @"\uea01",
    .phoneAndroid24px = @"\uea02",
    .phoneAndroid48px = @"\uea03",
    .phoneIphone24px = @"\uea04",
    .phoneIphone48px = @"\uea05",
    .phonelink24px = @"\uea06",
    .phonelink48px = @"\uea07",
    .phonelinkOff24px = @"\uea08",
    .phonelinkOff48px = @"\uea09",
    .security24px = @"\uea0a",
    .security48px = @"\uea0b",
    .simCard24px = @"\uea0c",
    .simCard48px = @"\uea0d",
    .smartphone24px = @"\uea0e",
    .smartphone48px = @"\uea0f",
    .speaker24px = @"\uea10",
    .speaker48px = @"\uea11",
    .tablet24px = @"\uea12",
    .tablet48px = @"\uea13",
    .tabletAndroid24px = @"\uea14",
    .tabletAndroid48px = @"\uea15",
    .tabletMac24px = @"\uea16",
    .tabletMac48px = @"\uea17",
    .tv24px = @"\uea18",
    .tv48px = @"\uea19",
    .watch24px = @"\uea1a",
    .watch48px = @"\uea1b",
    .addToPhotos24px = @"\uea1c",
    .addToPhotos48px = @"\uea1d",
    .adjust24px = @"\uea1e",
    .adjust48px = @"\uea1f",
    .assistantPhoto24px = @"\uea20",
    .assistantPhoto48px = @"\uea21",
    .audiotrack24px = @"\uea22",
    .audiotrack48px = @"\uea23",
    .blurCircular24px = @"\uea24",
    .blurCircular48px = @"\uea25",
    .blurLinear24px = @"\uea26",
    .blurLinear48px = @"\uea27",
    .blurOff24px = @"\uea28",
    .blurOff48px = @"\uea29",
    .blurOn24px = @"\uea2a",
    .blurOn48px = @"\uea2b",
    .brightness124px = @"\uea2c",
    .brightness148px = @"\uea2d",
    .brightness224px = @"\uea2e",
    .brightness248px = @"\uea2f",
    .brightness324px = @"\uea30",
    .brightness348px = @"\uea31",
    .brightness424px = @"\uea32",
    .brightness448px = @"\uea33",
    .brightness524px = @"\uea34",
    .brightness548px = @"\uea35",
    .brightness624px = @"\uea36",
    .brightness648px = @"\uea37",
    .brightness724px = @"\uea38",
    .brightness748px = @"\uea39",
    .brush24px = @"\uea3a",
    .brush48px = @"\uea3b",
    .camera24px = @"\uea3c",
    .camera48px = @"\uea3d",
    .cameraAlt24px = @"\uea3e",
    .cameraAlt48px = @"\uea3f",
    .cameraFront24px = @"\uea40",
    .cameraFront48px = @"\uea41",
    .cameraRear24px = @"\uea42",
    .cameraRear48px = @"\uea43",
    .cameraRoll24px = @"\uea44",
    .cameraRoll48px = @"\uea45",
    .centerFocusStrong24px = @"\uea46",
    .centerFocusStrong48px = @"\uea47",
    .centerFocusWeak24px = @"\uea48",
    .centerFocusWeak48px = @"\uea49",
    .collections24px = @"\uea4a",
    .collections48px = @"\uea4b",
    .colorLens24px = @"\uea4c",
    .colorLens48px = @"\uea4d",
    .colorize24px = @"\uea4e",
    .colorize48px = @"\uea4f",
    .compare24px = @"\uea50",
    .compare48px = @"\uea51",
    .controlPoint24px = @"\uea52",
    .controlPoint48px = @"\uea53",
    .controlPointDuplicate24px = @"\uea54",
    .controlPointDuplicate48px = @"\uea55",
    .crop3224px = @"\uea56",
    .crop3248px = @"\uea57",
    .crop5424px = @"\uea58",
    .crop5448px = @"\uea59",
    .crop7524px = @"\uea5a",
    .crop7548px = @"\uea5b",
    .crop16924px = @"\uea5c",
    .crop16948px = @"\uea5d",
    .crop24px = @"\uea5e",
    .crop48px = @"\uea5f",
    .cropDin24px = @"\uea60",
    .cropDin48px = @"\uea61",
    .cropFree24px = @"\uea62",
    .cropFree48px = @"\uea63",
    .cropLandscape24px = @"\uea64",
    .cropLandscape48px = @"\uea65",
    .cropOriginal24px = @"\uea66",
    .cropOriginal48px = @"\uea67",
    .cropPortrait24px = @"\uea68",
    .cropPortrait48px = @"\uea69",
    .cropSquare24px = @"\uea6a",
    .cropSquare48px = @"\uea6b",
    .dehaze24px = @"\uea6c",
    .dehaze48px = @"\uea6d",
    .details24px = @"\uea6e",
    .details48px = @"\uea6f",
    .edit24px = @"\uea70",
    .edit48px = @"\uea71",
    .exposure24px = @"\uea72",
    .exposure48px = @"\uea73",
    .exposureMinus124px = @"\uea74",
    .exposureMinus148px = @"\uea75",
    .exposureMinus224px = @"\uea76",
    .exposureMinus248px = @"\uea77",
    .exposurePlus124px = @"\uea78",
    .exposurePlus148px = @"\uea79",
    .exposurePlus224px = @"\uea7a",
    .exposurePlus248px = @"\uea7b",
    .exposureZero24px = @"\uea7c",
    .exposureZero48px = @"\uea7d",
    .filter124px = @"\uea7e",
    .filter148px = @"\uea7f",
    .filter224px = @"\uea80",
    .filter248px = @"\uea81",
    .filter324px = @"\uea82",
    .filter348px = @"\uea83",
    .filter424px = @"\uea84",
    .filter448px = @"\uea85",
    .filter524px = @"\uea86",
    .filter548px = @"\uea87",
    .filter624px = @"\uea88",
    .filter648px = @"\uea89",
    .filter724px = @"\uea8a",
    .filter748px = @"\uea8b",
    .filter824px = @"\uea8c",
    .filter848px = @"\uea8d",
    .filter924px = @"\uea8e",
    .filter948px = @"\uea8f",
    .filter9Plus24px = @"\uea90",
    .filter9Plus48px = @"\uea91",
    .filter24px = @"\uea92",
    .filter48px = @"\uea93",
    .filterBAndW24px = @"\uea94",
    .filterBAndW48px = @"\uea95",
    .filterCenterFocus24px = @"\uea96",
    .filterCenterFocus48px = @"\uea97",
    .filterDrama24px = @"\uea98",
    .filterDrama48px = @"\uea99",
    .filterFrames24px = @"\uea9a",
    .filterFrames48px = @"\uea9b",
    .filterHdr24px = @"\uea9c",
    .filterHdr48px = @"\uea9d",
    .filterNone24px = @"\uea9e",
    .filterNone48px = @"\uea9f",
    .filterTiltShift24px = @"\ueaa0",
    .filterTiltShift48px = @"\ueaa1",
    .filterVintage24px = @"\ueaa2",
    .filterVintage48px = @"\ueaa3",
    .flare24px = @"\ueaa4",
    .flare48px = @"\ueaa5",
    .flashAuto24px = @"\ueaa6",
    .flashAuto48px = @"\ueaa7",
    .flashOff24px = @"\ueaa8",
    .flashOff48px = @"\ueaa9",
    .flashOn24px = @"\ueaaa",
    .flashOn48px = @"\ueaab",
    .flip24px = @"\ueaac",
    .flip48px = @"\ueaad",
    .gradient24px = @"\ueaae",
    .gradient48px = @"\ueaaf",
    .grain24px = @"\ueab0",
    .grain48px = @"\ueab1",
    .gridOff24px = @"\ueab2",
    .gridOff48px = @"\ueab3",
    .gridOn24px = @"\ueab4",
    .gridOn48px = @"\ueab5",
    .hdrOff24px = @"\ueab6",
    .hdrOff48px = @"\ueab7",
    .hdrOn24px = @"\ueab8",
    .hdrOn48px = @"\ueab9",
    .hdrStrong24px = @"\ueaba",
    .hdrStrong48px = @"\ueabb",
    .hdrWeak24px = @"\ueabc",
    .hdrWeak48px = @"\ueabd",
    .healing24px = @"\ueabe",
    .healing48px = @"\ueabf",
    .image24px = @"\ueac0",
    .image48px = @"\ueac1",
    .imageAspectRatio24px = @"\ueac2",
    .imageAspectRatio48px = @"\ueac3",
    .iso24px = @"\ueac4",
    .iso48px = @"\ueac5",
    .landscape24px = @"\ueac6",
    .landscape48px = @"\ueac7",
    .leakAdd24px = @"\ueac8",
    .leakAdd48px = @"\ueac9",
    .leakRemove24px = @"\ueaca",
    .leakRemove48px = @"\ueacb",
    .lens24px = @"\ueacc",
    .lens48px = @"\ueacd",
    .looks324px = @"\ueace",
    .looks348px = @"\ueacf",
    .looks424px = @"\uead0",
    .looks448px = @"\uead1",
    .looks524px = @"\uead2",
    .looks548px = @"\uead3",
    .looks624px = @"\uead4",
    .looks648px = @"\uead5",
    .looks24px = @"\uead6",
    .looks48px = @"\uead7",
    .looksOne24px = @"\uead8",
    .looksOne48px = @"\uead9",
    .looksTwo24px = @"\ueada",
    .looksTwo48px = @"\ueadb",
    .loupe24px = @"\ueadc",
    .loupe48px = @"\ueadd",
    .movieCreation24px = @"\ueade",
    .movieCreation48px = @"\ueadf",
    .nature24px = @"\ueae0",
    .nature48px = @"\ueae1",
    .naturePeople24px = @"\ueae2",
    .naturePeople48px = @"\ueae3",
    .navigateBefore24px = @"\ueae4",
    .navigateBefore48px = @"\ueae5",
    .navigateNext24px = @"\ueae6",
    .navigateNext48px = @"\ueae7",
    .palette24px = @"\ueae8",
    .palette48px = @"\ueae9",
    .panorama24px = @"\ueaea",
    .panorama48px = @"\ueaeb",
    .panoramaFisheye24px = @"\ueaec",
    .panoramaFisheye48px = @"\ueaed",
    .panoramaHorizontal24px = @"\ueaee",
    .panoramaHorizontal48px = @"\ueaef",
    .panoramaVertical24px = @"\ueaf0",
    .panoramaVertical48px = @"\ueaf1",
    .panoramaWideAngle24px = @"\ueaf2",
    .panoramaWideAngle48px = @"\ueaf3",
    .photo24px = @"\ueaf4",
    .photo48px = @"\ueaf5",
    .photoAlbum24px = @"\ueaf6",
    .photoAlbum48px = @"\ueaf7",
    .photoCamera24px = @"\ueaf8",
    .photoCamera48px = @"\ueaf9",
    .photoLibrary24px = @"\ueafa",
    .photoLibrary48px = @"\ueafb",
    .portrait24px = @"\ueafc",
    .portrait48px = @"\ueafd",
    .removeRedEye24px = @"\ueafe",
    .removeRedEye48px = @"\ueaff",
    .rotateLeft24px = @"\ueb00",
    .rotateLeft48px = @"\ueb01",
    .rotateRight24px = @"\ueb02",
    .rotateRight48px = @"\ueb03",
    .slideshow24px = @"\ueb04",
    .slideshow48px = @"\ueb05",
    .straighten24px = @"\ueb06",
    .straighten48px = @"\ueb07",
    .style24px = @"\ueb08",
    .style48px = @"\ueb09",
    .switchCamera24px = @"\ueb0a",
    .switchCamera48px = @"\ueb0b",
    .switchVideo24px = @"\ueb0c",
    .switchVideo48px = @"\ueb0d",
    .tagFaces24px = @"\ueb0e",
    .tagFaces48px = @"\ueb0f",
    .texture24px = @"\ueb10",
    .texture48px = @"\ueb11",
    .timelapse24px = @"\ueb12",
    .timelapse48px = @"\ueb13",
    .timer324px = @"\ueb14",
    .timer348px = @"\ueb15",
    .timer1024px = @"\ueb16",
    .timer1048px = @"\ueb17",
    .timer24px = @"\ueb18",
    .timer48px = @"\ueb19",
    .timerAuto24px = @"\ueb1a",
    .timerAuto48px = @"\ueb1b",
    .timerOff24px = @"\ueb1c",
    .timerOff48px = @"\ueb1d",
    .tonality24px = @"\ueb1e",
    .tonality48px = @"\ueb1f",
    .transform24px = @"\ueb20",
    .transform48px = @"\ueb21",
    .tune24px = @"\ueb22",
    .tune48px = @"\ueb23",
    .wbAuto24px = @"\ueb24",
    .wbAuto48px = @"\ueb25",
    .wbCloudy24px = @"\ueb26",
    .wbCloudy48px = @"\ueb27",
    .wbIncandescent24px = @"\ueb28",
    .wbIncandescent48px = @"\ueb29",
    .wbIrradescent24px = @"\ueb2a",
    .wbIrradescent48px = @"\ueb2b",
    .wbSunny24px = @"\ueb2c",
    .wbSunny48px = @"\ueb2d",
    .checkBox24px = @"\ueb2e",
    .checkBox48px = @"\ueb2f",
    .checkBoxOutlineBlank24px = @"\ueb30",
    .checkBoxOutlineBlank48px = @"\ueb31",
    .radioButtonOff24px = @"\ueb32",
    .radioButtonOff48px = @"\ueb33",
    .radioButtonOn24px = @"\ueb34",
    .radioButtonOn48px = @"\ueb35",
    .star24px = @"\ueb36",
    .starHalf24px = @"\ueb37",
    .starOutline24px = @"\ueb38",
    .beenhere24px = @"\ueb39",
    .beenhere48px = @"\ueb3a",
    .directions24px = @"\ueb3b",
    .directions48px = @"\ueb3c",
    .directionsBike24px = @"\ueb3d",
    .directionsBike48px = @"\ueb3e",
    .directionsBus24px = @"\ueb3f",
    .directionsBus48px = @"\ueb40",
    .directionsCar24px = @"\ueb41",
    .directionsCar48px = @"\ueb42",
    .directionsFerry24px = @"\ueb43",
    .directionsFerry48px = @"\ueb44",
    .directionsSubway24px = @"\ueb45",
    .directionsSubway48px = @"\ueb46",
    .directionsTrain24px = @"\ueb47",
    .directionsTrain48px = @"\ueb48",
    .directionsTransit24px = @"\ueb49",
    .directionsTransit48px = @"\ueb4a",
    .directionsWalk24px = @"\ueb4b",
    .directionsWalk48px = @"\ueb4c",
    .flight24px = @"\ueb4d",
    .flight48px = @"\ueb4e",
    .hotel24px = @"\ueb4f",
    .hotel48px = @"\ueb50",
    .layers24px = @"\ueb51",
    .layers48px = @"\ueb52",
    .layersClear24px = @"\ueb53",
    .layersClear48px = @"\ueb54",
    .localAirport24px = @"\ueb55",
    .localAirport48px = @"\ueb56",
    .localAtm24px = @"\ueb57",
    .localAtm48px = @"\ueb58",
    .localAttraction24px = @"\ueb59",
    .localAttraction48px = @"\ueb5a",
    .localBar24px = @"\ueb5b",
    .localBar48px = @"\ueb5c",
    .localCafe24px = @"\ueb5d",
    .localCafe48px = @"\ueb5e",
    .localCarWash24px = @"\ueb5f",
    .localCarWash48px = @"\ueb60",
    .localConvenienceStore24px = @"\ueb61",
    .localConvenienceStore48px = @"\ueb62",
    .localDrink24px = @"\ueb63",
    .localDrink48px = @"\ueb64",
    .localFlorist24px = @"\ueb65",
    .localFlorist48px = @"\ueb66",
    .localGasStation24px = @"\ueb67",
    .localGasStation48px = @"\ueb68",
    .localGroceryStore24px = @"\ueb69",
    .localGroceryStore48px = @"\ueb6a",
    .localHospital24px = @"\ueb6b",
    .localHospital48px = @"\ueb6c",
    .localHotel24px = @"\ueb6d",
    .localHotel48px = @"\ueb6e",
    .localLaundryService24px = @"\ueb6f",
    .localLaundryService48px = @"\ueb70",
    .localLibrary24px = @"\ueb71",
    .localLibrary48px = @"\ueb72",
    .localMall24px = @"\ueb73",
    .localMall48px = @"\ueb74",
    .localMovies24px = @"\ueb75",
    .localMovies48px = @"\ueb76",
    .localOffer24px = @"\ueb77",
    .localOffer48px = @"\ueb78",
    .localParking24px = @"\ueb79",
    .localParking48px = @"\ueb7a",
    .localPharmacy24px = @"\ueb7b",
    .localPharmacy48px = @"\ueb7c",
    .localPhone24px = @"\ueb7d",
    .localPhone48px = @"\ueb7e",
    .localPizza24px = @"\ueb7f",
    .localPizza48px = @"\ueb80",
    .localPlay24px = @"\ueb81",
    .localPlay48px = @"\ueb82",
    .localPostOffice24px = @"\ueb83",
    .localPostOffice48px = @"\ueb84",
    .localPrintShop24px = @"\ueb85",
    .localPrintShop48px = @"\ueb86",
    .localRestaurant24px = @"\ueb87",
    .localRestaurant48px = @"\ueb88",
    .localSee24px = @"\ueb89",
    .localSee48px = @"\ueb8a",
    .localShipping24px = @"\ueb8b",
    .localShipping48px = @"\ueb8c",
    .localTaxi24px = @"\ueb8d",
    .localTaxi48px = @"\ueb8e",
    .locationHistory24px = @"\ueb8f",
    .locationHistory48px = @"\ueb90",
    .map24px = @"\ueb91",
    .map48px = @"\ueb92",
    .myLocation24px = @"\ueb93",
    .myLocation48px = @"\ueb94",
    .navigation24px = @"\ueb95",
    .navigation48px = @"\ueb96",
    .pinDrop24px = @"\ueb97",
    .pinDrop48px = @"\ueb98",
    .place24px = @"\ueb99",
    .place48px = @"\ueb9a",
    .rateReview24px = @"\ueb9b",
    .rateReview48px = @"\ueb9c",
    .restaurantMenu24px = @"\ueb9d",
    .restaurantMenu48px = @"\ueb9e",
    .satellite24px = @"\ueb9f",
    .satellite48px = @"\ueba0",
    .storeMallDirectory24px = @"\ueba1",
    .storeMallDirectory48px = @"\ueba2",
    .terrain24px = @"\ueba3",
    .terrain48px = @"\ueba4",
    .traffic24px = @"\ueba5",
    .traffic48px = @"\ueba6",
    .apps18px = @"\ueba7",
    .apps24px = @"\ueba8",
    .apps36px = @"\ueba9",
    .apps48px = @"\uebaa",
    .arrowBack18px = @"\uebab",
    .arrowBack24px = @"\uebac",
    .arrowBack36px = @"\uebad",
    .arrowBack48px = @"\uebae",
    .arrowDropDown18px = @"\uebaf",
    .arrowDropDown24px = @"\uebb0",
    .arrowDropDown36px = @"\uebb1",
    .arrowDropDown48px = @"\uebb2",
    .arrowDropDownCircle24px = @"\uebb3",
    .arrowDropDownCircle48px = @"\uebb4",
    .arrowDropUp18px = @"\uebb5",
    .arrowDropUp24px = @"\uebb6",
    .arrowDropUp36px = @"\uebb7",
    .arrowDropUp48px = @"\uebb8",
    .arrowForward18px = @"\uebb9",
    .arrowForward24px = @"\uebba",
    .arrowForward36px = @"\uebbb",
    .arrowForward48px = @"\uebbc",
    .cancel18px = @"\uebbd",
    .cancel24px = @"\uebbe",
    .cancel36px = @"\uebbf",
    .cancel48px = @"\uebc0",
    .check18px = @"\uebc1",
    .check24px = @"\uebc2",
    .check36px = @"\uebc3",
    .check48px = @"\uebc4",
    .chevronLeft18px = @"\uebc5",
    .chevronLeft24px = @"\uebc6",
    .chevronLeft36px = @"\uebc7",
    .chevronLeft48px = @"\uebc8",
    .chevronRight18px = @"\uebc9",
    .chevronRight24px = @"\uebca",
    .chevronRight36px = @"\uebcb",
    .chevronRight48px = @"\uebcc",
    .close18px = @"\uebcd",
    .close24px = @"\uebce",
    .close36px = @"\uebcf",
    .close48px = @"\uebd0",
    .expandLess18px = @"\uebd1",
    .expandLess24px = @"\uebd2",
    .expandLess36px = @"\uebd3",
    .expandLess48px = @"\uebd4",
    .expandMore18px = @"\uebd5",
    .expandMore24px = @"\uebd6",
    .expandMore36px = @"\uebd7",
    .expandMore48px = @"\uebd8",
    .fullscreen18px = @"\uebd9",
    .fullscreen24px = @"\uebda",
    .fullscreen36px = @"\uebdb",
    .fullscreen48px = @"\uebdc",
    .fullscreenExit18px = @"\uebdd",
    .fullscreenExit24px = @"\uebde",
    .fullscreenExit36px = @"\uebdf",
    .fullscreenExit48px = @"\uebe0",
    .menu18px = @"\uebe1",
    .menu24px = @"\uebe2",
    .menu36px = @"\uebe3",
    .menu48px = @"\uebe4",
    .moreHoriz18px = @"\uebe5",
    .moreHoriz24px = @"\uebe6",
    .moreHoriz36px = @"\uebe7",
    .moreHoriz48px = @"\uebe8",
    .moreVert18px = @"\uebe9",
    .moreVert24px = @"\uebea",
    .moreVert36px = @"\uebeb",
    .moreVert48px = @"\uebec",
    .refresh18px = @"\uebed",
    .refresh24px = @"\uebee",
    .refresh36px = @"\uebef",
    .refresh48px = @"\uebf0",
    .unfoldLess18px = @"\uebf1",
    .unfoldLess24px = @"\uebf2",
    .unfoldLess36px = @"\uebf3",
    .unfoldLess48px = @"\uebf4",
    .unfoldMore18px = @"\uebf5",
    .unfoldMore24px = @"\uebf6",
    .unfoldMore36px = @"\uebf7",
    .unfoldMore48px = @"\uebf8",
    .adb18px = @"\uebf9",
    .adb24px = @"\uebfa",
    .adb48px = @"\uebfb",
    .bluetoothAudio24px = @"\uebfc",
    .bluetoothAudio48px = @"\uebfd",
    .discFull24px = @"\uebfe",
    .discFull48px = @"\uebff",
    .dndForwardslash24px = @"\uec00",
    .dndForwardslash48px = @"\uec01",
    .doNotDisturb24px = @"\uec02",
    .doNotDisturb48px = @"\uec03",
    .driveEta24px = @"\uec04",
    .driveEta48px = @"\uec05",
    .eventAvailable24px = @"\uec06",
    .eventAvailable48px = @"\uec07",
    .eventBusy24px = @"\uec08",
    .eventBusy48px = @"\uec09",
    .eventNote18px = @"\uec0a",
    .eventNote24px = @"\uec0b",
    .eventNote48px = @"\uec0c",
    .folderSpecial24px = @"\uec0d",
    .folderSpecial48px = @"\uec0e",
    .mms24px = @"\uec0f",
    .mms48px = @"\uec10",
    .more24px = @"\uec11",
    .more48px = @"\uec12",
    .networkLocked24px = @"\uec13",
    .networkLocked48px = @"\uec14",
    .phoneBluetoothSpeaker24px = @"\uec15",
    .phoneBluetoothSpeaker48px = @"\uec16",
    .phoneForwarded24px = @"\uec17",
    .phoneForwarded48px = @"\uec18",
    .phoneInTalk24px = @"\uec19",
    .phoneInTalk48px = @"\uec1a",
    .phoneLocked24px = @"\uec1b",
    .phoneLocked48px = @"\uec1c",
    .phoneMissed24px = @"\uec1d",
    .phoneMissed48px = @"\uec1e",
    .phonePaused24px = @"\uec1f",
    .phonePaused48px = @"\uec20",
    .playDownload24px = @"\uec21",
    .playDownload48px = @"\uec22",
    .playInstall24px = @"\uec23",
    .playInstall48px = @"\uec24",
    .sdCard24px = @"\uec25",
    .sdCard48px = @"\uec26",
    .simCardAlert24px = @"\uec27",
    .simCardAlert48px = @"\uec28",
    .sms24px = @"\uec29",
    .sms48px = @"\uec2a",
    .smsFailed24px = @"\uec2b",
    .smsFailed48px = @"\uec2c",
    .sync24px = @"\uec2d",
    .sync48px = @"\uec2e",
    .syncDisabled24px = @"\uec2f",
    .syncDisabled48px = @"\uec30",
    .syncProblem24px = @"\uec31",
    .syncProblem48px = @"\uec32",
    .systemUpdate24px = @"\uec33",
    .systemUpdate48px = @"\uec34",
    .tapAndPlay24px = @"\uec35",
    .tapAndPlay48px = @"\uec36",
    .timeToLeave24px = @"\uec37",
    .timeToLeave48px = @"\uec38",
    .vibration18px = @"\uec39",
    .vibration24px = @"\uec3a",
    .vibration48px = @"\uec3b",
    .voiceChat24px = @"\uec3c",
    .voiceChat48px = @"\uec3d",
    .vpnLock24px = @"\uec3e",
    .vpnLock48px = @"\uec3f",
    .cake18px = @"\uec40",
    .cake24px = @"\uec41",
    .cake48px = @"\uec42",
    .domain18px = @"\uec43",
    .domain24px = @"\uec44",
    .domain48px = @"\uec45",
    .group18px = @"\uec46",
    .group24px = @"\uec47",
    .group48px = @"\uec48",
    .groupAdd18px = @"\uec49",
    .groupAdd24px = @"\uec4a",
    .groupAdd48px = @"\uec4b",
    .locationCity18px = @"\uec4c",
    .locationCity24px = @"\uec4d",
    .locationCity48px = @"\uec4e",
    .mood18px = @"\uec4f",
    .mood24px = @"\uec50",
    .mood48px = @"\uec51",
    .notifications24px = @"\uec52",
    .notifications48px = @"\uec53",
    .notificationsNone24px = @"\uec54",
    .notificationsNone48px = @"\uec55",
    .notificationsOff24px = @"\uec56",
    .notificationsOff48px = @"\uec57",
    .notificationsOn24px = @"\uec58",
    .notificationsOn48px = @"\uec59",
    .notificationsPaused24px = @"\uec5a",
    .notificationsPaused48px = @"\uec5b",
    .pages18px = @"\uec5c",
    .pages24px = @"\uec5d",
    .pages48px = @"\uec5e",
    .partyMode24px = @"\uec5f",
    .partyMode48px = @"\uec60",
    .people18px = @"\uec61",
    .people24px = @"\uec62",
    .people48px = @"\uec63",
    .peopleOutline24px = @"\uec64",
    .peopleOutline48px = @"\uec65",
    .person18px = @"\uec66",
    .person24px = @"\uec67",
    .person48px = @"\uec68",
    .personAdd18px = @"\uec69",
    .personAdd24px = @"\uec6a",
    .personAdd48px = @"\uec6b",
    .personOutline18px = @"\uec6c",
    .personOutline24px = @"\uec6d",
    .personOutline48px = @"\uec6e",
    .plusOne24px = @"\uec6f",
    .plusOne48px = @"\uec70",
    .poll18px = @"\uec71",
    .poll24px = @"\uec72",
    .poll48px = @"\uec73",
    .public24px = @"\uec74",
    .public48px = @"\uec75",
    .school24px = @"\uec76",
    .school48px = @"\uec77",
    .share24px = @"\uec78",
    .share48px = @"\uec79",
    .whatshot18px = @"\uec7a",
    .whatshot24px = @"\uec7b",
    .whatshot48px = @"\uec7c"
};

@end
