# Copyright 2018 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""Convenience helpers for managing Params for datasets and models.

Typical usage will be to define and register a subclass of ModelParams
for each dataset.
"""

import functools
import inspect
from lingvo import model_imports
import lingvo.compat as tf
from lingvo.core import base_model_params
from lingvo.core import program

tf.flags.DEFINE_string(
    'model_params_override', '', 'Optional text specifying'
    ' model_class.Model() params to be overridden in'
    ' Params.FromText format.  Each param must be separated by'
    ' a newline or a semicolon.  This is only intended for'
    ' quick experimentation.  Only one of'
    ' --model_params_override and --model_params_file_override'
    ' may be specified.')
tf.flags.DEFINE_string(
    'model_params_file_override', '', 'Optional text file for'
    ' overwriting model_class.Model() params in Params.FromText'
    ' format. Each param must occur on a single line.  Only one'
    ' of --model_params_override and'
    ' --model_params_file_override may be specified.')
tf.flags.DEFINE_string(
    'executor_datasets_to_eval', None, 'Semicolon separated dataset list for '
    'executor to eval and decode.')
tf.flags.DEFINE_integer('executor_train_steps_per_loop', None,
                        'To override ProgramSchedule.train_steps_per_loop.')
tf.flags.DEFINE_integer(
    'executor_train_executions_per_eval', None,
    'To override ProgramSchedule.train_executions_per_eval.')
tf.flags.DEFINE_integer('executor_eval_steps_per_loop', None,
                        'To override ProgramSchedule.eval_steps_per_loop.')
tf.flags.DEFINE_integer(
    'executor_decode_steps_per_loop', None,
    'To override ProgramSchedule.decode_steps_per_loop, '
    '-1 indicate decode_until_out_of_range.')
tf.flags.DEFINE_bool(
    'executor_multi_inputs_decoder', None,
    'Whether to use MultiInputsDecodeProgram for decoding datasets.')
tf.flags.DEFINE_string(
    'executor_oneoff_checkpoint_to_load', None,
    'Typically used for oneoff eval/decode executor run, to override '
    'p.checkpoint_to_load.')
tf.flags.DEFINE_string(
    'decode_summary_emails', None,
    'Semi-colon separated list of email addresses to send Decode summary to.')

FLAGS = tf.flags.FLAGS


class _ModelRegistryHelper:
  """Helper class."""

  _MODEL_PARAMS_ALLOW_REDEF = False

  # Global dictionary mapping subclass name to registered ModelParam subclass.
  _MODEL_PARAMS = {}
  # Global set of modules from which ModelParam subclasses have been registered.
  _REGISTERED_MODULES = set()

  @classmethod
  def _ClassPathPrefix(cls):
    return 'lingvo.tasks.'

  @classmethod
  def _ModelParamsClassKey(cls, src_cls):
    """Returns a string key used for `src_cls` in the model registry.

    The returned key is a period separated string. E.g., image.mnist.LeNet5. It
    roughly reflects how params files are organized. We put some of the
    directory information into the key to avoid future model name conflicts.

    Args:
      src_cls: A subclass of `~.base_model.BaseModel`.
    """
    path = src_cls.__module__
    # Removes the prefix.
    path_prefix = cls._ClassPathPrefix()
    path = path.replace(path_prefix, '')

    # Removes 'params.' if exists.
    if 'params.' in path:
      path = path.replace('params.', '')
    if inspect.getfile(src_cls).endswith('test.py'):
      return 'test.{}'.format(src_cls.__name__)
    return '{}.{}'.format(path, src_cls.__name__)

  @classmethod
  def _GetSourceInfo(cls, src_cls):
    """Gets a source info string given a source class."""
    info_str = '%s@%s' % (cls._ModelParamsClassKey(src_cls),
                          inspect.getsourcefile(src_cls))
    try:
      return f'{info_str}:{inspect.getsourcelines(src_cls)[-1]}'
    except OSError:
      # The class definition is not found in the source file, as it might be
      # dynamically created. We omit the line number in this case.
      return info_str

  @classmethod
  def _RegisterModel(cls, wrapper_cls, src_cls):
    """Registers a ModelParams subclass in the global registry."""
    key = cls._ModelParamsClassKey(src_cls)
    module = src_cls.__module__
    if not cls._MODEL_PARAMS_ALLOW_REDEF and key in cls._MODEL_PARAMS:
      raise ValueError('Duplicate model registered for key {}: {}.{}'.format(
          key, module, src_cls.__name__))

    tf.logging.debug('Registering model %s', key)
    # Log less frequently (once per module) but at a higher verbosity level.
    if module not in cls._REGISTERED_MODULES:
      tf.logging.info('Registering models from module: %s', module)
      cls._REGISTERED_MODULES.add(module)

    # Decorate param methods to add source info metadata.
    cls._MODEL_PARAMS[key] = wrapper_cls
    return key

  @classmethod
  def _CreateWrapperClass(cls, src_cls):
    """Creates a wrapper class for model params that adds source info."""

    # Python2 class decorators that return a different class are fundamentally
    # broken (technically, they are fine but canonical use of super() is
    # broken). Also, fallback mechanisms don't exist in python3. So, we only
    # decorate the version of the class that we register, but any decorators
    # should return the original class for maximum compatibility.
    # When the python3 super() is used, it should be possible to return this
    # from the decorators too.

    class Registered(src_cls):
      """Registered model wrapper."""

      @functools.lru_cache()
      def _registered_source_info(self):
        # This is lazily calculated only when needed. _GetSourceInfo is
        # expensive since it calls inspect APIs and is best to be avoided at
        # import time.
        return cls._GetSourceInfo(src_cls)  # pylint: disable=protected-access

      # Extend model to annotate source information.
      def Model(self):
        """Wraps BaseTask params into SingleTaskModel params."""
        p = super().Model()
        p.model = self._registered_source_info()
        return p

    # So things show up in messages well.
    Registered.__name__ = src_cls.__name__
    return Registered

  @classmethod
  def MaybeUpdateParamsFromFlags(cls, cfg):
    """Updates Model() Params from flags if set."""
    if FLAGS.model_params_override and FLAGS.model_params_file_override:
      raise ValueError('Only one of --model_params_override and'
                       ' --model_params_file_override may be specified.')

    if FLAGS.model_params_override:
      params_override = FLAGS.model_params_override.replace(';', '\n')
      tf.logging.info('Applying params overrides:\n%s\nTo:\n%s',
                      params_override, cfg.ToText())
      # FromText() inferred param type from the default_value of the param.
      # init_from_checkpoint_override's default_value is None, type cannot be
      # inferred, so we have to explicilty set type_overrides. Otherwise,
      # this override does not succeed.
      # TODO(xingwu): Consider to refactor to a more elegent way.
      cfg.FromText(
          params_override,
          type_overrides={'task.train.init_from_checkpoint_override': 'str'})
    if (FLAGS.model_params_file_override and
        tf.io.gfile.exists(FLAGS.model_params_file_override)):
      params_override = tf.io.gfile.GFile(FLAGS.model_params_file_override,
                                          'r').read()
      tf.logging.info('Applying params overrides from file %s:\n%s\nTo:\n%s',
                      FLAGS.model_params_file_override, params_override,
                      cfg.ToText())
      cfg.FromText(params_override)

  @classmethod
  def RegisterSingleTaskModel(cls, src_cls):
    """Class decorator that registers a `.SingleTaskModelParams` subclass."""
    if not issubclass(src_cls, base_model_params.SingleTaskModelParams):
      raise TypeError('src_cls %s is not a SingleTaskModelParams!' %
                      src_cls.__name__)
    cls._RegisterModel(cls._CreateWrapperClass(src_cls), src_cls)
    return src_cls

  @classmethod
  def RegisterMultiTaskModel(cls, src_cls):
    """Class decorator that registers a `.MultiTaskModelParams` subclass."""
    if not issubclass(src_cls, base_model_params.MultiTaskModelParams):
      raise TypeError('src_cls %s is not a MultiTaskModelParams!' %
                      src_cls.__name__)
    cls._RegisterModel(cls._CreateWrapperClass(src_cls), src_cls)
    return src_cls

  @staticmethod
  def GetAllRegisteredClasses():
    """Returns global registry map from model names to their param classes."""
    all_params = _ModelRegistryHelper._MODEL_PARAMS
    if not all_params:
      tf.logging.warning('No classes registered.')
    return all_params

  @classmethod
  def GetClass(cls, class_key):
    """Returns a ModelParams subclass with the given `class_key`.

    Args:
      class_key: string key of the ModelParams subclass to return.

    Returns:
      A subclass of `~.base_model_params._BaseModelParams`.

    Raises:
      LookupError: If no class with the given key has been registered.
    """
    all_params = cls.GetAllRegisteredClasses()
    if class_key not in all_params:
      for k in sorted(all_params):
        tf.logging.info('Known model: %s', k)
      raise LookupError('Model %s not found from list of above known models.' %
                        class_key)
    return all_params[class_key]

  @classmethod
  @functools.lru_cache(maxsize=8)
  def GetCachedObject(cls, class_key):
    """Returns a ModelParams class object with the given `class_key`.

    To reuse the same object for given `class_key`, it uses
    functools.lru_cache annotation.

    Args:
      class_key: string key of the ModelParams subclass to return.

    Returns:
      A instance of `~.base_model_params._BaseModelParams`.
    """
    return cls.GetClass(class_key)()

  @classmethod
  def GetParamsFromModelParamsObject(cls, model_params_obj, dataset_name):
    """Returns a `Params` object for given _BaseModelParams instance.

    In case of default model, params may be updated based on the flags
    `--model_params_override` or `--model_params_file_override`.

    Args:
      model_params_obj: A ~.base_model_params._BaseModelParams instance.
      dataset_name: Method to generate dataset params (i.e. 'Test').

    Returns:
      Full `~.hyperparams.Params` for the
        ~.base_model_params._BaseModelParams instance.

    """
    cfg = model_params_obj.Model()
    if dataset_name:
      cfg.input = model_params_obj.GetDatasetParams(dataset_name)
      # Overwrite dataset specific Task parameters.
      try:
        dataset_task_params = model_params_obj.GetDatasetParams('Task_' +
                                                                dataset_name)
        # Overwrite task params with dataset specific ones.
        if isinstance(model_params_obj,
                      base_model_params.SingleTaskModelParams):
          cfg.cls.CopyTaskParams(dataset_task_params, cfg)
        elif isinstance(model_params_obj,
                        base_model_params.MultiTaskModelParams):
          cfg.task_params = dataset_task_params
      except base_model_params.DatasetError:
        tf.logging.info('No dataset specific task parameters for %s',
                        dataset_name)

    cls.MaybeUpdateParamsFromFlags(cfg)
    return cfg

  @classmethod
  def GetParams(cls, class_key, dataset_name, cache=True):
    """Constructs a `Params` object for given model and dataset, obeying flags.

    In case of default model, params may be updated based on the flags
    `--model_params_override` or `--model_params_file_override`.

    Args:
      class_key: String class key (i.e. `image.mnist.LeNet5`).
      dataset_name: Method to generate dataset params (i.e. 'Test').
      cache: Whether to cache the params given class_key and dataset_name.

    Returns:
      Full `~.hyperparams.Params` for the model class.
    """
    if cache:
      # Reuse the cached object to reused the cached obj.GetAllDatasetParams().
      model_params_obj = cls.GetCachedObject(class_key)
    else:
      model_params_obj = cls.GetClass(class_key)()
    return cls.GetParamsFromModelParamsObject(model_params_obj, dataset_name)

  @classmethod
  def GetProgramScheduleFromModelParamsObject(cls, model_params_obj):
    """Retrieve the ProgramSchedule and a dict of task params.

    Args:
      model_params_obj: A ~.base_model_params._BaseModelParams instance.

    Returns:
      Full `~.hyperparams.Params` for the model class.
    """
    program_schedule_cfg = model_params_obj.ProgramSchedule()
    datasets_to_eval = None
    if FLAGS.executor_datasets_to_eval is not None:
      datasets_to_eval = FLAGS.executor_datasets_to_eval.split(';')
    decode_summary_emails = []
    if FLAGS.decode_summary_emails is not None:
      decode_summary_emails = FLAGS.decode_summary_emails.split(';')

    program_schedule_cfg = program.UpdateProgramSchedule(
        program_schedule_cfg,
        datasets_to_eval,
        FLAGS.executor_train_executions_per_eval,
        FLAGS.executor_train_steps_per_loop,
        FLAGS.executor_eval_steps_per_loop,
        FLAGS.executor_decode_steps_per_loop,
        FLAGS.executor_multi_inputs_decoder,
        decode_summary_emails,
        FLAGS.executor_oneoff_checkpoint_to_load,
    )
    return program_schedule_cfg

  @classmethod
  def GetProgramSchedule(cls, class_key):
    """Retrieve the ProgramSchedule and a dict of task params.

    Args:
      class_key: String class key (i.e. `image.mnist.LeNet5`).

    Returns:
      ProgramSchedule.Params()
    """
    model_params_cls = cls.GetClass(class_key)
    return cls.GetProgramScheduleFromModelParamsObject(model_params_cls())

# pyformat: disable
# pylint: disable=invalid-name
RegisterSingleTaskModel = _ModelRegistryHelper.RegisterSingleTaskModel
RegisterMultiTaskModel = _ModelRegistryHelper.RegisterMultiTaskModel
# pylint: enable=invalid-name
# pyformat: enable


def GetAllRegisteredClasses():
  model_imports.ImportAllParams()
  return _ModelRegistryHelper.GetAllRegisteredClasses()


def GetClass(class_key):
  model_imports.ImportParams(class_key)
  return _ModelRegistryHelper.GetClass(class_key)


def GetParams(class_key, dataset_name):
  model_imports.ImportParams(class_key)
  return _ModelRegistryHelper.GetParams(class_key, dataset_name)


def GetProgramSchedule(class_key):
  model_imports.ImportParams(class_key)
  return _ModelRegistryHelper.GetProgramSchedule(class_key)
