# Copyright 2018 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""Tokenizers."""

import lingvo.compat as tf
from lingvo.core import base_layer
from lingvo.core import ops
from lingvo.core import py_utils
from lingvo.core import wpm_encoder

import tensorflow_text as tf_text


class BaseTokenizer(base_layer.BaseLayer):
  """The base tokenizer."""

  @classmethod
  def Params(cls):
    """Defaults params for tokenizers."""
    p = super().Params()
    p.name = 'tokenizer'
    p.Define('vocab_size', 64, 'The size of the vocabuary.')
    p.Define(
        'append_eos', True, 'Whether to append </s> at the end and treat '
        'it as a non-padded label.')
    p.Define('pad_to_max_length', True,
             'If True, output ids will be padded to max_length.')
    # TODO(ciprianchelba): there should be a check in __init__ that the ids
    # below are consistent with the ones assigned by the vocabulary.
    p.Define('target_unk_id', 0, 'Target unknown token id.')
    p.Define('target_sos_id', 1, 'Target start of sequence id.')
    p.Define('target_eos_id', 2, 'Target end of sequence id.')
    p.Define('target_wb_id', -1, 'Target word boundary id.')
    return p

  def __init__(self, params):
    super().__init__(params)
    p = self.params

    self.sos_id = p.target_sos_id  # <S>
    self.eos_id = p.target_eos_id  # </S>
    self.unk_id = p.target_unk_id  # <UNK>

  def Initialize(self, sess):
    """Run any tokenizer initialization that requires a session."""
    pass

  def StringsToIds(self,
                   strs,
                   max_length,
                   external_append_eos=None,
                   languages=None):
    """Tokenize strs into vocab ids.

    Args:
      strs: A vector of strings.
      max_length: An int providing the max_length for strs.
      external_append_eos: Bool or None. If None, will be ignored and
        `params.append_eos` will be used. If bool, will determine if an eos
        symbol will be added to tokens.
      languages: A vector of strings with the same length as `strs`.

    Returns:
      A tuple (ids, labels, paddings) with the same shape [batch, maxlen].

      - ids[i, j] is the input token id of i-th sample for j-th step.
      - labels[i, j] is the target token id of i-th sample for j-th step.
      - paddings[i, j] is 1 iff i-th sample's j-th step is padded.

    Raises:
      ValueError: If unknown token type.
    """
    p = self.params

    if external_append_eos is None:
      append_eos = p.append_eos
    else:
      append_eos = external_append_eos

    return self._StringsToIdsImpl(strs, max_length, append_eos, languages)

  def StringsToIdsWithSegments(self,
                               strs,
                               max_length,
                               external_append_eos=None,
                               languages=None):
    """Same as StringsToIds, but can process strs that contain segments...

    separated by <segment> delimiter, and return the segment ids corresponding
    to each label.

    Args:
      strs: A vector of strings.
      max_length: An int providing the max_length for strs.
      external_append_eos: Bool or None. If None, will be ignored and
        `params.append_eos` will be used. If bool, will determine if an eos
        symbol will be added to tokens.
      languages: A vector of strings with the same length as `strs`.

    Returns:
      A tuple (ids, labels, paddings) with the same shape [batch, maxlen].

      - ids[i, j] is the input token id of i-th sample for j-th step.
      - labels[i, j] is the target token id of i-th sample for j-th step.
      - paddings[i, j] is 1 iff i-th sample's j-th step is padded.
      - segment_ids[i, j] is the segment id of i-th sample for j-th step.

    Raises:
      ValueError: If unknown token type.
    """
    # TODO(cjzheng) make segment delimiter configurable
    p = self.params

    if external_append_eos is None:
      append_eos = p.append_eos
    else:
      append_eos = external_append_eos

    return self._StringsToIdsWithSegmentsImpl(strs, max_length, append_eos,
                                              languages)

  def StringsToIdsWithOffsets(self,
                              strs,
                              max_length,
                              external_append_eos=None,
                              languages=None):
    """Tokenize strs into vocab ids, and also return byte-level offsets.

    Args:
      strs: A vector of strings.
      max_length: An int providing the max_length for strs.
      external_append_eos: Bool or None. If None, will be ignored and
        `params.append_eos` will be used. If bool, will determine if an eos
        symbol will be added to tokens.
      languages: A vector of strings with the same length as `strs`.

    Returns:
      A tuple (ids, labels, paddings, start_offsets, end_offsets). Each tensor
      has the same shape [batch, maxlen].

      - ids[i, j] is the input token id of i-th sample for j-th step.
      - labels[i, j] is the target token id of i-th sample for j-th step.
      - paddings[i, j] is 1 iff i-th sample's j-th step is padded.
      - start_offset[i, j] is the byte-level offset of the start of the j-th id
          in the i-th original string
      - end_offset[i, j] is the byte-level offset of the end of the j-th id
          in the i-th original string

    Raises:
      ValueError: If unknown token type.
      Exception: If the tokenizer does not support offsets.
    """
    p = self.params
    if external_append_eos is None:
      append_eos = p.append_eos
    else:
      append_eos = external_append_eos

    return self._StringsToIdsWithOffsetsImpl(strs, max_length, append_eos,
                                             languages)

  def _StringsToIdsImpl(self, strs, max_length, append_eos, languages):
    raise NotImplementedError('Abstract method.')

  def _StringsToIdsWithOffsetsImpl(self, strs, max_length, append_eos,
                                   languages):
    raise NotImplementedError('Abstract method.')

  def _StringsToIdsWithSegmentsImpl(self, strs, max_length, append_eos,
                                    languages):
    raise NotImplementedError('Abstract method.')

  def IdsToStrings(self, ids, lens, languages=None):
    """Converts ids back to strings.

    Args:
      ids: A matrix of shape [batch, seqlen]. ids[i, :] is the i-th sample's
        ids.
      lens: A vector of shape [batch]. lens[i] is the sequence length of the
        i-th sample. Only the first lens[i] tokens in ids[i, :] are valid tokens
        for the i-th sequence.
      languages: A vector of strings of shape [batch].

    Returns:
      sequences - A vector of shape [batch]. The converted string sequence.

    Raises:
      ValueError: If unknown token type.
    """
    raise NotImplementedError('Abstract method.')

  def IdsToTokens(self, ids, languages=None):
    """Converts ids back to tokens (as separate strings).

    Args:
      ids: A matrix of shape [batch, seqlen]. ids[i, :] is the i-th sample's
        ids.
      languages: A vector of strings of shape [batch].

    Returns:
      tokens - A matrix of shape [batch, seqlen] of bytes.

    Raises:
      ValueError: If unknown token type.
    """
    raise NotImplementedError('Abstract method.')


class AsciiTokenizer(BaseTokenizer):
  """A simple grapheme tokenizer.

  Maps a small vocabulary of character tokens for (lower case) letters, digits,
  and punctuation symbols.
  """

  def _StringsToIdsImpl(self, strs, max_length, append_eos, languages):
    p = self.params
    return ops.ascii_to_token_id(
        strs,
        maxlen=max_length,
        pad_to_maxlen=p.pad_to_max_length,
        append_eos=append_eos)

  def IdsToStrings(self, ids, lens):
    return ops.id_to_ascii(ids, lens)


class VocabFileTokenizer(BaseTokenizer):
  """Tokenizers that use vocab files for look-up."""

  @classmethod
  def Params(cls):
    p = super().Params()
    p.Define('token_vocab_filepath', None,
             'If set, specifies a filepath to the token vocab file.')
    p.Define('ngram_vocab_filepath', None,
             'If set, specifies a filepath to the Ngram vocab file.')
    p.Define('ngram_separator', '',
             'string separator to use when joining ngrams.')
    p.Define('tokens_delimiter', ' ',
             'The delimiter to split a string to tokens with.')
    p.Define(
        'load_token_ids_from_vocab', True,
        'Whether token ids are present in vocab (i.e. vocab contains two '
        'colums, one for IDs and one for words).  If false, line numbers '
        'are used.')
    return p

  @property
  def _vocab_file_params(self):
    return ['token_vocab_filepath', 'ngram_vocab_filepath']

  def _CheckParams(self):
    p = self.params
    num_params_specified = sum(
        [getattr(p, x) is not None for x in self._vocab_file_params])
    if num_params_specified != 1:
      raise ValueError('Exactly one vocab file should be specified!')

  def _StringsToIdsImpl(self, strs, max_length, append_eos, languages):
    self._CheckParams()
    p = self.params

    if p.token_vocab_filepath:
      return ops.str_to_vocab_tokens(
          strs,
          maxlen=max_length,
          pad_to_maxlen=p.pad_to_max_length,
          append_eos=append_eos,
          vocab_filepath=p.token_vocab_filepath,
          load_token_ids_from_vocab=p.load_token_ids_from_vocab,
          delimiter=p.tokens_delimiter)
    elif p.ngram_vocab_filepath:
      raise NotImplementedError('ngram vocab StringsToIds is not supported.')

  def IdsToStrings(self, ids, lens):
    self._CheckParams()
    p = self.params
    if p.token_vocab_filepath:
      ngram_vocab_filepath = p.token_vocab_filepath
      ngram_separator = p.tokens_delimiter
    elif p.ngram_vocab_filepath:
      ngram_vocab_filepath = p.ngram_vocab_filepath
      ngram_separator = p.ngram_separator

    return ops.ngram_id_to_token(
        token_ids=ids,
        seq_lengths=lens,
        ngram_vocab_filepath=ngram_vocab_filepath,
        ngram_separator=ngram_separator)


class BpeTokenizer(BaseTokenizer):
  """Tokenizers that use BPE vocab files and word to id lists for look-up."""

  @classmethod
  def Params(cls):
    p = super().Params()
    p.Define('codes_filepath', None,
             'Specifies a filepath to the list of bpe codes vocab file.')
    p.Define('words_to_ids_filepath', None,
             'Specifies a filepath to the word bpe vocab file.')
    return p

  def _StringsToIdsImpl(self, strs, max_length, append_eos, languages):
    p = self.params

    return ops.bpe_words_to_ids(
        strs,
        maxlen=max_length,
        append_eos=append_eos,
        tokenization_filepath=p.words_to_ids_filepath)

  def IdsToStrings(self, ids, lens):
    p = self.params

    return ops.bpe_ids_to_words(
        token_ids=ids, seq_lengths=lens, vocab_filepath=p.codes_filepath)


class WpmTokenizer(BaseTokenizer):
  """Tokenizer for word-piece models."""

  @classmethod
  def Params(cls):
    p = super().Params()
    p.Define(
        'vocab_filepath', None,
        'Specifies a filepath to the WPM vocab. The vocab is sorted by '
        'descending merge score.')
    p.Define(
        'merge_prob', 1.,
        'Probability of merging WPMs. If less than 1, then decomposition '
        'of words into wordpieces will no longer be deterministic, and '
        'result in longer ID sequences. At 0, it will be graphemes.')
    return p

  def __init__(self, params):
    super().__init__(params)
    p = self.params
    self._wpm_encoder = wpm_encoder.WpmEncoder(p.vocab_filepath, p.merge_prob)
    assert p.target_unk_id == self._wpm_encoder.unk_id
    assert p.target_sos_id == self._wpm_encoder.sentence_start_id
    assert p.target_eos_id == self._wpm_encoder.sentence_end_id

  def _StringsToIdsImpl(self, strs, max_length, append_eos, languages):
    """Takes a tensor of strings and returns id/padding tensors.

    This generates `token_ids`, `target_ids`, and `paddings` in the format that
    is expected for tokenizers. This performs padding to a fixed length and
    appends the end-of-sentence token as appropriate.

    Args:
      strs: a string Tensor.
      max_length: a python integer. The second dimension of the returned arrays.
        All sequences are padded or truncated to that length.
      append_eos: a python bool. See `BaseTokenizer` for explanation.
      languages: A vector of strings with the same length as `strs`.

    Returns:
      A tuple of 3 tensors:

      - token_ids: a tensor of sequences of WPM ids starting with SOS. Sequences
        always end with EOS unless the sequence exceeds the maximum length.
        Always padded with EOS.
      - target_ids: a tensor of sequences of WPM ids not starting with SOS
        but ending with EOS. Always padded with EOS.
      - paddings: a tensor of floats indicating, at each position, whether
        the corresponding position is padded.
    """
    p = self.params
    if append_eos is None:
      append_eos = p.append_eos

    batch_size = py_utils.GetShape(strs)[0]
    token_ids_ta = tf.TensorArray(tf.int32, batch_size)
    target_ids_ta = tf.TensorArray(tf.int32, batch_size)
    paddings_ta = tf.TensorArray(tf.float32, batch_size)

    def _TokenizeOneSentence(i, strs, token_ids_ta, target_ids_ta, paddings_ta):
      """Tokenizes a single sentence."""
      ids, _ = self._wpm_encoder.Encode(strs[i])

      if append_eos:
        ids = tf.concat([ids, [self.eos_id]], axis=0)

      # This truncates after the eos is added, so some sentences might
      # not have </s> at the end.
      token_ids_ta = token_ids_ta.write(
          i,
          py_utils.PadOrTrimTo(
              tf.concat([[self.sos_id], ids], axis=0), [max_length],
              self.eos_id))
      target_ids_ta = target_ids_ta.write(
          i, py_utils.PadOrTrimTo(ids, [max_length], self.eos_id))
      paddings_ta = paddings_ta.write(
          i,
          py_utils.PadOrTrimTo(
              tf.zeros_like(ids, dtype=tf.float32), [max_length], 1.))

      return i + 1, strs, token_ids_ta, target_ids_ta, paddings_ta

    _, _, token_ids_ta, target_ids_ta, paddings_ta = tf.while_loop(
        lambda i, *_: i < batch_size,
        _TokenizeOneSentence,
        loop_vars=(tf.constant(0, tf.int32), strs, token_ids_ta, target_ids_ta,
                   paddings_ta),
        parallel_iterations=30,
        back_prop=False)

    token_ids = token_ids_ta.stack()
    target_ids = target_ids_ta.stack()
    paddings = paddings_ta.stack()

    if not p.pad_to_max_length:
      maxlen = tf.cast(
          tf.round(tf.reduce_max(tf.reduce_sum(1.0 - paddings, axis=1))),
          tf.int32)
      token_ids = token_ids[:, :maxlen]
      target_ids = target_ids[:, :maxlen]
      paddings = paddings[:, :maxlen]

    return token_ids, target_ids, paddings

  def IdsToStrings(self, ids, lens):
    """Takes integer matrices and returns vectors of strings."""
    ids = py_utils.with_dependencies([py_utils.assert_same_dim0([ids, lens])],
                                     ids)
    return tf.map_fn(
        lambda inputs: self._wpm_encoder.Decode(inputs[0][:inputs[1]]),
        (ids, lens),
        dtype=tf.string,
        parallel_iterations=30,
        back_prop=False)


class SentencePieceTokenizer(BaseTokenizer):
  """SentencePiece tokenizer (https://arxiv.org/abs/1808.06226).

  This is a wrapper around tf_text.SentencepieceTokenizer.

  NOTE: this tokenizer is incompatible with GenericInput (b/191804185).
  """

  @classmethod
  def Params(cls):
    p = super().Params()
    p.vocab_size = 0
    p.Define('spm_model', None, 'The file path to the SentencePiece model.')
    p.Define('alpha', 1.0,
             'Inverse temparature for probability rescaling used in eval.')
    p.Define(
        'nbest_size', 0, 'A integer parameter for sampling.'
        ' nbest_size = {0,1}: No sampling is performed.'
        ' nbest_size > 1: samples from the nbest_size results.'
        ' nbest_size < 0: assuming that nbest_size is infinite and samples from'
        '                 all hypothesis (lattice) using'
        '                 forward-filtering-and-backward-sampling algorithm.')
    return p

  def __init__(self, params):
    super().__init__(params)
    p = self.params
    with tf.io.gfile.GFile(p.spm_model, 'rb') as f:
      self._tokenizer = tf_text.SentencepieceTokenizer(
          model=f.read(),
          out_type=tf.int32,
          nbest_size=p.nbest_size,
          alpha=p.alpha,
          reverse=False,
          add_bos=False,
          add_eos=p.append_eos)

  def StringsToIds(self,
                   strs,
                   max_length,
                   external_append_eos=None,
                   languages=None):
    """Tokenize strs into vocab ids."""
    if (external_append_eos is not None and
        external_append_eos != self.params.append_eos):
      raise ValueError('external_append_eos is not supported.')
    if languages is not None:
      raise ValueError('languages is not supported.')
    return self._StringsToIdsImpl(
        strs, max_length, append_eos=None, languages=None)

  def _StringsToIdsImpl(self, strs, max_length, append_eos, languages):
    del append_eos
    del languages

    p = self.params

    tokens = self._tokenizer.tokenize(strs)
    num_tokens = tokens.row_lengths(-1)

    if max_length is None:
      labels = tokens.to_tensor(default_value=p.target_unk_id)
    else:
      output_shape = tf.convert_to_tensor(strs).shape + [max_length]
      labels = tokens.to_tensor(
          default_value=p.target_unk_id, shape=output_shape)
      num_tokens = tf.minimum(num_tokens, max_length)

    ids = tf.concat([
        tf.expand_dims(tf.ones_like(strs, tf.int32) * p.target_sos_id, -1),
        labels[:, :-1]
    ], -1)

    paddings = 1.0 - tf.sequence_mask(
        num_tokens, maxlen=max_length, dtype=tf.float32)
    return ids, labels, paddings

  def IdsToStrings(self, ids, lens, languages=None):
    return self._tokenizer.detokenize(tf.RaggedTensor.from_tensor(ids, lens))

  def IdsToTokens(self, ids, languages=None):
    return self._tokenizer.id_to_string(ids)
