# Copyright 2018 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""Tests for inference_graph_exporter."""

from lingvo import model_registry
import lingvo.compat as tf
from lingvo.core import base_input_generator
from lingvo.core import base_model
from lingvo.core import base_model_params
from lingvo.core import inference_graph_exporter
from lingvo.core import inference_graph_pb2
from lingvo.core import predictor
from lingvo.core import py_utils
from lingvo.core import test_utils


class DummyLegacyModel(base_model.BaseTask):

  def Inference(self):
    if py_utils.use_tpu():
      raise NotImplementedError('TPU is not supported.')
    with tf.name_scope('inference'):
      feed1 = tf.placeholder(name='feed1_node', dtype=tf.float32, shape=[1])
      fetch1 = tf.identity(feed1, name='fetch1_node')
      return {
          'default': (
              py_utils.NestedMap({
                  'fetch1': fetch1,
                  'fetch_op': fetch1.op,  # Tests that ops are supported.
              }),
              py_utils.NestedMap({
                  'feed1': feed1,
              })),
          'unused': (py_utils.NestedMap({}), py_utils.NestedMap({})),
      }


@model_registry.RegisterSingleTaskModel
class DummyLegacyModelParams(base_model_params.SingleTaskModelParams):

  @classmethod
  def Test(cls):
    p = base_input_generator.BaseSequenceInputGenerator.Params()
    p.name = 'input'
    return p

  @classmethod
  def Task(cls):
    p = DummyLegacyModel.Params()
    p.name = 'testing'
    return p


class DummyModel(base_model.BaseTask):

  def Inference(self):
    with tf.name_scope('inference'):
      feed1 = tf.placeholder(name='feed1_node', dtype=tf.float32, shape=[1])
      fetch1 = tf.identity(feed1, name='fetch1_node')
      inference_graph = inference_graph_pb2.InferenceGraph()
      subgraph = inference_graph.subgraphs['default']
      subgraph.feeds['feed1'] = feed1.name
      subgraph.fetches['fetch1'] = fetch1.name
      # Tests that ops are supported.
      subgraph.fetches['fetch_op'] = fetch1.op.name
      return inference_graph


@model_registry.RegisterSingleTaskModel
class DummyModelParams(base_model_params.SingleTaskModelParams):

  @classmethod
  def Test(cls):
    p = base_input_generator.BaseSequenceInputGenerator.Params()
    p.name = 'input'
    return p

  @classmethod
  def Task(cls):
    p = DummyModel.Params()
    p.name = 'testing'
    return p


class InferenceGraphExporterTest(test_utils.TestCase):

  def testExportModelParamsWithSubgraphDict(self):
    params = model_registry.GetParams('test.DummyLegacyModelParams', 'Test')
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params, subgraph_filter=['default'])

    # Should populate subgraphs.
    self.assertIn('default', inference_graph.subgraphs)
    self.assertNotIn('unused', inference_graph.subgraphs)
    subgraph = inference_graph.subgraphs['default']
    self.assertIn('feed1', subgraph.feeds)
    self.assertIn('fetch1', subgraph.fetches)

    self.assertEqual(subgraph.feeds['feed1'], 'inference/feed1_node:0')
    self.assertEqual(subgraph.fetches['fetch1'], 'inference/fetch1_node:0')
    self.assertEqual(subgraph.fetches['fetch_op'], 'inference/fetch1_node')

  def testSubgraphFilterNotValid(self):
    params = model_registry.GetParams('test.DummyLegacyModelParams', 'Test')
    with self.assertRaises(ValueError):
      _ = inference_graph_exporter.InferenceGraphExporter.Export(
          params, subgraph_filter=['not-present'])

  def testExportModelParamsWithInferenceGraph(self):
    params = model_registry.GetParams('test.DummyModelParams', 'Test')
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params)

    # Should populate subgraphs.
    self.assertIn('default', inference_graph.subgraphs)
    subgraph = inference_graph.subgraphs['default']
    self.assertIn('feed1', subgraph.feeds)
    self.assertIn('fetch1', subgraph.fetches)

    self.assertEqual(subgraph.feeds['feed1'], 'inference/feed1_node:0')
    self.assertEqual(subgraph.fetches['fetch1'], 'inference/fetch1_node:0')
    self.assertEqual(subgraph.fetches['fetch_op'], 'inference/fetch1_node')

  def testExportModelDoesNotAffectFlagsOnException(self):
    initial_flags = {k: tf.flags.FLAGS[k].value for k in tf.flags.FLAGS}
    params = model_registry.GetParams('test.DummyLegacyModelParams', 'Test')
    with self.assertRaises(NotImplementedError):
      inference_graph_exporter.InferenceGraphExporter.Export(
          params,
          device_options=inference_graph_exporter.InferenceDeviceOptions(
              device='tpu',
              retain_device_placement=False,
              var_options=None,
              gen_init_op=True,
              dtype_override=None,
              fprop_dtype_override=None))
    self.assertDictEqual(initial_flags,
                         {k: tf.flags.FLAGS[k].value for k in tf.flags.FLAGS})


class NoConstGuaranteeScopeTest(test_utils.TestCase):

  def testNoConsting(self):
    with inference_graph_exporter.ConstGuaranteeScope():
      wp = py_utils.WeightParams(
          shape=[1],
          init=py_utils.WeightInit.Constant(0.0),
          dtype=tf.float32,
          collections=['v'])
      v = py_utils.CreateVariable('v', wp)
      self.assertIsInstance(v, tf.Tensor)
      with inference_graph_exporter.NoConstGuaranteeScope():
        with tf.variable_scope('', reuse=True):
          v = py_utils.CreateVariable('v', wp)
        self.assertIsInstance(v, tf.tf2.Variable)


class LinearModel(base_model.BaseTask):
  """A basic linear model."""

  @classmethod
  def Params(cls):
    p = super().Params()
    p.name = 'linear_model'
    return p

  def _CreateLayerVariables(self):
    super()._CreateLayerVariables()
    p = self.params
    w = py_utils.WeightParams(
        shape=[3],
        init=py_utils.WeightInit.Gaussian(scale=1.0, seed=123456),
        dtype=p.dtype)
    b = py_utils.WeightParams(
        shape=[],
        init=py_utils.WeightInit.Gaussian(scale=1.0, seed=234567),
        dtype=p.dtype)
    self.CreateVariable('w', w)
    self.CreateVariable('b', b)

  def Inference(self):
    """Computes y = w^T x + b. Returns y and x, as outputs and inputs."""
    # Add a dummy file def to the collection
    filename = tf.convert_to_tensor(
        'dummy.txt', tf.dtypes.string, name='asset_filepath')
    tf.compat.v1.add_to_collection(tf.compat.v1.GraphKeys.ASSET_FILEPATHS,
                                   filename)

    with tf.name_scope('inference'):
      x = tf.placeholder(dtype=tf.float32, name='input')
      r = tf.random.stateless_uniform([3],
                                      seed=py_utils.GenerateStepSeedPair(
                                          self.params))
      y = tf.reduce_sum((self.vars.w + r) * x) + self.vars.b
      return {'default': ({'output': y}, {'input': x})}


class LinearModelTpu(LinearModel):
  """A basic linear model that runs inference on the TPU."""

  def Inference(self):
    """Computes y = w^T x + b. Returns y and x, as outputs and inputs."""
    with tf.name_scope('inference'):
      x = tf.placeholder(dtype=tf.bfloat16, name='input')

      def InferenceFn(x):
        return tf.reduce_sum(self.vars.w * x) + self.vars.b

      y = tf.tpu.rewrite(InferenceFn, [x])
      return {'tpu': ({'output': y[0]}, {'input': x})}


@model_registry.RegisterSingleTaskModel
class LinearModelParams(base_model_params.SingleTaskModelParams):

  @classmethod
  def Test(cls):
    p = base_input_generator.BaseSequenceInputGenerator.Params()
    p.name = 'input'
    return p

  @classmethod
  def Task(cls):
    p = LinearModel.Params()
    p.name = 'testing'
    return p


@model_registry.RegisterSingleTaskModel
class LinearModelTpuParams(base_model_params.SingleTaskModelParams):

  @classmethod
  def Test(cls):
    p = base_input_generator.BaseSequenceInputGenerator.Params()
    p.name = 'input'
    return p

  @classmethod
  def Task(cls):
    p = LinearModelTpu.Params()
    p.name = 'testing'
    return p


@model_registry.RegisterSingleTaskModel
class LinearModelTpuParamsWithEma(base_model_params.SingleTaskModelParams):

  @classmethod
  def Test(cls):
    p = base_input_generator.BaseSequenceInputGenerator.Params()
    p.name = 'input'
    return p

  @classmethod
  def Task(cls):
    p = LinearModelTpu.Params()
    p.name = 'testing'
    p.train.ema_decay = 0.99
    return p


class InferenceGraphExporterLinearModelTest(test_utils.TestCase):

  def testExport(self):
    """Test basic export."""
    params = model_registry.GetParams('test.LinearModelParams', 'Test')
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params, subgraph_filter=['default'], export_graph_collections=True)
    self.assertIn('default', inference_graph.subgraphs)
    self.assertEqual(1, len(inference_graph.asset_file_def))
    # Check the GLOBAL_VARIABLES graph collection which is needed for
    # eager to lift variables from a GraphDef.
    self.assertIn('variables', inference_graph.collection_def)

  def testExportFreezeDefault(self):
    """Test exporting frozen graph."""
    params = model_registry.GetParams('test.LinearModelParams', 'Test')
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params, freeze_defaults=True, subgraph_filter=['default'])
    self.assertIn('default', inference_graph.subgraphs)
    # Test graphs are well-formed and importable.
    with tf.Graph().as_default():
      tf.import_graph_def(inference_graph.graph_def)

  def testTpuBfloat16OverrideExport(self):
    """Test that we can export with tf.bfloat16 dtype."""
    params = model_registry.GetParams('test.LinearModelTpuParams', 'Test')
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params,
        subgraph_filter=['tpu'],
        device_options=inference_graph_exporter.InferenceDeviceOptions(
            device='tpu',
            retain_device_placement=True,
            var_options='ON_DEVICE',
            gen_init_op=True,
            dtype_override=tf.bfloat16,
            fprop_dtype_override=None))
    self.assertIn('tpu', inference_graph.subgraphs)

  def testTpuBfloat16OverrideExportWithEma(self):
    """Test that we can export with tf.bfloat16 dtype."""
    params = model_registry.GetParams('test.LinearModelTpuParamsWithEma',
                                      'Test')
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params,
        subgraph_filter=['tpu'],
        device_options=inference_graph_exporter.InferenceDeviceOptions(
            device='tpu',
            retain_device_placement=True,
            var_options='ON_DEVICE',
            gen_init_op=True,
            dtype_override=tf.bfloat16,
            fprop_dtype_override=None))
    self.assertIn('tpu', inference_graph.subgraphs)

  def testExportWithRandomSeeds(self):
    """Test the effect of setting random seeds on export."""
    params = model_registry.GetParams('test.LinearModelParams', 'Test')
    # Default -- use random_seed = None.
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params, subgraph_filter=['default'])
    pred = predictor.Predictor(inference_graph)
    [no_op_seed_1] = pred.Run(['output'], input=3)
    [no_op_seed_2] = pred.Run(['output'], input=3)
    self.assertNotEqual(no_op_seed_1, no_op_seed_2)
    pred = predictor.Predictor(inference_graph)
    [no_op_seed_3] = pred.Run(['output'], input=3)
    self.assertNotEqual(no_op_seed_1, no_op_seed_3)

    # Use a fixed random_seed.
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params, subgraph_filter=['default'], random_seed=1234)
    pred = predictor.Predictor(inference_graph)
    [fixed_op_seed_1] = pred.Run(['output'], input=3)
    [fixed_op_seed_2] = pred.Run(['output'], input=3)
    self.assertEqual(fixed_op_seed_1, fixed_op_seed_2)
    pred = predictor.Predictor(inference_graph)
    [fixed_op_seed_3] = pred.Run(['output'], input=3)
    self.assertEqual(fixed_op_seed_1, fixed_op_seed_3)

    # A different seed gives different results.
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params, subgraph_filter=['default'], random_seed=1235)
    pred = predictor.Predictor(inference_graph)
    [fixed_op_seed_4] = pred.Run(['output'], input=3)
    self.assertNotEqual(fixed_op_seed_1, fixed_op_seed_4)


class GetOutputNamesTest(test_utils.TestCase):

  def _TestGraph(self):
    params = model_registry.GetParams('test.LinearModelParams', 'Test')
    inference_graph = inference_graph_exporter.InferenceGraphExporter.Export(
        params)
    graph = tf.Graph()
    with graph.as_default():
      tf.import_graph_def(inference_graph.graph_def, name='')
    return graph, inference_graph

  def testDefault(self):
    graph, inference_graph = self._TestGraph()
    output_op_names = inference_graph_exporter.GetOutputOpNames(
        graph, inference_graph)
    self.assertEqual(
        output_op_names,
        [
            # pyformat: disable
        'inference/add_2',
        'inference/input',
        'testing/b/var',
        'testing/b/var/Initializer/random_normal',
        'testing/b/var/Initializer/random_normal/RandomStandardNormal',
        'testing/b/var/Initializer/random_normal/mean',
        'testing/b/var/Initializer/random_normal/mul',
        'testing/b/var/Initializer/random_normal/shape',
        'testing/b/var/Initializer/random_normal/stddev',
        'testing/w/var',
        'testing/w/var/Initializer/random_normal',
        'testing/w/var/Initializer/random_normal/RandomStandardNormal',
        'testing/w/var/Initializer/random_normal/mean',
        'testing/w/var/Initializer/random_normal/mul',
        'testing/w/var/Initializer/random_normal/shape',
        'testing/w/var/Initializer/random_normal/stddev',
            # pyformat: enable
        ],
    )

  def testNoPreserveColocationNodes(self):
    graph, inference_graph = self._TestGraph()
    output_op_names = inference_graph_exporter.GetOutputOpNames(
        graph, inference_graph, preserve_colocation_nodes=False)
    self.assertEqual(
        output_op_names,
        [
            # pyformat: disable
        'inference/add_2',
        'inference/input',
            # pyformat: enable
        ],
    )

  def testPreserveSaverRestoreNodes(self):
    graph, inference_graph = self._TestGraph()
    output_op_names = inference_graph_exporter.GetOutputOpNames(
        graph,
        inference_graph,
        preserve_colocation_nodes=False,
        preserve_saver_restore_nodes=True)
    self.assertEqual(
        output_op_names,
        [
            # pyformat: disable
        'inference/add_2',
        'inference/input',
        'save/Const',
        'save/restore_all',
            # pyformat: enable
        ],
    )

  def testPreserveExtraOps(self):
    graph, inference_graph = self._TestGraph()
    output_op_names = inference_graph_exporter.GetOutputOpNames(
        graph,
        inference_graph,
        preserve_colocation_nodes=False,
        preserve_extra_ops=[
            'init_all_tables', 'init_all_variables', 'tpu_init_op'
        ])
    self.assertEqual(
        output_op_names,
        [
            # pyformat: disable
        'inference/add_2',
        'inference/input',
        'init_all_tables',
        'init_all_variables',
            # pyformat: enable
        ],
    )

  def testPreserveSaverNodesAndExtraOps(self):
    graph, inference_graph = self._TestGraph()
    output_op_names = inference_graph_exporter.GetOutputOpNames(
        graph,
        inference_graph,
        preserve_colocation_nodes=False,
        preserve_saver_restore_nodes=True,
        preserve_extra_ops=[
            'init_all_tables', 'init_all_variables', 'tpu_init_op'
        ])
    self.assertEqual(
        output_op_names,
        [
            # pyformat: disable
        'inference/add_2',
        'inference/input',
        'init_all_tables',
        'init_all_variables',
        'save/Const',
        'save/restore_all',
            # pyformat: enable
        ],
    )


if __name__ == '__main__':
  test_utils.main()
