// Copyright (c) 2023 Tim <tbckr>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// SPDX-License-Identifier: MIT

package fs

import (
	"io"
	"os"
	"path/filepath"
	"sync"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestGetAppCacheDir(t *testing.T) {
	existingEnv := os.Getenv("XDG_CACHE_HOME")
	t.Cleanup(func() {
		_ = os.Setenv("XDG_CACHE_HOME", existingEnv)
	})

	tempDir := t.TempDir()

	err := os.Setenv("XDG_CACHE_HOME", tempDir)
	require.NoError(t, err)

	var cacheDir string
	cacheDir, err = GetAppCacheDir()
	require.NoError(t, err)
	require.Equal(t, filepath.Join(tempDir, "sgpt"), cacheDir)
	require.DirExists(t, cacheDir)
}

func TestGetAppConfigDir(t *testing.T) {
	existingEnv := os.Getenv("XDG_CONFIG_HOME")
	t.Cleanup(func() {
		_ = os.Setenv("XDG_CACHE_HOME", existingEnv)
	})

	tempDir := t.TempDir()

	err := os.Setenv("XDG_CONFIG_HOME", tempDir)
	require.NoError(t, err)

	var cacheDir string
	cacheDir, err = GetAppConfigPath()
	require.NoError(t, err)
	require.Equal(t, filepath.Join(tempDir, "sgpt"), cacheDir)
	require.DirExists(t, cacheDir)
}

func TestReadString(t *testing.T) {
	reader, writer := io.Pipe()

	var wg sync.WaitGroup

	wg.Add(1)
	go func() {
		defer wg.Done()
		_, errWrite := writer.Write([]byte("Hello World"))
		require.NoError(t, writer.Close())
		require.NoError(t, errWrite)
	}()

	out, err := ReadString(reader)
	require.NoError(t, err)
	require.Equal(t, "Hello World", out)
	require.NoError(t, reader.Close())

	wg.Wait()
}
func TestLoadBase64ImageFromFile(t *testing.T) {
	const (
		base64MarvinJPEG = "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"
		base64MarvinPNG  = "iVBORw0KGgoAAAANSUhEUgAAAPoAAAGaCAIAAACpDcmEAAAYrXpUWHRSYXcgcHJvZmlsZSB0eXBlIGV4aWYAAHjarZppkmM5joT/8xRzBO4Ej8MFNJsbzPHnc0Vk1pLZNd1mk1ERUklPfCTgcLiTCv4///3Cf/GvWbRQ27A+e4/8q7POvHhi8evf/PxNsX7+fv699f1e+uvrYa7vp5nHwmP5emN8fyDp/fbHB37cI+2/vh7s+51s3wN9v8HAn39Fd9bz++dJ8nr+ej3VHzPyryd92vjzVPf3QOf7ws9Uvn/rz2l9Pej/w19eGETpNm5UcvaSSvz8ta8ZFP2Wsngl8TcX4zqu+TyvgYdS5vdgBOQvy/vxGOOfA/SXIM/zvbS/R//ns78FP/9ISvlbLPuPgfrv30jtb6+Xn7fJf75x+X4WePkvb4yZ2i/L+f5979p7/rW6VTsR7d+I+gQ7/RiGCzchL5+PdX4Gv43n4/Mz+bG44iHlN564+TlppkzEX0g13bTSS/55POkwxZo9Dx5zPrl8XrMy8synKE9VP+nlUWa5xcjZyR7IWS3551zS577zc7+TjDvfxKU5MVjiI//yJ/zTm//JT3hPQEgp2lecwAXzysI101Dm9JerSEh633lrnwD/+PlOf/wTfoAqGWyfMBsLXHF/DbFb+gNb5ZPnwnWNx68SSmHc7wEIEfduTIYSqCn2VFrqKY6cR0rE0UjQYubUQ95kILWWL5PMtZSew8iUDPfmMyN9rs0t96yX4SYS0Uovg9zMskhWrQ38jGpgaLXSamutt9EstNlWL7321nsfXSS3Rhl1tNHHGDbmWFasWrNuw8ymrZlngQPb7HNMm3OulcPiRouxFtcvXtl5l113232PbXvudYDPqaedfsaxM8+6+ZYLTdx+x7U77/IUHKbw6s27Dzefvh5Ye+XV115/49mbb/3MWvou27///AdZS99Zy59M6brxM2u8Gsb4MUQSnTTljIzlmsj4UAYAdFbOoqVaszKnnMUplmuZSTblJtykjJHC6im3l37m7o/M/Vt5C83+rbzl/ytzQan7/8hcIHW/5u03Wbvqc586/K5CxTQWqo9rVrbAb4z8+dvjtJdz7avNe1urY+QRmae3O/rppRCQeGxAmS2P11ZlRm3z6lyjEv3HHU6uMCOhyONQicyJRbCw0u2M6LnVt1Yfb7bsvTJ27/um0Jpff/26W+9pzvQuKDn2Tl6n92U11cZ/6qn/9Bi+n1h9ar50w3XSrePw4ml5Em3Ix+Fjbv5IJPi8ffhODuJKXbe/vu60MBEjMLtTGKKtdYZ56bMxAp/nM3n23dVxfZ0NYe/S7lmkhnC04o/Fe2o5QD/c+VAWmZTWPC4xHWv2it54i9fLIclIoOm3Wk9n9JVPAq2pkvc9QfLeKYihCGcXtDZdYe+82vHeTTPch3tGPUtrUc63jZH81Uvk09rU/VERtVNDzdXmKSu5gf3dTgPrr05V2b7VwfzbgGu7hmvLj1KSh83cmA8ZQeDcvkMao1BRu+T6QdE/pCkyZWbVCRYr6JYvpW4AJ1muYby4nnLxWOCo3WumjtBrwvcF3sgWwH7zOaUQPU+DYN0N8ln/6i3vjMBYwQ8Jqm/WkRohfml2lFBe4LQCR0hivzUALTgXZGe7s99rjMbYMx1bN7adw4k+KfNkE/6h06tWSXn0ergu3524rrVJ+867r9r45HmTpENGTkdx761OUS0VNNpJuW/A3pw4LLhspqm3uu09p59DQB+UQg7XfDmBfArGXi+kuJ3Sw+leKO/Wz7MB+83VoS3fwNJ3G/GCsrbSHYJ/WbQyAvObfISvJ709swWSlxvRJN6njNlbG764L0M6nJep0Hre8L63QFVSe9GdZaYdWGErj+kSKsoI9QNYzhqs8IhiOxicEVlft3jvxAVvAXFC5USrV80qmyFrIL0PP5L6hATL87WdkFT3DUncCd4bxDTubvUSf0X5EHUSUA4BbaWS2QDPwiAsgDi1N5OCs+CyiZKf9pk+UyzAqlV197gMOt+k4cLk6byzSccSsuexcuCCM00EneA7a9QiGbkik0LDeh2GWqS50UcemIP+oGf+B2bxTUAQo2LXFX959P5Ov9RVOcW8xuPAr9nboyv4RxCTJGxOc/AWdtrrUj9wCqCGTQGmrzpygUtJnEjo7rvd57y5NFFX2cifB6e1cik7GiYx6g6pcivQOIkHILS1x0vZbad31lsbnu+UFgudwBD2IM3m45DsziwX7eGVAMgRm3NSS2hWanR0j0+cR8N1o7uzLD7rg+Q5dxALQ1EpwWDOMiGXgZQOSy1inM7ovubI4mAyMt5IVC5toQnQFOTMTGgsoMi9YQTK0ZcRymUTVgkHCjkJiBSNzVQ/j7YWrNV4E8dY+kKUiECgGjpRUdnP6/UNOuS8Dcz0ANg8Yq5u22CPC1AiMMujJaJzJ30Bpoi0MrKMWnDKL966CasJsIfPg+6CiLjEkxaH2DfWvciFqV7UT6hP+PjGV2kkjXTXStgMeqX5MhZ1BwwndfM20s9550KO6Pz2qeb4y6NBzOUuJkwMqFO4TXpPTeIggBAtOfjOxLKrEWcUECQAHDZzoRmdmj41Bt9ttEylk6WDOXr0bfdMeTJYbdc8I7Ro4UWYSnTRi/Eg1VCufYLCn0KbotTiJFnUElCn28HiBxGUeq3xSZXnEnxW1j4dMYRSYHwoEYkAKzJSrHcAvH5uoUOKUepR7Bfd+yVzwO/oHooJnT2Y/SnAPzJFCRVMLryQ4V957U6AKDEifxAmJV/QkdXuEWGTRnRA+kvnhtOI40SToLTgr3hQIVyymWTHXWAk0EJf+J1boCnbZB9oTTg8G+uV1mDqgEdvBNvOnahRh/oafNSB0kEQkN3OQGvXpb7D/0CRdOWM6ICzHnxDU6cgUrC44RPCTPv1kkkSDecBMYTYAxS8kmCOst6OcN09PqWgDeGdzZUqxBw1HmiYH4oHGeo4uc9EyvqCPiI2yMhjxzch7DBP87wCG5LXNO2vYAu/oC9R7ro59ar+AntRDY42P5KpqOfrBH6tStHmDelQ6rsja5rDCwh9MDJN3YtbohoZYCBa0M3qyZVaV28lJ6+Cj4ugzgOtlbMfBw3IY8YgTx37yV1jY/hzxP+FXJxBB4IgOz1oLegwYQLr3DE7OnyfAUPtxK3aDCMuKL3AkTAwWOv0Ny7c0hOAkehAP8loeKamP+j6BIsrEA2fgpJgtHcDQseZh0wIcZGw2WlsAjquYoPbeZuxIDqqoZGZ6/sAmXyBkMQ0y80wdaABAcmEHkFLQpLaN8ABkKPXin0aoFBcxuX6rY/SB1GkqHWrslv4y7fXCuXmBf3QKi76Frux7oh7CDwPj7Iz2UcUIDWpIRQ+C8j0nEERdsB5d63UMUXLiieMb40A05LnYTy6R0Hap/4MlHaoUT2rc8NKjJEDVFA9eG1tPHn5/IbIknhCsNF3He0q0m0yFte1iF6Y3qaA+3yQd6YkcRsPxEQ+hCyr9QPN8A1Fu+hrBZFSru80mv7QPAz7hibDnqDDn9CBOduFMVgynZz+/nlYgY/HXT+d4xL/jOJn+vXKUooC4HI8JjQKkmU/l+Tpg4mQ3XQrdI0UK1lzadlp10/WplGz1C/vcg84KMGlYPzVfJNMA2MCHeBBwHdEEyIPK5W8zwo0IZwuWbpGSmT7qjQXuXSwJJNHCYz5kFAHqqW9XqQqkAIXSEW4BEmMRYZG6II2QU6Rkhw/HFs6fW5vcUtwmhQFzcKQJjgGbrQi6bsLkVTao4YpWuOFy73LRoxNUJ7oTNhkw8AarpZSScB4F5A3WTSDKi8CJJomYS1YbBzB+leoka4MZ0WV2Q5QjQlXnFS/WLaO9MAsgKj9RORI0XgXbuQAUGnMFOBfvyZfkoFhwo1VZMfGug7GRqrCYNKcH9YeUdym2+KogO8tcgK55O34NaTFm6IESMaXKpjqgB9rv9sSbNUjuohSRQij3sBsw6dvcoUvgsQBCT07vES0KmqgYQgv3mkO2BTvwGfIaI8DyVYhdZCZlT9D2IO9BaUZhdUKjo2wBbX0NMBFrZlZYelAIzEkVauIB1Z15RNnmTo9ELNNt82Et8K0Jw+Mq88C+QOIK/TY4t444ORgKplEZF3IwY0VAj63TbzhfFFa1Q+ajlcevR0aLpBlWMhBB4/gHu5Hqq8Dhzud0R7LoL5RlfUxFrPGR99MJAZ6HEwAGG4pAV5nwL4xmYd/gxsIr3ft6VTMAOI2V4wDygNjTRek6SpjqCOE/eNd2iZjUStMMJx7K//v8FOWk8Gknw7Vth4liqgBdMvFd4FsMCkNvhIOrm/wfA2PFWnpKYWtNRBZl1tbDN6AlGxuvR4PQPzq+ziLSc97CROCVhjaKqJe8O69iNRXQMbgv+UpKSFXfBvtFlfUGHFLMFBndDS81c2DiQxcGA7jZggAe4pEQzn1E9AGGF28iMwL1U8PUhBIJLr0wO9I14cohMfI1pKHiKgYRJUar+waJU5Bhsgkj9AL3gHvuC/LxKDScbpSECiri4LZzTLeJENcSIcaMesUvmEKCg3ijIAVGA64dM5xCRvqTGJjwi6wDCqHCCBh9xWtGwwLDhbqlSa5J7SN3kDstBauW0cf4OUI5QGtxtQJ/1A/U+9qXfvzBwOCyWjSaLGLw934HX4JW403gyPEG56AlkrXwLjS7/PBLbhE6YA0LmaqVR+YGmgbx3To5PYS6hgNRIvo6LaMYGe5tLUK6FggTc5lgZVEfO3D4RLFq85xC3oSnSsJf2VuSTd9jEaI1h54ETH5B7F0GPwFlHy2fzCcivbuSIp0KJN5LrZq1gZWHhmENKbkWXjC9xMODGCflFtvkPJeSYsYeO+L1hhI8o1pHgC13a5iYTX4IbqoThJiJVewc4RGIBOXHxyDG8FI2qODklATFTXQ8P00rv6oajtw5WOimYkKuJdCfaxvpInx21KaOD13miPla9r0QmFehHtH1yP2qnX05KPIqWqKTxEgrT2RMfw+DgWhVWT7B8WNtRjo0Ix1gsP2mha1qXTQMOAMnh40BJ8Q8q0LKXhMlo4I3oneZ0aYTkTgoyYyecd/4AjUI7H2SVsbqEI85UQLiJ1ZGxVPM8HkODfXJgETHQHlhwjBfR2BVEyECo4EIyYpcGwF1fbZEUQ8V4zT6xFrqc6bMWb0ef7zWwO2EJWCitq8DJsVnftp73bSlXkT+VB+bL3C1UBVxylVWzJgyT6sBbxqWI4hBijIM4JW8LSEq2sbg1l2bcDlCMnABa7uvXyur+0Fr5duQKMflMXD007tC6IQVsQII5OHJATAR/PSCvaEZ3QHGn+VAmiouHrwj0yEErnFLnJglkBT5SNLa5tW0GXUZYc8nmCFLJ/k/1hDPBG8NYZIsNDf6BVqC3zqATZPQRul9G8WIqa2bZMKORLKuBWGQVk8ihkuyZ/VQ0mIWe2MYlyp/SPnFLXJ4iKIi8rVbaRromGdksClLSMc9kAmgy5B+2WoCIigXQgyDguzq53EZ2hIbU0vbZ3CV04PRL5BVmhn/qb8QCz6DVtKnFH9ItaM4nAMEkyHfihVqA30XpCw/CrGmDE+RBaRrXhcVsOqO4IvNtBN72Xcp44w0wRdSDQYBteFWgrcVkcUcitYYy3yznWlPhbE+mBTRirawgHFJ0/pWHqhTFGBdjrSgtrYWxoSIMpDodYGphe4gJczTPvNE6lJ2cye1TpgFjqPKGSo3yztuKI1tG0dA04EQbQoYpLaGR32wjSAJjl9MKh95ZIyrQkEaM+1viXPpP1ysPTjvCH8egDxecyEumQ5q44Hc20qifhxthgpdBRkhC9+W5TEEgEkThc2R6gjmlFGjyo5VYdGUKkOD5AgYOvKh+ssCk1UmznuT9t72iCv6kaXGN2RtM9xIXFaqlwqLE6WqINOQzOosMhjSNveixNgdRd/S7VI1G7kLzMdoakN62VX5LTTijSb+aPciDXWHVYEjmhTaM0THofLUIhgf3FNGygFFBxWVB2Svmva1UYfazcPV4IMhbi2pvkxm/4cQkIwMk/M0IgAExleOw94fw8dzvQjuscsQxWfQB5COgu25iItkQlIuqYd36OtLXwyzRyaLVCeozGq9GqQO6B3YaF7b5nUZ5qVaY8XlbjTR4AuhMA5lP3Cfp1FXyUG+2qf4OdWQvjdBtaPrQT70o29X26DsQQME/598Ww6HwY3L6RJwWXZDqARATEiHTIiQQfOW8cYOlO9VZ50M7v7kZQXtUbvbbS5hZtkbDQj7m4qimGjOyk+FDXu/kjdoHLXEablqKnwfD4U09SWqJW5IIQZdRaBipFgQSC6B0p1kXR0GMoVbw1qtNn2Hp10SUjwS4BxlUDsaEsmxQM94Icx/olujax6ZYRnXF7qTMlchyWHstvxAfHFPbXr1o4uRbogJA62pMFYiFF610Heo+Xfo2ACcIH8n75voj1UlCIygnAi0QUb6DiL4/bOL13UJYYBqUbZQZuAgDp6FPVc4VFfINzRTRAmhI5LpysX2n28FFE7lf5K+WKxsZeShqScRVzazsfCIepvuwGazFIdpBmVgLtC1udW3ByZSSehLTtAVrMYg/7Tk4tB741Fe0yWqUUkeglPe880Ykkb2it2a0g8TGlxANFYMt520FfQfQQeuZLkz7SZ1HCG+IRO1/VwCZLlBkOmnSdCsg6BnjLRrggehqygg4GEOHqrdEE4qU81oy3T1aGrTtephNHOlcrVacbn/AoMoXsOYo3VZaNZoKSxnzshM/0SX3UdcJrhJWnVzzltz/FfsORPttTOP4nKasPwmM4Mj9CrfSL77FCXhmAvpLhEWWocE0LGdcJKogcSrmJFHHGuoza16aWRkKdP2wI6UqI9UYoIqgBFQV6mPmdGRelgQfuE4E8NYmjXHqPjrtMpmBW6AtDIZ3otgmHQJy4KHXcEwVXcAvVkRBXkdQWMjCKxbFU5DyTSScD3RSk59RzMeh/arzNIHFOaPdDI0CclQrGzIUK0f4pe1o4X7iZRBNIwauaTOirgRzvzprLVUVuFOwAa8hjfyax1/qYNVliAFyJ/sQtSoIn3oCD0Dnw2UZPIZYrRZeUhw5chKNxGQkR8Wossw9FZLywKK1paOrbXORZavjBnvBsWjSYfuSXMKnARSR3KPnT5agH0Y8e1CVjTHuAEr06OEJQCGhI4bV/Yu1SRj8kjHKONPdYrya+CA/dovfDZywLrMi3am577IbnSrqgkIFx11EQNsBQXOTCLCbcyUyIu59eYOrKihaVqpE/EpqN7bfhrP4/FkgGkJ+qJkWlhXUll0Vfegrddwh2qoJquDsEDbRK9wYV0TPG0TlkAXjk65yHJwKdGWaIkhJfLlOW+qk7e6eRy77nyN/z2bO3n4xLiJ4ryap9kYoRXRJQ0bWfR5uG/j6MFIMHiPOgllDwcj6esPaVxxxwZ2TxWRYYixSQL++dLAnvdefH9fdIstZmrrUCMYJA2i2mAXCpHu0qoCbooH2/os4OigJ+Jr74O89m00sbTJvc54zfIr8wISlpidIN5DMbTwYq2yFlS1+5Rb/gW9BctERh2lxShIvG3iY8iF6aARfK01xu4euaOfRlGW8QxQttLh83qykVbaaSdZgiNJCafsqO6dNBP4So1Ov41RHIoOsOBfRFe88p8UUzM15CtuSEryP9n35qOjSCZirLno1005k7AeJGCGiMAdX2vAlIuaEzYho6PqcAK1URnnxBlxxNZuoM8QpZLlufCdTqn197ppora0TeiJkZOO1jo6EWRHVS2gWgsuXgeGB3R0wQ5ci84uDZQjxj7RfOtZdLgmodxi6Q6eEQpPEwtlmhQZhmVrW/aSdIiONQUiHGdGMX+fSSCNPmDlMNvWbpQcUkH7Qrfq/Wp8MklJZE+felt7UjQ4DNGyBoGOQxtK4KYq91UyhG3zdpXYhBkXVoiWIIODXykB4VZwC5S9aGHae2lxe6ZTjvI0WR9KJp3IaIHUnWYqY0S2j6I0ik7GfmcA0buB0SRs4UKbAn4fU69xgmZJoF/j/WaDToiDEvvwjAlZF0eDSBrqxCq1P7Sxn4cGgRqXd+prAxORHWmF57OmfdGatEa9cUNYIkWQHxufTPPJLIi5ob057QMbXCcWIGejDntXduIua8VMFsVBWY63CXzbUo4FBQL5VtK4hKii+rCuj4q+5IHVx6mvmdJszXKewDecPSVBlRlJkE0rajjd7zFZgmMPFf8+nIT6qQlfe8CneEFN0JLPw3hcBEjaq7h0RqhPcOv412I6dY3KZDECGvstb7aU1vDzlEd+8mRXCCJ4ztgTxZKh1VWc6jCP94raoevxbXxeuvS/KTpB8qIfGYdYeirQF+y2X7Hg+F3xIgPw/SAFe0Ab53sfjmvie3U1+dwdvGztaXTIMIgHxi036lvkOgrfZazzicdbmva4FG8tWoozMiC9/QQUFebqElf4+ny7iB/TWTNr+XxQCgC/H8BZzAR8YwLTmQAAAGEaUNDUElDQyBwcm9maWxlAAB4nH2RPUjDQBzFX1PFr4qDRYp0yFCdLIiKOGoVilCh1AqtOphc+gVNGpIUF0fBteDgx2LVwcVZVwdXQRD8AHF1cVJ0kRL/lxRaxHhw3I939x537wChXmaq2TEOqJplpOIxMZNdFbte0YcwehDCkMRMfS6ZTMBzfN3Dx9e7KM/yPvfn6FdyJgN8IvEs0w2LeIN4etPSOe8TB1lRUojPiccMuiDxI9dll984FxwWeGbQSKfmiYPEYqGN5TZmRUMlniKOKKpG+ULGZYXzFme1XGXNe/IXBnLayjLXaYYRxyKWkIQIGVWUUIaFKK0aKSZStB/z8A87/iS5ZHKVwMixgApUSI4f/A9+d2vmJyfcpEAM6Hyx7Y8RoGsXaNRs+/vYthsngP8ZuNJa/kodmPkkvdbSIkfAwDZwcd3S5D3gcgcIPemSITmSn6aQzwPvZ/RNWWDwFuhdc3tr7uP0AUhTV4kb4OAQGC1Q9rrHu7vbe/v3TLO/H1zucp7I+fTLAAAACXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH5wscCRgpZfJjFgAAIABJREFUeNrsvVmTZNlxJubu59wllozcKjtr6eqNAKaJjYTEXcSMgZKG1EgaG1F84IM0HFE0k8S/wBf+AD7yB0iUwSjYjImkAA5GwFA0EoAocMYoAgIbRG+1opasrIyM/S7nuOvB771xY63M2tDVHdfSrkVGRsZ2v+PH/XP3z1FEYHNsjo/GQZuvYHNs4L45NscG7ptjc2zgvjk2xwbum2NzbOC+OTbHBu6bY3Ns4L45NscG7ptjc2zgvjk2xwbum2NzbOC+OTZw3xybYwP3zbE5XvjDfiQ/NZ99tT/r8mjcYHAD93MfAoBnOgtwDe71J0AAwAX4+RWg9CyICAA4+z+eBQAQce5+5xmLo3gvIgDAFknfj4CIoIiAECISbZbHMzAuL3x7x7ne/nK4U93S6/cx/Vaesbvn8kSXDREhGkJav6Q3cN/A/bxwP+MTCwDkea7WurTZqLf1e6u+veqG/rV+j94Wkerfiahm/XlhVZGArAL2Bu4fbbifC/FYd3y47uaAiCAToLo8UgLLO1dBHBEBHxNv3rn6kpiuHKTyPRACeGHxwCChDQQAFfjleYP3Ddyf0NdX48pZmgIBoQCiIcDSpZ7aXRHwnpmZWUT6/X5l4PXQ24XlRiQiY0z1a2t7e/o8IlCsHEomKRoySGRt6ckQAIDQTNSxMe8buD/+VlBZb2QQLi0ng3gBj8IehIRFpH/SY+fzPM/KI89z732v15tzV6bfKSLNHoi4vb2tCyAMwziOG41GHMfGBhDEQFRzZhAEAAmkvtJqAfcG7hu4nwPri3AHhbsDcT7LkmySJuM0z/I08d4/uHskDrz3rjy898zc6XQqNx1rHs7cndWvx8fHiGitDcOw0Wg0m81Go2HDaPfgkIyJgtiGgTEWyBRvUtdABX2sv+cNbjdwPzvcZ5xgBvQAPB6c5Hk2SUbDYX8w6A1Hg/F47LO8FTVJgMAACYFBAygEJAatIFf3621BNmgZPDAyePGgtwXZUpD7zGU+c6nPmcETGDaIJg4b8fb27u7u7s7OTrvVMVFU2nUsvJqKJNrAfQP3OY9Cz0SkJtlaa4zRB+RpFgRhzRVmySbDUW88Ht4/upNlyXg8TNIxs7eWoigKgyAAiwIkKAQGiFH0LI4ZRe+vn8GL3r/0jAz1s0foj/PcO5d7Imo2m3t7F/b399vtTmdnBygArLkvQoDALLl3ImKMCYKg+uCIm0XwEYZ7RacAQJZlzBwEgSEDnoEF2HmfTUbjh6cPjo/u9wan1gCDF3aCYsmYgMLAWAogFxKoP6GevfdzHouemfmRb7J6n4xowmbiXJakSZay80gmMJZs8ImPfTxutrZa7SBuAhKAAIOggA0EqP7p8jx3zjUajQ2aPwpw56Vwn/tcGj4q9oenpw+Ojo5PHoyHo9xnIJ6BxTsKMLTGhjawliyq64AOVsF9EeuLr7tqQeoNRgQbsCA779j73DkWdp4FrDGNZnuns721vdNpbzXbLdNoA0LuPJLVN6NGnWhT+PSRhzsATCaTPM+bzWYYhmrjT48fjE5P+qfdhw8enA5OkSVsRJ1WO4iDdDJCawJCCqwl9OCRkZlDGyyFu1rxRcSv+j7nElJ69gBploGxARk0gSUCIhJihIdHx2DIorVh2Gm3d/cPDi8c2O0dEBBjAUCD5rpLszk+InDnMhCdcWYUkYaMgPT7/bt3754eH9+78Z4liYIwjKMoCIHQZXnus1ajKQgIXLAgKAQEANbaan9YCuuly+As7tbU0gOACAsSoCAYtILQiJq5c8k4UScnjBtbrXbUal/92D+wcUMhrsEJABhjqve5OdYc9sXH+vLDe2+NBYAsz27cuHH79u3BYEDsosha9NaCNYKk/reUtLt48RrmGjKGAjLr4LvUmTljyFgFl2EYMrP3HrxnLwDg0SOi954QG42GMSZN0zxNjicJn3R7Sd7e2b1w4cLu7q5G4VVEvkHzh966cx30deuepqn3fjgcHh0d3b9/v9/vA0Bs6eJeM5+M1M8hoiiKwjAyxqiZLKsRiYiMCRDFCy+14nXQP3aoCsCEU0a9eB4wADAcjq21QRAhonfsnGNmj/Z4MA4azd3d3YODg/39/a2trQ3QP8xwn0u5ALhpkr2EEQKMR6MbN2689957g8Gg1WrtdLaNMT6fQD4CKYg8TfKLgPfeWouIhBYRlfbWYgEbmjVwP7vvvtKTYYcIREZDT2b2Tpg5DENEIyLOOe9FU1QYBOPUjbPcOReF8f7+/uHh4fbebqPRKop5hBZY+aXLj5Z+qxu4/2j9E6pfjwLdAuLEibNokXxgBHzOzpO1QAQiw2631z397t9+x1obB2EYhsYYBGBm5owlF3SzkDWVj74IX+U9zk44np8n4aUrIc/zpcXHWhOfOZdlDpA6nc5LFy/t7x+0dvYAtRih/C9cMAeAhTEQKhZn+WCeyWxtfPcf9eG9EKEgEAIEaCAAAALL+ZgCJLIAIJPx0dHRvTt3u91uZMgaigwFhAYLF8IDZsUqeoSr/cj7n25aR037mZaNeAOou1KWu8mwd3RXhv3B5ZfzRtyKWi0wATB4BmMIEABltntLq+YZwCz1CM0G7h8Im88MQMbgnK0jrakCyEejWzdu3rx58+GD4yRJXr3yspZkVU4IIwBi0Z604J/UcfzIGHQN1s+/DLBO8lQh7Cq4s2cwFKrzAy7Lsl6vNxiOJ2m2t3vhpUuXWp0dIIPinfOIYoLF9/fRzb++MHC31lSVs8Um7Zh9GoUEeXra7d66dev2zVvdbpcAW40GGtJ6WhHx3hdusfg5rM8BfRHlc396Fta9nqJaNPOLjJOWWIZhCBhYa3MvAHh0726WZfqnZnubjAHOXZ4ZGxTOzKbK4MUKVbXWHBG1BkZE2OeQjbsPj27fvn3nzp1+r0dEnVa72WwqNI2yLPpgZhEWEiCZ6Sqqgf7svvvTjVUWffeCKVqEu3PVrsViRISBEKk/GAGZMIh3d/cODg62d/eDKAJCAAfABdbR1Pxzqnz3ehSycWY+MJyMeGAgW1wgBAHvbt++2T/tnp6eMvPW1lYQBHEQGmPyPEdEAUZEU5pMD0wlWuutdHNOyxpnZtHML6VfzmtxKjemcmZWPY8mmJhZGBkEEQURgF862D/tDfq9bq/XG/R6l19OLl68aFtNYNlY9xcL7lyQjOINEiKCsDjX7/dHw8Hx0X2XZ4Zwq93SbL/3Ps8zIgJAD0BIIhWljYuOxJpq9TWE49ODO8xhvaroXH7BgsB7D4KCjLo5MzsQrXdoROE4SU+7DwGAndvd39va2QLaAP1FcmYYgIGrnjcA5v7pydHdo5PuMXIOIMjivc/zXMsDlbeu0Fy2QoNWeyHKYnNdZVYXrfhSlK8MJZkfZzEv8vErrgsRMbPSkUrMswdmnkySuNWMg3CcZOPhMPey3d5q7XQ+9ROfFQQkAsSNM/OBg7sAMBcBmTVlr76mNr1zeZ6nk9PT0+Pjo/FwIuK3W808z9IkmSSJFvpCWf9YE7RAITSAgExEADzXXFeH7xy+6/c/0sOBR/Hxy2o2eSnK9XkWH89cXxtTt4fQKqUOSCjgWLIkTVx+ePXqy69c3dnfB+9zL0EYAuAkmTTi1sZ3/2AsQQRjDKLGpo5YEAWt9hHxZDQ8PrrfPTmOomh3Z2cyGDjnsixzec7es0jlixdOMKH+IogAWGKb6hIaiz7MI5kZeKyamaV+y+IDVv265n6c3hACMOINikHpn57cEE7T9PDwMAhDfUwj/ogWx3/g4M4MQEAIIgUVAywiPrAmHY1Ojh+cdh8OBz1xjsIQvR8O++K8c057fJRWRiIygAgMoNs46GYuZAwtspCLnsxZItQnKRFbH6rOLYaZEJYAhEQEUHOkRdmAAWHhSs2GUAyiIRyPx6fDkXOu3W63OjsAwMLrxJs2cH+enoxIYaj0WgfWIgh4AJfd/eHt69feE+e2trZaO500TY/u3lEHRvXmSElKY9CQMYYBDKK2/kxNviUSWGrOz+W7L66H9YzK0r9qlfIi717H9zwfLzS3ALBW8wwiCAxgCIRQLJI3JsvywWBw9+7dQ6A4jtEYMhu4f8AO55whQCJ13K+9994Pb98+PT1tRhGhoPgsGQ/6PSXmEAANGkNoyFqDasI1QC0VuxCRsLD7i1TMKn59lU+/ysCvCmEXTfUUo7XtZW6fWWxCnbm/OPtCnclrkMMIiMCWgAEhCE0UO+du3bo1TrJXX321s7PzkTXwHzi468VlBma2xgBAniQnD4/+/q3vC2ftRiOOgmQ87KUTl+VhaCsTTERk1bQXcGeEoquzSicVJhDWp5bWp1rXZ2HP67uvN+eLf51ZbOUjCYAAPRQqUQQMiAaFCJhdGLeZudfrOYbt7e1Gq2WM+WhWEnxArXvFRbjc3blz5913/n4ymYQBUhAAQJqmk9HYEEVhyB7UhFcCRlNrPQt3IUSZVgfOJVbrtvaMcF9q8s9OUNYRvH4NzLnygoBQe8MypWtFxICG5QgIBkhLiK21zWbTeX/37l005sKFC83GR7Hljz6Y70mDLQCYTCb379299t57OzsdRByO+qPRAFHiRoQGkywBEiAB0lJJ4LJeoEAz6NkQGAOGwNS95zmhrzmsr68vWLokipVDiGiWngFIz4yAaKra+vUFmLUzIal3DoDq9JeuP2lBL3gQj6CdI2oFvMuswZ2dThSY4+Pjuz/84WgwLDPTxRc+Wzq84d2fJPycf83C2i0qjKr0ZzKZhGFoDB4/uP9X3/rmyfGDw4N971IUJmAERgQqyWKyISISmSJtZAwVjfqmgAiRAkthYVBQlkB5EeLVpsHLDHndNs89lSDAgqaMKs1U9zB4FAJgdlll7Gf5dYaaAGWdhgfQNJNnBhFf0O1zuiMsAOAF2VoGFNb7SQgRDJD58Tc/GcSNRtyyYVAQYgKA8OGWNfgAOTMehAC8c41GBAD9XvfGtfeT8SgOLbNTewYCglLUa6MwgiAAgqCowtycaUQwBd8OGqwilXZxlfGe8XAQpPL4Vzvrs1inwmBqb3d5Fk3qougZwWirIaKRsk5zDStf+5UV/yJTz0wWzlxoHTMKIHDh5wOTIKIR4X6v1wFoN9r6D+IZkIz5kFt3+0HbaLIssbYpuXvv7Xfe+cHbeZ62m7H3HkGKqr4y1lTCeRGyiyHjjO+BiGgBAJeVQ84dVHgmS7A+5+tPXxHXBSQLgH5E/LrI6jDL0iBnXQ2PrgwREKUxRQAePnyIRFtbu4G1+jDCD38P3wcF7poiAZZmo8FZeuPGjRvXr52enjabMREJe8KyC1MKJwihSJDOedJQFxKD+cRQRTjWXfY1iAezrohgCUWIcC5+HRC1gm2RgVkckVB3e+rOzzq4g6/75CgAyCg0HA6iKOq1unvmAtnABkYY2AtZ3MD9SYw2rAnCpkNhWECE2YGxP7x9662/+17vpNuMw0YUiHfabScilQs9LfoigoW5GtNfV7jpS4PURWcGAMCQ4DlqIXEZ57iaX0dZ9oT1f1kE+iLuHxU9eWDtUiUQAhFEEJHhcPjwwbExZnfvAiAhQZbkkQ03cH/mtGMRjYkkw8HN6+//8NYNY8xuu0NUEfBa+1UhnpRuW2xKWjTG9UBToVxhvd7dt2jaGQENrSEol4oLnIdfhwL0Cw9Y3KzqJn8R9+u/28KjERFiZBCC0JosS3v9bhiGcdxstNvwEZCCfuZwl7XmXvWMhAVYgAVFvv+9vzu6d1e8s4ElFJ3hYgmBuahYRyxox5rnvmoDQVjwc4gAUYjAGCACY7DG1hdDNcofg4ikUeaKrGrlHJcmE+F8/PpSZ2axZgYWuJc5l2YlqyssIgLoUWNjMyVhPWdZNhz1B72WtRREjSgKNnB/Iqw/coRpUb+qN1jefecHeTZuxxFazJMULcWhtQSOec4xWkKEAz4yqTkTiS74/fPMDADQSmZmDmeVfwKzVS5zwcOqKsjFmjBYViC51JyfRdyGimyXR0Jhj2SQBEWyLBsOh3EcB0EMhj7c0erzcGamZd1VFgmAmQkQiFAkTZK42QKWP/u//s9hv2dJrDHWYGCMDY0loypIWqdeA+jijq8QKSwfEVljKzBZGxhjgCyQFpKZSlmpvgDqqVkAIGt42eUnIlk4kD0IzhldvV3115Zds8VfrbUltzjDr1e5Xn2hKimrejj6DNWgKE2dLq4KRDSEAGAIWRCAKuWpLMtsGAU2JJHxsP+QhIjaO9veFzS+unlQpu3Mh4KkfH7OTN1qyHQSBQBR3GiC9zdu3Oh1TwnYgNfUIREDgyAgE1IhQnA+12W21IQRjCE82wEADCK0cr9YEkqSAS/1u1fxiXXfvW6a19v19bXyayJdEUEQYSEkYSmcNWEEFiFmFuddmuVpasNY8NGfdAP31STjrF2ffoOI4D2QAcCjO3f//u/eGvT6AYheBgIA8SiCotnWacelIAlSObwO6lMb12tpFEQkGSTCopisEiuAOlej/24AAElwOdYLf71ISAqAkIaBwnObwGp0LteGX8O7L/pmjwA9gBp2X+Q2vIqNFU1eIihePOmgHhya7QvN6inPQvts4D6P9Tm7XtzPTMYAi7rtN2/evHn9RjMiEiEEKlR5GYVQAFi0DqCcvr5cCoaQStIGl3rqRIExgTozRGStrZMzi84MQ8HMrCE96l5N0X8hsOjn1An42nsmlCkZu8rLnyPsV3n8i+EsliGT+u4ePIouP2/AAjCwZwAEdlmeJAkjbO9fgNrLLdJEG7ifyZ+Z+8Kcc6ExQJRPJv3u6b07d7M0Pdi74MYTQ2AK8AISQEE4aoZpPumzLPpUvM4w6wpuNMYEVlR+xhg0hkpmpvp/rJeLiQAhr8Y7AghzMSpVBIFReBHuUE77qPviBVKXZYiWeyMLFTtzm8AiPS/a8gIMQCKCAlxkV6tsqyCLoDBznmYikiSJDYNKVXi9LMIG7ktMe52lofIsZQK93z29/v614XDYbDYJMDCGUIiACCwZxAJ/JCA0HV29ilGp4I5o6loDKvhPNiBjpPzTqlC1uroq5ILr+OwZ604AuMK668PmnJypxvWC0zLnycwhe5Ge13vmQ2cQIBQuuFIARPEChELMHkU8obWIJLrtOOeGo37MrWazqd/Dh2zI2XPy3WkVu+X9ycnJ7du3DdFuZ7vXe7jb0p4jJINkAAlKL73gwgt+sCj7XmXgEXEK6BrcLRBJ7U/rmZk1F7oyzxWjosZWn6ZOs8DqDuuy4W4dz7jUxq99zrqB16ZW5ZaQp36OsPeFPgOAQTJUMFqDwUCAoiiaE2j4cBz0rJ8dyzPO/MpkAHzOeTYanvZ7D4MAm62w130AZIksGktkBQ2iQTR1gRSupfQFgWfXkRDqD+gGYYofsYTa1Ee2toxW/5ABMggkqNvFuh8NHJT3EzIC2kuH1bMhEEiVHiMQ1B8CUzbbKd1Jetb6+Po9el68ZEvrGqbqHQIMJEhcDSgpy9orKlOQAA0YC0SMkCRJnuf1hboJVc/juHu2piBVhL3LHaEEhAEKENy/f+u73/n3V65c2el0vvWNv7x09SIDe7LWWEZkEe89ERoia22pBIQABERERufMiJbdkpFiRzCAaOPYGAM2AGMAUQyxsWgDNIE2+emkl8quV6IUZ7QHZGr+zDRT74EVS4wiwFx49izF/D7PonNXmUGABZAMIAqXXk0xL9t7YUFiAa1W91L43VperPtc6WwUhcTqjTjnRJTcRCJMvQMAESYhRGOQjLEI5NkBhTaIyYRM1mlpvzFhGGdZNh6PoyiCUpfzQzMg5Jl/DItUgYYFiMACgQHwBJzeunG91W6EFo+O7m7vtCeTkW01Kp6xkPwqG5YK0w7KSlaNODx1YctebEDUHCwRAiEjGiIxBIbIBIgGySBZKAODx93oZP4HQZALSlITRoo1KBS/ymQtl//udfUKCkh11ip5Kc5Mgqx/FWV+6qLdK9LG5SBLECAWQSwU3Vn0qxFgFBHvWdgLe5AgNIEJrCI7z/Msy1SJ7cPkzzxzuBPNjuMgQiRABsLu0fHNmze3dzrMfO/evZ3tzsOHD6HVWLx+U5J4MUItqGujDU1EUwTXiPbCnAdBgGiKFm5j6lHpYxxLq7iW0PNVyTKUNT/Tj1AUsjzyharHw6OI8JniUPVfUIQRivSFgDDr98MCLOI8o8dYgiCIm40oCLMsc84lSVKxtBu4nw0QM8yMKLtYjAzj/Pr71zzncRwNTntJOva+WR/9XNUJUA2+snBRVUNphldRF6UseDTGABVYt9YiWiJbJ22e6APOF8ZgNda9LP+arYIs4a5Sj4jkhdevpbOkUedMQ/X4qhgStbSBmXWUJZYRefmNElEcx+12uxk3xuOxIl5HVm189/OHw1JO2gBNq8iw17t27dru7q44f9rtNqJ4MhztX9hNkmTRUNUQv6h/ZKb8IVpDlkyAiGistYGhwJBFq556YE0ItX3gqWzTszZ+NieKOB2WUfgYrMPPmLngImXd0y7w6wiwbkxxHfGzFcJeymKeYsWZUqQEEUmMwSgKmnEURZExZjwe1wt7dF7xhpk5h5tbOOHC4nKfjH946+Zw0GtG0XDUH4+HjUbE4urf6SLWYZnW1yJlrpZbbbkeld+ytAryKcJ9sUi4+nXx1c/yHpZ2f69/fJ1LJZmWnyF7ZsfF4VE8AhsolpPOe1Jbbq1tNBpBEKwqwNxY97X+jIhWAai8b5ZlLkuuXbvWbDa73e54NDCABnGnsz3onZINVnmi0wQtzkwhRSAEg2CILJElNIRkbWittTYkKmkcMGUG6ilnxVe1a6xxZqbOz6Oe8yxuzKrlUeghg4CwFoyStnUTOufQBEY3EM/ee+9zYfZQDJ4/v3L3xroX+Q4PyOU0dxGf52lydP/ubqdz/OD+ZDIJI5vn6c5O5/T0dGmItojO9YIZpVakWSzoXeMJPEXcn6Xs/lys9hkfv/DMXHnnIIziBTwgA4vPnfc5ixPwzJ69F+e9zyeTUSUaXicYNtb9jNdJ2DMZC94l43HcbkVR9Fff+mar1bp//75CzRgbBbZ7crKzs6NlWwVMizG/4lzh55T3Uz1+tTYMgsAYoxkcImutBSH2Rb0NobXGWhsgYhAEi4v8yQPWylo7zs9iAqrbxhhAqZI+Uw5xRlimepOIpU68PrKaslavTa9Ku4wxwOI59955L+BFy5HQQBhEwuxzh0AIwuzJYBTHoWjCgOvOzAbu5zDvzM4YAhRjCICH/d5o2AfvBTwAGK0yUf+egGDBFa5v6/O0jKkkk85Sv45onrNUlhKRj3zAeo+l7iwtBrL6a5Zl9ZEkROScY5ZJkjE7nzvxXuWcGDx5Ox5OwJANo6DRDKLYpkGWpD7NTBRjTS21aiJR6dkN3M9gyTyDYSAKbABZ9uDBg9PT09AYYDGl3AtVOcUVUozz/KOGfVBV/OqPmf2hmmAMlSJ1zwXis777DFTPtEvAKnGliuicC2yqSrgKo9577514J+KqZ8GyPi8IAs1WoEiWpF7YHBsGuXL1NcCZ6p0PU5XYs4Y7K1UA7IEsBDTpDh8c3RsOB51mi5mRxBRY10gCl+gLLBNIKrwaIKpRMrggp7HUxj97cz7blAQyUwi2xHIjPEqoYy4UXoR7o9FQ38Y5pzlR5xyzC8OQhcCAeFa1MgOEaAitAzRBgJa8ILC4LBuPx8Ph0JTO4VOkaz9CvLuxFrjIe/f7/QcPHujMMAA2NeKMgGF2ENKqqHSK6QruaKkcojqDeCAEKhibJ04qnTmmXJJmWv/4xfr1Vd1MFfimFe1EyuJXQNcar6IONBARAwQqMIiMwCAAw+FQyARhFFEzimITha1Wq91sJUkSRkX7iz7JBu7nM/BgULwgMAh3u92TkxOLlOdpYNBYU6apGRYU8Jao79YI7CJohXXWfd7SP5crtzTNJGf7l6X8Y22VztTH1/c6TQzp1MGqUYsInBuTtnAggrBnp66NMcYJe++cc2gcE2ZJmkRJ0GhV+QH12qsi6g3cz+i/A3iGwIBz3W631+tF1jqXGzQAplSmFqx1Ds0weissPREhIKEhLMgcUJcd6hR7LZYFU9TIP0dn5ly7waLUXj0EwlKRZqkzk6ZplV+r/ZcnIgQB45VGZ2bvhJlb7Z0kz1jQOeclAZcDgEeI2x0b+KWdhxu4n/XwnFuHPstGw346Hjd3Opw7YEAMS63dJepF66NVWC3G+2ymuZ85l1HUSWhfSznbur78EeodKsUtUPlLFC4UMEVlGIpqIxQGAGJgRIM1lRFEFHGIxhgKw7AqaczTLHdZaANBQiExznkQNEBeBJI8yTLHIIYgMGiNjULbiIJ0PDLGREFoIiJAImNt8KFRnnnWcCdAAO+8oDV054e3rr377sFeZzwcBEGgTiEz50iIgCbQDKiaKGOMlF1JRKSTVrX8S3dtay2SkEEkIaOCBlIsHBJjDSIaS8aU0zvAg4DK/z5L342VGSnK3JmxNJCqWlz0gqi1RpNlSanXLgIoAoIk4oFMMVAKWAj1MSic5zmzgACRiRqB97530j3pHl88vAzI4pk9MztgCYwJTJymqQh6j56JgZEMWCBDmUsFERDEpy73hD4Z+jxJ9i9GEmcWxBpSPSBgBiLvXEEVFyFEsVJfrIXwXKKQshgjSRL2RRSF5cBTIAQhjyRgBIhXv6VHsivPk4FZ+R6k+FofOQyj2s20Tr/erIRF9xbV/6p7YLvdEUEk297aZpY7d+72+/0L+y8V3ygizLw0lfSjyq9i2UelPzrfxmP5I+zyLPFZzi4DdqpkyN6D9/VwCOFFdeSfR5oJAEwYgs96vZ733pcz2s8FylWOzYu+vZ73s+R5rkWLaZr2ej3nXLPR6HQ6Wkla6DTVkx5ny40oiUnISZJEcZrnecN7sEYHJ3jvgygqSWJ4ccVTn0eaSZPf436/2+0ys/cy30Zd6iquvzrTVp1lRTLPmV9/inCvBDnlmrUkAAAgAElEQVRmJWiWz2Htn/YODw+dc9evX8+z5MqVK1vNZr/fL4p+QScM41letwpDNSdljBHv0zRN01SpTNIp454r2YgX3bzQc7NeSkFWzQeLA8DOZd3XDwx7kS7AMtmPRWGF6s5ms+mc6/V6k8mk0Wjs7u7GcawVBOdd6tWKqmVhvXa75nnuvQdEKN+P916WDA/cWPeFLxW8B8KTk5N+vx8EAbtMhGcvD57RCi5W0SxevxfOuq8yvbBMI/ull166devWycnJ/v7+3u72ZDLJkyQMw0KtSc5RGV8lqooSeO8RPHpf5aoC76FWpUdEBIExL7CFfx6hqgb1/X5/PB5rnd1i1+nZ9/252x8OZ+bMJW44Ho/TNA3DcGdnJ45jdTwajcbSD/7owR41nZzKwGdZliSJKnBA2dOkC8B7/0KXwT8XWSVmwzyZTPI8t3YrxyLJVxNKPx8+ZK1pfHGjVZgb27RMJ/769eutVqvT6YjIeDwOw9BGESzpmj1ffFV1tVZVN3meF3XXZDSQBQBS9L+wX/szh3ueZUEcTwb9Bw8etFotY0yv17v40gHp1F9m8QCCglgMRS3J+KoooPp1zXJSdXODRT2gc84E5pkSOKtUeddDas7pEhHvXT3qqALWPM9brZaacxGJoijP816vp358FEVhGLLP0zRN0rSiVth7EbHWVtXw9QlWdXV57329Mh6KCnsOjMmy7PT01BgTRI1GowFBQABpt0tEaGwUhQDgHDCIfdHmlj2nrGrV2V5dgGdtnhcHCjyj11ryzDUN3qV+hSx7wKLJr+oC1HtWv2VnZ0cdQuecsIdSblKt77n4x5nB85XwGDNg4dgUajNkqivFzGrcX1A68jl0MyEAjEYj5bayLCOiBSMHz4hXWQQZPhvEF58IZKmTsIiqRyKSiPI8Vz0MIhqNRmmaAsDe3p4C0XuPKilFVNnpSnf7kbUuS+vpC11LLLCepmmSJAxYb5ViZjL0gnqSzwnu/X4/yzKFu7YgzM85etpf3wzUamn8p/i5loGYUQQX5p6uOa+8MNYqF26MURU7Zu50Onp/oWVnEBHZ+7k26rNUdy3qZes3VVl3hftkMmFAU9Zu6EuDIcQXso7mOZUyD4dDdWPyPLfW1pmZZ8eoLKLq6Rb31XnrepPoohVfj/5V1j0MwzAMRWQymSRJYq3d3d2dTCZqOOriatqxumYzeSTTVU85VZFAlmVKyGhoVFE3AC9qsPo8lAgAYDweI6K1VvU1V01Yf4pwXzqT8ZmyT/VBYnMmdg3617x/raLLskyd+DiOgyBIksQ5V2knqZmHFaO0z471xXeliNeFVG0F5fAzwBezbuY5Wfc0TVUFQBsLqs75Z4R1mJ1gsWh0n+kLLUXzIhDXI1LdhjzPJ5MJALRarTAMx+MxAFT9FsoYVmP9Fjexs7R7L1qH+vzxqaJBuQBqWtgbuK+wslq+W4lnLJ0t+lSM7NxZprf5ibAuy89YnlU6as2YpPN6U0pkaQULEbXbbSIa9gcqdhkYiwIKd3X068/MAP5su9miqisKazG9iumVosoEOgGuWqMI+AIOYH3moWoQhiDc7/eZeTiZbG1tJcm4GUcVoV5Uo5e/2sCo7FdVQ6Ij9+rjRU1QNJ6yOBES0fhQBW09MxCBYzYiaMAzSu6JyNpAvSlAWmQOly43XA30mlaIAWEW0ElLzmcEPB0WUBtWM+c2lOOfyjvVaqqSJmKj1e52u1oYE8exy3LvfRAEqojEIsxskchYEVbQ64QeYRYGQSBrA2NclimFKOKrgeEArC64iKg0gRaxC0AUkA2NseA5E5EwDG0coQ1iGwhNdPlFURCFESILM6DdWPflNr5e9nR2vnyNR14+Zhqoqb0V7blHnv0r16iap2DaCwPPOtGgnNol53OZ6gFu3dzWg9G5C0aP2iWKoBOEz82NszCz5MDFJ5SyARhtgDaw1qIBFABkEAbgF4t9fy717jUt2anNPidfvnLEHCwP0WQZMcfMCA5RkAhqdDIWm/OqLX/1Ryt7lgR0vIYavHnf/ZHTT+cSEerMaHBfr9Rd/kmX/frYh/deW7KpvFgqKwvGOBcge2NfYG2CZz9XlRlryepKoXetdYcZvlxqxd8LsWD9roKmQCbdtmvEu/4JkTU1A08uBFcOl5xlIblYA2fmglaFMXmea3BfD3XWEJ1LOZbHgH41eF6/ompgGxCEYYjsdUzfU424PkRwr2d5Fimw9RwiLno4y6g9xfE8K4LTv3I5v7cO9znEn/vKMdfBUV9zgPx4VraO3SzL1Kyu4exXHfAEA695disuLRQwgDEk1kJ9zizixplZye/OXldceclxdoMWWDovt/BQiwEBwsxUs7WANLcD1OFeeQiPQLk8gn+sm3ZmLtwrOGulAMC8PF31wbMsC4IgCALtt1jc+tY4M2cvVVjzfhabB7TZQ+Axop+PGNyXXtQ1zkx9W0CoVXfMjcmdzWvWn7y+M7BwZd0JGAmEBdDA6lU39dpl1fusR8ClJ1PQnUv8k1UfuWKoYFa/V313zVHMwhfXcJ2Lhh+f4HrNmarqE5Rf/oZ3X2otZsfBnWWfPSNXvTCSRRbvmf0T12mcJ4nqnpbTXO9grEcgugCqMoE1XP4ztVDTDzjbqicb617H39QwIHrnWq3W8T3farXGo0EcWs0I2rJc2nsvKIbIWlNdd+891RScK/vnvRecmajIzDp7UU2ZUVGuohFBdPKePtI5FxoUzhkRPAkRl6youGXCeiKFbGVtY4FSmVFnmqpRB2CRAqBV2DD1uJjXWPdyGKqoANhoNOr3+xcuXNDb9bliiOjL8hURUU0Z3Vse6XTNQbmKOqosqT6oKicOEaMo0npM7wUNMrMwB6EN0HjOxXm0ZmPd57/vx5DZWHX9Fqy1LHV8V0VvsKyUZX7Pqc4VRT8bba/bo1BmIrlzGojqyauk/dJXebrM41lc0Bfaoj9vZqYazznbbrzuXxad8rl09xxeEaeeANI0bEWVo+Cp3qoQAwMSlQOLBKbDklBY9E+wpnC3nBBVLoVitDXSY8Zv9a/Fl+1Ij6xHWOVELfWy4My1OksVEEDmEV/IpWyYmUViS0e61XMoVYvAGmZmVZgls2w34cwMUaX59YURUYEuleC/iHiGakYXTssTFu1ZCY6V28uT8H1zS7eqcKx7Ncu5xdW701JYr0L/etNepUfqDQkb6342uBtbCRDAo9Rx9ZLWEbyUMSywTlSyllLdwyXcAUAnJUjlJyAyukItCAgQAUihDtVP5eDJfJZ8xeY+Nyr+MaOdyokHAC1zX+p/y9rekVWPWV+tuQru00FupUM3nRz7QlaIPRe4A0CZDOel4lirELBYULVo3YFmAKFZVREBs8S6swgj4JmtO+v7p3kqehXOHlsFoG7diSgIgizLllJPq7j/RzozcOYeWT1X0ztmnMz6v2+a91Y5poXvzqV1Xy1WMufM1J3FVYGpfvN1uFeTJ4pLUuahSvYd60Cv5t3hdHZ72ZpZEOGwJm57Ks5MvS9JS2WyLDuLqX6kQw+re6nWW/e5YeUi8777i5hpevbWXRCQiEi1kgXnIYICgoxClSfAAAZUHH05nqr68vntlAWokI8mYQYiZBEhQQE9I4D1IlRzlgp5XNX6LH+dAY2+IgEKACGw6FlQawNB34nWvp8fAloSQzX1C6hqikQEgPU1yveyLlSFtQoIZ3ozAIhUUb2mrOsnBhJmKD+yAJABeME8mqcG96XdwQAQx7EIfuzjn/jmN7/ZaUYGIM/yRlyIuyMIIgZEgiCcZWkeBEYQGQBBvDAKA6Ny8/WpKSxgEA0ZYEYwiEKExUx09ALMhIAEQmgNoScCg4RkBADRMAAhMWBRXCNChFDwRTI/JUH3G65IIBBBmi4uIESGwuXh2bmncxpJSyPLra1tEBoMe+PxJI6bcRzmeW4MlrwPshdAEBZEIIMilXgqVXnW8iWWBBhz2avqqPyoakKloAFjcseBQBAEcRyTMSCOANmzy1MAIIrUirFnY8yL5dE8e94dCRHB2CiK9A4P4oWnE1egaKIpLTsUhYVrG+GociCF6l84aqG2ek0gUAxtlWLchXgA8NpgVauFr167fnvuOzrLeVVQu+j6T9cSGBASEVVnn/easNY0BXz2gOcs9AssG5EiQFATlim2K/YsTkmw6b+jkRctWn0ORQSgoarqvNXzeYtqzouEzJq6gKXhGjyqYBAYdWD94pnK2R96W1OldO7Pi2sgtbRDtzK9dXGBJ4mUzs7A1L/5xfc8/SyVaF5NI+hF1Jl5TnX61tq9vb0q914l1Rd15ObraVdXt65qf67v3YuqGMwMM8X0tfq+2dvl3kPnX+FLBLiXzpOqVrjS7RXVzefXHT1XMLpqKS6O8oTSDdP+wHqjwqZEbOWVULjDrFjhKuu+FKZnORbNW91PLSwoCzjQH/SoP9VtYqruRI+Fa3R+rMOjpH1htnxFO/zr7UtPaN3hbNobc+tzfkpzjfyt1D42cF9/AYooU617FWvOFOvOluDN2eaqPPCpODP1vORcA/V54bLeaq53ZubIUIX7XD3cY1j3s6eTVglazRURVBRvITpbFvNUF3cD9znGpkh2bm1tVQT8IlNRJ9cf25k5A9Znqr7qP/qG5s74WJdzzRBMWDaGpCqVeRLf/clD1cVtpz6hu7LuT7L5fBTgXgRhysxU5d1LezSX1nafKz5bg/jywTwjRCO+OpfNGfXzYwfoj5CLWlrx9uQT2R87+bVqFtDcWnrRBfWfGtxpxaERmPd+//Dw8uXL+muWZaPRSPXfjDGqVl5pOgNAEARaFVjJFKZpmmWZJh31rwCgqsKwIIJV1Z/4hcNYRIOCOi/UM7AgA4GeySIaJItAwOAdu2Yr1vcZxzEAJEmCiFE5LQPK6p1qsNGyIrPpTJhFD4qIdi5c2N3dzbKs3W7fv39fZ8rWkVcVbBlj5ty/KtqZq3KpzEp9LIfuIfrv1W3VGjDG6FkP/QKzLPNpOhmNer2efudVsUP5vy8Y3J+9rFJg8yw1NgDxOzs7vYcPiMiG4VazpQkd55wkCSJ6KYLaOtulolkqDlphSFGuiNne2jPGqOizrgonLCJ7F/Z1aCsRoTHl5afesIdgDCIaExiDxgTIAphPUkG0RGiM1WnGiER4586d3d1dZh4Oh61WS0QGg0EVc9fHhukbHo1Gi436dXjNuRwPHj7YShKd5CMizWYziqLxeFyOjJRVzsP6+bJLH/PI3abecaapvSzL0izP89xGMbz4x3PoVQXvPUAAABcvXnzvB98noiiIkiRBLFx5U0ytKPaBSoiwXuKioqpqhPSIoggRdZofluCLomhnq9NoNO7ev0dEZI211gSB/gsitloNICRANGSQ9AyElsKiKIBQ77dkEHEysY3tbU6S0WhEcdyMYyKK2m3JsqWQajQaUz+4lJsDRKi1V9fP2/uHAKCmPYqiyWSSpqn3XquVZ18Ca9VZuKiiDKBjtmcozjkffXGmzdyMN6m9N5UaTrM8y7Ko2Zoj3WUD96VHNVjipZdeIiLPYK3tnvasNc1mUxGswVCaZVEUFSa53Mf1T6q4Um21ejEAYG/vwFqr1r3SVXTOXb16Va17OfxWbwCieNCGOu8493kxs2Vra6va96dSoGCA6frbbysvfvfu3UKsj7kuwFsJrlel6msM8FyM4VLf7XaPj48//elPB40G9fuDwUBLA+ozCUvglm0osxXRj7TiqwRo5xeMIr6krVR8OHfTATjwgle9Pw+4V997p9PZ2trqnZ4AQKvV0uFLCtMKBJPJBE2hXGWMUYeYmY0xQRA0Go0oiiqxIXX3RSSKY70/TdNRMsnz/PrNG8zs2Od5npaDtby4xKeCXDnc6uIz8/3796s7p9QnI6Tmnb9/59VXX33llVdu3LghIru7u3fu3Kmc5mo7UrirYO+iRuRS0w4AyMTMly5d+p3f+Z1Pmp8IgmAyGW1tbTmXzc0hqxVmTqeUzVtxWG7F62a+ctmrq1O5+1K26up71oFkAqhzuusylxURibiB++xRVctgFF26dGk07KsV9N6pFx4BNhqNZjtGY7IsYyhEljViUzzt7e1Vsa/ODNK4dtifdLvdu/fuHR0ddbvdwWBwOuhPJpN2Z6uy7lB1+5NABGzYUoAGCAyDFw9eXKvRBhICAyTAqPcjw/vfv2HYvvHGGz/90z999epVIrpw4cKtW7dWaeJNJpOlVn+p7y4irbidpikzHx4eQrnswzD0PgcAgboVn3FmFuUxqvnji2ndNTMj5lh2JJJqDJNzeZ4HYdRoNOrWHTeT91YdznEQBGkyjqIAEF9++eWbN66xS1yei3BFC6hz4kWstWma6igidU7UAN++fTtN08FgcHp62u12+/3+aDTKsiwZs3MgAO12sLOzY4zpj4aDwXBre8taGzXiVqvVaLWazWYcxxjS2I8whMhGNrKhCcEACTFynuRgIKAADCBj5jPOmR0Mh8Pf/G//+1//9V+/fPmyzmJQ7ZdWuz3V9J92Qs2zsBWvz1PF3Zlz0NgCgO//7d9evXp12Ovp0A4dw1Q35LPU+IxTXvfmzxLCLoV75cwYY1y5J+kYHGutjkx8cbXynh/cCw/VuSgKAWn34NDY0DlnQgyCoBGFOmzotD84Pj4+7fdv3r51enp6dHR8fHw0Gk2SJMsycA6UmCGCKMJ2u7O9vXXx4uVGo3Hp0qX9/f3XX/uxq6+9CgDf+c53vvzlL/+7v/n7ZtsDoXEuzXOb50mWAhEJMgkyehALJIRIxAAAJmgWHJzqpiP7nHOfeQD4lX/yT9787GcH3e7Ozo73/vTkZP/SJTceMwCKeBE9E4AXCa3V+6szaQU/YvWY6iwi9+6+d/X1N4IgGAwGzmdXrlwB4H6/32g0ABgEQQCBRBjRiIi2b2GVO1DRbEJgZBQUw2WFsxbwM+S4dlXMZZqICMtaZ21Cr1gBWPDdHylL9REkIk3uuNXuAAB4Zgpe+8SP//u//vbe3t61a+9de/fd4wf3j4+P7927d3p66hykOcQxbLW39vZ3Xn3lx/b2dzpbO1Ec7O7sx42w3eps72xtd3a3Oq04agahYYCtTksYkyxtNBqXX7nc2dvZPvjXX//6t15+Zd9E0dHxMRwfv/zKK41Wx7Nvtbdzn4EYQzELpuOMAttqtZjZi3AuYRDsbO9ev3790ktX/vSPv/Kf/xf/9LU3Xpc8Pzp+0Gw2gXA0mTSGA/GsTE51FgFBcFy7H0AQWEAQcu8FwbPUzwCw1WlNhqdO8tQlBwcHp/2etTaI4qIbppb1FREULXEXEfDCIsiCgEYtAYIX4KLbhEiYBdjaUP2qUqAaBLGSJQRt2yUia4kIyWoqQFn2zLEWgISNxsOHJ865lw4vQdF/bPLcB4HZWPdluVWVt6CwsbXVbG9jEH/xD/+3+/fv3v/h3SQddZqty5cv/tzP/dzh4eHhxcuaeIrjWJX8NU7S711peFPy6CI8SRIwUgw5C4Otra2f/dmfvfDSS4eHh1//s39748bDq69eaLc7779//fadu5/85Js7dsdaawJrgxAQAU3Ofjia7O3tJZNJFAV5mg0Go6uXr37n//3bTmfnC1/4woULF9BQEASOvSUbxpF2lELVnPUE50clAesd36JNRgDAKIhY9N0WrVoMZReAWuWysFN3l3WezEz2ADGwASJmWeYYoijS0Ov27dsi0mp32u22/u9oNNrZ6WzgvsRlFAHPLkBjg2B3d3dra+v27duXL1/82f/wpy9dfulgd6/dbupojc727qKjqVxkNdt2mmlHDqKInQdCZh72B8zcbrc/97nPHRwctDtbX/7yV04ePnSO1Q95++23j46PLl99+eLlS4A0GA29cNxstlotg9Y7CZvxZJTk3l166dIPfvDOf/KPfukLX/iCjaJsMtH+USJqNpvrp3g/t2NpPms2VEVAFDSqG4CEMF/hDEiEZJAMEmn+LQxDRJxMJmTDnZ2dZrMJzL1er6Ic1MBrYnsD95kjz71K4Xknzjsb4s7OzsWLF3/t137t8PDgYHefjKBna0lTj8PhcPkbtbYqGKwIeKSiHSkMwjAMk0l2etIdj5NOp3P58uX/7jf+xWuvvf7FL37xnbff23/poNFoHB3dS9NUCZ8wjsjaRqsZhaGuJWAGFgKM4vjm9RutuPHLv/zLBwcHo34/TVMd026tjeNY2cYfIcTnaJl6f+AM94KISIhSicTPTMeeLW03xoAxSCYMQ01Rx0HUbrcbjYbL8z//8z//zGc+86lPf1b/VzN6G7gvwj1XD89am44mNgzjZvPll19+8MYbxuBkMukPuvkkiaKg1WoVIdFCBU69K75if733GiOyOGtts9m0JpxMJmgsIna73Z2dnf/sl3/l0qVLf/xH/8c3vvGNk5OTvb0dE9hut3t6erp7Yf/q66+1Wq00z7rd7t7eXhiGms+Kw+iv/vKbP/MzP/P5z39+Mpn0+/0wDJvNZpIkurf8yFMti4XEK5YBVKm6ue2o/rCqVEaITEk9aVVfq9UiYyaTyTe+8Y2LFy9qEq3KsvX7w+1OewP3mvtJpLPBVLeavScyh4eHk8lExDeC6PDwMEBK04lWgwGaKm1ZYb3OJ9StOxkIoshzrskma0JrrTGBiKDIycnJ9vbOP/r8P/yxNz7+qU996mtf+9q7197Pej1BCcPQGPO+f/f+/fv7Bxd29/fEcxhGLssDYwanvWF/8Cv/+JcvHV5Mx4kxRtNbSozqbMcfLdBXkd8Ly8AICfLyQstFywLG6DBQEYnjeGtrK4oiYZ5MJr/6q7/6C7/wC1EUee9Ho9H29nYYWqLmi2Xdze/+7u8+2/VkTZ47oxJ5LALeGGOJ8iwdjYa97imLJ8AsSxGx0+kAUr06r7LrVdWedsg3m81Wq9VsNdj7wBgEyFzuHevQpSzPoigKrE2zzDn36iuv/uTnPndwcOCyPJmMEQmJBOD0tHd8fEzGHBwcJOMEAdNxYtC8/YO3f+yNH/uf/of/sRnHeebiuGGMqVdf1knoVf70UkJ2WRTP1tper6f0tk7krELhJf8+q2uw9FVmde5FtBIGioEEQITGSMnJoDFoDOkPURhGmhtrNpv7Fw7iVitL08Fg8B//p//48PAQAHXNl+XcLxgD/zziLe99mhbjP/WaGGsPDw/jONYdU23GcDjs9/tr5koXdTVpOh6Pq3xTkiQ6UToOwjgMm3EjDiODxM6HYdiM4uFw+Pbbb3dPTj7/i7/4e7/3e7/1W7/1yTffnIzGWZIe7O/v7ex2jx/+u2//9ag/GJz2JqNxliQ3r9/4xV/4j1577fXJOPW+yHz1ej3FolYgf3DC00f694vppDk2pkbLGLKBpoGVHINyWDEAHD94kCRJEASdTgcARqMJvGjHM+9MqT87sgA6YAEQcNm3vvWN7/9/fxeE1Gm20nTCzK1WyzPM9UrOVXhrtFp0RZFEQaihGBoismgMoilrB4oqAmMCCqz+4+uvv/7d7373y1/5yr/5+tfeee9dAWi0WzYMmnHDe/+pN3/8zg9/uLu987/+z/+LMEdRVLDa54fjot1dWkQQksmy7OTkBAD29/eJSMPiJElgdp5HcaXYQ60mZ66Xd1FGRuuRFicuVQ3gVWJbv/Ct7Z3j4+Nms/nGG29EW9uTweD44cnJyclnfvJzACBANZHUsmhh48ysWFs6j1O9HDMZDvqnvSxLLFEQ2EKwjsxShbpFFWYiQgJDhbIPIRESIhJQNTMEBJCQyBAiIRLg7Vu3rly+/DM/+zMf/9jHrbX9QX88Gjvvs0liyQhz9+HJr/1Xv/oPP/95n+XJeBIEkTwluC+6HABAgCKSJImINBoNLYG21rrZgmGYyWSu7KZdfJW5Vut6A0dVTV3dg4i5881mM8/zy1evsvPGmG//9V9/8pOfjBvNarGUH/PFg/uPkjq9cuXKydHxu+8N0zTd2ekg4mAwiGw4lxBZTHrPXWCCQgRvnuyv1hiLlGPKDvZfGg8ngvATn/nJz37uP3j7nXf+5E/+5Gv/9uv9097Jcffenfv/4BOf+NV/9l+TEDA249aq9uZz++izf60TgpU2ZUXnP0Zrdi3FMWPFp5PdF7rgK2K33iKcZVmr1ep0OuA9hfE73//+gwcPdi9cYH7h1a5/xHBv7+298sord+/dPj1+yMxxHGtlL5RdQrDQuD3fO6xDq0nnVouIIAuQIANhUVVSqimJkIjn8XBERCawKNKI4k/++I/v7uz80i/90le+8pXvfuc7N2/e/Gf/5T99/eMf7969x85vbW0Nk8lTuc6rhiMwixrXPM81cwxPMKSy+qKmLXazHcD1dliN+3WladmphiXMfHj1atLvx2H8ta997Sd+8nPwYTl+ZHCXNMUouHjx4sHBwfC0pw5rs9n0PIPsetvyUrkiqBRSNTageaxU+qbiWYhcnu/tXYjiuD8c3H9wM4yjK5evfubTP/H6q29873vfe+t73/vn/81vSJK7nJuNZq/bp9iey7o/0jAvNrBqWUS9s+SxrTssdHUwoJepz1GwMWrdbWCCEABEHIOoyGUjtBqJxs3m22+9de3atX/+G/8CRDZwf6IjTdM4CkyrdeXKld7Dk37/NEmSdrudO66zy6sEmMolUY4/0JgbAVmKWWSgheuzs7Y9d1qdZDTu9XqAeLC3H7eak8nkrbfeevXVVz/2xhuf/eSnrl6+8v677126eLGx1Tl65+3tcHdpcct5nZk5rZFFZ6ZeE/94cF9o9gNBQDbTkotZDqAaU6xOlHr2jTjcv3Qp6feDIPj617/+5ptvbu3suDSlINzA/fGPMAyBGYAuXbrUe3gyGg3G47H28s0NYa1YhWU9DSW2eDqeCRZnhLAAiMqi3r17d3d3d3t7m0WyPE9OT4EwDsOje/euXr168eLFe/fuNRoNEfHjcbvdXuW7P3JY2hpQzg3lq6YGLGoJPrb7XnAvIF5ZBpQAACAASURBVFSK9NZj1opO1TolDY6ttWEYthqRcsff/va3b9y48du//dvw4uttPFfevXIqBKtXRIqb7AAQGzv7uwcHGEST3KdewJj6jxDpDUasfnRAJCMCGg/IggKWxTASI1UFhIx1BtSr3O/BwX4Y2tFodNo7ScYTSxCHUSMKXZb3T7vifL/XvfTaa71e9/r163t7O1TFu2c7A5ehwuy5+mv98SigckqCqO/PizCAr6ZXzv7ULtn8uZxkRQCABXlKBMbYKRtTdbUXiTwbkLFaH0bG2iAMwqjR2gKg1u7+V/71v4kbzdc+9nEA+HCY9udh3XF+QzdgivsoaiSTSRxHr33i07ePTt966y2HYStq+jxTwZYwjDTFMx6PtZymvhezqNC/8QAohgTIgwE0SEQwyTIyxlhCRBYGAQMABjOfgQc0NgwtkRXv8nSCKK1GlIyHBGZrq3X/1nUT0Pbe9slpFw1pc4aOHBBVcxcQ76vWjbmzasbPnQWKx0v5GH08GjuaJAwYNZqOJXPeWuvynKjQaxfAqgiYWcALggijAAijCAojIAkzIIIuq7ICWMAXvC9ZYwxSAXntmRqMJxSEcavhMTHG7O7uBkHAPs9z/6UvfemPv/Kv//Iv/1IEkCjL8sWCMNzA/exbirIQQRg7LyJ06eprvVHSPTkOgiAKTNyMdFPOPQtSGDdQs4CqSVvfW1nbIECABEGQGASERBhnJ5lpyxGjRzQgHrEY9EGikPCICFi0KAEgo0dA1Q6ScuBHdYbz6L6v0YOvxmMDkXcu915vc5mkKwTNVINJY06ddDJ7RpRigrLMnkEqJ8bUapC8zlQxgQ2iKKYwDKO4IQKDXu/k5ORf/qv//ed/4RcPL15GoocPH+7v72+s+xMd2hFjjNEajFdffVVE/p+/Op5MJpYazWZT7bqa+TiOlSabk6Wtu6oF31gOYNKxZ8h1qaDKb54X2Vune/gsSQnRZS9TLlJL6tWlXqptv8qm0uzUkKkfj0xEhKKUDOG0ZriqtIsiajQaiJBl+WQy+aM/+qO/+Iu/+OIXv0hEeZ6rcs4G7k90BEGQ57lGSzow/uWXX758+fLD+3e1SEPFw/R66/5bz7auGS/MLPMjMGpRWh3uVSHrYn3sFF7IsHLE5BOv+dooMi0o0j50LUuc+1xrsA7L2pREBHS56yya6kvR6WvioczpWmO0SHswGFy7du0P/uAPDg4O3njjDVULbLfbg8FAFW03oerjH1p4BKXocxzHb775Zmd7h4EGo8kkzU0QRY2WDWM0gQ1jE0RkQ7IhmgBNAGQJ7dwaqDImc2TOVEplQaVxTll7laL8onTMUzkAGNiheEtgUICd+BzF138IWH+APSCf46cIiMsol533eZ5nWZ6Ox0Pvc0BGFGsJAbxzR/fvfulLX3r33Xd//ud//qtf/aoKYm6YmadzaG5FLX2SJN77y1euXL16tdVqVVV4auOD2lGvZxIqlGSwpn21auYWLOiSrjoWcb9UzfTJDxSxZFRYCkr1dF2usFrEeA2/uWwMhJ/RkMqyJEmS8WQ0HIrzobGBtZYMANy6detv/uZv/vAP//Dq1av7+/u///u//+DBgyiK+v1+1Z+6cWYe/9C630KezpgsyxqN6LXXXnPOpVkugM4zCSCS9hlNryt4EEDSSfBc9dWXZ54OvQGu7DoAlaPqltQYn913f7oC5zagPM89AxFZg4YAxHuXkVIzZWu2CGPBuGAtiNUxlIUG4NKsbdErA44ZBakAPTMzGEuNRmRCO5kk3aPuX/3f3/xX//JLw+Hwp37qp27evHn79u0/+7M/+83f/E11KT8cBv5HZt1VOQgAtO+zBDQ1O9u7u7uq46OtQ2rj55qbVm2y1WV25aEXuFJ8hrUC8KtGg8BsdddTPKp8qlr3+q+LH4pk5YSFR74QM7Pzum26LAOA0AYmDAFwMhzdvHnz29/+9le/+tU333wziqL333//4x//+J/+6Z/+/+y965Pk6VUmds55L79b3iqzrt3V97lopGHUg9BIyFpASGsCO4zDn0w4gs/8G0QQDvsDXmOztsDBYu9iDEhgtBIgIMQIBUajZUYIeq7dMz3VPd1d1XXN++/2Xo4/vJnZ1dXTs9JIGknL5FRkVFdlTWVlnt95z3nOc56nLMs4jre3t9/P7t9tag9vcJrOFsBCM+SsPXPhYqPduXr16v7+fqPRaCQpSiWPuYcujmZw1pYlw8x6F0l6762dycB7742ZjQwXYpRSqIUf74nsfi8XHmtbvXNv603wvVIiqItSEggd9kVYEZa1GefTzc3N4XBorU3TlBDGkwkAZFnmjIG348Evhsr3s8HQ1LUQwnkw1oZe11pX13UEkGUJABfTydWrr/7DP/zDH3/uD9M0XV9fv3bt2vLy8vb29sc+9rHAnzl9+vRilRveVxH7Xh43RM7aKIpOnz5NROPxeDqdhgp+8SovtoqABFjn2Cwy2AOw3XHuzSKm4cTq/gmQ5wT55IRR/bt7sx9S/4RpKS3WAkNdp5Q6PDwMGjsBnQxXrLWW5498GFJ5ojwLDVJAdayxlam9ByBsNpvNLDva23vzzTcP9vd/9Vd/dWNjQ46GZV6UZbm7u1vX9U/91E+FnirLsvez+/dnCkvS2TqK09ObmZB6a2urNsZYHyfZooYJ2V1Yy96SdxYxaKK/s/XkO+Du71C7M7xLbsy393gmAGA/e27eE4KSQgra399fX1/XcVQWuamrIEoaVpzmDcyxj8VFOL+6F79RS+Xq2nvPhMZWk8mYSAYt+TzPb25dv3Fj69d+7V/FkX7xpVd/9jM/0z8aJ0lydHQkpfzJn/xJAAi2Je+H+/ftOc0NPLrdblmWw+HwhDABzC2f2FvjfT1vTJ13J0ttvsczQ0QAz8RBinWBu4dv3auPj1UpzDNa7DEYHhc1z/ckuxPd81Ve/Pnh8aGrCWrJi+b7wZXtty0w5pt+DM6HpXIhhDfWGqMkKaUE0ssvvwwA/+e/+Tfb29txHJ/aWJ2OxqPxIIpT7/2lS5darZbWOuwT/gDXc/9TDvfFO+qc01qfPXv26Ohod3d3UeospkXMDCylc7TgEloPAMfj8HisBx6BIH98TfNta/cFQo+IPK9ZT1Q732nt/pDLwzsO2f3ek0QAImo3W/lkWkzzVquVxkld18ZaKQTQyTWAE/X6fe6czFVdWFM556ygYAoiFSFynk/yyfj/+ne/e/369SiKxuPxM888881/+BYpPRiOAeAXfuEXtNbGmFBQvR/u369wD+9ZwASklMvLy2VZLtYrjy+bAYOIIpwXskhc17Vz+GADdyzi7ytmFoPVEwlyfhqcLGb4HXUv3kW4IzCzP44bhifZbDbzPK+qKrg1hfDVWvuHPB83b6kXlX2AYkxVeOeMMbawDBRJRUT5dGqM+6u/+qs/+ZM/OXfu3K237lx67NFXXnkljmOPdHDYP3PmzM/8zM/0er1QPgVTqvfD/ftSvHsG70FHSVCDQaLVtY2iKE7U7t47cBBrDfM9IEAfIDzv/XxezovYDeSCH7ba3Tpz3OZo4VYfRVGjkRYFlmXunJFSai0RmecF2wkI1dp6sZgXMChrrbeuqipJwlo7mUyE1M1m03k/GAxef+PN3/iNzz7yyPmrr994/PFLw/7heFo0W21r7fr6+sWLF7e2tj7xiU8EOsOPtJfqDwXu/tBoR6hrw8xSEiLkeVlVJopUmqbHnWpmb6p3KIVQcvYhtRAi6KDP/jTGY5gjnoCxHwaoPwh0fHt+3N85g8Cz4/sK9wUJwhjTbDabzWZZlqPRKIwmPDxkbOy8t+74BxvLxoaoZwTHvqjKuq6Z0FrbHwx+43/71812evX1Gz/xzI+PJpPSurWN9cNB/2D/6PzZC6vLa3/0uT/e3dkLKWZB92AMivMnPwCA38/u7ybcAeLoXpmYpTNYQKt77vJKUi1QEFiLJEgEE0pBIMizra2x3ggk7613zjv23qMAREKGuq6J5EKpZlG0nKhkFrV7WP675/k4d48hehgc+fYtrJ8TV05eZggeQgTNlBMAERBQyNo6Dxin2XQ6PTjqJ0WZprGU0rEN2xwCyTlnKuNqY+o6S1IErorCWitJkLVlWUwmY8up0hELqryNsuSVq1f/+//xf/CMKk7Wz7Vv7+1ZJAPuzt4eCdVuZL1O78q3rty5c+eNq2+sra2WZR3HOjw5Bw7n2ikeAIEJBBzj+h9/Hx/ymtD74f6dngA4U6wFyKsSvXPeB7U9klqQIpIQRFx4QRcHcJ6REOhhcnMnyJXv8M93LGa+I+ENH1w+HnxkUCoOexgLmQDnwnKHm41jg/UsEgjRabXLsswnk7qsBJITwtZ1URbtdru2fuvmjdXV9YuPXPr3X/jiZ/+P3yrKOs4ajMIzeu+dB49ESgsQj5x5pMzLfJIvd5e11s6BrU2FKLVgZgcOZmuBwOCYMUIBx23IfwgLhh/pcA+DmBB/zrkinzAzhkYQiaQgJYVV3loAZARmQvDM7ADIe6T7AJYTatHHp0vzr7zN499FuPPbyyHdmwOcuNV1HcyTw/g56LA652prmGfUfxlUOrz33h8dHYX1UyFEMc3zaaGl6vV6N3d2xkV57tyF7lLvS1/803/7b3/34GC8tr7iGBHJe3YevPcCBEmphW40GteuXTs42P/EJz7R7XaFhEYjAwLHXpDA2WqVA2ACQhTBSuHBv/+Hc/D6o5rdAzYfbIOstd7MbO6AUUoZRUnlp8wC0UFYckN+GEnmHaaqC2znOMz3H0NmHhrub9cGeAb/tj+xqMvDH7vAdqoq50BVJ/LOMbMz1puZh6GWMriehAvDT4vheLq+carXXf3rrz7767/+rwej/AMffLyu62lRApBz3jv2AEJgpJRW0Wg02rq51e0sffzjH19ZWVlw0oICHM0oaYzABIjgkentefjvh/t3GeIP5ngAaLVaVVVNp+CdD7E425Oog8ai8sAMngGPB9zDJFcf/KX8oK7BOwGLcN8W9X2Pf3vnvUVNf+IWTrDAA138XYhovWKWsxVvQGbvHVgPy6trBwcHg9FYl3UjSZWOp/lg93Dw+GMfbC4t/eEffP63f/u3EfHChXMH+4eIAqUAxrA3i8wIAkEQiMPDQyXkU5c//KlP/2yjkQKANc6xFWK2B+hgLoaDnkjgPQGrd0BB6P1w/97clNZBV9F7j342XEdEKTRi4NIoz2ZmO4RwIosf3/178NJaFDMPe+S3GesPL2YeetmEizmE+3FJqTiOAxSzgNtD6Pf7/SzL0jQdDYZ7hwdCiGan3VvbQJn837/7+5//3B8XlV1fX7eG69oSgQBChIXcGAJ4ywbMwe7umTNnPvnJT16+/BQJEdxuCFhKYnDeOzkbbIdu1YN3J442PFndvJ/dv6dYqpA6ilNArKXEEiEHx46UZIciDBelCtvWgAQP0Hq/HYrYiYh/Bxwd4O1JuoG2uZAIWMD5D7t4FpfB4lkFjJKI5lk5bGlhaGTu3t07c+ZMEseTcW4dCykYqKjMF/7oD/6fP/gja+2lRx/b2dmpqmp9fb2ubJ7npISUQhEFJNQa453z3j79kcuf+MTH02ZWTou4kQghgn+xM7aua5IUR3E4l6qyTHV03FTN49z6DPCHsHH90Qv34xp693K8UkQkSTGzrSxCTSSYmVAShUU+8bDs/h/DZ+BhEf/uxkzf5njr+Cr6ggR/nK+/IMeH2+bmZl3XOzs74+lkdXml0+m8+OKLf/3s1/7g8188c/ZiWZZbb97s9Xpp0jg6HDQaDWu9Jo9qNpkKKx9szalTpz760Y8++eSTgDCZjnRERV3VdRlFUZ5Py7KUUrYaDVJUluV0Ok1X1wEAmOYlPjI4ZAASP4Tl+48kMnM82pwHIgjmE4jowQGATrSUsqyKqXPekgcG9J7Z+5nIzSJDh6haoD0LJsyieHDOHxf/P369vV1x4hHvZfcTco0n1gLDFx9UXj+2fnWPDhCeoZQyYO2MMyKdc64GAEJr7dGgX9bV+qmNLElfeuXlP/n3X/jKX/9td3llMJ4QUWupaxnq2hDJqjJJEnnvwVlEcHVdlWUURa1W6xd/8Rc/9ZlPrW9ugPc3b9788z//07feemt/f9d73x8cZVl25sxpW5vtuzvtdvvJDz3x05/8F41GY6ndFVkKAAjExljv0DkZJ+FCIiHDK1GWdRLr98P9u8z3M9hLCKVUhAlZrb3xzFyrOkAWzIQAgD4Ad8ENJmA7Abw7nji/m5m5cw7w7dP/gy3yg+qnJzqEB5XuwHO320VBR0dH4/F4wYPf399nhLWNdWZ+9m+++v9+4Qt3bt06f+nceMpkGOZ8uPls2SMKQO+8AecZnFYiS+NWMwP0RVHcvXPnuef+7ktf+tLrr1/N87ysis3NTQK8u33nyj9+Swhx/sLZpz74xObGqa3rb8RRlKZp1mi1Wq32UjdqNhUocB7AAwdDRS8lhUP4/ez+XSZ78HOBWyLQcaSUYo6YGSV67xkAy9I6460Dz0KKkNQDJ3ZBAD5RKy+IaN8poubZAT+0Wz3B9T1BJF786kW1Ju6/IaIkGUgWWZJFOh6PxweHh6PRCBFPnz2T5/mf/9mffeELX7i9fae31E3ShmFLlXfeIIJ3ntEzshBCKmID3lrrLAAkOmo3k1Y7feGFv++P+kT0wgsvvPbyK1GkVpaXm1nj2mtXu51W8CG7ePH8h5/8sfNnzxXTiQSo83w6GvWjfpZlS6NRd3ml0epIpcPAFRGddYiIAoXA98P9u8jrAAsxZ56leSQhAESWNqWUipTWejqd5nle5NPKcBwrb+/tv4U11pB4FmJDi8Jj5sT7rhoMeMBL4/gK0oPhfqJIWzBAg/jCQmlHS72zs2OM6XQ6VVXt7u4WVdntdlUc3bp16ytf+coX//RLu7tHp8+sry4vF5UDQkZ/jNPviZAEIDIRePTeu0iqNIuarbTRyE5vntna2rp27dr29rZEiKKGMXVR5JES/X6fiB65eOEzn/7Mx575GLC7efPG6bXVcCkCkinLvd2do6MjofTFRx5Ls0aUZkSCiGeEewDxfrh/90EfdBgtAwIIAMTgEoxSSp3ESTIeDAazJXxrvbULiWeYC10cr5iPG7h+p03prHa/f20KAKyzfL+IHwAgiwe5x8fz+kLEdPFsR6NRrGLv4M7tncP+kXNuZW11c3Pz/3vu67/3e7/37Fe/ljbTi49cKOtqd/9wqbtcVaOiNhD+Xg5RjkJgUU6IgAgipRppstxr95ZaWTPZ2Dx97c2tw8P9ONZLrXZd13t7e865Ry6ev3HjRpbEj1y8dOnCRfbWVvXG6hqyFUiSAAg9o3XO+QqM3dnZabba3ZXVNGtKKZzz/r5VmffD/XtUx3tmAJACpdRSCK11PPchiyNdTMc8j7wQVcf54scr5gVj/t1hRycq9cVu4X2/AiyAOC7hfdw2eXHOhAs1/Oz+zn6YrA0Gg2azuba2dndv9/Of//zn/viP3tzaUrE6depUo9Ws9/cn+YQH/aK2xjhkVkogINJMq6OqiiyNG2mapvFSu7O80ms2myTk3/zNV1+7+kZRFL1eTwhhbK2UWlnp3b17N1LyqSd/7OMf//jG2krI9OfPnZ0Oj5id91DXpXGMJIVGIdXwqG+sk1Gso0QqRUSOf/Cj1v9Ewp2DZ+gcC7sHIwIIKeOsgVJFUVRXnf7hfj4Zj8djY0wQoFxYIy3C7t621LtsVd9Gfem4UtKJydGDLelxh6ZQ7i/kQ8KutPW+2WzqOLp58+aX//IvvvjFL968fXDm7Mrmcq+qqv39fSFlnCRHR0cqShk8EKIUyB4Z2HvPNk3jTqfV6yw1mlmr0Ww0UmSe5pOtrS0G12q1nHPD0aCuay1VXdd7e4fP/MTlT33qU099+MlmmmmpnLcAvt1uOm+rsvYlG2estR7RAQrCgFS22rVUigh+GMydfuTD3TMc93+j+Xxj1hGyIyIkipMkjmO2Jtayf6jM/LYAHx/mrjEjus6GJ2L+zxOf40Lyml3QkueTYgH+ftoCAhICMxEzhqERE/Dinp1lQvDogb11laltbYzzKooqY5vNppTyG3//H37/93//ypUrKNQHHr80yad37+wRESMaM2GGSCWMiESCQAgkIGc9sgVrV7pLvV6vt9RNkijWkRBiMpkM+oMyn8Rp01o7Go2UoF6vp6UqimmrlT36+AfOnj9nHRvH65tnvK37/SMmj4gq0ikJQaayznlm61VK6JytSmsqhITnZn0/2BT/ox3uCCDx5Fdmn8wQdHH8e6iwtbyaZVnaaB4eHo5Go6qqju3pgQc0zjsGwSAYENlVpZSkVCQlCcKgse69FYH3GlSzAT077x07SwDM5L33znt3D2UPEvUBh1ZKBSmLssyBLAhSUglJioDJg3fWMwlZ1xY9k5KKBKGrinF/nDea3bS1dHt39ytf+cqzX3l2e3tbyjSKkuEgV6RTGbNHZ53giEiAgEpaihi9N7YEazRBGus4Uk88dikfT9aW2pNJjoo3NzavXLly+9YdW7uScyJi6zbPnrtz506v12MkIPkzP/uZJy9/JI7jVqs1nU4VZGd7vf7BXVMVVVUxijgWyrm6NGVd54Mjxx7BLvc60Mlglg+EZ0CmH5RezY8+EPkdHwdeKNXtdtM0HY1Gg8FgNBoVRXGikHDO1XXtvV1aWgqLU8bgwtRFiKSu6zl1BGca6uiRCLwEcLNEhowIRMjM3c5ScAkvimJBdBFhhE9cVZUpp+RRxqoRZ3EWH+4e9NaWm0ljv39w882tytUXzpz/0OXLL7765p//xV988Ytfeumll8BBu91uNttKRaYwABDcP9gJ9AQeHbi0kY2rYTkZg7PNOOq12yvdpWaaSOCV5a63drnbyUsbRdHh4ZEzXmvNgFprKWWe58xsjBkORh/5yEfOX7jY7S0LIbJGA5AQUUlsdDpVKcVEmaoG5wmRJYcd8KK208no6OBu3IyyRpOALFgF0fvFzHvbzEoppMySJHCqsizL8/zw8HDRFC4qaSFUWZYA/vjk9bhn2InKG4AdITjyzAzEEAZgjAij8bQ2Loi4G2OEsEIIQO8RlVRREgMxO3BsHWBZ1c3O0s23bg37o2a7ceb8BUb/xrXrf/6VZ5/92tdf33rr4OCw011K4yzP8/3DAyFEb2nZ2cCPY6EEzjelfF1hbYWHNI5Wer3N9bXV7lKaRNPp9PT6Rr8/7PV6clwg4t7eXmWcEKI2NjjEF0URx3GYwX36059+7LHHlno9dg6lzLKMhABvsNUqBKEN3E7LzCS8EIJQVNZMJpPt7W2VxnGaIElnrJLvh/t7eRPiHmlRqfbSUrvdZuY7d+6UZRlEy0LmVkoJISeTiRC42JH13ldVxcxB1extDb7hAYgdEbMsC+dGWNqYW2jwaDLEipQyC1KAMbm1VgjhPKXNlme8+sb1a9euff3rX7/y0it3D4Y6ijudTpIkVVEbY7Is63a7k9HUe09MQsjZDMF543jY348StbK6strrLi8ttZtJpLQATqO42Wwyo4qSjebS3f39aVFZAMdojAmLBFVVdTodpVS32/3oRz+6vLICx9U5idihFEoIAUQoCDyRJ6GkAnbORVGU1/XR0VF0N20vLS11e1oEtjC9H+7vYfnDOBOjQQSpQ0W0efERX9eL8ibP87quy7omJQGxdt6yIXuPtmWr+oRXFACgdzTXFTjBAtBaV1UVLqR5WJvKmqXllWlZlGWJxsdxLGVETOxgMJoAwHA4vHr16je/+c1r164Nh0NAsbyyTFI55wajQaSitY1V8DiajowxhBIFgkCPzOzZO+dNLNRap3fuzOmVXjeJNLAJLoVp1pAk1tbWisqsrK899/zzIlJlUTm+B/+HNsMYc/r06fPnzwMRBMVm7733ImBHzABEJKXQAiUrJmOEVePxWEdRbOpiXB7s7bW7nSRLkqjxg8Vn/vmFu/eAIcfPaxvnnHNCSlKqs7zc6XbrshyPx6PRqCxLY6qqLqbT6XQ6DaPBoDd/XNhsEfHEoAXxMUuFBaw5nU5DW7yQBENEHcVFbRxSFDdIETH1x8O9nb39o/3DvcPrN65fe/Xazt4OehRaNJpLtbPTadHq6DRNrbVaaill//Dozp3dU2urQqIgZDC1qZhZotRSXNi8sNptr60up0nM1jhGHas4jtM0lVJ3l5YPBkNAuvHWLZIayKJjrcWCfemcGw6HzzzzTLfbDf8UM4fQxeuHYQbMkSaGcHzVxkymUxCk40gWMs/z/bu7vV4vWU1/sHPVf37hftLJzBvrnHNFWSmlojgGEjrNemnWWeoaW08mo6qqQrgHnSPnnC1nIsNh9nlPhht85VyQMVqEe6hSsixLtQ6fl2UZVE4bWXb3aOAQbO0Go/727Z03b1x/68atw/7BnVvbjJ4d1NaCR02YNltLjaw/HARDq8lkUuaFEKLVaD766IUqL5QSBOic895qrZdarU6z8djZM1kkoyhCYBBMSqZpHGdpo9UUKpJSrqys7O0f7h/2UUgEYrbBVhYRgzCltfby5cv3KUXOZ3DOOW+td0ENaEZIBu+ZkJR0zgF4JaiqzHBwNDo6XFtZQ/5BSnT8cwt3AgJgD94HuWBEVFqrhWmRc96YwCAQQgiVxVkWan22djgc7u/vHxwcTCaTUJYcJ2+FMWmdT70zYYbFc4+xBSMgpPlbt25tb28LITq95f/mv/3v9g8PXn311ddeu/bCCy/cvHnTGKO1TrK5Dji1QhgZaw8ODpIkGY3H7Hyj0WikWVWURJTGka0rKQARPHshodVITp1e21xdXW42NfvQtgohlRZRFEmpGmlGOnIeltaWr7z2emUsOPYI4fwJ44g0TYOG2RNPPIGIMG/QQ6DL4CRVW2Ocs7PJgmc2zlWmllJWpgz+igLJlEU+HtVFoeL3W9X3OsETCMK3Wz8gIe9vo+aLyABAsr3Ua3W6lx59lxiYPQAAIABJREFUHAACk8RaW1VVKMrrugZvCbwzynNBQu3s7BDRqVOnrl+//vzzzwfphG9961uj0YiZL1269Gv/6n9qt9t39/aMsRcvXmo2W9/4xjdeeuklYyyRCA2AMTZcOaFqOjo6iiKVNNJgHqazhJmL6SSNo7osnXNBo/30xqler9eMImWdJpUkSRJpY6vaVnEcL3W7Ko7iRrMGmkwmV65cieN4v9/vdVdqorIsg7671rrf7y/Nu3kQQgoB3ltjiKiYTMKf761nDHNosN4674HQsQcAKWUUq6LEMp8e7u1ut2+ff7SLeJ/o7HuJvv+zDPfv6DSYu8GcuK1vbByHY2YilcwkBXhnptOyLLe2tkLFP5lMnn/++bW1NWttIKn3er1nnnnmscc+IKU0xmipWq1WmRcrKysba+uH/SNT1R44wPNKSA8cpMK6S21FImyyeuuc8wSA6PNRHsd6dXl5dXV1bXW12WxGUoC3OtISQSlBEiUJknEcR3ESqSSTOjLGlpUp67oy1jsIuwFSyAWxgoiCeFtwRwQh/LwtCQM0BBHMCRnnht1kwYPUiqxAb8KcmBhMWY37fXaG58ONhUbnexbx8p9lBH8HjS0E2PwBaTg3txW4h8AshMVIeMDaOkYiqY4ODl557er+4dHq+sbRYFjWptmMu8srH376x9udTpnnw36fgNM4FgjtZvP0xvpkMkbnamuZvRRCSwlE7JwxJpE6/HpjjLX17EQS2GpmvV73zKnT6+vrrWaTiJwz4HySRFKgVkopoUAiQbPZTJsNGaUgZGHdcDgsy7qsK0aqrUMPpGaLLyHEW61WkiRVVYUvhrkBLFT2UQAyIwK64CDlERhRKDUj7jNoIa0gW5aDo4M8z1U8032byRTPtwveD/cfkmvDvc03jrG4FucyA/raILvBYDAcDouiCDOawWDQ6XR6vd7+/j4ARFG0urr66KOPOu/Hw1E1zbVURGRrk0Xx6Y1To8Hw8PAwz/PCWPaWkSSSByTAYjoFz+y895aIolg1kjhJ4osXLjSbWbfdSdKI2KJHraRKVaQjqShSWggKeGij2UyzDFVk3UzCoKwra7xW8cxSgSh42YbKe2lpKWDwAT8NTLUALnnvJQlmEWwVPDpABBSMBgCCMgQAaCU5UuzsZDQeD0dNEYdwX/CUAuT1frj/kAxiaQZZwj0jPgQBAIJOvEneWuM9eiAgCSSN86NJvn/YlzouKjPJS8fY6S4/8aEfe+SxDwgdl/kk1jKKorquBXIzS5baTW/rtyI1HA6Pjo6KomBnHDvvPRuLDBJJSKFUnCRJq9VY7nazLD1/9lwUKy2FZ+tqg4CpVkkjQylkpCOliVBJmSRRI2vpOGIQpjIO4GgwNLUFgCRJvHPG1yGUlVJB9qPT6SBigJJmaMzcvgoRieSsJkHyPGcfEVrjwgMEoFRasPe1qb3t9/sqbQXK0IKA/Z7JC78f7u/yluf5Qld1wT7wzhztHxBCWZbhi4PBYGdnZ3d3t91uHx4eBneAzc3Np59+utPp2LwY9gexVpHUZT7VJJ1WkYrXV1faWXMw6h/uH/WHR3VpjKu9ZedcorQWUmuZxHGWJe12u7vUybJEAEaRirUCZBc7AZgkiYpjjrVOk1grItJSZFmWRikRGQdgLCJOJhNmRhRpnE3z8bw/NiHdImKj0TDGhNQeaBSL8RnPrCUkETiwwILBMyF4mFuBU8gJkrQDAIf7+/tZZ7nX68ExPfv3W9XvW6Z+Fz+C8GDtLpQUQoTs7tlXps7LwtRlVVXsnbU2WGeOx+OyLJVS7Xb77t27SZJkWfbII498+MMfBoCDw73peJSkEXpvTdVpNXRFtrJJrDuNZruTrXZ7o+moyqvKVuCAmSPSSkgdSS2VUiKO40aaRJFyxupIxkoKSYgoCbXWFMc+1lErS6IIESMtm2mmZeScA8tUGxIiL8tQs0kpnb1n0xlEy4gojuOqquI4PjEnDjImhLOOFvkeEZsBQJDUirzzAGgYpBQanIX9/f3e+mawAwmUux9+INIDE+D34f4+uBDgJDnaw0K05122nu9yEnss1Od8AQbwCIhh5WI4HI0Gw6qY9trN6WRUjKY5ws7tO2/evLG9vZ3nuRBiOBx2Op2lpaWzZ8+dPncePFvjUcis0aqqSirVW+2lRXpwcBD7OB9PCCHL4jiStmVDXSGRCEgJOTtVIOx/2LrmbrcL6JEBkJFIaK2TRMTaRCpKsiiJBGKkdJRlAsnXVtCM3cnOIXjBntA5W9JcjyRMf8OYrK7rOI6DtvtM9IFm2T0odAc6O957cVCRYBl5DbVxHKAbAQScj47qYmTqXFAcOlUSyi90au57w7/3isLvFO4hLR3vIYL71xybnqscMwOyrSpSRKSOqx+zt954Ecn7Hg/gnSFSztZCaPYWUQ76h51OD7wHoKD1ZY0zVYFCxGkUVFDm+i2zcHfgTG0jnRx/RRYzO3wINTg8AL894rBny+gRGIAceAIQJL3jqqoESqkkMLjSH+0cDQZDidSO21BWVNWuLPqDQT4djwZHN27caDQa07ww1jHDxYuXfuoTPwWebl+/3uq0h/lkZGpT16C1cw6l6vS6LeuGcZ+dcca62jhbe6+AHTCRFAu9byGUEALDsl+spdQ0W+OWWuskSUSk41bDA4cmNYliIQR4T8oW5URFcV0fdFpNU4ybseR6rMhZ58d5sbraBCHrum6324LQmtrUFUdSoAYAZC8Y2DN4b9B7wR48O8feCeBIKi2pmOZKxSi1FlFVTMuyBGQd4YWz6mj35ptkPvDBD0VxwzHXzkoh3TzWab6fMxN0wO9lxL9TuC/mxtPpNIyUAbyrjTf3Vn6ISCkFKGSkAIBtXdd1aLSllCSl0EFq3eejMTNnjQYQEIm6mOokAW+L6TRtNjudJbb14WFfoEySLM5SKYVUjRB1YRwdloY8u/kAHyN9UqNnEcfOu/lg/+102fFkxAdp0hndb/4tQsrrItIawSNyWVQIpKSMtBZCg4PB0VH/aFjlpfAAyFVRGjPNi2k1neTT8WQ0nk6ntqoBYDQaRVGkZHT5wz9+9uxZsDaJ0p3tXRbEJDwhC4lEIIXwghCyLGNnnKmdls4IYCZmj+D8bN4khCAZ2IgSEVWcSimF1uGVV1qLOFWRllozohAiiqIoihDRhX5XaUArpVRKKEl1ZYmtQo9KGc+hzAjs/LAEkyWR9wl4h4jAjgPVy3vUApgAHABhMMACACZFQpAgBGZ0zkkPzjkGx95IZFdP88mwFac4l4fHH2wxMxsrAASyaxDEQSSdJG+r+lnleXi5o/ursf27dxuNRtJopM3mLLCMGY1GoTefpW2A/sHBV7/61em0YMZ2a+nUqVOXLl1qNFOdJOA5TTPAWUVBSCSJwYWgdGwB6MGwFnRPj2OR1I/N8JjvF8Y4If9CRIQEALGOAaAsc+dcI2sGws10kmth9/cP93f3jHGNJM2amTWmKifD/mFZTYbD8dHR0eHh/mBwVJsSkYsyT9J4eaX70Wc+0uh2j+7e1VqOJ8O43UCcMcmklN57Zs+eREbsjDPKW+ut8d5jEOoI7Z2QC25CCPfoWLgLOROGl1qJSOI81iOlvPcG65DMwukdx3HQJglJCoUyVW2MCT5w0+k0QKKNNA7CeovTPrxcROQ9hGFTcA0EcPfYE4REhOyZua5rz6HmgbIs9/f3hc6yVlcJnJRl+p64tz403APpecYqUQoA3nrrrYODvVOra0KgUiqKIiIqiqLf749GozNnziilIud0FAFRGDU751bW12fXT1WFV4GE6HS7gMjWRmkKADeuX3/ttdeee+45reOyrBtZa2dnp6qL9fX1M2dOj8fj5bWVo8NDKUUorhjcbNXUWiE0AwiUKIBAoACBktELlEAcvrK4Z2QEZPAI5NiGdQrw6MHFOvHg2IEHhyzmEqo8Gg477XYaZwCQF7l3LtZRliXXXn2tqgyzSxIVRcpznefj8ehoMupXZT4cjvv9fhD6WsAacRw/9dRTFy9eBGPG43GioyzLLDMyIKIgACmBHaNm8K4yINAhsTBeSAAfaOYzAuY83FEIIgmEWsWkpNaRlJKECj7jUisWJCMZJqOSyBjjrRNCsJThZM6yLIqihX5g6EHrug7QZPgrRqNRu5mF1d6F9s49v08GZvYOjksCIgpEJBQoUKloRhF1TihJzEVR7O7uyjhNmy3ESODDzW3uVfXfz3BPkiQEfZgIvPrqq5/73Of+6Z++5aoakReu7UVRBIL40tKSUqrVam1sbDz66KMf+tCHHnvssbW1tWIyCe3OQuICiACgmEwGg8HG5ubg8PCzn/1sXdcHBwfew9JST+tyMDw6ODiQUi4vd69du1aWeZ7nzgd9mJn2eVUVRW0aWQsICYgkCZz9BwR1WZOkSEVREiVRomMdqQgRvDOMs7AnRBICGRkJkQUgSPQz7Q5mb4EwS9OqqoApjuMsaQDA3Z07N9/ckkJkWdputQFgMhyFmLBV6fK8LKbj8XA8HoZNkQBcFEWxvLz86U//7PJybzweMLi9vb1WpzmqxuwtgiciJpJSAhIgeyB25Ek4R6iD885sowpoVrujUCKEO6JSWmilVRwgIyml1BEpYgCloyhKtNbE4GwIREEkEYVSUaPRSpMGMFnjAWcCZnVdW2uTJBmPx4PBYH9/v9tpNRqN0MLCPTEcqlyo4Y8toTMxW1qsumNww5VEEqxVUnt2eVnlg0GcHvRW1hoNlUb6OELxA0NmFlPDl19++ctf/vKVK//4+KVHnLvHbl2oRRdFsVDk0lo3Go1msxnH8fLy8tmzZy9fvvzkk0+ura2FXR6hddJohN7gS1/60h/+4R+ePXtWKXV4eKh1vLq63m63janyfLKxsfLii/+0f3D3iSeesNZWVeGcC+mLBKaRNlXhgSQRGJIUwp48gDMGDLmqLsuikBMUQgnBiGkcA5ESgqSMlAqpEQGBfVi0IyQQAMzA3jmvVey9z6fTUVUR0WQ4ePXVV1+7+urF8xeYl6Qk6+r+YL8/OPLeR5EYDsuiKAJbeC5Pyd77PM8vXbr08Y9/XCXJ4LBPRMPhME4j55xnz8wEHggQBBIiA8bSe83OsIvQM9I9ebN50JOUGgUJUkwohJRKRVEcMgtJIaVGIRgh1KJEApxblGpMFOa77Xa71WoJIUKhYr0PbvRlWTaWOlLKwWCwtbW1vrrcaDTqdktKSTiTIEZE9sfUFVAIwZ4BWREsLJ0DPXi+GCAEA6CQtiqHw8Hh3n4UJUolDyXz8XsS7mGmEIZnxpibN2/u7+83m80sy7y3oZQIeSuKIq11XdcLezBjTFmW/X6/rmshRKPReO655zY2NlZWVs6dO3f58uUPfvCD3W633+9Pp9Ovfe1rnU4HAHZ3d1ut1tU3rk/LAj3v7u7cvn271Wo8+9d/3em0V5dXokglQVnc1sBOkY7jeDieEgEBA3hkQADwHE57xoDSlabwgWLFCK1GEwVJElIrJaTUSpJAQQRIUiipQIo5AMoEsHPr9sbm2Uaj/fw3vv47v/M7r1977YknnvjYx565c+dWUY7zYiSlrOocyROGfQhbVWWeT6u6dN4AeGZnbJUk0U//9L9YXV31dV2WRZ5P0zSeTideOgCH7EPzR4QEAhko2I0gsXQwb0+PK+EEZIaEIBJMKEhJraIokVrN1F6FBgFAqFQkhF6gHAhCCvToiWQcp+32UqvViaKkqiopdOkqHUV5nhdFAUsdIcTR0dEbb7zxxOOPtlqtPG8LIQTNnOwBAZEQIPA3AdF7AnKBLeeCLmeQoychpfLeV75CwDiOvYO6LHfvbkdar6yuCx0/IAlP72mrGgQKAWAwGNy5cyfP87W1tbqupaRQyVRVFfYYwoQ5fBKquiRJWq0WEU2n00DM2Nvbm06nSqlLly49/vjjm5ubW1tbg8Hgy1/+8kc/+tH9/f3r16///M//fNJoZln21taNGzfeHA77G6dW//Iv/yKO48uXnzpz+vSZM2eSSE9rM51OwwQECD2GpMiEEokRBKBnj+ErJEAIpQSE7xZFAegRBM2/En620+4qLTBKpZyryhCh443TmyYvvvGNb/zWb332Dz//eQDY29t1rn7yxz44HqOUmDUSIZDID4ejcpprFtbaUAwsXOettWHd03s/HA7zPB+NRmu91cP+IQhmH4AORoQQ6LOZw7x4DRspiAKRQ04VQqAgKRQIEkICItHMXDZskAARCkFS8Ny95zjbZ3HBKDUzrA2UASkl1FXomMNKLgBMJpOdnZ1wAdR1bYxhMV9IJyQpQ+0+O+mFBy+ZHYTyBoP2MgrBocTKCxaCtBbe++mkGA36g6zRaDQaSgPSsenK96W4Eb/yK7/y9teBlHVdh+ZSKfXCCy+8+OKL0+lES+WPGTcvVhzmTboP10k4EPf394OiZ1VVzWbTORcglz/7sz+7cuXKnTt3XnjhBa11OA3Onj1blmVVV1IKEsQAkdbXrl31zpdlzs791//VLzSydHfnbiNrIDIAx3HEyISAAMj+3n2YBgSPXmectbauTF3XVWGNQfZKSiVIEDlTV2VZ5JPpZDIeDkbDYT4auroW4IWUIARXta3N//Lr//O/+93fXWplj1w6d/vWW6+9+vIHP/D4ubNnBIEgkJE66h969qc21k1eDwZ9z240GTGiTqK7d++OxqPPfOYz//JffmZlZeWtt24aU5HAw4NDHSvnagCHgCKsAwUVKJod4iSFUFIrrXSkdKR1JKSSxz6ElIKUkCpJMqUjrWIVxTqKoyhWSpOgKI4RKJQc4XVy1lnrnHFSKIHCOV8U5dFRfzyeeO+N9yRlGAmbuup0OkkS37hxYzQcbGxstFvNqqoirRCxLEulI+NAx0kcJ0QokJSSUkgAlGJWuwRUzFnvrHfeSi2MrW1twzVdV3VZ5JPJeGNlGaUADmpjoRsmaywcY9R893SDd6rdQ5NaVZWU8pd+6Ze63e7f/u3XXrnyYhBHD4OMkOOrqgreQJ1OJ8uysiwHg4EQotfrBcfnME6v6/rRRx9dX18/c+bM66+/3mg0FvMsa22r1UJEmO2OOkQQktgSgCcALZW3xltN4JGd8ACM6BmYmRg8MDF6YAHhnh0zMXp/4j7WMSlSglAgOA/I4B2DP9zfQ4kSpYpVoodJI0mjVKmo1ez2j45ajWRzfTWK1XQ6ds60Wu0Xvvn86c31xx9/lIgYfK/X8+wUSqR7ipBB6zlc/+fPn8+yzBpT17WzdgacA+OMdDbTKlswSPCYsQKQACJB87yOIIQIeZ3myAxJIYSYRxmhIJhN9WcnRZj5L4JGaz0ej9n5dru9vLwcRZFzDpCIZFjaMMY4rQAgoDdvvfXW1tbW2c3TQoiiKJRSSqmyLKNGJ3xORCA8InMQEDeWKODwgmEhSEjAcFylkJDZW1tX08k4A0Cdmqo21mkdyyiRSj3Mqu37RSIIRfmFCxd++Zd/+fz5s//7//obu7s729vbVVVlWRbU2wDg0qVLBwcH+/v7Yd1rgcPM5YegLEtjzMbGRsDmAg0jvFIB4QqnMAEbZ601c3Z1FdrlNI2ttbauCRA8I3uB4KxhBGYPHpk9z3pMAcDOekaPTCfuScfI3jvrnfOWy7oIBKwinzJ6AkE55jJSY6llJKVcXam2t7c3T63+l//Ff353d/vq61fjSEZaPf/8f3j66Q//+I9fllJOikm32/Xej/sDIkJk54z3FpGqqjC2zrL0A088piM5Gg8m0xExRFGEyNZaIMBZNPBsizbsSciZyX1gHQohEAmJKVBZZqi2IjkLd0EK5UwxmIQkJCQRXH/D/2RWGXC4AFBrbe0AGdrtzubpM2ur62+8fj0viyiKLM90dWZnNWGWZbffuvGtb33r9MZ6mqaRlkmSaK2neRk3EUEEX0BADxT6U7aevZdEYdgnZnQyJrBEJEEYAFCEnhC9s2Wxe+d2b2Wt05NKCWbw3jtrhZLvUasa8noURc1mczqdBpeLn/u5n/uJy0+/+eYbV65c2draGo1GdV2XZVlV1dbWVlVVk8lkNBodrw6XlpbCzCJwXK21L7744s2bN1utVhRF3vsoigL/LkmSaVmERcn5kgsbY5wxyH6p3fG2rgpG9uBdsJ7zpgZJnpkAPDOiZ49hqcDb2h8zL1/c12XuAZDZOOeMKarK1nVlDDvnAZBrX0GNRUB4ELGYTPv9/qlTa8vLS9def817o6WY5NPRwF+/fr2u606nU5paqagsS2NcKOQWiTmoD6yvrm5sbNR1XUxHRVFEUoU/s6zyKFGLB+NsCIMLG81FuC/Ylzi3BkHEgEiiIESUkQ44ulAyQJOAFCzw7vlJzcG0cFNK2doAwOrq6oc+9KFbt269/OorkZQInKYpEYF30+k0jnQQFXvppZdOra9tbm6mSRQUlwDIWu+cU8wBKYV5Ly2lYmZvHZEDmrXaxBRQ6eAly0IoKZzzpi7v3HrLe58kadTpaRkBoLFgylpr+Z6Ge0BP9/f3V1dXA7L+4Q9/+Omnn17QaQJK89xzzzUajfF4/M1vfvPv/u7vbty44ZzTWhPRYDAAgEaj0Wq19vb2dnd3iWh5eTmorQdmXJIkcRwf9I+SJDbOLjqBIH0hpVxeXnbOldax96aqna211sZYAMHgQr3nGVCQ9wiEztShUDhxX+YuoDTGWVubIDVqvQPPQXzUsWfnw+IcM+eTsbFVq9WK42il1/3gBx6LE/3atavdnnjj9at37+6dOXNO6xiAauMAhRAiitR8BoR1XUpJFy+dz7J0PB4V00KImTYBEdTGgZdIHmfhSfMZDgaHPSQRihkSAmcupoyIJJSYyfcJkirELhEJqYgEkJwZg93jS8ypE8jAhCBM7ZI4K3ye52WWNZ9++iPb23ffun1nXBVMM2Gcusgnkwn7pNlsnj59+uq1G1euXLl8+fJSpzWb2qrEWuv9nHiBGFr84G7A7J2wzklGHyTZmJnIILFjz9aBIFLCMFhrq3p6dLCvomQdKG52mCR/HzQL3incm/OxfxzHoc6eTCaNZvP40Ivn+/af+bmfGx4dPfvsszdu3NjZ2RkMBt77IEYVqvY8z8+ePbu1taWUCsVikKcLla6UMk1TAHA8878WEq21VVEyc7PZXF1bBs91VRKgoTIwbJ1zhJ5hpukesAGgmWHFCdZuuHc8Y/y5YGoAuLCsYWYHiM5Zz+C8c86zdUYA23wyrIwhwadPrSFynk8QcXt7e39/f4YJkiKSUZQYGoWxvFLKIodhzRNPPBEkF511SZKoQKZTyjo5J0fc098LKX6+Oj6j285E+RClFIKAxIwLJpQUQgERBTg1WBQ82NHdb5ewyGW1EJPJpNVonjt37qmnnrq+9eY3X/ynoq4WLiYBbXPOLS0tRfrG9vb2yy+/vLa6rLWO47jRjGajJTxOUPVACshL760wRMQkZ7I8UkiWjACeHdVEJLUmMOxtmsRlWW7fuj2Zls2l5aTZzhqdNEvYw8Osar/H4R6qt6qqwoQ1lNfgXOBszdZ4pQxEAFfXv/mbv/nZz342nAPtdlsplSTJ/v7++vp6yPFpmu7s7CwvLx8cHIRmKBxxoZhpt9tLS0tFVRhnrTcKhHPO2CqItnW7XUQ0xkgSVeVnGJfznnCxJBZyJDAhIL/dxh3MPU0ZHBILRBJCo1yQQBYOH0E5w3vBYBB9XVsEaLebdV1P8uaZM2dq627dugUARNIan6RaSlVxHgYRUaSUFuzZe5+m6aOPPhomcQJFFEXeWGtrCVIIQewxzHV51qnicQVvmtfziGEChohBjjQwZ4gkoQzhDoCAYg7nATPyA4YZi0omkEQWoxKt9fnz5z/5yU8eTUa3tu8cHR0FnkggBRRFoSWtrq6Oh4MXX3zx3NnN1dXVuq6n0ynq1FrrrL9Pw4EZCFEIorkkCQoijxwGZODN7ADXUhKwtVLruCiruiqODvfHRZ21i41TOs2S93SqGuZEodEsimJpacnZWggSQolIH2erj0aTN2/e2N3dby21VlfXD/oHR0eDXm9pZWVlc3OTiF599dWNjY3bt2+Ha2B1dTUAvaGKlVK22+1er3d963rI7hxUX7xXSoXxFhCa4NEcFogIHHvBFN7XWQ5D5BD2/PZDCjczjZ8B3GGDM6QxZgYUDOQZBaPnEBZWS0GSGo1Go9U8ODgC3ut0Oq1WK47SLGsSUV0bgSSQqqJEZKWl0lIIkuyIfRTHG6srYE1dFVpoULKsCmdspOMZx3XmL3WfxyUgAyOyB0TkUCh4ALLeIXtGAI/oHCN6IAJGErRwdJgz8vlhBjHoPVtjKwAfRQrA13XZajWefPKDr994vTTFZNQXxHEcM/himuf5xNu62+1WVXXz5q0723etY+vYTAuVVqoqYi2RFQkEImD2ribSwWECkWbvCwIFEW5AIPThqhVKKiLhAEmoCBUCSVuX+WRYFmOEHoNn9Agi3H9/w33BAY7jOHwupGZ2yOgZkMEDIzMje6BXXr165tyFypSvvHbt1OZG2lCNdieNo9u3b4f9gNdffz3Lsu3t7Y2NjTCWCofm2tracDj82te+trKyIpCSNNZaG1u9fv3NVqs1HBydPX9OxdHhcFCb2iqVyAgJ+9OxkjJoGiIelwxH5oUI5H32L4wQKk15zCE1cJrCBB+IhBJAkqQTynnvY2rmeY6I08rlB8Mkbp0+c2F87aqp3ebm2cFRv91sIXtbVwQca+W9LgvvnNk8tb5186ZC/hc/+bFOKxsNh5kSQGBNEWuFUVBFleztTNgMJSHNu1VgZ4VQQhAQEgASawlSSYtIKIUQUikhtdRKUHBuEgDAlu8NqJh44R4wM+P0iIzEiJCmSV3X1lYkANEaW5NwS93Gp372P0ub0tWTmzdvFr5MkgxBj0dT49zewUHaaBWT6d8//48feOKpjVPn+qNh01RlPtKCKctIRKGcIa0AGCTpJDKmRqZIxVjTNB+d2c2+AAAgAElEQVQroTy7SCeyrZ2xzjlWnHZiRLTTqUCRNjL2UJnaFUNTDkFqkgqAS1MSU6QjALDOKvH/s/fmMZKd173Y+Za71V7dXb3OPpyNI1ISRUqmrViLl/jJkGzrAYIsW8gLYBiwIcGIgvxjG4aenpO8JEaUPMDwP7ITGTD98hzIepYcit4kRRIpSlxGJGfI2adnpveluta7fMvJH+fW7VtVPcPRcETKzlyMSj3Nnupb9577fef8zu/8fnJ0mPgO2lJ3V/ZyZHQ1B6/AkUEcq17YL5fL5apRyvR6vVJU9B03NQAjfiljGfi42+saZM9RFNVqNSFEkiRoGam3eZ43NTNNiyD9MdRpBNCAAgQfoGy7cBuAlO5wY4JZS+wNgSn4QdUrEq8PckIayDj9YYQRCY/If9Za0tLwvKBSqa2urlerVaVUMSigsXEYOa7QmnEBQeABgGA4UatMT00yrcnaHdEyYIwawSSeijztqrDhDH4AxjNkQCskQ2AopcO4lFIKKaXjSCk5E4zzweaQzyi4YJDuwOMDjMwCs8R4Apb+FZiZaUycOvlAt9Ni3N68sdxuN30/mJ2b7vcixli/H0ZKbe+0L1259u6fUOVS1RhjtNJaGaMdK7Noa25u1qemmOOUqpXNzc1ev8cYc32fW8XTDUxbC0Yb0o2i3Mz3fc55u9vudDoMsFgpNuYOxlZT2swZ16CZZZnb81s2q7q8vLy1tbW+vk5oer/f73Q6tVqlHBQoIc5m1wlPyOpUAt1IZLTf70vPEULQFDMVx77nTE9P75GFZ02TnA4MGwjwCrmbkZOY2yBV4Fnqm/+BrHVHjyW1ugh4JgwqtVFHpJrE87xisXjgwIF+vx8EgdY6iqLAlygEZ7JYLhljhHDm5/cdPHjYGGTE62EcgFlCvy2dB0/7L+koTxr99J+IRjuArgUDLqTM6lTHcbgQwEQu0PlQVLNbhsWwVffuBGqhUDh+/GQQFH3ffwafvX79OmPM9/1+L/I8r9lsSSl6/c5LL5/5wMoHjh49mlU7aWctbZrZeqMB1sb9vlKqVCrVarUwDDudlud5SieZUxChICRfQ9wT13VpFqLVanXOnfvJxqzWhnPuSAeAGWsRLefiLQ73drtdrVapqKXwJUYkVSQZapn5BeTV0qhg0lr3+/2oFUsp19fXGWPFwOv3+1OTC41GI6+1m28m5x/0PZvM+fXSMuCcU0WYMa5GnLIz6y8K92z/IQFoelx93+92u/Pz88ePHydSUPbpGAia/Irj2HGcqenphYUFZTTn3DJKTLK9ZfQMR44RWlhmqcA4T4fFhAB+bziD2QWJIxUUgqNHjwohpHDPnDmzsrLa73eNVY5T9jyvVqu1Wq2lpaXllZtHHzicjiPt0txZpk0CnHvFou10tre3lVJBENSr1auXLxRLhWq1SjHd7XZJk8fzvG6322q1qE6bnZ1dW1vb3NpevHZtojFdrVYNGmOMFI7gwlhzdxF/z8LdcZyTJ08S/ymKIrpVOzs7E5Uq7VC0ZPb7fcI9RoKVok0plZGraNCGc75//37KcCjnyfxkaE3K5NfGKW4joUN8wyy4R56c7H0In0m7IjleUOayTd9pt9s/9VM/Nbuw8NrZs5Rxua6bxD2GlktBi3FQLBOZeau5LQXprnLggzKaLKQx5X8BJ1AdGGcIwHj6nfTZ4Aw4Y4IjIh+3URiCAu8m1vPXxCSKc35w3/6CV5iemnru+89funQJDEgOvitdyV3Jw15vc23Nox2GA1iNRoHVACItjo2ySiFiUCoE5SIAxJ1Op9M6cuRInET9fp+6Ma7r1ut1wu7iOI6iiPhaFACIeO7cK+985NGJarXVaYdhODExJRzXGg08t6exNz3c3/Oe9/zar/1asVhcXFzsdruQmp2nc6U0DkLRTBBsHMdZ79DkpGVpmaeMiNjz+/fvzwx1jZQZI4360tmtyicklGenEzeDOOacPEWR4dCSllHisnDPHpK8Jl5mYE17d6lUeuc73wmZ7rO1juNEfWDWOo5jbU8ZU6/XaxN1LgUIzh3JMvglLXgoeRoy5RqJvD1X+hx6MyhDrYW7VeEa+i3AK6Vyp9eOoqhQKBw6dKhcLvteMDU19dprF3Z2dnzfbbd3er2+MebylYu9fmdmZoaWdoJuXccBIYClMx1Rp9PpdLa2ti5evHj27NmlpRtRv7fd3FpZWel0Op7n7d+//+jRo7Ozs4888siBAwcOHjwIngcASafTbrddz1laW9ve2igVg1YrlTCZqE3eNVfsnoW7UuratWvPP//81tYWDe8AAKJp1Cc4T4EdEjCh3Kbb7RK1hhZ1egaklBpNkiREMCZI5PDhwwMDMJGt8Vm4I8+mZtjAJwZT8B3AMkBAQGQWGZLgI6Qy5Dlf9sx3CYZ18DIxrYFIkCRybBRFp06dOn369OryMs0702PsOI5KYtcvJHrbWpiamiqXy1oZ1/GEcIYqe/pdAESoIt4UMD74ggHjgz8s+2vee5Vxvts8+uFvf34/pAd715QBBEMOljlCTk813v7wQ/Nzs/Ozc6+99loURRcuXJBcIOLVy5duXl/cvzCvlNUqSRgkjnQEZ44DADqKwjC8fv36N77xjS9/+cvf//5zQvCFhYVmc6taq8zMzBw/frxcLnPO19fXl5aWLly4cPTo0QcffPCBBx44fPiwWy4fOHBAGb3Tbm+urxoVcyYRcYuDw3ilPvkWh3uz2VxcXLx48aLjONPT0xTurutubW1RuBMNw1rrui4lNoMdADP+oBBCWJGJnITGBEFw6NAhos6PmOPlV/fxfCZbtkeCmEuRmaeOJ/fjOTRl7UTfyxAkpdTx48fn5ubOnDnTaDQYY81mk05bK+a6jrXWAhbLJem5yhrpOlkpyagBNFjqzXDLcNwtPv8dy7jkMr0CWdGST5ffQCZDr51uh6acaJvlnE9OTtZqtX379lVrFcGlH3gb65tRHC7dXN7cXBeCKQXUdqQF3uEcGGs2m1/84hf/7M/+bG1t7eTJk5/85K/v27fP990Hjh31PK9UKhWLRVrUaHPodrtbW1srKyvUgjz99rc7jiO5aEzWt3e2+u32xFTD9/12c6fg+ZV6fVeh4y0J92q1+tRTT01MTHS7XeIPSSnL5WJ3p0Usn8zfi7pxlABk9U3Wr+71e5SrbGxsGBU//vjjjz32WLPZJB6yKBSstYAwuOOcwpcNW7MPRTP9p8Hv0omhojNLkCh6snojy4WyapVmXCgI6ItisfjBD35weXm5Wq2GYUiJZhiG3W7HEazbDwvFku120UKtWudcRlFYrdQTrQB4lCTUqHZdt9vv+UFh9JFjDAGE6yGiRuCUCnoe8bQQUh9MaugCAKYKdVJFESJzgwAA+p2OIz0ncNFaAiLz+qOEOKWT8kRDHWxf5WKJuKvILQ1JKqWM0labh992OoqSaqV4+fLVGzcWjx194PriNc91Wzs75XK52WzOzU47vtvd2fmLv/iLP/iDPzhx4sS/+oWff+ihh44cOdJoNIhDb9Bk+wmBXXRfOOcPP/zw0aNHz5w5893vfrfT6Zw+ffrEiWMXLp6bbUytLK3eWLxy4sSpdrd94Xyr0Zjx/AICI5q0tkYrK4Qj5es8Afcs3F955RUAmJqaklISH3pubu7YsaMvfO/7UvKsdqRlW2u9s7OT5ccjgCCl+4VCIYmg0WhEUVQqlbqtHcpWOee091OpuucqlQXEyEqZPQy0Wmd9rpSmN9AaH4cmM0oFVaXT09NBEFDlTXKQ9H3OIQo7YIBzzh0pXIcmjISWSyvLMzMzTDhFx5mdnetHYbvdLpVKYS/Ku4Bk1jdUtKW0eM6zsPB8jwjAMMDaGWcMIOx2g0IBmEjC0HGcQrkMCN1Wq1guj1sG7H4H975E+SWDvl0sBlHErLWVaml+fpZx0+v1OJMXLpzfv/9AGIb79u0TQjzx53/+J3/yJ4j4iU984siRIydPnqRrRQw2znkYh3tm3q7rbm5uuq574sSJGzdunD9/PkmSRx995Gc/+IGvf/3rKunPzi9srK2GcVKvTbz0gxff8c53IRNSMk7ShZIEmUHwNyXcaU5vdXWVlsx+v3/x4sV+v+s4jpScFss999D8Vc7/mBCiUCicPn1aKVUul/uddsp4GShV0OqOaYKQqiDtsn0gBy+S/jg5+g40ofLT5VkE5AGZ7GZTuBM4Y4wpl8tE58weg0xxRQgBwC0w5MLzAs8LHMeTwnUcOznp+4VSs7ljjCkUK0ZjEuti2avWCvn+rrGojQEAPyhmn0VISdQGbdCGYa5ypeFoAIAgnZVJZwxI7cd13V3+VlbZDAOvI+EOBqnHxiwwixxBMHSEcBwHjTKOEKWiMy98l29ubna7/ZvXrx09dDg05ubi4pNPPvnkk09ev3r1E5/4xM/93M/Nz8/PzMyQ6mqSJCZVGWTjAUCXt9PpICIh9Ovr64uLiwXfe/Txx2anG2h0v9tRygSFEmNsbW3NGAOcGWMoFIRgBsAoI24LUN5LMjFJxU5NTVUqFcLXgyBApaXkWeqSYbQjUGAW69Za3/eNMWEYzs00Hn30UYJo0jQDkASGaAtmglscTbiHGN55pkBqsLGb0GcADuSUPvNJfz4gaIklKluj0Wg2m7Qb+L5PRLowDMOwR58iBaN8j3QyhBCOF3Q6HSFEsVxCxFKpVKyUEVGAGMmd6KBR0Yw+lMUHYbhZZQEADDgACOFYa0ulyszMTBAEKooAuRsEdtjxOPvgab2eLjSjOFX+ptDmY4wiDgLn3POccqWUqJg27WvXrrVarS9/+ctf/OJfvPe9P/H5z3/+0Ucf7Xa7NJ5GTj50B5G0Nfke5Rbd9yRJlpeXpZSnTp3a3Nw8c+ZMErYfe+xdniOfefZ7B/YfbExNdHvxwuyMIyQT0gIaYySX9KBbxt+k3F0I8Ru/8RuXL1++fPnyiy++yDkvlUqMMdfzaHWn5IFe86B4Rr6lW46IxWKx0+n0+/39+/efPn06jvqdTidHjmUZCWCwkGfksLQhiQA8m7Ggy2otWMsQ0wn/wTOQx9ppIR93ksiDla7rlsvlcrlMQ4n0kxmD0hiD1hqDyMBxHEd6gByRAXCtTRwnBw4eDsPwb//2b9c2N3y/cOXKlbXVjWwUJnulKMnCnS4LHYJhrsgWbDCsFARFxtiBA4fe9773/eRP/mSpWgWN3VYrKBaBDQDW4RSFc26tGS+Ls2c+XZg4ElWO4j5RcaJiABsEHt3N69evLy4uPvXUU0eOHPjYxz72kY985NVXX63X60II8hgk9jwAJEZbazG3HmW/NEkSMk2gHG96eppz3txeP3Pmhbc9eHx2emqmMWWMaTabXHhTU1PCcUBIjhAnWilDTlVCsDcp3B9++OGHH374ueeee+KJJ86ePdtoNBzHaTa3Asd1Xck5p92fqKdUtmaxnj3f9DUh9KVS6dFHH61UKi2royhy+CiyTrPGbFcFD7Ok6Faq4RYgM3rPtpoRbI7iPv8O1PWgaZVarVYoFJIkoT2HHMgo1qmWjcM0mIjyzgSnT1X0fQvY6/X+8R//8b//9//jzZsb8/ON1dWNfQsL4+2wTLltvFXsynyuxdMWFGNaW0R86aVXzp49u7Oz87Ff/VUQbAC5plcmX9BnaOxIYzWPfdGTxgEYQ2u1lNJYjaHROiEqJed8c6M1Mz1XKBROnTr167/+6w8++ODKyoq1tlwuIyI9tNmNtkpn6oYjyQz9MOe8UqlorSkrPnny5OLFV595+tunH3zo3Y8+8r3nzixfWj5w6IFmszk315fFCjAGwI1Jw52zNwuZWV9f/9a3vvXHf/zHV65cWV1d5Zz3+/2dne3Jaq1Q8H3fz64mrYUjeF8WgllZdvTo0fe97329Xo94wmh0TqcqzUUBkfFdGC4fu/kdOZ/MCCFvyb3ZC9AkXgeBjxTuQgjqCI4wEdIwkoJpLtF1Hd/1vSxhoxGWb3zjG3/6f/4fy8sbtVqxPjExOzenYp1PZjL/YeImjJMmJIecAz1LhekYq1briLi+vnnjxg1aKX/+536hUC5rpXY/YG6x4Ln3zC+0QjhgGTKb1VIpQ412TgSwyIE5UqK1VhsOzHXEvoW5X/vEx9/1yDuMMc3tzUMHD2oVc1J9B6u1BjQpLw2GuBsjyhcpP6ffb7VarutOTVZnZ6dvXL9y8dL5Y8dPzs/PJ0p3u121dKPRmJkSruMHQgirbY779qaE+/T09CuvvPL1r3/98OHDhw4d6vV6nU7nyJEjy0tLTDLHcaTnOtJhUjDGNFqTKMtSGZW0H0QePYwZpRjikUMHHnrooebWNoLxPC/qqd2oSulU6YolMsglt1nn8266cykWyQUOH/lcFm06AJKSFBlDTEVAiZRfLBaVUpSF0wCE4zg0pNvv95MkCgqeUIIBUAtWCMemokKuAHj66e8+8+0XpqfrBw8cWllfE6Ifx1H+MctvX/lwzB5X33Wt1dbu4kUU7ktLK6VSyXVdzuGFF174whe+YK395V/+ZYaGgWVACgkAOFSrpBeHUVY9oB9zBE5CT2isZYLTnIBFo4w2aJngDEAb0wv71Xrt8tUrhULh0Xc/tr6+Pjk56QX+zeUl8mQlPRw0QNmdy71uv49A+1JaOzDGEI3nuAxtFCVJFDLg1XIpDOObi9cLHl+YP3jl2mK3q97/Mz9Tq089/d3vGcu2t7eL5brjB2mqTE0Mg3DbfOZuwn2kHKQb0+l0FhcXDx06RASvUqn09re/fWdnGxhzPJdJkWhljPG4bwGV0a7n9qNQKRUUC0wKq9Ay8IsFh7FuuzNZr330l3+lXCxtrCzn+0HSdahZi4jGWNRI4g75iifNcXNNdQbAOQMuAARYQ8bmNArK+YDDpy0J9Ivd7r0AsJ4fbG9ulEqlYuBLzna2t1qtVqVUDMPQc9KBwy6DhMSwU6ytvzC/PwzjOI4bjZnVlbVisdyYaPw3/+1/93df+4cHDh9ut7uXLiwWi8VWs21QU/03AkTuSR0bUD4tYe354/Dhg+1223WdWq2ysrL2d3/3NSl5EDg/+zMfJL0g2jRKBd9zRHunJ4tFzrlwXGOMpX0VgHNmObMMDKAFS5PfltTABY+TKEqUZdwyrpRCLgrlShwaLyhONqa1hYmp6Y3NTSklWWrGyoAyCFw4HgAk2gJoLkn0ChBQIDNI2m/W6iRl3ViNBoxRAtFz3ESZBMAvTly6dnPjr/7m0fe8+7F3P/7CC2c2N7ddv+B5nlcsuY4EsAwRrAbhvRmruzFmbW3txo0b8/Pzma/a1NRUsVgMox4pxE9OThaLxe3t7e3t7dOnT2dVYBzHEEKskjiMJqvVrfXNn//Exx55xzt0khArOJuaHUozGKSVfg6EYTjQuLZ4508vInImUwybMc5lFu7U3qIqijTS6EGilhMJDdD3acCeOAUbGxsnTpwSwllbW2s0ZqrV+le/8v+88oOX15c3S6WSUsb3CpVClTPZmK5irg00klONJLgjBWX+r8RDJvGfSqWys7OzuHj18//b/zpVqz36Uz9hwnB1dXNh//7m1tbS0tKxY8d6vV5W5xDWAIxZGAz4coJ8Bqw1JoAb4AJ4uk8YC0pbpY1B9H3PLwTIwBjDpeBSGJq3yhKMbCUfDJ8wjoCMPjgHmoM3xJpDxpEZBDBWG4OdbuS6bqFULVXi1Y2N733/uQceOH7oyOFms9XpdDqdDmPMcT0mJQDIQcHzZgCRlBJMTEwgYrPZbDabQeD1ej3HFZVKhRCMdrttrZ2cnFxbW8u81amfVygUGMLm5malUv6lj/zK/v37r1275vuFJEmIknCLX5oOZJNUC+382TTTXsGNYx17luEbA355xjLncaQoLdnZ2dFaT05OUkOKRstJEIpQNsdxgDNErNcmO51eEAQkSlGv169du/7EE09cuHCBACshnHK5XK3UvZ7reQ6yDH7cBWHyCOltOA55FQ2i3bfbbVpfe73epUuX/uiP/ujf1f7tsVOnELHVbDqOU6vVSNZvpD4eocflkOJRkjDtovQmgJLkb6l7SODVLnA82P9HcoGhtWaYGCeEsAPMgK5tq9stFAq+7yfKrK6uHzhw5PjJU7Vqe3K6UavVpeum48dcvC4z9J6F+4ULF7rdrlJqe3ubKDFCiGKx2Ov1qGEEACSNSx3TiYkJimOay6YLEcdxv9//tY//6iOPPLK5sb22trZv3756vb60tETeIUD67EQ2GazxkMNkRpLyvQDTfJtjd/rpVqwTorgqZba3t4OgSHNS/X43CIqIyhgqr6XrcsYEqCjRampqqlqtb21tCeEcOnh4c3PzC1/4wrPPPhvH8cxMY3Z2jnPpui4DoZSKoq4Fk8V6lqOPFNDjK0uuYEUAKBQKJEjdarUQcWJiolAo1Ov1f/iHf3Bc8ZnPfOadjzyyubFhrZ2ZmVlZWcnpqo5aPWZhPWhm7W4C430S3wvK5XKpVCLkjUbVqEGxV6zvjuNkzLaR1nhGB6L39zyvtbUVRZHjuvPz8/V6fX7fQhAEk40Zr1jcpc0Min3HfVPaTHNzcx/84AeVUmtra81mkzhG1tpHHnlkZXVpa2uL/FIAoNfrdbvdYrFIGtnUhaXR7ziOH330sX/zb/5rAFhZWfE8b2dnh/T2x9u4OUBy18bKIlKSc6tw50zk86Fs7M3SNCfL844YAAsKJYsqjOJWp4sgLDAEpgzt4wCMC8cVXCJYi0qidTzXGGw0GktLS/V6sTo19ZWvfPEv//Ive92wUCgGQaCU0jrudrtxHG/tbAWBZ8HAGOcnP7YyUraO/DAdW1tbRDKlySCajxGST01N/ae/+s+I+Id/+IcHDx7c2dkhAhIlnNlQIow4gA/ocZyn3m8jjI/sacnIXtQOz0ijI4nZ0EAPY0OM1NfzVXUcx1gQwmk0po6fPEEblxcUwRgQEqw1CEIIUut/k5KZ/fv3/9Zv/db73//+8+fPX79+nXDoTqdVLBaBWRprItVwz/MmJiYy0hVpSrZarTiOa7XaJz/5yYcffvjixQvW2pnpufMXXu12u41Go9NppSQ4TCWoxi9lPhRelxKddasGXhT5kpAPZufA89wkQc6t4BKRqURbCwx4FCUAluTgGBNokXMpha1OVFdWlowx9fpkvTZx8dxrX/7y32xttWdnGp7nxUnY6XSIf2CtNWDCMMxydza22uEw1jSOkWf/ql6vk4A46bUTmYeIirVq8OSTTxpjfvd3f/ehd72ru73dbreznCT/q/PdZdj1bmd7DhK4rssYn5yczOb3szo7k0zM97bzb5V+kxoCww9G3jleCBH1o3Kl4vsFRCxVyrVazff9tY2NqWmBgAxRW4PIxJ0ZVt6zcO/1eo1GY2Zm5v3vfz9BAb1eb3Nz/dvf/vaJk8eI/Xv58uXnn3++2WxWKpUDBw5sbW0RTfzq1autVmt+fv7x9/zEe9/73mazqZSWUlhrA79InflRi8l0TpsWcio3AYAUZ+gK7t1PtphL2bmgPgVjTEoyISNfOzGIANQWheNW/OLEVMNaCOMEkUnXixNNpB1AtFohGsaEdD1r7eRkI45VtVILw/CJJ5549tlnp6aq1KsKw1AlmQEGlkvlfr87npGPL+T5v2Y/QG07+muhUIiiqNfrEbWTbCO4YBvLq/MLs2tra3/9138bBMHv/d7vHT1+vNLvU3k94lGTT9NH+oC0mmS5Fk3DSOlMTEzQyAvR7Ogt6MOOsDkGegh7JDN5Tgd1punhkRKk5zmuXyiUpOtUKhXf9w0yra1XLKWEbsmNxnSLtpa/ORSxIAiy/Zc2Ss/zSqXC/v37e/1OEATFQnF5ZXliYuKpp566cePG8vLy5uYmZXtra2vW2ne/+92f/vSngyC4cvkKSQevra0vLCwsLS21Wi0pJQAHwMGrudXIwm0S99vgqnT/KNxTJ7K0uR37gVupVCYmpjqdThwrxlgQVMKwBSAGemSaMea6XEqv3eo8+OCDi4uLiHjz5vKTTz65sdE6enT/+tqmlFIpTaPH1Fp2HFGfqMIY7JhHVEeGx/OJR5a+A8D169dLpRIpcpKvGABYNPv377967fL8/Pz0tPjSl74kpfzsZz87v7CwvLSU/fOMpLV7DsButdtkuZbjOIhElDK0mVP6mvGLxu/Crfzg85/aWgu7lDuo1Yqtbq8fhVOl4uTkZKlSUcogAxVF0nMZY0pryCb5+I+SMzN+OcZZnQAQBB4CGmvm5uY+/OEPCyGefPJJklDtdDqzs7Mf/ehHP/CBDzz00EO1Wu3GleuBX9TKVqtVxliv16MKbNAmAUj1ctM8LUziPK2AXHSBgbK4N3438D7PgjudxjLU6wbBOWOS0b7MkVldKFWk688t7K+FId3FdrePTAjHE0KAUsgEIUue58zNzfR64fT0bLfb/cu//L9WV9ff9a6HXnvttUJQ8n2f6pDdnUdway3Vp6mywEAjn4a7qctIzAvHcUqlUr/fH+nC0inRCAEhBKRwaIwxVl+/fr1cLkdRRDP/X/zz/9hqtf7Df/gP9Xq91WqRDwVl3lEUIRoOzFqNYPOZEhkXZBgUnY/v+9VqjbrO9IwhYhAEQgiSFRiZock4GsS/R0Q01hhDhXo+Q8tuqBAsVqZUqpSKFYPgF0rCC7a3V2vVuuP6JFsipctycNubxHe/1UH7S9ox5WLfwr4Pf/jDp06deu6554ggsW/fvoWFBc/zWq3WpUuXFqbnwVillEoMInIBQshUl4ZlNso233zec13PC3uM5Ka5cN/9vhDkiS4452IwMcQYm5go9Pu91dW1OI4qlerU1KTjuHEcbWxsMsaEkP1+//r1G1EUnjhx8u3veDsIRqf3zW9+8zvf+c7GxnYURVLKIAjoVLXWAIxCR6MVQuhUz55l8+lKqYmJCZKmoDFfmu4NwzBz76DR9QxaKRaLSZJQc4C2jjiOlU4OzC0As61WyxhTq9WiKHr66ad/+7d/+xeVTjkAACAASURBVE//9E9prIzExyk/EcKJ+qG1lvy481xoY9Pps4w4TSc2zsIYudrj3WLGYE/0bIQTtcveo7YIzSgzMNbaNJMdKHi/+dNMr8OGZ1xZFSex53pzs3Nzs3OnTp1qtVrb29txHO/s7FA6S7witGBtOq+NWZYtBIAYRLwYML5AIkMG4zlu1ufI0IB8vojE3EjrJAEAPA13yTlnXDLOGeeMYT+Mq7X6/MIBrZN+P4rjsNuLtE727T/YmJ0+8/yL//Zz/+6551741//6V37yp34ahGxvrlEof+c7zxhj3vvex7XWc3NzruPnSbYpvczoJFHamGyuh8Bs4pzRjFWv1yNnHq11GIZ0icYJw7VajaKQVmt6tIQQ7Xbb8x06pUajUa1WV1ZWvvnNb37mM5/5zd/8zZ/46Z9Wvd61a9eI8ba+vlEpla01JH88qDLTE6PHjB5CwtkKhUIGAWfktrysw62yzaHvjAX9CBkEkAkhXc+TjgtISoFiYEBJvcY79ZP/kYe7MakkCKGNhCsR9C6E8H2fRsUGvpvQ7fYdLqSUrlsiNoUxxlpSZcJcxAMAswy4FCNrw8hw6tggz+7WN4LnjOTEkAq0q3a72+vFSRIlifY8Z2JiqlotLy7e+NznPvc3f/PVt73twT/+4z95//t/2vOC7Y2ticY0gN1cXn788cc/9KEPkRiJ7/uO9LK8lmJRKZUYrZQ2OTvSDH2ndoRSikT0KX/o9/vb29uUeFATg9BbrTUxw0hrn7LEKIoSFVeCYhmK1ANqNpskLJ4kyV888X9T1L7tbW87cOAA9UDq9bpO1MApNasNbEbizzIcat+SoxhdurT4GYR7vrAeqTcARkvwkYPl1ixkHBC5EDQSZYxOjHY9D3Prus0xXm8/wfpmJDO0GAsuCoUCS73xrOu6haBQr9UBQBvdbrc7nU4cJi5zJJNZCmetNsZY5LlhqN01nsgwFnandQZUSRAO32NpIaIfKWMDMBAZhqNJOs4gR2TEereWMXbw8NFmc6vXC4Og2JipKRXfWFr53nPPffaznzt27Oj/8O//51/8xX9Vq9c77ba2MNGYQRVapX5w5uVKufa20w8bY3yvEPZDURY0m2IsaJOkjRFrgyBA2MW5KYOn0PGDADgHYygVTrE2ui6IOCzfRVtBJ3d0u90w6p9/5VwU9+k5WV9fJ0aG7/tHjhz52te+9tprr33qU5/60Ic+RAY1MzMzK0vLAByH+aQZtks5HnVwaRxZWbQW85UufZ3dr/Fqe5wJN5LY5BYgxjk3CKSrzKRDFuLFkgepKhvYgVwy3sGo9o883D3PI7eljG1CuQ0XnAKdVndCjp0pN+5GKlJhGJKRAWNMCCaFy5ga4niyVPfb7M6mDa3uIzn97ivkBhqAytZdubk8IkHHSy+9JIQol8tCOKurq2fOnPnWt7517ty5T3ziEx//+Mfn5mfDfrSyvDo3PwsAa8vLM9MNzi1jbGZmxnEc8ikpkBcxgDGGxIOUUogGuLQW2MCDjECbSqVC7BeVJI7jgBBOrntiM+60EI6UGUcEtaYJ/7m5uQw0NFaXPvlftTbXqOptNpvU2ut0Os1m85VXXllZWVleXj537tzRo0d939/Z2XFd1xhGDJZBTjKKTlIa47qutZaxdEaELjhtAhTuI/NTubuwB8A69EjkGm5CCoJbqY5XShm0rutiilq8RUoEtwNwgI1POpNMhRSylAKogIAq1pVKRbmKGhZJwqk855zrXeBxN+ItAEewey0S+cVpqGXIaeAtDfRcZPNUQz3HmQGA+fn5QqFQqVQogB544IEjR46USqWHHn5bp91tbu/U67WgMAsAKtGVSkUrtbm+Vi5XANjOzg7doUajsbW9QVu81rrbbff7fc65dP1eLyQlR1rAaPaHtBo3NzeJWQ1CWBItGgjtZxquu8iSUjRIkA1AKaW0Ud1ma2NzjWAZ3/eJ87O6unrixImPfOQjNFJMdUKxWNzZiT3PM4Zro3LjMix/PTnnnucWCgWC2wX3spmvLG/JJtz3zN33bCbcivohpVSJQS5ApIU+le9Db/tjE+7cGGVRC+5Qnw7AWgbArOv45KeEJOjLEZA7jsOY9BzXKxZqxigVa621TowxnU4n05LPPiZnYLURe13B8b14wFkXuX4q/RgfgSbToTgAxlipVKJAL5WLxdLBhYUFAHBciRbKlRIAWIME/zmudNwSoO31elNTU71eNwzDycnJGzdu+IGrlHKlAGvR6CSK47BP9JJSqWQBwYJFJAI9sV86nc7y8rLW2isE1eoEd73Ufdr1BuutBeTp2MRgSms32jgAIgM0VhGFi3w/kyQplUo0ebi8vOy67sTERKfTOXPmTLlcPnr0sOSMMQlsd9orTceZRDAAnDHhOK7repyLJEm4x6lzR746CBpyO+RIioKICGZXkXusck3T/d0bR0+O5WgFIEdrjAE0UkoGFoFz3E1m7mjlRcQf/fq+pynYnq+3fFSjMLSoVWKUjmlkjtJWY+2egONtFBNGWFCEzFCjJEtPU9sjzuv1+m3eZ49PavT25sbW1pbWWgBDMByYEEwIQfjBoAQ0YKxGDBMDnDup0bvwPM8PCp7nccdNR1eFkNwhv7F8QpiGe1a5qEQppZKYJNmMUVZZbRKlFNhdACfb+jID+0wJsNPp9Hq9kyePc8GohKC2Ec2bRlEUhr1ut0uA5tTUlOu6sdK9MBHSJUwTBhoykgHBEoDWWguWSl4DABpTWIJa3NZak6Sjj8YYo7RSymqd7lHIDHelXywWi8ePH5+ambly6XK32334kXdl+SeOphJvPRDJ7/j1lufrFwIAwAKODLkRYWG87TKi85ErnVO2jLUwgCgtdYXzXtJZ0+cuem+p95MxxEVAa6xNx0hENiqOgGAFMN91mJApF1o41Px3HIelhpJOJvw0pOtNV4yNdiEGl9sigCUqDu4S//MtfdK3IjyUHgPHcYLAe+aZZw4eOnDs2LFyvW7ieHt7m0Kc1gXX9TnnQVBkTMSx6ocRCBe4RCYscM5obUbLdpMZxtienmJ5BGZPAD5LbSQHzxGCoVXKGs05B7TA0iWS/TApzZuEu9+zMoAx4tzlgc4svDLfBDs84rTnZOoIR4V4TpnC0Yjm3g91ioRSa61J4A5TIezRrYd69Y50mJBZ9ZlNRguZdgCyk8G7vWIj3LIsJ86LeA0QIV6tVre3t59//vl6vT49PV2pVIjysLKyQhU2Ze20N3qel9icbGV6bfOdplzQ4x5hndEl9kQkKZmhTY80Q6m7fNfx888m3Gm1Hs9YqP5LpytuMeu5Z7iPTwmNJJR3KTNrrRm0jYZYfmO/KPtcu3AQMmoncc4dP8iPU7zBNWJEPWqcCzlQY+YnT568fmPx6tWr165dazQaBw4cmJyczGQ9s2YTpVVF1zP9OB/uiEiuyIPZX7w9xTdPDcpz/XfDnTEq4hGx0+lorUtB4V9+uItbUJnzOhkjQ6u3CffbEH7ybZE7YR2NHHEKMo4iRbTh2MG4EGJqFGasGTiqMrrjI7X13T94wx9zZIqCIA6iLuZ3s16vNz09XS6XV1ZWrl+/funSpUKhUK1W9+3bR3NSjUaDpBmiKBLSoeoiTcZo5pXl1BMYyxNaxrunWTMrr0aRD/dMndgYQ2nVXsMP/+LCfU9yRcZEvdVtvvPv30EZekcHCcRlCzPx+zjfhVBy6DLREQVk+dVgvEiOOO0MRLER4Yc9tVslM5k8WP6xz6bas2HLmzdv7uzsbGxsLC4uJkkyPz8/Nzfnui61b7k2XrHCBzT3NJnJr+4w+riO8yuz5DO/qg/6zKlqFQ0EkxiJ7/uvLwb5zz3c92gYId7mx+4uZN/gUgoA5Jk80ItEYwwHlFJaYzOll3QNs8YiGTTZDC8amdu4Q7jph0pmcqnLULpM51woFHr9bhiGjLEDBw7MzMxsbGxsbW1dv349Y/JkO21WRqe7EzWE8kDkHo3tPTDi2+TutBERjy0Mw2Kx6DiO0XpcL+hfVLi/Ltl45MduhczcaiguC7hxacgf6jxphiiNCavpBnPO0bIx2AGtRWXULrWZD02pEhqTi1R4w2k8jEujEZY1qNH55uam48pisZiNWk9NTU1MTBw7dmxtbS0IgomJCZpGpX1g6JkcXsVvI+d2m017vNmUxyEyrea78yr5Z5nMvJFc/3UTmzu/Q4R+ZolvGIZR2Ov1eq4jSeRRMPQ8jwMqpRwpEBEHiDIlM3ty+FJlAdcdwmdfz6GD5e1jcberACDysoQj8r8UPXnWNFkHE0aZdWopU69UKpOTkwCwvb0dhiF9ailllmClCJLdzb+tHRVZALb7DGQAfHYNjdIZSY5OLwzDQqlMmqGFQoFgoqBYxIFx7h0mqP/Cw/3N2XDGp+iLxaIUfHDDqDeZY1kO8/7urrS4iyTwVnnzeE4ohEBgI/lGxoTJM3vzizob9hK8wz15ZAjhVuefNcjEwOIF9vpd98P9TUqx8uE+MTEBaFP9GaWNMcAIYgP6X54Hi8M0vry6y5twwnsHhJQIu7VjNt2XmZ3kMauBF/lQ4A91tcalTwFHsGDYS51zIMKdPnKpF4aHI12X++H+1izwmdlLvV43WgFAv9/XCYEqlPakvK5cM58oyzhSUOZ11n90T+b4PEBemmFcr08IQe1e2LXKSQ9kA7csGMrj8+o0IxjR0PsPRHBhL54wPXLZbFp2DvfD/a1c2rM82HGkVJxm8gcYIqSKloiQtycAS8o2wwbW9wCKuRMMPhvMG/mvNLk30u4hmaQs1PImnvlwz0u35fOcoYgfPh/OOVrMZgJHsRoYCn1qr/L74f4WRnweyCO/9iFxXUQcxBBiKo+X7dgAuyJveeG4e7i6j6uRZav1SA9rQCWyw6NMmIEwFO4pHSjnv2AHGtb55xVzycy4T+3Qzw60PfZMZiCnhknjuQTWwFvrmv3/52O3dOMCrMk7zGS3N02CB6tmpmi5x+b+I1vdYVghbLxlm09yRgrNXSfnnPtNGq4ZIWxsKHt8AxwpMTN7uVsNPWW8vfwwuNZauPfD/a1b4HfRyV2okUg+VJJqRHoADEI2MLHH/DL7IZG1u4v1bK0l/4ihHxAmxffQAkfgyBhyDoylczZAGvGctNYEYyRNxRgDywSAwXSqgTOGNl2fGVpgTFjQYDmAAeQAhoEYEL/NEDEWOYBFy4Do2VwK7iBnUjpcOgCgEcTwtB7eD/c37diNGGTc8UwUdbvdKIq0jhAsB0QwgnEEA8hSRTQUFhEYR4Z75tB3cxK5LCiDumlUCnIDnZQUU/JBmjy7J4CWSWG0VVYZNJZZEADWWNRCeEHRt2C6/Z7V1NaQxUIVhWScEfMejWHoELDuuEKrWNkYgQHjjAtr0FowqZ6P4JybgcaBsUYrawYgEOcMLIDghrEo0RN+sd/tub43N7/PAms1d+oTU4NMZyBvPtKkuNVFuh+s9/xI6zygTd8iGACLoIEhoklvDbMs89G+RznM+GNzh2+4u95zBgPhfEwHR6xlaBkSSXlIFBsk0upLs44gBs7OHMjfmITsBhpSyLJKhbY1ln18mpkETL8/8E1DRCalA8CBC8fxhONx7khnlyI2GOsaXPn7q/ubn8jnc/fd5H5gBjaeucKwDdhdxv1eSdGdg0uDWjOljnHGEJjNPYdEj7EW0QLjjEuxe6okHgsASNQJCwAZJW78mdzT/HnEoisdHWTM9V3ChUh4AwbSdPeRmR+LwyhFDL6cpsrufD3kXWAh03V6w7Geq/JG1uzX5f0PUd6BJWg4MAaCMZJ/45wJzi0pmVGdKgSXwhWOHDiX79o4I0cQgjFmtbkT9HZ8QDsjHdAbZ14BJEwEd0YPuR/ub9KRGawO+pGQl5Ngwz1UGAbCd6mFdxvu4/nJbX58BIan/7dWA+fIOUNu0QJjljEE7ngBCK4TpdFKLrkjmRSMS7xFXnSbEmOEy2CHj5G9keZdisViKbUFf0N1/P1wv8dHPtZhaIhhLMhyXch0YC/zIfvh7+iIsf0dRsb4k8aYYmAGcgzIwDKwjBnXdTmXaJOBNJ8juEOTikxwNkaFuNVvzNZm2ihGbEvSEeRdPQVO04+FQoHUAuGNKQncL1V/hOG+h2DQXjv78Pr6RiHI8SzlTjL4nH0wy/7kKmghpTuQsWbABReCSwFc5kW38ydwm3DPuwtmJOSRyfqRJaBUKpHlPBHu76/uP0awzJ6xPi6xMN7pfCP4zA87Qj6SzAwfggGpuwjLLGOWMSOEYCDIMGJgTihgV1WBihK8zeRNHrTN2qh27MiW8Gy8i3Hh+z4f8BfeyIpwf3W/+7Cm12wylQ4yXSJOATUpMw5tNqaQ/4L6hZl4QUpGeL0FLC8aQ1FD+mEUrXnKeLaajhh802w/CQ5nlC/BeMEvhr0+GxQhHFgYhkFQ5ExGUYQMfL8AyJAzx/MQkEuxO8kBTPBbkIQH55w3ORs58krCRLJ3XVdrPT8/Pz09TZ/V8/27eLbvr+5v9MjciyhwiUwSh1F+U7bGIFi0BtEynk0ksTvHzu98qc7zW0Z0jGG4pZ+J9JKMWfa89ft9nSjJBWp0A5f5zFoTJbFSioBIpbRW1gCC4Fw4kM5/CQDD7ji/GsnuRmQIxv+t7/upZtO9OO6H+xs9sqUrv3JnJReChYGVw2iNCGxP17G7SEvGj5GECoaV10dSqd1vInjcTdyQIagobvc6q+trG1ubjuMdOOTGcaKUQsuk60jXAQaZNNX4+YxodI6D6yMP50jWnmE4pVIpc4K4H+5vZTKT3RXIsfZGlDPyiyt9OV6nwrgq9G3tcFNMc9itbmT9zr7Iez6O2C1l3EbXdT3PC4KAA9vZaMaR6nQ6m5vrm9sbVxavhnF07NgxKaWxkBgLXDiun6n6cGAwGDkf2aCGPk7OymEka8+r2mfmapn2N+khw/3V/a09MkFnSnyNMWEY9nu9MVg9/TOi1jQ+yzyUgWQOFjnuU/YEIA4tz/nG/gjtFoYtG/LPXjajlOXKcRyjsdevLUb9fpiEW1sb2ztbN28ueYHXaDRgQI5wXcfzPMGFBTCAgjj7u5bwe8xn5MN9xIU4j2Jl4S4YI6sLwh8zI8tsOv2uq9X74X73OQwbyEHS/HKn0+m2OyInlwK7OhMMx4i1sFeLMQt5dgs+N2aMyuHEfTxrz/sDj6AxkOPWMsaSJOl2u61Wq9/tqW6skyQMe81ma2u7efPmzXq9XigUtNbkJOX7geu69nVhnyG3mdut7iPhTnC767oU6yR1v7ujvrFl/n64v6GUPbuFFPdKKRAs25fBWmCWgUW0eSf7oUT2Fin4rVb3wcI8+q8I1bFGG2NsTiszz5HMFwmkhErFhlKKrG/iMCrKII7jlZWVi1cur66vnDt37tixY4FfUMqQPaDneY7jQepQu0clkB88z389kryNxPqQlJ+QZBBSLpdd12WDUan049zvqr75R7ZkZp5Evu+rIEn6PdQGlAFlkPjeSJ5SgjEExgE4I6djPhj8oP/L4iN1ureU6WPuFQAwFawZeLQBAlhEo7XG1NonrZgxNbGhd0cEBIsWkCFYYL1Otx+F3XYnjCOG4Hiu57h+pd7e3tnY2j5/+crLL7+0vrV5/cb64SMP1KZnt9c3mWWCp2AOBzCIPJVAzdc0mMHb9CizgQIx5ViDszH0hyNoRECiRAIi0s9zzqXn+r4PIo1vs5dRa4am2zvD1O+H+10eeWiMpmw8z5usVjZvLvUBMQqt5YNBVY4AxloAxrlFDtwoZpFJRCEdz7MMlDEouEBtUCsVA4DPfc458MxnAdAqsmNI48oorRTZ9JEHDsAgIDjjTADnwKykQGOgaBWPojiKEmU21zcczxWM+67jOb7re3GYbG1tXbl27e///u/7/f6ly5cL5VKhUprbfwhAWmBKGQZcciEYWKMEopDSGg27yLpBbdAYhhDHsVHKKg0WGaNSHixYbVWsoijuJ3FkdYLWMKNRax3G3KDLJJeiUCjWJ6dq9Qm3UHS8wAJyBC6FYKPhynIRz+6H+486mRkBIhhj09PTYRAEjmx1OyqKE5NYpZU1UqRjHRbAWpsYba213FjpWC4U59LojDciZUJuydQkyrIRRExtAgbpU6bx7TgOpnzxTMgGAdFaTVpOSZLEsQrDMEkUka6SJAmT0HVdr+ShxZWVlfMXLnzvxRdevXghCAKQUmmjja3VJ8BYlWi0ICSXjANazgCtBbRD+bpFm25OGWkdUmnjdONCBGPAZLx/RMMGFQmjEUHHcTzP8Vzp+o7npi6irxfLd5jf3A/3NwpH5iNeSskct+y5Ukrpe2G/H8WxSiJtDUNr0KI1xlpAg4zmPtBqRcTyzFOJpOqiKMpMRPIY9sh0XyYil68Cc5RxprU2KiH/syhKkiRRShtjrAEhRLFYFkL0euHy8vKLL774ytmzr12+1O31PM+rVqudTscYs7Cw0Ov1yEptV+ZlGNPEnPblrmDYeBUOZJRtEBk9EdljnPWAPc8LgmKhUCJzv3vrLXM/3N9QrFPTm+6xEIJxBtaCFE7BD8AyKUTsKuUOijJShNPZJCsyroFjzhwqD0Jn09P5UN5F5Yb3FnIvHAHj0Zo4jnVCsR7FMXHxEQbC1pVKrdlsnj179vnnn79w4cLW9jZjrFKpkKh0v9/XWs/MzFhr4zgmhH6o45PjAuVjPWM75ndCRLRotdY4KOXzOJLWmnHuOtLxPL8QuL7veC68AY2N++F+j0vVrCdCnGwaWQKjwVjGuXQcaR1tDTIwRnFERG6tlSB3u+WMG2Ak60zsP1o+aXgnbxGVoTr0i0agRgAg4aEM30gZOFZHcWK0TrRRxhqLwLiQgjE2MVUQjru5vfXaa69999lnX3rppXa743iuEMJYS+7EURQFQUAGHlrrIAg8zxuCAhEZ5/ktJesow1hlOTh/a3TWT2XWgjVgDRiLkjPBHc8LKJkRjgtMAN4P9x+biM9QyCiKGGPlUgEYZw7jAjhY1NIwSKwxFrkARJ4KRnIOQGkJcx0PIDUgIR4LjcnR9FDeOWdE+xxyEqdZ2yvjG6YOIkZpre2ALkY0LLLXnJpqXLp06cUXXzx79uy1q9fDMKSCuxeFvX7f9/04juv1er1eJ01jktJ3XTfzMd59HU6iKNwZjkiMDDYuNToPYCFdOIQjHd/zAt8LfOk6wHmKSd0P9x8f3J0SCUpzkySanJgABkwIl0GilVAuN3pQaRma1rM5ZF0Ov1u2XWTpb87fNFXzyifu4y40I5FHgD3jgnELnKG1FgEQL16+9PwLz3/3me8tLy8bY6Xr0tZBzvS1Ws0YMzs7yzn/1re+Va/XZ2dnMztvQIScmDCM2xBYy2Goj5sVOcYYWtHRMrQMgFuLBlE6nuv4nl/wgqIXFKTrMyEHYOb9cP/xiHiKMK11v9+P49gYM1GvE/mRO67j+Y5SFsAi4ScCjQIDiMaiRkQAbgwCpPxbCou8r0Z+5oPCK4qicULVOHV+0JniyAVHzNOzSL7vm9/85qVLl1ZWVpIkcRyXoBtlTaFYBIAgCKIo8n1/fX39K1/5ShzHv/M7v7MLE93C7C0f9JzvZTxmAalItwiZCSEA+V0K15Ge63iucBwuBfwIhDLvh/sbgt7JCK5QKBhjrl27BgDlcrHRaGitLeqgUGh1Wmsb64cOHWo2tyyisWDQIgMmJGe7E0Mw0InJimBCZtKeDuf5hROGbanzpMKsmZqXSNfANAKTTrlYiOP4/Pnzr7766veff1EpFSm10257blCtVrmF1vo649z3/VqtVq/Xl5eX6an7yle+UiwWP/axjxF5ptVqVet1sBaEIB4LGScxa1zX5QyUUnlij7VGa03AkLU2SbTVlnMBAEZba8CRnvR8z/cd13dc3/UD4XiWMUAruLyHUxn3w/3eLPMUrNbaVqvVaDQ4571O7Fb92Zk5a3F7e1tKhzE9UPo0g+ZIOsY/0oTP3EfoNcvUs/wEbu3bMy5r4Xke2c93Op2bN2+eO3fu7Nmza2trSZIgQqVSEdyJ41hrXa/XpSMajcbm5ma/3y8Wi7VazXGcTqfzxBNPkHXzoUOHpqengTEQIglD6fkZhzFJ4iiKjFZKKem7cHsjRJMjEZC/npNmNMJxhCOZEAACc86p98P9raxT82kDQSiIePnilVqlPtVoOFKG/X5Q8CfrE61m03c9C4xJYADWUogjYwzM3m8+zusasUd+XZuqFLExxvU8IZxuL7yxdPOll1964cwPLl++LIWjjeVcul4Qx2pnp+U4ztTURK1WmZ6evnjx4tzc3MzMTKvV4pxHUVSr1b761a8CwEc/+tGFAwdQayalGwSkokoeqyaJwzBEazzPQ7R507VdshAyhoxZQAAiOhhgggkhHOl4buATCimkY0Hcc0P3++F+b47Mav3mzZvlcrlaLhcK5X6n02l3BbBqtcYsag5gheWWkf4h0QZH1P6HifKZcwbsJRcKY0Mh43MeNKHX64U3b9589dVXz549u7S0FIbh1GTJWquU6ff7ShnXdavVar0+GYa9GzdubG5uHjp0SCm1ubl5+vTpJEnK5fJ3v/vdL33pS0mSkGtNvV4PgsAtFJVShPqnG50QnuepONyd2kY7Pm6SZVz0T5gUJPDreJ50HARmfwSjpfdnVe9+dc9uG4W74zgcWCkoXbl45ZWXzgJAoVS2Skf9sFIskXwoB8aRMwRmESzb1Z8be2c+jGfbWxOkstPY83D8oN3rX7p65QevvPyDl89eXbzRj+KgWNraaSmLBlmUaC8oHjx8dN+BQ14QhGHYbrepGrly5Uqj0fj4xz/+i7/4izdv3nzooYeklF/96lc/97nP7ezslEolarUSY4c6r3TCSZLAXqT23GbFMZXLBD6Q0nZ9z/V86bogpAVmAfXry+DdX93f3HCnLyjcAWB+fv78a68999xzhULhxOlT1fpUa2sDDXDOwYJlhgNDJjgaaurANAAAIABJREFUApUZ32PCY0zF4Ja/N3tNMcfhMXBENAZXVlbOnj378ssvX7t2bWdnhzEeBIFSKgiCQuBba8vlarVaTZJkfX01SZKFhblardbtdtfX10+cOGGtvXnzZhzHs7OzUsqNjY2XXnrpn/7pn3q93sLCghsUXNdljPV6PaUSx3EArdZacshNq9gRK6gREoTje0EQEGvAcRwGTA8gfQsg7t1dE5/97Gfvx+5d95iyJn+SJNvb21EYCsY5wsrS8vrq2tx0o1ytOQBaKVIeIrMO+oJz2vv3kN8YWbCznjyF8rjXALVRR9r49PWNpZvnzp37wZmXF69d73Z7WqdPBU1peK5fLBZd14uiaGtra2enyRlMTNQR8caNG0opzjlRDKanp/v9PmOsVqsBwNmzZ8+fP7+xsTHVmC4UCrTSR2GfesJaa06OVMSYsCY7T2sgM0zQyiKC6wWFUqlcqpQq5UK15gUBAjOACFyA4Pc0A7m/ur/R1T1NCgcw4srNxcOHDq2u3Hj5pRfm5xpzs1PClY4RoDJlFYkI1FgVlAsxCxaRDXoqFkEwq43NMdSRAQcGnKGxA8760CsaS+9g0aZ9LGMt2utXr125fO369evtdpsxTu3SJEkqlQJRYjzPs1a32604jqrV6r75WSn56upqHMfveMc7qBp5z3veo5RqtVqVSqXb7Z44ceLatWurq6tf/vKXFxYWHM58R8b9XqfdKhQKgedYnYDrgjW3KjbykBbxJrzA9zzPd1wODAEkMKIS3dts+/7qfg96TDTjo5Ta2dooB87GyuLJ4w+srdx8+un/d9/8/NzCHEPwC0Fzu8kdWSpX+2FkARwnaO7seK6LFlFbRJBMMMY5MmMRDFoEZiEVgQfGLFi0DBHBMmstWrILEAwYBwHAOFiltNGeIxlnWxvrFy5ceua7z165dGVrc4shCMaTOGYIxaAwWa+vraycOH6sUi5fX7zW3N6qlEu/8su/JCUnU1hSemk0Gvv27Wu32+12O4qi8+fPT0xM/OzP/izlb5126x/+7snAEadPnZhemHPAdtsttKZSLnXaLd9zOGf9fk8pRQN43W7Xc1ydmG4/DKMYmPCDguv5wNj09GwQFD0vAC44AkMg3aZ76+twv1S9B4h7Bs74vmt1FHiyubX+6GPvdAT/T3/1H3Uci4K/cnPJ81zf93u9ngVWLle9oOi6PtHEERGMNcagNvSafX/olRLfse9zhCiKkjBSSqE2cRy3mzvLy8uLi4vrq2utVos6vvmScXV19dSpU2EYvvrqWd93Z2YaDz548pd+6cPtdjsMQwJMqFvc7Xb7/T5lMp7n9Xq9VqtFSFSv1ysVgr/5z3/9R//L/3TplZdqM40jhw+i1ZsbazPTU5TgWWuJxUlfoGVD1DfOhSO9wOdCpINLNKBiEew9JszcX93fUKDnk3gASJJExeHa0s3DBw9ev7kkhZyenn32e88ur6w+/thjxtigUOiFYT/sFYpFKUUUhgA2NZnODZJmPII95GPSNMWOt2+IEUmsMq31xsbGhQsXzl+4ePXqtXa7kyRJRjUjvK9cLjPGoihqt9udTudDH/rQpz71qSRJvva1rxGbkjqg1N8lMxLXdZMkWV1d1Vpvb28LIfq93kSt2mm3FhcXl5eXy0Gwf//++tSU5LzVaoVh6HlevV4HgO3t7SiKCoVCnOjd8VxgjuMUi8ViqVSpVFMWJBUziMB4KuNwP9x/DLMaay0abeKw3WlZi51O5+Dhw0mSfOWrf1sulx9516OA0G61meATtYlOt7O1tV0uFsGaLHzzqkO30JZAxD0MGWHAiKRWV6fTWVxcPHfu3IWLlzqdbqIUWefRw+D7PulHt1qt7e3tQqHwwAMPfOYzn3n8ve/93z//+WvXrtFTQRGfJEkYhmEYdjodKWW5XKbQ39jYEEIwAN91qtWKtfbKlSsXLlyI43huZmZyYUFHUfb00oQhAHAm4zhJBXkAAZnruuVqpVKpBKWS4ziMCO6p4TBj98P9xzbcGWOSQ61Seubb35mdnZ+YmlpcXJyamnI9/6mnnlrYt79arVaqVSElAIb9nlG6WAqMVsAsos0J7wIAWqTHYOQP5ruV+XCndEVr3ev1iClw7ty5GzeXhJBmQJ6hcCeQe3l5OQiC9fX1d7zjHb//+7//X7z//a+dPUtMAWIgE8uFyAWkQZAkyfz8fLFYrFQqYRgqpfr9nta63Wq7nj85ObWz03rxzA/W1zeqxdKRB0/7jtvt9be2mxahVp/w/SCMQq0MxTpRkf1CUKvVKpWKHxS4EHn7yMxQ5H64/xjhM3mOlyNEd2enUqm02q0gCFqtlkF88MHTr7zyyssvvzI3N3f4yBGrTafVdjy3WCogolZRlk9n3MYRw8fh37h3uNObJEnSbDavXr362muvLS0t9cMIEfSALpaN3mmtm81mr9d75zvf+elPf/q//IVfUHH89NNPP/PMMxn5J45jeoRooO7IkSOdTofo73zghKqSRCX/X3tvGiTXdZ0Jnru8/eWelbWjUAWABAiCi2iJFEmJIdqyFtvj6Oj2JnkUEz3jmOkIj8NWyP3DP3p+TMQ4JmLmh/+Ne6ajJ7xIEW21xt3WjNRabFmyTFGkKFIkSBAAsRdqyz3z7e/eOz9O5sVDAYRIGSQBKV8wKoqFqsysyu+ed853vvOdREmhdWx5nnc6nZMnT5Zd1/f9lbW1SqmEPjxT30ylppuTDdMqlUrVatUrl7hhAaMwdfKYiIlm0f2OpSMJIZTRQad74MCBzatbo3FQrdWzLAeASq362qungiAgAK7jUEIYpYzSKAzzPBMik2ibOrEVU0pJIfKbRXd5026UzmfiON7d3T19+vTZs2cHg4FhWkEQCiEJIWhUZBgG3gF8319ZWfnsZz/7yU9+Mo7jwWBw8uTJ7373u47j5HmOo0xYD2grL8MwHnjggcFg0Ol0kGyhhHBGy5WKkHKv3c7yvFavC6nOnT9/+syZLM8XFxYbcy0AMhyN4yTlhnktW2PUdbxKpVKp1yzPm24I1HC//Vifwf22wX26NVuVa9Xnv/dMq9WilLbb7UajMRgNbdv2XP973/ve5ubmPffcs7i4GIyD8Wic5jHAdWLdosn/LZ70Rqxj4jEcDjc3N0+dOvXGG2+EYWiY1ngcACGcc+1ShJ5htm3/0R/90T//9V+/urmJUP7Sl7509uxZx3EwX8fQzjn3PK9UKr3yyivr6+uf+tSnCCE7Ozs4vRXHief5w0E/y7JKpaKU2tzcDMOw1Wr1er0LFy5cuXLFMIz5+XnfR4lOxhkTSgopGWOe65eqlVKpTNjEew8UgetWgczgfucl7gU6UmVRVPZLYRgFQSgVBGEEilDCPN+xLPNHL7546rVXDxxYnW/NjUZDxpiQOWMEQAmREwKmaRACaZoYBicEMMxLKTD8A6gkSfcZ56IRO2qGt7a2Xnrppddff308Hiul4iRNktT1vIWFBdu2u93ucDjELsHnPve53/qt3xJZppQaDAZKKWyUUkrxMFiW1W63hRDVajWO46WlpTNnzliWZdv2qVOnxuOxaZpLS0vBeEwoI5QpIIwbtuMKqbq9vmU7uZCbV7dOvX46ipOl5ZVSudIfDKMoqFQqcRwDpa35hVqtxm0nTxLKDSAEgELBNep2p+4zuN/ucE+VStMkiuMcuQhKJyoAkIyxSrl8/vz555577tChQ+9/9NGrVy4DSMooOpIibaeUQpqvqAwrDPVdc/Td5xkdBMHW1tbFixcx+koppQIh5Mrq6mg0Go/H5XL50qVLpmn+/u///q/8yq+Ypul4nmPbhJBer/f5z3++3+/j3KBpmqPRiFK6tLTkuq5t20899RSyjVeuXAGAgwcPCiEuXrzYqNfxpKEuUo+0nj9/HjUwV65c+eY3v/nss88ahvHggw826jXTtk3L8jyPUsYMblm2UJJN/Tz27zabwf0ORjshlGZZGicJKDBMkzImhciyjAIwSufnGiLPXvjB8+feOOs59tLiPGE0TiPGWblcEiJvt/cIgYPrB7vdjpRCKqnQHJKgmIxIeZ11elH82O12L168ePHixU6ng5pEqcC2nUajGYYhhnDbtp9++unPfe5z80tLo8HA9f0kjoUQQRB88Ytf7HQ6KGPmnLfb7Wq1ur6+nmVZs9ksl8tXr17t9/v9fp9zXqvVcH48SVPKmGXblmVTxoSUWZ5ned5qzQdhOB4H5XJlrtUKw+h7zz77H/7qP8Rx/Nprr1UqlbX1g5btOI7DuJHnGedGEd5y6hM4g/sdnt9AFqdJnCqiTMOglGZpmiSJFLlt20kcVavV+fn5Z5995pnvfrc517z3vqNZngVBkOe567qYAbfb7X3KwSmyhW6E78N6lmW7u7vnzp27fPnycDicpP6ENhrN4WjEGIuiKI7jX//1X//sZz/bbDazNDVNk1G6u7vLGCuVShcuXHjppZds246iiFKKKXi1Wh0MBocOHXrllVfCMETP9TiO2+22lHJxcTGOY8uyUA2aZRmuvkGdYxzHKDLDfxVC5Ln49t9/++zZM8eO33f//febpmU6DhBCGSUELTMJFJZf33bEz0QEt/2iigBQQpnBTAOTXTSNcRwHO/nHjx39+C9+zOT83/1f/+ef/MmfdDqdw4cP27bdbrdRQd7pdIob6nSOnqbpjXsv8PM8z8fjMUoAdMsJKfbhcNjv90ul0pNPPvnpT3/60D339Hq9fr9vex72SiuVim3bv/iLv9hsNhGaeZ7X63Xf98fjMQCgPBgJHF0k7O3ttdvtUrVm2E6uYBzF4yhOhWSm5filwTjgls1Ma2t377XTZ3rDUaM1f+99x7yS73huo9EwTTObluOUXMNhEeu3/72ZwfP2hnZQihBGKSeEgKKEM8dzK5VKuVzu9/vz8/PLy4vtdnthcf7jH/9YrVb7whe+8Kd/+qd/93d/h5Tf1tZWv98/dOjQvnWnhVGJTBRcuPQz6yaobgUgXx4EAaU0SZL777//d3/3d48dOzbq97GxCgBhGFYqFWoYOzs7i4uLR44cYYx5nqeUqtfrpmnGcdxoNFAxhk/HGEuShHO+vr6+sLCwu7s7Ho8ZY7VabW5urlwuCyH6/T76sqM/VKvVWlhYME2z1+uVy+Xl5eXV1VVmmq7rAkAYhRrc8p0ORTOI3nayBnA9IiHITrquW6vVGo0Ggmyu0bRNM43iubm5j3705z/60Z//wQ+e+9f/+nP/9t/+H+32ru+7Uua9Xgf9oAEkIYpSYIxQOtl8tm+XJYJ74t01dSPTcA/DUCn18MMPf+pTn3rsscewD+W6ruu6eZLgDUfleZZlrus+9NBDcRyXSiU0nEnTlBBy4MCBy5cv4w2nUqkgddNqtR599NHl5eVOp4MSNJQYlEoly7KQ7cHiG1G+tbVVrVY/9rGP1Wq1gwcPNudbACj5hyiK9ldA70xoh5ne/Z0I8IwZhDFCmADBCOG2bdsARD7wwP2vnTx5sb135Og90Tg4ffqUJHDvocPzc60f/OAHn/+zv/j+P37vtz/zXz/66KNZkhqGQQgBOiFkkPGQEtIkI0RRIKrgrQSFkT9qcGBU5gCMGoahCARx9PRHf+Gf/dq/6HW73V53dXU1zTKLW3Gc+OXS3u5unKXzS4uU0nvvOxYEQbPZTNMUu6q+77dareeff17v+UABzNLSUrPZ/Na3vtVqtVBC0263Xdf1PK/ZbLZarZ2dnfF4LIRwHEdKube399JLL+V5+olf+qXDhw/Pzy+NhkNCqF8q5Xmuwy46tb9jtPusVL19zSbtgUgoBUKTLI3iSOQZJYRTCkQZBi+XK4bB9nZ3R8GwVC7XypVoNDYpm6vVm9Xa1pXNv/+7b5059bqScnlxCX1MpRCUUUppKvI0iQ3GRZ6nSZLnGSFgmabj2q7rZFLu7O62u504SdI0NwyTMt4dDAfj8b/5n/7Np3/7t4MgiOKoWqsxzhlnuchdz1O484wQymgYRaVyaa/dfumHL66vr3e73SAIHnjggVOnTu3t7aVpOhwOK5VKFEXdbvcP//APv/vd7547d65SLlGAcsl3bfvc2bPLS4siy7I0sU3TYMwyjTxLszSplsuUwGunTs3PL/7L//a/syw7z6Xvl/Jc2JbDmEHg+l1W70xfdQb32x0/GFVKZlmaZRn6oBMCQkoCinGDUnQhAMY4EJoLUa9WGaEG59VKtVQupUl64dz573//+xcvXVRSrqysliuV4WDY7rQNxufn56MwtE2zVPIr5ZJlOwAqjpMwDGuNRhAEO7u73W53FIaD0VBIaDQaH/v4xz/y9NMrKytplgEA7n/FRFwqxQ3DMM0kTYejEQDUajWR56++chJtvXAqb2trK0kS3B5DCNnb2/vABz7QbDbfeOMNIQTehTjnWZbZtv3+97///PnzSqlGozGxiQVAkZlt2+VKZWlx9cNPPeW6nhDSMEykVou7Id45rM+SmXcidVdSZGkSKZlbnAkCIs+EEJwCpdSw7Vq9aVnWeDzu9Xqj/kBJmWYiy6VbKt979L5yvXHp0qVer/fCD158+Ucnv/Jfvv7YY4898cQTq6sHRqPRyy+9YnJ2bf0v45Zl+X7JcZxxFC8sLKytrSdJRrk96I+yLFtdXf3MZz6zvr4OSmVpahiGkjLLcwBozbX6g34Uhr7v25aVJgmKhA8dOoT91Lm5Oc/zcLADR1E55yhbf/zxx1955ZXd3V08AEihXrly5emnn65UKmhMsLW1hV+v1WqVSqXb7SZJYnOGBk9QWPIDha2d7/Q1g/vtzdslKMiiYNTr4Cptvf5cylxKyQzDdj3TdmyvTLilgFqMUoPbg0GSJGEY2o6zvr6+cejQ/MLCpUuXTr926oc//OH/9zdffv9jjz766KPr6+uMEc45ptFoQyekyvOcW/bCwsLaWj+KolwC2t18+MMfPnz4MKUUC0fTNJHeMQxDSGHbdrEbiq/2tVdOolgSPVCRa6/X6/gISZKsr6+Xy+U33ngDq1utnQzD8Od+7ue+8Y1vIDGPhwShjJ+EYchMY3l52TAMbTGpbezfnTdoBvfbn8mH42GvsyeEwPY759wyWBiLTCgJgjFmGJZnWBIoY6zb3jFdp0SJEceCgJlNJDGc8wcffPCRRx7Z29u7eOHC9//xmddefqXRbH7oQx+qzzUXFxer1arnG0opoExKqFUbjuOEURJFkVBkPB6vr2/88i//ShiGAICKLtzzgZttOp1Oa66FPGAQBEgaxnH81a9+NUmSUqmEhCMaYSPrEoahaZqPPvro5uYmEpSYw3iet7m5efz48SRJLly4IIQol8ucc9d1O53O3t5eo9Eol8uu6/qV8tGjRxH9eKFgYQb3u/eSSTgaD7pCCCVymWe26zmOg7VsLqRUANy0LMurcGbwOIvFQORxCJw25pqVWnU8HI3H4+Xl5b29va2rV/MsW1hYmG/OhWHY7XS++MUvVhv15eXlpaWl5lyr2WxW6w3P83Z2dlqt1srKSqPRuPdY98iRI9VqbXFx8fzFC7hkHZtWSFYSQiqVCgDESTwajer1uuu45y+c/94/PnPq1CnsEiDcMfEIw1BKqfn1F154AQBs2x6Pxyg4S5LkySeffPbZZwnBk7Y+Ho8pnWiBUHJsGEajNXfo0CHkanQzAf4Ja4FncH9PkxklQWQyy1Seg5RZGiulJGDDiDHGcF2HkKCAGqYNhNz/0IO9zt7O1nav18uTNEmSLM3TNDt79g3LspaXVlZXaJYkwWg8Go3iNBmGQZJn589fOHv2DdNyKpVKrdH0fX9pZaXZbK4fOry2tra2drBWq6PULE8z5nqMUVwlAFJRICCVZVpJHI+GQ06ZZZjtvb1vfv0bf/VXf5WmabPZxGRJL5ySUvZ6vSRJXNe9ePHi9va2aZo4uD03N3flypWHH37Ytu1XX30Vu1eovMfEqdls1mq1nZ0dy7KWl5fr9TqefNTAaWrr3UH8DO63PZcRoASjAASIVGkaE8I4545XopxRYiglAahUhBJGObt85arrGEsry9V6rdfp9no9TGdXPrwy7PU7e+0gCAghnudxzqMkrrfmgjgKgiCOYyEBpTK7u7t7nU6e50srrx47dmzlwNrS0hKO2GHsxPIRl+EgDbKzvV2v1+daLQC4dPHi1772tS9/+cunTp1qVGuoGGOM4TgIXmma4oDIpUuXDMPAhAS3yb5++o3f+Z3fOX36dBzHKKwfDAbz8/Mo3cH2KpqQPf300+jAgb+jnt7C/31X4tG7lTb9LPDuVOVAshf/4du7u7v1ZiOOskTk1WqTm4bjlYASUExSQqnJDcOyLMMkSTziBs2SdDQaySynhISjcb/bS+MkSZIkipIozpM0TdM0SdIsC7Mkm3riSSCEEMoZpbRcrS8vLzfnWkePHmWmhXpGzjmIvLjvANW5hmFgLu75fq/b/cIXvvD5z39+OBwuLi52u12saJVSlUqlXq+j78Dp06eXl5cXFhYIIcPhEJu1jLFOp7O2tlav18fjcb/fh6ntvVIKx7FRb9Pv93/1V3/1f/7j/0UJqgjs2x/4rl2z6H6br7/6i89ffOMMABw6dKg5P19yK6ZtmaY5GAwsy7Jsh1FTgRR5moIUOWnNz7f3trrdrlKq7PmObXueV6lUuu1OlqRRFGVJkqdZlmVZnKQiT7I0n4oICOOMMWZwxliztbC4uFiuVBcXF6M0Q6NqxlgcRzqOIkeEr7PX69Xn5gCg3W5/85vfNE0T05LFxUVEKiYke3t7WZaFYbiwsFCtVtHmIE1T13UJIWhSMBwOMWXHBN2yLHSkCcMQfXVGo9GDDz749NNPv+fvzgzutzXME/rP/vm/+Ie//9s///M///d/9meUGWtra/fdf+LYsWOPvP9R13VtxyWEpEJKKSlRnNE0TixuzdXnpJRJFO/utPM8twyjXK1lSWo5bpakQgiZ5aitTQWm1HSyu8swDNPmnJdrtfn5edvxXL8cd/tpkmeWmHB803UBhBCdQJRLJQBIw/D1U6dKvn/50iXDMBr1uu/7uKkYWZd+v4/T2dVqFYlLLF71EGC5XMbWFRbBKMOklEZRhHXtcDgEgMcee+wjTz8t8pwScwb3nybEs40jR+878dAbFy73er3XXj/97HPP27Zdb7aOHz/+1FNPPfLII0tLS6bjCCGkELtXtyzLoJRmSYqNScxi0zgGSoASYjBGCWHU4JTkOZemUgLVMczgpmmaloP0CI6cAiFIpUspGTBODcau7W/SKnns8gRB8OKLL957772vv/46ACwuLsZJgk0f/AYkIsvlMk5LoYwZPx+NRkEQbGxsmKaJR8I0TcyUKKU7Ozu+7+MLO3r06MMPPwyUqlwAmUX3n6LLcNz5xWW/XM0lhHEKlLt+2ff9Yb/7zHe/8/3v/WO5XD506NAHP/jBj3zkI/fff7/vuI7jhGEYDAMQUKlWpJSdTgcYzSVIoIowxagiUhFCCFCpiOJIZTBqUMIJoYRQx/Ys02GGCUAp5ZRyzk3DMLIkJnLS5Jq4FptCGcZuuN1aWalVa81a/W//9m+btXqpVGr3uoxzIYTWHmN1i61QhL7neXEc93q9NE0rlQqalqHlWHGbmmEYSqkLFy7UarWnn376xIkTcMstI+/ONdPM3ObSX4gcKHP9UmthcWN9fWl52XFsKWWtWrFtG6QIxuPd7auvv/bqP3zn7//mP/1nAmBb9vLSkuM4cRQNR8MgCLIssxxbEVAAilLCKGEMl6yaDBN2xhjj3DBN0zBNwzAq1Zrv+0AoM4wwCLvdLuecgMrSROTZtRbsdBeIbdujfp8z9sADD/zxH/8xuouNg0Br5ZFP1LbxGMLx5hAEAW5uWlpaQo0kCoZxOBUrBM/zgiDodDpHjx799Kc/vbaxAQBC5Iwa8N6VqjO43+4mkwLG+MLi0j333Hvo8KEHT5x438MP33PPPfOtufn5ed9zDU6pUkkU9bvdne3tr/+Xb5w5fSZPs2qt6jhOnuWmZS0uLgZRKJXKpUQ3eGBEglJSMQIUEJCIdtO0TMMw/FLZtm0hFTfNKIy63S6lVIpcCiFFjqpdzGFs27Ztuz43d+XSJQCozc9vXbmytbU1GAxK5bKU0rJt7PNjcxcAcAzPsixK6Wg0Qk18rVZjjOmFM6gnQx+bNE1t25ZSvu997/uN3/iNRx55xJy6olLKYcbM/PQkM4YRRYltM9O0CbByuVqr1VzXe/zxx9MoDsbDwaC3u71z7vzZM6de39zcCsbRM888c+7ihc/0up/85Cdt2+4PB2EYEs5yked5LoRA2y0hhBQZUYQoAEUZAAGlgBJgQLlpmpybSuQAYNs47G9QkBKUktd2ZqAfqud5SRgeOXIkTdNvfuUrv/d7vxeG4YsvvpgnKVCCggKU1uDIaRzH29vb6+vryKCj4NEwjN3dXc/zBChGGOGMUp7kSRjGw2CoFCnXqr/925/5hY9/FCQZj4e+XyGEqfc0d5/x7j85116cm9bWSAYzCYU0ya5evToeDA2TYcCzLAukIkTpEnDUH4yDYRyGz//g+ydPnqzX60899dRD73uYUrq3t2eapiq4LCHu9ViqUsQ0Lct0DMOyLIubxsbGBlAKSgGFfqdz8uTLQubHjx/fvroNQqIEF/MN3/cdx0FNZRRF/X4fp2AvXrx45syZ02fP9Xo9HFFF+1+M8WgLjKcliqJer8c5rzcbnX7P8/2SV5YgglEYJSEFZlj8xPEHfvm/+qWPPPW0aRtJlDqeTYDlIp/s3Z6Vqnd3zj51VpJZLqWMwhCEZIwpSaQASjgBhvsUcbLOpSY3HL9Wc026sDz/2OMfTJLEcZzhcOg4Tq1WwxltNTVBpTjSRohUOWHAmWkYJmFUgpKKMGoAZSqXSRrZnuO6dpKGo9FIKeE7Tp4LjNAYsCmlKFzTa21M06zX681mc2NjI07/X0LIaDQaDoeUUt/35+fn5+bmLl2VFudvAAAgAElEQVS6hAcD5WJSSmTcbcu1LIcwmidZmmecm41Go9ls/vf/6n84cuSI47oAADSbrhx+j2PrLLr/5NEdrl+sNVniPg6w4aKHStGPBRlGLAELHqgQDvuWbQDAaDTqdrtZlvm+jz5b+qahl2VLgCxPGOe27VumLYTKc+nYXrlcbi3OA4DIUmbycDT43/73//Vv/uZvNjY2fu9f/Y+lUrlarWKtiUoYSunCwgK2S3d3d/M89zwPh1OZYW1tbV25ciUIAsuyWq3W2tra4uLiX/zFX5w+ffrcuXNoETwxoWe03my1FuaXlpYqlYrneYuLi/fdd9/Gxob+Q+EzWpZ1J7xrs+j+T0I8VmnF5WGYHoRhiAlDnucIdzRjQU1iFEXYrKEUhsOhGk4YD/w2nH3mnBeHr/FnFSEkVdwwfN93bDeO0yhKkByMxqHju+jF5ZZKnudtb29fvXr1r//jlw9vHPzQhz706KOP3nvvvYuLi41GA6W5lmV5njc/P4+8CkLT9cvlcnljYwMFw0jnW5b1+OOPe56XZdnm5iYhpF6vr6yszM23PvPf/MtytYKeAkEQoLQGDznufsIfv0Peshnc/0lw1xk8Nt6zLMPVLprHwGoPQzv6TWuWOk1TpUS1Wg2jcRRF2HjHXibK0zXW9c8CpWkWw3Q/I2MTNa9pmpgTizzPskSIbGlp6fjx45ubmwcWV4Mg/MpXvvKlL33J87wjR448/PDDhw8ffvLJJz3Pw2FqvdOPUjrsdw3DsEyTECKlCMfDQa+TZdnxY/fWq+W11WXcQlytVjc2NlbX1qMoopyh64HneaiInGCLc/3CAAAtlt4dKdgM7u9g1g4AeZ6jgCQIAnynMepbloUBcpKNSAkAjLHpzGiOa2TQZhrn/3EwIptaTE6kw5gIUYrnKkkS23K0AwfBVQoAzDSjKNjd3WGMHThwYDQanXn1tG07juNgbD558uTzzz+fZdn8/Hy1Wl1aWlpbWzt48ODa2trS0lKpVPI8D/ekYqMUxzIYY71eb3FxcXl5GVN2NA8jVLmerZQaj8dogY2zfHmeo9uw/ivdCVif5e7/1EwGVYHITPf7/dFoNOp1MUgj1svlsu/7qO3G7Rr4EQ3UsywZ9/umxTnnqNFFG0dc86udla7twwEIo3GSpo5TqlXrUsJ4HJb8yvLKSpokeZ67ZQ9Abm9evnDxjW63u729feHM+Xa7c/ny5e3tbTyKyDPi6AbeZ/AGgjeiQ+vreC/K8xxL2EqlYprmwYMHy+Vys9lEyzGlVBRFYRyvHz1KmaFzniK+8Y+g+f73HOuz6H4bQrs2tcM2O9aa2HXHAVDTsgCAUaqiCAOznhDFMTkFAsk+vO8rpXDyX2Ndc525lJZl5ULozTN49wAKpmMl/QRf1eXLl3u93qFDhx566KHRB4ZhGI3H4/F4PBqNOp3O9vZ2t9vFdUsYp/GugsYBzz33XLlcxr2Z6O+OOf3hw4cty2o2m0eOHDlx4sTa2prjOEDpcDj0/MkdoGhUr9M87fGEj1MM+TO43zVsDKX0zJkz29vbzWZzaWkJKWqs+cbjIYCPrrmGYUwtgyRjhHOa5wSACAFS5mma2pxjvEdk4F4kpVQcxzriQnGDthSu63JuRVHEmDFJlxXIXGDenCeJ53kKGnrjDe4jwHQIc6E8z0ejEZ46HEhFw9Q8z3e3t0EJnSPpI2cYxs7OTqPROHPmzJXLF3/zN39zYWEBQKEbq65VihoEvPSCZbyxzKL7XXlJKbe2tr7xjW9sbW3V63XMfctlf+PAKtoslkollEkRIYAxfOOxehuNRjjhv7K63G/vaVdHhIUmfLTuXKvVOaVRnAAhlEoyXVEPAKCAGkxmE32LZVlSuVj16mJXL5rE83PgwAFEKqZVk3UdWZbde0RPGGE5gccjTdNyuYyzHePx+OrVq67rSkU8QvXIiC5O7thrBvef8KKUbmxsHDhw4Ny5c88888y3v/3tPM/jODx8cO3wkUNPPPHE448/vriygsBRIlNK9ft9vJublgGB6g96/UHPs8x9cRSjOKb72A3VbSyFI6QAlObACCFs/9YaKQHAcRxuEGyaIoczseADwPzHNE1MqRHKumWLZKjBruUeAcg8jUWWlDzXd+3hcHjwwMr29vbeztbhjYPlag0JU71apzhBMoP7T9W1trb2C7/wC5TSZ5999vLly1tbW51ORyTxCz/8wV/+5V+i0PeJJ574xCc+8XPvfz+htD43l8Vxp9PBxRiNRoNSOu73NJp1dNeribV3+2RcTynbMXGtB7ACHyol/jzQyeNg6q9vKROaUgiM7o7jjEYjKCwFmcR+xrI0Jmryg7qExTq7XC7j1hohxO7ubhzHddPknAdhiIV1qVSawf2n88qyDBNr5NqVUvPz841G7fVXXwaQtm1SCpcuXbhy5dKXv/yfS6XSJz/5yQ9+8IP333+/41h5nuIj9Ptd/ibzDno1JMJ3ssN6mtgU55onyU+mdIoshACisBIg0+xZpxmIb3S/wEUdmu4UWUYgZ0Rh+Q0gOae2bRoGC4LAMJhlGeVyaTQqX758cXv7qlepznkVdFya3FU4f/d1jjO4v+NXmqaGYVQqleXlZZy6b7fbw2H/8OHDURSgQAWrQFww/e1vP/P+9z/05JNPPvjggw888MCBAwdwY+PiXAvrQt2gxSvLMsQrTn/iatJMCMrAtCzDMDjjlE4kCVrFToGh+pwbE8oINwUU+X4sWCeFrxCYzGC5KbKMMUbUhDDFL2JChevkdZhvNBphGAqRoYdMGIaNRuMOT9xhpnf/ia8oilA4fuDAgY2NjY2NjaWlpcXFBaIkRjfs7ZdKpUqlUqvV5uYa29vbzz777Msvv7y7uyuEqNfrhw4dCsfBPqd2HbOL6J9A3zQZp6ZlWZaDcKeUcWbgv0oppcjG41EcR5ZtMsbG4zEjk6QHs3a0waCcc8syOOeFuwROsjqWYRmGaZgG55xxRhmjjDOWJkmapIyyPMst0+SMxVG8uLTMLGcwGjHG5ubmcGGTNsSbRfe7lYYBRYFIooiafqxWqkJk4TiKkrDVnGs0ag+eeCCKg93tzb29nfbu3igYj4ejnb3dvZ3d4XikhBRCuLYTRdFff+n/+frXv/7Jj3/i137t15r1ut5NQKYD1JhwT7j2PNfptSIEXV8455QwKUFv5WWcZ1GUZjFqy7A2TZKEO6aQQhHGGeWmbTo24xZQBbkSSuYSciUzoTIp0lxKIZI0pUAopQqYIEIQmgORChaXV3q9jmEYw/7AMAzHta9cuTIYDMxSFR1pkOvEkvfd8Tedwf2dwTpIAFAyJ5JIEAyYAgkKpMgoSCXzNErjLJaZBJlXy5VKyT+yvpHmeZYkQRTh2Nyg1+sPh+3d3e3d3X63OxiNnnv22e9+5zt/8Ad/8NBDDy0uLuJeu7m5Odu2cfupbtboQlYRwjlDyTujwLnJ2KRz6VBI09RxnW63bdt2uVw+c+YMIUQxahgWZ6akEKaZZKRccgiFHIQAkilIpcqBSMqIYRJOZcbktHhQCoQBAJTyHLhRqtUtk7darW63s9fdmWs1e/3OyqF7MKIjxJFQetdcwWZwv+1xffIJAQJEMaCgJFWQJCFqAfIkzbJEZrkQQiphUEYoRQWBct1yuYxZcumBB9AmNwxDpLpx4dH29varr766urrqeR4ayqFsRpeVN31ZxT3xOnkQQpw/f34w6K2sLCml8KmDKM2FMpi0HNN2TMu2CaUAkhsGkSByhXvpCaWcEwmMGiba60mVSyopNxhjIE2lhOW4Zd+rVkq1Zt2yrAsXLmzv7B2OIm39rrOmWTJz9wJeAlGT1RKE4LgyjidrURfWc0opBVIJSZTSui70GELhFJa2tVoNoYmSFfS/DYIAlzcNBgM0LboxoSeEwFQmqYEOIPTQU61ZH40HjrPQaDS2t69ubm4GQXDPkWO5kgZltmX4rm3YJkiRZYlh2IwRRhQoIfM0S2OVq1xJ27YzJaXI8izNsoRIBURRIAypHtOwHa82N+d5fpSmW51+u9f3S5VKpaIXst6xmcwM7m/hIuQaVTjd9ohuoLp/PsG6UkCAUooaGFS6I4OOXrhafYB6Q2Q/Tpw40W63sywLgkA7wKAn440xsviVa+EdFACEo8AteaPRqNmsm45jmuahQ4c45xfOXcSi2XMtzgCUAqIMw8jjQEqZZ5lBCNgGZ4DHOBMZA2kxYliG4NdI9CgYCSGCICKkv+S4/tzCxpGj/SDd29vzSxXdQr7D38wZ3N8C3EGhRAvtKzCcY1dSdz0nOKZgcgZEFpGNcPd9H/k+PXKKfAsKD5Mk2dvbw/F+1BsWka2juyrqdsh1/+Q4zksvvPjv/+9/Z1nGPfcczvL44MGD9VpleWHZcRzXdS3LIkpAKoBSoJTbJgCYyhJxPB6LJEqD8ThNU9u0CCGcEAlKgUyzia+GBGookmRCjsNSEFVst7WwdPhI/NwLP8TMraiHuWOr1Rncf9wlhFJCq6kQ7qjo2rftEeEuckGoKm5FxTdeZz74s/hFQkgQBLZtoxYFFYhobqF3WtyQzMjiMaCUMspQ7/6d73znq1/9qhBZq9UUMjUMI02iiuvX69WVpdWFpfmF1uLcfHOhtViullzbo5yY3GIGNSmvV0u1UkmBEGmGxynO0jCIsyRO4yRKYtfxCSGcG9wwpaIAjBt2pd6o1Zu4JLmI7zs2zM/g/mOuJEmkzPWSXt12gWmHvzjTNAlsBIrrfzHV0fS5bmFq/t5xHMuy8NGQRryFDYtGkmYnGWWU0t2tna2trSeeeGJ5eRFAxkmwt7c36He7OztXN69cuXAxSuMsTlOREQmKQtkrGbZZ8cu1Zn2+2Wq0mvVKzXHsQxsbJuOmY1uWxQ3Tc1zP8RWBdqeXZRkAYdzERQ1KgeP6R44cqVQqmkTSd60Z3O/KK01T7B1O+o7T7RfFNe0IQSklJcRxHMpArxHWObpOY4oHAw8Dmkfj94RhaFlWrVYbDAY3puyEECWvHQZ9clBl0Gw2f+mXP3HffUcvXDg3HPX6/T5nyuXmeDDAZb9hGOLWpCRJ+v1+HMej0ejiufOvvXISSwtKiMzSSrm8uLS0sbFx6MjR9UMbS4sHvHKJU5ZGcRSmQoLvVdM0N03Tcby1tTWt39RpzB1brc6mmYocjMR0AvGUJEkUBXk8ie5aPIgwRVFUUV9FCCEUDKBA5L4L0Vw8Gzo/SdPUcRzci4Qad/SBiaJIl7n4pJ7n+eVymsVxkghBDG7atsuYQQmnlF69ejUIAsqgVPKUEqNxXylVLjlBbwBT7zuYztTiqCEeRf3y0jQVeR4H4ygMe71ep9sN48ywzFp1zq+U7z/x4KEjhw3L3drZduzSsfuPl8vlvb29MA4OHDhAKY3jGL1/Z6Xq3XHtC0uI/iRJUDxSxPr+lPra/07KWk0U7lvofmNmotuiRflAgWcE7KHingzDMPySG0ZRHOdK7n8oxhg3sB1Lpqk/dT1PSQFCSaIYUAFS5TIVmW07mcxlJgRIIiGTuUjzPM9yz7dNy7IsoeQoCAeD0Wgcpmn6/HPPbW5uHr7nnsXlVcfxBp02UaI1P3fxQhAEAZr93sn91Bncb12gTgRSURTpZAbv2ppvKfY7NdylkApuEt2LON5Xa+ov4ue6D69n4fSClziOS2VPKiVlmsSpEAItF/EyDMOyDcuypMzRVYZSatiulLkSAFRxagBVSkAus9wRucxEJiUIkESoXGRSyIxJMDnnnAulymnebCZpJqSUnf5gb2/vtZOvjoejAwcO1up1lXqgJOd8PB4XNfT4W8y6qnf0FccxKlUwPuHgKaa5QmRFu/5r6vDrRV0wlZ5ruOsAr/XrNxagWp+o8xbM9VFaXLRVStM0l9K0uFQqz3GiQjEmCUgcGjRN0/Vsx7GyLMFhU0opAFMKFFOEEMDfjgNVJmEZyQ3CcjzDXClhCpkL17CyNMVhPikE49z3HdM0V1YObO3uXLl46crlS1EYHD16tFbx42HfNnkirt3x7vDceAb36wItghKbPoPBYDQaqCxXStzIo2PE1e/uNEJP4K4DuT4POorvy4V0aq5V6Yh4y7L0aYHpKF2SZb1ez7JtSifPLiX6+6ZoyeT7rmEwAIkeGJRSkSsJVCpJgFCglHJCCCglMyGBSqBq0jAGShglkhimgapLL8/iJI7j4XCcZVmaxvV6/ei9R/b29ra2tl790UsyTQ4fPUYsG19qkZOZiQju9EubRqRpOhqNer3eYDAIgpFBKCEKFQGYRuOgUBGO18pQkEoI3Wbal9zfFO56cRfeMTCZ0UyOPk44oASUxnFsmCbnOOanlFJCyCRJ0BDG8xwAmWUJCiqRq5dSkFwBA0IN4IwRLomkmQSAieiNcEUVU1RxBYoAkYRSg3PmctTMpGlqcDocDFLbrPhe4NuXLlzM40BIubBxeG5xCX1H6vU6iiJnROTdwcyMx+PhcDgYDIIgwAkdSinn1JxeOB6Bk/l6POIa9wISpARybbq0mOcUbgKgj4GO+kUPDx3UYaqumeTxhARBYJgmAI/jRCnCuYkn0LZt13UN25JZovkiAFCEUkGFlECUIgwIU4QRSoFyoiRQQoAA4YwTCkwpRRVkcZKkuRKSG9TzvGq5RAjxXPvyhYvt9m4i8pLrNhv10Wj43Pe/93O23VpaHgwGW1tbaMmE61dncL+jryzLwmDU6/W63e54PMbs2TJNizPDMBzbth3HtizGOSUECNETRFJKgmhGr14pCVUEGKFAgFEKBBgQKQUoEKAoEAGKEoLfgyWdVAoIQZZGKEWUEoSYmAnneZrneZYlUgJhbDgYc8PKc0jilDHDdX3T4kAsw2SmxWHar0WeCYAKIZVSikhQIEGAoooIgiEcw7sCIJISzgA1P5wQYtgWIcqgTIHIkjRLEynlgY31pdWVq1evDHqdheVlefXq2fOX93Y7QohgNN7Z2Tl8+HCpVEKJ234OCoDM4P7uX5gckKlpep7nnBHDMPZ2t0f9XrvdHo/HoKRlWZxTg6jFhQXbtsuez1wHFIDI4yCM0iRP0lTkIs3iLFW5EKCoAk4osx1MXtBlTORCykxKwRiflsIm7rQRQuap4IYwDJbnMo5DpYjjWHkug2BkWRYhSkqRZYpzkxAVReEoDAgYUZRcurj5wQ9+cGdnZzwelkoe57br2sDw5AgAUHibIYpSwhnQSZ0AILM8l7jKTwnBckEI4ZRwUIwQCSoRkQKlQCmpMgkEKOHUMFygNAIgpllbXnVrzU6nY45zr5E8+9zzxx98qFap5Gk2Ho7q1Vq1Wu31eij81HWrhIndDpnB/d288lxyPhnfxMkgANne2zv92qtJkqRJzDmvV8tzc3P1et3zPMNwgFJgbPLWScU5dwAGUYzbwUzGJaHmtITFFmmxBqaUoCKSMUaIpJRrhyOlZBgNhRBSioJnkySECJHh8+W5mCb3pFQqddqDPBOO47Tb7U6nMzc3Nxz2KaXlsg9KwLTdCwCgCK6FIVSBklDofIIUjCgFUoIkQKiSRAoJuCZECIQ7KFCUEEmASgCDcUkZJZxwRm1qlUSpGTcysXvy5XNnzx4/frzkeVh1UM6KxqiIdf2RzeD+bl6M0SnuJy7P7b29H/3oR9nUNU7TIKgJM2wfCEEAqCzTikjU6xZt4rTblu6hFr+OOjCdjk+bVsIwOcC1tisWqXme9/t9JGqSJMFCQgihKJOCcmPIOT99+jQh5IEHHiiXy9yyVJ5j/aHr5iLpNIE6IdraEgrfoL8oQUmCu20myuKpMm3SKFYw6Qm4rttoNAghp0+9+uqrr544caLVaukD9Z67hc3grt/7a4R3FEXj8fjK5ctBEGRxRAixTMP3/Xq93mq1qs0mUAqKXkONaZIkyfMcu/2aOizKtvbVr5qk9zxvwhtOYY1wz/JMyhy/qItXxthwODRN03GcSqVSrVYnBKUCgztxkoZhePLkyWazKYQYj8dVwyCUglJyKtHROKaMSQFFthRfAy1Uz7rvK5QUFMQ1uGMBjUVFKqUERcXUoRJ/o6WlpatXr0opW60WmzqRoKPYDO53Av0CaZraloHtwHPnzo2G/Wazub15BfkNXOSCRSKYJkgJUoosQ3EVmkygGd0+zh6D6ORnr/dlZ4zt7OwUo6kWCoRRiEtP9cFAmNm2myTJaNTGk4Mrqsdh7DjeYDgKw/DMmTM///M/jyPYeZZx04QpZ1/QacK+Llix0SulVKjuAjW550ghKQECasoLKQBCQAEAw7EVKfJ8stLMMKnnLy4unjt3rtPpLC8ve6USZUzkcqZ3v1MuSq/1hjqdzssvv9zt7C0uLs43G6Zpuo6NWUeSJMPh0LIsyyklUYTsZDi1y8Jp6KIgUS9d6vV6RbJcu5w2m82ir8tkpZ4iS0tLWZbglU0/y7LszJkze3t7V65cabfb6FcahuE4jKvVupDKtu1Wq3X8+PFqo1E8RVqAqSP3tdGQgoxHCIGdX6Jw/Ko4EFhIZkDhTAkhhBVMyyYn2TQNw6jVapzzra2tVqtlOw4A6J0lM7jfKR0lQkApNRgMNjc3X3n5JSHE4fWDvu/PNRvNZrNZr9br9Wq16nlee6+nnUFxTwaiHE/FPnIdAI4cOaJhV4C16vV6mAXh/QEtSPM8++GLP8CR7Wh6hWGIOw7QqRTH+SzL4twol800TRk3sixbW1s7duxYHAQ4/G+5LkxlbUV7o6LyDIrCNVThKyDF6UQAKAibJ8kfASAkS1IwDACqhAQgjFDTMMAwarXa8vIyjjICQBLHlmPPuqp3VoDHUhX3ZIzH47Nnz37vu//g+369Vq1UKiXPQSNpy7JWVw7atu37vu/7WIEhiMvlss6DdRNUCNHr9YQQOvPRu2uiKEK4I6Zxm4CUslwuSTU5EkWJmJQShe/F2lcoQkiO+XGtVqs3m912u1arkYJ1cBGs+xQsRYE+5vf0BgX49SJ+pcnbNE2UUoQwrW7Hg1Sv1w8cOIA3MQCIoshy7Dv2rf9ZhHueS4NTwzDK5TL2Ssvlcsl1hBBBEARBwClod1LH/kERRjpe4i6xYqMUP+IWgxvTek3gaE2Bbdv4CaemwclUhaYs0yzaKg3DMZJItm1bjtVoNI8eO2aa5oc//OHRYFCpVAjnSRharouuHvh6UC5vWkaSJJQBpvhZnuhCdjJoQogQQgk5+UUIAAAFEFJOumYAhFBCiEE5VUAJoYSCkGkUW9yyPbdUKq2urh48eDAMQ5Hn1VoN/wiO48zgfmd0mvbruoAQQlHbBIox5tomDtRRSqMwLe6Z0bhHsO4LqEqpUqlUZGD0c+HA9b7dkUqpNE21NA0/weq23W57ntdoNObm5lrTyyuXlpaWHde9evVqo9HArQeeaVquGwcBDitp3hMADJOnea5P3/XSfHITgYNUQEDBhCOSk1erpJSMGoQQ07Q9z8P1O0mSDofDtY21ZrPZarXG4zGKz4CQ93ZX8Azu1zEzhc+l7/sHDx6s1+vnzpxO0zSJJyMdyFQwxgzDRp4B47QGN0br4igGgmYwGOzTtSOwut2urlxRaoZlgFIKCUdcyYSr8CzLOnHiBMpg0NURzwDhbDQa4dwQskBxHBtRpMU2cRzD1DcYALIsY5wzPq1Wc8EYIwowcZp4GxQISqVUmmdYiRJCGKGUEQFSAkmSpNlslkvVOI6vXtk0DGN9/VDj2L3f+uY3hJIbGxu2bVPGlJQEyIyIvKOYGaq7Oa1W65FHHjEM44VaNQzD0XAQBEEah1ibJkmSJqKIbw1fPBL76EgA8H2/SEEiUhE9WHEipvHuwTlvNpsId7wQ7rZth2GoDxjWykEQZFIIIQbDoZSy3W4fOXLE8zzTcQAg6HSwDhZC6DNDGbEchzJA6FM0ECG5lFJwDgBEgZRSgdI1rp6gBQAliT628/Pz/X6/vdet1WqHDx/2fb/XG/zw2ee+9rWvLa+ufOADH/B9HwDI1JZ+xszcOYgnSgK+o57nmQbzPK/suUmShME4CII4HGMSnySJFOTG3L3wUHTfQBNa+mujahzzwQlU7XWqF0cSQkSudJ0ghEiTXOTjYByhZ3xRZ08IkQRwaVmz2czz3C+XQSmZZaPRCHlSLH/x96pWq4ZlKABCFXZniVRSykwBbosHAMzaUUY22fENRFzj7wEAKGGEkN3tHUJIrVpbWV4ul0qbm5t//63vfP/555jNl5RyHIdMF7sSRtM0xWxqBvf3+BJCMUYoY1ikcs6jMB+NRrZtm6ZZ8j2lFBqco9OGY/sYX4XQGzQUsss3Gc0uyHoR0Bjjizst9jEhbsmFqV5NOx0opSqVivY+wEdDuAtJiBCMsR/96EeVSqVSqfi+j7SPYRiu62Jde83GmlIASem1hpfAOSzTVEpJIjTo8XkxHcJ70URdzE1KaRQl9XrdcbxLly6dPHny7Nmzw8HYsizDMXFNdp4klHMhhGnwOxPrP5u5u6SUEQKccwyBjIIQIk8mbzNjzOSmnqlTkhbhXlSi7yv+NKlflLZjj7NIzxfJGQC4cuVKcSQcG5bFyT3cLYO9XklgaXmh3el0u90vfvGLL7zwwokTJz784Q8vLCxcvHgRrcK013uWZULmpm0Teq1E1odtwm8CUUpJKvbdsnR/N03TMIiklLVa4/XXXz937kK328VhFN/34zTBZ8Q/kUnpHVuk/ozCXQ/aSClt256bm6vXKqhkFGkyiabkGhHJGFdKoVpdSVQXUkKVyBV+RX8ERRUIAgy/AkTid6KuPUuFVr0LAVLlmBk351sTM+vpSiaMwbZt4xnTX8H0+vTp067nzbdaAHD+3LnLly//8IUXciEee/TRhcXFhfl5v1QyOMfCFxhLooAQBUBFluW5VAKluJIzU6ocBBFUAGFACQBVRFQqNUWkzFUQhoN+f3dvr7PXGQXjky+/WqqUF1rzzdaca3tRHPe63VEwXDywVK5W7eslkGEYFrX/X10AAAuzSURBVCXvtCAAfm+vn0WfGb3dRSkVhmG320U9FiWqaP2FaEPbaJErqXIpQMOXMxMBrYGOHy3TAYKjoVIfAyAShzwmIx3Tz2G6RqbI3Ot3pLjcFJkimadSZJVK5dyZs1/6T3996uSrtueKNLtw+VLFLwlQvuPOLcwvzS/Umo1GteaVS/V63TQN07QoJQCEKmUYJufMMEwphZQKQAkhoygcDUbjcHz61OlOr7O1ubWztxuOA0XAYJxy1mrOLa+uzM+1oiTevroTJ8nC/Pzi6sqRe48urx1YXztYqVUn63FuVqTOxjvew1J1YiKAZZ9pmhMXUsrI9D5u2dfcyovmGcXy9EaMTlWNDBgonIQD9HthSilCOAAowKb9VKlCJkmMFujCtZXTZCLRogQAlARFCeTUZLzZbK6trF5441y/0zVNc3lhEV3BxoNhp9M5+/ppnRchR442Hro20FpOLc9MkmQ8HsdxjOIIvKVYluU6juu6pmlW67XhcDgajUql0qEjG7hHtlStRKksl8uWYxfXBt7Eu3iWzNwJiEcSEMO8Lj1xqh+zUg2IGxP3m0blfU3WYqje1+EqZv83ue1OWUid6lBKQVFJCPoUrK6urq6uBkEQRRFqOVE4CQC6dZplmeM4+CtMeNVp07dINGGhotfSFxdToo29aZpBEHDOG41Gq9W67777VldXDcOIs7Rareqp9jv/+lmfVcWNAwCAXqQ6kCM1gd0cKGwf0Jao13nLFJB90yF8rYfRykT99ckA9fUp5bWBjAILqZRSgvm+hWe12Wy+733vq1arp0+fvnjxIu7PQMMCZPRRzDwcDvXvpadPoNAP1jwM/rJIqqDIJ45j3Mzhed7y8jKK0g4cOOC6Lv5BQFH0b4qiSD/CbNHkHXTtk75gPoNMhRZ7FYMrxv59TmDFGL8P7tfkh9d/vBEHRaM8nVwVUa4/ueY0xpjjua7tKMJs119dW19cXl1YWnnxxRe73S7t9/v9fhinw3GILWHTNBcWFoQQAIIwwikvvh49tJUJkYkcn+jS2XOWZbmu67ru8tz80tLSwYMHFxYWVldX5+fnW60WqjujKPJ9v96qUm5mIkcP+Dt2iOlnt1TVFr4acwjlwWCAmzkQ7jpGYjtmn15c9x1/LNyLGf9Nkxl9N9h3QoruYvoOI7LcNvlcoxkEwalTp/r9fqVSUUrt7u52Op12u727u9vr9fr9fr/fx74v/kZo3VGUPLDrV0xiOmSa5tzcnO/7zemFip1yuZxlGVpVapMz3/dd3ytVaqNgDAD1eh1lYUj8z0zz7ozzfb3zKExXZ/m+n01HURHKeDBwz6Me1dPUzU3zdZ2jk2vl5rVDVQzkxc9vvP8UE3fNzOD/cm4mWZ7mghkmUBanme/7Bw6uHzi4Ph6Px+Mxyg36/X4YhrjNHZeO6FXDWIqgSxQ2fTGc+75v2/bq6ioqHZDpn4wjKmVYtpSSUOqXPbRwwoOaCZEkyb7zj8dpBvc7pUjVd3P9leKCc10a6v8tHpIiS3PTx7/pvxYlNzc2m4ppzI3fcO3wUNJstYLRKM9zHCPSpSoC2rZtdLxAMbCU8qGHHsKtIXjHQBTiyUEjBjzDenBxNBrhb6FVNMjJTow0lMKgYFmW7/vUMC5dvjQcDgkhlUqlVCrdNG2bwf3Ogv7kb8G54zi4BkxXpVjb6TlrbYOox59vTJZuStcUk5ZiAYAdeF1KajkNciz6MGi7VqWUYVlmlgVRlEsJlBqGYZqmMd3/gS/AchzLcfD12K67b8sIuf7C3ysTIg9D23X174uiGWYYzDAo516pVCqVUHyGUjBQKs9zdOLGM4Z50WyN8F2T6miZbjHA71OH3yKA3Tqw3chJ31R5dqNths6jkiTB2Iw1NAZ1nUvoU7HvjnSjIFlbjBQLEqVUkiT66fj0QnIWk3vO+WR4avqz+pTum2Ocwf3ugDtyNVp8eyOOb+Ff/mbtp2LGUvy4D9m3qDcQTHGaAQBh3CuVHc/Xy2ckkH2w2+dBWXyooj+rTlr0VzTWtYpO22Jy02KGgU0joUAIOXGRNwz9bbO9qncHQbnv/o4IQIXgTanxWz8avIX2U9F66cabxr4dTFN37EkpjEDEiBuGISo09bPrvtjUruy6lnBx2FR7GuOP41HX7sS6Rocb1mFjLEAm17IsbRY7i+53WYDHt1n7+sL+gWV4M2XILYjIInMPNzME3pdS61x/X57NOZcCQIGShACzTIMAMw07NMJitNZwf7PoXszc9mmYi2ebEvSdIQAgBYhcCQacg1IgBaDBhmVZmsYpVt4zuN/RKL9pIasXxRRz3FtE9xsHYfdx7TeVGNyanSym4IyaBLR3Rq4l9deckq5fowA3a43d9B6C/6u3e+u0pPjJ9C9ApQSk8zmfCC7u5B01M7i/1Qs5hxt3Lb1duN80cdcPcmORt2/bh64sCVVACaUM50qpFJg3c9OQUgImMNMqU2cyxQT9xiN03bI0OvGdwZRO/4cMFZKhioBQKhN5nCRYwmp3kOto0xncfzooyzfbpHfrcvPNKJobvZn0ebhRHixExjmnlAuRIS2OTtaMMbS6ljJXilEKAJwQJaWSMpcShMikBClz1J8TwlCFrh3l8XPX9YpfIUQRQgGkZZlC5GpqK6aXb7qOpUc67vwO/Wyv6k94YZW2r+VECHkzPv7Ncgk9LLcvq36zx5kGbFx5wAhR+NGynDxPs0wgTPEJhcgQ1vojjnpMxy1u8pFSjsdDKYGsJucmYwShTyknROW5TBL0hooWFxeLFpk6FsxGs386M/596cFbfJv3/dQtkuk3uUvgvITUH3HJh5QoZ1eEYGDmUk6+B9eE4EvGfQ43/UgIIlUqXDEycecguA0KAISQWYbm4OIOH9Wbwf02l7bF5j/csn+uv+1GSVkR3z825b1pFYt3G92a1acO1yjceIpu8SxFEUQxbGuhMg6tY5V8FyndZ3C/zXzOWynO9pWGRXHBvod9izWAToG0NG3fc92I8lu/zn0Ei/4EH1+7qRFCkH+cwf1nC+j7wLFv3Glf7n5tT8Y06S/C/W3VUcVWa7FRqp+uKMPcdw95K49cbHvpOSmk87EvgfqifVb3M7j/rCQzRS3KLfL1Ym8fbtDB33iE3qzkLTaJKKWodbnx6fZt635boC+mXpqEBQC9c/Oue9dmcL8NpWoRQ29Wqhb17vtK1X0FwK1Ja81tF4VcyOTsWzu8j/9+i3Df10QrVhrI3KNKXrcjZnD/2SVn4E1mVW+R6O8jHH9sPnPjjnl8KF2qanWN9kl9W6H9RgVbscbQU7z4eZIkdxc5M+Pdb/OlFZRv8Xoz54If+/06umsBWRiG4/EYN1piAC76k73F11n0z9iXzLCp0yCeBJwKmHgFzwTAM67mLcL9RnHB230KHXR1qx8ZQ/LmVuu3SLrgTWZqbwT0W68EZnCfXTepAX4s4vf9a5GNwehLppuEse97C13+jy2Fi5SOdngtMj/71JozuM+i+9tjeN6WvurG7yyWrUUu5Raw3ndpzfo+UYN26IYpnQpTU5C3xXLO4D47HrdC8JslP/uie5qmN6Xh325tpufTb5Tx6BvCdI0UFOdOZtF9Bt+3FKHhZkYdP/bx96XX2P3RhIz+5jeL4m+WzBRz/Zsyp5gm6WbTvvvADO4zuL+NZOatPMi+3F3/iPYb04BG+eTbep1vBnHt8VSck7rpzCHMpplm19uqWX+C77m2/abgFrYvWhevNyMibxw+3Pd5sSS465iZGe/+Hl8/tpTcR8zfOglR119wy7bXW4H7jTF+n/zzzveFvC6FmwHuzk+K3mIQfVteCf/0l3cXBfUZ3O8axL8trcstpkZuO9DvRsTPcvc7Ool/62T8jUnpO4rFIt0+g/vsus35zJth+qb/9E6gcJ9SUgvUZsnM7LrNOcNbTGZuugLknXttd0traQb3n7bE/d3M3eGu6ivN4H5XIv6tZCnvDjNzN6L82ouf8e531/V2RQGzaxbdZ9fP6PX/AyrkUt3Rjwr1AAAAAElFTkSuQmCC"
	)

	type args struct {
		inputFile string
	}
	tests := []struct {
		name    string
		args    args
		want    string
		wantErr bool
	}{
		{
			name: "TestEncodeMarvinJPEG",
			args: args{
				inputFile: "testdata/marvin.jpg",
			},
			want:    base64MarvinJPEG,
			wantErr: false,
		},
		{
			name: "TestEncodeMarvinPNG",
			args: args{
				inputFile: "testdata/marvin.png",
			},
			want:    base64MarvinPNG,
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := LoadBase64ImageFromFile(tt.args.inputFile)
			if (err != nil) != tt.wantErr {
				t.Errorf("LoadBase64ImageFromFile() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("LoadBase64ImageFromFile() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestGetImageFileType(t *testing.T) {
	type args struct {
		inputFile string
	}
	tests := []struct {
		name    string
		args    args
		want    string
		wantErr bool
	}{
		{
			name: "MarvinJPEG",
			args: args{
				inputFile: "testdata/marvin.jpg",
			},
			want:    "image/jpeg",
			wantErr: false,
		},
		{
			name: "MarvinPNG",
			args: args{
				inputFile: "testdata/marvin.png",
			},
			want:    "image/png",
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := GetImageFileType(tt.args.inputFile)
			if (err != nil) != tt.wantErr {
				t.Errorf("GetImageFileType() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("GetImageFileType() got = %v, want %v", got, tt.want)
			}
		})
	}
}
