﻿using Microsoft.Extensions.Options;
using Moq;
using Synopackage.Model.Caching.Enums;
using Synopackage.Model.Caching.Exceptions;
using Synopackage.Model.DTOs;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Synopackage.Model.Tests;

public class CacheOptionsManagerTests
{
  private readonly CacheSettings testDefaults;
  private readonly VersionDTO testVersion = new VersionDTO()
  {
    Major = 6,
    Minor = 2,
    Micro = 4,
    Build = 25556,
    Version = "6.2.4-25556",
    ShortVersion = "6.2.4"
  };

  public CacheOptionsManagerTests()
  {
    testDefaults = new CacheSettings()
    {
      CacheIconExpirationInDays = 20,
      CacheSpkServerResponse = true,
      CacheSpkServerResponseTimeInHours = 21,
      CacheSpkServerResponseTimeInHoursForRepository = 22,
      ArchCacheLevel = Caching.Enums.ArchCacheLevel.None,
      VersionCacheLevel = Caching.Enums.VersionCacheLevel.Major,
      ChannelCacheLevel = Caching.Enums.ChannelCacheLevel.Fixed
    };
  }

  #region Test autogenerated code
  [Fact]
  public void GetCacheIconExpirationInDaysFromDefaults()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheIconExpirationInDays();

    //assert
    result.ShouldBe(20);
  }

  [Fact]
  public void GetCacheIconExpirationInDaysFromDefaultsWhenSourceIsNotDefined()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheIconExpirationInDays("other");

    //assert
    result.ShouldBe(20);
  }

  [Fact]
  public void GetCacheIconExpirationInDaysFromSource()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { CacheIconExpirationInDays = 30 } }
        }
    });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheIconExpirationInDays("test");

    //assert
    result.ShouldBe(30);
  }

  [Fact]
  public void GetCacheSpkServerResponseFromDefaults()
  {
    //arrange
    testDefaults.CacheSpkServerResponse = true;
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponse();

    //assert
    result.ShouldBe(true);
  }

  [Fact]
  public void GetCacheSpkServerResponseFromDefaultsWhenSourceIsNotDefined()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponse("other");

    //assert
    result.ShouldBe(true);
  }

  [Fact]
  public void GetCacheSpkServerResponseFromSource()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { CacheSpkServerResponse = false } }
        }
    });
    CacheOptionsManager manager = new CacheOptionsManager(options);


    //act
    var result = manager.GetCacheSpkServerResponse("test");

    //assert
    result.ShouldBe(false);
  }

  [Fact]
  public void GetCacheSpkServerResponseTimeInHoursFromDefaults()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponseTimeInHours();

    //assert
    result.ShouldBe(21);
  }

  [Fact]
  public void GetCacheSpkServerResponseTimeInHoursFromDefaultsWhenSourceIsNotDefined()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponseTimeInHours("other");

    //assert
    result.ShouldBe(21);
  }

  [Fact]
  public void GetCacheSpkServerResponseTimeInHoursFromSource()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { CacheSpkServerResponseTimeInHours = 30 } }
        }
    });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponseTimeInHours("test");

    //assert
    result.ShouldBe(30);
  }

  [Fact]
  public void GetCacheSpkServerResponseTimeInHoursForRepositoryFromDefaults()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponseTimeInHoursForRepository();

    //assert
    result.ShouldBe(22);
  }

  [Fact]
  public void GetCacheSpkServerResponseTimeInHoursForRepositoryFromDefaultsWhenSourceIsNotDefined()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponseTimeInHoursForRepository("other");

    //assert
    result.ShouldBe(22);
  }

  [Fact]
  public void GetCacheSpkServerResponseTimeInHoursForRepositoryFromSource()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { CacheSpkServerResponseTimeInHoursForRepository = 30 } }
        }
    });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetCacheSpkServerResponseTimeInHoursForRepository("test");

    //assert
    result.ShouldBe(30);
  }

  [Fact]
  public void GetArchCacheLevelFromDefaults()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetArchCacheLevel();

    //assert
    result.ShouldBe(ArchCacheLevel.None);
  }

  [Fact]
  public void GetArchCacheLevelFromDefaultsWhenSourceIsNotDefined()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetArchCacheLevel("other");

    //assert
    result.ShouldBe(ArchCacheLevel.None);
  }

  [Fact]
  public void GetArchCacheLevelFromSource()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { ArchCacheLevel =  ArchCacheLevel.None} }
        }
    });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetArchCacheLevel("test");

    //assert
    result.ShouldBe(ArchCacheLevel.None);
  }

  [Fact]
  public void GetVersionCacheLevelFromDefaults()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetVersionCacheLevel();

    //assert
    result.ShouldBe(VersionCacheLevel.Major);
  }

  [Fact]
  public void GetVersionCacheLevelFromDefaultsWhenSourceIsNotDefined()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetVersionCacheLevel("other");

    //assert
    result.ShouldBe(VersionCacheLevel.Major);
  }

  [Fact]
  public void GetVersionCacheLevelFromSource()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { VersionCacheLevel =  VersionCacheLevel.Build} }
        }
    });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetVersionCacheLevel("test");

    //assert
    result.ShouldBe(VersionCacheLevel.Build);
  }

  [Fact]
  public void GetChannelCacheLevelFromDefaults()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetChannelCacheLevel();

    //assert
    result.ShouldBe(ChannelCacheLevel.Fixed);
  }

  [Fact]
  public void GetChannelCacheLevelFromDefaultsWhenSourceIsNotDefined()
  {
    //arrange
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetChannelCacheLevel("other");

    //assert
    result.ShouldBe(ChannelCacheLevel.Fixed);
  }

  [Fact]
  public void GetChannelCacheLevelFromSource()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { ChannelCacheLevel =  ChannelCacheLevel.Requested} }
        }
    });
    CacheOptionsManager manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetChannelCacheLevel("test");

    //assert
    result.ShouldBe(ChannelCacheLevel.Requested);
  }



  [Fact]
  public void CacheSpkServerResponseTimeInHours_SetToNegativeNumber_ShouldThrowException()
  {
    //arrange
    testDefaults.CacheSpkServerResponseTimeInHours = -1;
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { CacheSpkServerResponseTimeInHoursForRepository = 30 } }
        }
    });

    //act
    var exception = Record.Exception(() => new CacheOptionsManager(options));

    //assert
    exception.ShouldBeOfType<FluentValidation.ValidationException>();
    exception.Message.ShouldContain("'Cache Spk Server Response Time In Hours'");
  }

  [Fact]
  public void CacheSpkServerResponseTimeInHoursForRepository_SetToNegativeNumber_ShouldThrowException()
  {
    //arrange
    testDefaults.CacheSpkServerResponseTimeInHoursForRepository = -1;
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { CacheSpkServerResponseTimeInHoursForRepository = 30 } }
        }
    });

    //act
    var exception = Record.Exception(() => new CacheOptionsManager(options));

    //assert
    exception.ShouldBeOfType<FluentValidation.ValidationException>();
    exception.Message.ShouldContain("'Cache Spk Server Response Time In Hours For Repository'");
  }

  [Fact]
  public void CacheSpkServerResponseTimeInHoursForSource_SetToNegativeNumber_ShouldThrowException()
  {
    //arrange
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
      {
        { "test", new CacheSettings() { CacheSpkServerResponseTimeInHours = -1} }
      }
    });

    //act
    var exception = Record.Exception(() => new CacheOptionsManager(options));

    //assert
    exception.ShouldBeOfType<FluentValidation.ValidationException>();
    exception.Message.ShouldContain("'Cache Spk Server Response Time In Hours'");
  }

  [Fact]
  public void CacheSpkServerResponseTimeInHoursForRepositoryForSource_SetToNegativeNumber_ShouldThrowException()
  {
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
        {
          { "test", new CacheSettings() { CacheSpkServerResponseTimeInHoursForRepository = -1 } }
        }
    });
    //act
    var exception = Record.Exception(() => new CacheOptionsManager(options));

    //assert
    exception.ShouldBeOfType<FluentValidation.ValidationException>();
    exception.Message.ShouldContain("'Cache Spk Server Response Time In Hours For Repository'");
  }

  #endregion

  #region cache levels tests

  [Theory]
  [InlineData(ArchCacheLevel.CPU, "apollolake", "apollolake")]
  [InlineData(ArchCacheLevel.CPU, "armada37xx", "armada37xx")]
  [InlineData(ArchCacheLevel.None, "apollolake", "allCPUs")]
  [InlineData(ArchCacheLevel.None, "armada37xx", "allCPUs")]
  [InlineData(ArchCacheLevel.OnlyListed, "apollolake", "apollolake")]
  [InlineData(ArchCacheLevel.OnlyListed, "armada37xx", "allCPUs")]
  public void GetGetArchStringForCacheFileFromDefaults(ArchCacheLevel archCacheLevel, string arch, string expected)
  {
    //arrange
    testDefaults.ArchCacheLevel = archCacheLevel;
    testDefaults.ArchList = new List<string>() { "apollolake" };
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetArchStringForCacheFile(arch);

    //assert
    result.ShouldBe(expected);
  }

  [Theory]
  [InlineData(ArchCacheLevel.CPU, "apollolake", "apollolake")]
  [InlineData(ArchCacheLevel.CPU, "armada37xx", "armada37xx")]
  [InlineData(ArchCacheLevel.None, "apollolake", "allCPUs")]
  [InlineData(ArchCacheLevel.None, "armada37xx", "allCPUs")]
  [InlineData(ArchCacheLevel.OnlyListed, "apollolake", "apollolake")]
  [InlineData(ArchCacheLevel.OnlyListed, "armada37xx", "allCPUs")]
  public void GetGetArchStringForCacheFileFromDefaulsWithNotDefinedSource(ArchCacheLevel archCacheLevel, string arch, string expected)
  {
    //arrange
    testDefaults.ArchCacheLevel = archCacheLevel;
    testDefaults.ArchList = new List<string>() { "apollolake" };
    var options = Options.Create(new CacheOptions() { Defaults = testDefaults });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetArchStringForCacheFile(arch, "test");

    //assert
    result.ShouldBe(expected);
  }

  [Theory]
  [InlineData(ArchCacheLevel.CPU, "apollolake", "allCPUs")]
  [InlineData(ArchCacheLevel.CPU, "armada37xx", "allCPUs")]
  [InlineData(ArchCacheLevel.None, "apollolake", "allCPUs")]
  [InlineData(ArchCacheLevel.None, "armada37xx", "allCPUs")]
  [InlineData(ArchCacheLevel.OnlyListed, "apollolake", "allCPUs")]
  [InlineData(ArchCacheLevel.OnlyListed, "armada37xx", "allCPUs")]
  public void GetGetArchStringForCacheFileFromSource_ArchCacheLevelSetToNone(ArchCacheLevel archCacheLevel, string arch, string expected)
  {
    //arrange
    testDefaults.ArchCacheLevel = archCacheLevel;
    testDefaults.ArchList = new List<string>() { "apollolake" };
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
      {
        { "synotest", new CacheSettings() { ArchCacheLevel = ArchCacheLevel.None } }
      }

    });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetArchStringForCacheFile(arch, "synotest");

    //assert
    result.ShouldBe(expected);
  }

  [Theory]
  [InlineData(VersionCacheLevel.Build, "25556")]
  [InlineData(VersionCacheLevel.Major, "DSM6")]
  [InlineData(VersionCacheLevel.Minor, "DSM6-2")]
  [InlineData(VersionCacheLevel.Micro, "DSM6-2-4")]
  public void GetVersionStringForCacheFileFromDefaults(VersionCacheLevel versionCacheLevel, string expected)
  {
    //arrange
    testDefaults.VersionCacheLevel = versionCacheLevel;
    testDefaults.ArchList = new List<string>() { "apollolake" };
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults
    });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetVersionStringForCacheFile(testVersion);

    //assert
    result.ShouldBe(expected);
  }

  [Theory]
  [InlineData(VersionCacheLevel.Build, "25556")]
  [InlineData(VersionCacheLevel.Major, "DSM6")]
  [InlineData(VersionCacheLevel.Minor, "DSM6-2")]
  [InlineData(VersionCacheLevel.Micro, "DSM6-2-4")]
  public void GetVersionStringForCacheFileFromDefaultsWithNotDefinedSource(VersionCacheLevel versionCacheLevel, string expected)
  {
    //arrange
    testDefaults.VersionCacheLevel = versionCacheLevel;
    testDefaults.ArchList = new List<string>() { "apollolake" };
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults
    });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetVersionStringForCacheFile(testVersion, "synotest");

    //assert
    result.ShouldBe(expected);
  }

  [Theory]
  [InlineData(VersionCacheLevel.Build, "DSM6")]
  [InlineData(VersionCacheLevel.Major, "DSM6")]
  [InlineData(VersionCacheLevel.Minor, "DSM6")]
  [InlineData(VersionCacheLevel.Micro, "DSM6")]
  public void GetVersionStringForCacheFileFromSource(VersionCacheLevel versionCacheLevel, string expected)
  {
    //arrange
    testDefaults.VersionCacheLevel = versionCacheLevel;
    testDefaults.ArchList = new List<string>() { "apollolake" };
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
      {
        { "synotest", new CacheSettings() { VersionCacheLevel = VersionCacheLevel.Major } }
      }
    });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetVersionStringForCacheFile(testVersion, "synotest");

    //assert
    result.ShouldBe(expected);
  }

  [Theory]
  [InlineData(ChannelCacheLevel.Requested, false, "stable")]
  [InlineData(ChannelCacheLevel.Requested, true, "beta")]
  [InlineData(ChannelCacheLevel.Fixed, false, "stable")]
  [InlineData(ChannelCacheLevel.Fixed, true, "stable")]
  public void GetChannelStringForCacheFileFromDefaults(ChannelCacheLevel channelCacheLevel, bool isBeta, string expected)
  {
    //arrange
    testDefaults.ChannelCacheLevel = channelCacheLevel;
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults
    });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetChannelStringForCacheFile(isBeta);

    //assert
    result.ShouldBe(expected);

  }

  [Theory]
  [InlineData(ChannelCacheLevel.Requested, false, "stable")]
  [InlineData(ChannelCacheLevel.Requested, true, "beta")]
  [InlineData(ChannelCacheLevel.Fixed, false, "stable")]
  [InlineData(ChannelCacheLevel.Fixed, true, "stable")]
  public void GetChannelStringForCacheFileFromDefaultsForNotDefinedSource(ChannelCacheLevel channelCacheLevel, bool isBeta, string expected)
  {
    //arrange
    testDefaults.ChannelCacheLevel = channelCacheLevel;
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults
    });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetChannelStringForCacheFile(isBeta, "synotest");

    //assert
    result.ShouldBe(expected);

  }

  [Theory]
  [InlineData(ChannelCacheLevel.Requested, false, "stable")]
  [InlineData(ChannelCacheLevel.Requested, true, "stable")]
  [InlineData(ChannelCacheLevel.Fixed, false, "stable")]
  [InlineData(ChannelCacheLevel.Fixed, true, "stable")]
  public void GetChannelStringForCacheFileFromSource(ChannelCacheLevel channelCacheLevel, bool isBeta, string expected)
  {
    //arrange
    testDefaults.ChannelCacheLevel = channelCacheLevel;
    var options = Options.Create(new CacheOptions()
    {
      Defaults = testDefaults,
      SourcesOverrides = new Dictionary<string, CacheSettings>()
      {
        { "synotest", new CacheSettings() { ChannelCacheLevel = ChannelCacheLevel.Fixed } }
      }
    });
    var manager = new CacheOptionsManager(options);

    //act
    var result = manager.GetChannelStringForCacheFile(isBeta, "synotest");

    //assert
    result.ShouldBe(expected);

  }

  #endregion

}

