package co.uk.swarmbit.docker.cli.impl;

import co.uk.swarmbit.docker.api.nodes.NodesApi;
import co.uk.swarmbit.docker.api.nodes.parameters.NodeDeleteParameters;
import co.uk.swarmbit.docker.api.services.parameters.ServiceInspectParameters;
import co.uk.swarmbit.docker.api.tasks.parameters.TasksListParameters;
import co.uk.swarmbit.docker.cli.model.Node;
import co.uk.swarmbit.docker.api.common.json.NodeJson;
import co.uk.swarmbit.docker.api.common.json.ServiceJson;
import co.uk.swarmbit.docker.api.common.json.TaskJson;
import co.uk.swarmbit.docker.api.common.json.inner.ManagerStatusJson;
import co.uk.swarmbit.docker.api.nodes.parameters.NodeRole;
import co.uk.swarmbit.docker.api.nodes.parameters.NodeUpdateParameters;
import co.uk.swarmbit.docker.api.nodes.parameters.NodesListParameters;
import co.uk.swarmbit.docker.api.services.ServicesApi;
import co.uk.swarmbit.docker.api.tasks.TasksApi;
import co.uk.swarmbit.docker.api.tasks.parameters.TasksFilters;
import co.uk.swarmbit.docker.cli.NodeCli;
import co.uk.swarmbit.docker.cli.impl.helper.TaskJsonHelper;
import co.uk.swarmbit.docker.cli.model.NodeSummary;
import co.uk.swarmbit.docker.cli.model.State;
import co.uk.swarmbit.docker.cli.model.Task;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Component
public class NodeCliImpl implements NodeCli {

    private final NodesApi nodesApi;

    private final TasksApi tasksApi;

    private final ServicesApi servicesApi;

    @Autowired
    public NodeCliImpl(NodesApi nodesApi, TasksApi tasksApi, ServicesApi servicesApi) {
        this.nodesApi = nodesApi;
        this.tasksApi = tasksApi;
        this.servicesApi = servicesApi;
    }

    @Override
    public List<NodeSummary> ls(String swarmId) {
        List<NodeSummary> nodeSummaryList = new ArrayList<>();
        TasksFilters filters = new TasksFilters();
        filters.setDesiredState(TasksFilters.RUNNING_STATE);
        TasksListParameters parameters = new TasksListParameters();
        parameters.setFilters(filters);
        List<TaskJson> taskJsons = tasksApi.listTasks(swarmId, parameters);
        nodesApi.listNodes(swarmId, new NodesListParameters()).forEach(nodeJson -> {
            NodeSummary nodeSummary = new NodeSummary();
            nodeSummary.setId(nodeJson.getId());
            nodeSummary.setHostname(nodeJson.getDescription().getHostname());
            nodeSummary.setAvailability(nodeJson.getSpec().getAvailability());
            nodeSummary.setStatus(nodeJson.getStatus().getState());
            ManagerStatusJson managerStatusJson = nodeJson.getManagerStatus();
            Long runningTasks = taskJsons.stream()
                    .filter(task -> {
                        boolean filter = StringUtils.equals(task.getStatus().getTaskState(), TasksFilters.RUNNING_STATE)
                                && StringUtils.equals(task.getNodeId(), nodeJson.getId());
                        return filter;
                    }).count();
            nodeSummary.setNumberOfRunningTasks(runningTasks);
            nodeSummary.setManager(StringUtils.equals(nodeJson.getSpec().getRole(), NodeRole.MANAGER.toString().toLowerCase()));
            if (managerStatusJson != null) {
                nodeSummary.setLeader(managerStatusJson.isLeader());
                nodeSummary.setManagerReachability(managerStatusJson.getReachability());
            }
            nodeSummaryList.add(nodeSummary);
        });
        return nodeSummaryList;
    }

    @Override
    public State ps(String swarmId, String nodeId) {
        TasksFilters filters = new TasksFilters().setNode(nodeId);
        TasksListParameters parameters = new TasksListParameters();
        parameters.setFilters(filters);
        NodeJson nodeJson = nodesApi.inspectNode(swarmId, nodeId);
        List<TaskJson> taskJsons = tasksApi.listTasks(swarmId, parameters);
        List<Task> tasks = TaskJsonHelper.getTasks(taskJsons);
        Map<String, List<Task>> tasksByService = new HashMap<>();
        tasks.forEach(task -> {
            task.setNodeHostname(nodeJson.getDescription().getHostname());
            List<Task> tasksAux = tasksByService.computeIfAbsent(task.getServiceId(), k -> new ArrayList<>());
            tasksAux.add(task);
        });
        for (Map.Entry<String, List<Task>> entry : tasksByService.entrySet()) {
            ServiceJson serviceJson = servicesApi.inspectService(swarmId, entry.getKey(),
                    new ServiceInspectParameters()
                    .setInsertDefaultsQueryParam(true));
            if (serviceJson != null) {
                for (Task task : entry.getValue()) {
                    task.setServiceName(serviceJson.getSpec().getName());
                }
            }
        }
        TaskJsonHelper.sortTasks(tasks);
        State state = new State();
        state.setTasks(tasks);
        return state;
    }

    @Override
    public Node inspect(String swarmId, String nodeId) {
        NodeJson nodeJson = nodesApi.inspectNode(swarmId, nodeId);
        Node node = new Node();
        node.setId(nodeId);
        node.setAddr(nodeJson.getStatus().getAddr());
        node.setAvailability(nodeJson.getSpec().getAvailability());
        node.setEngineVersion(nodeJson.getDescription().getEngine().getEngineVersion());
        node.setLabels(nodeJson.getSpec().getLabels());
        node.setLeader(nodeJson.getManagerStatus().isLeader());
        node.setHostname(nodeJson.getDescription().getHostname());
        node.setMemoryBytes(nodeJson.getDescription().getResources().getMemoryBytes());
        node.setNanoCPUs(nodeJson.getDescription().getResources().getNanoCPUs());
        node.setReachability(nodeJson.getManagerStatus().getReachability());
        node.setRole(nodeJson.getSpec().getRole());
        node.setState(nodeJson.getStatus().getState());
        return node;
    }

    @Override
    public void rm(String swarmId, String nodeId, boolean force) {
        nodesApi.deleteNode(swarmId,nodeId, new NodeDeleteParameters().setForceQueryParam(force));
    }

    @Override
    public void update(String swarmId, String nodeId, Node node) {
        NodeJson nodeJson = nodesApi.inspectNode(swarmId, nodeId);
        if (node.getRole() != null) {
            nodeJson.getSpec().setRole(node.getRole());
        }
        if (node.getAvailability() != null) {
            nodeJson.getSpec().setAvailability(node.getAvailability());
        }
        if (node.getLabels() != null) {
            nodeJson.getSpec().setLabels(node.getLabels());
        }
        NodeUpdateParameters parameters = new NodeUpdateParameters();
        parameters.setVersionQueryParam(nodeJson.getVersion().getIndex());
        parameters.setNode(nodeJson.getSpec());
        nodesApi.updateNode(swarmId, nodeId, parameters);
    }
    
}
