/** @import { AST, ExpressionMetadata, SvelteNode } from '#compiler' */
/**
 * All nodes that can appear elsewhere than the top level, have attributes and can contain children
 */
const element_nodes = [
	'SvelteElement',
	'RegularElement',
	'SvelteFragment',
	'Component',
	'SvelteComponent',
	'SvelteSelf',
	'SlotElement'
];

/**
 * Returns true for all nodes that can appear elsewhere than the top level, have attributes and can contain children
 * @param {SvelteNode} node
 * @returns {node is AST.Component | AST.RegularElement | AST.SlotElement | AST.SvelteComponent | AST.SvelteElement | AST.SvelteFragment | AST.SvelteSelf}
 */
export function is_element_node(node) {
	return element_nodes.includes(node.type);
}

/**
 * @param {AST.RegularElement | AST.SvelteElement} node
 * @returns {node is AST.RegularElement}
 */
export function is_custom_element_node(node) {
	return node.type === 'RegularElement' && node.name.includes('-');
}

/**
 * @param {string} name
 * @param {number} start
 * @param {number} end
 * @param {AST.Attribute['value']} value
 * @returns {AST.Attribute}
 */
export function create_attribute(name, start, end, value) {
	return {
		type: 'Attribute',
		start,
		end,
		name,
		value,
		parent: null,
		metadata: {
			expression: create_expression_metadata(),
			delegated: null
		}
	};
}

/**
 * @returns {ExpressionMetadata}
 */
export function create_expression_metadata() {
	return {
		dependencies: new Set(),
		has_state: false,
		has_call: false
	};
}
