import 'package:flutter/foundation.dart';
import 'package:flutter/services.dart';
import 'package:flutter/widgets.dart';
import 'package:super_editor/src/core/document.dart';
import 'package:super_editor/src/core/document_selection.dart';
import 'package:super_editor/src/infrastructure/_logging.dart';
import 'package:super_editor/src/infrastructure/platforms/ios/ios_document_controls.dart';
import 'package:super_editor/src/infrastructure/platforms/mac/mac_ime.dart';

import 'document_delta_editing.dart';
import 'document_serialization.dart';
import 'ime_decoration.dart';

/// Sends messages to, and receives messages from, the platform Input Method Engine (IME),
/// for the purpose of document editing.

/// A [TextInputClient] that applies IME operations to a [Document].
///
/// Ideally, this class *wouldn't* implement [TextInputConnection], but there are situations
/// where this class needs to care about what's sent to the IME. For more information, see
/// the [setEditingState] override in this class.
class DocumentImeInputClient extends TextInputConnectionDecorator with TextInputClient, DeltaTextInputClient {
  DocumentImeInputClient({
    required this.selection,
    required this.composingRegion,
    required this.textDeltasDocumentEditor,
    required this.imeConnection,
    required this.onPerformSelector,
    this.floatingCursorController,
  }) {
    // Note: we don't listen to document changes because we expect that any change during IME
    // editing will also include a selection change. If we listen to documents and selections, then
    // we'll attempt to serialize the document change before the selection change is made. This
    // results in a new document with an old selection and blows up the serializer. By listening
    // only to selection, we void this race condition.
    selection.addListener(_onContentChange);
    composingRegion.addListener(_onContentChange);
    imeConnection.addListener(_onImeConnectionChange);

    if (attached) {
      _sendDocumentToIme();
    }
  }

  void dispose() {
    selection.removeListener(_onContentChange);
    composingRegion.removeListener(_onContentChange);
    imeConnection.removeListener(_onImeConnectionChange);
  }

  /// The document's current selection.
  final ValueListenable<DocumentSelection?> selection;

  /// The document's current composing region, which represents a section
  /// of content that the platform IME is thinking about changing, such as spelling
  /// autocorrection.
  final ValueListenable<DocumentRange?> composingRegion;

  final TextDeltasDocumentEditor textDeltasDocumentEditor;

  final ValueListenable<TextInputConnection?> imeConnection;

  /// Handles a selector generated by the IME.
  ///
  /// For the list of selectors, see [MacOsSelectors].
  final void Function(String selectorName) onPerformSelector;

  // TODO: get floating cursor out of here. Use a multi-client IME decorator to split responsibilities
  late FloatingCursorController? floatingCursorController;

  /// Whether the floating cursor is being displayed.
  ///
  /// This value is updated on [updateFloatingCursor].
  bool _isFloatingCursorVisible = false;

  /// Whether or not a `TextInputAction.newline` was performed on the current frame.
  bool _hasPerformedNewLineActionThisFrame = false;

  void _onContentChange() {
    if (!attached) {
      return;
    }
    if (_isApplyingDeltas) {
      return;
    }

    _sendDocumentToIme();
  }

  void _onImeConnectionChange() {
    client = imeConnection.value;

    if (attached) {
      // This is a new IME connection for us. As far as we're concerned, there is no current
      // IME value.
      _currentTextEditingValue = const TextEditingValue();
      _platformTextEditingValue = const TextEditingValue();

      _sendDocumentToIme();
    }
  }

  /// Override on [TextInputConnection] base class.
  ///
  /// This method is the reason that this class extends [TextInputConnectionDecorator].
  /// Ideally, this object would be exclusively responsible for responding to IME
  /// deltas, and some other object would be exclusively responsible for sending the
  /// document to the IME. However, in certain situations, the decision to send the
  /// document to the IME depends upon knowledge of recent deltas received from the
  /// IME. As a result, this class is not only responsible for applying deltas to
  /// the editor, but also making some decisions about when to send new values to the
  /// IME. This method provides an override to do that, with minimal impact on other
  /// areas of responsibility.
  @override
  void setEditingState(TextEditingValue newValue) {
    if (_isApplyingDeltas) {
      // We're in the middle of applying a series of text deltas. Don't
      // send any updates to the IME because it will conflict with the
      // changes we're actively processing.
      editorImeLog.fine("Ignoring new TextEditingValue because we're applying deltas");
      return;
    }

    editorImeLog.fine("Wants to send a value to IME: $newValue");
    editorImeLog.fine("The current local IME value: $_currentTextEditingValue");
    editorImeLog.fine("The current platform IME value: $_currentTextEditingValue");
    if (newValue != _platformTextEditingValue) {
      // We've been given a new IME value. We compare its value to _platformTextEditingValue
      // instead of _currentTextEditingValue. Why is that?
      //
      // Sometimes the IME reports changes to us, but our document doesn't change
      // in ways that's reflected in the IME.
      //
      // Example: The user has a caret in an empty paragraph. That empty paragraph
      // includes a couple hidden characters, so the IME value might look like:
      //
      //     ". |"
      //
      // The ". " substring is invisible to the user and the "|" represents the caret at
      // the beginning of the empty paragraph.
      //
      // Then the user inserts a newline "\n". This causes Super Editor to insert a new,
      // empty paragraph node, and place the caret in the new, empty paragraph. At this
      // point, we have an issue:
      //
      // This class still sees the TextEditingValue as: ". |"
      //
      // However, the OS IME thinks the TextEditingValue is: ". |\n"
      //
      // In this situation, even though our desired TextEditingValue looks identical to what it
      // was before, it's not identical to what the operating system thinks it is. We need to
      // send our TextEditingValue back to the OS so that the OS doesn't think there's a "\n"
      // sitting in the edit region.
      editorImeLog.fine(
          "Sending forceful update to IME because our local TextEditingValue didn't change, but the IME may have:");
      editorImeLog.fine("$newValue");
      imeConnection.value?.setEditingState(newValue);
    } else {
      editorImeLog.fine("Ignoring new TextEditingValue because it's the same as the existing one: $newValue");
    }

    _currentTextEditingValue = newValue;
    _platformTextEditingValue = newValue;
  }

  @override
  AutofillScope? get currentAutofillScope => null;

  @override
  TextEditingValue get currentTextEditingValue => _currentTextEditingValue;
  TextEditingValue _currentTextEditingValue = const TextEditingValue();

  // What the platform IME *thinks* the current value is.
  TextEditingValue _platformTextEditingValue = const TextEditingValue();

  void _updatePlatformImeValueWithDeltas(List<TextEditingDelta> textEditingDeltas) {
    // Apply the deltas to the previous platform-side IME value, to find out
    // what the platform thinks the IME value is, right now.
    for (final delta in textEditingDeltas) {
      _platformTextEditingValue = delta.apply(_platformTextEditingValue);
    }
  }

  bool _isApplyingDeltas = false;

  @override
  void updateEditingValue(TextEditingValue value) {
    editorImeLog.shout("Delta text input client received a non-delta TextEditingValue from OS: $value");
  }

  @override
  void updateEditingValueWithDeltas(List<TextEditingDelta> textEditingDeltas) {
    if (textEditingDeltas.isEmpty) {
      return;
    }

    if (_isFloatingCursorVisible && textEditingDeltas.every((e) => e is TextEditingDeltaNonTextUpdate)) {
      // On iOS, dragging the floating cursor generates non-text deltas to update the selection.
      //
      // When dragging the floating cursor between paragraphs, we receive a non-text delta for the previously
      // selected paragraph when our selection already changed to another paragraph. If the previously selected
      // paragraph is bigger than the newly selected paragraph, a mapping error occurs, because we try
      // to select an offset bigger than the paragraph's length.
      //
      // As we already change the selection when the floating cursor moves, we ignore these deltas.
      return;
    }

    if (_hasPerformedNewLineActionThisFrame && defaultTargetPlatform == TargetPlatform.iOS) {
      // On iOS, pressing the new line action button can trigger the IME to try to apply suggestions
      // after we have already processed the new line insertion. This causes the deltas related to suggestions
      // to have offsets that are invalid for us. Ignore any new deltas on the same frame and forcefully
      // update the IME with our current state.
      imeConnection.value?.setEditingState(_currentTextEditingValue);
      return;
    }

    editorImeLog.fine("Received edit deltas from platform: ${textEditingDeltas.length} deltas");
    for (final delta in textEditingDeltas) {
      editorImeLog.fine("$delta");
    }

    final imeValueBeforeChange = currentTextEditingValue;
    editorImeLog.fine("IME value before applying deltas: $imeValueBeforeChange");

    _isApplyingDeltas = true;
    editorImeLog.fine("===================================================");
    // Update our local knowledge of what the platform thinks the IME value is right now.
    _updatePlatformImeValueWithDeltas(textEditingDeltas);

    // Apply the deltas to our document, selection, and composing region.
    textDeltasDocumentEditor.applyDeltas(textEditingDeltas);
    editorImeLog.fine("===================================================");
    _isApplyingDeltas = false;

    // Send latest doc and selection to IME
    _sendDocumentToIme();
  }

  bool _isSendingToIme = false;

  void _sendDocumentToIme() {
    if (_isApplyingDeltas) {
      editorImeLog
          .fine("[DocumentImeInputClient] - Tried to send document to IME, but we're applying deltas. Fizzling.");
      return;
    }

    if (_isSendingToIme) {
      editorImeLog
          .warning("[DocumentImeInputClient] - Tried to send document to IME, while we're sending document to IME.");
      return;
    }

    if (textDeltasDocumentEditor.selection.value == null) {
      // There's no selection, which means there's nothing to edit. Return.
      editorImeLog.fine("[DocumentImeInputClient] - There's no document selection. Not sending anything to IME.");
      return;
    }

    _isSendingToIme = true;
    editorImeLog.fine("[DocumentImeInputClient] - Serializing and sending document and selection to IME");
    editorImeLog.fine("[DocumentImeInputClient] - Selection: ${textDeltasDocumentEditor.selection.value}");
    editorImeLog.fine("[DocumentImeInputClient] - Composing region: ${textDeltasDocumentEditor.composingRegion.value}");
    final imeSerialization = DocumentImeSerializer(
      textDeltasDocumentEditor.document,
      textDeltasDocumentEditor.selection.value!,
      textDeltasDocumentEditor.composingRegion.value,
    );

    editorImeLog
        .fine("[DocumentImeInputClient] - Adding invisible characters?: ${imeSerialization.didPrependPlaceholder}");
    TextEditingValue textEditingValue = imeSerialization.toTextEditingValue();

    editorImeLog.fine("[DocumentImeInputClient] - Sending IME serialization:");
    editorImeLog.fine("[DocumentImeInputClient] - $textEditingValue");
    setEditingState(textEditingValue);
    editorImeLog.fine("[DocumentImeInputClient] - Done sending document to IME");

    _isSendingToIme = false;
  }

  @override
  void performAction(TextInputAction action) {
    editorImeLog.fine("IME says to perform action: $action");
    if (action == TextInputAction.newline) {
      textDeltasDocumentEditor.insertNewline();

      // Keep track that we have performed a new line action on this frame to work around an iOS timing issue,
      // where the iOS IME might report text deltas related to keyboard suggestions after we already processed
      // the new line action.
      //
      // See https://github.com/superlistapp/super_editor/issues/2007 for more information.
      _hasPerformedNewLineActionThisFrame = true;
      WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
        _hasPerformedNewLineActionThisFrame = false;
      });
    }
  }

  @override
  void performSelector(String selectorName) {
    editorImeLog.fine("IME says to perform selector: $selectorName");

    onPerformSelector(selectorName);
  }

  @override
  void performPrivateCommand(String action, Map<String, dynamic> data) {}

  @override
  void showAutocorrectionPromptRect(int start, int end) {}

  @override
  void updateFloatingCursor(RawFloatingCursorPoint point) {
    if (floatingCursorController == null) {
      return;
    }

    switch (point.state) {
      case FloatingCursorDragState.Start:
        _isFloatingCursorVisible = true;
        floatingCursorController!
          ..onStart()
          ..onMove(point.offset);
        break;
      case FloatingCursorDragState.Update:
        floatingCursorController!.onMove(point.offset);
        break;
      case FloatingCursorDragState.End:
        _isFloatingCursorVisible = false;
        floatingCursorController!.onStop();
        break;
    }
  }

  @override
  void connectionClosed() {
    editorImeLog.info("IME connection was closed");
  }
}
