import React, { useState, useMemo } from 'react';
import PropTypes from 'prop-types';
import { rgba } from 'polished';
import { styled } from '@storybook/theming';
import { ButtonToggle, Icon, Link, styles, TagLink, TagList } from '@storybook/design-system';
import useSiteMetadata from '../../../lib/useSiteMetadata';
import { SocialGraph, ListHeadingContainer, ListHeading } from '../../../basics';
import { AddonItem, AddonsGrid } from '../../../layout/integrations/addons';
import { RecipesList } from '../../../layout/integrations/recipes';
import { IntegrationsLayout } from '../../../layout/integrations';
import buildTagLinks from '../../../../util/build-tag-links';
import GatsbyLink from '../../../basics/GatsbyLink';

const { background, breakpoint, spacing, color, pageMargins, typography } = styles;

const PageHeaderContainer = styled.header`
  ${pageMargins}
  padding-top: 3rem;
  padding-bottom: 20px;
  display: flex;
  flex-direction: column;

  @media (min-width: ${breakpoint * 1.333}px) {
    padding-bottom: 40px;
    padding-top: 4rem;
    padding-bottom: ${spacing.padding.medium}px;
  }

  @media (min-width: ${breakpoint * 1.5}px) {
    flex-direction: row;
    padding-bottom: 30px;
  }
`;

const PageHeadingLeft = styled.div`
  flex: 1 1 auto;
`;

const PageHeading = styled.h1`
  font-size: ${typography.size.l1}px;
  line-height: ${typography.size.l2}px;
  font-weight: ${typography.weight.bold};
  margin-bottom: 1.25rem;
  color: ${color.darkest};

  @media (min-width: ${breakpoint * 1}px) {
    font-size: ${typography.size.l2}px;
    line-height: ${typography.size.l3}px;
    margin-bottom: 0.75rem;
  }
`;

const PageSubheading = styled.p`
  margin: 0; //reset
  font-size: ${typography.size.s3}px;
  line-height: 28px;
  color: ${color.darker};
`;

const PageHeadingRight = styled.div`
  display: none;
  font-size: ${typography.size.s2}px;
  line-height: ${typography.size.m1}px;
  font-weight: ${typography.weight.bold};

  @media (min-width: ${breakpoint * 1.5}px) {
    display: block;
    align-self: flex-start;
    justify-self: flex-end;
  }
`;

const PageHeader = () => (
  <PageHeaderContainer>
    <PageHeadingLeft>
      <PageHeading>Integrations</PageHeading>
      <PageSubheading>
        Integrate your tools with Storybook to connect workflows and unlock advanced features.
      </PageSubheading>
    </PageHeadingLeft>
    <PageHeadingRight>
      <GatsbyLink to="/docs/react/addons/integration-catalog/">
        <Icon icon="add" />
        Add your integration
      </GatsbyLink>
    </PageHeadingRight>
  </PageHeaderContainer>
);

const PopularTagsList = styled(TagList)`
  margin-bottom: calc(70px - 32px - 10px);
`;

const SectionHeader = styled.div`
  display: flex;
  align-items: center;
  justify-content: space-between;
  margin-bottom: ${spacing.padding.medium}px;
`;

const Title = styled.h3`
  font-weight: ${typography.weight.bold};
  font-size: ${typography.size.m2}px;
  line-height: ${typography.size.m3}px;
  color: ${color.darkest};
`;

const AddonCallout = styled(AddonItem)`
  background-color: ${background.positive};
  border-color: ${rgba(color.positive, 0.1)};
  margin-bottom: 3rem;

  &:hover {
    border-color: ${rgba(color.positive, 0.3)};
  }
`;

const PopularAddons = styled(AddonsGrid)`
  margin-bottom: 3rem;
`;

const PopularRecipes = styled(RecipesList)`
  margin-bottom: 3rem;
`;

export const IntegrationsCatalogHomeScreen = ({
  pageContext: { popularAddons = [], popularRecipes = [], trendingTags = [], vta = {} },
}) => {
  const { title, ogImageAddons, urls = {} } = useSiteMetadata();
  const { home } = urls;

  const tagLinks = useMemo(() => buildTagLinks(trendingTags), [trendingTags]);

  return (
    <>
      <SocialGraph
        title={`Integrations | ${title}`}
        desc="Integrations enable advanced functionality and unlock new workflows. Contributed by core maintainers and the amazing developer community."
        url={`${home}/integrations`}
        image={ogImageAddons}
      />
      <IntegrationsLayout currentPath="/integrations/" RenderHeader={PageHeader}>
        <PopularTagsList
          limit={6}
          tags={tagLinks.map(({ link, name }) => (
            <TagLink LinkWrapper={GatsbyLink} key={link} to={link}>
              {name}
            </TagLink>
          ))}
        />

        <SectionHeader>
          <Title>New to Storybook 8</Title>
        </SectionHeader>
        <AddonCallout key={vta.id} orientation="horizontal" {...vta} />

        <PopularAddons title="Popular addons" addonItems={popularAddons} />

        <section>
          <ListHeadingContainer>
            <ListHeading>Popular recipes</ListHeading>
          </ListHeadingContainer>
          <PopularRecipes recipeItems={popularRecipes} />
        </section>
      </IntegrationsLayout>
    </>
  );
};

IntegrationsCatalogHomeScreen.propTypes = {
  pageContext: PropTypes.shape({
    popularAddons: AddonsGrid.propTypes.addonItems.isRequired,
    popularRecipes: RecipesList.propTypes.recipeItems.isRequired,
  }).isRequired,
};

export default IntegrationsCatalogHomeScreen;
