﻿using System.Collections.Generic;
using Sitemapify.Umbraco.Config;
using Umbraco.Core.Models;
using Umbraco.Core.Models.PublishedContent;

namespace Sitemapify.Umbraco.Extensions
{
    public static class ContentExtensions
    {
        /// <summary>
        /// Determines whether this node should be shown in the sitemap generated by Sitemapify
        /// </summary>
        /// <param name="node">the node to inspect</param>
        /// <param name="propertyAlias">the property alias to inspect, the default will be retrieved from sitemapify settings</param>
        /// <returns></returns>
        public static bool ShowInSitemap(this IPublishedContent node, string propertyAlias = null)
        {
            return !node?.HideFromSitemap(propertyAlias) ?? false;
        }

        /// <summary>
        /// Determines whether this node should be hidden in the sitemap generated by Sitemapify
        /// </summary>
        /// <param name="node">the node to inspect</param>
        /// <param name="propertyAlias">the property alias to inspect, the default will be retrieved from sitemapify settings</param>
        /// <returns></returns>
        public static bool HideFromSitemap(this IPublishedContent node, string propertyAlias = null)
        {
            return node?.GetProperty(propertyAlias ?? SitemapifyUmbracoContentProviderSettings.Current.ExcludedFromSitemapPropertyAlias) != null && (bool) node?.GetProperty(propertyAlias ?? SitemapifyUmbracoContentProviderSettings.Current.ExcludedFromSitemapPropertyAlias).GetValue();
        }

        /// <summary>
        /// Determines whether this node's children should be hidden in the sitemap generated by Sitemapify
        /// </summary>
        /// <param name="node">the node to inspect</param>
        /// <param name="propertyAlias">the property alias to inspect, the default will be retrieved from sitemapify settings</param>
        /// <returns></returns>
        public static bool HideChildrenFromSitemap(this IPublishedContent node, string propertyAlias = null)
        {
            return node?.GetProperty(propertyAlias ?? SitemapifyUmbracoContentProviderSettings.Current.ExcludedChildrenFromSitemapPropertyAlias) != null && (bool) node?.GetProperty(propertyAlias ?? SitemapifyUmbracoContentProviderSettings.Current.ExcludedChildrenFromSitemapPropertyAlias).GetValue();
        }

        internal static IEnumerable<IPublishedContent> DescendantSitemapNodes(this IPublishedContent node, string hideFromSitemapPropertyAlias = null, string hideChildrenFromSitemapPropertyAlias = null)
        {
            if (!node.HideFromSitemap(hideFromSitemapPropertyAlias)) yield return node;
            if (!node.HideChildrenFromSitemap(hideChildrenFromSitemapPropertyAlias))
            {
                foreach (var child in node.Children)
                {
                    foreach (var grandChild in child.DescendantSitemapNodes(hideFromSitemapPropertyAlias, hideChildrenFromSitemapPropertyAlias))
                        yield return grandChild;
                }
            }
        }
    }
}
