// Copyright 2020 The SQLFlow Authors. All rights reserved.
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package testdata

// TextCNSQL is .sql format data sample of chinese news dataset (pre-processed)
var TextCNSQL = `CREATE DATABASE IF NOT EXISTS text_cn;
DROP TABLE IF EXISTS text_cn.train;
CREATE TABLE text_cn.train (
  id bigint(20) NOT NULL,
  class_id int(3) NOT NULL,
  class_name varchar(100) COLLATE utf8_unicode_ci NOT NULL,
  news_title text COLLATE utf8_unicode_ci,
  news_keywords varchar(255) COLLATE utf8_unicode_ci NOT NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;

INSERT INTO text_cn.train VALUES (6551700932705387022,0,'news_culture','1,2,3,4,5,6,7,8,9','保利集团,马未都,中国科学技术馆,博物馆,新中国');
INSERT INTO text_cn.train VALUES (6552368441838272771,0,'news_culture','10,11,8,12,13,14,15,16,17,18,19,16','');
INSERT INTO text_cn.train VALUES (6552407965343678723,0,'news_culture','20,21,22,23,24,25,26,27,28,29,16','');
INSERT INTO text_cn.train VALUES (6552332417753940238,0,'news_culture','30,31,32,33,34,35,36,37,38,39,40,16','');
INSERT INTO text_cn.train VALUES (6552475601595269390,0,'news_culture','41,42,31,43,16','');
INSERT INTO text_cn.train VALUES (6552387648126714125,0,'news_culture','20,21,44,45,46,47,48,27,28,29,16','');
INSERT INTO text_cn.train VALUES (6552271725814350087,0,'news_culture','31,42,49,50,51,16','');
INSERT INTO text_cn.train VALUES (6552452982015787268,0,'news_culture','52,53,54,48,29,16','');
INSERT INTO text_cn.train VALUES (6552400379030536455,0,'news_culture','20,21,55,56,57,48,29,27,28,16','');
INSERT INTO text_cn.train VALUES (6552339283632455939,0,'news_culture','20,21,58,59,60,61,26,62,28,29,16','');
INSERT INTO text_cn.train VALUES (6552387367334838792,0,'news_culture','63,58,48,62,64,65','林风眠,黄海归来步步云,秋山图,计白当黑,山水画,江山万里图,张大千,巫峡清秋图,活眼,山雨欲来图');
INSERT INTO text_cn.train VALUES (6552314684383429128,0,'news_culture','66,67,42,68,48,69,70,71,72,48,73,74','牡丹,收藏价值');
INSERT INTO text_cn.train VALUES (6552128476109865229,0,'news_culture','14,15,75,76,77,8,78,79,16','');
INSERT INTO text_cn.train VALUES (6552447749072093453,0,'news_culture','20,48,80,81,82,48,62,28,29,16','');
INSERT INTO text_cn.train VALUES (6552301380562846215,0,'news_culture','83,84,85,86','叶浅予,田世光,李苦禅,花鸟画,中央美术学院');
INSERT INTO text_cn.train VALUES (6552263394420850951,0,'news_culture','87,88,89,48,62,90,91,16','');
INSERT INTO text_cn.train VALUES (6552290314294395139,0,'news_culture','20,21,58,59,60,61,26,62,28,29,16','');
INSERT INTO text_cn.train VALUES (6552462208314376462,0,'news_culture','20,21,92,39,93,48,62,28,29,16','');
INSERT INTO text_cn.train VALUES (6552311866947797262,0,'news_culture','29,94,95,96,97,98,26,20,42,99,100,16','');
INSERT INTO text_cn.train VALUES (6552466638304707079,0,'news_culture','101,102,48,103,104,48,105,106,107,108,109,8,110,111,112,113','荷花,西湖,金粟词话,采莲女,念奴娇·赤壁怀古,林逋,荷叶');
INSERT INTO text_cn.train VALUES (6552431613437805063,1,'news_entertainment','114,39,115,116,117,118,48,119,120,8,121,122,123,124,125,126','佟丽娅,网络谣言,快乐大本营,李浩菲,谢娜,观众们');
INSERT INTO text_cn.train VALUES (6552320560913711629,1,'news_entertainment','114,127,39,128,129,130,131,74,114,132,133,74,128,134,135,136,74','汪涵,火星情报局,杨迪,主办方,谢娜,刘维');
INSERT INTO text_cn.train VALUES (6552390546051039747,1,'news_entertainment','137,138,139,140,16,141,142,34,48,143,144,145,34','飞纱,新娘,脱口秀,中国网,婚礼');
INSERT INTO text_cn.train VALUES (6552150358678831624,1,'news_entertainment','146,147,14,45,148,149,34,48,150,151,152,48,120,153,154,155,156,157','陆贞传奇,大红大紫,楚乔传,微博热搜,赵丽颖,花千骨,迪丽热巴,Angelababy');
INSERT INTO text_cn.train VALUES (6552408585177924099,1,'news_entertainment','158,159,160,161,162,163,8,164,165,166,167,48,4,2,168,17,159','戴上眼镜,刘德华,张翰,远大前程,杜志国,刘亦菲');
INSERT INTO text_cn.train VALUES (6552147830184608263,1,'news_entertainment','169,8,170,48,171,172','电影院,前任3,刘若英,张一白,田壮壮');
INSERT INTO text_cn.train VALUES (6552472345548685837,1,'news_entertainment','173,174,175,176,177,178,48,179,180,181,182,183,48,184,185,186','金刚狼3,休·杰克曼,神奇女侠,绯红女巫,超人,金刚狼');
INSERT INTO text_cn.train VALUES (6552385284682547716,1,'news_entertainment','187,188,189,190,191,192,193,194,195,196,197,198,199,200,201','张绍刚,新组合,腾讯视频,无限歌谣季,毛不易,父子');
INSERT INTO text_cn.train VALUES (6552364464715334151,1,'news_entertainment','202,203,204,205,206,207,208,209,48,210,209,120,144,42,27,211,8,212','中岛美嘉,滨崎步,张靓颖,演唱会,林子祥');
INSERT INTO text_cn.train VALUES (6552310157706002702,1,'news_entertainment','213,214,215,216,217,34,48,218,219,220,123,221,222,48,4,27,172,16','');
INSERT INTO text_cn.train VALUES (6552286735408038403,1,'news_entertainment','223,199,224,199,225,48,226,159,151,227,228,229,74','王全安,张柏芝,张雨绮,吴卓林,谢霆锋');
INSERT INTO text_cn.train VALUES (6552269871697101315,1,'news_entertainment','230,231,8,232,233,74,234,228,235,236,237,238,2,74,239,240,241','天天向上,毒鸡汤,大张伟,美男子,综艺,我去上学了,百变大咖秀');
INSERT INTO text_cn.train VALUES (6552418723179790856,1,'news_entertainment','114,242,243,48,244,245,48,246,245,48,247,248,228,249','杜海涛,谢娜,何炅,沈梦辰,吴昕,快本');
INSERT INTO text_cn.train VALUES (6552283654494617859,1,'news_entertainment','62,250,146,16','');
INSERT INTO text_cn.train VALUES (6552453686398812423,1,'news_entertainment','14,15,251,252,8,253,254,16','');
INSERT INTO text_cn.train VALUES (6552383324696871427,1,'news_entertainment','199,255,256,201,8,257,48,258,86,228,259','高富帅,阿尔维斯,颜值,吴彦祖,罗德里戈');
INSERT INTO text_cn.train VALUES (6552390851157295629,1,'news_entertainment','260,261,262,263,264,193,228,265,266,267,268,269,270,271','杨幂,徐冬冬,背带裙,大唐荣耀,唐嫣,景甜');
INSERT INTO text_cn.train VALUES (6552445894711575043,1,'news_entertainment','272,21,4,273,274,14,275,48,150,276,8,275,277,278,4,124,74','孤儿院,吴卓林,小龙女,加拿大');
INSERT INTO text_cn.train VALUES (6552351831668818435,1,'news_entertainment','187,279,280,238,281,142,191,282,283,284,285,286,287,288,289,290,291,8,292,16','欢乐颂,西王母,老戏骨,赵雅芝,杨紫,刘嘉玲,新白娘子传奇');
INSERT INTO text_cn.train VALUES (6552474018920792580,1,'news_entertainment','128,42,233,293,294,48,295,187,296,191,297,48,298,299,300,301,302,74','浩南哥,黄秋生,龙虎风云,东陵大盗,爱恋狂潮,李子雄,锣鼓巷,古惑仔,学校风云,郑伊健,张耀扬,监狱风云,无间道2');
INSERT INTO text_cn.train VALUES (6552309039697494532,2,'news_sports','303,304,305,21,306,307,308,309,193,137,310,129,311,312','土库曼斯坦,乌兹别克斯坦,亚洲杯,赔率,小组赛');
INSERT INTO text_cn.train VALUES (6552477789642031623,2,'news_sports','313,314,315,316,317,318,319,320,193,321,322,323,324,325','黄紫昌,武磊,卡佩罗,惠家康,韦世豪');
INSERT INTO text_cn.train VALUES (6552495859798376712,2,'news_sports','326,327,328,304,48,329,330,331,332,16','');
INSERT INTO text_cn.train VALUES (6552202204621570574,2,'news_sports','173,333,74,334,335,333,336,337,338,339,340','替补出场,助攻,科斯塔,赫迪拉,博洛尼亚');
INSERT INTO text_cn.train VALUES (6552409639911162381,2,'news_sports','341,164,342,343,34,344,16,345,346,347,339,314,348,349,350,351','骑士,半决赛,猛龙,德罗赞,乐福');
INSERT INTO text_cn.train VALUES (6552470008188895491,2,'news_sports','326,352,353,354,355,356,48,15,357,358,128,16','');
INSERT INTO text_cn.train VALUES (6552316209847599367,2,'news_sports','331,359,360,48,4,27,172,16','');
INSERT INTO text_cn.train VALUES (6552295807171691022,2,'news_sports','165,361,362,363,74,364,365,366,367,368,16,369,370,371,97,372,373,34','PJ-塔克,斯奈德,大魔王,火箭,爵士');
INSERT INTO text_cn.train VALUES (6552452056043487748,2,'news_sports','374,375,376,16,377,378,48,379,380,8,381,42,382,383,384','威廉,曼联,穆里尼奥,布莱顿,马夏尔');
INSERT INTO text_cn.train VALUES (6552287279061139972,2,'news_sports','385,386,387,330,388,389,390,21,391,392,393,8,394','尤文,博洛尼亚,施泰纳,利希施泰纳,拉齐奥');
INSERT INTO text_cn.train VALUES (6552325997813825805,2,'news_sports','395,396,397,398,154,399,400,401,8,330,402,16','');
INSERT INTO text_cn.train VALUES (6552468503452975630,2,'news_sports','403,404,405,406,129,407,408,409,48,127,154,128,410,411,412,8,413,414,415,416','NBA,科勒,卡戴珊,四季酒店,克利夫兰骑士队');
INSERT INTO text_cn.train VALUES (6552312104492204301,2,'news_sports','417,418,419,420,48,421,422,423,424,425,426,4,27,172,16','');
INSERT INTO text_cn.train VALUES (6552332687678374151,2,'news_sports','326,427,341,304,48,331,428,429,8,430,14,353,165,16','');
INSERT INTO text_cn.train VALUES (6552466727995703815,2,'news_sports','431,432,433,434,435,164,436,437,193,438,439,128,440,441,16','阿洛,阿洛伊西奥,华夏幸福,埃尔纳内斯,穆里奇');
INSERT INTO text_cn.train VALUES (6552416247298916612,2,'news_sports','442,443,178,444,14,445,16','');
INSERT INTO text_cn.train VALUES (6552354211424633352,2,'news_sports','4,446,447,448,449,441,16,450,451,452,453,8,454,455','双冠王,欧冠赛场,贝里索,巴尔韦德,欧冠冠军');
INSERT INTO text_cn.train VALUES (6552314862163198471,2,'news_sports','456,74,331,457,458,338,459,460,461,48,462,463,464,465,466,161,467','猛龙,詹姆斯,伊巴卡,德罗赞,科沃尔');
INSERT INTO text_cn.train VALUES (6552304431164031502,2,'news_sports','468,456,74,469,8,470,471,27,472','德罗赞,詹姆斯');
INSERT INTO text_cn.train VALUES (6552404265724281357,2,'news_sports','473,348,474,475,476,477,48,478,479,480,481','J罗,科隆,拜仁,假动作,托利索');

CREATE DATABASE IF NOT EXISTS text_cn;
DROP TABLE IF EXISTS text_cn.train_processed;
CREATE TABLE text_cn.train_processed (
  id bigint(20) NOT NULL,
  class_id int(3) NOT NULL,
  class_name varchar(100) COLLATE utf8_unicode_ci NOT NULL,
  news_title text COLLATE utf8_unicode_ci,
  news_keywords varchar(255) COLLATE utf8_unicode_ci NOT NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;


INSERT INTO text_cn.train_processed VALUES (6551700932705387022,1,'news_culture','1,2,3,4,5,6,7,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','保利集团,马未都,中国科学技术馆,博物馆,新中国');
INSERT INTO text_cn.train_processed VALUES (6552368441838272771,1,'news_culture','10,11,8,12,13,14,15,16,17,18,19,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552407965343678723,1,'news_culture','20,21,22,23,24,25,26,27,28,29,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552332417753940238,1,'news_culture','30,31,32,33,34,35,36,37,38,39,40,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552475601595269390,1,'news_culture','41,42,31,43,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552387648126714125,1,'news_culture','20,21,44,45,46,47,48,27,28,29,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552271725814350087,1,'news_culture','31,42,49,50,51,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552452982015787268,1,'news_culture','52,53,54,48,29,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552400379030536455,1,'news_culture','20,21,55,56,57,48,29,27,28,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552339283632455939,1,'news_culture','20,21,58,59,60,61,26,62,28,29,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552387367334838792,1,'news_culture','63,58,48,62,64,65,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','林风眠,黄海归来步步云,秋山图,计白当黑,山水画,江山万里图,张大千,巫峡清秋图,活眼,山雨欲来图');
INSERT INTO text_cn.train_processed VALUES (6552314684383429128,1,'news_culture','66,67,42,68,48,69,70,71,72,48,73,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','牡丹,收藏价值');
INSERT INTO text_cn.train_processed VALUES (6552128476109865229,1,'news_culture','14,15,75,76,77,8,78,79,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552447749072093453,1,'news_culture','20,48,80,81,82,48,62,28,29,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552301380562846215,1,'news_culture','83,84,85,86,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','叶浅予,田世光,李苦禅,花鸟画,中央美术学院');
INSERT INTO text_cn.train_processed VALUES (6552263394420850951,1,'news_culture','87,88,89,48,62,90,91,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552290314294395139,1,'news_culture','20,21,58,59,60,61,26,62,28,29,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552462208314376462,1,'news_culture','20,21,92,39,93,48,62,28,29,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552311866947797262,1,'news_culture','29,94,95,96,97,98,26,20,42,99,100,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552466638304707079,1,'news_culture','101,102,48,103,104,48,105,106,107,108,109,8,110,111,112,113,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','荷花,西湖,金粟词话,采莲女,念奴娇·赤壁怀古,林逋,荷叶');
INSERT INTO text_cn.train_processed VALUES (6552431613437805063,2,'news_entertainment','114,39,115,116,117,118,48,119,120,8,121,122,123,124,125,126,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','佟丽娅,网络谣言,快乐大本营,李浩菲,谢娜,观众们');
INSERT INTO text_cn.train_processed VALUES (6552320560913711629,2,'news_entertainment','114,127,39,128,129,130,131,74,114,132,133,74,128,134,135,136,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','汪涵,火星情报局,杨迪,主办方,谢娜,刘维');
INSERT INTO text_cn.train_processed VALUES (6552390546051039747,2,'news_entertainment','137,138,139,140,16,141,142,34,48,143,144,145,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','飞纱,新娘,脱口秀,中国网,婚礼');
INSERT INTO text_cn.train_processed VALUES (6552150358678831624,2,'news_entertainment','146,147,14,45,148,149,34,48,150,151,152,48,120,153,154,155,156,157,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','陆贞传奇,大红大紫,楚乔传,微博热搜,赵丽颖,花千骨,迪丽热巴,Angelababy');
INSERT INTO text_cn.train_processed VALUES (6552408585177924099,2,'news_entertainment','158,159,160,161,162,163,8,164,165,166,167,48,4,2,168,17,159,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','戴上眼镜,刘德华,张翰,远大前程,杜志国,刘亦菲');
INSERT INTO text_cn.train_processed VALUES (6552147830184608263,2,'news_entertainment','169,8,170,48,171,172,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','电影院,前任3,刘若英,张一白,田壮壮');
INSERT INTO text_cn.train_processed VALUES (6552472345548685837,2,'news_entertainment','173,174,175,176,177,178,48,179,180,181,182,183,48,184,185,186,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','金刚狼3,休·杰克曼,神奇女侠,绯红女巫,超人,金刚狼');
INSERT INTO text_cn.train_processed VALUES (6552385284682547716,2,'news_entertainment','187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','张绍刚,新组合,腾讯视频,无限歌谣季,毛不易,父子');
INSERT INTO text_cn.train_processed VALUES (6552364464715334151,2,'news_entertainment','202,203,204,205,206,207,208,209,48,210,209,120,144,42,27,211,8,212,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','中岛美嘉,滨崎步,张靓颖,演唱会,林子祥');
INSERT INTO text_cn.train_processed VALUES (6552310157706002702,2,'news_entertainment','213,214,215,216,217,34,48,218,219,220,123,221,222,48,4,27,172,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552286735408038403,2,'news_entertainment','223,199,224,199,225,48,226,159,151,227,228,229,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','王全安,张柏芝,张雨绮,吴卓林,谢霆锋');
INSERT INTO text_cn.train_processed VALUES (6552269871697101315,2,'news_entertainment','230,231,8,232,233,74,234,228,235,236,237,238,2,74,239,240,241,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','天天向上,毒鸡汤,大张伟,美男子,综艺,我去上学了,百变大咖秀');
INSERT INTO text_cn.train_processed VALUES (6552418723179790856,2,'news_entertainment','114,242,243,48,244,245,48,246,245,48,247,248,228,249,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','杜海涛,谢娜,何炅,沈梦辰,吴昕,快本');
INSERT INTO text_cn.train_processed VALUES (6552283654494617859,2,'news_entertainment','62,250,146,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552453686398812423,2,'news_entertainment','14,15,251,252,8,253,254,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552383324696871427,2,'news_entertainment','199,255,256,201,8,257,48,258,86,228,259,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','高富帅,阿尔维斯,颜值,吴彦祖,罗德里戈');
INSERT INTO text_cn.train_processed VALUES (6552390851157295629,2,'news_entertainment','260,261,262,263,264,193,228,265,266,267,268,269,270,271,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','杨幂,徐冬冬,背带裙,大唐荣耀,唐嫣,景甜');
INSERT INTO text_cn.train_processed VALUES (6552445894711575043,2,'news_entertainment','272,21,4,273,274,14,275,48,150,276,8,275,277,278,4,124,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','孤儿院,吴卓林,小龙女,加拿大');
INSERT INTO text_cn.train_processed VALUES (6552351831668818435,2,'news_entertainment','187,279,280,238,281,142,191,282,283,284,285,286,287,288,289,290,291,8,292,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','欢乐颂,西王母,老戏骨,赵雅芝,杨紫,刘嘉玲,新白娘子传奇');
INSERT INTO text_cn.train_processed VALUES (6552474018920792580,2,'news_entertainment','128,42,233,293,294,48,295,187,296,191,297,48,298,299,300,301,302,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','浩南哥,黄秋生,龙虎风云,东陵大盗,爱恋狂潮,李子雄,锣鼓巷,古惑仔,学校风云,郑伊健,张耀扬,监狱风云,无间道2');
INSERT INTO text_cn.train_processed VALUES (6552309039697494532,3,'news_sports','303,304,305,21,306,307,308,309,193,137,310,129,311,312,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','土库曼斯坦,乌兹别克斯坦,亚洲杯,赔率,小组赛');
INSERT INTO text_cn.train_processed VALUES (6552477789642031623,3,'news_sports','313,314,315,316,317,318,319,320,193,321,322,323,324,325,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','黄紫昌,武磊,卡佩罗,惠家康,韦世豪');
INSERT INTO text_cn.train_processed VALUES (6552495859798376712,3,'news_sports','326,327,328,304,48,329,330,331,332,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552202204621570574,3,'news_sports','173,333,74,334,335,333,336,337,338,339,340,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','替补出场,助攻,科斯塔,赫迪拉,博洛尼亚');
INSERT INTO text_cn.train_processed VALUES (6552409639911162381,3,'news_sports','341,164,342,343,34,344,16,345,346,347,339,314,348,349,350,351,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','骑士,半决赛,猛龙,德罗赞,乐福');
INSERT INTO text_cn.train_processed VALUES (6552470008188895491,3,'news_sports','326,352,353,354,355,356,48,15,357,358,128,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552316209847599367,3,'news_sports','331,359,360,48,4,27,172,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552295807171691022,3,'news_sports','165,361,362,363,74,364,365,366,367,368,16,369,370,371,97,372,373,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','PJ-塔克,斯奈德,大魔王,火箭,爵士');
INSERT INTO text_cn.train_processed VALUES (6552452056043487748,3,'news_sports','374,375,376,16,377,378,48,379,380,8,381,42,382,383,384,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','威廉,曼联,穆里尼奥,布莱顿,马夏尔');
INSERT INTO text_cn.train_processed VALUES (6552287279061139972,3,'news_sports','385,386,387,330,388,389,390,21,391,392,393,8,394,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','尤文,博洛尼亚,施泰纳,利希施泰纳,拉齐奥');
INSERT INTO text_cn.train_processed VALUES (6552325997813825805,3,'news_sports','395,396,397,398,154,399,400,401,8,330,402,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552468503452975630,3,'news_sports','403,404,405,406,129,407,408,409,48,127,154,128,410,411,412,8,413,414,415,416,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','NBA,科勒,卡戴珊,四季酒店,克利夫兰骑士队');
INSERT INTO text_cn.train_processed VALUES (6552312104492204301,3,'news_sports','417,418,419,420,48,421,422,423,424,425,426,4,27,172,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552332687678374151,3,'news_sports','326,427,341,304,48,331,428,429,8,430,14,353,165,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552466727995703815,3,'news_sports','431,432,433,434,435,164,436,437,193,438,439,128,440,441,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','阿洛,阿洛伊西奥,华夏幸福,埃尔纳内斯,穆里奇');
INSERT INTO text_cn.train_processed VALUES (6552416247298916612,3,'news_sports','442,443,178,444,14,445,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552354211424633352,3,'news_sports','4,446,447,448,449,441,16,450,451,452,453,8,454,455,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','双冠王,欧冠赛场,贝里索,巴尔韦德,欧冠冠军');
INSERT INTO text_cn.train_processed VALUES (6552314862163198471,3,'news_sports','456,74,331,457,458,338,459,460,461,48,462,463,464,465,466,161,467,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','猛龙,詹姆斯,伊巴卡,德罗赞,科沃尔');
INSERT INTO text_cn.train_processed VALUES (6552304431164031502,3,'news_sports','468,456,74,469,8,470,471,27,472,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','德罗赞,詹姆斯');
INSERT INTO text_cn.train_processed VALUES (6552404265724281357,3,'news_sports','473,348,474,475,476,477,48,478,479,480,481,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','J罗,科隆,拜仁,假动作,托利索');
INSERT INTO text_cn.train_processed VALUES (6552375446955098627,4,'news_finance','482,483,484,353,485,486,487,488,489,490,48,491,492,493,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','金融,民间借贷,法治');
INSERT INTO text_cn.train_processed VALUES (6552249357389791758,4,'news_finance','137,2,494,8,495,48,496,497,498,154,499,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','互联网企业,养猪,分布式,家庭农场,新兴县,温氏集团,温北英,可口可乐,技术创新,周其仁,农业');
INSERT INTO text_cn.train_processed VALUES (6552392199802192391,4,'news_finance','500,501,502,125,503,504,505,506,193,169,391,507,508,509,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','信用卡,股票,银行,P2P,利息,余额宝');
INSERT INTO text_cn.train_processed VALUES (6552423505156112899,4,'news_finance','183,510,511,512,182,48,513,514,515,516,517,518,48,519,520,365,183,521,522,523,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','汇算清缴,纳税人,残疾人,汇总纳税企业,企业所得税税款');
INSERT INTO text_cn.train_processed VALUES (6552393713924964872,4,'news_finance','524,525,526,527,528,529,530,210,531,8,532,533,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','股权解禁,新五丰,鲁阳节能,解禁,大小非');
INSERT INTO text_cn.train_processed VALUES (6552284551496859908,4,'news_finance','183,534,535,48,536,537,31,538,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552465616924574222,4,'news_finance','539,540,541,129,542,543,544,545,546,547,548,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','还款日,账单日,还款,信用卡,免息期');
INSERT INTO text_cn.train_processed VALUES (6552379453605937422,4,'news_finance','549,550,551,48,552,553,554,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552381513936142862,4,'news_finance','555,556,21,557,558,559,560,561,193,562,563,524,564,565,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','心静如水,大阳线,MACD,操作,黄金,原油');
INSERT INTO text_cn.train_processed VALUES (6552128165496488456,4,'news_finance','566,484,567,568,569,570,571,21,572,573,199,574,575,576,70,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','海兰信,安永,中国企业,上海电气,一带一路,交通银行,创造可持续价值,上市,华建集团,阿里巴巴,国药控股');
INSERT INTO text_cn.train_processed VALUES (6552367589337596174,4,'news_finance','577,209,8,578,579,14,15,580,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552171081761817091,4,'news_finance','578,21,140,76,581,288,582,583,193,474,584,144,532,585,586,583,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','美国,金融界,巴菲特,民主党,洛克菲勒');
INSERT INTO text_cn.train_processed VALUES (6552135862929326600,4,'news_finance','587,199,588,589,201,48,590,199,591,592,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','中国电子信息产业集团,电子对抗,美亚柏科,雷科防务,华力创通,中国电子,旋极信息,军民融合,民营企业,中国电子科技集团公司');
INSERT INTO text_cn.train_processed VALUES (6552348488141636099,4,'news_finance','593,21,199,594,201,595,596,597,598,599,600,601,193,199,602,603,201,604,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','5050计划,高新区,创新创业,阿里巴巴,滨江,杭州市');
INSERT INTO text_cn.train_processed VALUES (6552493681381736963,4,'news_finance','605,606,607,576,608,609,48,210,531,610,607,611,612,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','季度业绩,业绩预告,华帝股份,上证指数,净利润增幅,业绩增长,高增长,有色金属');
INSERT INTO text_cn.train_processed VALUES (6552479093923774734,4,'news_finance','613,14,614,615,616,48,617,154,618,619,620,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552387539792036360,4,'news_finance','621,21,622,623,624,625,626,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','中兴通讯,BIS,中兴通讯股份有限公司');
INSERT INTO text_cn.train_processed VALUES (6552474712801280520,4,'news_finance','265,627,628,629,8,630,193,631,144,42,632,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','巴菲特,虚拟货币,伯克希尔哈撒韦公司,伯克希尔哈撒韦,比特币');
INSERT INTO text_cn.train_processed VALUES (6552356541591192067,4,'news_finance','633,634,199,323,635,201,636,636,137,637,427,638,639,640,641,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','中国房地产,最强大脑,超级CFO,房地产行业,房地产');
INSERT INTO text_cn.train_processed VALUES (6552451858437243395,4,'news_finance','642,643,644,645,646,48,150,105,647,648,649,607,650,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','巨人网络,药明康德,分众传媒,招股书,独角兽企业');
INSERT INTO text_cn.train_processed VALUES (6552303945463628301,6,'news_house','651,499,652,199,475,615,458,201,8,653,16,654,17,159,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','摇号,未来科技城,阳光城,小高层,酒店式公寓');
INSERT INTO text_cn.train_processed VALUES (6552357571120857607,6,'news_house','655,656,657,658,659,660,48,661,662,663,74,664,665,183,666,667,668,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','徐碧街道,宏图花园,城乡规划局,村民委员会,二手房');
INSERT INTO text_cn.train_processed VALUES (6552472831328780803,6,'news_house','669,670,21,671,672,128,8,673,48,674,8,538,48,675,676,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','期房,何鸿燊,分层产权制度,霍官泰,楼花');
INSERT INTO text_cn.train_processed VALUES (6552418506741121544,6,'news_house','677,678,679,680,8,681,682,34,74,683,172,48,684,681,8,685,74,42,686,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','房价,山东省第一季度薪酬表,二手房价格行政区二手房,山东,二手房');
INSERT INTO text_cn.train_processed VALUES (6552413070562427144,6,'news_house','687,8,688,689,690,691,48,536,27,692,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552300361216950531,6,'news_house','693,694,283,695,119,696,697,698,699,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552327554336817678,6,'news_house','700,701,702,404,703,704,705,636,636,706,707,708,709,48,710,711,712,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','都江堰,彭州,楼盘,天立世纪华府,㎡三室两厅两卫户型');
INSERT INTO text_cn.train_processed VALUES (6552296976585589251,6,'news_house','713,714,715,427,349,716,717,48,678,718,719,74,720,721,722,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','当涂,城市发展规划,金色里程,马鞍山,当涂县,马鞍山市');
INSERT INTO text_cn.train_processed VALUES (6552301698050687501,6,'news_house','723,724,21,674,725,726,727,728,193,729,514,730,731,732,601,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','税改革,土地财政,产权房,胡光辉,瞭望,不动产,房地产税,税费');
INSERT INTO text_cn.train_processed VALUES (6552494403947069704,6,'news_house','733,734,14,15,583,8,735,736,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6551907587439198723,6,'news_house','737,415,738,739,740,48,741,742,743,48,652,31,744,745,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','行政行为,诉讼时效,行政机关');
INSERT INTO text_cn.train_processed VALUES (6552272429509509389,6,'news_house','746,747,748,749,8,750,3,751,441,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552403967991611662,6,'news_house','752,8,681,42,753,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552316609770291726,6,'news_house','754,755,756,391,48,365,757,758,8,750,759,760,48,440,761,762,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','房价,户型,售楼,买房');
INSERT INTO text_cn.train_processed VALUES (6552397419261198606,6,'news_house','763,681,288,764,765,31,766,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552387849344254222,6,'news_house','763,681,288,764,765,31,766,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552295326462509575,6,'news_house','427,209,8,681,767,31,768,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','房地产泡沫,海南岛,房价,大洗牌,房地产开发商');
INSERT INTO text_cn.train_processed VALUES (6552327314498126083,6,'news_house','769,42,770,8,583,771,441,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','');
INSERT INTO text_cn.train_processed VALUES (6552114549741322766,6,'news_house','772,8,773,774,48,775,226,8,776,632,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','保利新武昌,首套房,购房者,购房合同,利率上浮');
INSERT INTO text_cn.train_processed VALUES (6552236298344595982,6,'news_house','777,428,778,34,74,532,779,780,307,781,782,782,783,784,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0','世茂,银座,高新区,奥特莱斯,青岛高新区,项目投资协议');`
